
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	Some virtual modules :
;;;;	
;;;;	ML : ml primitives.
;;;;
;;;;	Data : syntax for terms to be passed as data.
;;;;



(defun match-line (string line)
  (let ((line (string-left-trim '(#\space #\tab) line))
	(l (length string)))
    (and (>= (length line) l)
	 (string= string line :end2 l))))
    

;; write-line
;; soft-break
;; hard-break
;; very-hard-break

(defvar *outport*)

(defmacro with-outport ((port hardp) &body body)
  `(let ((*outport* ,port)
	 (*comment-line-buf* nil)
	 (*comment-firstp* t)
	 (*comment-hardp* ,hardp))
    ,@body))
  
(defvar *comment-line-buf* nil)
(defvar *comment-firstp* nil)
(defvar *comment-hardp* nil)
(defvar *comment-page-size* 63)

(defun comment-flush ()
  (when *comment-line-buf*
    (if *comment-firstp*
	(setf *comment-firstp* nil)
	(write-char #\page *outport*))

    (let ((count 0))
      (do ((lines (mapcan #'(lambda (x) (when x (list x)))
			  (list-divide (reverse *comment-line-buf*)
				       #'(lambda (x) (eql 'break x))
				       #'(lambda (x) x)
				       #'(lambda (x) (declare (ignore x)) nil)))
		  (cdr lines)))
	  ((null lines))

	(when (car lines)
	  (if (and (> (+ count (length (car lines))) *comment-page-size*)
		   (not (zerop count)))
	      (progn
		(write-char #\page *outport*)
		(setf count 0))
	      (write-char #\newline *outport*))

	  (dolist (line (car lines))
	    (write-line line *outport*))

	  (incf count (length (car lines))))))
	    
    (setf *comment-line-buf* nil)))

			      
(defun comment-push-line (line)
  (push line *comment-line-buf*))


;; expect type of hard or soft.
(defun comment-page (type)
  (if (eql 'hard type)
      (if *comment-hardp*
	  (comment-flush)
	  (unless (or (null *comment-line-buf*)
		      (eql 'break (car *comment-line-buf*)))
	    (push `break *comment-line-buf*)))
      (unless (or (null *comment-line-buf*)
		  (eql 'break (car *comment-line-buf*)))
	(push `break *comment-line-buf*))))

  

;;;;;;;;;;;;;;;;;;;;;;  extracting comments


(defvar *in-path* "./sys/src/")
(defvar *out-path* "./doc/src/")
(defvar *out-file* "foo.text")


(defparameter *files-with-extractable-comments* 
  `(("../../doc/src/doc-overview.text" "../../doc/src/doc-stats.text")
    ("bsc-lisp.lsp" "bsc-io.lsp")
    ("trm-defs.lsp" "trm-vars.lsp" "trm-term.lsp" 
     "trm-func.lsp" "trm-sos.lsp")
    ("bml-defs.lsp")
    ("mlt-defs.lsp" "mlt-term.lsp")
    ("com-defs.lsp" ;"com-ddag.lsp"
     "com-abs.lsp" "com-thm.lsp" "com-ref.lsp" "com-edd.lsp" )
    ("lib-defs.lsp" "lib-ref.lsp" "lib-edd.lsp" "lib-inf.lsp")
    ("ref-defs.lsp" "ref-mono.lsp" "ref-eq.lsp" "ref-simp.lsp" "ref-arth.lsp"
     "ref-dc.lsp" "ref-itrp.lsp" "ref-ref.lsp" "ref-ml.lsp" "ref-itd.lsp")
    ("edd-defs.lsp" "edd-form.lsp" "edd-tree.lsp")
    ("dms-defs.lsp")
    ("io-defs.lsp" "io-asc.lsp")
    ("itd-defs.lsp")
    ("ptr-defs.lsp")
    ("prs-defs.lsp")
    ("tst-defs.lsp" "tst-edd.lsp" "tst-lib.lsp" )))


(defparameter *primitives-for-tactics* 
  `(
    ("bml-defs.lsp")
    ("mlt-defs.lsp" "mlt-term.lsp")
    ("com-defs.lsp" ;"com-ddag.lsp"
     "com-abs.lsp" "com-thm.lsp" "com-ref.lsp" "com-edd.lsp" )
    ("ref-defs.lsp" "ref-mono.lsp" "ref-eq.lsp" "ref-simp.lsp" "ref-arth.lsp"
     "ref-dc.lsp" "ref-itrp.lsp" "ref-ref.lsp" "ref-ml.lsp" "ref-itd.lsp")
    ;; can refiner expect component eval? NO. for now at least.
    ;;("io-defs.lsp")
    ;;("itd-defs.lsp") 
    ))




;; would be nice to hack defsystem to provide map fucntion we could use.
;; then make module name in defsystem be the 3 char names.
(defvar *files-for-stats* 
  `(("../../doc/src/doc-overview.text" "../../doc/src/doc-stats.text")
    ("bsc-vend.lsp" "bsc-lisp.lsp" "bsc-io.lsp")
    ("trm-defs.lsp" "trm-vars.lsp" "trm-term.lsp" 
     "trm-func.lsp" "trm-sos.lsp")
    ("bml-lisp.lsp"
     "bml-io.lsp"
     "bml-defs.lsp"
     "bml-pars.lsp"
     "bml-pml.lsp"
     "bml-prin.lsp"
     "bml-type.lsp"
     "bml-dml.lsp"
     "bml-form.lsp"
     "bml-writ.lsp"
     "bml-tml.lsp"
     "bml-lis.lsp"
     "bml-comp.lsp"
     "bml-conv.lsp"
     "bml-run.lsp"
     "bml-chrn.lsp"
     "bml-tran.lsp"
     "bml-eval.lsp"
     )
    ("mlt-defs.lsp" "mlt-term.lsp")
    ("mbs-pkg.lsp" "mbs-reg.lsp" "mbs-vars.lsp" "mbs-trms.lsp" "io-mbs.lsp")
    ("io-defs.lsp"  "io-term.lsp" "io-comp.lsp" "io-db.lsp")
    ("orb-defs.lsp" "orb-eval.lsp" "orb-ml.lsp" "orb-jnl.lsp" "orb-mini.lsp"
     "orbm-bsc.ml"
     )
    ("com-defs.lsp" "com-ent.lsp" "com-tabl.lsp" "com-term.lsp" ;"com-ddag.lsp"
     "com-gc.lsp" "com-orb.lsp" "com-abs.lsp" "com-thm.lsp" "com-ref.lsp" "com-edd.lsp"
     "com-lib.lsp"
     "ini-defs.lsp"

     "general.ml" "primitives.ml" "term.ml" "subst.ml"
     "comm-bsc.ml" "comm-def.ml" "comm-dag.ml" "comm-graph.ml" "comm-ref.ml" "comm-od.ml"
     "comm-ned.ml" "comm-nrf.ml"
     )

    ("lib-defs.lsp" "lib-ref.lsp" "lib-edd.lsp" "lib-objc.lsp" "lib-dir.lsp" "lib-inf.lsp"
     "lib-abs.lsp" "lib-itd.lsp" "lib-wm.lsp" "lib-nv4.lsp" "mig-v4-5.lsp" "lib-mpdf.lsp"
     "ini-lib.lsp"
     "libm-bsc.ml" "libm-ref.ml" "libm-wm.ml" "libm-utl.ml" "libm-mig.ml" "libm-mp.ml"
     )

    ("ref-defs.lsp" "ref-mono.lsp" "ref-eq.lsp" "ref-simp.lsp" "ref-arth.lsp"
     "ref-dc.lsp" "ref-itrp.lsp" "ref-ref.lsp" "ref-ml.lsp" "ref-itd.lsp"
     "ini-ref.lsp")
    ("dms-defs.lsp"  "ini-dms.lsp")
    ("ptr-defs.lsp" "ptr-font.lsp")
    ("edd-defs.lsp" "edd-form.lsp" "edd-tree.lsp" "edd-layt.lsp" "edd-itd.lsp" "edd-prs.lsp"
     "ini-edd.lsp")
    ("prs-defs.lsp" "prs-edd.lsp")
    ("tst-defs.lsp" )
    ("win-defs.lsp" "win-init.lsp")
    ))


(defun doc-module (tag &optional (hardp t))
  (when (eql tag 'doc)
    (doc-stats))
  (with-open-file (out-file (concatenate 'string *out-path* (format-string "~a.text" tag))
			    :direction :output
			    :if-exists :new-version
			    :if-does-not-exist :create)
    (with-outport (out-file hardp)
      (extract-comments *files-with-extractable-comments* (list tag)))))


(defun doc-tac (&optional (hardp t))
  (with-open-file (out-file (concatenate 'string *out-path*  "tac.text")
			    :direction :output
			    :if-exists :new-version
			    :if-does-not-exist :create)
    (with-outport (out-file hardp)
      (extract-comments *primitives-for-tactics* (list 'ml)))))

(defun doc-all ()
  (doc-mods)
  (doc-stats)
  (with-open-file (out-file (concatenate 'string *out-path* "all.text")
			    :direction :output
			    :if-exists :new-version
			    :if-does-not-exist :create)
    (with-outport (out-file t)
      (extract-comments (append (directory (pathname  (concatenate 'string *in-path* "*.lsp")))
				(directory (pathname  (concatenate 'string *in-path* "*.ml"))))
			(list 'doc 'bsc 'trm 'bml 'mlt 'com 'lib 'ref 'io 'itd 'ptr 'edd 'prs 'tst 'int)
			t))
    (with-outport (out-file nil)
      (extract-comments (append (directory (pathname  (concatenate 'string *in-path* "*.lsp")))
				(directory (pathname  (concatenate 'string *in-path* "*.ml"))))
			(list 'ml)
			t))))

(defun doc-mods ()
  (terpri)
  (terpri)
  (dolist (mod (list 'doc 'bsc 'trm 'bml 'mlt 'com 'lib 'ref 'io 'itd 'ptr 'edd 'prs 'data 'int 'prfcmd 'dform
		     ;; 'ml 'eddml ???
		     ))
    (doc-module mod)
    (format t "finished ~a~%" mod))
  (doc-module 'ml nil)
  (format t "finished ML~%")
  (terpri))

(defun expand-tab (s &optional (w 8) (trimmed 4))
  (declare (ignore trimmed))
  (with-output-to-string (news)
    (let ((first-p t)
	  (newl 0))
      (dotimes (i (length s))
	(if (char= #\tab (char s i))
	    (let ((pad (- w (rem (-  newl 2) w))))
	      (if first-p
		  (progn
		    (write-char #\space news)
		    (incf newl)
		    (write-char #\space news)
		    (incf newl))
		  (dotimes (j pad)
		    (write-char #\space news)
		    (incf newl))))
	    (write-char (char s i) news))
      
	(setf first-p nil)))))
	
	  
(defun comment-read-from-string (s)
  (let ((s (string-left-trim '(#\space #\tab) s)))
	
    (unless (string= "" s)
      (read-from-string s))))


(defvar *lines*)
(defvar *doc-lines*)
(defvar *doc-ml-funcs*)
(defvar *doc-lisp-funcs*)
;; other than doc lines comments starting at left margin.
;; if not at left margin than considered code.
(defvar *comment-lines*)
(defvar *top-forms*)
(defvar *defuns*)
(defvar *defunmls*)
(defvar *code-lines*)
(defvar *code-chars*)
(defvar *lets*)

(defvar *module-tags*)

(defvar *total-lines*)
(defvar *total-doc-lines*)
(defvar *total-doc-ml-funcs*)
(defvar *total-doc-lisp-funcs*)

;; other than doc lines comments starting at left margin.
(defvar *total-comment-lines*)
;; ( at left margin. includes defuns and defunmls.
(defvar *total-top-forms*)
;; (defunml at left margin
(defvar *total-defunmls*)
;; (defun at left margin
(defvar *total-defuns*)
;; any non empty line which is not a comment or doc line.
(defvar *total-code-lines*)
;; all chars in code lines except lhs whitespace.
(defvar *total-code-chars*)
;; top level ml defs.
(defvar *total-lets*)


(defun doc-stats ()
  (let ((prefix  ";;;; "))
    (setf (char prefix 4) #\tab)

  (with-open-file (out-file (concatenate 'string "./doc/src/doc-stats.text")
			   :direction :output)

    (labels ((format-stat (str &rest rest)
	       (apply #'format out-file (concatenate 'string "~a" str) prefix rest))

	     (visit (files)
	       (unless (null files)
		 (let ((*lines* 0)
		       (*doc-lines* 0)
		       (*doc-ml-funcs* 0)
		       (*doc-lisp-funcs* 0)
		       ;; other than doc lines comments starting at left margin.
		       ;; if not at left margin than considered code.
		       (*comment-lines* 0)
		       (*top-forms* 0)
		       (*defunmls* 0)
		       (*defuns* 0)
		       (*code-lines* 0)
		       (*code-chars* 0)
		       (*module-tags* nil)
		       (*lets* 0)
		       )
		 
		   (dolist (file (car files))
		     (let ((l (length file)))
		       (if (string= "ml" (subseq file (- l 2) l))
			   (doc-stats-ml-file file)
			   (doc-stats-file file))))

		   (incf *total-lines* *lines*)
		   (incf *total-doc-lines* *doc-lines*)
		   (incf *total-doc-ml-funcs* *doc-ml-funcs*)
		   (incf *total-doc-lisp-funcs* *doc-lisp-funcs*)
		   (incf *total-comment-lines* *comment-lines*)
		   (incf *total-top-forms* *top-forms*)
		   (incf *total-defunmls* *defunmls*)
		   (incf *total-defuns* *defuns*)
		   (incf *total-code-lines* *code-lines*)
		   (incf *total-code-chars* *code-chars*)
		   (incf *total-lets* *lets*)

		   (format out-file ";;;;  -page- ~%")
		   (format-stat "~%")
		   (format-stat "Module Statistics [~a]: ~a~%"
				(datetime-string (get-universal-time))
				(reverse *module-tags*))
		   (format-stat "~%")
		   (format-stat "Count of documentation lines: ~a.~%" *doc-lines*)
		   (format-stat "Count of lisp functions documented: ~a.~%" *doc-lisp-funcs*)
		   (format-stat "Count of ml functions documented: ~a.~%" *doc-ml-funcs*)
		   (format-stat "Count of other comment lines: ~a.~%" *comment-lines*)
		   (format-stat "Count of top level forms: ~a.~%" *top-forms*)
		   (format-stat "Count of lisp functions: ~a.~%" *defuns*)
		   (format-stat "Count of ml primitive functions: ~a.~%" *defunmls*)
		   (format-stat "Count of lines of code: ~a.~%" *code-lines*)
		   (format-stat "Count of chars of code: ~a.~%" *code-chars*)
		   (format-stat "Count of all lines : ~a.~%" *lines*)
		   (format-stat "~%")

		   )
	       
		 (visit (cdr files)))))

      (let ((*total-lines* 0)
	    (*total-doc-lines* 0)
	    (*total-doc-ml-funcs* 0)
	    (*total-doc-lisp-funcs* 0)
	    (*total-comment-lines* 0)
	    (*total-top-forms* 0)
	    (*total-defuns* 0)
	    (*total-defunmls* 0)
	    (*total-code-lines* 0)
	    (*total-code-chars* 0)
	    (*total-lets* 0))
		 
	(format out-file ";;;; -docs- (mod doc) ~%")
	(visit (subseq *files-for-stats* 0 10))

	(format out-file ";;;;  -page- ~%")
	(format-stat "~%")
	(format-stat "Module Statistics Totals [~a]:~%" (datetime-string (get-universal-time)))
	(format-stat "~%")
	(format-stat "Count of documentation lines: ~a.~%" *total-doc-lines*)
	(format-stat "Count of lisp functions documented: ~a.~%" *total-doc-lisp-funcs*)
	(format-stat "Count of ml functions documented: ~a.~%" *total-doc-ml-funcs*)
	(format-stat "Count of other comment lines: ~a.~%" *total-comment-lines*)
	(format-stat "Count of top level forms: ~a.~%" *total-top-forms*)
	(format-stat "Count of lisp functions: ~a.~%" *total-defuns*)
	(format-stat "Count of ml primitive functions: ~a.~%" *total-defunmls*)
	(format-stat "Count of ml functions: ~a.~%" *total-lets*)
	(format-stat "Count of lines of code: ~a.~%" *total-code-lines*)
	(format-stat "Count of chars of code: ~a.~%" *total-code-chars*)
	(format-stat "Count of lines: ~a.~%" *total-lines*)
	(format-stat "~%")
	(format out-file ";;;; -doce- ~%")
	(terpri)
	)))))


(defun doc-stats-file (file)

  (with-open-file (in-file (concatenate 'string *in-path* file)
			   :direction :input)
    (format t ".")

    (let ((indoc nil)		; after docs
	  )

      (loop
       (let ((line (read-line in-file nil nil)))
	 (incf *lines*)
	 (when (null line)
	   (return))
	 (if indoc
	     (progn
	       (when (match-line ";;;;" line)
		 (incf *doc-lines*))
	       (cond
		 ((match-line ";;;; -doce" line)
		  (setf indoc nil))
		 ((match-line ";;;; -doct-" line)
		  (dolist (tag (comment-read-from-string (subseq line 11)))
		    (pushnew tag *module-tags*)))
		 (t (let ((trimmed (string-left-trim '(#\space #\tab #\;) line)))
		      (cond
			;; foogoo \(      ==>	lisp func doc
			((let ((first-lparen (search "(" trimmed))
			       (first-space (search " " trimmed))
			       (first-tab (search (string #\tab) trimmed)))
			   (and first-lparen
				(or (null first-tab)
				    (> first-tab first-lparen))
				(or (null first-space)
				    (= first-lparen (1+ first-space))
				    (> first-space first-lparen))))
			 ;;(setf a trimmed) (break)
			 (incf *doc-lisp-funcs*))
			;; oof-oog : ooh -> ==>	ml func doc
			((let ((first-colon (search ":" trimmed))
			       (first-arrow (search "->" trimmed)))
			   (and first-colon
				first-arrow
				(> first-arrow first-colon)))
			 (incf *doc-ml-funcs*)))))))
	     (cond
	       ((string= "" line))
	       ((match-line ";;;; -docs-" line)
		(incf *doc-lines*)
		(setf indoc t)
		(dolist (tag (comment-read-from-string (subseq line 11)))
		  (pushnew tag *module-tags*))
		(incf *doc-lines*))
	       ((eql (char line 0) #\;)
		(incf *comment-lines*))
	       ((eql (char line 0) #\()
		(cond
		  ((match-line "(defunml" line)
		   (incf *defunmls*))
		  ((match-line "(defun" line)
		   (incf *defuns*)))
		(incf *code-chars* (length (string-trim '(#\space #\tab) line)))
		(incf *code-lines*)
		(incf *top-forms*))
	       (t (let ((trimmed (string-trim '(#\space #\tab) line)))
		    (when (not (string= trimmed ""))
		      (incf *code-chars* (length trimmed))
		      (incf *code-lines*)))))))))))


(defun doc-stats-ml-file (file)

  (with-open-file (in-file (concatenate 'string *in-path* file)
			   :direction :input)
    (format t "M")

    (let ((indoc nil)			; after docs
	  (incode nil)
	  )

      (loop
       (let ((line (read-line in-file nil nil)))

	 (when (null line)
	   (return))

	 (incf *lines*)

	 (when (and (not indoc)
		    (> (length line) 3)
		    (string= "let" (subseq line 0 3)))
	   (setf incode t)
	   ;;(format t "l")
	   (incf *lets*)
	   (incf *top-forms*))

	 (let ((noteindoc indoc));; if start and end indoc then count whole line as doc.
	   
	   ;; scan for comments char at a time.
	   (dotimes (i (length line))

	     (when (eql #\% (char line i))
	       ;;(format t "H")
	       (setf indoc (not indoc)))

	     (when  (and incode (not indoc))
	       (incf *code-chars*)))

	   (if (and indoc noteindoc)
	       (progn
		 ;;(format t "%")
		 (incf *comment-lines*))
	       (if incode
		   (progn
		     ;;(format t "C")
		     (incf *code-lines*))
		   ;;(format t "?")
		   )
	       )))))))


(defun extract-comments (files tags &optional pathname-p)
    (labels
	((visit (files)
	   (unless (null files)
	       (dolist (file (car files))
		 (extract-file file))
	     (comment-flush)
	     (visit (cdr files))))

	 (extract-file (file)
	   (with-open-file (in-file (if pathname-p
					file
					(concatenate 'string *in-path* file))
				    :direction :input)
	     (format t ".")
	     (let ((indoc nil)
		   (file-tags nil)
		   (tagged-p nil)
		   (file-header-p nil))
	       (loop
		(let  ((line (read-line in-file nil nil)))
		  (when (null line)
		    ;;(write-char #\page out-file)
		    ;;(write-char #\newline out-file)
		    (return))
		  (when indoc
		    (cond ((match-line ";;;; -doce" line)
			   (comment-page 'hard)
			   (setf file-tags nil)
			   (setf tagged-p nil)
			   (setf indoc nil))
			  ((and tagged-p
				(match-line ";;;;  -page-" line))
			   (comment-page (comment-read-from-string (subseq line 12))))
			  ((match-line ";;;; -doct-" line)
			   (setf file-tags (comment-read-from-string (subseq line 11)))
			   ;;(break)
			   (setf tagged-p (some #'(lambda (file-tag) (and (member file-tag tags) t))
						file-tags)))
			  ((and tagged-p
				(>= (length line) 4))
			   ;;(setf g line)
			   ;;(break "l")
			   (comment-push-line (expand-tab (concatenate 'string "" (subseq line 4)) 8 4)))))
		  (unless indoc
		    ;;(format t "~a~%" line) (incf i)
		    ;;(when (= i 100) (setf d line) (break))
		    (when (or (match-line ";;;; -docs" line)
			      (match-line ";;;; -doct" line))
		      ;;(setf a tags b line c file) (break "a")
		      (setf file-tags (when (> (length line) 11)
					(let ((tags (comment-read-from-string (subseq line 11))))
					  ;;(setf a tags b line c file) (break)
					  (when (listp tags) tags))))
		      (setf tagged-p (some #'(lambda (file-tag) (and (member file-tag tags) t))
					   file-tags)))
		    (when tagged-p
		      (setf indoc t)
		      (if file-header-p
			  (comment-push-line (format nil ""))
			  (progn
			    (comment-push-line (format nil "~a: Comments from file ~a. "
						       (datetime-string (get-universal-time))
						       file))
			    (comment-push-line (format nil ""))
			    (setf file-header-p t))))) ))))))


      (visit files)))




#+foo
(defun extract-code (&optional (mask 255))
  (with-open-file (out-file (concatenate 'string *out-path* "code.text")
			    :direction :output
			    :if-exists :new-version
			    :if-does-not-exist :create)
    (do ((files *files-with-extractable-code* (cdr files)))
	((null files))
      (with-open-file (in-file (concatenate 'string *in-path* (car files))
			       :direction :input)
	(write-line (format nil "Comments and Code from file ~a.~%" (car files))
		    out-file)	
	(let ((in-extract nil)
	      (level nil))
	  (loop
	    (multiple-value-bind (line eof-p) (read-line in-file nil nil)
	      (when (or eof-p (null line))
		(return))
	      (when in-extract
		(cond ((match-line ";;;;  -code-" line)
		       (setf in-extract nil)
		       (write-char #\page out-file)
		       (write-char #\newline out-file))
		      ((and (or (eql 255 mask) (logtest mask (car level)))
			    (match-line ";;;;  -page-" line))
		       (write-char #\page out-file))
		      ((match-line ";;;;  -level-"  line)
		       (push (string-to-hex (subseq line 13 15)) level))
		      ((or (eql 255 mask) (logtest mask (car level)))
		       (write-line line out-file))))
	      (when (and (null in-extract)
			 (match-line ";;;;  -cods-" line))
		(setf in-extract t)
		(setf level '(255))) )))))))



				    
