
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	oed-defs : v4 io-defs and macros etc.
;;;;	oed-edit : v4&5 shared code
;;;;	oed-edd : v5 support for oed-edit.
;;;;	
;;;;	
;;;;	
;;;;	


;;;;	
;;;;	
;;;;	view-abs is problem
;;;;	  either need desc to determine where to find abs
;;;;	  or match against source to find abs.
;;;;	  or define some psuedo-abs table for edit.
;;;;	
;;;;	
;;;;	



;;;;	
;;;;	Stacks
;;;;	
;;;;	Cmd stack
;;;;	Term stack.
;;;;	
;;;;	
;;;;	


;;;
;;;	General stack funcs.
;;;

;; stack will have special top element

(define-primitive |!edit-stack-top|)

(defvar *edit-stack-top-term* (iedit-stack-top-term))

(eval-when (compile)
  (proclaim '(inline
	      edit-stack-top-p
	      edit-stack-push edit-stack-pop
	      edit-stack-clear
	      edit-stack-peek
	      edit-stack-rotate edit-stack-reverse-rotate
	      edit-stack-unrotate 
	      )))

(defun edit-stack-top-p (stack)
  (and stack (iedit-stack-top-term-p (car stack))))

(defun edit-stack-push (item stack)
  ;;(setf -item item -stack stack) (break "esp")
  (when (null item) (break "edit-stack-push null item" ))
  (if (and stack (iedit-stack-top-term-p (car stack)))
      (progn
	(setf (cdr stack) (cons item (cdr stack)))
	stack)
      (push item stack)))

(defun edit-stack-pop (stack)
  (if (and stack (iedit-stack-top-term-p (car stack)))
      (progn
	(setf (cdr stack) (cddr stack))
	stack)
      (cdr stack)))

(defun edit-stack-peek (stack)
  (if (and stack (iedit-stack-top-term-p (car stack)))
      (cadr stack)
      (car stack)))

(defun edit-stack-peek-ahead (stack)
  (if (and stack (iedit-stack-top-term-p (car stack)))
      (edit-stack-peek (cddr stack))
      (edit-stack-peek (cdr stack))))

(defun edit-stack-peek-n (stack n)
  (if (and stack (iedit-stack-top-term-p (car stack)))
      (edit-stack-peek (nthcdr (1+ n) stack))
      (edit-stack-peek (nthcdr n stack))))

(defun edit-stack-peek-r (stack)
  (let ((peek (edit-stack-peek stack)))
    (when (null peek)
      (raise-error (error-message '(edit stack peek empty))))
    peek))

(defun edit-stack-peek-ahead-r (stack)
  (let ((peek (edit-stack-peek-ahead stack)))
    (when (null peek)
      (raise-error (error-message '(edit stack peek-ahead empty))))
    peek))

(defun edit-stack-rotate (stack)
  (cond
    ((null stack) nil)
    ((iedit-stack-top-term-p (car stack))
     (nconc (cddr stack) (list *edit-stack-top-term* (cadr stack))))
    (t (nconc (cdr stack) (list (car stack))))))

(defun edit-stack-reverse-rotate (stack)
  (labels
      ((reverse-rotate (s)
	 (cons (car (last s)) (butlast s))))

  (if (null (cdr stack))
      stack
      (let ((stack (reverse-rotate stack)))
 	(if (and stack (iedit-stack-top-term-p (car (last stack))))
	    (reverse-rotate stack)
	    stack)))))

(defun edit-stack-unrotate (stack)
  (do ((s stack (cdr s))
       (acc nil (cons (car s) acc)))
      ((or (null s) (iedit-stack-top-term-p (car s)))
       (if (null s)
	   (cons *edit-stack-top-term* (nreverse acc))
	   (append s (nreverse acc))))))


;;;
;;;  Cmd Stack
;;;

(defvar *edit-cmd-stack* nil)
				   
(eval-when (compile)
  (proclaim '(inline
	      cmd-stack-push cmd-stack-pop
	      cmd-stack-clear
	      cmd-stack-peek
	      cmd-stack-rotate cmd-stack-reverse-rotate
	      cmd-stack-unrotate)))

(defun cmd-stack-top-p ()
  (edit-stack-top-p *edit-cmd-stack*))

(defun cmd-stack-push (item)
  (let ((peek (edit-stack-peek *edit-cmd-stack*)))
    (when (or (null peek)
	      (not (compare-terms-p item peek)))
      (setf *edit-cmd-stack*
	    (edit-stack-push item *edit-cmd-stack*)))))

(defun cmd-stack-pop ()
  (setf *edit-cmd-stack*
	(edit-stack-pop *edit-cmd-stack*)))

(defun cmd-stack-clear ()
  (setf *edit-cmd-stack* nil))

(defun cmd-stack-peek ()
  (edit-stack-peek *edit-cmd-stack*))

(defun cmd-stack-peek-r ()
  (edit-stack-peek-r *edit-cmd-stack*))

(defun cmd-stack-rotate ()
  (setf *edit-cmd-stack*
	(edit-stack-rotate *edit-cmd-stack*)))

(defun cmd-stack-reverse-rotate ()
  (setf *edit-cmd-stack*
	(edit-stack-reverse-rotate *edit-cmd-stack*)))

(defun cmd-stack-unrotate ()
  (setf *edit-cmd-stack*
	(edit-stack-unrotate *edit-cmd-stack*)))


(defun edit-cmd-push (state)
  (let ((point (point-of-edit-state state)))
    (unless (dtree-leaf-p point)
      (cmd-stack-push (dtree-to-term point))))
  state)
    
(defun edit-cmd-pop (state)
  (cmd-stack-pop)
  state)

(defun edit-cmd-rotate (state)
  (cmd-stack-rotate)
  state)

(defun edit-cmd-reverse-rotate (state)
  (cmd-stack-reverse-rotate)
  state)

(defun edit-cmd-unrotate (state)
  (cmd-stack-unrotate)
  state)

(defun edit-cmd-print (state)
  (message-emit (inform-message '(edit cmd stack peek) (cmd-stack-peek-r)))
  state)

(defun edit-cmd-yank (state)
  (let ((point (point-of-edit-state state)))
    (if (dtree-leaf-p point)
	state
	(oed-edit-replace state point (cmd-stack-peek-r)))))


;;;
;;;	term-stack
;;; 

(defvar *edit-term-stack-ephemeral* nil)

(defvar *edit-term-stack* nil)

(eval-when (compile)
  (proclaim '(inline
	      term-stack-push term-stack-pop
	      term-stack-clear
	      term-stack-rotate term-stack-reverse-rotate
	      term-stack-unrotate
	      term-stack-peek)))


(defvar *edit-term-stack-cut-length* 32)

(defun set-edit-term-stack-cut (i)
    (setf *edit-term-stack-cut-length* i))

(defun show-edit-term-stack-cut ()
    *edit-term-stack-cut-length*)

(defun edit-term-stack-cut (state)

  (let ((nth (nthcdr *edit-term-stack-cut-length* *edit-term-stack*)))

    (when nth
      (setf (cdr nth) nil)))
  state)

(defun term-stack-push (item)
  ;;(break "tsp")
  (setf *edit-term-stack*
	(edit-stack-push item *edit-term-stack*)))

(defun term-stack-pop ()
  ;;(break "tsp")
  (setf *edit-term-stack* (edit-stack-pop *edit-term-stack*)))

(defun term-stack-clear ()
  (setf *edit-term-stack* nil))

(defun term-stack-top-p ()
  (edit-stack-top-p *edit-term-stack*))

(defun term-stack-peek ()
  (edit-stack-peek *edit-term-stack*))

(defun term-stack-peek-ahead-r ()
  (edit-stack-peek-ahead-r *edit-term-stack*))

(defun term-stack-peek-r ()
  (edit-stack-peek-r *edit-term-stack*))

(defun term-stack-rotate ()
  (setf *edit-term-stack*
	(edit-stack-rotate *edit-term-stack*)))

(defun term-stack-reverse-rotate ()
   (setf *edit-term-stack*
	(edit-stack-reverse-rotate *edit-term-stack*)))

(defun term-stack-unrotate ()
   (setf *edit-term-stack*
	(edit-stack-unrotate *edit-term-stack*)))

(defun edit-stack-print (state)
  (message-emit (inform-message '(edit term stack peek) (term-stack-peek-r)))
  state)


(defun eterm-stack-push (item)
  ;;(break "tsp")
  (setf *edit-term-stack-ephemeral*
	(edit-stack-push item *edit-term-stack-ephemeral*)))

(defun eterm-stack-pop ()
  ;;(break "tsp")
  (setf *edit-term-stack-ephemeral* (edit-stack-pop *edit-term-stack-ephemeral*)))

(defun eterm-stack-clear ()
  (setf *edit-term-stack-ephemeral* nil))

(defun eterm-stack-top-p ()
  (edit-stack-top-p *edit-term-stack-ephemeral*))

(defun eterm-stack-peek ()
  (edit-stack-peek *edit-term-stack-ephemeral*))

(defun eterm-stack-peek-ahead-r ()
  (edit-stack-peek-ahead-r *edit-term-stack-ephemeral*))

(defun eterm-stack-peek-r ()
  (edit-stack-peek-r *edit-term-stack-ephemeral*))

(defun eterm-stack-rotate ()
  (setf *edit-term-stack-ephemeral*
	(edit-stack-rotate *edit-term-stack-ephemeral*)))

(defun eterm-stack-reverse-rotate ()
   (setf *edit-term-stack-ephemeral*
	(edit-stack-reverse-rotate *edit-term-stack-ephemeral*)))

(defun eterm-stack-clear ()
   (setf *edit-term-stack-ephemeral* nil))





(defunml (|oed_cmd_stack_push| (term))
    (term -> unit)

  ;;(setf -term term) (break "opc")
  (cmd-stack-push term))

(defunml (|oed_eterm_stack_pop| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)
  (eterm-stack-pop)
  nil)

(defunml (|oed_eterm_stack_push| (term))
    (term -> unit)
  (eterm-stack-push term)
  nil)

(defunml (|oed_eterm_stack_peek| (unit) :declare ((declare (ignore unit))))
    (unit -> term)
  (eterm-stack-peek-r)
  )


(defunml (|oed_term_stack_pop| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)
  (term-stack-pop)
  nil)


(defunml (|oed_term_stack_push| (term))
    (term -> unit)
  (term-stack-push term)
  nil)


(defunml (|oed_term_stack_rotate| (dir))
    (bool -> unit)
  (if dir
      (term-stack-rotate)
      (term-stack-reverse-rotate)))

(defunml (|oed_eterm_stack_rotate| (dir))
    (bool -> unit)
  (if dir
      (eterm-stack-rotate)
      (eterm-stack-reverse-rotate)))


(defunml (|oed_term_stack_peek| (pos))
    (int -> term)
 
  (if (> pos 0)
      (or (edit-stack-peek-n *edit-term-stack* (1- pos))
	  (raise-error (error-message '(stack null))))
      (raise-error (error-message '(peek pos) pos))))

(defunml (|oed_term_stack_snap| (unit) :declare ((declare (ignore unit))))
    (unit -> (term list))
 
  *edit-term-stack*)



;;;;
;;;;  Preprocessed expand macro:
;;;;
;;;;    rhs		: (simple-macro | collection) list
;;;;    simple-macro	: unconditional expand macro | non-expand(leaf) macro
;;;;	collection	: A list of macros such that
;;;;			  Every macro in the collection has same lhs token
;;;;                      AND first macro in the collection is not unconditional.
;;;;      


(defstruct edit-macro
  (type nil)
  (token nil)
  (string nil)
  (rhs nil))

(defstruct (edit-leaf-macro (:include edit-macro))
  (before nil))


(defstruct (edit-dynamic-macro (:include edit-leaf-macro))
  (version nil)
  )

(defstruct (edit-degenerate-macro (:include edit-dynamic-macro))
  (library-prefix nil)
  (unprefixed-string nil)
  )

(defstruct (edit-library-macro (:include edit-leaf-macro))
  (unprefixed-string nil)
  )

(defstruct (edit-node-macro (:include edit-macro))
  (version nil)				; degenerate token version
  (library-prefix nil)
  (expand-prefix nil)
  (lhs nil)
  (conditions nil)
  (children nil)
  (collection nil))


(eval-when (compile)
  (proclaim '(inline
	      prefix-of-macro
	      type-of-macro token-of-macro string-of-macro rhs-of-macro before-of-macro
	      degenerate-macro-p
	      function-of-static-macro
	      dform-of-library-macro
	      version-of-expand-macro
	      expansion-of-expand-macro
	      macros-of-expand-macro
	      conditions-of-expand-macro
	      collection-of-expand-macro
	      function-of-dynamic-macro
	      version-of-dynamic-macro
	      collect-macros macro-collection-p macros-of-collection)))
	    

(defvar *macro-library-token-prefix* nil)
(defvar *macro-expand-token-prefix* nil)
(defvar *macro-degenerate-token-version* nil)

(defmacro with-macro-prefixes ((lib-prefix exp-prefix version) &body body)
  `(let ((*macro-library-token-prefix* ,lib-prefix)
	 (*macro-expand-token-prefix* ,exp-prefix)
	 (*macro-degenerate-token-version* ,version))
    ,@body))

(defun macro-library-token-prefix () *macro-library-token-prefix*)
(defun macro-expand-token-prefix () *macro-expand-token-prefix*)
(defun macro-degenerate-token-version () *macro-degenerate-token-version*)
(defun set-degenerate-token-version (v) (setf *macro-degenerate-token-version* v))


(defun make-leaf-macro (type token rhs &optional before)
  (if before
      (make-edit-leaf-macro :type type :token token
			    :string (implode-to-string (estring-to-istring token))
			    :rhs rhs :before before)
      (make-edit-leaf-macro :type type :token token
			    :string (implode-to-string (estring-to-istring token))
			    :rhs rhs)))

(defun make-node-macro (type token rhs conditions &optional children)
  (make-edit-node-macro :library-prefix (macro-library-token-prefix)
			:expand-prefix (macro-expand-token-prefix)
			:version (macro-degenerate-token-version)
			:type type
			:token token
			:lhs token
			:string (implode-to-string (estring-to-istring token))
			:rhs rhs
			:conditions conditions :children children))


(defun type-of-macro (macro)
  (edit-macro-type macro))

(defun token-of-macro (macro)
  (edit-macro-token macro))

(defun string-of-macro (macro)
  (edit-macro-string macro))

(defun rhs-of-macro (macro)
  (edit-macro-rhs macro))

(defun before-of-macro (macro)
  (edit-leaf-macro-before macro))

(defun expand-prefix-of-macro (m)
  (edit-node-macro-expand-prefix m))

(defun lhs-of-macro (m)
  (edit-node-macro-lhs m))


(defun make-degenerate-macro (buffer)
  (make-edit-degenerate-macro :type 'degenerate :token buffer
			:library-prefix (macro-library-token-prefix)
			:string (implode-to-string (estring-to-istring buffer))
			:rhs nil
			:version (macro-degenerate-token-version)
			:unprefixed-string (when (istring-prefixed-p buffer
								  (macro-library-token-prefix))
					     (unprefix-istring buffer
							       (macro-library-token-prefix)))))

(defun degenerate-macro-p (macro)
  (eql 'degenerate (type-of-macro macro)))

(defun library-prefix-of-degenerate-macro (m)
  (edit-degenerate-macro-library-prefix m))

(defun unprefixed-string-of-degenerate-macro (m)
  (edit-degenerate-macro-unprefixed-string m))


;;; prefixed-p
;;; unprefixed-istring

(defun unprefix-istring (istring prefix)
  (do ((pre prefix (cdr pre))
       (tok istring (cdr tok)))
      ((or (null pre) (not (equal (car tok) (car pre))))
       (implode-to-string (if (null pre) tok istring)))))
  
(defun istring-prefixed-p (istring prefix)
  (if (null prefix)
      nil
      (do ((pre prefix (cdr pre))
	   (tok istring (cdr tok)))
	  ((or (null pre) (not (equal (car tok) (car pre))))
	   (if (null pre) t nil)))))


(defun make-static-macro (token function &optional before)
  (make-leaf-macro 'static token function before))

(defun static-macro-p (macro)
  (eql 'static (type-of-macro macro)))

(defun function-of-static-macro (macro)
  (rhs-of-macro macro))


(defun make-library-macro (token term unprefixed-string)
  ;;(setf -term term -token (implode-to-string (estring-to-istring token))) (break "mlm")
  (make-edit-library-macro :type 'library :token token
			   :string (implode-to-string (estring-to-istring token))
			   :rhs term
			   :unprefixed-string unprefixed-string))

(defun library-macro-p (macro)
  (eql 'library (type-of-macro macro)))

(defun term-of-library-macro (macro)
  (rhs-of-macro macro))

(defun unprefixed-string-of-library-macro (macro)
  (edit-library-macro-unprefixed-string macro))


(defun expand-macro-p (macro)
  (eql 'expand (type-of-macro macro)))

(defun expansion-of-expand-macro (macro)
  (rhs-of-macro macro))

(defun conditions-of-expand-macro (macro)
  (edit-node-macro-conditions macro))

(defun library-prefix-of-expand-macro (macro)
  (edit-node-macro-library-prefix macro))

(defun version-of-expand-macro (macro)
  (edit-node-macro-version macro))

;; one of
;;  - a macro : edit-macro-p
;;  - a list of conditional macros : macro-collection-p
;;  - a token - to be run through lib/dynamic : not collection & list-p

(defun collect-macros (l token)
  (make-leaf-macro 'collection token l))

(defun macro-collection-p (c)
  (eql (type-of-macro c) 'collection))

(defun token-of-collection (c)
  (token-of-macro c))

(defun macros-of-collection (c)
  (rhs-of-macro c))



(defun macros-of-expand-macro (macro)
  (edit-node-macro-children macro))

;; the collection this expand macro heads.
(defun collection-of-expand-macro (macro)
  (edit-node-macro-collection macro))


(defun edit-consume-token (buffer macro)
  (nthcdr (length (token-of-macro macro))
	  buffer))

;;; dynamic

(defun make-dynamic-macro (token function &optional (version 'auto))
  (make-edit-dynamic-macro :type  'dynamic
			   :token token
			   :string "dynamic"
			   :rhs function
			   :version version))

(defun dynamic-macro-p (macro)
  (eql 'dynamic (type-of-macro macro)))

(defun function-of-dynamic-macro (macro)
  (rhs-of-macro macro))

(defun version-of-dynamic-macro (macro)
  (edit-dynamic-macro-version macro))


(defstruct edit-condition
  (id )
  (not nil)
  (parent nil)
  (ancestor nil))

(defun new-edit-condition (id not parent ancestor)

  (let ((istring (istring id)))
    (make-edit-condition :id (cond
			       ((null istring)
				(process-err "Macro has null condition id."))
			       ((forall-p #'numeric-ichar-p istring)
				(with-string-scanner ((string id))
				  (scan-num)))
			       (t id))
			 :not not
			 :parent parent
			 :ancestor ancestor)))
  
(defun id-of-condition (c) (edit-condition-id c))
(defun not-condition-p (c) (edit-condition-not c))
(defun parent-condition-p (c) (let ((p (edit-condition-parent c))) (and p (> p 0))))
(defun count-of-parent-condition (c) (or (edit-condition-parent c) 0))
(defun ancestor-condition-p (c) (edit-condition-ancestor c))






;;;  However, the following additional
;;;  macro syntax should allow users to define macros which conditionally 
;;;  switch modes.
;;;
;;;  <new-macro-rhs> :: (<bool>) <old-macro-rhs> | <old-macro-rhs>
;;;  <bool>  	     :: <bool-constant> | some simple boolean formulas
;;;  <bool-constant> :: TERM-CURSOR | TEMPLATE-CURSOR | TEXT-CURSOR 
;;;			| TERM-CHARACTER | STRING-TERM
;;;			etc.
;;;
;;;     when macro expanding, if boolean evaluates to false then
;;;   macro will not be used to expand.
;;;     this would allow for overloading of macro lhs for different editor
;;;   states.
;;;


;;;
;;; 
;;;


;; h,m,c,s

;;;;
;;;; Expand Macros
;;;;
;;;;  Syntax
;;;; 
;;;;   <edit-char>	:: <simple-char>
;;;;			| (c-<simple-char>)	; control
;;;;			| (m-<simple-char>)	; meta
;;;;			| (mc-<simple-char>)	; meta-control
;;;; 
;;;;   <simple-char>	:: <char> | \( | \) | \=
;;;;			| (tab) | (return) | (linefeed)
;;;;			| (newline) | (page)
;;;; 
;;;;   <char> :: what you expect minus ( ) =

;;;;
;;;;  <macro>		:: <edit-istring>==<edit-istring>
;;;;  <edit-istring>	:: <edit-ichar> | <edit-ichar><edit-istring>
;;;;  <edit-ichar>	:: <ichar>
;;;;			| ((:meta) . <ichar>)
;;;;			| ((:control) . <ichar>)
;;;;			| ((:meta :control) . <ichar>)
;;;;


;;;;
;;;;	expand-prefix 
;;;;
;;;;	load-macros(<file-spec>; <string{prefix}>)	: NULL
;;;;	 ** reads <emacro> list from file.  
;;;;	 ** macros defined subsequently need to prefix references with prefix.
;;;;
;;;;
;;;;	<emacro-disk>	: (<symbol{version}>
;;;;		 	   <string{lib prefix}>
;;;;			   <string{expand-prefix}>
;;;;			   <string{macro}>)
;;;;
;;;;	save-macros(<file-spec>)			: NULL
;;;;	restore-macros(<file-spec>)			: NULL
;;;;	 ;; removes current expand macros and replaces with saved macros.


(defvar *edit-macro-special-ichars* (list iat inot icomma ilparen irparen ipercent iequals))


(defun string-to-estring (s) 
  (with-string-scanner (s)
    (scan-estring)))

(defun scan-edit-char ()
  (if (scan-at-unescaped-ichar-p ilparen)
      (scan-edit-parens-char)
      (prog1 (scan-cur-byte)
	(scan-next-char))))

(defvar *edit-macro-sbits*
  (standard-character-sbits (map 'list #'char-code ",()")))

(defun scan-edit-macro-condition ()
  (let ((not nil)
	(parent nil)
	(ancestor nil)
	(string nil))

    (when (scan-at-ichar-p inot)
      (scan-ichar inot)
      (setf not t))

    (when (scan-at-ichar-p istar)
      (scan-ichar istar)
      (setf ancestor t))

    (setf parent (do ((i 0 (1+ i)))
		     ((not (scan-at-unescaped-ichar-p iat))
		      i)
		   (scan-ichar iat)))
    
    (when (scan-at-unescaped-ichar-p inot)
      (scan-ichar inot)
      (setf not t))

    (setf string (scan-string *edit-macro-sbits*))

    ;;(setf -string string) (break "semc")
    (new-edit-condition (intern (string-upcase string))
			not
			parent
			ancestor
			)))

  
(defvar *keych-keysym*
  '((left . #xff51)
    (up . #xff52)
    (right . #xff53)
    (down . #xff54)
    (insert . #xff63)
    (help . #xff6a)
    (numlock . #xff7f)
    (keypadspace . #xff80)
    (keypadtab . #xff89)
    (keypadenter . #xff8d)
    (keypadf1 . #xff91)
    (keypadf2 . #xff92)
    (keypadf3 . #xff93)
    (keypadf4 . #xff94)
    (keypadmultipication . #xffaa)
    (keypadplus . #xffab)
    (keypadseparator . #xffac)
    (keypadminus . #xffad)
    (keypadpoint . #xffae)
    (keypaddivision . #xffaf)
    (keypadzero . #xffb0)
    (keypadone . #xffb1)
    (keypadtwo . #xffb2)
    (keypadthree . #xffb3)
    (keypadfour . #xffb4)
    (keypadfive . #xffb5)
    (keypadsix . #xffb6)
    (keypadseven . #xffb7)
    (keypadeight . #xffb8)
    (keypadnine . #xffb9)
    (keypadequals . #xffbd)
    (f1 . #xffbe)
    (f2 . #xffbf)
    (f3 . #xffc0)
    (f4 . #xffc1)
    (f5 . #xffc2)
    (f6 . #xffc3)
    (f7 . #xffc4)
    (f8 . #xffc5)
    (f9 . #xffc6)
    (f10 . #xffc7)
    (l1 . #xffc8)
    (l2 . #xffc9)
    (l3 . #xffca)
    (l4 . #xffcb)
    (l5 . #xffcc)
    (l6 . #xffcd)
    (l7 . #xffce)
    (l8 . #xffcf)
    (l9 . #xffd0)
    (l10 . #xffd1)
    (r1 . #xffd2)
    (r2 . #xffd3)
    (r3 . #xffd4)
    (r4 . #xffd5)
    (r5 . #xffd6)
    (r6 . #xffd7)
    (r7 . #xffd8)
    (r8 . #xffd9)
    (r9 . #xffda)
    (r10 . #xffdb)
    (r11 . #xffdc)
    (r12 . #xffdd)
    (r13 . #xffde)
    (r14 . #xffdf)
    (r15 . #xffe0)

    ))

(defun keysym->keych (keysym)
  (car (rassoc keysym *keych-keysym*)))

(defun keych->keysym (keych)
  (cdr (assoc keych *keych-keysym*)))

(defun keych-p (keych)
  (assoc keych *keych-keysym*))

(defun scan-edit-parens-char ()
  (labels
      ((scan-conditions ()
	 (scan-ichar idash)
	 (cons (list :condition) 
	       (scan-undelimited-list #'scan-edit-macro-condition
				      #'(lambda () (scan-at-unescaped-ichar-p irparen))
				      #'(lambda () (scan-ichar icomma)))))

       (string-to-modifiers (string)
	 (sort (delete-duplicates
		(map 'list
		     #'(lambda (ch)
			 (case ch
			   (#\C :control)
			   (#\M :meta)
			   (#\S :shift)
			   (#\A :alt)
			   (otherwise
			    (process-err
			     (format-string
			      "~a is not valid modifier string for macro character."
			      string)))))
		     string))
	       #'string-lessp))
    
       (convert (istring)
	 (string-upcase (implode-to-string istring))))

    (prog2 (scan-ichar ilparen)
	(let ((cstring (convert (scan-istring #.'(list idash	; mc-
						  irparen)))))	; (tab)
	  (cond
	    ((and (scan-at-unescaped-ichar-p idash)
		  (string= "" cstring))
	     (scan-conditions))
	    ((scan-at-unescaped-ichar-p idash)
	     (cons (string-to-modifiers cstring)
		   (let ((ech (progn (scan-ichar idash) (scan-edit-char))))
		     (when (modified-edit-char-p ech)
		       (scan-error '(edit)
			(format-string "char of modified char may not be a modified char ~a"
				       (implode-to-string (edit-char-to-istring ech)))))
		     (if (integerp ech)
			 (char->ichar (char-upcase (ichar->char ech)))
			 ech))))
	    ((string= "MOUSELEFT" cstring) imouse-left)
	    ((string= "MOUSEMIDDLE" cstring) imouse-middle) 
	    ((string= "MOUSERIGHT" cstring) imouse-right)
	    ((string= "TAB" cstring) (ichar #\tab))
	    ((string= "PAGE" cstring) (ichar #\page))
	    ((string= "SPACE" cstring) (ichar #\space))
	    ((string= "RETURN" cstring) (ichar #\return))
	    ((string= "NEWLINE" cstring) (ichar #\newline))
	    ((string= "BACKSPACE" cstring) (ichar #\backspace))
	    ((string= "LINEFEED" cstring) (ichar #\linefeed))
	    ((string= "RUBOUT" cstring) (ichar #\rubout))
	    ((string= "FOCUS1" cstring) 'focus1)
	    (t (let ((sym (intern cstring)))
		 (if (keych-p sym)
		     sym
		     (scan-error '(edit) (format-string "Not expecting ~a to be in parens."
					      cstring)))))))
      (scan-ichar irparen))))


(defun edit-char-to-istring (ech)
  (labels
      ((modified-char-to-istring (ech)
	 (if (equal (car ech) '(:condition))
	     (conditions-to-istring (cdr ech))
	     `(,ilparen
	       ,@(mapcar #'(lambda (modifier)
			     (cond
			       ((eql modifier :shift) (ichar #\S))
			       ((eql modifier :alt) (ichar #\A))
			       ((eql modifier :meta) (ichar #\M))
			       ((eql modifier :control) (ichar #\C))))
		  (car ech))
	       ,idash
	       ,@(edit-char-to-istring (cdr ech))
	       ,irparen))))
	    
  (if (modified-edit-char-p ech)
      (modified-char-to-istring ech)
      (cond
	((eql imouse-left ech) (istring "(MouseLeft)"))
	((eql imouse-middle ech) (istring "(MouseMiddle)"))
	((eql imouse-right ech) (istring "(MouseRight)"))
	((eql itab ech) (istring "(TAB)"))
	((eql ipage ech) (istring "(PAGE)"))
	((eql ispace ech) (istring "(SPACE)"))
	((eql ibackspace ech) (istring "(BACKSPACE)"))
	((eql ireturn ech) (istring "(RETURN)"))
	((eql inewline ech) (istring "(NEWLINE)"))
	((eql ilinefeed ech) (istring "(LINEFEED)"))
	((eql ilinefeed ech) (istring "(RUBOUT)"))
	((eql ilparen ech) (list iescape ilparen))
	((eql iequals ech) (list iescape iequals))
	((member ech *edit-macro-special-ichars*) (list iescape ech))
	((symbolp ech) (istring ech))
	(t (list ech))))))

(defun modified-edit-char-p (ch)
  (and (consp ch)
       (listp (car ch))
       (forall-p #'(lambda (modifier)
		     (member modifier '(:condition :control :shift :meta :alt)))
	      (car ch))))

(defun scan-estring ()
  (let ((estring nil))
    (do ()
	((or (scan-at-end-p)
	     (scan-at-unescaped-ichar-p (char->ichar #\%))
	     (scan-at-unescaped-ichar-p iequals))
	 (nreverse estring))
      (push (scan-edit-char) estring))))

(defun estring-to-istring (estring)
  (mapcan #'edit-char-to-istring estring))
  
(defun estring-to-string (estr)
  (implode-to-string (estring-to-istring estr)))


(defun find-conditions (estring)
  (do ((estr estring (cdr estr))
       (acc nil)
       (conds nil))
      ((null estr) (values conds (nreverse acc)))
    (if (and (consp (car estr))
	     (equal (caar estr) '(:condition)))
	(setf conds (cdar estr))
	(push (car estr) acc))))
    

(defun scan-edit-expand-macro (&optional expand-macros)
  (let ((lhs (scan-estring)))
    (with-error-prefix (format-string "After Lhs[~a]: "
				      (estring-to-string lhs))
      (scan-ichar iequals)
      (scan-ichar iequals)
      (let ((rhs (scan-estring)))
	(unless (or (scan-at-end-p)
		    (scan-at-unescaped-ichar-p (char->ichar #\%)))
	  ;;(break)
	  (scan-error '(edit)
	   (format-string "extraneous characters at end of macro def: ~a"
			  (implode-to-string (scan-istring nil)))))
	(mlet* (((lhs-conditions lhs-token) (find-conditions lhs))
		((rhs-conditions rhs-token) (find-conditions rhs)))
	       (make-expand-macro lhs-token
				  rhs-token
				  (or rhs-conditions lhs-conditions)
				  expand-macros))))))

(defun conditions-to-istring (conditions)
  (list-to-istring conditions 
		   #'(lambda (condition)
		       `(,@(do ((i (count-of-parent-condition condition) (1- i))
				(acc nil (cons iat acc)))
			       ((= i 0) acc))
			 ,@(when (not-condition-p condition) (list inot))
			 ,@(when (ancestor-condition-p condition) (list istar))
			 ,@(escaped-istring (id-of-condition condition) *edit-macro-special-ichars*)))
		   (list icomma)
		   (list ilparen idash)))

(defun unscan-edit-expand-macro (macro)
  `(,@(estring-to-istring (token-of-macro macro))
    ,@(when (conditions-of-expand-macro macro)
	(conditions-to-istring (conditions-of-expand-macro macro)))
    ,iequals
    ,iequals
    ,@(estring-to-istring (expansion-of-expand-macro macro))))

(defun print-edit-expand-macro (macro)
  (implode-to-string (unscan-edit-expand-macro macro)))


(defun text-character-macro-p (macro)
  (let ((token (token-of-macro macro)))
    (and (null (cdr token))
	 (printable-ichar-p (car token))
	 (edit-macro-condition-p 'text macro))))



;;;
;;;  library macros
;;;

(defvar *edit-static-macros* nil)
(defvar *edit-library-macros* nil)
(defvar *edit-expand-macros* nil)

(defvar *edit-text-character-macros* (make-array 255 :initial-element nil))

(defvar *macro-library-token-prefixes* (list nil))
(defvar *edit-library-hash-tables* (acons nil (make-hash-table :test #'equal :size 1000) nil))

(defvar *edit-preprocess-macros-p* t)



(defun set-library-token-prefix (s)
  (let ((prefix (string-to-estring s)))
    (add-macro-library-token-prefix prefix t)
    (setf *macro-library-token-prefix*  prefix)))

(defun get-library-token-prefix ()
  (estring-to-string *macro-library-token-prefix*))

(defun make-library-token (itoken)
  (append *macro-library-token-prefix* itoken))

(defun tokens-of-display-form (dform)
  (unless (or (dform-orphaned-p dform) (null (object-name-of-dform dform)))
    (let ((disp-dforms (dforms-of-dform dform)))
      (let* ((dforms (permuted-list-of-dforms disp-dforms))
	     (tokens (let ((*macro-library-token-prefix* nil))
		       (edit-macro-tokens-of-dform dform dforms
						   (istring (object-name-of-dform dform))
						   0))))
	(if (eql (car dforms) dform)
	    (cons (object-name-of-dform dform)
		  (mapcar #'implode tokens))
	    (mapcar #'implode tokens))))))


(defun edit-macros-of-defined-term (term name)
 (when term
   (list (make-library-macro
	  (make-library-token (istring name))
	  term
	  (string name)))))

(defun edit-macro-tokens-of-defined-term (term name)
  (when term
    (list (make-library-token (istring name)))))

(defun edit-macros-of-dform-list (dforms iname index)
  ;;(setf -dforms dforms -iname iname -index index) (break "emodl")
  (when dforms
    (let ((dform (car dforms)))
      (when dform
	`(,@(when (and (zerop index)
		       (equal (butlast iname 3)
			      (istring (id-of-term (model-term-of-dform dform)))))
		  (let ((niname (butlast iname 3)))
		    (list (make-library-macro (make-library-token niname)
					      (dform-to-placeholder-term dform)
					      (implode-to-string niname)))))
	  ,@(when (zerop index)
		  (list (make-library-macro (make-library-token iname)
					    (dform-to-placeholder-term dform)
					    (implode-to-string iname))))
	  ,@(let ((itoken (append iname (istring (1+ index)))))
		 (list (make-library-macro (make-library-token itoken) 
					   (dform-to-placeholder-term dform)
					   (implode-to-string itoken))))
	  ,@(when (macro-name-of-dform dform)
		  (list (make-library-macro (make-library-token
					     (istring (macro-name-of-dform (car dforms))))
					    (dform-to-placeholder-term (car dforms))
					    (string (macro-name-of-dform (car dforms))))))
	  ,@(edit-macros-of-dform-list (cdr dforms) iname (1+ index)))))))

(defun edit-macros-of-dforms (dforms name)
  (when dforms
    (with-error-backtrace "LibraryMacroOfDisplayForm"
      (edit-macros-of-dform-list (list-of-dforms dforms)
				 (istring name) 0))))

(defun edit-macro-tokens-of-dform (dform dforms iname index)
  (when dforms
    (if (eq (car dforms) dform)
	`(,@(when (macro-name-of-dform (car dforms))
	      (list (make-library-token (istring (macro-name-of-dform (car dforms))))))
	  ,@(when (and (zerop index) iname) (list iname))
	  ,@(when iname (list (make-library-token (append iname (istring (1+ index)))))))
	 (edit-macro-tokens-of-dform dform (cdr dforms) iname (1+ index)))))


(defun edit-macro-tokens-of-dform-list (dforms iname index)
  ;;(setf -iname iname -dforms dforms -index index) (break "emtodl")
  (when dforms
    `(,@(when (and (exists-p #'identity dforms)
		   (equal (butlast iname 3)
			  (istring (id-of-term (model-term-of-dform
						(find-first #'(lambda (d) d) dforms))))))
	  (list (make-library-token (butlast iname 3))))
      ,@(when (and (car dforms) (= 0 index))
	  (list (make-library-token iname)))
      ,@(when (car dforms)
	  (list (make-library-token (append iname (istring (1+ index))))))
      ,@(when (and (car dforms) (macro-name-of-dform (car dforms)))
	  (list (make-library-token (istring (macro-name-of-dform (car dforms))))))
      ,@(edit-macro-tokens-of-dform-list (cdr dforms)
	 iname (1+ index)))))
  
(defun edit-macro-tokens-of-dforms (dforms name)
  (when dforms
    (with-error-backtrace "LibraryMacroOfDForm"
      (edit-macro-tokens-of-dform-list (permuted-list-of-dforms dforms)
				       (istring name) 0))))


;; fttb, we do not define default dforms for abs and thms.
(defun edit-macros-of-object (id name kind)
  (with-ignore
      (if (eql 'disp kind)
      
	  (edit-macros-of-dforms (dforms-lookup id) name)

	  (let ((term (defined-term-of-obj id kind)))
	    (when term
	      (edit-macros-of-defined-term term name))))))

	 
(defun edit-macro-tokens-of-object (id name kind)

  (if (eql 'disp kind)

      (edit-macro-tokens-of-dforms (dforms-lookup id) name)
      
      (let ((term (defined-term-of-obj id kind)))
	(when term
	  (edit-macro-tokens-of-defined-term term name)))))



(defun add-macro-library-token-prefix (prefix &optional (add-hash-p nil))
  ;;(break "amltp")
  (unless (member prefix *macro-library-token-prefixes* :test #'equal)
    (when add-hash-p (edit-hash-library-macros prefix))
    (setf *macro-library-token-prefixes* (cons prefix *macro-library-token-prefixes*))))

(defun table-of-macro-library-token-prefix  (&optional (prefix t))
  (cdr (assoc (if (eql prefix t)
		  (macro-library-token-prefix)
		  prefix)
	      *edit-library-hash-tables*
	      :test #'equal)))

(defun edit-hash-library-macro (macro table)
  (setf (gethash (string-of-macro macro) table) macro))

(defun edit-unhash-library-token (token table)
  (remhash (estring-to-string token) table))

(defun edit-gethash-library-macro (macro)
  (let ((table (table-of-macro-library-token-prefix (library-prefix-of-degenerate-macro macro))))
    (when table
    (gethash (string-of-macro macro) table))))
    
 
(defun edit-rehash-library-macros ()
  (mapc #'edit-hash-library-macros *macro-library-token-prefixes*)
  (setf *edit-library-macros*
	(let ((acc nil))
	  (map-library #'(lambda (id name kind)
			   (setf acc (nconc (edit-macros-of-object id name kind)
					    acc))))
	  acc))
  nil)

;; kludge since rehash at start_edd time does not see ostates.
(defunml (|oed_edit_library_rehash| (unit)  :declare ((declare (ignore unit))))
    (unit -> unit)
  (edit-rehash-library-macros))


(defun edit-hash-library-macros (prefix)
  (let ((*macro-library-token-prefix* prefix)
	(table (make-hash-table :test #'equal :size 1000)))
    
    (map-library #'(lambda (id name kind)
		     (mapc #'(lambda (m)
			       (edit-hash-library-macro m table))
			   (edit-macros-of-object id name kind))))

    (setf *edit-library-hash-tables* (acons prefix table *edit-library-hash-tables*))))


(defun add-edit-macros-of-object-aux (id name kind)
  (let ((macs (edit-macros-of-object id name kind)))
    (mapc #'(lambda (prefix)
	      (let ((*macro-library-token-prefix* prefix))
		;;(setf -macs macs)  (break "aemoa")
		(dolist (m macs)
		  (let ((table (table-of-macro-library-token-prefix)))
		    (when table
		      (edit-hash-library-macro m table))))))
	  *macro-library-token-prefixes*)

    (setf *edit-library-macros* (nconc macs *edit-library-macros*))))


(defun delete-edit-macros-of-object-aux (id name kind)
  (let ((tokens (edit-macro-tokens-of-object id name kind)))
    (mapc #'(lambda (prefix)
	      (let ((*macro-library-token-prefix* prefix))
		(dolist (token tokens)
		  (let ((table (table-of-macro-library-token-prefix)))
		    (when table
		      (edit-unhash-library-token token table))))))
	  *macro-library-token-prefixes*)
	      
    (setf *edit-library-macros*
	  (delete-if #'(lambda (macro)
			 (member (token-of-macro macro) tokens :test #'equal))
		     *edit-library-macros*
		     :count 1))))


;;;
;;;   dynamic macros
;;;

(defun throw-extendable (buffer)
  (throw 'extendable-throw buffer))

(defun edit-try-term-sig-macro (buffer)
  (mlet* (((term-sig token)
	   (handle-process-err #'(lambda (err) (declare (ignore err)) nil)
	     (parse-term-sig buffer)))) ; token is text scanned.
     (when term-sig   
       (list (make-dynamic-macro token
				 #'(lambda (token version state)
				     (declare (ignore token version))
				     (edit-insert-replace-term state
							       (template-term-to-placeholder-term
								(term-sig-to-term term-sig)))))))))

(defun construct-dynamic-macros (buffer extendable-p)
  (or (when (forall-p #'printable-ichar-p buffer) (edit-try-term-sig-macro buffer))
      (if (not (identifier-ichar-p (car buffer)))
	  nil
	  (list
	   (if (numeric-ichar-p (car buffer))
	       (make-dynamic-macro
		(do ((b buffer (cdr b))
		     (token nil (cons (car b) token)))
		    ((or (null b)
			 (not (numeric-ichar-p (car b))))
		     (when (and (null b) extendable-p)
		       (throw-extendable buffer))
		     (nreverse token)))
		#'edit-numeric-token)
	       (make-dynamic-macro
		(do ((b buffer (cdr b))
		     (istring nil (cons (car b) istring)))
		    ((or (null b)
			 (not (identifier-ichar-p (car b))))
		     (when (and (null b) extendable-p)
		       (throw-extendable buffer))
		     (nreverse istring)))
		#'edit-alpha-token))))))


;;;;
;;;;  Macro Tokenizing/Lookup
;;;;
;;;;
;;;;  Library tokens are always alpha-numeric.
;;;;

;; returns nil, t, or 'advance :  advance means t plus bump buff.
(defun max-tok-compare (buff tok)
)


;;; returns macro with maximal token match,
;;; or throws extendable.
(defun find-maximal-macro-token (buffer macros-list extendable-p dynamic-p)
  (labels
      ((macro-token-match (token)
	 ;;(setf -buffer buffer -token token) (break "fmmt")
	 (do ((i 0 (1+ i))
	      (buff buffer (cdr buff))
	      (tok token (cdr tok)))
	       
	     ((let ((b (car buff))
		    (k (car tok)))
		(or (not (or (equal b k)
			     (let ((focus-b-p (focusich-p b)))
			       (or (when focus-b-p
				     (or (focusich-p k)
					 (when (equal (cadr buff) k)
					   ;;(progn (setf -buff buff) (break "fmmt") t)
					   (setf buff (cdr buff))
					   t)))
			   
				   ;; matching focus changes is optional.
				   (when (and focus-b-p
					      (equal (cadr buff) k)
					      ;;(progn (setf -buff buff) (break "fmmt") t)
					      )
				     (setf buff (cdr buff))
				     t)))))
		    (null buff)
		    (null tok)))
	      (cond
		((null tok) i)
		((and (null buff) extendable-p)
		 (throw-extendable buffer))
		(t 0)))))

       (find-max (macros maximal maximal-length)
	 (if (null macros)
	     (values maximal maximal-length)
	     (let ((l (macro-token-match (token-of-macro (car macros)))))
	       (when l
		 (if (> l maximal-length)
		     (find-max (cdr macros) macros l)
		     (find-max (cdr macros) maximal maximal-length)))))) )

    (let ((maximal nil)
	  (maximal-length 0))

      (do ((macros macros-list (cdr macros)))
	  ((null macros))
	(multiple-value-setq (maximal maximal-length)
	    (find-max (car macros) maximal maximal-length)))

      (when dynamic-p
	(multiple-value-setq (maximal maximal-length)
	  (find-max (construct-dynamic-macros buffer extendable-p) maximal maximal-length)))

      maximal)))



;;;;
;;;;	find-macro (<istring> <emacro> list list <bool{extendable-p})
;;;;	 : <emacro> list
;;;;	 ** first <emacro> of result is found macro.
;;;;	 ** throws 'extendable if there is a macro which contains <istring> as prefix.
;;;;	 ** found macro will first <emacro> which is longest prefix of <istring>
;;;;		
;;;;	
;;;;

;;;  Search macro table for prefix of buffer which is token.
;;;    - if none, nil
;;;    - if extendable, throws
;;;      - Extendable if buffer is proper prefix of a macro.
;;;    - otherwise, returns macro, and segment if expand macro.

(defun find-macro (buffer macro-lists extendable-p &optional dynamic-p)
  (find-maximal-macro-token buffer
			    macro-lists
			    extendable-p
			    dynamic-p
			    ))

(defvar *dynamic-char-macros* nil)

(setf *dynamic-char-macros*
      (let ((a (make-array 256)))
	(dotimes (i 256)
	  (setf (aref a i)
		(make-dynamic-macro i
				    #'(lambda (ich version state)
					(declare (ignore version))
					(edit-text-insert-next state ich)))))
	a))



;; seems pretty wasteful to construct this closure/macro for each
;; char entered.
(defun find-text-macro (state ich)
  (let ((macro (aref *edit-text-character-macros* ich)))
    (if (and macro (edit-test-macro-conditions macro state))
	macro
	(aref *dynamic-char-macros* ich))))


;;	(make-dynamic-macro ich
;;			    #'(lambda (ich version state)
;;				(declare (ignore version))
;;				(edit-text-insert-next state ich)))




;;;
;;;  Static/Command macros
;;;

(defun make-m-x-token (string)
  `(((:meta) . ,(char->ichar #\X)) ,@(istring string)))
	   
(defun make-m-x-static-macro (string function)
  ;;(format t "static ~a~%" string)
  (make-static-macro (make-m-x-token string)
		     (cond
		       ((symbolp function)
			(symbol-function
			 (intern-system (string-upcase function))))
		       ((stringp function)
			(symbol-function
			 (intern-system (string-upcase function))))
		       ((functionp function)
			function)
		       (t (break) (message-emit (warn-message '(edit static macro function not)))
			  #'(lambda (s) s)))))



(defun make-insert-ichar-macro (ichar)
  (make-static-macro (make-m-x-token (format-string "insert_ichar_~a" ichar))
		     #'(lambda (state)
			 (edit-text-insert-next state ichar))))

(defun make-screen-macro (tok-func)
  (make-m-x-static-macro (car tok-func)
			 (cdr tok-func)))


(defun make-command-macro (tok-func)
  (make-m-x-static-macro (car tok-func) (cdr tok-func)))


(defun edit-rehash-command-macros-aux (cmds)
  (mapcar #'make-command-macro cmds))

(defun edit-rehash-screen-macros-aux (cmds)
  (mapcar #'make-screen-macro cmds))






(defun edit-rehash-static-macros-aux (cmd screen)
  
  (nconc

   (let ((accumulator nil))
     (dotimes (i 256 accumulator)
       (push (make-insert-ichar-macro i) accumulator)))

   (edit-rehash-command-macros-aux cmd)
   (edit-rehash-screen-macros-aux screen)))


(defvar *screen-commands-alist* nil)
(setf *screen-commands-alist*
      `(("no-op"		. "edit-no-op")
	("screen-left"		. "edit-screen-left")
	("screen-right"		. "edit-screen-right")
	("screen-up"		. "edit-screen-up")
	("screen-down"		. "edit-screen-down")
	("screen-page-down"	. "edit-screen-page-down")
	("screen-page-up"	. "edit-screen-page-up")
	("screen-scroll-up"	. "edit-screen-scroll-up")
	("screen-scroll-down"	. "edit-screen-scroll-down")
	("screen-eol"		. "edit-screen-eol")
	("screen-sol"		. "edit-screen-sol")))


(defun edit-rehash-static-macros ()

  ;;(edit-refiner-primitives)
  
  (setf *edit-static-macros*
	
	(nconc

	 (edit-rehash-static-macros-aux (primitive-commands-alist)
					*screen-commands-alist*)
					
	 ;; not certain why this was here. was irubout and edit-text-rubout.
	 ;;(list (make-static-macro (list ibackspace) #'edit-text-delete-prev))
	 ))
    
    ;; In order for expand macro preprocess to work on these
    ;; *edit-static-macros* must be set before they are processed.
    (setf *edit-static-macros*
	  (nconc (list (string-to-expand-macro "(space)==(-text)(m-x)insert_ichar_32")
		       (string-to-expand-macro "(space)==(-^text)(m-x)no-op")
		       (string-to-expand-macro "(m-x)library-macro-hook==(m-x)library-token(m-x)down-left")
		       (string-to-expand-macro "(m-x)numeric-hook==(m-x)yank")
		       (string-to-expand-macro "(m-x)alpha-hook==(m-x)yank")
		       (string-to-expand-macro "(m-x)cleanup-hook==(m-x)no-op"))
		 *edit-static-macros*))
    nil)



     
;;;
;;;
;;;

;;;;
;;;;	Macros:
;;;;
;;;;	find-macro
;;;;
;;;;	Preprocess: tokenizes and thins.
;;;;	Dynamic:
;;;;


(defvar *preprocess-ask-p* t)

(defun preprocess-expand-macro (m expand-macros)
  (do ((buffer (expansion-of-expand-macro m))
       (macros nil))
      ((null buffer) (nreverse macros))
    (let* ((collection (let ((macros (find-macro buffer (list expand-macros) nil)))
			(when macros (collection-of-expand-macro (car macros)))))
	   (static (car (find-macro buffer (list *edit-static-macros*) nil)))
	   (degenerate (when (and (null collection) (null static))
			 (let ((token (do ((b buffer (cdr b))
					   (tok nil (cons (car b) tok)))
					  ((not (printable-ichar-p (car b)))
					   (nreverse tok)))))
			   (when token
			     (make-degenerate-macro token))))))
      
      ;;(format t "~a~%" (edit-node-macro-string m))
      (format t "m")
      (cond
	((and collection
	      (or (null static)
		  (<= (length (token-of-macro static))
		      (length (token-of-macro collection)))))
	 (let ((collected (macros-of-collection collection)))
	   (if (null (conditions-of-expand-macro (car collected)))
	       (push (car collected) macros)
	       (push collection macros))))
	(static (push static macros))
	(degenerate (push degenerate macros))
	(t
	 ;;(setf -a collection -b static -c degenerate -d macros -e m -f buffer -g expand-macros) (break "pem")      
	 (let ((msg (format-string "PreProcessMacro[~a]: Cannot tokenize ~a."
				     (implode-to-string
				      (estring-to-istring (token-of-macro m)))
				     (estring-to-string buffer))))

	     (format t msg)
	     #|
	     (when *preprocess-ask-p*
	       (format t msg)
	       (when (y-or-n-p "Write current macros to file?")
		 (save-macros "~/preprocess-error.macro"))
	       (setf *preprocess-ask-p* (not (y-or-n-p "Would you like this break not to occur?")))
	       (unless *preprocess-ask-p*
 		 (format t "(edit-rehash-macros) and (edit-rehash-expand-macros) will change bit back.~%")))
	     (if *preprocess-ask-p*
		 (process-err "preprocess")
		)
	      |#


	     (process-err msg)))))
    (setf buffer (edit-consume-token buffer (car macros)))))

(defun make-expand-macro (token expansion conditions &optional new)
  (labels ((collect (macros token)
	     (if (null macros)
		 (let ((static (find-macro token (list *edit-static-macros*) nil)))
		   (when static
		     (list (car static))))
		 (cons (car macros)
		       (let ((macros (find-macro token
						 (list (cdr macros))
						 nil)))
			 ;;(setf a token b macros ) (break)
			 (if (and macros (expand-macro-p (car macros)))
			     (macros-of-collection (collection-of-expand-macro (car macros)))
			     (collect macros token)))))))

    (let* ((m (make-node-macro 'expand
			       token
			       expansion
			       conditions))
	   (macros (cond
		     ((eql t new) nil)
		     ((null new) *edit-expand-macros*)
		     (t new)))
	   (expansions
	    (when *edit-preprocess-macros-p*
	      ;;(setf -a m) (break "mem")
	      (handle-process-err
	       #'(lambda (err) (message-emit (inform-message '(oed macro preprocess) err)) nil)
	       (preprocess-expand-macro m
					macros)))))
      (when expansions
	(setf (edit-node-macro-children m) expansions))
      ;;(setf c m d macros e expansions)
      (setf (edit-node-macro-collection m) (collect-macros (collect (cons m macros) (token-of-macro m))
							   (token-of-macro m)))
      m)))


(defun string-to-expand-macro (s &optional (preprocess t))
  (when (null *edit-static-macros*) (edit-rehash-static-macros))
  (with-string-scanner (s)
    (with-error-prefix (format-string "ScanMacro[~a]: " s)
      (scan-edit-expand-macro preprocess))))


(defun add-expand-macro (macro)
  (if (text-character-macro-p macro)
      (setf (aref *edit-text-character-macros* (car (token-of-macro macro))) macro)
      (setf *edit-expand-macros* (cons macro *edit-expand-macros*))))

(defun delete-expand-macro (token)
  (let ((found nil))
    (setf *edit-expand-macros*
	  (delete-if #'(lambda (macro)
			 (when (equal (token-of-macro macro) token)
			   (setf found t)))
		     *edit-expand-macros*))
    (not (if found
	     nil
	     ;;(setf *edit-expand-macros*
	     ;;(edit-rehash-expand-macros *edit-expand-macros*))
	     (if (and (null (cdr token))
		      (printable-ichar-p (car token))
		      (aref *edit-text-character-macros* (car token)))
		 (setf (aref *edit-text-character-macros* (car token)) nil)
		 (progn 
		   (display-msg
		    (format-string "UnMacro: macro ~a not found"
				   (estring-to-istring token)))
		   t))))))

(defun edit-delete-macros ()
  (setf *edit-expand-macros* nil)
  (setf *edit-text-character-macros* (make-array 255 :initial-element nil))
  ;;(edit-rehash-macros)
  )

(defun edit-macro-dump (port)
  (dolist (macro (reverse *edit-expand-macros*))
    (write-line (implode-to-string
		 (unscan-edit-expand-macro macro)) port)))


(defun edit-macro-load (port expand-prefix)
  (when (null *edit-static-macros*) (edit-rehash-static-macros))

  (with-macro-prefixes (nil (when expand-prefix (string-to-estring expand-prefix)) nil)
    (let ((loaded nil))
      (prog1 (do ((n 0))
		 ((null (peek-char nil port nil nil)) n)
	       (let ((line (read-line port)))
		 (cond
		   ((string= "" line)
		    nil)
		   ((eql #\% (char line 0))
		    (when (and (> (length line) 1) (eql #\# (char line 1)))
		      (set-library-token-prefix (subseq line 2)))
		    (when (and (>= (length line) #.(length "%%AutoLibToken"))
			       (eql #\% (char line 1))
			       (string= "AUTOLIBTOKEN" (string-upcase (subseq line 2 #.(+ 2 (length "AUTOLIBTOKEN"))))))
		      (if (and (> (length line) #.(length "%%AutoLibToken"))
			       (eql #\- (char line #.(length "%%AutoLibToken"))))
			  (set-degenerate-token-version nil)
			  (set-degenerate-token-version 'auto))))
		   (t
		    ;;(setf -a line) (break "sm")
		    (with-string-scanner (line)
			(unless (scan-at-unescaped-ichar-p (ichar #\%))
			  (incf n)
			  (handle-process-err #'(lambda (err-str)
						  (display-msg err-str))
					      (with-error-prefix (format-string "ScanMacro[~a]: " line)
						(let ((macro (scan-edit-expand-macro)))
						  ;;(display-msg (format-string "Macro ~a loaded"
						  ;;(print-edit-expand-macro macro)))
						  (push macro loaded)
						  (add-expand-macro macro)))))))))
      
	       (let ((prefix (macro-expand-token-prefix)))
		 (when prefix (dolist (m loaded)
				(setf (edit-macro-token m) (append prefix (lhs-of-macro m))))))

	       (add-macro-library-token-prefix (macro-library-token-prefix) t))))))

(defun edit-macro-def (s prefix)
  (handle-process-err #'(lambda (err-str)
			  (display-msg err-str)
			  nil)

		      
		      (with-error-prefix (format-string "ScanMacro[~a]: " s)
			(with-string-scanner (s)
			  (let ((macro (scan-edit-expand-macro)))
			    ;;(print-edit-expand-macro macro)))
			    (when (edit-node-macro-children macro)
			      (when (and prefix (not (string= "" prefix)))
				(setf (edit-macro-token macro) (append prefix (lhs-of-macro macro))))
			      (add-expand-macro macro)
			      t
			      ))))))




;;;;	
;;;;	token : 'AUTO or not.
;;;;	  - if auto then literal text in macro interpreted as library token or alpha insert depending on context.
;;;;	  - otherwise just inserted on term stack.
;;;;	prefix : ??? if text prefixed with prefix then check for library token and insert auto if avaliable???
;;;;	

(defunml (|define_macro| (version prefix s))
    (token -> (string -> (string -> bool)))

  (let ((*macro-degenerate-token-version* version))
    (edit-macro-def s prefix)))


(defunml (|undefine_macro| (s))
  (string -> bool)

  (delete-expand-macro (string-to-estring s)))




(defun edit-rehash-expand-macros ()
  (let ((macros *edit-expand-macros*))
    (setf *edit-expand-macros* 
	  (let ((current-expand-prefix nil)
		(processed nil)
		(*edit-expand-macros* nil))

	    (setf *preprocess-ask-p* t)
  
	    (do* ((macros (reverse macros) (cdr macros)))
		 ((null macros) *edit-expand-macros*)

	      (with-macro-prefixes ((library-prefix-of-expand-macro (car macros))
				    (expand-prefix-of-macro (car macros))
				    (version-of-expand-macro (car macros)))

		(unless (equal current-expand-prefix (macro-expand-token-prefix))
		  (when current-expand-prefix
		    (dolist (m *edit-expand-macros*)
		      (setf (edit-macro-token m) (append current-expand-prefix (lhs-of-macro m)))))

		  (setf current-expand-prefix (macro-expand-token-prefix)
			processed nil))
      
		(add-macro-library-token-prefix (macro-library-token-prefix))

		(let ((m (or (make-expand-macro (lhs-of-macro (car macros))
						(expansion-of-expand-macro (car macros))
						(conditions-of-expand-macro (car macros))
						*edit-expand-macros*)
			     (progn;;(setf a macros b *edit-expand-macros*)
			       (break)))))
		  (push m processed)
		  (push m *edit-expand-macros*)))))))
  nil)



;; It is assumed that the library macro tokens never have modified
;; characters.
(defun edit-macros ()
  (list	*edit-expand-macros*
	*edit-library-macros*
	*edit-static-macros*))

;;;;
;;;;  rehash : produces list of macros.
;;;;

(defun edit-rehash-macros ()
  (format t "~%Rehashing edit macros:~%~%")  ;; (break "erm")
  (time (progn
	  (setf *macro-library-token-prefixes* (list nil)
		*edit-library-hash-tables* nil)
	  (edit-rehash-static-macros)
	  (edit-rehash-expand-macros)
	  (edit-rehash-library-macros)
	  ))
  (format t "~%Rehash edit macros completed.~%")
  nil)



;;; leaf : lhs - function - args
;;;    function would be command function, alpha-numeric, template.
;;;    args would be alpha string or template name.
;;;    arg could be lhs itself!
;;;   static - commands
;;;   library - template
;;;   dynamic    - alpha/numeric
;;; expand : lhs - rhs
;;;


(defun edit-condition-test-p (point condition)
  ;;(format t "edit-condition-test-p ~a~%" condition)
  (without-dependencies
   ;;(setf -point point -condition condition) (break "ecp")
   (let ((dform (dform-of-dtree-c point)))
     (or (and dform (dform-condition-p dform condition))
	 (member condition (conditions-of-dtree point))
	 (let ((term (term-of-dtree point)))
	   ;;(setf -term term -condition condition ) ;;(break)
	   (let ((st (string-upcase (id-of-term term)))
		 (sc (string condition)))
	     ;;(format t "~a ~a ~a~%" st sc (string= st sc))
	     (or (string= st sc)
		 (and (eql #\! (char st 0))
		      (string= (subseq st 1) sc))
		 (abstraction-condition-p term condition))))))))

(defvar *edit-condition-table* (make-hash-table :size 16))

(defun edit-macro-condition-p (condition-id macro)
  (and (expand-macro-p macro)
       (some #'(lambda (condition)
		 (and (eql (id-of-condition condition) condition-id)
		      (not (not-condition-p condition))))
	     (conditions-of-expand-macro macro))))


(defvar *oed-new-focus-p*)

(defun set-new-focus-p () (when (boundp '*oed-new-focus-p*) (setf *oed-new-focus-p* t)))
(defun new-focus-p () (and (boundp '*oed-new-focus-p*)  *oed-new-focus-p*))

;;;;	
;;;;	v5 changes :
;;;;	 
;;;;	textvariable textnatural textlevel textstring texttoken removed
;;;;	but type-ids will be valid conditions.
;;;;	objectid -> object-id.

(defvar *current-conditions* nil)
(setf *current-conditions*
      '(text screen leaf top left insert dtree ml_string slot
	textfirst textlast last first termfirst termlast termleaf

	variable natural level-expression string token time oid bool
	abstraction-meta display-meta
	meta

	ObjectIdDAG
	
	;; need general purpose test of ostate-properties.
	;; for each property present and  not !bool(false) or !void() then
	;; considered a condition.
	dynamicrefresh

	toplevellist listtop listelement list listfirst listlast

	stacklist stacktext stacktop cmdstacktop 
	stackempty cmdstackempty

	dispobject absobject thmobject mlobject comobject latobject ruleobject

	;;rulebox transformation cmd msg

	verified evaled

	modified modified-s object-modified object-modified-s object-current

	zoom orphandform
	newfocus pointsuppression
	))

(defvar *current-edit-conditions* nil)
(setf *current-edit-conditions*
      '(stacklist stacktext stacktop cmdstacktop stackempty cmdstackempty
	;;todo
	newfocus
	))

(defvar *current-object-conditions* nil)
(setf *current-object-conditions*
      (append '(text screen insert dtree

		walked evaled

		ObjectIdDAG

		dispobject absobject thmobject mlobject comobject latobject ruleobject

		verified
		modified modified-local modified-remote modified-both

		toploop
		;;rulebox transformation msg

		;; todo :
		modified modified-s object-modified object-modified-s object-current
		pointsuppression)
	      
	      *library-object-content-types*))


(defvar *current-dtree-conditions* nil)
(setf *current-dtree-conditions*
      `(leaf top left ml_string slot
	textfirst textlast last first termfirst termlast termleaf

	variable natural level-expression string token time oid
	abstraction-meta display-meta
	meta

	toplevellist listtop listelement list listfirst listlast

	;;new with v5 :
	non-modifiable
	
	;; todo
	zoom orphandform
	))
	   



;;;;	
;;;;	End macro defs.
;;;;	




;;;;	
;;;;	edit-state/view-state
;;;;	
;;;;	
;;;;	

(defun edit-reset-dyn (v oid)

  (dyneval-set oid (lookup-term-not-dyn oid))
  (view-touched v)

  v)
  

(defun edit-reset (state)
  
  ;; todo try to move labels to similar positions?
  (let ((old-dtree (dtree-of-view state))
	(oid (oid-of-view state)))

    (if (edit-indirect-oid-p oid)
	(edit-reset-dyn state oid)
	(let ((vo (object-of-view state)))
	  (let ((term (term-of-vobject vo)))

	    (view-dtree-init state term (implicit-of-vobject vo))
	    (view-refresh-title state)

	    ;; should be the same already. restore stack appears pointless.
	    (setf (edit-state-restore-stack state) (list term)))

	  (handle-process-err #'display-msg
			      (edit-asynch-macro *object-initialize-hook* state))

	  ;; should attempt to move marks from old to new
	  ;; but for now just set point to top.
	  (tag-dtree (dtree-of-view state) `point t)
	  (point-rehash state 'point)
      
	  ;; what about the point.
	  (view-flag-set-present-required state t)
	  (view-present state)

      
	  state))))

(defun edit-save (state)
  )

(defun edit-init-state (state)

  (let ((vo (object-of-view state)))
    (let ((term (term-of-vobject vo))
	  )

      ;; probably should be elsewhere:
      (setf (edit-state-restore-stack state)	(list term)
	    )))

    (handle-process-err #'display-msg
			(edit-asynch-macro *object-initialize-hook* state))

    ;; what about the point.
    (view-flag-set-present-required state t)
    (view-present state)
    
    state)

(defun edit-find-ml (point)
  (if (and (not (dtree-leaf-p point))
	   (iml-string-point-p point))
      point
      (if (null (parent-of-dtree point))
	  nil
	  (edit-find-ml (parent-of-dtree point)))))

(defun get-condition-value-aux (state id point)
  ;;(when (or t (eql id 'mklink)) (setf -a id -b state -c point) (break "gcv"))
  (if (integerp id)
      (eql id
	   (if (parent-of-dtree point)
	       (let ((pos (position point (children-of-dtree (parent-of-dtree point)))))
		 (if (null pos)
		     nil
		     (1+ pos)))
	       1))

      (case id
	      
	(text (dtree-leaf-p point))

	(leaf (or (dtree-leaf-p point)
		  (null (child-of-dtree point (left-mode)))))

	(screen (let ((cursor (cursor-of-edit-state state)))
		  (if (echo-cursor-p cursor)
		      (screen-cursor-p (shadowed-cursor-of-echo-cursor cursor))
		      (screen-cursor-p cursor))))
	      
	(top  (null (parent-of-dtree point)))
	      
	(left (left-edit-state-p state))
	(insert (insert-edit-state-p state))
	(dtree (dtree-edit-state-p state))

	(evaled (edit-state-evaled-p state))
	(walked (edit-state-walked-p state))
	(ml_string (edit-find-ml point))

	(new-focus (new-focus-p))

	(modified		(and (view-flag-modified-q state) t))
	(modified-local	(eql 'local (view-flag-modified-q state)))
	(modified-remote	(eql 'remote (view-flag-modified-q state)))
	(modified-both	(eql 'both (view-flag-modified-q state)))

	(slot (if (dtree-leaf-p point)
		  (dtree-flag-slot-p point)
		  (iplaceholder-term-p (term-of-dtree point))))

	(textfirst (oed-leaf-label-first-p 'point point))
	(textlast (oed-leaf-label-last-p 'point point))

	(first (let ((parent (parent-of-dtree point)))
		 (or (null parent)
		     (eql (child-of-dtree parent (left-mode))
			  point))))
	(last  (let ((parent (parent-of-dtree point)))
		 (or (null parent)
		     (eql (child-of-dtree parent (right-mode))
			  point))))

	(termfirst (and (not (dtree-leaf-p point))
			(let ((parent (parent-of-dtree point)))
			  (or (null parent)
			      (eql (dtree-child t t *tree-ce* parent) point)))))
	(termlast  (and (not (dtree-leaf-p point))
			(let ((parent (parent-of-dtree point)))
			  (or (null parent)
			      (eql (dtree-child t nil *tree-ce* parent) point)))))
	(termleaf (and (not (dtree-leaf-p point))
		       (null (child-of-dtree point (left-mode)))))
	      
	(abstraction-meta  (and (dtree-leaf-p point)
				(eql 'abstraction (meta-bit-of-dtree-leaf point))))
	(display-meta  (and (dtree-leaf-p point)
			    (eql 'display (meta-bit-of-dtree-leaf point))))

	;; following two for compatability purposes would be nice to
	;; remove.
	(textparameter  (and (dtree-leaf-p point)
			     (eql 'abstraction (meta-bit-of-dtree-leaf point))))
	(texttemplate  (and (dtree-leaf-p point)
			    (eql 'display (meta-bit-of-dtree-leaf point))))

	(non-modifiable (dtree-flag-non-modifiable-p point))
	      
	(meta  (and (dtree-leaf-p point)
		    (let ((mbit (meta-bit-of-dtree-leaf point)))
		      (or (eql 'display mbit)
			  (eql 'abstraction mbit)))))

	(objectiddag (let ((o (oid-of-view state)))
		       (when o (dag-directory-p o))))


	(listfirst (when (edit-ilist-element-p point)
		     (if (dtree-leaf-p point)
			 (edit-ilist-top-p (parent-of-dtree point))
			 (edit-ilist-top-p point))))
	(listlast (when (edit-ilist-element-p point)
		    ;;(setf -point point) (break "ll")
		    (edit-ilist-bottom-p point)))
	(toplevellist (and (edit-ilist-list-p point)
			   (null (parent-of-dtree point))))
	(listtop (and (edit-ilist-list-p point)
		      (not (edit-ilist-list-p (parent-of-dtree point)
					      (operator-of-term
					       (term-of-dtree point))))))
	(listelement (if (and (dtree-leaf-p point)
			      (edit-text-carrier-p (parent-of-dtree point) t))
			 (edit-ilist-element-p (parent-of-dtree point))
			 (edit-ilist-element-p point)))
	(list (edit-ilist-list-p point))
	      

	(stacklist (let ((top (term-stack-peek)))
		     (and top (weak-ilist-list-p top))))
	(stacktop (term-stack-top-p))
	(stacktext (let ((top (term-stack-peek)))
		     (and top (text-carrier-p top))))
	(cmdstacktop (cmd-stack-top-p))
	(stackempty (null *edit-term-stack*))	      
	(cmdstackempty (null *edit-cmd-stack*))

	((cmd toploop)
	 (let ((o (oid-of-view state)))
	   (when o
	     (let ((tp (property-of-ostate `|TopLoop| o)))
	       (when (and tp (ibool-term-p tp))
		 (bool-of-ibool-term tp))))))

	(absobject (let ((o (oid-of-view state)))
		     (when o (eql 'abs (kind-of-ostate o)))))
	(dispobject (let ((o (oid-of-view state)))
		      (when o (eql 'disp (kind-of-ostate o)))))
	(thmobject (let ((o (oid-of-view state)))
		     ;;(setf -o o) (break "kos")
		     (when o (or (eql 'stm (kind-of-ostate o))
				 (eql 'prf (kind-of-ostate o))))))
	(mlobject (let ((o (oid-of-view state)))
		    (when o
		      (and (eql 'code (kind-of-ostate o))
			   (eql 'ml (let ((p (property-of-ostate 'language o)))
				      (when (and p (itoken-term-p p)) (token-of-itoken-term p))))))))
	(comobject (let ((o (oid-of-view state)))
		     (when o (eql 'com (kind-of-ostate o)))))
	(latobject (let ((o (oid-of-view state)))
		     (when o (eql 'prec (kind-of-ostate o)))))
	(ruleobject (let ((o (oid-of-view state)))
		      (when o (eql 'rule (kind-of-ostate o)))))
	(verified (let ((o (oid-of-view state)))
		    (when o (active-of-ostate o))))


	;; todo
	(otherwise
	 ;;(setf -point point) (break)
	 (or (let ((sid (string id)))
	       (let ((l (length sid)))
		 
		 (when (and (> l 5) (string= "_MODE" (subseq sid (- l 5))))
		   ;;(setf -sid sid) (break "sid")
		   (with-ignore (ml-text sid)))))

	     (if (dtree-leaf-p point)
		 (eql id (intern (string-upcase (string (type-id-of-dtree-leaf point)))))
		 (or (edit-condition-test-p point id)
		     (and (dform-of-dtree point)
			  (member id (tokens-of-display-form (dform-of-dtree point))))))

	     (let ((o (oid-of-view state)))
	       (when (and id o) (eql id (kind-of-ostate o))))
	     (gethash id *edit-condition-table*)
	     (condition-property-of-ostate-p state id))))))


(defun get-ancestor-condition-value (state id notp)
  
  (labels ((visit (point)
	     (when point
	       (let ((v (get-condition-value-aux state id point)))
		 (or (if notp (null v) v)
		     (visit (parent-of-dtree point)))))))

    (visit (let ((p (point-of-edit-state state)))
	     (when p (parent-of-dtree p))))))


(defun get-condition-value (state id count)
  (let* ((point (do ((pre-point (point-of-edit-state state) (parent-of-dtree pre-point))
		     (i count (1- i)))
		    ((or (null count) (= i 0) (null pre-point)) pre-point))))
    (when point
      (get-condition-value-aux state id point))))

(defun get-condition-value-old (state id count)
  (let* ((point (do ((pre-point (point-of-edit-state state) (parent-of-dtree pre-point))
		     (i count (1- i)))
		    ((or (null count) (= i 0) (null pre-point)) pre-point))))

    ;;(when (or t (eql id 'mklink)) (setf -a id -b state -c point) (break "gcv"))
    (if (null point)
	nil
	(if (integerp id)
	    (= id
	       (if (parent-of-dtree point)
		   (1+ (position point (children-of-dtree (parent-of-dtree point))))
		   1))

	    (case id
	      
	      (text (dtree-leaf-p point))

	      (leaf (or (dtree-leaf-p point)
			(null (child-of-dtree point (left-mode)))))

	      (screen (let ((cursor (cursor-of-edit-state state)))
			(if (echo-cursor-p cursor)
			    (screen-cursor-p (shadowed-cursor-of-echo-cursor cursor))
			    (screen-cursor-p cursor))))
	      
	      (top  (null (parent-of-dtree point)))
	      
	      (left (left-edit-state-p state))
	      (insert (insert-edit-state-p state))
	      (dtree (dtree-edit-state-p state))

	      (evaled (edit-state-evaled-p state))
	      (ml_string (edit-find-ml point))

	      (new-focus (new-focus-p))

	      (modified		(and (view-flag-modified-q state) t))
	      (modified-local	(eql 'local (view-flag-modified-q state)))
	      (modified-remote	(eql 'remote (view-flag-modified-q state)))
	      (modified-both	(eql 'both (view-flag-modified-q state)))

	      (slot (if (dtree-leaf-p point)
			(dtree-flag-slot-p point)
			(iplaceholder-term-p (term-of-dtree point))))

	      (textfirst (oed-leaf-label-first-p 'point point))
	      (textlast (oed-leaf-label-last-p 'point point))

	      (first (let ((parent (parent-of-dtree point)))
		       (or (null parent)
			   (eql (child-of-dtree parent (left-mode))
				point))))
	      (last  (let ((parent (parent-of-dtree point)))
		       (or (null parent)
			   (eql (child-of-dtree parent (right-mode))
				point))))

	      (termfirst (and (not (dtree-leaf-p point))
			      (let ((parent (parent-of-dtree point)))
				(or (null parent)
				    (eql (dtree-child t t *tree-ce* parent) point)))))
	      (termlast  (and (not (dtree-leaf-p point))
			      (let ((parent (parent-of-dtree point)))
				(or (null parent)
				    (eql (dtree-child t nil *tree-ce* parent) point)))))
	      (termleaf (and (not (dtree-leaf-p point))
			     (null (child-of-dtree point (left-mode)))))
	      
	      (abstraction-meta  (and (dtree-leaf-p point)
				      (eql 'abstraction (meta-bit-of-dtree-leaf point))))
	      (display-meta  (and (dtree-leaf-p point)
				  (eql 'display (meta-bit-of-dtree-leaf point))))

	      ;; following two for compatability purposes would be nice to
	      ;; remove.
	      (textparameter  (and (dtree-leaf-p point)
				      (eql 'abstraction (meta-bit-of-dtree-leaf point))))
	      (texttemplate  (and (dtree-leaf-p point)
				  (eql 'display (meta-bit-of-dtree-leaf point))))

	      (non-modifiable (dtree-flag-non-modifiable-p point))
	      
	      (meta  (and (dtree-leaf-p point)
			  (let ((mbit (meta-bit-of-dtree-leaf point)))
			    (or (eql 'display mbit)
				(eql 'abstraction mbit)))))

	      (objectiddag (let ((o (oid-of-view state)))
			     (when o (dag-directory-p o))))


	      (listfirst (when (edit-ilist-element-p point)
			   (if (dtree-leaf-p point)
			       (edit-ilist-top-p (parent-of-dtree point))
			       (edit-ilist-top-p point))))
	      (listlast (when (edit-ilist-element-p point)
			  ;;(setf -point point) (break "ll")
			  (edit-ilist-bottom-p point)))
	      (toplevellist (and (edit-ilist-list-p point)
				 (null (parent-of-dtree point))))
	      (listtop (and (edit-ilist-list-p point)
			    (not (edit-ilist-list-p (parent-of-dtree point)
						    (operator-of-term
						     (term-of-dtree point))))))
	      (listelement (if (and (dtree-leaf-p point)
				    (edit-text-carrier-p (parent-of-dtree point) t))
			       (edit-ilist-element-p (parent-of-dtree point))
			       (edit-ilist-element-p point)))
	      (list (edit-ilist-list-p point))
	      

	      (stacklist (let ((top (term-stack-peek)))
			   (and top (weak-ilist-list-p top))))
	      (stacktop (term-stack-top-p))
	      (stacktext (let ((top (term-stack-peek)))
			   (and top (text-carrier-p top))))
	      (cmdstacktop (cmd-stack-top-p))
	      (stackempty (null *edit-term-stack*))	      
	      (cmdstackempty (null *edit-cmd-stack*))

	      ((cmd toploop)
	       (let ((o (oid-of-view state)))
		 (when o
		   (let ((tp (property-of-ostate `|TopLoop| o)))
		     (when (and tp (ibool-term-p tp))
		       (bool-of-ibool-term tp))))))

	      (absobject (let ((o (oid-of-view state)))
			     (when o (eql 'abs (kind-of-ostate o)))))
	      (dispobject (let ((o (oid-of-view state)))
			     (when o (eql 'disp (kind-of-ostate o)))))
	      (thmobject (let ((o (oid-of-view state)))
			   ;;(setf -o o) (break "kos")
			   (when o (or (eql 'stm (kind-of-ostate o))
				       (eql 'prf (kind-of-ostate o))))))
	      (mlobject (let ((o (oid-of-view state)))
			  (when o
			    (and (eql 'code (kind-of-ostate o))
				 (eql 'ml (let ((p (property-of-ostate 'language o)))
					    (when (and p (itoken-term-p p)) (token-of-itoken-term p))))))))
	      (comobject (let ((o (oid-of-view state)))
			   (when o (eql 'com (kind-of-ostate o)))))
	      (latobject (let ((o (oid-of-view state)))
			   (when o (eql 'prec (kind-of-ostate o)))))
	      (ruleobject (let ((o (oid-of-view state)))
			   (when o (eql 'rule (kind-of-ostate o)))))

	      (verified (let ((o (oid-of-view state)))
			   (when o (active-of-ostate o))))


	      ;; todo

	      (otherwise
	       ;;(setf -point point) (break)
	       (or (if (dtree-leaf-p point)
		       (eql id (intern (string-upcase (string (type-id-of-dtree-leaf point)))))
		       (or (edit-condition-test-p point id)
			   (and (dform-of-dtree point)
				(member id (tokens-of-display-form (dform-of-dtree point))))))

		   (let ((o (oid-of-view state)))
		     (when o (eql id (kind-of-ostate o))))
		   (gethash id *edit-condition-table*)
		   (condition-property-of-ostate-p state id))))))))





(defvar *edit-ostate-condition-properties* '(editrefresh))

(defun edit-verify-modified (state)
  (view-verify-modified state)
  state)


(defun ostate-condition-property-p (v)
  (or (ivoid-term-p v)
      (and (ibool-term-p v)
	   (bool-of-ibool-term v))
      (itoken-term-p v)))

;; RLE TODO.
;; has unfortunate expectation that property value be !void for condition
;; to be significant. Need to change that, but involves updating objects with !void{} properties first.
(defun condition-properties-of-ostate (state)
  (let ((o (oid-of-view state)))
    (when o
      (mapcan #'(lambda (prop)
		  (when (and (member (car prop) *edit-ostate-condition-properties*)
			     (ostate-condition-property-p (cdr prop)))
		    (list (car prop))))
	      (properties-of-ostate o)))))


(defun condition-property-of-ostate-p (state cond)
  (let ((o (oid-of-view state)))
    ;;(setf -state -state -o o) (break "cpoo")
    (when o
      (exists-p #'(lambda (prop)
		    (when (and (eql (car prop) cond)
			       (ostate-condition-property-p (cdr prop)))

		      t))
		(properties-of-ostate o)))))



(defun edit-fail (state)
  (declare (ignore state))
  ;;(break "ef")
  (raise-error (error-message '(edit macro fail)))
  )

(defun edit-loop (state)
  (break "loop")
  (internal-edit-error "loop command should not be executed.")
  state)

(defun edit-status (state)
  (let* ((point (point-of-edit-state state))
	 (conditions (mapcan #'(lambda (c)
				 (when (get-condition-value state c nil)
				   (list c)))
			     *current-dtree-conditions*))
	 (parent (parent-of-dtree point))
	 (parent-conditions (mapcan #'(lambda (c)
					(when (get-condition-value state c 1)
					  (list c)))
				    *current-dtree-conditions*))
	 (object-conditions (append (mapcan #'(lambda (c)
						(when (get-condition-value state c 0)
						  (list c)))
					    *current-object-conditions*)
				    (condition-properties-of-ostate state)))
	 (edit-conditions (mapcan #'(lambda (c)
						(when (get-condition-value state c 0)
						  (list c)))
					    *current-edit-conditions*))
	 )
    
    (setf conditions (append conditions
			     (conditions-of-dtree point)))
    
    (unless (dtree-leaf-p point)
      (setf conditions
	    (append conditions
		    (conditions-of-dform (dform-of-dtree point))
		    (conditions-of-abstraction-term (term-of-dtree point)))))
    (setf conditions
	  (cons (if parent
		    (dtree-child-position point parent)
		    1)
		    conditions))
	  
    (when parent
      (setf parent-conditions
	    (append parent-conditions
		    (conditions-of-dform (dform-of-dtree parent))
		    (conditions-of-abstraction-term (term-of-dtree parent))
		    (conditions-of-dtree parent)))
      (let ((grand-parent (parent-of-dtree parent)))
	(setf parent-conditions
	      (cons (if grand-parent
			(dtree-child-position parent grand-parent)
			1)
		    parent-conditions))))
    
    (display-msg (format-string "~%Global ~a~%Object: ~a~%Parent: ~a~%Point: ~a"
				edit-conditions object-conditions parent-conditions conditions))
    
    state))

;;;;	
;;;;	Edit code.
;;;;	
;;;;	
;;;;	





;;;;	
;;;;	Macro application :
;;;;	
;;;;	state/conditions/dispatch/debug


(defmacro handle-extendable (error-handler &body body)
  (let ((local-tag (gensym)))
    `(block ,local-tag
       (funcall ,error-handler (catch 'extendable-throw
				 (return-from ,local-tag (progn ,@body)))))))

(defvar *print-macro-debug* nil)
(defvar *macro-debug* t)
(defvar *macro-debug-size* 256)
(defvar *macro-debug-index* 0)
(defvar *macro-debug-array* (make-array *macro-debug-size*))


(defunml (|toggle_macro_debug| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)
  (setf *print-macro-debug* (not *print-macro-debug*)))

(defmacro macro-debug (s)
  (let ((a (gensym)))
    `(when *macro-debug*
      (let ((,a ,s))
	(when *print-macro-debug*  (princ ,a) (terpri))
	(setf *macro-debug-index* (mod (1+ *macro-debug-index*) *macro-debug-size*))
	(setf (aref *macro-debug-array* *macro-debug-index*) ,a)))))

(defmacro macro-debug-wrap ((m prefix) &body body)
  (let ((a (gensym)))
    `(if *macro-debug*
      (let ((,a (string-of-macro ,m)))
	(prog2 (macro-debug (concatenate 'string ,prefix " Begin " ,a))
	    (progn ,@body)
	  (macro-debug (concatenate 'string "End " ,a))))
      (progn ,@body))))

(defun show-macro-debug (m)
  (let ((n (if (or (null m) (> m *macro-debug-size*))
	       *macro-debug-size*
	       m)))
    (dotimes (i n)
      (let ((j (1- (- n i))))
	(format t "~a ~a~%" j
		(aref *macro-debug-array*
		      (mod (- *macro-debug-index* j) *macro-debug-size*)))))))



(defvar *object-initialize-hook* (string-to-estring "(m-x)object-initialize"))
(defvar *edit-numeric-hook* (string-to-estring "(m-x)numeric-hook"))
(defvar *edit-alpha-hook* (string-to-estring "(m-x)alpha-hook"))
(defvar *edit-library-macro-hook* (string-to-estring "(m-x)library-macro-hook"))
(defvar *edit-cleanup-hook* (string-to-estring "(m-x)cleanup-hook"))


;;;;
;;;;  dynamic macro functions
;;;;

(defun edit-numeric-token (token version state)
  (if (edit-macro-expansion-p)
      (if (eql 'auto version)
	  (edit-text-insert-istring-next state token)
	  (progn
	    (term-stack-push (itext-term (implode-to-string token)))
	    state))
      (progn
	(term-stack-push (itext-term (implode-to-string token)))
	(edit-asynch-macro *edit-numeric-hook* state))))

(defun edit-alpha-token (token version state)
  (if (edit-macro-expansion-p)
      (if (eql 'auto version)
	  (edit-text-insert-istring-next state token)
	  ;; kludge : is eterm-stack really want we want. Assuming this is text in macro defs
	  ;; for not autolibtoken macros.
	  (progn (eterm-stack-push (itext-term (implode-to-string token)))
		 state))
      (progn
	;;(format t "A")
	(term-stack-push (itext-term (implode-to-string token)))
	(edit-asynch-macro *edit-alpha-hook* state)
	)))


;;;;
;;;;  Macro hooks.
;;;;

(defun edit-asynch-macro (estring state)
  (edit-lex-and-execute estring
			(edit-macros)
			state))


;;;;
;;;;  library macro functions
;;;;

(defvar *library-term* nil)

(defun edit-dtree-concrete-library-token (state)
  (let* ((term (term-stack-peek))
	 (string (when (and term (itext-term-p term)) (string-of-itext-term-r term)))
	 (macro (when string (gethash string
				      (table-of-macro-library-token-prefix nil)))))

    (if macro
	(if (dtree-edit-state-p state)
	    (edit-insert-replace-dtree state (term-of-library-macro macro))
	    (edit-insert-replace-term state (term-of-library-macro macro)))
	(process-err "EditInsertLibraryTerm: There is no term to insert."))))


(defun edit-library-token (state)
  (let* ((term (eterm-stack-peek))
	 (string (when (and term (itext-term-p term)) (string-of-itext-term-r term)))
	 (macro (when string (gethash string
				      (table-of-macro-library-token-prefix nil)))))

    ;;(setf a macro b term c string) (break)
    (if macro
	(progn (eterm-stack-pop)
	       (edit-insert-replace-term state (term-of-library-macro macro)))
	(raise-error (error-message '(edit insert library term))))))


(defun edit-dtree-library-token (state)
  (let* ((term (eterm-stack-peek))
	 (string (when (and term (itext-term-p term)) (string-of-itext-term-r term)))
	 (macro (when (and string (stringp string))
		  (gethash string
			   (table-of-macro-library-token-prefix nil)))))
    
    (setf a macro b term c string) (break)
    (if macro
	(progn (eterm-stack-pop)
	       (edit-insert-replace-dtree state (term-of-library-macro macro)))
	(raise-error (error-message '(edit insert library term dtree))))))
  

(defun edit-execute-library (macro state)
  (let ((term (term-of-library-macro macro)))
    ;;(setf a macro b term) (break)
    (if (edit-macro-expansion-p)
	(edit-insert-replace-term state
				  ;;(template-term-to-placeholder-term term)
				  term)
	(progn
	  ;;(setf a m ) (break)
	  ;;(format t "L")
	  (eterm-stack-push (itext-term (unprefixed-string-of-library-macro macro)))
	  (edit-asynch-macro *edit-library-macro-hook* state)))))



(defvar *edit-macro-expansion* nil)
(defmacro with-macro-expansion-p (&body body)
  `(let ((*edit-macro-expansion* t))
    ,@body))
(defun edit-macro-expansion-p ()
  *edit-macro-expansion*)

(defun edit-execute-collection (collection state &optional token)
  (if (null collection)
      (progn
	(format-string "Edit macro ~a has no versions satisfying conditions."
		       (estring-to-string token))
	state)
      (progn
	;;(setf a collection b state c token) (break "exc")
	(if (or (not (expand-macro-p (car collection)))
		(edit-test-macro-conditions (car collection) state))
	    (edit-execute (list (car collection)) state)
	    (edit-execute-collection (cdr collection)
				     state
				     (token-of-macro (car collection))))
	)))



(defvar *macro-loop* nil)
(defvar *macro-loop-point* nil)

(defun loop-macro-p (m)
  (and (static-macro-p m)
       (string= (string-of-macro m) "(M-X)loop")))

(defun edit-execute-macro (macro state)

  ;;(setf -m macro -v state) (break "exm")
  (cond

    ((not (edit-macro-p macro))
     (process-err
      (format-string "EditExecute: Unexpected object: a." macro)))

    ((expand-macro-p macro)
     (macro-debug-wrap (macro "Expand")
		       (let ((*macro-loop* macro)
			     (*macro-loop-point* (point-of-edit-state state)))
			 (edit-execute (macros-of-expand-macro macro) state))))

    ((loop-macro-p macro)
     ;;(break)
     (if (eql *macro-loop-point* (point-of-edit-state state))
	 (progn (format t "MacroLoop: Original Point encountered.~%")
		state)
	 (macro-debug-wrap (macro "LoopExpand")
			   (edit-execute-macro (collection-of-expand-macro *macro-loop*)
					       state))))


    ((macro-collection-p macro)
     ;;(setf -m macro -v state) (break "exm")
     (macro-debug-wrap (macro "Collection")
		       (edit-execute-collection (macros-of-collection macro) state)))
    (t (edit-apply-macro macro state))))

			
(defun edit-execute (macros state)
  (if (null macros)
      state
      (with-macro-expansion-p
	  (edit-execute (cdr macros)
			(edit-execute-macro (car macros) state)))))


(defun auto-focus-state (tok buffer state)
  ;;(setf -tok tok -buffer buffer) (break "afs")
  (if (and (not (focusich-p (car tok)))
	   (focusich-p (car buffer)))
      (progn
	(oed-focus-on (window-of-focus-index 1 *edit-buffer*))
	(view-of-window *oed-focus*))
      state))


;; 
(defun edit-interactive-consume-token (buffer macro extendable-p)
  (let ((newb (let ((buff buffer)
		    (tok (token-of-macro macro)))

		(do ()
		    ((null tok)
		     buff)

		  (cond
		    ((and (focusich-p (car buff)) (focusich-p (car tok)))
		     (setf buff (cdr buff) tok (cdr tok)))
		    
		    ((focusich-p (car buff))
		     (setf buff (cdr buff)))

		    (t (setf buff (cdr buff) tok (cdr tok))))))))

    
    (do ((b newb (cdr b)))
	((or (null b)
	     (not extendable-p)
	     (not (or (delimiter-ichar-p (car b))
		      (and (focusich-p (car b))
			   (delimiter-ichar-p (cadr b)))
		      )))
	 b))))


(defun edit-top-lex-and-ex (buffer macros-list state)
  (if buffer
      ;; note that extendable-throw will only occur when processing edit buffer
      ;; and not on a recursion processing an expansion.
      (let ((macros (handle-extendable #'(lambda (buffer)
					       (return-from edit-top-lex-and-ex
						 (values state buffer)))
				       (find-macro buffer macros-list t t))))
	;;(setf -b buffer -m macros) (break)
	(if (null macros)
	    (progn
	      (display-msg
	       (format-string "Edit buffer ~a has no prefix which is a macro,~%~a"
			      (estring-to-string buffer)
			      " nor is the buffer extendable to any macro.") t)
	      (values state nil))

	    (progn
	      (setf *edit-macro-dispatched* t)
	      (echo-set-ifocii *edit-buffer* (token-of-macro (car macros)))
	      (let ((new-state
		     (cond
		       ((expand-macro-p (car macros))
			(edit-execute-macro (collection-of-expand-macro (car macros)) state))
		       (t (edit-apply-macro (car macros) state)))))

		(when (null new-state)
		  (setf -m (car macros))
		  (break "elet"))

		;;(clear-new-focus-p)
		(edit-lex-and-execute (edit-interactive-consume-token buffer (car macros) t)
				      macros-list
				      new-state
				      t)))))
      (values state nil)))  

;; returns list of macros, last is remainder
(defun edit-lex-and-execute (buffer macros-list state
				    &optional (extendable-p t) conditional-p)
  (when (null state) (break "ele1"))
  (if buffer
      ;; note that extendable-throw will only occur when processing edit buffer
      ;; and not on a recursion processing an expansion.
      (let ((macros (handle-extendable #'(lambda (buffer)
					       (return-from edit-lex-and-execute
						 (values state buffer)))
				       (find-macro buffer macros-list extendable-p t))))
	;;(setf -b buffer -m macros) (break)
	(if (null macros)
	    (progn
	      (if conditional-p
		  (display-msg
		   (format-string "Edit macro ~a has no versions satisfying conditions."
				  (estring-to-string buffer)) t)
		  (display-msg
		   (format-string "Edit buffer ~a has no prefix which is a macro,~%~a"
				  (estring-to-string buffer)
				  " nor is the buffer extendable to any macro.") t))
	      (values state nil))

	    (let ((new-state
		   (cond
		     ((expand-macro-p (car macros))
		      (edit-execute-macro (collection-of-expand-macro (car macros)) state))
		     (nil (if (edit-test-macro-conditions (car macros) state)
			      (with-macro-expansion-p
				  (cond
				    ((macros-of-expand-macro (car macros))
				     (edit-execute (macros-of-expand-macro (car macros)) state))
				    (t (edit-lex-and-execute
					(expansion-of-expand-macro (car macros))
					(cons (cdr macros) (cdr macros-list))
					state nil))))
			      (edit-lex-and-execute (token-of-macro (car macros))
						    (cons (cdr macros) (cdr macros-list))
						    state nil t)))
		     (t (edit-apply-macro (car macros) state)))))

	      ;;(clear-new-focus-p)
	      (when (null new-state)  (setf -m (car macros) ) (break "ele"))
	      (edit-lex-and-execute (edit-interactive-consume-token buffer (car macros) extendable-p)
				    macros-list
				    new-state
				    extendable-p))))
      (values state nil)))






(defun edit-test-macro-condition (condition state)
  ;;(break "etmc")
  (if (ancestor-condition-p condition)
      (get-ancestor-condition-value state
				    (id-of-condition condition)
				    (not-condition-p condition))
      (let ((value (get-condition-value state
					(id-of-condition condition)
					(count-of-parent-condition condition))))
	(if (not-condition-p condition)
	    (not value)
	    value))))


(defun construct-degenerate-dynamic-macro (buffer version)
  (if (forall-p #'numeric-ichar-p buffer)
      (make-dynamic-macro buffer #'edit-numeric-token version)
      (make-dynamic-macro buffer #'edit-alpha-token version)))


(defun apply-degenerate-macro (m state)

  (let ((lib-macro (edit-gethash-library-macro m)))
    ;;(setf a m b lib-macro) (break "adm")
    (cond

      ((and lib-macro (eql 'auto (version-of-dynamic-macro m)))
       (edit-apply-macro lib-macro state))

      ;; for normal alpha macros
      ((and (null lib-macro)
	    (null (unprefixed-string-of-degenerate-macro m)))
       (edit-apply-macro (construct-degenerate-dynamic-macro (token-of-macro m)
							     (version-of-dynamic-macro m))
			 state))

      (t ;;(format t "D")
	 (eterm-stack-push (itext-term (or (unprefixed-string-of-degenerate-macro m)
					  (string-of-macro m))))
	 state))))


(defun edit-test-macro-conditions (macro state)
  ;;(setf a macro b state c (conditions-of-expand-macro macro)) (break "etmcs")
  (forall-p #'(lambda (c) (edit-test-macro-condition c state))
	    (conditions-of-expand-macro macro)))


(defvar *virtual-macro* nil)

(defun edit-apply-macro (macro state)
  (macro-debug (string-of-macro macro))
  (when *virtual-macro* (return-from edit-apply-macro state))
  (labels
      ((call-list (l state)
	 (if (null l)
	     (dispatch state)
	     (call-list (cdr l) (funcall (car l) state))))

       (dispatch (state)
	 ;;(setf -m macro -v state) (break "eam")
	 (case (type-of-macro macro)
	   (static	(funcall (function-of-static-macro macro) state))
	   (library	(edit-execute-library macro state))
	   (dynamic	(funcall (function-of-dynamic-macro macro)
				 (token-of-macro macro) (version-of-dynamic-macro macro) state))
	   (expand	(process-err (format-string
				      "Non-dispatchable (ie defined) macro dispatched: ~a."
				      (print-edit-expand-macro macro))))
	   (degenerate  (apply-degenerate-macro macro state))
	   (otherwise	(internal-edit-error
			 (format-string
			  "Unknown macro dispatched ~a."
			  (estring-to-string (token-of-macro macro))))))))

    (call-list (before-of-macro macro) state)))
		       


;;;;	
;;;;	
;;;;	COMMANDS.
;;;;	
;;;;	


;;;;
;;;; mode comands.
;;;;

;;;
;;; left/right
;;;

(defun edit-mode (state)
  (toggle-edit-state-left-right state))

(defun edit-left-mode (state)
  (left-edit-state state))

(defun edit-right-mode (state)
  (right-edit-state state))

;;;
;;; insert/replace
;;;

(defun edit-insert-replace-mode (state)
  (toggle-edit-state-insert-replace (text-edit-state state)))

(defun edit-insert-mode (state)
  (insert-edit-state (text-edit-state state)))

(defun edit-replace-mode (state)
  (replace-edit-state (text-edit-state state)))


;;;
;;;  move point.
;;;

(defun edit-up (state)
  (when (edit-move-label-up 'point state)
    (point-rehash state 'point))
  state)

(defun edit-top (state)
  (when (edit-move-label-top 'point state)
    (point-rehash state 'point))
  state)


(defun edit-down-aux (state mode &optional ce)
  (when (edit-move-label-down t mode (or ce *true-ce*) 'point state)
    (point-rehash state 'point))
  state)

(defun edit-down (state)
  (edit-down-aux state (left-right-of-edit-state state)))


(defun edit-down-left (state)
  (edit-down-aux state (left-mode)))

(defun edit-down-right (state)
  (edit-down-aux state (right-mode)))


(defun edit-down-tree (state)
  (edit-down-aux state
		 (left-right-of-edit-state state)
		 *tree-ce*))


(defun edit-next-sibling-tree (state)
  (when (edit-move-label-to-sibling t
				    (left-right-of-edit-state state)
				    *tree-ce*
				    'point
				    state)
    (point-rehash state 'point))
  state)

(defun edit-prev-sibling-tree (state)
  (when (edit-move-label-to-sibling t	
				    (toggle-mode (left-right-of-edit-state state))
				    *tree-ce*
				    'point
				    state)
    (point-rehash state 'point))
  state)

(defun edit-left-sibling-tree (state)
  (when (edit-move-label-to-sibling t (right-mode) *tree-ce* 'point state)
    (point-rehash state 'point))
  state)

(defun edit-right-sibling-tree (state)
  (when (edit-move-label-to-sibling t (left-mode) *tree-ce* 'point state)
    (point-rehash state 'point))
  state)

(defun edit-left-sibling (state)
  (when (edit-move-label-to-sibling t (right-mode)  *true-ce* 'point state)
    (point-rehash state 'point))
  state)

(defun edit-right-sibling (state)
  (when (edit-move-label-to-sibling t (left-mode) *true-ce* 'point state)
    (point-rehash state 'point))
  state)


(defvar *walk-ceiling-ce* (string-to-cond-expr "|(#WalkCeiling $WalkCeiling)"))
(defvar *parameter-ce* (string-to-cond-expr "parameter"))
(defvar *not-instantiated-ce* (string-to-cond-expr "!(instantiated)"))

(defun edit-walk-aux (dir stop-ce view)
  (setf (edit-state-walked-p view) nil)
  (when (edit-walk nil			; permuted order
		   stop-ce	
		   nil			; no stop hook
		   *walk-ceiling-ce*	
		    (false-ce)		; no floor
		    ;;*not-instantiated-ce* ; floor 
		   'point		; move point 
		   dir			
		   view)
    (setf (edit-state-walked-p view) t)
    (point-rehash view 'point))
  view)

(defun edit-walk-leaf (dir view)
  (edit-walk-aux dir *parameter-ce* view))


(defun edit-walk-prev-leaf (v)
  (edit-walk-leaf (toggle-mode (left-right-of-edit-state v)) v))

(defun edit-walk-next-leaf (v)
  (edit-walk-leaf (left-right-of-edit-state v) v))


(defun edit-walk-left-leaf (v)
  ;; in right mode next is left.
  (edit-walk-leaf (right-mode) v))

(defun edit-walk-right-leaf (v)
  ;; in left mode next is right.
  (edit-walk-leaf (left-mode) v))


(defun edit-walk-any (dir view)
  (edit-walk-aux dir *true-ce* view))

(defun edit-walk-next (view)
  (edit-walk-any (left-right-of-edit-state view) view))

(defun edit-walk-prev (view)
  (edit-walk-any (toggle-mode (left-right-of-edit-state view)) view))


(defun edit-walk-slot (dir view)
  ;;(break "ews")
  (edit-walk-aux dir *slot-ce* view))

(defun edit-walk-next-slot (v)
  (edit-walk-slot (left-right-of-edit-state v) v))

(defun edit-walk-prev-slot (v)
  (edit-walk-slot (toggle-mode (left-right-of-edit-state v)) v))

(defun edit-walk-right-slot (v)
  (edit-walk-slot (left-mode) v))

(defun edit-walk-left-slot (v)
  (edit-walk-slot (right-mode) v))


(defvar *point-ce* (string-to-cond-expr "~point"))

(defun edit-walk-down-aux (dir stop-ce view)
  (when (edit-walk nil			; permuted order
		   stop-ce	
		   nil			; no stop hook
		   *point-ce*		; ceiling
		   (false-ce) ;*not-instantiated-ce* ; floor 
		   'point		; move point 
		   dir			
		   view)
    (point-rehash view 'point))
  view)

(defun edit-walk-down-slot (dir view)
  ;;(setf -v view) (break "ewds")
  (edit-walk-down-aux dir *slot-ce* view))

(defun edit-walk-down-next-slot (v)
  (edit-walk-down-slot (left-right-of-edit-state v) v))

(defun edit-walk-down-prev-slot (v)
  (edit-walk-down-slot (toggle-mode (left-right-of-edit-state v)) v))


(defun edit-mark (state)
  (edit-move-label-to-label state 'mark 'point)
  state)

(defun edit-clear-mark (state)
  (edit-remove-label state 'mark)
  (point-rehash state 'point)
  state
  )

(defun edit-swap-point-mark (state)
  (edit-move-label-to-label state '!temp 'mark)
  (edit-move-label-to-label state 'mark 'point)
  (edit-move-label-to-label state 'point '!temp)
  (edit-remove-label state '!temp)
  
  (point-rehash state 'point)
  state)


(defun edit-mark1 (state)
  (edit-move-label-to-label state 'mark1 'point)
  state)

(defun edit-mark2 (state)
  (edit-move-label-to-label state 'mark2 'point)
  state)

(defun edit-mark3 (state)
  (edit-move-label-to-label state 'mark3 'point)
  state)

(defun edit-mark4 (state)
  (edit-move-label-to-label state 'mark4 'point)
  state)

(defun edit-mark5 (state)
  (edit-move-label-to-label state 'mark5 'point)
  state)

(defun edit-mark6 (state)
  (edit-move-label-to-label state 'mark6 'point)
  state)

(defun edit-mark7 (state)
  (edit-move-label-to-label state 'mark7 'point)
  state)

(defun edit-mark8 (state)
  (edit-move-label-to-label state 'mark8 'point)
  state)


;;;;	
;;;;	
;;;;	screen
;;;;	
;;;;	
;;;;	

(defun edit-no-op (state)
  state)


(defun edit-screen-left (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r (row-of-xcursor cursor))
	 (c (col-of-xcursor cursor))
	 (w (width-of-oed-window (window-of-view state))))

    (if (<= c 1)
	(if (<= r 1)
	    (if (zerop (offset-of-edit-state state))
		(set-edit-state-cursor (screen-cursor state 1 1)
				       state)
		(shift-screen (set-edit-state-cursor (screen-cursor state 1 w)
						     state)
			      -1))
	    (set-edit-state-cursor (screen-cursor state (1- r) w) state))
	(set-edit-state-cursor (screen-cursor state r (1- c)) state)))
  
  (view-flag-set-cursor-present-required state t)
  
  state)

(defun right-row-col (row col indent lines)
  (cond
    ((null lines)
     (values row col))
    ((null (cdr lines))
      (values row
	      (1- (+ col (length (text-of-line (car lines)))))))
    (t (values (1- (+ row (length lines)))
	       (let ((last (car (last lines))))
		 (if (continued-line-p last)
		     (length (text-of-line last))
		     (+ indent (length (text-of-line last)))))))))

(defun edit-screen-right (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r nil)
	 (c nil)
	 (w (width-of-oed-window (window-of-view state)))
	 (h (height-of-oed-window (window-of-view state))))

    (multiple-value-setq (r c)
      (right-row-col (row-of-xcursor cursor) (col-of-xcursor cursor)
		     (if (line-cursor-p cursor)
			 (indent-of-line-cursor cursor)
			 0)
		     (when (line-cursor-p cursor)
		       (lines-of-line-cursor cursor))))

    ;;(format t "ScreenRight ~a ~a ~a~%" r c (offset-of-edit-state state))

    (if (>= c w) ;; should never be >.
	(if (>= r h)
	    (shift-screen (set-edit-state-cursor (screen-cursor state h 1)
						 state)
			  1)
	    (set-edit-state-cursor (screen-cursor state (1+ r) 1)
				   state))
	(set-edit-state-cursor (screen-cursor state r (1+ c)) state))))



(defun edit-screen-up (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r (row-of-xcursor cursor))
	 (c (col-of-xcursor cursor))
	 )

    (if (<= r 1)
	(shift-screen (set-edit-state-cursor (screen-cursor state 1 c)
					     state)
		      -1)
	(set-edit-state-cursor (screen-cursor state (1- r) c)
			       state))))

(defun edit-screen-down (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r nil)
	 (c nil)
	 (h (height-of-oed-window (window-of-view state))))

    (multiple-value-setq (r c)
      (right-row-col (row-of-xcursor cursor) (col-of-xcursor cursor)
		     (if (line-cursor-p cursor)
			 (indent-of-line-cursor cursor)
			 0)
		     (when (line-cursor-p cursor)
		       (lines-of-line-cursor cursor))))

    (if (>= r h)
	(shift-screen (set-edit-state-cursor (screen-cursor state h c) state)
		      (1+ (- r h)))
	(set-edit-state-cursor (screen-cursor state (1+ r) c) state))))


(defun edit-screen-page-up (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r (row-of-xcursor cursor))
	 (c (col-of-xcursor cursor))
	 (h (height-of-oed-window (window-of-view state)))
	 (offset (offset-of-edit-state state)))

    (if (= 0 offset)
	state
	(shift-screen (set-edit-state-cursor (screen-cursor state r c)
					     state)
		      (- h)))))

(defun edit-screen-page-down (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r nil)
	 (c nil)
	 (h (height-of-oed-window (window-of-view state))))

    (multiple-value-setq (r c)
      (right-row-col (row-of-xcursor cursor) (col-of-xcursor cursor)
		     (if (line-cursor-p cursor)
			 (indent-of-line-cursor cursor)
			 0)
		     (when (line-cursor-p cursor)
		       (lines-of-line-cursor cursor))))

    (shift-screen (set-edit-state-cursor (screen-cursor state r c) state)
		  h)))



(defun edit-screen-eol (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r nil)
	 (c nil)
	 (w (width-of-oed-window (window-of-view state)))
	 )

    (multiple-value-setq (r c)
      (right-row-col (row-of-xcursor cursor) (col-of-xcursor cursor)
		     (if (line-cursor-p cursor)
			 (indent-of-line-cursor cursor)
			 0)
		     (when (line-cursor-p cursor)
		       (lines-of-line-cursor cursor))))

    (set-edit-state-cursor (screen-cursor state
					  r
					  (1+ (last-non-white-of-draw-line
					       (buffer-state-of-oed-window (window-of-view state))
					       (1- r) w)))
			   state)))


(defun edit-screen-sol (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r (row-of-xcursor cursor)))

    (set-edit-state-cursor (screen-cursor state r 1)
			   state)))


(defun edit-screen-scroll-down (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r nil)
	 (c nil))

    (multiple-value-setq (r c)
      (right-row-col (row-of-xcursor cursor) (col-of-xcursor cursor)
		     (if (line-cursor-p cursor)
			 (indent-of-line-cursor cursor)
			 0)
		     (when (line-cursor-p cursor)
		       (lines-of-line-cursor cursor))))

    ;;(setf a r b c d h) (break)
    (shift-screen (set-edit-state-cursor (screen-cursor state r c)
					 state)
		  1)))

(defun edit-screen-scroll-up (state)
  (let* ((cursor (cursor-of-view-c state))
	 (r (row-of-xcursor cursor))
	 (c (col-of-xcursor cursor))
	 (offset (offset-of-edit-state state)))

    (if (= 0 offset)
	state
	(shift-screen (set-edit-state-cursor (screen-cursor state r c)
					     state)
		      -1))))


;;;;	
;;;;	
;;;;	undo
;;;;	
;;;;	
;;;;	

(defun oed-edit-undo (v)
  (when (edit-undo v)
    (point-rehash v 'point))
  v)

(defun oed-edit-redo (v)
  (when (edit-redo v)
    (point-rehash v 'point))
  v)

(defun oed-edit-abort-undo (v)
  (when (edit-abort-undo v)
    (point-rehash v 'point))
  v)


;;;;	
;;;;	Term stack commands.
;;;;	
;;;;	
;;;;	

(defun edit-term-pop (state)
  (term-stack-pop)
  state)

(defun edit-term-rotate (state)
  (term-stack-rotate)
  state)

(defun edit-term-reverse-rotate (state)
  (term-stack-reverse-rotate)
  state)

(defun edit-term-unrotate (state)
  (term-stack-unrotate)
  state)


(defvar *mark-filter* (new-tags-filter nil nil nil '(mark mark1 mark2 mark3 mark4 mark5 mark6 mark7 mark8) nil)) 
(defvar *point-filter* (new-tags-filter nil nil nil '(point) nil)) 
(defvar *point-mark-filter* (new-tags-filter nil nil nil '(point mark mark1 mark2 mark3 mark4 mark5 mark6 mark7 mark8 !walk found) nil)) 
(defvar *temp-point-mark-filter* (new-tags-filter nil nil nil '(temp point mark mark1 mark2 mark3 mark4 mark5 mark6 mark7 mark8 !walk found) nil)) 

(defun edit-term-push (state point clear-mark-p)
  (let ((term (dtree-to-term point)))
    (when (null term) (break "about to push null term"))
    ;;(setf -v state -point point -term term) (break "etp")
    (term-stack-push (if clear-mark-p
			 (filter-term-tags term *point-mark-filter*)
			 (filter-term-tags term *point-filter*))))
  state)

(defun edit-push (state)
  (let ((point (point-of-edit-state state)))
    (when point
      (unless (dtree-leaf-p point)
	(edit-term-push state point nil)))
    state))


(defun edit-push-clear (state)
  (let ((point (point-of-edit-state state)))
    (unless (dtree-leaf-p point)
      (edit-term-push state point t))
    state))



;;;
;;;
;;;

(defun edit-term-epop (state)
  (eterm-stack-pop)
  state)

(defun edit-term-erotate (state)
  (eterm-stack-rotate)
  state)

(defun edit-term-ereverse-rotate (state)
  (eterm-stack-reverse-rotate)
  state)

(defun edit-term-epush (state point clear-mark-p)
  (let ((term (dtree-to-term point)))
    (when (null term) (break "about to push null term"))
    ;;(setf -v state -point point -term term) (break "etp")
    (eterm-stack-push (if clear-mark-p
			 (filter-term-tags term *point-mark-filter*)
			 (filter-term-tags term *point-filter*))))
  state)

(defun edit-parameter-epush (state)
  (let ((point (point-of-edit-state state)))
    (when (dtree-leaf-p point)
      (eterm-stack-push (iparameter-term (dtree-leaf-to-parameter point nil))))
    state))

;; grab top of ephemeral stack
(defun edit-grab (state)
  (term-stack-push (eterm-stack-peek-r))
  state)



(defun edit-epush (state)
  (let ((point (point-of-edit-state state)))
    (when point
      (unless (dtree-leaf-p point)
	(edit-term-epush state point nil)))
    state))


(defun edit-epush-clear (state)
  (let ((point (point-of-edit-state state)))
    (unless (and point (dtree-leaf-p point))
      (edit-term-epush state point t))
    state))

(defun edit-term-eyank (state)
  (when (edit-paste-term 'point (eterm-stack-peek-r) state)
    (point-rehash state 'point))
  state)

(defun edit-parameter-eyank (state)
  (let ((point (point-of-edit-state state))
	(p (car (parameters-of-term  (eterm-stack-peek-r)))))
    (when (and p (dtree-leaf-p point))
      (oed-leaf-replace state point
			(tag-dtree (new-modified-dtree-leaf point p)
				   'point t))
      (point-rehash state)))

  state)

;;;;	
;;;;	Structure Modify
;;;;	

(defun edit-term-delete (state)
  (when (edit-cut 'point state)
    (point-rehash state 'point))
  state)

(defun edit-term-yank (state)
  (when (edit-paste-term 'point (term-stack-peek-r) state)
    (point-rehash state 'point))
  state)

(defun edit-term-text-fix (state)
  (let ((point (point-of-edit-state state)))
    (unless (dtree-leaf-p point)
      (let ((term (dtree-to-term point)))
	(when (edit-paste-term 'point (term-fix-text term) state)
	  (point-rehash state 'point)))))
  state)

(defun edit-yank-merge (state)
  (edit-insert-replace-term state (term-stack-peek-r)))

(defun edit-yank-dtree-merge (state)
  (edit-insert-replace-dtree state (term-stack-peek-r)))

(defun edit-yank-dtree-structure-merge (state)
  (if (dtree-edit-state-p state)
      (edit-insert-replace-dtree state (term-stack-peek-r))
      (edit-insert-replace-term state (term-stack-peek-r))))

(defun edit-prf-term-push (state term clear-mark-p)
  (term-stack-push (if clear-mark-p
		       (filter-term-tags term *point-mark-filter*)
		       (filter-term-tags term *point-filter*)))
  state)

(defun edit-prf-push (state)
  (let ((point (point-of-edit-state state)))
    (when point
      (unless (dtree-leaf-p point)
	(if (iproof-node-term-p (term-of-dtree point))
	    (let* ((proof (eddtop-proof-of-proof-object (object-of-edit-state state)))
		   (term (oed-filter-point-mark (term-of-dtree point)))				   
		   (node (unlimit-depth term proof)))
	      (edit-prf-term-push state node nil))
	    (edit-term-push state point nil)))))
  state)

(defvar *proof-paste-function* (cons (cons #'(lambda (term) term) 1) nil))

(defunml (|set_proof_paste_f| (function))
    ((term -> term) -> unit)
  (setf *proof-paste-function* function)
  nil)

(defun edit-prf-yank-aux (state term node)
  (labels ((visit (node)
	     (let ((refinement (refinement-of-iproof-node-term node)))
		  
	       (modify-proof-node node
				  :refinement
				  (icons-left-term
				   (progn (format t "P-paste-f~%")
				   (funmlcall *proof-paste-function* (icar refinement)))
				   (icdr refinement))
				  :subgoals
				  (map-list-to-isexpr
				   (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						       (iproof-node-cons-op)
						       #'visit)
				   (iproof-node-nil-term))))))
			       
    (let* ((cursor (cursor-of-edit-state state))
	   (old (oed-filter-point-mark (term-of-dtree (point-cursor-point cursor)))))
	    
      ;;(setf -node node -old old) (break "epy")
      (if (iproof-node-term-p old)
	  (let* ((goal (goal-of-iproof-node-term old))
		 ;;lal and also keep old status
		 (n (modify-address-of-iproof-node-term node (address-of-proof-node-term node)))
		 (n2 (prf-yank-adjust-address n (address-of-proof-node-term old)))
		 (refinement (refinement-of-iproof-node-term term))
		 (tac (icar refinement))
 
		 (new (modify-proof-node
		       n2
		       :goal goal
		       :refinement (icons-left-term (funmlcall *proof-paste-function* tac)
						    (icdr refinement))
		       :subgoals (map-list-to-isexpr
				  (map-isexpr-to-list (subgoals-of-iproof-node-term n2)
						      (iproof-node-cons-op)
						      #'visit)
				  (iproof-node-nil-term))))
	     
		 (obj (object-of-edit-state state))
		 (proof (eddtop-proof-of-proof-object obj))
		 )
		
	    ;;(setf pp proof oo obj  nn new) ;;(break) 
	    (when (edit-paste-term 'point (limit-depth new)
				   ;;new
				   state)
	      (setf (proof-object-eddtop-proof (object-of-edit-state state))
		    (replace-proof-node proof new))
	      (point-rehash state 'point)))
	      
	  (when (edit-paste-term 'point term state)
	    (point-rehash state 'point))))))


(defun edit-prf-yank (state)
  (let* ((term (term-stack-peek-r))
	 (node (oed-filter-point-mark term)))

    (when (iproof-editor-term-p node)
      (raise-error (error-message '(edit-prf-yank stack-term not proof-node))))

    (if (iproof-node-term-p node)
	(edit-prf-yank-aux state term node)
	(when (edit-paste-term 'point term state)
	  (point-rehash state 'point))))
  state)

(defvar *proof-paste-function* (cons nil #'(lambda (term) term)))

(defunml (|set_proof_paste_f| (function))
    ((term -> term) -> unit)
  (setf *proof-paste-function* function)
  nil)

(defun edit-prf-yank (state)

  ;; visits a proof node, calling the transformation function on the tactic
  ;; of that node and all of its subnodes
  (labels ((visit (node)
	     (let ((refinement (refinement-of-iproof-node-term node)))		  
	       (modify-proof-node node
				  :refinement
				  (icons-left-term
				   (funmlcall *proof-paste-function* (icar refinement))
				   (icdr refinement))
				  :subgoals
				  (map-list-to-isexpr
				   (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						       (iproof-node-cons-op)
						       #'visit)
				   (iproof-node-nil-term))))))
    
			       
    (let* ((term (term-stack-peek-r))
	   (node (oed-filter-point-mark term)))

      (when (iproof-editor-term-p node)
	(raise-error (error-message '(edit-prf-yank stack-term not proof-node))))

      (if (iproof-node-term-p node)	
	  (let* ((cursor (cursor-of-edit-state state))
		 (old (oed-filter-point-mark (term-of-dtree (point-cursor-point cursor)))))
	    
	    ;;(setf -node node -old old) (break "epy")
	    (if (iproof-node-term-p old)
		(let* ((goal (goal-of-iproof-node-term old))
		       ;;lal and also keep old status

		       ;; here, first we look at the node we want to paste in,
		       ;; and what the address term is for that node. We update all the addresses in
		       ;; that node so that the top node that we paste in has nil address
		       (n (modify-address-of-iproof-node-term node (address-of-proof-node-term node)))

		       ;; next, we look at the node which we are pasting over and it's top-address.
		       ;; and we update the addresses in n so that they match the paste-over address.
		       (n2 (prf-yank-adjust-address n (address-of-proof-node-term old)))
		       
		       (refinement (refinement-of-iproof-node-term term))
		       (tac (icar refinement))
 
		       (new (modify-proof-node
			     n2
			     :goal goal
			     :refinement (icons-left-term (funcall (caar *proof-paste-function*) tac)
							  (icdr refinement))
			     :subgoals (map-list-to-isexpr
					(map-isexpr-to-list (subgoals-of-iproof-node-term n2)
							    (iproof-node-cons-op)
							    #'visit)
					(iproof-node-nil-term))))
	     
		       (obj (object-of-edit-state state))
		       (proof (eddtop-proof-of-proof-object obj))
		       )
		
		  ;;(setf pp proof oo obj  nn new) ;;(break) 
		  (when (edit-paste-term 'point (limit-depth new)
					 ;;new
					 state)
		    (setf (proof-object-eddtop-proof (object-of-edit-state state))
			  (replace-proof-node proof new))
		    (point-rehash state 'point)))
	      
		(when (edit-paste-term 'point term state)
		  (point-rehash state 'point))))
		   
	  (when (edit-paste-term 'point term state)
	    (point-rehash state 'point))))
    state))

;;;	
;;;	Text
;;;	

(defun edit-next-text (state)
  (edit-move-label-text (left-right-of-edit-state state)
			'point
			state)
  state)

(defun edit-prev-text (state)
  (edit-move-label-text (toggle-mode (left-right-of-edit-state state))
			'point
			state)
  state)


(defun edit-text-left-word (state)
  (edit-move-label-word (left-mode)
			'point
			state)
  state)

(defun edit-text-right-word (state)
  (edit-move-label-word (right-mode)
			'point
			state)
  state)
  

(defun edit-text-first (state)
  (edit-move-label-text-to-end (left-mode) 'point state)
  state)

(defun edit-text-last (state)
  (edit-move-label-text-to-end (right-mode) 'point state)
  state)



;;;;	
;;;;	
;;;;	Text Modify 
;;;;	
;;;;	
;;;;	

(defun nlist-insert (l i item)
  (if (zerop i)
      (cons item l)
      (progn
	(do ((list l (cdr list))
	     (j 0 (1+ j)))
	    ((= i (1+ j)) (setf (cdr list) (cons item (cdr list)))))
	l)))

(defun nlist-delete (l i)
  (cond
    ((< i 0) l)
    ((= i 0) (cdr l))
    ((> i 0)
     (do ((list l (cdr list))
	  (j 0 (1+ j)))
	 ((= i (1+ j)) (setf (cdr list) (cddr list))))
     l)))


(defun edit-text-delete-next (state)
  (edit-delete-text-at-label (left-right-of-edit-state state) 'point state)
  (point-rehash state)
  state)


(defun edit-text-delete-prev (state)
  (edit-delete-text-at-label  (toggle-mode
				    (left-right-of-edit-state state))
			       'point state)
  (point-rehash state)
  state)


(defun increment-istring (text)
  (let ((acc nil))

    (do ((ichars (reverse text) (cdr ichars)))
	((not (numeric-ichar-p (car ichars))))
      (push (car ichars) acc))
	   
    (unless (null acc)
      (append (subseq text 0 (- (length text) (length acc))) 
	      (istring (1+ (with-string-scanner ((implode-to-string acc)) (scan-num))))))))


(defun decrement-istring (text)
  (let ((acc nil))

    (do ((ichars (reverse text) (cdr ichars)))
	((not (numeric-ichar-p (car ichars))))
      (push (car ichars) acc))
	   
    (unless (null acc)
      (append (subseq text 0 (- (length text) (length acc))) 
			     (istring (let ((n (1- (with-string-scanner ((implode-to-string acc)) (scan-num)))))
					(if (<= n 0) 0 n)))))))
(defun edit-text-increment (state)
  (when (oed-modify-text-at-label #'increment-istring 'point state)
    (point-rehash state))
  state)

(defun edit-text-decrement (state)
  (when (oed-modify-text-at-label #'decrement-istring 'point state)
    (point-rehash state))
  state)


(defun edit-text-delete-segment (state)
  (edit-delete-text-segment-at-labels 'point 'mark state)
  (point-rehash state)
  state)

(defun edit-text-delete-entire (state)
  (edit-delete-text-segment-entire 'point state)
  (point-rehash state)
  state)

(defun edit-text-fix (state)
  (edit-fix-text 'point state)
  (point-rehash state)
  state)

(defun edit-text-insert-null (state)
  (oed-text-insert-null 'point state)
  (point-rehash state)
  state)

(defun edit-meta-parameter (meta state)
  (let ((point (point-of-edit-state state)))
    (when (dtree-leaf-p point)
      
      (oed-leaf-change-meta state point meta))

    ;;(setf -point point -meta meta -v state) (break "emp")
    (point-rehash state))

  state)

(defun edit-display-meta-parameter (state)
  (edit-meta-parameter 'display state))

(defun edit-abstraction-meta-parameter (state)
  (edit-meta-parameter 'abstraction state))

(defun edit-not-meta-parameter (state)
  (edit-meta-parameter nil state))


(defun edit-parameter-yank (state)
  (let ((point (point-of-edit-state state))
	(p (car (parameters-of-term  (term-stack-peek-r)))))
    (when (and p (dtree-leaf-p point))
      (oed-leaf-replace state point
			(tag-dtree (new-modified-dtree-leaf point p)
				   'point t))))

  state)


(defun edit-text-insert-next (state ich)
  ;;(break "etin")
  (edit-text-insert (left-right-of-edit-state state) 'point ich state)
  (point-rehash state)
  state)

(defun edit-text-insert-prev (state ich)
  ;;(break "etip")
  (edit-text-insert (toggle-mode (left-right-of-edit-state state)) 'point ich state)
  (point-rehash state)
  state)




;;;;	
;;;;	insert "abc" into "xx|yy" in left mode : 
;;;;	   xxabc|yy point at 5.
;;;;	
;;;;	insert "abc" into "xx|yy" at 2 in right mode : 
;;;;	   xx|abcyy point at 2.
;;;;	



;;; need to know left to right as we expect iistr
;;; to always be left to right.
;;; have caller reverse.


(defun edit-text-insert-istring-next (state iistr)
  (let ((mode (left-right-of-edit-state state)))
    ;; what if not text.
    (edit-text-insert-istring mode 'point
			      (if (right-mode-p mode)
				  (reverse iistr)
				  iistr)
			      state))
  (point-rehash state)
  state)

(defun edit-text-insert-istring-prev (state iistr)
  (let ((mode (left-right-of-edit-state state)))
    (edit-text-insert-istring mode 'point
			      (if (left-mode-p mode)
				  (reverse iistr)
				  iistr)
			      state))
  (point-rehash state)
  state)


(defun edit-text-yank (state)
  (let ((term (term-stack-peek-r)))
    (if (text-carrier-p term)
	(edit-text-insert-istring-next state (istring (string-of-text-carrier term)))
	state)))



;;;;
;;;; text segment commands
;;;;


(defun edit-char-push (state)
  (let ((ich (char-of-dtree-leaf state 'point)))
    (when ich
      (term-stack-push
       (itext-term (implode-to-string (list ich))))))
  
  state)
	

(defun edit-text-push (state)
  (let ((istr (segment-of-dtree-leaf 'point 'mark state)))
    (when istr
      (term-stack-push
       (itext-term (implode-to-string istr)))))
  
  state)

(defun edit-text-push-entire (state)
  (let ((istr (entire-segment-of-dtree-leaf 'point state)))
    (when istr
      (term-stack-push
       (itext-term (implode-to-string istr)))))
  
  state)

(defvar *mlid-delimiters*
  (standard-character-sbits
   (list idash idot icomma ispace inewline istringquote ibackquote iquote
	 iplus iescape isplat iampersand istar iquestion iequals
	 ibar icolon isemicolon 
	 ilparen irparen ilangle irangle ilsquare irsquare)))

(defun find-delimited-istring (istr pos delimiters &optional require-delimiters)
  (let ((begin nil)
	(end nil)
	(escapedp nil))

    (dotimeslist (i e istr)
		 (unless end
		   (if (and (not escapedp)
			    (test-standard-character-bit e delimiters))
		       (if (<= i pos)
			   (setf begin i)
			   (setf end i))
		       (setf escapedp
			     (if escapedp
				 nil
				 (eql iescape e))))))

    (if (and require-delimiters
	     (or (null begin) (null end)))
	(values 0 -1) ;; will be recognized by calllers as invalid.
	(values (if (null begin) 0 (1+ begin))
		(or end (length istr))))))

(defun strip-escapes (istr)
  (labels ((aux (escape-p istr)
	     (cond
	       ((null istr) nil)
	       (escape-p
		(cons (car istr) (aux (cdr istr))))
	       (t (if (eql iescape (car istr))
		      (aux t (cdr istr))
		      (cons (car istr) (aux nil (cdr istr))))))))

    (aux nil istr)))

(defun edit-text-push-id (state delimiters)
  (mlet* (((istr pos) (entire-segment-of-dtree-leaf 'point state))
	  ((begin end) (find-delimited-istring istr pos delimiters)))
	 ;;(setf -istr istr -pos pos -d delimiters) (break "etpi")

	 (eterm-stack-push
	  (if (> end begin)
	      (itext-term
	       (implode-to-string (strip-escapes (subseq istr begin end))))
	      (itext-term ""))))
  
  state)

(defun edit-text-push-and-delete-id (state delimiters)
  (mlet* (((istr pos) (entire-segment-of-dtree-leaf 'point state))
	  ((begin end) (find-delimited-istring istr pos delimiters t)))

	 ;;(setf -istr istr -pos pos -d delimiters -b begin -e end) (break "etpi")

	(eterm-stack-push
	 (cond
	   ((> end begin)
	    (edit-delete-text-segment-at 'point state begin end))
	    (itext-term
	     (implode-to-string (strip-escapes (subseq istr begin end))))
	   (t (itext-term ""))))

    state))
  

(defun edit-text-push-mlid (state)
  (edit-text-push-id state *mlid-delimiters*))

(defvar *quote-delimiters*
  (standard-character-sbits
   (list inewline istringquote ibackquote
	 ilparen irparen ilangle irangle ilsquare irsquare)))

(defun edit-text-push-quoted (state)
  (edit-text-push-id state *quote-delimiters*))

(defun edit-text-push-quoted-and-delete (state)
  (edit-text-push-and-delete-id state *quote-delimiters*))

;;;;	
;;;;	Suppress
;;;;	
;;;;	

(defun edit-suppress-dform (state)
  (let ((point (point-of-edit-state state)))
    (unless (dtree-leaf-p point)
      (suppress-dform-in-view (dform-of-dtree point) state)))

  state)


(defun edit-unsuppress-dform (state)
  (let ((point (point-of-edit-state state)))
    (unless (dtree-leaf-p point)
      (let ((tsig (term-sig-of-term (term-of-dtree point))))
	(let ((dform (find-first #'(lambda (dform)
				     (when (term-sig-of-term-p tsig (model-term-of-dform dform))
				       dform))
				 (suppressed-dforms-of-view state))))
	  (when dform
	    (unsuppress-dform-in-view dform state))))))

  state)




;;;;	
;;;;	Mouse
;;;;	
;;;;	

(defun edit-mouse-set-mark (state)
    (let ((point (row-col-to-point state
				 (mouse-row-of-edit-state state)
				 (mouse-col-of-edit-state state))))

    (edit-move-label state 'mark point)
    
    state))

(defun edit-mouse-set-term-mark (state)
  (let ((point (row-col-to-point (term-edit-state state)
				 (mouse-row-of-edit-state state)
				 (mouse-col-of-edit-state state)
				 nil))
	(state (text-edit-state state)))

    (when point
      (edit-move-label state 'mark point)
      (point-rehash state))))


(defun edit-mouse-set-point (state)
  (let ((point (row-col-to-point state
				 (mouse-row-of-edit-state state)
				 (mouse-col-of-edit-state state))))

    (edit-move-label state 'point point)
    (point-rehash state)
    
    state))

(defun edit-mouse-set-term-point (state)
  (let ((point (row-col-to-point (term-edit-state state)
				 (mouse-row-of-edit-state state)
				 (mouse-col-of-edit-state state)
				 nil)))

    ;; want to keep in text mode by default?
    (text-edit-state state)

    ;;(setf -point point) (break "emstp")
    (when point
      (edit-move-label state 'point point)
      (point-rehash state))
    state))


(defun edit-mouse-move-point (state)
  (let* ((point (point-of-edit-state state))
	 (mouse (row-col-to-point (term-edit-state state)
				  (mouse-row-of-edit-state state)
				  (mouse-col-of-edit-state state)))
	 (ppath (path-to-root point))
	 (mpath (path-to-root mouse)))
     
    (text-edit-state state)

    (cond
      ((eql point mouse) state)
      ((member mouse ppath)
       (edit-move-label state 'point (parent-of-dtree point))
       (point-rehash state))
      ((member point mpath)
       (do ((path mpath (cdr path)))
	   ((eql (cadr path) point)
	    (edit-move-label state 'point (car path))
	    (point-rehash state))))
      (t (do ((qpath (nreverse ppath) (cdr qpath))
	      (npath (nreverse mpath) (cdr npath)))
	     ((not (eql (car qpath) (car npath)))
	      (edit-move-label state 'point (car npath))
	      (point-rehash state)))))))

;;;;	
;;;;	Eval :
;;;;	
;;;;	  * evals term on stack.
;;;;	
;;;;	(m-x)eval
;;;;	  - pop stack and push result.
;;;;	  - fail    : 
;;;;	(m-x)eval-term-to-term
;;;;	  * evals term 
;;;;	(m-x)eval-apply-to-term
;;;;	(m-x)eval-apply-to-view
;;;;	  * applies term to an arg.

;;;;	
;;;;	kludge alert : uses eterm-stack : will cause problems for sfa.
;;;;	 ie we will need to mill his macros to use eterm-stack
;;;;	 or define alternate that work the way he expects.
;;;;	
;;;;	

(define-primitive |!ml_fail| () (src failure))

(defun edit-ml-eval (term &optional description)
  (with-handle-error-and-message (()
				  #'(lambda (m)
				      ;;(setf -m m) (break "eme")

				      (let ((fterm (ifail-term
						    (iml-fail-term term 
								   (message-to-term
								    (tag-message
								     *environment-path*
								     (tag-message
								      '(edit eval)
								      (cons m (flush-message '(edit eval))))))))))
					(view-show fterm)
					fterm
					)))
    ;;(setf -term term) (break "eme")
    (setf -emer
	  (without-dependencies
	   (if description
	       (orb-eval-by-description description (iexpression-term term) 'one)
	       (ml-eval 
		(setf -emes
		      (source-reduce-iml term
					 (cons 'ml (reduction-tags-of-environment
						    (current-environment)))))
		t))))))

(define-primitive |!ml_cmd_description| () (input output description))

;; any result is good.
(defun edit-eval (state)
  (setf (edit-state-evaled-p state) nil)
  (let ((result (with-oids ((let ((o (oid-of-view state)))
			      (when o (list o))))
		  (let ((term (term-from-dtree-tags-term (eterm-stack-peek-r))))
		    (if (iml-cmd-description-term-p term)
			(edit-ml-eval (iml-woargs-term nil t (input-of-iml-cmd-description-term term))
				      (description-of-iml-cmd-description-term term))
			(edit-ml-eval (iml-woargs-term nil t term)))))))

    ;; result may contain termilized failure.
    ;; if failure flash msg?
    (when (ifail-term-p result)
      (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					 (when o (list o)))
				      '(edit ml eval fail)
				      result)))
    
    ;;(setf a result b state) (break)
    ;; there is always a result as errors are converted to strings.
    (when result
      (setf (edit-state-evaled-p state) t)
      (eterm-stack-pop)
      (eterm-stack-push result)))
  state)


;; result must not be fail to be good.
(defun edit-eval-ok (state)
  (setf (edit-state-evaled-p state) nil)
  (let ((result (with-oids ((let ((o (oid-of-view state)))
					 (when o (list o))))
		  (edit-ml-eval (iml-woargs-term nil t (eterm-stack-peek-r))))))

    ;; result may contain termilized failure.

    ;; if failure flash msg?
    (when (ifail-term-p result)
      (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					 (when o (list o))) '(edit ml eval fail)
				      result))
      (setf result nil))
    
    (when result
      (setf (edit-state-evaled-p state) t)
      (eterm-stack-pop)
      (eterm-stack-push (if (ivalue-term-p result)
			   (result-of-iresult-term result)
			   result))))
  state)



;; result must be term to be good.
(defun edit-eval-to-term (state)
  (setf (edit-state-evaled-p state) nil)
  (let ((result (with-oids ((let ((o (oid-of-view state)))
					 (when o (list o))))
		  (edit-ml-eval (iml-woargs-term nil t (eterm-stack-peek-r))))))

    (cond
      ((ivalue-term-p result)
       (setf (edit-state-evaled-p state) t)
       (eterm-stack-pop)
       (eterm-stack-push (result-of-iresult-term result)))

      ((ifail-term-p result)
       (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					 (when o (list o)))
				       '(edit ml eval term fail)
				       result)))

      (t (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					   (when o (list o)))
					 '(edit ml eval term not)
					 result)))))
    state)

(defvar *term-to-term-prefix* (itext-term "\\l. ("))
(defvar *term-to-term-suffix* (itext-term " (hd l))"))

(defun construct-term-to-term (term)
  (itext-cons-term *term-to-term-prefix*
	      (itext-cons-term term
			  *term-to-term-suffix*)))

(defun edit-eval-apply-to-term-aux (state)
    
  (let ((result (with-oids ((let ((o (oid-of-view state)))
			      (when o (list o)))
			    )
		  (edit-ml-eval (iml-term nil t (setf -ctt (construct-term-to-term (eterm-stack-peek-r)))
				     (list (eterm-stack-peek-ahead-r)))))))

    ;;(setf -v state -r result) (break "eeatta")

    (cond
     
      ((ivalue-term-p result)
       (setf (edit-state-evaled-p state) t)
       ;;(break "atta")
       (eterm-stack-pop)
       (eterm-stack-pop)
       (eterm-stack-push (result-of-iresult-term result)))


      ((ifail-term-p result)
       (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					 (when o (list o)))
				       '(edit ml eval term fail)
				       result)))

      (t ;;(setf -result result -state state) (break "emea tn")
       (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					   (when o (list o)))
					 '(edit ml eval-apply term not)
					 result))))))

;; differs from eval-apply-to-term as pops func and arg from term stack prior to eval.
;; needed a version which left stack as is prior to call to allow top loop to access stack.
;; this probably makes more sense in general but need other version for compatability
(defun edit-eval-apply-aux (state)
    
  (let ((result (with-oids ((let ((o (oid-of-view state)))
			      (when o (list o)))
			    )
		  (let ((f (eterm-stack-peek-r))
			(a (eterm-stack-peek-ahead-r)))

		    (eterm-stack-pop)
		    (eterm-stack-pop)
		    ;;(setf -f f -a a)

		    (edit-ml-eval (iml-term nil t (setf -ctt (construct-term-to-term f))
					    (list a)))))))

    ;;(setf -r result) (break "eeaa")
    (cond
      ((ivalue-term-p result)
       (setf (edit-state-evaled-p state) t)
       ;;(break "atta")
       (eterm-stack-push (result-of-iresult-term result)))

      ((ifail-term-p result)
       (message-emit (oid-warn-message (let ((o (oid-of-view state)))
					 (when o (list o)))
				       '(edit ml eval term fail)
				       result)))

      (t (message-emit (oid-inform-message (let ((o (oid-of-view state)))
					     (when o (list o)))
					   '(edit ml eval)
					   result))))))


(defun edit-eval-apply (state)
  (setf (edit-state-evaled-p state) nil)

  (edit-eval-apply-aux state)
  state)



(defun edit-eval-apply-to-term (state)
  (setf (edit-state-evaled-p state) nil)

  (edit-eval-apply-to-term-aux state)
  state)


(defun edit-eval-apply-to-view-aux (state)

  ;;(break "eeatva")

  (let ((term (eterm-stack-peek)))
    (eterm-stack-pop)
    (with-handle-error-and-message (()
				    #'(lambda (m)
					;;(setf -m m) (break "eme")
					(view-show (message-to-term
						    (oid-warn-message
						     (let ((o (oid-of-view state))) (when o (list o)))
						     '(edit ml eval term fail)
						     (cons (tag-message *environment-path*
									(tag-message '(edit eval) (cons term m)))
							   (messages-flush)  ))))
					(ifail-term (itext-term "it bombed, window pop-up shows error" ))))

      (setf -eeatva term)
      (when (null term)
	(raise-error (error-message '(apply-to-view stack null))))

      (without-dependencies
       (funmlcall
	(ml-term
	 (source-reduce term
			(cons 'ml (reduction-tags-of-environment
				   (current-environment)))))
	state)
       (setf (edit-state-evaled-p state) t)
       ))))


;; applies token on stack to view. 
;; result is ignored, ie nothing pushed on stack.
;; if failure then evaled bit is not set.
;;
;; stack popped whether failure or not.
;; if top of stack not token then cmd fails (stack still popped).
;; 
;; similar to ml primitive except evaled bit is avail and ???
;;;;	
;;;;	
;;;;	TODO check type of functions to ensure view -> unit.
;;;;	
;;;;	

(defun edit-eval-apply-to-view (state)
  (setf (edit-state-evaled-p state) nil)

  (edit-eval-apply-to-view-aux state)
  state)


;;;;	
;;;;	Misc
;;;;	

(defun edit-expand (v)
  (let* ((point (point-of-edit-state v)))
    (if (dtree-leaf-p point)
	state
	(let* ((term (dtree-to-term point))
	       (abs (abstraction-of-term term))
	       (exp (when abs (expansion-of-abstraction abs))))
	  (when exp
	    (edit-paste-term 'point 
			     (expand-term-aux term abs exp)
			     v)
	    (point-rehash v 'point)))))
  v)

(defun edit-explode (v)
  (let ((point (point-of-edit-state v)))
    (unless (dtree-leaf-p point)
 
     ;;(oed-edit-replace v point (explode-term (dtree-to-term point)))
      (edit-paste-term 'point (explode-term (dtree-to-term point)) v)
      (point-rehash v 'point)
      ))
  v)


(defun edit-implode (v)

  (let ((point (point-of-edit-state v)))
    (unless (dtree-leaf-p point)

      ;;(oed-edit-replace v point (implode-term (dtree-to-term point)))
      (edit-paste-term 'point (implode-term (dtree-to-term point)) v)
      (point-rehash v 'point)
      ))

  v)


(defun iml-string-point-p (point)
  (edit-condition-test-p point 'ml_string))

(defun term-of-iml-string-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))
       
(defun edit-find-ml (point)
  (if (and (not (dtree-leaf-p point))
	   (iml-string-point-p point))
      point
      (if (null (parent-of-dtree point))
	  nil
	  (edit-find-ml (parent-of-dtree point)))))

(defun edit-jump-ml (state)
  (let ((point (edit-find-ml (point-of-edit-state state))))
    (when point
      (edit-move-label state 'point point)
      (point-rehash state))
    state))


(defun edit-wait-cursor (state)
  (edit-change-cursors t)
  state)

(defun edit-normal-cursor (state)
  (edit-change-cursors nil)
  state)


(defun edit-goto-term (state)
  (let ((cursor (cursor-of-edit-state state)))

    (cond
      ((point-cursor-p cursor)
       (let ((point (point-cursor-point cursor)))
	 (if (dtree-leaf-p point)
	     (edit-up state)
	     state)))
      
	;; let is necessary as term-edit-state text-edit-state
	;; have destructive side effects and text-edit-state  must
	;; be done last.
      (t
       ;; moves label:
       (row-col-to-point (term-edit-state state)
					(row-of-cursor cursor)
					(col-of-cursor cursor))
       (point-rehash (text-edit-state state))))))


(defun edit-prelayout (v)
  (let ((dtree (dtree-of-view v)))

    ;;(setf -a dtree -v v) (break "el")

    ;; want to commit text changes.
    (let ((ndtree (dtree-lift-text dtree)))
      (clear-dtree-dform-cache dtree)
      (setf -b ndtree)

      (set-view-dtree v ndtree))

    ;; lift text above is vestigial dlift?
    ;;(break "el ")

    (point-rehash v 'point)    
    v))

(defun edit-layout (v)
  (edit-prelayout v)
  (view-relayout v)
  v)

(defvar *oed-blink-interval* .25)

(defun edit-blink (v)

  (let ((visible (view-flag-cursor-visible-p v)))
    (view-flag-set-cursor-visible v (not visible))
    (view-flag-set-cursor-display-required v t)
    (view-flag-set-cursor-present-required v t)

    (view-present v)
    (xwin-flush)

    (sleep-aux *oed-blink-interval*)

    (view-flag-set-cursor-visible v visible)
    (view-flag-set-cursor-display-required v t)
    (view-flag-set-cursor-present-required v t)

    (view-present v)
    (xwin-flush)
    )
  v)


(defun edit-exit (v)
  (set-current-view nil)
  (quit-prl-loop)
  v)

(defvar *primitive-commands-alist* nil)

(setf *primitive-commands-alist*
      '(("mode"			.	"edit-mode")
	("left-mode"		.	"edit-left-mode")
	("right-mode"		.	"edit-right-mode")

	("insert-replace-mode"  .	"edit-insert-replace-mode")
	("insert-mode"		.	"edit-insert-mode")
	("replace-mode"		.	"edit-replace-mode")

	("goto-term"		.	"edit-goto-term")

	("library-token"	.	"edit-library-token")

	("wait-cursor"		.	"edit-wait-cursor")
	("normal-cursor"	.	"edit-normal-cursor")

	("mark"			.	"edit-mark")
	("swap-point-mark"	.	"edit-swap-point-mark")
	("clear-mark"		.	"edit-clear-mark")

	("mouse-point"		.	"edit-mouse-set-point")
	("mouse-point-term"	.	"edit-mouse-set-term-point")
	("mouse-mark"		.	"edit-mouse-set-mark")
	("mouse-mark-term"	.	"edit-mouse-set-term-mark")
	("mouse-move-point"	.	"edit-mouse-move-point")
	
	("up"			.	"edit-up")
	("down"			.	"edit-down")
	("down-left"		.	"edit-down-left")
	("down-right"		.	"edit-down-right")
	("down-term"		.	"edit-down-tree")

	("top"			.	"edit-top")

	("next-sibling-term"	.	"edit-next-sibling-tree")
	("prev-sibling-term"	.	"edit-prev-sibling-tree")
	("right-sibling-term"	.	"edit-right-sibling-tree")
	("left-sibling-term"	.	"edit-left-sibling-tree")
	("right-sibling"	.	"edit-right-sibling")
	("left-sibling"		.	"edit-left-sibling")

	("next-leaf"		.	"edit-walk-next-leaf")
	("prev-leaf"		.	"edit-walk-prev-leaf")
	("right-leaf"		.	"edit-walk-right-leaf")
	("left-leaf"		.	"edit-walk-left-leaf")
	("next"			.	"edit-walk-next")
	("prev"			.	"edit-walk-prev")

	("jump-next-open-slot"	.	"edit-walk-next-slot")
	("jump-prev-open-slot"	.	"edit-walk-prev-slot")
	("jump-right-open-slot"	.	"edit-walk-right-slot")
	("jump-left-open-slot"	.	"edit-walk-left-slot")
	("jump-down-next"	.	"edit-walk-down-next-slot")
	("jump-down-prev"	.	"edit-walk-down-prev-slot")

	("jump-ml"		.	"edit-jump-ml")

	;;("jump-next-conditional-slot" . "edit-jump-next-conditional-slot")
	;;("jump-prev-conditional-slot" . "edit-jump-prev-conditional-slot")

	("text-next"		.	"edit-next-text")
	("text-prev"		.	"edit-prev-text")	
	("text-last"		.	"edit-text-last")
	("text-first"		.	"edit-text-first")
	("text-last"		.	"edit-text-last")
	("text-first"		.	"edit-text-first")
	("text-left-word"	.	"edit-text-left-word")
	("text-right-word"	.	"edit-text-right-word")

	("text-join"		.	"edit-text-join")
	("text-split"		.	"edit-text-split")
	("text-push"		.	"edit-text-push")
	("char-push"		.	"edit-char-push")
	("text-push-entire"	.	"edit-text-push-entire")

	("oid-yank"		.	"edit-parameter-yank")	
	("parameter-yank"	.	"edit-parameter-yank")	

	("text-delete-next"	.	"edit-text-delete-next")
	("text-delete-prev"	.	"edit-text-delete-prev")	
	("text-delete"		.	"edit-text-delete-next")
	("text-rubout"		.	"edit-text-delete-prev")	
	("text-delete-segment"	.	"edit-text-delete-segment")
	("text-delete-entire"	.	"edit-text-delete-entire")
	("text-insert-null"	.	"edit-text-insert-null")
	("text-template"	.	"edit-display-meta-parameter")
	("text-parameter"	.	"edit-abstraction-meta-parameter")
	("text-value"		.	"edit-not-meta-parameter")
	("text-yank"		.	"edit-text-yank")	
	("text-increment"	.	"edit-text-increment")
	("text-decrement"	.	"edit-text-decrement")
	("text-fix"		.	"edit-text-fix")
	("term-text-fix"	.	"edit-term-text-fix")

	("undo"			.	"oed-edit-undo")
	("redo"			.	"oed-edit-redo")
	("redo-all"		.	"oed-edit-abort-undo")


	("delete"		.	"edit-term-delete")
	("yank"			.	"edit-term-yank")
	("yank-merge"		.	"edit-yank-merge")
	("yank-dtree-merge"	.	"edit-yank-dtree-merge")
	("yank-structure-merge"	.	"edit-yank-dtree-structure-merge")
	("prf-yank"		.	"edit-prf-yank")
	("prf-push"		.	"edit-prf-push")
	
	("expand"		.	"edit-expand")
	("explode"		.	"edit-explode")
	("implode"		.	"edit-implode")

	("push"			.	"edit-push")
	("push-clear"		.	"edit-push-clear")
	("stack-cut"		.	"edit-term-stack-cut")
	("pop"			.	"edit-term-pop")
	("rotate"		.	"edit-term-rotate")
	("reverse-rotate"	.	"edit-term-reverse-rotate")
	("unrotate"		.	"edit-term-unrotate")

	("grab" 		.	"edit-grab")
	("epush"		.	"edit-epush")
	("epush-clear"		.	"edit-epush-clear")
	("epop"			.	"edit-term-epop")
	("erotate"		.	"edit-term-erotate")
	("ereverse-rotate"	.	"edit-term-ereverse-rotate")
	("eyank"		.	"edit-term-eyank")
	("eparameter-yank"	.	"edit-parameter-eyank")	
	("eparameter-push"	.	"edit-parameter-epush")	
	("text-push-mlid"	.	"edit-text-push-mlid")
	("text-push-quoted"	.	"edit-text-push-quoted")
	("text-push-quoted-and-delete"	.	"edit-text-push-quoted-and-delete")

	("cmd-push"		.	"edit-cmd-push")
	("cmd-pop"		.	"edit-cmd-pop")
	("cmd-rotate"		.	"edit-cmd-rotate")
	("cmd-reverse-rotate"	.	"edit-cmd-reverse-rotate")
	("cmd-unrotate"		.	"edit-cmd-unrotate")
	("cmd-yank"		.	"edit-cmd-yank")
	("cmd-print"		.	"edit-cmd-print")

	("list-normalize"	.	"edit-ilist-normalize")
	("list-up"		.	"edit-ilist-up")
	("list-down"		.	"edit-ilist-down")
	("list-last"		.	"edit-ilist-last")
	("list-right"		.	"edit-ilist-right")
	("list-left"		.	"edit-ilist-left")
	("list-delete-right"	.	"edit-ilist-delete-right")
	("list-delete-left"	.	"edit-ilist-delete-left")
	("list-insert"		.	"edit-ilist-insert")
	("list-add"		.	"edit-ilist-add")
	("list-yank"		.	"edit-ilist-yank")
	("list-push"		.	"edit-ilist-push")
	("list-transpose"	.	"edit-ilist-transpose")
	("list-delete-segment"	.	"edit-ilist-delete-segment")
	
	("conjoin"		.	"edit-conjoin")

	("layout"		.	"edit-layout")
        ("prelayout"		.	"edit-prelayout")

	("suppress"		.	"edit-suppress-dform")
	("unsuppress"		.	"edit-unsuppress-dform")

	("eval"			.	"edit-eval")
	("eval-with-fail"	.	"edit-eval-ok")
 	("eval-term-to-term"	.	"edit-eval-to-term")
	("eval-apply"	.		"edit-eval-apply")
	("eval-apply-to-term"	.	"edit-eval-apply-to-term")

	("eval-apply-to-view"	.	"edit-eval-apply-to-view")

	("stack-eval"		.	"edit-eval")
	("stack-print"		.	"edit-stack-print")

 	("state-rotate"		. "edit-rotate-focus")
	("state-reverse-rotate"	. "edit-reverse-rotate-focus")
	("raise"		. "edit-raise")
	("lower"		. "edit-lower")
	("select-prev"		. "edit-swap-focus")
	("focus-select"		. "edit-focus")
	("select-focus"		. "edit-select-focus")
 	("state-rotate"		. "edit-rotate-focus")
	;; hmm rotate-focus may not do what sfa expects.
 	("region"		. "edit-rotate-focus")
	("state-reverse-rotate"	. "edit-reverse-rotate-focus")

	("focus1"		. "edit-focus1")
	("focus-rotate"		. "edit-rotate-focus")
	("focus-reverse-rotate"	. "edit-reverse-rotate-focus")
	("focus-swap"		. "edit-swap-focus")
	("warp"			. "edit-warp")


	("verify-modified"	. "edit-verify-modified")
	("dyn-refresh"		. "edit-touch-dynamic")
	("reset"		. "edit-reset")

	("status"		. "edit-status")
	("loop"			. "edit-loop")
	("fail"			. "edit-fail")
	("exit"			. "edit-exit")
	;;("cmd"			. "edit-exit")
	("blink"		. "edit-blink")

	("mark1"	        .	"edit-mark1")
	("mark2"		.	"edit-mark2")
	("mark3"		.	"edit-mark3")
	("mark4"		.	"edit-mark4")
	("mark5"		.	"edit-mark5")
	("mark6"		.	"edit-mark6")
	("mark7"		.	"edit-mark7")
	("mark8"		.	"edit-mark8")
	
	))

;;;;	
;;;;	
;;;;	
;;;;	Direct ML edit extensions.
;;;;	
;;;;	  - defines primitive edit command which calls ml closure.
;;;;	  - works on v5 view and not v4 states thus 
;;;;	    REQUIRES lazy point-rehash to preclude v4 primitives from
;;;;	    being confused about the point/mark.
;;;;
;;;;	define-ml-edit-primitive (tok closure)
;;;;	  - closure : view -> unit
;;;;	
;;;;	

(defvar *edit-primitive-commands-hash-table* (make-hash-table :size 300))

(defun define-edit-primitive (name closure)
  (let ((old (gethash name *edit-primitive-commands-hash-table*)))
    (setf (gethash name *edit-primitive-commands-hash-table*)
	  closure)
    (when (member (string name) *primitive-commands-alist* :test #'string= :key #'car)
      (message-emit (inform-message '(edit primitive lisp command redefined) name)))
    (when old
      (message-emit (inform-message '(edit primitive hash command redefined) name)))
    ))

(defun define-ml-edit-primitive (name closure)
  (define-edit-primitive name
      #'(lambda (state)
	  (funmlcall closure state) state)))

(defun primitive-commands-alist ()
  (append *primitive-commands-alist*
	  (let ((acc nil))
	    (maphash #'(lambda (k v)
			 (push (cons (string k) v) acc))
		     *edit-primitive-commands-hash-table*)
	    acc)))


(defunml (|define_ml_edit_command| (pair))
    ((tok |#| (view -> unit)) -> unit)
  (define-ml-edit-primitive (car pair) (cdr pair)))

(defunml (|oed_edit_rehash| (unit)  :declare ((declare (ignore unit))))
    (unit -> unit)
  (edit-rehash-macros))

(defun edit-macro-reset ()
  (setf *edit-text-character-macros* (make-array 255 :initial-element nil)
	*edit-expand-macros* nil)
  (edit-rehash-macros))
    
(defunml (|oed_macro_reset| (unit)  :declare ((declare (ignore unit))))
    (unit -> unit)
  (edit-macro-reset))


;;;;	
;;;;	In addition to previous commands we have following
;;;;	functions: 
;;;;	
;;;;	edit-text-insert-next (state ichar)
;;;;	edit-text-insert-prev (state ichar)
;;;;	
;;;;	edit-text-insert-istring-next (state istring)
;;;;	edit-text-insert-istring-prev (state istring)
;;;;	
;;;;	Also have 
;;;;	(m-x)insert_ichar_xx where xx is one of 0 - 256.
;;;;	
;;;;	TODO  : 
;;;;	require unicode entry method.
;;;;	\nnnn
;;;;	
;;;;	ATM, somewhat of a mystery as to how non-standard unicode chars are displayed by old layout.
;;;;	






















;;;
;;; the characters will be ichars or a cons with some modifier (ie control, meta)
;;; and an ichar.
;;;

(defconstant *edit-delete* (list ibackspace irubout))
(defconstant *edit-escape*  (cons (list ':control) (char->ichar #\\)))

(defvar *edit-abort* (cons (list ':control) (char->ichar #\G)))
(defvar *edit-token-delimiters* (list ispace))


(defun set-edit-abort (s)
  (let ((estring (string-to-estring s)))
    (when (cdr estring)
      (process-err (format-string "Edit Abort should be single char: \"~a\" is extra."
				  (estring-to-string (cdr estring)))))
    (setf *edit-abort* (car estring))))

(defun show-edit-abort ()
  (estring-to-string (list *edit-abort*)))


(defun edit-token-delimiter-p (ich)
  (member ich *edit-token-delimiters*))

(defun edit-abort-ichar-p (ich)
  (equal *edit-abort* ich))

(defun edit-delete-ichar-p (ich)
  (member ich *edit-delete* :test #'equal))


;;;;
;;;;  edit buffer
;;;;

(defvar *edit-buffer* (make-echo))

(defun edit-buffer-clear ()
  (echo-clear *edit-buffer*))

(defun edit-buffer-insert (ich)
  (echo-insert *edit-buffer* ich))

(defun edit-buffer-delete ()
  (echo-rubout *edit-buffer*))

(defun edit-buffer-p ()
  (not (echo-null-p *edit-buffer*)))

(defun edit-buffer () (buffer-of-echo *edit-buffer*))

(defun set-edit-buffer (buffer)
  (set-echo-buffer *edit-buffer* buffer))



(defun edit-backspace-ichar-p (ich)
  (equal ibackspace ich))

;; returns buffer.
(defun update-edit-buffer (ich)
  (cond
    ((null ich) (edit-buffer))

    ((edit-buffer-p)
     (cond
       ((edit-delete-ichar-p ich)
	(edit-buffer-delete))

       ((edit-backspace-ichar-p ich)
	(edit-buffer-delete))

       ((edit-abort-ichar-p ich)
	(edit-change-cursors nil)
	(edit-buffer-clear))

       (t (edit-buffer-insert ich))))

    (t (edit-buffer-insert ich))))

(defun edit-focus1 (state)
  (let ((v (view-of-window (window-of-focus-index 1 *edit-buffer*))))
    (or v
	state)))

(defun edit-select-focus-aux ()
  (setf -v (view-of-window (oed-current-focus)))
  ;;(break "esfa")
  -v
  )

(defun edit-select-focus (state)
  (declare (ignore state))

  (edit-select-focus-aux))

;; sets focus to window of view.
(defunml (|view_focus| (v))
    (view -> unit)

  (edit-focus v)
  nil)


;;;;
;;;;  read loop 
;;;;

(defvar gstate nil)

(defvar *edit-macro-dispatched* nil)

(defun edit-text-p (state)
  (unless (edit-buffer-p)
    (dtree-leaf-p (point-of-edit-state state))))


(defun show-edit-error (term)
  ;;(break "see")
  (funmlcall (ml-text "show_edit_error") term))

(defun receive-edit-char (ich)
 
  ;;(format t " ich ~a~%" ich) ;;(break "rec")
  (when ich
    (with-dummy-transaction
	;;(prof:with-profiling (:type :time :count t)
	;;)
	(let* ((win (xwin-event-window))
	       (state (view-of-window win ;;(oed-current-focus)
				      )))

	  (with-handle-error-and-message (()
					  #'(lambda (m)
					      ;;(setf -m m) (break "rec")
					      (edit-buffer-clear)
					      (unless (and (basic-message-p m)
							   (equal (tags-of-message m) '(error edit macro fail)))
						(show-edit-error (message-to-term
								  (oid-warn-message
								   (let ((o (oid-of-view state))) (when o (list o)))
								   '(edit execute fail)
								   (cons (tag-message *environment-path* m)
									 (messages-flush) )))))))
    
	    #|
	  (with-handle-error (('(edit execute))
			      (edit-buffer-clear)
			      )

		(handle-process-err #'(lambda (err-str)
				  (display-msg err-str t)
				  (edit-buffer-clear)
				  ;;(flush-msgs)
				  (setf a err-str) (break "rec")
				  state))
            |#
				  
	    ;; for debugging
      
	    (if (and (not (edit-buffer-p))
		     (printable-ichar-p ich)
		     (edit-text-p (view-of-window win)))
		(progn
		  (oed-focus-on win)
		  (edit-execute-macro (find-text-macro state ich) state))
		(let ((*edit-macro-dispatched* nil)
		      (*oed-new-focus-p* nil)
		      )


		  ;;(setf -a ich) (break "erc")
		  (update-edit-buffer ich);; may change focus.
		  (let ((state (view-of-window (oed-current-focus))))
				      
		    (when (null state)
		      (reset-current-view)
		      (return-from receive-edit-char))
		    (setf gstate state)

		    (mlet* (((state buffer)
			     ;;(break "erc2")
			     (edit-top-lex-and-ex (edit-buffer)
						  (edit-macros)
						  state)))
							       
			   ;;(setf -a ich) (break "erc3")
			   (when (and state *edit-macro-dispatched*)
			     (setf state 
				   (edit-asynch-macro *edit-cleanup-hook* state)))

			   (when (or (null buffer) *edit-macro-dispatched*)
			     (when *print-macro-debug* (terpri) (terpri))
			     (set-edit-buffer buffer))
					   
			   ;;(setf -b buffer) (break "3")
			   (oed-focus-on (window-of-view state t))
			   (set-current-view state)
			   nil
			   )))))

	  (when (messages-p)
	    (mapcar #'print-message (setf -mf (messages-flush)))))

      (without-dependencies (refresh-views))

      ;; (maybe-pending-flash new-state)
      nil
      )))


(defun refresh-views ()

  (eterm-stack-clear)

  (when (null (view-of-window (oed-current-focus)))
    ;;(break "rv")
    (focus-on-stack))
    
  (set-current-view (view-of-window (oed-current-focus)))
  ;;(when (null (current-view)) (break "rvn"))

  (let ((v (current-view)))
    (set-view-echo v *edit-buffer*)
    (unless v
      (let ((buf (edit-buffer)))
	(when buf
	  (message-emit (warn-message '(edit view not buffer notnot) (estring-to-string buf)))
	  (edit-buffer-clear))))

    ;; null view ok here.
    (focus-on-view v))
  ;; (break "rv2")
  (dolist (v *views*)
    (with-ignore
	(when (and (view-window-open-p v)
		   (or (view-flag-echo-display-required-p v)
		       (view-flag-layout-required-p v) ; t
		       (view-flag-present-required-p v) ; t
		       (view-flag-display-required-p v)
		       (view-flag-cursor-layout-required-p v) ; t
		       (view-flag-cursor-present-required-p v)
		       (view-flag-cursor-display-required-p v)
		       (view-flag-touched-p v)
		       ))

	  ;;(setf -v v) (break "ep")
	  (when (view-flag-touched-p v)
	    (or (maybe-dyneval-set v (view-flag-touched-p v))
		(view-touched v)
		))
	  (view-present v)))
    (view-update-history v))

   (xwin-flush))


(defun load-macros (fname &optional expand-prefix)
  (with-open-file (s fname :direction :input)
    (when (null s)
      (raise-error (error-message '(macro load file not) fname)))
    (let ((count (edit-macro-load s expand-prefix)))
      ;; no error if here.
      (message-emit (inform-message '(macro load done) fname count)))))


(defunml (|load_macros| (fname))
    (string -> unit)

  (unless (probe-file fname)
    (raise-error (error-message '(macro load file not) fname)))
  
  (load-macros fname)
  
  nil)



(defun non-irrelevant-ancestor (dtree)
  (cond
    ((or (null (parent-of-dtree dtree))
	 (not (dtree-flag-irrelevant-p dtree)))
     dtree)
    (t (non-irrelevant-ancestor (parent-of-dtree dtree)))))

(defun maybe-set-dtree-leaf-index (dtree index)
  (when (and index (dtree-leaf-p dtree))
    (set-dtree-leaf-index dtree 'point
	  (min index (length (istring-of-dtree-leaf dtree)))))
  dtree)

(defun layout-lookup (v &key ignore-offset character enter exit eol)
  (let ((d (dtree-of-view-c v)))

    (when (and nil ;; probably is an error but message is annoying.
	       (view-layout-stale-p v d))
      (setf -d d -v v) (break "ll")
      (point-rehash v)
      (raise-error (oid-error-message (let ((o (oid-of-view v)))
					 (when o (list o)))
				      '(layout lookup modified))))

    (let ((w (window-of-view v)))
    
      ;; do not want to refresh layout as we need to pick from old.
      (layout-visit (layout-of-dtree d)
		    (width-of-oed-window w)
		    (height-of-oed-window w)
		    (offset-of-edit-state v)
		    nil
		    :ignore-offset ignore-offset
		    :character character
		    :enter enter
		    :exit exit
		    :eol eol))))


(defun row-col-to-nodes (state row col &optional (nulls-p t))
  (let* ((offset (offset-of-edit-state state))
	 (row (if offset
		  (+ row offset)
		  (if (> row 0) row 1)))
	 (eot-node nil)
	 (nodes nil)
	 (null-nodes nil))

    ;;(format t "~a ~a" row offset)
    (layout-lookup state
		  :character
		  #'(lambda (r c ich node i)
		      ;;(setf a r b c d ich e node f i)
		      (cond
			((and (= r row) (= c col))
			 (if (dtree-leaf-p node)
			     (push (maybe-set-dtree-leaf-index node i) nodes)
			     (if (null ich)
				 (when nulls-p (push node null-nodes))
				 (push node nodes))))
			((and (null ich)
			      (dtree-leaf-p node)
			      (= r row) (< c col)) ; ie to the right of editable text.
			 (setf eot-node (maybe-set-dtree-leaf-index node i)))
			(t (setf eot-node nil))))

		  :eol
		  #'(lambda (r c node)
		      (declare (ignore r c node))
		      ;;(break)
		      (when eot-node
			(push eot-node nodes))))

    (mapcar #'(lambda (node)
		(if (dtree-flag-irrelevant-p node)
		    (non-irrelevant-ancestor node)
		    node))
	    (or (append null-nodes nodes)
		(when eot-node (list eot-node))
		(list (dtree-of-edit-state state))))))


(defun row-col-to-point (state row col &optional (nulls-p t))
  (let* ((dtrees (row-col-to-nodes state row col nulls-p)))

    ;;(setf a dtrees b state) (break "to")
    
    (let ((point    
	   (if (term-edit-state-p state)
	       (let ((term-dtrees (mapcan #'(lambda (dtree)
					      (when (not (dtree-leaf-p dtree))
						(list dtree)))
					  dtrees)))
		 (or (car term-dtrees)
		     (parent-of-dtree (car dtrees))))
	       (let* ((text-dtrees
		       (mapcan #'(lambda (dtree)
				   (when (dtree-leaf-p dtree)
				     (list dtree)))
			       dtrees))
		      (preferable-text-dtrees
		       (mapcan #'(lambda (dtree)
				   (when (or (null (istring-of-dtree-leaf dtree))
					     (< (index-of-dtree-leaf dtree 'point)
						(length (istring-of-dtree-leaf dtree))))
				     (list dtree)))
			       text-dtrees)))
		 (car (or preferable-text-dtrees
			  text-dtrees
			  dtrees))))))

      (let ((npoint (do ((p point (parent-of-dtree p)))
			((or (null (parent-of-dtree p))
			     ;; traverseable ?!
			     (not (dtree-flag-non-modifiable-p (parent-of-dtree p))))
			 p))))
	(when npoint
	  ;;(setf -npoint npoint -p point) (break "rctp")
	  (edit-move-label state 'point npoint))
	npoint
	))))






  
(defunml (|edit_layout| (view))
    (view -> unit)

  (edit-layout view)
  
  nil)
(defunml (|edit_push| (view))
    (view -> unit)
  (edit-push view))

(defunml (|term_stack_push| (term))
     (term -> unit)
  (term-stack-push term))
