
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

(defun set-dtree-ttree (dtree ttree)
  (setf (dtree-layout dtree) ttree))

(defun ttree-of-dtree (dtree w)
  (layout-of-dtree-c dtree w))


(defun dtree-elided-p (dtree) (dtree-flag-elided-p dtree))
(defun minw-of-dtree (dtree) (dtree-minw dtree))
(defun indent-of-push-format (f) (amt-of-push-format f))
;;(defun istring-of-text-format (f) (istring (string-of-text-format f)))


;; RLE ??? PERF
(defun minw-of-dform (d) (declare (ignore d)) 0)
(defun minw-of-child-format (d) (declare (ignore d)) nil)



;;;;	
;;;;	Dtree leaf
;;;;	

;; hopefully at some point layout will not blow up all the leafs.
(defun istring-of-dtree-leaf (leaf)
  (when (not (dtree-flag-instantiated-p leaf))
    (instantiate-dtree-leaf leaf))
  (dtree-leaf-istr leaf))

(defun empty-dtree-leaf-instantiation-p (leaf)
  (when (not (dtree-flag-instantiated-p leaf))
    (instantiate-dtree-leaf leaf))
  (dtree-flag-empty-instantiation-p leaf))

(defun meta-bit-of-dtree-leaf (leaf)
  (when (not (dtree-flag-instantiated-p leaf))
    (instantiate-dtree-leaf leaf))
  (dtree-leaf-meta leaf))

(defun dtree-leaf-set-meta-bit (m leaf)
  (when (not (dtree-flag-instantiated-p leaf))
    (instantiate-dtree-leaf leaf))
  (setf (dtree-leaf-meta leaf) m)
  )

(defun descriptor-of-dtree-leaf (leaf)
    (when (not (dtree-flag-instantiated-p leaf))
    (instantiate-dtree-leaf leaf))

  (or (dtree-leaf-descriptor leaf)
      (setf (dtree-leaf-descriptor leaf)
	    (let ((format (format-of-dtree leaf)))
	      (cons ilsquare (nconc (or (istring (when format
						   (descriptor-of-dform-parameter-child format)))
					;; note that this catches descriptor of "" as well as null format.
					(istring "slot"))
				    (list irsquare)))
		))))

;; strips [ and ]
(defun raw-descriptor-of-dtree-leaf (leaf)
  (cdr (butlast (descriptor-of-dtree-leaf leaf))))



;;;;
;;;;	dtree-leaf-to-parameter-parts : tok tok tok string marks
;;;;	dtree-leaf-to-parameter-parts : tok tok tok string marks
;;;;	
;;;;	instantiate-dtree-leaf (<dtree> <format{child}>)	: NULL
;;;;	  * adds text edit buffer to leaf.
;;;;	  * moves marks from parameter to dtree. could we be overwriting dtree marks?
;;;;	
;;;;	
;;;;	dtree-leaf-to-parameter (<dtree> <bool{include-tags?}>)	: <parameter>
;;;;	  * incorporates buffered text edits.
;;;;	  * incorporates marks of dtree into parameter including tags and labels and edit indices
;;;;	  * include-tags nil strips tags though technically it needn't.
;;;;	  
;;;;	parameter-value-to-dtree-string (<*> <tok>{typeid})	: <string{display}>
;;;;	
;;;;	
;;;;	

;;; parameter field contains real oid/time values.
;;; seems like marks should be filtered.
(defun dtree-leaf-to-parameter (leaf tags-p &optional meta-p meta-bit)

  (let ((lmarks (mark-values leaf))
	(parameter (parameter-of-dtree leaf)))

    ;; lmarks supercede parameter marks in union below.
   
    (when *dtree-flag-trace*
      ;;(break "dltp")
      (format t "dtree-leaf-to-parameter point? ~a~%" (dtree-labeled-p 'point leaf))
      (format t "dtree-leaf-to-parameter text-modified? ~a~%" (dtree-flag-text-modified-p leaf))
      )

    ;;(when (dtree-labeled-p 'point leaf) (setf -tags-p tags-p -leaf leaf) (break "dltpal"))
    ;;(when (dtree-flag-text-modified-p leaf) (setf -m (dtree-flag-non-modifiable-p leaf) -leaf leaf -n tags-p) (break "dltp"))

    (if (and (or (not (dtree-flag-text-modified-p leaf))
		 (dtree-flag-non-modifiable-p leaf))	; shouldn't have to check this, but safer.
	     (null meta-p)
	     )

	(let ((indices (indices-of-dtree-leaf leaf)))
	  #-nodebug
	  (dtree-flag-trace
	    (format-string "dtree-leaf-to-parameter dtree-indices ~a ~%" indices)
	    (format-string "dtree-leaf-to-parameter parameter-indices ~a ~%"
			   (mark-of-parameter-value (parameter-value (parameter-of-dtree leaf)) 'edit-indices)))
	  
	  ;;(when indices (setf -indices indices -lmarks lmarks -tags-p tags-p -leaf leaf) (break "dltp"))
	  ;;(format t "DLTP : ~a ~a ~a~%" indices (mark-of-parameter-value (parameter-value (parameter-of-dtree leaf)) 'edit-indices) (parameter-of-dtree leaf))

	  ;;(setf -tags-p tags-p -lmarks lmarks -indices indices -parameter parameter) (break "yo")
	  (if (and tags-p lmarks)
	      (parameter-marks-union parameter
				     (if indices
					 (acons 'edit-indices
						indices
						(remove 'edit-indices lmarks :key #'car))
					 lmarks))
	      (if tags-p
		  parameter
		  ;; if null tags-p remove tags from parm although not strictly required.
		  (set-parameter-tags-and-labels parameter nil))))

	(progn
	  ;;(when (oid-parameter-p parameter) (break  "wtf"))
	  (setf -np
		(make-edit-parameter (when (dtree-flag-slot-p leaf) 'slot)
				     (if meta-p
					 meta-bit
					 (meta-bit-of-dtree-leaf leaf))
				     (type-id-of-parameter parameter)
				     (implode-to-string (istring-of-dtree-leaf leaf)) ; nfg if non-modifiable (eg oid,time).
				     (when tags-p lmarks)))
	  ;;(break "dltpnp")
	  -np))
    ))



;; assumes real value
(defun parameter-value-to-dtree-string (value typeid)
  
  (if (eql '|oid| typeid)
      (or (string (name-property-of-ostate value)) "ObId")
      (if (eql '|time| typeid)
	  (datetime-string value)
	  (raise-error (error-message (list 'oed 'dtree 'layout 'constant 'unexpected-type typeid))))))


(defun dtree-leaf-clean-text (dtree)

  (setf (dtree-leaf-istr dtree) nil
	(dtree-leaf-descriptor dtree) nil
	(dtree-leaf-meta dtree) nil)

  (dtree-flag-set-slot dtree nil)

  ;; technically above resets shouldn't be needed since fields should not be accessed
  ;; unless instantiated and instantiation should set.

  ;; probably shouldn't need to be done. but if there is a layout
  ;; it certainly is not up to date. would expect caller to clear layout
  ;; and set layout required as desired.
  (when *dtree-flag-trace*
    (format t "*dtree-leaf-clean-text layout-required -> t, text-modified -> nil~%"))

  (dtree-path-layout-required dtree)  

  (dtree-flag-set-instantiated dtree nil)
  (dtree-flag-set-text-modified dtree nil)
  nil)

;; could use dtree-flag-text-modified-p to guide/limit traversal.
(defun dtree-clean-text (dtree)

  (when (dtree-flag-text-modified-p dtree)
    (when *dtree-flag-trace*
      (format t "*dtree-clean-text text-modified -> nil~%"))
    (dtree-flag-set-text-modified dtree nil)
    
    (if (dtree-leaf-p dtree)
	(dtree-leaf-clean-text dtree)
	(map-dtree-children #'dtree-clean-text t t dtree))
    nil))


;;;;	
;;;;	dtree-sharing : 
;;;;	
;;;;	  - memoized
;;;;	  - history
;;;;	
;;;;	text-mod - initial text mod of leaf should replace leaf in tree,
;;;;	subsequent text mods then are destructive, lift is destructive but
;;;;	then next text-mod would be constructive. refresh after constructive update
;;;;	updates history but then destructive updates implicitly modify historied tree.
;;;;	
;;;;	lifting - any constructive update causes destructive lift, thus initial text mod of leaf
;;;;	lifts prior mod. History checkpoints last constructive update at view refresh.
;;;;	
;;;;	
;;;;	memos should be local to view, so as to preclude sharing amongst views.
;;;;	so as to prevent concurrent access of dtree-layout.
;;;;	
;;;;	
;;;;	orphan - just steal it, ie destructively update parent. If former use restored then
;;;;	 it must steal back.
;;;;	
;;;;	
;; orphan : parent pointer does not point to parent.
;;  lazyize parent and non-destructively replace in tree.
(defun dtree-shared-p (dtree parent) (not (eql parent (parent-of-dtree dtree))))

;; clean all labels from descendents of shared but then lose the ones we desired.
;; or accumulate labels and note if from shared or not and lose duplicates in shared.
(defun dtree-clean (dtree)

  ;;(setf -dt dtree) (break "dc")

  (let ((unshared nil)
	(shared nil))

    (labels ((visit (dtree parent sharedp)

	       (let ((shared-p (or (when (dtree-shared-p dtree parent)
				     (set-dtree-parent dtree parent)
				     t)
				   sharedp)))

		 (dolist (l (labels-of-dtree dtree))
		   (if shared-p
		       (push (cons l dtree) shared)
		       (push l unshared)))

		 (unless (dtree-leaf-p dtree)
		   (map-dtree-children #'(lambda (child) (visit child dtree shared-p))
				       nil t
				       dtree)))))
      
      (visit dtree nil nil)

      (dolist (l shared)
	(if (member (car l) shared)
	    (let ((d (cdr l)))
	      (dtree-path-layout-modified d 'tag)
	      (untag-dtree d (car l) t))
	    (push (car l) shared)))
      
      (values))))



(defvar *variable-istring-to-istringf* nil)

(defun variable-latex-munge (istr)
  (let ((pos (position iunder istr :from-end t)))
    (if pos
	(let ((pref (subseq istr 0 pos))
	      (suff (subseq istr (1+ pos))))
	  (let ((l (+ (length pref) (length suff))))
	    (cons (nconc pref
			 (istring "\\ff24\\ff5f\\ff7b")
			 suff
			 (istring "\\ff7d\\ff24"))
		  l)))
	(cons istr (length istr)))))

(defunml (|with_latex_variable_strings| (f) :declare ((declare (ignore unit))))
    ((unit -> *) -> *)

  (let ((*variable-istring-to-istringf* #'variable-latex-munge))
    (funmlcall f nil)))

(defun length-of-dtree-leaf (l) (dtree-leaf-length l))

(defun instantiate-dtree-leaf (dtree)

  ;;(setf -idtree dtree) (break "idl")

  (let ((parameter (parameter-of-dtree dtree)))
    ;; (edit-type meta-type type-id . string)
    (let* ((parts (edit-parameter-parts parameter))
	   (istr (istring (string-of-parameter-parts parts)))
	   (edit-type (edit-type-of-parameter-parts parts))
	   (meta-type (meta-type-of-parameter-parts parts))
	   (type-id (type-id-of-parameter-parts parts))
	   )

      (dtree-flag-set-length dtree nil)
      ;;(format t "idl ~a ~a ~a ~a~%" edit-type meta-type type-id (implode-to-string istr))
      (when (and (null edit-type)
		 (null meta-type)
		 *variable-istring-to-istringf*
		 (eql '|variable| type-id))

	;;(format t "idlf~%")
	(let ((sl (funcall *variable-istring-to-istringf* istr)))
	  (setf istr (car sl))
	  (setf (dtree-leaf-length dtree) (cdr sl))
	  ;;(format t "dtree l ~a~%" (cdr sl))
	  (dtree-flag-set-length dtree t)
	  (dtree-flag-set-non-modifiable dtree t)))
	

      (when (and (null edit-type)
		 (null meta-type)
		 (or (eql '|oid| type-id)
		     (eql '|time| type-id)))

	(setf istr (istring (parameter-value-to-dtree-string (value-of-parameter parameter)
							     type-id)))
	(dtree-flag-set-non-modifiable dtree t))

      (setf (dtree-leaf-istr dtree) istr
	    (dtree-leaf-descriptor dtree) nil ; set if requested.
	    (dtree-leaf-meta dtree) meta-type)

      ;;(setf -nt dtree) (break "idl")
      #-nodebug
      (dtree-flag-trace
       (format-string "instantiate-dtree-leaf !TEXT-EDIT ? ~a, POINT ? ~a~%"
		      (dtree-labeled-p '!TEXT-EDIT dtree)
		      (dtree-labeled-p 'POINT dtree)
		      ))
      
      (when (dtree-labeled-p '!TEXT-EDIT dtree)
	(if (dtree-labeled-p 'POINT dtree)
	    (dtree-path-text-modified dtree)
	    (progn (untag-dtree dtree '!text-edit t)
		   ;; could probably get by without this:
		   ;; need history update (caused by remove !text-edit label
		   (dtree-path-layout-modified dtree 'tag)
		   )))
      
      (dtree-flag-set-slot dtree (eql 'slot edit-type))
      (dtree-flag-set-instantiated dtree t)

      (when *dtree-flag-trace*
	(format t "instantiate-dtree-leaf path layout-required -> t~%"))
      (dtree-path-layout-required dtree)

      ;; maybe find parent edit indices. somewhat kludgey.
      ;; prefer term as that is not default method, thus someone went to some trouble to get them there.
      (let ((indices (or (let ((parent (parent-of-dtree dtree)))
			   (when parent
			     (mark (term-of-dtree parent) 'edit-indices)))
			 (mark-of-parameter-value (value-of-parameter-n parameter) 'edit-indices))))
	(dolist (ind indices)
	  (set-dtree-leaf-index dtree (car ind) (cdr ind))))

      ;; done at instantiation (make-dtree-leaf). Could be overwriting tags in dtree that
      ;;  not in parameter. ie make then at label then layout would lose label.
      ;;(marks dtree (marks-of-parameter-value (value-of-parameter-n parameter)))

      (values))))


;; returns new dtree
(defun dtree-leaf-set-parameter (dtree parameter)
  (new-modified-dtree-leaf dtree parameter))







(defvar layout-trace nil)

(defmacro with-ttree-update (item &body body)
  (let ((l (gensym))
	(c (gensym))
 	(litem (gensym)))
    `(let ((,litem ,item))
      (when layout-trace
	(format t "ttree-update: ~a~%" ,litem))
      (mlet* (((,l ,c) (progn ,@body)))
       (when (eql 0 ,l) (break "wtu"))
       (values
	(if ,litem
	    (cons ,litem ,l)
	    ,l)
	,c)))))


(defmacro with-ttree-updates (list &body body)
  (let ((l (gensym))
	(c (gensym))
 	(llist (gensym)))
    `(let ((,llist ,list))
      (when layout-trace
	(format t "ttree-update: ~a~%" ,llist))
      (mlet* (((,l ,c) (progn ,@body)))
       (when (eql 0 ,l) (break "wtus"))
       (values
	(if ,llist
	    (nconc ,llist ,l)
	    ,l)
	,c)))))


(defmacro with-fail (f &rest b)
  `(handle-error 'layout #'(lambda (w)
			     (throw 'layout (funcall ,f w)))
    ,@b))

;;(macroexpand '(with-try test a b))
;;(block #:g205664
;;  (funcall (function (lambda (val)
;;	     (if (funcall test val)
;;		 (progn b)
;;		 (throw (quote layout) val))))
;;	   (catch (quote layout)
;;	     (return-from #:g205664 (progn a)))))

;; tries b if value caught satisfies test, or there is no test.
;; otherwise continues throw. if there is no second clause, b,
;; the execution result is nil.
(defmacro with-try (test a &rest b)
  (if test
      `(handle-error 'layout
	(function (lambda (val)
	  (if (funcall ,test val)
	      (progn ,@b)
	      (throw 'layout val))))
	,a)
      `(handle-error 'layout
 	(function (lambda (val)
	  (declare (ignore val))
	  (progn ,@b)))
 	,a)))


;;;;
;;;;  ttree
;;;;

(eval-when (compile)
  (proclaim '(inline
	      text-ich-p text-push text-push-p
	      indent-of-text-push text-pop text-pop-p
	      indent-of-text-pop text-newline text-newline-p
	      text-ttree-p)))

;; ttree syntax
;;;
;;; ttree :: dtree . format list |
;;; format :: 'space | 'newline | string | i (+ -> push/- -> pop) | ttree

(defun text-tree-p (x)
  ;; only meaningful in istring of text-tree
  (and (consp x) (not (text-istring-p x))))
(defun text-tree (dtree formats) (cons dtree formats))
(defun dtree-of-text-tree (ttree) (car ttree))
(defun formats-of-text-tree (ttree) (cdr ttree))

(defun text-istring-p (x) (or (null x) (and (consp x) (integerp (car x)))))
(defun text-istring (x) x)
(defun istring-of-text-istring (x) x)

(defun text-space () 'space)
(defun text-space-p (x) (eql 'space x)) 

(defun text-push (x) x)
(defun text-pop (i) (- i))
(defun text-indent-p (x) (integerp x))
(defun indent-of-text-indent (x) x)

(defun text-newline () 'newline)
(defun text-newline-p (x) (eql 'newline x))



(defvar *skinny-height* 5)
(defvar *skinny-width* 8)

(defun infinite-width-p (w) (eql t w))

(defun set-dtree-minw (dtree w)
  (when (and dtree
	     (not (infinite-width-p w))
	     (or (null (minw-of-dtree dtree))
		 (< (minw-of-dtree dtree) w)))
    (setf (dtree-minw dtree) w)
    ;;(set-dtree-minw (parent-of-dtree dtree) w)
    ))

(defun skinny-w-p (w) (and w (not (infinite-width-p w)) (< w *skinny-width*)))
(defun skinny-l-p (l) (> l *skinny-height*))

(defun dtree-update-env (env key value &rest rest)
  ;;(when (eql :width key)
  ;;(when (= value 0) (break))
  ;;(format t "~a ~a~%" key value))
  (acons key value (if (null rest)
		       env
		       (apply #'dtree-update-env env rest))))

(defun dtree-env-lookup (env key)
  (cdr (assoc key env)))

(defun ncompute-dtree-minima (dtree) (declare (ignore dtree)) nil)




;;;;
;;;;  dtree layout
;;;;


(eval-when (compile)
  (proclaim '(inline
	      ndtree-init-cenv ndtree-init-renv
	    
	      ndtree-last-ich ndtree-col ndtree-line ndtree-softs ndtree-breaks ndtree-dtrees
	    
	      ndtree-indents ndtree-width ndtree-dtree
	      ndtree-break-control ndtree-break-control-mode ndtree-break-control-soft
	      ndtree-indent-amt ndtree-indent-renv ndtree-indent-cenv
	      ndtree-init-indent ndtree-push-indent ndtree-pop-indent

	      ndtree-width-fail-p ndtree-fail ndtree-fail-elide ndtree-fail-r
	      target target-dtree target-softs

	      ndtree-depth ndtree-new-depth ndtree-cleanup-elide-depth

	      ndtree-new-breaks ndtree-break-line ndtree-init-break-control
	      ndtree-push-break-control ndtree-pop-break-control
	      )))


;;; environment

(defun ndtree-init-cenv ()
  (dtree-update-env nil
		    :col 0
		    :line 1
		    :last-ich inewline
		    :skinnyl 0))

(defun ndtree-init-renv (break-control width)
  (dtree-update-env nil
		    :break-control (list (cons break-control nil))
		    :width (if (null width) t width)))
	    


;;; continuation enviroment accessors.
(defun ndtree-last-ich (cenv) (dtree-env-lookup cenv :last-ich))
(defun ndtree-col (cenv) (dtree-env-lookup cenv :col))
(defun ndtree-line (cenv) (dtree-env-lookup cenv :line))
(defun ndtree-softs (cenv) (dtree-env-lookup cenv :softs))
(defun ndtree-breaks (cenv) (dtree-env-lookup cenv :breaks))
(defun ndtree-dtrees (cenv) (dtree-env-lookup cenv :dtrees))

(defun ndtree-soft (cenv renv)
  (dtree-update-env cenv
		    :wrap nil
		    :softs (cons (cons renv cenv)
				 (ndtree-softs cenv))))
(defun ndtree-elide-depth (cenv)
  (dtree-env-lookup cenv :depth))

(defun set-ndtree-elide-depth (cenv depth)
  (dtree-update-env cenv :depth depth))

(defun ndtree-update-col (cenv col)
  ;;(format t "col ~a~%" col)
  (dtree-update-env cenv :col col))
  

;;;;
;;;;  Wrapping
;;;;
;;;;    Wrap
;;;;    WrapThenBroke
;;;;    WrapThenSoft

(defun ndtree-wrapped-p (cenv) (dtree-env-lookup cenv :wrap))
(defun ndtree-wrap-cenv (cenv) (dtree-update-env cenv :wrap t))


;;;
;;; recursion environment accessors.
;;;

(defun ndtree-parent-parens (renv) (dtree-env-lookup renv :parens))

(defun ndtree-indents (renv) (dtree-env-lookup renv :indents))
(defun ndtree-width (renv) (dtree-env-lookup renv :width))
(defun ndtree-dtree (renv) (dtree-env-lookup renv :dtree))
(defun ndtree-break-control (renv) (dtree-env-lookup renv :break-control))
(defun ndtree-break-control-mode (renv) (caar (dtree-env-lookup renv :break-control)))
(defun ndtree-break-control-soft (renv) (cdar (dtree-env-lookup renv :break-control)))

(defun ndtree-depth (renv) (dtree-env-lookup renv :depth))
(defun ndtree-new-depth (dtree renv cenv)
  (let ((elide-depth (ndtree-elide-depth cenv))
	(depth (ndtree-depth renv))
	(parent (parent-of-dtree dtree)))
    ;;(setf a elide-depth b parent) (break)
    (if (and (eql parent (cdr elide-depth)) (car elide-depth))
	(dtree-update-env renv :depth (car elide-depth))
	(if (and depth
		 parent
		 (not (iterate-match-p (model-term-of-dform (dform-of-dtree-c dtree t))
				       (model-term-of-dform (dform-of-dtree parent)))
		  ;;(equal (term-sig-of-dform (dform-of-dtree dtree)) (term-sig-of-dform (dform-of-dtree parent)))
		  ))
	    (dtree-update-env renv :depth (1- depth))
	    renv))))

;; returns cenv
(defun ndtree-cleanup-elide-depth (cenv renv)
  (let ((elide-depth (ndtree-elide-depth cenv))
	(dtree (ndtree-dtree renv)))
    (if (and elide-depth (eql dtree (cdr elide-depth)))
	(set-ndtree-elide-depth cenv nil)
	cenv)))

(defun ndtree-depth-elide-p (renv)
  ;;(format t "depth-elide ~a~%" (ndtree-depth renv))
  (let ((depth (ndtree-depth renv)))
    (and depth (<= depth 0))))


;; indents
(defun ndtree-indent-amt (in) (car in))
(defun ndtree-indent-renv (in) (cadr in))
(defun ndtree-indent-cenv (in) (caddr in))

(defun ndtree-init-indent (renv cenv)
  (declare (ignore cenv))
  (let ((indents (ndtree-indents renv)))
    (dtree-update-env renv
		      :indents
		      (cons nil indents))))

;; 
(defun ndtree-push-indent (amt renv cenv)
  (let* ((indents (ndtree-indents renv))
	 (w (ndtree-width renv))
	 (width (if (infinite-width-p w)
		    w
		    (- w amt))))
    (dtree-update-env renv
		      :width width
		      :indents (cons (cons (list amt renv cenv) (car indents))
				     indents))))
			    
(defun ndtree-pop-indent (renv cenv) 
  (declare (ignore cenv))
  (let ((indents (ndtree-indents renv)))
    (dtree-update-env renv
		      :indents (cons (cdr (car indents))
				     (cdr indents)))))


;;;
;;; search
;;;

;; find soft break prior to an indent which gains the
;; most if broken.
(defun max-target-search (indents indents-history max-gain)
  (cond
    ((and (null indents) (null indents-history))
     max-gain)
    ((null indents)
     (max-target-search (car indents-history)
			(cdr indents-history)
			max-gain))
    (t 
     (let* ((cenv (ndtree-indent-cenv (car indents)))
	    (soft (car (ndtree-softs cenv)))
	    (s-cenv (when soft (cdr soft))))
       (if (and soft
		(= (ndtree-line cenv) (ndtree-line s-cenv))
		(progn (unless (and (integerp (ndtree-col s-cenv)) (integerp  (car max-gain)))
			 (process-err (format-string "max-target-search ~a ~a" (ndtree-col s-cenv) (car max-gain)))) t)
		(>= (ndtree-col s-cenv) (car max-gain)))
	   (max-target-search (cdr indents) indents-history (cons (ndtree-col s-cenv) soft))
	   (max-target-search (cdr indents) indents-history max-gain))))))


(defun max-target-search-breaks (breaks max-gain)
  (if (null breaks)
      max-gain
      (max-target-search-breaks (cdr breaks)
				(max-target-search nil
						   (ndtree-indents (car breaks))
						   max-gain))))

	   
(defun target-break-control (bcon)
  (cond
    ((null bcon) nil)
    ((cdar bcon) (cdar bcon))
    (t (target-break-control (cdr bcon)))))
       
(defun target-line (softs line)
  (if (null softs)
      nil
      (if (not (= line (ndtree-line (cdar softs))))
	  nil
	  (let ((max (target-line (cdr softs) line)))
	    (if (null max)
		(unless (zerop (ndtree-col (cdar softs)))
		  (car softs))
		(if (> (ndtree-col (cdr max)) (ndtree-col (cdar softs)))
		    max
		    (car softs)))))))

(defun target-softs (cenv renv min)
  (declare (ignore renv))
  ;;(let* ((softs (ndtree-softs cenv))
  ;;(s-cenv (cdar softs))
  ;;(s-renv (caar softs))))
  ;; set minw if no possible break on current line.
  ;;(when (and s-cenv (not (= (ndtree-line cenv) (ndtree-line s-cenv))))
  ;;      (set-dtree-minw (ndtree-dtree s-renv)
  ;;(+ (ndtree-width s-renv) (- (ndtree-col cenv) (ndtree-width renv)))))
  (let ((soft (cdr (max-target-search-breaks (ndtree-breaks cenv) (cons 0 nil)))))
    (when (and soft (> (ndtree-width (car soft)) min))
      soft)))
	 
(defun target-dtree (renv cenv)
  ;;(setf a renv b cenv c (car (ndtree-dtrees cenv))) (break)
  (do ((renvs (ndtree-dtrees cenv) (cdr renvs))
       (w (ndtree-width renv)))
      ((or (null renvs)
	   (let ((dtree (ndtree-dtree (car renvs))))
	     (and (not (dtree-leaf-p dtree))
		  (not (dtree-elided-p dtree))
		  (progn (unless (and (integerp w) (integerp (ndtree-width (car renvs))))
			   (process-err (format-string "target-dtree ~a ~a" w  (ndtree-width (car renvs))))) t)
		  (>= (ndtree-width (car renvs)) w)
		  )))
       (if (null renvs)
	   nil
	   (ndtree-dtree (car renvs))))))

;; choose soft in history to backtrack to.
;; may need to look at mode when targeting.
(defun target (renv cenv min)
  (or (ndtree-elide-depth cenv)
      (target-break-control (ndtree-break-control renv))
      (target-line (ndtree-softs cenv) (ndtree-line cenv))
      (target-softs cenv renv min)
      (target-dtree renv cenv)))




;;;
;;; fail
;;;

(defun ndtree-width-fail-p (w a)
  (and (not (infinite-width-p w))
       a w
       (< w a)))

;; find term on line or find term on next longer line.
(defun ndtree-fail-elide (dtree renv cenv)
  (throw 'layout (or (ndtree-elide-depth cenv)
		     (target-dtree renv cenv)
		     (parent-of-dtree dtree)
		     dtree)))

(defun ndtree-fail (renv cenv)
  ;;(when ndtree-fail-break (break "ndtree-fail"))
  ;;(format t "ndtree-fail.~%")
  ;; (format t "ndtree-col: ~a. width ~a. line: ~a. term ~a~%" 	  
  ;;(ndtree-col cenv) (ndtree-width renv) (ndtree-line cenv)
  ;;(when (and (ndtree-dtree cenv) (not (dtree-leaf-p (ndtree-dtree cenv))))
  ;;(id-of-term (dtree-term (ndtree-dtree cenv)))))

  ;;(setf -renv renv -cenv cenv) (break "ndtree-fail")
  (throw 'layout (target renv cenv (or (minw-of-dtree (ndtree-dtree renv)) 0))))

(defun ndtree-fail-r (dtree renv cenv)
  (let ((minw (max (minw-of-dform (dform-of-dtree-c dtree t)) (or (minw-of-dtree dtree) 0)))
	;;(w (ndtree-width renv))
	)
    (when (ndtree-width-fail-p (ndtree-width renv) (+ minw (ndtree-col cenv)))
      ;;(setf -renv renv -cenv cenv) (break "ndtree-fail-width")

      (ndtree-fail renv cenv))))
       

;;;
;;; break control
;;;

(defun ndtree-new-breaks (renv cenv)
  (if (infinite-width-p (ndtree-width renv))
      (cons renv nil)
      (let* ((w (ndtree-width renv))
	     (breaks (do ((b (ndtree-breaks cenv) (cdr b)))
			 ((or (null b)
			      (<= w (ndtree-width (car b))))
			  b))))
	(cond
	  ((null breaks)
	   (cons renv nil))
	  ((= w (ndtree-width (car breaks)))
	   breaks)
	  ((< w (ndtree-width (car breaks)))
	   (cons  renv breaks))
	  (t (break))))))

;; cenv
(defun ndtree-break-line (renv cenv)
  (let* ((soft (car (ndtree-softs cenv)))
	 (s-cenv (cdr soft))
	 ;;(s-renv (car soft))
	 (rwidth (ndtree-width renv))
	 (cwidth (ndtree-width cenv)))
    (when (and s-cenv
	       (not (= (ndtree-line cenv) (ndtree-line s-cenv)))
	       (not (infinite-width-p rwidth))
	       (not (infinite-width-p cwidth)))
      ;;(set-dtree-minw (ndtree-dtree s-renv) (ndtree-width s-renv))
      ;;(setf a (ndtree-dtree renv) b (ndtree-dtree renv) c (ndtree-width renv)
      ;;d (ndtree-col cenv) e (- b (- c d)))
      ;;(break)
      (set-dtree-minw (ndtree-dtree renv)
		      (- (ndtree-width cenv)
			 (- (ndtree-width renv) (ndtree-col cenv))))))
	 
  ;;(format t "line:~a" (dtree-env-lookup cenv :line))
  (dtree-update-env cenv
		    ;; :wrap nil
		    :depth nil		; for quick lookup.
		    :breaks (ndtree-new-breaks renv cenv)
		    :width (ndtree-width renv)
		    :indents (ndtree-indents renv)
		    :line (1+ (dtree-env-lookup cenv :line))
		    :col 0
		    :last-ich inewline
		    :skinnyl (if (and (not (ndtree-wrapped-p cenv)) (skinny-w-p (dtree-env-lookup cenv :width)))
 				 (let ((l (dtree-env-lookup cenv :skinnyl)))
				   (if (skinny-l-p (1+ l))
				       (ndtree-fail renv cenv)
				       (1+ l)))
				 0)))


;; renv
(defun ndtree-init-break-control (renv cenv)
  (declare (ignore cenv))
  (let ((top (car (ndtree-break-control renv))))
    (dtree-update-env renv
		      :break-control (list top))))


;; renv
(defun ndtree-push-break-control (renv cenv new &optional soft)
  ;;(setf a renv b cenv c new d soft) (break "npubc")
  (when (null new) (break "break-control"))
  (let* ((current (ndtree-break-control renv))
	 (top (caar current)))
    (dtree-update-env renv
		      :break-control
		      (cons (cons (case new
				    (linear 'linear)
				    (break (case top
					     (linear
					       (ndtree-fail renv cenv)
					      ;'linear
					      )
					     (otherwise top)))
				    (multilinear (case top
						   (linear 'linear)
						   (otherwise new))))
				  soft)
			    current))))


;; renv
(defun ndtree-pop-break-control (renv cenv)
  (declare (ignore cenv))
  ;;(when (null (caar (cdr (ndtree-break-control renv)))) (break "npbc"))
  (dtree-update-env renv
		    :break-control (cdr (ndtree-break-control renv))))


;; cenv
(defun ndtree-text (renv cenv text &optional l)
  (let ((ich (car (last text)))
	(ncenv (dtree-update-env cenv :col (+ (or l (length text)) (ndtree-col cenv)))))

    ;;(format t "text ~a, ~a, ~a.~%"
    ;;(implode-to-string text) (ndtree-width renv) (ndtree-col ncenv ))

    ;;(break "text")
     (if (and (ndtree-width-fail-p (ndtree-width renv) (ndtree-col ncenv))
	      (not (ndtree-wrapped-p cenv)))
	 (ndtree-fail renv ncenv)
	 (if ich
	     (dtree-update-env ncenv :last-ich ich)
	     ncenv)
	 )))


;;;;	
;;;;	w(a(b,c), x(y,z))
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
(defstruct dtree-continuation
  ;dtree ; access via renv
  continuation
  children
  renv
  )

(defmacro new-dtree-continuation (continuation children renv)
  `(make-dtree-continuation
    :continuation ,continuation
    :children ,children
    :renv ,renv))
(defmacro continuation-of-dtree-continuation (c) `(dtree-continuation-continuation ,c))
(defmacro children-of-dtree-continuation (c) `(dtree-continuation-children ,c))
(defmacro renv-of-dtree-continuation (c) `(dtree-continuation-renv ,c))
  
 
(defun ndtree-continue (cenv continuations)
  (if (null continuations)
      (values nil nil)
      (mlet* (((continue) (car continuations))
	      ((renv) (renv-of-dtree-continuation continue))
	      ((cont) (continuation-of-dtree-continuation continue))
	      ((l c) (cond
		       ((null cont)
			(with-ttree-update (text-pop
					    (reduce #'+ (mapcar #'ndtree-indent-amt
								(car (ndtree-indents renv)))))
			  (ndtree-continue (ndtree-cleanup-elide-depth cenv renv)
					   (cdr continuations))))
		       (t ;;(setf -continue continue) (break "nc")
			  (ndtree-layout-formats cont (children-of-dtree-continuation continue)
						 cenv renv
						 (cdr continuations))))))
	     (values nil
		     ;; stack up continuation result.
		     (cons l c)))))

(defvar *max-linear-depth* 4)

(defun ndtree-top-linear (dtree cenv renv continuations)
  (let ((target (cons nil (ndtree-dtree renv))))
    (with-try #'(lambda (bullet) (eql target bullet))
      (ndtree-layout dtree
		     *top-child-format*
		     (set-ndtree-elide-depth cenv target)
		     renv nil continuations)
      (ndtree-try-top-linear-depth dtree cenv renv continuations 0))))


(defun ndtree-try-top-linear-depth (dtree cenv renv continuations i)
  (if (= i *max-linear-depth*)
      (progn;; (setf a cenv b renv) (break)
	(ndtree-fail renv (set-ndtree-elide-depth cenv nil)))
      (let ((target (cons (- (1- *max-linear-depth*) i) (ndtree-dtree renv))))
	(with-try #'(lambda (bullet) (eql target bullet))
	  (ndtree-layout dtree
			 *top-child-format*
			 (set-ndtree-elide-depth cenv target)
			 renv nil continuations)
	  (ndtree-try-top-linear-depth dtree cenv renv continuations (1+ i))))))

(defun ndtree-linear-layout (formats children cenv renv continuations)
  (let ((target (cons nil (ndtree-dtree renv))))
    (with-try #'(lambda (bullet) (eql target bullet))
      (ndtree-layout-formats formats children
			     (set-ndtree-elide-depth cenv target)
			     renv continuations)
      (ndtree-try-linear-depth formats children cenv renv continuations 0))))

(defun ndtree-try-linear-depth (formats children cenv renv continuations i)
  (if (= i *max-linear-depth*)
      (progn ;; (setf a cenv b renv) (break)
	(ndtree-fail renv (set-ndtree-elide-depth cenv nil)))
      (let ((target (cons (- (1- *max-linear-depth*) i) (ndtree-dtree renv))))
	(with-try #'(lambda (bullet) (eql target bullet))
	  (ndtree-layout-formats formats children
				 (set-ndtree-elide-depth cenv target)
				 renv continuations)
	  (ndtree-try-linear-depth formats children cenv renv continuations (1+ i))))))
		


(defmacro ndtree-layout-formats-recurse (formats children cenv renv continuations)
  (let ((lformats (gensym))
	(lrenv (gensym))
	(lcenv (gensym)))
    `(let ((,lformats ,formats)
	   (,lcenv ,cenv)
	   (,lrenv ,renv))
      (if ,lformats
	  (ndtree-layout-formats ,lformats ,children ,lcenv ,lrenv ,continuations)
	  (with-ttree-update (text-pop (reduce #'+ (mapcar #'ndtree-indent-amt
							   (car (ndtree-indents ,lrenv)))))
	    (ndtree-continue (ndtree-cleanup-elide-depth ,lcenv ,lrenv)
			     ,continuations))))))



;;;;		
;;;;	layout (dtree . index) index is index into formats.
;;;;	
;;;;	
;;;;	
;;;;	<layout-continuation> : (INT{format-index} . <format-array> list*
;;;;	
;;;;	parens can contain break control. Really want to inline.
;;;;	
;;;;	formats arg is a <layout-continuation>

(defun destruct-layout-continuation (lc)
  (if (consp lc)
      (values (car lc) (cdr lc))
      (values lc nil)))

(defun do-simple-format (format cenv renv acc-f)
  (cond
    ((null format)
     (values cenv renv))
    
    ((text-format-p format)
     (let ((istr (istring-of-text-format format)))
       (funcall acc-f
		(text-istring istr))
       (setf cenv (ndtree-text renv cenv
			       istr
			       (length-of-text-format format)))
       (values cenv renv)))

    ((space-format-p format)
     (unless (member (ndtree-last-ich cenv) #.'(list ispace inewline))
       (funcall acc-f (text-space))
       (setf cenv (ndtree-text renv cenv (list ispace) 1)))
     (values cenv renv))
					
    ((push-format-p format)
     ;;(setf a format b renv)(break "pushm")
     (if (or (eql 'break (ndtree-break-control-mode renv))
	     (eql 'linear (ndtree-break-control-mode renv))  ;; to get tabbing effect.
	     )

	 ;; floor x 13 is hack since we work with pixels and not chars.
	 (let* ((indent (indent-of-push-format format))
		;;(floor (indent-of-push-format format) 13)
		(amt (+ (ndtree-col cenv) indent)))
	   ;;(format t "push amt ~a indent ~a width ~a col ~a ~%"
	   ;;amt indent (ndtree-width renv) (ndtree-col cenv))
	   (funcall acc-f (text-push amt))
	   (let ((ncenv (ndtree-update-col cenv (- indent)))
		 (nrenv (ndtree-push-indent amt renv cenv)))
	     (setf cenv ncenv renv nrenv))))
     
     (values cenv renv))

    ((pop-format-p format)
     (when (member (ndtree-break-control-mode renv) '(break linear))
       (let* ((local-indents (car (ndtree-indents renv)))
	      (amt (if local-indents (ndtree-indent-amt (car local-indents)) 0))
	      ;;(ncenv )
	      )
	 ;;(format t "pop amt: ~a width ~a col ~a ~%"
	 ;;amt (ndtree-width renv) (ndtree-col cenv))
	 (funcall acc-f (text-pop amt))
	 (setf cenv (ndtree-update-col cenv (+ (ndtree-col cenv) amt)))
	 (setf renv (ndtree-pop-indent 
		     (dtree-update-env renv
				       :width
				       (let ((w (ndtree-width renv)))
					 (if (infinite-width-p w)
					     w
					     (+ amt w))))
		     cenv))))
     (values cenv renv))
		       
    ;; should be nil as indication that not simple format
    (t nil)))



;;;;	
;;;;	continuation is made when a child is visited.
;;;;	continuation is resolved when unwinding recursion by updating ttree via result of continue.
;;;;	
;;;;	
;;;;	CutBreak : resets envrionment except for line count and accumualted ttree to initial.
;;;;	
;;;;	Desire to abort backtracking at certain junctures as though 
;;;;	we were laying out a sequence of terms and then concatenating
;;;;	the layouts.
;;;;	  - hide continuation environment and return
;;;;	  - record ttree and then continue with mostly initial envs.
;;;;	      * the continuations correspond to partial ttrees.
;;;;		need to be able to complete those ttrees after cut.
;;;;		cut needs to encode unwinding as well as continue.
;;;;		cut needs to return partial values 
;;;;	        cut-continue needs to complete correct trees.a
;;;;	desire ttrees accumulated via dtree updates rather than via return.
;;;;	
;;;;	
;;;;	or modify ttree accumulation strategy to allow throw.
;;;;	
;;;;	 not only need
;;;;	
;;;;	continue	: (l{child} . c{continuation})
;;;;	
;;;;	
;;;;	
;;;;	continuation	: formats, dtrees, renv
;;;;	
;;;;	
;;;;	continuations	: continuation list
;;;;	resolution	: ttree segment
;;;;	result		: ttree segment list.
;;;;	
;;;;	
;;;;	iterate over continuations, updateing corresponding ttrees of dtrees of continuation renv.
;;;;	  - append continuation which forces notation of cenv for next iteration.
;;;;	
;;;;	
;;;;	
;;;;	


;; assume called after break-line

(defun cut-break-cenv (cenv)
  (dtree-update-env (ndtree-init-cenv)
		    :line (ndtree-line cenv)))

(defun cut-break-renv (renv)
  (ndtree-init-renv 'break (ndtree-width renv)))

(defstruct dtree-continuation-suspend
  ;; continuation  ; (int # formats)
  ;; children	; dtree array
  cenv
  ;; renv
  continuations
  )


; assume parens is cons.
(defun ndtree-layout-continuations (cenv continuations)
  (let ((ncontinuation (car continuations)))
    (mlet* (((l c) (ndtree-layout-formats (continuation-of-dtree-continuation ncontinuation)
					  (children-of-dtree-continuation ncontinuation)
					  cenv
					  (renv-of-dtree-continuation ncontinuation)
					  (cdr continuations))))
	   
	   ;;(setf -l l -c c)
	   (cons l c))))

(defun ndtree-layout-formats (continuation children cenv renv continuations)
	      
  (unless (or (null continuation) (integerp (car continuation)))
    (break "trap layout-formats-continuation"))

  (labels
      ((layout-child-format (format child cenv renv continuations)
	 ;;(break "nlflcf")
	 (let ((nrenv (let ((minw (minw-of-child-format format))
			    (w (ndtree-width renv)))
			(if (or (infinite-width-p w)
				(and minw (< minw w)))
			    (dtree-update-env renv :width (or minw w))
			    renv))))
	   (with-try #'(lambda (target) (declare (ignore target)) nil)
	     (if (dtree-leaf-p child)
		 (ndtree-layout-leaf child format cenv renv continuations)
		 (ndtree-layout child
				format
				cenv
				(ndtree-init-break-control (ndtree-init-indent nrenv cenv) cenv)
				(parens-of-dform-term-child format)
				continuations))
	     (ndtree-fail-elide child renv cenv))))

       ;; visit (NAT <format-array> list* f)  : <layout-continuation>
       ;; f (<format>)	: BOOL { true -> suspend }
       (visit (index formats f)
	 (mlet* (((a-formats rest) (destruct-layout-continuation formats)))
		
		;;(setf -a-formats a-formats -rest rest) (break "aoji")
		(if (dformat-p a-formats)
		    (progn
		      (funcall f a-formats)
		      (when rest (visit 0 rest f))
		      )
			     
		    (let ((l (length a-formats)))

		      (do ((i index (1+ i)))
			  ((= i l)
			   (when rest (visit 0 rest f)))

			(when layout-trace (format t "~a~%" (aref a-formats i)))

			(when (funcall f (aref a-formats i))
			  (return-from visit
			    (cons i formats))))))))
       )


    (let ((ncontinuation nil))
      (with-ttree-updates
	  (when continuation
	    (let ((acc nil))
	      ;;(setf a continuation) (break "l")
	      (setf ncontinuation
		    (visit (car continuation) (cdr continuation)
			   #'(lambda (format)
			       (mlet* (((ncenv nrenv)
					(do-simple-format format cenv renv
							  #'(lambda (x) (push x acc)))))
				      (if ncenv
					  (progn
					    (setf cenv ncenv renv nrenv)
					    nil)
					  t)))))
	      (nreverse acc)))

	;;(setf -ncontinuation ncontinuation -continuation continuation -continuationsx continuations) ;;(break "ncont")

	(if (null ncontinuation)
	    (with-ttree-update (text-pop (reduce #'+ (mapcar #'ndtree-indent-amt
							     (car (ndtree-indents renv)))))
	      (ndtree-continue (ndtree-cleanup-elide-depth cenv renv)
			       continuations))

	    (mlet* (((a-formats rest) (destruct-layout-continuation (cdr ncontinuation))))

		   (let* ((index (car ncontinuation))
			  (format (aref a-formats index))
			  (nncontinuation (let ((nindex (1+ index)))
 					    (if (= nindex (length a-formats))
						(when rest
						  (cons 0 rest))
						(cons nindex (cdr ncontinuation))))))
 
		     ;;(setf -nncontinuation nncontinuation -index index -format format) ;; (break "nncont")
		     ;;(when t (format t "format : ~a~%" format))
		    
		     (cond

		       ((dform-child-p format)
			(mlet* (((l c) (layout-child-format format
							    (aref children (dtree-index-of-dform-child format))
							    cenv renv
							    ;; add-continuation even if null nncontinuations
							    ;; since that keys ndtree-continue to pop margins.
							    (cons (new-dtree-continuation nncontinuation
											  children
											  renv)
								  continuations))))
			       (values (cons l (car c)) (cdr c))))

		       ;; if {push}(<c>){break}then <a>{break}else <b>{break}{pop}fi
		       ((break-control-format-p format)
			(let ((type (type-of-break-control-format format)))
			  (cond
			    ((or (null type) (eql type '||))
			     (let* ((old-mode (ndtree-break-control-mode renv))
				    (nrenv (ndtree-pop-break-control renv cenv))
				    (mode (ndtree-break-control-mode renv)))
		
			       (ndtree-layout-formats-recurse nncontinuation children
							      (if (and (eql old-mode 'linear)
								       (not (eql mode 'linear)))
								  (set-ndtree-elide-depth cenv nil)
								  cenv)
							      nrenv
							      continuations)))
			    ((eql 'soft type)
			     (let ((ncenv (ndtree-soft cenv renv)))
			       (with-try #'(lambda (bullet)
					     ;;(setf a renv b cenv c bullet d (car (ndtree-softs ncenv)))
					     ;;(break "test")
					     (eql bullet (car (ndtree-softs ncenv)))
					     ;;t
					     )
				 (ndtree-layout-formats-recurse nncontinuation children
								ncenv
								(ndtree-push-break-control renv cenv 'linear
											   (car (ndtree-softs ncenv)))
								continuations)
				 (ndtree-layout-formats-recurse nncontinuation children cenv
								(ndtree-push-break-control renv cenv 'break)
								continuations))))
	     
			    (t (let* ((old-mode (ndtree-break-control-mode renv))
				      (nrenv (ndtree-push-break-control renv cenv type))
				      (mode (ndtree-break-control-mode nrenv)))
				 ;;(setf a old-mode b nrenv c mode d renv e type f format) (break "here")
				 (if (and (not (eql old-mode mode)) (eql 'linear mode))
				     (ndtree-linear-layout nncontinuation children cenv nrenv continuations)
				     (ndtree-layout-formats-recurse nncontinuation 
								    children cenv nrenv continuations)))))))

		       ((cut-break-format-p format)

			(if (cut-break-continuable-p)
			    (with-ttree-update (text-newline)
			      ;;(setf -continuations continuations -nncontinuation nncontinuation) (break "cbcs")
			      (cut-break-continuation-set
			       (make-dtree-continuation-suspend
				:cenv (cut-break-cenv (ndtree-break-line renv cenv))
				:continuations (cons (new-dtree-continuation nncontinuation children renv)
						     continuations)))

			      nil)
			      
			    (if (eql 'linear (ndtree-break-control-mode renv))
				(ndtree-layout-formats-recurse nncontinuation children
							       (ndtree-break-line renv cenv)
							       renv
							       continuations)
				(with-ttree-update (text-newline)
				  (ndtree-layout-formats-recurse nncontinuation children
								 (ndtree-break-line renv cenv)
								 renv
								 continuations)))))

		       ((cut-continuation-format-p format)
			(cut-break-continuation-set
			 (make-dtree-continuation-suspend
			  :cenv (cut-break-cenv cenv)
			  ))
			nil)

		       
		       ((break-format-p format)
			(let ((break-control (ndtree-break-control-mode renv)))
			  ;;(when (ndtree-indents renv) (setf a renv b cenv c break-control d format) (break "what1"))
			  (case break-control
			    (break
			     (let ((diff (ndtree-col cenv)))
			       ;;    (reduce #'+ (mapcar #'ndtree-indent-amt
			       ;;			 (car (ndtree-indents renv))))

			       ;;(setf a renv b cenv c break-control d format e diff) (break "what")
			       ;;(format t "break : ~a ~a ~a~%"
			       ;;(reduce #'+ (mapcar #'ndtree-indent-amt
			       ;;(car (ndtree-indents renv))))
			       ;;(ndtree-col cenv)  diff)
			       (if (< diff 0)
				   (let ((ttree nil))
				     (mlet* (((ncenv nrenv)
					      (do-simple-format (new-text-format
								 (blank-string (- diff)))
						cenv renv
						#'(lambda (x) (setf ttree x)))))
					    (with-ttree-update ttree
					      (ndtree-layout-formats-recurse ncontinuation children
									     ncenv nrenv continuations))))

				   (with-ttree-update (text-newline)
				     (ndtree-layout-formats-recurse nncontinuation children
								    (ndtree-break-line renv cenv)
								    renv
								    continuations)))))

			    (linear
			     (let ((ttree))
			       (mlet* (((ncenv nrenv)
					(do-simple-format (new-text-format (string-of-break-format format))
					  cenv renv
					  #'(lambda (x) (setf ttree x)))))

				      ;; jump to margin if margin is to the right of position.
				      ;; indents is stack of indents. col is position from indent at last break.
				      ;; renv width is avail width at last break.
				      ;; assume this is relative to cenv col
				      
				      (let ((diff (ndtree-col cenv)
					      ;;(- (ndtree-col cenv)
					      ;;(reduce #'+ (mapcar #'ndtree-indent-amt
					      ;;(car (ndtree-indents renv)))))
					      ))


					;;(format t "linear : ~a ~a ~a~%"
					;;(reduce #'+ (mapcar #'ndtree-indent-amt
					;;(car (ndtree-indents renv))))
					;;(ndtree-col cenv)
					;;diff)
			       
					(if (< diff 0)
					    (mlet* (((nncenv nnrenv)
						     (do-simple-format (new-text-format
									(blank-string (- diff)))
						       ncenv nrenv
						       #'(lambda (x) (setf ttree x)))))

						   (with-ttree-update ttree
						     (ndtree-layout-formats-recurse nncontinuation children
										    nncenv nnrenv continuations)))
					    (with-ttree-update ttree
					      (ndtree-layout-formats-recurse nncontinuation children
									     ncenv nrenv continuations)))))))

			    (multilinear;; needs more stuff.
			     (let ((ttree))
			       (mlet* (((ncenv nrenv) (do-simple-format (new-text-format
									 (string-of-break-format format))
							cenv renv
							#'(lambda (x) (setf ttree x)))))
				      (with-ttree-update ttree
					(ndtree-layout-formats-recurse nncontinuation children
								       ncenv nrenv continuations)))))
			    (otherwise (and nil (break
						 (format-string
						  "unknown element[~a] on break control stack."
						  break-control)))
				       (ndtree-layout-formats-recurse nncontinuation children
								      cenv renv continuations)))))
	      
		       (t;; unknown format ? probably vertical
			(when layout-trace
			  (format t "unknown format: ~a~%." format))
			(ndtree-layout-formats-recurse nncontinuation children cenv renv continuations))))))))))
	     				  

(defvar *ndtree-wrap-p* t)

(defun ndtree-wrap-p (dtree renv cenv)
  (declare (ignore dtree renv cenv))
  *ndtree-wrap-p*)

(defvar *elide-istring* (list (list idot idot idot)))

(defun ndtree-try-elide (dtree format cenv renv parens continuations &optional no-wrap)
  ;;(break "e")	   	
  (cond
    ((dtree-elided-p dtree)
     (dtree-flag-set-elided dtree nil)
     (ndtree-fail-elide dtree renv cenv))
    ((and (null no-wrap) (ndtree-wrap-p dtree renv cenv))
     ;;(format t ".") ;;(break)
     (ndtree-wrap-layout dtree format (ndtree-wrap-cenv cenv) renv
		    parens continuations))
    ((ndtree-width-fail-p (ndtree-width renv) (+ 3 (ndtree-col cenv)))
     ;;(> 3 (- (ndtree-width renv) (ndtree-col cenv)))
     (ndtree-fail-elide dtree renv cenv))
    (t (dtree-flag-set-elided dtree t)
       ;;(setf a dtree) (break)
       (mlet* (((l c) (ndtree-continue (ndtree-text renv cenv
						    (car *elide-istring*) 3)
				       continuations)
		(declare (ignore l))))
	      (values (set-dtree-ttree dtree (text-tree dtree *elide-istring*))
		      c)))))


(defun text-of-dtree-leaf (leaf)
  (if (dtree-flag-slot-p leaf)
      (descriptor-of-dtree-leaf leaf)
      (istring-of-dtree-leaf leaf)))

(defun text-of-dtree-leaf-c (leaf)
  (unless (dtree-flag-instantiated-p leaf)
    (instantiate-dtree-leaf leaf))
  
  (if (dtree-flag-slot-p leaf)
      (descriptor-of-dtree-leaf leaf)
      (istring-of-dtree-leaf leaf)))

;; istring of dtree-leaf should be font-index string.
;; at some point though font-index will be a pair font and font-index pair
;; so either edit deals with font/index pairs(two sixteen bit values)
;;      and unmarshalls font/index pairs to unicode or edit deals directly
;;    with embedded unicode and lets layout present.
;;
;; middle ground may be when editing then show exploded version.

;; see istring-of-text-format in edd-defs for temp hack.


(defun istring-to-font-index-list (istr)
  ;; fttb font index will just be hex implosion of unicode chars.
  (with-byte-list-scanner (istr)
    (scan-unicode-istring))))


(defun ndtree-layout-leaf (leaf format cenv renv continuations)
  (unless (dtree-flag-instantiated-p leaf)
    (instantiate-dtree-leaf leaf))
    
  ;;(setf -l leaf) (break "nll")

  (let ((meta (meta-bit-of-dtree-leaf leaf)))
    (cond
      ((eql meta 'display)
       (mlet* (((l c) (ndtree-continue (ndtree-text renv cenv
						    (append (list ilangle)
							    (text-of-dtree-leaf leaf)
							    (list irangle)))
				       continuations)
		(declare (ignore l))))
	      (values (text-tree leaf
				 (list (text-istring (list ilangle))
				       (text-tree leaf nil)
				       (text-istring (list irangle))))
		      c)))
      
      ((eql meta 'abstraction)
       (mlet* (((l c) (ndtree-continue (ndtree-text renv cenv
						    (append (list idollar)
							    (text-of-dtree-leaf leaf)))
				       continuations)
		(declare (ignore l))))
	      (values (text-tree leaf
				 (list (text-istring (list idollar))
				       (text-tree leaf nil)))
		      c)))
      
      ((consp meta)
       (mlet* (((l c) (ndtree-continue (ndtree-text renv cenv
						    (text-of-dtree-leaf leaf))
				       continuations)
		(declare (ignore l))))
	      (values (text-tree leaf
				 (list (text-istring (istring (car meta)))
				       (text-tree leaf nil)
				        (text-istring (istring (cdr meta)))))
		      c)))
      
      (t
       (mlet* (((l c) (ndtree-continue (ndtree-text renv cenv
						    (text-of-dtree-leaf leaf)
						    (when (dtree-flag-length-p leaf)
						      (length-of-dtree-leaf leaf)))
				       continuations)
		(declare (ignore l))))
	      (values (text-tree leaf nil) c))))))

(defun parens-parent-dform (dtree)
  (let* ((parent (parent-of-dtree dtree))
	 (dform (when parent (dform-of-dtree parent))))
    (cond
      ((null dform)
       nil)
      ((dform-parens-passthru-p dform)
       (parens-parent-dform parent))
      (t dform))))

(defun parent-parens (formats dtree parens)
  (if (parenthesize-p dtree parens)
      (let ((wrapper (wrapper-of-dform-parentheses parens)))
	;;(setf -wrapper wrapper ) (break "pp")
	(list* (car wrapper)
	       formats
	       (cdr wrapper)))
      formats))


(defun wrap-slot (format)
  (let ((a (make-array 3)))
    (setf (aref a 0) (car *layout-slot-wrapper*)
	  (aref a 1) format
	  (aref a 2) (cdr *layout-slot-wrapper*))
    a))

(defun ndtree-layout (dtree format cenv renv parens continuations)
  ;;(when t (format t "layout: ~a" dtree))
  (set-dtree-ttree dtree nil)
  
  (let ((renv (ndtree-new-depth dtree (dtree-update-env renv :dtree dtree) cenv)))
    (dtree-flag-set-elided dtree nil)
    
    (if (ndtree-depth-elide-p renv)
	(ndtree-try-elide dtree format cenv renv parens continuations)
	(with-try #'(lambda (target) (eq dtree target))
	  (let ((cenv (dtree-update-env cenv :dtrees (cons renv (ndtree-dtrees cenv))))
		(formats (if (and (iplaceholder-term-p (term-of-dtree dtree))
				  (dform-variable-child-p format))
			     (wrap-slot (descriptor-of-dform-variable-child format))
			     (formats-array-of-dform (dform-of-dtree-c dtree t)))))

	    ;; if dtree-of-term is iplaceholder-p then subst in descriptor formats?
	    ;;
	    ;;(when (iplaceholder-term-p (term-of-dtree dtree))
	    ;;(setf -a format -b dtree) (break "ndl"))
	      
	    ;;(ndtree-fail-r dtree renv cenv)
	    (let ((r (mlet* (((l c)
			      (ndtree-layout-formats (cons 0 (parent-parens formats dtree parens))
						     (children-of-dtree-c dtree)
						     cenv
						     renv
						     continuations)))
			    (let ((ttree (text-tree dtree l)))
			      ;;(setf a ttree b dtree d c e l) (break)
			      (set-dtree-ttree dtree ttree)
			      (list ttree c)))))
	      (if r
		  (values-list r)
		  (ndtree-fail renv cenv))))
	  (ndtree-try-elide dtree format cenv renv parens continuations)))))


(defun ndtree-wrap-layout (dtree format cenv renv parens continuations)
  (when layout-trace (format t "wrap-layout: ~a" dtree))
  (let ((renv (ndtree-new-depth dtree (dtree-update-env renv :dtree dtree) cenv)))
    (dtree-flag-set-elided dtree nil)
    (if (ndtree-depth-elide-p renv)
	(ndtree-try-elide dtree format cenv renv parens continuations)
	(let ((cenv (dtree-update-env cenv :dtrees (cons renv (ndtree-dtrees cenv))))
	      (formats (if (and (iplaceholder-term-p (term-of-dtree dtree))
				(dform-variable-child-p format))
			   (wrap-slot (descriptor-of-dform-variable-child format))
			   (formats-array-of-dform (dform-of-dtree-c dtree t)))))

	  ;;(when (iplaceholder-term-p (term-of-dtree dtree))
	  ;;(setf -a formats -b dtree) (break "ndl yo"))

	  (let ((r (mlet* (((l c)
			    (ndtree-layout-formats (cons 0 (parent-parens formats dtree parens))
						   (children-of-dtree-c dtree)
						   cenv
						   renv
						   continuations)))

			  (let ((ttree (text-tree dtree l)))
			    ;;(setf a ttree b dtree d c e l) (break)
			    (set-dtree-ttree dtree ttree)
			    (list ttree c)))))

	    (if r
		(values-list r)
		(progn (format t ",") (ndtree-fail renv cenv))))))))


(defvar *cut-break-continuation*)

(defun cut-break-continuable-p ()
  (if (and (boundp '*cut-break-continuation*)
	       (null *cut-break-continuation*))
      t
      (progn
	;;(break "cbcp")
	nil)))

(defun cut-break-continuation-set (c)
  (if (not (boundp *cut-break-continuation*))
      (break "ubcs not bound")

      (if (not (null *cut-break-continuation*))
	  (break "ubcs not null")

	  (progn
	    ;;(format t "C")
	    (setf *cut-break-continuation* c)))))
      
(defvar *cut-continuation*
  (list (new-dtree-continuation
	 (cons 0 (make-array 1 :initial-element (make-dformat :kind 'cut-continuation)))
	 nil
	 nil)))

(defun cut-continuation-format-p (format)
  (eql 'cut-continuation (kind-of-format format)))

(defun cut-break-wrapper (dtree format-array cenv renv)

  (set-dtree-ttree dtree nil)
  (let ((continuations (list (new-dtree-continuation (cons 0 format-array)
						     (make-array 1 :initial-element dtree)
						     renv)))
	(ncenv cenv))

    (let ((*cut-break-continuation* nil))
      
      (tagbody continue

	 ;;(format t "cbw continue ~%")
	 
	 (let ((ttrees (ndtree-layout-continuations ncenv
						    (append continuations *cut-continuation*))))

	   ;;(setf -ttrees ttrees -ncontinuations continuations) ;;(break "cbw1")
	   (mapcar #'(lambda (ttree continuation)
		       (let* ((renv (renv-of-dtree-continuation continuation))
			      (dtree (ndtree-dtree renv)))
			 ;; initial will be nil.
			 (when dtree
			   ;;(setf -ttree ttree -continuation continuation)  (break "cbwtt")
			   (set-dtree-ttree dtree (nconc (dtree-layout dtree) ttree)))))
		   ttrees
		   continuations
		   ))

	   (setf continuations nil)

	 ;;)))
	 
	   (let ((args *cut-break-continuation*))

	     (setf *cut-break-continuation* nil)
	     ;;(setf -args args) (format t "cbww")

	     (if (dtree-continuation-suspend-continuations args)
		 ;; cut-break
		 (setf continuations (if continuations
					 (append (dtree-continuation-suspend-continuations args) continuations)
					 (dtree-continuation-suspend-continuations args))
		       ncenv (dtree-continuation-suspend-cenv args))
		 ;; cenv
		 (setf ncenv (dtree-continuation-suspend-cenv args))
		 )

	     ;;(setf -args args -continuations continuations) (break "cbw2")

	     (when continuations
	       (go continue))))))

    nil
    )
    
		   ;;(mapcan #'(lambda (c)
		   ;;(when (continuation-of-dtree-continuation c)
		   ;;(list c)))
		   ;;continuations))

(defun nlayout-dtree (dtree width &optional (mode 'break))
  ;;(format t "layout~%")
  (ncompute-dtree-minima dtree)
  
  (handle-error 'layout
		#'(lambda (x)
		    (declare (ignore x))
		    (handle-error 'layout
				  #'(lambda (x)
				      (declare (ignore x))
				      (format t "Can not display term in window, probably window is too narrow.~%"))
				    
				  (ndtree-try-elide dtree *top-child-format* (ndtree-init-cenv)
						    (ndtree-init-renv 'linear width)
						    nil nil t)))

		;;(let ((dtree-size (count-dtree-nodes dtree)))
		;;(when (> dtree-size *dtree-layout-limit*)
		;;  (format t "Layout limit [~a] exceeded [~a].~%" *dtree-layout-limit* dtree-size)
		;;(throw 'layout "layout limit")))
				
		(if (eql mode 'linear)
		    (ndtree-top-linear dtree
				       (ndtree-init-cenv)
				       (ndtree-init-renv 'linear width)
				       nil)
		    (cut-break-wrapper dtree *top-child-format-array*
				       (ndtree-init-cenv)
				       ;;(dtree-update-env (ndtree-init-renv 'break width) :dtree dtree)
				       (ndtree-init-renv 'break width)
				       )
		    #|
		    (ndtree-layout dtree
				   *top-child-format*
				   (ndtree-init-cenv)
				   (ndtree-init-renv 'break width)
				   nil nil)
                    |#
		    ))
    dtree)


;;; layout-text-visit
;;;    dtree
;;;    width height
;;;    enter(row col address) exit(row col address)
;;;      called at entrance/exit of dtree node.
;;;    character (row col address char)
;;;      called for each text character and possibly at end of text slot.
;;;
;;;    row,col 1,1 is first character.
;;;
;;;    returns text-lines

;;;;  callers may assume that the character function will be called
;;;;  for each col position from left to right up to the last character on
;;;;  the line. then eol will be called before character is called for col
;;;;  positions on the next line.

;;;; character function has five args: row, col, node, ich, and i.
;;;; if i is nil, then the ich is not part of an editable string.
;;;; if i is an integer then it is the index into the editable string
;;;; of the node. an index of 0 would place the cursor at the first
;;;; char of the string such that insertion would prepend to the string.
;;;; an index of n where n is the length of the string would cause insertion
;;;; to append chars to the string.

;;;; row is absolute ie not relative to offset.

(defun text-of-line (line)
  (if (eql 'continue (car line))
      (cdr line)
      line))

(defun continued-line-p (line)
  (eql 'continue (car line)))

(defconstant icontinue 12)

(defun equal-dtree-p (subtree dtree)
  (eql subtree dtree))

;; if margin, first line is not padded but subsequent are?
(defun layout-text-visit (ttree limit offset margin width
                                &optional (result-p t) enter exit character eol no-right-pad-p)
  (let ((recorded-p nil) ;; hack to allow multiple occurences on lhs of same id.
	(row 1)
	(col 1)
	(cur-lines nil)
        (cur-line nil)
	(length 0)
	(indentation margin)
	(record-p nil))

    (labels
	(
	 (push-ich (ich)
	   (when (> row offset)
	     (when record-p (push ich cur-line)))
	   (incf col)
	   )

	 (break-line (dtree &optional continue-p)
	   ;;(setf a row b col c cur-lines d cur-line e limit f indentation g record-p)
	   ;;(break)
	   (when eol (funcall eol row col dtree))
	   (when (and (> row offset)
		      record-p)
	     (push (nreverse cur-line)
		   cur-lines)
	     (incf length))
	   (setf cur-line (if continue-p (list continue-p) nil))
	   (incf row)
	   (setf col (if (and record-p (not continue-p))
			 (+ record-p 1)
			 1))
	   (when (and limit (>= length limit))
	     (return-from layout-text-visit (nreverse cur-lines))))

	 (flush-line ()
	   (let ((line (nreverse cur-line)))
	     (if (text-of-line line)
		 (push line cur-lines))))

	 (newline (dtree)
	   (when (and (not no-right-pad-p) width (not (infinite-width-p width)))
	     (dotimes (i (1+ (- width col)))
	       (visit-ich ispace dtree)))
	   (break-line dtree)
	   (dotimes (i (- indentation (or record-p 0)))
	     (visit-ich ispace dtree)))
       
	 (add-ich (ich dtree i)
	   ;;(setf x ich u width v col w character) (break "aich")
	   (if ich
	       (progn
		 (when (and width (not (infinite-width-p width)) (> col width))
		   ;; break-line does not indent. so here we wrap.
		   ;; newline should be called when indentation is required.
		   (push-ich icontinue)
		   (break-line dtree 'continue))
		 (when character
		   (funcall character row col ich dtree i))
		 (push-ich ich))
	        (when character
		   (funcall character row col ich dtree i))))
       
	 (flush-lines ()
	   (prog1
	       (nreverse cur-lines)
	     (setf cur-lines nil)))

	 (visit-istring (istring dtree)
	   ;;(setf c cur-lines e cur-line f istring g dtree) (break "ltv")	   (break "vi")
	   (dolist (ich istring)
	     (visit-ich ich dtree)))

	 (visit-leaf (dtree)
	   (do ((i 0 (1+ i))
		(istr (text-of-dtree-leaf-c dtree) (cdr istr)))
	       ((null istr) (visit-ich nil dtree i)) ;; nil at end-of-text for leafs.
	     (visit-ich (car istr) dtree i)))

	 (visit-ich (ich dtree &optional i)
	   (add-ich ich dtree i))
	 
	 (visit-text-format (text dtree)
	   (cond
	     ((text-istring-p text)
	      (visit-istring (istring-of-text-istring text) dtree))

	     ((text-tree-p text)
	      ;;(setf e text) (break "ttp")
	      (visit-ttree text))

	     ((text-newline-p text)
	      (newline dtree))

	     ((text-space-p text)
	      (visit-ich ispace dtree))

	     ((text-indent-p text)
	      (setf indentation
		    (+ indentation (indent-of-text-indent text))))))


	 (visit-ttree (ttree)
	   ;;(setf c cur-lines d ttree e cur-line) (break "ltv")
	   (let ((dtree (dtree-of-text-tree ttree))
		 (crow row)
		 (ccol col))
	     (when enter
	       (funcall enter row col 0 dtree)
	       ;;(funcall enter row col indentation dtree)
	       ;;(setf a row b col c indentation) (break)
	       )
	     (when (and (not recorded-p) (equal-dtree-p result-p dtree))
	       (setf record-p 0)
	       ;;(setf record-p indentation)
	       )
	     ;;(setf g dtree h record-p)
	     (if (and (dtree-leaf-p dtree) (null (formats-of-text-tree ttree)))
		 (visit-leaf dtree)
		 (mapc #'(lambda (format)
			   (visit-text-format format dtree))
		       (formats-of-text-tree ttree)))
	     (when (and (not recorded-p) (equal-dtree-p result-p dtree))
	       (flush-line)
	       (setf recorded-p t)
	       (setf record-p nil))
	     (when (and (= crow row) (= ccol col))  ;;null display?
	       (unless (dtree-leaf-p dtree) (visit-ich nil dtree))) 
	     (when exit (funcall exit row col dtree)))))

      (visit-ttree ttree)
      (flush-lines))))


(defun layout-of-dtree-c (dtree w)

  ;;(setf a dtree b w)(break "ldc")

  ;;(format t "layout ~a ~%" (dtree-flag-layout-required-p dtree))
  

  (when (or (not (dtree-flag-instantiated-p dtree)) (dtree-flag-layout-required-p dtree))

    (nlayout-dtree dtree w)

    ;; At some point it may be possible for layout to no-op. At which point
    ;; layout should set-path presentation-required flag.

    (when (or nil *dtree-flag-trace*)
      (format t "layout-of-dtree-c present-required -> t, layout-required -> nil~%"))

    (dtree-flag-set-present-required dtree t)
    (dtree-flag-set-layout-required dtree nil))

  (layout-of-dtree dtree))


#|
(defun layout-visit (state record-p &key ignore-offset character enter exit eol)
  (layout-text-visit (layout-of-dtree-c (dtree-of-edit-state state) (width-of-edit-state state))
		     (height-of-edit-state state)
		     (if ignore-offset 0 (offset-of-edit-state state))
		     0
		     (width-of-edit-state state)
		     record-p
		     enter exit character eol))
|#

(defun layout-visit (ttree w h o record-p &key ignore-offset character enter exit eol)
  (layout-text-visit ttree h
		     (if ignore-offset 0 o)
		     0
		     w
		     record-p
		     enter exit character eol))


(defvar *default-layout-width* 80)

(defun default-layout-width () *default-layout-width*)

(defun term-to-pretty-string-aux (term &optional w)
  (implode-to-string
   (term-to-pretty-istring term
			   (or *default-layout-width* w))))


(defun term-to-ttree (term &optional (w 80) (m 0))
  (cons 'ttree (term-to-pretty-istring term w m)))

;; null or t width -> infinite width
(defun term-to-pretty-istring (term w &optional (m 0) mode)
  (let* ((width (cond ((null w) t)
		      ((eql 0 w) t)
		      ((integerp w) w)
		      (t (process-err
			  (format-string
			   "error: termtostring: width[~a] is not nil, t, or integer."
			   w)))))
	 (dtree (nlayout-dtree (new-dtree term nil)
			       (if (infinite-width-p width)
				   width
				   (- width m))
			       mode)))
    (mapcan #'(lambda (istring)
		(append (if (eql 'continue (car istring))
			    (cdr istring)
			    istring)
			(list inewline)))
	    (layout-text-visit
	     (ttree-of-dtree dtree width) nil 0 m width dtree nil nil nil nil t))))

(defun row-col-to-term-node (dtree width row col)
  (layout-text-visit (ttree-of-dtree dtree width) nil 0 0 width nil nil nil
		     #'(lambda (r c ich node i)
			 (declare (ignore i ich))
			 (when (and (= r row) (= c col))
			   (return-from row-col-to-term-node
			     (if (dtree-leaf-p node)
				 (parent-of-dtree node)
				 node))))
		     #'(lambda (r c node)
			 (declare (ignore c))
			 (when (= r row)
			   (return-from row-col-to-term-node
			     (if (dtree-leaf-p node)
				 (parent-of-dtree node)
				 node))))
		     t)

  dtree)


(defun term-to-string-list (term w &optional (m 0) mode)
  (let* ((width (cond ((null w) t)
		      ((eql 0 w) t)
		      ((integerp w) w)
		      (t (process-err
			  (format-string
			   "error: termtostring: width[~a] is not nil, t, or integer."
			   w)))))
	 (dtree (nlayout-dtree (new-dtree term nil)
			       (if (infinite-width-p width)
				   width
				   (- width m))
			       mode)))
    (mapcar #'implode-to-string
	    (layout-text-visit
	     (ttree-of-dtree dtree width) nil 0 m width t nil nil nil nil t))))

(defvar *ml-output-width* nil)
(defvar *ml-output-margin* nil)

(defun term-to-ml-output (term)
  (istring-to-standard-character-string 
   (butlast (term-to-pretty-istring term
				    (cond
				      ((null *ml-output-width*) 80)
				      (t *ml-output-width*))
				    (or *ml-output-margin* 0)
				    (when (eql 0 *ml-output-width*)
				      'linear)))))


(defun shift-screen (state amt)
  (setf (edit-state-offset state)
	(max 0 (+ amt (edit-state-offset state))))
  ;;(format t "ShiftOffset ~a ~%" (offset-of-edit-state state))
  ;;(when (zerop (offset-of-edit-state state)) (break))
  (unless (zerop amt)
    (view-flag-set-present-required state t))
  state)

(defun limit-length (l i)
  (subseq l 0 (min i (length l))))

(defun find-elided-ancestor (point)
  (if (null point)
      nil
      (if (dtree-leaf-p point)
	  (find-elided-ancestor (parent-of-dtree point))
	  (or (find-elided-ancestor (parent-of-dtree point))
	      (when (dtree-elided-p point)
		point)))))

;;
;;ttree-of-dtree-c?? dtree-of-edit-state-c
(defun edit-layout-point (state)
  ;; (setf -v state) (break "elp")
  ;; (setf -vc (view-state-label-cache state))

  (set-view-label-cache state nil)	; kludge to avoid messed up cache bug.
  (let* ((win (window-of-view state))
	 (w (width-of-oed-window win))
	 (h (height-of-oed-window win))
	 (ttree (ttree-of-dtree (dtree-of-view-c state) w))
	 (point (or (progn (maybe-point-rehash state 'point) (point-of-edit-state state))
		    (find-elided-ancestor (point-of-edit-state state))
		    ))
	 (entered nil);; hack to allow multiple occurences on lhs of same id.
	 (row nil)
	 (col nil)
	 (indentation 0)
	 (lines nil))

    (if (dtree-leaf-p point)
	;; should make spot{|}-cursor as layout??
	(layout-text-visit ttree h (offset-of-edit-state state) 0 w nil nil nil
			   #'(lambda (r c ich dtree i)
			       (when (and (not (and (eql dtree (car entered))
						    (eql i (cdr entered))))
					  (eql dtree point)
					  i
					  (= i (index-of-dtree-leaf dtree 'point)))
				 (setf entered (cons dtree i))
				 (setf row r
				       col c
				       lines (when ich
					       (list (list ich)))))
			       (when (and row col (null lines) ich
					  (= row r) (= col c))
				 (setf lines (list (list ich))))))
	(setf lines
	      (layout-text-visit ttree h 0 0 w
				 point
				 #'(lambda (r c indent dtree)
				     (when (and (not entered)
						(equal-dtree-p point dtree))
				       (setf entered t)
				       (setf row r col c
					     indentation indent))))))

    (setf d lines e col f row g indentation) ;; (break)
    (when (null row)
      (setf -a point -b state)
      ;;(break "elp")
      (display-msg "Editor lost in term, reset at top.")
      (when point (set-dtree-tags-and-labels point nil))
      (return-from edit-layout-point
	(progn (tag-dtree (dtree-of-edit-state state) 'point t)
	       (point-rehash state)
	       (edit-layout-point state))))


    ;; move text cursor past window width to start of next line.
    (when (null lines)
      (when (= (1- col) w)
	(setf row (1+ row)
	      col 1)))

    ;; invariant: offset < row < offset + height + 1
    (let ((offset (offset-of-edit-state state))
	  (r row)
	  (l lines))


      ;;(setf -a h -b r -c l -d offset) (break)
      ;; shift row, limit lines.
      (cond

	((<= row offset)
	 (shift-screen state (1- (- row offset)))
	 (setf r 1 l (limit-length lines h)))

	((>= row (+ offset h 1))
	 (shift-screen state (- row (+ offset h)))
	 (setf r h l (limit-length lines 1)))

	(t
	 (setf r (- row offset))
	 (setf l (limit-length lines (- (1+ h) r)))))

      (let ((cursor (cursor-of-edit-state state)))
	(cond
	  ((dtree-leaf-p point)
	   (setf (cursor-layout cursor)
		 (new-text-cursor r col)))

	  ((null l)
	   (setf (cursor-layout cursor)
		 (new-null-cursor r col)))

	  (t (setf (cursor-layout cursor)
		   (new-line-cursor r col indentation l))
	     ))))))

(defun edit-point-text-p (point)
  (and (dtree-leaf-p point)

       ;; traversable ?
       (not (dtree-flag-slot-p point))
       (not (dtree-flag-non-modifiable-p point))))



(defun cursor-of-view-c (v)
  ;;(break "covc")
  (let ((c (or (cursor-of-edit-state v)
	       ;; probably is a better way but kludge to avoid returning nil.
	       (progn (point-rehash v 'point)
		      (cursor-of-edit-state v)))))
    (when c
      (when (view-flag-cursor-layout-required-p v)

	(cond
	  ((point-cursor-p c)
	   (view-flag-set-cursor-present-required v t)
	   (edit-layout-point v))

	  ((screen-cursor-p c)
	   (view-flag-set-cursor-present-required v t)
	   (let ((point (point-of-screen-cursor c)))
	     (when (edit-point-text-p point)
	       (edit-layout-point v))))

	  (t nil)))
      

      (view-flag-set-cursor-layout-required v nil)

      ;; layout may have set new cursor.
      (layout-of-cursor (cursor-of-edit-state v)))))
    
(defun edit-layout-echo (state cursor echo)
  (labels
      ((divide (buffer w line col)
	 (if (null buffer)
	     (unless (null line)
	       (list (nreverse line)))
	     (if (> col w)
		 (cons (nreverse line)
		       (divide buffer w nil 1))
		 (divide (cdr buffer)
			 w
			 (cons (car buffer) line)
			 (1+ col) )))))
			   
    (let* ((win (window-of-view state))
	   (w (width-of-oed-window win))
	   (col (col-of-cursor cursor))
	   (lines (divide (nconc (list ispace)
				 (estring-to-istring (buffer-of-echo echo))
				 (list ispace ispace))
			  w nil 1))
	   (l (length lines))
	   (ll (length (car lines))))

      (new-line-cursor (row-of-cursor cursor)
		       (cond
			 ((> l 1) 1)
			 ((> (+ col ll) w) (1+ (- w ll)))
			 (t col)) 0 lines))))


(defun echo-of-view-c (v)
  (let ((c (cursor-of-edit-state v))
	(e (echo-of-edit-state v)))
    
    (when (and c e)
      (edit-layout-echo v c e))))



#|
;; currently no caller, needs to be called when ability to edit parameters added. (not!)
;; needs to be worked into dtree-to-term, may want to modernize parameter editing
;; ie unicode string or bytearray  rather than istr, etc.  Later, fttb maybe
;; just have hook call this. FTTB could still share istr and ttree to avoid layout
;; with text mods.

;; bad idea: violates dtree term representation invariant.
(defun uninstantiate-dtree-leaf (leaf)

  (let ((lmarks (mark-values leaf)))
  
    (if (or (not (dtree-flag-modified-p leaf))
	    ;; really shouldn't be modified in non-modifiable but heh you never know.
	    (dtree-flag-non-modifiable-p leaf))

	;; need to union in dtree marks.
	(if lmarks
	    (parameter-marks-union (parameter-of-dtree leaf) lmarks)
	    (parameter-of-dtree leaf))


	(progn
	  ;; ie haven't done tags
	  (when *dtree-flag-trace*
	    (format t "uninstantiate-dtree-leaf term-modified -> nil~%"))

	  (dtree-flag-set-term-modified leaf nil)

	  ;; danger, danger, danger. nfg. can't do this.
	  ;; violates dtree term representation invariant. actully this is ok since?
	  (setf (dtree-leaf-parameter leaf)
		(make-edit-parameter (when (dtree-flag-slot-p leaf)
				       'slot)
				     (meta-bit-of-dtree-leaf leaf)
				     (type-id-of-parameter (parameter-of-dtree leaf))
				     (implode-to-string (istring-of-dtree-leaf leaf))
				     lmarks))
	  ))))


(defun oed-dtree-leaf (parameter istring descriptor meta 
			   &optional instantiated parent irrelevant-p)

  ;; PERF avoid consing by calling init-dtree repeatedly.
  ;; PERF not important till flags efficient.
  (when *dtree-flag-trace*
    (format t "oed-dtree-leaf term-modified -> structure~%"))  
  (init-dtree-flags (make-dtree-leaf :parameter parameter
				     :istr istring
				      
				     :index (length istring)
				     :descriptor (nconc (list ilsquare)
							(istring descriptor)
							(list irsquare))
				     :parent parent
				     :meta meta)
		    (list*
		     (cons 'slot (not instantiated))
		     (cons 'irrelevant-p  irrelevant-p)
		     '((modified . t)
		       (term-modified . 'structure)

		       ))))



;; not called ?
;;(not (template-variable-id-type-p (type-of-parameter parameter)))
(defun parameter-to-dtree-leaf (parameter descriptor irrelevant-p)

  ;; (edit-type meta-type type-id . string)
  (let* ((parts (edit-parameter-parts parameter))
	 (istr (istring (string-of-parameter-parts parts)))
	 (edit-type (edit-type-of-parameter-parts parts))) 

    (let ((leaf (oed-dtree-leaf parameter
				istr
				descriptor
				(meta-type-of-parameter-parts parts)
				(not (eql 'slot edit-type))
				nil irrelevant-p
				)))

      ;;(setf b parameter d leaf) ;;trm (break "b")
      (marks leaf (marks-of-parameter-value (value-of-parameter-n parameter)))
      leaf)))

|#
