
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;; Latex gags on large paragraphs.
(defparameter *max-paragraph-length* 100) ; in lines

(defparameter *latex-preamble* 
  "\\documentstyle[amssymbols]{article}

{\\obeyspaces\\global\\let =\\ }

\\newenvironment{bogustabbing}{\\begin{tabbing}\\=\\hspace{2em}\\=\\=\\=\\kill}%
{\\end{tabbing}}

\\newcommand{\\ttlbrace}{{\\tt \\char`\\{}}
\\newcommand{\\ttrbrace}{{\\tt \\char`\\}}}
\\newcommand{\\ttuscore}{{\\tt \\char`\\_}}
\\newcommand{\\enablettchars}{%
\\tt%
\\def\\_{\\ttuscore}% 
\\def\\{{\\ttlbrace}%
\\def\\}{\\ttrbrace}%
\\obeyspaces}

% program is indented 2em
\\newenvironment{program}{\\enablettchars\\begin{bogustabbing}\\+\\kill}{\\end{bogustabbing}}
\\newenvironment{program*}{\\enablettchars\\begin{bogustabbing}}{\\end{bogustabbing}}


\\newcommand{\\mcap}{{\\boldmath$\\cap$}}
\\newcommand{\\muparrow}{{\\boldmath$\\uparrow$}}
\\newcommand{\\mdownarrow}{{\\boldmath$\\downarrow$}}
\\newcommand{\\minfty}{{\\boldmath$\\infty$}}
\\newcommand{\\mleftarrow}{{\\boldmath$\\leftarrow$}}
\\newcommand{\\mrightarrow}{{\\boldmath$\\rightarrow$}}
\\newcommand{\\mrightleftharpoons}{{\\boldmath$\\rightleftharpoons$}}
\\newcommand{\\mexists}{{\\boldmath$\\exists$}}
\\newcommand{\\mpartial}{{\\boldmath$\\partial$}}
\\newcommand{\\mwedge}{{\\boldmath$\\wedge$}}
\\newcommand{\\mcup}{{\\boldmath$\\cup$}}
\\newcommand{\\msubset}{{\\boldmath$\\subset$}}
\\newcommand{\\mforall}{{\\boldmath$\\forall$}}
\\newcommand{\\mvee}{{\\boldmath$\\vee$}}
\\newcommand{\\msupset}{{\\boldmath$\\supset$}}
\\newcommand{\\malpha}{{\\boldmath$\\alpha$}}
\\newcommand{\\mbeta}{{\\boldmath$\\beta$}}
\\newcommand{\\mdelta}{{\\boldmath$\\delta$}}
\\newcommand{\\mepsilon}{{\\boldmath$\\epsilon$}}
\\newcommand{\\mgamma}{{\\boldmath$\\gamma$}}
\\newcommand{\\mlambda}{{\\boldmath$\\lambda$}}
\\newcommand{\\mpi}{{\\boldmath$\\pi$}}
\\newcommand{\\mleq}{{\\boldmath$\\leq$}}
\\newcommand{\\mgeq}{{\\boldmath$\\geq$}}
\\newcommand{\\mint}{{\\boldmath$\\int$}}
\\newcommand{\\mcdot}{{\\boldmath$\\cdot$}}
\\newcommand{\\mneg}{{\\boldmath$\\neg$}}
\\newcommand{\\mneq}{{\\boldmath$\\neq$}}
\\newcommand{\\mequiv}{{\\boldmath$\\equiv$}}
\\newcommand{\\mpm}{{\\boldmath$\\pm$}}
\\newcommand{\\moplus}{{\\boldmath$\\oplus$}}
\\newcommand{\\motimes}{{\\boldmath$\\otimes$}}
\\newcommand{\\mdiamond}{{\\boldmath$\\diamond$}}
\\newcommand{\\msim}{{\\boldmath$\\sim$}}
\\newcommand{\\mbackslash}{{\\boldmath$\\backslash$}}

\\newcommand{\\mBbbP}{{\\boldmath$\\Bbb{P}$}}
\\newcommand{\\mBbbR}{{\\boldmath$\\Bbb{R}$}}
\\newcommand{\\mBbbN}{{\\boldmath$\\Bbb{N}$}}
\\newcommand{\\mBbbC}{{\\boldmath$\\Bbb{C}$}}
\\newcommand{\\mBbbQ}{{\\boldmath$\\Bbb{Q}$}}
\\newcommand{\\mBbbZ}{{\\boldmath$\\Bbb{Z}$}}
\\newcommand{\\mBbbU}{{\\boldmath$\\Bbb{U}$}}
\\newcommand{\\mBbbB}{{\\boldmath$\\Bbb{B}$}}
\\newcommand{\\mkleeneopen}{{\\boldmath$^{\\lceil}$}}
\\newcommand{\\mkleeneclose}{{\\boldmath$^{\\rceil}$}}

\\newcommand{\\mLeftarrow}{{\\boldmath$\\Leftarrow\\!$}}
\\newcommand{\\mRightarrow}{{\\boldmath$\\Rightarrow$}}
\\newcommand{\\mvdash}{{\\boldmath$\\vdash$}}
\\newcommand{\\mmember}{{\\boldmath$\\in$}}
\\newcommand{\\mrho}{{\\boldmath$\\rho$}}
\\newcommand{\\mSigma}{{\\boldmath$\\Sigma$}}
\\newcommand{\\mPi}{{\\boldmath$\\Pi$}}
\\newcommand{\\mDelta}{{\\boldmath$\\Delta$}}
\\newcommand{\\mtimes}{{\\boldmath$\\times$}}
\\newcommand{\\mdiv}{{\\boldmath$\\div$}}
\\newcommand{\\msupplus}{{\\boldmath$^{+}$}}
\\newcommand{\\msupminus}{{\\boldmath$^{-}$}}
\\newcommand{\\msupzero}{{\\boldmath$^{0}$}}
\\newcommand{\\msubzero}{{\\boldmath$_{0}$}}
\\newcommand{\\msubone}{{\\boldmath$_{1}$}}
\\newcommand{\\msubtwo}{{\\boldmath$_{2}$}}
\\newcommand{\\msubthree}{{\\boldmath$_{3}$}}
\\newcommand{\\msuba}{{\\boldmath$_{a}$}}
\\newcommand{\\msubb}{{\\boldmath$_{b}$}}
\\newcommand{\\msubc}{{\\boldmath$_{c}$}}
\\newcommand{\\msubq}{{\\boldmath$_{q}$}}
\\newcommand{\\msubz}{{\\boldmath$_{z}$}}
\\newcommand{\\msubseteq}{{\\boldmath$\\subseteq$}}
\\newcommand{\\msupseteq}{{\\boldmath$\\supseteq$}}

% avoid spaces between program sections 

% \\topsep=0in
% \\partopsep=0in

\\oddsidemargin 0in
\\evensidemargin 0in
\\topmargin -.5in
\\headheight 0in
\\headsep 0in
\\textwidth 6.5in
\\textheight 9.5in 

\\begin{document}

"
  )


(defparameter *latex2e-preamble* 
  "\\documentclass{article}
   \\usepackage{amssymb}


{\\obeyspaces\\global\\let =\\ }

\\newenvironment{bogustabbing}{\\begin{tabbing}\\=\\hspace{2em}\\=\\=\\=\\kill}%
{\\end{tabbing}}

\\newcommand{\\ttlbrace}{{\\tt \\char`\\{}}
\\newcommand{\\ttrbrace}{{\\tt \\char`\\}}}
\\newcommand{\\ttuscore}{{\\tt \\char`\\_}}
\\newcommand{\\enablettchars}{%
\\tt%
\\def\\_{\\ttuscore}% 
\\def\\{{\\ttlbrace}%
\\def\\}{\\ttrbrace}%
\\obeyspaces}

% program is indented 2em
\\newenvironment{program}{\\enablettchars\\begin{bogustabbing}\\+\\kill}{\\end{bogustabbing}}
\\newenvironment{program*}{\\enablettchars\\begin{bogustabbing}}{\\end{bogustabbing}}


\\newcommand{\\mcap}{{\\boldmath$\\cap$}}
\\newcommand{\\muparrow}{{\\boldmath$\\uparrow$}}
\\newcommand{\\mdownarrow}{{\\boldmath$\\downarrow$}}
\\newcommand{\\minfty}{{\\boldmath$\\infty$}}
\\newcommand{\\mleftarrow}{{\\boldmath$\\leftarrow$}}
\\newcommand{\\mrightarrow}{{\\boldmath$\\rightarrow$}}
\\newcommand{\\mrightleftharpoons}{{\\boldmath$\\rightleftharpoons$}}
\\newcommand{\\mexists}{{\\boldmath$\\exists$}}
\\newcommand{\\mpartial}{{\\boldmath$\\partial$}}
\\newcommand{\\mwedge}{{\\boldmath$\\wedge$}}
\\newcommand{\\mcup}{{\\boldmath$\\cup$}}
\\newcommand{\\msubset}{{\\boldmath$\\subset$}}
\\newcommand{\\mforall}{{\\boldmath$\\forall$}}
\\newcommand{\\mvee}{{\\boldmath$\\vee$}}
\\newcommand{\\msupset}{{\\boldmath$\\supset$}}
\\newcommand{\\malpha}{{\\boldmath$\\alpha$}}
\\newcommand{\\mbeta}{{\\boldmath$\\beta$}}
\\newcommand{\\mdelta}{{\\boldmath$\\delta$}}
\\newcommand{\\mepsilon}{{\\boldmath$\\epsilon$}}
\\newcommand{\\mgamma}{{\\boldmath$\\gamma$}}
\\newcommand{\\mlambda}{{\\boldmath$\\lambda$}}
\\newcommand{\\mpi}{{\\boldmath$\\pi$}}
\\newcommand{\\mleq}{{\\boldmath$\\leq$}}
\\newcommand{\\mgeq}{{\\boldmath$\\geq$}}
\\newcommand{\\mint}{{\\boldmath$\\int$}}
\\newcommand{\\mcdot}{{\\boldmath$\\cdot$}}
\\newcommand{\\mneg}{{\\boldmath$\\neg$}}
\\newcommand{\\mneq}{{\\boldmath$\\neq$}}
\\newcommand{\\mequiv}{{\\boldmath$\\equiv$}}
\\newcommand{\\mpm}{{\\boldmath$\\pm$}}
\\newcommand{\\moplus}{{\\boldmath$\\oplus$}}
\\newcommand{\\motimes}{{\\boldmath$\\otimes$}}
\\newcommand{\\mdiamond}{{\\boldmath$\\diamond$}}
\\newcommand{\\msim}{{\\boldmath$\\sim$}}
\\newcommand{\\mbackslash}{{\\boldmath$\\backslash$}}

\\newcommand{\\mBbbP}{{\\boldmath$\\mathbb{P}$}}
\\newcommand{\\mBbbR}{{\\boldmath$\\mathbb{R}$}}
\\newcommand{\\mBbbN}{{\\boldmath$\\mathbb{N}$}}
\\newcommand{\\mBbbC}{{\\boldmath$\\mathbb{C}$}}
\\newcommand{\\mBbbQ}{{\\boldmath$\\mathbb{Q}$}}
\\newcommand{\\mBbbZ}{{\\boldmath$\\mathbb{Z}$}}
\\newcommand{\\mBbbU}{{\\boldmath$\\mathbb{U}$}}
\\newcommand{\\mBbbB}{{\\boldmath$\\mathbb{B}$}}
\\newcommand{\\mkleeneopen}{{\\boldmath$^{\\lceil}$}}
\\newcommand{\\mkleeneclose}{{\\boldmath$^{\\rceil}$}}

\\newcommand{\\mLeftarrow}{{\\boldmath$\\Leftarrow\\!$}}
\\newcommand{\\mRightarrow}{{\\boldmath$\\Rightarrow$}}
\\newcommand{\\mvdash}{{\\boldmath$\\vdash$}}
\\newcommand{\\mmember}{{\\boldmath$\\in$}}
\\newcommand{\\mrho}{{\\boldmath$\\rho$}}
\\newcommand{\\mSigma}{{\\boldmath$\\Sigma$}}
\\newcommand{\\mPi}{{\\boldmath$\\Pi$}}
\\newcommand{\\mDelta}{{\\boldmath$\\Delta$}}
\\newcommand{\\mtimes}{{\\boldmath$\\times$}}
\\newcommand{\\mdiv}{{\\boldmath$\\div$}}
\\newcommand{\\msupplus}{{\\boldmath$^{+}$}}
\\newcommand{\\msupminus}{{\\boldmath$^{-}$}}
\\newcommand{\\msupzero}{{\\boldmath$^{0}$}}
\\newcommand{\\msubzero}{{\\boldmath$_{0}$}}
\\newcommand{\\msubone}{{\\boldmath$_{1}$}}
\\newcommand{\\msubtwo}{{\\boldmath$_{2}$}}
\\newcommand{\\msubthree}{{\\boldmath$_{3}$}}
\\newcommand{\\msuba}{{\\boldmath$_{a}$}}
\\newcommand{\\msubb}{{\\boldmath$_{b}$}}
\\newcommand{\\msubc}{{\\boldmath$_{c}$}}
\\newcommand{\\msubq}{{\\boldmath$_{q}$}}
\\newcommand{\\msubz}{{\\boldmath$_{z}$}}
\\newcommand{\\msubseteq}{{\\boldmath$\\subseteq$}}
\\newcommand{\\msupseteq}{{\\boldmath$\\supseteq$}}

% avoid spaces between program sections 

% \\topsep=0in
% \\partopsep=0in

\\oddsidemargin 0in
\\evensidemargin 0in
\\topmargin -.5in
\\headheight 0in
\\headsep 0in
\\textwidth 6.5in
\\textheight 9.5in 

\\begin{document}

"
  )

(defparameter *special-standard-char->latex-macro* 
  '((#\# . "\\#")
    ;; following is continuation character. ignore fttb.
    (#\page . "") ;; \\end{program*}\\newpage\\begin{program*}
    (#\$ . "\\$")
    (#\% . "\\%")
    (#\& . "\\&")
    (#\~ . "\\msim{}")
    (#\_ . "\\_")
    (#\^ . "\\^{}")
    (#\{ . "\\{")
    (#\} . "\\}")
    (#\\ . "\\mbackslash{}")))

;;; Roughly the lispm's "symbol" characters.  The macros get defined
;;; in a header put out by functions in io.lisp.  
(defparameter *nonstandard-graphic-code->latex-macro*
  `((,(code-char 127) . "\\mint{}");; symbolics 127
    (,(code-char 128) . "\\mBbbP{}");; 
    (,(code-char 129) . "\\mBbbR{}");; 
    (,(code-char 130) . "\\mBbbN{}");; 
    (,(code-char 131) . "\\mBbbC{}");; 
    (,(code-char 132) . "\\mBbbQ{}");; 
    (,(code-char 133) . "\\mBbbZ{}");; 
    (,(code-char 134) . "\\mBbbU{}");; 
    (,(code-char 135) . "\\mLeftarrow{}");; 
    (,(code-char 136) . "{}");; 
    (,(code-char 137) . "\\mRightarrow{}");; 
    (,(code-char 138) . "\\mkleeneopen{}");; 
    (,(code-char 139) . "\\mkleeneclose{}");; 
    (,(code-char 140) . "\\mvdash{}");; 
    (,(code-char 141) . "\\mint{}");; 
    (,(code-char 142) . "\\mcdot{}");; symbolics 0
    (,(code-char 143) . "\\mdownarrow{}");; symbolics 1
    (,(code-char 144) . "\\malpha{}");; symbolics 2
    (,(code-char 145) . "\\mbeta{}");; symbolics 3
    (,(code-char 146) . "\\mwedge{}");; symbolics 4
    (,(code-char 147) . "\\mneg{}");; symbolics 5
    (,(code-char 148) . "\\mmember{}");; symbolics 6
    (,(code-char 149) . "\\mpi{}");; symbolics 7
    (,(code-char 150) . "\\mlambda{}");; symbolics 8
    (,(code-char 151) . "\\mgamma{}");; symbolics 9
    (,(code-char 152) . "\\mdelta{}");; symbolics 10
    (,(code-char 153) . "\\muparrow{}");; symbolics 11
    (,(code-char 154) . "\\mpm{}");; symbolics 12
    (,(code-char 155) . "\\moplus{}");; symbolics 13
    (,(code-char 156) . "\\minfty{}");; symbolics 14
    (,(code-char 157) . "\\mpartial{}");; symbolics 15
    (,(code-char 158) . "\\msubset{}");; symbolics 16
    (,(code-char 159) . "\\msupset{}");; symbolics 17
    (,(code-char 160) . "\\mcap{}");; symbolics 18
    (,(code-char 161) . "\\mcup{}");; symbolics 19
    (,(code-char 162) . "\\mforall{}");; symbolics 20
    (,(code-char 163) . "\\mexists{}");; symbolics 21
    (,(code-char 164) . "\\motimes{}");; symbolics 22
    (,(code-char 165) . "\\mrightleftharpoons{}");; symbolics 23
    (,(code-char 166) . "\\mleftarrow{}");; symbolics 24
    (,(code-char 167) . "\\mrightarrow{}");; symbolics 25
    (,(code-char 168) . "\\mneq{}");; symbolics 26
    (,(code-char 169) . "\\mdiamond{}");; symbolics 27
    (,(code-char 170) . "\\mleq{}");; symbolics 28
    (,(code-char 171) . "\\mgeq{}");; symbolics 29
    (,(code-char 172) . "\\mequiv{}");; symbolics 30
    (,(code-char 173) . "\\mvee{}");; symbolics 31
    (,(code-char 174) . "\\mleftarrow{}");; 
    (,(code-char 175) . "{}");;
    (,(code-char 176) . "\\mrightarrow{}");; 
    (,(code-char 177) . "\\mSigma{}");; 
    (,(code-char 178) . "\\mDelta{}");;
    (,(code-char 179) . "\\mPi{}");;
    (,(code-char 180) . "\\mtimes{}");; 
    (,(code-char 181) . "\\mdiv{}");; 
    (,(code-char 182) . "\\msupplus{}");; 
    (,(code-char 183) . "\\msupminus{}");; 
    (,(code-char 184) . "\\msupzero{}");; 
    (,(code-char 185) . "\\msubseteq{}");; 
    (,(code-char 186) . "\\msupseteq{}");; 
    (,(code-char 187) . "\\msubzero{}");; 
    (,(code-char 188) . "\\msubone{}");; 
    (,(code-char 189) . "\\msubtwo{}");; 
    (,(code-char 190) . "\\msubthree{}");; 
    (,(code-char 192) . "\\mBbbB{}");; 
    (,(code-char 193) . "\\mrho{}");; 
    (,(code-char 194) . "\\msuba{}");; 
    (,(code-char 195) . "\\msubq{}");; 
    (,(code-char 196) . "\\msubb{}");; 
    (,(code-char 197) . "\\msubz{}");; 
    (,(code-char 198) . "\\msubc{}");; 
    ))

(defparameter *nonstandard-graphic-ichar->string*
  `((128 . "[BoldP]")			;; \\mBbbP{}
    (129 . "[BoldR]")			;; \\mBbbR{}
    (130 . "[BoldN]")			;; \\mBbbN{}
    (131 . "[BoldC]")			;; \\mBbbC{}
    (132 . "[BoldQ]")			;; \\mBbbQ{}
    (133 . "[BoldZ]")			;; \\mBbbZ{}
    (134 . "[BoldU]")			;; \\mBbbU{}
    (135 . "<=")			;; \\mLeftarrow{}
    (136 . "=")				;; {}
    (137 . "=>")			;; \\mRightarrow{}
    (138 . "[UpperLeftRightAngle]")	;; \\mkleeneopen{}
    (139 . "[UpperRightRightAngle]")	;; \\mkleeneclose{}
    (140 . "[TurnStyle]")		;; mvdash{}
    (141 . "[Integral]")		;; \\mint{}
    (142 . "[CenterDot]")		;; \\mcdot{}
    (143 . "[DownArrow]")		;; \\mdownarrow{}
    (144 . "[Alpha]")			;; \\malpha{}
    (145 . "[Beta]")			;; \\mbeta{}
    (146 . "[Or]")			;; mwedge{}
    (147 . "[Not]")			;; \\mneg{}
    (148 . "[Member]") 			;; \\mmember{}
    (149 . "[Pi]")			;; \\mpi{}
    (150 . "[Lambda]")			;; \\mlambda{}
    (151 . "[Gamma]")			;; \\mgamma{}
    (152 . "[Delta]")			;; \\mdelta{}
    (153 . "[UpArrow]")			;; \\muparrow{}
    (154 . "[PlusMinus]")		;; \\mpm{}
    (155 . "[CirclePlus]")		;; \\moplus{}
    (156 . "[Infinity]")		;; \\minfty{}
    (157 . "[Parital]")			;; \\mpartial{}
    (158 . "[SubSet]")			;; \\msubset{}
    (159 . "[SuperSet]")		;; \\msupset{}
    (160 . "[Intersection]")		;; \\mcap{}
    (161 . "[Union]")			;; \\mcup{}
    (162 . "[All]")			;; \\mforall{}
    (163 . "[Exists]")			;; \\mexists{}
    (164 . "[CircleTimes]")		;; \\motimes{}
    (165 . "<->")			;; \\mrightleftharpoons{}
    (166 . "<-")			;; \\mleftarrow{}
    (167 . "->")			;; \\mrightarrow{}
    (168 . "[NotEqual]")		;; \\mneq{}
    (169 . "[Squiggle]")		;; \\mdiamond{}
    (170 . "=<")			;; \\mleq{}
    (171 . ">=")			;; \\mgeq{}
    (172 . "==")			;; \\mequiv{}
    (173 . "[And]")			;; \\mvee{}
    (174 . "<-")			;; \\mleftarrow{}
    (175 . "--")			;; {}
    (176 . "->")			;; \\mrightarrow{}
    (177 . "[Sigma]")			;; \\mSigma{}
    (178 . "[Delta]")			;; \\mDelta{}
    (179 . "[BigPi]")			;; \\mPi{}
    (180 . "[Times]")			;; \\mtimes{}
    (181 . "[Divides]")			;; \\mdiv{}
    (182 . "[SuperScript+]")		;; \\msupplus{}
    (183 . "[SuperScript-]")		;; \\msupminus{}
    (184 . "[SuperScript0]")		;; \\msupzero{}
    (185 . "[SubSetEqual]")		;; \\msubseteq{}
    (186 . "[SuperSetEqual]")		;; \\msupseteq{}
    (187 . "[SubScript0]")		;; \\msubzero{}
    (188 . "[SubScript1]")		;; \\msubone{}
    (189 . "[SubScript2]")		;; \\msubtwo{}
    (190 . "[SubScript3]")		;; \\msubthree{}
    (192 . "[BoldB]")			;; \\mBbbB{}
    (193 . "[rho]") 			;; \\mrho{}
    ))

;;; These aren't that great, but they do result in 
;;; more readable text.

(defparameter *nonstandard-graphic-ichar->pretty-string*
  `((128 . "PP")			;; \\mBbbP{}
    (129 . "RR")			;; \\mBbbR{}
    (130 . "NN")			;; \\mBbbN{}
    (131 . "CC")			;; \\mBbbC{}
    (132 . "QQ")			;; \\mBbbQ{}
    (133 . "ZZ")			;; \\mBbbZ{}
    (134 . "UU")			;; \\mBbbU{}
    (135 . "<=")			;; \\mLeftarrow{}
    (136 . "=")				;; {}
    (137 . "=>")			;; \\mRightarrow{}
    (138 . "`")	                        ;; \\mkleeneopen{}
    (139 . "'")	                        ;; \\mkleeneclose{}
    (140 . "|-")         		;; mvdash{}
    (141 . "Intg")         		;; \\mint{}
    (142 . ".")         		;; \\mcdot{}
    (143 . "!")	                	;; \\mdownarrow{}
    (144 . "%a")          		;; \\malpha{}
    (145 . "%b")  			;; \\mbeta{}
    (146 . "/\\")			;; mwedge{}
    (147 . "~")   			;; \\mneg{}
    (148 . "in") 			;; \\mmember{}
    (149 . "%p")			;; \\mpi{}
    (150 . "\\") 			;; \\mlambda{}
    (151 . "%g") 			;; \\mgamma{}
    (152 . "%d")        		;; \\mdelta{}
    (153 . "^")		        	;; \\muparrow{}
    (154 . "+-")	        	;; \\mpm{}
    (155 . "o+o")	        	;; \\moplus{}
    (156 . "oo")        		;; \\minfty{}
    (157 . "%pd ") 			;; \\mpartial{}
    (158 . "c=")			;; \\msubset{}
    (159 . "=o")          		;; \\msupset{}
    (160 . "|^|")          		;; \\mcap{}
    (161 . "|_|")  			;; \\mcup{}
    (162 . "All ")			;; \\mforall{}
    (163 . "Ex ")			;; \\mexists{}
    (164 . "oxo")          		;; \\motimes{}
    (165 . "<->")			;; \\mrightleftharpoons{}
    (166 . "<-")			;; \\mleftarrow{}
    (167 . "->")			;; \\mrightarrow{}
    (168 . "=!=")	        	;; \\mneq{}
    (169 . "~")         		;; \\mdiamond{}
    (170 . "=<")			;; \\mleq{}
    (171 . ">=")			;; \\mgeq{}
    (172 . "==")			;; \\mequiv{}
    (173 . "\\/")			;; \\mvee{}
    (174 . "<-")			;; \\mleftarrow{}
    (175 . "-")	        		;; {}
    (176 . "->")			;; \\mrightarrow{}
    (177 . "Sgm ")			;; \\mSigma{}
    (178 . "Dlt ")			;; \\mDelta{}
    (179 . "Pi ")			;; \\mPi{}
    (180 . "X") 			;; \\mtimes{}
    (181 . "/")  			;; \\mdiv{}
    (182 . "^+")          		;; \\msupplus{}
    (183 . "^-")		        ;; \\msupminus{}
    (184 . "^0")        		;; \\msupzero{}
    (185 . "c==")	        	;; \\msubseteq{}
    (186 . "==o")		        ;; \\msupseteq{}
    (187 . "_0")		;; \\msubzero{}
    (188 . "_1")		;; \\msubone{}
    (189 . "_2")		;; \\msubtwo{}
    (190 . "_3")		;; \\msubthree{}
    (192 . "BB")			;; \\mBbbB{}
    (193 . "_a") 			;; 
    (194 . "_q") 			;; 
    (195 . "_b") 			;; 
    (196 . "_z") 			;; 
    (197 . "_c") 			;; 
    ))

     
(defun new-line-and-latex-string (line out terminate-current-line?)
  (when terminate-current-line?
    (princ "\\\\" out))
  (terpri out)
  (princ "\\>" out)
  (latex-line line out))

(defun latex-line (line out)
  (let ((bs nil)
	(j 0)
	(s ""))
    (labels ((visit (ch)
	       (if bs
		   (if (= j 4)
		       (progn (setf bs nil) (setf j 0)
			      ;;(setf ss s) (break)
			      (princ (or (cdr (assoc (code-char (hex-string-to-int s))
						     *nonstandard-graphic-code->latex-macro*
						     :test #'char=))
					 (cdr (assoc (code-char (hex-string-to-int s))
						     *special-standard-char->latex-macro*
						     :test #'char=))
					 (code-char (hex-string-to-int s));;(break "foo")
					 )
				     out)
			      (if (char= #\\ ch)
				  (progn 
				    (setf bs t) (setf s ""))
				  (princ ch out)))
		       (progn (setf j (1+ j)) (setf s (concatenate 'string s (string ch)))))
		   (if (char= #\\ ch)
		       (progn;;(break s)
			 (setf bs t) (setf s ""))
		       (princ (if (alphanumericp ch)
				  ch
				  (or (cdr (assoc ch *nonstandard-graphic-code->latex-macro*
						  :test #'char=))
				      (cdr (assoc ch *special-standard-char->latex-macro*
						  :test #'char=))
				      ch))
			      out)))))
		   
      (map nil #'visit line))))
     
(defun new-line-and-latex-string-old (line out terminate-current-line?)
  (when terminate-current-line?
    (princ "\\\\" out))
  (terpri out)
  (princ "\\>" out)
  (map nil
       #'(lambda (ch) 
	   (princ (if (alphanumericp ch)
		      ch
                    (or (cdr (assoc ch *nonstandard-graphic-code->latex-macro*
                                    :test #'char=))
                        (cdr (assoc ch *special-standard-char->latex-macro*
                                    :test #'char=))
                        ch))
		  out))
       line))
     
(defun string-of-blanks? (string)
  (every #'(lambda (ch) (char= ch #\space)) string))

(defvar *tbreak* nil)

(defun latex-paragraph (in out)
  (let ((non-blank-line-read? nil)
	(lines 0))
    (do ((l (read-line in nil) (read-line in nil)))
	((or (null l)
	     (and non-blank-line-read?
		  (string-of-blanks? l))
	     (>= lines (1- *max-paragraph-length*)))
	 
	 (when (and l (not (string-of-blanks? l)))
	   (when *tbreak* (setf myl l)(break l))
	   (new-line-and-latex-string l out t))
	 (when non-blank-line-read?
	   (terpri out)
	   (latex-paragraph-conclusion out))
	 (and l t))
      (unless (string-of-blanks? l)
	(incf lines)
	(cond (non-blank-line-read?
	       (when *tbreak* (setf myl l)(break l))
	       (new-line-and-latex-string l out t))
	      (t
	       (setf non-blank-line-read? t)
	       (latex-paragraph-intro out)
	       (new-line-and-latex-string l out nil)))))))

(defun latex-paragraph-old (in out)
  (let ((non-blank-line-read? nil)
	(lines 0))
    (do ((l (read-line in nil) (read-line in nil)))
	((or (null l)
	     (and non-blank-line-read?
		  (string-of-blanks? l))
	     (>= lines (1- *max-paragraph-length*)))
	 (when (and l (not (string-of-blanks? l)))
	   (new-line-and-latex-string l out t))
	 (when non-blank-line-read?
	   (terpri out)
	   (latex-paragraph-conclusion out))
	 (and l t))
      (unless (string-of-blanks? l)
	(incf lines)
	(cond (non-blank-line-read?
	       (new-line-and-latex-string l out t))
	      (t
	       (setf non-blank-line-read? t)
	       (latex-paragraph-intro out)
	       (new-line-and-latex-string l out nil)))))))

(defun latex-paragraph-conclusion (out)
  (princ "\\end{program*}" out)
  (terpri out)
  (terpri out))

(defun latex-paragraph-intro (out)
  ;;(break "lpi")
  (princ "\\begin{program*}" out))

(defun latex-preamble (out &optional old)
  (princ (if old
	     *latex-preamble*
	     *latex2e-preamble*)
	 out))

(defun latex-postamble (s)
  (format s "~%\\end{document}"))

(defun latexize-file (input-file output-file
                                 &optional
				 (nonstandard-chars-only? nil)
				 (no-ambles-p nil)
				 (old-latex-p nil)
				 )
  (handle-file-error
   (with-open-file (in
		    input-file
		    :direction :input)
     (with-open-file (out
		      output-file
		      :direction :output
		      :if-exists :new-version
		      :if-does-not-exist :create)
       (cond (nonstandard-chars-only?
	      (do ((ch (read-char in nil nil) (read-char in nil nil)))
		  ((null ch))
		(if (alphanumericp ch)
		    (write-char ch out)
		    (let ((macro
			   (cdr (assoc ch *nonstandard-graphic-code->latex-macro*))))
		      (if macro
			  (princ macro out)
			  (write-char ch out))))))				
	     (t
	      (unless no-ambles-p (latex-preamble out old-latex-p))
	      (do ((paragraph? (latex-paragraph in out)
			       (latex-paragraph in out)))
		  ((not paragraph?)))
	      (unless no-ambles-p (latex-postamble out)))))))
  (values))


(defun latexize-file-old (input-file output-file
                                 &optional
				 (nonstandard-chars-only? nil)
				 (no-ambles-p nil)
				 (old-latex-p nil)
				 )
  (handle-file-error
   (with-open-file (in
		    input-file
		    :direction :input)
                   (with-open-file (out
                                    output-file
                                    :direction :output
                                    :if-exists :new-version
                                    :if-does-not-exist :create)
                                   (cond (nonstandard-chars-only?
                                          (do ((ch (read-char in nil nil) (read-char in nil nil)))
                                              ((null ch))
                                            (if (alphanumericp ch)
                                                (write-char ch out)
                                              (let ((macro
                                                     (cdr (assoc ch *nonstandard-graphic-code->latex-macro*))))
                                                (if macro
                                                    (princ macro out)
                                                  (write-char ch out))))))				
                                         (t
                                          (unless no-ambles-p (latex-preamble out old-latex-p))
                                          (do ((paragraph? (latex-paragraph-old in out)
                                                           (latex-paragraph-old in out)))
                                              ((not paragraph?)))
                                          (unless no-ambles-p (latex-postamble out)))))))
  (values))


;;;
;;;  latexize
;;;

(defvar *output-for-latexize* t)

(defun set-output-for-latexize (x)
  (setf *output-for-latexize* x))

(defun output-for-latexize-p ()
  *output-for-latexize*)


(defunml (|set_output_for_latexize| (x))
  (bool -> void)
  (set-output-for-latexize x))


(defunml (|latexize_file| (input-file output-file)) (string -> (string -> void))
  (latexize-file input-file output-file))

(defconstant *newline-string* (format-string "~%"))

(defunml (|latexize_char| (chsym)) (tok -> string)
  (let ((istr (string-to-unicode-istring (string chsym))))
    (if (not (and istr (null (cdr istr))))
	(breakout evaluation
		  (format-string "latexize-char: Char[~a] must be of length 1." chsym))
	(let ((ich (car istr)))
	  (cond
	    ((eql ich #x000C) "\\000C")
	    ((eql ich #x000A) *newline-string*)
	    ((eql ich #xff24) "$")
	    ((eql ich #xff5c) "\\")
	    ((eql ich #xff5e) "^")
	    ((eql ich #xff5f) "_")
	    ((eql ich #xff7b) "{")
	    ((eql ich #xff7d) "}")
	    ((eql ich #xff7e) "~")

	    (t (let ((ch (code-char (car istr))))
		 (or (cdr (assoc ch *nonstandard-graphic-code->latex-macro*
				 :test #'char=))
		     (cdr (assoc ch *special-standard-char->latex-macro*
				 :test #'char=))

		     (if (> ich 256)
			 (raise-error (error-message '(latexize_char unexpected unicode) ich))
			 (coerce (list ch) 'string))))))))))
