%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

let point_tag = tok_to_tag `POINT`;;
let mark_tag = tok_to_tag `MARK`;;


%
;;;;	view_edit cmds could be combined but want to do set_point_cursor and refresh only once.
;;;;	
;;;;	Permuted ? (perp)	- order of child occurence in format list. 
;;;;	Direction ? (dir)	- left to right ?
;;;;	Filter : cond_expr	- use first node satisfying expression.
;;;;	Tag :			- point, mark, etc.
;;;;	
;;;;	
;;;;	Result :		- false if no-op, true of any change.
;;;;	
;;;;	edit_down	: bool{perp} -> bool{dir}
;;;;			    -> cond_expr -> tag -> view -> bool
;;;;	  - view_point_down		: bool{dir} -> view -> bool
;;;;	     * (m-x)down
;;;;	     * (m-x)down-left
;;;;	     * (m-x)down-right
;;;;	  - view_point_down_oid		: bool{dir} -> view -> bool
;;;;	  - view_point_down_nat		: bool{dir} -> view -> bool
;;;;	  - view_point_down_bool	: bool{dir} -> view -> bool
;;;;	  - view_point_down_tree	: bool{dir} -> view -> bool
;;;;	     * (m-x)down-term
;;;;	     * stops at structures but not leafs.
;;;;	
;;;;	edit_up		: tag -> view -> bool
;;;;	  - view_up		:  view -> bool
;;;;	
;;;;	edit_sibling	: bool{perp} -> bool{dir}
;;;;			    -> cond_expr -> tag -> view -> bool
;;;;	  - view_point_sibling	: bool -> view -> bool
;;;;	     * (m-x)next-sibling
;;;;	     * (m-x)prev-sibling
;;;;	  - view_point_sibling_tree
;;;;	     * (m-x)next-sibling-term
;;;;	     * (m-x)prev-sibling-term
;;;;	
;;;;	
;;;;	edit_cut		: tag -> view -> bool
;;;;	  - view_point_cut	: view -> bool
;;;;	edit_paste_term		: tag -> term -> view -> bool
;;;;	  - view_paste_term	: term -> view -> bool
;;;;	
;;;;	edit_undo		: view -> bool
;;;;	  - view_undo		: view -> bool
;;;;	edit_redo		: view -> bool
;;;;	  - view_redo		: view -> bool
;;;;	edit_abort_undo		: view -> bool
;;;;	  - view_abort_undo	: view -> bool
;;;;	edit_commit_undo	: view -> bool
;;;;	  - view_commit_undo	: view -> bool
;;;;	 * not necessary to call commit, as implicit if view modified. However, it may
;;;;	   be convenient to ensure undo session is terminated.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	macros as ml :
;;;;	
;;;;	(m-x)my-edit-cmd
;;;;	  - binds to ?edit?view?_my_edit_cmd 
;;;;	  - view -> bool
;;;;	  - fails if view_my_edit_cmd not defined.
;;;;	  
;;;;	! args to macros ! : but then no need to extend macro syntax since 
;;;;	  we are replacing macros with ml?
;;;;	(m-x)fu{<parm list>}
;;;;	(cm-x)gu{$s:s}==(m-x)fu{<$s>}
;;;;
;;;;	==(-"<cond-expr>") | ==(_<cond-expr>)
;;;;	  or embed some syntax for cond_expr inside of condition list syntax.
;;;;
;;;;	
;;;;	
;;;;	
%


let view_cmd f v =
 let r = (f v) in
    view_set_point_cursor %`POINT`% point_tag v
  ; view_refresh v
  ; r
;;

% represents the address of goal slot, where we place the cursor initially when
  opening a proof window %
letref prf_init_addr = [1; 6; 1];;
			
let view_init_point v =
  if not (view_label_q point_tag v)
     then (if ((kind_of_oid (oid_of_view v) = `PRF`) ? false) then
	      view_cmd (edit_tag_address point_tag prf_init_addr true) v
           else view_cmd (edit_tag_address point_tag nil true) v)
;;

let view_point_down lrmode = edit_down true lrmode true_ce point_tag ;;
let view_point_down_cmd lrmode  = view_cmd (view_point_down lrmode);;

let view_point_up = edit_up point_tag ;;
let view_point_up_cmd   = view_cmd (view_point_up);;

let view_point_sibling lrmode = edit_sibling true lrmode true_ce point_tag ;;
let view_point_sibling_cmd lrmode  = view_cmd (view_point_sibling lrmode);;

let tree_ce = edit_ce_string_to_expression "tree";;

let view_point_down_tree lrmode = edit_down true lrmode tree_ce point_tag ;;
let view_point_down_tree_cmd lrmode  = view_cmd (view_point_down_tree lrmode);;
 

let oid_ce = edit_ce_string_to_expression ",o";;
let view_point_down_oid lrmode = edit_down true lrmode oid_ce point_tag ;;
let view_point_down_oid_cmd lrmode  = view_cmd (view_point_down_oid lrmode);;

let nat_ce = edit_ce_string_to_expression ",n";;
let view_point_down_nat lrmode = edit_down true lrmode nat_ce point_tag ;;
let view_point_down_nat_cmd lrmode  = view_cmd (view_point_down_nat lrmode);;

let bool_ce = edit_ce_string_to_expression ",b";;
let view_point_down_bool lrmode = edit_down true lrmode bool_ce point_tag ;;
let view_point_down_bool_cmd lrmode  = view_cmd (view_point_down_bool lrmode);;


let view_point_cut = edit_cut point_tag ;;
let view_point_cut_cmd = view_cmd view_point_cut;;

let view_point_paste_term = edit_paste_term point_tag ;;
let view_point_paste_term_cmd term = view_cmd (view_point_paste_term term);;

let view_undo = edit_undo ;;
let view_undo_cmd = view_cmd view_undo;;

let view_redo = edit_redo ;;
let view_redo_cmd = view_cmd view_redo;;

let view_abort_undo = edit_abort_undo ;;
let view_abort_undo_cmd = view_cmd view_abort_undo;;



%
;;;	
;;;	Text
;;;	
%

let view_move_point_text lrmode = edit_move_label_text lrmode point_tag ;;
let view_move_point_text_cmd lrmode  = view_cmd (view_move_point_text lrmode);;

let view_delete_point_text lrmode = edit_delete_label_text lrmode point_tag;;
let view_delete_point_text_cmd lrmode  = view_cmd (view_delete_point_text lrmode);;


let view_insert_point_text lrmode s = edit_insert_label_text lrmode point_tag s;;
let view_insert_point_text_cmd lrmode s  = view_cmd (view_insert_point_text lrmode s);;

let dowhile f =
 letrec aux i =
   if (f ()) then aux (i + 1) else i in
 aux 0
;;



let view_quit v =
  view_close_window v;
  view_disassociate_object false v;
  view_discard v
;;

let view_of_oid oid =
 find (\v. (equal_oids_p (view_object_q v) oid) ? false) (views ())
;;

 let view_open_prf soid poid =
 ((let v = view_of_oid poid in
        view_raise v;
	v)
   ?
   (let view = (let v = new_view false in
        view_associate_prf_object soid poid v;
        view_open_window v;
        view_init_point v;
        v) in 

  proof_view_add_oid view poid;
  view))
;;

let proof_stats soid =
 lib_eval_to_term (oid_ap (begin_ap "proof_stats") soid)
;;

let stm_to_prf soid =
 lib_eval_to_object_id (oid_ap (begin_ap "stm_to_prf") soid)
;;

let stm_to_prfs soid =
 lib_eval_to_object_ids (oid_ap (begin_ap "stm_to_prfs") soid)
;;

let stm_to_new_prf soid =
 lib_eval_to_object_id (oid_ap (begin_ap "stm_to_new_prf") soid)
;;

let stm_to_next_prf soid =
 lib_eval_to_object_id (oid_ap (begin_ap "stm_to_next_prf") soid)
;;

let view_prfs_of_stm soid = 
 map (\p. view_open_prf soid p) (stm_to_prfs soid)
;;
 
let view_prf_of_stm soid = 
 view_open_prf soid (stm_to_prf soid)
;;

let view_new_prf_of_stm soid = 
 view_open_prf soid (stm_to_new_prf soid)
;;

let view_open oid =
  if ((`STM` = (object_state_kind oid)) ? false) then
  view_prf_of_stm oid
  else
  ((let v = view_of_oid oid in
   view_raise v;
   v)
   ?
   (let v = new_view false in
   view_associate_object oid v ;
   view_open_window v;
   view_init_point v;
   v))
;;

 
let view_open_stm oid =
 let vwoid = stm_to_prf oid in
 (let v = view_of_oid vwoid in
  view_raise v
  ; v
   )
 ?
 (let v = new_view false in
   view_associate_object vwoid v ;
   view_open_window v;
   view_init_point v;
   % view_set_oid_list v [oid; poid]; %
 v)
;;

let view_ephemeral oid term =
 (let v = new_view false in
    view_associate_object_lite oid v ;
    view_open_window v;
   view_set_term term true v;
   view_init_point v;
 v)
;;

letref view_show_wrap_cache = inl () : unit + term;;

let view_show_wrap_term () =
  outr
   (if (isr view_show_wrap_cache)
       then view_show_wrap_cache
       else view_show_wrap_cache := 
	     (inr ( (term_lookup (descendent_s [`system`; `support`; `templates`; `view show buttons`]))
		  ? (term_lookup (descendent_s [`system`; `support`; `show`; `templates`; `view show buttons`]))
		  ? failwith `view_show_wrap`
                  )))
;;

let view_show_wrap t = 
 (replace_term 2 (view_show_wrap_term()) t)
 ? t
;;


let view_ap f v = view_set_term (f (source_reduce (view_term_q v) ``EditEphemeral``)) true v;; 


let view_show term =
 let v = new_view false in
   view_associate_object (dummy_object_id ()) v;
   view_set_term (view_show_wrap term) true v;
   view_open_window v;
   view_init_point v;
 v
;;

let view_showd name t = view_show t; ();;

letref view_show_eval_wrap_cache = inl () : unit + term;;

let view_show_eval_wrap_term () =
  outr
   (if (isr view_show_eval_wrap_cache)
       then view_show_eval_wrap_cache
       else view_show_eval_wrap_cache := 
	     (inr ( (term_lookup (descendent_s [`system`; `support`; `templates`; `view show eval buttons`]))
		  ? (term_lookup (descendent_s [`system`; `support`; `show`; `templates`; `view show eval buttons`]))
		  ? failwith `view_show_eval_wrap`
		  )))
;;

let buttons_wrap buts t = replace_term 2 buts t;;
					      
%let view_show_eval_wrap t =
  (buttons_wrap (view_show_eval_wrap_term()) t) ? t
;;
%
let view_show_eval_aux term =
 let v = new_view false in
   view_associate_object (dummy_object_id ()) v;
   view_set_term term true v;
   view_open_window v;
   view_init_point v;
 v
;;
  
let view_show_eval t =
  view_show_eval_aux  ((buttons_wrap (view_show_eval_wrap_term()) t) ? t)
;;  
let view_show_eval_but buts t =
  view_show_eval_aux (buttons_wrap buts t)
;;  

let view_show_aux geo title term =
 let v = new_view false in
   view_associate_object (dummy_object_id ()) v;
   view_set_term term true v;
   set_view_title title v;
   set_view_geo geo v;
   view_open_window v;
   view_init_point v;  v
;;

letref edit_error_view = (inl ()) : unit + view;;

let show_edit_error term =
 let v = if ((isr edit_error_view) & (view_window_open_p (outr edit_error_view)))
	    then outr edit_error_view
	    else outr (edit_error_view := inr (view_show ivoid_term))
  in

   view_set_term term true v;
   view_init_point v;

   ()
;;	    


let local_eval_wrap_show = view_show_wrap o local_eval;;
   
let edit_indirect_p oid = 
  directory_p oid
  or ((object_state_property `EDITREFRESH` oid; ostate_active_p oid)  ? false)
;;

let edit_put_check oid =
   if (edit_indirect_p oid) then failwith `EditPutFail`
   ; ()
;;
 
% need force_put for dynamic objects which strip's computed values (save-dynamic-term), %

let iint_term i =
 if (i < 0) 
    then make_term (`!negative`, nil) [[],inatural_term (abs i)]
    else inatural_term i
;; 
  

let view_put v =
 let oid = (view_object_q v) in
  edit_put_check oid;
  put_term_and_properties
    oid
    (view_term_q v)
    [ `GEOMETRY` , map_to_ilist iint_term icons_op (view_window_geometry_q v)
    ; `IMPLICIT` , map_to_ilist ivariable_term icons_op (view_implicit_q v)]
;;    

let view_put_term v =
 let oid = (view_object_q v) in
  edit_put_check oid;
  put_term oid (view_term_q v)
;;    

let view_put_geometry v =
  put_property
    (view_object_q v)
    `GEOMETRY`  
    (map_to_ilist iint_term icons_op (view_window_geometry_q v))
;;    

let view_put_implicit v =
  put_property
    (view_object_q v)
    `IMPLICIT` 
     (map_to_ilist ivariable_term icons_op (view_implicit_q v))
;;    

let view_point_disp v =
 let (oid, pos) = view_dform_q point_tag v in
   view_open oid
   ; ()
;;

let view_point_abs v =
   let term = edit_term_at_label point_tag false v in
   tty_print "new view_abs";
   let oid = (abstraction_obid_of_term term ? last_oid term) in
     oed_eterm_stack_push (ioid_term oid);
     view_open oid
     ; ()
;;

let view_point_oid v =
 let oid = view_object_id_q point_tag v in
   oed_eterm_stack_push (ioid_term oid);
   view_open oid
   ; ()
;;

let view_point_stm v =
 let oid = view_object_id_q point_tag v in
   view_open_stm oid
   ; ()
;;

let view_push_point_oid v =
 let oid = view_object_id_q point_tag v in
   oed_eterm_stack_push (ioid_term oid)
   ; ()
;;

let view_push_view_oid v =
 let oid = view_object_q v in
   oed_eterm_stack_push (ioid_term oid)
   ; ()
;;

% force :: ie activate_f %
let view_activate v =
    activate (view_object_q v)
;;    

letref cursearch = (\b v.()) : bool -> view -> unit;;

letref SEARCH_MODE = false;;
let set_search_mode v = SEARCH_MODE := true;;
  

let set_search_pattern s =
 cursearch := make_edit_search s
; ()
;;

let set_term_search_predicate f =
 cursearch := make_edit_term_search f
;; 

let view_set_search_pattern v =
 let s = first_string (oed_term_stack_peek 1) in
   SEARCH_MODE := false;
   %tty_print ("set_search_pattern :" J s J ":");%
   set_search_pattern s
;;


let oed_ml_command_primitives () = 
 map define_ml_edit_command 
  [ `quit`, view_quit
  ; `save`, view_put
  ; `verify`, view_activate
  ; `view-disp`, view_point_disp
  ; `view-abs`, view_point_abs
  ; `view-search-forwards`, (\v. cursearch true v)
  ; `view-search-backwards`, (\v. cursearch false v)
  ; `view-object-id-stm`, view_point_stm
  ; `view-object-id`, view_point_oid
  ; `view-push-object-id`, view_push_point_oid
  ; `view-push-view-object-id`, view_push_view_oid
  ; `put-term`, view_put_term
  ; `put-geometry`, view_put_geometry
  ; `put-implicit`, view_put_implicit
  ; `toggle-macro-debug`, (\v. toggle_macro_debug())
  ; `toggle-process-break`, (\v. (toggle_process_break(); ()))
  ; `cmd`, (\v. ((view_open (descendent_s ``system pui MLtoploop toploop``)); ()))

  % ; `view-toploop`, (\v. (view_toploop(); ())) %
  ]
;;



let oed_putedesc ioid =
 putedesc (first_oid_of_term ioid)
;;

let oed_putldesc ioid =
 putldesc (first_oid_of_term ioid)
;;

let oed_putrdesc ioid =
 putrdesc (first_oid_of_term ioid)
;;

let oed_putrledesc ioid =
 putrledesc (first_oid_of_term ioid)
;;


let oed_putmllang ioid =
  putmllang (first_oid_of_term ioid)
;;

let iml_cmd_term term =
  make_term (`!ml_cmd`,nil) [[],term; [],ivoid_term]
;;

let null_cmd = itext_term "";;

let oed_term_stack_get_oid () =
 first_oid_of_term (oed_eterm_stack_peek ())
;;

let topoid () =  first_oid_of_term (oed_term_stack_peek 1);;

%
;;;;	
;;;;	nuprl		: Paul's original.
;;;;	stui-buttons	: some of Stuarts stuff but modified.
;;;;	oed		: general purpose eval and point motion
;;;;	navigator	: 
;;;;	
%			     
let pui_macro_files () =
 [ (make_filename (system_path_prefix ()) ["library"; "macros"] "nuprl" "macro")
 ; (make_filename (system_path_prefix ()) ["library"; "macros"] "oed" "macro")
 ; (make_filename (system_path_prefix ()) ["library"; "macros"] "navigator" "macro")
 ; (make_filename (system_path_prefix ()) ["library"; "macros"] "misc" "macro")
 ; (make_filename (system_path_prefix ()) ["library"; "macros"] "proof" "macro")
 ; (make_filename (system_path_prefix ()) ["library"; "macros"] "keys" "macro")
]
;;


%
;;;;	
;;;;	This is not optimal:
;;;;	  twould be better to have booleans not be reset by recompilation.
;;;;	
%

letref pui_macro_files_loaded = false;;
letref mykeys_macro_file_loaded = false;;
letref oed_inited = false;;


let load_pui_macro_files () =
 (mlbreak `load_pui_macro_files`);
 if not pui_macro_files_loaded
     then ( map load_macros (pui_macro_files())
	  ; pui_macro_files_loaded := true
	  ; ((load_macros (make_filename (system_site_prefix ()) nil "edit" "macro")) ? ())
	  ; () )
;;

let top_oed_macro_reset () =
    oed_macro_reset()
  ; oed_inited := false			     
  ; pui_macro_files_loaded := false
  ; mykeys_macro_file_loaded := false
;;


let oed_init () =
  if not oed_inited then			      
  ( oed_inited := true			     
  ; fdl_editor()
  ; top_oed_macro_reset()
  ; oed_ml_command_primitives ()
  ; oed_edit_rehash()
  )	       
;;

let oed_key_init () =
 oed_init ();			     
 oed_edit_library_rehash();
 load_pui_macro_files ();			     

 mykeys_macro_file_loaded := true;
 (load_macros "~/mykeys.macro" ? ())
;;

%
  Need some sort of init hook at end of start_edd which loads mykeys.
%


let oed_rehash () = 
   top_oed_macro_reset()
  ; oed_ml_command_primitives ()
  ; oed_edit_rehash()
  ; load_pui_macro_files ()
  ; mykeys_macro_file_loaded := true
  ; (load_macros "~/mykeys.macro" ? ())
;;

let oed_reset () = 
    oed_x_reset ()
  ; map (\v. view_disassociate_object true v; view_discard v) (views())
  ; fdl_editor ()
;;

let oedmacro = define_macro `NIL` "";;


%
 temp utils for manip edit objs.
%

let migd old new = 
 let oid = (descendent (root `system`) [`pui`; `dforms`; `boot`; old]) in
 let toid = descendent_s ``system pui DForms`` in
   insert_object_id toid new oid;
   put_property oid `DESCRIPTION` nuprl5_edit_description_term;
   activate oid
;;

let make_mig_aux aord srcext desc destext old new = 
 let oid = (descendent (root `system`) [`pui`; aord; srcext; old]) in
 let toid = descendent_s [`system`; `pui`; destext] in
   insert_object_id toid new oid;
   put_property oid `DESCRIPTION` desc;
   activate oid
;;

let make_mig = make_mig_aux `dforms` `boot` nuprl5_edit_description_term;;


let make_dform_boot_mig = make_mig_aux `dforms` `boot` nuprl5_edit_description_term;;
let make_abs_boot_mig  = make_mig_aux `abstractions` `boot` nuprl5_edit_description_term;;

let migma = make_abs_boot_mig `MLtoploop`;;
let migm = make_dform_boot_mig `MLtoploop`;;

let migc = make_mig `Conditions` ;;

let rm_df tok = 
 let toks = (rev (explode tok)) in
  let ntoks =
   (if ((hd toks) = `f` & (hd (tl toks)) = `d` & (hd (tl (tl toks))) = `_`)
      then (rev (tl (tl (tl toks))))
    else (rev toks)) in
    implode (if (hd ntoks) = `i` then (`!` . (tl ntoks)) else ntoks)
  ;;  


let lvd_mig old new = 
 let oid = (descendent (root `theories`) [`misc`; `dag`; old]) in 
 let toid = descendent_s [`development`; `libview`; `display`] in
   insert_object_id toid new oid;
   put_property oid `DESCRIPTION` nuprl5_edit_description_term;
   activate oid
;;

 

let mign () = 
  let obids = directory_obids (descendent_s ``theories misc dag``) in
    map (\ oid . (if `DISP` = object_state_kind oid  
		    then (let n = name_property oid in lvd_mig n (rm_df n); ())))
        obids
;;

let migna a = 
  let oid = (descendent (descendent_s ``theories misc dag``) [a]) in
    if `ABS` = object_state_kind oid  
	then (let n = name_property oid in

	 let toid = descendent_s [`development`; `libview`; `display`; `abstractions`] in
   insert_object_id toid n oid;
   put_property oid `DESCRIPTION` nuprl5_edit_description_term;
   activate oid)
;;

letref noid = dummy_object_id ();;
letref moid = dummy_object_id ();;
letref toid = dummy_object_id ();;

let sysinit ()  =

 oed_key_init ();

 noid := (descendent (root `system`) ``view filters Navigator``);

 view_open noid; 
 ((view_open (descendent_s [`system`; `pui`; `MLtoploop`; `ML TopLoop`]); ()) ? ());

 ()
;;


let win_init () =
 if (equal_oids_p noid (dummy_object_id()))
     then sysinit ()
     else
       ( if not mykeys_macro_file_loaded then oed_key_init()
       ; view_open noid
       ; ((view_open (descendent_s [`system`; `pui`; `MLtoploop`; `ML TopLoop`]); ()) ? ())
       ; ()
	    )

 %; ((view_open (descendent_s [`system`; `pui`; `MLtoploop`; `Refiner ML TopLoop`]); ()) ? ())%
 ; ((view_open (descendent_s [`doc`; `Getting Started`]); ()) ? ())
; ()
;;


% would be good to mill callers once facility to easily find callers is available %
let replace_subterm = replace_term;;


% conversts all top layer of !ml_tlx_cons !text_cons !ml_text_cons to !ml_text_cons. %
let coerce_text_cons op t =
 letrec aux t =
 let op = operator_of_term t in
    if (  op = itext_cons_op 
       or op = iml_cons_op 
       or op = iml_tlx_cons_op)
     then flatten (map_isexpr_to_list op aux t)
     else [t]
  in map_to_ilist id op (aux t)
;; 

let coerce_to_ml_text_cons = coerce_text_cons iml_cons_op;;
let coerce_to_text_cons = coerce_text_cons itext_cons_op;;
let coerce_to_tlx_cons = coerce_text_cons iml_tlx_cons_op;;

let find_labeled_subterm l t =
 letrec aux bts i =
   if null bts then fail else
   let s = term_of_bterm (hd bts) in
   if ( (is_term `!label` s)
      & (can (find (\p. l = (destruct_token_parameter p))) (parameters_of_term s)))
      then i
      else aux (tl bts) (i + 1) in
      
  aux (bound_terms_of_term t) 1   
;;
  
letref menu_fill_hooks = [] : (tok # ((term # int) -> term)) list;;
letref menu_fill_funcs = [] : ((term # int) -> term) list;;

let add_menu_fill_hook name hook =
 menu_fill_hooks := update_alist menu_fill_hooks name (uncurry hook);
 menu_fill_funcs := map snd menu_fill_hooks;
 ()
;;

letref menu_fill_debug = ivoid_term ;;
let menu_fill v = 
 let t = edit_term_at_label point_tag true v in
 menu_fill_debug := t;
 let i = find_labeled_subterm `MARK` t in
 let tt = source_reduce t ``EDITEPHEMERAL`` in

 ((let m = first_fun menu_fill_funcs (tt,i) in
   edit_paste_term mark_tag m v)
 ? false)
;;

 
let delete_nil_subterm t =
 let ((topid,topparms),topbts) = destruct_term t in
 let s = snd (hd topbts) in
 let (opid, parms), bts = destruct_term s in
  if (null parms) & (null bts) then
     if topid = `!abstraction` then
        make_term (topid,topparms)
	 ((nil, make_term (`!condition`,[string_slot]) nil) . (tl topbts))
     else if topid = `!dform` then
        make_term (topid,topparms)
	 ((nil, make_term (`!placeholder`,[]) nil) . (tl topbts))
     else t
  else t
;;
 
