
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      convert-list-in-place convert-arms-in-place convert-test-exp-in-place
	      strip-straints
	      )))

;;; Contains the functions which convert the abstract syntax tree into the
;;; desired form.  Each variable reference is turned into a reference of a
;;; descriptor.  In processing, we fill in the various fields of the descriptors.


(defun convert-list-in-place (l)
  ;; Replace every member of the list l with its conversion.
  (hack-list #'convert l))

(defun convert-test-exp-in-place (test-exp)
  (convert-list-in-place (cdr test-exp)))

(defun convert-arms-in-place (arms)
  (mapc #'convert-test-exp-in-place (first arms))
  (when (cdr arms)
    (convert-test-exp-in-place (second arms))))


(defun convert (exp)
  ;; Convert the given expression, and return it.
  (cond
    ((abstract-type-mkid-p (node-kind exp))
     exp)
    ((abstract-type-mkid-p (node-kind (constant-to-abstract-type-mkid (node-kind exp))))
     exp)
    (t
      (case (node-kind exp)
	((mk-boolconst mk-intconst mk-fail mk-tokconst mk-empty
		       mk-stringconst)
	 exp )
	(mk-failwith
	 (convert-list-in-place (cdr exp))
	 exp)
	(mk-var
	 (let ((desc (lookup-name (second exp))))
	   (when (is-internal-function desc)
	     (setf (aux-function-all-uses-full-appl desc) nil))
	   (mk-var desc)))
	(mk-dupl
	 (convert-list-in-place (cdr exp))
	 exp)
	(mk-list
	 (convert-list-in-place (cdr exp))
	 (mk-ml-list (cdr exp)))
	(mk-straint (convert (second exp)))
	(mk-appn (convert-appn exp))
	(mk-binop
	 (case (second exp)
	   (|%&| (mland (convert (third exp)) (convert (fourth exp))))
	   (|%or| (mlor (convert (third exp)) (convert (fourth exp))))
	   (otherwise (convert-appn
		       `(mk-appn (mk-appn (mk-var ,(second exp)) ,(third exp)) ,(fourth exp))))))
	(mk-unop
	 (convert-appn `(mk-appn (mk-var ,(second exp)) ,(third exp))))
	(mk-assign
	 (setf (second exp) (conv-vs-in-current-env (second exp)))
	 (setf (third exp) (convert (third exp)))
	 exp)
	(mk-seq
	 (convert-list-in-place (second exp))
	 (setf (third exp) (convert (third exp)))
	 exp )
	(mk-test (convert-test exp))
	(mk-trap (convert-trap exp))
	(mk-abstr (convert-abstr exp))
	(mk-in (convert-in exp))
	(mk-ina (convert-in exp))
	(mk-ind (convert (third exp)))
	((mk-let mk-letrec mk-letref mk-abstype mk-absrectype)
	 (convert-decl exp))
	(otherwise nil)))))



(defun strip-straints (exp)
  ;; Return exp with all top level type constraints stripped off.
  (do ((e exp (second e)))
      ((neq (node-kind e) 'mk-straint) e)))


(defun convert-appn (exp)
  ;; Collect the arguments of the application.
  (do ((exp (strip-straints (second exp)) (strip-straints (second exp)))
       (args (ncons (convert (third exp))) (cons (convert (third exp)) args)))
      ((neq (node-kind exp) 'mk-appn)
       (mk-appn (if (neq (node-kind exp) 'mk-var)
		    (convert exp)
		    ;; This has to be a special case as convert unconditionally bashes
		    ;; the all-uses-full-appl field.
		    (let ((desc (lookup-name (second exp))))
		      (when (and (is-internal-function desc)
				 (< (length args) (ml-function-arity desc)))
			(setf (aux-function-all-uses-full-appl desc) nil))
		      (mk-var desc)))
		args))))

(defvar *conversion-env* nil)

(defmacro with-extended-env (descs &body body)
  `(let ((*conversion-env* (cons ,descs *conversion-env*)))
     ,@body))

(defun lookup-global (id)
  (description-of-mldef (find-mldef id)))

(defun lookup-local (id)
  (dolist (descs *conversion-env*)
    (let ((result (assoc id descs)))
      (when result
	(return-from lookup-local result)))))

(defun lookup-name (id)
  (cond ((lookup-local id))
	((lookup-global id))
	(t (break) (syserror `(lookup-name ,id)))))


(defun walk-varstruct (varstruct f)
  (labels ((walk-vs (vs)
	     (case (node-kind vs)
	       (mk-var
		(funcall f vs))
	       (mk-dupl
		(walk-vs (pair-left vs))
		(walk-vs (pair-right vs)))
	       (mk-binop
		(walk-vs (binop-left vs))
		(walk-vs (binop-right vs)))
	       (mk-list
		(dolist (l (ml-list-list vs))
		  (walk-vs l))))))
    (walk-vs varstruct)))

(defun descs-of-varstructs (varstructs)
  (let ((descriptors nil))
    (dolist (varstruct varstructs)
      (walk-varstruct
	varstruct (prllambda (vs) (push (var-desc vs) descriptors))))
    descriptors))

(defun convert-arms (arms)
  (mapcar
   (prllambda (arm)
	      (mk-arm (first arm)
		      (convert (second arm))
		      (convert (cddr arm))))
   arms))

(defun convert-else (else)
  (mk-else (first else)
	   (convert (cdr else))))

(defun convert-test (exp)
  (mk-test (convert-arms (second exp))
	   (if (not (null (cddr exp)))
	       (convert-else (third exp))
	       '(EMPTY nil))))

(defun convert-trap (exp)
  (let ((binding-id nil)
	(else-exp (fourth exp)))
    (setf else-exp
	  (cond ((null else-exp)
		 '(EMPTY nil))
		((symbolp (first else-exp))
		 (convert-else else-exp))
		(t
		 (setf binding-id (cdr (first else-exp)))
		 (setf (first else-exp) (car (first else-exp)))
		 (with-extended-env (list (make-value :id binding-id :name binding-id))
		   (convert-else else-exp)))))
    (mk-trap (convert (second exp))
	     (convert-arms (third exp))
	     else-exp
	     binding-id)))

(defun convert-abstr (exp)
  ;; Collect the parameter list for the abstraction, then produce the converted
  ;; abstraction form.  The parameter list is reversed.  This is necessitated by
  ;; the runtime closure environment format.
  (do ((params (ncons (convert-varstruct (second exp)))
	       (cons (convert-varstruct (second exp)) params))
       (exp (strip-straints (third exp)) (strip-straints (third exp))))
      ((neq (node-kind exp) 'mk-abstr)
       (with-extended-env (descs-of-varstructs params)
	 (mk-abstr (convert exp) params)))))


(defun convert-top-level-of-abstr (exp)
  ;; Process only the parameters of the abstraction.  This is necessary for
  ;; implementing recursion.  The order of the collected parameters is the
  ;; reverse of that in which they are processed.
  (do ((params (ncons (convert-varstruct (second exp)))
	       (cons (convert-varstruct (second exp)) params))
       (exp (strip-straints (third exp)) (strip-straints (third exp))))
      ((neq (node-kind exp) 'mk-abstr)
       (mk-abstr exp params))))

(defun convert-body-of-abstr (abstr)
  ;; Convert the body of the given abstraction in an environment given
  ;; by the descs field of the abstraction.  Used in implementing recursion.
  (with-extended-env (descs-of-varstructs  (abstr-params abstr))
    (setf (cadr abstr)			; (abstr-body abstr)
	  (convert (abstr-body abstr)))
    abstr))
  
(defun convert-in (exp)
  ;; Convert the body of the in expression in an environment in which the
  ;; names defined in the declaration of the in expression are available.
  (let ((decl (convert-decl (second exp))))
    (with-extended-env (descs-of-varstructs (decl-varstructs decl))
      (mk-in decl (convert (third exp))))))

(defun convert-decl (decl)
  (setf decl (strip-straints decl))
  (case (node-kind decl)
    ((mk-abstype mk-absrectype)
     (with-extended-env (mk-isomorphism-descriptors (second decl))
       (convert-decl (third decl))))
    (otherwise
     (multiple-value-bind (varstructs values)
	 (multiple-value-call
	   (case (node-kind decl)
	     (mk-let #'process-let)
	     (mk-letrec #'process-letrec)
	     (mk-letref #'process-letref))
	   (chop-vs-and-value (second decl) (third decl)))
       (mk-decl (node-kind decl) varstructs values)))))

(defun mk-isomorphism-descriptors (type-defs)
  ;; Returns a list of descriptors for the absract type isomorphisms for the
  ;; types defined in type-defs.
  (mapcan
    (prllambda (type-def)
      (list*
	(mk-isom-desc (concat '|abs_| (first type-def)))
	(mk-isom-desc (concat '|rep_| (first type-def)))
	nil))
    type-defs))

(defun process-let (varstructs values)
  ;; Convert each of the varstructs and values.
  (dolists ((vs varstructs)
	    (val values))
	   ((conv-varstructs nil)
	    (conv-values nil))
	   (values (nreverse conv-varstructs) (nreverse conv-values))
    (let ((cvs (convert-varstruct vs))
	  (cval (convert val)))
      (when (and (eql (node-kind cval) 'mk-abstr)
		 (eql (node-kind cvs) 'mk-var))
	;; The function value is to be associated with a name.  Replace the
	;; descriptor with a function descriptor.
	(let ((fdesc (mk-function-desc
		       (desc-id (var-desc cvs))
		       (length (abstr-params cval)))))
	  (setf (cadr cvs) fdesc)))	; (var-desc cvs)
      (push cvs conv-varstructs)
      (push cval conv-values))))

(defun process-letrec (varstructs values)
  ;; We make two passes over the arguments.  In the first pass, we
  ;; define the descriptors for each varstruct .  In the second pass we
  ;; convert each of the value components in an environment in which the
  ;; descriptors are visible.
  (multiple-value-bind (nvarstructs nvalues)
      (letrec-define-descriptors varstructs values)
    (with-extended-env (descs-of-varstructs nvarstructs)
      (letrec-convert-values nvalues)
      (values nvarstructs nvalues))))

(defun letrec-define-descriptors (varstructs values)
  (dolists ((vs varstructs)
	    (val values))
	   ((conv-varstructs nil)
	    (conv-values nil))
	   (values conv-varstructs conv-values)
    (let ((id (find-id-for-abstr-match vs))
	  (top (convert-top-level-of-abstr val)))
      (when id
	(push top conv-values)
	(push (mk-var (mk-function-desc id (length (abstr-params top))))
	      conv-varstructs)))))

(defun letrec-convert-values (values)
  (hack-list
    (prllambda (val) (if (eql (node-kind val) 'mk-abstr) (convert-body-of-abstr val)))
    values))

(defun process-letref (varstructs values)
  (hack-list (prllambda (vs) (convert-varstruct vs)) varstructs)
  (hack-list #'convert values)
  (values varstructs values))

(defun general-varstruct-converter (varstruct var-action)
  (labels ((conv-vs (vs)
	     (case (node-kind vs)
	       (mk-empty vs)
	       (mk-var
		(setf (cadr vs)		; (var-desc vs)
		      (funcall var-action (second vs)))
		vs)
	       (mk-straint (conv-vs (second vs)))
	       (mk-dupl (conv-varstructs (cdr vs)) vs)
	       (mk-binop (conv-varstructs (cddr vs)) vs)
	       (mk-list
		(conv-varstructs (cdr vs))
		(mk-ml-list (cdr vs)))
	       (otherwise (syserror (cons vs '(bad varstruct))))))
	   (conv-varstructs (varstructs)
	     (hack-list #'conv-vs varstructs)))
    (conv-vs varstruct)))

(defun convert-varstruct (varstruct)
  (general-varstruct-converter
    varstruct
    #'mk-value-desc))

(defun conv-vs-in-current-env (varstruct)
  (general-varstruct-converter
    varstruct
    #'lookup-name))

(defun find-id-for-abstr-match (vs)
  ;; The unconverted varstruct vs is known to match a function value, and thus
  ;; must either define an identifier or be the empty varstruct.  In the first case
  ;; return the identifier, and in the second return nil.
  (setf vs (strip-straints vs))
  (case (node-kind vs)
    (mk-empty nil)
    (mk-var (second vs))
    (otherwise (syserror (cons vs '(bad varstruct for abstraction))))))

(defun chop-vs-and-value (vs val)
  ;; Returns as values a list of varstructs and a list of values.  Each corresponding
  ;; varstruct and value are subtrees of vs and val, and occur in corresponding positions
  ;; in vs and val.  The order of the elements of the result lists is the order in which
  ;; they occur in an in-order traversal of vs.  The arguments may be unconverted.
  (let ((vs-result nil)
	(val-result nil))
    (labels ((chop (vs val)
	       (setf val (strip-straints val))
	       (setf vs (strip-straints vs))
	       (case (node-kind vs)
		 ((mk-empty mk-var) (add-pair vs val))
		 (mk-dupl
		  (if (eql (node-kind val) 'mk-dupl)
		      (mapc #'chop (cdr vs) (cdr val))
		      (add-pair vs val)))
		 (mk-binop
		  (cond ((eql (node-kind val) 'mk-binop)
			 (mapc #'chop (cddr vs) (cddr val)))
			((and (eql (second vs) '|.|)
			      (eql (node-kind val) 'mk-list)
			      (not (null (cdr val))))
			 (chop (third vs) (second val))
			 (chop (fourth vs) (cons 'mk-list (cddr val))))
			(t (add-pair vs val))))
		 (mk-list
		  (let ((lval (canonicalize-list val)))
		    (if (and (neq lval 'non-list)
			     (= (length (cdr vs)) (length lval)))
			(mapc #'chop (cdr vs) lval)
			(add-pair vs val))))
		 (otherwise (syserror (cons vs '(bad varstruct))))))
	     (add-pair (vs val)
	       (push vs vs-result)
	       (push val val-result)))
      (chop vs val)
      (values (nreverse vs-result) (nreverse val-result)))))


;;; This function won't compile in some versions of Franz lisp.  
;;; (defun canonicalize-list (l) 'non-list) will do in its place
;;; (although it's not the best solution).
(defun canonicalize-list (l)
  ;; If the expression l represents a list whose top level structure is known,
  ;; return the list.  Otherwise return the atom 'non-list.
  (block canonical-list
    (labels ((can-list (l)
	       (case (node-kind l)
		 (mk-straint (can-list (second l)))
		 (mk-list (cdr l))
		 (mk-binop
		  (if (eql (second l) '|.|)
		      (if (is-nil (fourth l))
			  (ncons (third l))
			  (cons (third l) (can-list (fourth l))))
		      (return-from canonical-list 'non-list)))
		 (otherwise (return-from canonical-list 'non-list))))
	     (is-nil (v)
	       (case (node-kind v)
		 (mk-straint (is-nil (second v)))
		 (mk-list (null (cdr v)))
		 (mk-var (eql (second v) '|nil|))
		 (otherwise nil))))
      (can-list l))))
