
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; -docs- (mod com prfcmd)
;;;;	
;;;;	Refiner : 
;;;;	
;;;;	  - abstractions
;;;;	  - statements
;;;;	  - proofs
;;;;	  - rules
;;;;	  - comments
;;;;	  - code
;;;;
;;;;  -page-
;;;;	
;;;;	Dependencies :
;;;;	  - Statements reference proofs.
;;;;	  - Proofs reference
;;;;	     * statements as lemmas.
;;;;	     * proofs as extracts.
;;;;	     * abstractions via expansion.
;;;;	     * code via eval.
;;;;	  - Comments can be used to group dependencies so that references may be 
;;;;	    collected more concisely.
;;;;	
;;;;	References should not be made unless object is active.
;;;;	
;;;;	The Library is used to store data, activation states,
;;;;	and dependencies of objects. A complete proof can be assembled
;;;;	from diverse inf trees within the library.
;;;;
;;;;  -page-
;;;;
;;;;	Security : It is necessary that proofs be securely identified with an instance
;;;;	  of a refiner. It is also necessary that editors have access to a 
;;;;	  visible proof. It is convenient if editors can interact directly with
;;;;	  a refiner.  
;;;;
;;;;	The literature needs to be reviewed to determine the best solution for these
;;;;	requirements. Security issues are addressed by CORBA and it may be as easy
;;;;	as using some security component on a CORBA bus.
;;;;	
;;;;  -page-
;;;;	
;;;;	Proofs:
;;;;
;;;;	Without loss of generality, we require all top-level calls to the
;;;;	refiner to be tactic calls. This simplifies some assumptions in the
;;;;	implementation and recognizes that primitive top-level calls are
;;;;	unusual.
;;;;	
;;;;	Recursive invocation of the refiner, ie having a tactic call the refiner
;;;;	with a tactic rule, is actively supported.
;;;;
;;;;  -page-
;;;;
;;;;	Some examples of recursive proof structures:
;;;;
;;;;	Arrows are interiors.
;;;;	Lines are children.
;;;;	p(X) nodes mean primitive refinements of tactic X.
;;;;	o is unrefined leaf.
;;;;	Capitals are tactics.
;;;;	"T" means recusive call of tactic T.
;;;;
;;;;
;;;;	T = A :
;;;;
;;;;        T ---> p(A)
;;;;        |      |
;;;;        o      o
;;;;              
;;;;
;;;;	T = "A" :
;;;;
;;;;        T ---> A ---> p(A) 
;;;;        |      |      |
;;;;        o      o      o
;;;;
;;;;
;;;;	T = A THEN B
;;;;
;;;;        T ---> p(T)
;;;;        |      |
;;;;        o      o
;;;;              
;;;;              
;;;;	T = A THEN "B"
;;;;
;;;;        T ---> p(A) 
;;;;        |      |          
;;;;        o      B --> p(B) 
;;;;               |     |   
;;;;               o     o   
;;;;              
;;;;
;;;;	T = "A" THEN "B"
;;;;
;;;;        T ---> A ----------> p(A)
;;;;        |      |             |
;;;;        o      B --> p(B)    o
;;;;               |     |   
;;;;               o     o  
;;;;
;;;;  -page-
;;;;
;;;;	Inference Tree:
;;;;
;;;;	An inference tree contains :
;;;;	  - normal proof structure, ie goal, tactic, and subgoals.
;;;;	  - any information in an expanded proof which is deemed worthy of user
;;;;	    inspection.
;;;;	     * space and time stats, etc.
;;;;	  - any information in the proof required to maintain the library
;;;;	    representation of the proof.
;;;;	     * dependencies, etc.
;;;;	  - in the library domain, inference trees are decorated with object
;;;;	    attributes.
;;;;	     * preferred window sizes, etc.
;;;;
;;;;  
;;;;	Inference trees play two roles:
;;;;	 * communication with refiner.
;;;;	 * format for saving inference tree to disk.
;;;;
;;;;	A uniform syntax is presented which meets the requirements of both these
;;;;	roles.
;;;;
;;;;	An proof is a two dimensional recursive structure.
;;;;
;;;;	<inf-tree>	: INF_TREE (<goal> <node> <inf-tree> list)
;;;;	<node>		: NODE(<rule> <inf-tree>)
;;;;
;;;;
;;;;	Definitions: 
;;;;
;;;;	 * Exterior Proof : the outer one dimensional recursive inf-tree
;;;;	    structure, ie an inf-tree in the above syntax if you omit the
;;;;	    inf-tree in the node.
;;;;	 * Interior Proof	: an inf-tree  rooted in a node refinement.
;;;;	 * Frontier		: the list of unproven leafs of an inf-tree.
;;;;
;;;;
;;;;	Observations :
;;;;
;;;;	* An interior proof is also an exterior proof. The outermost exterior
;;;;	  proof is not an interior proof.
;;;;	* A proof contains a lot of shared structure:
;;;;	  - The goal of an interior proof is identical to the
;;;;	    the goal of the exterior proof directly containing interior proof.
;;;;	  - the the goals of the children of an exterior proof are identical to the
;;;;	    goals of the frontier of the interior proof of the node of the exterior
;;;;	    proof.
;;;;
;;;;  -page-
;;;;
;;;;	We can elaborate <node> to contain extract and dependency information
;;;;
;;;;	<node>		: TOP       (<dependendencies> <inf-tree> <tactic>)
;;;;			| ABBREV    (<dependendencies> <extract> <tactic>)
;;;;			| PRIMITIVE (<dependendencies> <extract>)
;;;;			| UNPROVEN
;;;;
;;;;
;;;;	An abbrev node is a top node whose inf-tree has been abbreviated to be
;;;;	the extract.
;;;;
;;;;	In an inf-tree which represents a proof, no nodes of the outermost
;;;;	external tree will be primitive. This is a consequence of the fact that
;;;;	we require all top-level calls to the refiner to be tactic calls.
;;;;
;;;;	There is no requirement that dependences be maintained. However if a
;;;;	node does contain dependencies, then they must be cumulative.  Ie
;;;;	dependenecies in top are union of all dependendencies in inf-tree of
;;;;	top. Though there is no restriction on inclusion of dependencies, it is
;;;;	suggested that dependencies only occur on the leaves or on the outermost
;;;;	top nodes. The former contains more finely grained information however
;;;;	the latter will be more efficient wrt space. A tree which contains
;;;;	dependencies in both tops and included nodes is redundant.
;;;;
;;;;	Dependencies include :
;;;;	 * abstractions referenced to produce tactic text.
;;;;	 * abstractions referenced by computation, evaluation, or tactics.
;;;;	 * lemmas referenced in primitive rules or tactics.
;;;;	 * extracts referenced by computation, evaluation, or tactics.
;;;;
;;;;	We can elaborate <dependencies>:
;;;;
;;;;	<dependencies>	: DEPENDENCIES
;;;;			| DEPENDENCIES(<dependency{parse}> list
;;;;				       <dependency{refine}> list)
;;;;
;;;;	The degenerate DEPENDENCIES implies that dependencies are not
;;;;	maintained, not that the dependency lists are empty.
;;;;
;;;;	The parse dependencies are the abstractions referenced to produce tactic
;;;;	text.  The refine dependencies include all others. If necessary, the
;;;;	type of reference can be deduced by examining the object referenced.
;;;;
;;;;	TODO : need to update comments wrt events.
;;;;	TODO : <dependencies> should be dependency-store, allow for random types
;;;;	TODO : not just parse and refine.
;;;;	TODO : 
;;;;	TODO : 
;;;;	TODO : 
;;;;	TODO : 
;;;;
;;;;  -page-
;;;;	
;;;;	Simlarly there is no requirement to maintain extracts, thus we elaborate the
;;;;	<extract> syntax:
;;;;
;;;;	<extract>	: EXTRACT (<term>)
;;;;			| EXTRACT
;;;;
;;;;  -page-
;;;;	
;;;;	It is possible for the tactics to annotate a proof node during refinement.
;;;;	We can elaborate the <goal> syntax to allow for preservation of such
;;;;	annotations.  Also, recall the observation that goals are shared.  We can
;;;;	elaborate the <goal> syntax such that identical goals need not be repeated:
;;;;
;;;;
;;;;	<goal>		: GOAL (<sequent> <annotations>)
;;;;			| GOAL
;;;;
;;;;	Thus, the null goal would be an indication to use an alternate source for the
;;;;	goal.  One must take care that only semantically sensible null goals are
;;;;	included.
;;;;	
;;;;  -page-
;;;;	
;;;;	The ability to access each inf-tree through the object interface is supplied.
;;;;	Also the ability for the refiner to annotate the inf-tree is supplied. One typical
;;;;	annotation would be timing and space usage statistics during refinement.
;;;;	Both these goals are accomplished by extending the <inf-tree> syntax:
;;;;
;;;;	<inf-tree>	: INF_TREE(<goal> <node> <inf-tree> list <annotations>)
;;;;
;;;;  -page-
;;;;
;;;;	In summary, we arrive at the following abstract syntax:
;;;;
;;;;	<inf-tree>	: INF_TREE(<goal> <node> <inf-tree> list <annotations>)
;;;;
;;;;	<node>		: TOP       (<dependencies> <inf-tree> <tactic>)
;;;;			| ABBREV    (<dependencies> <extract> <tactic>)
;;;;			| PRIMITIVE (<dependencies> <extract>)
;;;;			| UNPROVEN
;;;;
;;;;	<dependencies>	: DEPENDENCIES
;;;;			| TEXT_DEPENDENCIES(<dependency> list)
;;;;			| ABSTRACTION_DEPENDENCIES(<dependency> list)
;;;;			| STATEMENT_DEPENDENCIES(<dependency> list)
;;;;			| PROOF_DEPENDENCIES(<dependency> list)
;;;;
;;;;	<goal>		: GOAL (<sequent> <annotations>)
;;;;			| GOAL
;;;;
;;;;	<extract>	: EXTRACT (<term>)
;;;;			| EXTRACT
;;;;
;;;;  -page-
;;;;
;;;;	We can elaborate this syntax to encode the semantic requirements for sharing <goal>'s
;;;;	and barring primitive nodes from outermost exterior trees:
;;;;
;;;;	<inf-tree>	: <ex-inf-tree>
;;;;
;;;;	<ex-inf-tree>	: INF_TREE(<goal> <ex-node> <ex-inf-tree> list <annotations>)
;;;;
;;;;	<ex-node>	: TOP       (<dependendencies> <top-inf-tree> <tactic>)
;;;;			| ABBREV    (<dependendencies> <extract> <tactic>)
;;;;			| UNPROVEN
;;;;
;;;;	<top-inf-tree>	: INF_TREE(GOAL <in-node> <in-inf-tree> list <annotations>)
;;;;
;;;;	<in-inf-tree>	: INF_TREE(<goal> <in-node> <in-inf-tree> list <annotations>)
;;;;			| INF_TREE(GOAL UNPROVEN <in-inf-tree> list <annotations>)
;;;;
;;;;	<in-node>	: TOP       (<dependendencies> <in-inf-tree> <tactic>)
;;;;			| ABBREV    (<dependendencies> <extract> <tactic>)
;;;;			| PRIMITIVE (<dependendencies> <extract>)
;;;;			| UNPROVEN
;;;;
;;;;	<dependencies>	: DEPENDENCIES
;;;;
;;;;	<goal>		: GOAL (<sequent> <annotations>)
;;;;
;;;;	<extract>	: EXTRACT (<term>)
;;;;			| EXTRACT
;;;;
;;;;
;;;;	An inf-tree conforming to the preceding syntax can be easily transformed
;;;;	to an inf-tree with no degenerate goals by a procedure which recognizes
;;;;	the sharing.
;;;;
;;;;  -page-
;;;;	
;;;;	TODO : hiding
;;;;	elaborate so that subtrees can be squashed into interior trees in order to hide
;;;;	steps. (or for describing search or interpreted RefineTrees).
;;;;	  - when to unhide?
;;;;	  - how to recover hiding when copying via replay?
;;;;	  * maybe just a predicate which tests tactic term for replayability
;;;;	    but then more difficult to treat uniformly 
;;;;	    but also not uniform ie varying causes of hiding and reasons to unhide, ie search can be replayed, 
;;;;	     RefineTrees probably not, squashed exterior trees ???.
;;;;	inf_hide{<kind>:t}(<dependencies> <inf-tree> <source>)
;;;;	 source : <tactic{search}> | <RefineTree> or Squash{<depth>:n}()
;;;;	
;;;;	
;;;; -doct- (mod com data)
;;;;  -page-
;;;;
;;;;	The smaller abstract syntax as terms:
;;;;
;;;;	<inf-tree>	: !inf_tree(<goal>; <node>;
;;;;				    <inf-tree> !inf_tree_cons list;
;;;;				    <annotations>)
;;;;
;;;;	<node>		: !inf_top{<direct>:b}(<dependencies>; <inf-tree>; <tactic>)
;;;;			| !inf_abbrev{<direct>:b}(<dependencies>; <extract>; <tactic>)
;;;;			| !inf_primitive(<dependencies>; <extract>)
;;;;			| !inf_unrefined()
;;;;	  * <tactic> is reduced, source can be put in annotations.
;;;;
;;;;	<dependencies>	: <dependencies-term>
;;;;
;;;;	<goal>		: !inf_goal(<sequent>; <annotations>)
;;;;			| !inf_goal()
;;;;	<extract>	: !inf_extract(<term>)
;;;;			| !inf_extract()
;;;;
;;;;	<tactic>	: <term>
;;;;	<sequent>	: !inf_sequent{<hidden>:bool}(<type>; <id>.<sequent>)
;;;;			| <term>
;;;;
;;;;	<annotations>	: <term> !annotation_cons list
;;;;
;;;;	Though this syntax is documented in the COM module, except for term
;;;;	constructors there is no related code in the COM module. REF contains
;;;;	code to export expanded proofs to inf-tree syntax. LIB contains code to
;;;;	import/export inf-tree syntax to objects.
;;;;
;;;;	<inf-step>	: !inf_tree(!inf_goal(<sequent>; <annotations>);
;;;;			    	    <node>;
;;;;	
;;;;
;;;; -doct- (mod com)
;;;;  -page-
;;;;
;;;;	TODO:
;;;;
;;;;	Abbreviated inf-tree consisting only of goal/extract/dependencies
;;;;	and outermost annotations(includes tactics).
;;;;
;;;;	All interior annotations are lost and are not recoverable.
;;;;
;;;;	<ab-proof>	: ABBREV-PROOF(<goal> <extract> <dependencies> <ab-inf-tree>)
;;;;
;;;;	<ab-inf-tree>	: AB-INF-TREE(<annotations>  AB-INF-TREE !inf_tree_cons list)
;;;;
;;;;	Possibly can annotations and just include tactic.
;;;;
;;;;  -page-
;;;;
;;;;	<rule-definition>	: <definition>
;;;;	 * REF module uses an extended definition.
;;;;
;;;;	export-rule-definition(<rule-definition>)		: <rule-spec>
;;;;	import-rule-definition(<rule-spec>)			: <rule-definition>
;;;;
;;;;	rule-table (<process-id> <tag> import/export-f???)	: <definition-table{rules}>
;;;;
;;;;	with-rules ((<definition-table{rules}>) &body body) 
;;;;	 ** dynamically binds *rules* variable.
;;;;
;;;; -doce-
;;;;	
;;;;	mention statement and proof broadcasts should return enough dependency info to detect cycles.
;;;;	comment dependencies can be used to pool dependencies and may be necessary also.
;;;;	
;;;;	

;;;;
;;;;	<proof-anno>	: !proof_anno{<tok>:t, <decide>:b, <int>:n}
;;;;				     (!proof_anno_arg{tok:t}
;;;;						     (<arg>) !paa_cons ilist)
;;;;
;;;; OR
;;;;
;;;;	<proof-anno>	: !proof_anno{<tok>:t}
;;;;				     (!inl (<int>);
;;;;				      !proof_anno_arg{tok:t}
;;;;						     (<arg>) !paa_cons ilist)
;;;;			| !proof_anno{<tok>:t}
;;;;				     (!inr ();
;;;;				      !proof_anno_arg{tok:t}
;;;;						     (<arg>) !paa_cons ilist)
;;;;
;;;;	The first is more compact but the second is more representative, you
;;;;	choose (or do something completely different). Keep in mind that these
;;;;	will be visible in proofs.  At proof display time, there will be
;;;;	opportunity for further translation, but it would be convenient if this
;;;;	form were amenable to direct display.  It is for this reason that I
;;;;	chose to use an icons list instead of inserting the !proof_anno_args
;;;;	directly as subterms. Similarly, the first syntax may be a better choice
;;;;	as it allows choice of display form for !proof_anno term depending on
;;;;	bool.
;;;;
;;;;	* It would be possible to have the !proof_annotation_arguments directly as subgoals
;;;;	* instead of an icons list. However, these annotations will be visible when viewing
;;;;	* proof and this representation is directly displayable.

;;;;	Proof annotations may be rule arguments, thus arguments may be
;;;;	marshalled to terms:
;;;;
;;;;	argument_to_term	: argument -> term
;;;;	term_to_argument	: term -> argument
;;;;
;;;;


;;;;	if inf-object is direct member of lib then occurence in inf tree should be as obid.
;;;;	
;;;;	otherwise rolled into inf-tree and refinement should be in tree not in objc.
;;;;	


;;;;	
;;;;	callers should supply goal, if not needed then ignored.
;;;;	but if needed and not supplied then throws off frontier.
;;;;	

;;;;	
;;;;	null goal means pop-frontier???
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;

(defun extract-of-iinf-node (node)

  (cond

    ((iinf-primitive-term-p node)
     (term-to-inf-extract
      (extract-of-iinf-primitive-term node)))

    ((iinf-abbrev-term-p node)
     (term-to-inf-extract
      (extract-of-iinf-abbrev-term node)))

    (t nil)))



;; ((goal # annos) # tactic) tree
(defun iinf-tree-term-to-inf-tree (term frontier-goals &optional goal ephp)
  ;;(setf -term term) (break "itttit")

  ;;(format t "~% term -> tree ~%")(break)
  
  (let ((remaining-frontier frontier-goals))

    (labels
	((visit (term &optional goal)
	   ;;(setf -term term -goal goal -remaining-frontier remaining-frontier) (break "itttit")
	   (let* ((annotations nil)
		  (preprocess-source nil)
		  (node-term (node-of-iinf-tree-term term))
		  (unrefinedp (iinf-unrefined-term-p node-term)
		    )
		  (primp (iinf-primitive-actual-term-p node-term))
		  )

	     ;; objc anno's should not be present as inf-objc-trees are not exported
	     ;; into !inf_trees any longer.

	     (dolist (anno (term-to-annotations (annotations-of-iinf-tree-term term)))
	       (if (iannotation-term-p anno)
		   (cond
		     ((eql 'objc (label-of-iannotation-term anno))
		      (message-emit (warn-message '(!inf_tree objc annotation)
						  anno)))
		     ((eql 'export-preprocess-source (label-of-iannotation-term anno))
		      (setf preprocess-source (new-source (ivoid-term) 'inf))
		      (data-import preprocess-source (term-of-iannotation-term anno))
		      ;;(setf -preprocess-source preprocess-source) (break "iiipps")
		      )
		      
		     (t (push anno annotations)))
		   (raise-error (error-message '(term iinf-tree !annotation not)
					       anno))))

	     (let* ((children (map-isexpr-to-list (children-of-iinf-tree-term term)
						  (iinf-tree-cons-op)
						  #'visit))

		    (step (inf-step (or goal
					(let ((goal-term (goal-of-iinf-tree-term term)))
					  
					  (if (null-iinf-goal-term-p goal-term)

					      ;; RLE TODO remove following after suitable testing.
					      ;; LAL problem here with inv_image
					      
					      (progn (unless remaining-frontier
						       (setf a term b remaining-frontier)
						       (break "term-to-inf-tree"))
						     (pop remaining-frontier))
					      
					      (iinf-goal-term-to-goal goal-term))))))

		    (subgoals (mapcar #'(lambda (inf-tree)
					  (goal-of-inf-step
					   (step-of-inf-objc-r
					    (objc-of-inf-tree inf-tree))))
				      children))
		    
		    (references (unless (or unrefinedp primp)
				  (let ((deps (dependencies-of-iinf-top-term node-term)))
				    (when deps (term-to-environment-dependencies deps))))))
				    
		 
		  
	       ;;(setf s step c children g subgoals) (break "s")
	       (setf (inf-step-info-annotations step) annotations

		     (inf-step-refinement step)
		     (cond
		       ((iinf-top-term-p node-term)
			;; RLE TODO is this right direct-p check?? I think it is safe.
			(make-top-refinement :direct-p (direct-of-iinf-top-term node-term)
					     :subgoals subgoals
					     :references references
					     :tactic (tactic-of-iinf-top-term node-term)
					     :inf-tree (new-inf-tree-proxy (tree-of-iinf-top-term node-term))))

		       ((iinf-primitive-term-p node-term)
			(make-primitive-refinement :direct-p nil
						   :subgoals subgoals 
						   :references references
						   :extract (term-to-inf-extract
							     (extract-of-iinf-primitive-term node-term))))

		       ((iinf-primitive-actual-term-p node-term)
			(make-rule-refinement :direct-p nil
					      :subgoals subgoals 
					      :references references
					      :rule (rule-of-iinf-primitive-actual-term node-term)))

		       ((iinf-abbrev-term-p node-term)
			(make-abbrev-refinement :direct-p (direct-of-iinf-abbrev-term node-term)
						:subgoals subgoals
						:references references
						:tactic (tactic-of-iinf-abbrev-term node-term)
						:extract (term-to-inf-extract
							  (extract-of-iinf-abbrev-term node-term))))
		       (unrefinedp nil)
			 
		       (t (system-error "iinf-tree-term-to-inf-tree"))))

	       (let* ((tactic (unless unrefinedp ; LAL need this test for mp
				(rule-of-inf-step step t)))

		      (objc (if preprocess-source
				(objc-modify-source (object-contents 'inf (or tactic (ivoid-term)))
						    preprocess-source)
				(object-contents 'inf (or tactic (ivoid-term))))))

		 (set-inf-objc-step objc step)
		 ;;(setf -objc objc -step step)
		 (when (iinf-top-term-p node-term)
		   (setf objc (xref-objc-modify-source objc (xrefs-of-inf-step step))))

		 ;;(setf -objc objc -step step) (break "iiii")

		 ;; kludge alert : need binding of inf tree to persist even if transaction fails
		 ;;  and other effects are undone. Not sure if really need persistence, however
		 ;;  if not sneaked it must be completed elsewhere. This unrolling is side
		 ;;  effect of inf-step lookup.
		 (if (or ephp (eql 'rule (type-of-inf-step step)))
		     (inf-objc-tree objc children)
		     (let ((oid (bind-sneak objc)))
		       (inf-tree oid children))))))))

      (prog1 (visit term goal)
	;; RLE TODO remove following after suitable testing.
	(when remaining-frontier
	  (setf -rf remaining-frontier)
	  (format t "~% term-to-inf-tree leftover frontier ")
	  )))))


(defvar *prim-refine-ap* nil)
(defvar *prim-refine-apt* nil)


(defunml (|prim_refine| (eoid goal tac oids))
    (oid -> (term -> (term -> ((object_id list) -> inf_tree))))

  (unless *prim-refine-ap*
    (setf *prim-refine-ap*
	  (funcall #'ml-text "lib_refine_prim")))
  
  (iinf-tree-term-to-inf-tree (funmlcall *prim-refine-ap* goal tac eoid oids) nil))

(defunml (|prim_refinet| (eterm goal tac))
    (term -> (term -> (term -> inf_tree)))

  (unless *prim-refine-apt*
    (setf *prim-refine-apt*
	  (funcall #'ml-text "lib_refine_primt")))
  
  (iinf-tree-term-to-inf-tree (funmlcall *prim-refine-apt* goal tac eterm) nil))

#|
(defun iinf-tree-term-to-inf-tree (term frontier-goals &optional goal)

  (let ((remaining-frontier frontier-goals))

    (labels
	((visit (term &optional goal)
	   (setf -term term -goal goal -remaining-frontier remaining-frontier) (break "itttit")
	   (let* ((annotations nil)
		  (node-term (node-of-iinf-tree-term term))
		  (unrefinedp (iinf-unrefined-term-p node-term))
		  )

	     ;; objc anno's should not be present as inf-objc-trees are not exported
	     ;; into !inf_trees any longer.

	     (dolist (anno (term-to-annotations (annotations-of-iinf-tree-term term)))
	       (if (iannotation-term-p anno)
		   (if (eql 'objc (label-of-iannotation-term anno))
		       (message-emit (warn-message '(!inf_tree objc annotation)
						   anno))
		       (push anno annotations))
		   (raise-error (error-message '(term iinf-tree !annotation not)
					       anno))))


	     (let* ((children (map-isexpr-to-list (children-of-iinf-tree-term term)
						  (iinf-tree-cons-op)
						  #'visit))
		    (step (inf-step (or goal
					(let ((goal-term (goal-of-iinf-tree-term term)))
					  ;;(setf g goal-term) (break)
					  (if (null-iinf-goal-term-p goal-term)
					      ;; RLE TODO remove following after suitable testing.
					      ;; LAL problem here with inv_image
					      (progn (unless remaining-frontier
						       (setf a term b remaining-frontier)
						       (break "term-to-inf-tree"))
						     ;;(setf -a remaining-frontier) (break)   
						     (pop remaining-frontier))
					      (iinf-goal-term-to-goal goal-term))))))
		   

		    (subgoals (mapcar #'(lambda (inf-tree)
					  (goal-of-inf-step
					   (step-of-inf-objc-r
					    (objc-of-inf-tree inf-tree))))
				      children))
		    (references (unless unrefinedp
				  (let ((deps (dependencies-of-iinf-top-term node-term)))
				    (when deps (term-to-environment-dependencies deps))))))
				    
		 
		  
	       ;;(setf s step c children g subgoals) (break "s")
	       (setf (inf-step-info-annotations step) annotations

		     (inf-step-refinement step)
		     (cond
		       ((iinf-top-term-p node-term)
			;; RLE TODO is this right direct-p check?? I think it is safe.
			(make-top-refinement :direct-p (direct-of-iinf-top-term node-term)
					     :subgoals subgoals
					     :references references
					     :tactic (tactic-of-iinf-top-term node-term)
					     :inf-tree (new-inf-tree-proxy (tree-of-iinf-top-term node-term))))
		       ((iinf-primitive-term-p node-term)
			(make-primitive-refinement :direct-p nil
						   :subgoals subgoals 
						   :references references
						   :extract (term-to-inf-extract
							     (extract-of-iinf-primitive-term node-term))))
		       ((iinf-abbrev-term-p node-term)
			(make-abbrev-refinement :direct-p (direct-of-iinf-abbrev-term node-term)
						:subgoals subgoals
						:references references
						:tactic (tactic-of-iinf-abbrev-term node-term)
						:extract (term-to-inf-extract
							  (extract-of-iinf-abbrev-term node-term))))
		       (unrefinedp nil)
			 
		       (t (system-error "iinf-tree-term-to-inf-tree"))))
	       
	       (let ((objc (object-contents 'inf (tactic-of-inf-step step))))
	         (set-inf-objc-step objc step)

		 ;; kludge alert : need binding of inf tree to persist even if transaction fails
		 ;;  and other effects are undone. Not sure if really need persistence, however
		 ;;  if not sneaked it must be completed elsewhere. This unrolling is side
		 ;;  effect of inf-step lookup.
	         (let ((oid (bind-sneak objc)))
		   (inf-tree oid children)))))))
      (prog1 (visit term goal)
	;; RLE TODO remove following after suitable testing.
	(when remaining-frontier
	  (setf rf remaining-frontier)
	  (format t "~% term-to-inf-tree leftover frontier ")
	  )))))
		
|#

;;;
;;;	inf-tree-to-term : orphans, replace by inf-objc-tree-to-term
;;;
;;;	remain as someday may want capability of exporting inf-objc-tree's to !inf_tree's.

(defun step-to-node-term (step)
  (case (type-of-inf-step step)
	(unrefined (iinf-unrefined-term))
	(top (iinf-top-term (direct-inf-step-p step)
			    (environment-dependencies-to-term (references-of-inf-step step))
			    (iinf-tree-of-inf-step step)
			    (tactic-of-inf-step step)))

	(primitive (iinf-primitive-term (environment-dependencies-to-term (references-of-inf-step step))
					(inf-extract-to-term (extract-of-inf-step step))))
	(abbrev	(iinf-abbrev-term nil
				  (environment-dependencies-to-term (references-of-inf-step step))
				  (inf-extract-to-term (extract-of-inf-step step))
				  (tactic-of-inf-step step)))))


(define-primitive |!export_preprocess_source| () (data))

(defun inf-tree-to-iinf-tree-term (inf-tree interior-p root-p)

  (format t "~% tree -> term ~%")
  (let* ((objc (objc-of-inf-tree inf-tree))
	 (step (step-of-inf-objc-r objc)))

    (iinf-tree-term

     (let ((frontier-p (not (refined-inf-step-p step))))
       (cond
	 ((and interior-p (or frontier-p root-p))
	  (instantiate-term (iinf-goal-op)))

	 ((and interior-p frontier-p root-p)
	  (raise-error '(inf-tree term interior frontier root)))

	 ((or (and interior-p (not frontier-p) (not root-p))
	      (not interior-p))
	  (goal-to-term (goal-of-inf-step step)))

	 ;; should not be reachable
	 (t (raise-error '(inf-tree term ???)))))

     (step-to-node-term step)

     (map-sexpr-to-isexpr (children-of-inf-tree inf-tree)
			  (iinf-tree-nil-term)
			  #'(lambda (it) (inf-tree-to-iinf-tree-term it interior-p nil)))

     (let ((source (source-of-objc objc)))
       ;;(data-export-aux source (ivoid-term))
       (annotations-to-term
	(cons (iannotation-term 'export-preprocess-source (data-export-aux source (ivoid-term)))
	      (annotations-of-inf-objc-src objc)))))))

;; ?? unused and unwanted AFAIK
(defun inf-objc-to-term (step)
  (iinf-tree-term
   (goal-to-term (goal-of-inf-step step))
   (step-to-node-term step)
   (if (refined-inf-step-p step)
       (progn (break "ic") (map-sexpr-to-isexpr (subgoals-of-inf-step step)
			    (iinf-tree-nil-term)
			    #'(lambda (subgoal)
				(iinf-tree-term (goal-to-term subgoal)
						(iinf-unrefined-term)
						(iinf-tree-nil-term)
						(iannotation-nil-term)))))
       (iinf-tree-nil-term))
   (map-sexpr-to-isexpr (annotations-of-inf-step step)
			(iannotation-nil-term))))


;;;;
;;;;	interior-inf-tree-match-p(<inf-tree> <inf-tree>)	: <bool>
;;;;	 * To match the tree structure (modulo ABBREV),
;;;;	   all goals, all tactic annotations, all tactics (modulo source reduction),
;;;;	   all extracts, and [??all references??] must be identical.
;;;;
;;;;
;;;;
;;;;	goal-match-p(<goal> <goal>)	: <bool>
;;;;	 * either identical or something similar to equal-sequents-p
;;;;
;;;;	inf-tree-match-p(<inf-tree> <inf-tree>)			: <bool>
;;;;	 * t when goals match.
;;;;	 
;;;;	inf-tree (<objc{inf}> <inf-tree> list)		: inf-tree
;;;;	 * checks that subgoals of inf match goals of inf-tree.
;;;;



(defun direct-inf-tree-match-p (a b)
  (let ((step-a (step-of-inf-objc (objc-of-inf-tree a)))
	(step-b (step-of-inf-objc (objc-of-inf-tree b))))

  (if (refined-inf-step-p step-a)
      (and (refined-inf-step-p step-b)
	   (goal-match-p (goal-of-inf-step step-a)
			 (goal-of-inf-step step-b))
	   (forall-p #'direct-inf-tree-match-p
		       (children-of-inf-tree a)
		       (children-of-inf-tree b)))
      (and (not (refined-inf-step-p step-b))
	   (goal-match-p (goal-of-inf-step step-a)
			 (goal-of-inf-step step-b))))))



(defun indirect-inf-tree-match-p (a b)
  (let ((step-a (step-of-inf-objc (objc-of-inf-tree a)))
	(step-b (step-of-inf-objc (objc-of-inf-tree b))))

    (if (refined-inf-step-p step-a)
	(and (refined-inf-step-p step-b)
	     (equal-stamps-p (stamp-of-inf-step step-a)
			     (stamp-of-inf-step step-b))
	     (forall-p #'indirect-inf-tree-match-p
		       (children-of-inf-tree a)
		       (children-of-inf-tree b)))
	(and (not (refined-inf-step-p step-b))
	     (goal-match-p (goal-of-inf-step step-a)
			   (goal-of-inf-step step-b))))))

    
(defun inf-step-modify-inf-tree (step inf-tree)
  (let ((old-inf-tree (inf-tree-of-inf-step step)))
    (unless (direct-inf-tree-match-p old-inf-tree inf-tree)
      (raise-error (error-message '(inf-step modify inf-tree match direct))))
    (unless (indirect-inf-tree-match-p old-inf-tree inf-tree)
      (raise-error (error-message '(inf-step modify inf-tree match indirect))))

    (new-inf-step-inf-tree step inf-tree)))


(defun instantiate-inf-tree (oid children)
  (let* ((objc (oc oid))
	 (inf-step (step-of-inf-objc-r objc)))

    (if (not (refined-inf-step-p inf-step))

	(progn
	  (when children
	    (raise-error (error-message '(inf-tree objc unrefined))))

	  (inf-tree oid nil))

	(progn
	  ;; check uniformity of direct/indirect.
	  (let ((direct-p (direct-inf-step-p inf-step)))
	    (unless (forall-p #'(lambda (c)
				  (or (not (refined-inf-tree-p c))
				      (eql direct-p (direct-inf-tree-p c))))
			      children)
	      (raise-error (error-message '(inf-tree direct indirect mixed) direct-p))))
  
	  ;; check subgoals of inf-step match children 
	  (let ((subgoals (subgoals-of-inf-step inf-step)))
	    ;;(setf a subgoals b children) (break "mt")
	    (when children;;is this children test ok, lal added
	      (unless (and (= (length subgoals) (length children))
			   (forall-p #'(lambda (goal child)
					 (goal-match-p (goal-of-inf-step (step-of-inf-objc
									  (objc-of-inf-tree child)))
						       goal))
				     subgoals
				     children))
		;;(setf a subgoals b children) (break "mt")
	   
		(raise-error (error-message '(inf-tree goals match not))))))

	  (inf-tree oid children)))))


(defun match-inf-trees (l1 l2)
  (let ((l (mapcar #'(lambda (x)
		       (cons x (goal-of-inf-step (step-of-inf-objc-r (objc-of-inf-tree x)))))
		   l2)))
    (labels ((match (tree)
	       (let ((goal (goal-of-inf-step (step-of-inf-objc-r (objc-of-inf-tree tree))))
		      (finds nil))
		      
		       (mapc #'(lambda (x)
				 (when (goal-match-p (cdr x) goal)
				   (setf finds (cons (cons (car x) (if (itree-complete-p (car x)) 0 1)) finds))))
			     l)
		       
		     (let ((found (when finds (car (car (sort finds #'< :key #'cdr))))))
		 (or found tree)))))
      (mapcar #'match l1))))

    
		 
(defun inf-tree-replace-children (t1 t2)
   (let ((children (match-inf-trees (children-of-inf-tree t1) (children-of-inf-tree t2))))
     (inf-tree (oid-of-inf-tree t1) children)))

					     
(defun inf-tree-replace-tree (target tree address &optional children-p)
  ;;(setf -target target -tree tree -address address) (break "itrt")

  (inf-tree-modify target address 
		   #'(lambda (target)
		       (cond
			 ((and (or (unrefined-inf-tree-p target) (direct-inf-tree-p target))
			       (direct-inf-tree-p tree))
			  (unless (goal-match-p (goal-of-inf-step
						 (step-of-inf-objc-r (objc-of-inf-tree target)))
						(goal-of-inf-step
						 (step-of-inf-objc-r (objc-of-inf-tree tree))))
			    
			    (setf g1 (goal-of-inf-step (step-of-inf-objc-r (objc-of-inf-tree target)))
				  g2 (goal-of-inf-step (step-of-inf-objc-r (objc-of-inf-tree tree))))
			    ;;(break)
			    
			    (raise-error (error-message '(inf-tree replace tree match))))
			  
			  (if children-p
			      (inf-tree-replace-children tree target)
			      tree))
			  
			 ((and (not (direct-inf-tree-p target)) (not (direct-inf-tree-p tree)))
			  (unless (indirect-inf-tree-match-p target tree)
			    (raise-error (error-message '(inf-tree replace tree match interior))))

			  (if children-p
			      (inf-tree-replace-children tree target)
			      tree))
			 
			 (t (raise-error (error-message '(inf-tree replace match direct)
							(cons (direct-inf-tree-p target)
							      (direct-inf-tree-p tree)))))))))


(defun inf-tree-replace-tree-unconditional (target tree address)
  (inf-tree-modify target address #'(lambda (target) tree)))

(defun inf-tree-update (source place)
  (let ((oid (oid-of-inf-tree place)))
    (labels ((visit (tree)
	       (let ((ioid (oid-of-inf-tree tree)))
		 (if (equal-oids-p oid ioid)
		     place
		     (let ((hitp nil))
		       (let ((children (mapcar #'(lambda (child)
						   (let ((rchild (visit child)))
						     (when (not (eq child rchild)) (setf hitp t))
						     rchild))
					       (children-of-inf-tree tree))))
			 (if hitp
			     (inf-tree ioid children)
			     tree)))))))
      (visit source)))) 



(defun inf-tree-replace-node (target oid address)
  (inf-tree-modify target address 
		   #'(lambda (target)
		       (let* ((target-objc (objc-of-inf-tree target))
			      (target-step (step-of-inf-objc-r target-objc))
			      (objc (oc oid))
			      (step (step-of-inf-objc-r objc)))
			 
			 (unless (goal-match-p (goal-of-inf-step target-step)
					       (goal-of-inf-step step))
			   (raise-error (error-message '(inf-tree replace node match goal))))

			 (cond
			   ((unrefined-inf-step-p step)
			    (unless (unrefined-inf-step-p target-step)
			      (raise-error (error-message '(inf-tree replace node unrefined))))
			    (inf-tree oid nil))
			  
			   ((not (direct-inf-step-p step))
			    (unless (equal-stamps-p (stamp-of-inf-step target-step)
						    (stamp-of-inf-step step))
			      (raise-error (error-message '(inf-tree replace node indirect))))
			    (inf-tree oid (children-of-inf-tree target)))

			   (t;;(direct-inf-step-p step)
			    ;; (instantiate-inf-tree oid (children-of-inf-tree target))
			    (inf-tree oid (children-of-inf-tree target))
			    ))))))

			  

(defun inf-tree-modify (target address replacement-r)

  (labels ((visit (target address)
	     (if (null address)
		 (or (funcall replacement-r target)
		     (raise-error (error-message '(inf-tree modify match not))))

		 (let ((index (car address)))
		   (with-tag index
		     (let ((children (children-of-inf-tree target)))

		       (cond
			 ((zerop index)
                          ;; below should not be reached, won't work anyway
                          (raise-error (error-message '(inf-tree modify zero addr)))
			  (let* ((objc (objc-of-inf-tree target))
				 (step (step-of-inf-objc-r objc)))
			    (unless (eql 'top (type-of-inf-step step))
			      (raise-error (error-message '(inf-tree modify top not))))
		
			    (inf-tree
			     (inf-objc-modify-step objc
						   (new-inf-step-inf-tree step
									  (visit (inf-tree-of-inf-step step)
										 (cdr address))))
			     children)))
	      
	      
			 ((> 0 index)
			  (visit target
				 (cons (+ 1 (length children) index)
				       (cdr address))))
	      
			 ((< 0 index)
			  (unless (<= index (length children))
			    (raise-error (error-message '(inf-tree modify bounds))))
			  (let ((tree (visit (nth (1- index) children) (cdr address))))
			    (inf-tree (oid-of-inf-tree target)
				      (replace-in-list tree index children)))))))))))
    (visit target address)))

(defun inf-tree-find (tree address)
  (if (null address)
      tree
      (let ((index (car address)))
	(with-tag index
	  (let ((children (children-of-inf-tree tree)))
	    (cond
	      ((zerop index)
	       (let* ((objc (objc-of-inf-tree tree))
		      (step (step-of-inf-objc-r objc)))
		 (unless (eql 'top (type-of-inf-step step))
		   (raise-error (error-message '(inf-tree replace top not))))
		 (inf-tree-find (inf-tree-of-inf-step step)
				(cdr address))))
	      ((> 0 index)
	       (inf-tree-find tree (cons (+ 1 (length children) index)
					 (cdr address))))
	      ((< 0 index)
	       (unless (<= index (length children))
		 (raise-error (error-message '(inf-tree replace bounds))))
	       (inf-tree-find (nth (1- index) children) (cdr address)))))))))


;; invariant
;; in extract of a tree iincomplete index correlates to position in frontier of that tree.

(defun extract-from-inf-tree-aux (refined-p node-extract-f children-f inf-tree)
  (labels
      ((visit-tree (tree)
	 ;;(setf -tree tree) (break "efita")
	 (if (funcall refined-p tree)
	     (let ((running-count 0))
	       (values (let ((ext (funcall node-extract-f #'visit-tree tree)))

			 ;;(incf running-count l)
			 (extract-replace
			  ext
			  (mapcar #'(lambda (stree)
				      (mlet* (((ext l) (visit-tree stree)))
					     (if (zerop l)
						 ext
						 (progn
						   (incf running-count l)
						   (setf -running-count running-count -l l)
						   (extract-replace
						    ext
						    (do ((j l (1- j))
							 (acc nil (let ((i (- running-count (- l j))))
								    ;;(unless (> i 0) (break "rc"))
								    (cons (iincomplete-term i)
									  acc))))
							((zerop j) (setf -acc acc) acc)))))))
				  (funcall children-f tree))))
		       running-count))
	     (values (iincomplete-term 1) 1))))
    (visit-tree inf-tree)
    ))



(defvar *inf-tree-dummy-term*
  (iinf-tree-term (iinf-goal-op)(iinf-unrefined-term)(iinf-tree-cons-op)(iannotation-nil-term)))


(defun xref-of-inf-step (step)
  (let ((x (find-first #'(lambda (x) (when (xref-ianno-p x) x)) (annotations-of-inf-step step))))
    (when x (xref-of-ianno x))))

;;(xref-of-inf-step step)
;;(references-of-refinement r)
;; node : 
;;   extract
;;   refs
;; annos :
;;   xref
;;   stats
;; recurse through tops?
;; f (<bool{term|inf-step}> <step>)
;; step is not represented locally as term in itp but is instead
;;  constructed from various parts of !inf_tree term
;;
;;  dependencies occur in node and in annos?? 
;;    maybe oversight in that collected as anno and then extracted from annos
;;    but then also by default included as anno.


(defun inf-tree-walk-aux (f it &optional addr-p interior-p)
  
  (labels
      (
       (visit (mode v vt it top children addr)

	 (let ((i 0))
	   (let ((top-contf (when top
			      #'(lambda ()
				  (funcall (or vt v) top (when addr-p (cons 0 addr)))) ))
		 (child-contf #'(lambda (valuep)
				  (if valuep
				      (mapcar #'(lambda (c)
						  (funcall v c (when addr-p (cons (incf i) addr))))
					      children)
				      (dolist (c children)
					(funcall v c (when addr-p (cons (incf i) addr))))))) )

	     (if addr-p
		 (funcall f mode it top-contf child-contf addr)
		 (funcall f mode it top-contf child-contf)))))

       (visit-itp (itp addr)

	 (let ((node (node-of-iinf-tree-term itp))
	       (children (map-isexpr-to-list (children-of-iinf-tree-term itp)
					     (iinf-tree-cons-op))))

	   (visit 'term #'visit-itp nil
		  itp
		  (when (iinf-top-term-p node)
		    (tree-of-iinf-top-term node))
		  children
		  addr)))

       (visit-it (it addr)

	 (let* ((step (step-of-inf-objc (objc-of-inf-tree it)))
		(r (refinement-of-inf-step step))
		(vt nil))

	   (let ((top (when (top-refinement-p r)
			(let ((it (top-refinement-inf-tree r)))
			  (if (inf-tree-p it)
			      it ;;(inf-tree-of-top-refinement r (goal-of-inf-step step))
			      (progn
				;;(setf -r r -it it)
				;;(break "xitp")
				(setf vt #'visit-itp)
				(tree-of-inf-tree-proxy (provide-data it 'inf-tree-proxy))))))))

	     (visit 'inf_tree #'visit-it vt
		    it
		    top
		    (children-of-inf-tree it)
		    addr)))))
    

    (if (inf-tree-p it)
	(visit-it it nil)
	(visit-itp (tree-of-inf-tree-proxy (provide-data it 'inf-tree-proxy))
		   nil))))

(defun inf-tree-walk-step-aux (f refinement &optional addr-p)
  (inf-tree-walk-aux f (top-refinement-inf-tree refinement) addr-p))


(defun inf-tree-walk-step (f refinement  &optional addr-p)
  (inf-tree-walk-step-aux #'(lambda (mode it topf childf &optional addr)
			      (if addr-p
				  (funcall f mode it addr)
				  (funcall f mode it))
			      (when topf (funcall topf))
			      (when childf (funcall childf nil)))
			  refinement addr-p))


(defun walk-inf-tree-references (f refinement)
  (inf-tree-walk-step
   #'(lambda (mode it)
       (if (eql mode 'term)
	   (let ((node (node-of-iinf-tree-term it)))
	     (let ((deps (dependencies-of-iinf-top-term node)))
	       (when deps
		 (let ((ddeps (term-to-environment-dependencies deps)))
		   (when ddeps
		     (funcall f ddeps))))))
	   (let* ((step (step-of-inf-objc (objc-of-inf-tree it)))
		  (r (refinement-of-inf-step step)))
	     (let ((refs (cond
			   ((top-refinement-p r)	(references-of-refinement r))
			   ((abbrev-refinement-p r)	(references-of-refinement r))
			   ((primitive-refinement-p r) (references-of-refinement r))
			   (t nil))))
	       (when refs (funcall f refs))))))
   refinement))

(defun refs-of-top-refinement (r)

  (let ((acc nil))
    (walk-inf-tree-references
     #'(lambda  (refs)
	 (if (environment-dependencies-p refs)
	     (setf acc (append (list-of-stamp-dependencies refs) acc))
	     ;; at the moment do not expect dependency store but is
	     ;;  a possibility later.
	     (dolist (env-d (list-of-dependency-store refs))
	       (setf acc (append (list-of-stamp-dependencies env-d) acc))
	       ;;(visit-refs env-d)
	       )))
     r)

    (environment-dependencies-normal (transaction-stamp) acc)))


;; sometimes want to return interior proof of a node.
;; particularly, !refine_tree but sometimes search tactics (ie obvious, jprover)
(defun test-special (a)
  (labels ((fu (a)
	     (declare (special b))
	     (if (eql b 2)
		 nil
	     (unless (null a)
	       (let ((b (car a)))
		 (declare (special b))
		 (bar (cdr a))))))

	   (bar (a)
	     (declare (special b))
	     (cons b (fu a))))
    (fu a)))


(defun tactic-tree-from-inf-tree-or-proxy (tree &optional interior-p)
  (inf-tree-walk-aux
   #'(lambda (mode it topf childrenf &optional addr)
       (declare (ignore addr) (special children))
       ;;(setf -it it -mode mode) (break "ttfitop")
       
       (if (if (eql mode 'term)
	       (iinf-unrefined-term-p (node-of-iinf-tree-term it))
	       (not (refined-inf-tree-p it)))
	   (if (null children)
	       (break "too few")
	       (let ((c (pop children)))
		 (setf -c c -children children)
		 (break "fu")
		 c))

	   (let ((tac (if (eql mode 'term)
			  (tactic-of-iinf-top-term
			   ;; should we check for itop-term-p??
			   ;; should be top since we do not look inside top.
			   (node-of-iinf-tree-term it))
			  ;; commented code resulted in losing line breaks in inf source when replaying.
			  ;;  (tactic-of-inf-step
			  ;;    (step-of-inf-source (source-of-objc (objc-of-inf-tree it))))
			  (or (term-of-source (source-of-objc (objc-of-inf-tree it)))
			      (raise-error (error-message '(proof not complete))))))
		 (children (funcall childrenf t)))
	     (declare (special children))
	     (if (and interior-p (funcall interior-p tac))
		 (prog1 
		     (funcall topf)
		   (when (not (null children))
		     (setf -children children -tac tac)
		     (break "too many")))
		 (tactic-tree-term tac (map-list-to-ilist children *tactic-tree-nil-term*))))))
   tree nil))

(defun tactic-goal-tree-from-inf-tree-or-proxy (tree)
  (inf-tree-walk-aux
   #'(lambda (mode it topf childrenf &optional addr)
       (declare (ignore addr topf) (special children))
       ;;(setf -it it -mode mode) (break "ttfitop")
       
       (if (if (eql mode 'term)
	       (iinf-unrefined-term-p (node-of-iinf-tree-term it))
	       (not (refined-inf-tree-p it)))
	   (if (null children)
	       (break "too few")
	       (let ((c (pop children)))
		 (setf -c c -children children)
		 (break "fu")
		 c))

	   (let ((tac (if (eql mode 'term)
			  (icons-term
			   (tactic-of-iinf-top-term
			    ;; should we check for itop-term-p??
			    ;; should be top since we do not look inside top.
			    (node-of-iinf-tree-term it))
			   (map-list-to-ilist (mapcar #'(lambda (it)
							  (goal-of-iinf-tree-term it))
						      (map-isexpr-to-list
						       (children-of-iinf-tree-term it)
						       (iinf-tree-cons-op)))
					      (inil-term)))
			  ;; commented code resulted in losing line breaks in inf source when replaying.
			  ;;  (tactic-of-inf-step
			  ;;    (step-of-inf-source (source-of-objc (objc-of-inf-tree it))))
			  (let ((isrc (source-of-objc (objc-of-inf-tree it))))
			    (icons-term
			     (or (term-of-source isrc)
				 (raise-error (error-message '(proof not complete))))
			     (inf-objc-to-step-subgoals-term isrc)))))
		 (children (funcall childrenf t)))
	     (declare (special children))
	     (tactic-tree-term tac (map-list-to-ilist children *tactic-tree-nil-term*)))))
   tree nil))

(defun tactic-tree-of-inf-tree-or-proxy (tree)
  ;;(setf -tree tree) (break "ttoitop")
  (cond
    ((inf-tree-p tree)
     (tactic-tree-from-inf-tree-or-proxy tree))
    ((complete-inf-tree-proxy-p tree)
     (tactic-tree-from-inf-tree-or-proxy tree))
    (t (raise-error (error-message '(tactic-tree inf-tree complete not)))))  )

(defun tactic-tree-of-prf-objc (pobjc)
  (tactic-tree-of-inf-tree-or-proxy
   (inf-tree-or-proxy-of-proof-source (source-of-objc pobjc))))

(defunml (|tactic_tree_of_prf_objc| (objc))
    (object_contents -> term)
  (require-objc-kind objc 'prf)
  (tactic-tree-of-prf-objc objc))

(defunml (|tactic_goal_tree_of_prf_objc| (objc))
    (object_contents -> term)
  (require-objc-kind objc 'prf)
  (tactic-goal-tree-from-inf-tree-or-proxy
   (inf-tree-or-proxy-of-proof-source (source-of-objc objc))))

(defunml (|tactic_tree_of_prf_objc_wint| (p objc))
    ((term -> bool) -> (object_contents -> term))
  (require-objc-kind objc 'prf)
  (tactic-tree-from-inf-tree-or-proxy
   (inf-tree-or-proxy-of-proof-source (source-of-objc objc))
   #'(lambda (tac) (funmlcall p tac))))


;; another kludge.
(defunml (|expose_inf_from_refine_tree| (objc))
    (object_contents -> object_contents)

  ;; just flip direct bit in node top.
  (let ((step (step-of-inf-objc-r objc)))
    (let ((newr (copy-top-refinement (refinement-of-inf-step step))))
      (setf (refinement-direct-p newr) t)
      (let ((nstep (copy-inf-step step)))
	(setf (inf-step-refinement nstep) newr)
	(inf-objc-modify-step objc nstep)))))


#|
(defun visit-inf-tree-refinement-references (inf-tree)
  (let ((acc nil))
    (labels
	((visit-refs (refs)
	   (when refs
	     (if (environment-dependencies-p refs)
		 (setf acc (append (list-of-stamp-dependencies refs) acc))
		 ;; at the moment do not expect dependency store but is
		 ;;  a possibility later.
		 (dolist (env-d (list-of-dependency-store refs))
		   (visit-refs env-d)))))

	 (visit (inf-tree)
	   (let* ((step (step-of-inf-objc (objc-of-inf-tree inf-tree)))
		  (r (refinement-of-inf-step step)))
	     (cond
	       ((top-refinement-p r)
		(visit-refs (references-of-refinement r))
		(visit (inf-tree-of-top-refinement r (goal-of-inf-step step))))
	       ((abbrev-refinement-p r)
		(visit-refs (references-of-refinement r)))
	       ((primitive-refinement-p r)
		(visit-refs (references-of-refinement r)))))
	   (dolist (child (children-of-inf-tree inf-tree))
	     (visit child))))

      (visit inf-tree))
    (environment-dependencies-normal (transaction-stamp) acc)))|#


(defun walk-inf-tree-annotations (f refinement)
  (inf-tree-walk-step
   #'(lambda (mode it)
       (if (eql mode 'term)
	   (funcall f (term-to-annotations (annotations-of-iinf-tree-term it)))
	   (funcall f (annotations-of-inf-step (step-of-inf-objc (objc-of-inf-tree it))))))
   refinement))

(defun walk-inf-tree-xref (f refinement)
  (walk-inf-tree-annotations
   #'(lambda (annos)
       (let ((x (find-first #'(lambda (x)
				(when (xref-ianno-p x) x))
			    annos)))
	 (when x (funcall f (xref-of-ianno x)))))
   refinement))


    
(defun xrefs-of-top-refinement (r)

  (let ((acc nil))

    (walk-inf-tree-xref #'(lambda (x)
			    (push x acc))
			r)

    (map-list-to-ilist (reduce #'nconc
			       (mapcar #'(lambda (x)
					   (map-isexpr-to-list x (xref-cons-op)))
				       (nreverse acc)))
		       (xref-nil-term))))


(defun xrefs-of-refinement (r)
  (if (top-refinement-p r)
      (xrefs-of-top-refinement r)
      (raise-error (error-message '(xref refinement top not)))) )

(defun inf-tree-extract-sexpr (it)

  ;;(setf -it it) (break "ites")

  (inf-tree-walk-aux

   #'(lambda (mode it topf childrenf &optional addr)
       ;;(format t "~% ~a" (reverse addr))
       ;;(setf -it it -mode mode -topf topf) (break "efr00")
       (if (eql mode 'term)
	   (let ((node (node-of-iinf-tree-term it)))
	     (if (iinf-unrefined-term-p node)
		 nil
		 (let ((tope (if topf
				 (funcall topf)
				 (let ((ext (extract-of-iinf-node node)))
				   (when (null ext) (setf -it it)
					 (fooe) ;;(break "fu")
					 )
				   ext)))
		       (childrene (funcall childrenf t)))
		   (cons tope childrene))))
	   (if (not (refined-inf-tree-p it))
	       nil
	       (let ((tope (if topf
			       (funcall topf)
			       (let ((ext (extract-from-inf-objc (objc-of-inf-tree it))))
				 (when (null ext) (setf -it it)  (break "fu"))
				 ext)))
		     (childrene (funcall childrenf t)))
		 ;;(when (equal-oids-p (oid-of-inf-tree it) -moid)
		 ;;  (break "aughrg"))
		 ;;(setf -tope tope)
		 ;;(when (equal-terms-p (car tope) (car -mytermext))
		 ;;(setf -iit it -addr addr))
		 (cons tope childrene)))))

   it
   nil ;; no addresses
   ))

(defun extract-from-inf-tree (inf-tree)
  (extract-from-inf-tree-aux
   #'(lambda (tree) (not (null tree)))
   #'(lambda (vt tree)
       (if (null tree)
	   (progn;;(break "null tree")
	     (iincomplete-term 1))
	   (let ((s (car tree)))
	     (if (consp s)
		 (funcall vt s)
		 s))))
   #'cdr
   (inf-tree-extract-sexpr inf-tree)))


(defun extract-from-inf-step (step)
  (with-ignore
      (extract-from-inf-tree
       ;; returns inf-tree or inf-tree-proxy.
       ;; inf-tree-of-inf-step inflates tree.
       (top-refinement-inf-tree (refinement-of-inf-step step)))))

(defun extract-from-inf-objc (iobjc)
  (let ((step (step-of-inf-objc-r iobjc)))
    (let ((type (type-of-inf-step step)))
      (case type
	(unrefined (iincomplete-term 1))
	((abbrev primitive) (extract-of-inf-step step))
	(top (extract-from-inf-step step))
	(otherwise (raise-error (error-message '(extract unexpected type) type)))))))

;; should unopquote quoted incompletes?? not here I don't think.
(defun extract-from-inf-tree-r (inf-tree)
  (or (extract-from-inf-tree inf-tree)
      (raise-error (error-message '(extract inf-tree not)))))


;;%
;;;;	
;;;;	<extract-tree>		: !extract_tree(<extract>; <extract-tree> list)
;;;;	
;;;;	<extract-node>		: !extract_node(<address>; <extract>)
;;;;				| !extract_node(<address>; <extract-tree>)
;;%
(define-primitive |!extract_tree| () (node children))
(define-primitive |!extract_node| () (addr extract))
(define-primitive |!extract_cons| () (car cdr))

(defunml (|inf_tree_extract_tree| (itree))
    (inf_tree -> term)

  (labels ((visit-tree (addr x)
	     ;; have unnatural case of  ((!incomplete{1} . (NIL)) . (NIL)) instead of (!incomplete{1} . (NIL)) ?
	     ;; probably as result of Id tac kludge.
	     (if (and (not (null (cdr x)))
		      (forall-p #'null (cdr x)))
		 
		 (visit-node (cons 0 addr) (car x))
		 
		 (iextract-tree-term (visit-node addr (car x))
				     (let ((i 0))
				       (map-list-to-ilist (cdr x)
							  (iextract-nil-term)
							  #'(lambda (child)
							      (incf i)
							      (visit-tree (cons i addr) child)))))))

	   (visit-node (addr node)

	     (iextract-node-term (map-list-to-ilist addr (ipui-addr-nil-term) #'ipui-addr-term)
				 (cond
				   ((null node) (break "null node"))

				   ((not (consp node)) node)
				   
				   ((forall-p #'null (cdr node))
				    (let ((a (car node)))
				      (if (consp a)
					  (visit-node (cons 0 addr) a)
					  a)))

				   (t (visit-tree (cons 0 addr) node))))))

    (setf -a (visit-tree nil (setf -sexpr (inf-tree-extract-sexpr itree))))
    ;;(break "a")
    -a))

(defunml (|raw_inf_extract_tree| (itree))
    (inf_tree -> term)

  (labels ((visit (x)
	     (if (consp x)
		 (instantiate-term (instantiate-operator `|!sexpr_cons| nil)
				   (list (instantiate-bound-term (visit (car x)))
					 (instantiate-bound-term (visit (cdr x)))))
		 (if (null x)
		     (instantiate-term (instantiate-operator `|!sexpr_cons| nil))
		     (if (not (term-p x))
			 (progn (setf -x x) (break))
			 x)))))
    (visit
     (inf-tree-extract-sexpr itree))))

;; following was used for debugging extracts.
#|
(defunml (|inf_extract_tree| (iobjc))
    (object_contents -> term)

  (labels ((visit (x)
	     (if (consp x)
		 (instantiate-term (instantiate-operator `|!sexpr_cons| nil)
				   (list (instantiate-bound-term (visit (car x)))
					 (instantiate-bound-term (visit (cdr x)))))
		 (if (null x)
		     (instantiate-term (instantiate-operator `|!sexpr_cons| nil))
		     (if (not (term-p x))
			 (progn (setf -x x) (break))
			 x)))))
    (visit
     (inf-extract-sexpr iobjc))))

(defunml (|inf_extract_tree| (iobjc))
    (object_contents -> term)

  (labels ((visit (x)
	     (if (consp x)
		 (instantiate-term (instantiate-operator `|!sexpr_cons| nil)
				   (list (instantiate-bound-term (visit (car x)))
					 (instantiate-bound-term (visit (cdr x)))))
		 (if (null x)
		     (instantiate-term (instantiate-operator `|!sexpr_cons| nil))
		     (if (not (term-p x))
			 (progn (setf -x x) (break))
			 x)))))
    (visit
     (inf-extract-sexpr iobjc))))

(defunml (|sexpr_to_extract| (sexpr))
    (term -> term)
  
  (labels ((visit (x)
	     (if (eql `|!sexpr_cons| (id-of-term x))
		 (let ((bts (bound-terms-of-term x)))
		   (when bts
		     (cons (visit (term-of-bound-term (car bts)))
			   (visit (term-of-bound-term (cadr bts)))
			   )))
		 x)))

    (extract-from-inf-tree-aux
     #'(lambda (tree) (not (null tree)))
     #'(lambda (vt tree)
	 (if (null tree)
	     (progn ;;(break "null tree")
		    (iincomplete-term 1))
	     (progn ;;(setf -tree tree) (break "tree")
	     (let ((s (car tree)))
	       (if (consp s)
		   (funcall vt s)
		   s)))))
     #'cdr
     (setf -sexpr (visit sexpr)))))|#

(defunml (|extract_from_inf_step| (step))
    (inf_step -> term)
  (let ((type (type-of-inf-step step)))
    (or (case type
	  (unrefined (iincomplete-term 1))
	  ((abbrev primitive) (extract-of-inf-step step))
	  (top (extract-from-inf-tree (inf-tree-of-inf-step step))))
	(raise-error (error-message '(extract_from_inf_step))))))

#|
;; may return nil



(defun extract-from-refinement (r)

    (let ((x-sexpr
	   (inf-tree-walk-step-aux
	    #'(lambda (mode it topf childrenf)
		;;(setf -it it -topf topf) (break "efr00")
		(if (eql mode 'term)
		    (let ((node (node-of-iinf-tree-term it)))
		      (if (iinf-unrefined-term-p node)
			  nil
			  (let ((tope (if topf
					  (funcall topf)
					  (let ((ext (extract-of-iinf-node node)))
					    ext)))
				(childrene (funcall childrenf t)))
			    (cons tope childrene))))
		    (if (not (refined-inf-tree-p it))
			nil
			(let ((tope (if topf
					(funcall topf)
					(let ((ext (extract-from-inf-step
						    (step-of-inf-objc-r (objc-of-inf-tree it)))))
					  ext)))
			      (childrene (funcall childrenf t)))
			  (cons tope childrene)))))
	    r)))

      ;; compute extract from sexpr
      (labels
	  ((main-visit (tree)
	     ;;(setf -tree tree) (break "xfrmv")
	     (let ((i 0))
	       (labels
		   ((visit (tree)
		      (if (null tree)
			  (progn
			    ;;(setf -i i) (break "ii")
			    (iincomplete-term (incf i)))
			  (extract-replace (if (consp (car tree))
					       (main-visit (car tree))
					       (car tree))
					   (mapcar #'visit (cdr tree))))))
		 (visit tree))))

	   )
	
	;;(setf -x-sexpr x-sexpr) (break "xfr")
	(main-visit x-sexpr)))
      )
|#
   

(defun xrefs-of-inf-step (s)
  (let ((r (refinement-of-inf-step s)))
    (when r
      (setf -r r)
      (xrefs-of-refinement r))))

(defun extract-of-refinement (r)
  (cond
    ((abbrev-refinement-p r) (extract-of-abbrev-refinement r))
    (t (raise-error (error-message '(extract refinement not))))))

(defun direct-refinement-top-p (r)
  (let ((it (top-refinement-inf-tree r)))
    (if (inf-tree-p it)
	(direct-inf-tree-p it)
	(let ((node (node-of-iinf-tree-term (tree-of-inf-tree-proxy (provide-data it 'inf-tree-proxy)))))
	  (and (iinf-top-term-p node)
	       (direct-of-iinf-top-term node)))
	)))

(defun inf-step-abbreviate (step)

  (when (unrefined-inf-step-p step)
    (raise-error (error-message '(inf-step abbreviate unrefined))))
  
  (let ((r (refinement-of-inf-step step)))

    (when (direct-refinement-top-p r)
      (raise-error (error-message '(inf-step abbreviate direct))))

    (let ((extract (extract-from-inf-step step))
	  (store (refs-of-top-refinement r)))

      (let ((new-step (copy-inf-step step)))

	(setf (inf-step-refinement new-step)
	      (abbrev-refinement (stamp-of-refinement r)
				 (direct-refinement-p r)
				 store
				 (subgoals-of-refinement r)
				 (tactic-of-top-refinement r)
				 extract))
	new-step))))


#|
(defun inf-step-abbreviate (step)
  (let ((inf-tree (inf-tree-of-inf-step step t)))

    (when (direct-inf-tree-p inf-tree)
      (raise-error (error-message '(inf-step abbreviate direct))))

    (let ((extract (extract-from-inf-tree inf-tree))
	  (store (visit-inf-tree-refinement-references inf-tree)))

      (let ((new-step (copy-inf-step step))
	    (refinement (refinement-of-inf-step step)))

	(setf (inf-step-refinement new-step)
	      (abbrev-refinement (stamp-of-refinement refinement)
				 (direct-refinement-p refinement)
				 store
				 (subgoals-of-refinement refinement)
				 (tactic-of-top-refinement refinement)
				 extract))
	new-step))))
|#
	    

(defvar *lib-refine* nil)
(defvar *lib-refinet* nil)

(defun lib-refine (desc tactic goal envoid oids)
  ;;(setf a desc b tactic c goal d oids) (break "lr")
  ;;(format t "lib-refine")
  (funmlcall (or *lib-refine*
		 (setf *lib-refine* (ml-text "lib_refine")))
	     desc tactic goal envoid oids))

(defun lib-refinet (desc tactic goal envterm)
  ;;(setf a desc b tactic c goal d envterm) (break "lr")
  ;;(format t "lib-refine")
  (funmlcall (or *lib-refinet*
		 (setf *lib-refinet* (ml-text "lib_refinet")))
	     desc tactic goal envterm))

(define-primitive |!refine_failure| () (tactic error))
(define-primitive |!pui_hyp| ((natural . numeral) (bool . repeat-p)  (bool . invisible-p))) 
(define-primitive |!pui_nat| ((natural . numeral)))

(defun inf-step-refine (desc step tactic envoid oids)
  ;;(setf -tactic tactic -step step) (break "is")
  (with-handle-error-and-message
      (('(inf refine))
       #'(lambda (msg)
	   ;;(setf -msg msg) (break "isre")
	   (inf-step-annotate
	    step
	    (irefine-failure-term
	     tactic
	     (apply #'ifail-term
		    (message-to-term
		     (tag-message *environment-path*
				  (tag-message '(eval) msg)))
		    (mapcar #'message-to-term (messages-flush)))))))

    ;;(setf g (goal-to-term (goal-of-inf-step step))) (break "sending goal")
    (setf -istep
    (inf-step-update-refinement step
				(lib-refine desc
					    (goal-to-term (goal-of-inf-step step))
					    tactic
					    envoid
					    oids)
				tactic)
    ) ;;(break "isr")
      -istep
    ))

(defun inf-step-refinet (desc step tactic envterm)
  ;;(setf -tactic tactic -step step) (break "is")
  (with-handle-error-and-message
      (('(inf refine))
       #'(lambda (msg)
	   ;;(setf -msg msg) (break "isre")
	   (inf-step-annotate
	    step
	    (irefine-failure-term
	     tactic
	     (apply #'ifail-term
		    (message-to-term
		     (tag-message *environment-path*
				  (tag-message '(eval) msg)))
		    (mapcar #'message-to-term (messages-flush)))))))

    ;;(setf g (goal-to-term (goal-of-inf-step step))) (break "sending goal")
    (setf -istep
    (inf-step-update-refinement step
				(lib-refinet desc
					    (goal-to-term (goal-of-inf-step step))
					    tactic
					    envterm
					    )
				tactic)
    ) ;;(break "isr")
      -istep
    ))

(defunml (|inf_tree| (oid children))
     (object_id -> ((inf_tree list) -> inf_tree))

  (let ((objc (oc oid)))  
    (unless (eql 'inf (kind-of-objc objc))
      (raise-error (error-message '(kind inf) (kind-of-objc objc))))

    (instantiate-inf-tree oid children)))

;; doesn't perform checks that steps are refined and goals match
(defunml (|inf_tree_force| (oid children))
     (object_id -> ((inf_tree list) -> inf_tree))

  (let ((objc (oc oid)))  
    (unless (eql 'inf (kind-of-objc objc))
      (raise-error (error-message '(kind inf) (kind-of-objc objc))))

    (inf-tree oid children)))


(defunml (|inf_step_refine| (desc inf-step tactic envoid oids))
    (term -> (inf_step -> (term -> (object_id -> ((object_id list) -> inf_step)))))

  (inf-step-refine desc inf-step tactic envoid oids))

(defunml (|inf_step_refinet| (desc inf-step tactic envterm))
    (term -> (inf_step -> (term -> (term -> inf_step))))
  
  (inf-step-refinet desc inf-step tactic envterm))

(defunml (|inf_step_modify_annotations| (inf-step annos))
    (inf_step -> ((term list) -> inf-step))
  (inf-step-set-annotations inf-step annos))


(defunml (|inf_object_contents_refine| (oc envoid oids))
    (object_contents -> (object_id -> ((object_id list) -> object_contents)))

  (inf-objc-refine oc envoid oids))

(defunml (|inf_object_contents_refinet| (oc envterm))
    (object_contents -> (term -> object_contents))

  (inf-objc-refinet oc envterm))



;;;;	NatInd 4 proof shape of nat_ind thm of int_1 thy.
;;;;	#pn
;;;;	  #pn1
;;;;	    *pn11
;;;;	      *pn111
;;;;	      *pn112
;;;; 	    *pn12
;;;;	      *pn121
;;;;	      *pn122
;;;;	    #pn13
;;;;	      #pn131
;;;;	        #pn1311
;;;;	        #pn1312
;;;;	      *pn132
 

;; inf-tree skeleton : for debugging, allows easier inspection of inf-tree shape.
(define-primitive its () (goal rule children))

(defun inf-tree-term-to-its (it)
  (its-term
   (goal-of-iinf-tree-term it)
   (let ((node (node-of-iinf-tree-term it)))
     ;;(setf -node node) (break "ittti")
     (if (eql (id-of-term node) '|!inf_top|)
 	 (icons-term (inf-tree-term-to-its (tree-of-iinf-top-term node))
 		     (or (let ((bt (caddr (bound-terms-of-term node))))
 			   (when bt (term-of-bound-term bt)))
 			 (ivoid-term)))
 	 (ivoid-term)))
   (map-sexpr-to-isexpr
    (map-isexpr-to-list (children-of-iinf-tree-term it) (iinf-tree-cons-op))
    (iinf-tree-nil-term)
    #'inf-tree-term-to-its)))

		  
(defun inf-step-update-refinement (step inf-tree &optional tactic)
  ;;(setf -step step -inf-tree inf-tree -tactic tactic)  (break "iu")
  (let ((new-step (copy-inf-step step))
	(top-node (node-of-iinf-tree-term inf-tree)))

    (setf -st step -it inf-tree) ;;(break "u")
    (setf (inf-step-info-annotations new-step)
	  (map-isexpr-to-list (annotations-of-iinf-tree-term inf-tree)
			      (iannotation-cons-op))
	    
	  (inf-step-refinement new-step)
	  (make-top-refinement
	  
	   :direct-p (direct-of-iinf-top-term top-node)

	   :subgoals (map-isexpr-to-list (children-of-iinf-tree-term inf-tree)
					 (iinf-tree-cons-op)
					 #'(lambda (term)
					     (iinf-goal-term-to-goal
					      (goal-of-iinf-tree-term term))))

	   :references (term-to-environment-dependencies
			(dependencies-of-iinf-top-term top-node))

	   :tactic  (or tactic (tactic-of-iinf-top-term top-node))

	   :inf-tree (let ((term (tree-of-iinf-top-term top-node)))
		       ;; if should be !inf_tree term -> make proxy
		       ;;(setf -itt inf-tree) (break "sbittmp")
		       (new-inf-tree-proxy term))))
    new-step))



(defunml (|inf_tree_object_contents| (inf-tree))
     (inf_tree -> object_contents)
  (objc-of-inf-tree inf-tree))

(defunml (|inf_tree_object_id| (it))
    (inf_tree -> object_id)
  (oid-of-inf-tree it))


(defunml (|inf_tree_children| (inf-tree) :error-wrap-p nil)
     (inf_tree -> (inf_tree list))
  (children-of-inf-tree inf-tree))

(defunml (|inf_tree_extract| (inf-tree))
     (inf_tree -> term)
  (extract-from-inf-tree-r inf-tree))


(defunml (|inf_tree_replace_tree| (target inf-tree address))
     (inf_tree -> (inf_tree -> ((int list) -> inf_tree)))
  
  (inf-tree-replace-tree target inf-tree address t))

(defunml (|inf_tree_replace_tree_ns| (target inf-tree address))
    (inf_tree -> (inf_tree -> ((int list) -> inf_tree)))
  
  (inf-tree-replace-tree target inf-tree address nil))

(defunml (|inf_tree_replace_tree_unconditional| (target inf-tree address))
     (inf_tree -> (inf_tree -> ((int list) -> inf_tree)))
  
  (inf-tree-modify target address 
		   #'(lambda (target) (declare (ignore target)) inf-tree)))

(defunml (|inf_tree_replace_node| (target oid address))
     (inf_tree -> (object_id -> ((int list) -> inf_tree)))
  
  (inf-tree-replace-node target oid address))

(defunml (|inf_tree_find| (inf-tree address))
     (inf_tree -> ((int list) -> inf_tree))
  
  (inf-tree-find inf-tree address))


(defunml (|inf_tree_to_iinf_tree_term| (inf-tree))
  (inf_tree -> term)
  
  (inf-tree-to-iinf-tree-term inf-tree nil t))

#|
(defunml (|inf_tree_to_psterm| (inf-tree))
    (inf_tree -> term)
  
  (inf-tree-to-psterm inf-tree))
|#
;; see default_view_mode
(defunml (|inf_tree_to_iproof_node_term| (inf-tree status address view depth))
    (inf_tree ->  (term -> (term -> (tok -> (int -> term)))))
  (inf-tree-to-iproof-node-term inf-tree status address view nil depth))

(defunml (|primitive_inf_tree_to_iproof_node_term| (inf-tree status address view depth))
    (inf_tree ->  (term -> (term -> (tok -> (int -> term)))))
  
  (primitive-inf-tree-to-iproof-node-term inf-tree status address view nil depth))

(defunml (|interior_inf_tree_to_iproof_node_term| (inf-tree status address view depth))
    (inf_tree ->  (term -> (term -> (tok -> (int -> term)))))
  
  (interior-inf-tree-to-iproof-node-term inf-tree status address view nil depth))

;; see default_view_mode
(defunml (|inf_tree_to_iproof_node_term_g| (inf-tree status address view depth))
    (inf_tree ->  (term -> (term -> (tok -> (int -> term)))))
  
  (inf-tree-to-iproof-node-term inf-tree status address view nil depth)
  ;;(inf-tree-to-iproof-node-term inf-tree status address '|t| nil depth)
  )

(defunml (|inf_tree_to_iproof_node_term_p| (inf-tree status address view parent depth))
    (inf_tree ->  (term -> (term -> (tok -> (term -> (int -> term))))))
  ;; (setf -parent parent ) (break "itp")
  (inf-tree-to-iproof-node-term inf-tree status address view parent depth))

(defunml (|iproof_node_term_to_inf_tree| (term))
    (term -> inf_tree)
  ;;(setf -term term) (break)
  (iproof-node-term-to-inf-tree term))

#|
(defunml (|iproof_node_term_to_inf_tree_modify| (term))
    (term -> inf_tree)
  ;;(setf -term term) (break)
  (iproof-node-term-to-inf-tree term t))
|#

(defunml (|iinf_tree_term_to_inf_tree| (term))
  (term -> inf_tree)
  
  (iinf-tree-term-to-inf-tree term nil))

(defunml (|goal_of_iinf_tree_term| (term)) 
    (term -> goal)
  (iinf-goal-term-to-goal (goal-of-iinf-tree-term term)))

(defunml (|make_inf_tree| (oid children))
  (object_id -> ((inf_tree list) -> inf_tree))
  (make-inf-tree :oid oid :children children))



;;;
;;;	inf_step ;
;;;

(defunml (|inf_step| (goal))
    ((term |#| (term list)) -> inf_step)
  (inf-step goal))

(defunml (|inf_step_type| (inf-step))
     (inf_step -> tok)
  (type-of-inf-step inf-step))

(defunml (|inf_step_goal| (inf-step))
     (inf_step -> (term |#| (term list)))
  (goal-of-inf-step inf-step))

(defunml (|inf_step_info_annotations| (inf-step))
     (inf_step -> (term list))
  (annotations-of-inf-step inf-step))

(defunml (|inf_step_refined_p| (step))
     (inf_step -> bool)
  (refined-inf-step-p step))


(defunml (|inf_step_tactic| (inf-step))
     (inf_step -> term)
  (tactic-of-inf-step inf-step))

(defunml (|inf_step_subgoals| (inf-step))
     (inf_step -> ((term |#| (term list)) list))
  (subgoals-of-inf-step inf-step))

(defunml (|inf_step_references| (inf-step))
    (inf_step -> ((tok |#| (dependency list)) list))

  (setf -is inf-step)  (break "isr")
  (normalize-dependencies (list-of-stamp-dependencies (references-of-inf-step inf-step)))
  )


(defunml (|inf_step_inf_tree| (inf-step))
    (inf_step -> inf_tree)
  (inf-tree-of-inf-step inf-step))

(defunml (|inf_step_extract| (inf-step))
    (inf_step -> term)
    (setf -inf-step inf-step) (break "ise")
    (or (extract-of-inf-step inf-step)
	(progn (setf -inf-step inf-step) (break "ise") nil)
	(raise-error (error-message '(inf_step extract not)))))
  
(defunml (|inf_step_modify_inf_tree| (inf-step inf-tree))
     (inf_step -> (inf_tree -> inf_step))
  (inf-step-modify-inf-tree inf-step inf-tree))

(defunml (|inf_step_abbreviate| (inf-step))
    (inf_step -> inf_step)
  (inf-step-abbreviate inf-step))




;;;;	stats
;;;;	
;;;;	walk over list of proof objects and collect/print stats.
;;;;	

(defun collect-inf-stats (it acc)

  ;; only need to collect from top level tree ?
  
  (let ((inf-objc (objc-of-inf-tree it)))
    (mapc #'(lambda (ianno)
	      (when (and (iannotation-term-p ianno)
			 (eql 'refine-stats (label-of-iannotation-term ianno)))
		(refine-stats-acc acc (term-to-refine-stats (term-of-iannotation-term ianno)))))
		  
	  (annotations-of-inf-step (step-of-inf-objc-r inf-objc)))

    (mapc #'(lambda (it) (collect-inf-stats it acc))
	  (children-of-inf-tree it))))


(defun collect-proof-group-stats (stream descriptor oids)
  (let ((acc (new-refine-stats-acc)))

    (dolist (oid oids)

      (with-ignore
	
	  (let ((objc (objc-of-library-object (library-lookup oid))))
	    (when (eql `stm (kind-of-objc objc))
	      (dolist (prf (proofs-of-statement-source (source-of-objc objc)))
		(let ((pobjc (objc-of-library-object (library-lookup prf))))
		  (when (eql `prf (kind-of-objc pobjc))
		    (let ((pacc (new-refine-stats-acc)))

		      (collect-inf-stats (inf-tree-of-proof-source-r (source-of-objc pobjc) t) pacc)

		      (when stream (report-refine-stats stream
							pacc
							(let ((itok (property-of-objc objc 'name)))
							  (when itok (token-of-itoken-term itok)))))
		      (refine-stats-acc acc pacc)))))))))
    
    (when stream
      (report-refine-stats stream acc descriptor))

    acc))


;; lists : (descriptor . oids) list
(defun report-proof-stats (fname lists descriptor)

  (with-open-file (s fname :direction :output)
    (cond

      ((null lists) (report-hdr s descriptor "None"))

      ((null (cdr lists))
       (report-hdr s descriptor "")
       (collect-proof-group-stats s (caar lists) (cdar lists)))

      (t (let ((acc (new-refine-stats-acc)))
	   (report-hdr s descriptor "Start")
	   (dolist (l lists)
	     (refine-stats-acc acc (collect-proof-group-stats s (car l) (cdr l))))
	   (report-refine-stats s acc descriptor))))))
  

;; report_proof_stats "~/fu.report" ((`all` , (lib_statements (lib_list()))) . nil) `all`;;
(defunml (|report_proof_stats| (fname lists desc))
    (string -> (((token |#| (object_id list)) list) -> (token -> unit)))

  (report-proof-stats fname lists desc)
  nil)


(defvar *ignore-annos* nil);; lal: can ignore these for metaprl, but need for analogies in nuprl
(defvar *max-subgoal-count* 32)
(defvar *max-depth* 4)
(defvar *subgoal-offset* 0)

(defun invisible-binding-p (binding)
  (unless (dummy-variable-id-p binding)
    (char= #\% (char (variable-id-to-string binding) 0))))

;; LAL should be done in the editor, 
;; numbers hyps and marks hyps repeated if same as their parent
(defun modify-sequent-term (sequent &optional (count 1) parent)
  
  (if (iinf-sequent-term-p sequent);;nv5 refiner
      
      (let* ((seq (sequent-of-iinf-sequent-term sequent))
	     (type (type-of-iinf-sequent-term sequent))
	     (binding (binding-of-sequent-of-iinf-sequent-term sequent))
	     (repeat-p (and parent
			    (iinf-sequent-term-p parent)
			    (equal-terms-p type (type-of-iinf-sequent-term parent))
			    (equal-bindings binding (binding-of-sequent-of-iinf-sequent-term parent)))))
	
	(ipui-sequent-term count
			   repeat-p 
			   (invisible-binding-p binding)
			   (iinf-sequent-term (hidden-of-iinf-sequent-term sequent)
					      type
					      binding
					      (if (iinf-sequent-term-p seq)
						  (modify-sequent-term seq (1+ count)
								       (if (and parent (iinf-sequent-term-p parent))
									   (sequent-of-iinf-sequent-term parent)))
						  (ipui-sequent-term 0 nil nil seq)))))

      ;; need to account for mp-sequent-term
      (if (imp-msequent-term-p sequent);;metaprl refiner

	  (let* ((assums (map-isexpr-to-list (assums-of-imp-msequent-term sequent) 
					     (imcons-op) #'(lambda (x) x)))
		 (num count)
		 (nassums (map-sexpr-to-isexpr assums (imcons-nil-term)
					       #'(lambda (x) (prog1 (inum-sequent-term (ipui-hyp-term num nil nil) x)
							       (setf num (1+ num))))))
		 (goal (goal-of-imp-msequent-term sequent)))
	    
	    (imp-msequent-term nassums
			       (if (mp-sequent-p goal)
				   (mp-sequent (term-of-bound-term (car (bound-terms-of-term goal)))
					       (number-goal-term (term-of-bound-term (cadr (bound-terms-of-term goal)))))
				   goal)))
	  sequent)))


#|
;;not used
(defun step-to-refinement-term (step &optional top-p)
  ;;(break "p")
  (case (type-of-inf-step step)
    (unrefined (icons-left-term (itext-term "") (iinf-unrefined-term)))
    (top (let ((tac (tactic-of-inf-step step)))
	   (icons-left-term tac
			    (iinf-top-term (direct-inf-step-p step)
					   (environment-dependencies-to-term (references-of-inf-step step)) 
					   (inf-tree-to-term (inf-tree-of-inf-step step))
					   ;;(inf-tree-to-iinf-tree-term (inf-tree-of-inf-step step) t t)
					   ;;last arg nil lal? redundant here too, maybe just send oid? or oid tree (with childs)
					   tac))))

    (primitive (icons-left-term	(itext-term "")
				(iinf-primitive-term (environment-dependencies-to-term
						      (references-of-inf-step step))
						     (inf-extract-to-term (extract-of-inf-step step)))))
    (abbrev (let ((tac (tactic-of-inf-step step)))
	      (icons-left-term tac
			       (iinf-abbrev-term nil
						 (environment-dependencies-to-term
						  (references-of-inf-step step))
						 (inf-extract-to-term (extract-of-inf-step step))
						 tac))))))
|#

;; (oid # goal # annos # status # tactic) tree
;; abstractly-to-iproof-node-term
;;   goalf:(node -> oid # goal # annos)
;;   statusf:(node -> status)
;;   tacticf:node->tactic 
;;   childrenf: tree -> tree list
;;   tree

(defun inf-tree-to-iproof-node-term (itree status addr view &optional parent (depth 0) (number-p t))
  ;;(setf ii itree ss status aa addr vv view) (break "itpet")
  (labels

      ((visit (inf-tree status address sequent d)
	 (let* ((oid (if (inf-objc-tree-p inf-tree)
			 *null-oid*
			 (oid-of-inf-tree inf-tree)))
		(objc (if (inf-objc-tree-p inf-tree)
			  (objc-of-inf-tree inf-tree)
			  (oc oid)))
		(step (step-of-inf-objc-r objc))
		(gterm (goal-to-term (goal-of-inf-step step)))
		(seq (sequent-of-iinf-goal-term gterm))
		(children (children-of-inf-tree inf-tree))
		(count (length children))
		(stat (if (and (unrefined-inf-step-p step) children)
			  (iproof-status-term (intern-system "unknown"))
			  status)))
		    
	   (iproof-node-term oid d view count stat
			     address				 
			     (iinf-goal-term (if number-p (modify-sequent-term seq 1 sequent) seq)
					     (if *ignore-annos* 
						 (iannotation-nil-term)
						 (annotations-of-iinf-goal-term gterm)))
	    
			     (icons-left-term (or (term-of-source (objc-source objc))
						  (if (unrefined-inf-step-p step)
						      (itext-term "")
						      (or (tactic-of-inf-step step)
						          (itext-term ""))))
					      (ioid-term oid))
	       
			     (let* ((index 1)				    
				    (subgoals
				     (map-sexpr-to-isexpr
				      (if (<= d *max-depth*) 
					  children
					  (progn (format t "~%Tree depth exceeds max.~%") children))
				      (iproof-node-nil-term)
				      #'(lambda (inf-tree)
					  (prog1 (visit inf-tree
							(if inf-tree
							    (itree-status inf-tree)
							    (iproof-status-term `|complete|)) 
							(ipui-addr-cons-term (ipui-addr-term index) address)
							seq (1+ d))

					    (setf index (1+ index)))))))
       
			       (when (> count *max-subgoal-count*)
				 (format t "~%~%Number of subgoals is ~s, exceeded max.~%~%" count)
				 ;;(break)
				 ;;(raise-error (error-message '(too many subgoals) (1- count)))
				 )
			       subgoals)
	     
			     (if *ignore-annos* 
				 (iannotation-nil-term)
				 (annotations-to-term (annotations-of-inf-objc-src objc)))))))
   
    (visit itree status addr parent depth)))

#|
(defun primitive-inf-tree-to-iproof-node-term (itree status addr view &optional parent (depth 0))
  ;;(setf ii itree ss status aa addr) (break "pit")

  (labels
      ((visit (inf-tree stat address sequent d)
	   
	 (let* ((oid (oid-of-inf-tree inf-tree))
		(objc (oc oid))
		(step (step-of-inf-objc-r objc))
		(gterm (goal-to-term (goal-of-inf-step step)))
		(seq (sequent-of-iinf-goal-term gterm))
		(subgoals (children-of-inf-tree inf-tree))
		(count (length subgoals)))
		    
	   (iproof-node-term oid d view count stat address
			     (iinf-goal-term (modify-sequent-term seq 1 sequent)
					     (if *ignore-annos* 
						 (iannotation-nil-term)
						 (annotations-of-iinf-goal-term gterm)))
	    
			     (icons-left-term (if (unrefined-inf-step-p step)
						  (itext-term "")
						  (or (term-of-source (objc-source objc))
						      (tactic-of-inf-step step)
						      (itext-term "")))
					      (ioid-term oid))
	       
			     (let* ((index 1)				    
				    (children
				     
				     (let* ((refinement (refinement-of-inf-step step))
					    (it2 (when (and (top-refinement-p refinement)
							    (top-refinement-direct-p refinement))
						   (inf-tree-of-inf-step step))))
				       ;;(setf ii it2)(break)
				       (if it2 (append subgoals (list it2)) subgoals)))
					 
				    (subs (map-sexpr-to-isexpr (if (<= d *max-depth*) 
								   children
								   (progn;;(format t "~%Tree depth exceeds max.~%")
								     children))
							       (iproof-node-nil-term)
							       #'(lambda (it)
								   (prog1
								       (visit it
									      (if it (itree-status it) (iproof-status-term `|complete|)) 
									      (ipui-addr-cons-term (ipui-addr-term index) address)
									      seq (1+ d) t)

								     (setf index (1+ index)))))))
       
			       ;;(when (> count *max-subgoal-count*)
			       ;;(format t "~%~%Number of subgoals is ~s, exceeded max.~%~%" count)
			       ;;(break)
			       ;;(raise-error (error-message '(too many subgoals) (1- count)))
			       ;;)
			       subs)
	      
			     (if *ignore-annos* 
				 (iannotation-nil-term)
				 (annotations-to-term (annotations-of-inf-objc-src objc)))))))
    
    (visit itree status addr parent depth)))
|#

(defun interior-inf-tree-to-iproof-node-term (itree status addr view &optional parent (depth 0))
 ;; (setf ii itree ss status aa addr) (break "pit")

  (labels
      ((visit (inf-tree stat address sequent d)
	   
	 (let* ((oid (oid-of-inf-tree inf-tree))
		(objc (oc oid))
		(step (step-of-inf-objc-r objc))
		(gterm (goal-to-term (goal-of-inf-step step)))
		(seq (sequent-of-iinf-goal-term gterm))
		(subgoals (children-of-inf-tree inf-tree))
		(count (length subgoals))
		(goal (iinf-goal-term (modify-sequent-term seq 1 sequent)
				      (if *ignore-annos* 
					  (iannotation-nil-term)
					  ;;(cons (iannotation-term (ianno-label (intern "internal")))
					  (annotations-of-iinf-goal-term gterm))))
		(ref (icons-left-term (if (unrefined-inf-step-p step)
					  (itext-term "")
					  (or (term-of-source (objc-source objc))
					      (tactic-of-inf-step step)
					      (itext-term "")))
				      (ioid-term oid)))
		(index 1)				    
		(children (let* ((refinement (refinement-of-inf-step step))
				 (it2 (when (and (top-refinement-p refinement)
						 (top-refinement-direct-p refinement))
					(inf-tree-of-inf-step step))))			    
			    (if it2 (append subgoals (list it2)) subgoals)))
					 
		(subs (map-sexpr-to-isexpr (if (<= d *max-depth*) 
					       children
					       (progn (format t "~%Tree depth exceeds max.~%")
						      children))
					   (iproof-node-nil-term)
					   #'(lambda (it)
					       (prog1 (visit it (itree-status it)
							     (ipui-addr-cons-term (ipui-addr-term index)
										  address)
							     seq (1+ d))
						 (setf index (1+ index))))))
		(annos (if *ignore-annos* 
			   (iannotation-nil-term)
			   (annotations-to-term (annotations-of-inf-objc-src objc)))))
       
	   (iproof-node-term oid d view count stat address goal ref subs annos))))
    
    (let* ((objc (objc-of-inf-tree itree))
	   (step (step-of-inf-objc-r objc))
	   (refinement (refinement-of-inf-step step))
	   (it2 (when (and (top-refinement-p refinement)
			   (top-refinement-direct-p refinement))
		  (inf-tree-of-inf-step step))))
      ;;(setf ii it2)(break)
      (visit (or it2 itree)
             ;;itree
	     status addr parent depth))))

(defun primitive-inf-tree-to-iproof-node-term
    (itree status addr view &optional parent (depth 0))
  ;;(setf ii itree ss status aa addr) (break "pit")

  (labels
      ((visit (inf-tree stat address sequent d)
	   
	 (let* ((objc (objc-of-inf-tree inf-tree))
		(step (step-of-inf-objc-r objc))
		(gterm (goal-to-term (goal-of-inf-step step)))
		(seq (sequent-of-iinf-goal-term gterm))
		(subgoals (children-of-inf-tree inf-tree))
		(count (length subgoals))
		(goal (iinf-goal-term (modify-sequent-term seq 1 sequent)
				      (if *ignore-annos* 
					  (iannotation-nil-term)
					  ;;(cons (iannotation-term (ianno-label (intern "internal")))
					  (annotations-of-iinf-goal-term gterm))))
		(ref (icons-left-term (if (unrefined-inf-step-p step)
					  (itext-term "")
					  (or (term-of-source (objc-source objc))
					      (tactic-of-inf-step step)
					      (itext-term "")))
				      (ioid-term *null-oid*)))
		(index 1)				    
		(children (let* ((refinement (refinement-of-inf-step step))
				 (it2 (when (and (top-refinement-p refinement)
						 (top-refinement-direct-p refinement))
					(inf-tree-of-inf-step step))))			    
			    (if it2 (append subgoals (list it2)) subgoals)))
					 
		(subs (map-sexpr-to-isexpr (if (<= d *max-depth*) 
					       children
					       (progn (format t "~%Tree depth exceeds max.~%")
						      children))
					   (iproof-node-nil-term)
					   #'(lambda (it)
					       (prog1 (visit it (itree-status it)
							     (ipui-addr-cons-term (ipui-addr-term index)
										  address)
							     seq (1+ d))
						 (setf index (1+ index))))))
		(annos (if *ignore-annos* 
			   (iannotation-nil-term)
			   (annotations-to-term (annotations-of-inf-objc-src objc)))))
       
	   (iproof-node-term *null-oid* d view count stat address goal ref subs annos))))
    
    (let* ((objc (objc-of-inf-tree itree))
	   (step (step-of-inf-objc-r objc))
	   (refinement (refinement-of-inf-step step))
	   (it2 (when (and (top-refinement-p refinement)
			   (top-refinement-direct-p refinement))
		  (inf-tree-of-inf-step step))))
      ;;(setf ii it2)(break)
      (visit (if it2 it2 itree) status addr parent depth))))

;;modifies step, 
(defun iproof-node-term-to-inf-tree-modify (node)  
  (let ((oid (oid-of-ioid-term (right-of-icons-left-term (refinement-of-iproof-node-term node)))))    
    (let ((inf-step (step-of-inf-objc-r (oc oid))))
      (setf (inf-step-goal inf-step) 
	    (cons (sequent-of-iinf-goal-term 
		   (goal-of-iproof-node-term node)) nil))) 

    (inf-tree oid (map-isexpr-to-list (subgoals-of-iproof-node-term node)
				      (iproof-node-cons-op)
				      #'iproof-node-term-to-inf-tree))))    

(defun iproof-node-term-to-inf-tree (node)
  ;;(setf -node node) (break "pnt")
  (let ((ioid (oid-of-ioid-term (right-of-icons-left-term (refinement-of-iproof-node-term node)))))
    (inf-tree ioid (map-isexpr-to-list (subgoals-of-iproof-node-term node)
				       (iproof-node-cons-op)
				       #'iproof-node-term-to-inf-tree))))    

(defun find-annotation (label annos)
  (mapcan #'(lambda (iannotation)
		   (if (iannotation-term-p iannotation)
		       (when (eql label (label-of-iannotation-term iannotation))
			     (list (term-of-iannotation-term iannotation)))
		     (message-emit (warn-message '(!annotation not) iannotation))))
	       annos))

(defunml (|address_of_goal| (goal))
  (term -> term)
  (car (find-annotation 'ADDRESS (annotations-of-goal (iinf-goal-term-to-goal goal)))))
  
(defun itree-complete-p (itree)
  (and (refined-inf-step-p (step-of-inf-objc-r (objc-of-inf-tree itree)))
       (forall-p #'itree-complete-p (children-of-inf-tree itree))))

(defun itree-status (itree)
  (if (itree-complete-p itree)
      (iproof-status-term `|complete|)  
    (iproof-status-term `|incomplete|))) 

(defun my-list (start finish)  
  (do ((i start (1+ i))
       (l nil (cons i l)))
      ((= i finish) (reverse (cons i l)))))

(defun my-list-out (start finish)  
  (do ((i start (1+ i)))     
      ((> i finish))
      (format t "~s~%" i)))

;; assumes binding is nil at the address fttb
(defun iinf-tree-replace-term-at-address (term new-term address)
  (let* ((goal (goal-of-iinf-tree-term term))
	(node (node-of-iinf-tree-term term))
	(c (children-of-iinf-tree-term term))
	(ch (map-isexpr-to-list c (iinf-tree-cons-op)))
	(annos (annotations-of-iinf-tree-term term)))

    (if (null address)
	(let ((d (dependencies-of-iinf-abbrev-term node))
	      (e (extract-of-iinf-abbrev-term node))
	      (b (direct-of-iinf-abbrev-term node)))
	  (iinf-tree-term goal 
			  (iinf-abbrev-term b d e new-term)
			  c
			  annos))
				
	(iinf-tree-term goal 
			node 
			(map-list-to-ilist
			 (replace-in-list (replace-term-at-address (nth (1- (car address)) ch)
								   new-term
								   (cdr address))
					  (car address)
					  ch)
			 (iinf-tree-nil-term))
			annos))))

(defunml (|iinf_tree_replace_term| (term new-term ints))
    (term -> (term -> ((int list) -> term)))
  (iinf-tree-replace-term-at-address term new-term ints))






;;;;
;;;; meta prl proof refinement
;;;;


(defun make-metaprl-term (params bterms)
  (instantiate-term
   (instantiate-operator (intern-system "!metaprl_implementation") params)
   bterms))

(defun metaprl-term-p (term)
  (equal (id-of-term term) '|!metaprl_implementation|))

(defun metaprl-term-with-opname-p (term opname)
  (let* ((parameter (car (parameters-of-term term)))
	 (value (value-of-parameter parameter)))
    (and value
	 (parameter-list-typeid-p (type-id-of-type (type-of-parameter parameter)))
	 (equal-parameters-p  (car (last value)) (string-parameter opname)))))

(defun mp-context-p (term)
  (and (metaprl-term-p term)
       (let ((parameters (parameters-of-term term)))
	 (and (= (length parameters) 2)
	      (equal-parameters-p (car parameters)
				  (parameter-list-parameter (list (string-parameter "context"))))))))
  
(defun mp-concl-p (term)
  (and (metaprl-term-p term)
       (metaprl-term-with-opname-p term "concl")))
  
(defun mp-sequent-p (term)
  (and (metaprl-term-p term)
       (metaprl-term-with-opname-p term "sequent")))
  
(defun mp-hyp-p (term)
  (and (metaprl-term-p term)
       (metaprl-term-with-opname-p term "hyp")))

;;!metaprl_implementation{{context:s}:parameter-list, H:variable}
(defun mp-context (params bterms)
  (make-metaprl-term (cons (parameter-list-parameter (list (string-parameter "context")))
			    params)
		      (list bterms)))
  
(defun mp-sequent (term1 term2)
  (make-metaprl-term (list (parameter-list-parameter (list (string-parameter "Perv")
							    (string-parameter "sequent"))))
		      (list (cons nil term1) (cons nil term2))))
  
(defun mp-hyp (bterms)
  (make-metaprl-term (list (parameter-list-parameter (list (string-parameter "Perv")
							    (string-parameter "hyp"))))
		      
		      bterms))
  
;;term is !metaprl_implementation{{Perv:s,sequent:s}:parameter-list}
(defun number-goal-term (goal &optional (count 1))
  (cond ((mp-concl-p goal)
	 goal)
	
	((mp-context-p goal)
	 (let ((length (length (bound-terms-of-term goal))))
           (cond ((= length 1) 
		  (inum-sequent-term (ipui-hyp-term count nil nil)
				     (make-metaprl-term (parameters-of-term goal)
							(list (cons nil (number-goal-term (term-of-bound-term (car (bound-terms-of-term goal))) (1+ count)))))))
		 ((= length 2)
		  (inum-sequent-term (ipui-hyp-term count nil nil)
				     (make-metaprl-term (parameters-of-term goal)
							(let ((bts (bound-terms-of-term goal)))
							  (list (car bts)
								(cons nil (number-goal-term (term-of-bound-term (cadr bts))
											    (1+ count))))))))
		 (t (break "number-goal context term exceeds 2")))))
	
	((mp-hyp-p goal)
	 (inum-sequent-term (ipui-hyp-term count nil nil) 
			    (make-metaprl-term (parameters-of-term goal)
					       (let ((bts (bound-terms-of-term goal)))
						 (list (cons nil (term-of-bound-term (car bts)))
						       (cons (bindings-of-bound-term (cadr bts))
							     (number-goal-term (term-of-bound-term (cadr bts))
									       (1+ count))))))))
	(t goal)))

;;used for mp
(defun unnumber-goal-term (goal)
  (cond ((inum-sequent-term-p goal)
	 (let ((seq (sequent-of-inum-sequent-term goal)))   
	   (cond ((mp-concl-p seq) goal)
	
		 ((mp-context-p seq)
		  (let ((length (length (bound-terms-of-term seq))))
		   
		    (cond ((= length 1)
			   (make-metaprl-term (parameters-of-term seq)
					      (list (cons nil (unnumber-goal-term
							       (term-of-bound-term (car (bound-terms-of-term seq))))))))
			  ((= length 2)
			   (make-metaprl-term (parameters-of-term seq)
					      (list (car (bound-terms-of-term seq))
						    (cons nil (unnumber-goal-term
							       (term-of-bound-term (cadr (bound-terms-of-term seq))))))))
			  (t (break "unnumber-goal context term exceeds 2")))))
		 
		 ((mp-hyp-p seq)
		  (make-metaprl-term (parameters-of-term seq)
				     (let ((bts (bound-terms-of-term seq)))
				       (list (cons nil (term-of-bound-term (car bts)))
					     (cons (bindings-of-bound-term (cadr bts))
						   (unnumber-goal-term (term-of-bound-term (cadr bts)))))))))))
	(t goal)))

;;used for mp
(defun unnumber-sequent-term (sequent)
  ;;(setf -sequent sequent) (break)	    

  (if (imp-msequent-term-p sequent);;metaprl refiner

      (let* ((assums (map-isexpr-to-list (assums-of-imp-msequent-term sequent) 
					 (imcons-op)
					 #'(lambda (x) x)))
	     (uassums (map-sexpr-to-isexpr assums
					   (imcons-nil-term)
					   #'(lambda (x) (sequent-of-inum-sequent-term x))))		       
	     (goal (goal-of-imp-msequent-term sequent)))
		  
	(imp-msequent-term uassums
			   (mp-sequent (term-of-bound-term (car (bound-terms-of-term goal)))
				       (unnumber-goal-term (term-of-bound-term (cadr (bound-terms-of-term goal)))))))
				  
      sequent))

(defvar *imcons* '|!mcons|)
(defun imcons-op () (instantiate-operator *imcons* nil))
(defun imcons-nil-term () (instantiate-term (imcons-op) nil))

;; works for faux refine test
(defun dummy-up-nl (goal tac icons &optional extras)
  (setf a goal b tac c icons d
	(let ((subgoals (map-isexpr-to-list icons (icons-op) #'(lambda (x) x))))

	  (let ((dummy-deps (ienvironment-dependencies-term (stamp-to-term
							     (dummy-transaction-stamp))
							    (idependencies-nil-term))))
    
	    (iinf-tree-term (iinf-goal-term goal (iannotation-nil-term))
			    (iinf-top-term t
					   dummy-deps
					   (iinf-tree-term (null-iinf-goal-term)
							   (iinf-primitive-term dummy-deps
										(null-iinf-extract-term))
							   (map-list-to-ilist subgoals
									      (iinf-tree-nil-term)
									      #'(lambda (g)
										  (declare (ignore g))
										  (iinf-tree-term 
										   (null-iinf-goal-term)
										   (iinf-unrefined-term)
										   (iinf-tree-nil-term)
										   (iannotation-nil-term))))
							   (iannotation-nil-term))
					   tac)
			    (map-list-to-ilist subgoals
					       (iinf-tree-nil-term)
					       #'(lambda (g)
						   (iinf-tree-term g
								   (iinf-unrefined-term)
								   (iinf-tree-nil-term)
								   (iannotation-nil-term))))
			    (iannotation-nil-term)))))

  ;;(break "dummy")
  d)

(defun dummy-nl-to-inf-tree (goal tac subgoals extras)
  (setf a goal b tac c subgoals d
	(let ((subs (map-isexpr-to-list subgoals (icons-op) #'(lambda (x) x))))

	  (let ((dummy-deps (ienvironment-dependencies-term (stamp-to-term
							     (dummy-transaction-stamp))
							    (idependencies-nil-term))))
    
	    (iinf-tree-term (iinf-goal-term goal (iannotation-term 'extras extras))
			    (iinf-top-term t
					   dummy-deps
					   (iinf-tree-term (null-iinf-goal-term)
							   (iinf-primitive-term dummy-deps
										(null-iinf-extract-term))
							   (map-list-to-ilist subs
									      (iinf-tree-nil-term)
									      #'(lambda (g)
										  ;;(declare (ignore g))
										  (iinf-tree-term
										   (iinf-goal-term g (iannotation-nil-term))
										   ;;(null-iinf-goal-term)
										   (iinf-unrefined-term)
										   (iinf-tree-nil-term)
										   (iannotation-nil-term)
										   )))
							   (iannotation-nil-term))
					   tac)
			    (map-list-to-ilist subs
					       (iinf-tree-nil-term)
					       #'(lambda (g)
						   (iinf-tree-term (iinf-goal-term g (iannotation-nil-term))
								   (iinf-unrefined-term)
								   (iinf-tree-nil-term)
								   (iannotation-nil-term))))
			    (iannotation-nil-term)))))

  ;;(break "dummy")
  d)

(defun dummy-nl-to-proof (goal tac subgoals extras)
  ;;(setf a goal b tac c subgoals) (break "df")
  (let ((subs (map-isexpr-to-list subgoals (icons-op) #'(lambda (x) x))))

    (let ((dummy-deps (ienvironment-dependencies-term (stamp-to-term
						       (dummy-transaction-stamp))
						      (idependencies-nil-term))))
    
      (iinf-tree-term
       (iinf-goal-term goal (if (inil-term-p extras)
				(iannotation-nil-term)
				(iannotation-term 'extras extras)))
       (if (ivoid-term-p tac)
	   (iinf-unrefined-term)
	   (iinf-top-term t
			  dummy-deps
			  (iinf-tree-term (null-iinf-goal-term)
					  (iinf-primitive-term dummy-deps
							       (null-iinf-extract-term))
					  (map-list-to-ilist subs
							     (iinf-tree-nil-term)
							     #'(lambda (g)
								 ;;(declare (ignore g))
								 (iinf-tree-term 
								  (iinf-goal-term g (iannotation-nil-term));;(null-iinf-goal-term)
								  (iinf-unrefined-term)
								  (iinf-tree-nil-term)
								  (iannotation-nil-term))))
					  (iannotation-nil-term))
			  tac))
       (map-list-to-ilist subs
			  (iinf-tree-nil-term)
			  #'(lambda (g)
			      (dummy-nl-to-proof (goal-of-inl-prf-term g)
						 (tactic-of-inl-prf-term g)
						 (subgoals-of-inl-prf-term g)
						 (extras-of-inl-prf-term g))))
       (iannotation-nil-term)))))

					 

(defunml (|dummy_up_mp| (g tac l))
    (term -> (term -> (term -> term)))

  (dummy-up-nl g tac l))

(defunml (|dummy_mp_to_inf_tree| (g tac s e))
    (term -> (term -> (term -> (term -> term))))

  (dummy-nl-to-inf-tree g tac s e))

(defunml (|dummy_mp_to_proof| (g tac s e))
    (term -> (term -> (term -> (term -> term))))

  (dummy-nl-to-proof g tac s e))


;;;;
;;;; end of metaprl stuff
;;;;

  
(defunml (|unnumber_sequent_term| (term))
   (term -> term)
 (unnumber-sequent-term term))

(defunml (|sequent_of_goal| (term))
  (term -> term)
  ;;(setf tt term)(break)
  (sequent-of-iinf-goal-term term))
  
(defunml (|symaddr_of_goal| (goal))
    (term -> term)
  (or (car (find-annotation 'SYMBOLICADDRESS (annotations-of-goal (iinf-goal-term-to-goal goal))))
      (break "symaddr?") (raise-error '(symaddr of goal not))))
  
(defunml (|address_to_annotation| (address))
  (term -> term)
  (iannotation-term 'ADDRESS address))

(defunml (|symaddr_to_annotation| (addr))
  (term -> term)
  (iannotation-term 'SYMBOLICADDRESS addr))

(defun unmodify-sequent-term (sequent)
  ;;(setf -sequent sequent) (break)	    
  (if (iinf-sequent-term-p sequent);;nv5 refiner      
      (break)	    
      (if (imp-msequent-term-p sequent);;metaprl refiner
	  (let* ((assums (map-isexpr-to-list (assums-of-imp-msequent-term sequent) 
					     (imcons-op) #'(lambda (x) x)))
		 (uassums (map-sexpr-to-isexpr assums (imcons-nil-term)
					       #'(lambda (x) (sequent-of-inum-sequent-term x))))
		 (goal (goal-of-imp-msequent-term sequent)))
	    (imp-msequent-term uassums
			       (mp-sequent (term-of-bound-term (car (bound-terms-of-term goal)))
					   (unnumber-goal-term (term-of-bound-term
								(cadr (bound-terms-of-term goal)))))) 
	    sequent))))

(defvar *primitive-tree* nil)

(defunml (|set_primitive_tree| (val))
    (bool -> unit)
  (setf *primitive-tree* val))
