
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;
;;;; -docs- (mod lib)
;;;;
;;;;	
;;;;	Database : (db) Low level data store.
;;;;
;;;;	DB is a common resource to all environments in process. DB functions are
;;;;	available to all environments in a process via direct calls.
;;;;	Inter-process calls can be effected by sending a request to any
;;;;	environment in the process(eg ORB).
;;;;
;;;;	Caching, log record keeping, and garbage collection functionality are provided.
;;;;
;;;;
;;;;    db-init(<pathname{master}>)	: NULL
;;;;	  * sets global db directory
;;;;
;;;;    db-query (&optional <pathname{master}>)
;;;;      : <log-description> list
;;;;	  * if pathname not specified, uses pathname from db-init
;;;;
;;;;    db-persistent-p (<stamp> <tok{type}>)   : BOOL
;;;;	  * true if stamp and type have associated data in db.
;;;;	
;;;;	db-read (<stamp> <tok{type}>)		: <term> 	  
;;;;	  * stamp can be for any process. 
;;;;	  ** emits warning and returns nil if file is either not readable,
;;;;	    still open for writing, or does not exist
;;;;
;;;;	db-read-ahead (<stamp> <tok{type}>)	: NULL
;;;;	  * if MTT then read into cache. Otherwise, ignore.
;;;;
;;;;	db-write (<stamp> <tok{type}> <term>)	: NULL
;;;;	  * stamp will always be for current process.
;;;;	  ** creates a new file (could append in future)
;;;;
;;;;	
;;;;	Users can label points in a log via checkpoints.
;;;;	Checkpoint broadcasts will be written to log. In addition the db will 
;;;;	maintain an assoc list of checkpoints to logs. This will aid in finding
;;;;	logs containing checkpoints and listing checkpoints in the log.
;;;;
;;;;	db-inform-checkpoint (<stamp{log}>, <stamp{checkpoint})	: NULL
;;;;	  * writes log and checkpoint stamps to checkpoint log file
;;;;	db-query-checkpoint (<stamp{checkpoint}>)		: <stamp{log}>
;;;;	  * fails if checkpoint not found for stamp.
;;;;	db-query-checkpoints (<stamp{log}>)			: <stamp{checkpoint}> list
;;;;	  * fails if checkpoint not found for stamp.
;;;;	
;;;;	db-log-p (<stamp>)					: BOOL
;;;;	  * true if there is a log file associated with this stamp.
;;;;	
;;;;	There may be multiple checkpoints per log or none.
;;;;	
;;;;
;;;;	*db-cache-size*
;;;;	*db-assoc-list*
;;;;	  * associates checkpoints to logs
;;;;
;;;;	db-process-list()		:process list
;;;;	db-files-of-process(process)	:data file list
;;;;	db-logs-of-process(process)	:log file list
;;;;
;;;;	db-extend-directory(directory list)		:directory
;;;;	db-extend-pathname(&optional directories filename type)		:pathname
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Logs:
;;;;
;;;;	Logs are a record of references to terms in a particular environment
;;;;	within a particular process.  Logs are either opened for writing or
;;;;	reading (never bidirectional).  Once a write log is closed, it is not modified.
;;;;	
;;;;	<wlog>		: output <log>
;;;;	<rlog>		: input <log>
;;;;
;;;;	<log>		: <term> record
;;;;	  * filename derived from stamp is process-id/sequence.LOG
;;;;
;;;;	Information is available for each log:
;;;;	 <log-info> : log-info [<log-date{accessed}>
;;;;                            <log-date{modified}>
;;;;                            <env-description>]
;;;;	 * fields may be added or modified in the future if more data becomes desirable
;;;;	   (ie possibly include #checkpoints of each log, unbinds?, #references/descendents, userid)
;;;;
;;;;	<log-header>	: log-header [<stamp{parent}> <stamp{log}> <env-description> <log-properties>]
;;;;	  * logs are connected via parent pointers in log header, parent may be !void
;;;;	  ** header is maintained in separate file, ie for each sequence.LOG file there
;;;;	     exists a sequence.HDR file
;;;;     LAL perf, write parent 3rd to avoid reading at query
;;;;
;;;;	<log-properties> : !property {count}(!natural{<p>:n})
;;;;			   !property {owner}(!bool{<p>:b}) t if active to owner
;;;;			   !property {permanent}(!bool{<p>:b})
;;;;			   !property {garbage}(!bool{<p>:b})
;;;;
;;;;
;;;;	  * reference count is number of active immediate children of log
;;;;	  ** recorded to insure safe garbage collection within a shared database 
;;;;
;;;;
;;;;	log-description-to-term (<log-description>)	:term
;;;;	 * display forms are provided for log description terms
;;;;
;;;;	log-info-to-term (<log-info>)	:term
;;;;	 * display forms are provided for log info terms
;;;;	
;;;;    <env-description>       : !environment  ( <tokens{address}>
;;;;                                            ; <tokens{purposes}>
;;;;                                            ; <tokens{resources}>
;;;;                                            ; <tokens{table-types}>
;;;;                                            ; <tokens{reduction-tags}>
;;;;                                            )
;;;;
;;;;    <tokens>        : !tokens{<tok:t> list}()
;;;;
;;;;	<log-tree>	: db-tree~ [<stamp>
;;;;				   <db-tree{child}> list
;;;;				   <db-tree{parent}>
;;;;				   <INT{ancestor-count}>
;;;;				   <BOOL{active-p}>
;;;;				   <INT{ownerid}>]
;;;;      * log-tree only constructed for garbage collection purposes
;;;;
;;;;
;;;;    new-log-open (<stamp> <env-description>)      : <wlog>
;;;;	  * creates process directory if does not already exist
;;;;	  ** creates a log file without parent, creates and writes header file
;;;;    log-open-write (<stamp> <env-description> <stamp{parent}>)      : <wlog>
;;;;	  * creates process directory if does not already exist
;;;;	  ** creates a log file with parent, creates and writes header file
;;;;
;;;;    log-write-record (<wlog> <term>)	: NULL
;;;;	  
;;;;    log-open-read (<stamp>)                 : <rlog>
;;;;    log-read-record (<rlog>)                : <term>
;;;;
;;;;    log-close (<rlog>|<wlog>)               : NULL
;;;;
;;;;    log-query (<stamp>)                     : <env-description>
;;;;
;;;;
;;;;	root-log-p (<stamp{log}>)		: bool
;;;;	  * true if log has no parent
;;;;	build-log-tree (logs)			: NULL
;;;;	  * visits logs, gathering parent/child relationship data into *log-tree*
;;;;	ancestors-of-log (<stamp{log}> &optional INTEGER{depth}): <log> list
;;;;	  * returns list of ancestors of length <= depth	
;;;;
;;;;  -page-
;;;;

;;;;	
;;;;	Compression. at some point should move to 3byte indices.
;;;;	
;;;;	



;;;;    open process cannot record data persist term from non persist term reference

;;;; idiosyncrasies of c access time, libc-atime(file):
;;;;  case 1.  File opened with call to open:  
;;;;  time only changes when file is read-lock and even then only after first read-char 
;;;;  (maybe since only first byte is locked) (this is ok)
;;;;  case 2.  File opened using with-open-file:
;;;;  time changes with every read (as expected)--
;;;;  don't want access time to change due to query so keep header separate (better) or
;;;;  don't lock and call open.

;;;;	
;;;;	
;;;;	IO buffering : 
;;;;	  - term-writes buffered.
;;;;	  - link writes inlined if stamp is bufferred.
;;;;	
;;;;	
;;;;	IO taxonomy
;;;;	
;;;;	- db-write-term
;;;;	    - db-write
;;;;		- persist-data : buffer persist
;;;;	    - db-inform-checkpoint : flush then write
;;;;		- journal checkpoint stuff
;;;;	    - db-write-header : flush then write
;;;;		- db-log-file-open-write
;;;;	    - log-write-record-aux : interleave with persist buffer or give persist-buffer pri
;;;;		- log-buffer-write
;;;;		- log-write-record
;;;;	
;;;;	- db-write-aux : flush then write
;;;;	    add-log-derivation   
;;;;	
;;;;	- log-write-append : flush then write
;;;;	    - loglog
;;;;	        - db-log-open
;;;;	            - journal checkpoint stuff
;;;;	

;;;; FFI / file system functionality

#+lucid (load-foreign-libraries nil)


#+allegro-v3.1
(require :foreign)


#+cmu
(progn
  (defunff (libc-mkdir) int ((name string)))
  (defunff (umask libc-umask) int ((name int)))
  (defunff (rename libc-rename) int ((old string) (new string)))
  
  (defunff (read-lock-test) int ((fd int)))
  (defunff (write-lock-test) int ((fd int)))
  (defunff (read-lock) int ((fd int)))
  (defunff (write-lock) int ((fd int)))
  (defunff (isread-lock) int ((fd int)))
  (defunff (iswrite-lock) int ((fd int)))
  (defunff (iswrite-lock-file) int ((name string)))
  (defunff (un-lock unlock-fd) int ((fd int)))
  (defunff (un-lock-read unlock-fd-read) int ((fd int)))

  (defunff (owner-uid) int ((name string)))
  (defunff (libc-atime) int ((name string)))
  (defunff (libc-asctime) int ((name string)))
  (defunff (libc-create) int ((name string)))
  (defunff (libc-rmdir) int ((name string)))
  )  


#+lucid
(progn
  (def-foreign-function (libc-mkdir (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_libc_mkdir"))
      (name :simple-string))
  (def-foreign-function (libc-chmod (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_libc_chmod"))
      (name :simple-string))
  (def-foreign-function (libc-umask (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_umask"))
      (name :signed-32bit))
  (def-foreign-function (libc-rename (:return-type :signed-32bit)
				     (:language :c)
				     (:name "_rename"))
      (oldname :simple-string)
    (newname :simple-string))
  (def-foreign-function (read-lock-test (:return-type :signed-32bit)
					(:language :c)
					(:name "_read_lock_test"))
      (fd :signed-32bit))
  (def-foreign-function (write-lock-test (:return-type :signed-32bit)
					 (:language :c)
					 (:name "_write_lock_test"))
      (fd :signed-32bit))
  (def-foreign-function (read-lock (:return-type :signed-32bit)
				   (:language :c)
				   (:name "_read_lock"))
      (fd :signed-32bit))
  (def-foreign-function (write-lock (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_write_lock"))
      (fd :signed-32bit))
  (def-foreign-function (isread-lock (:return-type :signed-32bit)
				     (:language :c)
				     (:name "_isread_lock"))
      (fd :signed-32bit))
  (def-foreign-function (iswrite-lock (:return-type :signed-32bit)
				      (:language :c)
				      (:name "_iswrite_lock"))
      (fd :signed-32bit))
  (def-foreign-function (iswrite-lock-file (:return-type :signed-32bit)
					   (:language :c)
					   (:name "_iswrite_lock_file"))
      (name :simple-string))
  (def-foreign-function (unlock-fd (:return-type :signed-32bit)
				   (:language :c)
				   (:name "_un_lock"))
      (fd :signed-32bit))
  (def-foreign-function (unlock-fd-read (:return-type :signed-32bit)
					(:language :c)
					(:name "_un_lock_read"))
      (fd :signed-32bit))
  (def-foreign-function (set-modes (:return-type :signed-32bit)
				   (:language :c)
				   (:name "_read_modes"))
      (name :simple-string))
  (def-foreign-function (owner-uid (:return-type :unsigned-32bit)
				   (:language :c)
				   (:name "_owner_uid"))
      (name :simple-string))
  (def-foreign-function (libc-atime (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_libc_atime"))
      (name :simple-string))
  (def-foreign-function (libc-asctime (:return-type :simple-string)
				      (:language :c)
				      (:name "_libc_asctime"))
      (name :simple-string))
  (def-foreign-function (libc-create (:return-type :signed-32bit)
				     (:language :c)
				     (:name "_libc_create"))
      (name :simple-string))
  (def-foreign-function (libc-rmdir (:return-type :signed-32bit)
				    (:language :c)
				    (:name "_rmdir"))
      (name :simple-string)))
  
#+allegro
(progn
  (ff:defforeign 'libc-mkdir
		 :entry-point (ff:convert-to-lang "libc_mkdir")
		 :arguments '(string)
		 :return-type :integer)
  (ff:defforeign 'libc-chmod
		 :entry-point (ff:convert-to-lang "libc_chmod")
		 :arguments '(string)
		 :return-type :integer)
  (ff:defforeign 'libc-rmdir
		 :entry-point (ff:convert-to-lang "rmdir")
		 :arguments '(string)
		 :return-type :integer)
  (ff:defforeign 'libc-umask
		 :entry-point (ff:convert-to-lang "umask")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'read-lock
		 :entry-point (ff:convert-to-lang "read_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'write_lock
		 :entry-point (ff:convert-to-lang "write_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'isread-lock
		 :entry-point (ff:convert-to-lang "isread_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'iswrite-lock
		 :entry-point (ff:convert-to-lang "iswrite_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'unlock-fd
		 :entry-point (ff:convert-to-lang "un_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'libc-create
		 :entry-point (ff:convert-to-lang "libc_create")
		 :arguments '(string)
		 :return-type :integer)
  (ff:defforeign 'libc-atime
		 :entry-point (ff:convert-to-lang "libc_atime")
		 :arguments '(string)
		 :return-type :integer)
  (ff:defforeign 'write-lock
		 :entry-point (ff:convert-to-lang "write_lock")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'owner-uid
		 :entry-point (ff:convert-to-lang "owner_uid")
		 :arguments '(string)
		 :return-type :integer)
)


(defun fd-of-stream (stream output-p)
  #+lucid (extract-stream-handle stream)
  #+allegro (if output-p
		 (excl:stream-output-fn stream)
	       (excl:stream-input-fn stream))
  #+cmu (system:fd-stream-fd stream)
)
   
(defun create-directory (name &optional error-p)
  (let* ((length (length name))
	 (len (length *path-separator-string*));; test for "/" may not be necessary
	 (directory
	  (if (string= *path-separator-string* name :start2 (- length len))
	      (subseq name 0 (- length len))

	    name)))

    (if (minusp (libc-mkdir directory))
	(if error-p
	    (raise-error (error-message '(create directory) directory))
	    nil)
	t)))

(defun file-owner-p (filename)
  (= (getuid) (owner-uid filename)))

(defvar *character-type*
  #+:lucid 'base-character
  #+:allegro 'character)
					     
(define-primitive |!natural| ((natural . numeral)))    
(define-primitive |!bool| ((bool . bool)))    

(defvar *itokens* '|!tokens|)

(defun itokens-term-p (toks)
  (and (or (eql (id-of-term toks) *itokens*)
	   (eql (id-of-term toks) *itoken*))
       (null (bound-terms-of-term toks))
       (forall-p #'token-parameter-p (parameters-of-term toks))))

(defun itokens-term (toks)
  (instantiate-term (instantiate-operator *itokens*
					  (mapcar #'token-parameter toks))))
(defun tokens-of-itokens-term (term)
  (mapcar #'value-of-parameter-f (parameters-of-term term)))


;;;; DB

(defun directory-listing (pathname)
  #+lucid(directory pathname :sorted nil)
  #-lucid(directory pathname))

;; temp kludge
;; fttb expect nil.
;; then expect constant.
(defvar *db-compression-levels* nil)

;;true iff process directory has been initialized
(defvar *process-init* nil) 
;; only work with logs that I own, includes queries (independent of collection)
(defvar *personal-db* nil)

(defvar *db-assoc-list* nil)

(defvar *db-cache* (make-hash-table :test #'equal))
(defvar *db-cache-size* 1200)
(defvar *db-cache-max* 1200)
(defvar *db-cache-count* 0)
(defvar *db-cache-init* t)

(defun db-cache-reset ()
  (clrhash *db-cache*)
  (setf *db-cache-size* 1200)
  (setf *db-cache-max* 1200)
  (setf *db-cache-count* 0)
  (setf *db-cache-init* t))

(defstruct db-cache-entry
  term
  count)

(defun term-of-db-cache-entry (entry) (db-cache-entry-term entry))
(defun count-of-db-cache-entry (entry) (db-cache-entry-count entry))

(defvar *master-pathname* nil) ;;set by db-init
(defvar *passive-dir* "old")
(defconstant *data-dir* "data")
(defconstant *derivation-dir* "derivations")

;;non-process files and dirs
(defvar *db-extras*
  (list nil "levels" "closed" "collection" "checkpoints" ".pid" "derivations" "process.LOG" 
	"COMPRESSION" "README" "README-GC" "archived" "." *passive-dir* "term"))

(defun key-of-file (pathname &optional log-p)
  (let* ((ptype (pathname-type pathname))
	 (type (if ptype (intern-system ptype)
		 (progn;; (break "k")
		   '|no_type|)));; tokens easier to compare then strings
	 (sequence (intern-system (pathname-name pathname)))
	 (process-id (car (last (if log-p (pathname-directory pathname)
				    (butlast (pathname-directory pathname)))))))
    (cons type (cons sequence process-id))))

(defun terminate-path-name (p)
  (if (eql #\/ (char p (1- (length p))))
      p
      (concatenate 'string p "/")))

(defun db-root-path (&optional ascii-p path)
  (declare (ignore ascii-p))
  (if path
      (terminate-path-name path)
      (namestring (extend-pathname *system-path-prefix* '("FDLdb")))))
	

;; can't call prl-extend-x since need *master-pathname* var
(defun db-extend-directory (dirs)
  (unless *master-pathname*
	  (raise-error (error-message '(db extend directory init not))))
  (append (pathname-directory *master-pathname*) dirs))

(defun db-extend-pathname (&optional dirs name type)
  (unless *master-pathname*
	  (raise-error (error-message '(db extend pathname init not))))
  (make-pathname :name name
		 :type (when type (if (string= type "lst") type (string-upcase type))) ;; LAL this is a hack, probably just let lst upcase too but file is already with low case
		 :directory (db-extend-directory dirs)))

(defun process-id-to-string (process-id)
  (string process-id))

(defun process-id-to-pathname (process-id)
  (unless *master-pathname*
	  (raise-error (error-message '(db process pathname init not) process-id)))
  (db-extend-pathname (list (process-id-to-string process-id))))

(defun pathname-to-directory (pathname)
  (make-pathname :directory
		 (append (pathname-directory pathname) (list (file-namestring pathname)))))

(defun pid-to-path-aux (pid extensions)
  (db-extend-pathname (cons (process-id-to-string pid) extensions)))

(defun stamp-to-path-aux (stamp extensions)
  (pid-to-path-aux (process-id-of-stamp stamp) extensions))

(defun pid-to-filename-aux (pid extensions name type)
  (db-extend-pathname (cons (process-id-to-string pid) extensions)
		      name type))

(defun stamp-to-filename-aux (stamp extensions type)
  (pid-to-filename-aux (process-id-of-stamp stamp) extensions
		       (princ-to-string (sequence-of-stamp stamp))
		       type))

  
(defun stamp-to-datafile (stamp &optional type)
  (db-extend-pathname (list (process-id-to-string (process-id-of-stamp stamp))
			    *data-dir*)
		      (princ-to-string (sequence-of-stamp stamp))
		      type))

(defun stamp-to-pathname (stamp &optional type data-p)
  (if data-p
      (stamp-to-datafile stamp type)
      (db-extend-pathname (list (process-id-to-string (process-id-of-stamp stamp)))
			  (princ-to-string (sequence-of-stamp stamp))
			  type)))

(defun db-process-list (&optional master personal-p)
  (let ((process-list (filter #'(lambda (x)
		      (let* ((fname (pathname-name x))
                             (name (or fname (car (last (pathname-directory x))))))
			(not (or (and personal-p
                                      (not (file-owner-p (namestring x))))
				 (member name *db-extras* :test #'string=)))))
		  (directory-listing (or master *master-pathname*)))))
  #-cmu (mapcar #'pathname-to-directory process-list)
  #+cmu process-list))
	  

(defun data-directory (process-directory)
  (make-pathname :directory (append (pathname-directory process-directory)
				    (list *data-dir*))))

(defun db-files-of-process (process-directory)  
  (filter #'(lambda (x)
	      (let ((name (pathname-name x)))
		(not (or (equal "." name) (equal nil name) (equal nil (pathname-type x))))))
	  (directory-listing (data-directory process-directory))))

(defun file-exists-p (filename)
  (with-open-file (intermediate-stream
		   filename
		   :direction :probe
		   :if-does-not-exist nil)
		  (when intermediate-stream
			t)))
		    
(defun db-log-p (stamp)
  (file-exists-p (stamp-to-pathname stamp "LOG"))) ;;lal was 'log but cmu complained
		   
(defun db-read-term (stream)
  ;;(break "drt")

  (io-echo "I")
  (cprl-stream-read stream))
  
(defvar *num-writes* 0)

(defun db-write-term (term stream)  

  (incf *num-writes*)

  (io-echo "O")
  (cprl-stream-write stream term))


(defun db-finish-output (stream)  

  (cprl-stream-finish-output stream))


(defun db-init (master)
  ;;(break "init")
  (if nil ;; take out this test to avoid adverse effects of disksave
      ;;*master-pathname*
      (message-emit (warn-message '(db init called) *master-pathname* master))
    (progn
      (setf *master-pathname*
	    (if (string= *path-separator-string* master :start2 (- (length master) 
								   (length *path-separator-string*)))
		master
	      (concatenate 'string master *path-separator-string*)))
      
      ;; todo need better method.
      (setf *db-compression-levels* (get-levels))
      nil)))

(defun db-query-old (&optional master)
  (mapcan #'get-log-info (get-hdr-files (db-process-list master *personal-db*))))

(defun db-query (&optional master)
  (walk-db-hdrs #'(lambda (stream)
		    (if stream
			(with-handle-error (('(db log header)) ())
					   (let ((parent (db-read-term stream))
						 (stamp (term-to-stamp (db-read-term stream)))
						 (description (db-read-term stream)))
					     (when (and parent stamp description)
						   (list (cons description stamp)))))))
		master))

(defun db-persistent-p (stamp type)
  (if (gethash (cons stamp type) *db-cache*)
      t
      (with-open-file (intermediate-stream
		       (stamp-to-pathname stamp type t)    
		       :direction :probe
		       :element-type *character-type*
		       :if-does-not-exist nil)
	(if intermediate-stream t nil))))

(defun db-persistent-not-inline-p (stamp type)
  (with-open-file (intermediate-stream
		   (stamp-to-pathname stamp type t)    
		   :direction :probe
		   :element-type *character-type*
		   :if-does-not-exist nil)
    (if intermediate-stream t nil)))


(defmacro with-db-input-file ((stream filename &optional error-p) &body body)
  `(with-cprl-open-in-file (,stream ,filename) ,@body))
   

(defvar *umask-mode* 2) ;;call libc-create first instead? protect umask

(defmacro with-db-output-file ((stream filename &optional error-p not-log-p gc-p) &body body)
  `(let ((umask (libc-umask *umask-mode*)))
    (with-cprl-open-out-file (,stream ,filename *db-compression-levels* ,not-log-p ,gc-p)
      ,@body)
    (libc-umask umask)))


(defmacro with-acl-open-file ((stream filename in-or-out) &body body)
  (cond
    ((eql in-or-out 'in)
     `(let ((,stream (open ,filename
			   :direction :input
			   :element-type 'character
			   :if-does-not-exist nil)))
       ,@body
       (close ,stream)))
	     
    ((eql in-or-out 'out)
     `(let ((,stream (open ,filename
			   :direction :output
			   :element-type 'character
			   :if-does-not-exist nil)))
       ,@body
       (close ,stream)))))

(defun db-read-ahead (stamp type)
  (let ((key (cons stamp type)))
    (unless (gethash key *db-cache*)	
    (with-db-input-file (intermediate-stream (stamp-to-pathname stamp type t) t)   
				(setf (gethash key *db-cache*)
				      (make-db-cache-entry :term (db-read-term intermediate-stream)
							   :count (setf *db-cache-count*
									(+ 2 *db-cache-count*)))))
	    nil)))

(defun update-db-cache ()
  (format t "updating db cache...")
  (let ((odd-p (oddp *db-cache-count*)))
    (if *db-cache-init*
	(setf *db-cache-init* nil)
	(maphash  #'(lambda (key val)
		      (when (and (db-cache-entry-p val)
				 (or (and odd-p (evenp (count-of-db-cache-entry val)))
				     (and (not odd-p) (oddp (count-of-db-cache-entry val)))))
			(remhash key *db-cache*)))
		  *db-cache*))
    (setf *db-cache-count* (if odd-p 0 1))))

(defun maybe-update-db-cache ()
  (when (>= *db-cache-count* *db-cache-max*)
    (update-db-cache)
    (setf *db-cache-max* *db-cache-size*))
  nil)
			     
(defun flush-db-cache ()
  (clrhash *db-cache*)
  (setf *db-cache-init* t *db-cache-count* 0)
  nil)

(defun db-read-file (filename)
  (with-db-input-file (intermediate-stream filename t)
		      (db-read-term intermediate-stream)))

(defvar *db-cache-hit-count* 0)
(defun db-read (stamp type)
  (maybe-update-db-cache)
  
  (let ((key (cons stamp type)))

    (or (let ((val (gethash key *db-cache*)))
	  (when val
	    (setf (db-cache-entry-count val) (incf *db-cache-count* 2))
	    (incf *db-cache-hit-count* 0)
	    (incf *db-cache-max* 2)
	    (term-of-db-cache-entry val)))
	
	(when *io-db-buffered-persist*
	  (gethash key *io-db-buffered-persist*))
	
	(let ((term (db-read-file (stamp-to-pathname stamp type t))))
	  (setf (gethash key *db-cache*)
		(make-db-cache-entry :term term
				     :count (incf *db-cache-count* 2)))
	  term))))

(defvar *db-init-process-dir-cache* nil)

(defun db-init-process-dir-aux (pid)
  (let ((a (member pid *db-init-process-dir-cache*)))
    (unless a
      (let ((mypath (make-pathname :directory
				   (db-extend-directory (list (process-id-to-string pid))))))
	(if (not (probe-file mypath))
	    (progn
	      (libc-mkdir (namestring mypath))
	      (libc-mkdir
	       (namestring
		(make-pathname :directory
			       (db-extend-directory (list (process-id-to-string pid)
							  "data"))))))
	    (let ((dpath (namestring
			  (make-pathname :directory
					 (db-extend-directory (list (process-id-to-string pid)
								    "data"))))))
	      (unless (probe-file dpath)
		(libc-mkdir dpath)))))
      (push pid *db-init-process-dir-cache*))
    ))

(defun db-init-process-dir (stamp)
  (db-init-process-dir-aux (process-id-of-stamp stamp)))


(defun db-write-aux (path stamp type term)
  (io-db-buffer-flush)
  (with-cprl-open-out-file (s (make-pathname :name (princ-to-string (sequence-of-stamp stamp))
					     :type (when type (string-upcase type))
					     :directory path)
			      nil nil nil)
    (cprl-stream-write s term t))
  )


(defun db-write-check-cache (stamp-type term)
  (maybe-update-db-cache)
  (let* ((key stamp-type)
	 (val (gethash key *db-cache*)))
    (cond
     (val  ;; LAL verify file exists
      (setf (db-cache-entry-count val) (setf *db-cache-count* (+ 2 *db-cache-count*)))
      (setf *db-cache-max* (+ 2 *db-cache-max*))
      t)
     (t (setf (gethash key *db-cache*)
	      (make-db-cache-entry :term term
				   :count (setf *db-cache-count*
						(+ 2 *db-cache-count*))))
	nil))))

(defun db-write-buf (stamp-type term)

  (let ((stamp (car stamp-type))
	(type (cdr stamp-type)))

  ;; this is normal due to changing inlines to persist of imported data.
  (unless (eql *process-id* (process-id-of-stamp stamp))
    (db-init-process-dir stamp)
    ;;(break "db-write other")
    (format t "~% db-write to other dir ~a ~a ~a~%" (process-id) (process-id-of-stamp stamp) (cadr stamp)))

  ;; possible dir does not exist.
  ;; proble dir and mkdir if not present?

  ;; could be worthwhile to split into open, write, finish/close to try to avoid
  ;; waits
  (with-db-output-file (intermediate-stream (stamp-to-pathname stamp type t) t t nil)
		       (when intermediate-stream
			 (db-write-term term intermediate-stream)))))


(defun db-write (stamp type term &optional gc-p (cachep t))
  ;; this is normal due to changing inlines to persist of imported data.
  (unless (eql (process-id) (process-id-of-stamp stamp))
    (format t "~% db-write to other dir ~a ~a ~a~%" (process-id) (process-id-of-stamp stamp) (cadr stamp))
    ;;(break "db-write tod")
    )

  (db-init-process-dir stamp)

  (unless (and cachep
	       (db-write-check-cache (cons stamp type) term))
    
    (with-db-output-file (intermediate-stream (stamp-to-pathname stamp type t) t t gc-p)
			 (when intermediate-stream
			   (db-write-term term intermediate-stream)))))


(defvar *io-db-buffer* nil)
(defvar *io-db-buffered-persist* nil)
(defvar *io-db-buffering* nil)


(defstruct io-db-buffer-entry
  term
  )

(defstruct (io-db-buffer-persist-entry (:include io-db-buffer-entry))
  stamp-type
  cachedp)

(defun iodb-be-term (iodbbe) (io-db-buffer-entry-term iodbbe))
(defun iodb-be-cached-p (iodbbe) (io-db-buffer-persist-entry-cachedp iodbbe))
(defun iodb-be-stamp-type (iodbbe) (io-db-buffer-persist-entry-stamp-type iodbbe))

(defun maybe-inline-db-buffered-persist (term)
  (or (when *io-db-buffered-persist*
	(when (idata-persist-term-p term)
	  (let ((tstamp (stamp-of-idata-persist-term term))
		(type (type-of-idata-persist-term term)))
	    (let ((cterm (gethash (cons (term-to-stamp tstamp) type) *io-db-buffered-persist*)))
	      (when cterm
		(idata-persist-inline-term type
					   tstamp
					   cterm))))))

      term))

(defun inline-db-buffered-persist (term)
  (if (zerop (hash-table-count *io-db-buffered-persist*))
      term
    (term-walk-d term
		 #'(lambda (x) (and (idata-persist-term-p x)
				    (not (itemplate-term-p
					  (term-of-bound-term (car (bound-terms-of-term x)))))
				    ))
		 #'(lambda (term contf)
		     (let ((tstamp (stamp-of-idata-persist-term term))
			   (type (type-of-idata-persist-term term)))
		       ;;(setf -tstamp tstamp -type type -term term) (break "ipd")
		       (let ((cterm (gethash (cons (term-to-stamp tstamp) type) *io-db-buffered-persist*)))
			 (if cterm
			     (idata-persist-inline-term type
							tstamp
							(funcall contf cterm))
			   term)))))))

(defun new-db-persist-be (st term c)
  (make-io-db-buffer-persist-entry
   :stamp-type st
   :term term
   :cachedp c))

(defun begin-db-persist-buffering ()
  (begin-io-db-buffering)
  (setf *link-term-hook* #'inline-db-buffered-persist)
  (setf *io-db-buffering* t
	*io-db-buffered-persist* (make-hash-table :test #'equal)))

(defun db-write-buffer-persist (stamp type term cachep)
  (format t "~%WBI ~a " (length-of-queue *io-db-buffer*))
  (let ((stamp-type (cons stamp type)))
    (setf (gethash stamp-type *io-db-buffered-persist*) term)
    (queue-push *io-db-buffer* (new-db-persist-be stamp-type term
						  (when cachep
						    (db-write-check-cache stamp-type term)))) ))

(defun db-write-persist-buffered (e)
  (let ((st (iodb-be-stamp-type e)))
    (remhash st *io-db-buffered-persist*)
    (unless (iodb-be-cached-p e)
      (db-write-buf st (iodb-be-term e)))
    (format t "~%BPR ~a " (length-of-queue *io-db-buffer*))
  ))
  
(defun db-write-persist (stamp type term cachep)
  (if *io-db-buffering*
      (db-write-buffer-persist stamp type term cachep)
    (db-write stamp type term nil cachep)))

(defun db-inform-checkpoint (log-stamp checkpoint-stamp)
  (io-db-buffer-flush)
  (with-db-output-file (intermediate-stream (db-extend-pathname nil "checkpoints" "LOG") nil nil t)

  ;;(with-cprl-open-out-file (intermediate-stream (db-extend-pathname nil "checkpoints" "LOG") nil nil))
    (when intermediate-stream
      (db-write-term (stamp-to-term checkpoint-stamp)
		     intermediate-stream)
      (db-write-term (stamp-to-term log-stamp)
		     intermediate-stream)
      (db-finish-output intermediate-stream)))
      			
  (setf *db-assoc-list* (acons checkpoint-stamp log-stamp *db-assoc-list*))
  nil)

(defvar *checkpoint-file-position* 0)
(defvar *checkpoint-query-time* 0)

(defun db-reset ()
  (setf *checkpoint-file-position* 0)
  (setf *checkpoint-query-time* 0)
  (setf *db-assoc-list* nil))

(defun modified-p (pathname time)
  (let ((date (file-write-date pathname)))
    (or (not date) (> date time))))

(defun db-stream-file-position (stream pos)
  (cprl-stream-file-position stream pos))
	
     
;;;; only read checkpoints.log if necessary. (could have gc handle this instead?)
;;;; if modified since last read then
;;;;	if collection.log modified since last read (garbage collection occurred) then
;;;;	 read entire file -or remove moved logs from list?
;;;;	else read starting where finished at last read
;;;; else don't read
(defun db-query-checkpoint (checkpoint)
  (let ((pathname (db-extend-pathname nil "checkpoints" "LOG")))
    ;;(setf p pathname) (break "query")
    (or (if (modified-p pathname *checkpoint-query-time*)
	    (with-db-input-file (intermediate-stream pathname)
	      ;;(with-cprl-open-in-file (intermediate-stream pathname))
	      (when intermediate-stream
		(if (modified-p (db-extend-pathname nil "collection" "LOG")
				*checkpoint-query-time*)
		    (setf *db-assoc-list* nil)
		    (db-stream-file-position intermediate-stream
					     *checkpoint-file-position*))
		(setf *checkpoint-query-time* (get-universal-time))
		(do* ((term (db-read-term intermediate-stream)
			    (db-read-term intermediate-stream))
		      (cp (when term (term-to-stamp term))
			  (when term (term-to-stamp term)))
		      (log (when cp (term-to-stamp
				     (db-read-term intermediate-stream)))
			   (when cp (term-to-stamp
				     (db-read-term intermediate-stream)))))
		     ((or (null term) (equal checkpoint cp))
		      (when term
			(progn (setf *checkpoint-file-position*
				     (db-stream-file-position intermediate-stream nil))
							 
			       (setf *db-assoc-list*
				     (acons cp log *db-assoc-list*))
			       log))))))
	    (cdr (assoc checkpoint *db-assoc-list* :test #'equal)))
	(raise-error (error-message '(db query checkpoint not) (stamp-to-str checkpoint))))))

;; maybe put in basic-lisp
(defun assoc-all (x list)
  (let ((l nil))
    (mapc #'(lambda (p)
	      (when (equal (car p) x)
		    (setf l (cons (cdr p) l))))
	  list)
    l))

(defun update-checkpoint-list ()
  (with-db-input-file (intermediate-stream (db-extend-pathname nil "checkpoints" "LOG"))
  ;;(with-cprl-open-in-file (intermediate-stream (db-extend-pathname nil "checkpoints" "LOG")))
    (when intermediate-stream
      (if (modified-p (db-extend-pathname nil "collection" "LOG")
		      *checkpoint-query-time*)
	  (db-stream-file-position intermediate-stream
				     *checkpoint-file-position*)
	  (setf *db-assoc-list* nil))
	      
      (setf *checkpoint-query-time* (get-universal-time))
      (do* ((term (db-read-term intermediate-stream)
		  (db-read-term intermediate-stream))
	    (cp (when term (term-to-stamp term))
		(when term (term-to-stamp term)))
	    (log (when cp (term-to-stamp (db-read-term term)))
		 (when cp (term-to-stamp (db-read-term term)))))
	   ((null term)
	    (progn (setf *checkpoint-file-position*
			 (db-stream-file-position intermediate-stream nil))
		   nil))
			    
	(setf *db-assoc-list* (acons cp log *db-assoc-list*))))))
  
(defun db-query-checkpoints (log-stamp)
  (when (modified-p (db-extend-pathname nil "checkpoints" "LOG") *checkpoint-query-time*)
	(update-checkpoint-list)) 
  (or (assoc-all log-stamp *db-assoc-list*)
      (raise-error (error-message '(db query checkpoints not) (stamp-to-str log-stamp)))))

		    
;;;;
;;;; Logs
;;;;

;;default log properties
(defvar *logs* nil)
(defvar *log-infos* nil)
(defvar *log-properties* (list (cons (intern-system "count") (inatural-term 0))
			       (cons (intern-system "owner") (ibool-term t))
			       (cons (intern-system "permanent") (ibool-term nil))
			       (cons (intern-system "garbage") (ibool-term nil))
			       ))
(define-primitive |!open_log| () (stamp))
(define-primitive |!close_log| () (stamp))
(define-primitive |!reference| ((natural . count) (bool . owner-p) (bool . root-p))) ;;root t if permannet
(define-primitive |!log_description| () (environment types))
(define-primitive |!environment| () (address purposes resources table-types reduction-tags))

(define-primitive |!tok_cons| ((token . head)) (tail))
(define-primitive |!token_cons| ((token . head)) (tail))
(define-primitive |!log_info| ((string . access-time) (string . modify-time)) (address))
(define-primitive |!log_cons| () (head tail))
(define-primitive |!tok_nil| () ())


(defvar *open-wlog-table* (make-hash-table :test #'equal))
(defvar *open-rlog-table* (make-hash-table :test #'equal))


;;;; log file locking

(defun write-lock-p (log )
  (or (gethash (key-of-file log) *open-wlog-table*)
      (with-open-file (stream
		       log
		       :direction :input
		       :element-type *character-type*
		       :if-does-not-exist nil)
		      (unless stream (raise-error (error-message '(read lock p) log)))
		      (let* ((fd (fd-of-stream stream nil))
			     (pid (iswrite-lock fd)))
			(< 0 pid)))))

(defun read-lock-p (log)
  (or (gethash (key-of-file log) *open-rlog-table*)
      (with-open-file (stream
		       log
		       :direction :output
		       :element-type *character-type*
		       :if-does-not-exist nil
		       :if-exists :overwrite)
		      (unless stream (raise-error (error-message '(write lock p) log)))
		      (file-position stream 0)
		      (let* ((fd (fd-of-stream stream t))
			     (pid (isread-lock fd)))
			(> pid 0)))))
				

(defstruct log-header
  parent
  stamp
  description
  reference
  )

(defun parent-of-log-header (l) (log-header-parent l))
(defun stamp-of-log-header (l) (log-header-stamp l))
(defun description-of-log-header (l) (log-header-description l))
(defun reference-of-log-header (l) (log-header-reference l))

(defun walk-db-hdrs (stream-f &optional master)
  (let ((l nil))
    (labels ((walk-hdr (hdr)
	       (with-db-input-file (intermediate-stream hdr)
		 (funcall stream-f intermediate-stream)))
	     (walk-process (p)
			   (setf l (append (mapcan #'walk-hdr
						   (filter #'hdr-file-p (directory-listing p)))
					   l))))
	  
	    (mapc #'walk-process (db-process-list master *personal-db*)))
    l))


(defun log-file-p (pathname)
  ;;(setf -p pathname) (break "lfp")
  (and (string= "LOG" (pathname-type pathname))
       (not (string= "collection" (pathname-name pathname)))
       (file-exists-p (header-of-log pathname));; remove after testing
       ))

(defun hdr-file-p (pathname)
  (string= "HDR" (pathname-type pathname)))

(defun header-of-log (log)
  (make-pathname :name (pathname-name log)
		 :type "HDR" ;;lal was 'hdr but cmu complained
		 :directory (pathname-directory log)))
		       
(defun root-log-p (log)
  (unless (log-file-p log)
	  (raise-error (error-message '(root log not log) log)))
  (with-db-input-file (intermediate-stream (header-of-log log) t)
		      (istamp-term-p (db-read-term intermediate-stream))))

;; process : pathname
(defun get-process-hdr-files (process)
  (filter #'hdr-file-p (directory-listing process)))

(defun get-hdr-files (process-list)
  (flatten (mapcar #'get-process-hdr-files process-list)))

;; 
(defun db-logs-of-process (process)
  (filter #'log-file-p (directory-listing process)))

(defun db-read-list (fname)
  (let ((acc nil))
    (with-db-input-file (s fname)
      (do ((r (db-read-term s) (db-read-term s)))
	  ((null r)
	   (nreverse acc))
	(push r acc)))))

(defun db-read-map (f fname)
  (with-db-input-file (s fname)
      (do ((r (db-read-term s) (db-read-term s)))
	  ((null r) nil)
	(when (funcall f r)
	  (return-from db-read-map t)))))

(defun get-log-info (hdr)
  (with-db-input-file (intermediate-stream hdr)
		      (if intermediate-stream
			  (with-handle-error (('(db log header)) ())
			    (let ((header (read-log-header intermediate-stream)))
			      (when header
				(list (cons (description-of-log-header header)
					    (stamp-of-log-header header))
				      ;; (parent-of-log-header header)
				      ))))
		    
			(progn (message-emit (warn-message '(get log info not)
							   hdr))
			       nil))))

(defun get-log-header (hdr)
  (with-db-input-file (intermediate-stream hdr)
		      (if intermediate-stream
			  (with-handle-error (('(db log header)) (nil))
					     (read-log-header intermediate-stream)))))


#| ;; not used,  maybe later if log info put in 1 file per process
(defun get-db-logs (process-directory)
  (let ((pair-list nil))
    (with-open-file (intermediate-stream
		     (make-pathname :name "db.logs" :directory process-directory)
		     :direction :input
	     :if-does-not-exist nil)
		    (do ((parent-node (read-node intermediate-stream)
				      (read-node intermediate-stream))
			 (stamp (term-to-stamp (db-read-term intermediate-stream))
				(term-to-stamp (db-read-term intermediate-stream)))
			 (env-description (db-read-term intermediate-stream)
					  (db-read-term intermediate-stream)))
			((null parent-node) pair-list)
			(setf pair-list (cons (cons env-description stamp) pair-list))))))

|# 


(defun read-log-header (stream)
  (let* ((parent (db-read-term stream))
	 (stamp (db-read-term stream))
	 (description (db-read-term stream))
	 (reference (db-read-term stream)))
	  
    ;;(setf p parent s stamp d description) (break  "rlh")
    (if (and parent stamp description)
	(make-log-header :parent (when (istamp-term-p parent) ;;not ivoid-term
				       (let ((pstamp (term-to-stamp parent)))
					 (unless (equal pstamp (term-to-stamp stamp))
						 pstamp)))
			 :stamp (term-to-stamp stamp)
			 :description description
			 :reference reference)
      ;;(raise-error (error-message '(read log header not)))
      (progn
	(message-emit (warn-message '(read log header not)
				    stream))
	nil))))

(defun log-query (stamp)
  (with-db-input-file (intermediate-stream (stamp-to-pathname stamp "HDR") t)
		      (let ((header (read-log-header intermediate-stream)))
			(when header (description-of-log-header header)))))

(defun gc-file (stamp)
  (db-extend-pathname (list (process-id-to-string (process-id-of-stamp stamp)))
		      "closed"
		      "GC"))

#|
(defun db-write-header-old (create-p stamp desc &optional parent reference lock-p)
  (let ((pathname (stamp-to-pathname stamp "HDR")))

    (when create-p (libc-create (namestring pathname)))
    
    (with-db-output-file (header-stream pathname t nil create-p)	  
		       (when lock-p
			     (write-lock (fd-of-stream (prl-stream-stream
							(cprl-stream-prl-stream header-stream))
							      t)))
		       (db-write-term (or parent (ivoid-term)) header-stream)
		       (db-write-term (stamp-to-term stamp) header-stream)
		       (db-write-term desc header-stream)
		       (db-write-term (or reference (properties-to-term *log-properties*)) header-stream)
		       (db-finish-output header-stream))))
|#


(defun db-write-header (create-p stamp desc &optional parent reference lock-p)
  (io-db-buffer-flush)
  (let ((pathname (stamp-to-pathname stamp "HDR")))

    (when create-p (libc-create (namestring pathname)))
    
    (let* ((h-stream (open pathname
			   :direction :output
			   :element-type '(unsigned-byte 8)
					 
			   :if-does-not-exist nil
			   :if-exists (if create-p :append :overwrite)))
	   (header-stream (new-cprl-out-stream (new-prl-out-stream h-stream)
					       *db-compression-levels*
					       nil)
	     ))
      (when lock-p
	    (write-lock (fd-of-stream (prl-stream-stream
				       (cprl-stream-prl-stream header-stream))
					
				      t)))
      (db-write-term (or parent (ivoid-term)) header-stream)
      (db-write-term (stamp-to-term stamp) header-stream)
      (db-write-term desc header-stream)
      (db-write-term (or reference (properties-to-term *log-properties*)) header-stream)
      (db-finish-output header-stream)
      (close h-stream))))

(defun log-write-append (filename term)
  (io-db-buffer-flush)
  (let ((umask (libc-umask *umask-mode*)))
    (with-cprl-open-out-file (s filename nil nil t)
      (cprl-stream-write s term t))
    (libc-umask umask)))


(defun lite-log-open-write (stamp)
  (let ((pathname (stamp-to-pathname stamp "LOG" t)))

    (libc-create (namestring pathname))

    (let ((wlog-stream (open pathname
			     :direction :output
			     :element-type '(unsigned-byte 8)
			     :if-does-not-exist nil
			     :if-exists :append)))

      (unless wlog-stream
	(raise-error (error-message '(log open write fail) stamp)))
      
      (new-cprl-out-stream (new-prl-out-stream wlog-stream)
			   *db-compression-levels*
			   nil) )))


(defun db-log-file-open-write (stamp desc &optional parent)
  (let ((pathname (stamp-to-pathname stamp "LOG")))

    (libc-create (namestring pathname))
    (let ((wlog-stream (open pathname
			     :direction :output
			     :element-type '(unsigned-byte 8)
			     :if-does-not-exist nil
			     :if-exists :append)))

      (unless wlog-stream
	      ;;(setf p pathname) (break)
	      (raise-error (error-message '(db log open write fail) stamp)))
      
      (write-lock (fd-of-stream wlog-stream t))
      (db-write-header t stamp desc parent)      
      (setf (gethash (key-of-file pathname) *open-wlog-table*) t)

      (new-cprl-out-stream (new-prl-out-stream wlog-stream)
			   *db-compression-levels*
			   nil))))

#|
(defun update-reference-old (stamp closure)
  (let ((header-file (stamp-to-pathname stamp "HDR"))
	(header nil))
    (with-db-input-file (intermediate-stream header-file t)
			(setf header (read-log-header intermediate-stream)))
    (let ((reference (reference-of-log-header header)))
      (when reference
	    (db-write-header nil
			     (stamp-of-log-header header)
			     (description-of-log-header header)
			     (stamp-to-term (parent-of-log-header header))
			     (funcall closure (count-of-ireference-term reference);;(property-of-term (intern-system "count") reference)
				      nil;;lal remove later;; (owner-p-of-ireference-term reference)
				      nil;;lal remove later;; (root-p-of-ireference-term reference)
				      )  
			     t)))  				      
    ))
|#

;;temp to account for header term change
(defun ireference-term-to-properties (reference)
  (list (cons (intern-system "count")
	      (inatural-term (count-of-ireference-term reference)))
	(cons (intern-system "owner")
	      (ibool-term (with-unwind-error (nil) (owner-p-of-ireference-term reference))))
	(cons (intern-system "permanent")
	      (ibool-term (with-unwind-error (nil) (root-p-of-ireference-term reference))))
	(cons (intern-system "garbage")
	      (ibool-term nil))))

(defun ireference-old-term-to-properties (reference)
  (list (cons (intern-system "count")
	      (inatural-term (count-of-ireference-term reference)))
	(cons (intern-system "owner")
	      (ibool-term (when (> (length (parameters-of-term reference)) 1)
				(owner-p-of-ireference-term reference))))
	(cons (intern-system "permanent")
	      (ibool-term  nil))
	(cons (intern-system "garbage")
	      (ibool-term nil))))

(defun property-of-term (tag term)
  (assoc tag (if (ireference-term-p term)
		 (ireference-term-to-properties term)
	       (if (icons-term-p term)
		   (term-to-properties term)
		 (progn;;(break "unknown ref term")
		   (ireference-old-term-to-properties term))))))


(defun update-reference (stamp closure)
  (let ((header-file (stamp-to-pathname stamp "HDR"))
	(header nil))
    (with-db-input-file (intermediate-stream header-file t)
			(setf header (read-log-header intermediate-stream)))
    (let ((reference (or (reference-of-log-header header) (properties-to-term *log-properties*))))
      (db-write-header nil
		       (stamp-of-log-header header)
		       (description-of-log-header header)
		       (stamp-to-term (parent-of-log-header header))
		       (if (ireference-term-p reference)
			   (properties-to-term (funcall closure
							(ireference-term-to-properties reference)))
			 (if (icons-term-p reference)
			     (properties-to-term (funcall closure (term-to-properties reference)))
			   (progn ;; (setf h header r reference) (break "unknown ref term")
				  (properties-to-term (funcall closure *log-properties*)))))
		       t))))

(defun increment-reference-count (stamp)
  (update-reference stamp #'(lambda (props)
			      (let ((count (assoc '|count| props)))
				(cond
				 ((null count)
				  ;; optionally:
				  (message-emit (warn-message '(db hdr property required missing) 'count))
				  ;; or (break "dhprm") while in development mode.
				  (acons '|count| (inatural-term 1) props))
				 (t
				  ;; rplacd is ok too.		
				  (setf (cdr count)
					(inatural-term (1+ (numeral-of-inatural-term (cdr count))))))))
			      props)))
		  
(defun decrement-reference-count (stamp &optional val)
  (update-reference stamp  #'(lambda (props)
			       (let ((count (assoc '|count| props)))
				 (cond
				  ((null count)
				   ;; optionally:
				   (message-emit (warn-message '(db hdr property required missing) 'count))
				   ;; or (break "dhprm") while in development mode.
				   (acons '|count| (inatural-term 0) props))
				  (t
				   ;; rplacd is ok too.		
				   (setf (cdr count)
					 (inatural-term (max 0 (- (numeral-of-inatural-term (cdr count))
							   (or val 1))))))))
			       props)))
		  
(defun update-reference-owner-p (stamp &optional owner)
  (set-log-property stamp (cons '|owner| (ibool-term owner))))
		  
(defun update-reference-root-p (stamp val)
  (set-log-property stamp (cons '|permanent| (ibool-term val))))
	
(defun set-log-property (stamp prop)
  (update-reference stamp #'(lambda (props)
			      (let* ((tag (car prop))
				     (p (assoc tag props)))
				(cond
				 ((null p)
				  ;; optionally:
				  (message-emit (warn-message '(db hdr property required missing) tag))
				  ;; or (break "dhprm") while in development mode.
				  (acons tag (cdr prop) props))
				 (t
				  ;; rplacd is ok too.		
				  (setf (cdr p) (cdr prop)))))
			      props)))

(defun db-derivation-path-pid (pid)
  (db-extend-pathname (list (process-id-to-string pid)
			    *derivation-dir*)))

(defun db-derivation-path (stamp)
   (db-derivation-path-pid (process-id-of-stamp stamp)))

(defun db-local-old-path-pid (pid)
  (db-extend-pathname (list (process-id-to-string pid)
			    "old")))



(define-primitive |!loglog| ((tok . kind) (time . time)) (term))

;; journal close
;; journal open
;; journal modified at least once.
(defun loglog (kind term)
  ;;(setf -kind kind -term term) (break "loglog")
  (log-write-append (process-log-name (process-id))
		    (iloglog-term kind (get-universal-time) term)))

(defun db-log-open (stamp env-description &optional parent)
  (if *master-pathname*;; db-init called
      (progn
	(unless *process-init*
	  (let ((pid (process-id-of-stamp stamp)))
	    (create-directory (directory-namestring (process-id-to-pathname pid)))
	    (create-directory (directory-namestring
			       (db-extend-pathname
				(list (process-id-to-string pid) *data-dir*))))
	    (create-directory (directory-namestring (db-derivation-path-pid pid)))
	    (create-directory (directory-namestring (db-local-old-path-pid pid)))

	    (loglog 'pid (itoken-term pid))
		    
	    (setf *process-init* t)))
	
	;; check for garbage collection in progress
	(if (file-exists-p (gc-file stamp))
	    (message-emit (warn-message '(new log open collecting) stamp))
	    (db-log-file-open-write stamp env-description parent)))
      
    (message-emit (warn-message '(log open write db init not) stamp))))


(define-primitive |!derivation| ((time . time)) (parent child))

(defun process-pathname (stamp)
  (process-id-to-pathname (process-id-of-stamp stamp)))


(defun add-log-derivation (parent child)

  (let ((ddir (db-derivation-path parent)))

    ;;(setf -ddir ddir) (break "ald")

    ;; warn if derivation dir is not present.
    (if (not (probe-file ddir))
	(message-emit (warn-message '(add log derivation dir not) (stamp-to-term parent)))
	(let ((umask (libc-umask *umask-mode*)))
	  (db-write-aux (namestring ddir) child 'der
			(iderivation-term (get-universal-time) (stamp-to-term parent) (stamp-to-term child)))
	  (libc-umask umask)))))
    

(defun new-log-open (stamp env-description)
   (db-log-open stamp env-description))		  
	  		    
(defun log-open-write (stamp env-description parent-stamp)
  (prog1
      (db-log-open stamp env-description (stamp-to-term parent-stamp))
    (add-log-derivation parent-stamp stamp)
    (increment-reference-count parent-stamp)))

;; may want optional finish-output after checkpoints
;; could check term
;;
;; insert-index is last filled
;; insert-index is next free slot
;; write-index is last written
;;   then empty 1+ write = insert
;;   then full 1+ write = insert
;;   full when 1+ last filled = 

(defvar *io-db-log-buffering* nil)

(defstruct (io-db-buffer-log-entry (:include io-db-buffer-entry))
  log)

(defun iodb-be-log (iodbbe) (io-db-buffer-log-entry-log iodbbe))


(defun begin-log-buffering ()
  (begin-io-db-buffering)
  (setf *io-db-log-buffering* t)
  )

(defun end-log-buffering ()
  (setf *io-db-log-buffering* nil)
  (end-io-db-buffering)
  )
  
(defun log-buffer-insert (log term)
  (queue-push *io-db-buffer* (make-io-db-buffer-log-entry :term term :log log))
  (format t "~%LBI ~a " (length-of-queue *io-db-buffer*)))

(defun io-db-buffer-listen ()
  (and *io-db-buffer*
       (not (zerop (length-of-queue *io-db-buffer*)))))

(defun log-write-record-aux (wlog term)
  ;; (setf w wlog m term) (break)
  (db-write-term term wlog)
  (db-finish-output wlog))

(defun log-buffer-write (e)
  (log-write-record-aux (iodb-be-log e) (iodb-be-term e))
  (format t "~%LBR ~a " (length-of-queue *io-db-buffer*)))

(defun io-db-buffer-write ()
  ;;(sleep 1)
  (let ((e (queue-pop *io-db-buffer*)))
    (cond
     ((io-db-buffer-persist-entry-p e) (db-write-persist-buffered e))
     ((io-db-buffer-log-entry-p e) (log-buffer-write e))
     (t (setf -e e) (break "idbw")))))
      
(defun log-write-record (wlog term)
  (if *io-db-log-buffering*
      (log-buffer-insert wlog term)
    (log-write-record-aux wlog term)))

(defun io-db-buffer-flush ()
  (do ()
      ((not (io-db-buffer-listen)))
      (io-db-buffer-write)
      ))

(defun end-db-persist-buffering ()
  
  (end-io-db-buffering)

  (setf *link-term-hook* nil
	*io-db-buffering* nil
	*io-db-buffered-persist* nil)
  )

(defun io-db-buffer-sleeper (q)
  (if (io-db-buffer-listen)
      (io-db-buffer-write)
    (sleep-aux q)))

(defun begin-io-db-buffering ()

  (db-init-process-dir-aux (process-id))

  (unless *io-db-buffer*
    (setf *io-db-buffer* (new-queue)))
  (set-orb-sleeper #'io-db-buffer-sleeper))

(defun end-io-db-buffering ()
  (io-db-buffer-flush)
  )

(defun start-db-buffering ()
  (format t "~%~%~%~tStarting DB Buffering~%~%")
  (begin-log-buffering)
  (begin-db-persist-buffering))

(defun stop-db-buffering ()
  (format t "~%~%~%~tEnding DB Buffering~%~%")
  (end-log-buffering)
  (end-db-persist-buffering)
  (setf *io-db-buffer* nil))

(defun db-log-file-open-read (stamp)
  (let ((rlog-stream (open (stamp-to-pathname stamp "LOG")
			   :element-type '(unsigned-byte 8)
			   :direction :input
			   :if-does-not-exist nil)))
    
    (unless rlog-stream
      (raise-error (error-message '(log open read file not) stamp)))

    (read-lock (fd-of-stream rlog-stream nil))
  
    (new-cprl-in-stream (new-prl-in-stream rlog-stream))
    ))

(defun log-open-read (stamp)
  (if (null *master-pathname*)
      (message-emit (warn-message '(log open read db init not) stamp))
      (progn
	(let ((s (db-log-file-open-read stamp)))
	  (db-read-term s)		; access file
	  (log-close s))

	;;(sleep 1) 
	;; repeat for race condition
	(db-log-file-open-read stamp))))


(defun lite-log-open-read-aux (fname)
  (let ((rlog-stream (open fname
			   :element-type '(unsigned-byte 8)
			   :direction :input
			   :if-does-not-exist nil)))
    
    (unless rlog-stream
      (raise-error (error-message '(log open read file not) fname)))

    (new-cprl-in-stream (new-prl-in-stream rlog-stream))))


(defun lite-log-open-read (stamp)
  (if (null *master-pathname*)
      (message-emit (warn-message '(log open read db init not) stamp))
      (lite-log-open-read-aux (stamp-to-pathname stamp "LOG" t))))


(defun log-read-record (rlog)
  (db-read-term rlog))

;; 3/2003  This should be obsolete. see recover-data-list in lib-defs.
;; and read_recover_file and callers in objects.
#|
(defvar *recovered-objc-data* nil)
(defun recover-objc-data (pathname)
  (let ((dl (directory pathname)))
    (format t "~a" (length dl))
    ;;(setf -dl dl) (break)
    (let ((loacc nil)
	  (oacc nil)
	  (kinds nil))
      
      (dolist (fname dl)
	(let ((k (pathname-type fname)))
	  (unless (member k kinds :test #'string=)
	    (push k kinds))
	  (when (string= "OBJC" k)
	    (push (cons fname (db-read-file fname)) oacc))
	  (when (string= "LOBJ" k)
	    (push (cons fname (db-read-file fname)) loacc))))
      (setf *recovered-objc-data* (cons loacc oacc))
      (list* (length loacc) (length oacc) kinds))))

(defvar *recovered-graph* nil)
(defun recovered-objc-graph ()
  (setf *recovered-graph*
	(make-hash-table :test #'equal
			 :size (* 2 (length (cdr *recovered-objc-data*)))))
  (dolist (oobjc (cdr *recovered-objc-data*))
    ;; recover stamp from filename
    ;; name is sequence and last dir of path is process_id
    ;;(setf -a oobjc) (break "rog")
    (let ((newstamp (let ((pid (intern-system
				(car (last (pathname-directory (car oobjc)) 2))))
			  (seq (read-from-string (pathname-name (car oobjc)))))
		    (cons (cons 0 pid) (cons seq (get-universal-time))))))
    
      ;; history points from later to earlier to recover latest
      ;; want to follow chain from early to latest.
      (let ((hterm (history-of-iobjc-term
		       (term-of-idata-term (cdr oobjc)))))
	(when (and hterm (not (ivoid-term-p hterm)))
	  (let ((oldstamp (term-to-stamp
			   (stamp-of-idata-persist-term
			    hterm))))
	    (setf -old oldstamp -new newstamp)
	    (setf (gethash oldstamp *recovered-graph*) newstamp)))))))

(defvar *recovered-inf-data* nil)
(defun recover-inf-data (pathname)
  (let ((dl (directory pathname)))
    (format t "~a" (length dl))
    ;;(setf -dl dl) (break)
    (let ((acc nil)
	  )
      
      (dolist (fname dl)
	(let ((k (pathname-type fname)))
	  (when (string= "INF-SOURCE" k)
	    (push fname acc))))
    
    (setf *recovered-inf-data* acc)
    (length acc))))

(defvar *recovered-inf-list* nil)
(defun recover-inf-list ()
 (let ((acc nil))

   (dolist (fname (cdr *recovered-inf-data*))
    ;; recover stamp from filename
    ;; name is sequence and last dir of path is process_id
    ;;(setf -a oobjc) (break "rog")
    (let ((stamp (let ((pid (intern-system
				(car (last (pathname-directory fname) 2))))
			  (seq (read-from-string (pathname-name fname))))
		    (cons (cons 0 pid) (cons seq (get-universal-time))))))

      (let ((d (make-instance 'inf-source
			      'stamp stamp
			      'type 'inf-source)))
	(data-import d (db-read stamp 'inf-source))

	(let* ((step (step-of-inf-source d))
	       (tac (term-of-source d))
	       (goal (goal-of-inf-step step))
	       (refinement (refinement-of-inf-step step))
	       (subgoals (when refinement (subgoals-of-refinement refinement))))

	  (when (and refinement (top-refinement-p refinement))
	    ;;(setf -d d -g goal -t tac -s subgoals) (break "ril")
	    (push (list stamp goal tac subgoals) acc))))))

   (setf *recovered-inf-list* acc)
   (format t "~%~aInf objects found. " (length acc))))

(defun save-recovered-inf-list ()
  (write-terms-to-file
   "~/recovered-inf-data-markb.trm"
   (mapcar #'(lambda (a)
	       (instantiate-term
		(instantiate-operator `inf_data` nil)
		(list
		 (instantiate-bound-term (stamp-to-term (first a)))
		 (instantiate-bound-term (goal-to-term (second a)))
		 (instantiate-bound-term (third a))
		 (instantiate-bound-term (map-list-to-ilist (fourth a) (isubgoal-nil-term) #'goal-to-term)))))
	   (sort *recovered-inf-list* #'>
		 :key #'(lambda (x)
			  (sequence-of-stamp (car x))
			  ))))
  nil)
|#
(defun db-stream-close (stream)
  (let* ((s (prl-stream-stream (prl-stream-of-cprl-stream stream)))
	 (input-p (input-stream-p s))
	 (pathname (pathname s)))
    (remhash (key-of-file pathname)
	     (if input-p *open-rlog-table* *open-wlog-table*))
    (close s)))

(defun log-close (log)
  (db-stream-close log))
    


;;LAL todo, redefined in orb, need to put in right place, mv fns to here since now needed?
(defun db-environments ()
  (mapcar #'(lambda (dbe)
	      (cons (tokens-of-itokens-term
		     (address-of-ienvironment-term
		      (environment-of-ilog-description-term (car dbe))))
		    (cdr dbe)))
	  (sort (db-query)
		#'>
		:key #'(lambda (dbe) (time-of-stamp (cdr dbe ))))))


(defvar *db-environments* nil)

(defun update-db-environments (env)
  (push (cons (address-of-environment env)
	      (stamp-of-environment env))
	*db-environments*))
	 
(defun reset-db-environments ()
  (setf *db-environments* (db-environments)))
	 
(defun match-environment-in-list (tags envs)
  (find-first #'(lambda (env)
		  (let ((address (address-of-environment env)))
		    (when (forall-p #'(lambda (type) (member type address))
				    tags)
		      env)))
	      envs))




(defun match-db-environment (pname)
  (unless (and (boundp '*db-environments*) *db-environments*)
    (setf *db-environments* (db-environments)))
  (find-first #'(lambda (env)
		  (let ((address (car env)))
		    (when (forall-p #'(lambda (type) (member type address))
				    pname)
		      env)))
	      *db-environments*))


(defun match-db-environment-all (pname)
  (unless (and (boundp '*db-environments*) *db-environments*)
	  (setf *db-environments* (db-environments)))
  (mapcan #'(lambda (env)
		  (let ((address (car env)))
		    (when (forall-p #'(lambda (type) (member type address))
				    pname)
		      (list env))))
	      	  *db-environments*))

(defun match-db-environment-all-sub (pname)
  (unless (and (boundp '*db-environments*) *db-environments*)
	  (setf *db-environments* (db-environments)))
  (mapcan #'(lambda (env)
		  (let ((address (car env)))
		    (when (forall-p #'(lambda (type)
					(find-first #'(lambda (a) (string= (subseq (string a) 0 (length (string type))) (string type)))
						    address))
				    pname)
		      (list env))))
	      	  *db-environments*))

;; p addr stamp -> bool
(defun filter-db-environments-aux (p)
  (unless (and (boundp '*db-environments*) *db-environments*)
    (setf *db-environments* (db-environments)))

  (filter #'(lambda (env)
	      (funcall p (car env) (cdr env)))
	  *db-environments*))

;; paddr is list of tokens.
(defun filter-db-environments (paddr)
  (unless (and (boundp '*db-environments*) *db-environments*)
    (setf *db-environments* (db-environments)))
  (let ((patterns (mapcar #'(lambda (n) (string-pattern-search #'identity (string n))) paddr)))
    (filter #'(lambda (env)
		(let ((address (car env)))
		  (forall-p #'(lambda (pattern)
				(exists-p #'(lambda (n) (funcall pattern (string n))) address))
			    patterns)))
		
	    *db-environments*)))




;;;;	TODO would like to be able to randomly access terms associated with stamps.
;;;;	TODO However, inlined data makes it difficult to find associated terms
;;;;	TODO without scanning the log file (which may be acceptable since infrequent). 

(defun stamp-filenames (stamp)
  (directory (merge-pathnames (stamp-to-pathname stamp nil t)  ;; nil could be "*"
			      (make-pathname :type :wild))))


