;;; -*- Mode:Lisp; Package:CL-User; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			    MathBus Package Definition
;;; ===========================================================================
;;; (c) Copyright 2000 Cornell University

;;; $Id: mbs-reg.lsp,v 1.1 2004/02/26 23:12:03 eaton Exp $

#+lucid
(in-package 'mathbus)

#-lucid
(in-package "MATHBUS")

;;;; export/import mathbus symbols

(defvar *mathbus-exported-symbols*
  '("*STREAM-MODE*"

    "MAKE-MBNODE"
    "MBNODE"
    "MB-INTEGER"
    "INTEGER-VALUE"
    "LISP-INTEGER"
    "MB-STRING"
    "STRING-VALUE"
    "SUBTERM-TYPES"
    "MBNODE-LABEL"
    "MBNODE-NSUBTERMS"
    "MBNODE-SUBTERM"
    "NUMERIC-LABEL"
    "SYMBOLIC-LABEL"
    "SELECT-CONTEXT"
    "DECLARE-LOCAL-STRINGID"
    "HASH-CODE"
    "BIND-ONE"
    "UNBIND"
    "GLOBAL-ASSIGN"
    "LOOP-OVER-SUBTERMS"
    "NODE-ITERATE"
    "PRINT-NODE"
    "WRITE-NODE"
    "READ-NODE"
    "GENERATE-LABEL-DECLARATIONS"
    "INITIALIZE-BASE64"))
    

(export (mk::intern-in-package 'MATHBUS
			       '("*STREAM-MODE*"

				 "MAKE-MBNODE"
				 "MBNODE"
				 "MB-INTEGER"
				 "INTEGER-VALUE"
				 "LISP-INTEGER"
				 "MB-STRING"
				 "STRING-VALUE"
				 "SUBTERM-TYPES"
				 "MBNODE-LABEL"
				 "MBNODE-NSUBTERMS"
				 "MBNODE-SUBTERM"
				 "NUMERIC-LABEL"
				 "SYMBOLIC-LABEL"
				 "SELECT-CONTEXT"
				 "DECLARE-LOCAL-STRINGID"
				 "HASH-CODE"
				 "BIND-ONE"
				 "UNBIND"
				 "GLOBAL-ASSIGN"
				 "LOOP-OVER-SUBTERMS"
				 "NODE-ITERATE"
				 "PRINT-NODE"
				 "WRITE-NODE"
				 "READ-NODE"
				 "GENERATE-LABEL-DECLARATIONS"
				 "INITIALIZE-BASE64"
				 "READ-REGISTRY"
				 "PASCII2NUM"))
	"MATHBUS")


(import (mk::intern-in-package 'MATHBUS
			       '("*STREAM-MODE*"
				 "MAKE-MBNODE"
				 "MBNODE"
				 "MB-INTEGER"
				 "INTEGER-VALUE"
				 "LISP-INTEGER"
				 "MB-STRING"
				 "STRING-VALUE"
				 "SUBTERM-TYPES"
				 "MBNODE-LABEL"
				 "MBNODE-NSUBTERMS"
				 "MBNODE-SUBTERM"
				 "NUMERIC-LABEL"
				 "SYMBOLIC-LABEL"
				 "SELECT-CONTEXT"
				 "DECLARE-LOCAL-STRINGID"
				 "HASH-CODE"
				 "BIND-ONE"
				 "UNBIND"
				 "GLOBAL-ASSIGN"
				 "LOOP-OVER-SUBTERMS"
				 "NODE-ITERATE"
				 "PRINT-NODE"
				 "WRITE-NODE"
				 "READ-NODE"
				 "GENERATE-LABEL-DECLARATIONS"
				 "INITIALIZE-BASE64"
				 "READ-REGISTRY"
				 "PASCII2NUM"))
	cl-user::*system-package-name*)



;;; mbs extras, to compress frequently sent MetaPRL tokens
(defvar *token-file*  (extend-pathname (pathname cl-user:*system-path-prefix*) '("library" "orb") "mbs-mpl.txt"))
(defvar *token-table* (make-hash-table :test #'equal))

;; In this simple implementation, a registry is a collection of hash
;; tables, one or two for each registry type.  There are two
;; registries, one for the global information, and one for the local
;; information.


;; could hack up somehting to make cmu case behave like -cmu.
;; however I'm hoping it is just an efficiency hack.
(defun make-equalp-hash-table ()
  #+cmu(make-hash-table :test 'equal)
  #-cmu(make-hash-table :test #'equalp)
  )

(defvar *global-registry* (make-equalp-hash-table))

(defvar *local-registry*  (make-equalp-hash-table))

(defvar *registry-types* ()
  "All known registry types")

(defvar *registry-file*)

(defmacro define-registry-type (label &optional (bidirectional? nil))
  (unless (stringp label)
     (error "~A was expected to be a string" label))
  `(unless (member ,label *registry-types* :test #'string-equal)
     (setf (gethash ,label *global-registry*)
           ,(if bidirectional?
		`(list (make-equalp-hash-table) 
		  (make-hash-table :test #'eql))
		`(list (make-equalp-hash-table))))
    (setf (gethash ,label *local-registry*)
          ,(if bidirectional?
	       `(list (make-equalp-hash-table) 
		 (make-hash-table :test #'eql))
	       `(list (make-equalp-hash-table))))
     (push ,label *registry-types*)))

(define-registry-type "StringId" T)
(define-registry-type "SubTypes" nil)

(defun clear-registry (&optional (global nil) (local nil))
  (when global
    (do ((types *registry-types* (cdr types)))
	((null types))
      (clrhash (gethash (car types) *global-registry*)))
    ;;(loop for type in *registry-types*
    ;;do (clrhash (gethash type *global-registry*)))
    )
  (when local
    (do ((types *registry-types* (cdr types)))
	((null types))
      (clrhash (gethash (car types) *local-registry*)))
    ;;(loop for type in *registry-types*
    ;;do (clrhash (gethash type *local-registry*)))
    ))

(defun registry-lookup-value (id regtype)
  (unless (stringp id)
    (setq id (string id)))
  (unless (stringp regtype)
    (setq regtype (string id)))
  (or (gethash id (first (gethash regtype *local-registry*)))
      (gethash id (first (gethash regtype *global-registry*)))))

(defun registry-lookup-identifier (regtype value)
  (unless (stringp regtype)
    (setq regtype (string regtype)))
  (unless (integerp value)
    (error "Invalid registry value: ~S" value))

  (let ((reg (gethash regtype *local-registry*)))
    (unless (rest reg)
      (error "Not a bidirectional registry property: ~A" regtype))
    (or (gethash value (second reg))
	(gethash value (second (gethash regtype *global-registry*))))))

(defun registry-store-local (id regtype value)
  (unless (stringp id)
    (setq id (string id)))
  (unless (stringp regtype)
    (setq regtype (string id)))
  (unless (integerp value)
    (error "Invalid registry value: ~S" value))

  (let ((reg (gethash regtype *local-registry*)))
    (when (null reg)
      (error "Not a valid registry type: ~A" regtype))
    
    (setf (gethash id (first reg)) value)
    (when (rest reg)
      (setf (gethash value (second reg)) id))))

;;LAL added this
(defun registry-store-global (id regtype value)
  (unless (stringp id)
    (setq id (string id)))
  (unless (stringp regtype)
    (setq regtype (string id)))
  (unless (integerp value)
    (error "Invalid registry value: ~S" value))

  (let ((reg (gethash regtype *global-registry*)))
    (when (null reg)
      (error "Not a valid registry type: ~A" regtype))
    
    (setf (gethash id (first reg)) value)
    (when (rest reg)
      (setf (gethash value (second reg)) id))))

;; Load a registry file

;; Scan away all the white space, then determine if the next item is a
;; string or a number.  Return whichever it is.
(defun read-item (stream)
  (let ((char))
    (labels ((scan-white-space ()
	       (loop 
		(setq char (read-char stream nil 'eof))
		(when (eql char 'eof)
		  (return-from read-item nil))
		(when (char= char #\;)
		  (read-to-eol))
		(when (or (alphanumericp char)
			  (char= char #\-))
		  (return nil))))
	     (read-to-eol ()
	       (loop
		(setq char (read-char stream nil 'eof))
		(cond ((eql char 'eof)
		       (return-from read-item nil))
		      ((member char '(#\newline #\linefeed) :test #'char=)
		       (return nil)))))
	     (read-string ()
	       (let ((buffer (list char))
		     string)
		 (loop while (alphanumericp (setq char (read-char stream)))
		       do (push char buffer))
		 (setq string (make-string (length buffer)))
		 (loop for i downfrom (1- (length buffer))
		       while (not (minusp i))
		       do (setf (char string i) (pop buffer)))
		  string))
	     (read-number ()
	       (let ((base 10.)
		     (value 0)
		     (neg? nil))
		 (cond ((char= char #\-)
			(setq neg? t))
		       ((char= char #\0)
			(if (char= (setq char (read-char stream)) #\x)
			    (setq base 16.)
			    (if (alphanumericp char) (unread-char char))))
		       (t (setq value (- (char-code char) #.(char-code #\0)))))
		 (loop while (alphanumericp (setq char (read-char stream)))
		       do (setq value 
				(+ (* value base)
				   (- (char-code char) 
				      (cond ((char<= #\0 char #\9)
					     #.(char-code #\0))
					    ((char<= #\A char #\Z)
					     (- #.(char-code #\A) 10.))
					    ((char<= #\a char #\z)
					     (- #.(char-code #\a) 10.))
					    (t (error "Bad character")))))))
		 (if neg? (- value) value))))	     
		   
      (scan-white-space) 
      (if (or (digit-char-p char)
	      (char= char #\-))
	  (read-number)
	  (read-string)))))

(defun default-registry-files ()
  (loop with mathbus = (mk::find-system 'mathbus) and target
	for comp in (mk::component-components mathbus)
	do (when (string-equal (mk::component-name comp) "registry-file")
	     (setq target 
		   (first (mk::file-pathnames-in-component comp :source)))
	     (return (values (make-pathname :type "txt" :defaults target)
			     target)))))

(defvar *registry-header-text*
";;; -*- Mode:Lisp; Package:CL-User; Base:10; Lowercase:T; Syntax:Common-Lisp -*-~%
;;; ===========================================================================~%
 ;;;			    MathBus Term Labels~%
;;; ===========================================================================~%
;;; (c) Copyright 2000 Cornell University~%~%
;;;  DO NOT MODIFY. THIS IS AN AUTOMATICALLY GENERATED FILE!!! ~%~%
(in-package \"MATHBUS\")~%~%")

;; Load all the information in the registry into the currently running Lisp
(defun read-registry (&optional (file *registry-file*))
  (with-open-file (stream file :direction :input)
    (loop for ident = (read-item stream)
	  do (when (null ident)
	       (return t))
	  (registry-store-local ident (read-item stream) (read-item stream)))))

;; Load the static token hash table
(defun read-tokens (&optional (file *token-file*))
  (when (probe-file file)
    (with-open-file (stream file :direction :input)
      (loop for ident = (mread-string stream)
	    do (when (null ident)
		 (return t))
	    (let ((num (mread-number stream)))
	      (setf (gethash ident *token-table*) num)
	      (setf (gethash num *token-table*) ident))))))

;; Generated the (LISP) declaration file for label names.
(defun generate-label-declarations (&optional ofile file)
 (generate-registry-declarations ofile file))
(defun generate-registry-declarations (&optional ofile file)
  (when (or (null ofile) (null file))
    (multiple-value-bind (in out) (default-registry-files)
      (when (null ofile)
	(setq ofile out))
      (when (null file)
	(setq file in))))	
  (with-open-file (stream file :direction :input)
    (with-open-file (ostream ofile :direction :output)
    (format ostream *registry-header-text*)

    (loop for ident = (read-item stream)
	  with regtype and regval
	  do (when (null ident)
	       (return t))
	  (setq regtype (read-item stream))
	  (setq regval (read-item stream))
	  (when (string-equal regtype "StringId")
	    ;; Don't bother with MBS_ variables for single characters or Temps'
	    (unless (or (= (length ident) 1)
			(and (= (length ident) 5)
			     (char= (aref ident 0) #\T)
			     (char= (aref ident 1) #\e)
			     (char= (aref ident 2) #\m)
			     (char= (aref ident 3) #\p)))
	      (format ostream "~%(defvar MBS_~A #x~16R)~%(export 'mathbus::MBS_~A 'mathbus)~%(import 'mathbus::MBS_~A cl-user:*system-kind*)~%" 
		      ident regval (string-upcase ident) (string-upcase ident))))
	  (format ostream "(registry-store-local \"~A\" \"~A\" ~D)~%"
		  ident regtype regval))
    (format ostream "~%~%;;; End of automatically generated file.~%~%"))))
           
(defun mread-string (s)
  (let* ((char (read-char s nil 'eof))
	 (buffer (list char))
	 string)
    (when (eql char 'eof)
	  (return-from mread-string nil))
    (loop while (or (alphanumericp (setq char (read-char s))) (eq char #\_))
	  do (push char buffer))
    (setq string (make-string (length buffer)))
    (loop for i downfrom (1- (length buffer))
	  while (not (minusp i))
	  do (setf (char string i) (pop buffer)))
    string))


(defun mread-number (stream)
  (let ((base 10)
	(value 0)
	(char (read-char stream nil nil)))
    (setq value (- (char-code char) 48))
    (loop while (and (setq char (read-char stream nil nil)) (alphanumericp char))
	  do (setq value 
		   (+ (* value base)
		      (- (char-code char) 48))))
    value))
