;; -*- Mode:Lisp; Package:CL-User; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			    MathBus Package Definition
;;; ===========================================================================
;;; (c) Copyright 2000 Cornell University

;;; $Id: mbs-trms.lsp,v 1.1 2004/02/26 23:12:03 eaton Exp $

(in-package "MATHBUS")
(defvar *unicode-p* nil) ;;LAL added this to make node smaller fttb


(defconstant *minimum-global-numeric-label* #X00000000)
(defconstant *maximum-global-numeric-label* #X00FFFFFF)
(defconstant *minimum-local-numeric-label*  #X01000000)
(defconstant *maximum-local-numeric-label*  #XFFFFFFFF)

;; In this reference implementation, the local labels are allocated
;; from the maximum down, however this is not necessary in all
;; implementations.  The precise allocations of local labels should
;; be transparent to the user.
(defvar *next-local-label* *maximum-local-numeric-label*)

;; All of the information about labels is contained in the registry.
(defmacro symbolic-label (num)
  `(registry-lookup-identifier "StringId" ,num))

(defmacro numeric-label (sym)
  `(registry-lookup-value 
     (if (string ,sym) ,sym (string ,sym))
     "StringId"))

;; Subterm-Types returns the subtypes of a node (ie. which sub-terms are 
;; 32-bit integers and which are nodes themselves.) The meansings are:
;;
;; <n>  non-negative integer: the number of "leading" sub-terms that 
;;      are 32-bit integers. 
;; -1   All of the sub-terms are 32-bit integers
;; -2   The first sub-term is a node and the rest are 32-bit integers
;; -3   The even indexed sub-terms are 32-bit integers
;; -4   The odd indexed sub-terms are 32-bit integers
;; -5   The first sub-term is a string identifier and the remaining 
;;      sub-terms are references to nodes. 
;; -6   The even indexed sub-terms are string identifiers and the odd 
;;      indexed sub-terms are references to nodes. 
;; -7   The odd indexed sub-terms are string identifiers and the even
;;      indexed sub-terms are references to nodes. 

(defmacro subterm-types (num)
  `(registry-lookup-value 
    (registry-lookup-identifier "StringId" ,num)
    "SubTypes"))


;; The following macro should logically appear in the section on
;; iterators, but it is used much earlier.

;; Binds SUBTYPES to the sub-types of the node,
;; also binds TYPE for each sub-term.  
;; TYPE = nil for node reference
;; TYPE = :32bit for 32-bit integer
;; TYPE = :stringId for string identifier

(defmacro loop-over-subterms (node (ind subtype) &body body)
  
   `(let ((subtypes (or (subterm-types (mbnode-label ,node)) 0));;LAL or
	  (,subtype nil)
	  (len+1 (1+ (mbnode-nSubterms ,node))))
     
     (labels ((body (,ind) ,@body)
	      (alternate (first second)
		(loop for i upfrom 1 below len+1
		      do (setq ,subtype first)
		      (body i)
		      (unless (< (incf i) len+1)
			(setq ,subtype second)
			(body i)))))
       
       (cond ((= subtypes 0)		; All references to nodes
	      (setq ,subtype nil)
	      (loop for i upfrom 1 below (1+ (mbnode-nSubterms ,node))
		    do (body i)))
	     ((= subtypes -1)		;All sub-terms are 32-bit ints
	      (setq ,subtype :32bit)
	      (loop for i upfrom 1 below (1+ (mbnode-nSubterms ,node))
		    do (body i)))
	     ((= subtypes -2)
	      (setq ,subtype nil)
	      (body 1)
	      (setq ,subtype :32bit)
	      (loop for i upfrom 2 below (1+ (mbnode-nSubterms ,node))
		    do (body i)))
	     ((= subtypes -3)
	      (alternate nil :32bit))
	     ((= subtypes -4)
	      (alternate :32bit nil))
	     ((= subtypes -5)
	      (setq ,subtype :stringId)
	      (body 1)
	      (setq ,subtype nil)
	      (loop for i upfrom 2 below (1+ (mbnode-nSubterms ,node))
		    do (body i)))
	     ((= subtypes -6)
	      (alternate nil :stringId))
	     ((= subtypes -7)
	      (alternate :stringId nil))
	     ((< subtypes -7)
	      (error "Illegal SUBTYPES field: ~D" subtypes))
	     (t 
	      (setq ,subtype :32bit)
	      (loop for i upfrom 1 below (1+ subtypes)
		    do (body i))
	      (setq ,subtype nil)
	      (loop for i upfrom (1+ subtypes) 
		    below (1+ (mbnode-nSubterms ,node))
		    do (body i)))))))

;; The following routine creates a new local string identifer, if
;; needed, filling in the sub-types information, if provided, in the
;; registry.
(defun declare-local-stringId (symbolic-label &optional subtypes)
  (or (registry-lookup-value symbolic-label "StringId")
      (let ((numeric-label *next-local-label*))
	(when (< numeric-label *minimum-local-numeric-label*)
	  (error "Ran out of local-labels"))
	(unless (stringp symbolic-label)
	  (error "~S expected to be a string" symbolic-label))
	(when subtypes
	  (unless (and (numberp subtypes) (> subtypes -7))
	    (error "Invalid SubTypes field: ~S" subtypes)))
	
	(registry-store-local symbolic-label "StringId" numeric-label)
	(decf *next-local-label*)
	(when subtypes
	  (registry-store-local symbolic-label "SubTypes" subtypes))
	numeric-label)))

(defmacro mbnode-labelq (node) `(aref ,node 0))
(defun mbnode-label (node)
  (let ((lab (mbnode-labelq node)))
    (if (eql lab MBS_Attributes)
	(mbnode-labelq (aref node 1))
	lab)))

(defmacro mbnode-nSubtermsq (node)
  `(1- (array-dimension ,node 0)))
(defun mbnode-nSubterms (node)
  (mbnode-nSubtermsq
   (if (eql (mbnode-labelq node) MBS_Attributes)
       (aref node 1)
       node)))

(defmacro mbnode-subtermq (node i) `(aref ,node ,i))
(defun mbnode-subterm (node i)
  (mbnode-subtermq
   (if (eql (mbnode-labelq node) MBS_Attributes)
       (aref node 1)
       node)
   i))

(defun mbnode-set-subterm (node i val)
  (setf (aref (if (eql (mbnode-labelq node) MBS_Attributes)
		  (aref node 1)
		  node)
	      i)
	val))

(defsetf mbnode-subterm mbnode-set-subterm)

(defun mbnode-subterm-float (node i)
   (declare (ignore node i))
   (error "Not yet implemented"))

(defmacro make-mbnode-old (label length)
  `(let ((node (make-array (+ 1 ,length))))
     (setf (mbnode-labelq node) ,label)
     node))

(defmacro make-mbnode (label length)
  `(let ((node (make-array (+ 1 ,length))))
     (setf (mbnode-labelq node)  (if (numberp ,label) ,label (numeric-label ,label)))
     node))

(defmacro mbnode-old (label &rest args)
  `(let ((node (make-mbnode (numeric-label ,label) ,(length args))))
     ,@(loop for i below (length args)
	     collect `(setf (mbnode-subtermq node (1+ ,i)) ,(pop args)))
     node))

#+ignore
 ;; args should be a list
(defmacro mbnode (label args)
  `(let ((node (make-mbnode (if (numberp ,label) ,label (numeric-label ,label)) ,(length args))))
     ,@(loop for i below (length args)
	     collect `(setf (mbnode-subtermq node (1+ ,i)) ,(pop args)))
     node))

;; args should be a list
(defun mbnode (label args)
  (let ((node (make-mbnode (if (numberp label) label (numeric-label label)) (length args))))
     (loop for i below (length args)
	     collect (setf (mbnode-subtermq node (1+ i)) (pop args)))
     node))

(defun mb-integer (int &optional size-p label)
 (when size-p
       (when (> int #x3fffffff) (format t "~s" int) (error "mb-integer too big")))
  (let* ((ints 
	  (if (zerop int)
	      (list 0)
	    (loop for res = (abs int) then (floor res 1000000000)
		  while (not (zerop res))
		  collect (rem res 1000000000))))
	 (neg? (minusp int))
	 (len (length ints))
	 (node (make-mbnode (or label MBS_LongInteger) (length ints))))
    (loop for i below len
	  do (setf (mbnode-subtermq node (- len i))
		   (pop ints)))
    (when neg?
	  (setf (mbnode-subtermq node 1) (- (mbnode-subtermq node 1)))) 
    node))

(defvar *32bit-sign-bit* (ash 1 31))
(defvar *32bit-twos-complement* (ash 1 32))

(defun integer-value (node)
  (let ((base (mbnode-subtermq node 1))
	(sign 1))
    (when (minusp base)
      (setq base (- base))
      (setq sign -1))

    (loop for i upfrom 2
	  while (< i (1+ (mbnode-nSubterms node)))
	  do (setq base (+ (* base 1000000000) (mbnode-subtermq node i))))
    (if (minusp sign) 
	(- base)
	base)))

;; The following packs ASCII strings into 32-bit words, two characters
;; to a word.  So, every other byte is 0.  This is leave room for the
;; Unicode encoding that I will put in soon.
;; when added set unicode-p to t
(defun mb-string-with-unicode (str &optional label)
  (let* ((len (length str))
	 (node (make-mbnode (or label MBS_String) (1+ (ash (1+ len) -1)))))
    (setf (mbnode-subtermq node 1) len)
    (loop for i below len by 2
	  for j upfrom 2
	  do (setf (mbnode-subtermq node j)
		   (if (< (1+ i) len)
		       (logior (ash (char-code (char str i)) 16)
			       (char-code (char str (1+ i))))
		     (ash (char-code (char str i)) 16))))
    node))

(defun mb-string (str &optional label)
  (if *unicode-p*
      (mb-string-with-unicode str label)
    (let* ((len (length str))
	   (m (mod len 4))
	   (d (/ (- len m) 4))
	   (a (if (= m 0) (1+ d) (+ 2 d)))
	   (node (make-mbnode (or label MBS_String) a )))
      (setf (mbnode-subtermq node 1) len)
      (do ((i 0 (+ i 4))
	   (j 2 (1+ j)))
	  ((> (1+ i) len))
	  (setf (mbnode-subtermq node j)
		     
		(logior (ash (char-code (char str i)) 24)
			(if (> (+ 2 i) len) 0 (ash (char-code (char str (1+ i))) 16))
			(if (> (+ 3 i) len) 0 (ash (char-code (char str (+ 2 i))) 8))
			(if (> (+ 4 i) len) 0 (char-code (char str (+ 3 i)))))))
		    
      node)))

(defun string-value-unicode (node)
  (let* ((len (mbnode-subtermq node 1))
	 string)
    (setq string (make-string len))
    (loop with i = 0
	  for j upfrom 2
	  while (< i len)
	  do (setf (char string i) 
		   (code-char (ldb (byte 16 16) (mbnode-subtermq node j))))
	  (incf i)
	  (when (< i len)
	    (setf (char string i) 
		  (code-char (ldb (byte 16 0) (mbnode-subtermq node j))))
      (incf i)))
    string))

(defun string-value (node)
  (if *unicode-p*
      (string-value-unicode node)
    (let* ((len (mbnode-subtermq node 1))
	   string)
      (setq string (make-string len))
      (do ((i 0 (+ 4 i))
	   (j 2 (1+ j)))
	  ((>= i len))
	  (let ((n (mbnode-subtermq node j)))
	    (when (< (+ 3 i) len)
		  (setf (char string (+ 3 i))
			(code-char (logand n #xFF))))
	    (when (< (+ 2 i) len)
		  (setf (char string (+ 2 i))
			(code-char (logand (ash n -8) #xFF))))
	
	    (when (< (+ 1 i) len)
		  (setf (char string (1+ i))
			(code-char (logand (ash n -16) #xFF))))
	
	    (setf (char string i) 
		  (code-char (logand (ash n -24) #xFF)))
	    
	    ))
	
      
      string)))

(defun mb-float (fl)
  (declare (ignore fl))
  (error "Not yet implemented"))

(defun float-value (node)
  (declare (ignore node))
  (error "Not yet implemented"))

(defun print-node (node &optional (stream *standard-output*))
  (cond ((numberp node) (princ node stream))
	((= (mbnode-label node) MBS_LongInteger)
	 (format stream "[~D]" (integer-value node)))
	( (or (= (mbnode-label node) MBS_String) (= (mbnode-label node) MBS_Token))
	 (format stream "\"~D\"" (string-value node)))
	(t (format stream "(~A" (symbolic-label (mbnode-label node)))
	   (loop-over-subterms node (i type)
			       (princ " (" stream)
			       (cond ((null type)
				      (print-node (mbnode-subtermq node i) stream))
				     ((eql type :32bit)
				      (princ (mbnode-subtermq node i) stream))
				     ((eql type :stringId)
				      (format stream "'~A'"
					      (registry-lookup-identifier 
					       "StringId" (mbnode-subtermq node i)))))
			       (princ ")" stream))
	    (princ ")" stream)
	     nil)))

;; Contexts these are used by the procedural labels of the data
;; structure and during communications.
					 
(user::defclass context ()
		((user::globals :initform (make-hash-table) 
				:accessor globals-of)
		 (user::last-update-time :initform 0
					 :accessor update-time
					  )))
;;#+ignore
(user::defmethod update-time ((c context))
  (setf (update-time c) (get-universal-time))) ;; LAL replace update-time calls below with setf's

(defvar *contexts* (make-hash-table)
  "All contexts currently known about")

(defvar *current-context* nil)

(defun select-context (name)
   (let ((context (gethash name *contexts*)))
      (when (null context)
         (setq context (user::make-instance 'context))
         (setf (gethash name *contexts*) context))
      (setq *current-context* context)))

;; These are local bindings
(defmacro bind-one (key value &optional (context '*current-context*))
  `(push (cons ,key ,value) ,context))

;; Unbind n bindings
(defmacro unbind (&optional (n 1) (context '*current-context*))
  (cond ((eql n 1)
	 `(pop ,context))
	(t `(progn
	      (loop for i below n
		    do (when (atom ,context))
		    (error "Overpoped the stack"))
	      (pop ,context)))))
;;#+ignore
(defun lookup (key &optional (context *current-context*))
 (loop with c = context and k and value
	do (if (atom c)
	       (return (gethash key (globals-of c))) ;;LAL added globals-of fix
	     (when (eql key (first (first c)))
	       (return (rest (first c)))))))

(defmacro global-assign (key value &optional (context *current-context*))
  `(%global-assign ,context ,key ,value))

(user::defmethod %global-assign ((c context) key value)
  (setf (gethash key (globals-of c)) value)
  (update-time c))

;;#+ ignore
(defun assign (key value &optional (context *current-context*))
 
  (loop with c = context;;and k and value
	do (if (atom c)
	       (error "No local binding of ~S" key)
	     (when (eql key (first (first c)))
		   (setf (rest (first c)) value)
		   (return value)))))

;; Simple iterators

(defvar *attributes* nil)

(defun get-attribute (node attrib)
  (let ((attr (gethash node *attributes*)))
    (unless attr
      (loop for i upfrom 1 by 2 below (mbnode-nSubterms node)
	    do (when (eql (mbnode-subterm node i) attrib)
		 (return (mbnode-subterm node (1+ i))))))))

(defun store-attribute (node attrib value)
  (let ((attr (gethash node *attributes*)))
    (unless attr
      (loop for i upfrom 2 by 2 below (1+ (mbnode-nSubterms node))
	    do (when (eql (mbnode-subterm node i) attrib);;LAL nS
		 (setf (mbnode-subtermq node (1+ i)) value)
		 (return value))))))


(defun node-iterate (node func &optional 
				 (*current-context* *current-context*))
  (let ((context *current-context*)
	(*attributes* (make-hash-table)))
    (node-iterate1 node func context)))

(defun node-iterate1 (node func context)
  (let ((label (mbnode-labelq node)))
    (when (eql label MBS_ATTRIBUTES)
      (let ((new-node (mbnode-subtermq node 1))) 
	(setf (gethash new-node *attributes*) node)
	(setq node new-node)
	(setq label (mbnode-labelq node))))

    (cond ((eql label MBS_VOID)
	   (error "Illegal node label -- VOID"))
	  ((eql label MBS_NULL)
	   (values (funcall func node func :node context)
		   :done))
	  ((eql label MBS_MBVariable)
	   (multiple-value-bind (nodevalue continue-flag)
	       (funcall func node func :variable context)
	     (unless (eql continue-flag :done)
	       (let ((val (lookup node context)))
		 (when (null val)
		   (error "Variable encountered that has no value"))
		 (node-iterate1 val func context)))))
	  ((eql label MBS_MBBind)
	   (bind-one (mbnode-subtermq node 1) (mbnode-subtermq node 2))
	   (node-iterate1 (mbnode-subtermq node 3) func context)
	   ;; Note that when this function returns, there will no
	   ;; longer be a pointer to a context with this binding.
	   ;; Nonetheless, we explicitly do the unbind for clarity, and
	   ;; so that implementations that don't want to rely on
	   ;; garbage collection won't have (too many) problems.
	   (unbind))
	  ((eql label MBS_Attributes)
	   (setf (gethash (mbnode-subtermq node 1) *attributes*) node)
	   (node-iterate1 (mbnode-subtermq node 2) func context))
	  (t (multiple-value-bind (nodevalue continue-flag) 
		 (funcall func node func :node context)
	       (unless (eql continue-flag :done)
		 (loop-over-subterms node (i type)
                   (cond ((null type)
			  (node-iterate1 (mbnode-subtermq node i) 
					 func context))
			 ((eql type :stringId)
			  (funcall func (mbnode-subtermq node i)
				   func :stringId context))))
		 (values nodevalue :done)))))))

;; Canonicalize term structure
;; collect algvariables and algfunctions

(defconstant *rsl-stringId* 0)
(defconstant *rsl-subtypes* 1)

;; The following routine is used before transmitting a term in order
;; to ensure that it is actually a tree and to make sure that it is
;; self contained.

(defun canonicalize-term (term)
  (let ((*local-node-labels* nil)
	(*local-strings* nil)
	(*variables* nil))
    (node-iterate term
		  #'(lambda (node func state context)
		      (cond ((eql state :node)
			     (let ((label (mbnode-labelq node)))
			       (cond ((eql func :variable)
				      (pushnew node *variables*)
				      (values node :done))
				     ((>= label *minimum-local-numeric-label*)
				      (pushnew label *local-strings*)
				      (pushnew label *local-node-labels*)))))
			    ((and (eql state :stringId)
				  (>= node *minimum-local-numeric-label*))
			     (pushnew node *local-strings*)))))
    (when (or *local-node-labels* *local-strings*)
      (let* ((len (+ (length *local-node-labels*)
		     (length *local-strings*)
		     (length *variables*)
		     1))
	       (node (make-mbnode MBS_Sequence len))
	     (ind 1))
	(setf (mbnode-subtermq node len) term)
	(loop for nl in *local-strings*
	      for label = (symbolic-label nl)
	      do (setf (mbnode-subtermq node ind)
		       (mbnode 'RegistryStoreLocal 
			       (list nl
			       *rsl-stringId*
			       (mb-string label))))
	      (incf ind))

	(loop for nl in *local-node-labels* 

	     for label = (symbolic-label nl)
	      for mb-label = (mb-string label)
	      do 
	      (setf (mbnode-subtermq node ind)
		    (mbnode 'RegistryStoreLocal 
			     (list (registry-lookup-value label "SubTypes")
			    *rsl-subtypes*
			    mb-label)))
	      (incf ind))		       

	(setq term node)))
    term))

;; Hash code function.  The purpose of this function is to assign a
;; 32-bit integer to each term in a fashion that is both machine
;; architecture and language independent.

(defvar *hash-code-depth-limit*)
(defvar *hash-code-width-limit*)

(defvar *rotate-masks*
  (make-array 32 :initial-contents
	      '(0 
		#x80000000 #xC0000000 #xE0000000 #xF0000000
		#xF8000000 #xFC000000 #xFE000000 #xFF000000
		#xFF800000 #xFFC00000 #xFFE00000 #xFFF00000
		#xFFF80000 #xFFFC0000 #xFFFE0000 #xFFFF0000
		#xFFFF8000 #xFFFFC000 #xFFFFE000 #xFFFFF000
		#xFFFFF800 #xFFFFFC00 #xFFFFFE00 #xFFFFFF00
		#xFFFFFF80 #xFFFFFFC0 #xFFFFFFE0 #xFFFFFFF0
		#xFFFFFFF8 #xFFFFFFFC #xFFFFFFFE )))

(defun bit-rotate (num cnt)
  (setq cnt (logand cnt #x1F))		;Rotates cnt bits/mod 32

  (logior (ash (logandc1 (aref *rotate-masks* cnt) num) cnt)
	  (ash (logand (aref *rotate-masks* cnt) num) (- cnt 32))))

(defun hash-code (node &optional (*hash-code-depth-limit* 7)
		       (*hash-code-width-limit* 7))
  (hash-code-internal node 0))
    
(defun hash-code-internal (node depth)
  (if (>= depth *hash-code-depth-limit*)
      0
    (let* ((hash (mbnode-label node))
	   (subterm-limit (min *hash-code-width-limit*
			       (mbnode-nSubterms node))))
      (block loop
	(loop-over-subterms node (i type)
	  (when (<= i subterm-limit)
	    (return-from loop t))
	  (setq hash (logxor hash 
			     (bit-rotate
			      (if (null type)
				  (hash-code-internal (mbnode-subtermq node i)
						      (1+ depth))
				  (mbnode-subtermq node i))
			      i)))))
      hash)))

;;; To avoid problems with operating system differences, the byte
;;; stream generated can be restricted to consist solely of ASCII
;;; printing characters.  This format uses a radix 64 representation
;;; for the words, encoded as follows: 

(defvar *byte-count* 0)

(defvar *base64-translation-list*
   '(( 0 #\A) ( 1 #\B) ( 2 #\C) ( 3 #\D) ( 4 #\E) ( 5 #\F) ( 6 #\G) ( 7 #\H) 
     ( 8 #\I) ( 9 #\J) (10 #\K) (11 #\L) (12 #\M) (13 #\N) (14 #\O) (15 #\P) 
     (16 #\Q) (17 #\R) (18 #\S) (19 #\T) (20 #\U) (21 #\V) (22 #\W) (23 #\X) 
     (24 #\Y) (25 #\Z) (26 #\a) (27 #\b) (28 #\c) (29 #\d) (30 #\e) (31 #\f) 
     (32 #\g) (33 #\h) (34 #\i) (35 #\j) (36 #\k) (37 #\l) (38 #\m) (39 #\n) 
     (40 #\o) (41 #\p) (42 #\q) (43 #\r) (44 #\s) (45 #\t) (46 #\u) (47 #\v) 
     (48 #\w) (49 #\x) (50 #\y) (51 #\z) (52 #\0) (53 #\1) (54 #\2) (55 #\3) 
     (56 #\4) (57 #\5) (58 #\6) (59 #\7) (60 #\8) (61 #\9) (62 #\+) (63 #\/)))

(defvar *base64-by-num-table* (make-array 64))
(defvar *base64-by-char-table* (make-array 128))
(defvar *base64-char-count* 0)
(defvar *mask-table* (make-array 32))

(defun initialize-base64 ()
   (loop for i below 128
      do (setf (aref *base64-by-char-table* i) -1))
   (loop for (num char) in *base64-translation-list*
      do (setf (aref *base64-by-num-table* num) char)
      (setf (aref *base64-by-char-table* (char-code char)) num))
   (loop for i below 32
      do (setf (aref *mask-table* i) (1- (ash 1 i))))
   (setq *base64-char-count* 0))

(initialize-base64)

(defmacro num2pascii (num)
   `(aref *base64-by-num-table* ,num))

(defmacro pascii2num (char)
  `(let ((char ,char))
     (if (graphic-char-p char)
         (aref *base64-by-char-table* (char-code char))
         -1)))

(defmacro write-base64-char (char stream)
   `(progn
       (when (zerop *base64-char-count*)
          (write-char #\Newline ,stream)
          (setq *base64-char-count* 64))
       (write-char ,char ,stream)
       (decf *base64-char-count*)))

(defmacro write-base64-num (num stream)
   `(write-base64-char (num2pascii ,num) ,stream))
       
       
(defvar *stream-mode* :base64)

(defvar *base64-obuffer* 0)
(defvar *base64-ocount* 0)

;; This class translates between 8bit bytes streams on input and
;; base64 bytes streams on output.
(user::defclass base64-buffer ()
  ((stream :initarg :stream
	   :reader  stream-of)
   (buffer :initform 0 :accessor buffer-of)
   (cnt :initform 0 :accessor count-of)))

(user::defmethod write-byte64 (byte (buff base64-buffer))
  (setf (buffer-of buff) (logior (ash (buffer-of buff) 8)
				 (logand byte #xFF)))
  (when (= (incf (count-of buff)) 3)
    ;; Buffer is now full
    (flush-buffer buff)))

(user::defmethod flush-buffer ((buff base64-buffer))
  (user::with-slots (cnt buffer stream) buff
    (cond ((zerop cnt))
	  ((= cnt 1)
	   (write-base64-num (logand (ash buffer -2) #x3F)
			     stream)
	   (write-base64-num (logand (ash buffer 4) #x3F)
			     stream)
	   (write-base64-char #\= stream)
	   (write-base64-char #\= stream))
	  ((= cnt 2)
	   (write-base64-num (logand (ash buffer -10) #x3F)
			     stream)
	   (write-base64-num (logand (ash buffer -4) #x3F)
			     stream)
	   (write-base64-num (logand (ash buffer 2) #x3F)
			     stream)
	   (write-base64-char #\= stream))
	  (t (write-base64-num (logand (ash buffer -18) #x3F)
			       stream)
	     (write-base64-num (logand (ash buffer -12) #x3F)
			       stream)
	     (write-base64-num (logand (ash buffer -6) #x3F)
			       stream)
	     (write-base64-num (logand buffer #x3F)
			       stream)))   
    (setq cnt 0)
    (incf *byte-count* 4)
    (setq buffer 0)))

;;; Byte stream output
(defun write-32bit (num stream)
  (cond ((eql *stream-mode* :binary-byte)
	 (write-char (logand (ash num -24) #XFF) stream)
	 (write-char (logand (ash num -16) #XFF) stream)
	 (write-char (logand (ash num  -8) #XFF) stream)
	 (write-char (logand num #XFF) stream)
	 (incf *byte-count* 4))
	((eql *stream-mode* :base64)
	 (cond ((< num #xFD)
		(write-byte64 num stream))
	       ((<= num #xFFFF)
		(write-byte64 #xFD stream)
		(write-byte64 (ash num -8) stream)
		(write-byte64 (logand #xFF num) stream))
               ((<= num #xFFFFFF)
                (write-byte64 #xFE stream)
                (write-byte64 (ash num -16) stream)
                (write-byte64 (logand #xFF (ash num -8)) stream)
                (write-byte64 (logand #xFF num) stream))
	       (t (write-byte64 #xFF stream)
		  (write-byte64 (ash num -24) stream)
		  (write-byte64 (logand #xFF (ash num -16)) stream)
		  (write-byte64 (logand #xFF (ash num -8)) stream)
		  (write-byte64 (logand #xFF num) stream))))
	((eql *stream-mode* :debug-byte)
	 (format stream "[~16R ~16R ~16R ~16R]"
		 (logand (ash num -24) #XFF) (logand (ash num -16) #XFF)
		 (logand (ash num -8) #XFF) (logand num #XFF))
	 (incf *byte-count* 4))
	(t (error "Don't know how to write streams in mode: ~S"
		  *stream-mode*))))

(defvar *base64-ibuffer* 0)
(defvar *base64-icount* 0)

;; returns nil if eof
(defun base64-read-char (stream)
  (let* ((val (read-char stream nil nil))
	 (num (if val (pascii2num val) nil)))
    (loop while (and num (minusp num))
	  do (let ((val (read-char stream nil nil)))
	       (setq num (if val (pascii2num val) nil))))
    (when num (setq *base64-ibuffer* 
		    (logior (ash *base64-ibuffer* 6) num))
	  (incf *base64-icount* 6))))

;; returns nil if eof
(defun base64-read-8bit (stream)
  (when (base64-read-char stream)
	(if (< *base64-icount* 8)
	    (base64-read-char stream)) 
	(let ((temp *base64-ibuffer*))
	  (setq *base64-icount* (- *base64-icount* 8))
	  (setq *base64-ibuffer* (logand *base64-ibuffer* 
					 (aref *mask-table* *base64-icount*)))
	  (ash temp (- *base64-icount*)))))

;; returns nil if eof
(defun read-32bit (stream)
  (cond ((eql *stream-mode* :binary-byte)
         (let ((num (read-char stream nil nil)))
	   (when num
		 (logior 
		  (ash num 24)
		  (ash (read-char stream nil nil) 16)
		  (ash (read-char stream nil nil) 8)
		  (read-char stream)))))
        ((eql *stream-mode* :base64)
         (let ((char (base64-read-8bit stream)))
	   (when char
		 (cond ((= char #xFF)
			(logior (ash (base64-read-8bit stream) 24)
				(ash (base64-read-8bit stream) 16)
				(ash (base64-read-8bit stream) 8)
				(base64-read-8bit stream)))
		       ((= char #xFE)
			(logior (ash (base64-read-8bit stream) 16)
				(ash (base64-read-8bit stream) 8)
				(base64-read-8bit stream)))
		       ((= char #xFD)
			(logior (ash (base64-read-8bit stream) 8)
				(base64-read-8bit stream)))
		       (t char)))))
        (t (error "Don't know how to read streams in mode: ~S" 
		  *stream-mode*))))
;;#+ignore
;; Compute the maximum depth and width of a tree.
(defun depth-and-width (node)
  (let ((max-width 0))
    (labels ((func (node func state context) 
	       (when (eql state :node)
		 (let ((nsubterms (mbnode-nSubterms node))
		       (depth 0))
		   (setq max-width (max max-width nsubterms))
		   (loop-over-subterms node (i type)
		     (when (null type)
		       (setq depth 
			     (max (node-iterate1 (mbnode-subtermq node i)
						 #'func context)
				  depth))))
		   (1+ depth)))))
      (values (node-iterate node #'func) max-width))))

(defun write-node (node &optional (stream *standard-output*))
  (let ((*byte-count* 0))
    (when (eql *stream-mode* :base64)
      (setq stream (user::make-instance 'base64-buffer :stream stream)))

    ;; Write header information
    (write-32bit #xFADEBAC0 stream)
     ;;(multiple-value-bind (depth width) (depth-and-width node)
     ;;  (write-32bit depth stream)
      ;; (write-32bit width stream))

    ;; Write the term itself. 
    (cond ((numberp node)
	   (error "Term must be more than just a 32-bit integer"))
	  (t
	   ;; (%write-node (canonicalize-term node) stream) 
	   (%write-node node stream)
	  ))

    ;; Write trailer information
    ;; FIXTHIS: Need to compute the check-sum someplace...
    (write-32bit #XABCDEF00 stream)

    ;; Flush output if necessary
    (when (eql *stream-mode* :base64)
	  (flush-buffer stream))
    (setq *base64-char-count* 0)
    (format nil "~D bytes" *byte-count*)))

(defun %write-node (node stream)
  (let* ((op (mbnode-labelq node))
	 (nSubterms (mbnode-nSubterms node))
	 ;;(type (subterm-types op)) LAL
	 )
    (write-32bit op stream)
    (write-32bit nSubterms stream)
    (loop-over-subterms node (i type)
      (cond ((null type)
	     (%write-node (mbnode-subtermq node i) stream))
	    ((or (eql type :32bit) (eql type :stringId))
	     (write-32bit (mbnode-subtermq node i) stream))
	    (t (error "Illegal term sub-type: ~S" type))))))		    

(defvar *stringID-translation*)

;; returns nil if eof
(defun read-node (stream)
  (let ((*stringID-translation* (make-hash-table))) ;;need to be global
    ;;;;;;;;;
    ;; Handle header
    ;;;;;;;;
    (setq *base64-ibuffer* 0)
    (setq *base64-icount* 0)

    (let ((num (read-32bit stream)))
      (when num
	    (unless (eql num #xFADEBAC0)
		    (error "Bad header word for MathBus stream"))
	    ;; This implementation doesn't need to kow the depth or width of the
	    ;; node being constructed a priori.
	    ;;(read-32bit stream)
	    ;;(read-32bit stream)

	    (prog1 
		(read-node-internal stream)
	      ;; Trailer
	      (read-32bit stream))))))

(defun read-node-internal (stream) 
  (let* ((op (read-32bit stream))
	 (nSubterms (read-32bit stream))
	 (node (make-mbnode op nSubterms)))
    ;;(break)
    (loop-over-subterms node (i type) 
      (setf (mbnode-subtermq node i)
	    (cond ((null type)
		   (read-node-internal stream))
		  ((eql type :32bit) 
		   (read-32bit stream))
		  ((eql type :stringId)
		   (let ((strId (read-32bit stream)))
		     (or (gethash strId *stringID-translation*)
			 strId)))
		  (t (error "Illegal term sub-type: ~S" type)))))
    
    (when (eql op MBS_RegistryStoreLocal)
      (cond ((eql (mbnode-subtermq node 2) *rsl-stringId*)
	     (let ((new-op (declare-local-stringId 
			    (string-value (mbnode-subtermq node 3)))))
	       (unless (eql new-op op)
		 (setf (gethash op *stringID-translation*) new-op))))
	    ((eql (mbnode-subtermq node 2) *rsl-subtypes*)
	     (Error "not yet implemented"))))
    node))


;;; extra

(when (and (boundp '*token-file*) *token-file* (probe-file *token-file*))
  (read-tokens))
