
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;; twould be neat to test a client->client(A->B) request with serialized transaction.
;; so A does callback to server, then in callback sends req to B, b does callback to server.



(defvar *transaction-description* (description-to-term (new-description '(|TransactionClient|))))
(defvar *orb-description* (description-to-term (new-description '(|ORB| |TransactionClient|))))

;; Should cleanup if fails.

;;FTTB, assume in server starting server transaction.
(defun transaction-checkpoint (tstate)
  ;; make tstate weak to prevent lib change during transaction.
  (tstate-weaken tstate)
  (let ((stamp (new-transaction-stamp)))
    (broadcast stamp
	       (ipassport-term 'transaction
			       ;; table stamp?
			       (if (resource-p 'library)
				   (table-stamp-term-of-definition-table
				    (resource 'library))
				   (ivoid-term))
			       *transaction-description*
			       (icheckpoint-term (stamp-to-term stamp)))
	       'any)
    stamp))


(defun transaction-passport-term (table-stamp-term term)
  (ipassport-term 'transaction
		  table-stamp-term
		  *transaction-description*
		  term))

(defun orb-passport-term (stamp-term term)
  (ipassport-term 'orb
		  stamp-term
		  *orb-description*
		  term))

(defun transaction-begin (tstate)

  (let ((stamp (new-transaction-stamp)))

    (io-echo "B")

    (push-io-history (stamp-to-term stamp) 'begin)
    (show-telemetry "~%transation-begin ~a" (cadr stamp))

    ;;(setf a tstate b stamp) (break "tb")

    (tstate-set-begin tstate stamp)  
    (tstate-add tstate)

    ;; send begin broadcasts.
    ;;if transaction server-p
    (broadcast stamp
	       (transaction-passport-term
		(stamp-term-of-environment *orb*)
		;;(if (resource-p 'library)
		;;(table-stamp-term-of-definition-table (resource 'library))
		;;(ivoid-term))
		(ibegin-term (stamp-to-term stamp)
			     (tid)))
	       'any)

    ;;(break "tb")
    ))

;; To clear hung lib transactions :
;; b.
;; (transaction-cleanup (car *transactions-active*))

(defun transaction-cleanup (tstate &optional (kind '|lib|))
  (with-environment ((address-of-environment
		      (match-environment-in-list (list kind) *component*)))
    (transaction-end tstate nil)))

;; should cleanup if fails.
(defun transaction-end (tstate okp)
  ;; resolve pending completions
  ;;(break "te")
  ;;(with-io-echo-stats (nil "Transaction End"))

    (unwind-transaction okp tstate)

    (let ((stamp (begin-stamp-of-tstate tstate)))
      (show-telemetry "~%transaction-end ~a" (cadr stamp))
      (push-io-history (icons-term (ibool-term okp) (stamp-to-term stamp)) 'end)

      (tstate-delete tstate)

      ;; collect tents.
      ;; TODO: should touch history be table of transaction touches
      ;; so that touches from various transactions not intermixed.
      ;; 11/2002 moved history to tstate.
      (tstate-end tstate)

      (io-echo "E")
      ;; send end broadcast.
      (broadcast stamp
		 (transaction-passport-term
		  (stamp-term-of-environment *orb*)
		  ;;(if (resource-p 'library)
		  ;;(table-stamp-term-of-definition-table
		  ;;(resource 'library))
		  ;;(ivoid-term))
		  (iend-term (stamp-to-term stamp)))
		 'any) ))


;; transaction started, begin broadcast sent.
(defun transaction-serve (term)

  ;; have failure thrown if appropos.
  (interpret-result

   (let ((tstate (transaction-state)))

     (let ((cookie (when (checkpoint-of-itransaction-term term)
			 (term-to-standard-character-string
			  (stamp-to-term (transaction-checkpoint tstate))))))

       (bus-eval-callback
	(icommand-term
	 ;;(setf a tstate b term) (break "eh")
	 ;; send callback to tid-blink
	 (icallback-term (or cookie "")
			 (stamp-to-term (begin-stamp-of-tstate tstate))))))))

  ;; return ack to !transaction request.
  (iack-term))


;; appropriate means local if not server environment and no transaction to join.
(defmacro with-appropriate-transaction ((inter-orb-p readonlyp) &body body)
  ;; if client then want local transaction  if not tid, if tid then want global
  ;; t means join if there is one.
  `(if (let ((env (current-environment)))
	 (or (and ,inter-orb-p (not (orb-p env)))
	     (transaction-server-p env)
	     (and (tid) (not (orb-p env)) t)))
    (with-transaction (t ,readonlyp) ,@body)
    (with-local-transaction ,@body)))


;; broadcast eval needs to protect tstate so as to avoid with-transaction clash.
;; maybe with-e does this??

;;;;	oids : list of oids to restrict state to.
;;;;	  - for working map functions. Ie can export map this way.
;;;;	
;;;;	


(defmacro delay-broadcast-compiles (stamp cdescr &body body)
  `(do-delayed-compiles ,stamp
			,cdescr
			(let ((*broadcast-compiles-delayed* nil)
			      (*broadcast-compiles-orders* nil)
			      (*delay-broadcast-compiles* t))
			  ,@body
			  (cons *broadcast-compiles-delayed* *broadcast-compiles-orders*))))
	
(define-primitive |!subscribe| ((bool . un))
  (resources address stamp description synchro))
(define-primitive |!subscription| ((bool . un))
  (resources address stamp broadcasts))

;; current environment should be server
;; check term/stamp desc/term usage to make sure consistent.
;; lib-producer needs to be changed to take synchro as distinct arg.
(defun unsubscribe-server (sterm)
  (let* ((env (current-environment))
	 (blink (tid-blink))
	 (client-address (tags-of-ienvironment-address-term (address-of-isubscribe-term sterm)))
	 (client-synchro (synchro-of-isubscribe-term sterm))
	 )

    (let ((cenv (find-environment-in-list client-address
					  (imported-environments-of-bus-link blink))))

      (unless cenv
	(raise-error (error-message '(unsubscribe client-environment not)
				    client-address)))

      (let ((stampterm (when (and (ibool-term-p client-synchro)
				  (bool-of-ibool-term client-synchro))
			 (ml-text "record_event_checkpoint ()"))))

	;; 2/2003 was in scope of with-checkpoint callback hook, but I doubt that his really needs to be.
	;; probably put there at one time thinking the event-term might be used.
	(remove-broadcast-state-from-link-environment
	 (tokens-of-itokens-term (resources-of-isubscribe-term sterm))
	 (stamp-term-of-environment env)
	 (stamp-of-isubscribe-term sterm)
	 cenv
	 blink)

	(isubscription-term nil
			    (resources-of-isubscribe-term sterm)
			    (ienvironment-address-term (address-of-environment env))
			    (stamp-term-of-environment env)
			    ;; do we want event-term to go back??
			    (or stampterm (ivoid-term)))))))

(defun subscribe-server (sterm)
  
    (let* ((env (current-environment))
	   (address (address-of-environment env)))

      (let* ((types (tokens-of-itokens-term (resources-of-isubscribe-term sterm)))
	     (client-address (tags-of-ienvironment-address-term (address-of-isubscribe-term sterm)))
	     (client-stamp (stamp-of-isubscribe-term sterm))
	     (client-description (description-of-isubscribe-term sterm))
	     (client-synchro (synchro-of-isubscribe-term sterm))
	     (blink (tid-blink)))
	
	;; kludge : don't want to force journal subscribe to do a transaction thus
	;;  blink will be missing thus look it up.
	(when (eql 'journal (car client-address))
	  (setf blink
		(find-if #'(lambda (blink)
			  (member client-address (imported-environments-of-bus-link blink)
				  :test #'equal :key #'address-of-environment))
		      *bus-links*)))
	
	;;(setf -sterm sterm -client-synchro client-synchro -blink blink) (break "ss")

	;; MTT stop broadcasts until configure-term queued.

	;; update link to consumer
	(let* ((cenv (find-environment-in-list client-address
					       (imported-environments-of-bus-link blink))))
	      
	  (unless cenv
	    (raise-error (error-message '(subscribe client-environment not)
					client-address)))

      
	  (add-broadcast-state-to-link-environment
	   (new-broadcast-state types
				address
				(stamp-term-of-environment env)
				client-address
				client-stamp
				(term-to-description client-description))
	   cenv
	   blink)

	  ;; result
	  (isubscription-term nil
			      (resources-of-isubscribe-term sterm)
			      (ienvironment-address-term address)
			      (stamp-term-of-environment env)
			      (producer-broadcast-state types
							client-description
							(description-of-environment env)
							nil
							(unless (ivoid-term-p client-synchro)
							  client-synchro)))))))

(defvar *unsubscribe-store* nil)

;; current environment should be client
(defun subscribe-client (subscription &optional synchro)

  (let ((resources (tokens-of-itokens-term (resources-of-isubscription-term subscription)))
	(server-address (tags-of-ienvironment-address-term
			 (address-of-isubscription-term subscription)))
	)
	
    (let* ((env (current-environment))
	   (address (address-of-environment env))
	   (senv (find-bus-environment server-address)))

      (let ((sstamp (stamp-of-isubscription-term subscription)))

	;; add resources, fail if duplicate.
	(environment-add-resources env resources
				   (cdr (assoc synchro *unsubscribe-store* :test #'compare-terms-p)))

	;; update bus-link to server

	;; needed so that consumer can locate broadcast state by stamp on takedown
	;; initiated by consumer, ie configure-export of irevoke istart.
	;; maybe should part of config identity handshake.
	(setf (environment-base-stamp senv) sstamp)
    
	;;(setf -resources resources -env env -server-address server-address) (break "sc")
	(when (member 'transaction resources)
	  (environment-set-server-address env server-address))
	    
	;; update broadcast state
	(environment-broadcast-update
	 env
	 resources
	 sstamp 
	 (new-broadcast-state resources
			      server-address
			      sstamp
			      address
			      (stamp-term-of-environment env)
			      ;; should this really be client description
			      ;;   - broadcast state not very relevant to client side so probably not used.
			      (description-of-environment env))
	 t)
	      
	;; process broadcasts.
	(let ((bcasts (broadcasts-of-isubscription-term subscription)))
	  (delay-broadcast-compiles (term-to-stamp (stamp-of-ibroadcasts-term bcasts))
				    env
				    (do-broadcasts env bcasts))))))
  nil)

(defun unsubscribe-client (subscription)

  (let ((resources (tokens-of-itokens-term (resources-of-isubscription-term subscription)))
	(synchro (broadcasts-of-isubscription-term subscription))
	;;(server-address (tags-of-ienvironment-address-term
	;;		 (address-of-isubscription-term subscription)))
	)
	
    ;;(setf -synchro synchro) (break "-uc")
    (let ((env (current-environment)))
      (remove-broadcast-state-from-environment resources
					       (stamp-of-isubscription-term subscription)
					       (stamp-term-of-environment env)
					       env)

      ;; transaction subscription should be done by orbs not applications
      ;; so that application subscription happens in a transaction?
      (when (member 'transaction resources)
	(environment-set-server-address env nil))

      (let ((removed (environment-remove-resources env resources)))
	(unless (ivoid-term-p synchro)
	  (push (cons synchro (filter #'(lambda (x)
					  (not (eql (car x) 'transaction)))
				      removed))
		*unsubscribe-store*)))))

  nil)


(define-primitive |!interpret| () (command))

(defun eval-command (addr cmd &optional local-ok)
  (interpret-result (orb-eval 'one addr
			      (icommand-term (iinterpret-term cmd)
					     t ;; assume 
					     )
			      local-ok)))

(defun subscribe-client-initiate (resources server-address &optional synchro)

  (let* ((env (current-environment))
	 (address (address-of-environment env))
	 )

    (let ((subscription (eval-command server-address
				      (isubscribe-term nil
						       (itokens-term resources)
						       (ienvironment-address-term address)
						       (stamp-term-of-environment env)
						       (description-to-term
							(description-of-environment env))
						       (or synchro (ivoid-term))))))

      (subscribe-client subscription synchro))))

(defun subscribe-transactions (server-address)

  (let* ((env (current-environment))
	 (address (address-of-environment env))
	 )

    (let ((subscription (eval-command server-address
				      (isubscribe-term nil
						       (itokens-term '(transaction))
						       (ienvironment-address-term address)
						       (stamp-term-of-environment env)
						       (description-to-term
							(description-of-environment env))
						       (ivoid-term))
				      t)))

      (subscribe-client subscription))))

(defun unsubscribe-transactions (server-address)

  (let* ((env (current-environment))
	 (address (address-of-environment env))
	 )

    (let ((subscription (eval-command server-address
				      (isubscribe-term t
						       (itokens-term '(transaction))
						       (ienvironment-address-term address)
						       (stamp-term-of-environment env)
						       (description-to-term
							(description-of-environment env))
						       (ivoid-term))
				      t)))

      (unsubscribe-client subscription))))

(defun unsubscribe-client-initiate (resources server-address syncp)

  ;;(setf -resources resources -server-address server-address -syncp syncp) (break "uci")
  
  (let* ((env (current-environment))
	 (address (address-of-environment env))
	 ;;(senv (find-bus-environment server-address))
	 )

    (let ((subscription (eval-command server-address
				      (isubscribe-term t
						       (itokens-term resources)
						       (ienvironment-address-term address)
						       (stamp-term-of-environment env)
						       (description-to-term
							(description-of-environment env))
						       (ibool-term syncp)))))

      (unsubscribe-client subscription)

      (let ((sync (broadcasts-of-isubscription-term subscription)))
	(unless (ivoid-term-p sync)
	  sync)))))



(defun unsubscribe-server-initiate (resources client-address)

  (let* ((client-env (find-bus-environment client-address))
	 )
    (let ((subscription (unsubscribe-server
			 (isubscribe-term t
					  (itokens-term resources)
					  (stamp-term-of-environment client-env)
					  (ienvironment-address-term client-address)
					  (description-to-term
					   (description-of-environment client-env))
					  (ivoid-term)))))
      (eval-command client-address subscription))))


(defun producer-broadcast-state (types consumer-desc producer-desc &optional oids synchro)

  (ibroadcasts-auto-commit-term
   'state-init
   (current-transaction-stamp)
   (new-transaction-stamp)
   (let ((acc nil))
     (dolist (f (lookup-producer-state-snap-funcs
		 (purposes-of-description producer-desc)))
       (funcall f types consumer-desc
		#'(lambda (bt)
		    (push bt acc))
		oids
		synchro))

     (nreverse acc)))

    ;;(dms-tables-to-bound-terms types)
    )


;;;;
;;;;	eval
;;;;


(defun call-ml (term parse-p display-p)

  (cond
    ((iml-woargs-term-p term)
     (if parse-p
	 (vml-term (text-of-iml-term term) display-p)
	 (ml-term (text-of-iml-term term) display-p)))
    ((iml-term-p term)
     (if parse-p
	 (vml-term-apply (text-of-iml-term term)
			 (data-of-iml-term term)
			 display-p)
	 (ml-term-apply (text-of-iml-term term)
			(data-of-iml-term term)
			display-p)))
    (t (raise-error (error-message '(input) term)))))


(defun source-reduce-iml (term conditions)
  (if (or (iml-woargs-term-p term) (iml-term-p term))
      (maybe-instantiate-term
       term
       (operator-of-term term)
       (let ((bts (bound-terms-of-term term)))
	 (cons (maybe-instantiate-bound-term (car bts)
					     nil
					     (source-reduce (text-of-iml-term term)
							    conditions))
	       (cdr bts))))
      (raise-error (error-message '(source reduce ml not)))))

(defun ml-eval (term &optional suppress-display)
  (advance-sequence)
  
  ;; RLE TODO check for iml-term before accessing bits.

  ;; RLE PERF: should get type; then use that to determine if output should be produced.
  (let* ((parse-p (parse-p-of-iml-term term))
	 (result-p (result-p-of-iml-term term))
	 (display-p (and (not suppress-display) result-p 'not-term)))

    ;;(setf a term p parse-p c result-p d display-p)      (break)

    (mlet* (((value type output)
	     (call-ml term parse-p display-p)
	     (declare (ignore type))))
	 
	   ;;(setf a result-p b value c output d term -parse-p parse-p -display-p display-p) (break "me")
	   (if result-p
	       (if (term-p value)
		   (ivalue-term value
				(when (messages-p)
				  (message-to-term (flush-message
						    (append (current-environment-path)
							    '(ml eval))))))
		   (iprint-term (message-to-term output)
				;;(setf a output) (break)
				(when (messages-p)
				  (message-to-term (flush-message
						    (append (current-environment-path)
							    '(ml eval)))))))
		 
	       (progn
		 (when (messages-p)
		   (message-emit-asynch (message-to-term
					 (flush-message (append (current-environment-path)
								'(eval))))))
		 (iack-term))))))



(define-primitive |!LISP| ((bool . result-p)) (expr))

(defun lisp-eval (term)
  (advance-sequence)
  
  ;;(setf -term term) (break "le")

  (let* ((eof (cons 'eof 'eof))
	 (sexpr (with-input-from-string (s (text-to-string
					    (term-to-text
					     (expr-of-ilisp-term
					      (syntax-reduce 'lisp '(lispprimitive) term)))))
		  (format t "~%~a~%" s)
		  (read s nil eof))))

    (when (eq eof sexpr)
      (raise-error (error-message '(eval lisp eof)  term)))
	
    (let ((res (eval sexpr)))
      
      ;;(setf a res d term) (break)
      (if (result-p-of-ilisp-term term)
	  (if (term-p res)
	      (ivalue-term res
			   (when (messages-p)
			     (message-to-term (flush-message
					       (append (current-environment-path)
						       '(eval lisp))))))
	      (iprint-term (itext-term (princ-to-string res))
			   (when (messages-p)
			     (message-to-term (flush-message
					       (append (current-environment-path)
						       '(eval lisp)))))))
	  (progn
	    (when (messages-p)
	      (message-emit-asynch (message-to-term
				    (flush-message (append (current-environment-path)
							   '(eval))))))
	    (iack-term))))))


(defvar *eval-print-p* nil)

(defun eval-print-toggle () (setf *eval-print-p* (not *eval-print-p*)))








(defvar *ml-get-term* nil)

;; 
(defun orb-get-term (oid)
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-term oid not))))
  ;;(setf -a oid) (break)

  (funmlcall (or *ml-get-term*
		 (ml-text "\\oid. (get_term oid) ? default_placeholder_term"))
	     oid))

(defun orb-get-inf-term (oid)
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-inf-term oid not))))

  (funmlcall (or *ml-get-term*
		 (ml-text "\\oid. (get_inf_term oid) ? (proof-editor-template)"))
	     oid))

(defun view-show (term &optional name)
  (if name
      (funmlcall (ml-text "view_showd") name term)
      (funmlcall (ml-text "view_show") term)))



;;;;	
;;;;	
;;;;	asynch-eval
;;;;	
;;;;	

(defstruct (asynch-channel (:include soft-channel))
  recv-queue
  pend-queue
  )

(defstruct asynch-call
  environment
  hook
  inter-orb-p
  )

(defun new-notify (c inter-orb-p)
  (make-asynch-call :environment (current-environment)
		    :hook c
		    :inter-orb-p inter-orb-p))

(defstruct (asynch-req (:include asynch-call))
  term
  )

(defun term-of-asynch-req (r) (asynch-req-term r))
(defun hook-of-asynch-call (r) (asynch-call-hook r))
(defun environment-of-asynch-call (r) (asynch-call-environment r))
(defun inter-orb-asynch-call-p (r) (asynch-call-environment r))

(defun new-asynch-req (term c &optional inter-orb-p env)
  ;;(break "nar")
  (make-asynch-req :term term
		   :hook c
		   :environment (or env (current-environment))
		   :inter-orb-p inter-orb-p))


(defun find-asynch-req (ch rsp)
  (let ((a (assoc (sequence-of-irsp-term rsp)
		  (asynch-channel-pend-queue ch))))
    (when a
      (setf (asynch-channel-pend-queue ch) (delete a (asynch-channel-pend-queue ch))))
    ;;(setf -b a) (break "far")
    a))

;; imessage queued to be evaluated. ie recv from process?
(defun asynch-channel-recv (channel)
  
  (show-telemetry "~% asynch-channel-recv pend ~a recv ~a "
		  (length (asynch-channel-pend-queue channel))
		  (length (asynch-channel-recv-queue channel)))
  
 ;;(setf -c channel) (break)
  (let* ((areq (pop (asynch-channel-recv-queue channel)))
	 (ireq (term-of-asynch-req areq)))

    ;; kludge no pend if asynch-command as no rsp to asynch command.
    ;; see bus-asynch-command-eval 
    (unless (iasynch-command-term-p (expression-of-ireq-term (term-of-asynch-req areq)))
      (push (cons (sequence-of-ireq-term ireq)
		  areq)
	    (asynch-channel-pend-queue channel)))
    ;;(setf -channel channel -ireq ireq) (break "acr")
    ireq
    ))

;;(< (length (asynch-channel-pend-queue channel))
;;	  (length (mapcan #'(lambda (bl)
;;			      (filter #'(lambda (ie) (member '|ref| (address-of-environment ie))) (bus-link-imported-environments bl)))
;;			  *bus-links*)))
(defun asynch-channel-listen (channel)
  ;; do one at a time. Ie wait for pend to be nil before returning t?

  (when (and (null (asynch-channel-pend-queue channel))
	     (not (null (asynch-channel-recv-queue channel)))
	     ;;(null *transactions-active*)
	     )
    (show-telemetry "~% asynch-channel-listen pending ~a recvd ~a"
		    (length (asynch-channel-pend-queue channel))
		    (length (asynch-channel-recv-queue channel)))
    t))

(defun asynch-call (c term)
  (let ((channel (find-asynch-channel)))
    (show-telemetry "~%asynch-call pend ~a recv ~a "
		    (length (asynch-channel-pend-queue channel))
		    (length (asynch-channel-recv-queue channel)))
    ;;(when (not (zerop (length (asynch-channel-pend-queue channel))))  (break "ac"))
    )

  (let ((h (hook-of-asynch-call c)))
    (when h
      (with-environment-actual (environment-of-asynch-call c)
	(with-appropriate-transaction ((inter-orb-asynch-call-p c) nil) ;; ???
	  (without-dependencies
	   ;;(setf -c c -term term) (break "acs")
	   (funcall h term)))))))
    

;; send : ie sending to process
(defun asynch-channel-send (channel term)
  (show-telemetry "~% asynch-channel-send ~a pending ~a recvd ~a "
		  (id-of-term term)
		  (length (asynch-channel-pend-queue channel))
		  (length (asynch-channel-recv-queue channel)))

  (unless (and (irsp-term-p term) (iack-term-p (result-of-iresult-term term)))
    (message-emit (degenerate-term-message term)))

  ;;(setf a term) (break "tcs")
  (when (and (ireq-aux-term-p term)
	     (iconfigure-term-p (expression-of-ireq-term term))
	     (irequest-term-p (command-of-iconfigure-term (expression-of-ireq-term term)))
	     (inoack-term-p (info-of-irequest-term (command-of-iconfigure-term (expression-of-ireq-term term)))))
    (setf -term term) (break "asynch-channel-send")
    )
  
  (when (messages-p)
    (terpri)
    (mapcar #'print-message (messages-flush)))

  ;;(setf -c channel -a term) (break "acs2")
  (with-handle-error (('(asynch channel send)) nil)
    (when (irsp-term-p term)
      (let ((req (cdr (find-asynch-req channel term)))) ; deletes from pend queue.
	(when (null req) (setf -term term ) (break "fa"))
	(when req
	  ;;(setf -req req) (break "acs33")
	  (with-environment-actual (environment-of-asynch-call req)
	    (let ((desc-posure (asynch-call req (result-of-irsp-term term)))) ; call completion.
	      ;; can not queue new request itself since it wants itself as hook and does not have a handle on self.
	      ;; thus we allow for asynch call to return a new expression to be evaluated.
	      (when desc-posure
		;; need environment for description to be resolved to related environment.
		(orb-queue-asynch-request (car desc-posure) (cdr desc-posure)
					  (hook-of-asynch-call req)
					  )))))))

    nil
    ))


(defun asynch-channel-queue (channel term)
  (format t "~% asynch-channel-queue ~%")
  ;;(break "acq")
  (show-telemetry "~% asynch-channel-queue pend ~a recv ~a "
		  (length (asynch-channel-pend-queue channel))
		  (length (asynch-channel-recv-queue channel)))
  (unless (zerop (length (asynch-channel-recv-queue channel)))
    (format t "~%~%Warning AsynchChannelQueue has length ~a~%~%" (length (asynch-channel-recv-queue channel))))
  (setf (asynch-channel-recv-queue channel) (append (asynch-channel-recv-queue channel) (list term))))

(defun new-asynch-link ()
  (new-soft-link
   (make-asynch-channel :kind 'asynch)

   :send #'asynch-channel-send
   :recv #'asynch-channel-recv
   :listen #'asynch-channel-listen
   ))

(defun add-asynch-link ()
  (let ((a (find-asynch-link t)))
    (when a (remove-bus-link a))
    (let ((blink (new-bus-link (new-asynch-link))))
      (add-bus-link blink)
      blink)))

(defun asynch-blink-p (blink)
   (let ((ch (car (channels-of-link (link-of-bus-link blink)))))
     (and ch
	  (asynch-channel-p ch))))

(defun find-asynch-link (&optional dontmake)
  (or (find-first #'(lambda (x) (when (asynch-blink-p x) x)) *bus-links*)
      (unless dontmake
	(add-asynch-link))))

(defun find-asynch-channel ()
  (find-first #'(lambda (ch)
		  (when (asynch-channel-p ch)
		    ch))
	      (channels-of-link (link-of-bus-link (find-asynch-link)))))

					
(defun orb-queue-asynch-request-aux (tags posure
				     &optional completion inter-orb-p completion-inter-orb-p transprops)
  ;;(when transprops (break "oqara"))
  (asynch-channel-queue
   (find-asynch-channel)
   (new-asynch-req 
    (new-ireq-term (current-sequence) nil
		   'asynch
		   tags
		   (let ((et (iexpression-term (iml-term nil t (car posure) (cdr posure))
					       inter-orb-p)))
		     (if transprops
			 (ieval-property-term (iproperty-term 'transaction transprops)
					      et)
			 et)))
    completion
    completion-inter-orb-p)))



(defun orb-queue-asynch-request (description posure
				 &optional completion (properties '(main)) (inter-orb-p t) tprops)
  (orb-queue-asynch-request-aux (mapcar #'token-parameter
					(orb-choose-connected-environment description properties))
				posure completion
				inter-orb-p
				(properties-to-term tprops)))

(defun orb-queue-local-asynch-request (posure &optional completion (inter-orb-p t) addr)
  (orb-queue-asynch-request-aux
   (mapcar #'token-parameter (or addr (address-of-environment (current-environment))))
   posure completion
   inter-orb-p)
  )

;;;;	
;;;;	!notify(<description>; <cookie{stamp}>)
;;;;	
;;;;	!asynch-notify(<cookie>; <rsp>)
;;;;	
;;;;	!asynch-queue{inter-orb:b}(<posure>; <notify>)
;;;;	
;;;;	
;;;;	
;;;;	recv-orb-asynch-notify	(<term{!asynch-notify}>)
;;;;	send-orb-asynch-notify	(<blink> <term{cookie}> <term{rsp}>
;;;;	recv-orb-asynch-queue	(<tok list> <term{!asynch-queue}>)
;;;;	send-orb-asynch-queue	(<bool{inter-orb}> <tok list{addr}> <posure>
;;;;				  &optional <term{cookie}>)
;;;;	
(define-primitive |!asynch_queue| ((bool . inter-orb-p)) (posure cookie))
(define-primitive |!asynch_notify| ((bool . twice)) (cookie rsp))

(define-primitive |!asynch_command| () (command))

(defun iasynch-queue-term-p-aux (iaq)
  ;;(setf -iaq iaq) (break "iatpa")
  (and (equal (id-of-term iaq) *iasynch-queue*)
       (let ((p (parameters-of-term iaq)))
	 (and p (null (cdr p))
	      (bool-parameter-p (car p))))
       (let ((bts (bound-terms-of-term iaq)))
	 (let ((l (length bts)))
	   (cond
	     ((eql l 2)
	      (and (null (bindings-of-bound-term (car bts)))
		   (null (bindings-of-bound-term (cadr bts)))))
	     ((eql l 3)
	      (and (null (bindings-of-bound-term (car bts)))
		   (null (bindings-of-bound-term (cadr bts)))
		   (null (bindings-of-bound-term (caddr bts)))))
	     (t nil))))))

(defun properties-of-iasynch-queue-term (iaq)
  (let ((bt (caddr (bound-terms-of-term iaq))))
    (when bt (term-of-bound-term bt))))

(defun iasynch-queue-term-wprops (iop pos cook props)
  ;;(setf -props props) (break)
  (instantiate-term (instantiate-operator *iasynch-queue* (list (bool-parameter iop)))
		    (list (instantiate-bound-term pos)
			  (instantiate-bound-term cook)
			  (instantiate-bound-term props))))


(define-primitive |!posure| () (f args))

(defun term-to-posure (p)
  (cons (f-of-iposure-term p)
	(map-ilist-to-list (args-of-iposure-term p) (icons-op))))

(defun posure-to-term (p)
  (iposure-term (car p)
		(map-list-to-ilist (cdr p) (inil-term))))

;; notify or queue notify on asynch queue.
;; ie eval in command transaction ? that or avoid with-appropriate-transaction in environment-eval

;; ack should be sent back prior to eval of notify.
;; need to catch notify errors.
(defun recv-orb-asynch-notify (iasynch-notify)

  ;; tis possible that notify has come back prior to add-notify thus need to make it wait
  (if (can-notify-p (cookie-of-iasynch-notify-term iasynch-notify))
      (with-handle-error-and-message (()
				      #'(lambda (m)
					  (apply #'ifail-term
						 (message-to-term
						  (tag-message '(iasynch-notify) m))
						 (mapcar #'message-to-term (messages-flush)))))
	(notify (cookie-of-iasynch-notify-term iasynch-notify)
		(rsp-of-iasynch-notify-term iasynch-notify)))

      (notify-wait-list-push (cookie-of-iasynch-notify-term iasynch-notify)
			     (new-asynch-req 
			      (new-ireq-term (current-sequence) nil
					     'asynch
					     nil
					     (iasynch-command-term iasynch-notify))
			      nil)))
  nil)

(defun send-orb-asynch-notify (blink cookie rsp)
  ;;(break "soan")
  (bus-send-asynch-command-iob
   (new-callback-iob (tid)
		     blink
		     (iasynch-command-term (iasynch-notify-term nil cookie rsp)))))

(define-primitive |!asynch_posure| ((bool . inter-orb-p))
  (address posure completion))


(defun asynch-iterator (env c term) 

  (with-environment-actual env
    (with-appropriate-transaction (t nil)
      (without-dependencies

       ;;(setf -c channel -a term) (break "acs")
       (let ((desc-posure (funcall c term)))

	 (when desc-posure
	   (orb-queue-asynch-request (car desc-posure) (cdr desc-posure) c)))))) )


(defvar *notify-alist* nil)
(defvar *notify-wait-alist* nil)

(defvar *asynch-history* nil)
(defvar *debug-asynch* nil)

;; last resort.
(defun reset-asynch ()
  (setf *notify-alist* nil)
  (setf *notify-wait-alist* nil)
  ;; TODO if lib need to run in scope of lib env.
  (with-ignore (ml-text "bot_cookies := nil"))
  (add-asynch-link))

(defun notify-alist-push (s n)
  (show-telemetry "~%notify alist push ~a ~a ~%" (mapcar #'cadar *notify-alist*) (cadr s))
  (let ((a (cons s n)))
    (when *debug-asynch* (push (cons 'notify-push a) *asynch-history*))

    (push a *notify-alist*)))
  
(defun can-notify-p (cookiet)
  (member (term-to-stamp cookiet) *notify-alist*
	  :test #'(lambda (a b) (equal-stamps-p a b))
	  :key #'car))

(defun notify-alist-delete (s)
  (let ((n (assoc s *notify-alist*
		  :test #'(lambda (a b) (equal-stamps-p a b)))))

    (when *debug-asynch*
      (push (cons `notify-delete s) *asynch-history*))
    (show-telemetry "~%notify alist delete[~a ~a] ~a ~%" (if n "T" "F")
		    (mapcar #'cadar *notify-alist*)
		    (cadr s))

    (when n
      (setf *notify-alist* (delete n *notify-alist*)))
    n))

(defun notify-wait-list-push (cookiet cont)
  (let* ((s (term-to-stamp cookiet))
	 (a (cons s cont)))
    (show-telemetry "~%notify wait list push ~a ~a ~%" (mapcar #'cadar *notify-wait-alist*) (cadr s))
    (when *debug-asynch*
      (push (cons `notify-wait-list s) *asynch-history*))

    ;;(break "wln")
    (setf *notify-wait-alist*
	  (nconc *notify-wait-alist* (list a)))))

(defun notify-wait-list-delete (s)
  (let ((n (cdr (assoc s *notify-wait-alist*
		       :test #'(lambda (a b) (equal-stamps-p a b))))))

    (show-telemetry "~%notify wait list delete ~a ~a ~%" (mapcar #'cadar *notify-wait-alist*) (cadr s))
    (when *debug-asynch*
      (push (cons `notify-wait-list-delete s) *asynch-history*))

    (setf *notify-wait-alist*
	  (delete s *notify-wait-alist*
		  :test #'(lambda (a b) (equal-stamps-p a b))
		  :key #'car
		  :count 1))
    n))


(defun add-notify-aux (s f inter-orb-p)

  (notify-alist-push s (new-notify f inter-orb-p))
  
  ;; in case result came inprior to notify being added.
  (let ((n (notify-wait-list-delete s)))
    (when n
      ;;(setf -n n) (break "ana")
      (asynch-channel-queue (find-asynch-channel) n)) ))

(defun add-notify (f &optional (inter-orb-p t))
  (let ((s (new-transaction-stamp)))
    (add-notify-aux s f inter-orb-p)
    (stamp-to-term s)))

(defun add-notify-wcookie (f cookie &optional (inter-orb-p t))
  ;;(break "anwc")
  (add-notify-aux cookie f inter-orb-p))

(defun notify (cookie arg)
  (let ((notify (notify-alist-delete (term-to-stamp cookie))))
    (if notify
	(asynch-call (cdr notify) arg) 
	(progn
	  ;;(setf -cookie cookie) (break "n")
	  (message-emit (warn-message '(notify cookie not) cookie))
	  ))))

;; cookie, or address or void
;; if address must remember link (no can find link from address).
;; seems it would be useful to allow a handoff (a continuation), ie a rsp
;; which is another asynch call and a completion which continues
;;  (inter-orb-pdesc props posure) comp
;;   send-orb-asynch-queue (desc props posure #'(lambda (r) recv-orb-asynch-queue tags iasynch-queue blink))
(defun send-orb-asynch-queue (inter-orb-p addr posure &optional (cookie (ivoid-term)) props)
  ;;(when props (setf -props props) (break "soaq"))
  (bus-send-asynch-command-iob
   (new-out-req-iob 'asynch
		    addr
		    (iasynch-command-term
		     (if props
			 (iasynch-queue-term-wprops inter-orb-p
						    (posure-to-term posure)
						    cookie
						    (properties-to-term props))
			 (iasynch-queue-term inter-orb-p (posure-to-term posure) cookie)))))
  nil)


(defun recv-orb-asynch-queue (tags iasynch-queue &optional blink)
  (let ((blink (or blink (tid-blink))))
    (orb-queue-asynch-request-aux 
     (mapcar #'token-parameter tags)
     (term-to-posure (posure-of-iasynch-queue-term iasynch-queue))
     (let ((cookie (cookie-of-iasynch-queue-term iasynch-queue)))
       ;;(setf -cookie cookie) (break "roaq")
       (unless (ivoid-term-p cookie)
	 #'(lambda (res)
             ;;(setf -blink blink -rsp res) (break "roaq2")
	     (if (and (ivalue-term-p res)
		      (iasynch-posure-term-p (result-of-iresult-term res)))
		 (let ((rsp (result-of-iresult-term res)))
		   (send-orb-asynch-queue
		    (inter-orb-p-of-iasynch-posure-term rsp)
		    (tokens-of-itokens-term (address-of-iasynch-posure-term rsp))
		    (term-to-posure (posure-of-iasynch-posure-term rsp))
		    (add-notify
		     #'(lambda (r)
			 (recv-orb-asynch-queue
			  tags
			  (iasynch-queue-term nil
					      (progn
						;;(setf -r r -rsp rsp) (break "hoha")
						(funmlcall (ml-text "term_ap_update_asynch_posure_term")
							   (completion-of-iasynch-posure-term rsp)
							   r))
					      cookie)
			  blink))
		     nil)))
		 (if (ienvironment-address-term-p cookie)
		     (send-orb-asynch-queue nil
					    (tags-of-ienvironment-address-term cookie)
					    (term-to-posure res))
		     (send-orb-asynch-notify blink cookie res)))
	     nil)))
     (inter-orb-p-of-iasynch-queue-term iasynch-queue)
     nil
     (properties-of-iasynch-queue-term iasynch-queue)
     ))
  nil)


;; simply seeing if bot touched is not robust as it may be touched for
;; some other reason and the bot's transaction may still be active.
;; could combine with some status bit set by response so that even if wrong transaction
;; end we know bot is good to go. but then how does failure work. need response good or bad
;; to cause some processing. where does response go.
(defun bot-transaction-end-hook (thl)
  (declare (ignore thl))
  )


(defun orb-send-one-term (blink term)
  (link-send (link-of-bus-link blink)
	     term))

(defun orb-receive-one-term (blink &optional blockp)
  (link-recv (link-of-bus-link blink) blockp))

(defun orb-stateless-eval (blink addr term args)

  ;; send query
  (orb-send-one-term
	      blink
	      (new-ireq-term (progn (advance-sequence) (current-sequence))
			     nil 'stateless
			     (mapcar #'token-parameter addr)
			     (iexpression-term
			      (iml-term nil t term args))))

  ;; receive result
  (interpret-result (result-of-irsp-term  (orb-receive-one-term blink t))))


(defun orb-remote-lib-eval (term args)
  (let ((lib-blinks
	 (mapcan #'(lambda (blink)
		     (let ((a (mapcan #'(lambda (e)
					  (when (and
						 (member '|lib|
							 (address-of-environment e))
						 (not
						  (member 'JOURNAL
							  (address-of-environment e))))
					    (list (address-of-environment e))))
				      (imported-environments-of-bus-link blink))))
		       (when a
			 (list (cons (car a) blink)))))
		 *bus-links*)))

    (unless (and lib-blinks (null (cdr lib-blinks)))
      (raise-error (error-message '(orb-remote-lib-eval) (list term))))

    
    (or (orb-stateless-eval (cdar lib-blinks) (caar lib-blinks) term args)
	(raise-error (error-message '(orb-remote-lib-eval result not) (list term))))))



;;;;	
;;;;	reset :
;;;;	  - abort all transactions
;;;;	  - reply to pending requests with kill errors.
;;;;	  - throw to reset.
;;;;	
;;;;	top catches reset and restarts toploop.
;;;;	
;;;;	need someway to detect no responding bus-links so
;;;;	as to avoid getting hung up writing to link.
;;;;	
;;;;	that's a general purpose problem separate from restart.
;;;;	  - add primitive (ie at level of !req !rsp) acks to protocol
;;;;	    !ack(<seq>:n;<req-or-rsp>:b)
;;;;          * have special write which builds string directly then writes.
;;;;		  (format-string "!ack{~a:n,~b:b}" seq b)
;;;;		skip args and write constant.
;;;;	  - waiting-for-ack-p  - maybe save time when set (or seq transaction-id)
;;;;	    turn on at send, turn off when ack received
;;;;	    queue messages if on.
;;;;	
;;;;	at reset if waiting for long time then kill-link.
;;;;	

