
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;
;;;;	RLE TODO !!! journal should close in response to a revoke of broadcasts.
;;;;	

;;;;
;;;; -docs- (mod lib orb) 
;;;;
;;;;	
;;;;	Journal :
;;;;	  - logs bind/unbinds.
;;;;	 
;;;;	restore :
;;;;	  - restores bindings.
;;;;	
;;;;	*** Vaporware alert ***
;;;;	
;;;;	Library Journal : Object DataBase
;;;;	  Persistence and History.
;;;;
;;;;	  - Listens for lib broadasts and backs up on disks all commited actions.
;;;; 	*** Vaporware alert ***
;;;;	  - allows for unwind and playback
;;;;	  - allows for unwind of single object to access previous versions.
;;;;	  - can rebuild lib tables on restart.
;;;;	      * lib order
;;;;	      * activate order : might be hard?
;;;;
;;;;	**vapor
;;;;	Objects on Disk:
;;;;	  - two pieces: source ("o<n>") and substance ("u<n>")
;;;;	      * substance : data needed to activate object.
;;;;		  - non-substantative change does not alter substance.
;;;;		    in fact that is the defining criteria of substantive change.
;;;;		  - descriptions should be substance as needed to broadcast activation.
;;;;		  - dependencies should be substance.
;;;;	      * source : annotations + data needed to produce substance.
;;;;	  - desire multiple terms in a file ordered in increasing importance so
;;;;	    that partial reads may be done? (not to save io necessarily but rather
;;;;	    ephemeral space.) So something like stamps could be up front and then substance
;;;;	    and then source.
;;;;	  - desire chain of stamps assoc'd with oid be producible.
;;;;	 
;;;;	
;;;;	
;;;;	Broadcasts journal'd depend on table-types arg of journal-create.
;;;;	Expect library broadcasts.
;;;;	
;;;;	broadcasts affecting lib state:
;;;;
;;;;	'library insert		: oid def
;;;;	'library delete		: oid
;;;;	'library allow		: oid
;;;;	'library disallow	: oid
;;;;	'library activate	: oid
;;;;	'library deactivate	: oid
;;;;	
;;;;	
;;;;	
;;;;	
;;;; -doce- (mod lib)




;;;;
;;;;	Journal channel
;;;;

(defstruct (journal-channel (:include soft-channel))
  ilog			;; could subclass soft-link also and
  log			;;  ...
  stamp			;;   ... include these fields there instead.
  server-address
  recv-queue
  broadcast-stack	;; pop for undo
  broadcast-queue	;; pop for commit

  ;; at collection write in broadcast: close will dump state as collection of broadcasts.
  ;; open will init log with same collection. onlybcast is a method of sharing that
  ;; collection on disk to save disk and io (it can be quite a large term).
  ;; close : t -> write collection as persistent term-data 
  ;; open  : !data-persist # icollection -> write data-persist if (compare-term-p icollection new collection)
  onlybcast  )

(defun ilog-of-journal-channel (ch) (journal-channel-ilog ch))
(defun log-of-journal-channel (ch) (journal-channel-log ch))
(defun stamp-of-journal-channel (ch) (journal-channel-stamp ch))
(defun recv-queue-of-journal-channel (ch) (journal-channel-recv-queue ch))
(defun server-address-of-journal-channel (ch) (journal-channel-server-address ch))
(defun onlybcast-of-journal-channel (ch) (journal-channel-onlybcast ch))

;; queue until completion and only write if commited.
;; reuseable on bus send queue (or rely on send queue to do it).
;; undo pops stack
;; commit pops queue (reversed stack)
;; nreverse stack <-> queue as needed.

(defun broadcast-stack-of-journal-channel (ch)
  (let ((q (journal-channel-broadcast-queue ch))
	(s (journal-channel-broadcast-stack ch)))
    (cond
      ((and q s) (break "uh-oh"))
      (q (set-journal-channel-broadcast-stack ch (nreverse q)))
      (s s)
      (t nil))))

(defun broadcast-queue-of-journal-channel (ch)
  (let ((q (journal-channel-broadcast-queue ch))
	(s (journal-channel-broadcast-stack ch)))
    (cond
      ((and q s) (break "uh-oh"))
      (q q)
      (s (set-journal-channel-broadcast-queue ch (nreverse s)))
      (t nil))))

(defun set-journal-channel-broadcast-stack (ch s)
  (setf (journal-channel-broadcast-queue ch) nil
	(journal-channel-broadcast-stack ch) s)
  s)

(defun set-journal-channel-broadcast-queue (ch q)
  (setf (journal-channel-broadcast-stack ch) nil
	(journal-channel-broadcast-queue ch) q)
  q)



(defun journal-queue-broadcast (ch term)
  (set-journal-channel-broadcast-stack
   ch
   (cons term (broadcast-stack-of-journal-channel ch))))


(defun journal-commit (ch commit)
  (let ((seq (sequence-of-ibroadcast-term (broadcast-of-ipassport-term commit)))
	(result nil))
    (set-journal-channel-broadcast-queue
     ch
     (delete-if #'(lambda (bterm)
		    (let ((term (term-of-bound-term bterm)))
		      ;;(setf a term)
		      (when (= seq (sequence-of-ibroadcast-term
				    (broadcast-of-ipassport-term term)))
			(setf result bterm)
			t)))
		(broadcast-queue-of-journal-channel ch)  
		:count 1))
    result))


(defun journal-undo (ch undo)
  (let ((seq (sequence-of-ibroadcast-term (broadcast-of-ipassport-term undo))))
    (set-journal-channel-broadcast-stack
     ch
     (delete-if #'(lambda (bterm)
		    (let ((term (term-of-bound-term bterm)))
		      (= seq (sequence-of-ibroadcast-term (broadcast-of-ipassport-term term)))))
		(broadcast-stack-of-journal-channel ch)  
		:count 1)))
  (values))




;;;;
;;;;	broadcast journal
;;;;
;;;;
;;;;	!journal{lib-list}





(defun journal-send (blink term)
  (advance-sequence)
  (let ((ch (channel-of-soft-link (link-of-bus-link blink))))
    (setf (journal-channel-recv-queue ch)
	  (append (journal-channel-recv-queue ch)
		  (list (new-ireq-term (current-sequence) nil
				       'library
				       (mapcar #'(lambda (tag) (instantiate-parameter-r tag *token-type*))
					       (server-address-of-journal-channel ch))
				       term))))))


(defun journal-write (ch term)
  (log-write-record (log-of-journal-channel ch)
		    term))


(define-primitive |!collection| () ())

(defun icollection-term-weak-p (term) (eql (id-of-term term) *icollection*))


;;;;	Expect one of the following :
;;;;
;;;;		!definition_insert{<seq>:n}(<definition>)
;;;;		!definition_disallow{<seq>:n, <oid>:o}()
;;;;		!definition_activate{<seq>:n, <oid>:o}()
;;;;	
;;;;	for same oid : insert < disallow < activate
;;;;
;;;;	
;;;;	<entry>	: <cell> list
;;;;
;;;;	<cell>	: (<passport> . <bool>)
;;;;
;;;;
;;;;


(defun ccp-oid-of-passport (pass)
  (let ((bcast (broadcast-of-ipassport-term pass)))
    (cond
      ((idefinition-insert-term-p bcast)
       (oid-of-idependency-term
	(dependency-of-idefinition-term
	 (definition-of-idefinition-insert-term bcast))))
      ((idefinition-disallow-collection-term-p bcast)
       (oid-of-idefinition-disallow-collection-term bcast))
      ((idefinition-activate-term-p bcast)
       (oid-of-idefinition-activate-term bcast))
      ;; !definition-replace if lib consumes object attributes may be necessary.
      (t (raise-error (error-message '(compare collections broadcast unexpectd) bcast))))))


(defun compare-broadcasts-p (bcast-a bcast-b)
  (and (eql (id-of-term bcast-a) (id-of-term bcast-b))
       (apply-predicate-to-list-pair (cdr (parameters-of-term bcast-a))
				     (cdr (parameters-of-term bcast-b))
				     #'equal-parameters-p)
       (apply-predicate-to-list-pair (bound-terms-of-term bcast-a)
				     (bound-terms-of-term bcast-b)
				     #'(lambda (bt-a bt-b)
					 (compare-terms-p
					  (term-of-bound-term bt-a)
					  (term-of-bound-term bt-b))))))


(defun break-ccp () (break))

;; ignore !passport stamp and broadcast seq
(defun compare-collections-p (cterm-a cterm-b)
  ;;(setf a cterm-a b cterm-b) (break "ccp")

  (with-handle-error (('(compare-collections)) (break) nil)

    ;; place one collection in oid hash table.
    (let ((table (make-hash-table :test #'equal)))
      (dolist (bt (bound-terms-of-term cterm-a))
	(let ((pass-a (term-of-bound-term bt)))
	  (if (ienvironment-state-term-p pass-a)
	      (push (cons pass-a nil) (gethash 'state table))
	      (push (cons pass-a nil) (gethash (stamp-of-oid (ccp-oid-of-passport pass-a)) table)))))

    
      ;; set cdr cell to t if found comparable.
      ;; stops if no cell found of pass for same oid and broadcast op not comparable.
      (and (forall-p #'(lambda (bpass-b)
			 (let ((pass-b (term-of-bound-term bpass-b)))
			   
			   (if (ienvironment-state-term-p pass-b)
			       (mapc #'(lambda (cell)
					 (when (compare-terms-p (car cell) pass-b)
					   (setf (cdr cell) t)))
				     (gethash 'state table))

			       (let ((bcast (broadcast-of-ipassport-term pass-b)))
				 (let ((cell (assoc (id-of-term bcast) 
						    (gethash (stamp-of-oid (ccp-oid-of-passport pass-b)) table)
						    :key #'(lambda (pass-a)
							     (id-of-term (broadcast-of-ipassport-term pass-a)))
						    )))
				   ;;(unless cell (setf d cell e pass-b f table) (break "cell"))
				   (when cell
				     (setf (cdr cell)
					   (and (compare-terms-p (description-of-ipassport-term (car cell))
								 (description-of-ipassport-term pass-b))
						(compare-broadcasts-p (broadcast-of-ipassport-term (car cell))
								      (broadcast-of-ipassport-term pass-b))
						t))))))))
		     (bound-terms-of-term cterm-b))

	   ;; map over hash table check if all hashed passports have been checked.
	   (let ((allp t))
	     (maphash #'(lambda (oid entry)
			  (declare (ignore oid))
			  (unless (forall-p #'cdr entry)
			    (setf allp nil)))
		      table)
	     allp)
	   ))))
  

;;;;	
;;;;	desire a method of cheaply dumping messages to journal
;;;;	  - things that affect state of environment like definition logs.
;;;;	  - not necessarily part of a transaction.
;;;;	

(defun journal-state-out (term)
  (dolist (j (journals-of-environment (current-environment)))
    (let* ((blink (blink-of-journal j))
	   (l (link-of-bus-link blink)))
      
      ;;(setf -term term -l l) (break "jso")
      (journal-write (channel-of-soft-link l) term))))


(defun journal-broadcast (ch bterms auto-commit)
  ;;(break "jb")

  (let ((collection nil)
	(onlybcast-persist nil))

    (labels
	((collect (bterm) (push bterm collection))     

	 (flush ()

	   ;;(setf a collection)
	 
	   (cond
	     ((null collection) nil)
	   
	     ;; if singleton then do not wrap
	     ((null (cdr collection))
	      (journal-write ch (term-of-bound-term (car collection))))

	     ;; if begin/end with no intervening terms then no-op.
	     ((and (not (ienvironment-state-term (cadr collection)))
		   (not (ienvironment-state-term (car collection)))
		   (ibegin-term-p (broadcast-of-ipassport-term (term-of-bound-term (cadr collection))))
		   (iend-term-p (broadcast-of-ipassport-term (term-of-bound-term (car collection)))))
	      nil)
	 
	     (t
	      ;; if writing collection of start for ephemeral then may want data-persist.
	      ;; if writing collection of start after reading data persist then want data persist.
	      (let ((collection-term (instantiate-term (icollection-op)
						       (nreverse collection)))
		    (obcast (onlybcast-of-journal-channel ch)))

		(cond
		  ((null obcast)
		   (journal-write ch collection-term))

		  ((eql t obcast)
		   ;;(format t "jbt writing persist-data term")
		   ;; when making checkpoint desire the data-persist be passed forward
		   ;; to next journal.
		   (journal-write ch (setf onlybcast-persist
					   (persist-data (term-data collection-term) nil nil))))

		  
		  ;; checkpoint knows that persist term is good.
		  ((and (not (consp obcast)) (idata-persist-term-p obcast))
		   (journal-write ch obcast))

		  ;; a weaker form of equality may be ok, ie order may not be important.
		  ;; passport will have diff stamps?
		  ((compare-collections-p (car obcast) collection-term)
		   ;;(setf a collection-term b (car obcast)) (break "jbc")
		   (journal-write ch (cdr obcast)))

		  (t
		   (when obcast
		     ;;(setf a collection-term b obcast) (break "jbd")
		     )
		   (journal-write ch collection-term)))

		(when obcast
		  (setf (journal-channel-onlybcast ch) nil)))))
	      
	   (setf collection nil)))

      (dolist (bt bterms)
	(let* ((passport (term-of-bound-term bt))
	       (bcast (broadcast-of-ipassport-term passport)))
	  ;; PERF : collect writes and write once.
	  (cond
	    (auto-commit
	     (collect bt))

	    ((icommit-term-p bcast)
	     (collect (journal-commit ch passport)))

	    ((iundo-term-p bcast)
	     (journal-undo ch passport))

	    ((ibegin-term-p bcast) (collect bt))

	    ((iend-term-p bcast) (collect bt) (flush))

	    ((icheckpoint-term-p bcast)
	     (flush)
	     (journal-write ch passport)
	     (db-inform-checkpoint (stamp-of-journal-channel ch)
				   (term-to-stamp (stamp-of-icheckpoint-term bcast))))
	 
	    ;;((ilib-init-term-p bcast)
	    ;;(break "journal-broadcast")
	    ;;)
				 
	    (t (journal-queue-broadcast ch bt)))))

      ;; PERF could skip flush if we know we always get end notification.
      ;; believe this (that we get end notification) is true but not certain.
      (flush)

      onlybcast-persist)))


(defun new-journal-link (stamp ilog server-address &optional parent onlybcast)
  ;; "journals"
  (let ((wrotep nil))
    
    (new-soft-link
     (make-journal-channel :server-address server-address
			   :ilog ilog
			   :stamp stamp
			   :onlybcast onlybcast)
     :open #'(lambda (ch)
	       (setf (journal-channel-log ch)
		     (if parent
			 (log-open-write (stamp-of-journal-channel ch)
					 (ilog-of-journal-channel ch)
					 parent)
			 (new-log-open (stamp-of-journal-channel ch)
				       (ilog-of-journal-channel ch))))

	       ;; db reads first record at read-open.
	       ;; write noop to insure it is small record.
	       (log-write-record (log-of-journal-channel ch)
				 (inoop-term)))

     :close #'(lambda (ch)
		(log-close (log-of-journal-channel ch)))

     ;; send to virtual process that is journal, ie input to journal.
     :send #'(lambda (ch term)
	       ;;(setf -ch ch -term term) (break "njls")
	       (cond
		 ((ibroadcasts-term-p term)
		  ;;(when myfu (setf -term term -wrotep wrotep) (break "myfu"))
		  (unless wrotep
		    ;; need to look into interaction of onlybcast/attr-log
		    (setf wrotep t)
		    (loglog `wrote (stamp-to-term stamp)))

		  ;; extract broadcast from !broadcast.
		  (journal-broadcast ch
				     (broadcasts-of-ibroadcasts-term term)
				     (auto-commit-of-ibroadcasts-term term)))

		 ((ireq-aux-term-p term)
		  ;;(setf -term term) (break "njlreq")
		  (let ((rterm (expression-of-ireq-term term)))
		    (cond
		      ((iconfigure-term-p rterm)

		       ;; what type of configures do we expect?
		       ;; - compression.
		       ;; - broadcast stop.
		       ;;(when stop
		       ;;  (stop-lib-journal ??addr??))

		       ;;(break "jconfigure")
		       (journal-write ch rterm)

		       (setf (journal-channel-recv-queue ch)
			     (append (journal-channel-recv-queue ch)
				     (list (irsp-term (sequence-of-ireq-term term)
						      (iack-term))))))

		      (t (message-emit (basic-message '(journal received req)
						      term))
			 ;; rle todo find out where this(ie emitted msg) goes ???
			 (setf (journal-channel-recv-queue ch)
			       (append (journal-channel-recv-queue ch)
				       (list (irsp-term (sequence-of-ireq-term term) (iack-term)))))
			 ))))

		 ((irsp-term-p term)
		  (let ((rterm (result-of-irsp-term term)))
		    ;;(setf -term term -rterm rterm) (break "njlrsp")
		    ;; here we are looking for inform of broadcast start from producer.

		    (cond
		      ((and (ivalue-term-p rterm)
			    (isubscription-term-p (result-of-iresult-term rterm)))
		       (let ((bcasts (broadcasts-of-isubscription-term
				      (result-of-iresult-term rterm))))
			 ;;(setf a bcasts b term) (break "ojs")
			 (journal-broadcast ch
					    (broadcasts-of-ibroadcasts-term bcasts)
					    (auto-commit-of-ibroadcasts-term bcasts))))

		      (t
		       (raise-error
			(error-message '(journal received response unknown) rterm))))))
	       
		 (t (message-emit (basic-message '(journal received)
						 term))))

	       (values))

     :listen #'(lambda (ch)
		 (and (recv-queue-of-journal-channel ch) t))

     ;; receive from virtual process that is journal, ie output from journal.
     :recv #'(lambda (ch)
	       (pop (journal-channel-recv-queue ch)))
     )))




;;;;	journal create
;;;;	
;;;;	  - connect to bus 
;;;;	  - connect to data base
;;;;	

;; stamp : of env?
;; assume fttb that the table stamps are not important for restore.
;; get purpose and table-types from description of env???
;; all data is easily accessible in env  present in env

(defun new-ienvironment-term (address purposes resources table-types reduction-tags)
  (ienvironment-term (itokens-term address)
		     (itokens-term purposes)
		     (itokens-term resources)
		     (itokens-term table-types)
		     (itokens-term reduction-tags)))



(defstruct journal
  blink
  environment
  types)

(defun blink-of-journal (j) (journal-blink j))
(defun environment-of-journal (j) (journal-environment j))
(defun types-of-journal (j) (journal-types j))


;; restored lib gets broadcast types based on journal table-types not lib resources or broadcast types.!!
;; todo .
(defun journal-create (env table-types &optional parent onlybcast incp)

  ;;(break "jc")
  ;; open log file in data base.

  ;; note there is a distinction between types produce and types journal'd.
  (let* ((stamp (new-transaction-stamp))
	 (address (if incp
		      (increment-environment-address
		       (address-of-environment env))
		      (address-of-environment env)))
	 (jaddress (cons 'journal address))
	 (purposes (purposes-of-environment env))
	 (desc (new-description (cons 'journal purposes)))
	 (jlink (new-journal-link stamp
				  (ilog-description-term
				   (new-ienvironment-term address
							  purposes
							  (resource-names-of-environment env)
							  (produce-table-types-of-environment env)
							  (reduction-tags-of-environment env))
				   (itokens-term table-types))
				  address
				  parent
				  onlybcast))
	 (blink (new-bus-link jlink))
	 )



    ;; add to bus
    (link-open jlink)
    (loglog 'open (stamp-to-term stamp))
    (add-bus-link blink)

    (with-unwind-error ((remove-bus-link blink)
			(link-close jlink))

      (bus-link-add-import-environment jaddress blink)
    
      ;; if close was not quick then this should be same as single broadcast read.
      ;; we ought to be able to share, via data persist? save broadcast read and
      ;; check if compare-term-p to one created here?
      ;; if not single read (other than no-op) then do not test.

      (let ((onlybcast-persist nil))
      
      (if (eql onlybcast t)
	  (let ((bcasts (producer-broadcast-state table-types
						  desc
						  (description-of-environment env))))
	    ;;(setf a bcasts b term) (break "ojs")
	    (setf onlybcast-persist
		  (journal-broadcast (channel-of-soft-link jlink)
				     (broadcasts-of-ibroadcasts-term bcasts)
				     (auto-commit-of-ibroadcasts-term bcasts))))

	  (progn
	    ;;(setf -table-types table-types -address address -env env -stamp stamp -jaddress jaddress -desc desc -blink blink)
	    ;;(break "jc")
	    (journal-send blink
			  (icommand-term
			   (iinterpret-term (isubscribe-term nil
							     (itokens-term table-types)
							     (ienvironment-address-term jaddress)
							     (stamp-to-term stamp)
							     (description-to-term desc)
							     (ivoid-term))) t))

	    #|(let ((bcasts (producer-broadcast-state table-types
						    desc
						    (description-of-environment env))))
	      (journal-broadcast (channel-of-soft-link jlink)
				 (broadcasts-of-ibroadcasts-term bcasts)
				 (auto-commit-of-ibroadcasts-term bcasts)))|#
	    )

	   
	  #|
	  ;; why not just dump result of direct call here.
	  ;; presumeably there could be some indirection??
	  (journal-send blink
			(iconfigure-term-wtransaction
			 (irequest-term
			  (istart-term (itable-types-term table-types
							  (ienvironment-address-term address))
				       (stamp-to-term stamp)
				       (ienvironment-address-term jaddress)
				       (description-to-term desc)))))
	   |#)

      (values (make-journal :blink blink :environment env :types table-types)
	      onlybcast-persist
	      )))))




(defun journal-close (journal)
  
  ;; RLE TODO ??? send stop to end broadcasts, or assume removing link is adequate.
  ;; RLE TODO ??? is stop sent by environment close  already ?

  ;; RLE TODO wait for transactions to end?
  ;; even without MTT could be waiting on io and kill via toploop.
  (let* ((blink (blink-of-journal journal))
	 (l (link-of-bus-link blink))
	 (stamp (stamp-of-journal-channel (channel-of-soft-link l))))

    (link-close l)
    (remove-bus-link blink)

    (loglog `close (stamp-to-term stamp))

    ;; return stamp
    stamp))




;;;;	restore : 
;;;;	
;;;;	uses journal to replay lib list
;;;;	  - stores update stamps in base-objcs.
;;;;	  - activate/deactivate do not affect lib but are remembered.
;;;;	then activates objs active at end of replay.
;;;;	  - order of activation could be lib-order or dependency order.

;; run the bcasts through library.
;; need to update lib-import to do the right thing with light stores.
;; also activates and deactivates need to be delayed.

;;;;	This is done by replacing the import/activate/deactivate hooks
;;;;	in the library-table during restore.
;;;;	
;;;;	need to alter bcast sentry to accept library-light during restore.


;; RLE TODO could delay activation to allow browsing of lib. open_environment could return
;; a closure to be called to reactivate. Of course, no guarauntee of success if you've mucked up lib
;; in the meantime.

;;;;	
;;;;	only items in log AtTheMoment are bind and unbind broadcasts
;;;;	
;;;;	restore could track bind and unbinds and then simply bind those remaining binding
;;;;	but probably safer to just rerun the broadcasts. Restore must add passport wrappers.
;;;;	
;;;;	currently passport is journaled. but should save desc in ilog and not include
;;;;	desc in records.
;;;;	
;;;;	


;;;;	
;;;;	Union :
;;;;	  - import journal into an open environment.
;;;;	     * close, import, open, dump.
;;;;	  - drive state from one environment to another. In one env make
;;;;	    requests to duplicate lib in second. 
;;;;	     * connect, req loop, close.
;;;;	
;;;;	
;;;;	incremental vs batch
;;;;	
;;;;	When merging : only expect insert, allow, and activate.
;;;;	  insert : deactivate, allow, delete old def
;;;;	  allow  : allow may already be allowed.
;;;;	  activate : may already be active.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	import :
;;;;	  - apply-passport : updates local lib-tables.
;;;;	      * silently shadows current defs if conflict
;;;;	  - broadcast producer state (of local touch history) to self
;;;;	   
;;;;	  * conflicts do not appear to be a problem, but I suspect
;;;;	    some assumptions are being violated. EG, delete after import
;;;;	    may expose old value. Probably should detect and fail if conflict,
;;;;	    or at least delete old.
;;;;	
;;;;	  - no broadcasts to connected components.
;;;;	      * could do at open only.
;;;;	
;;;;	

;;;;	
;;;;	OnlyBCast :
;;;;	
;;;;	  - environment is initialized by reading previous log.
;;;;	  - then new log is created by dumping environment, 
;;;;	    and then incrementally updated.
;;;;	  - a graceful close will create a log (checkpoint) with a single bcast
;;;;	    pointing to a term containing the state of the environment
;;;;	    at time of close.
;;;;	  - a log derived from such a graceful close need only point
;;;;	    to the same term to dump the initial environment.
;;;;	  ? should ignore environment attr log updates ?
;;;;	  
;;;;	Thus journal create needs to be able to find the pointer to an onlybcast if available,
;;;;	to repoint the new journal. Thus journal-import makes a note if the environment
;;;;	was initialized from an onlybast.
;;;;	
;;;;	  - checkpoint : always onlybcast
;;;;	  
;;;;	
;;;;	

(defun configure-noack-term-p (term)
  (and (iconfigure-term-p term)
       (let ((cmd (command-of-iconfigure-term term)))
	 (and (iinform-term-p cmd)
	      (inoack-term-p (info-of-iinform-term cmd))))))

;; does not update journal. 
;; only updates environment state.
(defvar *jidebug* nil)
(defvar *jidebughook* nil)

(defun jiabort (term)
  ;;(setf -term term)
  (if (eql `|!collection| (id-of-term term))
      (let ((ipass (icar term)))
	(if (ipassport-term-p ipass)
	    (let ((ithing (term-of-bound-term (third (bound-terms-of-term ipass)))))
	      (let ((n (value-of-parameter-r (car (parameters-of-term ithing)))))
		;;(setf -n n)
		(when (> n 363550) (break "jia"))
		))
	    (break "not-passport")))
      (let ((ipass term))
	(if (ipassport-term-p ipass)
	    (let ((ithing (term-of-bound-term (third (bound-terms-of-term ipass)))))
	      (let ((n (value-of-parameter-r (car (parameters-of-term ithing)))))
		;;(setf -n n)
		(when (> n 363550) (break "jia"))
		))
	    (break "not-passport"))))

  term)

(defun journal-import (f stamp)

  ;;(setf a stamp) (break "ji")

  (let* ((log-stamp (if (or (db-log-p stamp)
			    (progn (reset-db-environments)
				   (db-log-p stamp))) 
			stamp
			(db-query-checkpoint stamp)))

	 (checkpoint-stamp (unless (db-log-p stamp) stamp))
	 ;; could check if ienv compatable with env.
	 ;;(ilog (log-query log-stamp))
	 ;;(ienv (environment-of-ilog-description-term ilog))
	 (log (log-open-read log-stamp)))


    ;; could read all and do single broadcast or read batchs.
    ;; read batches may work well with MTT to allow interleaving of io.
    (let ((onlybcast t))
      (let (
	    (donep nil)
	    )

	(labels
	    ((doit (bcast)
	       ;; broadcast : why not cheat and apply directly to table?
	       ;; this is more robust if we want a partial restore for example where some
	       ;; broadcast are filtered.
	       
	       (if (ienvironment-state-term-p bcast)
		   (progn (environment-state-in bcast)
			  nil)

		   (progn
		     ;;(setf b bcast) (break "jidoit")
		     (unless (ipassport-term-p bcast)
		       ;;(setf b bcast)
		       (raise-error (error-message '(journal read !passport not))))

		     (let ((imsg (broadcast-of-ipassport-term bcast)))

		       (cond
			 ((or (ibegin-term-p imsg) (iend-term-p imsg))
			  nil)
		
			 ((icheckpoint-term-p imsg)
			  (when (and checkpoint-stamp
				     (equal-stamps-p checkpoint-stamp
						     (term-to-stamp
						      (stamp-of-icheckpoint-term imsg))))
			    t))

			 (t;;(setf a bcast) (break "jib")
			  (funcall f bcast)
			  nil)))))))
    
	  (do ((term (log-read-record log) (log-read-record log)))
	      ((or donep (null term)))

	    ;;(when *jidebug* (setf -a term) (break "nidtp"))
	    ;;(when *jidebughook* (setf term (funcall *jidebughook* term)))
		  
	    (let ((bcast (if (idata-persist-term-p term)
			     (progn
			       ;;(setf a term b (term-to-data term)) ;;(break "idtp")
			       (term-of-term-data (provide-data (term-to-data term) 'term-data))
			       )
			     (progn
			       (unless (inoop-term-p term)
				 (setf onlybcast nil))
			       term))))

	      (unless (or (inoop-term-p bcast)
			  (configure-noack-term-p bcast))

		;;(setf b bcast c onlybcast d term) (break "jo")
		(if (eql t onlybcast)
		    (setf onlybcast (cons bcast term))
		    (setf onlybcast nil))

		(if (icollection-term-weak-p bcast)
		    (dolist (bt (bound-terms-of-term bcast))
		      (setf donep (doit (term-of-bound-term bt))))
		    (setf donep (doit bcast))))))))
	
      onlybcast)))


(defun increment-environment-address (address )
  (let ((s (string (car (last address)))))
    (if (every #'(lambda (ch) (digit-char-p ch)) s)
	(append (butlast address)
		(list (intern-system
		       (princ-to-string (1+ (with-string-scanner (s)
					      (scan-decimal-num)))))))
	(append address 
		(list (intern-system "0"))))))


;; needs to happen in scope of transaction, transaction done by orb? transaction manager.
(defun journal-open (stamp &optional new-address)
  (when (null (transaction-state t))
    (break "journal-open"))
  
  (stop-db-buffering)
  (prog1
      (with-io-echo-stats (nil "Journal Open")
       (let* ((log-stamp (if (or (db-log-p stamp)
				 (progn (reset-db-environments)
					(db-log-p stamp)));;try reset in case collection occurred or new jnl
			     stamp
			   (db-query-checkpoint stamp)))
	      (ilog (log-query log-stamp))
	      (ienv (environment-of-ilog-description-term ilog)))

	 (let ((address (or new-address
			    (increment-environment-address
			     (tokens-of-itokens-term (address-of-ienvironment-term ienv)))))
	       (purposes (tokens-of-itokens-term (purposes-of-ienvironment-term ienv)))
	       (resources (tokens-of-itokens-term (resources-of-ienvironment-term ienv)))
	       (table-types (tokens-of-itokens-term (table-types-of-ienvironment-term ienv)))
	       (reduction-tags (tokens-of-itokens-term (reduction-tags-of-ienvironment-term ienv)))
	       )

	   ;; kludge to get old libraries to do ObjectIdDAG
	   ;;(setf a purposes) (break "jo")
	   (when (and (member 'library purposes) (not (member 'terms resources)))
	     (setf resources (append resources '(terms)))
	     (setf purposes (append purposes '(|ObjectIdDAG|)))
	     )

	   (let ((env (library-environment address
					   purposes
					   resources
					   table-types
					   reduction-tags))
		 ;; stamp for open to pose as producer. but why not just use env stamp?
		 (consumer-types (broadcast-types-of-resource-names resources))
		 )

	     ;; update db-environments if new env.
	     (when new-address
	       (unless (boundp '*db-environments*)
		 (raise-error (error-message '(journal open db-environments not))))
	       (update-db-environments env))
	  
	     (add-environment env)

	     ;; enable broadcasts for restore.
	     ;; import will apply broadcasts to environment
	     ;;(setf -r resources -c consumer-types) (break "joc")
	     (environment-broadcast-update env consumer-types (stamp-term-of-environment env) ;;pstamp
					   (new-broadcast-state consumer-types
								address
								(stamp-term-of-environment env)
								address
								(stamp-term-of-environment env)
								(description-of-environment env)))
	
	       (with-environment-actual env
		 (let ((oids (new-oid-table)))
		   (with-local-touch-history ((transaction-state) env)
	  	     (let ((onlybcast
			    (let ((auto-stamp (new-transaction-stamp))
				  (stamp-term (stamp-term-of-environment env)))
			      (journal-import
			       #'(lambda (bcast)
				   (apply-passport env
						   (ipassport-term (table-type-of-ipassport-term bcast)
								   stamp-term ;; table-stamp
								   (description-of-ipassport-term bcast)
								   (broadcast-of-ipassport-term bcast))
						   (current-transaction-stamp)
						   auto-stamp))
			       stamp))))

		       (dolist (tr (touch-history-of-environment (transaction-state) env))
			 (let ((oid (oid-of-touch-record tr)))
			   (setf (gethash (stamp-of-oid oid) oids) t)))

		       ;; need to broadcast inserts (ie activates) to self.
		       ;; (setf a oids b env) (break "ji"

		       (do-broadcasts env
			 (producer-broadcast-state
			  (remove 'library (broadcast-types-of-resource-names
					    (resource-names-of-environment env)))
			  (description-of-environment env)
			  (description-of-environment env)
			  oids))

		       ;; create new journal and dump state.
		       (environment-push-journal env
						 (journal-create env
								 (tokens-of-itokens-term
								  (types-of-ilog-description-term ilog))
								 log-stamp
								 onlybcast)))))))
	   address)))

    (start-db-buffering)))


;;(defunml (|open_environment| (term))
;;    (term -> unit)
;;
;;  (journal-open (term-to-stamp term)))





(defunml (|open_environment| (s))
    (string -> (tok list))

  (journal-open (string-to-stamp s)))


(defun import-environment-aux (f s)
  (journal-import #'(lambda (bcast)
		      (ap f (broadcast-of-ipassport-term bcast)))
		  s))

(defunml (|import_environment| (f s))
    ((term -> unit) -> (string -> unit))

  (import-environment-aux f (string-to-stamp s))
  nil)


(defunml (|import_environment_by_match| (f paddr))
    ((term -> unit) -> ((tok list) -> unit))

  (reset-db-environments)
  (let ((e-addr (match-db-environment paddr)))

    ;;(setf a e-addr b paddr) (break "oebm")
    (if e-addr
	(import-environment-aux f (cdr e-addr))
	(raise-error (error-message '(open environment match not) paddr))))
  nil)

(defun open-environment-by-match (paddr) 
  (reset-db-environments)
  (let ((e-addr (match-db-environment paddr)))

    ;;(setf a e-addr b paddr) (break "oebm")
    (if e-addr
	(journal-open (cdr e-addr))
	(raise-error (error-message '(open environment match not) paddr)))))


(defunml (|open_environment_by_match| (paddr))
    ((tok list) -> (tok list))
 
  (open-environment-by-match paddr))


(defunml (|open_environment_by_match_as| (paddr newaddr))
    ((tok list) -> ((tok list) -> (tok list)))

  (reset-db-environments)
  (let ((e-addr (match-db-environment paddr)))

    ;;(setf a e-addr b paddr) (break "oebm")
    (if e-addr
	(journal-open (cdr e-addr) newaddr)
	(raise-error (error-message '(open environment match not) paddr)))))


(defunml (|environment_start_journal| (addr types))
    ((tok list) -> ((tok list) -> unit))

  (when (bus-environment-p addr)
    (message-emit (warn-message '(duplicate) addr)))

  (let ((e (find-environment addr)))
    (environment-push-journal e (journal-create e types))))


;;;;	checkpoint should only be done if no pending completions?
;;;;	doh! only committed bcasts are written to log.
;;;;	


(defunml (|environment_stop_journal| (addr))
    ((tok list) -> unit)

  (journal-close (environment-pop-journal (find-environment addr))))



;; this would be better done with an enviroment mutation, ie restore environment
;; then mutate. maybe mutate by having old environment broadcast state to new.
;;(defunml (|open_environment_detail| (addr purposes resources table-types reduction-tags))
;;    ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> unit)))))
;;  (lib-restore addr
;;	       purposes
;;	       resources
;;	       table-types
;;	       reduction-tags))


(defun journal-checkpoint-aux (env j)

  (let ((types (types-of-journal j))
	(parent (journal-close j)))

    (environment-delete-journal env j)
    
    (mlet* (((j onlybcast) (journal-create env types parent t t)))

	   (let ((chk (journal-close j)))
	     (loglog 'checkpoint (stamp-to-term chk))
	     (values chk onlybcast types) ))))


(defun journal-checkpoint (env anotherp)

  (let ((journals (journals-of-environment env)))

    (if (not (and journals (null (cdr journals))))
	(raise-error (error-message '(journal checkpoint single not) (length journals) (address-of-environment env)))

	(let ((j (car journals)))

	  (mlet* (((chk onlybcast types) (journal-checkpoint-aux env j)))
		 
		 ;; should make sure closed is removed from env.
		 (when anotherp
		  (environment-push-journal env
					    (journal-create env types chk onlybcast)))

		 chk)))))


