
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


(defunml (|quit_server_loop| (unit)  :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> unit)

  (quit-request-loop))

(defunml (|quit_request_loop| (unit)  :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> unit)

  (quit-request-loop))


(defunml (|cmd_eval_string| (address s))
    ((tok list) -> (string -> unit))
  (cmd-eval address (iexpression-term (iml-woargs-term nil t (text-to-term s))))
  nil)

(defunml (|cmd_eval_term_to_term| (address term arg-list))
    ((tok list) -> (term -> ((term list) -> term)))
  (cmd-eval address (iexpression-term (iml-term nil t term arg-list))))




(defunml (|directory| (dir))

   (string -> (string list))

  (mapcar #'pathname-name (directory (pathname dir))))


(defunml (|new_environment| (addr purposes resources table-types reduction-tags))
    ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> unit)))))

  (when (component-environment-p addr)
    (raise-error (error-message '(environment new exists) addr)))

  (add-environment (new-environment addr purposes resources table-types reduction-tags
				    (when (member 'library purposes)
				      (make-library-sub-environment)
				      )))
  )



(defunml (|orb_name| (unit) :declare ((declare (ignore unit))))
    (unit -> tok)
  (orb-name))

(defunml (|sys_version| (unit) :declare ((declare (ignore unit))))
    (unit -> tok)
  *sys-version*)


(defunml (|orb_address| (unit) :declare ((declare (ignore unit))))
    (unit -> (tok list))

  (orb-address))

(defunml (|current_environment_address| (unit) :declare ((declare (ignore unit))))
    (unit -> (tok list))

  (address-of-environment (current-environment)))

(defunml (|orb_local_environments| (unit) :declare ((declare (ignore unit))))
    (unit -> ((tok list) list))

  (mapcar #'(lambda (env)
	      (address-of-environment env))
	  *component*))

(defunml (|orb_match_local_environment| (tags))
    ((tok list) -> (tok list))

  (let ((env (match-environment-in-list tags *component*)))
    (unless env
      (raise-error (error-message '(match not) tags)))

    (address-of-environment env)))


(defunml (|environment_table_types| (addr))
    ((tok list) -> (tok list))
  (produce-table-types-of-environment
   (find-any-environment addr)))

(defunml (|orb_bus_environments| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> ((tok list) list))
  (orb-bus-environments))


(defun orb-match-bus-environment (tags &optional nil-ok-p)
  (let ((env (match-bus-environment tags)))
    (if env
	(address-of-environment env)
	(unless nil-ok-p (raise-error (error-message '(match not) tags))))))

(defunml (|orb_match_bus_environment| (tags))
    ((tok list) -> (tok list))

  (orb-match-bus-environment tags))

(defunml (|orb_match_bus_environments| (tags))
    ((tok list) -> ((tok list) list))

  (mapcar #'address-of-environment (match-bus-environments tags)))

(defunml (|orb_active_broadcasts| (addr))
    ((tok list) -> (((tok list) |#| ((tok list) |#| (tok list))) list))

  (show-broadcasts addr))


(defun orb-bus-orb (port)
  
  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(bus-link not)
				  port)))
        
    (let ((env (match-environment-in-list '(orb)
					  (imported-environments-of-bus-link blink))))
      (unless env
	(raise-error (error-message '(orb not))))

      (address-of-environment env))))

  
(defunml (|orb_bus_orb| (port))
    (int -> (tok list))
     (orb-bus-orb port))

(defunml (|orb_bus_port| (addr))
    ((tok list) -> int)

  (port-of-bus-link (bus-link-of-environment-address addr)))

(defunml (|orb_bus_exported_environments| (unit) :declare ((declare (ignore unit))))
    (unit -> ((tok list) list))
  (mapcan #'(lambda (blink)
	      (mapcar #'address-of-environment
		      (exported-environments-of-bus-link blink)))
	  *bus-links*))

(defunml (|orb_send_address| (port))
    (int -> unit)

  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(orb send-address bus-link not)
				  port)))
    
    (config-export-address blink (orb-address)))

  (values))

(defunml (|orb_unsend_address| (port))
    (int -> unit)

  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(orb send-address bus-link not)
				  port)))
    
    (config-unexport-address blink (orb-address)))

  (values))


;; note set's properties for both partners of blink.
(defunml (|orb_send_bus_link_properties| (port properties))
    (int -> (((tok |#| term) list) -> unit))

  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(orb send-properties bus-link not)
				  port)))
    
  (config-blink-inform-aux
   blink
   (ilink-properties-term (properties-to-term properties))
   'export-properties)
  
  nil))



(defunml (|orb_request_address| (port))
    (int -> (tok list))

  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(orb request-address bus-link not)
				  port)))

    (config-request-address blink nil)))

(defunml (|orb_unrequest_address| (port))
    (int -> (tok list))

  (let ((blink (bus-link-of-port port)))
    (unless blink
      (raise-error (error-message '(orb request-address bus-link not)
				  port)))

    (config-request-address-revoke blink nil)
))


;; appl must id to server prior to routing or reqs
;; thus no hardship to specify address in property req.
(defunml (|orb_send_environment_properties| (dest eaddr properties))
    ((tok list) -> ((tok list) -> (((tok |#| term) list) -> unit)))

  (config-blink-inform-aux
   (bus-link-of-environment-address dest)
   (ilink-environment-properties-term (itokens-term eaddr)
				      (properties-to-term properties))
   'export-properties)
  
  nil)

(defunml (|orb_send_environment_description| (dest eaddr))
    ((tok list) -> ((tok list) -> unit))
  (config-export-description (bus-link-of-environment-address dest) eaddr)
  nil)

(defunml (|orb_unsend_environment_description| (dest eaddr))
    ((tok list) -> ((tok list) -> unit))
  (config-unexport-description (bus-link-of-environment-address dest) eaddr)
  nil)

(defunml (|orb_request_environment_description| (dest eaddr))
    ((tok list) -> ((tok list) -> term))
  (config-request-environment-description (bus-link-of-environment-address dest) eaddr))

(defunml (|orb_unrequest_environment_description| (dest eaddr))
    ((tok list) -> ((tok list) -> unit))
  (config-unrequest-environment-description (bus-link-of-environment-address dest) eaddr)
  nil)


(defunml (|orb_send_environment_address| (dest source))
    ((tok list) -> ((tok list) -> unit))
  (config-export-address (bus-link-of-environment-address dest) source)
  nil)

(defunml (|orb_unsend_environment_address| (dest addr))
    ((tok list) -> ((tok list) -> unit))
  (config-unexport-address (bus-link-of-environment-address dest) addr)
  nil)


(defunml (|orb_request_environment_address| (dest addr))
    ((tok list) -> ((tok list) -> (tok list)))
  (config-request-address (bus-link-of-environment-address dest) addr))

(defunml (|orb_unrequest_environment_address| (dest addr))
    ((tok list) -> ((tok list) -> unit))

  (config-request-address-revoke (bus-link-of-environment-address dest) addr)
  nil)


(defunml (|orb_connect_environments| (local remote))
    ((tok list) -> ((tok list) -> unit))
  (config-connect-environments (bus-link-of-environment-address remote)
			       local remote))

(defunml (|orb_unconnect_environments| (local remote))
    ((tok list) -> ((tok list) -> unit))
  (config-unconnect-environments (bus-link-of-environment-address remote)
				 local remote))




;;;;	It is sensible to allow similar multiple tables within
;;;;	an environment. It is not sensible to allow similar
;;;;	multiple table-type to be produced by a single environment.
;;;;	Why? It would be sensible if producer transactions where parameterizable
;;;;	by a producer stamp. Producer could simplified as a transaction manager.
;;;;	RLE !!! Twould be nice to abstract out the transaction management functions
;;;;	   from the library. Ie lib becomes a client (intraprocess) of the transaction
;;;;	   manager rather then being the transaction manager.



(defunml (|subscribe| (synchro producer resource-names))
    ((unit |+| term) -> ((tok list) -> ((tok list) -> unit)))

  (subscribe-client-initiate resource-names producer (when (not (ml-isl synchro)) (ml-outr synchro)))
      
  (values))

(defunml (|subscribe_transaction| (un producer))
    (bool -> ((tok list) -> unit))

  (if un
      (unsubscribe-transactions producer)
      (subscribe-transactions producer))
      
  (values))


(defunml (|exit| (unit) :declare ((declare (ignore unit))))
  (unit -> unit)

  (orb-cleanup t)
  (update-process-activity-log "exit")
  (quit))

;; evaled in consumer ??
;; producer halt is handled at lower level.
(defunml (|unsubscribe| (syncp producer resource-names))
    (bool -> ((tok list) -> ((tok list) -> (unit |+| term))))

  (let ((sync (unsubscribe-client-initiate resource-names producer syncp)))
    (if sync
	(ml-inr sync)
	(ml-inl nil))))

(defunml (|close_environment| (addr quickp gcp))
    ((tok list) -> (bool -> (bool -> (string list))))

  (let ((env (find-environment addr)))
    (if (member 'library (environment-resources env) :key  #'car)
	(close-environment env quickp gcp)
	(close-environment-client env quickp))))

(defunml (|cleanup_bus_link| (addr))
    ((tok list) -> bool)

  (let ((blink (bus-link-of-environment-address addr)))
    (when blink
      (bus-link-close blink)
      t)))


;;;
;;;	connect
;;;

;; default compression-levels. TODO: needs higher level management.
(defvar *orb-compression-levels* nil)

(defun orb-start-lib-accept (port &optional (type '|fdl|) (compressp t))
  (let* ((socket (or (new-socket-listen port)
		     (raise-error (error-message '(orb start accept) port))))
	 (c (new-accept-channel socket
				(eql type `|mathbus|)
				(when (eql type '|fdl|) *orb-compression-levels*)
				(when (eql type '|fdl|) compressp)))
	 (l (new-soft-link c
			   :close #'accept-channel-close
			   :listen #'accept-channel-listen)))
    (link-open l)
    (add-bus-link (new-bus-link l))

    (make-orb-transaction-manager)
    ))

(defun orb-start-appl-server-accept (port &optional (type '|fdl|) (compressp t) api-kind)
  (let ((l (orb-start-appl-server-accept port type compressp api-kind)))
    (add-bus-link (new-bus-link l))
    l))

;; type is either `fdl`or `mathbus`
(defunml (|orb_start_accept| (port type))
  (int -> (tok -> unit))

  (case type
    ((|fdl-uncompressed| fdl-uncompressed)
     (orb-start-lib-accept port 'fdl nil))
    (otherwise		 (orb-start-lib-accept port type))))
   
(defunml (|orb_start_application_server_accept| (port))
  (int -> unit)

  (orb-start-appl-server-accept port 'fdl nil))

(defunml (|make_www_command_map| (name obids))
    (tok -> ((object_id list) -> unit))

  (make-www-command-map name obids))
			     
(defunml (|orb_start_restricted_application_server_accept| (port api-kind))
  (int -> (tok -> unit))

  (orb-start-appl-server-accept port 'fdl-uncompressed nil api-kind))


(defunml (|orb_stop_accept| (port))
  (int -> unit)
  
  (let ((blink (find-first #'(lambda (blink)
			       (let ((channel (car (channels-of-link (link-of-bus-link blink)))))
				 (when (and (accept-channel-p channel)
					    (equal (port-of-socket (socket-of-accept-channel channel)) port))
				       blink)))
			   *bus-links*)))
    (when blink
	   ;;(format t "found blink ~%")
	  (bus-link-close blink)))
  )

(defun accept-active-p (port)
    
  (and (find-first #'(lambda (blink)
		       (let ((channel (car (channels-of-link (link-of-bus-link blink)))))
			 (when (and (accept-channel-p channel)
				    (equal (port-of-socket (socket-of-accept-channel channel)) port))
			   blink)))
		   *bus-links*)
       t))

;; used to :
;; establish 2 distinct bi-directional streams with (1)remote server
;; at port on host and (2) local-port, assumes local-host variable is
;; set.
;; now makes one uncompressed bi-directional ascii stream and expects
;; further config for mathbus/compress/multiple streams.
(defunml (|orb_connect| (remote-port remote-host local-port))
    (int -> (string -> (int -> unit)))

  (orb-connect-uncompressed-stream remote-port remote-host
		      ;;local-port
		      ))


(defunml (|orb_compressed_stream| (port))
    (int -> unit)

  (config-compressed-blink (bus-link-of-port port))
  (values))

(defunml (|orb_uncompressed_stream| (port))
    (int -> unit)

  (config-uncompressed-blink (bus-link-of-port port))
  (values))

(defunml (|orb_mathbus_stream| (port))
    (int -> unit)

  (config-mathbus-blink (bus-link-of-port port))
  (values))
  

;;   A stream channel may have 2 ports associated with it (1 for
;;   in-stream, 1 for out) if port = output port, channel is closed.
;;   this solves possible ambiguity problem since we assume only 1
;;   connection is made to a particular server socket.  if client
;;   created socket for input, it is destroyed as well.  ( server
;;   socket is only destroyed when accept channel is closed)

(defunml (|orb_disconnect| (port))
    (int -> unit)
   
  (orb-disconnect port t))


(defunml (|with_transaction| (f) :declare ((declare (ignore unit))))
    ((unit -> *) -> *)

  (with-appropriate-transaction (t nil) (funmlcall f nil)))

(defunml (|with_environment_transaction| (ea f))
    ((tok list) -> ((unit -> *) -> *))

  (with-environment-actual (find-environment-in-list ea *component*)
   (with-appropriate-transaction (t nil) (funmlcall f nil))))


(defunml (|warn_message| (oids tags terms) :error-wrap-p nil)
    ((object_id list) -> ((tok list) -> ((term list) -> unit)))

  (message-emit (oid-warn-message oids tags terms)))


(defunml (|inform_message| (oids tags terms) :error-wrap-p nil)
    ((object_id list) -> ((tok list) -> ((term list) -> unit)))

  (message-emit (oid-inform-message oids tags terms)))


;;(defunml (|raise_error| (oids tags terms) :error-wrap-p nil))
(defunml (|raise_error| (oids tags terms))
    ((object_id list) -> ((tok list) -> ((term list) -> unit)))

  (raise-error (oid-error-message oids tags terms)))

(defunml (|with_error_hook| (h f a))
    ((term -> *) -> ((** -> *) -> (** -> *)))

  (with-handle-error-and-message (nil #'(lambda (m)
					  ;;(setf -m m) (break "weh")
					  (funmlcall h (message-to-term m))))
    (funmlcall f a)))

(defunml (|add_transaction_end_hook| (name h))
    (tok -> (((object_id list) -> unit) -> unit))

  (add-transaction-end-hook
   name
   #'(lambda (thl)
       (funmlcall h (mapcar #'oid-of-touch-record thl))))

  (values))
  


;;;;	
;;;;	Use of these needs to be restricted somehow.
;;;;	FTTB, used to import and export during restore/save in metaprl.
;;;;	

(defunml (|string_to_object_id| (s))
    (string -> object_id)

  (string-to-oid s))

(defunml (|object_id_to_string| (oid))
    (object_id -> string)

  (string-of-oid oid))

;; (slow-counter # processid ) # (fast-counter # time)
(defunml (|destruct_object_id| (oid))
    (object_id -> ((int |#| token) |#| (int |#| int)))

  (stamp-of-oid oid))

;; not completely robust, but good enough fttb.
(defun stamp-later-p (a b)
  (or (> (time-of-stamp a) (time-of-stamp b))
      (and (= (time-of-stamp a) (time-of-stamp b))
	   (eql (process-id-of-stamp a) (process-id-of-stamp b))
	   (> (sequence-of-stamp a) (sequence-of-stamp b)))))

	   
(defunml (|later_stamp_p| (a b))
    (term -> (term -> bool))
 (stamp-later-p (term-to-stamp a) (term-to-stamp b)))

(defunml (|get_stamp| (unit) :declare ((declare (ignore unit))))
    (unit -> term)

  (stamp-to-term (transaction-stamp)))
    
(defunml (|term_to_string| (term))
    (term -> string)
  (term-to-standard-character-string term))

(defunml (|string_to_term| (s))
    (string -> term)
 (standard-character-string-to-term s))

(defunml (|local_eval| (term))
    (term -> term)
  
  ;;(setf -leterm term) (break "le")
  (let ((result (ml-eval (without-dependencies
			  (iml-woargs-term nil t
					   (source-reduce term
							  (append '(ml code)
								  (reduction-tags-of-environment
								   (current-environment))))))
			 t)))
       
    ;;(setf -r result -term term) (break "le")

    (if (ifail-term-p result)
	(ifail-term result)
	result)))

(defunml (|mleval| (term))
    (term -> *)
  
  ;;(setf -leterm term) (break "le")
  (ml-term (source-reduce term
			  (cons 'ml (reduction-tags-of-environment
				     (current-environment))))
	   t))

;;;;	
;;;;	central library env may have several remote edd or ref environments.
;;;;	asynchrounous events in the library may be intend for some subset
;;;;	or more commonly a particular remote environmnet. Thus some method
;;;;	of filter remote environments is required. 
;;;;	
;;;;	  - pass properties from transaction to transaction when doing asynch.
;;;;	  - associate properties with remote environments.
;;;;	  - include properties when making requests.
;;;;	
;;;;	  - enable nmemonic identification of environments to allow
;;;;	    users to address other users.
;;;;	

(defun get-transaction-properties ()
  (let ((tstate (transaction-state)))
    (when tstate
      (mapcan #'(lambda (p)
		  (when (and (cdr p) (term-p (cdr p)))
		    (list p)))
	      (properties-of-tstate tstate)))))

(defun set-transaction-properties (props)
  (when props
    (let ((tstate (transaction-state)))
      (when tstate
	(dolist (prop props) 
	  (tstate-property-acons (car prop) (cdr prop) tstate))))))

;; maybe do not want to interpret result
(defun local-eval-apply-aux (term args &optional props)

  (set-transaction-properties props)
			     
  ;;(setf -leterm term) (break "le")
  (let ((result (ml-eval
		 (without-dependencies
		  (iml-term nil t
			    (source-reduce term
					   (cons 'ml (reduction-tags-of-environment
						      (current-environment))))
			    args))
		 t)))

    ;;(setf -leterm term -result result) (break "le")
      
    (or (interpret-result result)
	(setf -r result) (break "le")
	(raise-error (error-message '(result term not)))))
  )


(defunml (|local_eval_apply| (term args))
    (term -> ((term list) -> term))
  (local-eval-apply-aux term args))

;; props should be (tok # term) list ??? to be most general.
(defunml (|local_eval_apply_wprops| (props posure))
    (((tok |#| term) list) -> ((term |#| (term list)) -> term))

   (local-eval-apply-aux (car posure) (cdr posure) props))


(defunml (|get_environment_property| (name))
    (tok -> term)

  (or (let ((env (current-environment)))
	(when env
	  (cdr (assoc name (properties-of-environment env)))))

      (fail-ml `|get_environment_property|)))

  ;; first_tok (get_transaction_property `edd_connection`)
(defunml (|get_transaction_property| (name))
    (tok -> term)

  ;;(format t "~%gtp")
  (or (let ((tstate (transaction-state)))
	(when tstate
	  (let ((prop (property-of-tstate name tstate)))
	    (when (and prop (term-p prop))
	      prop))))

      (progn ;;(format t "~%gtp failed")
	     (fail-ml `|get_transaction_property|))))

;; first_tok (get_transaction_property `edd_connection`)
(defunml (|get_transaction_properties| (unit) :declare ((declare (ignore unit))))
    (unit -> ((tok |#| term) list))

  ;;(break "gtp")
  (get-transaction-properties))

(defunml (|add_transaction_property| (name prop))
    (tok -> (term -> unit))

  (let ((tstate (transaction-state)))
    (when tstate
      (tstate-property-acons name prop tstate))))


(defunml (|set_environment_property| (name prop))
  (tok -> (term -> unit))

  (let ((env (current-environment)))
    (if env
	(environment-property-acons name prop env)
	(fail-ml `|set_environment_property|))
    
    (values)))


(defunml (|without_dependencies| (f))
    ((unit -> *) -> *)

   (without-dependencies
    (funmlcall f nil)))

(defunml (|with_dependencies_report| (f))
    ((unit -> *) -> *)

   (with-ephemeral-dependencies
     (progn
       ;;(break "wdr")
       (let ((r (funmlcall f nil)))
	 ;;(break "wdr2")
	 r))))

(defunml (|ephemeral_dependencies| (unit) :declare ((declare (ignore unit))))
    (unit -> ((tok |#| (object_id list)) list))

  (mapcar #'(lambda (deps)
	      (cons (tag-of-dependencies deps)
		    (remove-duplicate-dependencies (dependencies-list deps) t)))
	  (ephemeral-dependencies)))


;; probably don't want to return full description.
(defunml (|current_description_aux| (unit) :declare ((declare (ignore unit))))
    (unit -> term)

  (description-to-term (description-of-environment (current-environment)))
  )


;;;;	
;;;;	
;;;;	local funcs to eval to modify props in connected env.
;;;;	  ? maintain local copy ?
;;;;	
;;;;	
;;;;	modify_environment_property	: tok -> term -> unit
;;;;	remove_environment_property	: tok -> unit
;;;;	
;;;;	set_environment_bool_property	: tok -> bool -> unit
;;;;	set_environment_priority_property	: tok -> int -> unit
;;;;	
;;;;	show_enviroment_properties	: unit -> (tok # term) list
;;;;	
;;;;	func which modifies props of a connected env.
;;;;	
;;;;	modify_environment_properties	: (tok list) -> ((tok # term) list -> (tok # term) list) -> unit
;;;;	
;;;;	

(defunml (|modify_environment_properties| (address f))
    ((tok list) -> ((((tok |#| term) list) -> ((tok |#| term) list)) -> unit))

  (let ((conns (find-connections-by-address address)))
    (unless (and conns (null (cdr conns)))
      ;;(setf -a address -f f) (break "mep")
      (raise-error (error-message '(|modify_environment_properties|)
				  (mapcar #'address-of-connection conns))))

    (let ((props (funmlcall f (properties-of-connection (car conns)))))
      (setf (environment-properties (environment-of-connection (car conns)))
	    props)))

  nil)

(defunml (|show_environment_properties| (address))
    ((tok list) -> ((tok |#| term) list))

  ;;(setf -addr address) (break "mep")
  (let ((conns (find-connections-by-address address)))
    (unless (and conns (null (cdr conns)))
      (break "sep")
      (raise-error (error-message '(|show_environment_properties|)
				  (mapcar #'address-of-connection conns))))

    (properties-of-connection (car conns))))


(defunml (|orb_eval| (tags expr-args))
    ((tok list) -> ((term |#| (term list)) -> unit))
   (orb-queue-asynch-request-aux (mapcar #'token-parameter tags) expr-args nil nil)
   (values))


;; desc is used to find suitable env for eval.
;; at some point may want to all multiple destinations if no result.
(defunml (|orb_eval_args| (fan desc expr-args))
    (tok -> (term -> ((term |#| (term list)) -> unit)))

  ;;(setf a fan b desc c expr-args)  (break "oea")

  (let ((rsps (orb-eval-by-description desc
				       (iexpression-term (iml-term nil t (car expr-args) (cdr expr-args)))
				       fan)))
    (if (listp rsps)
	(mapc #'interpret-result rsps)
	(interpret-result rsps))
    nil))

(defunml (|mother_of_all_eval_args| (fan toru connprops desc transprops expr-args))
    (tok -> (bool -> ((tok list) -> (term ->
					   (((tok |#| term) list) -> ((term |#| (term list))
								    -> (unit |+| term)))))))
	 
  (let ((rsps (orb-eval-by-description
	       desc
	       (let ((et (iexpression-term
			  (iml-term nil t (car expr-args) (cdr expr-args)))))
		 (if transprops
		     (ieval-property-term (iproperty-term 'transaction
							  (properties-to-term transprops))
					  et)
		     et))
	       fan
	       connprops)))

    ;; toru : term(inr) or unit(inl)
    (let ((p (listp rsps)))
      (let ((r (if p
		   (mapc #'interpret-result rsps)
		   (interpret-result rsps))))
	(cond
	  ((and toru p)
	   (raise-error (error-message '(result term list))))
	  ((and toru (null r))
	   (raise-error (error-message '(result term not))))
	  (toru (ml-inr r))
	  (t (ml-inl nil)))))))


(defunml (|orb_eval_args_wprops| (props desc expr-args))
    ((tok list) -> (term -> ((term |#| (term list)) -> unit)))

  ;;(setf -props props b desc c expr-args)  (break "oeap")

  (let ((rsps (orb-eval-by-description desc
				       (iexpression-term (iml-term nil t (car expr-args) (cdr expr-args)))
				       'one
				       props)))
    (if (listp rsps)
	(mapc #'interpret-result rsps)
	(interpret-result rsps))
    nil))

(defunml (|orb_eval_args_asynch| (desc posure))
    (term -> ((term |#| (term list)) -> unit))

  ;;(setf a fan b desc c expr-args)  (break "oea")
  (orb-queue-asynch-request desc posure)
  nil
  )

;;let mycomp a = orb_eval_args_asynch nuprl5_library_description_term (nat_ap (begin_ap "myplus") 2);;
;;orb_eval_asynch_with_completion  nuprl5_library_description_term (nat_ap (begin_ap "myplus") 2) mycomp;;
(defunml (|orb_eval_asynch_with_completion| (desc posure comp))
    (term -> ((term |#| (term list)) -> ((term -> unit)  -> unit)))

  ;;(setf -desc desc -posure posure -comp comp)  (break "oea")
  (let ((props (get-transaction-properties)))
    (orb-queue-asynch-request desc posure
			      #'(lambda (result)
				  (set-transaction-properties props)
				  (funmlcall comp result))
			      nil
			      nil
			      props
			      )
    ;;(break "oeaa")
    nil))


(defunml (|orb_queue_asynch_local| (inter-orb-p posure))
    (bool -> ((term |#| (term list)) -> unit))

  ;;(let ((props (get-transaction-properties))))
  ;;(setf -posure posure -comp comp)  (break "oea")
  (orb-queue-local-asynch-request posure nil inter-orb-p)
  nil)

(defunml (|set_server_address| (laddr saddr))
  ((tok list) -> ((tok list) -> unit))

  (environment-set-server-address (find-environment laddr) saddr))
	  

(defunml (|orb_queue_asynch_local_by_address| (inter-orb-p addr posure))
    (bool -> ((tok list) -> ((term |#| (term list)) -> unit)))

  ;;(let ((props (get-transaction-properties))))
  ;;(setf -posure posure -addr addr -inter-orb-p inter-orb-p)  (break "oqalba")
  (orb-queue-local-asynch-request posure nil inter-orb-p addr)
  nil)

;; if desc is for some other process then send request to remote env to queue async req
;; and add a notify to call completion.
(defunml (|orb_eval_asynch_wprops| (inter-orb-p desc props posure comp))
    (bool -> (term -> ((tok list)
		       -> ((term |#| (term list))
			   -> ((term -> unit)  -> unit)))))

  (let ((tprops (get-transaction-properties))
	(env (current-environment)))
    (let ((cdesc (when env (description-of-environment env))))

      ;;(setf -desc desc -cdesc cdesc -posure posure -comp comp)  (break "oea")
      (if (and t (match-descriptions-p desc cdesc))
	  (orb-queue-asynch-request desc posure
				    #'(lambda (result)
					(set-transaction-properties tprops)
					(funmlcall comp result))
				    props
				    inter-orb-p
				    tprops)

	    ;; add notify.
	    ;; send cookie and request to remote to be queued
	    ;; remote will call back with cookie when completed.
	    (let ((cookie (add-notify #'(lambda (result)
					  (set-transaction-properties tprops)
					  (funmlcall comp result)))))
	      ;;(term_ap (bool_ap (begin_ap "orb_eval_asynch") inter-orb-p) desc)
	      (break)
	      
	      )
	    )))
  nil)

(defunml (|orb_exists_idle_link| (desc props))
    (term -> ((tok list) -> bool))

  (let ((a (orb-choose-connected-environment desc props t)))
    (format t "~%orb_exists_idle_link ~a ~a~%" props a)
    ;;(show-ref-blinks '(|obvious|));;
    a))
    


(defunml (|choose_connected_environment| (desc props))
    (term -> ((tok list) -> (tok list)))

  (orb-choose-connected-environment desc props))


(defunml (|orb_queue_asynch| (inter-orb-p desc props posure notify-inter-orb-p comp))
    (bool -> (term -> ((tok list)
		       -> ((term |#| (term list))
			   -> (bool
			       -> ((term -> unit)  -> unit))))))

  ;; should be args and be passed to remote implicitly too. check callers.
  (let ((tprops (get-transaction-properties)))

    (let ((r (send-orb-asynch-queue inter-orb-p
				    (orb-choose-connected-environment desc props)
				    posure
				    (add-notify #'(lambda (result)
						    (set-transaction-properties tprops)
						    (funmlcall comp result))
						notify-inter-orb-p)
				    tprops)))
      (unless (null r)
	(break "oqa nn")
	(raise-error (error-message '(orb_queue_asynch nil not))))))
  nil)

(defunml (|orb_queue_asynch_by_address| (addr posure))
    ((tok list) -> ((term |#| (term list)) -> unit))

  (let ((r (send-orb-asynch-queue nil addr posure)))
    (unless (null r)
      (break "oqa nn")
      (raise-error (error-message '(orb_queue_asynch_by_address nil not)))))
  
  nil)
  
(defunml (|orb_queue_asynch_wo_completion| (fan inter-orb-p desc props posure))
    (tok -> (bool -> (term -> ((tok list)
			       -> ((term |#| (term list))
				   -> unit)))))
    (cond
      ;; use orb_queue_asynch_local
      #|((eql fan 'current)
       (send-orb-asynch-queue inter-orb-p
			      (address-of-environment (current-environment))
			      posure))|#
      ((member fan '(one any))
       (send-orb-asynch-queue inter-orb-p
			      (orb-choose-connected-environment desc props)
			      posure))
      (t (dolist (conn (car (sort-connections-by-properties props (orb-connections desc))))
	   (send-orb-asynch-queue inter-orb-p (address-of-connection conn) posure) )))
  nil)
	  

(defunml (|orb_eval_asynch_iterator| (comp))
   ((term -> (unit |+| (term |#| (term |#| (term list)))))  -> unit)

  ;;(setf a fan b desc c expr-args)  (break "oea")

  (asynch-iterator (current-environment)
		   ;; returns unit + (term # posure)
		   #'(lambda (result)
		       ;; + is cons where car t means inl and car nil means inr.
		       (let ((union-desc-posure (funmlcall comp result)))
			 (when (null (car union-desc-posure))
			   (cdr union-desc-posure))))
		   (ivoid-term))
  nil
  )


(defunml (|new_cookie| (unit) :declare ((declare (ignore unit))))
    (unit -> term)
  (stamp-to-term (new-transaction-stamp)))

(defunml (|add_notify_wcookie| (cookie f))
    (term -> ((term -> unit) -> unit))
  (add-notify-wcookie #'(lambda (a)
			  (funmlcall f a))
		      (term-to-stamp cookie))
  nil)
 
(defunml (|add_notify| (f))
    ((term -> unit) -> term)
  (add-notify #'(lambda (a)
		  (funmlcall f a))))


;; reset asynch may result in notify for non-existent pending req.
;; can this be distinquished from case where notify beats queueing of notify callback hook.
;; could remember dead cookies and fail notify of dead cookie
;; but FTTB, find way of resetting call when asynch reset.
(defunml (|notify_environment| (ea cookie arg))
    ((tok list) -> (term -> (term -> unit)))
  (if (can-notify-p cookie)
      (notify cookie arg)
      (notify-wait-list-push cookie
			     (new-asynch-req (new-ireq-term (current-sequence) nil
							    'asynch
							    nil
							    (iasynch-command-term
							     (iasynch-notify-term t cookie arg)))
					     nil
					     nil
					     (find-environment ea))))
  nil)

(defunml (|notify| (cookie arg))
    (term -> (term -> unit))
  (if (can-notify-p cookie)
      (notify cookie arg)
      (notify-wait-list-push cookie
			     (new-asynch-req (new-ireq-term (current-sequence) nil
							    'asynch
							    nil
							    (iasynch-command-term
							     (iasynch-notify-term t cookie arg)))
					     nil)))
  nil)

(defunml (|notify_cancel| (cookie))
    (term -> unit)

  (let ((notify (assoc (term-to-stamp cookie) *notify-alist* :test #'equal-stamps-pf)))
    (when notify
      (setf *notify-alist* (delete notify *notify-alist*))))

  nil)

;; kludge to make it possible for caller to sift through error message to find cookie to
;; cancel after failure.
(define-primitive |!cookie| () (stamp)) 
(defunml (|with_cookie| (f cookie))
    ((unit -> *) -> (term -> *))

  (with-handle-error-and-message
      (nil  #'(lambda (m)
		(raise-error (error-message '(cookie fail)
					    (icookie-term cookie) m))))
    (funmlcall f nil)))


;; this appears to be a kludge to allow top loop to build !expression terms directly.
;;   - allows !eval_property wrappers to pass refenv to evaluator.
;;  so what would not be a kludge?
;;   pushing refenv arg deeper into call might be a win.
;;   but what is truely appropriate level for refenv.
;;   could push into !expression ie  wrap expression subterm
;;   but effectively that's no real diff.
;;  so I don't have a good answer.
;; this is analogous to local_eval (ie term-> term, no interpretation of result).
(defunml (|orb_eval_expression| (desc expr) :error-wrap-p nil)
    (term -> (term -> term))

   (let ((r (orb-eval-by-description desc expr 'one)))
     ;; (setf -r r) (break "oee")
     r
     ))


(defun orb-eval-args-to-term (infan desc expr-args &optional props)
  (let ((readonlyp (and (eql infan `readonly) t)))
    (let ((fan (if (eql infan `readonly)
		   'any
		   infan)))
      (let ((rsps (orb-eval-by-description desc
					   (iexpression-term (iml-term nil t (car expr-args) (cdr expr-args))
							     nil readonlyp)
					   fan
					   props)))
	;;(setf -rsps rsps)
	(if (listp rsps)
	    (map-list-to-ilist (mapcar #'(lambda (rsp)
					   (or (interpret-result rsp)
					       (raise-error (error-message '(result term not list)))))
				       rsps)
			       (inil-term))
	    (or (interpret-result rsps)
		(raise-error (error-message '(result term not)))))))))


(defunml (|orb_eval_args_to_term| (fan desc expr-args))
    (tok -> (term -> ((term |#| (term list)) -> term)))

  ;;(setf a fan b desc c expr-args)  (break "oeat")
  (orb-eval-args-to-term fan desc expr-args '(main)))

(defunml (|orb_eval_args_to_term_wprops| (props desc expr-args))
    ((tok list) -> (term -> ((term |#| (term list)) -> term)))

  ;;(setf -props props b desc c expr-args)  (break "oeatp")
  (orb-eval-args-to-term 'one desc expr-args props))


(defunml (|orb_eval_command_to_term| (fan desc cmd))
    (tok -> (term -> (term -> term)))

  ;;(setf a fan b desc c cmd)  (break "oeat")

  (let ((rsps (orb-eval-by-description desc
				       (icommand-term cmd)
				       fan)))

    (if (listp rsps)
	(map-list-to-ilist (mapcar #'(lambda (rsp)
				       (or (interpret-result rsp)
					   (raise-error (error-message '(result term not)))))
				   rsps)
			   (inil-term))
	(or (interpret-result rsps)
	    (raise-error (error-message '(result term not)))))))


(defunml (|orb_eval_command| (fan desc cmd))
    (tok -> (term -> (term -> unit)))

  ;;(setf a fan b desc c expr-args)  (break "oeat")

  (let ((rsps (orb-eval-by-description desc
				       (icommand-term cmd)
				       fan)))

    (if (listp rsps)
	(mapc #'(lambda (rsp)
		  (or (interpret-result rsp)
		      (raise-error (error-message '(result term not)))))
	  
	      rsps)
	(interpret-result rsps))
    nil))

    
  

#|
;;ascii
(defunml (|lib_get_pd| (unit) :declare ((declare (ignore unit))))
  (unit -> term)

  (lib-get-pd))
 
(defunml (|lib_get_cterm| (unit) :declare ((declare (ignore unit))))
  (unit -> term)

  cterm)
 
(defun get-persist ()
  (io-db-buffer-flush)
  (let* ((process-list (db-process-list nil t))
	 (logs  (flatten (mapcar #'db-logs-of-process process-list)))
	 (pterm nil))
    (labels ((find-persist (ls)
		(if ls (with-db-input-file (intermediate-stream (car ls) t)
			(do ((found nil found)
			     (term (db-read-term intermediate-stream)
				   (db-read-term intermediate-stream)))
			    ((or pterm (null term))
			     (if pterm
				 (setf log (car ls) pt pterm)
			       (find-persist (cdr ls))))
			    (if (idata-persist-term-p term)
				(setf pterm term)))) (error))))
	    (find-persist logs))))
			      

;;ascii
(defunml (|get_persist| (unit) :declare ((declare (ignore unit))))
  (unit -> term)

  (get-persist))
|#
 

;;;;	RLE TODO : need ability to cleanup hanging environments when remote site crashes.
;;;;	RLE TODO : Prob want to recognize dead stream (unwind) and suspend link.
;;;;	RLE TODO : Need ability to report suspended links.
;;;;	RLE TODO : Need ability to cleanup suspended links.
;;;;	RLE TODO : Need ability of remote to reconnect to a suspended link.

(defunml (|list_environments| (paddr))
    ((tok list) -> ((int |#| (tok list)) list))

  (let ((dbes (filter-db-environments paddr)))
    (mapcar #'(lambda (dbe) (cons (time-of-stamp (cdr dbe)) (car dbe)))
	    dbes)))

(defunml (|list_environments_ugly| (paddr))
    ((tok list) -> (((tok list) |#| ((int |#| tok) |#| (int |#| int))) list))

  (filter-db-environments paddr))


(defunml (|map_oid_to_string| (u n m oid))
    ((tok -> (string -> unit)) -> (int -> (((tok |#| string) list) -> (object_id -> string))))
  (let ((s (stamp-of-oid oid)))
    (let ((pid (process-id-of-stamp s)))
      ;;(setf -s s -pid pid -m m -n n -oid oid) (break "mpoo")
      (or (let ((pids (or (cdr (assoc pid m))
			  (let ((newpids (concatenate 'string
						      "p"
						      (princ-to-string n)
						      "_"
						      )))
			    (funmlcall u pid newpids)
			    newpids)
			  )))
	    (concatenate 'string
			 pids
			 (princ-to-string (sequence-of-stamp s))
			 "_"))))))

;;;; graph funcs.

(defunml (|graph_reduce_proxy| (eq g))
    ((* -> (* -> bool))
     -> ((((object_id |#| (object_id list)) |#| (object_id list)) list)
	 -> ((object_id |#| (object_id list)) list)))

  (graph-reduce-proxy #'(lambda (a b) (funmlcall eq a b))
		      g))

(defunml (|layered_oids| (g l))
    (((object_id |#| (object_id list)) list) -> ((object_id list) -> ((object_id list) list)))
  ;;(setf -g g -l l) (break "lo")
  (oid-list-layers g l #'(lambda (x) (declare (ignore x)) t)))

(defunml (|layered_oids_cycles| (g l))
    (((object_id |#| (object_id list)) list) -> ((object_id list) -> (object_id list)))

  (mlet* (((l c) (oid-list-layers g l #'(lambda (x) (declare (ignore x)) t)) (declare (ignore l))))

	 c))

(defunml (|graph_cycle| (g))
    (((object_id |#| (object_id list)) list) -> (object_id list))

  (graph-cycle g))


;; some rigamorole to force call of make...func with ap of graph arg.
(defun make-ml-graph-dependency-closure-func (g)
  (let ((f (make-graph-dependency-closure-func g)))
    (mlclosure #'(lambda (filter seeds)
		   (funcall f #'(lambda (o) (funmlcall filter o)) seeds))
	       2)))
  
(dml |ddg_graph_dependency_closure_func|
     1
     make-ml-graph-dependency-closure-func
     (((object_id |#| (object_id list)) list)
      -> ((object_id -> bool) -> ((object_id list) -> (object_id list))))
     )


(defunml (|graph_flat| (g))
    (((object_id |#| (object_id list)) list) -> (object_id list))
  
  ;;(setf -g g) (break "gf")
  (mapcan #'(lambda (x) x) (graph-layers g)))

(defunml (|inconsistent_graph_objects| (g to))
    (((object_id |#| (object_id list)) list) ->
     ((object_id -> (object_id -> bool)) -> (object_id list)))

  (let ((l (mapcar #'car g))
	(po (inconsistent-objects-po g)))
    ;;(setf -po po -g g) (break "io")
    (setf -r (poto-inconsistents po #'(lambda (a b) (funmlcall to a b)) l))

    ;;(break "io2")
    -r))

(defunml (|fast_remove_duplicate_oids| (l))
    ((object_id list) -> (object_id list))

    (fast-remove-duplicate-oids l))

(defunml (|fast_diff_oids| (l ll))
    ((object_id list) -> ((object_id list) -> (object_id list)))

    (fast-diff-oids l ll))


% preserves order of first arg. %
(defunml (|fast_intersect_oids| (l ll))
    ((object_id list) -> ((object_id list) -> (object_id list)))

  ;;(format t "fast_intersect_oids called ~a ~a ~%" (length l) (length ll))

  (let ((ohash (new-oid-table)))

    (let ((acc nil))
      (dolist (oid ll)
	(unless (hashoid-get ohash oid)
	  (hashoid-set ohash oid t)))

      (dolist (oid l)
	(let ((g (hashoid-get ohash oid)))
	  (when g
	    (unless (eql 'acc g)
	      (push oid acc)
	      (hashoid-set ohash oid 'acc)))))

      (nreverse acc))))

(defunml (|fast_filter_graph| (listorderp l g))
    (bool ->
     ((object_id list) ->
      (((object_id |#| (object_id list)) list) ->
       ((object_id |#| (object_id list)) list))))

  (format t "Fast filter graph called~%")

  (let ((ohash (new-oid-table)))

    (if listorderp
	(progn
	  (dolist (e g)
	    (hashoid-set ohash (car e) e))

	  (mapcan #'(lambda (oid)
		      (let ((e (hashoid-get ohash oid)))
			(when e
			  (list e))))
		  l))
	
	(progn
	  (dolist (oid l)
	    (hashoid-set ohash oid t))

	  (mapcan #'(lambda (e)
		      (when (hashoid-get ohash (car e))
			(list e)))
		  g)))))

(defunml (|fast_thin_graph| (reducekinds l g))
    ((tok list) ->
     ((object_id list) ->
      (((object_id |#| (object_id list)) list) ->
       ((object_id |#| (object_id list)) list))))

  (format t "Fast thin graph called ~a ~a ~%" (length l) (length g)) 
  (let ((ohash (new-oid-table)))

    (if (null reducekinds)
	;; remove edges not in list.
	(let ((acc nil))
	  (dolist (oid l)
	    (hashoid-set ohash oid t))
	  (dolist (e g)
	    (when (hashoid-get ohash (car e))
	      (push (cons (car e)
			  (filter #'(lambda (d) (hashoid-get ohash d)) (cdr e)))
		    acc)))
	  (format t "Fast thin graph done~%")
	  (nreverse acc))

	(let ((khash (new-oid-table))
	      (esum 0))
	  ;;(setf -khash khash -ohash ohash)

	  (dolist (e g)
	    (when (member (kind-of-ostate (car e)) reducekinds)
	      (incf esum (length (cdr e)))
	      (hashoid-set khash (car e) (or (cdr e) t))))

	  (format t "  graph hashed ~a ~a~%" (length g) esum) 
	  (dolist (oid l)
	    (unless (hashoid-get khash oid)
	      (hashoid-set ohash oid t)))

	  (format t "  list hashed~%")
	  (setf esum 0)
	  (let ((acc nil))
	    (dolist (e g)
	      (when (hashoid-get ohash (car e))
		(push (cons (car e)
			    (let ((newe (mapcan #'(lambda (o)
						    (let ((ks (hashoid-get khash o)))
						      (if ks
							  (if (eql t ks) nil (copy-list ks))
							  (list o))))
						(cdr e))))
			      (incf esum (length newe))
			      newe))
		      acc)))
	    
	    (format t "  graph done ~a ~%" esum)
	    (nreverse acc))))))





(defunml (|lookup_re_index_kludge| (term) :error-wrap-p nil)
    (term -> object_id)

  ;;(break "lreik")
  (or (cdr (assoc term *re-index-kludge-assoc* :test #'compare-terms-p))
      (breakout evaluation '|lookup_re_index_kludge|)
  ))

(defunml (|is_re_index_kludge| (oid) :error-wrap-p nil)
    (object_id -> bool)

  ;;(break "ireik")
  (and (member oid *re-index-kludge-assoc* :key #'cdr :test #'equal-oids-p)
       t)
  )


(defunml (|with_make_ref_environment| (makef term))
    ((object_id -> (term -> unit)) -> (term -> ((term -> *) -> *)))

  ;;(setf -term term)   (break "wmre2")
  (mlclosure #'(lambda (f)

		 (when (member term *re-index-kludge-assoc* :test #'compare-terms-p :key #'car)
		   (raise-error (error-message '(|with_make_ref_environment|) term)))

		 (if (ioid-term-p term)
		     (funmlcall f term)
		     (let ((re (get-temp-re-obid term)))
		       (unwind-protect
			    (progn (funmlcall makef re term)
				   (funmlcall f term))
			 (ref-env-remove re)
			 (free-temp-re-obid term)))))
	     1))



(defunml (|double_print| (b))
    (bool -> int)

  (setf *max-print-depth*
	(if b
	    (* 2 *max-print-depth*)
	    (floor *max-print-depth* 2)))
  

  *max-print-depth*)


(defunml (|inject_trace_message| (action data))
    (tok -> (term -> unit))

  ;;(break "itm")
  (when (io-trace-file-p)
    (write-io-trace-file action data))
  
  (values))

(defunml (|export_term| (oidenprefix encode-bindings term))
    ((unit + string) -> (bool -> (term -> term)))

  (if (ml-isl oidenprefix)
      (www-term-out term encode-bindings)
      (let ((*obid-encoding-prefix* (ml-outr oidenprefix)))
	(www-term-out term encode-bindings))))

(defunml (|export_obid_encoding| (unit) :declare ((declare (ignore unit))))
    (unit -> term)
  (obid-encoding-table-to-term (current-environment)))


(defunml (|import_obid_encoding| (term))
    (term -> unit)
  (term-to-obid-encoding-table term env))


(defunml (|kill_bus_link| (addr))
    ((tok list) -> unit)

  (let ((blink (bus-link-of-environment-address addr)))
    (unless blink
      (raise-error (error-message (append '(kill-bus-link link not) addr))))

    (kill-bus-link blink (itext-term "kill_bus_link")))

  (values))

(defunml (|liblib_eval| (term-args))
    ((term |#| (term list)) -> term)

  (orb-remote-lib-eval (car term-args) (cdr term-args)))

(defunml (|config_data| (k))
    (tok -> term)

  (let ((vs (get-config-data k)))
    (when (> (length vs) 1)
      (raise-error (error-message '(config_data multiple values))))
    
    (let ((v (car vs)))
      (cond
	((null v) (raise-error (error-message '(config_data null value))))
	((integerp v) (iint-term  v))
	((symbolp v) (itoken-term v))
	((stringp v) (itext-term v))
	(t (raise-error (error-message '(config_data unexpected type) k)))))))

(defunml (|config_data_list| (k))
    (tok -> (parameter list))

  (let ((l (get-config-data k)))
    (unless (null l)
      (let ((v (car l))
	    vcheck
	    vparm)
	(cond
	  ((null v) (raise-error (error-message '(config_data list null value))))
	  ((and (integerp v) (>= v 0))
	   (setf vcheck #'integerp vparm #'natural-parameter))
	  ((symbolp v) (setf vcheck #'symbolp vparm #'token-parameter))
	  ((stringp v) (setf vcheck #'stringp vparm #'string-parameter))
	  (t (raise-error (error-message '(config_data list unexpected type) k))))
	
	(dotimeslist (i v l)
		     (unless (funcall vcheck v)
		       (raise-error (error-message '(application config-data list type) (list i)))))

	(mapcar vparm l)))))

(defunml (|applications_config_data| (k))
    (tok -> ((tok |#| string) list))

  (let ((v (get-config-data k)))
    (dotimeslist (i l v)
		 (unless (eql (length l) 2)
		   (raise-error (error-message '(application config-data element length) (list (length l) i))))

		 (unless (symbolp (car l))
		   (raise-error (error-message '(application config-data token not) (list i))))
		 (unless (stringp (cadr l))
		   (raise-error (error-message '(application config-data string not) (list i)))))
    (mapcar #'(lambda (e)
		(cons (car e) (cadr e)))
	    v)))


(defunml (|full_gc| (unit))
    (unit -> unit)
  #+cmu(gc :full t)
  #+allegro(excl:gc t)
  #+(and (not cmu) (not allegro))(gc)
  )