
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;; -docs- (mod com data)
;;;;
;;;;	<disp-export-term>	: !dforms(<sig>; <dform-sexpr>)
;;;;
;;;;	<sig>			: <term>	
;;;;	
;;;;	** all model terms of dforms in sexpr will have same term-sig
;;;;	** as <sig> term.
;;;;	
;;;;
;;;;	<prec-export-term>	: <prec-expr>
;;;;	
;;;; -doce-




;;;;	Formats and attributes are terms. 	
;;;;
;;;;	Some checking of dform syntax is performed by lib. However, the
;;;;	edd component receiving the dform def is arbiter of correctness
;;;;	for a dform spec. Not all edd components will recognize all dforms.
;;;;
;;;;	  - !dform(0;0;0)
;;;;	  - all models have same term-sig.
;;;;	  
;;;;	
;;;;	One might expect edd to check dform prior to installing in lib.
;;;;	A dform which is active according to lib does not mean it is usable by edd.
;;;;	



;;; substance

;;  might expect
;;
;;

;; todo need imports for substance classes.
(defclass display-substance (substance)
  ((model :reader model-of-display-substance
	  :writer set-display-substance-model
	  :initform nil
	  :initarg model)

   (name :reader name-of-display-substance
	 :writer set-display-substance-name
	 :initform nil
	 :initarg name)
   ))


(defun display-substance (term properties model)
  (make-instance 'display-substance 'term term 'properties properties 'model model))

(defun display-substance-wname (term properties model name)
  (make-instance 'display-substance 'term term 'properties properties 'model model 'name name))

(define-primitive |!display_substance| ((token . name)) (model))


(defmethod data-import ((substance display-substance) super)
  (let ((term (call-next-method substance super)))
    ;; at one point term was simply the model term, so need to check for
    ;; compatability with old data.
    (if (idisplay-substance-term-p term)
	(progn
	  (set-display-substance-name (name-of-idisplay-substance-term term) substance)
	  (set-display-substance-model (model-of-idisplay-substance-term term) substance))
	(set-display-substance-model term substance))))

(defmethod data-export ((substance display-substance) sub)
  (call-next-method substance
		    (idisplay-substance-term (name-of-display-substance substance)
					     (model-of-display-substance substance))))


(define-primitive |!dforms| () (sig list))


;;;
;;; Dforms
;;;

(define-primitive |!dform| () (attributes formats model))
(define-primitive |!dform_cons| () (car cdr))


(defparameter *floatdown-variable* (get-display-meta-variable-id "FLOATDOWN"))
(defparameter *floatup-variable* (get-display-meta-variable-id "FLOATUP"))
(defparameter *iterate-variables* (list (get-display-meta-variable-id "ITERATE")
					(get-display-meta-variable-id "#")))

(defun floatdown-p (v) (eql v *floatdown-variable*))
(defun floatup-p (v) (eql v *floatup-variable*))
(defun iterate-p (v) (member v *iterate-variables*))
    


(define-primitive |!template| ((string . id)))

(defun meta-id-of-itemplate-term (term)
  (get-display-meta-variable-id (id-of-itemplate-term term)))

;;;
;;; Dform Attributes
;;;

(define-primitive |!dform_attr_cons| () (car cdr))

;; Conditions

(define-primitive |!dform_cond_expr| () (expression))


;; Child hiding.

(define-primitive |!dform_hidden_cond_expr| () (expression))



    
;; checks for syntactically valid condition expression term.
(defun icond-expr-term-p (term)
  (declare (ignore term))

  ;; there is not much that can be checked for. We could check for valid
  ;; text terms without term literals. 
  ;;
  ;;(raise-error (error-message '(cond-expr syntax) term))

  t
  )


;; parens and precedence 

(define-primitive |!dform_precedence_exception|)
(define-primitive |!dform_precedence_sensitivity|)

(define-primitive |!dform_precedence_passthru|)

(define-primitive |!dform_precedence_injection| () (pointer))


;; Iterate

(define-primitive |!dform_families| () (list))

(define-primitive |!dform_family| ((string . name)))

(define-primitive |!dform_family_cons| () (car cdr))

;; Misc

(define-primitive |!dform_macro_name|  ((string . string)))

(define-primitive |!dform_name| ((string . name)))

(define-primitive |!dform_text| ((string . text) (natural . length)))

;;;
;;; Dform Formats.
;;;


(define-primitive |!dform_label_wrap| ((token . label)) (formats))


;;(define-primitive |!dform_depth| ((token . type) (n . amt)))
(defun idform-depth-term-p (term)
  (and (eql '|!dform_depth| (id-of-term term))
       (null (bound-terms-of-term term))
       (let* ((parameters (parameters-of-term term))
	      (p (car parameters)))
	 (and p
	      (token-parameter-p p)
	      (let ((parameters (cdr parameters)))
		(or (null parameters)
		    (and (natural-parameter-p (car parameters))
			 (null (cdr parameters)))))))))

(defun type-of-idform-depth-term (term)
  (value-of-parameter (car (parameters-of-term term))))

(defun amt-of-idform-depth-term (term)
  (let ((p (cadr (parameters-of-term term))))
    (if p
	(value-of-parameter p)
	(raise-error (error-message '(!dform_depth amt not) term)))))

(defun depth-type-value-p (type) (member type '(nodepth new min max + -)))

;;; format_cons used to be lhs_cons. It can be changed by migration,
;;; however migration should also add an abs to expand lhs_cons to format_cons.
(define-primitive |!dform_format_cons| () (car cdr))


(define-primitive |!dform_space|)


(define-primitive |!dform_push| ((n . amt)))
(define-primitive |!dform_pop|)

;;;
;;; Break formats.
;;;

;;lal added || 5/22
(defun break-control-value-p (x) (member x '(linear break soft multilinear nil ||)))

(define-primitive |!dform_break_control| ((token . type)))

;;(define-primitive |!dform_break| ((string . string) (string . prefix) (string . suffix)))
(define-primitive |!dform_cut_break|)

(defun idform-break-term-p (term)
  (and (eql '|!dform_break| (id-of-term term))
       (null (bound-terms-of-term term))
       (let* ((parameters (parameters-of-term term))
	      (p (car parameters)))
	 (and p
	      (string-parameter-p p)
	      (let ((parameters (cdr parameters)))
		(or (null parameters)
		    (and (string-parameter-p (car parameters)))
		    (let ((parameters (cdr parameters)))
		      (or (null parameters)
			  (and (string-parameter-p (car parameters))
			       (null (cdr parameters)))))))))))

(defun string-of-idform-break-term (term)
  (value-of-parameter (car (parameters-of-term term))))
			  
(defun prefix-of-idform-break-term (term)
  (let ((p (cadr (parameters-of-term term))))
    (if p
	(value-of-parameter p)
	"")))

(defun suffix-of-idform-break-term (term)
  (let ((p (caddr (parameters-of-term term))))
    (if p
	(value-of-parameter p)
	"")))

(defun idform-break-term (s &optional prefix suffix)
  (instantiate-term
   (instantiate-operator '|!dform_break|
			 (cons (instantiate-parameter s *string-type*)
			       (when prefix
				 (cons (instantiate-parameter prefix *string-type*)
				       (when suffix
					 (list (instantiate-parameter suffix *string-type*)))))))))

;;;
;;; child formats
;;;


;; includes parameters. currently no applicable attributes but we should allow for it.
(define-primitive |!dform_variable_child| ((string . id) (string . descriptor)) (attributes))

(defun meta-id-of-idform-variable-child-term (term)
  (get-display-meta-variable-id
   (id-of-idform-variable-child-term term)))


(define-primitive |!dform_library_child|  ((string . pointer)) (attributes))

(define-primitive |!dform_constant_child| () (attributes term))

(defun attributes-of-idform-child-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))


;;;
;;; child attributes
;;;

(define-primitive |!dform_child_attr_cons| () (car cdr))


(defun parentheses-relation-value-p (x) (member x '(all equal less)))

(define-primitive |!dform_child_parentheses| ((token . relation)) (pointer (1 . formats)))

;;(define-iprimitive |!dform_child_width| ((natural . width)))


;;;
;;; Precedence
;;;


(define-primitive |!precedence_label| ((token . token)))

(define-primitive |!precedence_object| ((oid . oid)) ())

(define-primitive |!precedence_ordered| nil (left right))
(define-primitive |!precedence_equal| nil (left right))
(define-primitive |!precedence_unrelated| nil (left right))

(define-primitive |!dform_address| ((string . id) (oid . oid)))

;; (define-primitive |!dform_address| ((string . id)))
(defun relative-idform-address-term-p (term)
  (let ((sig (term-sig-of-term term)))
    (and (eql (car sig) (car *idform-address-term-sig*))
	 (equal (caadr sig) (caadr *idform-address-term-sig*))
	 (equal (arities-of-term-sig sig)
		(arities-of-term-sig *idform-address-term-sig*)))))

(defun idform-precedence-pointer-term-p (term &optional (void-ok-p nil))
  (or (and (idform-address-term-p term)
	   (real-itext-term-p term)
	   (oid-p (oid-of-idform-address-term term)))
      (and (relative-idform-address-term-p term)
	   (real-itext-term-p term))
      (and (iprecedence-label-term-p term)
	   (real-itext-term-p term))
      (and (ivoid-term-p term) void-ok-p)))


(defun term-sig-to-model-term (term-sig)
  (instantiate-term
   (instantiate-operator (id-of-term-sig term-sig)
			 (let ((i 0))
			   (mapcar #'(lambda (type-id)
				       (let ((type (lookup-typeid type-id)))
					 (unless type
					   (raise-error
					    (error-message '(termsig term typeid) type-id)))
					 
					 (instantiate-parameter (get-display-meta-variable-id
								 (with-output-to-string (s)
								   (princ "p" s)
								   (princ (incf i) s)))
								type)))
				   (parameters-of-term-sig term-sig))))

   (let ((i 0))
     (mapcar #'(lambda (arity)
		 (instantiate-bound-term (itemplate-term (with-output-to-string (s)
							   (princ "t" s)
							   (princ (incf i) s)))
					 (let ((limit (1+ arity))
					       (bindings nil))
					   (do ((j 1 (1+ j)))
					       ((= j limit))
					     (push (get-display-meta-variable-id
						    (with-output-to-string (s)
						      (princ "b" s)
						      (princ i s)
						      (princ "v" s)
						      (princ j s)))
						   bindings))
					   (nreverse bindings))))
	     (arities-of-term-sig term-sig)))))



