
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	
;;;;	Client view of library is a set of lookup tables.
;;;;	 
;;;;	A table contains definitions.
;;;;
;;;;	Transactions supply a static view of the tables
;;;;	  - only modifications caused by the transaction are visible during the transaction.
;;;;	    In other words, modifications made by other transactions are not visible by a
;;;;	    a transaction.
;;;;	  - commits are atomic. Once modifications are committed they are visible
;;;;	    to future transactions.
;;;;	  - a transaction may commit without exiting, ie a transaction may commit
;;;;	    multiple times.
;;;;	  - Transactions need never fail, however, if desired commit will fail the 
;;;;	    transaction when there is an object which has been modified since the
;;;;	    committing transaction started.
;;;;
;;;;	Commits are serializable.
;;;;
;;;;
;;;;	Library
;;;;	  - transaction : library -> (library_handle -> unit) -> unit
;;;;
;;;;	libary_handle
;;;;	  lemma_lookup : ID -> term {sequent} ??
;;;;	  proof_store : ID -> int list -> term -> unit
;;;;	  term_lookup : ID -> term
;;;;

;;;;	Implementation details :
;;;;	
;;;;	Lib serves transactions stamps. Ie lib must be called to provide
;;;;	a view of the library. It would be possible to provide a lightweight
;;;;	transaction which did not allow modification of library and which
;;;;	was limited to accessing local data. Ie no communication needed with
;;;;	lib. Browsing.
;;;;
;;;;	transaction stamp : transaction id # time 
;;;;	  - transaction id is unique.
;;;;	      * as unique as pratical.
;;;;	  - time is unique within process. If two transactions stamps generated by
;;;;	    the same lib process have distinct ids they will have distinct times and
;;;;	    the stamps can then be ordered by time.
;;;;
;;;;	All relevant data is cached locally. Lib provides broadcasts to
;;;;	maintain local cache.
;;;;
;;;;	For Each transaction lib sends begin, update, and end broadcasts, all
;;;;	broadcasts for a transaction will have same transaction id.
;;;;
;;;;	All commit broadcasts generated by a single commit will have identical transaction stamp.
;;;;
;;;;	All transactions emit begin/end broadcasts. However, begin should be
;;;;	queued until some other io queued. Then if no intervening messages
;;;;	between begin and end the begin/end messages can be discarded.
;;;;	Or if only broadcasts between begin and end then they can be
;;;;	compressed. In fact, even if multiple transaction broadcasts mixed on queue
;;;;	they can be serialized and compressed at queue end.
;;;;	
;;;;	

;;;;	lookup table maintenance :
;;;;
;;;;	Lookup tables are hash tables.
;;;;	Hash table entries are parameterized by transaction stamps.
;;;;	  - pending modifications : visible only to transaction causing modification.
;;;;	  - committed modifications : if no pending mod for transaction, then most recent
;;;;	    commit prior to transaction start is visible.
;;;;	

;;;;	Simple entry model : entry is a stack of triples : (<data> <tag> list <committed?>
;;;;	  - start transaction with tag t :  atomically tag the latest committed entry.
;;;;	  - during transaction :
;;;;	      * modify: push (<data>, [t], false)
;;;;	      * lookup: find first triple with tag t.
;;;;	  - commit modification : find oldest uncommitted triple with tag t and set committed? to true.
;;;;	  - unwind mofification : remove newest tagged triple.
;;;;	  - end transaction : remove t from all triples in all entry stacks.
;;;;	  - garbage collection : for each entry stack thin all untagged triples from stack and
;;;;	    then place newest(last) of the untagged at bottom of stack.
;;;;	
;;;;
;;;;	Efficient entry model :
;;;;	  - lazy tagging :
;;;;	      * tag modifications and commits.
;;;;	      * use time of transaction stamps to find appropriate committed data.
;;;;	  - collect at after end of each transaction but only examine entries
;;;;	    modified by transaction.



;;;;	** transaction commit broadcast is atomic and contains stamp.
;;;;	  - allow for begin/commit/end as single atomic broadcast.
;;;;	stale committed collected by end-transaction processing.
;;;;	

;;;;	Library transaction broadcast invariants :
;;;;	
;;;;	Delete Oid will match Insert Oid of insert deleted.
;;;;	There will not be contiquous deletes or inserts for same oid.
;;;;	A visible delete is equivalent to absence of a value.

;;;;	Broadcast/Transaction observations:	
;;;;	
;;;;	Commit broadcasts should be treated atomically.
;;;;	All transaction broadcasts happen between a begin/end broadcast, 
;;;;	allowing cleanup/rehashing to be triggered by end broadcast.
;;;;
;;;;	As long as there are no other transmissions, Library may collect broadcasts
;;;;	and all broadcasts for a transaction in a single message.
;;;;
;;;;
;;;;	<definition>	: the data.
;;;;
;;;;	
;;;;	Tent : Transaction ENTry.
;;;;	  - holds definitions in a hashtable or assoc list.
;;;;	  - lookup parameterized by stamp and/or OID
;;;;	  - there may be more than one tent containing similar definitions.
;;;;	
;;;;
;;;;	<tent>		: <committed>, <pending>
;;;;
;;;;	<committed>	: <s-entry> list
;;;;	<pending>	: <s-entry> list
;;;;
;;;;	<s-entry>	: <s-entry-i> | <s-entry-d>
;;;;	
;;;;	<s-entry-i>	: <stamp>, <int{sequence}>, <oid>, <data>, <deletes>
;;;;	<s-entry-d>	: <stamp>, <int{sequence}>, <oid>
;;;;
;;;;	** <s-entry-i> created by insert.
;;;;	** <s-entry-d> created by delete.
;;;;
;;;;	An s-entry is visible wrt a transaction stamp if the s-entry was created by the
;;;;	transaction or the commit stamp of the s-entry is older than the stamp and
;;;;	there is no later committed s-entry which is visible.
;;;;	
;;;;	visible-p (<s-entry> <stamp{begin}>) : BOOL
;;;;	
;;;;	Pending s-entries stamp should be stamp of broadcast causing insert/delete.
;;;;	Not necessarily, pending stamps never compared with inequality just member,
;;;;	so any stamp of transaction ok. Ie, begin stamp of transaction state suitable.
;;;;	
;;;;
;;;;	new-tent () : <tent>
;;;;
;;;;	tent-lookup (<tent> <stamp{begin}>)) : <data>
;;;;	  - returns data from visible <s-entry-i>
;;;;
;;;;	tent-lookup-woid (<tent> <stamp{begin}> <oid>)) : <s-entry-i>
;;;;	  - returns first visible s-entry-i for oid.
;;;;
;;;;	tent-contains-oid-p (<tent> <stamp{begin}> <oid>)) : BOOL
;;;;	  - returns first visible s-entry-i for oid.
;;;;
;;;;	tent-contains-committed-oid-p (<tent> <stamp> <oid>)) : <s-entry-i>
;;;;	  - returns true if contains visible comitted s-entry-i for oid.
;;;;	    Used by commit to find entry to delete.
;;;;
;;;;	tent-contains-pending-p (<tent> <stamp{begin}> <int{sequence}>) : BOOL
;;;;	  - returns true if contains pending visible s-entry with same sequence.
;;;;	    Only pending entries for same transaction will be visible.
;;;;	    Used to find s-entry to undo.
;;;;
;;;;	tent-contains-visible-p (<tent> <stamp>) : <s-entry>
;;;;	  - returns true if contains visible s-entry-i.
;;;;	    Used to find tent to insert into.
;;;;
;;;;	tent-insert (<tent> <stamp> <int{sequence}> <oid> <data>)	: <tent>
;;;;	tent-delete (<tent> <stamp{begin}> <int{sequence}> <oid>)	: <tent>
;;;;	  * if oid not-applicable use t. If you then make a call requiring an oid
;;;;	    say hello to lisp debugger.
;;;;
;;;;	tent-undo-s-entry (<tent> <stamp> <int{sequence}>) : (values <tent> <s-entry>)
;;;;	  * returns s-entry to allow caller to access undone data and oid.
;;;;
;;;;	tent-undo (<tent> <stamp> <int{sequence}>) : <data>
;;;;	  * returns s-entry to allow caller to access undone data and oid.
;;;;
;;;;	tent-commit (<tent> <stamp> <int{sequence}>) : <data>
;;;;	  * updates s-entry stamp to be commit stamp.
;;;;
;;;;	tent-collect (<tent> <stamp> list{active})
;;;;	  - remove all insert/delete pairs such that delete is visible to all
;;;;	    active transactions.
;;;;
;;;;	transaction-end :
;;;;	  - atomically remove from active list and then collect all modified
;;;;	    by transaction.
;;;;	

;;;
;;;	FTTB : include new term access functions.
;;;
 

(define-primitive |!oid| ((oid . oid)))

(defun ioids-term (oids)
  (instantiate-term
   (instantiate-operator *ioid* (mapcar #'oid-parameter oids))))

(defun oids-of-ioids-term (term)
  (mapcar #'value-of-parameter-f (parameters-of-term term)))

(defun ioids-term-p (term)
  (and (eql *ioid* (id-of-term term))
       (null (bound-terms-of-term term))
       (forall-p #'oid-parameter-p (parameters-of-term term))))
  

(define-primitive |!dependency| ((oid . oid)) (objc data))


#|
(defvar *heritage-term-embedding-opid* '|!!uGh|)

(defun leaves-of-term (term)
  (let ((id (id-of-term term)))
    (if (eql id *heritage-term-embedding-opid*)
	(parameters-of-term term)
	;; TODO later make more efficient by adding leaves to term structure (but not cons struct).
	(cons (instantiate-parameter id *token-type*)
	      (parameters-of-term term)))))
|#



(defun transaction-less-than (commit other)
  ;; null other, other is null when local has requested server to start
  ;; transaction but transaction has not yet begun. 
  (if (or (null other) (dummy-transaction-stamp-p other))
      t
      (if (eql (process-id-of-stamp commit) (process-id-of-stamp other))
	  (transaction-< commit other)
	  (tent-order-less-than commit other))))



(defclass tent ()
  (
   (pending
    :initform nil
    :reader pending-of-tent
    :writer set-tent-pending
    )

   (committed
    :initform nil
    :reader committed-of-tent
    :writer set-tent-committed
    )
   ))

(defun tent-empty-p (tent)
  (and (null (pending-of-tent tent))
       (null (committed-of-tent tent))))

(defun new-tent ()
  (make-instance 'tent))

(defclass s-entry ()
  ((stamp :reader stamp-of-s-entry
	  :writer set-s-entry-stamp
	  :initarg stamp)
   (sequence :reader sequence-of-s-entry
	     :initarg sequence)
   (oid :reader oid-of-s-entry
	:initarg oid)
   (data :reader data-of-s-entry
	 :initarg data
	 :writer set-s-entry-data)
   (note :reader note-of-s-entry
	 :initarg note)
   ))

(defclass s-entry-i (s-entry)
  ((deletes :reader deletes-of-s-entry
	    :writer set-s-entry-deletes
	    :initform nil)))


(defun new-s-entry (stamp seq oid data &optional note)
  (make-instance 's-entry-i 'stamp stamp 'data data 'sequence seq 'oid oid 'note note))

(defun new-s-entry-d (stamp seq oid &optional note)
  (make-instance 's-entry 'stamp stamp 'sequence seq 'oid oid 'note note))

(defun s-entry-i-visibly-deleted-p (tent sent stamp)
  (let ((deletes (deletes-of-s-entry sent)))
    (and deletes
	 (exists-p-optimized (dsent deletes)
			     (let ((dstamp (stamp-of-s-entry dsent)))
			       (or (in-transaction-p dstamp stamp)
				   (and (member dsent (committed-of-tent tent))
					(transaction-less-than dstamp stamp))))))))

(defun committed-s-entry-i-visible-p (tent sent stamp)
  (and (or (transaction-less-than (stamp-of-s-entry sent) stamp)
	   (in-transaction-p (stamp-of-s-entry sent) stamp))
       (typep sent 's-entry-i)
       (not (s-entry-i-visibly-deleted-p tent sent stamp))
       t))


;; returns first data value of undeleted s-entry with applicable stamp.
;; PERF : learned that find-first/exists-p on nil still creates closure
;;   so checkinf for nil args first is a win.
;;   also labels instead of closures somtimes helps.
;;   labels inside a let though ie : (let ((x a)) (labels ((aux ()))))
;;    are not advantageous?
(defmethod tent-lookup ((tent tent) stamp &optional nil-ok-p p)
  ;;(setf a tent b stamp) (break "tl")
      (or (let ((pending (pending-of-tent tent)))
	    (and pending
		 ;;(find-first #'pending-aux pending)
		 (find-first-optimized
		  (sent pending)
		  (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
			     (typep sent 's-entry-i)
			     (let ((deletes (deletes-of-s-entry sent)))
			       (not (and deletes
					 (exists-p-optimized (sent deletes)
							     (in-transaction-p (stamp-of-s-entry sent) stamp))
					 ))))
			  
		    (when (typep sent 's-entry-i)
		      (let ((d (data-of-s-entry sent)))
			(if (null p) d
			    (when (funcall p d)
			      ;;(setf -d d) (break "fu")
			      d)))))
		  )))
	  (let ((committed (committed-of-tent tent)))
	    (and committed
		 (find-first-optimized
		  (sent committed)
		  (when (committed-s-entry-i-visible-p tent sent stamp)
		    (when (typep sent 's-entry-i)
		      (let ((d (data-of-s-entry sent)))
			(if (null p) d
			    (when (funcall p d)
			      ;;(setf -d d) (break "fu")
			      d))))))))
	  (unless nil-ok-p
	    ;;(setf a tent b stamp) (break "tln")
	    (raise-error (error-message '(tent lookup none))))))


;; finds s-entry with stamp and seq.
(defmethod tent-contains-pending-p ((tent tent) stamp seq)
  ;;(setf a tent b stamp) (break "tf")
  (find-first #'(lambda (sent)
		  (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
			     (eql seq (sequence-of-s-entry sent)))
			  
		    t))
	      (pending-of-tent tent)))

(defmethod tent-contains-committed-oid-p ((tent tent) stamp oid)
  ;;(setf a tent b stamp) (break "tf")
  (find-first #'(lambda (sent)
		  (when (and (let ((dstamp (stamp-of-s-entry sent)))
			       (or (in-transaction-p dstamp stamp)
				   (transaction-less-than dstamp stamp)))
			     (equal-oids-p oid (oid-of-s-entry sent)))
		    t))
	      (committed-of-tent tent)))


(defmethod tent-contains-visible-p ((tent tent) stamp)
  (or (find-first-optimized
       (sent (pending-of-tent tent))
       (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
		  (typep sent 's-entry-i)
		  (not (exists-p-optimized (sent (deletes-of-s-entry sent))
					   (in-transaction-p (stamp-of-s-entry sent) stamp))))
	 t))
      (find-first-optimized
       (sent (committed-of-tent tent))
       (committed-s-entry-i-visible-p tent sent stamp)) ))


;; returns first s-entry with applicable stamp and equal oid.
(defun tent-lookup-woid (tent stamp oid)
  ;;(setf a tent b stamp) (break "tlw")
  (or (find-first #'(lambda (sent)
		      (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
				 (equal-oids-p oid (oid-of-s-entry sent)))
			sent))
		  (pending-of-tent tent))
      (find-first #'(lambda (sent)
		      (when (and (or (in-transaction-p (stamp-of-s-entry sent) stamp)
				     (transaction-less-than (stamp-of-s-entry sent) stamp))
				 (equal-oids-p oid (oid-of-s-entry sent)))
			sent))
		  (committed-of-tent tent))
      (progn (break "tlw")
	     (system-error (error-message '(tent lookup woid none))))))
		      
(defun tent-contains-oid-p (tent stamp oid)
  (or (find-first #'(lambda (sent)
		      (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
				 (equal-oids-p oid (oid-of-s-entry sent)))
			t))
		  (pending-of-tent tent))
      (find-first #'(lambda (sent)
		      (when (and (transaction-less-than (stamp-of-s-entry sent) stamp)
				 (equal-oids-p oid (oid-of-s-entry sent)))
			t))
		  (committed-of-tent tent))))


(defvar *tent-debug* nil)
(defvar *tent-history* nil)

(defun sent-show (s &optional tent)
  (let ((o (oid-of-s-entry s)))
    (let ((oidseq (when o (sequence-of-stamp (stamp-of-oid o)))))
      (when (and oidseq tent)
	(unless (member oidseq *tent-history* :key #'car)
	  (push (cons oidseq tent) *tent-history*)))

      (format t "~%    ~a ~a Seq : ~a, OidSeq : ~a~%        Stamp : ~a"
	      (if (typep s 's-entry-i) "I" "D")
	      (transaction-of-stamp (stamp-of-s-entry s))
	      (sequence-of-s-entry s)
	      oidseq
	      (stamp-of-s-entry s))
      (if (typep s 's-entry-i)
	  (format t "~%      ~a" (length (deletes-of-s-entry s)))))))


(defun tent-show (tent)
  (format t "~%  tent pending ")
  (mapc #'(lambda (s) (sent-show s tent)) (pending-of-tent tent))
  (format t "~%  tent committed ")
  (mapc #'(lambda (s) (sent-show s tent)) (committed-of-tent tent))
  )
    

(defmethod tent-insert ((tent tent) stamp seq oid data &optional note)
  (when *tent-debug*
    (format t "~%tent-insert ~a ~a ~a " (transaction-of-stamp stamp) seq (sequence-of-stamp (stamp-of-oid oid)))
    (tent-show tent)
    (when (eql *tent-debug* `b)
      (setf -tent tent -stamp stamp -seq seq -data data -oid oid) (break "ti")
      ))

  (set-tent-pending (cons (new-s-entry stamp seq oid data note)
			  (pending-of-tent tent))
		    tent)
  tent)


;;;;	
;;;;	Trouble brewing if two simultaneous transactions modifying
;;;;	same object. True simultaneous mods should lock object and
;;;;	second should fail. But if t1 starts, reads, but does not
;;;;	write lock, then t2 starts, locks, writes, ends, and then
;;;;	t1 locks and writes then ends, then trouble.
;;;;	 trouble since t1 reads, then t2 writes then t1 writes
;;;;	thus t1 never saw t2's mods.  That is sort of ok given
;;;;	our not-optimal locking model, but it can be prevent here
;;;;	by examining tent and failing if multiple updates are 
;;;;	happening.

;;;;	
;;;;	proper solution might be
;;;;	  - detect at mod (commit?) if some other mod committed that is not
;;;;	    visible.
;;;;	

(defmethod tent-delete ((tent tent) stamp seq oid &optional note)
  (when *tent-debug*
    (format t "~%tent-delete ~a ~a ~a " (transaction-of-stamp stamp) seq (sequence-of-stamp (stamp-of-oid oid)))
    (tent-show tent)
    (when (eql *tent-debug* `b)
      (setf -tent tent -stamp stamp -seq seq -oid oid) (break "td")
      ))

  (let ((nsent (new-s-entry-d stamp seq oid note))
	(sent (tent-lookup-woid tent stamp oid)))

    ;; if sent is not of same transaction or is not first of committed then clash.
    (cond
      ((not (or (in-transaction-p (stamp-of-s-entry sent) stamp)
		;; need to check if first with same oid
		(eql sent
		     (find-if #'(lambda (c) (equal-oids-p oid (oid-of-s-entry c)))
			      (committed-of-tent tent)))))
       (raise-error (error-message '(tent delete clash
				     simultaneous-transactions-modifing-same-object))))
    
      ((not (typep sent 's-entry-i))
       ;;(setf -nsent nsent -sent sent) (break "td")
       (raise-error (error-message '(tent delete deleted))))

      (t (set-s-entry-deletes (cons nsent (deletes-of-s-entry sent)) sent)
	 (set-s-entry-data (data-of-s-entry sent) nsent)
	 (set-tent-pending (cons nsent (pending-of-tent tent))
			   tent)))
  tent))


(defmethod tent-undo ((tent tent) stamp seq)
  (when *tent-debug*
    (format t "~%tent-undo ~a ~a" (transaction-of-stamp stamp) seq)
    (tent-show tent)
    (when (eql *tent-debug* `b)
      (setf -tent tent -stamp stamp -seq seq) (break "tu")
      ))
  
  (let ((usent nil))
    (set-tent-pending (delete-if #'(lambda (sent)
				     (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
						(eql seq (sequence-of-s-entry sent)))
				       ;;(unless (eql seq (sequence-of-s-entry sent))
				       ;;(setf a tent b (pending-of-tent tent) c sent d stamp) (break "tu")
				       ;;(system-error (error-message '(tent undo sequence))))
				       (setf usent sent)
				       t))
			       (pending-of-tent tent)
			       :count 1)
		      tent)

    ;; if undoing delete then must remove from deletes list.
    (unless (or (null usent) (typep usent 's-entry-i))
      ;; lookup seems not as robust as it might be but all later pendings
      ;; will already be undone, ie it can not find a later pending thus in must
      ;; be finding the latest.
      (let ((sent (tent-lookup-woid tent stamp (oid-of-s-entry usent))))
	(set-s-entry-deletes (delete usent (deletes-of-s-entry sent))
			     sent)))
    (values tent usent)))

(defun tent-undo-data (tent stamp seq)
  (mlet* (((tent s-entry) (tent-undo tent stamp seq) (declare (ignore tent))))
	 (values (data-of-s-entry s-entry)
		 (typep s-entry 's-entry-i)
		 (note-of-s-entry s-entry))))

(defmethod tent-commit ((tent tent) stamp seq)
  (when *tent-debug*
    (format t "~%tent-commit ~a ~a " (transaction-of-stamp stamp) seq)
    (tent-show tent)
    (when (eql *tent-debug* `b)
      (setf -tent tent -stamp stamp -seq seq) (break "tc")
      ))
  
  (let ((commit nil))
    (set-tent-pending (delete-if #'(lambda (sent)
				     (when (and (in-transaction-p (stamp-of-s-entry sent) stamp)
						(eql seq (sequence-of-s-entry sent)))
				       ;;(unless (eql seq (sequence-of-s-entry sent)))
				       ;;(system-error (error-message '(tent commit sequence)))
				       (setf commit sent)
				       t))
				 (pending-of-tent tent)
				 :from-end t
				 :count 1)
		      tent)

    ;;(setf a tent b stamp)
    ;;(setf c commit) (break "tc")
    (when commit
      (set-s-entry-stamp stamp commit)

      ;; collect immediately if all older commits are part of same atomic commit.
      (let ((committed (committed-of-tent tent)))
	(if (or (null committed)
		(forall-p #'(lambda (s) (equal-stamps-p s stamp))
			  committed))
	    (set-tent-committed (list commit) tent)
	    (progn
	      (set-tent-committed (cons commit committed)
				  tent)

	      (transaction-queue-collect stamp tent)))))

    
    (values tent commit)))

(defun tent-commit-data (tent stamp seq)
  (mlet* (((tent sent) (tent-commit tent stamp seq) (declare (ignore tent))))
	 (values (data-of-s-entry sent)
		 (typep sent 's-entry-i)
		 (note-of-s-entry sent))))
    


;;;	
;;;	If deletes list of insert has more than one delete then ignore.
;;;	
;;;	
;;;	Commits are sequential and commit stamps are ordered.
;;;	

;; only delete/insert pairs should be collected as if not deleted older insert can be
;; exposed via deletes. 
;; ok if delete is visible to active transactions.
;; do not collect if delete not visible to any active transactions.

;;; todo collect deletes if oldest as delete == nil.


(defmethod tent-collect (tent transactions)
  (when *tent-debug*
    (format t "~%tent-collect ")
    (tent-show tent)
    (when (eql *tent-debug* 'b)
      (setf -tent tent -transactions transactions) (break "tcl")
      ))
  

  ;; can't take the first one out.
  (let ((committed (committed-of-tent tent))
	(s-entries-d nil)
	(removes nil))

    ;;(when (cdr committed)
    ;;(setf -tent tent -transactions transactions) (break "tcl"))

    ;; could be updated to handle case of multiple deletes on same insert.
    ;; i2 d2 i3[] d3 i1[d2;d3] - all could go except i3
    ;; although probably better to prevent this state from arising.
    (dolist (sent (cdr committed))
      (if (typep sent 's-entry-i)
	  (let ((deletes (deletes-of-s-entry sent)))
	    (when (and deletes
		       (null (cdr deletes))
		       (member (car deletes) s-entries-d))
	      (push sent removes)
	      (push (car deletes) removes)))
	  (when (forall-p #'(lambda (tstate)
			      (let ((bstamp (begin-stamp-of-tstate tstate)))
				;; transaction has not yet begun so delete will be visible when it begins.
				(or (null bstamp)
				    (transaction-less-than (stamp-of-s-entry sent)
							   (begin-stamp-of-tstate tstate)))))
			  transactions)
	      (push sent s-entries-d))))

    (when removes
      (set-tent-committed (delete-if #'(lambda (sent)
					 (member sent removes))
				     committed)
			  tent))
    tent))
      
  
(defclass tentoid (tent)
  ((oid
    :initarg oid
    :reader oid-of-tent
    )))

(defun new-tentoid (oid)
  (make-instance 'tentoid 'oid oid))
  


;;;;	
;;;;	TENT end.
;;;;	

;;;;	
;;;;	BASIS :
;;;;
;;;;	term-basis : the parameter type list and arity list of a term.
;;;;	
;;;;	basis-of-term (<term>)			: <basis>
;;;;	term-basis-p (<basis> <term>)		: BOOL
;;;;
;;;;	type-ids-of-term-basis (<basis>)	: <tok> list
;;;;	arities-of-term-basis (<basis>)		: INT list
;;;;
;;;;
;;;;	basis-alist : an assoc list using the term basis as key.
;;;;	
;;;;	<basis-alist>	: <basis-cell> list
;;;;	<basis-cell>	: (<basis> . (<key-alist>)
;;;;	
;;;;	<key-alist>	: (<key-cell> list)
;;;;	<key-cell>	: (<term{key}> . <value>)
;;;;
;;;;	
;;;;	lookup-basis (<basis-alist> <term>)	: <basis-cell>
;;;;	
;;;;	tixt-basis-list-search(<basis-list> <closure{matchp}> <term{arg}> <closure{search-f}>)
;;;;	  : <*>
;;;;	  * match-p (<term{arg}> <term{key}>)	: BOOL
;;;;	  * search-f (<term{key}> <value>) : <*>
;;;;
;;;;	tixt-basis-list-mod (<basis-list> <closure{matchp}> <term{arg}> <closure{mod-f}>)
;;;;	  : <basis-list>
;;;;	  * match-p (<term{arg}> <term{key}>)	: BOOL
;;;;	  * mod-f (<value{arg}> | nil) 			: (<value{result}> | nil)
;;;;	    if (not (eq arg result)) then result replaces current value.
;;;;	
;;;;	tixt-lookup-in-basis-list-if (<basis-list> <closure{matchp}> <closure{f}> <term{instance})
;;;;	  : <data>
;;;;	  * f (<term{key}> <value>) : <data>
;;;;	    if <data> returned from f closure then that <data> is returned from function.
;;;;	
;;;;	



(defun basis-of-term (term)
  (cons (mapcar #'type-id-of-parameter (leaves-of-term term))
	(mapcar #'length (bindings-of-term term))))

(defun type-ids-of-term-basis (b) (car b))
(defun arities-of-term-basis (b) (cdr b))

(defun equal-term-basis-p (term-a term-b)
  (and (eql (id-of-term term-a) (id-of-term term-b))
       (apply-predicate-to-list-pair-optimized
	(parameters-of-term term-a)
	(parameters-of-term term-b)
	(lambda (a b) (eql (type-id-of-parameter a)
			   (type-id-of-parameter b))))
       (apply-predicate-to-list-pair-optimized
	(bound-terms-of-term term-a)
	(bound-terms-of-term term-b)
	(lambda (a b) (eql (length (bindings-of-bound-term a))
			   (length (bindings-of-bound-term b)))))))
       

(defun term-basis-p (basis term)
  (and
   (apply-predicate-to-list-pair-optimized
    (leaves-of-term term)
    (type-ids-of-term-basis basis)
    (lambda (p typeid)
      (eql (type-id-of-parameter p) typeid)))

   (apply-predicate-to-list-pair-optimized
    (bindings-of-term term)
    (arities-of-term-basis basis)
    (lambda (binding-list arity)
      (= (length binding-list) arity)))))




;;;	
;;;	basis-list 
;;;	

(defun lookup-basis (blist key)
  (assoc-if #'(lambda (basis)
		      (term-basis-p basis key))
	    blist))


(defun tixt-basis-lookup-key (matchp mlist instance)
  (assoc-if #'(lambda (key)
		;; why the lambda wrapper instead of simply passing f to assoc if?
		;;  - the order of args to test func if assoc-if is unspecified.
		;;    and as the args are not symmetric, we must make sure they are
		;;    passed in the correct order.
		(funcall matchp key instance))
	    mlist))


;; mod-f (<value{arg}> | nil) : (<value{result}> | nil)
;; if (not (eq arg result)) then result replaces current value.
(defun tixt-basis-list-mod (blist matchp key mod-f)
  (if blist
	      
      ;; returns basis match list cons.
      (let ((bcell (lookup-basis blist key)))
	(if bcell
		    
	    (let* ((klist (cdr bcell))
		   (kcell (tixt-basis-lookup-key matchp klist key)))

	      (if kcell
		  (let* ((value (cdr kcell))
			 (nvalue (funcall mod-f (car kcell) value)))
		    (unless (eql nvalue value)
		      (setf (cdr kcell) nvalue)))
		  
		  (let ((value (funcall mod-f key nil)))
		    (when value
		      (setf (cdr bcell)
			    (acons key value
				   klist)))))

	      blist)
 
	    ;; new basis cell
	      (let ((value (funcall mod-f key nil)))
		(if value
		    (acons (basis-of-term key)
			   (acons key value nil)
			   blist)
		    blist))))

      ;; new hashtable slot.
      (let ((value (funcall mod-f key nil)))
	(when value
	  (acons (basis-of-term key)
		 (acons key value nil)
		 nil)))))

(defun tixt-basis-list-search (blist matchp key search-f)
  (when blist
	      
    ;; returns basis match list cons.
    (let ((bcell (lookup-basis blist key)))
      (when bcell
		    
	(let* ((klist (cdr bcell))
	       (kcell (tixt-basis-lookup-key matchp klist key)))

	  (when kcell
	    (funcall search-f (car kcell) (cdr kcell))))))))


(defun tixt-lookup-in-basis-list-if (blist matchp f key)
  ;;(break "tlibli")
  ;;(format t "~%BASE")
  (when blist
    (let ((bcell (lookup-basis blist key)))

      (when bcell
	(let* ((klist (cdr bcell))
	       (kcell (tixt-basis-lookup-key matchp klist key)))

	  (when kcell
	    (funcall f (car kcell) (cdr kcell))))))))




;;;;	
;;;;	TIXT :
;;;;	
;;;;
;;;;	new-term-index-table (<closure{meta-parameter-p}>
;;;;			      <closure{meta-term-p}>
;;;;			      <closure{match-p}>)
;;;;	  * match-p (<term{instance}> <term{key}>)
;;;;	    match-p not called unless terms have same basis.
;;;;	  * meta predicates used by implementation to do indexing.
;;;;	      - assumes we can hash on non-meta parameter values for OID and token parameter types.
;;;;	      - TIXT assumes that if t1 has parameter p and t2 has parameter q and t1 and t2
;;;;		have same basis and p and q occur in same position in parameter list then
;;;;		t1 can not match t2 if both are not meta and they are not equal (in the equality
;;;;		used to hash token and object-ids).
;;;;	  * clash-p : used to match keys. Should be similar to match-p. Diff is match is
;;;;	    expected to match keys with instances and clash matches keys with keys.
;;;;	    ??? Maybe key-match-p would be a better name.
;;;;	
;;;;	tixt-lookup-if (<tixt> <term{instance}> <closure{f}> &optional <bool{failp}>)
;;;;	  : <data>
;;;;	  * f (<term{key}> <value>) : <data>
;;;;	    f is called only if  key matches instance.
;;;;	
;;;;	tixt-search(<tixt> <term{arg}> <closure{search-f}>)	: <*>
;;;;	  * search-f (<term{key}> <value>) : <*>
;;;;	    search-f is called only if  key matches instance.
;;;;
;;;;	tixt-mod(<tixt> <term{arg}> <closure{mod-f}>)		: NULL
;;;;	  * mod-f (<value{arg}> | nil) 			: (<value{result}> | nil)
;;;;	    mod-f is called only if  key matches instance.
;;;;	    if (not (eq arg result)) then result replaces current value.
;;;;



(defclass term-index-table ()
  ((oids :reader oids-of-tixt
	 :initform (make-hash-table)
	 )
   (toks :reader toks-of-tixt
	 :initform (make-hash-table))
   (other :reader other-of-tixt
	  :writer set-tixt-other
	  :initform nil)
   (match-p :initarg match-p
	    :reader match-p-of-tixt)
   (meta-parameter-p :initarg meta-parameter-p
		     :reader meta-parameter-p-of-tixt)
   (meta-term-p :initarg meta-term-p
		:reader meta-term-p-of-tixt)
   (clash-p :initarg clash-p
	    :reader clash-p-of-tixt)))

(defun new-term-index-table (meta-parameter-p meta-term-p match-p clash-p)
  (make-instance 'term-index-table
		 'meta-parameter-p meta-parameter-p
		 'meta-term-p meta-term-p
		 'match-p match-p
		 'clash-p clash-p
		 ))



;;; MTT mutex need to control access to clash list of table. Ie multiple transactions
;;; MTT could be inserting simultaneously. This is true for any list in ttable so best to control
;;; access to ttable itself.
(defun tixt-constant-table (parameter ttable)
  (case (type-id-of-parameter parameter)
    (OID (oids-of-tixt ttable))
    (|token| (toks-of-tixt ttable))
    (otherwise nil)))
	

(defmethod map-basis-lists ((ttable term-index-table) f)
  (maphash #'(lambda (k v)
	       (declare (ignore k))
	       (funcall f v))
	   (oids-of-tixt ttable))
  (maphash #'(lambda (k v)
	       (declare (ignore k))
	       (funcall f v))
	   (toks-of-tixt ttable))
  (mapc f (other-of-tixt ttable)))
  

;; f (<key> <value>) : <data>
(defmethod tixt-lookup-if ((ttable term-index-table) key f &optional failp)
  ;;(format t "~%TIF2")
  (let* ((p (car (leaves-of-term key)))
	 (constant-table (unless (funcall (meta-parameter-p-of-tixt ttable) p)
			   (tixt-constant-table p ttable)))
	 (match-p (match-p-of-tixt ttable)))

    ;;(setf a p b constant-table c match-p d key e f) (break "tli2")

    (or (if constant-table
	    (tixt-lookup-in-basis-list-if (gethash (value-of-parameter p) constant-table)
					  match-p f key)
	    (tixt-lookup-in-basis-list-if (other-of-tixt ttable)
					  match-p f key))
	(when failp
	  (raise-error (error-message '(term-index-table lookup) key))))))


;; f (<term{key}> <value>)
(defun tixt-search (ttable key f)
  (let* ((p (car (leaves-of-term key)))
	 (constant-table (unless (funcall (meta-parameter-p-of-tixt ttable) p)
			   (tixt-constant-table p ttable)))
	 (clash-p (clash-p-of-tixt ttable)))

    (or (when constant-table
	  (tixt-basis-list-search (gethash (value-of-parameter p) constant-table)
				   clash-p key f))
	(tixt-basis-list-search (other-of-tixt ttable)
				 clash-p key f))))

	

(defun tixt-mod (ttable key mod-f)
  (let* ((p (car (leaves-of-term key)))
	 (constant-table (unless (funcall (meta-parameter-p-of-tixt ttable) p)
			   (tixt-constant-table p ttable)))
	 (clash-p (clash-p-of-tixt ttable)))

    (if constant-table
	(let* ((blist (gethash (value-of-parameter p) constant-table))
	       (nblist (tixt-basis-list-mod blist clash-p key mod-f)))
	  (unless (eq blist nblist)
	    (setf (gethash (value-of-parameter p) constant-table)
		  nblist)))
	
	(let* ((blist (other-of-tixt ttable))
	       (nblist (tixt-basis-list-mod blist clash-p key mod-f)))
	  (unless (eq blist nblist)
	    (set-tixt-other nblist ttable))))))



(defun tixt-test ()
  ;; todo :
  (let ((tixt (new-term-index-table #'abstraction-meta-parameter-p
				    #'(lambda (x)
					(declare (ignore x))
					(error "tixt-meta-term-p"))
				    #'compare-terms-p
				    #'compare-terms-p)))

    (list

     ;; add and then lookup returns value added.
     (progn 
      (tixt-mod tixt (ivoid-term)
		#'(lambda (key value)
		    (declare (ignore key value))
		    (cons t 1)))
       (eql 1 (tixt-lookup-if tixt (ivoid-term)
			      #'(lambda (key value)
				  (declare (ignore key))
				  (cdr value)))))

     ;; remove and then lookup returns nil.
     (progn
       (tixt-mod tixt (ivoid-term)
		 #'(lambda (key value)
		    (declare (ignore key))
		    (setf (cdr value) nil)))
       (null (tixt-lookup-if tixt (ivoid-term)
			      #'(lambda (key value)
				  (declare (ignore key))
				  (cdr value)))))

     ;; add then add and lookup returns second value.
     (progn
      (tixt-mod tixt (ivoid-term)
		#'(lambda (key value)
		    (declare (ignore key value))
		    (cons t 1)))
      (tixt-mod tixt (ivoid-term)
		#'(lambda (key value)
		    (declare (ignore key value))
		    (cons t 2)))
	    
      (prog1 (tixt-search tixt (ivoid-term)
			  #'(lambda (key value)
			      (declare (ignore key))
			      (eql 2 (cdr value))))
	(tixt-mod tixt (ivoid-term)
		  #'(lambda (key value)
		      (declare (ignore key))
		      (setf (cdr value) nil)))))
     )))




;;;;	
;;;;	Tixt-Pool
;;;;	
;;;;	no clash (==> clash-p == match-p) and lookup returns pool
;;;;	
;;;;
;;;;	new-pool-tixt-table (meta-parameter-p data-eq-p data-order-p)
;;;;	  data-order-p{<data> <data> : <bool>}	: used to sort.
;;;;	  data-eq-p   {<data> <data> : <bool>}	: used to remove.
;;;;	
;;;;	tixt-pool-add   (<tixt> <stamp> <term{meta}> <data> sexpr &optional <bool{sortp}>)
;;;;	
;;;;	tixt-pool-remove(<tixt> <stamp> <term{meta}> <data> sexpr)
;;;;
;;;;	tixt-pool-sort(<tixt> <stamp>)
;;;;	  * sorts all pools updated{add/remove} since last sort.
;;;;	
;;;;	
;;;;	
;;;;	Pool implementation facilitates local caching of pool data:
;;;;	
;;;;	<pool>		: (<stamp{pool}> . <asynch-pool>)
;;;;	<asynch-pool>	: (<stamp{update}> . <data> sexpr)
;;;;
;;;;	  * the aysnch pool may be destructively updated at add/remove/sort.
;;;;	    At any such update the asynch stamp will be updated.
;;;;	    The pool stamp however is not destructively modified. Thus
;;;;	    it will not change once returned. 
;;;;	    at lookup time the two stamps should be identical.
;;;;	      - after lookup if they differ you know the asynch-pool has been
;;;;		destructively modified.
;;;;	      - after lookup if the asynch stamp is void, then you know that the
;;;;		pool has been modified and the asynch pool does not contain mods.
;;;;		
;;;;	    Initial implementation will never void asynch stamp. If you want the
;;;;	    pool to reflect the transaction view of the transaction doing lookup
;;;;	    then voiding will be used. Currently such pool usage is not supported.
;;;;	
;;;;	
;;;;	tixt-pool-lookup(<tixt> <term>)		: <pool>
;;;;	

;;;;	
;;;;	Commit/Undo would need to be built upon transtixt with the 
;;;;	pools being in the tents.
;;;;	
;;;;	current implmentation built on tixt where tixt data is asynch pool.
;;;;	add/remove uses tixt-mod to find current pool and update it.
;;;;	must collect pools to be sorted, or maybe just unconditionally sort 
;;;;	at each update. Could assume that remove does not require resort.

(defun stamp-of-definition-pool (p) (car p))
(defun asynch-stamp-of-definition-pool (p) (cadr p))
(defun list-of-definition-pool (p) (cddr p))
(defun asynch-pool-of-definition-pool (p) (cdr p))

(defun new-definition-pool (stamp apool) (cons stamp apool))

(defun refresh-definition-pool (pool)
  (if (eql (car pool) (cadr pool))
      pool
      (new-definition-pool (asynch-stamp-of-definition-pool pool)
			   (asynch-pool-of-definition-pool pool))))

(defun stale-definition-pool-p (pool)
  (or (null pool)
      (not (equal-stamps-p (asynch-stamp-of-definition-pool pool)
			   (stamp-of-definition-pool pool)))))

(defclass pool-term-index-table (term-index-table)
  ((eq-p :reader eq-p-of-pool-tixt
	 :initarg eq-p)
   (less-than-p :reader less-than-p-of-pool-tixt
	    :initarg less-than-p)))
	    

(defun new-pool-tixt-table (meta-parameter-p match-p eq-p less-than-p)
  (make-instance 'pool-term-index-table
		 'meta-parameter-p meta-parameter-p
		 'meta-term-p #'(lambda (term)
				  (declare (ignore term))
				  (raise-error (error-message '(pool text meta-term-p))))
		 'match-p #'equal-term-basis-p
		 'clash-p match-p ;; basis-eq
		 'eq-p eq-p
		 'less-than-p less-than-p
		 ))

;; returns nil in no match.
(defmethod pool-tixt-lookup ((ttable pool-term-index-table) term)
  (tixt-lookup-if ttable term
		  #'(lambda (term apool)
		      (declare (ignore term))
		      (when apool
			(new-definition-pool (car apool) apool)))))
		      
(defmethod pool-tixt-add ((ttable pool-term-index-table) stamp term data &optional sortp)
  (declare (ignore sortp)) 
  (let ((lt-p (less-than-p-of-pool-tixt ttable)))
    (labels
	((aux (l)
	   (cond
	     ((null l)
	      (list data))
	     ((funcall lt-p data (car l))
	      (cons (car l) (aux (cdr l))))
	     (t (cons data l)))))
	       
      (tixt-mod ttable term
	      #'(lambda (term apool)
		  (declare (ignore term))
		  
		  ;; destructively update stamp so that pool holders know update has occurred.
		  (when apool
		    (setf (car apool) stamp))

		  (cons stamp
			(aux (cdr apool))))))))


(defmethod pool-tixt-remove ((ttable pool-term-index-table) stamp term data)

  (tixt-mod ttable term
	      #'(lambda (term apool)
		  (declare (ignore term))
		  (cons stamp
			(delete data (cdr apool)
				:test (eq-p-of-pool-tixt ttable))))))


(defmethod pool-tixt-sort ((ttable pool-term-index-table) stamp)

  (let ((sortp (less-than-p-of-pool-tixt ttable)))

    ;; sorted at insert.

    nil))




;;;;	
;;;;	Clash :
;;;;	
;;;;	
;;;;	A term index table(TIXT) is used to look up definitions associated with a term
;;;;	instance by match the instance to a template term key of the TIXT. In some
;;;;	applications it is desirable, that no instance match more than a single template
;;;;	term. If it is possible for some instance to match more than one template term,
;;;;	then the template terms are said to clash. We assume we have a predicate which
;;;;	given two keys can detect if they clash. With concurrent transactions, one strategy
;;;;	to avoid clashing keys is to check for key clash when committing an insert to the
;;;;	TIXT and if the key being inserted does clash with a key already committed, then
;;;;	the commit fails and the insert is undone. It is not sufficient to check prior to
;;;;	commit as it is always possible that a concurrent transaction will commit a clashing
;;;;	key after the check but prior to the primary transaction's commit.
;;;;	
;;;;	Unfortunately, it the TIXT is remote our current transaction model does not allow
;;;;	commit to fail. One of the primary motivations of the current transaction model is
;;;;	to prevent the library process from being affected by remote failures.
;;;;
;;;;		(That statement is perhaps best thought of as a corollary
;;;;		to Les Lamport's classical statement, ``A distributed
;;;;		system is one in which the failure of a computer you
;;;;		didn't even know existed can render your own computer
;;;;		unusable.)  PGN  (from comp.risks)
;;;;
;;;;	Thus enhancing the commit protocol to allow commit failures is to be
;;;;	avoided if possible. If the library must wait for a response to the
;;;;	commit, and the remote host crashes then it is possible that the libary
;;;;	will hang due to the remote failure.
;;;;
;;;;
;;;;	This model seeks to avoid this problem.
;;;;	
;;;;	We might allow discrimination among library clients such that an
;;;;	essential client would be allowed to fail commits. In a single user
;;;;	library this is probably what is desired, since if the library client
;;;;	fails the session is over anyways. However, this still allows the
;;;;	anonymous listener scenario, where the non-essential anonymous client
;;;;	can not hang the library.
;;;;
;;;;	Another solution is to export the clash detection predicate to the library so 
;;;;	that the library can fail locally rather than having the client fail.
;;;;	
;;;;	FTTB, we do not have concurrent transactions. Exporting the predicate would
;;;;	result in a simpler and more robust transaction model in the case where the library
;;;;	itself is not distributed (the only possibility at the moment).
;;;;
;;;;	A less attractive method of dealing with clashes is too have the later key shadow
;;;;	the earlier key. This may be insufficient for some tables. The current implementation
;;;;	does shadowing.
;;;;
;;;;
;;;;	Clash : For any two term index keys, if there exists a term instance where the
;;;;	  instance matches both term keys then the keys are said to clash.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Implementation notes:
;;;;	
;;;;	Clash is implemented as a basis list with add order which is checked prior
;;;;	to the tixt table.
;;;;
;;;;	  - key-clash-p : allows for detection of clashes and shadowing of clashed values.
;;;;	    In case of clash later is prefered over former. Former is still deleteable.
;;;;
;;;;	Implementation is somewhat unsatisfying in that removing shadowed entry does not
;;;;	move clash into usual table.
;;;;	
;;;;	TODO : maybe commit can rehash clash list. At least allow explicit rehash call.
;;;;	
;;;;	
;;;;		
;;;;	TIXT (clash) : TIXT which shadows clashing definitions.
;;;;	
;;;;	new-clash-term-index-table (<closure{meta-parameter-p}>
;;;;				    <closure{meta-term-p}>
;;;;				    <closure{match-p}>
;;;;				    <closure{clash-p}>)
;;;;	  - clash-p : detects if two keys clash.
;;;;	    (clash-p t1 t2) -> (match-p t1 t2) & (match-p t2 t1)
;;;;	    note : not <->.
;;;;	    (clash-p t1 t1) ->  true.
;;;;
;;;;	tixt-lookup-if (<tixt> <term{instance}> <closure{f}> &optional <bool{failp}>)
;;;;	  : <data>
;;;;	  * f (<term{key}> <value>) : <data> | nil
;;;;	    nil means keep looking.
;;;;	
;;;;	tixt-clash-mod(<tixt> <term{arg}> <closure{clash-f}> <closure{mod-f}>)	: NULL
;;;;	  * clash-f (<term{key}> <value>) : BOOL
;;;;	  * mod-f (<value{arg}> | nil) 			: (<value{result}> | nil)
;;;;	    if clash-f true then applies to clash-value. 
;;;;	
;;;;	tixt-clash-add(<tixt> <term{arg}> <closure{clash-f}> <closure{mod-f}>)	: NULL
;;;;	  * clash-f (<term{key}> <value>) : BOOL
;;;;	  * mod-f (<value{arg}> | nil) 			: (<value{result}> | nil)
;;;;	    if clash-f true then adds to clash list otherwise applies to tixt table.
;;;;	



;;;;	
;;;;	TRANSTIXT : combines clash-tixt and with tents.
;;;;	
;;;;	transtixt-lookup (<tixt> <term{instance}> <stamp> &optional BOOL{failp}) : <data>
;;;;	
;;;;	transtixt-insert (<tixt> <term{key}> <stamp> <int{sequence}> <oid> <data>
;;;;			  &optional BOOL{failp})
;;;;	
;;;;	transtixt-delete (<tixt> <term{key}> <stamp> <int{sequence}> <oid>)
;;;;	  * needs to be another arg to verify correct data is being deleted. ie to choose amoung clashing
;;;;	
;;;;	transtixt-commit (<tixt> <term{key}> <stamp> <int{sequence}>
;;;;			  &optional BOOL{failp})
;;;;	
;;;;	transtixt-undo (<tixt> <term{key}> <stamp> <int{sequence}>)
;;;;	
;;;;	




;;;;	Serializable :
;;;;	
;;;;	At any time there may be only one committed undeleted insert for any oid.
;;;;	If there is an insert then all older inserts must have been deleted.
;;;;	
;;;;	Commit undoes delete and then re-applies.
;;;;	  - No longer a suitable entry to delete -> fails ???
;;;;	      * could continue as deleting nothing is safe.
;;;;	  - Latest committed insert deleted. This may be an entry inserted by a concurrent
;;;;	    transaction which completed prior to current transaction.
;;;;
;;;;	Commit undoes insert and then re-applies.
;;;;	  - A clashing entry added -> adds current as clash.
;;;;	  - A clashing entry deleted -> adds as non-clash.
;;;;	  - A clashing entry for same oid added -> fail???
;;;;	      * transtixt : no - does not catch all possible conflicts so why catch any.
;;;;	      * oidixt : yes - catches unserializable inserts for same oid.
;;;;
;;;;	Serializability?
;;;;	  - Transaction view does not see concurrent effects, so the state is not serial.
;;;;	    However, the state of the library is serialized and re-running the broadcasts
;;;;	    will result in same lib state. However, re-running refinements may see different
;;;;	    state.
;;;;	
;;;;	




(defclass clash-term-index-table (term-index-table)
  ((clash :reader clash-of-tixt
	  :writer set-tixt-clash
	  :initform nil)))


(defun new-clash-term-index-table (meta-parameter-p meta-term-p match-p clash-p)
  (make-instance 'clash-term-index-table
		 'meta-parameter-p meta-parameter-p
		 'meta-term-p meta-term-p
		 'match-p match-p
		 'clash-p clash-p
		 ;;'remove-p (or remove-p clash-p)
		 ))

(defmethod tixt-lookup-if ((ttable clash-term-index-table) key f &optional failp)
  ;;(format t "~%TIF1")
  (or (tixt-lookup-in-basis-list-if (clash-of-tixt ttable)
				    (match-p-of-tixt ttable)
				    f key)
      ;;(progn  (break "tli") nil)
      (call-next-method ttable key f failp)))


(defmethod tixt-clash-add ((ttable clash-term-index-table) key clash-hook mod-hook)

  (let ((clash (clash-of-tixt ttable))
	(foundp nil))

    (let ((new-clash (tixt-basis-list-mod clash
					  (clash-p-of-tixt ttable)
					  key
					  #'(lambda (tkey value)
					      (if (and value
						       (funcall clash-hook tkey value))
						  (progn
						    (setf foundp t)
						    (funcall mod-hook tkey value))
						  value)))))

      ;;(setf -clash clash -foundp foundp -nc new-clash) (break "tca") 
      (if foundp
	  (set-tixt-clash new-clash ttable)
	  (if (tixt-search ttable key #'(lambda (tkey value)
					  (funcall clash-hook tkey value)))

	      (set-tixt-clash (tixt-basis-list-mod clash
						   (clash-p-of-tixt ttable)
						   key
						   mod-hook)
			      ttable)
	      (tixt-mod ttable key mod-hook))))))


(defmethod tixt-clash-mod ((ttable clash-term-index-table) key clash-hook mod-hook)

  (let ((clash (clash-of-tixt ttable))
	(foundp nil))

    (let ((new-clash (tixt-basis-list-mod clash
					  (clash-p-of-tixt ttable)
					  key
					  #'(lambda (tkey value)
					      (if (and value
						       (funcall clash-hook tkey value))
						  (progn
						    (setf foundp t)
						    (funcall mod-hook tkey value))
						  value)))))
      (if foundp
	  (set-tixt-clash new-clash ttable)
	  (tixt-mod ttable key
		    #'(lambda (tkey value)
			;;(setf a tkey b value) (break "tcm")
			(if (and value
				 (funcall clash-hook tkey value))
			    (progn
			      (setf foundp t)
			      (funcall mod-hook tkey value))
			    value))))
      (unless foundp
	(raise-error (error-message '(term-index-table clash mod none)))))))




  
(defun clash-tixt-test ()
  (labels
      ((tixt-add (tixt term data)
	 (tixt-clash-add tixt term
			 #'(lambda (key value)
			     (declare (ignore key))
			     (and value (car value)))
			 #'(lambda (key value)
			     (declare (ignore key))
			     (if value
				 (progn (setf (cdr value) data (car value) t) value)
				 (cons t data)))))

       (tixt-remove (tixt term)
	 (tixt-clash-mod tixt term
			 #'(lambda (key value)
			     (declare (ignore key))
			     (and value (car value)))
			 #'(lambda (key value)
			     (declare (ignore key))
			     (when value
			       (setf (cdr value) nil
				     (car value) nil))
			     value)))
 
       (tixt-lookup (tixt term)
	 (tixt-lookup-if tixt term
			 #'(lambda (key value)
			     (declare (ignore key))
			     ;;(setf a key b value) (break "tl")
			     (when (car value)
			       (cdr value))))))
    
    (let ((tixt (new-clash-term-index-table #'abstraction-meta-parameter-p #'variable-term-p
					    #'compare-terms-p #'compare-terms-p)))

    
      (list
     
       ;; add clash then remove former.
       (progn
	 (tixt-add tixt (ivoid-term) 1)
	 (tixt-add tixt (ivoid-term) 2)
	 (eql 2 (tixt-lookup tixt (ivoid-term))))

       ;;(tixt-clash-p tixt)

       (progn
	 (tixt-remove tixt (ivoid-term))
	 (eql 1 (tixt-lookup tixt (ivoid-term))))

       ;; remove shadowed
       (progn
	 (tixt-add tixt (ivoid-term) 2)
	 ;; expect to be able to search using clash-f and then have mod modify where search stopped.
	 ;; presently if any clash then modifies clash list
	 (tixt-clash-mod tixt (ivoid-term)
			 #'(lambda (key value)
			     (declare (ignore key))
			     (and value (car value) (eql 1 (cdr value))))
			 #'(lambda (key value)
			     (declare (ignore key))
			     ;;(setf a key b value) (break "arrgh2")
			     (when value
			       (setf (cdr value) nil
				     (car value) nil))
			     value))

	 (eql 2 (tixt-lookup tixt (ivoid-term))))

       ;;(not (tixt-clash-p tixt))

       ))))


    
;;;;	<value>		: <tent>
;;;;	
;;;;	Invariant : eqness of tent should not change within a transaction
;;;;	


(defun transtixt-lookup (ttable instance stamp &optional failp p)
  ;;(setf c stamp b instance) (break "ttl1")
  ;;(format t "~%TIXT")
  (tixt-lookup-if ttable
		  instance
		  #'(lambda (key value)
		      (declare (ignore key))
		      ;;(setf a value) (break "ttl")
		      (when value
			;;(setf -value value -stamp stamp)
			;;(format t "~%TENT ~a" (id-of-term (abstraction-lhs (data-of-s-entry (car (committed-of-tent -value))))))
			;;(tent-show value)
			(tent-lookup value stamp t p)
			))
		  failp))

;; assume that not clash k1 k2 -> for a : match a k1 -> notmatch a k2 and match a k2 -> not match a k1

(defun transtixt-insert (ttable key stamp seq oid data)
  (tixt-clash-add ttable key

		  #'(lambda (tkey tent)
		      (declare (ignore tkey))
		      (and tent
			   (tent-contains-visible-p tent stamp)))
		  #'(lambda (tkey tent)
		      (declare (ignore tkey))
		      (if tent
			  (tent-insert tent stamp seq oid data)
			  (let ((tent (new-tent)))
			    ;;(setf a tent b tkey) (break "tti")
			    (tent-insert tent stamp seq oid data))))))



;;; this is not correct it needs to shadow a particular s-entry not all s-entries in clashing stack.
;;; ie tent-insert of nil should be some tent-delete which affects an s-entry.
;;; even worse delete needs to shadow all s-entry for oid but not other clashing s-entries for other oids.
;;; ie current tent-lookup insufficient, and to fix add delete and/or oid to s-entry. (raise oid ?) or segregate
;;; sentries by oid into distinct tents?

;;;;	the fact that an entry in the tent is deleted does not mean lookup should return nil.
;;;;	There may be another entry being shadowed.  But sometimes it should? eg t1 starts and deletes then
;;;;	lokup is nil, also 
;;;;	
;;;;	t1, t2 both committed and t1 is insert, t2 is delete, skip delete but find t1 is deleted.
;;;;	worry about finding an older version,
;;;;	
;;;;	First undeleted s-entry with applicable stamp. 
;;;;	It is not possible to lookup an insert for a transaction not relevant to the transaction, ie 
;;;;	there exists a newer comitted transaction completed prior to start of current transaction shadowing
;;;;	irrelevant insert.  If deleted (wrt current transaction) that will be recognized. If not deleted
;;;;	then must be relevant as there can be no newer version as it would have to have been deleted first.
;;;;	
;;;;	so lookup is looking for latest not deleted insert
;;;;



;;;;
;;;;	It is not transtixt's responsibility to detect lock problems.
;;;;	transtixt behaves unpredictably wrt lock problems.
;;;;	
;;;;	Assume : if data is inserted for an oid then either there is no other data for
;;;;	         oid or it has been deleted by current transaction or an earlier.
;;;;	
;;;;	
;;;;	when deleted s-entry points to s-entry deleting. If deleted again then suspect problemo.
;;;;	transtixt should warn oid-table should object. Lib should have local oid table for all objects
;;;;	just to detect such conflicts. Locks insufficient 
;;;;	
;;;;	delete does  show as s-entry itself but also shows in s-entry.
;;;;	
;;;;	
;;;;	Concurrent transactions both delete same oid data. 
;;;;	  - lock conflict : one transaction should be killed at delete of commit of delete.
;;;;	Concurrent transactions both insert same oid data. 
;;;;	
;;;;	RLE TODO : make sure this is accounted for?
;;;;	T1 starts t2 starts modifies ends, then t1 modifies then ends ???
;;;;	 at t1 commit/modify we ought to be able to detect conflict and abort t1.
;;;;	 detect conflict be seeing two deletes committed to same entry or two contiguous inserts.
;;;;	 could detect at modify as can see the commit occured between current transaction begin
;;;;	and modify attempt.

;;;;	 if we do not the there will be some value deleted twice and two contiguous inserts.
;;;;	 contiguous inserts are bad as we assume they can not happen.

;; stamp is time of delete
(defun transtixt-delete (ttable key stamp seq oid)
  ;;(setf -key key -stamp stamp -seq seq -oid oid) (break "ttd")
  (tixt-clash-mod ttable key
		  #'(lambda (tkey tent)
		      (declare (ignore tkey))
		      (and tent
			   (tent-contains-oid-p tent stamp oid)))
		  #'(lambda (tkey tent)
		      (declare (ignore tkey))
		      (when (null tent) (break "transtixt-delete"))
		      (tent-delete tent stamp seq oid)
		      )))



;; tixt-mod if add-f returns value not eq to passed value then tixt-mod updates.
;;; stamp is commit stamp which is latest broadcast for transaction.

(defun transtixt-commit (ttable key stamp seq)

  ;; need to reinsert as concurrent transactions may have committed
  ;; since transaction began.

  (let ((sent nil)
	(foundp nil))

    ;; undo 
    (tixt-clash-mod ttable key
		    #'(lambda (tkey tent)
			(declare (ignore tkey))
			(and tent
			     (tent-contains-pending-p tent stamp seq)))

		    #'(lambda (tkey tent)
			(declare (ignore tkey))
			;; if tent empty return nil? no let collector handle it.
			(setf foundp t)
			;;(setf e tent) (break "ssu")
			(mlet* (((tent s-entry) (tent-undo tent stamp seq)))
			       ;;(setf a s-entry d tent) (break "sse")
			       (setf sent s-entry)
			       tent)))

    (unless foundp
      (system-error (error-message '(transaction-index-table commit))))
   
 
    (let ((insert-p (typep sent 's-entry-i)))

      (if insert-p
	  (tixt-clash-add ttable key
			  #'(lambda (tkey tent)
			      (declare (ignore tkey))
			      ;; if visible for same oid then could fail?
			      (and tent
				   (tent-contains-visible-p tent stamp)))
			  #'(lambda (tkey tent)
			      (declare (ignore tkey))
			      (if tent
				  (progn
				    (tent-insert tent stamp seq (oid-of-s-entry sent)
						 (data-of-s-entry sent))
				    (tent-commit tent stamp seq))
				  (let ((tent (new-tent)))
				    (tent-insert tent stamp seq (oid-of-s-entry sent)
						 (data-of-s-entry sent))
				    (tent-commit tent stamp seq)))))
	  
	  (tixt-clash-mod ttable key
			  #'(lambda (tkey tent)
			      (declare (ignore tkey))
			      (and tent
				   (tent-contains-committed-oid-p tent stamp (oid-of-s-entry sent))))
			  #'(lambda (tkey tent)
			      (declare (ignore tkey))
			      (tent-delete tent stamp seq (oid-of-s-entry sent))
			      (tent-commit tent stamp seq)))))))


(defun transtixt-undo (ttable key stamp seq)

  ;; undo
  (tixt-clash-mod ttable key
		    #'(lambda (tkey tent)
			(declare (ignore tkey))
			(and tent
			     (tent-contains-pending-p tent stamp seq)))
		  #'(lambda (tkey tent)
		      (declare (ignore tkey))
		      ;; if tent empty return nil? no let collector handle it.
		      (tent-undo tent stamp seq))))


;; needs (defun equal-oids-p (a b) (eql a b)) to work.
#+debug
(defun transtixt-test ()
  (let ((tixt (new-clash-term-index-table #'abstraction-meta-parameter-p #'variable-term-p
					  #'compare-terms-p #'compare-terms-p))
	(key (ivoid-term))
	(oid-a 100)
	(oid-b 200)
	(oid-d 400))

    (advance-sequence)
    (list 
     (with-transaction
	 (let ((stamp (transaction-stamp)))
	   (advance-sequence)
	   (transtixt-insert tixt key stamp 1 oid-a 'a)
	   (prog1 (eql 'a (transtixt-lookup tixt key
					    (current-transaction-stamp)))
	     (advance-sequence)
	     (transtixt-commit tixt key (transaction-stamp) 1))))
   
  
     (with-transaction
	 (advance-sequence)
       (eql 'a (transtixt-lookup tixt key
				 (current-transaction-stamp))))

     (with-transaction
	 (advance-sequence)
       (let ((stamp (transaction-stamp)))
	 (transtixt-insert tixt key stamp 2 oid-b 'b)
	 (advance-sequence)
	 (prog1 (eql 'b (transtixt-lookup tixt key
					  (current-transaction-stamp)))
	   (transtixt-undo tixt key stamp 2))))


       (with-transaction
	   (advance-sequence)
	   (eql 'a (transtixt-lookup tixt key
				     (current-transaction-stamp))))

       (let (a b c d e f)
	 (with-transaction (setf a (current-transaction-stamp)
				 b (transaction-stamp)
				 e (transaction-state)))
	 (advance-sequence)
	 (with-transaction (setf c (current-transaction-stamp)
				 d (transaction-stamp)
				 f (transaction-state)))
	 
	 (advance-sequence)
	 (let ((*transaction-state* e))
	   (transtixt-insert tixt key b 3 oid-b 'b))
	 (advance-sequence)
	 (let ((*transaction-state* f))
	   (transtixt-insert tixt key d 4 oid-d 'd))
	     
	 ;;(setf dd d bb b)
	 (and
	 (let ((*transaction-state* e))
	   (eql 'b (transtixt-lookup tixt key a)))
	 (let ((*transaction-state* f))
	   (eql 'd (transtixt-lookup tixt key c)))
	  (progn (advance-sequence)
		 (let ((*transaction-state* f))
		   (transtixt-commit tixt key  (let ((*transaction-id* (car d))) (transaction-stamp)) 4))
		 (advance-sequence)
		 ;;(break "ss")
		 (let ((*transaction-state* e))
		   (transtixt-commit tixt key (let ((*transaction-id* (car b))) (transaction-stamp))
				     3))
		 
		 ;;(break "st")
		 (with-transaction
		     (eql 'b (transtixt-lookup tixt key
					       (begin-stamp-of-tstate
						(transaction-state))))))))

       ;; do some deletes!
       (with-transaction
	   (advance-sequence)
	 (let ((stamp (transaction-stamp)))
	   (list (prog1 (eql 'b (transtixt-lookup tixt key
						  (current-transaction-stamp)))
		   (transtixt-delete tixt key stamp 5 oid-b)
		   (advance-sequence))
		 (prog1 (eql 'd (transtixt-lookup tixt key
						  (current-transaction-stamp)))
		   (transtixt-undo tixt key stamp 5))
		 (eql 'b (transtixt-lookup tixt key
					   (current-transaction-stamp))))))
   
       
       ;; t1 start
       ;; t2 start
       ;; t1 deletes b.
       ;; t1 completes
       ;; t2 reads (should see original)
       ;; t2 completes.
       ;; t3 starts, reads, completes.
       (let (a e f)
	 (with-transaction (setf a (current-transaction-stamp)
				 ;;b (transaction-stamp)
				 e (transaction-state)))
	 (advance-sequence)

	 (with-transaction (setf ;;c (current-transaction-stamp)
				 ;;d (transaction-stamp)
				 f (transaction-state)))
	 (advance-sequence)
	
	 ;; t1 deletes b.
	 (let ((*transaction-state* e))
	   (let ((stamp (transaction-stamp)))
	     (transtixt-delete tixt key stamp 6 oid-b)
	     (advance-sequence)
	     (transtixt-commit tixt key stamp 6)
	     (advance-sequence)))
	 
		
	 (list (let ((*transaction-state* f))
		 (eql 'b (transtixt-lookup tixt key a)))
	       
	       (with-transaction
		   (advance-sequence)
		 (eql 'd (transtixt-lookup tixt key
					   (current-transaction-stamp))))))


       (with-transaction
	   (advance-sequence)
	 (let ((stamp (transaction-stamp)))
	   (list (prog1 (eql 'd (transtixt-lookup tixt key
						  (current-transaction-stamp)))
		   (transtixt-delete tixt key stamp 7 oid-d)
		   (advance-sequence))
		 (prog1 (eql 'a (transtixt-lookup tixt key
						  (current-transaction-stamp)))
		   (transtixt-commit tixt key stamp 7))
		 (eql 'a (transtixt-lookup tixt key
					   (current-transaction-stamp))))))

     (with-transaction
	 (advance-sequence)
       (eql 'a (transtixt-lookup tixt key
				 (current-transaction-stamp))))
	 )))





;;;; vestigial doc : should be milled to see in any useful info or unresolved concerns contained.




;;;;	No Locks - No deadlocks - No killing of transactions. 
;;;;
;;;;	not possible still need write locks to prevent contiquous inserts/deletes.
;;;;
;;;;
;;;;	Total Recall : All data, even stale data is accessible 
;;;;	  even if not readily available. 
;;;;	
;;;;	not possible due to difficulty in merging commits of lib-list changes.
;;;;	however, define new lib-create primitive where position is not specified
;;;;	then no locks used.


;;;;	TODO : producer broadcast tables now can supply a consistent table even with
;;;;	TODO : active transactions. 
;;;;	TODO :  activate must use transaction entries! store etc as well.


;;;;	todo make sure collector is called. but how to collect empty tents? dont!.


;;;;	clash-insert
;;;;	  - table
;;;;	  - ckey
;;;;	  - data
;;;;	  - matchp (tkey ckey) : used to filter a values to apply clash-value-p to.
;;;;	  - clash-value-p (key{table}  value{table})  
;;;;	  - add (ckey value data) 
;;;;	     : value may be nil, if value nil then add returns a value to be inserted.
;;;;		returned values will be igonred (warn!) in other cases.
;;;;	
;;;;	  - stamp seq or implicit in match-p/clash-f
;;;;
;;;;	clash-delete
;;;;	  - table
;;;;	  - ckey 
;;;;	  - matchp (tkey ckey) : used to filter a values to apply clash-value-p to.
;;;;	  - value-p (key{table} key{insert} value{table} data{insert})  
;;;;	  - stamp/seq or implicit in match-p/clash-f
;;;;	
;;;;	Allow mulitple values with similar keys ?? If no clash then no problem.
;;;;	If clashes already exist then will add clash data to clash value
;;;;	
;;;;	
;;;;	clash-lookup
;;;;	  - table 
;;;;	  - instance
;;;;	  - match-f (tkey instance value) : lookup returns first non-nil value returned from match-f
;;;;	
;;;;	
;;;;	commit (table key complete-f (key key value))
;;;;	undo ( table key complete-f (key key value))
;;;;	  - either may return indication that value is empty.
;;;;	
;;;;	


;;;;	match (tent stamp key instance)
;;;;	clash (tent stamp key key def)
;;;;	
;;;;	




;;;;	arrgh match-p, clash-p, remove-p must take value as arg to allow
;;;;	multiplexing of tents as value list, key may index value but no part
;;;;	of value satisfies stamp so may need to get to other list.
;;;;	
;;;;	value is tent list or value is tent, multiple oids per tent ok ???
;;;;	there may be more than one tent per oid as the index term may change with an update
;;;;	by not mixing oids in a tent it is easier to move data
;;;;
;;;;	at commit the entry may need to be moved. as it may clash with some other entry
;;;;	committed since transaction begin. or may no longer clash with some entry committed at begin.
;;;;
;;;;	insert : add
;;;;	delete : tent is useful for undo as value is saved.
;;;;	  oid & stamp uniquely identify as only one active per oid per transaction.
;;;;		 need to mark as invisible to transaction. although could have eq check avail as
;;;;		 oid table will give eq value.
;;;	undo : remove
;;;	commit : or somehow check if would move and then remove and add or update.
;;;;
;;;;	committed deletes can be removed(collected) when there is no possibility of reference to deleted value.

     




;;;;	Term Index table : tixt.
;;;;	  - transactions supported.
;;;;	
;;;;	new-term-index-table (meta-parm-p meta-term-p match-p key-clash-p)
;;;;	  - key-clash-p : allows for detection of clashes and shadowing of clashed values.
;;;;	    In case of clash later is prefered over former. Former is still deleteable.
;;;;
;;;;	term index table : a table key allowing lookup by matching terms to key meta terms.
;;;;	  - insert (<ttable> <stamp> <OID> <term{key}> <value>)
;;;;	  - delete (<ttable> <stamp> <OID> <term{key}>)
;;;;	  - commit (<ttable> <stamp> <OID> <term{key}>)
;;;;	  - undo   (<ttable> <stamp> <OID> <term{key}>)
;;;;	      * key locates, OID identifies
;;;;	      * stamp is active transaction.
;;;;	      * ??? some difficulty in finding correct occurence for commit.
;;;;		it may be that same OID occurs in multiple tents in various lists, so
;;;;		tent does not contain all OID occurences. 
;;;;		that is what the key is for key locates tent then oid and stamp update tent.
;;;;
;;;;	clash-p (<ttable> <stamp>) : BOOL
;;;;	 * any clashes in transaction.
;;;;	clash-p (<ttable>) : BOOL
;;;;	  * any clashes among commits.
;;;;
;;;;
;;;;	lookup (<ttable> <stamp> <term{instance}>) : <value>
;;;;	  * returns first value matching instance.
;;;;
;;;;	
;;;;	term index table :
;;;;	  - clashes : alist { clash list }
;;;;	      * key : term :test matchf
;;;;	      * value : tent.
;;;;	  - hash-table list  { table list }
;;;;	      * key : paramter-type of first parameter of term.
;;;;		one of OID | TOKEN | STRING | true {other}
;;;;	      * value : hash-table { constant table}
;;;;	          - key : value of first parameter of term or true.
;;;;		  - value : alist { basis list }
;;;;		      ? could be hash table on term-basis where hash-value
;;;;		        computed explictly (esp for t above)
;;;;		      * key : term{key} :test term-basis-eq.
;;;;		      * value : alist { match list }
;;;;			  - key : term {key} :test matchf
;;;;			  - value : tent
;;;;			  - ordered if inequality specified for table, ordering
;;;;			    only valid for values in same transaction thus must
;;;;			    be reorded at commit.
;;;;
;;;;	** Any keys with equal term-basis's may clash?
;;;;	** If not a clash then there should be a unique match. Thus lookup stops
;;;;	   when it finds any match.
;;;;	** If clashes not detected, then lookup pool must examine other table
;;;;	   as well as the appropriate constant table.
;;;;
;;;;	** considered meta list in place of basis list where test would be meta-eq.
;;;;	  - meta-eq : may be multiple entries with potential clashes.
;;;;	  + basis-eq : shorter basis list but longer match list.
;;;;	  + basis-eq : test cheaper.
;;;;	  + basis-eq : simpler.
;;;;	  + common case : OID constant and single item in match list.
;;;;	
;;;;	** keying lookup of constant first parameter seems a slighty un-robust.
;;;;	   Considered alternative of of finding first non-constant parameter, but
;;;;	   that may require lookups in multiple constant tables.
;;;;	   EG;
;;;;		key {$o:OID; fu:t, 1:n}
;;;;		    {o1:OID; bar:t, 2:n}
;;;;		instance {o1, bar} will try OID, then must also try token table.
;;;;	  Decision based on suspicion that common case is that first parameter will
;;;;	  be constant oid or token. String included in case, metaprl does not
;;;;	  distinquish strings and tokens.
;;;;	
;;;;	** Considred requiring a constant parameter of OID|TOKEN|STRING in a key.
;;;;	   OTOH {$i:n} == natural_number{$i:n} seems reasonable.
;;;;	
;;;;	** meta-parameter-p used only to decide if constant table lookup appropriate.
;;;;	  
;;;;	** meta-term-p not used at the moment but would like to allow for other
;;;;	   term index table implementation which might require it.
;;;;
;;;;	** meta-eq : the obvious structure equality using meta predicates.
;;;;	   We require that
;;;;	     - clash-p (a,b) -> meta-eq (a,b)
;;;;	     - match-p (a,b) -> meta-eq (a,b)
;;;;	     - and trivially note that meta-eq -> basis-eq
;;;;
;;;;	** Considered hard coding match and clash predicates in terms of meta predicates.
;;;;	   However, reduces flexibility.
;;;;	   Eg, matchf may utilize identity of match variables which meta-eq does not.
;;;;
;;;;	??? ability to warn when term-of shadows abs ???
;;;;	
;;;;	



;;;;	 
;;;;	Glitches :
;;;;
;;;;	serializability  : weakly serializable. commits serializable, transactions not.
;;;;	  - during a transaction an entry may have a value which
;;;;	    differs from the value it would have if it were serialized.
;;;;	      * entry e has value a. Transaction t1 and t2 run concurrently and
;;;;		t1 updates e to b. T2 reads a. t1 commits, and then t2 commits.
;;;;		When serialized t2 would see e with value b rather a as it did
;;;;		at runtime
;;;;
;;;;	Shadows :
;;;;	  Tables with term indices may have entries shadowed. A shadow
;;;;	is wimpy way of dealing with inconsistencies.
;;;;	  - do not want broadcasts to fail thus must allow shadowed term
;;;;	    index lookups as lib not necessarily privy to term index conflicts.
;;;;	      * TODO : lib must save commit stamp of activate of object so as to
;;;;		reactivate in commit order on restore. if multiple commits use last.
;;;;	        also table dump at start should be in commit order so that shadows
;;;;		rebuilt correctly.
;;;;	      * TODO lib should have appropriate hooks to allow detection and failure 
;;;;		of conflicts. Could be part of start protocol to download or identify
;;;;		hooks for table types.
;;;;		Could conceivably do hooks as remote calls (like code compile!). <- TODO
;;;;	
;;;;	shadows are unavoidable consequence of not allowing client to fail broadcasts.
;;;;	  - must have commit and failure due to conflict be atomic. Otherwise it allows
;;;;	    a window to slip conflict in.
;;;;	FTTB: ack that this might not be best model but do shadows and  later add client
;;;;	  failure to broadcasts. If they do not like shadows then they should use lib
;;;;	  hooks.
;;;;	

    


;;;;	See Order problem comments below.
;;;;
;;;;	  - < compares values not keys.
;;;;
;;;;	The ability to order values is supplied as an extension:
;;;;	  ??? clash-p becomes optional ??? 
;;;;	
;;;;	TODO : pool interface needs work. FTTB Simplify by not allowing clash with pools.
;;;;	lookup-pool (<ttable> <stamp> <term{instance}>) : <value> generator
;;;;	      * pool lookup not available if clashf specified.
;;;;		??? Alternatively, allow clash and clash entries simply
;;;;		??? generated out of order, ie in commit/update order rather than < order.
;;;;		??? OR lookup checks clashes first so that if < is really <= clash will be
;;;;		??? first among =.
;;;;	      * returns all values matching instance.
;;;;	      * fails if there is not at least one applicable value.
;;;;	      * if < specified then generates in order.
;;;;	      * generator.next -> <value>
;;;;		  * fails if eofp true.
;;;;		generator.eofp -> BOOL
;;;;	lookup-generator (<ttable> <term>) : (<stamp> -> <value> generator)
;;;;	      ?? generator generator to cache dform lists on terms??
;;;;	      ??? does modification of table alter a value generator in the wild ???
;;;;		if clashes not checked then can cache match list pointer.
;;;;	

;;;;	
;;;;	Assume that match-p matches all keys with same basis.
;;;;	then basis-alist contains single basis-cell and key alist
;;;;	of basis-cell contains all possible members of pool.
;;;;	
;;;;	then ordering key-alist orders pool,
;;;;	  - transactions change order, view from two diff concurrent transactions
;;;;	    may require diff orders. Simplify by restricting order info to only committed
;;;;	    data and uncompleted orde
;;;;	generator walks pool and returns visible values.
;;;;	generator know when stale and refreshes self?
;;;;	generator belongs to a transaction?
;;;;	
;;;;	display would normally happen outside of scope of transactions?
;;;;	thus all committed visible. however allow for transaction then
;;;;	usual visibility applies but generator becomes stale when transaction ends.
;;;;	generators can be cached on terms to generate dforms.

;;;;	
;;;;	order parameterized by transaction data.
;;;;	thus concurrent transactions may desire distinct orders for same pool.
;;;;	

;;;;	simple version
;;;;	  - order is parameterized by committed data only.
;;;;	  - all transactions see same order but visibility of elements is affected by transaction.
;;;;	  - resort only when ordering parameters committed.
;;;;	  - insert/deletes of elements maintains order.
;;;;	
;;;;	generator sees current basis-cell and can be parameterized by transaction
;;;;	to effect visibility but not order.
;;;;	
;;;;	generators should be invisible in non-native env. Ie if cached on term and term
;;;;	shared in two environments should only be visible in env caching.
;;;;
;;;;	this can be effected by giving clash table a clash predicate always return nil
;;;;	and then writing gen to cache basis-cell (assume basis cells not replaced).
;;;;	and using tent funcs for transaction visibility testing. 
;;;;	

;;;;	generator-stale-p (g)	: bool
;;;;	  - re-sorted or insert/delete committed since last reset.
;;;;	generator-reset (g) : start at beggining.
;;;;	generate (g) : value | nil
;;;;	generate-p : bool
;;;;	  * true if more values can be generated
;;;;	  * redundant as can test for nil but allows flow control without generate.
;;;;


;;;;	Order < : problemmatical wrt transaction independence.
;;;;	  - at the moment ordering is used in non critical areas such as dform choice
;;;;	    if used elsewhere then needs more work.
;;;;
;;;;	can compare values not present in transaction?
;;;;	order changes at commit. after commit must be merged with new values
;;;;	not visible at update and still not visible in transaction.
;;;;	some possible orders:
;;;;	  - lib list order
;;;;	  - commit order
;;;;	  - precedence tree - tree must be specified outside of current transaction.
;;;;	     tree update changes order, how do we cause resort?
;;;;	  - directory tree.
;;;;	need sort function for table when order changes due to outside influence.
;;;;	runs as transaction and therefore sorts only visible entries.
;;;;	unvisible expected to be sorted at thier commit. Resort must be atomic, ie
;;;;	no commits during sort.
;;;;
;;;;	  - order should not depend on table calling <.
;;;;	  - transaction can not depend on consistent order, ie order is not invariant
;;;;	    within transaction.
;;;;	UnGood scenario - abstraction lookup depends on some directory tree structure,
;;;;	and order changes in midst of transaction and similar instances then start expanding
;;;;	differently.
;;;;
;;;;	 +++ tree must be built constructively such that resort does not change currently
;;;;	     referenced trees, or built using tents so that pointers be parameterized by transaction change
;;;;
;;;;	 would expect rehash to be called from end-transaction processing of some other 
;;;;	 table, and thus would expect t
;;;;
;;;;	Transaction pertubations :
;;;;	  - match, other and clash list may contain unapplicable tents.
;;;;	    ie need to skip some tents based on stamp.
;;;;	  - clash : clash list always checked first for match.
;;;; 
;;;;	



;;;;	
;;;;	
;;;;	following comments apply to using tents as conses, deemed inappropriate
;;;;	for term-index table but may be appropriate for lib list.
;;;;	
;;;;	
;;;;	'(a b c) and transactiont moves b after c  so t sees '(a c b)
;;;;	
;;;;	Thus cdr of a in one case points to '(b c) and in the other (c b)
;;;;	Thus it is the cdr which is the entry.
;;;;	
;;;;	t-cons : has mulitple cdrs. is a tent.
;;;;	tent-list : like a cons but cdr is tent.
;;;;	 value
;;;;	 next
;;;;	 insert : by order or at a point.
;;;;	 remove
;;;;

;;;
;;;;	
;;;;

;; head of list ie pool pointer must be tent.
;; undo must remove cons as well as thin s-entries in remaining.
;; term index list different beast then lib list.
;;

;;;;	
;;;;	l  = a (-> d)
;;;;	insert b before d (a b d)  (a (-> b (-> d) -> d))
;;;;	insert c before d (a c d)  (a (-> c (-> d) -> b(-> d) -> d))
;;;;	commit b (a (-> c (-> d) -> b (-> d)))
;;;;	 must update ptr of a to point to b rather than d.
;;;;	commit c (-> b (-> c (-> d)))
;;;;	 must update ptr of b not ptr of a to point to c

;;;;	in term index list
;;;;	multiple deletes and adds of same entry by diff transactions possible.
;;;;	want to detect dups. 
;;;;	  - ie must survive commit of delete where obj is already gone.
;;;;	  - and add of object already added.

;;;;	Problemo:
;;;;	t1 activates object a
;;;;	t2 deletes object a
;;;;	
;;;;	t1 commits - a active
;;;;	t2 commits - a deleted but still active in clients??
;;;;	  - can commit detect active and fix? No.
;;;;	  - delete can detect active and fail!
;;;;	Moral : tent's do not obviate failure but simply allow
;;;;	 concurrent views. However, no object occurs modified in more than one view!
;;;;	
;;;;	t1 starts : a0
;;;;	t2 starts : a0
;;;;	t2 ends   : a1
;;;;	t3 starts : a1
;;;;	t4 starts : a1
;;;;	t4 ends : a2
;;;;	t5 starts : a2
;;;;	
;;;;	t1, t3, and t5 see three diff a's. however never more than one modified at a time. 
;;;;	if t1,3,or5 modify a then they should fail, however they are no longer locked
;;;;	 so can not modify if locked or if modified since transaction started.
;;;;	so all updates must be for same transaction and can not update a tent
;;;;	committed since transaction began. Lib must treat activates as a table with tent control.
;;;	ie activate is a write op requiring a lock.
;;;;
;;;;	summary : no read failures but still write failures.
;;;;
;;;;	if we add read locks then we do not freeze entire view only freeze those read.
;;;;	but :
;;;;	  - does not allow mods of read objects.
;;;;	  - difficult to add read lock for client local reads.
;;;;	  - difficult to coordinate all potential parties to a transaction.
;;;;	    ie want transaction to be dynamic, so we can compute participants rather
;;;;	    than declare.
;;;;	  - want to allow anonymous participation ??
;;;;	
;;;;	single thread for entire transaction : allow thread to roam, may read
;;;;	many objects, DO NOT WANT OVERHEAD OF DISTRIBUTING READ LOCKS.
;;;;	
;;;;	

;;;;	Tables needed : 
;;;;	
;;;;	Oid : lookup data by oid.
;;;;	  - 1-to-1 data-to-oid.
;;;;	    Source/Rules
;;;;
;;;;	Term : lookup data by term 
;;;;	  - 1-to-many term to data
;;;;	    Dforms
;;;;	  - 1-to-1 term to data
;;;;	      * however, multiple clashing inserts possible.
;;;;	    Abstractions. 
;;;;	

;;;;	new-transtixt (meta-parameter-p key-match-f key-clash-f)


(defun new-transtixt (meta-parameter-p key-match-f key-clash-f)
  (new-clash-term-index-table  meta-parameter-p
			       #'(lambda (x) (declare (ignore x)) nil)
			       key-match-f
			       key-clash-f))



;;;;	
;;;;	
;;;;	Tok Definition table : used to lookup definitions  
;;;;	 by tok.
;;;;	
;;;;	tok-table-insert (<table> <tok> <oid> <stamp> <seq> <data>)
;;;;	tok-table-delete (<table> <tok> <oid> <stamp> <seq>)
;;;;
;;;;	tok-table-undo   (<table> <tok> <oid> <stamp> <seq>)	: <data>
;;;;	tok-table-commit (<table> <tok> <oid> <stamp> <seq>)	: <data>	
;;;;	
;;;;	<data> must be returned whether completing an insert or a delete, ie
;;;;	completion of a delete must return data deleted.
;;;;	
;;;;	tok-table-lookup (<table> <tok> <stamp>)
;;;;	
;;;;	

(defun get-tok-tent (table tok oid)
  (cdr (assoc oid (gethash tok table) :test #'equal-oids-p)))

(defun add-tok-tent (table tok oid tent)
  (let ((curtent (get-tok-tent table tok oid)))
    (when  curtent
      ;;(setf a curtent b table c tok d oid e tent)
      (break "stt"))
    (setf (gethash tok table) (acons oid tent (gethash tok table)))))

(defun remove-tok-tent (table tok oid tent)
  (let ((curtent (get-tok-tent table tok oid)))
    (unless (and curtent (eq tent curtent))
      ;;(setf a curtent b table c tok d oid e tent)
      (break "stt"))
    (setf (gethash tok table)
	  (remove oid (gethash tok table) :key #'car :test #'equal-oids-p))))


(defun new-tok-table ()
  (make-hash-table :test #'eql))

(defun tok-table-insert (table tok oid s i data &optional note)
  ;;(setf -tok tok -oid oid -data data) (break "tti")
  (let ((tent (or (get-tok-tent table tok oid)
		  (let ((nt (new-tent)))
		    (add-tok-tent table tok oid nt)
		    nt))))
    (tent-insert tent s i oid data note)))

(defun tok-table-delete (table tok oid s i &optional note)
  (tent-delete (get-tok-tent table tok oid) s i oid note))

;; returns data undone.
(defun tok-table-undo (table tok oid s i)
  (let ((tent (get-tok-tent table tok oid)))
    (multiple-value-prog1 (tent-undo-data tent s i)
      (when (tent-empty-p tent)
	(remove-tok-tent table tok oid tent)))))


;; returns data committed.
(defun tok-table-commit (table tok oid s i)
  (let ((tent (get-tok-tent table tok oid)))
    (multiple-value-prog1 (tent-commit-data tent s i)
      (when (tent-empty-p tent) (remove-tok-tent table tok oid tent)))))
    

;; returns list of data
(defun tok-table-lookup (table tok s)
  ;;(setf a table b tok c s) (break "ttl")
  (mapcan #'(lambda (oid-tent)
	      (let ((data (tent-lookup (cdr oid-tent) s t)))
		;;(setf -data data -oid-tent oid-tent) (break "ttlu")
		(when data (list data))))
	  (gethash tok table)))
    

(defun tok-table-bound-p (table tok s)
  (some #'(lambda (oid-tent)
	    (and (tent-lookup (cdr oid-tent) s t) t))
	(gethash tok table)))

;; f : tok -> oid -> data
(defun tok-table-map (table stamp f)
  (maphash #'(lambda (tok tents)
	       ;; key is tok val is tent
	       (dolist (oid-tent tents)
		 (let ((data (tent-lookup (cdr oid-tent) stamp t)))
		   (when data
		     (funcall f tok (car oid-tent) data)))))
	   table))


;; f : tok  -> data
(defun tok-table-map-2 (table stamp f)
  (maphash #'(lambda (tok tents)
	       ;; key is tok val is tent
	        
	       (let ((defs (mapcar #'(lambda (oid-tent) (tent-lookup (cdr oid-tent) stamp t)) tents)))
		       
		 (when (cdr defs)
		   (funcall f tok defs))))
	   table))


;;;;	
;;;;	
;;;;	Oid Definition table : used to lookup definitions  
;;;;	 by oid.
;;;;	
;;;;	oid-table-insert (<table> <oid> <stamp> <seq> <data>)
;;;;	oid-table-delete (<table> <oid> <stamp> <seq>)
;;;;
;;;;	oid-table-undo  (<table> <oid> <stamp> <seq>)	: <data>
;;;;	oid-table-commit(<table> <oid> <stamp> <seq>)	: <data>	
;;;;	
;;;;	<data> must be returned whether completing an insert or a delete, ie
;;;;	completion of a delete must return data deleted.
;;;;	
;;;;	oid-table-lookup (<table> <oid> <stamp>)
;;;;	
;;;;	

(defun oid-table-insert (table o s i data &optional note)
  (let ((tent (or (gethash (stamp-of-oid o) table)
		  (let ((nt (new-tent)))
		    (setf (gethash (stamp-of-oid o) table) nt)
		    nt))))
    ;;(setf c tent) (break "oti")
    (tent-insert tent s i o data note)))

(defun oid-table-delete (table o s i &optional note)
  (tent-delete (gethash (stamp-of-oid o) table) s i o note))

(defun oid-commit-ok-p (tent stamp seq)
  (exists-p #'(lambda (sent)
		(and (in-transaction-p (stamp-of-s-entry sent) stamp)
		     (eql seq (sequence-of-s-entry sent))))
	    (pending-of-tent tent)))

;; returns data undone.
(defun oid-table-undo (table o s i)
  (let ((tent (gethash (stamp-of-oid o) table)))
    (when (and tent (oid-commit-ok-p tent s i)) 
      (multiple-value-prog1 (tent-undo-data tent s i)
	(when (tent-empty-p tent) (remhash (stamp-of-oid o) table))))))


;; returns data committed.
(defun oid-table-commit (table o s i)
  (let ((tent (gethash (stamp-of-oid o) table)))
    (when (and tent (oid-commit-ok-p tent s i)) 
      (multiple-value-prog1 (tent-commit-data tent s i)
	(when (tent-empty-p tent) (remhash (stamp-of-oid o) table))))))
    
(defun oid-table-lookup (table o s &optional nil-ok-p)
  (let ((tent (gethash (stamp-of-oid o) table)))
    (if tent
	(tent-lookup tent s nil-ok-p)
	(unless nil-ok-p
	  ;;(setf a o) (break)
	  (raise-error (oid-error-message (list o) '(oid table lookup not)))))))

    
(defun oid-table-bound-p (table o s)
  (let ((tent (gethash (stamp-of-oid o) table)))
    (when tent
      (and (tent-lookup tent s t) t))))
  
;; todo : 
(defun oid-table-map (table stamp f)
  (maphash #'(lambda (s tent)
	       ;; key is oid val is tent
	       (let ((data (tent-lookup tent stamp t)))
		 (when data
		   (funcall f (new-oid s) data))))
	   table))


(defun def-table-oid-count (table stamp)
  (let ((count 0))
    (oid-table-map (oid-table-of-definition-table table) stamp
		   #'(lambda (oid data)
		       (declare (ignore oid data))
		       (incf count)))
    count))


;;;;	def-table
;;;;	
;;;;	def-table-insert (<table> <key> <stamp> <seq> <definition>)
;;;;	def-table-delete (<table> <key> <stamp> <seq> <oid>)
;;;;
;;;;	def-table-undo   (<table> <key> <stamp> <seq>)
;;;;	def-table-commit (<table> <key> <stamp> <seq>) 	
;;;;	

(defun new-def-table (meta-parameter-p key-match-f key-clash-f)
  (new-transtixt  meta-parameter-p
		  key-match-f
		  key-clash-f))

(defun def-table-insert (table key s i def)
  ;;(setf a table b key c s d i e def) (break "dti")
  (transtixt-insert table
		    key
		    s i
		    (oid-of-definition def)
		    def)
  (values))

(defun def-table-delete (table key s i oid)
  (transtixt-delete table
		    key
		    s i
		    oid)
  (values))

(defun def-table-undo (table key s i)
  (transtixt-undo table
		  key
		  s i))

(defun def-table-commit (table key s i)
  (transtixt-commit table
		    key
		    s i))
		    
  


;;;
;;; definition table includes transaction stamp at time of creation. This is 
;;; an implementation device to allow caching of abstraction defs on abs instances.
;;; the table stamp is used to verify that the cached def matches the table.

(defstruct table-visibility
  name
  ml-accessor-string
  ml-accessor-f
  (assoc nil))

(defun name-of-table-visibility (v) (table-visibility-name v))
(defun assoc-of-table-visibility (v) (table-visibility-assoc v))

(defvar *table-visibilities* nil)

(defun new-table-visibility (n a)
  (let ((tv  (make-table-visibility :name n :ml-accessor-string a)))
    (push tv *table-visibilities*)
    tv))

(defun table-visibility-reset ()
  (dolist (tv *table-visibilities*)
    (setf (table-visibility-ml-accessor-f tv) nil)))

(defun get-visible-definitions (v index)
  (funmlcall (or (table-visibility-ml-accessor-f v)
		 (setf (table-visibility-ml-accessor-f v)
		       (ml-text (table-visibility-ml-accessor-string v))))
	     index))



(defstruct definition-table
  (flags nil)
  
  (table-stamp nil)		;;; current transaction stamp at time of creation.
  (table-stamp-term nil)	;;; For remote tables stamp from library.
  ;;(table-stamp-string "")

  (process-id nil)
  
  ;; message tags
  (tags nil)
  (clash-tags nil)
  (lookup-tags nil)
  (oid-lookup-tags nil)

  ;; term-index table
  (table nil)
  (key-clash-f  #'(lambda (x y) (declare (ignore x y)) nil))
  (key-match-f nil)
  (order-f #'(lambda (x y) (declare (ignore x y)) t))
  (key-f nil)
  (keys-f nil)
  (visibility nil)

  ;; oid lookup
  (oid-table nil)		; optional hash table to allow lookup via oid.

  ;; export/import
  ;;(export-f nil)
  (import-f nil)
  (import-aux-f nil)
  (delete-f nil)
  (insert-f nil)
  (commit-f nil)
  (undo-f nil)

  (create-f nil)
  ;;(store-f nil)
  (activate-f nil)
  (deactivate-f nil)
  (allow-f nil)
  (disallow-f nil)
  (move-f nil)

  (touch-history-p t)
  
  ;; intertable clashes
  (inter-clash-tables-f nil)
  (caches nil) ;; cleard at any insert or delete.

  (quickdepref-index 0)
  )

(define-flags (definition-table)
    ((touched t nil))
  )


(defun definition-table-get-cache (ct tag)
  (assoc tag (definition-table-caches ct)))

(defun definition-table-put-cache (ct tag c)
  (setf (definition-table-caches ct)
	(cons (cons c tag) (definition-table-caches ct))))


;;;;	RLE TODO : definition-table table-stamp
;;;;	RLE TODO : I believe it is intended to be used by having remote tables
;;;;	RLE TODO : inherit a lib stamp and then verify incoming defs are appropriate.
;;;;	RLE TODO : 
;;;;	RLE TODO : Found where this is done:
;;;;	RLE TODO : This is being done by !connect protocol. Needs to be doc'd.

;; export is vestigial, lib writes substance to db, and tables import from substance read from
;; db, there is no role for export.

(defmacro define-definition-table (stamp
				   tags
				   term-lookup-p
				   ;;ordered-p
				   &key
				   meta-parameter-p
   				   key-match-f
				   key-f keys-f order-f key-clash-f
				   ;;export-f store-f
				   import-f create-f delete-f insert-f commit-f undo-f replace-f
				   import-aux-f
				   activate-f deactivate-f 
				   allow-f disallow-f 
				   inter-clash-tables-f
				   no-touch-history
				   visibility
				   (make-f 'make-definition-table))
  `(prog1
	(funcall #',make-f
	 :flags (init-default-definition-table-flags-a)	 
	 :table-stamp ,stamp
	 :table-stamp-term (when ,stamp (stamp-to-term ,stamp))
	 ;;:table-stamp-string (when ,stamp (stamp-to-string ,stamp))
	 :tags (append ,tags '(definition-table))
	 :clash-tags (append '(definition-table) ,tags '(insert clash))
	 :lookup-tags (append '(definition-table) ,tags '(lookup))
	 :oid-lookup-tags (append '(definition-table) ,tags '(oid-lookup))

	 :oid-table (new-oid-table)
      
	 ,@(when term-lookup-p
	     (unless (or key-match-f (eql 'group term-lookup-p))
	       (error (format-string "~a define-definition-table: term-lookup-p = t requires :key-match-f." 
				     term-lookup-p)))
	     (when (and key-clash-f (null meta-parameter-p))
	       (error "Define-definition-table: term-lookup-p and clash requires :meta-parameter-p."))

	     `(:table (if :key-clash-f
			  (new-def-table ,meta-parameter-p
					 ,key-match-f
					 ,key-clash-f))))
	 ,@(when visibility
	     `(:visibility ,visibility))
     
	 ,@(when key-f
	     `(:key-f ,key-f))

	 ,@(when keys-f
	     `(:keys-f ,keys-f))

	 ,@(when order-f
	     `(:order-f ,order-f))
	 
	 ;;,@(when export-f
	 ;;`(:export-f ,export-f))
	 ,@(when import-f
	     `(:import-f ,import-f))
	 ,@(when import-aux-f
	     `(:import-aux-f ,import-aux-f))
	 ,@(when create-f
	     `(:create-f ,create-f))
	 ,@(when delete-f
	     `(:delete-f ,delete-f))
	 ,@(when insert-f
	     `(:insert-f ,insert-f))
	 ,@(when commit-f
	     `(:commit-f ,commit-f))
	 ,@(when replace-f
	     `(:replace-f ,replace-f))
	 ,@(when undo-f
	     `(:undo-f ,undo-f))
	 ;;,@(when store-f
	     ;;`(:store-f ,store-f))
	 ,@(when activate-f
	     `(:activate-f ,activate-f))
	 ,@(when deactivate-f
	     `(:deactivate-f ,deactivate-f))
	 ,@(when allow-f
	     `(:allow-f ,allow-f))
	 ,@(when disallow-f
	     `(:disallow-f ,disallow-f))
	 ,@(when inter-clash-tables-f
	     `(:inter-clash-tables-f ,inter-clash-tables-f))


	 ,@(when no-touch-history
	     `(:touch-history-p (not ,no-touch-history))))

      (advance-sequence)))

(defmacro define-definition-table-make (stamp
				   tags
				   term-lookup-p
				   ;;ordered-p
				   &key
				   meta-parameter-p
   				   key-match-f
				   key-f keys-f order-f key-clash-f
				   ;;export-f store-f
				   import-f create-f delete-f insert-f commit-f undo-f replace-f
				   import-aux-f
				   activate-f deactivate-f 
				   allow-f disallow-f 
				   inter-clash-tables-f
				   no-touch-history
				   visibility
				   )
  `(prog1
	(make-definition-table
	 :flags (init-default-definition-table-flags-a)	 
	 :table-stamp ,stamp
	 :table-stamp-term (when ,stamp (stamp-to-term ,stamp))
	 ;;:table-stamp-string (when ,stamp (stamp-to-string ,stamp))
	 :tags (append ,tags '(definition-table))
	 :clash-tags (append '(definition-table) ,tags '(insert clash))
	 :lookup-tags (append '(definition-table) ,tags '(lookup))
	 :oid-lookup-tags (append '(definition-table) ,tags '(oid-lookup))

	 :oid-table (new-oid-table)
      
	 ,@(when term-lookup-p
	     (unless (or key-match-f (eql 'group term-lookup-p))
	       (error (format-string "~a define-definition-table: term-lookup-p = t requires :key-match-f." 
				     term-lookup-p)))
	     (when (and key-clash-f (null meta-parameter-p))
	       (error "Define-definition-table: term-lookup-p and clash requires :meta-parameter-p."))

	     `(:table (if :key-clash-f
			  (new-def-table ,meta-parameter-p
					 ,key-match-f
					 ,key-clash-f))))
	 ,@(when visibility
	     `(:visibility ,visibility))
     
	 ,@(when key-f
	     `(:key-f ,key-f))

	 ,@(when keys-f
	     `(:keys-f ,keys-f))

	 ,@(when order-f
	     `(:order-f ,order-f))
	 
	 ;;,@(when export-f
	 ;;`(:export-f ,export-f))
	 ,@(when import-f
	     `(:import-f ,import-f))
	 ,@(when import-aux-f
	     `(:import-aux-f ,import-aux-f))
	 ,@(when create-f
	     `(:create-f ,create-f))
	 ,@(when delete-f
	     `(:delete-f ,delete-f))
	 ,@(when insert-f
	     `(:insert-f ,insert-f))
	 ,@(when commit-f
	     `(:commit-f ,commit-f))
	 ,@(when replace-f
	     `(:replace-f ,replace-f))
	 ,@(when undo-f
	     `(:undo-f ,undo-f))
	 ;;,@(when store-f
	     ;;`(:store-f ,store-f))
	 ,@(when activate-f
	     `(:activate-f ,activate-f))
	 ,@(when deactivate-f
	     `(:deactivate-f ,deactivate-f))
	 ,@(when allow-f
	     `(:allow-f ,allow-f))
	 ,@(when disallow-f
	     `(:disallow-f ,disallow-f))
	 ,@(when inter-clash-tables-f
	     `(:inter-clash-tables-f ,inter-clash-tables-f))


	 ,@(when no-touch-history
	     `(:touch-history-p (not ,no-touch-history))))

      (advance-sequence)))


(defun table-stamp-of-definition-table (dt) (definition-table-table-stamp dt))
(defun table-stamp-term-of-definition-table (dt) (definition-table-table-stamp-term dt))
;;(defun table-stamp-string-of-definition-table (dt) (definition-table-table-stamp-string dt))

(defun tags-of-definition-table (dt) (definition-table-tags dt))
(defun clash-tags-of-definition-table (dt) (definition-table-clash-tags dt))
(defun lookup-tags-of-definition-table (dt) (definition-table-lookup-tags dt))
(defun oid-lookup-tags-of-definition-table (dt) (definition-table-oid-lookup-tags dt))

(defun table-of-definition-table (dt) (definition-table-table dt))
(defun order-f-of-definition-table (dt) (definition-table-order-f dt))
(defun key-f-of-definition-table (dt) (definition-table-key-f dt))
(defun keys-f-of-definition-table (dt) (definition-table-keys-f dt))

;;(defun export-f-of-definition-table (dt) (definition-table-export-f dt))
(defun import-f-of-definition-table (dt) (definition-table-import-f dt))
(defun import-aux-f-of-definition-table (dt) (definition-table-import-aux-f dt))
(defun create-f-of-definition-table (dt) (definition-table-create-f dt))
(defun delete-f-of-definition-table (dt) (definition-table-delete-f dt))
(defun insert-f-of-definition-table (dt) (definition-table-insert-f dt))
(defun replace-f-of-definition-table (dt) (definition-table-commit-f dt))
(defun commit-f-of-definition-table (dt) (definition-table-commit-f dt))
(defun undo-f-of-definition-table (dt) (definition-table-undo-f dt))
;;(defun store-f-of-definition-table (dt) (definition-table-store-f dt))
(defun activate-f-of-definition-table (dt) (definition-table-activate-f dt))
(defun deactivate-f-of-definition-table (dt) (definition-table-deactivate-f dt))
(defun allow-f-of-definition-table (dt) (definition-table-allow-f dt))
(defun disallow-f-of-definition-table (dt) (definition-table-disallow-f dt))

(defun oid-table-of-definition-table (dt) (definition-table-oid-table dt))
(defun visibility-of-definition-table (dt) (definition-table-visibility dt))

(defun quickdepref-index-of-definition-table (d) (definition-table-quickdepref-index d))

(defun inter-clash-tables-of-definition-table (dt)
  (when (definition-table-inter-clash-tables-f dt)
    (funcall (definition-table-inter-clash-tables-f dt))))
  
(defun set-definition-table-insert-f (dt f)
  (setf (definition-table-insert-f dt) f))

;; can only be reset if null or eq unless forced.
(defun definition-table-set-stamp (dt stamp-term forcep)
  (let ((cur-stamp (table-stamp-of-definition-table dt))
	(stamp (term-to-stamp stamp-term))
	)

    (unless (or forcep
		(equal-stamps-p cur-stamp stamp)
		(null cur-stamp))
	(raise-error (error-message '(definition table set stamp)
				    (table-stamp-term-of-definition-table dt))))

    (setf (definition-table-table-stamp dt) stamp
	  (definition-table-table-stamp-term dt) stamp-term
	  ;;(definition-table-table-stamp-string dt) (stamp-to-string stamp)
	  )))


;; equal test seems proper, but eql test suffcient since
;; updates happen from global which is set once bound.
(defun equal-ref-environments-p (a b) (eql a b))
(defun tag-of-definition-table (table) (car (tags-of-definition-table table)))

(defun update-definition-table-visibility (table)

  (let ((tv (visibility-of-definition-table table)))

    (when tv
      (let ((re *ref-environment*))
	(when re
	  (let ((re-index (car re))
		(re-assoc (cddr re)))

	    (let ((re-table-index (cdr (assoc (name-of-table-visibility tv) re-assoc))))

	      (when re-table-index
		(let ((tindex (cons (cons (tag-of-definition-table table) re-index)
				    re-table-index))
		      (oids (get-visible-definitions tv re-table-index)))

		  (setf (table-visibility-assoc tv)
			(cons (cons tindex oids)
			      (mapcan #'(lambda (v)
					  (when (member (caar v) *ref-environment-stack*
							:key #'car
							:test #'equal-ref-environments-p)
					    (list v) ))
				      (table-visibility-assoc tv))))

		  (dolist (oid oids)
		    (let ((def (with-error-to-warn
				   (definition-lookup-by-oid table oid nil
							     (current-transaction-stamp) t))))
		      ;;(when *process-break* (break "fu")) 
		      (when (and (boundp '|*ref-current-objects*|)
				 (member oid *ref-current-objects* :test #'equal-oids-p))
			(break "avoid env clash")
			)
		      ;;(unless def (setf -table table -oid oid ) (break "udtv nil")(fooe))
		      (when def
			(update-visibility-of-definition def tindex table))
		      )))

		re-table-index))))))))


(defmacro reid-of-table-index (a) `(cdar ,a))
(defmacro index-of-table-index (a) `(cdr ,a))
(defmacro table-index-of-table-visibility-element (a) `(car ,a))


(defun get-definition-table-visibility (table)
  (let ((tv (visibility-of-definition-table table)))
    (when tv
      (let ((re *ref-environment*))
	(when re
	  (let ((re-id (car re)))
	    ;;(setf -re-id (car re) -tv tv) (break "gdtv")
	    (let ((e (assoc re-id (assoc-of-table-visibility tv)
			    :test #'equal-ref-environments-p
			    :key #'(lambda (key) (reid-of-table-index key)))))
	      (if e
		  (index-of-table-index (table-index-of-table-visibility-element e))
		  (update-definition-table-visibility table)))))))))


;; a local view of import of substance and abstract structure to be data
;; in definiton table.
;;
;; should be used to derive structures.
;;

;;;;	
;;;;	when ref_environment is bound, it is paired with stamp in order to identify
;;;;	binding occurence. 
;;;;	
;;;;	the definition maintains an assoc list of transaction stamp and ref_environment binding
;;;;	stamp to visibility index. 
;;;;	
;;;;	When assoc list updated it is safe to remove elements for transactions which are
;;;;	 no longer active. Thus there is no need to eager remove elements when binding
;;;;	 scope of ref_environment is left.
;;;;	
;;;;	If def removed from table  then can not find def to check if visible so visibility
;;;;	  implicitly fails.
;;;;	If def added to table then insertion must check ref_environment binding and update
;;;;	  definition visibility accordingly.
;;;;
;;;;	The definition table has a ref_environment name to faciltate finding the 
;;;;	visibility status of member definitions at insertion time.
;;;;	  - but must look at all ref-environments at insertion. Since even if 
;;;;	    inserting in scope of one may want accessible in outer scope if same
;;;;	    same transaction.
;;;;	
;;;;	<ref-env-id>	: (<stamp{transation}> . <sequence-stamp>)
;;;;	<ref-env>	: <ref-env-id> . (<oid{ref_state index}> . <ref-assoc>)
;;;;	<ref-env-stack>	: <ref-env> list
;;;;	
;;;;	<ref-assoc>	: <tok> . <oid> list
;;;;	  * oid is index in a ref state.
;;;;	
;;;;	*ref-environment*	: <ref-env-id> . <oid{ref_env index}> . <ref-environment>
;;;;	 
;;;;	<table-index>	: <ref-env-id> . <oid{table-index}>
;;;;	  * if type is table-type then there exists (name . <oid{table-index}>) in <ref-environment>
;;;;	 
;;;;	<def-visibility>	: <table-index> list
;;;;	  * definition contains list of def visibility
;;;;	 
;;;;	<table-visibility>	: (<table-index> . <oid{visible}> list) list
;;;;	  * table contains list of table visibility
;;;;	
;;;;	visibility-of-definition (def)
;;;;	  * use ref-env-id to find table-visibility.
;;;;	
;;;;	update-definition-table-visibility (<table>)
;;;;	  * build table-index and push on table-visibility-assoc
;;;;	  * foreach visible def update-visibility-of-definiton with table-index.
;;;;	    
;;;;	update-visibility-of-definition 
;;;;	  * pushes <table-index> onto definition-visibility-assoc
;;;;	  * filters definition-visibility-assoc and removes
;;;;	
;;;;	

;;;;	
;;;;	ephemeral ref-envs. 
;;;;	  - if lemma or abs, add object to refenv of objects refenv and recurse.
;;;;	
;;;;	
;;;;	


(defvar *ref-environment* nil)		;; nil should allow global access.
(defvar *ref-environment-term* nil)	;; nil should allow global access.
(defvar *ref-environment-stack* nil)	;; nil should allow global access.
(defvar *ref-environment-abstractions-index* nil)

(defvar *enter-exit-ref-environment-f*)

(setf *enter-exit-ref-environment-f* nil)

(defun ref-environment-enter (index)
  (unless (ivoid-term-p index)
    (when (null *enter-exit-ref-environment-f*)
      (setf *enter-exit-ref-environment-f* (ml-text "ref_environment_enter, ref_environment_exit")))
    (funmlcall (car *enter-exit-ref-environment-f*) index)))

(defun ref-environment-exit ()
  (when (null *enter-exit-ref-environment-f*)
      (setf *enter-exit-ref-environment-f* (ml-text "ref_environment_enter, ref_environment_exit")))
    (funmlcall (cdr *enter-exit-ref-environment-f*) nil))

(defmacro with-reference-environment-term (indext &body body)
  (let ((rrt (gentemp)))
    `(unwind-protect
      (let ((,rrt (unless (or (null ,indext) (ivoid-term-p ,indext))
		    (ref-environment-enter ,indext))))
    
	(let* ((*ref-environment-term* (or ,indext (ivoid-term)))
	       (*ref-environment* (when *ref-environment-term*
				    (cons (cons (current-transaction-stamp) (new-sequence-stamp))
					  (cons *ref-environment-term* ,rrt))))
	       (*ref-environment-stack* (when *ref-environment-term* (cons *ref-environment* *ref-environment-stack*)))
	       (*ref-environment-abstractions-index* (when *ref-environment-term* 'not-initialized)))
	    ,@body))
      (progn (ref-environment-exit)
	     (when (and *ref-environment-term* (not (ivoid-term-p *ref-environment-term*)))
	       (ref-environment-enter *ref-environment-term*))))))

(defmacro without-ref-environment (&body body)
  `(unwind-protect
    (let ((*ref-environment-term* nil)
	  (*ref-environment* nil)
	  (*ref-environment-abstractions-index* nil))

      (ref-environment-exit)
      
      ,@body)

    (when (and *ref-environment-term* (not (ivoid-term-p *ref-environment-term*)))
      (ref-environment-enter *ref-environment-term*))))

(defun in-refenv-p ()
  (not (null *ref-environment*)))

(defmacro with-reference-environment (index &body body)
  `(with-reference-environment-term (unless (or (null ,index) (dummy-object-id-p ,index))
				      (ioid-term ,index))
    ,@body))

(defstruct def-visibility
  (assoc nil)	; assoc list of stamps and indices.
  )


(defstruct definition
  (table-stamp nil)			; for independent testing of validity of definition wrt a table.
  (keys nil)				; for term-sig lookup/insert

  (dependency nil)			; dependency of data from which def is derived.
  (substance nil)

  ;; not sure if this is best place for name but any def may be derived from object with
  ;; name, most sub structs will need a name field, but possibly some will not use it.
  (name nil)
  (visibility nil)

  (quickdepref-index nil)
  (table-containing nil) ;; up pointer to containing table.
  )

(defun table-stamp-of-definition (d) (definition-table-stamp d))
(defun dependency-of-definition (d) (definition-dependency d))
(defun name-of-definition (d) (definition-name d))
(defun quickdepref-index-of-definition (d) (definition-quickdepref-index d))
(defun table-containing-of-definition (d) (definition-table-containing d))

(defun visibility-of-definition (def)
  (when *ref-environment*
    (let ((dv (definition-visibility def)))
      (when dv
	(cdr (assoc (car *ref-environment*)
		    (def-visibility-assoc dv)
		    :test #'equal-ref-environments-p
		    :key #'cdr
		    ))))))

;; definition may be in more than one table,
;; ie stm is both a lemma and abstraction(termof).

;; adds new and thins current.
(defun update-visibility-of-definition (def tindex table)
  (let ((v (definition-visibility def))
	(tag (tag-of-definition-table table)))
    (when *ref-environment*
      (unless v
	(setf v (setf (definition-visibility def) (make-def-visibility))))

      (setf (def-visibility-assoc v)
	    (cons tindex
		
		  ;; filter stale
		  (mapcan #'(lambda (a)
			      (when (or (not (eql tag (caar a)))
					(member a
						(assoc-of-table-visibility
						 (definition-table-visibility table))
						:test #'equal-ref-environments-p :key #'car))
				(list a)))
			  (def-visibility-assoc v))
		  )))))


;; initializes from table
(defun init-visibility-of-definition (def table)
  (let ((vis (visibility-of-definition-table table)))
    ;;(when (definition-visibility def) (break "ivod"))
    (when vis
	  (let ((doid (oid-of-definition def)))
	    (if (null (definition-visibility def))
		(setf (definition-visibility def)
		      (make-def-visibility
		       :assoc
		       (mapcan #'(lambda (v)
				   (when (member doid
						 (cdr v)
						 :test #'equal-oids-p)
				     (list (car v))))
			       (assoc-of-table-visibility vis))))
		;; stm defs are used in botn abs and lemma tables and thus
		;; will be initted twice.
		(let ((dv (definition-visibility def)))
		  (setf (def-visibility-assoc dv)
			(append (mapcan #'(lambda (v)
					    (when (member doid
							  (cdr v)
							  :test #'equal-oids-p)
					      (list (car v))))
					(assoc-of-table-visibility vis))
				(def-visibility-assoc dv)))))))))


(defun definition-visible-p (def index)
  ;;(setf -def def -index index) (break "dvp")
  (let ((v (visibility-of-definition def)))
    (or (null index)
	(when v (equal-oids-p index v)))))


(defun substance-of-definition (d type)
  (let ((s (definition-substance d)))
    (when s
      (if (data-provided-p s)
	  s
	  (setf (definition-substance d) (provide-data s type))))))


(defun ephemeral-substance-of-definition (d type)
  (let ((s (definition-substance d)))
    (when s
      (if (data-provided-p s)
	  s
	  (provide-data s type)))))

(defun otable-p (dtable)
  (and (oid-table-of-definition-table dtable) t))
  
(defun oid-of-definition (d &optional table)
  (let ((dep (dependency-of-definition d)))
    (if dep
	(oid-of-dependency dep)
	(when (and table (otable-p table))
	  (raise-error (error-message (append (tags-of-definition-table table)
					      '(definition no-oid))))))))

;; caches value of key function, modification is safe.
(defun keys-of-definition (d table)
  (or (definition-keys d)
      (let ((key-f (key-f-of-definition-table table))
	    (keys-f (keys-f-of-definition-table table)))
	;;(setf a d b table)
	(cond
	  (keys-f (setf (definition-keys d) (funcall keys-f d))) ; safe : definition-key cache.
	  (key-f  (setf (definition-keys d) (list (funcall key-f d))))
	  (t (raise-error (oid-error-message (list (oid-of-definition d))
					     (append (tags-of-definition-table table) '(key missing)))))))))

;; caches value of key function, modification is safe.
(defun key-of-definition (d table)
  (car (keys-of-definition d table)))


(defun a-transaction-stamp ()
  (or (with-ignore (current-transaction-stamp))
      (local-transaction-stamp)))
  

(defun definition-lookup-by-oid (table oid &optional nil-ok-p tstamp dont-note)
  (let ((def (oid-table-lookup (oid-table-of-definition-table table)
			       oid
			       (or tstamp
				   (a-transaction-stamp)
				   )
			       nil-ok-p)))
    (when (and def (not dont-note))
      ;;(when (eql *statements* table) (break "dlbo"))
      ;;(setf -def def) (break)
      ;;(let* ((objc (library-object-objc def))
      ;;(prop (when objc (property-of-objc objc 'name))))
      ;;(when prop
      ;;(setf -def def -oid oid) (break "dlbo")
      ;;(format t "access-definition-object-id ~a~%" (value-of-parameter (car (parameters-of-term prop ))))))
      (dependency-note-reference 'access-definition-object-id
				 (dependency-of-definition def)))
    def))
    


(defun definition-lookup (table term &optional dont-note p)
  ;;(setf a term) (break)
  (let ((def (transtixt-lookup (table-of-definition-table table)
			       term
			       (a-transaction-stamp)
			       nil
			       p)))

    ;;(unless dont-note (break "dlbt"))
    (when (and def (not dont-note))
      ;;(break "dl")
      (dependency-note-reference 'access-definition-term
				 (dependency-of-definition def)))
    def))

(defun definition-table-map (table stamp f)
  (without-dependencies
   (maphash #'(lambda (ostamp tent)
		;; key is oid val is tent
		(let ((data (tent-lookup tent stamp t)))
		  (when data
		    (funcall f (new-oid ostamp) data))))
	    (oid-table-of-definition-table table))))



;;;;	
;;;;	hash table for oids.
;;;;	
;;;;	
;;;;	table_proxy {<type>:t; <name>:t}(<filter args>; <sort args>)
;;;;	
;;;;	type indicates a filter and sort method ?
;;;;	
;;;;	list
;;;;	refreshp ?
;;;;	
;;;;	array : list proxy for scroller 
;;;;	
;;;;	make_list_proxy (<term> list 
;;;;	
;;;;	
;;;;	

(defun hashoid-get (otable oid)
  (gethash (stamp-of-oid oid) otable))

(defun hashoid-set (otable oid v)
  (setf (gethash (stamp-of-oid oid) otable) v))


(defstruct proxy-table
  table
  (array nil)
  (stale t)
  )

(defun table-of-proxy-table (ptab) (proxy-table-table ptab))
(defun array-of-proxy-table (ptab) (proxy-table-array ptab))
(defun proxy-table-stale-p (ptab) (proxy-table-stale ptab))

(defun make-oproxy-table ()
  (make-proxy-table :table (make-hash-table :test #'equal)))

(defun oproxy-table-touch (ptable)
  (setf (proxy-table-stale ptable) t))

(defun oproxy-array-fill (ptable &optional filterp mungef sortf)

  (if (not (proxy-table-stale-p ptable))
      (array-of-proxy-table ptable)

      (let ((a (array-of-proxy-table ptable))
	    (table (table-of-proxy-table ptable)))
    
	(if (or (null a)
		(< (array-dimension a 0) (hash-table-count table)))
	    (setf a (setf (proxy-table-array ptable)
			  (make-array (hash-table-count table) :fill-pointer 0)))
	    (setf (fill-pointer a) 0))

	(maphash #'(lambda (oid def)
		     (when (or (null filterp)
			       (funcall filterp oid def))
		       (let ((term (term-of-term-def def)))
			 (vector-push (if mungef (funcall mungef term) term) a) )))
		 table)
    
	(when sortf
	  (sort a sortf))

	a)))

(defvar *proxy-tables-alist* nil)


(defun oproxy-table-find (oid)
  (or (cdr (assoc oid *proxy-tables-alist* :test #'equal-oids-p))
      (let ((ptable (make-oproxy-table)))
	(setf *proxy-tables-alist* (acons oid ptable *proxy-tables-alist*))
	ptable)))



;;;; following no longer is applicable

;;;
;;;	oid-map
;;;
;;;	having an index and using the index as oid seems like a win.
;;;	fttb do without and see how things shape up.
;;;;	
;;;;	needs to be under transaction control so that concurrent transactions 
;;;;	see consistent map.
;;;;	
;;;;	could be implemented as oid-table or definition-table without term lookup

;;;;	oid-map will be in environments resource list for broadcast
;;;;	routing purposes, but it will also be explicitly bound
;;;;	for oid-lookup purposes. In explicit case the hash-table itself
;;;;	is bound while in resource case the structure containing the
;;;;	hash table is present.



;;;;	oid-table is usual definition-table without term lookup.
;;;;    ie library table.
;;;;	
;;;;	if oid a binds oc A and b binds B and a mapped to b. then a is in 
;;;;	oid table binding A but is also mapped to b. should we require A be 
;;;;	deleted prior to map to b?? yes?
;;;;	
;;;;	
;;;;	lib_insert oid oc should
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Since lookup maps oid, it seems the oid table could
;;;;	be a usual hash-table on stamps! 
;;;;	  - lib generates broadcasts such that oid maps to its own stamp.
;;;;	  - stamp is unique to data.
;;;;	But, it is possible for an object to be active in one transaction
;;;;	but not another. Thus, lookup must be parameterized by transactions.
;;;;	Tents are overkill, but effective.
;;;;	
;;;;	
;;;;	We assume that broadcast oids map to themselves and thus do not
;;;;	implicitly map oids in table modifications. Lookup does implicitly
;;;;	map oids as they do not originate in broadcasts.
;;;;	

;;;;	
;;;;	oid-map-lookup (<oid> <oid-map>)	: <oid>
;;;;	
;;;;	oid-map-update (<stamp{transaction}> <int{sequence}>
;;;;			<oid{old}> <oid{new}> 
;;;;			<oid-map>)
;;;;	  : NULL
;;;;	
;;;;	oid-map-commit (<stamp{transaction}> <int{sequence}> <oid> <oid-map>)	: NULL
;;;;	oid-map-undo   (<stamp{transaction}> <int{sequence}> <oid> <oid-map>)	: NULL
;;;;	

(defstruct oid-map-table
  (map  (make-hash-table :test #'equal))
  )

(defun map-of-oid-map-table (omt)
  (oid-map-table-map omt))

;;;;	
;;;;	library is mapping of oids to object contents.
;;;;	mapping occures in two parts. Oids can be mapped to oids.
;;;;	and oids can be assoc'd with oc.
;;;;
;;;;	oid-map handles map library table handles oc.
;;;;	
;;;;	assoc (a O) followed by map a b, then a remains in definition table but
;;;;	can not be accessed??
;;;;	
;;;;	where is active bit?
;;;;	
;;;;	
;;;;	some components may want map updates but not oc updates.
;;;;	could have diff table types map to same table, but then 
;;;;	sentry can filter unwanted type.
;;;;	

;; todo : oids should be locked in transactions when mapped, bound or unbound.
;; thus if simultaneous transactions mess with same oid then one fails.

(defun allocate-oid-map (stamp name)
  (declare (ignore name stamp))
  
  (make-oid-map-table))

(defun oid-map-lookup (oid map)
  (let ((stamp (local-transaction-stamp)))
    (labels ((look (o)
	       (let* ((tent (gethash (stamp-of-oid o) map))
		      (e (when tent (tent-lookup tent stamp t))))
		 (if (oid-p e)
		     (look e)
		     o))))
      (look oid))))

(defun oid-lookup (oid)
  
  ;;(oid-map-lookup oid (oid-map))
  oid)
  
(defun oid-mapped-p (oid)
  (let* ((tent (gethash (stamp-of-oid oid) (oid-map)))
	 (e (when tent (tent-lookup tent (local-transaction-stamp) t))))

    (when e
      (oid-p e))))

  

;; stamp and seq from map bcast
(defun oid-map-update (stamp seq ooid noid map)
  (let ((tent (or (gethash (stamp-of-oid ooid) map)
		  (new-tent))))
    (tent-insert tent stamp seq t noid)))

(defun oid-map-commit (stamp seq ooid map)
  (tent-commit (gethash (stamp-of-oid ooid) map)
	       stamp seq))

(defun oid-map-undo (stamp seq ooid map)
  (tent-undo (gethash (stamp-of-oid ooid) map)
	     stamp seq))



;;;;	
;;;;	Would like to allow arbitrary terms to be ref-environment indices
;;;;	  - as kludge instead of making ref states take term indices, intead
;;;;	    assoc terms to obids	   
;;;;	
;;;;	
;;;;	- produce ref_env from term. (but this requires obid!)
;;;;	- assoc with some index (ie the term).
;;;;	- use index.
;;;;	
;;;;	
;;;;	with_ref_environment_index term (f:object_id -> *)
;;;;	  * allows using term as index, if ref_env defined using supplied id.
;;;;	
;;;;	
;;;;	kludge because right thing is to modify refstate to use terms as indices.
;;;;	

(defvar *re-index-kludge-assoc* nil)

(defun get-temp-re-obid (term)
  (let ((obid (new-object-id)))
    (push (cons term obid)
	  *re-index-kludge-assoc*)
    obid
    ))

(defun free-temp-re-obid (term)
  (let ((e (assoc term *re-index-kludge-assoc* :test #'compare-terms-p)))

    (unless e (break "ftro"))
    (setf *re-index-kludge-assoc* (remove term *re-index-kludge-assoc* :key #'car :test #'compare-terms-p))
    (values)
    ))

;; it is an error to use the object_id outside of scope of lambda closure.
;; might make sense to check that returned value is not a closure (nor contains a closure?)
;; to reduce posssibility of use. Of course still possible to assign to letref and use later.
;; since cannot guarauntee then do nothing.
;; another method might be to dynamically bind some global in closure  saying this is current and if used
;; when global not bound to same obid then problem. Still a hole in that obid could be reused and
;; bound.
;; don't reuse obids, but clear ref_state of obid at exit of closure!

(defvar *ref-env-remove* nil)

(defun ref-env-remove (oid)
  (unless *ref-env-remove*
    (setf *ref-env-remove* (ml-text "ref_env_remove") )  )
  
  (funmlcall *ref-env-remove* oid))


(defmacro with-make-ref-environment ((makef term) &body body)

  `(progn
    (when (member ,term *re-index-kludge-assoc* :test #'compare-terms-p :key #'car)
      (raise-error (error-message '(|with_ref_environment_index|) ,term)))
    
      (if (or (null ,term) (ioid-term-p ,term))
	  (progn ,@body)
	  (let ((re (get-temp-re-obid ,term)))
	    (unwind-protect
		 (progn (funcall ,makef re ,term)
			,@body)
	      (ref-env-remove re)
	      (free-temp-re-obid ,term))))))


       
