
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	
;;;;	Needs improvement. Need make and manage bin files.
;;;;	
;;;;	distinct-tables for various Languages?

;;;;	
;;;;	dependencies
;;;;	
;;;;	contains mapping from id -> dependency.
;;;;	

;;;;	
;;;;	
;;;;	
;;;;	

(defstruct (code-definition (:include definition))
  (global-env nil)
  )

(defun global-env-of-code-definition (def) (code-definition-global-env def))


;; idents : token list.
;; bins : term list.

;; substance term will be source (or source filename).
;; should include language?!? Could have as substantive property.
(defclass code-substance (substance)
  (
   ;; identifiers of globals defined, to update code dependency table
   (idents :reader idents-of-code-substance
	   :writer set-code-substance-idents
	   :initform nil
	   :initarg idents)

   ;; to locate compiled.
   (bins :reader bins-of-code-substance
	   :writer set-code-substance-bins
	   :initform nil
	   :initarg bins)

   ))

(define-primitive |!code_substance| () (idents bins))

(defun new-code-substance (term idents bins props)
  (make-instance 'code-substance 'term term 'idents idents 'bins bins 'properties props))

(defmethod data-import ((substance code-substance) super)
  (let ((term (call-next-method substance super)))
    (set-code-substance-idents (tokens-of-itokens-term (idents-of-icode-substance-term term))
			       substance)
    ;;(break "ncs")
    (set-code-substance-bins (subterms-of-iterms-term (bins-of-icode-substance-term term))
			     substance))
  (values))

(defmethod data-export ((substance code-substance) sub)
  (declare (ignore sub))
  (call-next-method substance
		    (icode-substance-term (itokens-term (idents-of-code-substance substance))
					  (iterms (bins-of-code-substance substance)))))



(defmethod clone-copy ((old-substance code-substance) (new-substance code-substance))
  (set-code-substance-idents (idents-of-code-substance old-substance) new-substance)
  (set-code-substance-bins (bins-of-code-substance old-substance) new-substance)
  (call-next-method old-substance new-substance))




(defstruct (code-table (:include definition-table))
  (dependencies (make-hash-table))
  (lookup-hints (make-hash-table))
  (compiles nil)
  )

(defun dependencies-of-code-table (ct) (code-table-dependencies ct))
(defun lookup-hints-of-code-table (ct) (code-table-lookup-hints ct))
(defun compiles-of-code-table (ct) (code-table-compiles ct))


(defun code-dependency-table-lookup (table id)
  (when table
    (gethash id table)))

(defun force-code-dependency (oid)
  (let ((table (resource 'code)))
    (definition-lookup-by-oid table oid (current-transaction-stamp))))

(defunml (|force_code_dependency| (oid)) (object_id -> unit)
  (force-code-dependency oid))

(defun code-dependency-table-add (table id d)
  (format t "code-dependency-table-add ~a ~%" id)
  ;;(setf -id id) (break "cdta")
  (when table
    (setf (gethash id table) d)))


;; compile-ml-code
;;   - insert-code-def
;;   - com_ml_compile
;; recompile-code-silent



;;;;	!!!! changed compile to not update state so this is no longer needed. 8/2002 !!!!
;;;;	
;;;;	code table bins : 
;;;;	  - ordered list of compilations
;;;;	  - <stamp> 
;;;;	
;;;;	recompile : move bin to front of bins list.
;;;;	
;;;;	each def in code table has stamp.
;;;;	 - thus can order code table by stamp.
;;;;	   position of stamp in bins list is order of compilation.
;;;;	
;;;;	allows avoidance of recompilation at insert by leaving bin at compile.
;;;;	  -if last bin is bin being inserted then can skip compile.
;;;;
;;;;	!!!! changed compile to not update state so this is no longer needed. !!!!

#|
;; returns bin matched in compiles of table.
(defun code-table-compile-bin-match (table bins)
  (exists-p #'(lambda (bin)
		(exists-p #'(lambda (dbin)
			      (when (compare-terms-p bin dbin) bin))
			  bins))
	    (compiles-of-code-table table)))

(defun code-table-note-compile (table bin)
  (push bin (code-table-compiles table)))

(defun code-table-note-compile-update (table bins)
  ;;(setf -table table -bins bins) (break "ctncu")
  (setf (code-table-compiles table)
	(append bins (compiles-of-code-table table))))

(defun code-table-unnote-compile (table bin)
  (setf (code-table-compiles table)
	(delete bin (compiles-of-code-table table))))

|#
;; produce list of defs in order of compilation
;; thins bins
;; broadcast might include bin from some other process. ok as filtered out.
;; prob when def in code table but no bins in compile list ?
;; Maybe: if code broadcast but compile fails then no bin?
;; thus if no bin then never compiled thus never less than any compiled object.
#|
(defun code-table-compilation-order (table)
  (let* ((bins (compiles-of-code-table table))
	 (dbins (make-hash-table :size (length bins) :test #'equal))
	 )
    ;;(break "ctco")

    (without-dependencies
     (definition-table-map table (current-transaction-stamp)
       #'(lambda (oid def)
	   (declare (ignore oid))
	   (dolist (b (bins-of-code-substance (substance-of-definition def 'code-substance)))
	     (setf (gethash (term-to-stamp b) dbins) t))
	   ))

     (setf bins
	   (delete-if-not #'(lambda (bin)
			      (let ((s (term-to-stamp bin)))
				(let ((e (gethash s dbins)))
				  (when (eq e t)
				    (setf (gethash s dbins) 'seen)
				    t))))
			  bins))
			   
     (setf (code-table-compiles table) bins))
    ;;(setf -bins bins -dbins dbins) (break "ctco")

    (dotimeslist (i bin bins)
		 (let ((s (term-to-stamp bin)))
		   (setf (gethash s dbins) i))) 

    ;; a < b ==> a compiled earlier than b.
    #'(lambda (aoid boid)
	;;(setf -aoid aoid -boid boid) (break "ctcoc")
	(without-dependencies
	 (let ((adef (lookup-code-def aoid t))
	       (bdef (lookup-code-def boid t)))
	   (let ((aposs (mapcan #'(lambda (bin)
				    (let ((pos (gethash (term-to-stamp bin) dbins)))
				      (when (integerp pos) (list pos))))
				(when adef
				  (bins-of-code-substance
				   (substance-of-definition adef 'code-substance)))))
		 (bposs (mapcan #'(lambda (bin)
				    (let ((pos (gethash (term-to-stamp bin) dbins)))
				      (when (integerp pos) (list pos))))
				(when bdef
				  (bins-of-code-substance
				   (substance-of-definition bdef 'code-substance))))))
	     (let ((amin (when aposs (apply #'min aposs)))
		   (bmin (when bposs (apply #'min bposs))))

	       (setf -amin amin -bmin bmin -dbins dbins)

	       (cond
		 ((and (not (integerp bmin)) (integerp amin))
		  t)
		 
		 ((and (not (integerp amin)) (integerp bmin))
		  nil)

		 ((and (not (integerp amin)) (not (integerp bmin)))
		 
		  (setf -aoid aoid -boid boid)
		  (format t "-")
		  ;;(break "code-table-compilation-order")
		  ;; say no as more like to create conflicts ??
		  nil)

		 (t (< bmin amin))))))))))

;; true when first oid compiled later than second?
;; a < b if a compiled later than b ? 
(defunml (|code_table_compilation_order| (unit) :declare ((declare (ignore unit))))
    (unit -> (object_id -> (object_id -> bool)))
  ;; twould be nice to arrange funmlcall mlclosure does less.
  (mlclosure (code-table-compilation-order (resource 'code)) 2))
|#


(defvar *make-explicit-ref-environment-f* nil)
(defun make-explicit-ref-environment (oid reterm)
  (unless *make-explicit-ref-environment-f*
    (setf *make-explicit-ref-environment-f* (ml-text "make_explicit_reference_environment")))
  
  (funmlcall *make-explicit-ref-environment-f* oid reterm)
  )

(defmacro with-compile-ref-environment (reterm &body body)
  (let ((ret (gentemp)))
    `(let ((,ret ,reterm))
      (if (null ,ret)
	  (progn ,@body)
	  (with-make-ref-environment (#'make-explicit-ref-environment ,ret)
	    (with-reference-environment-term ,ret
	      ,@body))))))

  
;; see also reference-environment-term-of-objc in lib-objc 
;;
(defun reference-environment-property-of-substance (substance)
  (or (property-of-substance substance '|reference_environment_minimal|)
      (property-of-substance substance '|reference_environment_theories_minimal|)
      (property-of-substance substance '|reference_environment_relative_minimal|)
      (property-of-substance substance '|reference_environment|)))


(defun nml-file-code-raise (filenames)

  ;;(setf -filenames filenames) (break "nfcr")
  (when (not (boundp' *current-ml-definitions*))
    (raise-error (error-message '(|nml-file-code-raise| current-ml-definitions bound not))))
  
  (maphash #'(lambda (k vv)
	       (dolist (v vv)
		 (let ((s (source-of-mldef v)))
		   (let ((ss (cond
			       ((symbolp s) (string s))
			       ((stringp s) s)
			       ((and (term-p s) (iml-file-description-term-p s))
				;;(setf -name (name-of-iml-file-description-term s)) (break "nfcr3")
				(name-of-iml-file-description-term s))
			       (t ;;(break "nfcr2")
				  ""))))
		     (when (member ss filenames :test #'string=)
		       (setf *current-ml-definitions*
			     (cons (cons k (copy-ml-definition v)) *current-ml-definitions*))))
		   )))
	     *global-ml-definiton-table*))

(dml |nml_file_code_raise| 1 nml-file-code-raise ((string list) -> unit))


;; TODO there has got to be a better way.
(defvar *nml-code-add-f* nil)
(defun nml-code-add (oid)
  (funmlcall *nml-code-add-f* oid (list oid)))
  
(defun compile-ml-code-object (oid code-def &optional ct)
  (if *delay-broadcast-compiles*
      (push oid *broadcast-compiles-delayed*)

      (let ((ctable (or ct (resource 'code)))
	    (substance (definition-substance code-def)))

	(without-dependencies
	 (with-oids ((list oid))
	   (with-ml-definitions (nil)
	     (with-compile-ref-environment (reference-environment-property-of-substance substance)
	       (mlet* (((a b msg) (ml-term (let ((*ref-environment-abstractions-index* nil))
					     (source-reduce (term-of-substance substance) '(ml)))
					   t)
			(declare (ignore a b))))

		      ;;(setf -msg msg -substance substance)
		      (when (and msg *recompile-code-print-message*)
			(print-message msg))))
	     
	     ;;(setf -oid oid -code-def code-def) (break "smo")
	     (setf (code-definition-global-env code-def)
		   (mapcar #'(lambda (d)
			       ;;(setf -d d -code-def code-def) (break "smo")
			       (set-mldef-source (cdr d) oid)
			       d)
			   (get-current-ml-definitions)))

	     (when (and (null (property-of-substance substance '|reference environment additions|))
			(exists-p #'(lambda (d) (not (null (id-of-mldef (cdr d)))))
				  (global-env-of-code-definition code-def)))
	       (nml-code-add (oid-of-definition code-def)))

	     (cache-ml-compile-lookup-hints code-def ctable))

	   ;; ;; unnote old, add a new
	   ;; (code-table-unnote-compile ctable (car (bins-of-code-substance substance)))
			 
	   #| ;; 9/02 mldef's made this obsolete.
	   (let ((dep (dependency-to-term (dependency-of-definition code-def)))
		 (table (dependencies-of-code-table ctable)))
	     (dolist (id (idents-of-code-substance substance))
	       (when *compile-print-names*
		 (format t "~a~%" id))
	       (code-dependency-table-add table id dep)))
	   |#
     
	   (let* ((s (stamp-to-term (new-transaction-stamp))))
	     ;;(setf -table table -s s -term term) (break "cmc")
	     (set-code-substance-bins (cons s (bins-of-code-substance substance)) substance)
	     ;;(code-table-note-compile ctable s)
	     ))))))


;;(funmlcall (ml-text "view_show_oids") (mapcar #'car *suppressed-compile-errors*));;
;;(funmlcall (ml-text "\\oids. view_showd `RICH` (ioids_term oids) ") (mapcar #'car *suppressed-compile-errors*));;

(defvar *suppressed-compile-errors* nil)

(defunml (|connection_compile_errors| (unit) :declare ((declare (ignore unit))))
    (unit -> (object_id list))
  (mapcar #'car *suppressed-compile-errors*))

(defun compile-ml-code-woxref-woerr (oid code-def ct)
  (with-handle-error-and-message (nil #'(lambda (msg)
					  ;;(break "icd")
					  (push (cons (oid-of-definition code-def)
						      (message-to-term msg))
						*suppressed-compile-errors*)
					  (message-emit
					   (oid-warn-message (list oid)
							     '(compile fail code)
							     (cons msg (messages-flush)))
					   'asynch)))
				 (compile-ml-code-object oid code-def ct))
   (mapc #'print-message (messages-flush)))

(defvar *failed-code-oids* nil)

(defun insert-code-def-aux (table def)
  (let* ((substance (substance-of-definition def 'code-substance))
	 (bins (bins-of-code-substance substance))
	 (dependency (dependency-to-term (dependency-of-definition def))))

	
    (with-handle-error-and-message (nil #'(lambda (msg)
					    ;;(break "icd")
					     (push (oid-of-definition def)  *failed-code-oids*)
					    (message-emit
					     (oid-warn-message (list (oid-of-definition def))
							       '(broadcast fail code)
							       (cons msg (messages-flush)))
					     'asynch)))


      (let* ((lprop (property-of-substance substance 'language))
	     (language (when lprop (token-of-itoken-term lprop))))

	;;(setf -l language -substance substance) (break "icd")

	(case language

	  (ml

	   ;;(setf dd def tt (term-of-substance substance) ss substance)
	   (format t "compiling/loading code~%")
	   (compile-ml-code-object (oid-of-definition def) def)
	   ;;(setf -def def) (break "nmlca")
	   )
	  
	  (lisp

	   ;; we always want to recompile at this stage since object was previously unactive
	   (unless (and nil ;;(code-table-compile-bin-match table bins)
			) 

	     ;;(setf dd def tt (term-of-substance substance) ss substance)
	     (format t "compiling/loading code~%")
	       
	     (let ((bin (bin-of-icode-compiled-term
			 (compile-lisp-code (term-of-substance substance)
					    (reference-environment-property-of-substance substance)))))

	       ;;(setf -bin bin -substance substance -bins bins) (break "cmc")

	       (set-code-substance-bins (cons bin bins) substance))

	     ;; does this do anything for lisp??
	     (let ((table (dependencies-of-code-table table)))
	       (dolist (id (idents-of-code-substance substance))
		 ;;(format t "~s " id)
		 (code-dependency-table-add table id dependency)))
	     ))

	  (sql nil)
	  
	  (otherwise (message-emit (warn-message '(code insert language not) language))))

	))))


;; if any bin present, then no-op else compile/load and add bin.
;; update def with bin so delete removes.
(defun insert-code-def (table def s i)
  (declare (ignore s i))
  ;;(when (definition-name def) 
  ;;(format t "~s " (definition-name def)))
  (insert-code-def-aux table def)
  )



;; delete any matching bins.
(defun delete-code-def (table def s i)
  (declare (ignore table def s i))

  ;;(code-table-unnote-compile table
  ;;(code-table-compile-bin-match table
  ;;(bins-of-code-substance
  ;;(substance-of-definition def 'code-substance))))
  )


(defun cache-ml-compile-lookup-hints (def ct)
  ;;(setf -def def) (break "cmclh")
  (let ((oid (oid-of-definition def))
	(hints (lookup-hints-of-code-table ct)))
    (dolist (e (global-env-of-code-definition def))
      (let ((n (car e)))
	(when n
	  ;;(format t "~%cmclh ~a " n) 
	  (let ((cur (gethash n hints)))
	    (unless (member oid cur :test #'equal-oids-p)
	      ;;(setf -n n -oid oid -cur cur) (break "cmclh")
	      (setf (gethash n hints)
		    (cons oid cur)))))))))

(defun import-code-def (term)
  (make-code-definition :substance (term-to-data term)))


(defmacro code-table (stamp tag &rest keys &key &allow-other-keys)
  `(define-definition-table
      ,stamp
      (list 'code ,tag)
    nil
    :make-f make-code-table
    :import-f #'import-code-def
    :insert-f #'insert-code-def
    :delete-f #'delete-code-def
    ,@keys))

(defun allocate-code (stamp tag)
  (code-table stamp tag))

(defun lookup-code-def (oid &optional nil-ok-p)
  (if (resource-p 'code)
      (definition-lookup-by-oid (resource 'code) oid nil-ok-p)
      (raise-error (oid-error-message (list oid) '(terms lookup not)))
      ))


;;;;	
;;;;	would like to lose mlxrefs : 
;;;;	- raw-mlxrefs-to -term creates xrefs term instead of mlxrefs term
;;;;	- ml import converts mlxref-term to xrefs.
;;;;	- ddg import ditto .
;;;;	
;;;;	raw-mlxrefs-to-term -> MLXREF
;;;;	  - compile-check-ml-code
;;;;	  - com_ml_compile
;;;;	  - lib_ml_code_compile
;;;;	  - lib-ml-code-compile
;;;;	  - compile-code-objc-source -> MLXREF
;;;;	      - term-to-mlxref 
;;;;	      - (set-code-source-xref xref csource)
;;;;	  - xref-of-code-source / code-xrefs-to-term
;;;;	     - build-object-attrs - ddg
;;;;	     - build-ddg-aux - ddg
;;;;	     - data-export
;;;;	  - xref-of-code-source / mlxrefs-to-term
;;;;	     - xrefs-of-ml-code-source
;;;;	  - xref-of-code-source / idents-of-xrefs
;;;;	     - idents-of-code-source
;;;;	  - xref-of-code-source dependencies-of-xrefs
;;;;	     - ref-dependencies-of-code-source
;;;;	
;;;;	ddg-lookup-xid
;;;;	this turned to oid for ddg xref ?? at the moment never?



#|
(defunml (|post_activate_code| (oid))
    (oid -> unit)

 (let ((def (lookup-code-def oid)))
    (let ((sub (substance-of-definition def 'substance)))
      (ml-term (term-of-substance sub)))
    nil))
|#

;; xref-id	: tok{id} list, oid
;;		| tok{id} list, term{this/file/}
;;		| tok{id} list, dependency  ;; backwards compatability.
;; xref-id-ids  : tok{id}
;;		| tok{id} list
;;
(defmacro xref-id (id src) `(cons ,id ,src))
(defmacro source-of-xref-id (xid) `(cdr ,xid))
(defmacro id-of-xref-id (xid) `(car ,xid))
(defun oid-of-xref-id (xid)
  (let ((s (source-of-xref-id xid)))
    (and s (oid-p s) s)))

(defstruct xref-entry
  term
  id
  source
  mutable
  type
  called
  callers
  )

(defun term-of-xref (x) (xref-entry-term x))
(defun id-of-xref (x) (xref-entry-id x))
(defun source-of-xref (x) (xref-entry-source x))
(defun mutable-of-xref (x) (xref-entry-mutable x))
(defun type-of-xref (x) (xref-entry-type x))
(defun called-of-xref (x) (xref-entry-called x))
(defun callers-of-xref (x) (xref-entry-callers x))

;; called : oid || term {file|this|token}
(defun new-xref-entry (mutable type id src called)
  (make-xref-entry :mutable mutable
		   :type type
		   :id id
		   :source src ;; source : oid | term{includes this} |
		   :called called))

(defun new-xref-pre-entry (id src)
  (make-xref-entry :id id
		   :source src ;; source : oid | term{includes this} |
		   ))


(define-primitive |xref| ((bool . mutable)) (xid type called callers))
(define-primitive |xref_cons| () (car cdr))
(define-primitive |xref_id_cons| () (car cdr))
;; xref_id token{id} (source{oid} | term | dependency{old} | this |)
;; for backwards compatability need to allow for oid in parameter.
;; and dependency in source
;;(define-primitive |xref_id| ((token . id) (oid . oid)))
;; list of tok parameters.
(define-primitive |xref_id| ((token . id)) (source))

(defun xref-id-source-to-term (src)
  (cond
    ((oid-p src) (ioid-term src))
    ((dependency-p src) (ioid-term (oid-of-dependency src)))
    ((term-p src) src)
    ((null src) (ivoid-term))
    (t   (setf -src src) (break "xistt")
	 (raise-error (error-message '(xref id source term not))))))

(defun term-to-xref-id-source (src)
  (cond
    ((idependency-term-p src) (oid-of-dependency (term-to-dependency src))) ; backwards compatability.
    ((ioid-term-p src) (first-oid-of-term src))
    (t src)))

(defun xref-id-to-term (xid)
  (let ((id (id-of-xref-id xid))
	(src (source-of-xref-id xid)))
    (xref-id-term id  (xref-id-source-to-term src))))

(defun term-to-xref-id (xidterm)
  ;; backwards compatability :
  ;;  - opid maybe be xref_id_dependency and subterm would then be !dependency.
  ;;  - op may have second parm which is obid.
  (let ((parms (parameters-of-term xidterm))
	(bts (bound-terms-of-term xidterm)))
    (cond
      ((and (cdr parms) (oid-parameter-p (cadr parms)))
       (xref-id (value-of-parameter-r (car parms))
		(value-of-parameter-r (cadr parms))))
      ((not (null bts))
       (xref-id (value-of-parameter-r (car parms))
		(term-to-xref-id-source (term-of-bound-term (car bts)))))
      (t (xref-id (value-of-parameter-r (car parms)) nil)))))
	   

(defun xref-to-term-aux (xref)
  (xref-term (mutable-of-xref xref)
	     (xref-id-to-term (xref-id (id-of-xref xref) (source-of-xref xref)))
	     (let ((xtype  (type-of-xref xref)))
	       (if xtype
		   (type-sexpr-to-term xtype)
		   (ivoid-term)))
	     (map-list-to-ilist (called-of-xref xref)
				(xref-id-nil-term)
				#'xref-id-to-term)
	     (map-list-to-ilist (callers-of-xref xref)
				(xref-id-nil-term)
				#'xref-id-to-term)) )


(defun imlxref-to-mlxref (mlxref)
  (let ((called (map-isexpr-to-list
		 (called-of-imlxref-term mlxref)
		 (ixref-called-cons-op)
		 #'(lambda (called)
		     (xref-id (id-of-imlxref-called-term called)
			      (let ((idependency (dependency-of-imlxref-called-term called)))
				(cond
				  ((ivoid-term-p idependency) nil)
				  ((idependency-term-p idependency)
				   (oid-of-idependency-term idependency))
				  (t idependency))))))))
			
    (map-isexpr-to-list
     (callers-of-imlxref-term mlxref)
     (ixref-caller-cons-op)
     #'(lambda (caller)
	 (new-xref-entry (refp-of-imlxref-term mlxref)
			 (term-to-type-sexpr
			  (type-of-imlxref-caller-term caller))
			 (id-of-imlxref-caller-term caller)
			 (ithis-term)
			 called)))) )

(defun xref-to-term (xref)
  (or (term-of-xref xref)
      (setf (xref-entry-term xref) (xref-to-term-aux xref))))

(defun term-to-xref (term)
  (if (imlxref-term-p term)
      (imlxref-to-mlxref term)
      (let ((xid (term-to-xref-id (xid-of-xref-term term))))
	(make-xref-entry :mutable (mutable-of-xref-term term)
			 :id (id-of-xref-id xid)
			 :type (term-to-type-sexpr (type-of-xref-term term))
			 :source (source-of-xref-id xid)
			 :called (map-isexpr-to-list (called-of-xref-term term)
						     (xref-id-cons-op)
						     #'term-to-xref-id
						     )
			 ))))

(defun term-to-xrefs (term)
  (map-isexpr-to-list term (xref-cons-op) #'term-to-xref))


(defun xrefs-to-term (xrefs)
  (map-list-to-ilist xrefs (xref-nil-term) #'xref-to-term))
  


(define-primitive |!ml_compiled| () (bin xref dependencies))
(define-primitive |!code_compiled| () (bin xref dependencies))

(define-primitive |!xref_cons| () (car cdr))
(define-primitive |!xref_called_cons| () (car cdr))
(define-primitive |!xref_caller_cons| () (car cdr))
  
(define-primitive |!mlxref| ((bool . refp)) (callers called))
(define-primitive |!mlxref_called| ((token . id)) (dependency))
(define-primitive |!mlxref_caller| ((token . id)) (type))

;; temp kludge
(define-primitive |!codebin| () (address))


;; TODO : if we created a bin file we could split compile and load such that
;; compile had no effect on state.

;; assumes dependency table holds terms.

;; add dependencies from  ml-xref-table
;; build result xref-term
;; if from bml-xref then called is idents, but if not then called is ident . dependency

(define-primitive |!this|)

#|
(defun raw-mlxrefs-to-term (xrefs)
  ;;(setf -xrefs xrefs) (break "rmtt")
  (map-list-to-isexpr
   xrefs
   (ixref-nil-term)
   #'(lambda (xref)
       (let ((this-ids nil))
	 (imlxref-term
	  (refp-of-mlxref xref)
	  (map-list-to-isexpr (callers-of-mlxref xref)
			      (ixref-caller-nil-term)
			      #'(lambda (caller)
				  (let ((id (id-of-mlxref-caller caller)))
				    ;;(format t "add ~a ~%" id)
				    (push id this-ids) ; if id not recursive this could be wrong.
				    (imlxref-caller-term
				     id
				     (type-sexpr-to-term (type-of-mlxref-caller caller))))))
	  (map-list-to-isexpr
	   (called-of-mlxref xref)
	   (ixref-called-nil-term)
	   #'(lambda (e)
	       (let ((id (car e)))
		 (setf -e e -this-ids this-ids -xref xref) (break "rm")
		 ;;(format t "look ~a ~%" id)
		 ;; if id defined in object just compiled then should not
		 ;; reference other object. 
		 (imlxref-called-term
		  id
		  (cond
		    ;;((member id this-ids) (ithis-term)) ;; recursive - maybe stamps catch this?
		    (t (or (let* ((asrc (cdr e))
				  (src (if (ml-definition-p asrc)
					   (source-of-mldef asrc)
					   asrc)))
						   
			     ;;(when (null src) (setf -e e -this-ids this-ids -xref xref) (break "raw"))
			     (cond
			       ((null src) nil)
			       ((dependency-p src) ; probably old.
				(ioid-term (oid-of-dependency src)))
			       ((equal src stamp) (ithis-term))
			       ((oid-p src) ; object
				(ioid-term src))
			       ;;(idependency-term src (ivoid-term) (ivoid-term))
			       ((symbolp src)
				(itoken-term src) ; TODO, confusing ddg, ddg needs to treat more abstractly.
				)	; built-in
			       ((term-p src)
				src ; TODO, confusing ddg, ddg needs to treat more abstractly.
				)	; file-description / stamp
			       (t ;;(break "afja")
				  nil)))
			   ;;code-dependency-table-lookup table
			   (ivoid-term)))))))))))))
|#

(defun mlxrefs-to-term (xrefs)
  ;;(setf -xrefs xrefs) (break "mxtt")
  (map-list-to-isexpr xrefs
		      (ixref-nil-term)
		      #'(lambda (xref)
			  (imlxref-term
			   (refp-of-mlxref xref)
			   (map-list-to-isexpr (callers-of-mlxref xref)
					       (ixref-caller-nil-term)
					       #'(lambda (caller)
						   (imlxref-caller-term
						    (id-of-mlxref-caller caller)
						    (type-sexpr-to-term (type-of-mlxref-caller caller)))))
			   (map-list-to-isexpr (called-of-mlxref xref)
					       (ixref-called-nil-term)
					       #'(lambda (called)
						   (let ((d (dependency-of-mlxref-called called)))
						     (imlxref-called-term (id-of-mlxref-called called)
									  (cond
									    ((null d) (ivoid-term))
									    ((eql 'this d) (ithis-term))
									    (t (dependency-to-term d)))))))))))


;; ?? where does !this() get mapped to oid.
(defun raw-mlxrefs-to-xrefs (rmlxrefs stamp)
  ;;(setf -xrefs rmlxrefs) (break "rmtt")

  (let ((itt (ithis-term)))

    (mapcan
     #'(lambda (mlxref)
	 (let ((called (mapcar
			#'(lambda (e)
			    ;;(setf -e e)
			    (xref-id (car e)
				     (let* ((asrc (cdr e))
					    (src (if (ml-definition-p asrc)
						     (source-of-mldef asrc)
						     asrc)))
						   
				       (cond
					 ((null src) nil)
					 ((dependency-p src) (oid-of-dependency src)) ; probably old.
					 ((equal src stamp) itt)
					 ((oid-p src) src)
					 ((symbolp src) (itoken-term src))
					 ((term-p src) (if (ioid-term-p src) (oid-of-ioid-term src) src))
					 (t ;;(break "afja") 
					    nil)))))
			(called-of-mlxref mlxref))))

	   (mapcar #'(lambda (caller)
		       (new-xref-entry (refp-of-mlxref mlxref)
				       (type-of-mlxref-caller caller)
				       (id-of-mlxref-caller caller)
				       itt
				       called
				       ))
		   (callers-of-mlxref mlxref))))
     rmlxrefs)))


;; xref : name . <bool{ref}> . called 
(defun lisp-xrefs-to-term (table xrefs &optional oid)
  (let ((this-ids nil))	;; kludge to avoid referencing other objects with same ids.
    (map-list-to-isexpr xrefs
			(ixref-nil-term)
			#'(lambda (xref)
			    (let ((id (cadr xref)))
			      (push id this-ids)
			      (xref-term
			       (member (car xref) '(defvar setf))
			       (xref-id-to-term (xref-id id oid))
			       (ivoid-term) ; type
			       (map-list-to-isexpr
				(caddr xref)
				(xref-id-nil-term)
				#'(lambda (id)
				    ;; if id defined in object just compiled then should not
				    ;; reference other object.
				    (let ((d (unless (member id this-ids)
					       (code-dependency-table-lookup table id))))
				      (xref-id-to-term (xref-id id (and d (oid-of-dependency d)))))))
			       (xref-id-nil-term)))))))


(defun term-to-mlxref (term)
  ;; may be proper 
  ;; stamp wrapper/oids/strings/terms/tokens

  (if (not (or (ixref-cons-term-p term)
	       (ixref-nil-term-p term)
	       (imlxref-term-p term)))
      (map-isexpr-to-list term (xref-cons-op)
			  #'term-to-xref)
      ;; backwards compatability
      (let ((l (map-isexpr-to-list term (ixref-cons-op))))
	(mapcan #'imlxref-to-mlxref l))))

(defun term-to-code-xrefs (term)
  (mlet* (((lang xterm) (if (code-xrefs-term-p term)
			    (values (lang-of-code-xrefs-term term) (xrefs-of-code-xrefs-term term))
			    (values 'ml term))))
    (case lang
      (ml (term-to-mlxref xterm))
      (lisp (term-to-xrefs xterm))
      (otherwise (break "term-to-code-xrefs"))
      )))

(define-primitive |code_xrefs| ((token . lang)) (xrefs))

(defun code-xrefs-to-term (lang xrefs)
  (code-xrefs-term lang
		   (case lang
		     (ml (xrefs-to-term xrefs))
		     (lisp (xrefs-to-term xrefs))
		     (otherwise (break "code-source xrefs-to-term")))  
		   ))

;;;;	should set it up so that back to back compile/activate does not
;;;;	recompile.
;;;;	
;;;;	
;;;;	dependencies : a dummy is made up at compile time.
;;;;	
;;;;	used as initial binding for code-table-compile 
;;;;	used as bin id for compile.
;;;;	

;;;;	
;;;;	compile : 
;;;;	  - compile object (not necessary to update state).
;;;;	      * returns xref (needs to remember obids at lookup).
;;;;	    compile-check
;;;;	  - load object (insert code-def) (updates state)
;;;;	    recompile-silent (refresh calls to modified code).
;;;;	      * updates code table state.
;;;;	    compile-ml-code-object
;;;;	  - compile-file (updates state)
;;;;	      * updates global-env state.
;;;;	    compile-file
;;;;	  - eval top-level / refine 
;;;;	      * updates global-env state.
;;;;		(should be similar to compile file in that new defs will have no obid)
;;;;	      * returns xref (needs to remember obids at lookup).
;;;;	    compile-eval
;;;;	
;;;;	

;; at refine time need with-ml-xref 

(defvar *compile-print-names* nil)

(defvar *code-ref-state-lookup* nil)

;; if called during typecheck messes up ml compiler global variables.
(defunml (|set_lisp_ml_forward_hook| (name))
    (tok -> unit)

  (case name
    (nml_code (setf *code-ref-state-lookup* (ml-text "nml_code_ref_state_lookup "))
	      (setf *nml-code-add-f* (ml-text "nml_code_add"))
	      )))



#|;; if no refenv then return all code oids.
;; when no refenv then oid-table-of-definition-table
(defun nml-ref-state-code-object-p (table oid)
  ;; TODO visibiity similar to abs ?
  (declare (ignore table))

  (member oid 
	  (funmlcall *code-ref-state-lookup*  nil)
	  :test #'equal-oids-p))
|#
(defun nml-ref-state-code-object-filter (table coids)
  ;; TODO visibiity similar to abs ?
  (declare (ignore table))
  (let ((oids (funmlcall *code-ref-state-lookup* nil)))

    (filter #'(lambda (oid-e) (member (car oid-e) oids :test #'equal-oids-p)) coids)))

(defun multiple-code-defs-list-by-hints ()
  (let ((myhash (make-hash-table :size 10000))
	(hints (lookup-hints-of-code-table (resource 'code))))


    (maphash #'(lambda (k v)
		 (dolist (vv v)
		   (when (term-p (source-of-mldef vv))
		     (push (source-of-mldef vv) (gethash k myhash)))))
	     *global-ml-definiton-table*)
    (maphash #'(lambda (k vv)
		 (dolist (v vv)
		   (push v (gethash k myhash))))
	     hints)
	       
    (let ((acc nil))
      (maphash #'(lambda (k v)
		   (unless (onep (length v))
		     (push (cons k v) acc)))
	       myhash)

      ;;(setf -acc acc)
      ;;(setf -oids (filter #'oid-p (remove-duplicates (mapcan #'(lambda (x) (copy-list (cdr x))) -acc) :test #'(lambda (a b) (or (eq a b) (and (oid-p a) (oid-p b) (equal-oids-p a b)))))))
      ;;(break "mcdlbh")
      (length acc))))

(defvar *fail-multiple-mldefs* nil)
(defun global-lookupper-aux (ct)
  (let ((hints (lookup-hints-of-code-table ct)))

    ;; what about local defs, not being updated to defs how to find?
    #'(lambda (id)
	(let ((coids (mapcan #'(lambda (hoid)
				 (let ((def (definition-lookup-by-oid ct hoid t nil t))) ;;(hashoid-get codes hoid)
				   (when def
				     ;; multiple occurences of id then what last defined, ie first on list.
				     (let ((e (cdr (assoc id (global-env-of-code-definition def)))))
				       (when e
					 (list (cons hoid e)))))))
			     (gethash id hints))))

	  (cond
	    ((null coids) nil)
	    ((null (cdr coids))
	     (let ((oid-e (car coids)))
	       ;;(setf -oid-e oid-e) (break "need to set obid in mldef?")
	       ;;(when *remember-global-lookups* (add-global-lookup id (car oid-e)))
	       (cdr oid-e)))
	    (t
	     ;;(setf -coids coids) (break "multiple-defs")
	     (let ((ncoids (nml-ref-state-code-object-filter ct coids)))
	       ;;(filter #'(lambda (oid-e) (nml-ref-state-code-object-p ct (car oid-e))) coids)
	       (when (and *fail-multiple-mldefs*
			  (or (null ncoids)
			      (cdr ncoids)))
		 ;;(setf -coids coids -ncoids ncoids)
		 (break "gm")
		 (raise-error (oid-error-message (mapcar #'car (or ncoids coids)) '(ml-global lookup multiple) id)))
	       (format t "~% nml multiple-defs ~a ~a ~a" id (length coids) (length ncoids))
	       (cdr (car (or ncoids coids)))
	       )))))))

  
(defun global-lookupper ()
  (global-lookupper-aux (resource 'code)))

;; still problem with local object defs for check case.
;; twould be nice to avoid (setf (symbol-value n) in updatevalues. maybe something similar for types ?
;; worry about capture ie need interleaved transactions to protect themselves may need to add
;;   to transaction state.
;;  * fixed by environment-eval rebinding lookupper.
(defun universal-lookupper ()
  (cond
    ((resource-p 'code)
     (global-lookupper))
    (t nil)))

(setf *universal-lookupper* #'universal-lookupper)

#|
(defun local-updater (descriptors)
  (setf -d descriptors) (break "ld")
  (setf *global-local-updates* (append descriptors *global-local-updates*)))
|#



;; compile-check : type and xref.
(defun compile-check-ml-code (term &optional env)
  (with-dependencies-vertical ('(compile-code ml-eval))
    (let (;;(table (resource 'code))
	  (s (stamp-to-term (new-transaction-stamp))))
      
      (mlet* (((xrefs stamp)
	       (with-ml-xref
		   ;; remember successful compilations for transaction.
		   ;; insert-code-def will not see these and that's a good thing.
		   ;; no so good is that old compiles will shadow more recent inserts.
		   ;; if multiple compiles with no commits then do not get oid info updated in mldefs for earlier.
		   ;; if multiple compiles with intervening commits then want mldefs from table update
		   ;;    not saved ones.
		   (with-ml-definitions (;;(tstate-stuff-get 'ml-definitions)
					 nil
					 )
		     (with-compile-ref-environment env
		       
		       ;; if function defined is produced by evaluation then
		       ;; this will miss any eval errors. Also misses declared dependencies
		       ;; vml-term was changed to ml-term to avoid these probs.
		       (ml-term (let ((*ref-environment-abstractions-index* nil))
				   (source-reduce term '(ml)))
				 t)
		       ;;(tstate-stuff-put 'ml-definitions (get-current-ml-definitions))
		       (values (get-ml-xref) *current-ml-definitions-stamp*))))))

	      ;;(setf -a xrefs) (break "cmc")
	      (when *compile-print-names*
		(terpri)
		(dolist (xref xrefs)
		  (dolist (c (callers-of-mlxref xref))
		    (format t "~a~%" (id-of-mlxref-caller c)))))
      
	      ;; (setf -table table -s s -term term -xrefs xrefs -name-values name-values) (break "cmc")
	      ;;(code-table-note-compile table s)
	      (iml-compiled-term s
				 (xrefs-to-term (raw-mlxrefs-to-xrefs xrefs stamp))
				 (event-dependencies-collected-term))))))


(define-primitive |!expression_cons| () (car cdr))
(defvar *compile-code-process-hooks* nil)

(defun add-compile-code-process-hook (lang name hook)
  (let  ((lhooks (assoc lang *compile-code-process-hooks*)))
    (if lhooks
	(setf (cdr lhooks) (acons name hook (delete name (cdr lhooks) :key #'car)))
	(setf *compile-code-process-hooks*
	      (acons lang (acons name hook nil) *compile-code-process-hooks*)))))

(defunml (|add_compile_code_process_hook| (lang name hook))
    (tok -> (tok -> ((term -> term) -> unit)))

  (add-compile-code-process-hook lang name
				 #'(lambda (term)
				     (handle-error 'evaluation
						   #'(lambda (err)
						       (declare (ignore err))
						       (break "cchf")
						       (format t "compile_code_hook ~a failed" name)
						       term)
						   (funcall (closure-func hook) term))))
  nil)

(defun apply-compile-code-process-hooks (lang term)
  (let ((lhooks (cdr (assoc lang *compile-code-process-hooks*))))
    (term-walk-ops term
		   #'(lambda (term)
		       (let ((nterm term)
			     (hits nil)
			     (done nil)
			     (abort nil)
			     )
			 (do ()
			     ((or done abort))
			   (setf done t)
			   (dolist (h lhooks)
			     ;;(setf -nterm nterm)
			     (let ((nnterm (funcall (cdr h) nterm)))
			       (unless (eq nterm nnterm)
				 ;;(setf -h h -nnterm nnterm) (break "doc")
				 (setf nterm nnterm)
				 (let ((n (car h)))
				   (setf done nil)
				   (if (member n hits)
				       (progn (format t "apply-compile-code-hooks ~a loop?" n)
					      (setf abort t))
				       (setf hits (cons n hits))))))))
			 nterm)))))

(defun syntax-reduce (lang tags term)
  (source-reduce (apply-compile-code-process-hooks lang term)
		 (list* 'syntax lang tags)))


;;;;	perf : use buffer to accumulate text
(defun compile-lisp-code (term reterm)
  ;; transform to list

  ;;(setf -term term) (break "clc")
  (let* ((red (syntax-reduce 'lisp '(lispprimitive) term))
	 (sexprs (map-isexpr-to-list red
				     (iexpression-cons-op)
				     #'term-to-text)))
    ;;(setf -sexprs sexprs);;(break "clc")

    
    (let ((exprs (with-compile-ref-environment reterm
		   (mapcar #'(lambda (expr)
			       (read-from-string
				;; perf : combine with term-to-text via term-to-text-aux which
				;;        takes accumulator arg.
				(with-text-scanner (expr)
				  (let ((s (scan-string *standard-character-sbits* t)))
				    (unless (scan-eof-p)
				      (raise-error (error-message '(compile-lisp-code scan not) s red)))
				    (format t "~%~a~%" s)
				    s))))
			   sexprs))))
      
      ;;(setf -exprs exprs);; (break "clc2")
      

      ;; eval
      (let ((xrefs (mapcar #'(lambda (e)
			       (let ((r (eval e)))
				 (when (and (symbolp r)
					    (fboundp r)
					    (not (compiled-function-p (symbol-function r))))
				   (compile r))
				 (let ((form (car e)))
				   (when (member form '(defun defmacro defvar setf setq))
				     (list form (cadr e)
					   ;; ?refs of body?
					   nil
					   )
				     ))))
			   exprs)))

	(let* ((s (stamp-to-term (new-transaction-stamp)))
	       (table (resource 'code)))
	  ;;(code-table-note-compile table s)
	  (icode-compiled-term s
			       (lisp-xrefs-to-term (dependencies-of-code-table table) xrefs)
			       (event-dependencies-collected-term))    
	  )))))

(defvar *recompile-code-print-message* nil)

(defun recompile-code-silent (oids)
   (let ((ctable (resource 'code))
	(stamp (current-transaction-stamp)))
    
    (mapcar #'(lambda (oid)
		(let ((code-def (definition-lookup-by-oid ctable oid t stamp t)))
		  ;;(setf -code-def code-def)
		  (when code-def
		    (format t "~a~%" (token-of-itoken-term
				      (cdr (assoc 'name (properties-of-substance (definition-substance code-def))))))
		    (compile-ml-code-object oid code-def))))
	    oids)))


(defunml (|recompile_code_silent| (oids))
    ((object_id list) -> unit)

  (recompile-code-silent oids))




;; kludge : expect compile/load to be distinct ops.
;;  compile is precursror to translate, load happens at activate.
;;  compile should not affect state.
;;
;;  currently, compile implicitly loads. also there may be no bin
;;   file at activate so compilation need occur instead.
;; 
;;  thus at activate need to know if compilation has been done
;;	by examining bin address (stamp or filename). Library should
;;   prevent multiple loads of same bin.
;;
;;  compilation returns a bin address (stamp or filename), activation
;;  will include bin address, so if we maintain list of bins loaded
;;  then skip if loaded. deactivate/activate may be used to recapture
;;  a global id. but activate will not load? so deactivate needs to 
;;  remove bin. 
;;
;;  so kludge is that compile adds bin prior to activation, then
;;  when first activated compile may be skipped. however if deactivatated
;;  then re-activated compile should happen at re-activate.
;;
;;  ATM, can not guarauntee that compile at activate time will succeed due to
;;  dependency tracking holes/errors. Thus need code table to survive
;;  activate time compile error, but warn so that dependency problems which
;;  allowed compile error can be tracked down.
;;
;;  for file proxies, we need to detect source file changes to cause compile, then
;;  allow for recompilation of all dependent files/objects.
;;

(defunml (|com_ml_compile| (reterm source))
    (term -> (term -> term))

  (compile-check-ml-code source (unless (ivoid-term-p reterm) reterm)))
    
(defunml (|com_lisp_compile| (reterm source))
    (term -> term -> term)

   (compile-lisp-code source (unless (ivoid-term-p reterm) reterm)))
    

;;;;
;;;;	Broadcast
;;;;

(defvar *bcast-trace-p* nil)
(defvar *bcast-trace* nil)

;; should be called only by originating transaction.
(defun broadcast (stamp bcast type)
  (when *bcast-trace-p* (push bcast *bcast-trace*))

  ;; try local before calling orb.
  ;;(setf -b bcast -t type -e (current-environment))
  ;;(when (eql '|!definition_insert| (id-of-term (subterm-of-term -b '(2))))(break "bl"))
  (apply-passport (current-environment) bcast (current-transaction-stamp) nil)

  ;; catching errors here prevents lib directives from failing when they should.
  ;;
  ;; I wonder why it was here in the first place??
  ;;
  ;; It might have been :prob is if processing broadcast in client fails
  ;; you night not want to fail out of read loop. 
  ;; However client bcast do not come through here?
  ;; this may no longer be a problem.
    
  (if nil
      (with-handle-error-and-message (nil #'(lambda (msg)
					      ;;(setf -msg msg -stamp stamp -bcast bcast -type type)(break "bcast")
					      (warn-message '(broadcast fail)
							    (cons msg (messages-flush)))))
	(orb-broadcast stamp type bcast)
	)
      (orb-broadcast type bcast stamp)))



(defun cmd-eval (tags term &optional (type 'orb))
  (interpret-result (orb-eval type tags term t)))



(defun define-ml-com-primitive-types ()
  (add-primitive-type '|dependency|
		      #'(lambda (d) (dependency-to-string d))
		      :member-p #'(lambda (d) (dependency-p d))
		      :eq-func #'(lambda (a b) (equal-dependencies-p a b)))
  (add-primitive-type '|object_id|
		      #'(lambda (o) (declare (ignore o)) "OBJECT_ID")
		      :member-p #'oid-p
		      :eq-func #'equal-oids-p
		      )
  )
 

(define-ml-com-primitive-types)


(defun rehash-ml-primitive-types ()
  (reset-defined-ml-primitive-types)
  (define-ml-term-primitive-types)
  (define-ml-float-primitive-type)

  (define-ml-com-primitive-types)
  (when (fboundp 'define-ml-ref-primitive-types)
    (funcall (intern "DEFINE-ML-REF-PRIMITIVE-TYPES" *system-package*)))
  (when (fboundp 'define-ml-lib-primitive-types)
    (funcall (intern "DEFINE-ML-LIB-PRIMITIVE-TYPES" *system-package*)))
  (when (fboundp 'define-ml-edd-primitive-types)
    (funcall (intern "DEFINE-ML-EDD-PRIMITIVE-TYPES" *system-package*)))
    )

(defunml (|stamp_string| (unit) :declare ((declare (ignore unit))))
    (unit -> string)
  (stamp-to-string (transaction-stamp)))

(defunml (|stamp_term| (unit) :declare ((declare (ignore unit))))
    (unit -> term)
  (stamp-to-term (transaction-stamp)))


(defunml (|destruct_object_id_parameter| (parm))
	  (parameter -> object_id)
  (if (oid-parameter-p parm)
      (value-of-parameter-r parm)
      (raise-error (error-message '(parameter-type bad) (type-id-of-parameter parm)))))


(defunml (|make_object_id_parameter| (oid) :error-wrap-p nil)
	  (object_id -> parameter)
  (oid-parameter oid))


(defunml (|term_lookup| (oid))
    (object_id -> term)

  (lookup-term oid))

(defunml (|message_emit| (toks oids strings terms))
    ((tok list) -> ((object_id list) -> ((string list) -> ((term list) -> unit))))

  (message-emit (if (or strings terms)
		    (oid-message oids toks strings terms)
		    (oid-message oids toks))))




(defunml (|directory_term| (oid))
    (object_id -> term)

  (lookup-directory-term oid))


(defunml (|dag_roots| (unit) :declare ((declare (ignore unit))))
    (void -> ((token |#| object_id) list))
  
  (mapcar #'(lambda (r) (cons (car r) (cdr r))) (dag-roots)))


(defunml (|dag_root_p| (oid))
    (object_id -> bool)
  (dag-root-p oid))


(defunml (|dag_root_name| (oid))
    (object_id -> token)

  (dag-root-name oid))


(defunml (|dag_directory_p| (oid))
    (object_id -> bool)

  (dag-directory-p oid))


(defunml (|dag_directory_children| (oid))
    (object_id -> ((tok |#| object_id) list))

  (mapcar #'(lambda (r) (cons (car r) (cdr r)))
	  (dag-directory-children oid)))



;;;;	
;;;;	Ostate : lift ostate.
;;;;	
;;;;	
;;;;	
;;;;	lookup_ostate		: oid -> term{ostate}
;;;;	
;;;;	property_of_ostate	: tok{name} -> oid
;;;;	
;;;;	


(defunml (|equal_oids_p| (a b))
    (object_id -> (object_id -> bool))

  (equal-oids-p a b))



(defun interpret-command (cmd)
  (with-handle-error-and-message
      (()
       #'(lambda (msg)
	   (apply #'ifail-term 
		  (cons (message-to-term
			 (tag-message '(config request) msg))
			(mapcar #'message-to-term (messages-flush))))))

    (let ((r (case (id-of-term cmd)
	       ('|!subscribe| (if (un-of-isubscribe-term cmd)
				  (unsubscribe-server cmd)
				  (subscribe-server cmd)))
	       ('|!subscription| (if (un-of-isubscription-term cmd)
				     (unsubscribe-client cmd)
				     (subscribe-client cmd))
				 (iack-term))
	       (otherwise (raise-error (error-message '(interpret-message unknown) cmd))))))
      (if (null r)
	  (iack-term)
	  (ivalue-term r)))))
      

;; ?? what if someone redefines abs to expand to something more liberal ??
;; TODO : 
;;   - restrict mod of objects.
;;   - record stamp of definition as well as obid.
(defun abs-list-api-sentry (name oids)
  (add-api-sentries name
		    (list (cons name
				#'(lambda (term)
				    (when (icommand-term-p term)
				      (let ((abs (abstraction-of-term (term-of-iexpression-aux-term term) t)))
					(when (and abs (member-oid (oid-of-definition abs) oids))
					  (format-string "Bad input for api ~a" name)
					  ))))))))
	  
(defunml (|make_abstraction_api_sentry| (api-kind abs-ok))
    (tok -> ((object_id list) -> unit))

  (abs-list-api-sentry api-kind abs-ok))


(defunml (|find_api_sentry| (api-kind))
    (tok -> int)

  (or (cdr (assoc api-kind (list-api-sentries)))
      (raise-error (error-message '(no sentry) api-kind))))



;; moved here from orb-eval to solve forward reference to with-default-dependencies.

(defvar *profile-io-p* nil)


(define-primitive |!eval_property| () (property term))

(defvar *make_explicit_reference_environment* nil)

(defun eval-with-property (p term)
  (case (tag-of-iproperty-term p)

    (profile (with-profile (:time environment-eval)
	       (environment-eval-aux term)))

    ;; gross gross gross  == with_make_ref_environment
    (|reference_environment|
     (let ((prop (term-of-iproperty-term p)))
       (if (ioid-term-p prop)
	   (with-reference-environment-term prop
	     (environment-eval-aux term))
	   (with-make-ref-environment (#'make-explicit-ref-environment prop)
	     (with-reference-environment-term prop
	       (environment-eval-aux term))))))

    (transaction
     (set-transaction-properties (term-to-properties (term-of-iproperty-term p)))
     (environment-eval-aux term))

    (otherwise (message-emit (warn-message '(eval property recognized not) p))
	       ;;(setf -t term -p p) (break "eprn")
	       (environment-eval-aux term))))


;;;;	
;;;;	want to limit commands from a particular blink to a fixed set. 
;;;;	initially worry not about blink.
;;;;	
;;;;	use command-map to define the set.
;;;;	
;;;;	maybe wrap some method of limiting evals, then limit to chosen command map.
;;;;	
;;;;	
;;;;	

(defun environment-eval-aux (term)
  (cond
    ((ieval-property-term-p term)
     (eval-with-property (property-of-ieval-property-term term)
			 (term-of-ieval-property-term term)))
					     
    ;; conservative-reduce 
    ((iexpression-term-p term)
     ;; (break "expression")
     (let ((expr (term-of-iexpression-aux-term term)))
       (cond
	 ((or (iml-term-p expr)
	      (iml-woargs-term-p expr))
	  ;;(setf -expr expr) (break "expression")
	  (ml-eval (without-dependencies
		    (source-reduce-iml expr
				       (cons 'syntax
					     (cons 'ml (reduction-tags-of-environment
							(current-environment))))))))
	 ((ilisp-term-p expr)
	  (lisp-eval (source-reduce expr
				    (cons 'syntax
					  (cons 'lisp (reduction-tags-of-environment
						       (current-environment)))))))
	 (t ;;(setf a expr) (break)
	  (raise-error (error-message '(eval expression unknown) expr))))))

    ((iquery-term-p term)
     (let ((mapped (map-query (term-of-iexpression-aux-term term))))
       ;;(setf a mapped b term) (break)
       (ml-eval (or mapped
		    (iml-woargs-term
		     nil t
		     (let ((reduced (source-reduce (term-of-iexpression-aux-term term)
						   (cons 'ml (reduction-tags-of-environment
							      (current-environment))))))
		       (when (compare-terms-p reduced (term-of-iexpression-aux-term term))
			 (raise-error (error-message '(request query unreducible))))
		       reduced))))))

    ((icommand-term-p term)
     (let ((cmd (term-of-iexpression-aux-term term)))
       (cond
	 ((itransaction-term-p cmd)
	  (transaction-serve cmd))
	 ((icallback-term-p cmd)
	  (transaction-callback cmd))
	 ((iinterpret-term-p cmd)
	  (interpret-command (command-of-iinterpret-term cmd)))
	 (t #|(let ((e (map-command (term-of-iexpression-aux-term term))))
	      (ml-eval (source-reduce e
				      (cons 'ml (reduction-tags-of-environment
						 (current-environment))))))|#
	  
	  ;;(setf -term term -cmd cmd) (break "eea")
	  (map-command #'(lambda (cmd)
			   (ml-eval (source-reduce-iml
				     cmd
				     (cons 'syntax
					   (cons 'ml (reduction-tags-of-environment
						      (current-environment)))))))
		       cmd)
	  ))))

    (t (raise-error (error-message '(request unknown)))))
  )

(defun environment-eval (term)

  (with-handle-error-and-message (()
				  #'(lambda (m)
				      (apply #'ifail-term
					     (message-to-term
					      (tag-message *environment-path*
							   (tag-message '(eval) m)))
					     (mapcar #'message-to-term (messages-flush)))))
    
    ;;(setf a term) (break "ee")
    (when *eval-print-p*
      (format t "~%~a~%" (term-to-pretty-string term)))
    ;;(setf a term) (break "eee")
    
    (with-default-dependencies ('eval)
      ;; (setf a term) (break "wdd")
      ;; if !eval_property  then these not work
      (with-appropriate-transaction ((inter-orb-transaction-expression-p term)
				     (readonly-transaction-expression-p term))
	;;(format t "~%inter-orb-p ~a~%" (inter-orb-transaction-expression-p term))
	;;(setf -term term) (break "eee")

	(environment-eval-aux term)))))



(defvar *aterm* (ivoid-term))

;; a kludgey way of getting a term into ml.
(defunml (|lift_aterm| (unit) :declare ((declare (ignore unit))))
  (unit -> term)

  *aterm*)

(defvar *atermlist* (list (ivoid-term)))

;; a kludgey way of getting a term into ml.
(defunml (|lift_atermlist| (unit) :declare ((declare (ignore unit))))
  (unit -> (term list))

  *atermlist*)



(defvar *dummy-stamp* (dummy-transaction-stamp))
(defvar *dummy-oid* (new-oid *dummy-stamp*))

(defunml (|dummy_object_id| (unit) :declare ((declare (ignore unit))))
    (unit -> object_id)

  ;; why not *dummy-oid* ??
  (new-oid *dummy-stamp*))

;; really nothing temp about except hopefully intended use.
(defunml (|temp_object_id| (unit) :declare ((declare (ignore unit))))
    (unit -> object_id)

  (new-oid (new-transaction-stamp)))
  

(defunml (|static_object_id| (name))
    (tok -> object_id)

  (new-oid (static-transaction-stamp name)))


(defunml (|new_dummy_object_id| (unit) :declare ((declare (ignore unit))))
    (unit -> object_id)
  (new-oid (dummy-transaction-stamp)))

(defun dummy-object-id-p (oid)
  (dummy-transaction-stamp-p (stamp-of-oid oid)))

(defunml (|dummy_object_id_p| (oid))
  (object_id -> bool)
  (dummy-object-id-p oid))

(defunml (|with_ref_environment_term| (f a envterm))
    ((* -> **) -> (* -> (term -> **)))

  ;;(break "wre") (setf -r 
  (with-reference-environment-term envterm
    ;;(break "wreb")
    (funmlcall f a))
  ;;)(break "wrer") -r
  )

(defunml (|with_ref_environment| (f a envoid))
    ((* -> **) -> (* -> (object_id -> **)))

  ;;(break "wre") (setf -r 
  (with-reference-environment envoid
    ;;(break "wreb")
    (funmlcall f a))
  ;;)(break "wrer") -r
  )

(defunml (|without_ref_environment| (f a))
    ((* -> **) -> (* -> **))

  (without-ref-environment 
   (funmlcall f a)))

(defunml (|reset_get_ref_environment_f| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)

  (table-visibility-reset)
  (setf *enter-exit-ref-environment-f* nil)
  )

(defunml (|current_ref_environment_index| (tok))
    (tok -> object_id)

  ;;(break "crei")
  (or (cdr (assoc tok (cddr *ref-environment*)))
      *dummy-oid*))

(defunml (|set_current_ref_environment_index| (tok oid))
    (tok -> (object_id -> unit))
  (setf (cddr *ref-environment*)
	(acons tok oid (cddr *ref-environment*))))


(defvar *really-break-p* nil)
(defunml (|really_break| (m))
    (tok -> unit)

  (unless (or (member m '(|graph_find|))
	      (not *really-break-p*))
    (break (format-string "really_break ~a" m)))
  nil)

(defunml (|current_ref_environment| (unit) :declare ((declare (ignore unit))))
    (unit -> object_id)

  (or (when (boundp '*ref-environment*)
	(let ((reterm (cadr *ref-environment*)))
	  (unless (and reterm (ioid-term-p reterm))
	    (raise-error (error-message '(|current_ref_environment| obid not))))
	  (oid-of-ioid-term reterm)))
      (raise-error (error-message '(|current_ref_environment|)))))

(defunml (|current_ref_environment_term| (unit) :declare ((declare (ignore unit))))
    (unit -> term)

  (or (when (and (boundp '*ref-environment*) *ref-environment*)
	(cadr *ref-environment*))
      (raise-error (error-message '(|current_ref_environment_term|)))))


(defunml (|note_dependency| (tok oid))
    (token -> (object_id -> unit))

  (dependency-note-reference tok (dependency oid nil nil)))

(defunml (|toggle_process_break| (unit) :declare ((declare (ignore unit))))
    (unit -> bool)

  (setf *process-break* (not *process-break*)))


(defunml (|oid_find_paths| (oid))
    (object_id -> ((int |#| (tok list)) list))

  (oid-find-paths oid))


(define-primitive |!io_history_header| ((time . time) (bool . tracep) (n . length)))
(define-primitive |!io_history_entry| ((tok . dir) (n . opcount)) (term))
(define-primitive |!double_break_cons| () (car cdr))

(defunml (|io_history_snap| (unit) :declare ((declare (ignore unit))))
    (unit -> (term |#|(term list)))

  (cons (iio-history-header-term (get-universal-time) (io-trace-file-p) (or *io-history-count* 0))
	(mapcar #'(lambda (ioh)
		    (iio-history-entry-term (car ioh)
					    (term-op-count (cdr ioh))
					    (cdr ioh))
		    )
		*io-history*
		;;(subseq *io-history* 0 10)
		)))



(defunml (|match_db_logs| (paddr))
    ((tok list) -> ((term |#| term) list))

  (mapcar #'(lambda (log)
	      (cons 
	       (map-list-to-ilist (car log)
				  (inil-term)
				  #'itoken-term)
	       (stamp-to-term (cdr log))))
	  
	  (match-db-environment-all paddr)))


(defun log-segment-aux (start len stamp)

  (let ((log (log-open-read (term-to-stamp stamp))))

    (dotimes (i start)
      (log-read-record log))

    (let ((acc nil))
      (dotimes (i len)
	(let ((e (log-read-record log)))
	  (when e
	    (push e acc))))

      (nreverse acc))))

(defunml (|log_segment| (start len stamp))
    (int -> (int -> (term -> (term list))))

  (log-segment-aux start len stamp))

    

(defunml (|config_string| (name))
    (tok -> (string list))

  (let ((data (get-config-data name)))
    (mapcar #'(lambda (data)
		(if (stringp data)
		    data
		    (if (and data (or (integerp data) (symbolp data)))
			(string data)
			(raise-error (error-message '(config_data string not) name)))))
	    data)))
	

(defunml (|config_sockets| (unit)  :declare ((declare (ignore unit))))
    (unit -> (int |#| int))

  (let ((sockets (get-config-data 'sockets)))

    ;;(setf -s sockets) (break)
    ;; could have list be accept edd, ref, .... port nums???
    (if (and (consp sockets)
	     (integerp (car sockets))
	     (integerp (cadr sockets)))
	(case *component-kind*
	  (|edd| (cons (car sockets) (- (cadr sockets) 1)))
	  (|ref| (cons (car sockets) (- (cadr sockets) 2)))
	  (|ref501| (cons (car sockets) (- (cadr sockets) 2)))
	  (otherwise (cons (car sockets) (cadr sockets))))
	(raise-error (error-message '(config-sockets not))))))


(defun config-accept ()

  (let ((sockets (get-config-data 'sockets)))
    (let ((port (cond
		  ((integerp sockets) sockets)
		  ((and (consp sockets)
			(integerp (car sockets)))

		   ;;(setf -s sockets) (break)
		   ;; could have list be accept edd, ref, .... port nums???
		   (car sockets)))))
      (or port
	  (raise-error (error-message '(config-accept not)))))))


(defunml (|config_accept| (unit)  :declare ((declare (ignore unit))))
    (unit -> int)
  (config-accept))


(defunml (|config_libhost| (unit)  :declare ((declare (ignore unit))))
    (unit -> string)

  (let ((libhost (car (get-config-data 'libhost))))

    (if libhost
	(string libhost)
	(raise-error (error-message '(config-libhost not))))))

(defun config-libenv (&optional nil-ok-p)
  (let ((libenv (car (get-config-data 'libenv))))

    (if libenv
	(string libenv)
	(unless nil-ok-p
	  (raise-error (error-message '(config-libenv not)))))))

(defunml (|config_libenv| (unit)  :declare ((declare (ignore unit))))
    (unit -> string)

  (config-libenv))


;;;;	
;;;;	
;;;;	
;;;;	
;;;;	


(defunml (|lookup_object_attr| (kind oid))
    (tok -> (object_id -> term))

  (data-of-object-attr-def (lookup-object-attr-def kind oid)))

(defunml (|map_object_attrs| (kind f))
    (tok -> ((object_id -> (term -> unit)) -> unit))


  (object-attr-table-map
   kind
   #'(lambda (oid def)
       (funmlcall f oid (data-of-object-attr-def def)))))
			 


(defunml (|term_fix_text| (term))
  (term -> term)

  (term-fix-text term))


(defunml (|with_object_id| (f oid))
    ((unit -> *) -> (object_id -> *))

  (let ((*current-dependency* (dependency oid nil nil)))
    ;;(fooe);;(break "woi")
    (funmlcall f nil)))


(defunml (|code_object_p| (oid))
    (object_id -> bool)
    
  (let ((code-def (definition-lookup-by-oid (resource 'code)
		      oid t
		      (current-transaction-stamp)
		      t)))
    (and code-def
	 t)))

(defunml (|substantive_property_of_code| (tag oid))
    (tok -> (object_id -> term))

  (let ((code-def (definition-lookup-by-oid (resource 'code)
		      oid t
		      (current-transaction-stamp)
		      t)))

    (if code-def
	(or (property-of-substance (ephemeral-substance-of-definition code-def 'substance) tag)
	    (ivoid-term))
	(raise-error (oid-error-message (list oid) '(substantive_property_of_code code not) tag)))))

(defunml (|ref_get_substantive_property| (tag oid))
    (tok -> (object_id -> term))

  (let ((stamp (current-transaction-stamp)))
    (let ((def (or (definition-lookup-by-oid (resource 'code) oid t stamp t)
		   (definition-lookup-by-oid (resource 'statements) oid t stamp t)
		   (definition-lookup-by-oid (resource 'abstractions) oid t stamp t))))
      (if def
	  (or (property-of-substance (ephemeral-substance-of-definition def 'substance) tag)
	      (ivoid-term))
	  (raise-error (oid-error-message (list oid)
					  '(ref_get_substantive_property ref-substance not)
					  tag))))))
				       
(defunml (|substance_term_of_code| (oid))
    (object_id -> term)

  (let ((code-def (definition-lookup-by-oid (resource 'code)
		      oid t
		      (current-transaction-stamp)
		      t)))

	(or (when code-def
	      (term-of-substance (ephemeral-substance-of-definition code-def 'substance)))
	    (raise-error (oid-error-message (list oid) '(substantive_term_of_code code not))))))



;; todo : might want to fold into ref_environment_ref_state
(defvar *proper-refenvs* (new-oid-table))

(defunml (|declare_refenv_proper| (oid))
    (object_id -> unit)

  (hashoid-set *proper-refenvs* oid t)
  (values))

(defunml (|is_refenv_proper| (oid))
    (object_id -> bool)

  (and (hashoid-get *proper-refenvs* oid)
       t))

(defvar *improper-cached-refenvs* nil)

(defunml (|declare_refenv_improper_cached| (oid codedeps))
    (object_id -> (object_ids -> unit))

  (push (cons oid codedeps) *improper-cached-refenvs*)
  (values))

(defunml (|is_refenv_improper_cached| (oid))
    (object_id -> (object_id list))

    (or (cdr (assoc oid *improper-cached-refenvs* :test #'equal-oids-p))
	(failwith '|is_refenv_improper_cached|)))
  

(add-transaction-end-hook 'improper-refenvs
			  #'(lambda (l)
			      (when l
				(when *improper-cached-refenvs*
				  (format t "~%improper-cached-refenvs reset ~a~%" (length *improper-cached-refenvs*))
				  (setf *improper-cached-refenvs* nil)))))


(defunml (|sanitize_object_name| (n))
    (string -> string)
  (sanitize-name-string n))

;; int or ascii codes.
(defunml (|sanitize_object_name_aux| (p f name))
    ((int -> bool) -> ((int -> string) -> (string -> string)))

  (implode-to-string
   (mapcan #'(lambda (ich)
	       (if (not (funmlcall p ich))
		   (istring (funmlcall f ich))
		   (list ich)))
	   (istring name))))

;; (setf *daily-sleeper* (list 8 18 (* 30 60)))
(defunml (|daily_sleep| (forcep))
    (bool -> unit)
  (daily-sleep forcep))







