
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;
;;;;	term table.
;;;;	

;;;;	
;;;;	<dir>		: !directory(<root>:s)(<dag-child> !dag_cons ilist)
;;;;			| !directory()(<dag-child> !dag_cons ilist)
;;;;
;;;;	<dag-child>	: !dag_child{<name>:s, <oid>:o}
;;;;	
;;;;	<dag-filter>	: !dag_filter(<term{input}>; <filter>)
;;;;	  * input of !global() produces dag list of all ostates.
;;;;	  * eval produces <dag-child> !dag_cons ilist
;;;;	    first oid of term is used as input. input should be either a directory
;;;;	    or another filter. If a filter the dynamic substance of the filter is used.
;;;;	
;;;;	<dag-path>	: !dag_path(<term{path})
;;;;	  * eval produces !oid{<o:oid> list} where root is last and leaf is first.
;;;;
;;;;	
;;;;	<dyneval>	: !dyneval{<id>:o,<active>:b}(<condition list>; <expression>, <stamp>, <value>)
;;;;	  * id is available to allow discrimination by dforms.
;;;;	
;;;;	<expression>	: <dag-filter>
;;;;			| <dag-path>
;;;;	
;;;;	<list-expression> : <expression>
;;;;	
;;;;	To be dynamically updatable the object must have the dynamicrefresh property.
;;;;	
;;;;	
;;;;	<scrollpos>	: !scroll_position{<len>:n, <offset>:n, <size>:n, <point>:n}(<sublist>)
;;;;	
;;;;	<scrolllist>	: !scroll_list{<id>:o, <offset>:n, <size>:n, <point>:n}(<op>; <list-expression>)
;;;;	
;;;;	<navigator>	: !dyneval{doc:Navigator:o, T:b}
;;;;				  ( EDITREFRESH
;;;;				  ; <scrolllist{<dag-filter{<oid-stack>; !void()}>}>
;;;;				  ; <stamp>
;;;;				  ; <scrollpos>
;;;;				  )
;;;;	oid-stack is arg to filter, list derived from hd of oid stack is arg to scroll-list.
;;;;	
;;;;	
;;;;	
;;;;	Detecting stale
;;;;	Compute value
;;;;	
;;;;	
;;;;	
;;;;	Dynamic refresh requires term to be updated when referenced data changes.
;;;;	
;;;;	Navigator case study: 
;;;;	  - ostate broadcast :
;;;;	      * head of oid-stack is dynamic term.
;;;;	      * head of oid-stack touched
;;;;	      * member of scrollpos sublist touched
;;;;	      * other member of dir of head of oid stack touched
;;;;		ie new entry added which should be visible, implies head touched.
;;;;	  - edit modification :
;;;;	      * point modified within visible window.
;;;;	      * size, point, or offset modified.
;;;;		   - re-evaluate scroll list
;;;;	 
;;;;	If there were a filter expression then change of non-visible important as
;;;;	may change result of filter eval.
;;;;	 
;;;;	It may then be sensible to have a third subterm to the dag-filter which is list raw list
;;;;	derived from input.
;;;;	 
;;;;	   If member of list derived from input is touched then must the input be touched?
;;;;	     - as any descendent of tree a dependency of the root? YES.
;;;;	     - can't think of a counter example, derived from input is kinda fuzzy.
;;;;	 
;;;;	 if navigator is positioned on open list and open list contains navigator is that 
;;;;	 a cycle? 
;;;;	 

Then dependences static as there will be no need to repor what if input is from some other

(defstruct term-dyneval
  (flags nil)
  (stamp nil)				; stamp at last eval.
  (term nil)				; result of last dyneval.
  (dependencies nil)			; dependencies of last dyneval.
  (prev nil)				; (term . dependencies) of previous dyneval.
  )
    

(define-flags (term-dyneval dyneval)

    ((inprogress nil t)			; used for eval cycle detection.

     ;; did refresh cause an eval?
     (evaled nil t)	; set to t during update if any dynamic eval happened.

     ;; allow lazy computation of dependencies :
     (dependencies nil t) ; if t then includes static value dependencies.

     ;;
     (modified asynch synch nil)
     ))

(defun term-of-term-dyneval (tde) (term-dyneval-term tde))
(defun stamp-of-term-dyneval (tde) (term-dyneval-stamp tde))
(defun dependencies-of-term-dyneval (tde) (term-dyneval-dependencies tde))
(defun prev-of-term-dyneval (tde) (term-dyneval-prev tde))

(defun old-term-of-term-dyneval (tdyn)
  (when tdyn
    (let ((prev (prev-of-term-dyneval tdyn)))
      (when prev
	(term-of-term-dyneval prev)))))


(defstruct (term-def (:include definition))
  (dyneval nil)		;; if dynamic refresh then should have value.
  )

(defun dyneval-of-term-def (td)
  (when (term-def-p td) (term-def-dyneval td)))

(defun dyneval-dependencies-of-term-definition (def)
  (dependencies-of-term-dyneval (dyneval-of-term-def def)))



(defun term-dyneval-inprogress-p (tdyn)
  (dyneval-flag-inprogress-p tdyn))

;; used to update interactively.
;; would be better to make new term-dyneval struct and save one level of old.
(defun dyneval-set-term (def term)
  ;;(setf -term term) (break "dst")
  (let ((tdyn (dyneval-of-term-def def)))
    
    ;;(setf -term term -tdyn tdyn) (break "dst")
    (when tdyn
      (when (term-dyneval-inprogress-p tdyn)
	(raise-error (error-message '(dyneval term set in-progess))))
      (setf (term-def-dyneval def) 
	    (make-term-dyneval :term term :prev tdyn))
      (setf (term-dyneval-prev tdyn) nil)
      )))


(defun dynamic-refresh-p (def)
  (and (term-def-p def)
       (not (null (dyneval-of-term-def def)))))

(defun dynamic-refresh-reset (def)
  (let ((tdyn (dyneval-of-term-def def)))
    (when tdyn
      (dyneval-flag-set-inprogress tdyn nil))))


;;;;	
;;;;	Hooks can be supplied to be called when definitions
;;;;	are dynamically updated.
;;;;	

(defvar *dyneval-hooks* nil)


(defun add-dyneval-hook (tok f)
  (setf *dyneval-hooks* (cons (cons tok f)
			      (delete tok *dyneval-hooks* :key #'car)))
  )

(defun dyneval-call-hooks (tdef)
  ;;(break "dch")
  (mapc #'(lambda (h) (funcall (cdr h)
			       (oid-of-definition tdef)
			       (term-of-term-def tdef)
			       tdef))
			       
	*dyneval-hooks*))


(defstruct (directory-def (:include term-def))
  (children nil)	;; alist of tok . oid
  )


(defstruct (root-directory-def (:include directory-def))
  rootname
  )

(defun children-of-directory-def (d) (directory-def-children d))
(defun name-of-root-directory-def (d) (root-directory-def-rootname d))


(define-primitive |!directory| () (children))
(define-primitive |!dag_child| ((token . name) (oid . oid)))
(define-primitive |!dag_child_state| ((token . name) (oid . oid)) (state))
(define-primitive |!dag_cons| () (car cdr))
(define-primitive |!dag_filter| () (input filter))
(define-primitive |!placeholder_filter| () (template))
(define-primitive |!dag_global|)
(define-primitive |!dag_root|)


(defun idirectory-root-term-p (term)
  (and (eql (id-of-term term) *idirectory*)
       (let ((parms (parameters-of-term term)))
	 (and parms
	      (null (cdr parms))
	      (token-parameter-p (car parms))))
       (let ((bts (bound-terms-of-term term)))
	 (and bts
	      (null (cdr bts))
	      (null (bindings-of-bound-term (car bts)))))))


(defun name-of-idirectory-root-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))

(defun idirectory-term-to-child-alist (term)
  (map-isexpr-to-list (children-of-idirectory-term term)
		      (idag-cons-op)
		      #'(lambda (ichild)
			  (if (idag-child-term-p ichild)
			      (cons (name-of-idag-child-term ichild)
				    (oid-of-idag-child-term ichild))
			      (progn
				(message-emit (warn-message '(|ObjectIdDAG| directory child not) ichild))
				nil)))))

(defstruct (persistent-event-def (:include term-def))
  (term nil)	
  )

(defmacro term-of-persistent-event-def (def) `(persistent-event-def-term ,def))

(define-primitive |!persistent_event| () (term))

(defun import-term-def (term)
  
  (let* ((substance (term-to-data term))
	 (tsubstance (provide-data substance 'substance)))

    (let ((idir (term-of-substance tsubstance))
	  (dyn-p (property-of-substance tsubstance 'dynamicrefresh)))
      ;;(setf -idir idir) (break "itd")
      (cond
	((idirectory-root-term-p idir)
	 (make-root-directory-def :substance substance
				  :rootname (name-of-idirectory-root-term idir)
				  :children (idirectory-term-to-child-alist idir)))

	((idirectory-term-p idir)
	 (make-directory-def :substance substance
			     :children (idirectory-term-to-child-alist idir)))

	(dyn-p
	 (make-term-def :dyneval (make-term-dyneval :term idir)
			:substance substance))

	((ipersistent-event-term-p idir)
	 (make-persistent-event-def :substance substance
				    :term (term-of-ipersistent-event-term idir)))
	 
	(t (make-definition :substance substance))))))


(defun term-of-term-def (td &optional not-dyn-p)
  (let ((dyneval (dyneval-of-term-def td)))
    (if (and dyneval (null not-dyn-p))
	(term-of-term-dyneval dyneval)
	(term-of-substance (substance-of-definition td 'substance)))))

(defun property-of-term-def (name td)
  (property-of-substance (substance-of-definition td 'substance) name))

;; event dependencies
(defun dependencies-of-term-def (td)
  (dependencies-of-substance (substance-of-definition td 'substance)))


(defstruct (term-table (:include definition-table))
  (listings nil)
  )
			       

;; for performance reasons touch history does not include members of
;; log files read. This should not generally be a problem since log
;; files are mostly read at connect and most touch-history hooks are
;; more concerned with incremental updates.
;; 
;; however, it would be good to have some log and type indication in the touch history
;; so that clients may perform global refresh if appropriate.
;; 
(defun tree-listings-end-hook (thl)
  (when (and thl (resource-p 'terms))
    (setf (term-table-listings (resource 'terms)) nil)))

(defmacro terms-table (stamp tag &rest keys &key &allow-other-keys)
  `(define-definition-table
      ,stamp
      (list 'terms ,tag)
    nil 

    :import-f #'import-term-def
    :make-f make-term-table

    ,@keys))
    

(defun allocate-terms (stamp tag)
  (add-transaction-end-hook 'tree-listings #'tree-listings-end-hook)
  (terms-table stamp tag)
  )

(defun lookup-term-def (oid &optional nil-ok-p)
  (if (resource-p 'terms)
      (definition-lookup-by-oid (resource 'terms) oid nil-ok-p)
      (progn
	;;(setf -a oid -b)   (break "ltd")
	(raise-error (oid-error-message (list oid) '(terms lookup not)))
	)))


(defun lookup-term (oid)
  (term-of-term-def (lookup-term-def oid)))

(defun lookup-term-not-dyn (oid)
  (term-of-term-def (lookup-term-def oid) t))

(defun replace-ioid-with-ostate (term)
  (labels ((visit-term (term)
	     (if (idag-child-term-p term)
		 (let ((ostate (lookup-ostate (oid-of-idag-child-term term))))
		   ;;(setf b ostate c term) (break "riwo")
		   (if ostate
		       (idag-child-state-term (name-of-idag-child-term term)
					      (oid-of-idag-child-term term)
					      ostate)
		       term))
		 (maybe-instantiate-term term
					 (operator-of-term term)
					 (mapcar #'(lambda (bt)
						     (maybe-instantiate-bound-term
						      bt
						      (bindings-of-bound-term bt)
						      (visit-term (term-of-bound-term bt))))
						 (bound-terms-of-term term))) )))

    (visit-term term)))


(defun lookup-directory-term (oid &optional shortp)
  (let ((def (lookup-term-def oid)))
    ;;(setf a def) (break "ldt")
    (if (directory-def-p def)
	(if shortp
	    (term-of-term-def def)
	    (replace-ioid-with-ostate (term-of-term-def def)))
	(raise-error (oid-error-message (list oid) '(|ObjectIdDAG| directory lookup not)
					(term-of-term-def def))))))

(defun lookup-dynamic-term (oid)
  (let ((def (lookup-term-def oid t)))
    ;;(setf a def) (break "ldt")
    (when (and def (term-def-p def))
      (term-of-term-def def))))

(defun persistent-events ()
  (unless (resource-p 'terms)
    ;;(break "pe")
    (raise-error (error-message '(terms lookup persistent-events))))
  
  (let ((acc nil))
    (definition-table-map (resource 'terms)
	(local-transaction-stamp)
      #'(lambda (oid def)
	  (when (persistent-event-def-p def)
	    (push (cons oid (term-of-persistent-event-def def))
		  acc))))
    acc))

;;;;	
;;;;	funcs to access roots, children, paths etc.
;;;;	
;;;;	dag_roots	: unit -> object_id list
;;;;	
;;;;	dag_root_p	: object_id -> bool
;;;;	dag_root_name	: object_id -> string
;;;;	
;;;;	dag_directory_p	: object_id -> bool
;;;;	  - true for roots as well.
;;;;	dag_directory_children	: object_id -> (string # object_id) list
;;;;

(defun dag-roots ()
  (unless (resource-p 'terms)
    ;;(break "dr")
    (raise-error (error-message '(terms lookup not roots))))
  
  (let ((acc nil))
    (definition-table-map (resource 'terms)
	(local-transaction-stamp)
      #'(lambda (oid def)
	  (when (root-directory-def-p def)
	    (push (cons (name-of-root-directory-def def) oid) acc))))
    acc))

(defun dag-root (name)
  (cdr (assoc name (dag-roots))))

(defun dag-root-p (oid)
  (let ((tdef (lookup-term-def oid t)))
    (and tdef
	 (root-directory-def-p tdef)
	 t)))

(defun dag-root-name (oid)
  (let ((def (lookup-term-def oid)))
    (if (root-directory-def-p def)
	(name-of-root-directory-def def)
	(raise-error (oid-error-message (list oid) '(dag root name not))))))

(defun dag-directory-p (oid)
  (let ((term-def (lookup-term-def oid t)))
    (and term-def (directory-def-p term-def) t)))

(defun dag-directory-children (oid)
  (let ((def (lookup-term-def oid)))
    (if (directory-def-p def)
	(children-of-directory-def def)
	(raise-error (oid-error-message (list oid) '(dag directory name not))))))

(defun dag-directory-child (oid name)
  (let ((def (lookup-term-def oid)))
    (if (directory-def-p def)
	(cdr (assoc name (children-of-directory-def def)))
	(raise-error (oid-error-message (list oid) '(dag directory name not))))))




;;;;	
;;;;	
;;;;	
;;;;	Object Attributes table :
;;;;	  - ostate
;;;;	  - ddag
;;;;	
;;;;	

;; kind may be redundant with definition wrapper which needed kind to find table ??

(define-primitive |!object_attr_state_not| ((oid . oid)))

(defstruct (object-attr-def (:include definition))
  (kind nil)
  (data nil)				; term
  )

(defun kind-of-object-attr-def (d) (object-attr-def-kind d))
(defun data-of-object-attr-def (d) (object-attr-def-data d))

(defun object-attr-def-not-p (d)
  (iobject-attr-state-not-term-p (data-of-object-attr-def d)))


(defun import-object-attr-def (term makef importf)
  (let ((def (funcall makef)))
    (setf (object-attr-def-data def) term)
    (if (object-attr-def-not-p def)
	def
	(funcall importf def))))


(defun object-state-def-not-p (def)
  (if (object-attr-def-p def)
      (object-attr-def-not-p def)
      (ostate-def-not-p def)))
      
(defmacro object-attr-table (kind makef importf tag stamp &rest keys &key &allow-other-keys)
  `(define-definition-table
    ,stamp
    (list ,kind ,tag)
    nil 
    :import-f #'(lambda (term)
		  ;;(setf -term term) (break "oat") ; may want to pass ioad subterm of term?
		  (import-object-attr-def term ,makef ,importf))
    ,@keys))

;;;;	
;;;;	table updates will be in form of definition replace.
;;;;	
;;;;	<def-replace>	: !definition_replace{<parms>}(<op>(<definition> bound term list))
;;;;	

(defvar *object-attr-table-types* '(ddg ostates))

(defvar *object-attr-states-op* (instantiate-operator '|object_attr_states| nil))

(defmacro allocate-object-attr (kind makef importf tag stamp &rest keys &key &allow-other-keys)
  `(object-attr-table
   ,kind
   (or ,makef #'make-object-attr-def)
   (or ,importf #'identity)
   ,tag ,stamp
   ,@keys))


;; by default do not note object attr lookups?!?
(defun lookup-object-attr-def (kind oid)
  (if (resource-p kind)
      (definition-lookup-by-oid (resource kind) oid t nil t)
      (raise-error (oid-error-message (list oid) (cons kind '(lookup not))))
      ))

(defun object-attr-table-map (kind f)
  (definition-table-map (resource kind)
      (current-transaction-stamp)
    f))

(defun ddg-def-max ()
  (let ((max (cons nil 0)))
    (object-attr-table-map 'ddg
			   #'(lambda (oid def)
			       (let ((c (term-op-count (ddg-def-data def))))
				 (when (> c (cdr max))
				   (setf max (cons oid c))
				   (format t "~%~a" (term-op-count (ddg-def-data def)))))))
    max))

(defun count-attrs (type)
  (let ((count 0))
    (object-attr-table-map type #'(lambda (&rest r) (incf count)))
    count))

;;;;	
;;;;	
;;;;	Dependency Directed Graph
;;;;	
;;;;	should elaborate ddg structure later, ie want more than default object-attr-def.
;;;;	

;; data terms of object-attr def term:
(define-primitive |!ddg_state| () (dependency store xref)) 

;; xref : key is symbol ->  value is (source . xref) alist
;;				source is oid | term | nil
;;      | key is oid    ->  value is xef list

(defstruct (ddg-def (:include object-attr-def))
  store
  xref
  (layers nil)
  )

(defun store-of-ddg-def (d) (ddg-def-store d))
(defun xref-of-ddg-def (d) (ddg-def-xref d))
(defun layers-of-ddg-def (d) (ddg-def-layers d))
(defun layer-dag-of-ddg-def (type d)
  (cdr (assoc type (ddg-def-layers d))))


(defstruct (ddg-table (:include definition-table))
  (references (make-hash-table :size 1000 :test #'equal))
  (need-ref-p t)
  (xref (make-hash-table :size 1000 :test #'equal))
  (need-xref-p t))

(defun import-ddg-def (def)
  (let ((term (data-of-object-attr-def def)))
    (setf (ddg-def-store def)
	  (provide-data (term-to-data (store-of-iddg-state-term term))
			'dependency-store))
    (let ((xterm (xref-of-iddg-state-term term)))
      (unless (ivoid-term-p xterm)
	(setf (ddg-def-xref def) (term-to-code-xrefs xterm))))
    def))

(defun ddg-rehash-references (table)
  (setf (ddg-table-need-ref-p table) t
	(ddg-table-need-xref-p table) t)
  (clrhash (ddg-table-references table))
  (clrhash (ddg-table-xref table)))


;; see comments near tree-listings end-hook
(defun ddg-transaction-end-hook (th)

  (let ((ddg (resource 'ddg)))
    (when ddg
      (when (exists-p #'(lambda (tr)
			  (declare (ignore tr))
			  (member 'ddg (tags-of-definition-table ddg)))
		      th)

	;;(with-local-transaction
	    (ddg-rehash-references ddg)))))
   ;;)



(defun ddg-table (tag stamp)
  (allocate-object-attr 'ddg
			#'make-ddg-def
			#'import-ddg-def
			tag stamp
			:make-f make-ddg-table))

(defun allocate-ddg (stamp name)
  (ddg-table name stamp))



;;;;	
;;;;	Object State defs.
;;;;	
;;;;	

(define-primitive |!object_state|
    ((oid . oid) (bool . active) (bool . allow) (token . kind) (bool . translated))
  (description objc source substance properties other))

(define-primitive |!object_state_not| ((oid . oid)))



(defstruct (ostate-def (:include definition))
  (state nil)	;; term
  (properties nil)
  )

(defun state-of-ostate-def (d) (ostate-def-state d))
(defun properties-of-ostate-def (d) (ostate-def-properties d))

(defun ostate-def-not-p (d)
  (iobject-state-not-term-p (state-of-ostate-def d)))

(defun import-ostate-def (term)
  (make-ostate-def :state term
		   :properties (unless (iobject-state-not-term-p term)
				 (term-to-properties
				  (properties-of-iobject-state-term term)))))



(defmacro ostates-table (stamp tag &rest keys &key &allow-other-keys)
  `(define-definition-table
    ,stamp
    (list 'ostates ,tag)
    nil 
    :import-f #'import-ostate-def
    ,@keys))

(defun allocate-ostates (stamp tag)
  (ostates-table stamp tag))


;; by default do not note ostate lookups?!?
(defun lookup-ostate-def (oid)
  (if (resource-p 'ostates)
      (definition-lookup-by-oid (resource 'ostates) oid t nil t)
      (raise-error (oid-error-message (list oid) '(ostates lookup not)))
      ))

(defun lookup-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
      (state-of-ostate-def def))))

(defun lookup-ostate-dependency (oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
      (dependency-of-definition def))))


(defun property-of-ostate-def (name def)
  (cdr (assoc name (properties-of-ostate-def def))))

(defun property-of-ostate (name oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
      (cdr (assoc name (properties-of-ostate-def def))))))


(defun properties-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
      (properties-of-ostate-def def))))


(defun name-property-of-ostate-def (def)
  (let ((prop (property-of-ostate-def 'name def)))
	(when (and prop (itoken-term-p prop))
	  (token-of-itoken-term prop))))
  
(defun name-property-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
	  (name-property-of-ostate-def def))))

(defun active-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when (and def (not (ostate-def-not-p def))) 
     (active-of-iobject-state-term (state-of-ostate-def def)))))

(defun substance-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    ;;(setf -oid oid -def def) (break "soo")
    (when def
      (let ((istamp (substance-of-iobject-state-term (state-of-ostate-def def))))
	(unless (ivoid-term-p istamp)
	  (term-to-stamp istamp))))))


(defun active-of-ostate-r (oid)
  (let ((def (lookup-ostate-def oid)))
    (if (and def (not (ostate-def-not-p def))) 
	(active-of-iobject-state-term (state-of-ostate-def def))
	(oid-error-message (list oid) '(|ObjectState| active not)))))

(defun kind-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when (and def (not (ostate-def-not-p def))) 
     (kind-of-iobject-state-term (state-of-ostate-def def)))))


(defun description-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when (and def (not (ostate-def-not-p def))) 
     (description-of-iobject-state-term (state-of-ostate-def def)))))


  
(defun edit-refresh-property-of-ostate (oid)
  (let ((def (lookup-ostate-def oid)))
    (when def
      (property-of-ostate-def 'editrefresh def))))




(defun first-oid-of-term (term)
  (find-first #'(lambda (p)
		  (when (oid-parameter-p p)
		    (let ((v (value-of-parameter p)))
		      (when (real-parameter-value-p v (type-of-parameter p))
			v))))
	      (parameters-of-term term)))


;;;;	
;;;;	Wrapper :
;;;;	
;;;;	
;;;;	  !wrapper{<kind>:t}(<term>)
;;;;	  !wrapper{<kind>:t}(<term>, <term{annotation}>)
;;;;	
;;;;	
;;;;	
(defconstant *ilabel* '|!label|)
(defconstant *itag* '|!tag|)


(defun real-ilabel-term-p (term)
  (and (eql (id-of-term term) *ilabel*)
       (forall-p #'(lambda (p)
		     (and (token-parameter-p p)
			  (real-parameter-value-p (value-of-parameter p) *token-type*)))
		 (parameters-of-term term))
       (let ((bound-terms (bound-terms-of-term term)))
	 (and bound-terms
	      (null (cdr bound-terms))
	      (null (bindings-of-bound-term (car bound-terms)))))))


(define-primitive |!wrapper| ((token . kind)) (term))

(defun ixwrapper-term-p (term)
  (and (eql (id-of-term term) *iwrapper*)
       (let ((parms (parameters-of-term term)))
	 (and parms (token-parameter-p (car parms)))
	 (and (bound-terms-of-term term)))))

(defun wrapped-term-p (term)
  (or (imark-term-p term)
      (real-ilabel-term-p term)
      (ixwrapper-term-p term)))

(defun term-of-wrapped-term (term)
  (if (wrapped-term-p term)
      (term-of-wrapped-term (icar term))
    term))

(defun wrapped-term (kind term &optional toks other-terms)
  (instantiate-term (instantiate-operator *iwrapper*
					  (cons (token-parameter kind)
						(when toks
						  (mapcar #'token-parameter toks))))
		    (cons (instantiate-bound-term term)
			  (when other-terms
			    (mapcar #'instantiate-bound-term other-terms)))))


(defun transfer-wrapper (old new &optional unwrap)
  (if (wrapped-term-p old)
      (if (and unwrap (eql (kind-of-iwrapper-term old) unwrap))
	  (transfer-wrapper (icar old) new unwrap)
	(instantiate-term (operator-of-term old)
			  (cons (instantiate-bound-term (transfer-wrapper (icar old) new unwrap))
				(cdr (bound-terms-of-term old)))))
    new))


  
(defun transfer-parameter-marks (old new)
  (let ((marks (marks-of-parameter-value old)))
    (if (null marks)
	new
	(let ((nnew new))
	  (dolist (mark marks)
	    (setf nnew (mark-parameter-value nnew (car mark) (cdr mark))))
	  ;;(setf a nnew) (break "tpm")
	  nnew))))


;;;	
;;;	Input can be either list or isexpr since different sources
;;;	may have diff reps. Output always is list.
;;;	


;; moves wrappers from old to new if first oid of term same.
;; old could be isexpr and wa
(defun oid-zip-wrapped-terms (new old &optional op unwrap)

  ;;(setf a new b old) (break "ozwt")

  (let ((wrappers nil))
    (labels
	((wrapper-collect (term)
			  ;;(setf e term) (break "ozwtwc")
			  (if (wrapped-term-p term)
			      (if (or (eql 'dummy (kind-of-iwrapper-term term))
				      (eql unwrap (kind-of-iwrapper-term term)))
				  (wrapper-collect (term-of-iwrapper-term term))
				(let ((id (first-oid-of-term (term-of-wrapped-term term))))
				  (when id
				    (push (cons id term) wrappers))))))

	 (rewrap (n)
		 (let ((term n)
		       (oid (first-oid-of-term (term-of-wrapped-term n))))
		   
		   (dolist (w wrappers)
		     (if (equal-oids-p (car w) oid)
			 ;; do need unwrap here too, as unwrap kind may be inside a persistent kind.
			 (setf term (transfer-wrapper (cdr w) term unwrap))))
		   term)))	 

      
      ;; collect.
      (if (term-p old)
	  (map-isexpr old (or op (icons-op))  #'wrapper-collect)
	  (mapc #'wrapper-collect old))

      ;;(setf d wrappers) (break "zip")

      ;; rewrap, listify term input even if no wrappers.
      (if (term-p new)
	  (map-isexpr-to-list new (or op (icons-op))
			      (when wrappers #'rewrap))
	(if wrappers
	    (mapcar #'rewrap new)
	  new)))))



;;;;	
;;;;	
;;;;	
;;;;	

;;;
;;; filter
;;; 

(define-primitive |!dyneval| ((oid . oid) (bool . liveness)) (conditions expression stamp value))

;;;;	
;;;;	table_proxy - may occur as expression of dyneval 
;;;;	  stands for a hash table
;;;;      can compute to a filtered sorted list or array
;;;;	
;;;;	array_proxy - may occur as value of dyneval
;;;;	  stands for a long list.
;;;;	  can access subseq 

(define-primitive |table_proxy| ((oid . oid) (token . nmemonic)) (filter sort))
(define-primitive |array_proxy| ((oid . oid)) (filter sort))

(defun value-of-table-proxy (term)
  (let* ((oid (oid-of-table-proxy-term term))
	 (ptable (oproxy-table-find oid)))

    ;;(oproxy-table-touch ptable)  ; ???

    (array-proxy-term (oid-of-table-proxy-term term)
		      (filter-of-table-proxy-term term)
		      (sort-of-table-proxy-term term))))
    

(defun expr-of-dag (term)

  (cond
   ((idirectory-term-p term)
    (children-of-idirectory-term term))

   ((idirectory-root-term-p term)
    (children-of-idirectory-term term))

   ((idyneval-term-p term)
    (let ((expr (expression-of-idyneval-term term)))
      (cond
	((idag-filter-term-p expr)
	 (value-of-idyneval-term term))

       (t (raise-error (error-message '(dag expression not dyneval)))))))
    (t  (raise-error (error-message '(dag expression not))))))    




(defun dag-of-oid-term (term &optional (failp t))
  (let ((oid (first-oid-of-term term)))

    ;;(setf a term) (break "lfi")

    (if (null oid)
	(and failp (raise-error (error-message '(dag filter input not oid))))
	    
	(let ((td (lookup-term-def oid t)))
	  (if td
	      (expr-of-dag (term-of-term-def td))
	      (and failp (raise-error (error-message '(dag filter input not term))))
	      )))))


(define-primitive |!oid_stack| () (op tree))

(defun head-of-ioid-stack-term (term)
  (let ((op (operator-of-term (op-of-ioid-stack-term term))))
    (labels
	((aux (term cont)
	   (if (equal-operators-p op (operator-of-term term))
	       (let ((bts (bound-terms-of-term term)))
		 (if bts
		     (aux (term-of-bound-term (car bts))
			  (if (cdr bts)
			      (cons (cdr bts) cont)
			      cont))
		     (when cont
		       (let ((bts (car cont)))
			 (aux (term-of-bound-term (car bts))
			      (if (cdr bts)
				  (cons (cdr bts) (cdr cont))
				  (cdr cont)))))))
	       term)))
      (aux (tree-of-ioid-stack-term term) nil))))


(defun list-of-dag-addr (term) 

  (cond

    ;; global 
    ((idag-global-term-p term)
     (let ((acc nil))
       (definition-table-map (resource 'ostates)
	   (current-transaction-stamp)
	 #'(lambda (oid def)
	     (push (idag-child-term (or (name-property-of-ostate-def def) '||)
				    oid)
		   acc)))
       acc))

    ((idag-root-term-p term)
     (mapcar #'(lambda (root)
		 (idag-child-term (car root) (cdr root)))
	     (dag-roots)))


    ;; returns nil of no dag found, otherwise returns list.
    ((dag-of-oid-term term nil))))



(defun list-of-dag-filter-input (interm)

  (let ((term (term-of-wrapped-term interm)))

    (cond

      ((ioid-stack-term-p term)
       (let ((h (head-of-ioid-stack-term term)))
	 (if h
	     (or (list-of-dag-addr h)
		 (raise-error (error-message '(filter input dag stack addr not) term)))
	     (raise-error (error-message '(filter input dag stack null) term)))))

      ;; filter
      ((idag-filter-term-p term)
       (dag-filter term nil nil))

      ;; dyneval
      ((idyneval-term-p term)
       (let ((v (value-of-idyneval-term term)))
	 (list-of-dag-filter-input v)))

      ((list-of-dag-addr term))

      ;; literal
      (t term))))




(defun placeholder-filter-match (filter term)
  (placeholder-match (template-of-iplaceholder-filter-term filter)
		     term))


;; returns idag_cons list of idag-child or idag-child state.
(defun dag-filter (expr old &optional (ilist-p t))

  (let* ((l (list-of-dag-filter-input (input-of-idag-filter-term expr)))
	 (filter (filter-of-idag-filter-term expr))
	 (placeholder-filter-p (iplaceholder-filter-term-p filter))
	 )

    ;;(setf i l g filter) (break "df")
    (labels ((test (idag-child)
	       (let ((ostate (if (idag-child-state-term-p idag-child)
				 (state-of-idag-child-state-term idag-child)
				 (lookup-ostate (oid-of-idag-child-term idag-child)))))
		 (when (cond
			    
			 ((ivoid-term-p filter) t)

			 (placeholder-filter-p
			  ;;(setf o ostate j idag-child ) (break "pfm")
			  (placeholder-filter-match filter ostate)
			  )

			 (t nil))

		   (list (if (idag-child-state-term-p idag-child)
			     idag-child
			     (idag-child-state-term (name-of-idag-child-term idag-child)
						    (oid-of-idag-child-term idag-child)
						    ostate)))))))


      (let ((new (if (consp l)
		     (mapcan #'test l)
	    
		     (let ((acc nil))
		       (term-walk l
				  #'(lambda (c)
				      (when (or (idag-child-state-term-p c)
						(idag-child-term-p c))
					(when (test c)
					  (push c acc))

					t)))
		       (nreverse acc)))))

	(if ilist-p
	    (map-sexpr-to-ilist (oid-zip-wrapped-terms new old (idag-cons-op))
				(idag-nil-term))
	    new)))))



;;;;	end
;;;;	
;;;;	Terms may be viewed but some filter though not viewed need to be refreshed.
;;;;	Thus the local definition should have field for term. 
;;;;	Then at end processing, the def are refreshed then visible refreshed from changed defs.
;;;;	
;;;;	Edit term may differ from def, edit is maintained but all references are to
;;;;	def versions.
;;;;	
;;;;	
;;;;	Thus, at end have touch list.
;;;;	
;;;;	find subset which are TERMs. (touch-terms)
;;;;	find list of other TERMs. (other-terms)
;;;;	start with null list of untouched. 
;;;;	start with null list of dynamic.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Then move other to either touched or untouched, dynevaling along the way.
;;;;	
;;;;	Assume Push implicitly deletes from other and sets progress true
;;;;	
;;;;	Extend :
;;;;
;;;;	Bool progress
;;;;	
;;;;	progress <- false
;;;;	FOREACH t IN other
;;;;	 IF dynamicrefresh THEN (Push t dynamic)
;;;;	 ELSEIF (Intersectp touch dependencies(t))
;;;;	    THEN (Push t touch)
;;;;	 ELSEIF Not And (Intersectp other dependencies(t))
;;;;			(Intersectp dynamic dependencies(t))
;;;;	    THEN (Push t untouched)
;;;;	
;;;;	IF progress THEN repeat.
;;;;	
;;;;	other contains only objects depending on dynamic or each other
;;;;	touched contains all objects known touched or depending on touched.
;;;;	untouched contains all objects not depending on touched dynamic or other.
;;;;	
;;;;	dependencies of dynamic are incomplete, as oids may be computed.
;;;;	
;;;;	Refresh :
;;;;	
;;;;	FOREACH t IN dynamic
;;;;	  DynamicUpdate t
;;;;	  IF (Intersectp touch (dependencies-of t))
;;;;	     THEN Push oid touch
;;;;	     ELSE Push oid other
;;;;
;;;;	Do Extend
;;;;
;;;;	** extend should complete as dynamic is nil.
;;;;	** actually at end, other contains cycles not containing a touched oid.
;;;;
;;;;
;;;;	LookupTerm oid		** all calls by Eval should come through here.
;;;;	  if DynamicRefresh THEN DynamicUpdate oid
;;;;	  termof oid
;;;;
;;;;	DynamicUpdate (oid) =
;;;;	  WHEN not (CompleteStamp oid)
;;;;	    IF (PartialStamp oid)
;;;;	      THEN Fail '(DynEval Cycle)
;;;;	      ELSE LET dep, term = DynEval (term-of oid)
;;;;		     (dependencies-of oid) := Union dep (dependencies-of oid)
;;;;		     (term-of oid) := term
;;;;	
;;;;	
;;;;	  
;;;;	DynEval term : dependencies # term
;;;;   	  WithCollect 
;;;;	  StartStamp
;;;;	  WALK (term collectp) :
;;;;	    IF DynEval-p term
;;;;	       THEN MAKE DynEval (expression-of term)
;;;;			         WALK (DoEval term (\d. collect d)) t
;;;;	       ELSE WHEN (And Oidp collectp)
;;;;	             collect (oid-of term)
;;;;		    WALK
;;;;	  CompleteStamp	    
;;;;
;;;;	  RETURN collection, term  
;;;;
;;;;	
;;;;	** can no-op if determines that no change.
;;;;	** no change if all oids in args are untouched.
;;;;	DoEval term oid-collector
;;;;	  LET dep, term = DynEval (expression-of term) 
;;;;	     CALL oid-collector dep
;;;;	     Eval term		** Eval may do LookupTerm and cause dynamic refresh of dependent.
;;;;	
;;;;	


;;;;
;;;;	dynamic evaluation.
;;;;	
;;;;	
;;;;	

(defun dynamic-refresh-oid-p (oid)
  (let ((term-def (lookup-term-def oid t)))
    (and term-def
	 (or (directory-def-p term-def)
	     (property-of-term-def 'dynamicrefresh term-def) t))))

(defun edit-refresh-oid-p (oid)
  (let ((term-def (lookup-term-def oid t)))
    (and term-def
	 (or (directory-def-p term-def)
	     (property-of-term-def 'editrefresh term-def) t))))

(defun lookup-term-dependency (oid)
  (let ((term-def (lookup-term-def oid t)))
    (when term-def
      (dependency-of-definition term-def))))


;;(define-primitive |!condition| ((s . id)))
(define-primitive |!condition_cons| () (car cdr))

(defconstant *icondition* '|!condition|)

(defun icondition-term-p (term)
  (and (eql (id-of-term term) *icondition*)
       (let ((parms (parameters-of-term term)))
	 (and parms
	      (null (cdr parms))
	      (let ((p (car parms)))
		(and (real-parameter-p p)
		     (or (token-parameter-p p)
			 (string-parameter-p p))))))))

(defun id-of-icondition-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))

(defun icondition-term (cond)
  (let ((condition (if (meta-variable-id-p cond)
		      cond
		      (intern-system (string-upcase (string cond))))))

    (instantiate-term (instantiate-operator *icondition*
					    (list (token-parameter condition))))))

(defun token-of-icondition-term (term)
  (let ((cond (id-of-icondition-term term)))
    (if (stringp cond)
	(intern-system cond)
	cond)))

(defun icondition-sexpr-p (term)
    (or (icondition-cons-term-p term)
	(icondition-term-p term)
	(itext-term-p term)))
  
(defun icondition-sexpr-r (term)
  (map-isexpr term
	      (icondition-cons-op)
	      #'(lambda (term)
		  (unless (and (or (icondition-term-p term)
				   (itext-term-p term))
			       (real-itext-term-p term))
		    (raise-error (error-message '(syntax condition list not) term))))))


;; assumes term is !condition_cons list of generalized itext terms.
(defun term-to-condition-list (term)
  ;;(setf a term) (break "tcl")
  (map-isexpr-to-list term
		      (icondition-cons-op)
		      #'(lambda (term)
			  (if (or (itext-term-p term) (icondition-term-p term))
			      (intern-system (string-upcase
					     (string-of-itext-term-r term)))
			      (progn
				;;(setf a term) (break "tcl")
				(raise-error (error-message '(condition not) term)))))))



(defun walk-note-oid-dependencies (tag term)
  (when (noting-dependencies)
    (term-walk term
	       #'(lambda (term)
		   (dolist (p (parameters-of-term term))
		     (when (oid-parameter-p p)
		       (let ((oid (value-of-parameter p)))
			 (when (real-parameter-value-p oid (type-of-parameter p))
			   ;; static reference is not using contents thus objc and data stamps irrelevant.
			   (dependency-note-reference tag (dependency oid nil nil))))))
		   nil))
    nil))

(defun collect-term-oids (term avoid-p acc-f)
  (term-walk term
	     #'(lambda (term)
		 ;;(setf -term term -ap (funcall avoid-p term)) (break "ctot")
		 (if (and avoid-p (funcall avoid-p term))
		     t
		     (progn
		       ;;(setf -term term) (break "cto")
		       (dolist (p (parameters-of-term term))
			 (when (oid-parameter-p p)
			   (let ((oid (value-of-parameter p)))
			     ;;(setf -term term -oid oid -avoid-p avoid-p) (break "cto")
			     (when (real-parameter-value-p oid (type-of-parameter p))
			       (funcall acc-f oid)))))
		       nil))))
  nil)



(defun static-dependencies-of-term (term)
  (with-dependency-event ('(static))
    (with-dependency-environment
	(with-dependencies
	    ;;(setf -term term) (break "sdot")
	    (walk-note-oid-dependencies 'static term)))))


(defun static-dependencies-of-substance (substance)
  (not-without-dependencies
   (with-dependency-event ('(static))
     (with-dependency-environment
	 (with-dependencies
	     ;;(format t "static-dependencies-of-substance");; (setf -s substance) (break "sdos")
	     (walk-note-oid-dependencies 'static (term-of-substance substance))
	     (dolist (p (properties-of-substance substance))
	       (walk-note-oid-dependencies 'static (cdr p)))))

   ;;(setf -a substance) (break "sdos")
   (event-dependencies-collected (new-event-description *system* *version* *dependency-event-tags*)
				 (transaction-stamp)))))


(defun note-oid-dependencies (tag term)
  ;;(setf a2 tag b2 term) (break "nod")
  (dolist (p (parameters-of-term term))
    (when (oid-parameter-p p)
		  
      (let ((oid (value-of-parameter p)))
	(when (real-parameter-value-p oid (type-of-parameter p))

	  ;; would these be different?
	  (let ((td (lookup-term-def oid t)))
	    (if td
		(dependency-note-reference tag (dependency-of-definition td))
		(let ((dd (lookup-ostate-dependency oid)))
		  (when dd (dependency-note-reference tag dd))))))))))

    



(define-primitive |!scroll_list|
    ((oid . oid) (n . offset) (n . size) (n . point))
  (op input))

(defun offset-parameter-value-of-iscroll-list-term (term)
  (value-of-parameter-n (second (parameters-of-term term))))

(defun point-parameter-value-of-iscroll-list-term (term)
  (value-of-parameter-n (fourth (parameters-of-term term))))


;; length of input list, size of visible. May be less than requested if lenght - offset less.
(define-primitive |!scroll_position| ((n . length) (n . offset) (n . size) (n . point)) (list))

(defun list-of-scroll-input (term)
  (cond
    ((idyneval-term-p term)
     (value-of-idyneval-term term))
    (t term)))


(defun scroll-list (iscroll old)
  (let ((nil-term (op-of-iscroll-list-term iscroll)))

    (let ((l (oid-zip-wrapped-terms (list-of-scroll-input (term-of-condition-cut
							   (input-of-iscroll-list-term iscroll)))
				    (if (iscroll-position-term-p old)
					(list-of-iscroll-position-term old)
					nil)
				    (operator-of-term nil-term)
				    'scrollpoint)))
      ;;(setf a l b old c iscroll) (break "sl")
      (let ((ll (length l))
	    (sloffset (offset-of-iscroll-list-term iscroll))
	    (size  (size-of-iscroll-list-term iscroll))
	    (slpoint (point-of-iscroll-list-term iscroll))
	    )
	(unless (and (integerp sloffset)
		     (integerp size)
		     (integerp slpoint))
	  (raise-error
	   (error-message '(scroll list parameters bad)
			  (mapcar #'iparameter-term (parameters-of-term iscroll)))))

	(if (zerop ll)
	    
	    (iscroll-position-term 0 0 0 0 nil-term)
	    
	    (let ((point (max (min slpoint (1- ll)) 0)))
	      ;; ensure offset shows point.
	      (let ((offset (cond
			      ((< point sloffset) point)
			      ((>= point (+ sloffset size))
			       ;; make point second to last if possible.
			       (max 0 (- (+ 2 point) size)))
			      ((>= sloffset ll)
			       (1- ll))
			      (t sloffset))))
			       
		;;(setf -iscroll iscroll -offset offset) (break "sle")

		(values
		 (transfer-wrapper
		  old 
		  ;; ensure point in range.
		  (let ((end (min ll (+ offset size)))
			(i offset))

		    ;;(setf a ll b offset c size d end e point f sloffset g slpoint) (break "sl")
		    (iscroll-position-term ll offset (- end offset)
					   point
					   (map-sexpr-to-ilist
					    (mapcar #'(lambda (term)
							(prog1
							    (if (= point i)
								(iwrapper-term 'scrollpoint term)
								(wrapped-term 'dummy term '(pad3)))
							  (incf i)))
						    (subseq l offset end))
					    nil-term))
		    ))
		(when (or (not (and (eql point slpoint)
				    (eql offset sloffset)))
			  (not (icondition-cut-term-p (input-of-iscroll-list-term iscroll))))
		  
		  ;;(setf -input (input-of-iscroll-list-term iscroll))
		  ;;(break "sle")

		  (iscroll-list-term (oid-of-iscroll-list-term iscroll)
				     (transfer-parameter-marks
				      (offset-parameter-value-of-iscroll-list-term iscroll)
				      offset)
				     (size-of-iscroll-list-term iscroll)
				     (transfer-parameter-marks
				      (point-parameter-value-of-iscroll-list-term iscroll)
				      point)
				     (op-of-iscroll-list-term iscroll)
				     (icondition-cut-term ; adds if not
				      (big-term (term-of-condition-cut ;   there.
						 (input-of-iscroll-list-term iscroll)))))
		  )))))))))




(define-primitive |!ml_apply| () (e args))
(define-primitive |!MARK| ((tok . mark)) (term))

;;;;	
;;;;	result-of-evaluating expression is term.
;;;;	
;;;;	often called without mod because arg to some higher expression modified.
;;;;	
;;;;	
;;;;	so if not t
;;;;	
;;;;	
;;;;	

;; should call environment eval??
(defun dynamic-eval (expr prev-value)
  ;; todo need more general solution for wrappers like this.
  ;; ie source reduce !? seems overkill.
  (let ((expression (term-of-wrapped-term  expr)))
    ;;(setf -a expression -b prev-value) (break "de")
    ;;(if (compare-terms-p expression (term-of-wrapped-term prev-value))
    ;;expr)
    (with-handle-error-and-message (() #'(lambda (msg)
					       (apply #'ifail-term
						      (message-to-term
						       (tag-message *environment-path*
								    (tag-message '(eval) msg)))
						      (mapcar #'message-to-term (messages-flush)))))

	  (cond

	    ;; produce !dag_cons ilist.
	    ((idag-filter-term-p expression)
	     (dag-filter expression prev-value)
	     )

	    ;; produces !scroll_position.
	    ((iscroll-list-term-p expression)
	     (scroll-list expression prev-value))

	    ((iml-apply-term-p expression)
	     (ml-terms-to-term (e-of-iml-apply-term expression)
			       (cons prev-value
				     (map-isexpr-to-list (args-of-iml-apply-term expression) (icons-op)))))

	    (t (setf a expression) (break "de")
	       (raise-error (error-message '(dynamic-eval expression  unknown))))))))
    
	
;; strip live values
(defun dyneval-term-abbrev (cond term)
  (term-walk-d term
	       #'(lambda (term)
		   (and (idyneval-term-p term)
			(liveness-of-idyneval-term term)))
	       #'(lambda (term walk)
		   (let ((conditions (funcall walk (conditions-of-idyneval-term term))))
		     ;;(setf a cond b conditions c term d  (conditions-of-idyneval-term term)) (break "dta")
		     (if (isexpr-exists-p conditions
					  (icondition-cons-op)
					  #'(lambda (term)
					      ;;(setf g term h cond) (break "dta2")
					      (eql cond (token-of-icondition-term term))))
			 (idyneval-term (oid-of-idyneval-term term)
					t
					conditions
					(funcall walk (expression-of-idyneval-term term))
					(ivoid-term)
					(ivoid-term))
			 (idyneval-term (oid-of-idyneval-term term)
					t
					conditions
					(funcall walk (expression-of-idyneval-term term))
					(stamp-of-idyneval-term term)
					(funcall walk (value-of-idyneval-term term))))))) )


(defun save-dynamic-term (term)
  ;;(break)
  (dyneval-term-abbrev 'editrefresh (dyneval-term-abbrev 'editrefresh term)))
			
(defvar *dyneval-collectp* nil)
			
(defun dyneval-set (oid term)
  (let ((def (lookup-term-def oid t)))

    ;;(setf a def) (break "ldt")
    (when (and def (term-def-p def))
      (dyneval-set-term def term))))


(defun dyneval-set-and-update (dyncond oid term)
  (let ((def (lookup-term-def oid t)))

    ;;(setf a def) (break "ldt")
    (when (and def (term-def-p def))
      (dyneval-set-term def term)
      (dyneval-term-update dyncond
			   def
			   #'(lambda (term)
			       (declare (ignore term)) t))
      ;;(setf a def b term)
      (break "dsau")
      (term-of-term-def def))))


(defun dyneval-refresh (dyncond oid)
  (dyneval-synch-refresh dyncond oid))

;; should only recompute if args structureally modified.
(defun dyneval-synch-refresh (dyncond oid &optional weak)
  ;;(format t "dyneval-sync-refresh ~a ~a ~a ~%" weak dyncond (name-property-of-ostate oid))
  (let ((def (lookup-term-def oid t)))
    (when def
      (if (not (null (dyneval-of-term-def def)))
	  (progn
	    ;;(format t "dyneval-refresh ~a ~%" (name-property-of-ostate oid))
	    (dyneval-term-update dyncond
				 def
				 ;; if expression contains any touched term oids then
				 ;; should be evaled. Expression args have been reduced.
				 ;; however at this time we do not know which oids were touched.
				 ;; could have dyneval-touched remember touced oids.
				 ;; fttb refresh unconditionally.
				 #'(lambda (term old)
				     ;;(declare (ignore term old))
				     (if weak
					 (not (compare-terms-p term old))
					 t)
				     ;;(setf -term term -old old -p (not (compare-terms-p term old))) (break "dsr")
				     ;;(not (compare-terms-p term old))
				     )))
	
	  ;; at least lookup term-def term if dir.
	  (progn
	    ;;(break "dr")
	    (dyneval-call-hooks def))
	  ))))




;;; only sees static dependencies, but seems feasible to accept that limitation.
;;; ie that only static dependencies are tracked.
(defun dependencies-of-dyneval (tde)

  (let ((term (term-of-term-dyneval tde)))
    (when term

      (if (dyneval-flag-dependencies-p tde)
	  (term-dyneval-dependencies tde)

	  (with-dependency-event ('(static))
	    (with-dependency-environment
		(with-dependencies
		    ;; for dyneval terms could walk args but include deps from value
		    ;; collected at eval and skip val walk?
		    (term-walk (term-of-term-dyneval tde)
			       #'(lambda (term)
				   (note-oid-dependencies 'static term)))))

	    ;; rather than collecting deps should directly test. Or
	    ;; collect dependencies since at a later date we'll want to cache.
	
	    (let ((evtd (event-dependencies-collected
			 (new-event-description *system* *version* *dependency-event-tags*)
			 (current-transaction-stamp))))

	      (dyneval-flag-set-dependencies tde t)
	      (setf (term-dyneval-dependencies tde) evtd)))))))



;; kludge to prevent walking of large hidden terms looking for labels.
(define-primitive |!condition_cut| () (term))

(defun term-of-condition-cut (term)
  (if (icondition-cut-term-p term)
      (term-of-icondition-cut-term term)
      term))
    

;; if execution of a !dyneval lookups up an oid which is not
;; statically visible in call to predicate then not assurance
;; that value returned is fresh.
(defun dyneval-term (cond term old predicate)
  (let* ((cstamp  (current-transaction-stamp))
	 (cstamp-term (stamp-to-term cstamp))
	 )

    (labels ((walk-value (term walk)
	       term
	       ;;(let ((*dyneval-collectp* t)))
	       ;;(setf w2 term w3 walk)(break "hap")
	       ;;(funcall walk term nil)
	       ))
      
      ;;(setf -term term) (break "dt")

      (with-dependencies

	  (terms-walk-d
	   term
	   old
	   #'(lambda (term oldp)
	       (declare (ignore oldp))
	       ;;(when www (setf c2 term c3 collectp) (break "dtw"))
	       (idyneval-term-p term)
	       ;;(if (idyneval-term-p term))
	       ;;t
	       ;;(when *dyneval-collectp*)
		     ;; what about refs of touched oids other than term defs.
		     ;; need to know we refd, special ostate dep of (oid void void)
		     ;;(setf a term) (break "nod")
	       ;;(note-oid-dependencies 'dyneval term)
	       ;;nil 
	       )
	 
	   #'(lambda (term old walk)
	       (if (and (not (ivoid-term-p (stamp-of-idyneval-term term)))
			(in-transaction-p cstamp
					  (term-to-stamp (stamp-of-idyneval-term term))))
		   term

		   (let* ((old-expression (when old (expression-of-idyneval-term old)))
			  (expression (funcall walk
					      (expression-of-idyneval-term term)
					      old-expression))
			  (conditions (funcall walk (conditions-of-idyneval-term term) nil))
			  (live-p (liveness-of-idyneval-term term)))
		      
		     ;;(setf x expression y conditions  z live-p w term) (break "dt")
		     ;; 'always condition ?
		     (if (or (not live-p)
			     (not (isexpr-exists-p conditions
						   (icondition-cons-op)
						   #'(lambda (term)
						       (and (icondition-term-p term)
							    (eql cond (token-of-icondition-term term)))))))
				
			 (idyneval-term (oid-of-idyneval-term term)
					live-p
					conditions
					expression
					(stamp-of-idyneval-term term)
					(walk-value (value-of-idyneval-term term) walk))

			 (if (or (ivoid-term-p (value-of-idyneval-term term))
				 (funcall predicate expression old-expression))
			     (mlet* (((value expr) (dynamic-eval expression (value-of-idyneval-term term))))
				    ;;(setf a value b expr c (value-of-idyneval-term term) d expression) (break "de")
				    ;;(format t "doing dyneval update~%")
				    (idyneval-term (oid-of-idyneval-term term)
						   live-p
						   conditions
						   (or expr expression)
						   cstamp-term
						   (walk-value value walk)))

			     (idyneval-term (oid-of-idyneval-term term)
					    live-p
					    conditions
					    expression
					    cstamp-term
					    (walk-value (value-of-idyneval-term term) walk))))))))))))

#|(defun dyneval-term-old (cond term old predicate)
  (let* ((cstamp  (current-transaction-stamp))
	 (cstamp-term (stamp-to-term cstamp))
	 )

    (labels ((walk-value (term walk)
	       (let ((*dyneval-collectp* t))
		 ;;(setf w2 term w3 walk)(break "hap")
		 (funcall walk term))))
      
      ;;(setf -term term) (break "dt")

      (with-dependencies

	  (terms-walk-d
	   term
	   old
	   #'(lambda (term old)
	       ;;(when www (setf c2 term c3 collectp) (break "dtw"))
	       (if (idyneval-term-p term)
		   t
		   (when *dyneval-collectp*
		     ;; what about refs of touched oids other than term defs.
		     ;; need to know we refd, special ostate dep of (oid void void)
		     ;;(setf a term) (break "nod")
		     (note-oid-dependencies 'dyneval term)
		     nil )))
	 
	   #'(lambda (term walk)
	       (if (and (not (ivoid-term-p (stamp-of-idyneval-term term)))
			(in-transaction-p cstamp
					  (term-to-stamp (stamp-of-idyneval-term term))))
		   term

		   (let ((expression (funcall walk (expression-of-idyneval-term term)))
			 (conditions (funcall walk (conditions-of-idyneval-term term)))
			 (live-p (liveness-of-idyneval-term term)))
		      
		     ;;(setf x expression y conditions  z live-p w term) (break "dt")
		     ;; 'always condition ?
		     (if (or (not live-p)
			     (not (isexpr-exists-p conditions
						   (icondition-cons-op)
						   #'(lambda (term)
						       (eql cond (token-of-icondition-term term))))))
				
			 (idyneval-term (oid-of-idyneval-term term)
					live-p
					conditions
					expression
					(stamp-of-idyneval-term term)
					(walk-value (value-of-idyneval-term term) walk))

			 (if (or (ivoid-term-p (value-of-idyneval-term term))
				 (funcall predicate expression))
			     (mlet* (((value expr) (dynamic-eval expression (value-of-idyneval-term term))))
				    ;;(setf a value b expr c (value-of-idyneval-term term) d expression) (break "de")
				    (idyneval-term (oid-of-idyneval-term term)
						   live-p
						   conditions
						   (or expr expression)
						   cstamp-term
						   (walk-value value walk)))

			     (idyneval-term (oid-of-idyneval-term term)
					    live-p
					    conditions
					    expression
					    cstamp-term
					    (walk-value (value-of-idyneval-term term) walk))))))))))))|#



;; for debugging aid
(defun oids-of-tdef-dyn-dependencies (tdef)
  (let ((acc nil))
    (walk-event-dependencies
     #'(lambda (dep)
	 (let ((oid (oid-of-dependency dep)))
	   ;;(setf ah1 oid ah2 acc) (break "ah")
	   (unless (member oid acc :test #'equal-oids-p)
	     (push oid acc))))
     (term-dyneval-dependencies (dyneval-of-term-def tdef)))
    (mapcar #'string-of-oid acc)))



;; term-touched-f : t if any oid in term refers to touched def.
;; if returns t then forces refresh of container.
(defun dyneval-term-update (cond term-def term-touched-f)
  (let* ((tdyn (dyneval-of-term-def term-def))
	 )

    (if (term-dyneval-inprogress-p tdyn)
	(raise-error (error-message '(dyneval cycle) cond))
	(with-dependency-event ('(dyneval))
	  (with-dependency-environment
	      (with-dependencies
		  (let ((old-term (old-term-of-term-dyneval tdyn))
			(term nil))

		    (unwind-protect
			 (progn (dyneval-flag-set-inprogress tdyn t)
				(setf term (dyneval-term cond
							 (or (term-of-term-dyneval tdyn)
							     (term-of-term-def term-def))
							 old-term
							 term-touched-f
							 )))

		      (dyneval-flag-set-inprogress tdyn nil))

		    ;;(setf a cond b term c term-def e (dyneval-of-term-def term-def)) (break "dtu")

		    ;; deps here are dependencies gleaned from produced values.
		    ;; should include those touched.
		    ;;(setf -term term  -oldterm old-term -tdyn tdyn) (break "tdu")
		    (let ((cstamp (current-transaction-stamp)))
		      (setf (term-dyneval-term tdyn) term
			    (term-dyneval-dependencies tdyn) (event-dependencies-collected
							      (new-event-description *system* *version*
										     *dependency-event-tags*)
							      cstamp)	
			    (term-dyneval-stamp tdyn) cstamp))

		    (dyneval-call-hooks term-def)
		    
		    ;;(setf a cond b term c term-def
		    ;;e (dyneval-of-term-def term-def)
		    ;;f (term-dyneval-dependencies tdyn))
		    ;;(break "dtu2")

		    ;; force refresh if previously uninitialized.
		    (null old-term))))))
    ))



;; see comments near tree-listings end-hook
(defun dyneval-transaction-end-hook (th)
  ;;(setf a th) (break "dvteh")
  (when th
    ;;(with-local-transaction
	;; should this be done automatically by term table?
	(without-dependencies
	 (refresh-dynamic-terms
	  (remove-duplicates (mapcar #'oid-of-touch-record th)
			     :test #'equal-oids-p)))))
    ;;)

(defun refresh-dynamic-terms (touched)

  ;;(setf tt (mapcar #'string-of-oid touched))

  (let ((dmap (make-hash-table :test #'equal)) ; map form stamp to dependency to choose canonical dep.
	(dynamic nil)
	(other nil))

    (labels
	(
	 (mapd (d &optional add)
	   (if (dependency-p d)
	       (let ((dm (gethash (stamp-of-oid (oid-of-dependency d)) dmap)))
		 (or dm
		     (when add
		       (setf (gethash (stamp-of-oid (oid-of-dependency d)) dmap) d))))
	       (if (oid-p d)
		   (gethash (stamp-of-oid d) dmap)
		   (progn (break) nil))))

	 (memberp (d l)
	   ;;(oid-of-dependency d) l :key #'oid-of-dependency :test #'equal-oids-p
	   ;; since all d's in l and d itself run thru mapd eq should be ok.
	   (member d l :test #'eq))

	 (awaiting-refresh-p (d)
	   (and (memberp d dynamic)
		(let ((stamp (stamp-of-term-dyneval
			      (dyneval-of-term-def (mark-value d 'term-touch-definition)))))
		  (not (and stamp (in-transaction-p stamp (current-transaction-stamp)))))))
		

	 ;; find dynamics referenced in closure of d.
	 ;; if cycles found in other which do not refer to any dynamic
	 ;; then they are marked as untouched.
	 (other-closure (d)
	   (let ((acc nil)
		 (dacc nil))
	     (labels ((visit (dd)
			(walk-event-dependencies
			 #'(lambda (ddd)
			     (let ((dddd (mapd ddd nil)))
			       (when dddd
				 (unless (or (markp dddd 'term-touch)
					     (memberp dddd acc))
				   (if (awaiting-refresh-p dddd)
				       (unless (memberp dddd dacc)
					 (push dddd dacc))
				       (progn
					 (push dddd acc)
					 (visit dddd)))))))
			 (dependencies-of-term-def (mark-value dd 'term-touch-definition)))))
	       (visit d))

	     (if dacc
		 dacc
		 (progn
		   (dolist (dd acc)
		     (mark dd 'term-touch nil))
		   nil))))	     
	   
	 ;; determine if touched, untouched, or still don't know.
	 ;; returns nil if touched or untouched (list d) otherwise.
	 (dispose (d dyn-deps)
	   (let ((touched-p nil)
		 (untouched-p t))
			      
	     (labels
		 ((visit (dep)
		    (let ((dd (mapd dep nil)))
		      ;;(setf a d b dep c dd) (break "rtdtt")
		      (when dd
			(if (markp dd 'term-touch)
			    (when (mark-value dd 'term-touch)
			      (setf touched-p t
				    untouched-p nil))
			    (setf untouched-p nil))))))
	       
	       (walk-event-dependencies	#'visit
					(dependencies-of-term-def
					 (mark-value d 'term-touch-definition)))

	       (walk-event-dependencies #'visit dyn-deps))

	     ;;(setf a d b touched-p c untouched-p  f #'mapd  ) (break "rdtd")
	     (cond
	       (touched-p
		(mark d 'term-touch t)
		nil)
	       (untouched-p
		(mark d 'term-touch nil)
		nil)
	       (t (list d)))))

	 (dependency-of-parameter (p)
	   (when (oid-parameter-p p)
	     (let ((oid (value-of-parameter p)))
	       (when (real-parameter-value-p oid (type-of-parameter p))
		 (mapd oid)))))

	 (refresh (d)
	   ;;(setf a d) (break "pre-dtu")
	   ;; force dyneval if direct touch.
	   (let* ((direct-touch-p (mark-value d 'term-touch))
		  (def (mark-value d 'term-touch-definition))
		  (force-touch
		   (when (dyneval-term-update
			  'refresh
			  def
			  ;; touched-p : t if term contains touched oid.
			  ;; assume t contains oid in other, such oid is
			  ;; dir containing a yet to be refereshed dynamic.
			  ;; if an oid is in dynamic then implicitly refreshed.
			  ;; if an oid is in other then it must refer to a dynamic
			  ;; find dynamic and refresh.
			  #'(lambda (term old)
			      (declare (ignore old))
			      (let ((bit direct-touch-p))
				(term-walk term
					   #'(lambda (term)
					       ;; kludge alert : might be more palatable to wrap in
					       ;;  some !always-touched(0) op
					       (if (idag-global-term-p term)
						   (setf bit t)
						   (dolist (p (parameters-of-term term))
						     (let ((dd (dependency-of-parameter p)))
						       (when (and dd (awaiting-refresh-p dd))
							 (refresh dd)
							 (if (markp dd 'term-touch)
							     (when (mark-value dd 'term-touch)
							       (setf bit t))
							     (progn
							       ;; (in map and (not (touched or untouched))
							       ;; and (evaled if dynamic)) thus must be in other.
							       ;; if in other then must indirectly ref some dyn
							       ;; or some cycle of others.
							       (dolist (ddd (other-closure dd))
								 (refresh ddd))

							       ;; dd may still remain in other but if so
							       ;; must be due to a cycle of others. Thus if it is
							       ;; touched it must be marked as touched now.
							       (when (mark-value dd 'term-touch)
								 (setf bit t))))))))

					       ;; must visit each node to ensure all needed are refreshed.
					       nil))
				;;(setf h bit) (break "yadda")
				bit))))))
		   
	     
	     ;; been refreshed, refreshed value may still refer to other
	     ;;(setf m force-touch) (break "yadda yadda")

	     (if (cond (force-touch
			(mark d 'term-touch)
			t)
		       ((mark-value d 'term-touch)
			t)
		       (t
			(let ((deps (dyneval-dependencies-of-term-definition def)))
			  (dispose d deps))))
		 (setf other (mapcan #'(lambda (o) (dispose o nil)) other))
		 (push d other))))
	 )

	   
      (with-label-memory 'term-touch-definition
	(with-label-memory 'term-touch

	  (setf touched
		(mapcan #'(lambda (oid)
			    ;;(setf a oid ) (break "ah")
			    (let ((d (or (lookup-term-dependency oid) (lookup-ostate-dependency oid))))
			      (when d
				(mark d 'term-touch t)
				(list (mapd d t)))))
			touched))

	  ;; map over all term-defs
	  (definition-table-map (resource 'terms)
	      (current-transaction-stamp)
	    #'(lambda (oid def)
		(declare (ignore oid))
		(let ((d (mapd (dependency-of-definition def) t)))
		  (mark d 'term-touch-definition def)
		  (cond
		    ((dynamic-refresh-p def)
		     (dynamic-refresh-reset def)
		     (push d dynamic))
		
		    ((markp d 'term-touch)
		     ;;(setf a def b d)  (break "rdtmp")
		     nil)
		     
		    ((null-event-dependencies-p (dependencies-of-term-def def))
		     ;;(setf a def b d)  (break "rdtnp")
		     (mark d 'term-touch nil))

		    (t 	;;(setf a def b d)  (break "rdto")
		     (push d other))))))

	  ;; extend
	  ;;(setf d dynamic e other m dmap) (break "rdtbp");;
	  (let ((progressp t))

	    (do ()
		((not progressp))
	  
	      (setf progressp nil)
	      (setf other (mapcan #'(lambda (d)
				      (let ((dd (dispose d nil)))
					(or dd
					    (progn
					      (setf progressp t)
					      nil))))
				  other))
	      ;;(setf c progressp e other) (break "rdtp")
	      ))


	  ;;(setf b dynamic c (remembered-marks-of-label 'term-touch)) (break "rdta")

	  ;; refresh dynevals
	  (dolist (d dynamic)
	    (when (awaiting-refresh-p d)
	      (refresh d)))
	  
	  ;; update touched dirs
	  (let ((term-touched (mapcan #'(lambda (d)
					  ;;(setf a d b dynamic )
					  ;;(format t "wha ~a" (and (member a b) t)) (break "yo")
					  (when (mark-value d 'term-touch)
					    (list (oid-of-dependency d))))
				      (remembered-marks-of-label 'term-touch))))

	    (dolist (oid term-touched)
	      (let ((term-def (lookup-term-def oid t)))
		(when (and term-def (directory-def-p term-def))
		  (dyneval-call-hooks term-def))))
	    
	    term-touched ))))))



;;;;	
;;;;	Refres
;;;;	
;;;;	
;;;;	

;; return those of arena which are touched or statically reference a touched oid.


(defun event-dependencies-reference-oids-p (evtd inoids &optional assume-oids)
  ;; oids can be list of dependencies or list of oids.
  (let ((ufos nil))
    (let ((oids (if assume-oids
		    inoids
		    (mapcan #'(lambda (o)
				(let ((d (if (oid-p o)
					     o
					     (when (dependency-p o)
					       (oid-of-dependency o)))))
				  (if d
				      (list d)
				      ;; convenient for debug. cost irrelevant since should be nil if
				      ;; doesn't need debuggin.
				      (push o ufos))))
			    inoids))))
      
      (walk-event-dependencies
       #'(lambda (dd)
	   (when (member (oid-of-dependency dd) oids :test #'equal-oids-p)
	     (return-from event-dependencies-reference-oids-p t)))
       evtd)))

  nil)



;; touched is list of oids
;; arena is oid term pair
(defun static-references (touched arena)

  (let* ((ufos nil)
	 (toids (mapcan #'(lambda (o)
				(let ((d (if (oid-p o)
					     o
					     (when (dependency-p o)
					       (oid-of-dependency o)))))
				  (if d
				      (list d)
				      ;; convenient for debug. cost irrelevant since should be nil if
				      ;; doesn't need debuggin.
				      (push o ufos))))
			    touched)))

  (let ((acc nil))

    (dolist (oid-term arena)
      (with-dependency-event ('(static))
	  (with-dependency-environment
	      (with-dependencies
		  ;; for dyneval terms could walk args but include deps from value
		  ;; collected at eval and skip val walk?
		  (term-walk (cdr oid-term)
			     #'(lambda (term)
				 (note-oid-dependencies 'static term)))))

	;; rather than collecting deps should directly test. Or
	;; collect dependencies since at a later date we'll want to cache.
	
	(let ((evtd (event-dependencies-collected
		     (new-event-description *system* *version* *dependency-event-tags*)
		     (current-transaction-stamp))))

	  (when (event-dependencies-reference-oids-p evtd toids t)
	    (push (car oid-term) acc)))))

    acc)))


;; only finds single path, ie if  (b c d) is path and (a c d) is also
;; only gets one.
(defun oid-find-paths (oid)
  (let ((count 0))
    (let ((oidhash (make-hash-table :test #'equal)))
      (labels ((visit (diroid a)
		 (incf count)
		 (when (> count 10000) (break "ofp"))
		 (unless (gethash (stamp-of-oid diroid) oidhash)
		   (setf (gethash (stamp-of-oid diroid) oidhash) t)
		   (let ((children (dag-directory-children diroid)))
		     (dotimeslist (i child children)
				  (if (equal-oids-p (cdr child) oid)
				      (progn
					;;(setf -a a -i i -child child) (break "found")
					(return-from visit (cons i a)))
				      (when (dag-directory-p (cdr child))
					(let ((r (visit (cdr child) (cons (car child) a))))
					  (when r
					    ;;(setf -r r) (break "hello")
					    (return-from visit r))))))
		     nil))))
       
	(without-dependencies
	 (mapcan #'(lambda (root) (let ((a (visit (cdr root) (list (car root)))))
				    (when a
				      (list
				       (cons (car a)
					     (reverse (cdr a)))))))
		 (dag-roots))))))
  )

     



