
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	
;;;;	Eval for evaluating lisp toploop input.
;;;;	

;;;; -docs- (mod top ml)
;;;;	
;;;;	TopLoop :
;;;;
;;;;	Toploop commands :
;;;;	  b. : enter lisp debugger.
;;;;	  x. : exit top loop.
;;;;	  p. : toggle process-break mode. When t enters debugger when error thrown.
;;;;	  e. : toggle eval echo mode.
;;;;	  r. : toggle io-report mode.
;;;;	  t. : close/open trace file. If open closes, if closed opens.
;;;;	  m. : switch to ml mode. 
;;;;	  l. : switch to lisp mode.
;;;;
;;;;	EG: 
;;;;	
;;;;	ML[(ORB)]> t.
;;;;	
;;;;	io journal file: ~/spool/trace-io1.jnl.
;;;;	
;;;;	ML[(ORB)]> t.
;;;;	
;;;;	closed io journal file: ~/spool/trace-io1.jnl.
;;;;	
;;;;	ML[(ORB)]> p.
;;;;	Process break is now T.
;;;;	ML[(ORB)]> p.
;;;;	Process break is now NIL.
;;;;	
;;;;
;;;;  -page-
;;;;
;;;;	ML eval:
;;;;	  * input which does not begin with a backslash and ends with
;;;;	    a period and contains no whitespace, will be interpreted as
;;;;	    input to the toploop, all other input will be evaluated.
;;;;
;;;;	FTTB: all input must be on a single line. Of course it may be a very long line.
;;;;	If the top loop appears to be stuck usually ;;<return> will clear. In worst
;;;;	case (c-c) to lisp debugger, :a to abort then (top) to restart.
;;;;	
;;;;
;;;;	Useful ML commands:
;;;;
;;;;	orb_start_accept	: int { <port> } -> tok {`mathbus`} -> unit
;;;;	mosa			: int { <port> }
;;;;	  : starts accepting connections on port.
;;;;	
;;;;	orb_local_environments	: unit -> tok list list
;;;;	  * lists the mnemonic tags for loaded library instances in this process.
;;;;
;;;;	orb_bus_environments	: unit -> tok list list
;;;;	  * lists the mnemonic tags for connected components.
;;;;	
;;;;	Useful LISP commands :
;;;;	
;;;;	(orb-cleanup)		: wipes all loaded libraries and all bus connections.
;;;;	(orb-cleanup nil)	: wipes all bus connections.
;;;;	(top)			: starts top-loop.
;;;;
;;;;	(orb-cleanup) is useful as a last ditch effort to recover the process
;;;;	and avoid killing and restarting lisp. You need to restart the accept
;;;;	with mosa <port> after cleanup. Sometimes even then the port is still
;;;;	corrupted. In that case, you can use a different port or give it up
;;;;	and kill lisp.
;;;;	
;;;;
;;;;  -page-
;;;;
;;;;	top
;;;;	  * input which does not begin with a backslash and ends with
;;;;	    a period and contains no whitespace, will be interpreted as
;;;;	    input to the toploop rather than input to be evaluated.
;;;;	
;;;;	  * while in the toploop expressions can be evaluated locally 
;;;;	    or remotely. This is controlled by changing the component
;;;;	    mode of the toploop. All evaluations in the toploop are
;;;;	    routed to a component matching the mode.  Eg, you connect to
;;;;	    lib as ref, you start top loop and goto ref mode.   Any
;;;;	    input to be evaluated will be wrapped as a request and sent
;;;;	    to the connected ref.  The mode is reflected in the prompt
;;;;	    unless the mode matches the type of the local interpreter.
;;;;	    The following toploop cmds control the mode:
;;;;	      - lib.  
;;;;	      - ref.
;;;;	      - edd.
;;;;	      - com.
;;;;	      - quit. : sets interpreter type to local interpreter.
;;;;
;;;;	  * while in the toploop expressions can be evaluated in lisp evaluator
;;;;	    or ml evaluator. m. or l. switches evaluators.
;;;;	
;;;;	
;;;;	  * To exit the top loop:
;;;;	      - exit.
;;;;
;;;;
;;;;	When the top-loop is shipping expressions to remote components to be
;;;;	evaluated the expr is simply a string to the toploop. IE, you can
;;;;	type CAML or SML expressions at the top loop if they are destined
;;;;	for a component of that type.
;;;;
;;;;  -page-
;;;;	
;;;;	Caveats :
;;;;	  - the responses are wrapped in a lot of crap you
;;;;	    eventually will not see.
;;;;	      * You can use something like the following to control how much is printed:
;;;;		(setf *max-print-depth* 20)
;;;;	  - somtimes the ML> prompt does not appear initially.
;;;;	    Entering ";;<return> " should get you the prompt.
;;;;	  - sometimes you get two ML> prompts in sequence. Not a problem but
;;;;	    it looks funny.
;;;;	  - Emacs & ILISP & lisp: ie running lisp in emacs buffer using ilisp.
;;;;	      * ilisp tries to interpret [] brackets without
;;;;	        passing througth as data.
;;;;	      * compiling lisp forms with ilisp while at ML prompt will hang
;;;;	        lisp buffer. Only solution I've found is to kill buffer and
;;;;		start over. Instead, break with b. first and then compile.
;;;;			
;;;; -doce-
;;;;	<remotes>	: ((<server-type> . <env-address>) . <resource-name> list) list



;;;;	RLE PERF :  Either figure how to lock pages in memory in order to
;;;;	RLE PERF :  lock critical io code or send small keep-alive messages
;;;;	RLE PERF :  at periodic (sleep quantum) intervals to keep code from
;;;;	RLE PERF :  being paged out. Like second idea with a switch to turn off.



;;;;	stdin channel : for toploop at listener prompt.
;;;;	
;;;;	output side of channel dumps to stdout.

;;;;	toploop :
;;;;	  add toploop channel to bus.
;;;;	  print emitted msgs. and prompts.
;;;;
;;;;	toploop dynamic vars:
;;;;	toploop-address-parameters
;;;;	 destination address tags as parameters
;;;;	interp-type
;;;;	
;;;;	
;;;;	whitespace or null input should re-prompt.
;;;;	

(defun console-get-line (tab &optional (blockp t))
  (when tab
    (princ tab))
  
    (let ((line (read-line t blockp)))
    (let ((l (length line)))
      (unless (or (zerop l)
		  (and (= 1 (length line))
		       (break-char-p (char line 0))))
	line))))



(defun console-get-lines (&optional blockp)
  (let ((tab (blank-string (toploop-prompt-length)))
	(ml-text nil))
    
    (do* ((line (console-get-line nil blockp)
		(console-get-line (when ml-text tab)))
	  )
	 ((when (not (listen))
	    ;;(setf a line b ml-text) (break "ngl2")
	    (or (and (null ml-text) (null line))
		(and line
		     ;; index of last non-whitespace character.
		     (let ((index (do ((i (1- (length line)) (1- i)))
				      ((or (< i 0)
					   (not (whitespace-char-p (char line i))))
				       i))))
		       (and (> index 0)
			    (or (when (and (char= #\; (char line index))
					   (char= #\; (char line (1- index))))
				  ;;(setf c ml-text)
				  (push (subseq line 0 (1- index)) ml-text)
				  ;;(setf d ml-text)(break "ngl2")
				  t)
				(when (and (null (cdr ml-text))
					   (char= #\. (char line index))
					   (not (char= #\\ (char line (1- index))))
					   (let ((p t))
					     (dotimes (i (1- index))
					       (when (whitespace-char-p (char line i))
						 (setf p t)))
					     p))
				  (setf ml-text
					(intern-system (subseq line 0 index)))
				  t)))))))
			 
	  (if (symbolp ml-text)
	      ml-text
	      (mapcar #'(lambda (item)
			  (if (stringp item)
			      item
			      ;; messes up ml scanner. should handle itab ireturn ok though.
			      ;;(string-to-standard-character-string item)
			      item))
		      (nreverse ml-text))))

      ;;(setf a ml-text) (break "nmt1")
      (when line
	(setf ml-text (if (null ml-text)
			  (list line)
			  (list* #\newline line ml-text))))
      ;;(setf a ml-text) (break "nmt")

      (when (null ml-text)
	(toploop-prompt)
	))))


(defvar *toploop-address-parameters*)
(defvar *toploop-addresses*)
(defvar *toploop-interp-type*)
(defvar *toploop-local-type*)
(defvar *toploop-prompt-length*)
(defvar *toploop-link*)
(defvar *toploop-eval-type*)  ;; 'lisp or 'ml
(defvar *toploop-inter-orb-p* nil)

;; assume not in server loop when called, elsewise will not be able to call (unless called
;; remotely.

(defun toploop (init-type)
  (insys
   (let* ((*toploop-addresses* (nconc (mapcar #'address-of-environment
					      *component*)
				      (mapcan #'(lambda (blink)
						  (mapcar #'address-of-environment
							  (imported-environments-of-bus-link blink)))
					      *bus-links*))
				      )

	  (*toploop-address-parameters* (mapcar #'(lambda (addr)
						    (address-parameters-of-environment
						     (find-any-environment addr)))
						*toploop-addresses*))

	  (*toploop-link* (new-bus-link (new-toploop-link)))

	  (*toploop-interp-type* (list init-type))
	  (*toploop-eval-type* 'ml)
	  (*toploop-local-type* init-type)
	  (*toploop-prompt-length* 0))

     ;; add toploop channel to bus.
     (add-bus-link *toploop-link*)
          
     (toploop-prompt)

     ;; orb loop.
     (unwind-protect
	  ;; if not looping aleady start loop, but this fails start toploop if looping already??
	  (when *request-loop-quit-p*
	    (orb-request-loop))

       (when *toploop-link*
	 (toploop-remove-bus-link))))))


(defun toploop-remove-bus-link ()
  (when *toploop-link*
    (remove-bus-link *toploop-link*)
    (setf *toploop-link* nil)))


(defun toploop-set-eval-type (type)
  (setf *toploop-eval-type* type))

;; type is parsed into tokens with #\. separators.
(defun toploop-set-interp-type (type)
  (when (eql '|lib| type)
    (lib-stats))
  
   (setf *toploop-addresses* (nconc (mapcar #'address-of-environment
					    *component*)
				    (mapcan #'(lambda (blink)
						(mapcar #'address-of-environment
							(imported-environments-of-bus-link blink)))
					    *bus-links*)))

   (setf *toploop-address-parameters* (mapcar #'(lambda (addr)
						  (address-parameters-of-environment
						   (find-any-environment addr)))
					      *toploop-addresses*))
   (let ((types nil)
	 (acc nil)
	 (s (string type))
	 (escapep nil))
     (dotimes (i (length s))
       (cond
	 ((and (not escapep) (eql (char s i) #\.))
	  (let ((type (make-string (length acc))))
	    (dotimeslist (i c (nreverse acc))
			 (setf (char type i) c))
	    (setf acc nil)
	    (push (intern-system type) types)
	    (setf escapep nil)))
	 ((and (not escapep) (eql (char s i) #\\))
	  (setf escapep t))
	 (t (push (char s i) acc)
	    (setf escapep nil))))
     (when acc 
       (let ((type (make-string (length acc))))
	 (dotimeslist (i c (nreverse acc))
		      (setf (char type i) c))
	 (push (intern-system type) types)))

     (if types
	 (if (not (find-first #'(lambda (address parameters)
				  (when (forall-p #'(lambda (type) (member type address))
						  types)
				    parameters))
			      *toploop-addresses*
			      *toploop-address-parameters*))
	     (message-emit (warn-message '(toploop address not) types))
	     (setf *toploop-interp-type* types))
	 (setf *toploop-interp-type* (list *toploop-local-type*)))))

;; () -> string
(defun toploop-prompt ()
  (let ((s (format-string (if *toploop-inter-orb-p*
			      "~%~a]~a[> "
			      "~%~a[~a]> ")
			  *toploop-eval-type*
			  (case *toploop-interp-type*
			    (library "LIB")
			    (edit "EDD")
			    (refiner "REF")
			    (otherwise *toploop-interp-type*)))))
    (setf *toploop-prompt-length* (1- (length s))) ; 1- for newline char.
    (format t s)))

(defun toploop-address-parameters ()
  (or  (find-first #'(lambda (address parameters)
		       (when (forall-p #'(lambda (type) (member type address))
				       *toploop-interp-type*)
			 parameters))
		   *toploop-addresses*
		   *toploop-address-parameters*)
       (raise-error (error-message '(toploop address not) *toploop-interp-type*))))


(defun toploop-prompt-length ()
  *toploop-prompt-length*)




;;;; toploop-link


(defun toploop-channel-send (channel term)
  (declare (ignore channel))

  (unless (and (irsp-term-p term) (iack-term-p (result-of-iresult-term term)))
    (message-emit (degenerate-term-message term)))

  ;;(setf a term) (break "tcs")
  (when (and (ireq-aux-term-p term)
	     (iconfigure-term-p (expression-of-ireq-term term))
	     (irequest-term-p (command-of-iconfigure-term (expression-of-ireq-term term)))
	     (inoack-term-p (info-of-irequest-term (command-of-iconfigure-term (expression-of-ireq-term term)))))
    (toploop-channel-recv-queue-push channel (irsp-term (sequence-of-ireq-term term) (iack-term)))
    (setf a term) (break))
  
  (when (messages-p)
    (terpri)
    ;;(break "toploop mess")
    (mapcar #'print-message (messages-flush)))

  ;; reprompt.
  (toploop-prompt)
  )

;; \\( -> \(
(defun top-loop-munge-input (input)
  ;;(setf a input) (break "tlmi")
  (mapcar #'(lambda (s)
	      (if (not (stringp s))
		  s
		  (let ((l (length s))
			(found 0))
		    (dotimes (i (- l 2))
		      (when (and (eql #\\ (char s i))
				 (eql #\\ (char s (1+ i)))
				 (eql #\( (char s (+ i 2))))
			(incf found)))
		    (if (zerop found)
			s
			(let ((news (make-string (- l found))))
			  (let ((j 0)
				(limit (- l 2)))
		
			    (dotimes (i l)
			      ;;(setf a limit b j c i d l e s)
			      (cond
				((>= i limit)
				 (setf (char news j) (char s i))
				 (incf j))
				((not (and (eql #\\ (char s i))
					   (eql #\\ (char s (1+ i)))
					   (eql #\( (char s (+ i 2)))))
				 (setf (char news j) (char s i))
				 (incf j))
				(t nil))))
			  news)))))
	  input))

(defun go-win ()
  (format t "go win")
  (toploop-set-interp-type `|edd|)
  (runprl *toploop-interp-type*))

;; may have to return !noop if input is for toploop itself.
(defun toploop-channel-recv (channel &optional blockp)
  
  (or (toploop-channel-recv-queue-pop channel)

      (let ((input (console-get-lines blockp)))

	;;(setf a input) (break "tcr")
	(cond
	  ((null input)
	   (new-ireq-term (current-sequence) nil
			  (car *toploop-interp-type*)
			  (toploop-address-parameters)
			  (inoop-term)))
	  ((symbolp input)
	   (case input
	     ((|m| |ml| ml)	(toploop-set-eval-type 'ml))
	     ((|l| |lisp| lisp)	(toploop-set-eval-type 'lisp))
	     ((|o| |orb| orb)	(toploop-set-interp-type 'ORB))
	     ((|b| |break| break)	(break))
	     ((|p| |P|)	(progn (setf *process-break* (not *process-break*))
			       (format t "Process break is now ~a." (if *process-break* "on" "off"))))
	     ((|r| |R|)	(progn (io-echo-report-toggle)
			       (format t "IO reporting is now ~a." (if *io-echo-report-p* "on" "off"))))
	     ((|e| |E|)	(progn (eval-print-toggle)
			       (format t "Eval echo is now ~a." (if *eval-print-p* "on" "off"))))
	     ((|t| |T|)	(if (io-trace-file-p)
			    (close-io-trace-file)
			    (open-io-trace-file "trace" t)))
	     ((|g| |G|)	(setf *toploop-inter-orb-p* (not *toploop-inter-orb-p*)))
	     ;;(lib	(toploop-set-interp-type 'library))
	     ;;(ref	(toploop-set-interp-type 'refiner))
	     ;;(edd	(toploop-set-interp-type 'edit))
	     ((|q| |quit| quit)	(toploop-set-interp-type 'ORB))
	     ((|w| |where| where)	(message-emit (basic-message (toploop-address-parameters))))
	     ((|win|) (go-win))
	     ((|go|) (go-go))
	     ((|start|) (go-go))
	     ((|stop|) (stop))
	     ((|x| |exit| exit)	(progn
				  (quit-request-loop)
				  (toploop-remove-bus-link)))
	       
	     (otherwise	(toploop-set-interp-type input)))
	   (new-ireq-term (current-sequence) nil
			  (car *toploop-interp-type*)
			  (toploop-address-parameters)
			  (inoop-term)))
	
	  (t
	   ;; kludge alert : converts #\\#\\#\( to #\\#\( need #\\#\\#\( to prevent ilisp getting confused
	   ;; about paren balance.
	   (advance-sequence)
	   ;;(format t "~% toploop-channel-recv 2 ~%")

	   (new-ireq-term (current-sequence) nil
			  (car *toploop-interp-type*)
			  (toploop-address-parameters)
			  (case *toploop-eval-type*
			    (ml (iexpression-term (iml-woargs-term nil t
								   (setf a (text-to-term
									    (top-loop-munge-input input))))
						  *toploop-inter-orb-p*))
			    (lisp (iexpression-term (ilisp-term t (text-to-term input))
						    *toploop-inter-orb-p*))
			    (otherwise (raise-error '(toploop eval-type unknown))))))))))
  

(defun new-toploop-link ()
  (format t "~% new-toploop-link ")
   (new-soft-link
   (new-toploop-channel)
   :send #'toploop-channel-send
   :recv #'toploop-channel-recv
   :listen #'(lambda (ch)
	       (declare (ignore ch))
	       (let ((val (listen t)))
		 ;;(mapc #'print-message (messages-flush))
		 ;;(format t "~% inside listen: ~s" val)
		 val))))



;;;;	lib startup :
;;;;	
;;;;	(orb-toploop)
;;;;	orb ML> start_accept <port>;; 
;;;;
;;;; then drive from edit.

;; default toploop:
(defun orb-toploop (&optional start)
  (setf *toploop-inter-orb-p* nil)
  (when start (funcall start))
  (toploop 'orb)
  )

(defun top (&optional start)
  (let ((utime (get-universal-time)))
    (format t "~%~%    CURRENT TIME : ~a.~%~%~%" (datetime-string utime))
    (update-process-activity-log "top" utime))
  (orb-init)
  (orb-toploop start))

(defun cl-user::top ()
  (in-system-package)
  (funcall (intern "TOP" cl-user:*system-package-name*))
  )

(defun count-tree (pairp tree)
  (labels ((count-aux (l)
	     (if (funcall pairp l)
		 (+ 1 (count-aux (cdr l)) (count-aux (car l)))
		 0)))
    
    (count-aux tree)))
  
(defun global-env-counts (&optional (max 2000))
  (let ((i 0))
    (dolist (e *global-env*)
      (incf i)
      (let ((c (count-tree #'consp e)))
	(when (> c max)
	  (format t "~a ~a~%" (car e) c))))
    i))

(defun package-counts-aux (package max)
  (labels ((mylength-aux (l)
	     (if (consp l)
		 (+ 1 (mylength-aux (cdr l)) (mylength-aux (car l)))
		 0))
	   (mylength (s)
	     (let ((ll (mylength-aux (symbol-value s))))
	       (when (> ll max)
		 (format t "MaxL ~a ~a~%" s ll))
	       ll)))
    
    
  (let ((funcs 0)
	(constants 0)
	(other 0)
	(nonnull 0)
	(conspp 0)
	(unbound 0)
	(i 0)
	(plist 0)
	)
    (do-symbols (s package)
      (when (eql package (symbol-package s))
	(incf i)
	(incf plist (length (symbol-plist s)))
	(unless
	    (or (when (fboundp s)
		  (incf funcs)
		  t)
		(when (boundp s)
		  (if (constantp s)
		      (incf constants)
		      (progn
			(incf other)
			(when (not (null (symbol-value s)))
			  (if (consp (symbol-value s))
			      (progn (setf -s s)
				     (incf conspp (mylength s)))
			      (incf nonnull)
			      ))))
		  t))
	  (incf unbound))))

    (format t "~%Total ~a~%Constants ~a~%PList ~a~%Other ~a~%NonNull ~a~%Cons ~a~%Funcs ~a~%Unbound ~a~%"
	    i constants plist other nonnull conspp funcs unbound))))

(defun system-package-counts (&optional (max 2000))
  (package-counts-aux *system-package* max))
(defun system-ml-package-counts (&optional (max 2000))
  (package-counts-aux *ml-runtime-package* max))

(defun clear-system-package-symbol-properties ()
  (let ((npackage *system-package*))
    (format t "~%---~%")
    (room nil)
    (let ((i 0)
	  (j 0))
      (do-symbols (s npackage)
	(when (and (eq (symbol-package s) npackage)
		   (not (fboundp s))
		   (boundp s))
	  (let ((plist (symbol-plist s)))
	    (when plist
	      (incf i)
	      (incf j (length plist))
	      (setf (symbol-plist s) nil)
	      (format t "symbol> ~a ~a <~%" (symbol-name s) (length plist))))))
      (format t "~%--- ~a ~a~%" i j)
      (room nil)
      )))

;; try to narrow down variables holding onto memory.
(defun clear-system-package-looking-for-space-aux (npackage max)
  (let ((i 0))
    (do-symbols (s npackage)
      (when (and (eq (symbol-package s) npackage)
		 ;;(not (fboundp s))
		 (or (and (boundp s) (symbol-value s))
		     (symbol-plist s)))

	(format t "~%symbol> ~a <~%" (symbol-name s))
	(unintern s)
	(unless (constantp s)
	  (setf (symbol-value s) nil)
	  (setf (symbol-plist s) nil)
	  )

	(incf i)
	(when (= 100 i)
	  (setf i 0)
	  (gc t)
	  (room nil)
	  (package-counts-aux npackage max))))))

(defun clear-system-package-looking-for-space ()
  (clear-system-package-looking-for-space-aux *system-package* 2000))

(defun clear-system-ml-package-looking-for-space ()
  (clear-system-package-looking-for-space-aux *ml-runtime-package* 2000))


