
% merged simp and new-simp removing duplicate defs %



%[ *************************************************************
**************** Simp "interface" ******************************

Everything below is of type tactic conv.



Creating simps.  A "body" of a simp formula is a maximal subterm which
is not all, and, or implies.  Bodies that are not instance of equiv relns
(is_eq_rel_term) are taken to be implicit eq_rel_terms, according to the
following sequence of rewrites.
  not(assert(P))  -->  P=bfalse in bool
  assert(P)       -->  P=btrue in bool
  not(P)          -->  P <=> False
  P               -->  P <=> True      if P not an eq_rel_term
Left hand sides of iff rewrites will additionally match terms which
match after swapping equands.  

The two main simp creation functions are the following.

  gen_lemma_simp name n l_to_r_p cond_tacs
  gen_hyp_simp i n l_to_r_p condition_tacs p

Clause n of the lemma/hyp is used, oriented left-to-right iff
l_to_r_p.  If cond_tacs is empty, then the lemma is unconditional --
conditions will become subgoals.  Otherwise, and if the lemma has
conditions, it is a condition rewrite using cond_tacs to check/prove
the conditions (the list is padded/truncated as necessary).  In the
following, "lemma" can be replaced by "rev_lemma".  
  lemma_simps name                                  -- [DefaultConditionSimp], left-to
                                                       right, all clauses.
  lemma_simp name
  lemma_clause_simp name n                          -- use only clause n

  dc_simp index t1 t2 c1 c2                         -- same args as MacroC
  simple_dc_simp index t1 t2 abstr_names            -- t1 is t2 after unfolding
  unfold_top_simp opid                              -- UnfoldTopC opid
  unfold_top_if_simp opid pred                      -- UnfoldTopC opid if pred
  pattern_unfold_op_simp index opid patterns        -- if term matches a pattern,
                                                       unfold opid in pattern occurrence

  hyp_simp i p                                      -- guesses orientation, filters out
                                                       possibly "insane" simps

  generic_simp index pred C                         -- apply C whenever pred is true;
                                                       use sparingly + w. extreme caution

  bogus_simp index                                  -- a failing simp, for cancelling
                                                       a global simp


Adding simps.
  
  simp_adds simps               --  add simps to the global simpset.
  simp_add                      
  simpset_adds name simps       --  add simps to named simpset.
  simpset_add


Simping using the global simpset.

  SSome clauses   --   Simp the specified clauses.
  S               --   Simp everything.
  DCS             --   Everything, but using only "direct computation" simps.


Simping using the global simpset plus the hypotheses.  An equality
hypothesis is oriented left-to-right unless it is of the form t=x where
t is not a variable.  A hypothesis assert(e) is treated as e=true in bool, 
and not(assert(e)) as e=false in bool.  Simping is from the largest numbered
hyp to the smallest, finishing with the conclusion (if it's in clauses in the 
case of HSSome).  When each clause i is simped, all hypotheses except i are
used;  if some of the other hyps have already been simped, the simped version
is used.  (The single "S" means simp globally.)

  HSSome clauses
  HS


Simping using just simpsets.

  SimpsetSome names clauses     --  apply exhaustively
  Simpset names
  Simpset1Some names clauses    --  apply in a single top-down pass
  Simpset1 names

Simping using simpsets together with global simps and hypotheses.

  SSet name
  SSetSome name clauses
  HSSet names
  HSSetSome names clauses


Using individual lemmas as simps (no global simps).  Lemmas with
conditions are turned into conditional simps.

  LemmaSimps names     --  turn lemmas into simps, apply exhaustively
  RevLemmaSimps names 
  LemmaSimp name
  RevLemmaSimp name
  LemmaSimp1s names    --  apply in a single top-down pass
  RevLemmaSimp1s names
  LemmaSimp1 name
  RevLemmaSimp1 name


Misc.

  reset_simp  --  clear all simp tables and recheck all ML objects.
                  Currently this may be necessary if WSimps is aborted.
  site_of_simp_add id
              --  the library object in which the simp id is added to
                  the global simplifier.
  sites_of_simpset_add id


To do: 

1.  HS should use contextual assumptions as well, e.g. use A when
rewriting B i.  These are already used for condition discharge, but
not to rewrite with.


***************************************************************]%


%************************************************************%
%                     Some general stuff                     %
%************************************************************%

% FDL Port defined differently in Nuprl5 tactics
let TopC = HigherC ;;
%


let num_simple_clauses fmla =
  length (process_general_formula fmla)
;;

let num_simple_clauses_of_lemma name =
  num_simple_clauses (main_goal_of_theorem name)
;;

let fo_matches pat inst =
  (fo_match (free_vars pat) pat inst ; true) ? false
;;

let matches pat inst =
  (match (free_vars pat) pat inst ; true) ? false
;;

let mst = mk_simple_term ;;

letrec addressed_subterm t l =
  if null l then t else addressed_subterm (subterm t (hd l)) (tl l)
;;

let third ((),(),z) = z ;;
let fourth ((),(),(),z) = z ;;

let ws = string_to_words ;;

let wmap f s = map f (ws s) ;;

let head_opid t = 
  opid_of_term (is_apply_term t => head_of_application t | t)
;;

let it_mk_apply = mk_iterated_apply ;;
let it_dest_apply = dest_iterated_apply ;;

let curried_arity t = length (it_dest_apply t) - 1 ;;

let dest_tlambda t = 
  let (op,[]),[[],A;[x],b] = dest_term t in
  if op = `tlambda` then (x,A),b
  else failwith `dest_tlambda`
;;

let it_dest_tlambda = unreduce dest_tlambda ;;

let is_tlambda_term t = (dest_tlambda t; true) ? false ;;

let mk_tlambda_term (x,A) b =
  mk_term (`tlambda`,[]) [[],A; [x],b]
;;

let it_mk_tlambda binders t = reduce mk_tlambda_term t binders ;;

let appending_results f l = reduce append [] (map f l) ;;

letrec first_suffix P l =
  if P l then l
  if null l then failwith `first_suffix`
  else first_suffix P (tl l)
;;

let following_member x l =
  hd (tl (first_suffix (\l. x = hd l) l))
  ? failwith `following_member`
;;

let succeeds f x = (f x; true) ? false ;;

letrec mapc f l =
  if null l then ()
  else (f (hd l); mapc f (tl l))
;;

letrec collect_successes f l =
  if null l then []
  else f (hd l) . collect_successes f (tl l) ? collect_successes f (tl l)
;;

let normalize_clause_number i p =
  if i<0 then num_hyps p + i + 1 else i
;;

let normalize_clause_numbers l p =
 map (C normalize_clause_number p) l
;;


let replace_var x y t =
  let xv = mvt x and yv = mvt y in
  let g vs = map (\z. if z=x then y else z) vs in
  letrec f t = 
    if alpha_equal_terms t xv then yv else
    if is_var_term t then t
    else let op,bterms = dest_term t in
         mk_term op (map (\vs,u. g vs, f u) bterms)  in
  f t
;;

let Last (T: int->tactic) p = T (num_hyps p) p ;;

let Lasts i T = RepeatMFor i (Last T) ;;

%let NormC C e t = RepeatC (TopC C) e t ;;%
let NormC C e t = RepeatC (SweepDnC (RepeatC C)) e t ;;

% simp2's version
let with_cleanup f x c = (let res = f x in c(); res) ?\id (c(); failwith id);;
%

let with_cleanup f x c = apply_and_cleanup f x c ;;

let hyp_id i p = 
  if i=0 then failwith `hyp_id: argument 0.`
  else int_to_tok (normalize_clause_number i p)
;;

% begin - FDL-port new-simp only  %
let True i p =
  if clause_type i p = mst `true` [] then
    (if i=0 then Trivial else Thin i) p
  else failwith `True`
;;

let is_bound_at_clause p i var =
  let i = if i=0 then num_hyps p + 1 else normalize_clause_number i p in
  member var (map var_of_declaration (firstn (i-1) (hyps p)))
;;

let is_closed_at_clause p i t =
  all (is_bound_at_clause p i) (free_vars t)
;;

let tracing string f x = 
  display_message ("Entering " J string) ;
  let res = f x in
  display_message ("Exiting " J string) ;
  res
;;

letrec map_on_nth f l n = 
  if null l then l 
  if n=1 then f (hd l) . tl l
  else hd l . map_on_nth f l (n-1)
;;

let env_of_mapped_sequent f p =
  decls_to_env
    (map (\i,x,A,b. let x',A' = f i x A in mk_declaration (x',A',b))
         (zip (upto 1 (num_hyps p))
              (map dest_declaration (hyps p))))
;;

% end - FDL-port new-simp only  %


%***********************************************************************%
%           Interface to the simp tables; simp state variables          %
%***********************************************************************%




% A simp has fields: (id, op, pred, C).
  id:     a unique identifier for the simplification.  
  o,pred: the simp is only attempted on a given term t in environment
          e if both of the following hold:
            1) op = ` `  (ie. one blank)
               or the opid of t is op
               or t is an iterated application term whose head has opid op.  
            2) pred e t = true.
  C:      a conversion.

  We maintain two kinds of tables for simps.  A hash table for simp lookup
  via op, and an ML alist for lookup via id.  The former is for rewriting,
  the latter for adding and deleting simps.
%


lettype simp =   tok #                    % opid %
                 object_id #              % op %
                 (env -> term -> bool) #  % pred %
                 convn                    % C %
;;


letref simp_trail = [] : object_id list ;;
letref simp_count = 0 ;;
letref simp_limit = 100 ;;

letref display_simp_trail_p = false ;;
letref beta_simp_disabled_p = false ;;

% The two simp tables are lifted from Lisp. %
letref DefaultSimpCondition = Trivial ;;
letref disabled_simps = [] : tok list ;;
letref dc_simps_only_p = false ;;
simp_trail := [] ;;  
simp_count := 0 ;;
simp_limit := 100 ;;
display_simp_trail_p := true ;;
letref beta_simp_disabled_p = false ;;
letref display_redef_warning = false ;;

letref simpsets = []: (tok # (simp # tok) list) list ;;

% Functions lifted from Lisp. %
%FDL PORT re
let op_to_simps_table_lookup: tok -> (simp list) = op_to_simps_table_lookup' ;;
let op_to_simps_table_replace: tok -> (simp list) -> unit = op_to_simps_table_replace' ;;
let id_to_simp_table_lookup: tok -> simp = id_to_simp_table_lookup' ;;
let id_to_simp_table_replace: tok -> simp -> unit = id_to_simp_table_replace' ;;
let list_of_id_to_simp_table: unit -> (simp list) = list_of_id_to_simp_table' ;;
let list_of_op_to_simps_table: unit -> (simp list) = list_of_op_to_simps_table' ;;
%
% lifted directly: %
%FDL PORT re
clear_simp_id_to_location_table:   unit->unit ;;
simp_id_to_location_table_lookup:  tok->tok ;; 
simp_id_to_location_table_replace: tok->tok->unit ;;
simp_id_to_location_table_delete:  tok->unit ;;
list_of_simp_id_to_location_table: unit->(tok list) ;;

let id_to_simp = id_to_simp_table_lookup  ;;
let op_to_simps = op_to_simps_table_lookup ;;

let list_simps = list_of_id_to_simp_table ;;

let simp_exists_p = succeeds id_to_simp ;;
%
% Never fails %
%FDL PORT re
let simp_delete s =
 (let (id,op,p) = s in
  let simps = op_to_simps_table_lookup op in
  op_to_simps_table_replace 
   op
   (remove_if (\id',v. id=id') simps) ;
  id_to_simp_table_delete id ;
  simp_id_to_location_table_delete id
 ) ? ()
;;
%

%FDL PORT re
let simp_add s = 
  let (id,op,p) = s in
  if simp_exists_p id
    then (if not display_redef_warning then () else
          display_message ("Warning: simp " J (tok_to_string id) J " being redefined.")
         ;simp_delete (id_to_simp id)
         )
    else () ;
  let simps = op_to_simps_table_lookup op in
  op_to_simps_table_replace op (s.simps) ;
  id_to_simp_table_replace id s ;
  ()
  (simp_id_to_location_table_replace id (ml_object_being_checked ())
   ? ()) 
;;
%

%FDL PORT one-simp merge%

%FDL PORT re
let add_simp = simp_add;;

let simp_adds ss = map simp_add ss ;;


let clear_simps () =
  clear_simp_id_to_location_table () ;
  clear_op_to_simps_table () ;
  clear_id_to_simp_table ()
;;
%

%
let reset_simp () =
  clear_simps();
  recheck_all_ml_obs()
;;
%


%************************************************************%
%                 Simp formulas.                             %
%************************************************************%


% Simp formula conventions.  A body of a simp formula is a maximal 
  subterm which is not all, and, or implies.  Bodies that are not
  rel_terms (is_rel_term) are taken to be implicit rel_terms, according 
  to the following sequence of rewrites:
  not(assert(P))  -->  P=bfalse in bool
  assert(P)       -->  P=btrue in bool
  not P           -->  P <=> False
  P               -->  P <=> True
%

let simple_formula_concl formula =
  let (),(),concl = dest_simple_formula formula in
  concl
;;

let nth_simple_formula_concl formula n =
  fourth (hd (nth_simple_formula n formula))
;;

let number_of_formula_clauses fmla =
  length (process_general_formula fmla)
;;

let number_of_lemma_clauses name =
  number_of_formula_clauses (main_goal_of_theorem name)
;;
let number_of_lemma_clauses_o obid =
  number_of_formula_clauses (statement_lookup obid)
;;

let is_eq_rel_term t =
( let r,t1,t2 = dest_rel_term_without_check t 
  in
  0 = fst (identify_rel r)
) ? false % is_rel_term moronically prints an error message when false! %
;;

let simp_ize_atomic_formula c =
  if is_eq_rel_term c then c
  if is_term `assert` c then mk_equal_term (mst `bool` []) 
                               (subterm c 1) (mst `btrue` [])
  if not is_not_term c then mk_iff_term c (mst `true` [])
  else let t = dest_not c in
       if is_term `assert` t then mk_equal_term (mst `bool` [])
                                    (subterm t 1) (mst `bfalse` [])
       else mk_iff_term t (mst `false` [])
;;


let simp_formula_relnands formula =
  snd (dest_rel_term_without_check (simp_ize_atomic_formula (simple_formula_concl formula)))
;;

% dir=true iff the formula is to be oriented left-to-right.
%
let simp_formula_lhs formula dir =
  let t1,t2 = simp_formula_relnands formula in
  if dir then t1 else t2
;;


% Fails if not a simp formula %
let simp_formula_opid formula dir =
  opid_for_simp (simp_formula_lhs formula dir)
;;


% Note: we don't check types during rewriting. %
let maybe_insane_simp_formula fmla n l_to_r_p =
  let (), ctxt, assums, concl  =  hd (nth_simple_formula n fmla) in 
  let t = simp_formula_lhs concl l_to_r_p in
  is_var_term t & member (dest_var t) (map fst ctxt)
  & all (\assum. not member (dest_var t) (free_vars assum)) assums
;;

let maybe_l_to_r_simp fmla n =
  let t = nth_simple_formula_concl fmla n in
  (let t1,t2 = simp_formula_relnands t in
   not (is_var_term t2 & not is_var_term t1)
  ) ? true
;;


% A formula is like a simple formula except that it may
  contain &'s.  %
letrec map_on_formula_concls g f =
 if is_term `all` f or is_term `implies` f then 
   apply_to_nth_immediate_subterm (map_on_formula_concls g) 2 f
 if is_term `rev_implies` f then
   apply_to_nth_immediate_subterm (map_on_formula_concls g) 1 f
 if is_term `guard` f then
   apply_to_nth_immediate_subterm g 1 f
 if is_term `and` f then
   map_on_immediate_subterms (map_on_formula_concls g) f
 else g f
;;

% Make implicit bodies into explicit ones (see comment above).%
let simp_ize_formula f =
  map_on_formula_concls simp_ize_atomic_formula f
;;

%************************************************************%
%           Creating atomic simps.                           %
%************************************************************%


%********** Conversions *************************************%

let UnfoldInPatternC opid patterns e t =
  let pattern = find (C matches t) patterns in
  AllC (map (C AddrC (UnfoldTopC opid) o snd)
            (rev (find_subterms_with_addrs (\(),u. is_term opid u) pattern)))
       e t
;;

% Patch up the atomic simp formula in the last hyp according to
  simp_ize_atomic_formula.
%
let SimpIzeAtomicLast p =
  let i = num_hyps p in
  let c = h i p in
  let T name =  FLemma name [i] 
                ORELSE FailWith `SimpIzeAtomicLast: must have theory hol loaded`  in
  (if is_eq_rel_term c then Id
   if is_term `assert` c then T `assert_imp_eq_btrue`
   if not is_not_term c then 
      Assert (mk_iff_term c (mst `true` []))
      THENL [D 0 THENM D 0 THENM Trivial
            ;Thin i
            ]
   if is_term `assert` (dest_not c) then T `assert_imp_eq_bfalse`
   else Assert (mk_iff_term (dest_not c) (mst `false` []))
        THENL [D 0 THENM D 0 THENM Trivial
              ;Thin i
              ]
  ) p
;;


%FDL port : added this def. Need to make sure appropriate. %
let swap_equands e =
  let T,a,b = dest_equal e in
   (mk_equal_term T b a)
;;

let SwapEquandsC (e:env) t =
  swap_equands t, iff_reln, form_tactic_just (D 0 THENM D 0 
                                              THENM SwapEquands 0 THENM Trivial)
;;


% A modification of GenLemmaThenLC to regard non-rel_term conclusions P as
  P <=> True etc. %
let GenLemmaSimpC lemma_name dir clause_num condition_tacs  =
  let body = simp_ize_formula (main_goal_of_theorem lemma_name) in
  let lhs = simp_formula_lhs body dir in
  let Conv =
      GenFormulaCondC 
        lemma_name 
        body
        (\bs subs. 
           InstGenLemmaByAddr (lemma_lookup lemma_name) bs subs 
           THENM SimpIzeAtomicLast)
        clause_num
        (if dir then `LR` else `RL`)
        `tactics`
        dummy_conv_enabler
        condition_tacs
        []  in
  if is_equal_term lhs then Conv ORELSEC (SwapEquandsC ANDTHENC Conv)
  else Conv
;;

let GenLemmaSimpC_o obid dir clause_num condition_tacs  =
  let body = simp_ize_formula (statement_lookup obid) in
  let lhs = simp_formula_lhs body dir in
  let Conv =
      GenFormulaCondC (name_of_lemma obid)
        body
        (\bs subs. 
           InstGenLemmaByAddr obid bs subs 
           THENM SimpIzeAtomicLast)
        clause_num
        (if dir then `LR` else `RL`)
        `tactics`
        dummy_conv_enabler
        condition_tacs
        []  in
  if is_equal_term lhs then Conv ORELSEC (SwapEquandsC ANDTHENC Conv)
  else Conv
;;

let LemmaSimpC lemma_obid condition_tacs  =
  GenLemmaSimpC_o lemma_obid true (-1) condition_tacs 
;;

let RevLemmaSimpC lemma_obid condition_tacs  =
  GenLemmaSimpC_o lemma_obid false (-1) condition_tacs 
;;

let GenHypSimpC i n l_to_r_p condition_tacs p =
  let i = normalize_clause_number i p in
  let fmla = simp_ize_formula (h i p) in
  let (), (), (), concl  =  hd (nth_simple_formula n fmla) in
  let lhs = simp_formula_lhs concl l_to_r_p in
  let Conv =
      GenFormulaCondC 
        (`hyp_conv` ^ int_to_tok i)
        fmla
        (\bs subs. InstGenHypByAddr i bs subs THENM SimpIzeAtomicLast)
        n
        (if l_to_r_p then `LR` else `RL`)
        `tactics`
        dummy_conv_enabler
        condition_tacs
        []   in
  if is_equal_term lhs then Conv ORELSEC (SwapEquandsC ANDTHENC Conv)
  else Conv
;;

let HypSimpC i condition_tacs  =
  GenHypSimpC i (-1) true condition_tacs 
;;

let RevHypSimpC i condition_tacs  =
  GenHypSimpC i (-1) false condition_tacs 
;;



%**************************** DC simps ***********************************%

let dc_simp id t1 t2 c1 c2  = 
 let ob = obid_for_simp t1 in
  ob, (\e t. true), MacroC id c1 t1 c2 t2
;;

let simple_dc_simp id t1 t2 abstr_names  = 
 let ob = obid_for_simp t1 in
  ob, (\e t. true), SimpleMacroC id t1 t2 abstr_names
;;

let unfold_top_simp op =
  (\e t. true), UnfoldTopC op
;;

let unfold_top_if_simp op pred =
 (\e t. pred t), UnfoldTopC op
;;

let pattern_unfold_op op patterns =
  let pred e t = exists (C matches t) patterns in
    pred, UnfoldInPatternC op patterns
;;

let pattern_unfold_simp_o (ob,patterns) = ob,(pattern_unfold_op (opid_of_obid ob) patterns);;

%
let dc_simp id t1 t2 c1 c2  = 
  id, opid_for_simp t1, (\e t. true), MacroC id c1 t1 c2 t2
;;

let simple_dc_simp id t1 t2 abstr_names : simp =
  id, opid_for_simp t1, (\e t. true), SimpleMacroC id t1 t2 abstr_names
;;

let unfold_top_simp opid : simp =
  opid, opid, (\e t. true), UnfoldTopC opid
;;

let unfold_top_if_simp opid pred : simp =
  opid, opid, (\e t. pred t), UnfoldTopC opid
;;

let pattern_unfold_op index opid patterns : simp =
  let pred e t = exists (C matches t) patterns in
  index, opid, pred, UnfoldInPatternC opid patterns
;;
%


% Ooops... %

let pattern_unfold_op_simp = pattern_unfold_op ;;
 

%**************************** Misc simps ***********************************%

% For cancelling existing simps. %

%FDL PORT re
;; hmmm, bogus opid wont match anything so only shadows simps
;; looked up by name which is not used by refinement?
;; so does this actually affect anything.
let bogus_simp id : simp = 
  id, `aBoGusOPPIDEEEE`, (\x y. false), FailC
;;

FDL PORT re : See varible proxy in simp-re for generic simp .
let generic_simp id pred C = (id, ` `, pred, C) ;;
%


%**************************** Lemma simps ***********************************%

let simp_formula_op_ob formula dir =
  op_ob_for_simp (simp_formula_lhs formula dir)
;;

let simp_formula_obid formula dir =
  obid_for_simp (simp_formula_lhs formula dir)
;;


let gen_lemma_simp name n l_to_r_p cond_tacs : simp = 
  let n = if n<1 then -1 else n in
  let (), context, assums, concl  =  hd (nth_simple_formula n
					 (main_goal_of_theorem name)) in
  %let simp_id = if n = -1 then name 
                else name ^ `_` ^ int_to_tok n in%
  let op,ob = simp_formula_op_ob concl l_to_r_p in
  (op, ob,
   (\e t. %not (member simp_id disabled_simps)
          &% not dc_simps_only_p),
   GenLemmaSimpC name l_to_r_p n cond_tacs    
  )
  ? failwith `Lemma `^name^`does not have the right form for a simp.`
;;

let gen_lemma_simp_o obid n l_to_r_p cond_tacs = 
  let n = if n<1 then -1 else n in
  let (), context, assums, concl  =  hd (nth_simple_formula n
					 (statement_lookup obid)) in
  let ob = simp_formula_obid concl l_to_r_p in
  (ob,
   (\e t. %not (member simp_id disabled_simps)
          &% not dc_simps_only_p),
   GenLemmaSimpC_o obid l_to_r_p n cond_tacs    
  )
  ? failwith ( `Lemma `^(name_of_lemma obid)
             ^ `does not have the right form for a simp.`)
;;


%begin FDL port from simp %
% For unconditional lemmas %
let lemma_to_simp lemma_name =
  let p =  main_goal_of_theorem lemma_name in
  (lemma_name,
   simp_formula_opid p true,
   (\e t. true),
   LemmaC lemma_name)
  ? failwith `Lemma `^lemma_name^`does not have the right form for a simp.`
;;

%FDL PORT re
let add_lemma_simp id = simp_add (lemma_to_simp id) ;;
%

%end FDL port from simp %



let lemma_clause_simp name n =
  gen_lemma_simp name n true [DefaultSimpCondition]
;;

let rev_lemma_clause_simp name n =
  gen_lemma_simp name n false[DefaultSimpCondition]
;;

let lemma_simps name = 
  let n = number_of_lemma_clauses name in
  let clause_nums = if n=1 then [-1] else upto 1 n in
  map (\i. gen_lemma_simp name i true [DefaultSimpCondition])
      clause_nums
;;

let lemma_simps_o obid = 
  let n = number_of_lemma_clauses_o obid in
  let clause_nums = if n=1 then [-1] else upto 1 n in
  map (\i. gen_lemma_simp_o obid i true [DefaultSimpCondition])
      clause_nums
;;

let lemma_simp name = 
  gen_lemma_simp name (-1) true [DefaultSimpCondition]
;;

let rev_lemma_simps name = 
  let n = number_of_lemma_clauses name in
  let clause_nums = if n=1 then [-1] else upto 1 n in
  map (\i. gen_lemma_simp name i false [DefaultSimpCondition])
      clause_nums
;;

let rev_lemma_simp name = 
  gen_lemma_simp_o name (-1) false [DefaultSimpCondition]
;;

let lemma_cond_simp lemma_obid condition_tacs =
  let condition_tacs = if null condition_tacs then [DefaultSimpCondition]
                       else condition_tacs in
  gen_lemma_simp_o lemma_obid (-1) true condition_tacs
;;

let rev_lemma_cond_simp lemma_obid condition_tacs =
  let condition_tacs = if null condition_tacs then [DefaultSimpCondition]
                       else condition_tacs in
  gen_lemma_simp_o lemma_obid (-1) false condition_tacs
;;


%**************************** Hyp simps ***********************************%

let gen_hyp_simp i n l_to_r_p condition_tacs p =
 let i = normalize_clause_number i p in
 let n = if number_of_formula_clauses (h i p) = 1 then -1 else n in
 let op,ob = simp_formula_op_ob (simp_ize_atomic_formula
			        (nth_simple_formula_concl (h i p) n)) 
                              l_to_r_p ? (` `, dummy_object_id ())  in
 %let index = n=-1 => `hyp_` ^ hyp_id i p 
                  |  `hyp_` ^ hyp_id i p ^ `_` ^ int_to_tok n  in%
 (hyp_id i p
 ,op
 ,ob
 ,(\e t. %not (member index disabled_simps)% true)
 ,GenHypSimpC i n l_to_r_p condition_tacs p
 )
;;

let hyp_simps i p =
  collect_successes (\n. gen_hyp_simp i n true [DefaultSimpCondition] p)
                    (upto 1 (number_of_formula_clauses (h i p)))
;;
  
let hyp_simp i p = 
  let l = hyp_simps i p in
  if length l = 1 then hd l
  else failwith `hyp_simp: more than one possible simp in hyp `^int_to_tok i
;;


% l1 is a list of l->r simps, l2 a list of r->l. %
let some_hyp_simps l1 l2 p =
  let simper b i = 
      collect_successes (\n. gen_hyp_simp i n b [DefaultSimpCondition] p)
                        (upto 1 (number_of_formula_clauses (h i p)))  in
  appending_results (simper true) l1 @ appending_results (simper false) l2
  ? failwith `some_hyp_simps: some specified hypothesis is not a simp formula.`
;;


% Almost certainly will be a bad simp. %
let undesirable_hyp_for_simp p i =
  not is_invisible_var (var_of_hyp i p)
;;

let potential_hyp_simps i p =
  if undesirable_hyp_for_simp p i then [] else
  collect_successes
   (\n. if maybe_insane_simp_formula (h i p) n true then fail
        else  gen_hyp_simp i n true [DefaultSimpCondition] p)
   (upto 1 (number_of_formula_clauses (h i p)))
;;


let SimpleCondCAux dir lemma_name =
  (if dir = `LR` then LemmaThenLC else RevLemmaThenLC)
  [Trivial]
  lemma_name
;;

let SimpleCondC lemma_name =
   SimpleCondCAux `LR` lemma_name
;;

let RevSimpleCondC lemma_name =
   SimpleCondCAux `RL` lemma_name
;;



%***************************************************************************%
%                    Adding/disabling simps temporarily                     %
%***************************************************************************%

%FDL PORT re

let WSimps simps (T:tactic) p =
  let clobbered_simps = collect_successes (id_to_simp o fst) simps in
  map simp_add simps ;
  with_cleanup T p (\(). map simp_delete simps; map simp_add clobbered_simps)
;;


let WSimpsC simps (C:convn) (e:env) (t:term) =
  let clobbered_simps = collect_successes (id_to_simp o fst) simps in
  map simp_add simps ;
  with_cleanup (C e) t (\(). map simp_delete simps; map simp_add clobbered_simps)
;;


let WOSimps simps T p =
  map simp_delete simps ;
  with_cleanup T p (\(). map simp_add simps)
;;

--port re- just use WOSimps?
let DisablingSimps l (T:tactic) p =
  disabled_simps := l ;
  let res = T p in
  disabled_simps := [] ;
  res
;;

let DisablingSimpsC l (C:convn) e t =
  disabled_simps := l ;
  let res = C e t in
  disabled_simps := [] ;
  res
;;
%

%***************************************************************************%
%                     The simplifier.                                       %
%***************************************************************************%


%*******************  Supporting conversions *******************************%

% f identity up to alpha conversion.  t ==> f t. %
let AlphaC f e t =
  let t' = f t in
  if alpha_equal_terms t t' then (let t,r,j = IdC e t in t',r,j)
  else fail
;;


let is_tbeta_redex t =
  is_apply_term t & is_tlambda_term (subterm_of_term t 1)
;;

let is_beta_redex t = 
  is_apply_term t & is_lambda_term (subterm_of_term t 1)
;;

% For simulating Nuprl's variable-name propagation in second-order substitution. %
let rename_tbeta_redex t =
  if not is_tbeta_redex t then fail
  else let f,arg = dest_apply t in
       (let (g,()),() = dest_tlambda f in
        let (y,()),() = dest_tlambda arg in
        let g_ap = find_subterm (\ (),t. is_apply_term t & 
                                        alpha_equal_terms (mvt g) (fst (dest_apply t)))
                                t  in
        let x = dest_var (snd (dest_apply g_ap)) in
	let res = mk_apply_term (replace_var x y f) arg in
        if alpha_equal_terms res t then res else fail
       ) ? t
;;


let TBetaC e t = 
  if not is_tbeta_redex t then fail
  else (NthSubC 1 (UnfoldTopC `tlambda`) ANDTHENC BetaC) e t
;;

let RenamingTBetaC = 
  AlphaC rename_tbeta_redex ANDTHENC TBetaC
;;


% Uses tlambda_evalC, defined in theory fun_1. %
let AnyBetaC e t  = 
  if beta_simp_disabled_p then fail
  if not is_apply_term t  then fail
  if is_beta_redex t then BetaC e t
  if is_tbeta_redex t then RenamingTBetaC e t
  else fail
;;

letref trace_RestrRewrite = false ;;
letref RestrRewrite_trail = [] : (tok # convn #  int # proof) list ;;
let init_RestrRewrite_trail = RestrRewrite_trail := [] ;;
let add_to_RestrRewrite_trail label C i p = 
  if trace_RestrRewrite then
     (RestrRewrite_trail := (label,C,i,p).RestrRewrite_trail; ())
  else ()
;;

% Conditional rewrites on a hyp can't use that hyp in checking
  conditions.  Never fails.  May move rewritten hyp to end (only
  when doesn't declare a used var).
%
let RestrRewrite C i p =
  add_to_RestrRewrite_trail `entering` C i p ;
  let i = normalize_clause_number i p in
  let e = env_of_mapped_sequent 
            (\i' x A. if i'=i & is_invisible_var x then x, mst `true` [] 
                      else x, A)
            p in
  let res = Rewrite (\e' t. C e t) i p ? Id p in
  add_to_RestrRewrite_trail `exiting` C i p ;
  res
;;
   
%********************* The basic simplifier *********************%

%FDL PORT re
let applicable_simps e term : simp list =
  filter (\(),(),pred,(). pred e term)
         (op_to_simps (opid_for_simp term) @ op_to_simps ` `)
;;
%

%let applicable_simps e term : convn list =
  let op = opid_for_simp term  in
  map fourth (op_to_simps op)
  @ 
  map fourth (filter (\(),(),pred,(). pred e term)
            	     (op_to_simps ` `))
;;%


let simp_count_and_cap () =
  simp_count := simp_count + 1 ;
  if simp_count > simp_limit then (display_message "Error: simp_limit exceeded." ;
                                   failwith `SimpC: simp_limit exceeded.`)
  else ()
;;

let init_simp_tracking () =
  simp_count := 0 ;
  simp_trail := []
;;


%BATCH begin following depends on simp-re %


let lemma_to_simp_o lemma_obid =
 ( let p =  statement_lookup lemma_obid in
   simp_formula_obid p true,
   (\e t. true),
   LemmaC_o lemma_obid)
  ? failwith `Lemma does not have the right form for a simp.`
;;

let add_lemma_simp_o staticp uobid obids =
 simp_add_o_aux staticp uobid (map lemma_to_simp_o obids)
;;


let add_simple_dc_simp_o iobid args =
 simp_add_o iobid 
   (map  (\id,t1,t2,absnames. simple_dc_simp id t1 t2 absnames)
         %   ( obid_for_simp t1
            , (\e t. true)
            , (SimpleMacroC id t1 t2 absnames)))%
         args)
;;


let update_simp_trail uobid n =
 % if display_simp_trail_p 
      then display_message ("Simp1C: " J tok_to_string (fst (hd simps))) 
      else ()%
		      
 if isr uobid then (simp_trail := (outr uobid) . simp_trail; ());
 if isr n then (outr n) ()
;;

let Simp1C e t =
  let f (uobid,n,c) = 
    (let t2,r,j = c e t in
       if dc_simps_only_p & not (name_of_rel r = `identity` 
                                 or is_comp_seq_just j) 
          then fail 
          else ( update_simp_trail uobid n
               ; (t2,r,j)
               )
         ) in
 (let convns = lookup_simp_convns e t in
   (first_value f convns)) ? AnyBetaC e t
;;

% end preceding depends on simp-re %



% Assumes hyp simps already put in tables. %
%let Simp1C e t =
  letrec f simps = 
    if null simps then failwith `Simp1C`
    else ( (let (),(),p,c = hd simps in
            if not p e t then failwith `Simp1C` else () ;
            let t2,r,j = c e t in
            if dc_simps_only_p & not (name_of_rel r = `identity` 
                                      or is_comp_seq_just j) 
            then fail else
            simp_trail := fst (hd simps) . simp_trail;
            if display_simp_trail_p 
             then display_message ("Simp1C: " J tok_to_string (fst (hd simps))) 
             else ();
            t2,r,j
           ) ? f (tl simps)
         )  in
  f (applicable_simps e t) ? AnyBetaC e t
;;
%
let NSimpC n = RepeatForC n (NthC 1 Simp1C) ;;

let DCSimpC e t =
  let b = dc_simps_only_p in
  dc_simps_only_p := true ;
  with_cleanup (NormC Simp1C e) t (\(). dc_simps_only_p := b)
;;

let ISimpC e t = 
  %dc_simps_only_p := false ;%
  (DCSimpC ANDTHENC NormC Simp1C) e t 
;;

let SimpC e t = 
  init_simp_tracking () ;
  ISimpC e t
;;

let SSome clauses p =
  SeqOnSameConcl (map (\i. RestrRewrite SimpC i %THEN (if (not i=0) then Try (True i) else Id)%)
		      (quicksort $> (normalize_clause_numbers clauses p)))
                 p
;;

let S p = SSome (0 . (upto 1 (num_hyps p))) p ;;

%*********************** DC and Hyp simping *****************************%

let DCS p = 
  let b = dc_simps_only_p in
  dc_simps_only_p := true ;
  let res = S p in
  dc_simps_only_p := false ;
  res
;;

let DCSSome l p = 
  let b = dc_simps_only_p in
  dc_simps_only_p := true ;
  let res = SSome l p in
  dc_simps_only_p := false ;
  res
;;



% Simp clause i using hyps and the global simpset.  Ignores suspect simps.
% 
let HSOneUsing hyps i p =
  let i.hyps = normalize_clause_numbers (i.hyps) p in
  let simps = appending_results (\j. potential_hyp_simps j p)
                                (remove i hyps) in
  (WESimps simps (SSome [i])
  ) p
;;

% The rewriter doesn't keep track of bound variables.  E.g. it
will happily apply a hyp x=2 to (Some x:N. x=3).
%
let warn_of_hyp_simp_danger i j p =
( if not null (intersection (free_vars (h i p))
	                    (bound_vars (j=0 => concl p | h j p))) then
    display_message ("Warning: hyp " J int_to_string i J " mentions vars bound in "
                     J int_to_string j J ".")
  else ()
) ? ()
;;

% Simp each clause in clauses using the global simpset and all hyps
  except clause.  Simp from largest numbered hyp to smallest, finishing 
  with the concl if it's in clauses.  At each simp, use the current hypotheses
  (some of which may have been simped).  
%
let HSSome clauses p =
  let clauses = quicksort $< (normalize_clause_numbers clauses p) in
  let conclp = not null clauses & hd clauses = 0 in
  let hyps = rev (if conclp then tl clauses else clauses) in
  let can_rewrite p i j = 
      is_invisible_var (var_of_hyp i p) 
      & (j=0 or
         let x = var_of_hyp j p in
         is_invisible_var x 
         or is_closed_at_clause p j (h i p)
         or every (concl p . map (C h p) (upto (j + 1) (num_hyps p)))
	          ($not o member x o free_vars))
      & (warn_of_hyp_simp_danger i j p; true)  in
  let possible_hyps_for p j = 
      filter (C (can_rewrite p) j)
             (remove_if ($= j) (upto 1 (num_hyps p))) in
  let T i p = HSOneUsing (possible_hyps_for p i) i p in
  (SeqOnSameConcl (map T hyps @ [if conclp then T 0 else Id])
  ) p
;;

% HSSome on everything. %
let HS p = HSSome (upto 0 (num_hyps p)) p ;;


%*********************************************************************%
%                     Simpsets.                                       %
%*********************************************************************%

%FDL PORT re

let bogus_location = `nonexistent location` ;;

let simpset_add name simp =
  let loc = %%FDL port ml_object_being_checked () ?%% bogus_location in
  let id = fst simp in
  letref found = false in
  (let simpsandlocs = apply_alist simpsets name in
   simpsets := update_alist simpsets name
                 ((simp,loc) 
                  .remove_if (\(id',()),(). id=id') simpsandlocs
                 ) ;
   ()

  )
  ? 
  (simpsets := (name, [simp,loc]) . simpsets ;
   ()
  )
;;

let simpset_adds tok simps = map (simpset_add tok) simps ;;

let simpset_delete name simpid =
  (let simpsandlocs = apply_alist simpsets name in
   simpsets := update_alist simpsets name
                 (remove_if (\(id,()),(). simpid=id) simpsandlocs) ;
   ()
  ) ? failwith `simpset_delete: simpset or simp name unknown`
;;

let delete_simpset id = remove_alist_entry simpsets id ;;

let get_simpset id = 
  map fst
      (apply_alist simpsets id ? failwith `get_simpset: no simpset `^id^`.`)
;;

let simpset_simp_adds names =
  map simp_add (appending_results get_simpset names) 
;;

let simpset_simp_deletes names =
  map simp_delete (appending_results get_simpset names) 
;;

let sites_of_simpset_add id =
   collect_successes 
    (\name,simpsandlocs. name, snd (find (\(id',()),(). id=id') simpsandlocs))
    simpsets
;;
%

let WSimpsets = WithSimpsets true;;
let WOSimpsets = WithSimpsets false;;
let WSimpsetsC = WithSimpsetsC;;

%FDL PORT re

let WSimpsets names T p =
  WSimps (appending_results get_simpset names) T p
;;

let WSimpsetsC names C e t =
  WSimpsC (appending_results get_simpset names) C e t
;;


let WOSimpsets names T p =
  WOSimps (collect get_simpset names) T p
;;
%

let ApplySimpsetC e t =
 first_value (\ (uobid,n,c). 
		 let r = c e t in
		  update_simp_trail uobid n;
		  r)
             (lookup_simpset_convns e t)
;;

let SimpsetSome names clauses p =
  let l = normalize_clause_numbers clauses p in
  let l' = rev (quicksort $< l) in
  let hyps = filter (\x. not x=0) l' in

  WithSimpsetsOnly names
   (let conv = NormC ApplySimpsetC in
    let Concl p = if member 0 l then RestrRewrite conv 0 p else Id p in
    let T i p = RestrRewrite conv i p in
      (Concl THENM SeqOnSameConcl (map T hyps))
   )
  p
;;

let Simpset names p = SimpsetSome names (0 . (upto 1 (num_hyps p))) p;;

%FDL PORT re
let ApplySimpC ( (op,ob,p,c):simp ) e t =

  if (op=` ` or op = opid_for_simp t) & p e t 
  then (let res = c e t in
        if display_simp_trail_p then
            display_message ("ApplySimpC: " J tok_to_string id) 
        else ()
       ;res
       )
  else failwith `ApplySimpC`
;;

let SimpsetSome names clauses p =
  let l = normalize_clause_numbers clauses p in
  let l' = rev (quicksort $< l) in
  let hyps = filter (\x. not x=0) l' in
  let conv = NormC (FirstC (map ApplySimpC (appending_results get_simpset names))) in
  let Concl p = if member 0 l then RestrRewrite conv 0 p else Id p in
  let T i p = RestrRewrite conv i p in
  (Concl THENM SeqOnSameConcl (map T hyps))
  p
;;
%

let Simpset names p = SimpsetSome names (0 . (upto 1 (num_hyps p))) p ;;

%FDL PORT re
let Simpset1Some names clauses p =
  let l = normalize_clause_numbers clauses p in
  let l' = rev (quicksort $< l) in
  let hyps = filter (\x. not x=0) l' in
  let conv = TryC (SweepUpC (FirstC (map ApplySimpC 
                                         (appending_results get_simpset names)))) in
  let Concl p = if member 0 l then RestrRewrite conv 0 p else Id p in
  let T i p = RestrRewrite conv i p in
  (Concl THENM SeqOnSameConcl (map T hyps))
  p
;;
% 
let Simpset1Some names clauses p =
  let l = normalize_clause_numbers clauses p in
  let l' = rev (quicksort $< l) in
  let hyps = filter (\x. not x=0) l' in

  WithSimpsetsOnly names
   (let conv = TryC (SweepUpC ApplySimpsetC) in
    let Concl p = if member 0 l then RestrRewrite conv 0 p else Id p in
    let T i p = RestrRewrite conv i p in
    (Concl THENM SeqOnSameConcl (map T hyps))
   ) p
;;

let Simpset1 names p = Simpset1Some names (0 . (upto 1 (num_hyps p))) p ;;

let SSetC names = WSimpsetsC names SimpC ;;


% can only affect hyps added by WithHypSimps %
let WOHypSimp i T p =
  WOESimps ([(hyp_id i p)] ? []) T p 
;;


let SimpSome clauses p =
  let l = normalize_clause_numbers clauses p in
  let l' = rev (quicksort $< l) in
  let hyps = filter (\x. not x=0) l' in
  let Concl p = if member 0 l then Rewrite SimpC 0 p else Id p in
  let T i p = WOHypSimp i (Rewrite SimpC i) p in 
  (Concl THENM SeqOnSameConcl (map T hyps))
  p
;;

let Simp p = SimpSome (0 . (upto 1 (num_hyps p))) p ;;
  
% SimpsetC is for compatibility with the old simplifier. %
% FDL Port modified from simp.ml
let SimpsetC names = WithSimpsetsC names SimpC ;;
let Simpsetp names = WithSimpsets names Simp ;;
%
let SimpsetC names = WSimpsetsC names SimpC ;;
let Simpsetp names = WSimpsets names Simp ;;

let SimpsetOnlyC names =
 WithSimpsetsOnly names (NormC ApplySimpsetC)
;;

let SSet names = WSimpsets names S ;; 
let SSetSome names clauses =  WSimpsets names (SSome clauses) ;;
let HSSet names = WSimpsets names HS ;;
let HSSetSome names clauses = WSimpsets names (HSSome clauses) ;;

%*********************************************************************%
%          Manual rewriting with simps.                               %
%*********************************************************************%

let ApplyLemmaSimps lsimps = 
 let C e t = first_value (\ (op,ob,pred,convn). 
                              if pred e t then convn e t else fail)
                         lsimps in
  All (RestrRewrite (RepeatC (SweepDnC C)))
;;

let LemmaSimps names = ApplyLemmaSimps (flatten (map lemma_simps names));;

let RevLemmaSimps names = ApplyLemmaSimps (flatten (map rev_lemma_simps names));;

let LemmaSimp name = LemmaSimps [name] ;;
let RevLemmaSimp name = RevLemmaSimps [name] ;;

let LemmaSimp1s names = 
  let cs = appending_results (map fourth o lemma_simps) names in
  All (RestrRewrite (SweepUpC (FirstC cs)))
;;

let RevLemmaSimp1s names = 
  let cs = appending_results (map fourth o rev_lemma_simps) names in
  All (RestrRewrite (SweepUpC (FirstC cs)))
;;

let LemmaSimp1 name = LemmaSimp1s [name] ;;
let RevLemmaSimp1 name = RevLemmaSimp1s [name] ;;



letref disabled_conditional_simps = [] : tok list ;;

letref dc_simps_only_p = false ;;


% For conditional rewrites.  If condition_tacs is empty then
  do the rewrite, leaving the condition as a subgoal.  If non-
  empty, it need not be the same length as the condition list.
  Padding or truncation is done.
%

let lemma_to_cond_simp lemma_obid condition_tacs =
  let p =  statement_lookup lemma_obid in
  let ob = simp_formula_obid p true in
  (ob,
   (\e t. %not member lemma_name disabled_conditional_simps 
          &% not dc_simps_only_p),
   LemmaThenLC_o condition_tacs lemma_obid
  )
  ? failwith ( `Lemma ` ^ name_of_lemma lemma_obid
             ^ `does not have the right form for a simp.`)
;;


%
let add_simple_dc_simp id t1 t2 abstr_names =
  add_simp (simple_dc_simp id t1 t2 abstr_names)
;;
%

let add_cond_lemma_simp uobid lemma_obid condition_tacs =
  simp_add_o uobid [lemma_to_cond_simp lemma_obid condition_tacs]
;;


% For conditional rewrites.  If condition_tacs is empty then
  do the rewrite, leaving the condition as a subgoal.  If non-
  empty, it need not be the same length as the condition list.
  Padding or truncation is done.
%
%let lemma_clause_to_cond_simp lemma_name n condition_tacs =
  if n<1 then failwith `lemma_clause_to_cond_simp: clause number < 1` else
  let (), context, assums, concl  =  hd (nth_simple_formula n
					 (main_goal_of_theorem lemma_name)) in
  (simp_formula_obid concl true,
   (\e t. %%not member simp_id disabled_conditional_simps
          &%% not dc_simps_only_p),
   GenLemmaWithThenLC n [] condition_tacs  lemma_name
  )
  ? failwith `Lemma `^lemma_name^`does not have the right form for a simp.`
;;

%
let lemma_clause_to_cond_simp lemma_obid n condition_tacs =
  if n<1 then failwith `lemma_clause_to_cond_simp: clause number < 1` else
  let (), context, assums, concl  =  hd (nth_simple_formula n
					 (statement_lookup lemma_obid)) in
  (simp_formula_obid concl true,
   (\e t. %not member simp_id disabled_conditional_simps
          &% not dc_simps_only_p),
   GenLemmaWithThenLC_o n [] condition_tacs lemma_obid
  )
  ? failwith ( `Lemma `^ (name_of_lemma lemma_obid)
             ^ `does not have the right form for a simp.`)
;;

% For unconditional lemmas %
let lemma_clause_to_simp lemma_obid n  =
  if n<1 then failwith `lemma_clause_to_simp: clause number < 1` else
  let (), context, assums, concl  =  hd (nth_simple_formula n
					 (statement_lookup lemma_obid)) in
  if not null assums then failwith `lemma_clause_to_simp: clause is conditional` else
  (simp_formula_obid concl true,
   (\e t. true),
   GenLemmaWithThenLC_o n [] [] lemma_obid)
  ? failwith ( `Lemma `^ (name_of_lemma lemma_obid)
             ^ `does not have the right form for a simp.`)
;;

let lemma_clauses_to_simps obid = 
  map (\x. lemma_clause_to_simp obid x ? lemma_clause_to_cond_simp obid x [Trivial])
      (upto 1 (number_of_lemma_clauses_o obid))
;;

let add_dc_simp uobid id t1 t2 c1 c2 = simp_add_o uobid [dc_simp id t1 t2 c1 c2] ;;

let add_lemma_simps_o uobid obid = simp_add_o uobid (lemma_clauses_to_simps obid) ;;
let add_lemma_simpset_o uobid obid = simpset_simp_add_o uobid (lemma_clauses_to_simps obid) ;;

let LemmaNC name n = GenLemmaWithThenLC n [] []  name ;;
let RevLemmaNC name n = RevGenLemmaWithThenLC n [] []  name ;;
let LemmaNRW name n = All (Rewrite (SweepDnC (LemmaNC name n))) ;;
let RevLemmaNRW name n = All (Rewrite (SweepDnC (RevLemmaNC name n))) ;;

let lemma_to_conversions name =
  map (LemmaNC name) ((-1) . (upto 1 (num_simple_clauses_of_lemma name)))
;;

let lemma_to_reverse_conversions name =
  map (RevLemmaNC name) (upto 1 (num_simple_clauses_of_lemma name))
;;

let LemmaRWs names = 
  let cs = flatten (map lemma_to_conversions names) in
  All (Rewrite (SweepDnC (FirstC cs)))
;;

let RevLemmaRWs names = 
  let cs = flatten (map lemma_to_reverse_conversions names) in
  All (Rewrite (SweepDnC (FirstC cs)))
;;

let LemmaRW name = LemmaRWs [name] ;;
let RevLemmaRW name = RevLemmaRWs [name] ;;

let RLemmaRWs names = 
  let cs = flatten (map lemma_to_conversions names) in
  All (Rewrite (RepeatC (SweepDnC (FirstC cs))))
;;

let RRevLemmaRWs names = 
  let cs = flatten (map lemma_to_reverse_conversions names) in
  All (Rewrite (RepeatC (SweepDnC (FirstC cs))))
;;


%FDL PORT re

letref tmp_simps = []: simp list ;;

let delete_tmp_simps (simps: simp list) = 
  tmp_simps := filter (\id,(). not exists (\id',(). id=id') simps) tmp_simps ;
  ()
;;

let add_tmp_simps simps =
  tmp_simps := simps @ tmp_simps ;
  ()
;;

letref tmp_simp_max_length = 200 ;;

let check_tmp_simps () =
  if length tmp_simps > tmp_simp_max_length 
  then display_message "Warning: there are a lot of temporary simps."
  else ()
;;  

let WithSimps simps T p =
  check_tmp_simps () ;
  map add_simp simps ;
  add_tmp_simps simps ;
  with_cleanup T p (\(). map simp_delete simps; delete_tmp_simps simps)
;;
%

% direction=true iff l to r. %
%let hyp_to_simp i direction guess_direction_p p =
 let i = normalize_clause_number i p in
 let fmla = h i p in
 let l_to_r_p = 
     if guess_direction_p then
       let t1,t2 = simp_formula_relnands fmla in
       if (not is_var_term t1 & not is_var_term t2)
          or (is_var_term t1 & t1=t2)
          then failwith (`hyp_to_simp: cannot guess simp direction for hyp `
                         ^ int_to_tok i)
          else is_var_term t1
     else direction in
   hyp_id i p,
   simp_formula_opid fmla l_to_r_p
   , (\e t.true)
   , if l_to_r_p then HypC i else RevHypC i
;;
%
let hyp_to_simp_o i direction guess_direction_p p =
 let i = normalize_clause_number i p in
 let fmla = h i p in
 let l_to_r_p = 
     if guess_direction_p then
       let t1,t2 = simp_formula_relnands fmla in
       if (not is_var_term t1 & not is_var_term t2)
          or (is_var_term t1 & alpha_equal_terms t1 t2)
          then failwith (`hyp_to_simp: cannot guess simp direction for hyp `
                         ^ int_to_tok i)
          else is_var_term t1
     else direction in

  let op,ob = simp_formula_op_ob fmla l_to_r_p
        
       ? %extrme kludge to catch variables % (` `, dummy_object_id()) in
   hyp_id i p
   , op, ob
   , (\e t.true)
   , if l_to_r_p then HypC i else RevHypC i
;;

let all_hyp_simps p =
  collect_successes (\i.  hyp_to_simp_o i true true p)
                    (upto 1 (num_hyps p)) 
;;

let WithSomeHypSimps l1 l2 T p = WESimps (some_hyp_simps l1 l2 p) T p ;;

let WithHypSimps T p = WESimps (all_hyp_simps p) T p ;;
let HSimpSome clauses = WithHypSimps (SimpSome clauses) ;;

let HSimp = WithHypSimps Simp ;;


%FDL PORT re

let DisablingCondSimps l T p =
  disabled_conditional_simps := l ;
  let res = T p in
  disabled_conditional_simps := [] ;
  res
;;
%