
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;; EQUALITY RULE -- for complete description of the congruence closure
;;;                  algorithm used see Johnson, A Computer System for
;;;                  Checking Proofs, TR 80-444, Chptr. 6, or Krafft, TR 81-458.
;;;                  Discussion of the use of congruence closure for the theory
;;;                  of LISP list structure can be found in Nelson and Oppen,
;;;                  Fast Decision Procedures Based on Congruence Closure,
;;;                  JACM, v.27, no.2 , April 1980.


;; pointers to nodes.  Each element of buckets$ is a pointer
;; to a doubly linked list (with header) of nodes linked 
;; through their previous and next fields.
(defvar num-buckets$ 117)

(defvar buckets$ (make-array 117))
(defvar unknown-terms (make-array 117))

(defvar unknown-term-count 0)
(defvar num-nodes$ 0)                               ;number of nodes now existing

(defvar free-node-list nil)

(defvar disequalities$ nil)                           ;a list of pairs of nodes
                                                  ;that are the lhs's and rhs's
                                                  ;of disequalities in H ^ ~C  


;; make all buckets empty, make disequality list empty 
;; no new variables used yet, no nodes created yet

(defun init-equality ()

  (dotimes (i num-buckets$)
    ;; (add-to-free-node-list (sel buckets$ (i)))
    (setf (aref buckets$ i) nil)
    (setf (aref unknown-terms i) nil))

  (setf disequalities$ nil)            
  ;;   (<- non-equality-assumption-formulas nil)
  ;;   (<- non-equality-conclusion-formula nil)
  (setf num-nodes$ 0)
  (setf unknown-term-count 0)
  )


(defun do-equality (terms)
  (if (onep (length terms))
      (eq-insert (car terms))
      (do ((tms (cdr terms) (cdr tms))
	   (tm (car terms) (car tms)))
	  ((null tms))
	(eq-assert tm (car tms)))))


(defun do-disequality (terms)
  (setf disequalities$ (cons terms disequalities$)))


(defun unsatisfiable-graph ()
  (let ((unsatisfiable      nil))

    (do ((diseqs disequalities$ (cdr diseqs)))
	((or (null diseqs) unsatisfiable)
	 unsatisfiable)

      (setf unsatisfiable
	    (let ((rep1       (eq-insert (caar diseqs)))
		  (all-equal  t))

	      (do ((terms (cdar diseqs) (cdr terms)))
		  ((or (null terms) (not all-equal))
		   all-equal)

		(let ((rep (eq-insert (car terms))))
		  (setf all-equal (eql rep rep1))
		  (setf rep1 rep))))))))




;;;  if necessary create a node corresponding to this term and enter it in the graph
;;;  return (pointer to) the node that is the representative of the equivalence
;;;  class to which this term (node) belongs (all this relative to type type)

;; nodes in the graph used in the congruence closure
;; algorithm correspond to terms
(defstruct eq-node
  label					; function identifier for the term corresponding to
					;   this node
					;   For some terms this label is a list. For
					;   other terms label is equal to the term's KIND
					;   field.             
  operands				; a list of pointers to the nodes representing the
					;   operands for this function node    
					;   (the node corresponding to the type of a term
					;   is considered an operand of that term)
  containers				; a list of pairs corresponding to operand fields 
					;   of nodes for which this is an operand
					;   The list is of the form (n i).  n is a pointer
					;   to a node whose i-th operand is this node.
  reprlink				; either a pointer to a node, or nil (marking the
					;   root) such that for all modes x and y,  x and y
					;   are in the same equivalence class iff their
					;   reprlinks lead to the same root node
  reprcount				; the number of nodes in this node's equivalence
					;   class (if this node is a representative)
  previous				; the previous entry in this hash bucket
  next					; the next entry in the hash bucket
  eligible				; a boolean value, = "this node is in a hash bucket"
  node-iden				; a unique integer associated with this node (used
					;   for hashing purposes; this value is used as a 
					;   "fake pointer" to this node)
  )

(defun label-of-eq-node (n) (eq-node-label n))
(defun operands-of-eq-node (n) (eq-node-operands n))
(defun containers-of-eq-node (n) (eq-node-containers n))
(defun reprlink-of-eq-node (n) (eq-node-reprlink n))
(defun reprcount-of-eq-node (n) (eq-node-reprcount n))
(defun previous-of-eq-node (n) (eq-node-previous n))
(defun next-of-eq-node (n) (eq-node-next n))
(defun eligible-of-eq-node (n) (eq-node-eligible n))
(defun node-iden-of-eq-node (n) (eq-node-node-iden n))

(defun eq-node (label operands containers reprlink reprcount previous next eligible node-iden)
  (make-eq-node :label label
		:operands operands
		:containers containers
		:reprlink reprlink
		:reprcount reprcount
		:previous previous
		:next next
		:eligible eligible
		:node-iden node-iden))

;; Make the nodes corresponding to terms u,v be in the same equivalence class
;;   (thus u and v are made equal)
(defun eq-assert (u v)
   
  (let ((ToDo (list (list (eq-insert u) (eq-insert v)))))

    (do ()
	((null ToDo))

      (let ((x (do ((link (caar ToDo) (reprlink-of-eq-node link)))
		   ((null (reprlink-of-eq-node link)) link)))
	    (y (do ((link (cadar ToDo) (reprlink-of-eq-node link)))
		   ((null (reprlink-of-eq-node link)) link))))

	(setf ToDo (cdr ToDo))
                 
	(when (not (eql x y))
	  ;; choose as the new representative the nodes who's
	  ;; current equivalence class is larger.  Let x be
	  ;; the name of the new representative.
	  (when (> (reprcount-of-eq-node y) 
		   (reprcount-of-eq-node x))
	    (let ((temp x))
	      (setf x y)
	      (setf y temp)))

	  (setf (eq-node-reprlink y) x)
	  (setf (eq-node-reprcount x) (+ (reprcount-of-eq-node x)
					 (reprcount-of-eq-node y)))    

	  ;; For each term(node) that had y as an operand have it
	  ;;   now use x as operand wherever it used y before.
	  ;;   Remove each node that gets changed in this way
	  ;;   from its bucket, rehash, and check if an identical
	  ;;   node exists in the new bucket.  If so, add the
	  ;;   node and its duplicate to the ToDo list. If not,
	  ;;   put the node in the bucket.
	  (dolist (node-op-pair (containers-of-eq-node y))
	    (let ((p-node (car node-op-pair))
		  (p-op (cadr node-op-pair)))
 
	      (setf (car (nthcdr (1- p-op) 
				 (operands-of-eq-node p-node)))
		    x)

	      (when (eligible-of-eq-node p-node)
		(bucket-remove$ p-node)
		(let* ((h (hash$ (label-of-eq-node p-node) 
				 (operands-of-eq-node p-node)))
		       (dup (search-bucket$ h
					    (label-of-eq-node p-node)
					    (operands-of-eq-node p-node))))
		  (if dup
		      (setf ToDo (cons (list p-node dup) ToDo))
		      (bucket-insert$ h p-node))))))
	  (setf (eq-node-containers x) (append (containers-of-eq-node x) 
					       (containers-of-eq-node y))))))))


;;; search in bucket 'bucket of the buckets array for a node which has
;;; a label-field equal to 'label' and has operand nodes EQ to those in 'operands'
;;; (return a pointer to the node if one is found, else 'nil')

(defun search-bucket$ (bucket label operands)
  (let ((head (aref buckets$ bucket))
	(duplicate  nil)
	(made-pass-p nil))
    
    (when head
	  (do ((current head (next-of-eq-node current)))
	      ((or duplicate (and made-pass-p (eql current head)))
	       duplicate)
	      (setf made-pass-p t)
	      (when (and (equal label (label-of-eq-node current))
			 (equal (length operands)
				(length (operands-of-eq-node current))) 
			 (forall-p #'eql 
				   operands
				   (operands-of-eq-node current)))
		    (setf duplicate current))))))




;;; delete node 'node' from the bucket array (and make the node ineligible)
(defun bucket-remove$ (node)
   (setf (eq-node-previous (next-of-eq-node node)) (previous-of-eq-node node))
   (setf (eq-node-next (previous-of-eq-node node)) (next-of-eq-node node))
   (setf (eq-node-next node) nil)
   (setf (eq-node-previous node) nil)
   (setf (eq-node-eligible node) nil))


;;;; return an integer to be associated with a node as its unique identifier
;;;  (used in hashing as a "fake pointer" to the node)
(defun get-node-iden$ ()
  (setf num-nodes$ (1+ num-nodes$)))


;; return a newly created node with all fields, except the identifier one, nil
(defun new-node$ ()
  (if (null free-node-list)
      (eq-node nil nil nil nil nil nil nil nil (get-node-iden$))
      (prog1 (car free-node-list)
	(setf free-node-list (cdr  free-node-list)))))

;;; insert node 'node' into a bucket 'bucket 'in the bucket array.  If the bucket 
;;;   has no header yet, make one. (Also, make the node eligible)
(defun bucket-insert$ (bucket node)

   (let ((head (aref buckets$ bucket)))

     (when (null head)        ;if no header present, make one
       (setf (aref buckets$ bucket) (new-node$))
       (setf head (aref buckets$ bucket))
       (setf (eq-node-previous head) head)
       (setf (eq-node-next head) head))

     (setf (eq-node-next node) (next-of-eq-node head))
     (setf (eq-node-previous node) head)
     (setf (eq-node-previous (next-of-eq-node head)) node)
     (setf (eq-node-next head) node)       
     
     (setf (eq-node-eligible node) t)))



;;; if necessary create a node corresponding to this term and enter it in the graph
;;; return (pointer to) the node that is the representative of the equivalence
;;; class to which this term (node) belongs (all this relative to type type)


(defun eq-insert (term)
   (let ((fcn-id (get-fcn-id$ term))
	 (arg-pointers nil)
         (h nil)
         (eq-ins nil))

     (when (not (and (consp fcn-id)
		     (eql (car fcn-id) 'UNKNOWN-TERM)))
       (setf arg-pointers
	     (mapcar #'(lambda (bound-term)
			 (eq-insert (term-of-bound-term bound-term)))
		     (bound-terms-of-term term))))

     (setf h (hash$ fcn-id arg-pointers))

     ;; see if a node already exists for this term.  If not make one.
     (let ((dup (search-bucket$ h fcn-id arg-pointers)))
       (if dup
	   (do ((dupp dup (reprlink-of-eq-node dupp)))
	       ((not (reprlink-of-eq-node dupp)) dupp))
	     
	   (progn
	     (setf eq-ins (new-node$))
	     (bucket-insert$ h eq-ins)
	     (setf (eq-node-label eq-ins) fcn-id)
	     (setf (eq-node-operands eq-ins) arg-pointers)
	     (do ((i 1 (1+ i))
		  (opnds arg-pointers (cdr opnds)))
		 ((null opnds))
	       (setf (eq-node-containers (car opnds))
		     (cons (list eq-ins i) 
			   (containers-of-eq-node (car opnds)))))

	     (setf (eq-node-containers eq-ins) nil)
	     (setf (eq-node-reprlink eq-ins) nil)
	     (setf (eq-node-reprcount eq-ins) 1)
	     ;;return eq-insert (= eq-ins)
	     eq-ins)))))



;; return an identifier for a term (to be used as the label of a node)
(defun get-fcn-id$ (term)
  
  (cond
   ((universe-term-p term) 
    (list 'u (level-of-universe-term term)))
	   
   ((variable-p term)
    (list 'v (id-of-variable-term term)))

   ((natural-number-term-p term)
    (list 'i (number-of-natural-number-term term)))

   ((token-term-p term)
    (list 'a (atom-of-token-term term)))
                       
   ((termof-p term)
    (list 't (id-of-term term)))

   ((or (void-term-p term)
	(atom-term-p term)
	(int-term-p term)
	(nil-term-p term)
	(axiom-term-p term)
	(object-term-p term))
    (id-of-term term))

   ;; PERF
   ((or (any-term-p term)
	(minus-term-p term)
	(add-term-p term)
	(subtract-term-p term)
	(multiply-term-p term)
	(divide-term-p term)
	(remainder-term-p term)
	(list-term-p term)
	(cons-term-p term)
	(union-term-p term)
	(inl-term-p term)
	(inr-term-p term)
	(pair-term-p term)
	(equal-term-p term)
	(apply-term-p term)
	(less-than-term-p term)
	(atom-eq-term-p term)
	(int-eq-term-p term)
	(less-term-p term))
    (id-of-term term))

	 
   (t (lookup-unknown-term term)
      ;;or (raise-error (error-message '(proof termof not) term))));;lal
      )))


(defun rot (x b)
  (logand #xffffffff
	  (if (minusp b)
	      (let ((b1 (logand #x1f (- b))))
		(dpb x (byte b1 (- 32 b1)) (ash x (- b1))))
	      (let ((b1 (logand #x1f b)))
		(logior (ash x b1) (ldb (byte b1 (- 32 b1)) x))))))


;; return a hash value for a term           
(defun hash$ (function-symbol operand-list)
  (let ((hval (sxhash function-symbol)))
    (dolist (opnd operand-list)
      (setf hval (logxor (node-iden-of-eq-node opnd) (rot hval 1))))
    (abs (rem hval num-buckets$))))


(defun lookup-unknown-term (term)
  (let ((bkt (rem (hash-term term) num-buckets$)))
    (do ((alist (aref unknown-terms bkt) (cdr alist)))
	((null alist)
	 (incf unknown-term-count)
	 (let ((uterm (list 'UNKNOWN-TERM unknown-term-count)))
	   (setf (aref unknown-terms bkt)
		 (cons (cons term uterm) (aref unknown-terms bkt)))
	   uterm))
      (let ((entry (car alist)))
	(when (equal-terms-p term (car entry))
	  (return-from lookup-unknown-term (cdr entry)))))))
	
    
    
