
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; -docs- (mod ref)
;;;;
;;;;	Refiner mechanism: consist of five primary functions.
;;;;
;;;;	Briefly :	
;;;;
;;;;	apply-tactic(<term> <sequent>) 		: <proof>
;;;;
;;;;	refine-primitive-rule(<sequent> <rule>)	: (values <rule> <sequent> list)
;;;;	 * INSTANTIATES sequents
;;;;
;;;;	refine-tactic(<sequent> <tactic>)	: (values <rule> <sequent> list)
;;;;	 * calls apply-tactic
;;;;	 * sequents are (ie eq) frontier of proof top.
;;;;
;;;;	refine(<sequent> <rule>) 		: (values <rule> <sequent> list)
;;;;	 * calls refine-primitive-rule or refine-tactic.
;;;;		
;;;;	refine 		: rule -> sequent -> (proof list # (proof list -> proof))
;;;;	 * calls refine.
;;;;	 * validation instantiates proof.
;;;;
;;;;  -page-
;;;;
;;;;	More details:
;;;;
;;;;	refine : rule -> sequent -> (proof list # (proof list -> proof))
;;;;	 * this is the normal entry point to the refiner from the tactics.
;;;;	 * calls refine to produce refinement and subgoal list.
;;;;	 * returns subgoal list and validation.
;;;;	   validation : : proof list -> proof
;;;;	     - fails if proof list does not match the subgoal list.
;;;;	       proof list matches subgoal list if when compared pair wise:
;;;;		* all assumptions ids are identical
;;;;		* all assumption types and the concls are alpha equal
;;;;	     - if successful the INSTANTIATES proof whose:
;;;;	        * goal is the sequent;
;;;;		* refinment is result from refine call;
;;;;		* subgoal list is proof list.
;;;;
;;;;  -page-
;;;;
;;;;	refine (<sequent> <rule> 		: (values <rule> <sequent> list)
;;;;	 * normal entry point to refiner from top-level refinement call,
;;;;	   ie from editor or library.
;;;;	 * also called by tactics via ml refine.
;;;;	 * calls refine-tactic or refine-primitive-rule.
;;;;
;;;;  -page-
;;;;
;;;;	apply-tactic (<term> <sequent>) 	: <proof>
;;;;	 * parses term to produce tactic.
;;;;	 * applies resulting tactic to sequent to produce validation and subgoal list
;;;;	   and records any references made to abstractions, statements, or proofs.
;;;;	 * applies resulting validation to resulting subgoal list to produce proof,
;;;;	   and annotates proof with references.
;;;;	 * returns annotated proof.
;;;;
;;;;  -page-
;;;;
;;;;	refine-tactic (<sequent> <tactic>)	: (values <rule> <sequent> list)
;;;;	 * calls apply-tactic to produce proof.
;;;;	 * verifies that goal of resulting proof matches sequent.
;;;;	 * possibly cleans primitive proofs from proof.
;;;;	   - all proof modifications done CONSTRUCTIVELY.
;;;;	   - primitive-extract-mode : controls treatement of primitive proofs.
;;;;	    * eager	: partial extracts are made from primitive proof trees
;;;;			  and the primitive proof trees are cleared from proof.
;;;;	    * lazy	: no extraction done, primitive trees are untouched.
;;;;	    * nil	: no extraction done, primitive trees cleared.
;;;;	 * returns tactic rule with possibly cleansed proof as proof top
;;;;		   and frontier of proof.
;;;;
;;;; -doce-


;;;;	In following code, there is no destructive modification of proofs outside
;;;;	of the scope of an instantiation(copy-of-proof-node) of the proof.
;;;;
;;;;	variables with id sequent are treated as sequents, ie it is assumed they
;;;;	have no rule or children.
;;;;
;;;;	variables with id proof are assumed to be proofs. As sequents are simply
;;;;	degenerate proofs in some circumstances such args may be sequents.


(defun add-ref-lettypes ()
  (ml-text "lettype VALIDATION = proof list -> proof")
  (ml-text "lettype validation = VALIDATION")
  (ml-text "lettype TACTIC = proof -> (proof list) # validation")
  (ml-text "lettype tactic = TACTIC"))

(insys (add-ref-lettypes))

(defvar *ref-tactic-type* (makety 'tactic))

(defun tactic-type-p (type) (equal type *ref-tactic-type*))


(defunml (|refine| (rule sequent))
    (rule -> tactic)

  (mlet* (((rule sequents) (refine sequent rule)))
	 ;;(setf d rule e sequents) (break "refine");;
	 ;; (subgoals . validation)
	 (cons sequents
	       (make-closure #'(lambda (proofs)
				 (unless (apply-predicate-to-list-pair sequents proofs
								       #'equal-sequents-p)
				   (progn ;;(break "f")
				     (breakout evaluation "refine: validation failed.")))
			    
				 (let ((proof (copy-of-proof-node sequent t)))
				   (setf (proof-node-children proof) proofs
					 (proof-node-rule proof) rule)
				   proof))
			     1))))

;;;
;;; returns rule and children and does not copy.
;;;

(defun refine (sequent rule)

  (with-tag 'refine

    (cond
      ((primitive-rule-p rule)
       (refine-primitive-rule sequent rule))

      ((nml-tactic-rule-p rule)
      (refine-tactic sequent rule))

      ((proof-rule-p rule)
       (refine-proof-rule sequent rule))

      ;;((prl-rule-p rule)
      ;;(refine-prl-rule proof-node rule))
      ;;((proof-cache-rule-p (rule-of-proof-node proof-node))
      ;;(refine-proof-cache-rule proof-node))
      ;;((reflection-rule-p rule)
      ;;(refine-reflection-rule proof-node rule))

      (t (raise-error (error-message '(refine rule-type unknown)))))))


(defun text-to-tactic (term)
  (let ((source (source-reduce term '(ml ref))))
    (with-dependency-environment
     (with-ml-evaluation text-to-tactic
			 (with-dependencies 
			  (mlet* (((tac type) (ml-term source nil)))

				 ;; TODO : work with-ml-xref in here and then note dependencies of idents in called.
		       
				 ;;(setf a tac b type c source) (break "b")
				 (unless (tactic-type-p type)
					 (raise-error (error-message '(ref text-to-tactic not-tactic-type))))
				 tac))))))

#|
(defun apply-tactic (term sequent)
      ;;(break "apply tactic")

    (with-dependency-environment
	(with-ml-evaluation apply-tactic
	  (let ((source (source-reduce term '(ml ref))))
	    (with-dependencies 
		(mlet* (((tac type) (ml-term source nil)))

		       ;; TODO : work with-ml-xref in here and then note dependencies of idents in called.
		       
		       (unless (tactic-type-p type)
			 ;;(setf a tac b type c source) (break "b")
			 (raise-error (error-message '(ref apply-tactic not-tactic-type))))
		       
		       (let ((validation (ap tac sequent)))
			 (setf -deps (environment-dependencies-collected (current-transaction-stamp)))
			;; (setf a validation b tac c sequent) (break "v")
			 
			 (mark-proof-node (ap (cdr validation) (car validation))
					  'dependencies
					  ;; do we get lib refs from callbacks ??
					  (environment-dependencies-collected (current-transaction-stamp))
					  -deps
					  ))))))))
|#

(defvar *source-list* nil)
(defvar *source-list-index* 0)
(defvar *source-list-count* nil)
(defun save-tactic-source (tac)
  (when *source-list-count*
    (push (cons (incf *source-list-index*) tac) *source-list*)
    (format t "~%SourceList updated ~a ~a . " *source-list-index* (length *source-list*))
    (setf *source-list-count* (mod (incf *source-list-count*) 64))
    (when (zerop *source-list-count*)
      (setf (cdr (nthcdr 48 *source-list*)) nil))))


(defun apply-tactic (term sequent)
  ;;(break "apply tactic")
  ;;(funmlcall (ml-text "view_show") (icons-term (conclusion-of-sequent sequent) term))

  (with-dependency-environment
      (with-ml-evaluation apply-tactic
	(let ((source (source-reduce term '(ml ref)))
	      (xrefs nil)
	      (ctac nil))
	  (with-dependencies
	      (with-ml-xref
		  (mlet* (((tac type) (ml-term source nil)))
			 (setf xrefs (get-ml-xref))
			 (setf ctac tac)

			 (save-tactic-source source)

			 ;;(setf a tac b type) ;;(break "apply tac")
			 (unless (tactic-type-p type)
			   ;;(setf a tac b type c source) (break "tactic type?")
			   (raise-error (error-message '(ref apply-tactic not-tactic-type))))))

	    ;;(when (equal-terms-p -idtac term) (setf -ctac ctac -sequent sequent) (break "at"))

	    (let ((validation (let ((*process-break* nil)) (ap ctac sequent))))
	      ;;(setf -deps (environment-dependencies-collected (current-transaction-stamp) t))
	      ;;(setf a validation b tac c sequent) (break "v")

	      (mark-proof-node-n 
	       (mark-proof-node (ap (cdr validation) (car validation))
				'dependencies
				(environment-dependencies-collected (current-transaction-stamp) t)
				;;-deps
				)
	       'xref
	       (xrefs-to-term (raw-mlxrefs-to-xrefs xrefs nil)))))))))


(defun refine-tactic (sequent tactic)
  (incf-tactic-count)
  (let* ((text (text-of-tactic-rule tactic))
	 (proof (apply-tactic text sequent)))

    ;;(setf a proof b sequent c tactic) (break "r")
    (unless (equal-sequents-p sequent proof)
      (raise-error (error-message '(refine tactic refined-goal-differs))))

    ;;(when (null (proof-node-rule proof)) (setf -proof proof)(break "rt"))

    (values (refined-nml-tactic-rule text (refined-primitive-proof proof))
	    (frontier-of-proof proof))))


;;;;	proof-rules 

(defun proof-rule-to-extract (proof children)
  (let ((rule (rule-of-proof-node proof)))
    ;;(format t "prte ~a~%" (length children))
    (case (id-of-rule rule)
      (permute-proof
       (let ((nchildren (let ((m (car (args-of-proof-rule rule))))
			  (if m
			      (inverse-permute-list-by-map m children)
			      children))))
	 ;; (setf -rule rule -p (proof-of-top-rule rule)) (break "prte")

	 (extract-replace (or (proof-rule-extract rule)
			      (setf (proof-rule-extract rule)
				    (extract-now (proof-of-top-rule rule) t)))
			  nchildren)))	
      (extract-proof (void-term))
      (otherwise (raise-error (error-message `(unknown proof rule) (id-of-rule rule)))))))


(defun permute-proof-rule (sequent rule)
  (let ((p (proof-of-top-rule rule))
	(map (car (args-of-proof-rule rule))))

    (unless (equal-sequents-p sequent p)
      (raise-error (error-message '(permute-proof-rule goals match not))))

    (values rule
       (let ((l (if map
                    (permute-list-by-map #'equal-sequents-p map (frontier-of-proof p))
                    (frontier-of-proof p) 
                 )))
          ;;(setf -l l -rule rule -sequent sequent -p p -map map) (break "ppr")
          l))
          
   ))

(defun extract-proof-rule (sequent rule)
  (let ((p (proof-of-top-rule rule)))
    (let ((concl (conclusion-of-proof-node sequent)))

      (unless (equal-term-p concl)
	(raise-error (error-message '(extract-proof-rule equal term not) concl)))

      (let ((equand (leftterm-of-equal-term concl)))
	(unless (equal-terms-p equand (rightterm-of-equal-term concl))
	  (raise-error (error-message '(extract-proof-rule equands equal not) concl)))

	(let ((ext (extract-now p nil)))
	  (unless (equal-terms-p ext (leftterm-of-equal-term concl))
	    (raise-error (error-message '(extract-proof-rule equal extract not) equand ext))))

	(values rule nil)))))


(defun refine-proof-rule (sequent rule)
  ;;(break "rpr")
  (case (id-of-rule rule)
    (permute-proof (permute-proof-rule sequent rule))
    (extract-proof (extract-proof-rule sequent rule))
    (otherwise (raise-error (error-message `(unknown proof rule) (id-of-rule rule))))))


(defunml (|make_permute_proof_rule| (proof map))
    (proof -> ((int list) -> rule))
  (proof-rule *permute-proof-rule-definition* (list map) proof))

(defunml (|make_extract_proof_rule| (proof))
    (proof -> rule)
  (proof-rule *extract-proof-rule-definition* nil proof))

;;;;	
;;;;	primitive-proof : proof tree which may contain primitive rules.
;;;;	top-proof	: proof tree which contains only tactics or primitive-tree rules.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	partial-extract : primitive-proof -> top-proof 
;;;;	  - does not examine top-proof of tactic rule. 
;;;;	  - top-proofs embedded in a primitive-proof become leaves of 
;;;;	    primitive-tree replacing primitive-proof
;;;;	
;;;;	
;;;;	Need some handwaving arguments about how these algorigthms manage
;;;;	to come up with the correct extract after the partials are
;;;;	combined. 
;;;;	
;;;;	collect primitive-references to rules/abs{compute}/stm{lemmas} in oid hash.
;;;;	
;;;;
(defvar *computed-hook* nil)
(defvar *primitive-hash* nil)

(defun partial-extract-aux (proof extract-p &optional metrics-p)
  (let ((i 0)
	(longest 0)
	(count 0)
	(children nil))
    
    ;; RLE perf could avoid calling primitive extract if could test judge if proof contributes.
    ;; (setf -p proof ) (break "pea")
    (labels ((compute-primitive-reference (dep kind)
	       (let ((oid (oid-of-dependency dep)))
		 ;;(format t "primref ~a~%" kind)
		 ;;(break "bar")
		 (let ((dep-kinds (hashoid-get *primitive-hash* oid)))
		   (if dep-kinds
		       (unless (member kind (cdr dep-kinds))
			 ;;(setf -dep dep -kind kind) (break "cpr")
			 (push kind (cdr dep-kinds)))
		       (hashoid-set *primitive-hash* oid
				    (cons dep (list kind)))))))
	     
	     (compute-primitives (term rule-id)
	       (let ((*computed-hook* #'(lambda (term)
					  (compute-primitive-reference
					   (dependency-of-definition
					    (abstraction-super-of-term term))
					   rule-id))))
		 ;; redo computation
		 (do-indicated-computations term)))

	     (compute-primitive-stm-reference (oid k)
	       ;;(setf -oid oid -k k) (break "cpsr")
	       (compute-primitive-reference (dependency-of-definition
					     (lookup-statement-def oid nil t))
					    k))

	     (mentions (term)
	       (term-walk term
			  #'(lambda (term)
			      (let ((abs (abstraction-of-term term)))
				(when abs
				  (compute-primitive-reference (dependency-of-definition abs)
							       'mentions)))
			      nil)))
	     
	     (abstraction-mentions (proof)
	       (dolist (assum (assumptions-of-proof-node proof))
		 (mentions (type-of-assumption assum)))
	       (mentions (conclusion-of-proof-node proof)))
	     
	     (visit (proof)
	       (let ((rule (rule-of-proof-node proof)))
		 (if (and (refined-rule-p rule)
			  (or (primitive-rule-p rule) (proof-rule-p rule)))
		     (progn
		       ;;(format t "~a~%" (id-of-rule rule))
		       ;;(setf -proof proof) (break "fu")
		       (when *primitive-hash*
			 (abstraction-mentions proof)

			 (let ((rule-dep (dependency-of-definition 
					  (definition-of-rule rule))))
			   (when rule-dep
			     (compute-primitive-reference rule-dep '|rule|)))
			  
			 ;;(format t "id-of-rule ~a~%" (id-of-rule rule))
			 (setf -rule rule)
			 (when (member (id-of-rule rule)
				       '(|lemma| |lemma_by_obid| |extract|
					 |direct_computation| |reverse_direct_computation| 
					 |direct_computation_hypothesis| |reverse_direct_computation_hypothesis|))
			   (setf -rule rule -proof proof) ;;(break "pea")
			   (cond
			     ((eql '|extract| (id-of-rule rule))
			      (compute-primitive-stm-reference
			       (lemma-lookup (atom-of-token-term (get-first-arg-value rule)))
			       '|extract|))
			     ((eql '|lemma| (id-of-rule rule))
			      (compute-primitive-stm-reference
			       (lemma-lookup (atom-of-token-term (get-first-arg-value rule)))
			       '|lemma|))
			     ((eql '|lemma_by_obid| (id-of-rule rule))
			      (compute-primitive-stm-reference
			       (first-oid-of-term (get-first-arg-value rule))
			       '|lemma_by_obid|))
			     ((eql '|direct_computation| (id-of-rule rule))
			      (compute-primitives (get-first-arg-value rule) (id-of-rule rule)))
			     ((eql '|reverse_direct_computation| (id-of-rule rule))
			      (compute-primitives (get-first-arg-value rule) (id-of-rule rule)))
			     ((eql '|direct_computation_hypothesis| (id-of-rule rule))
			      (compute-primitives (get-second-arg-value rule) (id-of-rule rule)))
			     ((eql '|reverse_direct_computation_hypothesis| (id-of-rule rule))
			      (compute-primitives (get-second-arg-value rule) (id-of-rule rule)))
			     (t (break "???")))))
			 
		       (when metrics-p
			 (incf count)
			 (setf longest (max longest
					    (length (assumptions-of-proof-node proof)))))
		       (if extract-p
			   (if (proof-rule-p rule)
			       ;; recurse on interior proof?
			       (proof-rule-to-extract proof
						      (mapcar #'visit (children-of-proof-node proof)))
			       (primitive-extract proof
						  (mapcar #'visit
							  (children-of-proof-node proof))))
			   (mapc #'visit (children-of-proof-node proof))))
		     (progn
		       (push proof children)
		       (when extract-p (iincomplete-term (incf i))))))))

      (let ((extract (visit proof)))
	(values (nreverse children)
		(when extract-p extract) count longest)))))

(defun extract-now (proof &optional incomplete-ok-p)
  (let ((i 0))
    ;; RLE perf could avoid calling primitive extract if could test judge if proof contributes.
    (labels ((visit (proof)
	       (let ((rule (rule-of-proof-node proof)))
		 
		 (if (not (refined-rule-p rule))

		     (if incomplete-ok-p
			 (iincomplete-term (incf i))
			 (raise-error (error-message '(extract incomplete))))

		     (let ((children (mapcar #'visit
				       (children-of-proof-node proof))))

		     (cond
		       ((primitive-rule-p rule)
			(primitive-extract proof children))

		       ((primitive-tree-rule-p rule)
			(extract-replace (extract-of-primitive-tree-rule rule) children))

		       ((proof-rule-p rule)
                        (proof-rule-to-extract proof children))
    
		       ((top-rule-p rule)
			(let ((ext (visit (proof-of-top-rule rule))))
			   (extract-replace ext children)))

		       (t (break "ext2") (raise-error (error-message '(extract unexpected rule) (id-of-rule rule))))

		       ))))))

      (visit proof)
      )))

(defunml (|extract| (p))
    (proof -> term)
  (extract-now p nil))

(defunml (|frontier_and_map| (p))
    (proof -> (proof |#| ((proof list) |#| (int list))))

  (mlet* (((newp l) (stick-frontier-of-proof p)))

	 (let ((i 0)
	       (unique nil)
               (somedupp nil))
           (cons newp
		 (cons l
		    (let ((m
		       (mapcar #'(lambda (pp)
				   (let ((ii (position pp unique :test #'equal-sequents-p)))
				     (if ii
                                         (progn (setf somedupp t) (1+ ii))
					 (progn
					   (setf unique (nconc unique (list pp)))
					   (incf i)))))
			       l)))
                         (if somedupp
			     m
                             nil)))))))


(defun note-primitive-refs (phash)
  (let ((refs nil))
    (maphash #'(lambda (stamp v)
		 (declare (ignore stamp))
		 (let ((dep (car v)))
		   (dolist (k (cdr v))
		     (let ((ka (assoc k refs)))
		       (if ka
			   (push dep (cdr ka))
			   (push (cons k (list dep)) refs))))))
	     phash)
    ;;(setf -refs refs) (break "npr2")
    (mapcar #'(lambda (k-deps)
		(new-dependencies (car k-deps) (cdr k-deps)))
	    refs)))


(defvar *primitive-refs* nil)

(defunml (|set_primitive_references| (b)) (bool -> unit)
  (setf *primitive-refs* b))


(defun partial-extract (proof extract-p &optional metrics-p)

  (labels ((visit (proof)
 	   ;;(setf -proof proof) (break "pe")

           (let ((rule (rule-of-proof-node proof)))
	     (if (or (primitive-rule-p rule)
                     (proof-rule-p rule))

		 (let ((p (copy-of-proof-node proof t)))
		   ;;(setf -proof proof) (break "pe")
		   (mlet* (((children extract count longest)
			    (partial-extract-aux proof extract-p metrics-p)))

			  (setf (proof-node-rule p) (primitive-tree-rule (when extract-p extract) count longest)
				(proof-node-children p) (mapcar #'visit children))
			  ;;(setf -c children -e extract -co count -lo longest) (break "pev")
			  p))
      
		 (let ((children (children-of-proof-node proof)))
		   (let ((nchildren (mapcar #'visit children)))
		     (if (apply-predicate-to-list-pair nchildren children #'eql)
			 proof
			 (let ((p (copy-of-proof-node proof t)))
			   (setf (proof-node-rule p) (rule-of-proof-node proof))
			   (setf (proof-node-children p) nchildren)
			   p))))))))

    ;;(setf -proof proof) (break "pe")

    (let ((*primitive-hash* (when *primitive-refs*
			      (make-hash-table :test #'equal :size 100))))
	 
      (let ((r (visit proof)))
	
	(when  *primitive-hash*
	  (let ((deps (note-primitive-refs *primitive-hash*))
		(odeps (proof-node-mark-value proof 'dependencies)))
	    (setf (environment-dependencies-list odeps)
		  (append deps (environment-dependencies-list odeps)))
	    ;;(setf -deps deps -odeps odeps -proof proof -r r) (break "pe")
	    ))
	  r))))
			
    

(defun refined-primitive-proof (proof)

  (case *primitive-extract-mode*
	
    (lazy 	proof)

    (eager	(partial-extract proof t t))

    (otherwise	(partial-extract proof nil t))))


#|(defun partial-extract (proof extract-p &optional metrics-p)
  (let ((i 0)
	(longest 0)
	(count 0)
	(children nil))
    ;; RLE perf could avoid calling primitve extract if could test judge if proof contributes.
    (labels ((visit (proof)
	       (let ((rule (rule-of-proof-node proof)))
		 (if (and (refined-rule-p rule) (primitive-rule-p rule))
		     (progn
		       (when metrics-p
			 (incf count)
			 (setf longest (max longest (length (assumptions-of-proof-node proof)))))
		       (if extract-p
			   (primitive-extract proof (mapcar #'visit (children-of-proof-node proof)))
			   (mapc #'visit (children-of-proof-node proof))))
		     (progn
		       (push proof children)
		       (when extract-p (iincomplete-term (incf i))))))))

      (let ((extract (visit proof)))
	(values (nreverse children) (when extract-p extract) count longest)))))

(defun refined-primitive-proof (proof)

  (case *primitive-extract-mode*
	
    (lazy 	proof)
	
    (eager	(let ((p (copy-of-proof-node proof t)))
		  (mlet* (((children extract count longest) (partial-extract proof t t)))
			 (setf (proof-node-rule p) (primitive-tree-rule extract count longest)
			       (proof-node-children p) children))
		  p))
	
    (otherwise	(let ((p (copy-of-proof-node proof t)))
		  (mlet* (((children extract count longest) (partial-extract proof nil t)
			   (declare (ignore extract))
			   ))
			 (setf (proof-node-rule p) (primitive-tree-rule nil count longest)
			       (proof-node-children p) children)
			 p)))))
|#
    



;;;
;;; 	other stuff.
;;; 

(defun make-frontier-validation (pf sequents)
  (flet ((rebuild-proof (pf proofs)
	   (labels
	       ((rebuild-proof-aux (pf)
		  (if (refined-proof-node-p pf)
		      (if (children-of-proof-node pf)
			  (let* ((children (children-of-proof-node pf))
				 (new-children (mapcar #'rebuild-proof-aux children)))
			    (if (forall-p #'eql children new-children)
				pf
				(let ((new-pf (copy-of-proof-node pf)))
				  (setf (proof-node-children new-pf) new-children)
				  new-pf)))
			   pf)
		      (pop proofs))))
	      (rebuild-proof-aux pf))))

    (make-closure
     #'(lambda (achievement)
	 ;; RLE ??? why not alpha-equal-sequents-p 
	 (if (apply-predicate-to-list-pair achievement sequents #'equal-sequents-p)
	     (progn
              (setf -npf (rebuild-proof pf achievement) -pf pf)
              (when (null -npf)
                  (break "mfvr"))
                  -npf)
	     (breakout evaluation '|frontier|)))
     1)))



(defunml (|frontier| (pf))
    tactic
  (let* ((sequents (frontier-of-proof pf)))
    (cons sequents (make-frontier-validation pf sequents))))


(defunml (|text_to_tactic| (term) :error-wrap-p nil)
    (term -> tactic)
  (text-to-tactic term))

