%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%
%[
******************************************************************************
******************************************************************************
AUTOTACTIC.ML
******************************************************************************
******************************************************************************
]%


%[
******************************************************************************
SupInf extras
******************************************************************************
]%

let OpenTypesWithArithInfo p = 
  let HTac i p = 
    let hinfo = find_supertypes p (h i p) in 
    let m = search_list 
       (\t,(),().member (opid_of_term t) types_with_arith_properties) hinfo
       ? failwith `OpenTypesWithArithInfo: no arith supertype found`
    in 
    let Tacs = map (\(),T,().T i) (firstn (m-1) hinfo)
    in 
    ( AddHiddenLabel `main`
      THENM SeqOnM Tacs
      THENM AddProperties i 
    ) p 
  in
  let CTac p = 
    let T',t,t' = dest_member_or_equal (concl p)
    in let T = unfold_soft_abs T'
    in if is_terms types_with_arith_properties T then
    ( MemTypeCD THEN IfLabL 
      [`set predicate`,AddHiddenLabel `main`
      ;`main`,AddHiddenLabel `aux`]
    ) p 
    else
      Fail p
  in
    Progress (TryOnAllMHyps HTac THENM Try CTac) p
;;


let SupInfAux options p = 
  if not member `si` options then failwith `SupInf not enabled` 
  if subset ``si_x si_at`` options then
  ( SupInfAux1 (remove `si_x` options)
    ORELSE 
    (AddHiddenLabel `main` 
     THEN Try OpenTypesWithArithInfo 
     THENM SupInfAux1 options
    )
  ) p
  if member `si_at` options then
  ( SupInfAux1 options
    ORELSE (OpenTypesWithArithInfo THENM (SupInfAux1 options))
  ) p
  if member `si_x` options then 
  ( SupInfAux1 (remove `si_x` options)
    ORELSE SupInfAux1 options
  ) p
  else
    SupInfAux1 options p
;;

let SupInf = SupInfAux [`si`] ;;

let SupInf' = SupInfAux ``si si_x``;;



%[
******************************************************************************
Additions to Auto tactic.
******************************************************************************
]%

% Desire additions managed by refenvs
  Desire use once and remove additions to prevent looping
   - pass list forward and remove if application is successful.
%

letref Auto_additions = [] : (tok # tactic) list ;;

let update_Auto_additions id T = 
 Auto_additions :=  update_alist Auto_additions id T ; ();;



let SqStableTac p =
  if is_term `sq_stable` (concl p) & not label_of_proof p = `squash stable?` then
  ( ProveSqStable
    ORELSE AddHiddenLabel `squash_stable?`
  ) p
  else
    Fail p
;;

let DecidableTac p =
  if is_term `decidable` (concl p) & not label_of_proof p = `decidable?` then
  ( ProveDecidable
    ORELSE AddHiddenLabel `decidable?`
  ) p
  else
    Fail p
;;

update_Auto_additions `SqStableTac` SqStableTac ;;
update_Auto_additions `DecidableTac` DecidableTac ;;
update_Auto_additions `ProveSubtyping` ProveSubtyping ;;

%[
*********************************************************************
Autotactic with monitoring and proof caching.
**********************************************************************
]%

%
Eq tried first to ensure that user-supplied typings as hyps 
override default typings. Probably should look at Eq to make sure
that works fast, since it's being called very often here.
%

let RepMemReflEqCD strength p =
  let MemEqCD = EqToMemberEq (\i.EqCDAux strength) 0 in
  let T,t,t' = dest_member_or_equal (concl p) in
  if alpha_equal_terms t t' then
    Progress (RepWith (\x.true) "=" ["Eq",Eq;"ME",MemEqCD]) p 
  else
    failwith `RepMemReflEqCD: equand opids not the same`
;;

%
The concl D's are restricted to the major logical connectives. We probably 
don't want to decompose other soft abstractions like nequal le gt not.
%

%[
Current Auto options:

`si` Run SupInf
`si_x` enable supinf option to infer xtra info from arith lemmas
       NB: use with caution, since can get divergent behaviour here.
`si_at` enable supinf option to use arithmetic info from non-arithmetic
        types (e.g. list(n))
`ar` run arith
`rec` eqcd on primitive recursive types
`ar_ty` decompose arith types
]%

let AutoWithOptions options p =
  let If opt T = if member opt options then T else Fail in 
  let strength = if member `rec` options then `strong` else `regular` in
  let ExtraTs = map snd Auto_additions 
  in
    RepWith (\x.true) "A"
    ["Tr",Trivial
    ;"HD",Progress (OnAllHyps RepeatAndHD)
    ;"CD",Progress WeakGenUnivCD
    ;"Eq",RepMemReflEqCD strength
    ;"Ar",If `ar` Arith
    ;"SI",SupInfAux options
    ;"Aq",RepeatEqCDForArith
    ;"AT",If `ar_ty` ArithMemberReflEqTypeCD
    ;"Ex",First ExtraTs
    ] p
;;


let AutoStepWithOptions options p =
  let If opt T = if member opt options then T else Fail in 
  let strength = if member `rec` options then `strong` else `regular` in
  let ExtraTs = map snd Auto_additions 
  in
   First
    [ Trivial
    ; Progress (OnAllHyps RepeatAndHD)
    ; Progress WeakGenUnivCD
    ; RepMemReflEqCD strength
    ; If `ar` Arith
    ; SupInfAux options
    ; RepeatEqCDForArith
    ; If `ar_ty` ArithMemberReflEqTypeCD
    ; First ExtraTs
    ] p
;;

let AutoStep p = AutoStepWithOptions ``ar ar_ty`` p;;




% 5/10/01 Variant of Auto which bounds the number of recursive calls to SupInf for Auto'
  and removes Monitor overhead. 

  TODO provide monitor variant which can be activated by compilation.
%

letref AutoBound = 5;;

let WithBound n T p =
  let n' = (get_int_arg `bound` p) %- 1% ? n in
  %(if n' < n then tty_print ("recursive WithBound " J (int_to_string n) J " " J (int_to_string n')));%
  (WithArgs [`bound`, int_to_arg n'] T p);;
 
let DecrementBoundWCutOff cutoff PreT T p =
  let n =  get_int_arg `bound` p in
    %tty_print ("WithBound Decrement " J (int_to_string n));%
  if n < 1 then 
    (tty_print ""; tty_print ("WithBound Decrement Fail " J (int_to_string n)); Fail p)
  else 
    let p' = update_current_args_entry `bound` (int_to_arg (n-1)) p in 
    (if (n < cutoff) then (PreT THEN T)
     else T) p'
;;
   
let WithCaching id T p = 
  if proof_cache_kind = `basic` then
    ((ApplyPfCache id) ORELSE
     (UpdatingMLPfCache_aux false false (thm_being_refined ()) id T)) p
  else T p
;;

letrec RepeatLabWithCaching id T p =
  (let Aux = (LabProgress T) THEN (RepeatLabWithCaching id T) in
   WithCaching id Aux ORELSE Id) p
;;

let CleanAutoWithOptions options p =
  let If opt T = if member opt options then T else Fail in
  let strength = if member `rec` options then `strong` else `regular` in
  let ExtraTs = map snd Auto_additions in
 
  WithBound AutoBound
  (RepeatLabWithCaching "A"
  (First
     [Trivial
     ; Progress (OnAllHyps RepeatAndHD)
     ; Progress WeakGenUnivCD
     ; RepMemReflEqCD strength
     ; If `ar` Arith
     ; DecrementBoundWCutOff 3 (AllHyps Thin) (SupInfAux options)
     ; RepeatEqCDForArith
     ; If `ar_ty` ArithMemberReflEqTypeCD
     ; First ExtraTs
     ]))
  p
;;



let sTHEN_onEach_aux (g : * -> * list -> **) 
                 (sT: (* -> proof -> (* # (proof list) # ((proof list) -> proof))))
		 (sF : proof list -> (* -> proof -> (* # (proof list) # ((proof list) -> proof))) list)
		 (s: *)
		 (p:proof) =

  let ns,pl,v = sT s p  in
  let sTs = sF pl in
  if not length sTs = length pl then
    failwith `sTHEN_OnEach_aux: Wrong number of tactics`
  else
  let nsl,pll,vl = unzip3 (map2 (\sT p. sT ns p) sTs pl)
  in
    g ns nsl, flatten pll,  (v o mapshape (map length pll) vl)
;;

let sTHEN_onEach (sT: (* -> proof -> (* # (proof list) # ((proof list) -> proof))))
		 (sF : proof list -> (* -> tactic) list)
		 (s: *)
		 (p:proof) =

  let ns,pl,v = sT s p in
  let Ts = sF pl in
  if not length Ts = length pl then
    failwith `THEN_OnEach: Wrong number of tactics`
  else
  let pll,vl = unzip (map2 (\T p. (T ns p)) Ts pl)
  in flatten pll,  (v o mapshape (map length pll) vl)
;;

% g : * -> * list -> *
   used to reduce list of results of second stactic.
%
let sTHEN_aux g sT1 sT2 s = sTHEN_onEach_aux g sT1 (\ps.replicate sT2 (length ps)) s ;;
let sTHEN sT1 T2 s p = sTHEN_onEach sT1  (\ps.replicate T2 (length ps)) s p;;

let sLabProgress (stac: (* -> proof -> (* # (proof list) # ((proof list) -> proof))))
                 s (p : proof) =
  let ns,ps,v = stac s p in
    if length ps = 1 then
      let p' = (hd ps) in
        if equal_sequents p' p & label_of_proof p' = label_of_proof p
	  then failwith `LabProgress`
          else ns,ps,v
    else ns,ps,v
;;

let sRepeatLabWithCaching id sT s p =
 letrec auxr s = WithCaching id
                   (sTHEN (sLabProgress sT) auxr s)
                 ORELSE Id in
  auxr s p
;;
  


let sDecrementBoundWCutOff cutoff PreT T (n,r) p =
  if n < 1 then 
    (tty_print ""; tty_print ("WithBound Decrement Fail " J (int_to_string n)); 
     (n,r),Fail p)
  else
   ((n - 1,r),
    (if (n < cutoff) then (PreT THEN T) p
     else T p))
;;
 
% fst of s is bound on supinf
  rest is alist of bounded additions.
%
%
let CleanAutoWithOptions_forward options s p =
  let If opt T = if member opt options then T else Fail in
  let strength = if member `rec` options then `strong` else `regular` in
  let ExtraTs = map snd Auto_additions in
 
  WithBound AutoBound
  (RepeatLabWithCaching "A"
  (First
     [Trivial
     ; Progress (OnAllHyps RepeatAndHD)
x     ; Progress WeakGenUnivCD
     ; RepMemReflEqCD strength
     ; If `ar` Arith
     ; sDecrementBoundWCutOff 3 (AllHyps Thin) (SupInfAux options)
     ; RepeatEqCDForArith
     ; If `ar_ty` ArithMemberReflEqTypeCD
     ; First ExtraTs
     ]))
  p
;;
%
%
letref max_si_depth = 0;;
letref saved_max_si_depth = 0;;
letref supinf_called = false;;
let update_si_depth n =
  if max_si_depth < n then
     (max_si_depth := n; tty_print ("Max si depth " J (int_to_string n)); ())
  else ()
;;
let SetSupInfCalled p =
  supinf_called := true; Id p
;;
let ReSetSupInfCalled p =
  supinf_called := false; Id p
;;

letrec RepeatLabWithCaching_old id n T p =
 let Aux q =
  if 5 < n then (max_si_depth := saved_max_si_depth; fail) 
  else
  let pl,v = supinf_called:= false; LabProgress T q in
  let n' = if supinf_called then (update_si_depth n ; n+1) else n in
  let Ts = replicate (RepeatLabWithCaching_old id n' T) (length pl) in
  let pll,vl = unzip (map2 ap Ts pl) in
    flatten pll, (v o mapshape (map length pll) vl)
  in 
    (WithCaching id Aux ORELSE Id) p
;;

let CleanAutoWithOptions_old options p =
  let If opt T = if member opt options then T else Fail in
  let strength = if member `rec` options then `strong` else `regular` in
  let ExtraTs = map snd Auto_additions in
  saved_max_si_depth := max_si_depth;
 
  (RepeatLabWithCaching_old "A" 5
  (First
     [Trivial
     ; Progress (OnAllHyps RepeatAndHD)
     ; Progress WeakGenUnivCD
     ; RepMemReflEqCD strength
     ; If `ar` Arith
     ; (SupInfAux options) THEN SetSupInfCalled
     ; RepeatEqCDForArith
     ; If `ar_ty` ArithMemberReflEqTypeCD
     ; First ExtraTs
     ]))
  p
;;
%
