%[
****************************************************************************
****************************************************************************
INC-TACTICS-3.ML
****************************************************************************
****************************************************************************

]%


%[
**********************************************************************
Functions for Supertype reasoning.
**********************************************************************
For tackling goals:

  |- Subtype(S;T)

where S and T have some supertypes S' and T' which have the same
outermost type constructor. 

The current strategy picks the closest super-types which are
not both set types, and on which some other Inclusion option can make
progress.

For each type A you wish to be recognized as a subtype,
you should add an ML object shortly after the type's definition with 
the following expression:

add_set_inclusion_info
  (opidA : tok)
  (opidB : tok)
  (DecompTac : int->tactic)   (Usually AbSetDForInc)
  (ProveTac : tactic)         (Usually Auto)
;;

opidA is the opid of type A. 
opidB is the opid of the supertype.

DecompTac should decompose the type A  in either the 
conclusion or hypothesis to the type B. DecompTac 0 should add the label
`set predicate` to each subgoal predicate.

ProveTac should be a tactic powerful enough to prove these set predicate
subgoals.

Multiple levels are handled properly. ie. you can declare A to be a subtype
of B and B to be a subtype of C. Each type should have at most one declared
supertype.
In the event of there being multiple supertypes, the smallest one is picked.


QUESTION:
Is it always going to be desirable to have the supertype be `1 level' up? If
so, we can assume that DecompTac is always the same, and figure out what
opidB is from the abstraction for opidA.

Probably is useful having ProveTac supplied explicitly, since
predicate proving could get quite specialized.

These functions build on ones defined in type-inc.ml
[%




%[
-----------------------------------------------------------------------------
Widening Types
-----------------------------------------------------------------------------
... i. x:S ... H[x] ... |- C[x]

   By WidenDecl T i 

`main`   ... i. x:T ... H[x] ... |- C[x]
`subtype`... i. x:S ... H[x] ... |- S c= T
(`wf`    several)

The way things are currently set up, the x:S and dependent hyps H[x] will 
get bunched up at the end of the hyp list
]%

% Assumes no dep hyps %

let WidenDecl1 T i p = 
  let i' = get_pos_hyp_num i p in
  let x = var_of_hyp i' p in
  let C = concl p in
  let all_xT_C = mk_all_term x T C in
  
  ( Assert all_xT_C THEN IfLabL
    [`assertion`,Thin i' THEN D 0
    ;`main`
     % x:S ... All x:T. Cx |- Cx %
     ,With (mvt x) (D (-1)) THENM Trivial 
     % x:S ... |- x in T %
      THEN MoveToConcl i' THEN FoldTop `subtype` 0
      THEN AddHiddenLabel `subtype`
    ]
  ) p
;;


% Widening and retaining dependent hyps is not always good idea because
they might not be well-formed for widened case. If they are not, the moving 
back of the hyps creates unprovable wf goals. 
%
%
let WidenDecl T i = MoveEndHypsToConclFor (WidenDecl1 T) i ;;
%

let WidenDecl T i = ThinBeyond i THEN WidenDecl1 T i ;;


%[
-----------------------------------------------------------------------------
Supertyping by soft abstraction.
-----------------------------------------------------------------------------
]%

let soft_ab_supertype_fun t (p:proof) = 
  if is_soft_ab t then 
    unfold_soft_ab t
    ,(\i.if i = 0 then UnfoldAtAddr [1] i else UnfoldTopAb i)
    ,`equal`
  else
    failwith `soft_ab_supertype`
;;

update_supertype_funs `soft_ab` soft_ab_supertype_fun ;;
 
%[
-----------------------------------------------------------------------------
Supertyping by redex contraction
-----------------------------------------------------------------------------
]%

let ab_redex_supertype_fun t (p:proof) = 
  if is_ab_redex t then
     apply_conv (RedexC ORELSEC AbRedexC) t
     ,AbReduceTypeForInc 
     ,`equal`
  else
    failwith `soft_ab_supertype`
;;
update_supertype_funs `ab_redex` ab_redex_supertype_fun ;;

%[
-----------------------------------------------------------------------------
Supertyping by hyp supertype assumption
-----------------------------------------------------------------------------
Looking for supertype of type S and have assumption of form one of:
 1. Subtype(S;T) 
 2. Suptype(T;S)

Warning: if typing expression is reflexive, then this causes looping.
]%

% look for hyp  subtype(t;s) or suptype(s;t) and return # of hyp
  & `_hyp_subtype`,s %

let find_suptyping_hyp p t = 
  let f (i,A) = 
    if is_terms ``subtype suptype`` A & alpha_equal_terms (sub_term 1 A) (sub_term 2 A) then
      fail
    if is_term `subtype` A & alpha_equal_terms t (sub_term 1 A) then
       i,sub_term 2 A
    if is_term `suptype` A & alpha_equal_terms t (sub_term 2 A) then
       i,sub_term 1 A
    else
      fail
  in
    first_value f (number (hs p)) ? failwith `find_suptyping_hyp`
;;
  

let WidenUsingHypSuptyping i p = 
  if i = 0 then failwith `WidenUsingHypSuptyping: cannot widen concl` 
  else
  let j = fst (find_suptyping_hyp p (h i p)) in
  let R = h j p in
  let T = if is_term `subtype` R then sub_term 2 R else sub_term 1 R 
  in
  (WidenDecl T i  THEN IfLabL [`subtype`,SoftNthHyp j]) p 
;;
    

let hyp_assum_supertype_fun t p = 
  let superT = snd (find_suptyping_hyp p t) 
  in
     superT,WidenUsingHypSuptyping,`maybe proper`
;;


update_supertype_funs `hyp_assum` hyp_assum_supertype_fun ;;

%[
-----------------------------------------------------------------------------
Supertyping by set_inc_info 
-----------------------------------------------------------------------------
]%

%
letref set_inc_info = [] : (tok # tok) list;;
letref set_inc_tactics = [] : (tok # (int->tactic) # tactic) list;;

let add_set_inclusion_info ida idb DTac PTac = 
  set_inc_tactics:=
    update_alist set_inc_tactics ida (DTac,PTac)
  ;
  set_inc_info :=
    update_alist set_inc_info ida idb
  ;
  ()
;;
%

letref set_inc_ref_state =
  new_alist_ref_state `set_inc_alist`
    (nil : (tok # tok # (int->tactic) # tactic) list)
;;

update_ref_state_view
 (\(). set_inc_ref_state)
 (ref_state_view_list_entry (\e. icons_term (itoken_term (fst e))
				            (itoken_term (fst (snd e)))))
;;     

let ref_add_set_inc index edges items =
 declare_ref_state_index index `set_inc_alist` index;
 set_inc_ref_state
   := declare_ref_state_data_indirect `set_inc_alist` set_inc_ref_state index items edges
;;
let ref_add_set_inc_aux index edges items =
 set_inc_ref_state
   := declare_ref_state_data_indirect `set_inc_alist` set_inc_ref_state index items edges
;;

let ref_add_set_inc_additions items index edges =
  set_inc_ref_state
    := ref_state_modify_state_aux (\data gedata. append gedata data)
          set_inc_ref_state items index edges
;;

let set_inc_add_data oid data = 
 set_inc_ref_state := ref_state_set_data set_inc_ref_state [oid, data]
;;

let SetInclusionInfo_add oid data =
 reset_ref_environment_data oid;
 add_ref_environment_data oid `set_inc_alist` set_inc_add_data data
;;

let SetInclusionInfo_add_o oid data =
 reset_ref_environment_data oid;
 add_ref_environment_data oid `set_inc_alist` set_inc_add_data
   (map (\oid, opid, f, g. id_of_term (fst (snd (abstraction_lookup oid))), opid, f, g) data)
;;

let set_inc_do_updates oid edges oids = 
  set_inc_ref_state := ref_state_do_updates set_inc_ref_state oid oids edges
 ; ()
;;
let undeclare_set_inc oid = 
 (set_inc_ref_state := ref_state_remove set_inc_ref_state oid; ())
 ? ()
;;

let lookup_set_inc_tactics id = 
  snd (apply_alist (ref_state_get set_inc_ref_state (current_ref_environment_index `set_inc_alist`)) id)
;;
let lookup_set_inc_info id = 
  fst (apply_alist (ref_state_get set_inc_ref_state (current_ref_environment_index `set_inc_alist`)) id)
;;



let SetIncInfoTac id i p = 
  let DTac,PTac = lookup_set_inc_tactics id
  in
  if i = 0 then
    ( DTac 0
      THEN IfLabL 
       [`set predicate`,
        Complete PTac 
        ORELSE AddHiddenLabel `unproved set predicate from Inclusion`
       ]
    ) p
  else 
   DTac i p
;;

let inc_info_supertype_fun t (p:proof) = 
  let ida = opid_of_term t in
  let idb = lookup_set_inc_info ida 
            ? failwith `inc_info_supertype_fun`
  in
  let t' = 
  ( if member idb ``1 2 3`` then
      nth (position idb ``1 2 3``) (subterms t) 
    else
     (repeatf unfold_ab thenf (fst o snd o dest_set)) t
  ) 
  in
    t'
    ,SetIncInfoTac ida
    ,`maybe proper`
;;

update_supertype_funs `inc_info` inc_info_supertype_fun ;;

%[
-----------------------------------------------------------------------------
Instances of inc_info tactics
-----------------------------------------------------------------------------
]%
  
let PrimSetDForInc i = 
  if i = 0 then 
    SetEqTypeCD 
  else 
    BasicSetHD i
;;

let AbSetDForInc i = 
  if i = 0 then
    AbSetEqTypeCD
  else
    BasicAbSetHD i
;;



% updated in autotactic.ml %

letref InclusionAuto = Id ;;
%
add_set_inclusion_info `set` `1` PrimSetDForInc (\p.InclusionAuto p);;
add_set_inclusion_info `unique_set` `1` AbSetDForInc (\p.InclusionAuto p);;
%

%[
-----------------------------------------------------------------------------
Supertyping by power type declaration
-----------------------------------------------------------------------------
Only good for hypotheses.
  
Opening Power type elements
---------------------------


... S: c= T ... i. x:S ... |- C[x]

WidenPowerTyEl i

  ... S: c= T ... i. x:T ... |- C[x]


or more generally

... S: x:A->y:B-> (c= Txy) ... i. x:(S a b)... |- C[x]

WidenPowerTyEl i

  ... S: x:A->y:B-> (c= Txy)  ... i. x:(Tab) ... |- C[x]

for arbitrary # of curried arguments.

Tactic assumes it is presented with above structure 
]%

% PostC: in main goal 'Sas c= Tas' is inserted before hyp i %


let find_var_type p v =
  (type_of_declaration (find (\h. v = var_of_declaration h) (hyps p)))
    %apply_alist (dest_hyps p) v %
  ? failwith `find_var_type`
;;


let WidenPowerTyElSetup i p = 
  let i' = get_pos_hyp_num i p in
  let Sas = h i' p in
  let S.as = dest_iterated_apply Sas in
  let Sv = dest_var S in
  if null as then
  ( let j = get_decl_num Sv p in
      AddProperties j THENM MoveToHyp (i'+1) (j+1)
  ) p
  else
  ( let F = find_var_type p Sv in 
    let xAs,subTxs = iterate_dest_quantifier dest_function F in 
    let subTas = subst (zip (map fst xAs) as) subTxs in
    ( Assert (mk_member_term subTas Sas) THEN IfLabL
      [`assertion`,AddHiddenLabel `wf`
      ;`main`,AddProperties (-1) THENM MoveToHyp i' (-1) THENM Thin (-1)]
    ) p
  ) 
;;

let WidenPowerTyEl i p = 
  let i' = get_pos_hyp_num i p in
  let Widen p = WidenDecl (sub_term 2 (h i' p)) (i' + 1) p
  in
  ( WidenPowerTyElSetup i' 
    THENM Widen
    THEN IfLabL [`main`,Thin i' ;`subtype`,NthHyp i']
  ) p
;;



let power_ty_supertype_fun t p = 
  let f.as = dest_iterated_apply t in
  let v = dest_var f ? failwith `power_ty_supertype_fun` in
  let T = find_var_type p v in 
  let xAs,B = iterate_dest_quantifier dest_function T in 
  if not is_term `power_ty` B then failwith `power_ty_supertype_fun`
  else

  % NB: since null vars never occur free, bindings for them will be ignored %

  let t' = subst (zip (map fst xAs) as) (sub_term 1 B) 
  in
    t'
    ,WidenPowerTyEl
    ,`maybe proper`  
;;

update_supertype_funs `power_ty` power_ty_supertype_fun ;;

%[
-----------------------------------------------------------------------------
Local Type Definition Supertyping
-----------------------------------------------------------------------------

A `Local type definition' is a definition for a type or a family of types
that is introduced locally into the context of a proof by a new variable and
an equality giving the expansion of the variable.  

Would be interesting to explore introducing such definitions by using 
an untyped equality: e.g. the `squiggle' relation.


...T:U, T = S in U, ... i. x:T ... |- ...

BY UnfoldLocalTypeDefn i 

  ...T:U, T = S in U, ... i. x:S ... |- ...
 
or in general:

...T:A->B->U, T = \x,y.Sxy in U, ... i. x:(T a b) ... |- ...

BY UnfoldLocalTypeDefn i 

  ...T:A->B->U, T = \x,y.Sxy in U, ... i. x:Sab ... |- ...

We allow for lambdas possibly being wrapped in tlambda abstractions...
]%


let UnfoldLocalTypeDefn i p = 
  let T = hd (dest_iterated_apply (h i p)) in
  let f A = 
    (let U,L,R = dest_equal A in L = T) ? false
  in
  let j = search_list f (hs p) 
  in
  (  HypSubst j i
     THENM (Try (Unfold `tlambda` i) THEN Reduce i)
  ) p
;;



let find_var_defn p v = 
  let V = mk_var_term v in 
  let f A = 
    let T,a,b = dest_equal A in 
    if alpha_equal_terms a V then b else fail
  in
    first_value f (hs p) ? failwith `find_var_defn`
;;
  
let local_typedef_supertype_fun t p = 
  let f.as = dest_iterated_apply t in
  let v = dest_var f ? failwith `local_typedef_supertype_fun` in
  let e = find_var_defn p v in

  % compute is probably too strong here. Will unfold other abstractions too%
  % e might contain tlambdas %

  let t' = compute (mk_iterated_apply (e.as))
  in
    t'
    ,UnfoldLocalTypeDefn
    ,`equal`
;;

update_supertype_funs `local_typedef` local_typedef_supertype_fun ;;

%[
-----------------------------------------------------------------------------
The Supertype Inclusion Tactic
-----------------------------------------------------------------------------
]%


% 
If as and bs have common start (not set) then make 1 step
of progress. This is needed when there is no inclusion lemma
for both types at start, yet there is one for some common supertype.

Otherwise, return prefixes of as and bs sufficient to 
get to common (non set) super-type

On entry: 
    1. both length > 0 and at least one length > 1

If simple_p is true, then check that all transformations on type
are equality transformations
%

let choose_inc_tactics simple_p as bs = 
  let atms,aTks = unzip as in
  let btms,bTks = unzip bs in
  let aTs,() = unzip aTks in
  let bTs,() = unzip bTks in

  let na,nb = find_common_supertype_depths atms btms in
  let aTks',bTks' = 
    if na = 1 & nb = 1 then
     [hd aTks], [hd bTks]
    else
      firstn (na-1) aTks, firstn (nb-1) bTks
  in
  if not simple_p
     or all (\T,k.k = `equal`) aTks' & all (\T,k.k = `equal`) bTks' 
  then
    map fst aTks',map fst bTks'
  else
    failwith `choose_inc_tactics: could not find simple inc`
;;

   
let AbSetSubtype p =
  %  ... >> Subtype(A;B) ...%
  let i,A,B = get_subtype_args p in 
  let i' = i = -1 => num_hyps p + 1 | i in
  let hinfo = find_supertypes p A in 
  let cinfo = find_supertypes p B in

  if null (tl hinfo) & null (tl cinfo) then
    failwith `AbSetSubtype: no progress possible`
  else
  let HypTacs,ConclTacs = 
    choose_inc_tactics (is_simple_inclusion_goal p) hinfo cinfo 
  in
  let Seq Ts i = SeqOnM (map (\T.T i) Ts) 
  in 
  ( SubtypeCD
    THENM Seq HypTacs i'
    THENM Seq ConclTacs 0
    THENM AddHiddenLabel `subterm`
  )  p
;;



%[
-----------------------------------------------------------------------------
Setting Subtype_additions 
-----------------------------------------------------------------------------
]%

% should fix later ones to not use IncToSubtype. 
  Only causes extra wf headaches when left type moved back to 
  hyps! 


Ordering here matters a little, since more than one can 
make progress on a given goal. In particular, the last
two could both work where either of the first two apply.

2/24/95: Does order of last two matter?
Found case in thm restrict_perm_using_txpose  in perms_1 theory in which had
to put Arith earlier (see below for original)
%

Subtype_additions := 
 [`SubtypeAbReduce` ,SubtypeAbReduce
 ;`SubtypeArithSimp`,IncToSubtype THEN SubtypeArithSimp
 ;`AbSetSubtype`, AbSetSubtype
 ;`DecomposeSubtype`,If is_simple_inclusion_goal
                        Fail                    
                        (IncToSubtype THEN DecomposeSubtype)
 ;`Hypothesis`, (OnSomeHyp SoftNthHyp)
 ] 
;;

% Used to be: %
%
Subtype_additions := 
 [`SubtypeAbReduce` ,SubtypeAbReduce
 ;`AbSetSubtype`, AbSetSubtype
 ;`DecomposeSubtype`,If is_simple_inclusion_goal
                        Fail                    
                        (IncToSubtype THEN DecomposeSubtype)
 ;`SubtypeArithSimp`,IncToSubtype THEN SubtypeArithSimp
 ] 
;;

%        
