%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%


%[
**********************************************************************
Integer Induction
**********************************************************************
These tactics will probably be made obselete by the ones in the 
following sections.
]%

%[

NonNegIntInd (k:term) (i:int) 

 Assumes that sequent is of form:

... #i:n:Int ... >> LowerBound(k,n) => E[n]

where LowerBound(k,n) <=> (k =< n),
where k is an expression such that k >= 0. is provable and
where variable n does not occur in any hypothesis terms.

If all goes well we get the following subgoals:
`basecase`: ... ... k = 0 >> E[0]            
`basecase`: ... ... n:Int,0 < n,k = n >> E[n]
`upcase`  : ... ... n:Int, k < n, E[n-1] >> E[n] 

We try running Arith on the two base cases, so if k is a constant or k > 0
at least one of them will always be knocked off.

Do we want to somehow merge these if k >= 0?
What about doing substitution for k in basecase when n = k? 

Arithmetic subgoals which should be trivially solvable but which Arith misses
are labelled `induction arith`. Various `wf` subgoals will also be 
generated.
]%

% ArithTac goals below should always be solvable %

let NonNegIntInd k i p =
 (let ArithTac = Arith ORELSE AddHiddenLabel `induction arith` in
  let i' = get_pos_hyp_num i p in
  let n = var_of_hyp i' p in
  let j = maybe_new_proof_var (mkv `zzj`) p in
  let j_tm = mk_var_term j in
  let zero = mk_integer_term 0 in
  IntInduction j i
  THEN
  IfLabL
  [% ... #i:n:Int ... j:Int, j < 0, ((j+1 >= k) => E[j+1]) >> (j>=k)=>E[j]%
   `downcase`, D 0 THENM ArithTac 
  ;% ... #i:n:Int ... >> (0>=k)=>E[0]%
   `basecase`, 
    D 0 
    THENM (DebugTry (Decide (mk_equal_term int_term k zero))
           THENLL 
           [`main`,
            [% ... #i:n:Int ...,(k=<0), k = 0 >> E[0]%
             Arith 
             ORELSE 
              (OnHyps ([-2;i'] @ if (alpha_equal_terms k zero) then [-1] else []) Thin 
               THEN AddHiddenLabel `basecase`
              )
             % ... ... k = 0 >> E[0]%
            ;% ... #i:n:Int ...,(k=<0), not(k = 0) >> E[0]%
             ArithTac
            ]
           ]
          )
  ;% ... #i:n:Int ... j:Int, 0 < j, ((j-1 >= k) => E[j-1]) >> (j>=k)=>E[j]%
   `upcase`,
   D 0 
   THENM
   % ... #i:n:Int ... j:Int, 0 < j, ((j-1 >= k) => E[j-1]), (j>=k) >> E[j]%
   ( DebugTry (Decide (mk_equal_term int_term k j_tm) )
     THENLL 
     [`main`,
      [%.#i:n:Int.. j:Int,0 < j,((j-1 >= k) => E[j-1]), (j>=k),k = j >> E[j]%
       Arith 
       ORELSE (OnHyps [-2;-2;i'] Thin
               THEN RenameVar n (-3)
               THEN AddHiddenLabel `basecase`
       %... ... n:Int,0 < n,k = n >> E[n]%
              )
      ;%.#i:n:Int.. j:Int,0 < j,((j-1 >= k)=>E[j-1]),(j>=k),not(k=j) >> E[j]%
       D (-3)
       THEN IfLabL
       [%.#i:n:Int.. j:Int,0 < j, (j>=k),not(k = j) >> j-1 >= k%
        `antecedent`,ArithTac
       ;%.#i:n:Int.. j:Int,0 < j, (j>=k),not(k = j), E[j-1] >> E[j] %
        `main`, OnHyps [-4;i'] Thin 
                THEN RenameVar n (-4)
                THEN AssertAtHyp (-1) (mk_less_than_term k (mk_var_term n))
                THEN IfLabL
                [`assertion`,ArithTac
                ;`main`,OnHyps [-3;-3] Thin THEN AddHiddenLabel `upcase`
                 % ... ... n:Int, k < n, E[n-1] >> E[n] %
                ]
       ]
      ]
     ]
   )
  ]
 ) p

;;


%[ 
induction over range a .. b with base case at a 
 Assumes that sequent is of form:

... #i:n:Int ... >> (a=<n) => (n=<b) => E[n]
(or more generally)
... #i:n:Int ... >> LowerBound(a,n) => UpperBound(b,n) => E[n]

]%

let NSubrangeInd a i p =
  %... #i:n:Int ... >> (a=<n) => (n=<b) => E[n]%
 (NonNegIntInd a i
  THEN IfLabL
  [
   `basecase`,
   %... #i:n:Int ... ... >> (n=<b) => E[n]%
   D 0 THENM (Thin (-1) THEN AddHiddenLabel `basecase`)
   %... #i:n:Int ... ... >> E[n]%
  ;`upcase`,
   %... #i:n:Int ... (n-1=<b) => E[n-1] >> (n=<b) => E[n]%
   D 0 THENM 
   %... #i:n:Int ... (n-1=<b) => E[n-1], (n=<b) >> E[n]%
   (D (-2) 
     THEN IfLabL 
       [`assertion`,% Arith ORELSE % AddHiddenLabel `induction arith`
       ;`main`,AddHiddenLabel `upcase`
        %... #i:n:Int ... (n=<b), E[n-1] >> E[n]%
       ]
   )
  ]
 ) p
;;

%[
Do induction over a subset of N (perhaps N itself...)
Handle extra predicate for upper bound...

Seems that # of cases we will want to handle is finite and small,
so OK to compile in info about particular subsets of integers.

Should have properties lemmas proved for these types so that AbSetHD is
pretty quick.

This easily generates MANY wf goals. e.g. when used to prove nth_tl_aux_wf
it generated 25 wf subgoals as well as 1 arithmetic subgoal, and the
2 desired subgoals! We should see if we can efficiently code up the effect
of this tactic in lemmas...
]%

let NSubsetInd i p =
  let i' = get_pos_hyp_num i p in
  let T = h i' p in
  let ind_type = opid_of_term T in
  let lower_bound,exists_upper_bound =
  ( if ind_type = `nat` then mk_integer_term 0,false
    if ind_type = `nat_plus` then mk_integer_term 1,false
    if ind_type = `int_upper` then hd (subterms T),false
    if ind_type = `int_seg` then hd (subterms T),true
    else failwith `NSubsetInd: unfamiliar type`
  )
  in
  ( if exists_upper_bound then
      AbSetHD i' 
      THENM
      ( D (i'+1)
        THEN MoveDepHypsToConcl i' 
        THEN NSubrangeInd lower_bound i'
      )
    else
      AbSetHD i' 
      THENM
      ( MoveDepHypsToConcl i' 
        THEN NonNegIntInd lower_bound i'
      )
  ) p
;;
  

let IntInd i p =
  let i' = get_pos_hyp_num i p in
  let n = var_of_hyp i' p in
  let j = maybe_new_proof_var (mkv `zzj`) p 
  in
  ( MoveDepHypsToConcl i'
    THEN IntInduction j i'
    THEN
    KeepingAnnotation
    ( IfLabL
      [`basecase`, Thin i'
      ;`downcase`, Thin i' THEN RenameVar n (-3)
      ;`upcase`, Thin i' THEN RenameVar n (-3)
      ]
    )
  ) p
;;


let ListInd i p = 
  let i' = get_pos_hyp_num i p 
  in
  ( MoveDepHypsToConcl i' THEN 
    ListInduction 
      (get_optional_var_arg `v1` p)
      (get_optional_var_arg `v2` p)
      i' 
  ) p
;;

% 
Thins the useless original declaration left by rule.
Has smarter induction variable choice.

Often give names to vars of <root>s. Then appropriate
to use the names <root> and <root>s for new variables.

Have to watch clash problem with original name.
%

let ListIndA i p = 
  let i' = get_pos_hyp_num i p in
  let strip_s v = 
    let vtoks = explode (var_to_tok v) in
    if last vtoks = `s` & length vtoks > 1 then 
      tok_to_var (implode (remove_last vtoks))
    else
      v
  in
  let oldv = var_of_hyp i' p in
  let hdv = get_var_arg `v1` p ? strip_s oldv in
  let tlv = get_var_arg `v2` p ? oldv in

  ( MoveDepHypsToConcl i' THEN 
    RenameVar (tok_to_var `****`)  i' THEN
    ListInduction hdv tlv i'
    THEN Thin i'
  ) p
;;




%[
**********************************************************************
Integer Induction
**********************************************************************
See int_1 theory for equivalent tactics defined by analogy
that take care of induction over naturals in a much cleaner way.

We provide some beefing up of the basic induction principles.
]%

% 
NInd
~~~~
Induction over the naturals. This tactic generates 5 wf/arith subgoals,
as oppose to the 11 which NSubsetInd generates.

... #i:n:N ... F[n] ... >> G[n] 

BY NInd #i

`basecase`: ... ... >> F[n] => G[n]
`upcase`: ... ... n:Z, 0 < n, F[n-1] => G[n-1]  >> F[n] => G[n]

`wf`    : 3 X ... >> p <= q for p,q either 0 or n.
`arith` : 2 X Simple subgoals for Arith.
%

let NInd i p =
  let i' = get_pos_hyp_num i p in
  if not opid_of_term (h i' p) = `nat` then
    failwith `NInd: induction must be over nat type`
  else
  % ... #i:n:N ... F[n] ... >> G[n] %
 (AbSetHD i'
  % ... #i:n:Int, n>=0, ... F[n] ... >> G[n] %
  THENM 
  ( IntInd i'
    THEN
    IfLabL
    [% let E[n] =def F[n] => G[n] %
     % ... ... n:Int, n < 0, ((n+1 >= 0) => E[n+1]) >> (n>=0)=>E[n]%
     `downcase`, D 0 THENM AddHiddenLabel `arith`
    ;% ... ... >> (0>=0)=>E[0]%
     `basecase`, D 0 THENM (Thin (-1) THEN AddHiddenLabel `basecase`)
    ;% ... ... n:Int, 0 < n, ((n-1 >= 0) => E[n-1]) >> (n>=0)=>E[n]%
    `upcase`, 
     D 0 THENM
       (D (-2) 
        THEN 
        IfLabL
        [`antecedent`,AddHiddenLabel `arith`
        ;`main`,Thin (-2) THEN AddHiddenLabel `upcase`
        ]
       )
    ]
  )
 ) p
;;

let NCompIndSetup i =
  MoveDepHypsToConcl i 
  % ... #i:n:N ... >> P[n] %
  THENM
  \p.
  let z = get_distinct_var (mkv `zz`) p in
  let n,N = dest_hyp i p in
  let n_term = mk_var_term n in
  let Pn = concl p in
  if not opid_of_term N = `nat` then
    failwith `NCompInd: induction must be over nat type`
  else
  
  ( Assert % All z:N. All n:N. n<z => P[n] %
 
           (mk_all_term
              z
              (mk_simple_term `nat` [])
              (mk_all_term
                 n
                 (mk_simple_term `nat` [])
                 (mk_implies_term
                    (mk_less_than_term (mvt n) (mvt z))
                     Pn
                 )
              )
            )

    THEN
    IfLabL
    [% ... #i:n:N ... All z:N. All n:N.n<z => P[n]  >> P[n] %
     `main`,
     DTerm (mk_add_term n_term (mk_integer_term 1)) (-1)
     THENM (DTerm n_term (-1) 
            THENM (D (-1) THEN
                   IfLab `main`  Hypothesis (AddHiddenLabel `arith`)))
     % `wf` ...  ... >> n+1 = n+1 in N %
     % `wf` ...  ... >> n = n in N %
     % `antecedent` ... >> n < n+1 %

    ;% ... #i:n:N ... >> All z:N. All n:N.n<z => P[n]  %
     `assertion`,
     Thin i THEN D 0 
       % ... ... z:N >> All n:N.n<z => P[n]  %
    ]
  ) p
;;



let NCompIndUpcase p =
  let n,(),imp = dest_all (concl p) in
  let (),Pn = dest_implies imp in
  let n' = maybe_new_var n (n . (bound_vars Pn @ declared_vars p)) in
  let Pn' = fo_subst [n,mk_var_term n'] Pn in

  % ... ... z:Z, 0 < z, All n:N. n<z-1 => P[n] >> All n:N. n<z => P[n] %
  ( SeqOnM
      [D 0
      ;D 0
       % ... ... z:Z, 0 < z, All n:N.n<z-1 => P[n], n:N, n<z >> P[n]  %

      ; Assert % All n':{0...n-}.P[n'] %
            (mk_all_term
               n'
               (mk_simple_term `int_seg` [mk_integer_term 0;mk_var_term n])
               Pn'
            )
          THEN IfLabL
          [`assertion`,
           % ... ... z:Z, 0 < z, All n:N.n<z-1 => P[n], n:N, n<z >>
                                                    All n':{0...n-}.P[n'] %
           SeqOnM
             [D 0
              % ... ... z:Z, 0 < z, All n:N.n<z-1 => P[n], n:N, n<z,
                                                  n':{0...n-} >> P[n'] %
             ;DTerm (mk_var_term n') (-4) 
             ;D (-1) THEN IfLab `main` Hypothesis (AddHiddenLabel `arith`)
             ]
          ;`main`,
           % ... ... z:Z, 0 < z, All n:N.n<z-1 => P[n], n:N, n<z,
                                              All n':{0...n-}.P[n'] >> P[n]  %
           OnHyps [-2;-3;-3;-3] Thin
           % ... ... n:N, All n':{0...n-}.P[n'] >> P[n]  %
          ]
        ]
  ) p
;;

% 
NCompInd
~~~~~~~~
Complete Induction (Course of values induction) over the naturals. 

... #i:n:N ... F[n] ... >> G[n] 

BY NCompInd #i

`main`: ... ... n:N, All n':{0..n-}. F[n'] => G[n'] >> F[n] => G[n]

`wf`    : 12 X   None involve well-formedness of F or G.
`arith` : 5 X    All soluble by Arith tactic.

If WF of F and G can be proven by Auto, it is going to be more efficient to
use the complete induction lemma.
%

let NCompInd i =
     NCompIndSetup i  
     THENM
       % ... ... z:N >> All n:N.n<z => P[n]  %
     ( NInd (-1)
       THEN IfLabL
       [`basecase`,
        % ... ... >> All n:N.0<0 => P[0]  %
        D 0 THENM (D 0 THENM AddHiddenLabel `arith`)
       ;`upcase`,
         % ... ... z:Z, 0 < z, All n:N. n<z-1 => P[n] >> All n:N. n<z => P[n] %
         NCompIndUpcase
       ]
     )
;;



%[
**********************************************************************
Finite Element Induction
**********************************************************************
]%

% Unit =def 0 in Int 

   . = it = unique element of unit.%

% 
...#i: x:Unit ... H[x] ... |- C[x]

  BY UnitInd i

  ... ... H[.] ... |- C[.]
%

let UnitInd i p =
  let i' = get_pos_hyp_num i p in
  let n = num_hyps p 
  in
  ( D i'
    THEN OnClauses (0.upto (i'+1) n) (\i.Try (Fold `it` i))
    THEN Thin i'
  ) p
;;

let BoolInd i p =
  let i' = get_pos_hyp_num i p in 
  let n = num_hyps p 
  in
  ( D i' 
    THEN UnitInd i'
    THEN OnClauses (0.upto i' (n-1)) (\i.Try (Folds ``btrue bfalse`` i))
  ) p
;;
