%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*************************************************************************
%

let is_identity_comp_seq_just j = 
   let cs = open_comp_seq_just j in
   if (length cs = 1) then
    let op,t,x = hd cs in
     (op = NOP & null x)
  else false
;;
   
   
let convert_comp_seq_to_sqjust j =
    if is_identity_comp_seq_just j then
      mk_id_sqjust
    else
    mk_tactic_sqjust (convert_comp_seq_to_tactic j)
;;

let convert_just_to_sqjust j =
  if is_sqjust j then open_sqjust j
  else if is_comp_seq_just j then
    convert_comp_seq_to_sqjust j
  else mk_tactic_sqjust (open_tactic_just j)
;;

let normalize_just j =
  if is_tactic_just j then j
  if is_sqjust j then
      form_tactic_just (ExecuteSqJust (open_sqjust j))
  else form_tactic_just (convert_comp_seq_to_tactic j)
;;

%
*****************************************************************************
Creating inversion justifications
*****************************************************************************
%


%
We only need to use inversion lemmas for equivalence relations. Inversion
of order relations is done by abstraction folding/unfolding.


Lemma for equivalence relation R should be of form:

All x1,y1:A1...xn,yn:An. B1...Bm  X R Y => Y R X

with name

<name of R>_inversion
%


%
letref inversion_lemma_alist = [] : 
  (reln # (just -> just)) list
;;
%
%
if the returned just_to_just proves  X R Y |- Y R_inv X

then R_inv should be supplied as the argument.
(The justification reverses this relation)

prepare_inversion_lemma name =
  [r,j_to_j] if name is that of equivalence rel

%

%
let prepare_inversion_lemma r 
  = 
  let just_wrap Tac = (\j.form_tactic_just
                       (Tac THEN_OnLast (DebugTry (open_tactic_just j)))) in
  let lemma_name = (name_of_rel r) ^ `_inversion` in
  let Invertor = BackThruGenLemma lemma_name (-1) in

   r,(just_wrap Invertor)
;;
%

let inversion_build index r name = 
  with_ref_environment
    (\name. (\j. form_tactic_just
                        ( (BackThruGenLemma name (-1))
                          THEN_OnLast (DebugTry (open_tactic_just j)))))

    name
    index
;;


letref inversion_cache =
  new_simple_ref_state `inversion`
   cache_ref_state_update_fail
   (nil : (reln # (unit + (just -> just))) list)
;;

update_ref_state_view
 (\(). inversion_cache)
 (ref_state_view_list_entry (reln_to_term o fst))
;;     

let declare_inversion_cache index = 
 inversion_cache := declare_ref_state_data inversion_cache index nil nil
;;
let undeclare_inversion_cache index = 
 (inversion_cache := ref_state_remove inversion_cache index; ())
 ? ()
;;

let inversion_cache_lookup_aux = 
  ref_cache_lookup_aux_aux
    alpha_equal_relns
    (let m = build_exact_cache_match "inversion" in
      cache_update_singleton_aux (\r. m (name_of_rel r)))
    (\(). inversion_cache)
    (\c. inversion_cache := c)
    inversion_build
;;

let inversion_cache_lookup i = outr (inversion_cache_lookup_aux i);;

%
let update_inversion_lemma_alist r =
  let r,just = prepare_inversion_lemma r in 
  inversion_lemma_alist :=
    update_alist_p 
      inversion_lemma_alist
      r
      just
      alpha_equal_relns
  ; ()
;;
%
let lookup_inversion_lemma r =
  inversion_cache_lookup r
  %apply_alist_p inversion_lemma_alist r alpha_equal_relns%
;;

let lookup_in_and_maybe_update_inversion_lemmas r =
  lookup_inversion_lemma r
;;

let InvertOrderRel r i p =
  let addr = if is_term `assert` (clause_type i p) then [1] else [] in
  if is_std_order_rel r then
    FoldAtAddr (name_of_rel (invert_rel r)) addr i p
  else
    UnfoldAtAddr addr i p
;;


let InvertIff i p =

  % Clause j' is: t1 <=> t2 %

  let ProveInversion j' p' =
  ( let j = get_pos_hyp_num j' p' in
    % ...,#j:ta <=> tb,... >> tb <=> ta %
     D j 
     THEN D 0 
     THENL 
     [% ...,#j:ta => tb, ta <= tb,... >> tb => ta %
      UnfoldTop `rev_implies` (j+1) THEN NthHyp (j+1)
     ;% ...,#j:ta => tb, ta <= tb,... >> tb <= ta %
      UnfoldTop `rev_implies` 0 THEN NthHyp j
     ]
  ) p'
  in
  
  let i' = get_pos_hyp_num i p in
  let t1,t2 = dest_iff (clause_type i' p) in
  
  if i' = 0 then 
  ( Assert (mk_iff_term t2 t1) 
    THENL [Id;ProveInversion (-1)]
  ) p
  else
  ( AssertAtHyp i' (mk_iff_term t2 t1) 
    THENL [ProveInversion i';Thin (i'+1)]
  ) p
;;

%
This is hooked up to inversion lemma database below
%

let InvertRelAux r i p =
 (if is_identity_rel r then
    Id 
  if is_equal_rel r then
    SwapEquands i 
  if is_sqequal_rel r then
    SqEq_Symmetry i
  if is_iff_rel r then
    InvertIff i
  if is_order_rel r then
    InvertOrderRel r i
  else
    failwith `InvertRelAux: no obvious way to invert rel`
 ) p
;;

let BasicInvertRel i p =
 ( let r,(),() = dest_rel_term_without_check (clause_type i p)
   in
    InvertRelAux r i p
 ) ? failwith `InvertRel`
;;


%
if j proves t r t'
then (get_inversion_just r j) proves t' rev_r t
%

let get_inversion_just r =
  let rname = name_of_rel r in
  if is_identity_rel r then
    id
  if is_equal_rel r then
    (\j.form_tactic_just (SwapEquands 0 THEN DebugTry (open_tactic_just j)))
  if is_sqequal_rel r then
    (\j.form_tactic_just (SqEq_Symmetry 0 THEN DebugTry (open_tactic_just j)))
  if is_iff_rel r then
    (\j.form_tactic_just (InvertIff 0 THEN DebugTry (open_tactic_just j)))
%
  if rname = `module_eq` then
    (\j.form_tactic_just (ModuleInvertEqv THEN DebugTry (open_tactic_just j)))
%
  if is_order_rel r then
    (\j.form_tactic_just (InvertOrderRel (invert_rel r) 0
                          THEN (DebugTry (open_tactic_just j))))
  else
    lookup_in_and_maybe_update_inversion_lemmas r
;;

let InvertRel i p =
 let rtm = clause_type i p in
 let r,t1,t2 = dest_rel_term_without_check rtm in
 if is_order_rel r then
    InvertOrderRel r i p
 else
 let inv_rtm = mk_similar_rel_term rtm t2 t1 in
 let sub_j = form_tactic_just (DebugTry Hypothesis) in 
 let r' = if i = 0 then invert_rel r else r in
 let T = open_tactic_just (get_inversion_just r' sub_j) in
 
 ( ReplaceWith inv_rtm i
   THENA T ) p
;;


%
*****************************************************************************
Creating weakening justifications
*****************************************************************************

A weakening justification takes a just ja for a rewrite w.r.t. reln ra and
generates a just jb for a rewrite w.r.t. a weaker reln rb.

Assume for now all weakening lemmas are of form

All x,y:T . x Ra y => x Rb y

with Ra nuprl equality and Rb some equivalence relation wearker than Ra.

Assume lemmas are named

<name of Rb>_weakening

or 

<name of Rb>_weakening_1
  .   .   .      .     .
  .   .   .      .     .
<name of Rb>_weakening_n

%


%
cache of prepared weakening lemmas. 

Typical entry:  ((rb,ra), ja_to_jb)
%

%letref weakening_lemma_alist = [] : ((reln # reln) # (just -> just)) list ;;
%

%
add new lemmas to cache
%

let prepare_weakening_lemma name = 

  let (),hyp_props,concl_prop = dest_simple_lemma name in
  let Tac = BackThruGenLemma name (-1) in
  let MaybeAddLab = IfLab `main` (AddHiddenLabel `weakening lemma antecedent`)
                                 Id
  in
  let TacThen T' = 
     Tac THEN_OnEach 
        (\ps.replicate MaybeAddLab (length ps - 1) @ [T'])
  in
  let justa_to_justb justa = 
    form_tactic_just (TacThen (DebugTry (open_tactic_just justa))) in
  let ra = fst (dest_rel_term (last hyp_props)) in
  let rb = fst (dest_rel_term concl_prop) 
  in
    (rb,ra),justa_to_justb
;;

  
let prepare_weakening_lemma_set name =
  let (rb,ra),justa_to_justb = prepare_weakening_lemma name in
  if not is_order_rel rb then
    [(rb,ra),justa_to_justb]
  else
    [(rb,ra),justa_to_justb
    ;let rev_rb = invert_rel rb in
     let rev_ra = invert_rel ra in
     (rev_rb,rev_ra)
     ,get_inversion_just rb o justa_to_justb o get_inversion_just rev_ra
    ]
;;

let weakening_list_build index names = 
  with_ref_environment (\names. flatten (map prepare_weakening_lemma_set names)) names index
;;

letref weakening_cache =
  new_simple_ref_state `weakening` cache_ref_state_update_fail
		(nil : (tok # ((reln # reln) # (just -> just)) list) list);;

update_ref_state_view
 (\(). weakening_cache)
 (ref_state_view_list_entry
   (\e. (icons_term (itoken_term (fst e))
		    (map_to_ilist (\(a,b),r. (icons_term (reln_to_term a) (reln_to_term b)))
			  (icons_op) (snd e)))))
;;     

let declare_weakening_cache index = 
 weakening_cache := declare_ref_state_data weakening_cache index nil nil
;;
let undeclare_weakening_cache index = 
 (weakening_cache := ref_state_remove weakening_cache index; ())
 ? ()
;;

let weakening_cache_lookup = 
  ref_cache_lookup_aux
    (cache_update_aux (build_cache_prefix_match "weakening"))
    (\(). weakening_cache)
    (\c. weakening_cache := c)
    weakening_list_build
;;

%
let update_weakening_lemma_alist rb 
  =
  let lemma_names = opid_alist_names (name_of_rel (get_std_rel rb)) `weakening` in 
  let alist_entry_sets = map prepare_weakening_lemma_set lemma_names 
  in
      weakening_lemma_alist :=
        multi_update_alist
           weakening_lemma_alist 
           (flatten alist_entry_sets)
      ; ()
;;
%
%
lookup_weakening_lemma (ra:rel) (rc:rel) = rb,jb_to_jc : rel # (just -> just)

Finds a lemma which reduces a goal of finding a justification for rc
to that of finding a justification for rb, where rb is no weaker than ra.

%

let lookup_weakening_lemma ra rc =
( let (rc',rb),jb_to_jc' = 
    find
      (\(rc',rb),(). rel_equal rc rc' & rel_less_or_eq ra rb)
      (weakening_cache_lookup (name_of_rel (get_std_rel rc)))
  in
    rb,jb_to_jc'
) ? failwith `lookup_weakening_lemma` 
;;

let lookup_and_maybe_update_weakening_lemma ra rc =
  lookup_weakening_lemma ra rc
 ?
  failwith (`lookup_and_maybe_update_weakening_lemma:` ^
        `\R  no lemma to weaken ` ^ name_of_rel ra ^ ` to ` ^ name_of_rel rc)
;;
%
get_weakening_just rela:rel relb:rel = justa_to_justb:just->just

We have justa proving (t1 rela t2) for some t1 t2. We desire a justb
that proves (t1 relb t2), where relb is a weaker relation than rela.

Special cases are

1. rela = relb ... return id
2. rela = identity, relb = equality , return id
3. rela = iff relb = implies
4. rela = iff relb = rev_implies
5. rela = sqequal relb = equality

3. rela, relb are both universe equalities, Ua lower or same as Ub
   ... use Refine (cumulativity level) THEN ...

Otherwise lookup weakening lemma, using the type of relb as an index.
Note that we might need several iterations to work down from relb
to rela.
%

let ImpToIff p =
  let (),t1,t2 = dest_rel_term_without_check (concl p) in

% >> t1 => t2 %

( Assert (mk_rel_term iff_reln t1 t2)
  THENL
  [
   % >> t1 <=> t2 %
   Id
  ;
   % t1 <=> t2 >> t1 => t2 %
   OnHyp (-1) D 
   % t1 => t2, t1 <= t2 >> t1 => t2 %
   THEN NthHyp (-2)]
) p
;;

let RevImpToIff p =
  let (),t1,t2 = dest_rel_term_without_check (concl p) in

% >> t1 <= t2 %

( Assert (mk_rel_term iff_reln t1 t2)
  THENL
  [
   % >> t1 <=> t2 %
   Id
  ;
   % t1 <=> t2 >> t1 <= t2 %
   OnHyp (-1) D 
   % t1 => t2, t1 <= t2 >> t1 <= t2 %
   THEN NthHyp (-1)]
) p
;;


let IffToEq p =
  let (),t1,t2 = dest_rel_term_without_check (concl p) in
  let Univ = U_lub (get_type p t1) (get_type p t2) in
  let level = snd (dest_lp_term Univ) in

% >> t1 <=> t2 %

( Assert (mk_equal_term (mk_U_term level) t1 t2)
  THENL
  [
   % >> t1 = t2 in Ui %
   AddHiddenLabel `main`
  ;
   % t1 = t2 in Ui >> t1 <=> t2 %
   D 0 
   % t1 = t2 in Ui >> t1 => t2 %
   % t1 = t2 in Ui >> t1 <= t2 %

   THEN 
   (\p'.
    (Repeat (UnfoldTopAb 0) THEN
     Refine `lambdaFormation` [mk_level_exp_arg level 
                              ;mk_var_arg (new_invisible_var p')] 
     % t1 = t2 in Ui, ta >> tb %
     % t1 = t2 in Ui, >> ta = ta in Ui'%

     % need PrimEq since Eq wrongly unfolds ta when ta is a `member` term %

     THENL [\p''.(UseEqWitness (mvt (var_of_hyp (-1) p'')) THEN PrimEq) p''
           ;Eq
           ]
    ) p'
   )
  ]
) p
;;

let EqToSqEq p =
  % >> A = B %
  let T,A,B = dest_member_or_equal (concl p) in
  (Try (UnfoldTop `member` 0) THEN (SqSubstInConclAtAddr [2] B)
   THENL [AddHiddenLabel `main`;  % A ~ B %
         Fold `member` 0 THEN AddHiddenLabel `wf`  % B in T %
         ]
  ) p
;;


%
Find a justification which weakens relb to relc where relb is no weaker than
rela.

get_weakening_just_step rela relc = relb, justb_to_justc 
%

let get_weakening_just_step rela relc 
  =
  if rel_equal relc untyped_equal_reln & rel_equal rela identity_reln then
    identity_reln,id
%
  if rel_equal relc module_eq_reln &
     rel_less_or_eq rela untyped_equal_reln then
       untyped_equal_reln
       , (\j.form_tactic_just (ModuleEqvToEq
                               THENM DebugTry (open_tactic_just j)))
%
  if rel_equal relc sqequal_reln &
     rel_less_or_eq rela identity_reln then
    identity_reln, (\j.form_sqjust mk_id_sqjust)

  if rel_equal relc untyped_equal_reln &
     rel_less_or_eq rela sqequal_reln then
    sqequal_reln, (\j.form_tactic_just (EqToSqEq
                                    THENM DebugTry (open_tactic_just j)))

  if rel_equal relc implies_reln &
     rel_less_or_eq rela iff_reln then
       iff_reln, (\j.form_tactic_just (ImpToIff 
                                      THEN DebugTry (open_tactic_just j)))

  if rel_equal relc rev_implies_reln &
     rel_less_or_eq rela iff_reln then
    iff_reln, (\j.form_tactic_just (RevImpToIff 
                                    THEN DebugTry (open_tactic_just j)))


  if rel_equal relc iff_reln &
     rel_less_or_eq rela untyped_equal_reln then
    untyped_equal_reln, (\j.form_tactic_just (IffToEq
                                    THEN DebugTry (open_tactic_just j)))

  else
    lookup_and_maybe_update_weakening_lemma rela relc 
;;


% get just to weaken rela to relc. Work iteratively down from relc %
 
letrec get_weakening_just rela relc =
    if rel_equal rela relc then id
    else
    let relb,justb_to_justc = get_weakening_just_step rela relc in
    let justa_to_justb = get_weakening_just rela relb
    in
      \ja. justb_to_justc (justa_to_justb ja)
;;

let apply_just_weakening r' r j =
  let j_weakener = 
   get_weakening_just r r'
     ? failwith 
       (`WeakenC: cannot weaken ` ^ name_of_rel r ^ ` to ` ^ name_of_rel r' )
    in
      j_weakener j 
;;

let StrengthenRelTo desired_r p = 
  let concl_r,() = dest_rel_term_without_check (concl p) in
  let j_to_j = get_weakening_just desired_r concl_r in

    open_tactic_just (j_to_j (form_tactic_just Id)) p
;;


let StrengthenRel = StrengthenRelTo untyped_equal_reln ;;

let StrengthenRelToHyp i p = 
  let hyp_r,() = dest_rel_term_without_check (h i p) in
  ( StrengthenRelTo hyp_r
    THENM (DebugTry (NthHyp i))
  ) p
;;

%
*****************************************************************************
Creating functionality justifications
*****************************************************************************

a functionality justification takes a list of justifications

[j1;...jn] where ji >> ti Ri ti'

and returns a just J such that

J >> T R T' where  T = op(t1;...;tn) , T'= op(t1';...;tn')


lemmas must have form

All x1...xn y1...yn A1...Am x1 r1 y1, ..., xn rn yn => 
  op(x1;...;xn) R op(y1;...;yn)

The Alls and Assumption As can be intermixed. The order
of the ri is not important, and one or more may be omitted.
In this event, the corresponding subterms of op on the 
lhs and rhs of R should either be non-variable, or identical.

lemmas in library are named

<opid of op>_functionality

where <rel kind> is the kind of the weakest relation of r1 ... rn



Functionality lemmas of form

All x1,y1:T1...xn,yn:Tn A1...Am x1 = y1 in T1, ..., xn = yn in Tn => 
  op(x1;...;xn) = op(y1;...;yn) in T

are derived from membership lemmas of form

All x1:T1...xn:Tn A1...Am op(x1;...;xn) in T

These membership lemmas should be named
or
<opid of op>_wf  (V4 Nuprl)




%

%
cache for functionality justifications

Typical entry:

index 1: opid : tok

index 2: [R1;...;Rn] : rel list

value : R, \[j1;...;jn]. J : 
         (rel list # rel # (just list -> just))


Entries for a given opid should be ordered (more specific R1 - Rn) to 
(more general R1 - Rn), so that by picking first applicable entry
we don't overgeneralize.

%

%
letref functionality_lemma_alist = 
  [] : (tok # 
         (reln list #
                      reln # (just list -> just) 
         ) list
       ) list

;;
%
%
Add in all functionality lemmas for term with op id opid 
%


let prepare_functionality_lemma name =
  let (),ant_props,concl_prop = dest_simple_lemma name in
  let Tac = BackThruGenLemma name (-1) in
  let R,t1,t2 = dest_rel_term concl_prop in
  let get_var t = dest_var t ? fst (dest_so_apply_of_var t) ? null_var in
  let get_bvs_v_pr t = map (id # get_var) (snd (rw_quick_dest_term t)) in

  % null_var indicates that lemma doesn't have info on subt functionality %

  let t1_v_bvs_prs = map2 
                     (\(vs,v) (vs',v').if v=v' then null_var,vs else v,vs) 
                     (get_bvs_v_pr t1)
                     (get_bvs_v_pr t2)
  in
  let n_func_args = length (remove_if (is_null_var o fst) t1_v_bvs_prs) in
  let dest_ant_prop aprop = 
    let r,tl,tr = dest_imbedded_rel_term aprop in
      r,get_var tl,is_term `guard` aprop
  in 
  let ant_Rel_v_gdp_trips = map dest_ant_prop (lastn n_func_args ant_props) in
  let num_rest_ants = length ant_props - n_func_args in
  let AntTs = replicate (AddHiddenLabel `rw func antecedent`) num_rest_ants in
  let ant_Rel_v_prs = map (\a,b,().a,b) ant_Rel_v_gdp_trips in
  let get_subt_r (v,bvs) = rev_apply_alist ant_Rel_v_prs v ? identity_reln in
  let Rels = map get_subt_r t1_v_bvs_prs in
  let justs_to_J justs = 
    let v_bvs_j_trips = map2 (\(v,bvs) j.v,bvs,j) t1_v_bvs_prs justs in
    let get_subt_j (ant_R,ant_v,is_guarded) = 
      let bvs,j = apply_alist v_bvs_j_trips ant_v in
       (if is_guarded then Id else RepeatAllImpCD [] %bvs%) 
       THEN_OnFirst 
       ( Try (UnfoldTop `guard` 0) 
         THEN DebugTry (open_tactic_just j) 
       )

    in 
      form_tactic_just 
        (Tac THENML (AntTs @ map get_subt_j ant_Rel_v_gdp_trips))
  in
     Rels,R,justs_to_J 
;;

%let old_prepare_functionality_lemma name =
  let (),hyp_props,concl_prop = dest_simple_lemma name in
  let Tac = BackThruGenLemma name (-1) in
  let R,t1,() = dest_rel_term concl_prop in
  let (),bterms = quick_dest_ho_term t1 in
  let num_subterms = length bterms in
  let Rels = 
    map (fst o dest_imbedded_rel_term) (lastn num_subterms hyp_props) in
  let justs_to_J justs = 
    form_tactic_just (Tac THEN_OnLastL 
                          (map2 
                              (\j (vs,()). 
                                  RepeatAllImpCD vs 
                                  THEN_OnFirst 
                                  ( Try (UnfoldTop `guard` 0) 
                                    THEN DebugTry (open_tactic_just j) 
                                  )
                              )
                              justs
                              bterms
                          ))
  in
     Rels,R,justs_to_J 
;;
%
%let prepare_functionality_lemma_set name =
  let Rels,R,justs_to_J = 
  (  prepare_functionality_lemma name 
     ?\x (display_message ("prep fun lemma failed: " J tok_to_string x)
          ; failwith x)
  )
  in
  if not is_order_rel R then
    [Rels,R,justs_to_J]
  else
  let rev_R = invert_rel R in
  let rev_Rels = map invert_rel Rels in
  let subterm_just_invertors = map get_inversion_just rev_Rels in
  let just_invertor = get_inversion_just R in

    [Rels,R,justs_to_J
    ;rev_Rels,rev_R,
       \js. (just_invertor o justs_to_J) (map2 ap subterm_just_invertors js) 
    ]
;;
%
let functionality_build name =
  let Rels,R,justs_to_J = 
  (  prepare_functionality_lemma name 
     ?\x (display_message ("prep fun lemma failed: " J tok_to_string x)
          ; failwith x)
  )
  in
  if not is_order_rel R then
    [Rels,R,justs_to_J]
  else
  let rev_R = invert_rel R in
  let rev_Rels = map invert_rel Rels in
  let subterm_just_invertors = map get_inversion_just rev_Rels in
  let just_invertor = get_inversion_just R in

    [Rels,R,justs_to_J
    ;rev_Rels,rev_R,
       \js. (just_invertor o justs_to_J) (map2 ap subterm_just_invertors js) 
    ]
;;

let functionality_list_build index names =  
 let l = (with_ref_environment (mapfilter functionality_build) names index) in
   % Using topolsort here guarantees that if several functionality
     lemmas are applicable, the one chosen will not have a weaker set
     of subterm relations than any of the others. Hopefully this
     ensures that the chosen one is always the most appropriate one.
   %
   (topolsort 
      (\(rs,()) (rs',()). rels_less_or_eq rs rs')
      (flatten l))
;;

letref functionality_cache =
  new_simple_ref_state `functionality` cache_ref_state_update_fail
		(nil : (tok # (reln list # reln # (just list -> just)) list) list);;

update_ref_state_view
 (\(). functionality_cache)
 (ref_state_view_list_entry
   (\e. (icons_term (itoken_term (fst e))
		    (map_to_ilist (\(l,b,r). (icons_term (map_to_ilist reln_to_term icons_op l)
							 (reln_to_term b)))
			  (icons_op) (snd e)))))
;;     
let declare_functionality_cache index = 
 functionality_cache := declare_ref_state_data functionality_cache index nil nil
;;
let undeclare_functionality_cache index = 
 (functionality_cache := ref_state_remove functionality_cache index; ())
 ? ()
;;
 
let functionality_cache_lookup =
 ref_cache_lookup_aux
    (cache_update_aux (build_cache_prefix_match "functionality"))
    (\(). functionality_cache)
    (\c. functionality_cache := c)
    functionality_list_build
;;

% 
let update_functionality_lemma_alist opid 
  =
  if is_bound opid functionality_lemma_alist then () else
  let lemma_names = opid_alist_names opid `functionality` in 
  let alist_entry_sets = 
        (mapfilter prepare_functionality_lemma_set lemma_names) 
  in
    functionality_lemma_alist :=
      update_alist 
        functionality_lemma_alist
        opid 
        %% Using topolsort here guarantees that if several functionality
          lemmas are applicable, the one chosen will not have a weaker set
          of subterm relations than any of the others. Hopefully this
          ensures that the chosen one is always the most appropriate one.
        %%
        (topolsort 
           (\(rs,()) (rs',()). rels_less_or_eq rs rs')
           (flatten alist_entry_sets))
  ; ()
;;
%

%
| Look through the cache for a functionality lemma.
| If it doesn't exist, clear the cache of this opid,
| and then try again.
%
let lookup_functionality_lemma opid rels =

  let fun_entries = functionality_cache_lookup opid in
    (find (\rels',R_and_Js_to_J. rels_less_or_eq rels rels')
	  fun_entries)
    ? failwith `lookup_functionality_lemma`
;;

%
| Special case for equality lemmas.
%
let lookup_equality_functionality_lemma opid rels =
  let fun_entries = functionality_cache_lookup opid in
    (find (\rels',R_and_Js_to_J. 
		 all (\r'.(is_identity_rel r' or is_equal_rel r') 
		      & rels_less_or_eq rels rels')
		 rels')
		fun_entries)
    ?  failwith `lookup_equality_functionality_lemma`
;;

%
get_best_functionality_just (opid:tok) ([r1;...;rn] : rel list) = 
  [r1';...;rn'], R,(\j1...jn.J)


Find most specific justification with (for all i)( ri equal to or more specific
than ri')

Takes care of 
1. For definition terms, as necessary, deriving equality functionality 
   entries from membership lemmas.
2. Using EqI for primitive terms.
3. for definition terms, looking up functionality lemmas in library.
4. Occasionally, it is desirable not to use EqCD for functionality.
   e.g. when want to use lemma to provide extra info about context.
   So even for equality, we check first if any suitable functionality
   lemma exists.
   See list_2 theory and mon_for_functionality_wrt_permr for an example. 
%



let get_best_functionality_just t rels =

  let LocEqCDThenL n Ts p =
    letrec Aux rev_Ts = 
      if null (tl (tl rev_Ts)) then
        EqCDThenL (rev rev_Ts)
      else
        EqCDThenL [Aux (tl rev_Ts);hd rev_Ts]
    in
    if n = 0 or is_term `infix_ap` (first_equand (concl p)) then
      EqCDThenL Ts p
    else
      Aux (rev Ts) p
  in
  let opid,kind = rw_opid_and_ap_arity_of_term t in

  if every rels (\r.is_identity_rel r or is_sqequal_rel r) then
      (map (\r.sqequal_reln) rels),
       sqequal_reln,
       (\js.
            let Tacs = map (\j.DebugTry (open_tactic_just j)) js in
              form_tactic_just (EqCD THENL Tacs))

  if every rels (\r.is_identity_rel r or is_equal_rel r) then
    ( lookup_equality_functionality_lemma opid rels
    ?
      (map (\r.untyped_equal_reln) rels),
       untyped_equal_reln,
       (\js.
            let Tacs = map (\j.DebugTry (open_tactic_just j)) js in
              form_tactic_just (LocEqCDThenL kind Tacs))
    )
%
  if some rels (\r.name_of_rel r = `module_eq`) & is_module_opid opid 
     & not is_module_eq_opid opid then
  (
    if is_module_eq_opid opid then
       (map (\r.module_eq_reln) rels),
       iff_reln,
       (\js.
            let Tacs = map (\j.DebugTry (open_tactic_just j)) js in
              form_tactic_just (ModuleEqvFun THENLL [`subterm`,Tacs])
       )
    else
       (map (\r.module_eq_reln) rels),
       module_eq_reln,
       (\js.
            let Tacs = map (\j.DebugTry (open_tactic_just j)) js in
              form_tactic_just (ModuleOpFun THENLL [`subterm`,Tacs])
       )
  )       
%
  else
  ( lookup_functionality_lemma opid rels
    ? failwith 
        `get_best_functionality_just: no lemma for ` ^ opid
     ^ ` with subterm relations` ^ concatl (map (\r.` ` ^ name_of_rel r) rels)
  )
;;



%
Have All i . (jai)|- ti rai ti'

Desire  (j)|- t r t'   where t = op(...vsi.ti...) , t' = ...

Functionality justification proves this from

All i (jbi)|- ti rbi ti', where  rbi is same or weaker than rai.

Weakening justifications take care of proving rbi's from rai's
%

letref quick_fiat_conv = false;;
let dummy_just_fiat = Fiat;;
let quick_dummy_reln = form_rel_with_parms `dummy_reln` nil nil;;
let quick_dummy_just = form_tactic_just dummy_just_fiat;;

let get_functionality_rel_and_just subterm_rel_just_prs t =
 if quick_fiat_conv then (quick_dummy_reln, quick_dummy_just)
 else
  let ras,jas = unzip subterm_rel_just_prs in
  if all is_identity_rel ras then
    identity_reln, merge_subterm_compute_seq_justs t jas
  else if all is_comp_seq_just jas then
    sqequal_reln, merge_subterm_compute_seq_justs t jas
  else if all (\r.is_identity_rel r or is_sqequal_rel r) ras then  
    %else if all (\j.is_sqjust j or is_comp_seq_just j) jas then  %
    sqequal_reln, form_sqjust (mk_list_sqjust (map convert_just_to_sqjust jas)) 
  else
    let tactic_jas = map normalize_just jas in
    let rbs,r,jbs_to_j =
      get_best_functionality_just t ras in
    let jbs = map3 get_weakening_just ras rbs tactic_jas
    in
      r, jbs_to_j jbs
;;


let RelArgCD p = 
  let r,ta,tb = dest_rel_term_without_check (concl p)
  in let ida,() = rw_opid_and_ap_arity_of_term ta 
  in let idb,() = rw_opid_and_ap_arity_of_term tb 
  in
  if not ida = idb then
    failwith `RelArgCD: outermost term constructors must be same`
  else
  let info_on_id = functionality_cache_lookup ida
                   ? failwith (`RelArgCD: no info found for id: ` ^ ida)
  in
  let proc (subt_rs,r',js_to_j) = 
    if rel_equal r' r then js_to_j else fail
  in
  let n = get_int_arg `n` p ? 1 in
  if n > length info_on_id then 
    failwith (`RelArgCD: no entry: ` ^ int_to_tok n ^ ` for id: ` ^ ida)
  else
  let subt_js = 
    map 
    (\n.form_tactic_just (AddHiddenLabelAndNumber `subterm` n))
    (upto 1 (length (subterms ta))) 
  in
    open_tactic_just (nth n (mapfilter proc info_on_id) subt_js) p
;;


%
*****************************************************************************
Creating transitivity justifications
*****************************************************************************

rely on lemmas of form

All x1:A1...xn:An  B1...Bm, t1 Ra t2, t2 Rb t3 => t1 Rc t3

Assume for now that Rc = lub (Ra,Rb)  

(This assumption affects how we name lemmas and how the alist is updated and
accessed)

Index lemmas by 

<name of Rc>_transitivity

or 

<name of Rc>_transitivity_1
  .     .       .
  .     .       .
<name of Rc>_transitivity_n



We need a transitivity justification when we have


Ja |- t1 Ra' t2   and Jb |- t2 Rb' t3

and we desire a J such that J |- t1 R' t3.

We deal with special cases when one or both of Ra' and Rb' are identity,
equal and iff directly, not using lemmas. (not fully implemented yet)

In the general case we use some transitivity lemma. The current 
strategy to pick lemmas with relations (Ra Rb Rc) is to choose the first one
with 

Rc = lub(Ra',Rb'), Ra =< Ra', Rb =< Rb' .

The implications of this strategy are

1. We won't be able to use some lemma which proves a weaker result, i.e. an Rc,
where Rc >= lub(Ra',Rb').

2. No matter how the alist entries are ordered, we will always pick the best
lemma.

We adopt this strategy because item 2 above is desirable, in particular 
since we have a lazy alist update strategy. (items are added only when needed.)



Final note: 
For integers, < =|= lub(<,<), though the current lub function returns < and it
is likely that the lemma for Ra,Rb = < will have Rc = < .It seems for integers
that x<y should really be treated like x+1 =< y. This simplifies things a lot.
%

%
Typical alist entry is

(Ra,Rb,Rc),t_to_jpr_to_j

With each R triple identifies a lemma uniquely, so if an update is repeated,
the old entry gets replaced. (is this desirable, or if updates are repeated
are we going to empty alist first anyway?)

Known Bug:
  The term binding t passed to the lemma being back chained through is
  generated in the environment created during justification generation.
  However, the environment during justification execution might differ if
  vars are renamed due to clashes. t will then have the wrong free variables.

%

%
letref transitivity_lemma_alist = [] : 
  ((reln # reln # reln) # (term -> (just # just) -> just)) list
;;
%
let prepare_transitivity_lemma name = 
  let (),hyp_props,concl_prop = dest_simple_lemma name in
  let Tac = BackThruGenLemma name (-1) in
  let [ra;rb] = map (fst o dest_rel_term) (lastn 2 hyp_props) in
  let rc = (fst o dest_rel_term) concl_prop in
  let tm_to_just_pr_to_just t (j1,j2) =
    form_tactic_just 
          ( (Using [null_var,t] Tac ORELSE AddLabel `foo`)
            THEN_OnLastL (map (\j. DebugTry (open_tactic_just j)) [j1;j2])
          ) 
  in
    (ra,rb,rc),tm_to_just_pr_to_just
;;

let prepare_transitivity_lemma_set name =
  let (ra,rb,rc),tm_to_just_pr_to_just =
    prepare_transitivity_lemma name in
  if not is_order_rel rc then
    [(ra,rb,rc),tm_to_just_pr_to_just]
  else
  let rev_ra = invert_rel ra in
  let rev_rb = invert_rel rb in
  let rev_rc = invert_rel rc in
  let just_rev_ra_to_just_ra = get_inversion_just rev_ra in
  let just_rev_rb_to_just_rb = get_inversion_just rev_rb in
  let just_rc_to_just_rev_rc = get_inversion_just rc in

  % note how order of relations and justs has to be inverted %

    [(ra,rb,rc),tm_to_just_pr_to_just
    ;(rev_rb,rev_ra,rev_rc),
     (\t (jb,ja). (just_rc_to_just_rev_rc o tm_to_just_pr_to_just t)
         (just_rev_ra_to_just_ra ja, just_rev_rb_to_just_rb jb))]
;;

let transitivity_list_build index names = 
 with_ref_environment  (\names. flatten (map prepare_transitivity_lemma_set names)) names index
;;
    
letref transitivity_cache =
  new_simple_ref_state `transitivity` cache_ref_state_update_fail
		(nil : (tok # ((reln # reln # reln) # (term -> (just # just) -> just)) list) list);;

update_ref_state_view
 (\(). transitivity_cache)
 (ref_state_view_list_entry
   (\e. (icons_term (itoken_term (fst e))
		    (map_to_ilist (\(a,b,c),r. (icons_term (reln_to_term a)
							   (icons_term (reln_to_term b) (reln_to_term c))))
			  (icons_op) (snd e)))))
;;     
let declare_transitivity_cache index = 
 transitivity_cache := declare_ref_state_data transitivity_cache index nil nil
;;
let undeclare_transitivity_cache index = 
 (transitivity_cache := ref_state_remove transitivity_cache index; ())
 ? ()		     
;;
 
let transitivity_cache_lookup =
 ref_cache_lookup_aux
    (cache_update_aux (build_cache_prefix_match "transitivity"))
    (\(). transitivity_cache)
    (\c. transitivity_cache := c)
    transitivity_list_build
;;


%
let update_transitivity_lemma_alist ra rb =
  let rc = compose_rels ra rb in
  let lemma_names = opid_alist_names (name_of_rel (get_std_rel rc)) `transitivity` in 
  let alist_entry_set = map prepare_transitivity_lemma_set lemma_names 
  in
    transitivity_lemma_alist :=
      multi_update_alist 
        transitivity_lemma_alist
        (flatten alist_entry_set)
    ; ()
;;
%
let lookup_transitivity_lemma ra rb =
  let rc = compose_rels ra rb in
    find
       (\(ra',rb',rc'),t_to_jpr_to_j.
          rel_equal rc rc' 
          & rel_less_or_eq ra ra'
          & rel_less_or_eq rb rb')
   (transitivity_cache_lookup (name_of_rel (get_std_rel rc)))
;;


let lookup_and_maybe_update_transitivity_lemmas ra rb =
  lookup_transitivity_lemma ra rb
  ? failwith 
    ( `lookup_and_maybe_update_transitivity_lemmas` 
      ^ `no lemma found for ` ^ name_of_rel ra ^`, ` ^ name_of_rel rb 
      ^ `transitivity`
    )
;;

% 
Not clear if this is any more efficient than using
lemma. If you want to use this, make sure backchain is
defined first.
%

let SplitIff t2 p =
  let n = num_hyps p in

% |- t1 <=> t3 %

  let (),t1,t3 = dest_rel_term_without_check (concl p) in
  
( AssertL [mk_iff_term t1 t2;mk_iff_term t2 t3]

%
|- t1 <=> t2
t1 <=> t2 |- t2 <=> t3
t1 <=> t2, t2 <=> t3 |- t1 <=> t3
%
  THENL 
  [Id
  ;Thin (n+1)
  ;D (n+1) THEN D (n+2) % THEN Backchain%] 
) p

;;

%
NB Above we don't account for well formedness subgoals. We need to know the
order of these before we use it. Would be interesting to see which approach
is faster.
%
%
Need to add too, conditions if one of ra and rb are EQUAL. See Dougs simp.ml
for details
%
let get_transitivity_just ra rb = 
  if is_equal_rel ra & is_equal_rel rb then
    (ra,rb,ra),(\t2 (ja,jb).form_tactic_just (SplitEq t2 
       THENL map (\j.DebugTry (open_tactic_just j)) [ja;jb]))
%
  if rname = `iff` then
    (\t2 (ja,jb).form_tactic_just
                   (SplitIff t2 THEN_OnFirstL map open_tactic_just [ja;jb]))
  if name_ra = `module_eq` or name_rb = `module_eq` then
    (module_eq_reln,module_eq_reln,module_eq_reln)
    ,(\t (ja,jb). 
       form_tactic_just
         (ModuleSplitEqv t 
          THENLL [`subterm`,map (\j.DebugTry (open_tactic_just j)) [ja;jb]]
         )
     )  
%
  else
    lookup_and_maybe_update_transitivity_lemmas ra rb
;;



% Ja proves ta ~ tb and Jb proves tb Rb tc.
  We want to prove that ta Rb tc
%
let chain_sq_just_first Ja tb Jb = 
 form_tactic_just (\p.
   let c = concl p in
   let n = length (subterms c) in %is there a more efficient way to compute n? %
   let t,ta,tc = dest_rel_term_without_check c in
   let ta_addr = [n-1] in
   let Ta = open_tactic_just (normalize_just Ja) in
   let Tb = open_tactic_just (normalize_just Jb) in
   (SqSubstInConclAtAddr ta_addr tb
    THENL [Ta; Tb]
    ) p
  )
;;
let chain_sq_just_second Ja tb Jb = 
 form_tactic_just (\p.
   let c = concl p in
   let n = length (subterms c) in %is there a more efficient way to compute n? %
   let t,ta,tc = dest_rel_term_without_check c in
   let tc_addr = [n] in
   let Ta = open_tactic_just (normalize_just Ja) in
   let Tb = open_tactic_just (normalize_just Jb) in
   (SqSubstInConclAtAddr tc_addr tb
    THENL [SqEq_SymmetryCD THEN Tb; Ta]
    ) p
  )
;;

let get_transitivity_rel_and_just (Ra,Ja) (Rb,Jb) e t2 =
 if quick_fiat_conv then (quick_dummy_reln, quick_dummy_just)
 else
  if is_identity_rel Ra then
    (Rb,Jb)
  if is_identity_rel Rb then
    (Ra,Ja)
  if is_comp_seq_just Ja & is_comp_seq_just Jb then
    sqequal_reln, join_comp_seq_justs Ja Jb
  if is_sqjust Ja & is_sqjust Jb then
    sqequal_reln, form_sqjust      
                  (chain_sqjusts (open_sqjust Ja) (open_sqjust Jb) t2) 
  if is_sqequal_rel Ra then
    if is_comp_seq_just Jb then
      sqequal_reln, 
      form_sqjust (chain_sqjusts (open_sqjust Ja) 
                                 (convert_comp_seq_to_sqjust Jb)
                                  t2)
    else
    (Rb, chain_sq_just_first Ja t2 Jb)
  if is_sqequal_rel Rb then
    if is_comp_seq_just Ja then
      sqequal_reln,
      form_sqjust (chain_sqjusts (convert_comp_seq_to_sqjust Ja) 
                                 (open_sqjust Jb) 
                                 t2)
    else
    (Ra,chain_sq_just_second Ja t2 Jb)
  else
  let tactic_Ja = normalize_just Ja in
  let tactic_Jb = normalize_just Jb in
  let (Ra',Rb',Rc'),t_to_J_pr_to_J =
     get_transitivity_just Ra Rb
  in
  let J_to_J_a = get_weakening_just Ra Ra' in
  let J_to_J_b = get_weakening_just Rb Rb' in
  let J_pr_to_J (J1,J2) =
    form_tactic_just
     (\p.let e' = env_of_sequent p in
         let t2' = term_in_env_fixup e t2 e'
         in
           open_tactic_just (t_to_J_pr_to_J t2' (J1,J2)) p
      )
  in
    Rc', J_pr_to_J (J_to_J_a tactic_Ja,J_to_J_b tactic_Jb)
;;

%
Are given.
t1 Ra t2  proved by Ja
t2 Rb t3 proved by Jb

We first find a basic justification for

t1 Ra' t2, t2 Rb' t3 |- t1 Rc' t3    : proved by J_pr_to_J

where Ra' no stronger than Ra, and Rb' no stronger than Rb.
Then we get weakening proofs:

t1 Ra t2 |- t1 Ra' t2   : proved by J_to_J_a
t1 Rb t2 |- t1 Rb' t2   : proved by J_to_J_b

To fill in the gaps.
%


% OLD FUNCTION %
%
let get_transitivity_rel_and_just (Ra,Ja) (Rb,Jb) t2 
  =
  if is_identity_rel Ra then
    (Rb,Jb)
  if is_identity_rel Rb then
    (Ra,Ja)
  if is_comp_seq_just Ja & is_comp_seq_just Jb then
    untyped_equal_reln, join_comp_seq_justs Ja Jb 
  if is_comp_or_subst_just Ja & is_comp_or_subst_just Jb then
    untyped_equal_reln, form_comp_subst_just [Ja; Jb]
  else
  let tactic_Ja = normalize_just Ja in
  let tactic_Jb = normalize_just Jb in
  let (Ra',Rb',Rc'),t_to_J_pr_to_J =
     get_transitivity_just Ra Rb  
  in
  let J_to_J_a = get_weakening_just Ra Ra' in
  let J_to_J_b = get_weakening_just Rb Rb' 
  in
    Rc', t_to_J_pr_to_J t2 (J_to_J_a tactic_Ja,J_to_J_b tactic_Jb)
;;
%

% 
  #i1.a R1 b ... #i2. b R2 c ... |- ...

  BY JoinRels i1 i2

  #i1.a R1 b ... #i2 b R2 c ... a R3 c |- ...

NB: JoinRels currently assumes that R3 will be same as R1 or R2.
This assumption also made in compose_rels function used in
update_transitivity_lemma_alist and lookup_transitivity_lemma above,
so is safe enough for now.
%

let JoinRels i1 i2 p = 
  let e = env_of_sequent p 
  in let i1' = get_pos_hyp_num i1 p and i2' = get_pos_hyp_num i2 p 
  in let R1 = h i1' p and R2 = h i2' p
  in let r1,a,b = dest_rel_term_without_check R1
  in let r2,b',c = dest_rel_term_without_check R2
  in 
  if not (alpha_equal_terms b b') then failwith `JoinRel: must have middle arg` else
  let j1 = form_tactic_just (DebugTry (NthHyp i1'))
  in let j2 = form_tactic_just (DebugTry (NthHyp i2'))
  in let r3,j = get_transitivity_rel_and_just (r1,j1) (r2,j2) e b
  in let R3 = 
   (if rel_equal r3 r1 then mk_similar_rel_term R1 a c  
    if rel_equal r3 r2 then mk_similar_rel_term R2 a c  
    else failwith `JoinRels: cannot guess rel to form`
   )
  in
  (  Assert R3 THEN IfLabL
     [`main`, AddHiddenLabel `main`
     ;`assertion`, DebugTry (open_tactic_just j)
     ]
  ) p
;;
  
% Use tb as intermediate term. %

let SplitRel tb p = 
  let e = env_of_sequent p in
  let r,ta,tc = dest_rel_term (concl p) in
  let j1 = form_tactic_just (AddHiddenLabel `main`) in
  let j2 = form_tactic_just (AddHiddenLabel `main`) in
  let (),j = get_transitivity_rel_and_just (r,j1) (r,j2) e tb
  in
    DebugTry (open_tactic_just j) p
;;
  

% R1 is  a r1 b  and T1 is a tactic such that
   H |- a r1 b   BY T1    
  R2 is  b r2 c, and T2 is a tactic such that
   H |- b r2 c   BY T2.

  then  from H |- a r3 c
  BY GeneralJoinRels H (R1, T1) (R2, T2)

NB: GeneralJoinRels currently assumes that R3 will be same as R1 or R2.
This assumption also made in compose_rels function used in
update_transitivity_lemma_alist and lookup_transitivity_lemma above,
so is safe enough for now.
%

let GeneralJoinRels e (R1, T1) (R2, T2) = 
  let r1,a,b = dest_rel_term_without_check R1 in
  let r2,b',c = dest_rel_term_without_check R2 in 
  if not (alpha_equal_terms b b') then failwith `JoinRel: must have middle arg` else
  let j1 = form_tactic_just (DebugTry T1) in
  let j2 = form_tactic_just (DebugTry T2) in
  let r3,j = get_transitivity_rel_and_just (r1,j1) (r2,j2) e b in
  let R3 = 
   (if rel_equal r3 r1 then mk_similar_rel_term R1 a c  
    if rel_equal r3 r2 then mk_similar_rel_term R2 a c  
    else failwith `GeneralJoinRels: cannot guess rel to form`
   ) in
  (  R3 , DebugTry (open_tactic_just j)) 
;;


%
*****************************************************************************
Creating identity justifications
*****************************************************************************
%

let get_identity_rel_and_just (t:term) =
  identity_reln, form_identity_comp_seq_just t
;;

%
*****************************************************************************
Cache initialization
*****************************************************************************
%

let initialize_rw_lemma_caches (():void) =
  %transitivity_lemma_alist := [] ;
  functionality_lemma_alist := [] ;
  inversion_lemma_alist := [] ;
  weakening_lemma_alist := [] ;
  %()
;;
