%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
| This package provides support for the case statement, and pattern matching.
| Tactic are provided to reduce the case statements, and
| to create patterns from templates.
]%

%
| Keep a list of patterns.
| Each 4-tuples has:
|    1. name of a contructor
|    2. name of a destructor
|    3. does destructor have continuation?
|    4. should destructor be updatable?
|       a. `none`: nothing is updatable
|       b. `all`: update abstractions and display forms
|       c. `display`: update display forms only
|       d. `abstractions`: update abstractions only
|    5. should state constructors be created?
|    6. Pattern itself
%

%letref (pattern_list : (tok # tok # bool # tok list # bool # term) list) =
    [`pair`, `case_pair`, false, [], false, void_term;
     `inl`, `case_inl`, true, [], false, void_term;
     `inr`, `case_inr`, true, [], false, void_term;
     `cons`, `case_cons`, true, [], false, void_term;
     `nil`, `case_nil`, true, [], false, void_term;
     `it`, `case_it`, false, [], false, void_term];;

let remove_pattern name =
    pattern_list := remove_alist_entry pattern_list name;
    ();;

let install_pattern name copid cont_flag update_flag state_flag pattern =
    pattern_list := update_alist pattern_list
                        name (copid, cont_flag, update_flag, state_flag, pattern);
    ();;

let is_pattern name =
    bound name pattern_list;;

let get_pattern name =
    apply_alist pattern_list name;;
%

letref pattern_list_ref_state =
  new_list_ref_state `pattern_list` (nil : (tok # tok # bool # tok list # bool # term) list);;

let pattern_list_do_updates oid edges oids =
 pattern_list_ref_state := ref_state_do_updates pattern_list_ref_state oid oids edges
 ; ()
;;

let undeclare_pattern_list oid = 
 (pattern_list_ref_state := ref_state_remove pattern_list_ref_state oid; ())
 ? () 			      
;;

let pattern_list_add_data oid data =
 pattern_list_ref_state := ref_state_set_data pattern_list_ref_state [oid, data]
;;

let pattern_list_add oid data =
  reset_ref_environment_data oid;
  add_ref_environment_data oid `pattern_list` pattern_list_add_data data
;;

let pattern_list_merge index edges items =
 pattern_list_ref_state
   := declare_ref_state_data_indirect `pattern_list` pattern_list_ref_state index items edges
;;

let lookup_pattern_list () =
  ref_state_get pattern_list_ref_state (current_ref_environment_index `pattern_list`);;

update_ref_state_view
 (\(). pattern_list_ref_state)
 (ref_state_view_list_entry (\e. itoken_term (fst e)))
;;

let is_pattern name =
    bound name (lookup_pattern_list())
;;

let get_pattern name =
    apply_alist (lookup_pattern_list ()) name
;;


let update_flags = [`case_display`;
		    `case_abstraction`;
		    `pattern_display`;
		    `pattern_abstraction`];;

let update_case_flags = [`case_display`; `case_abstraction`];;

let flat_flags =
    letrec aux space flags =
	if null flags then
	    `)`
	else
	    let f.l = flags in
		space ^ f ^ (aux ` ` l)
    in
	aux `(` update_flags;;


%
| Destruct a case bind term.
%
let dest_case_bind t =
    (let (opid, [v]), [(), body] = dest_term t in
	 if opid = `case_bind` then
	     dest_variable_parm v, body
	 else
	     fail)
    ? failwith `dest_case_bind: malformed case_bind`;;

%
| Simple terms.
%
let mk_hd_term x = mk_simple_term `hd` [x];;
let mk_tl_term x = mk_simple_term `tl` [x];;

%
| Disply form primitives.
%
let mk_pushm_term i =
    mk_term (`!dform_push`, [mk_natural_parm i]) [];;

let mk_popm_term =
    mk_simple_term `!dform_pop` [];;

let mk_break_term s =
    mk_term (`!dform_break`, [mk_string_parm s]) [];;

let mk_break_control_term s =
    mk_term (`!dform_break_control`, [mk_token_parm (string_to_tok s)]) [];;

let mk_szone_term =
    mk_break_control_term "soft";;

let mk_ezone_term =
    mk_break_control_term "NIL";;

%
| Reverse a list.
%
let reverse l =
    % Reverse a list %
    letrec aux l value =
        if null l then
	    value
	else
	    aux (tl l) ((hd l).value)
    in
	aux l [];;

%
| Possibly create an object.
| Return true if object was created.
%
%
let maybe_create_object obname type position =
    if is_lib_member (string_to_tok obname) then
	false
    else
	(create_object_nr obname type position;	true);;
%
%
| Set the term of an object, then check it.
%
%
let set_and_check_term_of_object obname term =
    set_term_of_object obname term;
    check_object_nr obname;
    ();;
%
%
| Transform an input pattern into a destructor for the pattern.
| Destructors have a procedure to check for the pattern
| and provide values for binding variables; a body
| to be executed if the pattern matches, and
| a continuation to be run if the match fails.
|
| In some case, the match will always succeed, and
| the continuation is not generated.
|
| Return a pair: whether the continuation was used,
|    and the destructor
|
| Walk the tree in pre-order.
%
let transform_pattern pattern final_term =
    % Scratch variables %
    let bv = (all_vars pattern) @ (all_vars final_term) in
    let x = maybe_new_var `x' bv in
    let xt = mk_var_term x in
    let y = maybe_new_var `y' (x.bv) in
    let yt = mk_var_term y in

    % This var is used for the continuation value %
    let z = maybe_new_var `z' (y.x.bv) in
    let zt = mk_var_term z in

    % This var is used for any continuation %
    let contv = maybe_new_var `cont' (z.y.x.bv) in
    let contt = mk_var_term contv in
    let cont_term = mk_apply_term (mk_apply_term contt zt) zt in

    % Get a new set of vars %
    let new_vars i =
	% Make a new set of vars %
	letrec aux i value avoid =
	    if i = 0 then
		value
	    else
		let v = maybe_new_var `x' avoid in
		    aux (i - 1) (v.value) (v.avoid)
	in
	    aux i [] (contv.z.y.x.bv)
    in

    %
    | This is the main function.
    | Match the term in input.
    | cont is a list of remaining terms to match.
    %
    letrec aux input cont =
        % input is the current input, cont is a list of further inputs %
	let opid = opid_of_term input in
	    if opid = `variable` then
		% Bind this variable to the term at this point %
		let v = dest_var input in
		    mk_lambda_term x
			(mk_apply_term
			    (mk_lambda_term v
			        (if null cont then
				     final_term
				 else
				     (mk_apply_term
				         (aux (hd cont) (tl cont))
					 (mk_tl_term xt))))
			    (mk_hd_term xt))
	    else if opid = `case_bind` then
		% Bind a variable to the input at this point %
		let v, body = dest_case_bind input in
		    mk_lambda_term x
		        (mk_apply_term
			    (mk_lambda_term v
			        (mk_apply_term
				    (aux body cont)
				    xt))
			    (mk_hd_term xt))

	    else if opid = `guard` then
		% This construction is not matched %
		let (), [const] = dest_simple_term input in
		    aux const cont

	    % Special cases for terms with significant parameters %
	    else if opid = `natural_number` or opid = `minus` then
		% Must match the number %
                mk_lambda_term x
                    (mk_apply_term
                        (mk_lambda_term y
                             (mk_simple_term `ifthenelse`
                                 [mk_simple_term `eq_int` [yt; input];
                                  if null cont then
                                      final_term
                                  else
                                      (mk_apply_term
                                          (aux (hd cont) (tl cont))
                                          (mk_tl_term xt));
                                  cont_term]))
                        (mk_hd_term xt))

	    else if opid = `token` then
		% Must match the token %
		mk_lambda_term x
		    (mk_apply_term
			(mk_lambda_term y
			     (mk_simple_term `ifthenelse`
				 [mk_simple_term `eq_atom` [yt; input];
				  if null cont then
				      final_term
				  else
				      (mk_apply_term
				          (aux (hd cont) (tl cont))
					  (mk_tl_term xt));
				  cont_term]))
			(mk_hd_term xt))

	    % General case for recursive patterns %
	    else if is_pattern opid then
		% Recursive pattern %
		let copid, cflag, () = get_pattern opid in
		let (), bargs = dest_term input in
		let (), args = unzip bargs in
		let argcount = length args in
		let nv = new_vars argcount in
		let mk_spread_term body =
		    % Analogous to 'spread' for pairs %
		    mk_term (copid, [])
		        (if cflag then
			     [nv, body; [], contt]
			 else
			     [nv, body])
		in
		let newaux arg xargs =
		    % Call aux recursively %
		    if argcount = 0 then
			if null cont then
			    final_term
			else
			    (mk_apply_term
			        (aux (hd cont) (tl cont))
				xargs)
		    else
			(mk_apply_term
			    (aux (hd arg) ((tl arg) @ cont))
			    xargs)
		in
		letrec consargs nv =
		    % New arguments to aux %
		    if null nv then
			mk_tl_term xt
		    else
			mk_cons_term
			   (mk_var_term (hd nv))
			   (consargs (tl nv))
		in
		    mk_lambda_term x
		        (mk_apply_term
			    (mk_apply_term
			        (mk_spread_term
			            (newaux args (consargs nv)))
				(mk_hd_term xt))
			    zt)
	    else
		failwith (`add_pattern: no pattern for ` ^ opid)
    in
    let term =
	mk_lambda_term x
	    (mk_lambda_term z
	        (mk_apply_term (aux pattern [])
		    (mk_cons_term xt mk_nil_term)))
    in
	if member contv (free_vars term) then
	    (true, (mk_lambda_term contv term))
	else
	    (false, term);;

%
| Create a display form pattern.
| this is just the text of the term, interspersed with
| slots for the identifiers.
|
| It would be better to update the display form package to
| deal with this directly.
%
let create_match_disp_form pattern =
    % Change the names of the variables to be unique strings %
    let prefix = `$$UniquePrefix<` in
    let suffix = `>UniqueSuffix$$` in
    let eprefix = explode prefix in
    let esuffix = explode suffix in
    let mk_unique_term v =
	mk_var_term (tok_to_var (prefix ^ (var_to_tok v) ^ suffix))
    in
    let vars = free_vars pattern in
    let newvars = map mk_unique_term vars in
    let newpattern = subst (zip vars newvars) pattern in

    % Get the display form for the term and explode it %
    let format = term_to_print_string 80 newpattern in
    let eformat = explode_string format in

    % Utilities for working on exploded strings %
    let nl = string_to_tok "\n" in
    let break_term = mk_break_term "" in
    let prepend_term value name =
	% Prepend name to value, reversed.  Look for imbedded newlines %
	letrec aux name accum term =
	    % name is the remaining input with newlines,
	    | accum is the scanned input without newlines,
	    | term is the processed input terms.
	    %
	    if null name then
		if null accum then
		    term
		else
		    (mk_text_term (tok_to_string (implode accum))).term
	    else
		let c.l = name in
		    if c = nl then
			(break_term 
			 . if null accum then
			       aux l [] term
			   else
			       (mk_text_term (tok_to_string (implode accum))).
			       (aux l [] term))
		    else
			aux l (c.accum) term
	in
	    aux name [] value
    in
    let prepend_tok value prefix c index =
	% Prepend a list of tokens to value (reversed) %
	letrec aux prefix =
	    if prefix = index then
		value
	    else
		(hd prefix).(aux (tl prefix))
	in
	    c.(aux prefix) 
    in
    let prepend_var value name =
	let vname = tok_to_string (implode (reverse name)) in
	    % Reverse name and turn it into a term %
	    mk_df_slot_format vname vname "*".value
    in

    %
    | Aux scans the exploded stroing and recovers variables.
    |    1. state = 0 means scanning prefix
    |       state = 1 means scanning variable name
    |    2. index is an index into the prefix or suffix explosions
    |    3. name is the input being collected into a string or var name
    |    4. data is the current exploded input
    |    5. value is the value to be returned
    %
    letrec aux state index name data value =
	if null data then
	    if state = 0 then
		prepend_term value name
	    else
		prepend_var value (prepend_tok name esuffix (string_to_tok "") index)
	else
	    let c = hd data in
		if state = 0 then
		    % Collect the prefix %
		    if c = hd index then
			if null (tl index) then
			    aux 1 esuffix [] (tl data) (prepend_term value name)
			else
			    aux 0 (tl index) name (tl data) value
		    else
			aux 0 eprefix (prepend_tok name eprefix c index) (tl data) value
		else
		    % Collect suffix %
		    if c = hd index then
			if null (tl index) then
			    aux 0 eprefix [] (tl data) (prepend_var value name)
			else
			    aux 1 (tl index) name (tl data) value
		    else
			% Start collecting the variable again %
			aux 1 esuffix (prepend_tok name esuffix c index) (tl data) value
    in
    let rformat = aux 0 eprefix [] eformat [] in
	(mk_szone_term
	 . mk_pushm_term 0
	 . (reverse rformat @ [mk_popm_term; mk_ezone_term]));;

%
| Turn the format into a display form.
%
let create_pattern_disp_form rformat =
    mk_df_format_list rformat;;

%
| Turn the format into a case display form.
%
let create_case_disp_form pattern bodyv cont_flag contv =
    let rformat = create_match_disp_form pattern in
    let body_slot = 
	[mk_text_term " =>";
	 mk_break_term " ";
	 mk_df_slot_format (tok_to_string (var_to_tok bodyv)) "True Case" "*";
	 mk_popm_term]
    in
    let cont_slot =
	if cont_flag then
	    [mk_break_term " ";
	     mk_df_slot_format (tok_to_string (var_to_tok contv)) "Continue" "*"]
	else
	    []
    in
	mk_df_format_list
	    (mk_pushm_term 4
	     . mk_text_term "Case "
	     . (rformat @ body_slot @ cont_slot));;

%
| Create pattern destructors and contrsuctors from a pattern template.
|
| Arguments:
|    1. force_flag: should display forms and abstractions be updated?
|    2. state_flag: should state versions of the patterns be created?
|    3. lib_position: where to put the patterm
|    4. opid: name of the pattern constructor
|    5. pattern: pattern template
|    6. update_flag: should this pattern respond to updates in the future?
%

let create_pattern_content pattern_p state_p case_p opid pattern format =

    let copid = `case_` ^ opid in
    let copids = tok_to_string copid in
    let popid = opid in
    let popids = tok_to_string popid in

    % Build a destructor %
    let bv = reverse (all_vars pattern) in
    let bodyv = maybe_new_var `body' bv in
    let contv = maybe_new_var `cont' (bodyv.bv) in
    let contt = mk_var_term contv in
    let final_term = mk_so_var_term bodyv (map mk_var_term bv) in
    let cont_flag, pterm = transform_pattern pattern final_term in

       %
       | Pattern.
       %

       % Term list for display form %
       let df_format =
         if null format then
	      create_match_disp_form pattern
         else format
	in
 
        %
	| The "pattern" form is the constructor.
	%
	let pattern_ab_term = mk_simple_term popid (map mk_var_term bv) in
	let pattern_df_lhs = create_pattern_disp_form df_format in
	let pattern_ab_def = apply_conv (TryC (UnfoldsC ``case_bind guard``)) pattern in
	let pattern_ab = mk_ab_object_term pattern_ab_term pattern_ab_def in
        let pattern_df_rhs = mk_df_rhs_for_term pattern_ab_term in
        let pattern_df = mk_df_def_list [mk_dform_def_term popids
						 pattern_df_lhs pattern_df_rhs] in

        let case_ab_term, case_ab_def1 =
	 if case_p then
             (if cont_flag then
                 mk_term (copid, []) [bv, final_term; [], contt],
                 mk_apply_term pterm contt
              else
                 mk_term (copid, []) [bv, final_term],
                 pterm)
         else ivoid_term, ivoid_term in

       (pattern_ab_term
       , case_ab_term)
       ,(append
	 %
	 | Create the pattern objects.
         ;;   - unconditional ? if called then assume we want it. incremental updates
         ;;      after selected deletes is broken. 
	 %
  	   [ create_df_obj_data pattern_df (popids J "_df")
           ; create_ab_obj_data pattern_ab popids
	   ]

         (append
	 %
         | State version.
	 %
         (if state_p then
	     (let state = `state' in
	      let statet = mk_var_term state in
	      let st_pattern_ab_term = mk_simple_term (`st_` ^ popid) (map mk_var_term bv) in
	      let st_pattern_ab_def =
		  mk_lambda_term `state'
		    (mk_simple_term popid
		      (map (\v. (mk_apply_term (mk_var_term v) statet)) bv))
	      in
	      let st_pattern_ab = mk_ab_object_term st_pattern_ab_term st_pattern_ab_def in
              let st_pattern_df_lhs = pattern_df_lhs in
              let st_pattern_df_rhs = mk_df_rhs_for_term st_pattern_ab_term in
              let st_pattern_df = mk_df_def_list
				    [mk_dform_def_term ("st_" J popids)
					 st_pattern_df_lhs st_pattern_df_rhs] in

	      %
	      | Create the pattern objects.
	      %
	      [ create_df_obj_data  st_pattern_df ("st_" J popids J "_df")
	      ; create_ab_obj_data   st_pattern_ab ("st_" J popids)
	      ]) 
	    else nil)

         %
	 | The "case" form is the destructor.
	 %
         (if case_p then
          (let case_ab_def = sweep_down_map compute case_ab_def1 in
  	   let case_ab = mk_ab_object_term case_ab_term case_ab_def in
           let case_df_lhs = create_case_disp_form pattern_ab_term bodyv cont_flag contv in
           let case_df_rhs = mk_df_rhs_for_term case_ab_term in
           let case_df = mk_df_def_list [mk_dform_def_term copids
							 case_df_lhs case_df_rhs] in

	   %
	   | Create the destructor objects.
	   %
           [ create_df_obj_data case_df (copids J "_df")
	   ; create_ab_obj_data case_ab copids
	   ])
       else nil)))
   , cont_flag
;;


let build_install_pattern opid contp statep updateflags pattern =
 indicate_insert_object_id
 (include_properties_term [`reference environment additions`, itoken_term `update`]
  (make_term (`install_pattern`, [ make_token_parameter opid
	  		         ; make_bool_parameter contp
			         ; make_bool_parameter statep])
    [[],map_to_ilist itoken_term itok_cons_op updateflags; [], pattern]))
;;

let create_pattern_aux oacc state_flag opid pattern update_flag format =

  if not null (diff update_flag update_flags) then
     failwith (`create_pattern: update_flag should be in ` ^ flat_flags);

   let ab_terms,content,contp = 
       (create_pattern_content 
           (member `pattern_abstraction` update_flag)
	   state_flag
	   (member `case_abstraction` update_flag)
	    opid pattern format)
     in
      ab_terms
      , (call_oacc oacc
         (append content
                [create_ml_obj_data (build_install_pattern opid contp state_flag update_flag pattern)
		    ((tok_to_string opid) J "_install_pattern")
		]))
;;


%
| This tactic is used for reducing case splits.
| The term at addr in clause i is presented as an application of
| a pattern to a value.  The application is reduced until
| another pattern application is uncovered, or until the
| body is uncovered.
%
let ReducePatternMatchC e t =
    let patterns = map (\(), y, (). y) (lookup_pattern_list ()) in
    let ReduceApplyC e t =
	% Preliminary reduction of the pattern application %
	(let tmp, value2 = dest_apply t in
	 let pattern, value1 = dest_apply tmp in
	 let opid = opid_of_term pattern in
	     if member opid patterns or opid = `case_default` then
		 % Fold subterms in guards, then unfold pattern %
		 let (), args = dest_term pattern in
		 let i = length args in
		 letrec aux j =
		     if j > i then
			 AddrC [2] (FoldTopC `guard`)
		     else
			 AddrC [1;1;j] (FoldTopC `guard`)
			 ANDTHENC aux (j + 1)
                 in
		     aux 1 ANDTHENC AddrC [1;1] (UnfoldTopC opid)
	     else if opid = `lambda` then
		 IdC
	     else
		 failwith `ReducePatternMatchC: not a pattern: ` ^ opid) e t
    in
	(if is_apply_term t then
	     ReduceApplyC
	     ANDTHENC ComputeGuardC
	     ANDTHENC (UnfoldTopC `guard`
		       ORELSEC (AddrC [1;1] (UnfoldTopC `guard`)
				ANDTHENC AddrC [1;2] (UnfoldTopC `guard`)
				ANDTHENC AddrC [2] (UnfoldTopC `guard`)))
	 else
	     failwith `ReducePatternMatchC: not a pattern`) e t;;

%
| Reduce the case statement, one case at a time.
%
let ReduceCaseC =
    UnfoldTopC `case`
    ANDTHENC ReducePatternMatchC
    ANDTHENC TryC (FoldTopC `case`);;
%add_AbReduce_conv `case` ReduceCaseC;;
%
