%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
| This is a package for creating enum's, which are implemented
| as natural numbers.
|
| To use "create_enum", you must provide the following argtuments:
|    name: base name of enum (as a tok)
|    values: enumeration values (as a list of tokens).
|
| The create_enum call provides the type of the enum,
| abstractions for each of the values,
| and a switch.
]%

%
| Remember the enumerations.
%
letref enumeration_list = [] : (tok # tok # tok list) list;;

% Add an enumeration %
let install_enum eopid opid values =
    enumeration_list := update_alist enumeration_list
        eopid (opid, values);;

% Get an enumeration %
let get_enum eopid = apply_alist enumeration_list eopid;;

%
| Base universe.
%
let enum_universe = mk_U_term (mk_const_level_exp 1);;

%
| Tactic for sqequality
%
let EnumSqTypeTac p =
    let opid, [type] = dest_simple_term (concl p) in
    let eopid, () = dest_simple_term type in
    let int_term = mk_simple_term `int` [] in
    let DTac = D 0 THENW Auto in
        (RepUnfolds [opid; eopid; `guard`] 0
	 THEN DTac
	 THEN DTac
	 THEN DTac
	 THEN With int_term SqEqual
	 THEN Auto) p;;

%
| Boolean reasoning.
%
let EnumEqTrueTac p =
    let i, enumtype, t = dest_all (concl p) in
    let j, () = dest_all t in
    let eopid, () = dest_simple_term enumtype in
    let DTac = D 0 THENW Auto in
	(Unfold eopid 0
	 THEN UnivCD
	 THENM InstLemma `eq_int_eq_true_intro` [mk_var_term i; mk_var_term j]
	 THEN Auto) p;;

let EnumEqFalseTac p =
    let i, enumtype, t = dest_all (concl p) in
    let j, () = dest_all t in
    let eopid, () = dest_simple_term enumtype in
    let DTac = D 0 THENW Auto in
	(Unfold eopid 0
	 THEN UnivCD
	 THENM InstLemma `eq_int_eq_false_intro` [mk_var_term i; mk_var_term j]
	 THEN IfLabL[`antecedent`, D 0 THENM D (-2)]
	 THEN Auto) p;;

%
| Case split tac.
| All the cases are listed in the concl as a large disjunction.
%
let EnumCasesTac p =
    let var, type, () = dest_all (concl p) in
    let vart = mk_var_term var in
    let name, () = dest_simple_term type in
    let prefix, branches = get_enum name in
    let names = name . (map (\s. prefix ^ `_` ^ s) branches) in

    % Use SupInf to wipe up %
    let Finish =
	RepeatFor (length branches) (Thin 2)
	THEN Assert (mk_simple_term `false` [])
	THENL [Fiat; Trivial]
    in

    % Split the cases %
    let univ = mk_level_exp_arg (mk_const_level_exp 1) in
    let SelLeft = RW (UnfoldTopC `or`) 0 THEN Refine `inlFormation` [univ] in
    let SelRight = RW (UnfoldTopC `or`) 0 THEN Refine `inrFormation` [univ] in
    let AutoThin p = (RepeatFor ((length (hyps p)) - 1) (Thin 2) THEN Auto) p in
    letrec Split bs =
        % Perform the decision %
        if null bs then
	    FailWith (`enumeration ` ^ name ^ ` is empty`)
	else
	    let h.t = bs in
	    let term = mk_equal_term type vart (mk_simple_term (prefix ^ `_` ^ h) []) in
		if null t then
		    Decide term THENL [Trivial; AutoThin; Trivial; Finish]
		else
		    Decide term
		    THENL [Trivial;
			   AutoThin;
			   SelLeft THEN Trivial;
			   SelRight THENL [Split t; AutoThin]]
    in
    letrec AssertWF bs Tac =
        % Cache the WF results %
        if null bs then
	    Tac
	else
	    let branch_to_eq name =
		(mk_equal_term type vart (mk_simple_term (prefix ^ `_` ^ name) []))
	    in
	    let term = mk_member_term enum_universe (mk_iterated_or (map branch_to_eq bs)) in
	    let h.t = bs in
	    let AssertTac =
		if null t then
		    Auto
		else
		    MemCD THENL [Auto; Trivial]
	    in
		AssertWF t (Assert term THENL [AssertTac; Tac])
    in
	(D 0 THENL [AssertWF branches Id THEN Split branches; Auto]) p;;
%
| Case split tac.
| All the cases are listed in the concl as a large disjunction.
%
let EnumInd i p =
    let i' = get_pos_hyp_num i p in
    let max = length (hyps p) in
    let v = var_of_hyp i' p in
    let vterm = mk_var_term v in
    let type = h i p in
    let name, () = dest_simple_term type in
    let prefix, branches = get_enum name in
    let names = (map (\s. prefix ^ `_` ^ s) branches) in

    % Handle a case %
    letrec SubstCase sim i =
	if i < max then
	    Try (Subst sim i) THEN SubstCase sim (i + 1)
	else
	    Try (Subst sim 0)
    in
    let Case name p =
	let caseterm = mk_simple_term name [] in
	let substterm = mk_sqequal_term vterm caseterm in
	    (Assert substterm
	     THENL [With type SqEqual THEN Trivial;
		    SubstCase substterm (i' + 1)
		    THENL [Trivial; Thin (-1) THEN Thin (-1) THEN Thin i']]) p
    in

    % Actual splitter %
    letrec Split names =
	if null names then
	    Id
	else
	    let hd.tl = names in
		if null tl then
		    Id
		else
		    D (-1) THENL [Case hd; Split tl]
    in
	(InstLemma (name ^ `_cases`) [vterm]
	 THENL [Trivial; Split names]) p;;

%
| Satisfy an enum conclusion.
%
let EnumFormation p =
    let name = opid_of_term (concl p) in
    let prefix, branches = get_enum name ?
	failwith (`EnumFormation: not an enum ` ^ name)
    in
    let member = get_term_arg `t1` p ?
	if null branches then
	    failwith (`Enum has no elements: ` ^ name)
	else
	    let h. () = branches in
		mk_simple_term (prefix ^ `_` ^ h) []
    in
	UseWitness member p;;

%
| General decomposition.
%
let EnumD i =
    if i = 0 then
	EnumFormation
    else
	EnumInd i;;
	    
%
| Create an n-way enum.
%

let create_enum_abs oacc eopid opid n =
    let eopids = tok_to_string eopid in
    let ab_term = mk_simple_term eopid [] in
    let ab_def = mk_simple_term `int_seg` [zero; mk_natural_number_term n] in
    let ab_wf_thm = mk_member_term enum_universe ab_term in
    let wf_text = "Unfold `" J eopids J "` 0 THEN Auto" in
     call_oacc oacc
	[ create_disp_content_for_new_def ab_term (eopids J "_df") eopids
	; create_ab_content_for_new_def ab_term ab_def eopids
	; create_thm_obj_data ab_wf_thm (itext_term wf_text) (eopids J "_wf")
        ]
;;

%
| Create the patterns for the elements in the enum
%

let create_enum_patterns oacc state_flag eopid opid p =
    let eterm = mk_simple_term eopid [] in
    let eopids = tok_to_string eopid in
    letrec create_proj oacc p i =
	if not null p then
	   (let h = hd p in
	    let popid = opid ^ `_` ^ h in
	    let popids = tok_to_string popid in
	    let dform = [mk_text_term popids] in
	    let ab_term = mk_simple_term popid [] in
	    let ab_def = mk_natural_number_term i in
            let ab_wf_thm = mk_member_term eterm ab_term in
	    let wf_text = "Unfolds ``" J popids J " " J eopids J "`` 0 THEN Auto" in
	    let ab_wf_thm2 = mk_member_term int_term ab_term in
	    let wf_text2 = "Unfold `" J popids J "` 0 THEN Auto" in
	      create_proj
                (call_oacc (snd (create_pattern_aux oacc state_flag popid ab_def update_case_flags dform))
		  [ create_thm_obj_data ab_wf_thm (itext_term wf_text) (popids J "_wf")
		  ; create_thm_obj_data ab_wf_thm2 (itext_term wf_text2) (popids J "_wf2")
		  ])
                (tl p) (i + 1))
          else oacc
    in
	create_proj oacc p 0
;;
	
%
| Create a switch for the enum.
| This switch has no defaults (for defaults, just use ifthenelse).
%
let create_enum_switch oacc eopid opid p forceflag =
    let eopids = tok_to_string eopid in
    let eterm = mk_simple_term eopid [] in
    let sopid = eopid ^ `_switch` in
    let sopids = tok_to_string sopid in

    % Create the abstraction for the switch %
    let vars = map tok_to_var p in
    let v = maybe_new_var `value' (map tok_to_var p) in
    let vt = mk_var_term v in
    let ab_term = mk_simple_term sopid (vt . map mk_var_term vars) in
    letrec create_def p =
        if not null p then
	    let val = hd p in
		mk_simple_term `switch_case`
		    [mk_simple_term (opid ^ `_` ^ val) [];
		     mk_var_term (tok_to_var val);
		     create_def (tl p)]
	else
	    mk_simple_term `switch_done` []
    in
    let ab_def = mk_simple_term `switch` [vt; create_def p] in
    let ab = mk_ab_object_term ab_term ab_def in

    % Create the display form for the switch %
    let break_term = mk_break_term " " in
    letrec create_format p =
	if null p then
	    [mk_text_term "EndSwitch";
	     mk_popm_term;
	     mk_ezone_term]
	else
	    let v.l = p in
	    let vs = tok_to_string v in
            let cases = term_to_print_string 80 (mk_simple_term (opid ^ `_` ^ v) []) in
		(mk_pushm_term 4
		 . mk_text_term ("Case " J cases J " =>")
		 . break_term
		 . mk_df_slot_format vs "Term" "*"
		 . mk_popm_term
		 . break_term
		 . create_format l)
    in
    let df_format =
	(mk_szone_term
	 . mk_pushm_term 0
	 . mk_text_term "Switch("
	 . mk_df_slot_format (tok_to_string (var_to_tok v)) "int" "*"
	 . mk_text_term ")"
	 . break_term
	 . create_format p)
    in
    let df_lhs = mk_df_format_list df_format in
    let df_rhs = mk_df_rhs_for_term ab_term in
    let df = mk_df_def_list [mk_dform_def_term sopids df_lhs df_rhs] in
    let df_name = (sopids J "_df") in
	% Create the display form %
        (call_oacc oacc
	 	[ create_df_obj_data df df_name
		; create_ab_obj_data ab sopids
		])
;;

%
| Extra theorems about enumerations.
|    1. Decidable equality
|    2. Squiggle type
|    3. equality => bool squiggle equality
|    4. equality => bool squiggle inequality
%

let create_enum_thms oacc eopid =

    % Local vars %
    let prefix, branches = get_enum eopid in
    let eopids = tok_to_string eopid in
    let enumtype = mk_simple_term eopid [] in
    let i = `i' in
    let j = `j' in
    let iterm = mk_var_term i in
    let jterm = mk_var_term j in

    % Decidable equality %
    let dec_name = "decidable__equal_" J eopids in
    let dec_term =
        (mk_all_term i enumtype
          (mk_all_term j enumtype
            (mk_simple_term `decidable` [mk_equal_term enumtype iterm jterm])))
    in
    let dec_tac = "Unfold `" J eopids J "` 0 THEN Auto" in

    % Squigggle type %
    let sq_name = eopids J "_sq" in
    let sq_term = mk_simple_term `sq_type` [enumtype] in
    let sq_tac = "EnumSqTypeTac" in

    % Boolean %
    let eqt_name = "eq_" J eopids J "_eq_true_intro" in
    let eqt_term =
        (mk_all_term i enumtype
          (mk_all_term j enumtype
            (mk_implies_term
              (mk_equal_term enumtype iterm jterm)
	      (mk_sqequal_term
	        (mk_simple_term `eq_int` [iterm; jterm])
		(mk_simple_term `btrue` [])))))
    in
    let eqt_tac = "EnumEqTrueTac" in

    let eqf_name = "eq_" J eopids J "_eq_false_intro" in
    let eqf_term =
        (mk_all_term i enumtype
          (mk_all_term j enumtype
            (mk_implies_term
              (mk_not_term (mk_equal_term enumtype iterm jterm))
	      (mk_sqequal_term
	        (mk_simple_term `eq_int` [iterm; jterm])
		(mk_simple_term `bfalse` [])))))
    in
    let eqf_tac = "EnumEqFalseTac" in

    % Induction %
    let ind_name = eopids J "_cases" in
    let ind_term =
        letrec aux branches =
            if null branches then
		% This is an empty enum %
		mk_simple_term `false` []
	    else
		let hd.tl = branches in
		let t = mk_equal_term
		    enumtype iterm (mk_simple_term (prefix ^ `_` ^ hd) [])
		in
		    if null tl then
			t
		    else
			mk_or_term t (aux tl)
	in
	    mk_all_term i enumtype (aux branches)
    in
    let ind_tac = "EnumCasesTac"
    
    in

     (call_oacc oacc
	[ create_thm_obj_data dec_term (itext_term dec_tac) dec_name
	; create_thm_obj_data sq_term (itext_term sq_tac) sq_name
	; create_thm_obj_data eqt_term (itext_term eqt_tac) eqt_name
	; create_thm_obj_data eqf_term (itext_term eqf_tac) eqf_name
	; create_thm_obj_data ind_term (itext_term ind_tac) ind_name
	])

;;

let build_install_enum eopid opid values =
 make_term (`install_enum`, [ make_token_parameter eopid
			    ; make_token_parameter opid])
  [[],map_to_ilist itoken_term itok_cons_op values]
;;

let build_enum_AbReduce args = 
 let sopid = (first_tok_of_term args) in
  [sopid,UnfoldTopC sopid ANDTHENC AbRedexC];;

let enum_AbReduce_additions_term sopid =
 mk_ab_reduce_adds sopid [sopid]
    [("UnfoldTopc `" J (tok_to_string sopid) J "`ANDTHENC AbRedexC")]
;;
% build_and_apply_additions_term 
  "ref_add_AbReduce_additions"
  "build_enum_AbReduce"
  (make_term (`args`, [make_token_parameter sopid]) [])
% 
 
let create_enum_aux oacc state_flag eopid opid values =

 % May create theorems %
 create_enum_thms

 % Remember the enumeration %
 (call_oacc

   % Must create patterns regardless %
   (create_enum_patterns 

      (let sopid = eopid ^ `_switch` in
       % let ReduceC = UnfoldTopC sopid ANDTHENC AbRedexC in
	    add_AbReduce_conv sopid ReduceC;
       %
       call_oacc % May create the enumeration type, and a switch for it %
		  (create_enum_switch (create_enum_abs oacc eopid opid (length values))
		   eopid opid values false)
           [create_ml_obj_data 
             (include_properties_term [`reference environment additions`, ibool_term true]
               (enum_AbReduce_additions_term sopid))
	     ((tok_to_string sopid) J "_AbReduce_conv_additions")])
       state_flag eopid opid values)

     [create_ml_obj_data (build_install_enum eopid opid values) 
	 ((tok_to_string opid) J "_install_enum")])

   eopid
;;
