%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
| This file provides a tactic to Split switch statements,
| similar to SplitITE.
]%

%
| Have a SwitchCase, decide it.
%
let SplitSwitchCase i addr enumname =
    let addr2 = addr @ [1] in
    let btrue = mk_simple_term `btrue` [] in
    let bfalse = mk_simple_term `bfalse` [] in
    let enumtype = mk_simple_term enumname [] in
    let true_name = `eq_` ^ enumname ^ `_eq_true_intro` in
    let false_name = `eq_` ^ enumname ^ `_eq_false_intro` in
    let DoTrueLemma name i p =
        let (), n, m = dest_equal (h i p) in
            InstLemma name [n; m] p
    in
    let DoFalseLemma name i p =
        let eq = dest_not (h i p) in
        let (), n, m = dest_equal eq in
            InstLemma name [n; m] p
    in
    let CaseWF name =
	InstLemma (name ^ `_wf`) [] THEN Trivial
    in
    let SplitSwitchEnumCleanup value p =
	let prefix, branches = get_enum enumname in
	let len = length branches in
	let i' = get_pos_hyp_num (-1) p in
	letrec EnumAux j =
	    if j = 0 then
		D i' THEN Trivial
	    else
		D (-1)
		THENL [D (i' - (2 * j)) THEN Trivial; EnumAux (j - 1)]
	in
	    (if len = 0 then
		 FailWith (`Empty enum: ` ^ enumname)
	     else
		 InstLemma (enumname ^ `_cases`) [value]
		 THENL [Trivial; EnumAux (len - 1)]) p
    in
    letrec Aux j p =
	(let ap = get_addressed_subterm addr (clause_type i p) in
	 let term, value = dest_apply ap in
	 let opid = opid_of_term term in
	     if opid = `switch_case` then
	         let (), [casev; body; cont] = dest_simple_term term in
		 let SplitSwitchCaseTrue =
		     SqSubstAtAddr addr2 btrue i
		     THENL [DoTrueLemma true_name (-1) THEN Trivial;
			    RW (AddrC addr AbRedexC) i
			    THEN RepeatFor (2 * j + 2) (Thin (-2))
			    THEN AddHiddenLabel `case`]
		 in
		 let SplitSwitchCaseFalse =
		     SqSubstAtAddr addr2 bfalse i
		     THENL [DoFalseLemma false_name (-1) THEN Trivial;
			    RW (AddrC addr AbRedexC) i
			    THEN Aux (j + 1)]
		 in
		     RW (AllC [AddrC addr2 (UnfoldTopC `switch_case`);
			       AddrC addr RedexC]) i
		     THEN Assert (mk_member_term enumtype casev)
		     THENL [CaseWF (opid_of_term casev) ORELSE AddHiddenLabel `wf`;
			    Decide (mk_equal_term enumtype value casev)
			    THENL [Trivial;
				   Trivial;
				   SplitSwitchCaseTrue;
				   SplitSwitchCaseFalse]]
	     else if opid = `switch_done` then
		 RW (AllC [AddrC addr2 (UnfoldTopC `switch_done`);
			   AddrC addr RedexC]) i
		 THEN (SplitSwitchEnumCleanup value ORELSE AddHiddenLabel `done`)
	     else if opid = `switch_default` then
		 RW (AllC [AddrC addr2 (UnfoldTopC `switch_default`);
			   AddrC addr RedexC]) i
		 THEN AddHiddenLabel `default`
	     else
		 FailWith `SplitSwitch: illegal switch format`) p
    in
	Aux 0;;

%
| Split a switch at a particular address.
%
let SplitSwitchAux i addr name =
    RW (AddrC addr (UnfoldTopC `switch`)) i
    THEN SplitSwitchCase i addr name;;

%
| Find a Switch subterm and split on it.
%
let SplitSwitch j p =
    let i = get_pos_hyp_num j p in
    let termlist =
	find_subterms_with_addrs
	    (\vs, t. is_term `switch` t & null (intersection vs (free_vars t)))
	    (clause_type i p)
    in
	(if null termlist then
	     FailWith `SplitSwitch: no switch terms`
	 else
	     let term, addr = hd termlist in
	     let (), [value; body] = dest_simple_term term in
		 Assert (mk_member_term int_term value)
		 THENL [AddHiddenLabel `wf`;
			SplitSwitchAux i addr `int`]) p;;

%
| Find a Switch subterm and split on it.
%
let SplitEnumSwitch enumname j p =
    let i = get_pos_hyp_num j p in
    let enumtype = mk_simple_term enumname [] in
    let switchname = enumname ^ `_switch` in
    let termlist =
	find_subterms_with_addrs
	    (\vs, t. is_term switchname t & null (intersection vs (free_vars t)))
	    (clause_type i p)
    in
	(if null termlist then
	     FailWith (`SplitEnumSwitch: no switch terms with name ` ^ switchname)
	 else
	     let term, addr = hd termlist in
	     let (), (value.cases) = dest_simple_term term in
		 Assert (mk_member_term enumtype value)
		 THENL [AddHiddenLabel `wf`;
			RW (AddrC addr (UnfoldTopC switchname)) 0
			THEN SplitSwitchAux i addr enumname]) p;;

%
| Reduce the switch statement, one statemnent at a time.
%
let ReduceSwitchC =
    let CheckTT e t =
	(if t = mk_simple_term `btrue` [] then IdC else FailC) e t
    in
    letrec AuxC e t =
        (AllC [UnfoldTopC `switch`;
	      AddrC [1] (UnfoldTopC `switch_case`);
	      RedexC;
	      AddrC [1;1] NormalizeC;
	      AddrC [1;2] NormalizeC;
	      AddrC [1] (UnfoldTopC `eq_int` ANDTHENC RedexC);
	      (AddrC [1] CheckTT ANDTHENC AbRedexC) ORELSEC (AbRedexC ANDTHENC FoldTopC `switch`)]
	 ANDTHENC TryC AuxC) e t
    in
	AuxC;;

%add_AbReduce_conv `switch` ReduceSwitchC;;
%
