%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%


%
*****************************************************************************
Justifications - rw-types-defs.ml
*****************************************************************************

Justifications are either
    1. tactics
    2. lists of compute sequences
    3. trees of tactics to prove sqequality (see file sqequal-just.ml)


A compute sequence also includes squiggle-equality rewrites,
which require tactics to prove the sqequal subgoals.  Each compute
sequence has a number tag, that determines what type it is, an
optional tagged term tagging the redices, and an optional tactic
to apply to the sqequal subgoals.
%

%
*****************************************************************************
environments 
*****************************************************************************
See file env.ml
%

%
*****************************************************************************
Relations - see rw-types-defs.ml
*****************************************************************************

Relations are pairs of

1. a relation name (usually the opid of the relation) and 
2. a list of bterms that are parameters to the relation.
   For the `equal' relation, this is a single bterm with no 
   bindings that is the type of the relation.
   Other relations might have, 0, 1 or more parameters.

Relations should be treated abstractly by other files; they shouldn't
be opening up the relation type and instead should be using the functions
defined later in this file. 

An exception is the use of name_of_rel in the just-manips file. There
the name is important for
a) forming abstraction names for order relation inversion
b) forming library object names
c) diagnostic messages


%

% don't bother checking parameters %

let rel_equal r1 r2 = (name_of_rel r1 = name_of_rel r2) ;;


%
equivalence relation info
~~~~~~~~~~~~~~~~~~~~~~~~~
%


%;;;	
;;;;	root_rel_relative_strengths 
;;;%

let add_equiv_relations index edges rnam stronger_rnam  =
 ref_add_rrrs_assoc index edges [rnam, stronger_rnam]
;;


let root_rel_less_or_eq r1 r2 = 
  exists_path_in_dag (lookup_rrrs ()) r2 r1 ;;




% only look at relative strength info %
let get_stronger_order_rel r = 
  snd (find (\wr,sr.rel_equal r wr) (lookup_order_rel_relative_strengths())) ;;
let get_weaker_order_rel r = 
  fst (find (\wr,sr.rel_equal r sr) (lookup_order_rel_relative_strengths())) ;;


%
OLD TREATMENT OF RELATIONS:
===========================
Relation ordering and equivalence.

We want to handle both equivalence and ordering relations.
Basic things we need to know are

1. If two relations are the same.
2. If one is weaker than another.

3. For ordering relations
    names of pairs, which are inverses of each other with one of
    each pair distinguished as being the standard.
 
The following scheme seems adequate for now.

The user can declare each relation with one of the following

1) add_equiv_rel_info `<name>` [<list of strictly stronger rels>] ;;

2) add_order_rel_info `<name>` `<name of inverse>` <is reflexive?>  
                                [<list of strictly stronger rels>] ;;

Names must be the same as the term opid's. (ignoring parameters and
types of equalities for now)

Built in is knowledge of nuprl equality terms (`equal`),
and identity (`identity`). These relations are assumed strictly stronger than all 
equivalence relations and reflexive ordering relations.

We store this info in a pair of structures.

1. order_rel_inverses = [n1,n1';...] : (tok # tok) list
  
   where the first element of each pair is the standard relation.

2. rel_weakness_table = [n1,[n1-1;n1-2;...]
                        ;n2,[             ]
                        .
                        .
                        ]   : (tok # tok list) list

if there is an entry ni, ni-j then relation ni is strictly weaker than 
relation ni-j.  The ni should be all standard names, not inverse names.
We compare inverse names by looking up their complements.


NEW TREATMENT OF RELATIONS: 
===========================
(as of September 15th 94)

Order relations are grouped into `families'. A family is a lattice
of order and equivalence relations of form:

      =<   >=
     /  \ /   \
    <    =    >

   2  1  0 -1 -2

   lt le eq ge gt

where weaker relations are higher in the lattice .
and relations within a family satisfy: 

For all a,b.

  a < b  <=>  b > a
  a =< b <=>  b >= a
  a = b  <=>  a =< b & b =< a
  a < b  <=>  a =< b & not(b =< a)


< and > are strict order relations.
=< and >= are order relations
= are equivalence relations 

Order relations are always partial or total when = is treated as the equality.

The relations in a family are identified by numbers as shown beneath them
in the above diagram, and sometimes by the two character mnemonic.
 
The = is treated as the `root relation' of a family.

I assume for the most part that one order relation being relatively stronger
than another can be determined by looking at 
 i) the relationship between relations in each family.
 ii) the relationship between root relations of families.

In case there are exceptions, I have added the order_rel_relative_strengths 
table. 

For keeping information on relations I have set up three tables:

1)  order_rel_families : (tok # reln list) list

    Every entry of form (fnam,[lt;le;eq;ge;gt]) records the relations of one
    family. 

    fnam is the family name (usually the opid of the lt relation). 
    fnam is used for quick equality testing of families.

2) root_rel_relative_strengths :  (tok # tok) list

    Every entry of form (rnam1,rnam2) where rnam2 is name of relation 
    strictly stronger than that with rnam1.

3) order_rel_relative_strengths : (reln # reln) list

Every equivalence relation (other than `identity`) should occur as the
left element of some pair in root_rel_relative_strengths.  However, an
equivalence relation need not be listed in an explicit family.

Every order relation should be in some family. 

order_rel_relative_strengths is only used when different families have
order relations of related strength.

Note that fnam's cannot be opids of equivalence relations in families,
since different families can have equivalence relations with the same
opids (e.g.  `equal`).

The lattice of relations can be visualized as having a backbone dag of
the ids of equivalence relations. The M-shaped sublattices for each
family are then appended to the appropriate ids of this backbone by
their central vertices.
%



% 
Matching and substituting with relations.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% 

let rel_match rpat rinst = 
  let pat  = rel_to_term rpat in
  let inst = rel_to_term rinst in
  
    match (free_vars pat) pat inst
;;

let rel_match_p rpat rinst = can (rel_match rpat) rinst ;;

let rel_subst sub rel = 
  term_to_rel (subst sub (rel_to_term rel)) 
;;


% 
Identifying Relations
~~~~~~~~~~~~~~~~~~~~~


identify_rel : rel -> int # tok # tok

Given relation r, returns a triple (k,fam,root) where

1.k  =  kind of relation (one of the 5 listed above)
2.fam=  order family rel is in (or `unknown`)
3.root= root rel kind (usually opid of equiv rel in family).
% 

% called after family not found %

let identify_equiv_rel rnam = 
  if rnam = `identity` or is_bound rnam (lookup_rrrs()) then
    0,`unknown`,rnam 
  else
    failwith (`identify_rel: unknown rel: ` ^ rnam)
;;

%
Used to measure `goodness' of match. 
A smaller number means that a match is better.
%

let rel_match_sub_size (sub : (var # term) list)= 
  reduce (\x y.x+y) 0 (map (term_size o snd) sub)
;;


let identify_rel r = 
  let add_root_kind (k,fam,rs) = k,fam,name_of_rel (nth 3 rs) in

  let rnam = name_of_rel r in
  let find_kind fam_rs = 3 - search_list (\r.name_of_rel r = rnam) fam_rs in
  let poss_idents = 
    mapfilter (\fid,frs.  find_kind frs, fid, frs) (get_order_rel_families())
  in
  if null poss_idents then
    identify_equiv_rel rnam
  if null (tl poss_idents) then 
    add_root_kind (hd poss_idents)
  else
  % got to do some more work. (e.g. `equal` in several families) %
  let poss_idents' = 
    mapfilter 
      (\k,fid,frs. rel_match (nth (3 - k) frs) r,k,fid,frs) 
      poss_idents 
  in
  if null poss_idents' then
    identify_equiv_rel rnam
  if null (tl poss_idents') then
    add_root_kind (snd (hd poss_idents'))
  else
  % do yet more work. Pick first family that gives the smallest match size % 
  let ranked_idents = map (rel_match_sub_size # id) poss_idents' in
  let pick_min (i,x) (i',x') = if i < i' then i,x else i',x' 
  in
    add_root_kind (snd (reduce1 pick_min ranked_idents))
;;


% 
Assume that parameter structure uniform across order relations
%

let change_rel fam_id old_kind old_r new_kind = 
  if new_kind = old_kind then old_r 
  else

  % should only get here if fam_id is not `unknown` %

  let fam = get_order_rel_family fam_id in
  if new_kind = 0 or old_kind = 0 then
    rel_subst
     (rel_match (nth (3-old_kind) fam) old_r)
     (nth (3-new_kind) fam)
  else
    form_rel_with_parms
      (name_of_rel (nth (3-new_kind) fam))
      (prms_of_rel old_r)
      (parms_of_rel old_r)
;;



let is_std_order_rel r = fst (identify_rel r) > 0 ;;
let is_rev_order_rel r = fst (identify_rel r) < 0 ;;
let is_order_rel r = not (fst (identify_rel r) = 0) ;;

let invert_rel r =
  let kind,fam,() = identify_rel r 
  in
    change_rel fam kind r (-kind)
;;

let get_std_rel r =
  let kind,fam,() = identify_rel r 
  in
    change_rel fam kind r (abs kind)
;;

let get_root_rel r = 
  let kind,fam,() = identify_rel r 
  in
    change_rel fam kind r 0
;;

let get_order_rel r = 
  let kind,fam,() = identify_rel r 
  in
    change_rel fam kind r 1
;;

% 
we consider the strongest relation at the bottom of the ordering,
the weakest at the top 
%
%
r1 stronger than r2 

More intelligent use should be made of 
order_rel_relative_strengths information here.
%


let basic_rel_less r1 r2 =
  
  let kind1,fam1,root1 = identify_rel r1 in
  let kind2,fam2,root2 = identify_rel r2 in

  if kind2 = 2 or kind2 = -2 then 
    false
  if kind1 = 2 then
    kind2 = 1 & fam1 = fam2
  if kind1 = -2 then
    kind2 = -1 & fam1 = fam2
  else
    kind1 = 0 & root_rel_less_or_eq root1 root2
;;

let basic_rel_less_or_eq r1 r2 = rel_equal r1 r2 or basic_rel_less r1 r2 ;;

let rel_less r1 r2 = 
  basic_rel_less r1 r2
  or (basic_rel_less_or_eq r1 (get_weaker_order_rel r2) ? false)
  or (basic_rel_less_or_eq (get_stronger_order_rel r1) r2 ? false)
;;


let rel_less_or_eq r1 r2 = rel_equal r1 r2 or rel_less r1 r2 ;;


let rels_less_or_eq r1s r2s = 
 accumulate (\x y.x & y) true (map2 rel_less_or_eq r1s r2s)
;;
      
%
This is approximate, but so far good enough..

Cases considered:

OR is Non symmetric relation
E is equiv relation
R is order relation (strict or not, equivalence or not)
SR is strict order relation.

Cases are considered in order
  r1 r2   r1@r2

  R1 R2    R1   R1 and R2 same.
  E  R     R    if E strength .le. R root strength
  R  E     R    if E strength .le. R root strength
  SR OR    SR   if SR strength .le. OR strength   
  OR SR    SR   if SR strength .le. OR strength   
%
let compose_rels r1 r2 = 
  let le r r' = rel_less_or_eq r r' 
  in
  if rel_equal r1 r2 then r1 else
  let k1,(),() = identify_rel r1 in
  let k2,(),() = identify_rel r2 in
  if k1 = 0 & le r1 (get_root_rel r2) then r2
  if k2 = 0 & le r2 (get_root_rel r1) then r1
  if k1 = 0 or k2 = 0 then failwith `compose_rels` 
  if (k1 = 2 or k1 = -2) & le r1 r2 then r1
  if (k2 = 2 or k2 = -2) & le r2 r1 then r2
  else failwith `compose_rels`
;;

%
The standard relations
%

let iff_reln = form_untyped_rel `iff` ;;
let identity_reln = form_untyped_rel `identity`;;
let implies_reln = form_untyped_rel `implies`;;
let rev_implies_reln = form_untyped_rel `rev_implies`;;
let untyped_equal_reln = form_untyped_rel `equal` ;;
let sqequal_reln = form_untyped_rel `sqequal` ;;
let module_eq_reln = form_untyped_rel `module_eq` ;;

%
In general, its probably a good idea to use these definitions as much
as possible, rather than make_untyped_rel, throughout the code.
%


let is_iff_rel r = name_of_rel r = `iff` ;;
let is_equal_rel r = name_of_rel r = `equal` ;;
let is_sqequal_rel r = name_of_rel r = `sqequal` ;;
let is_identity_rel r = name_of_rel r = `identity` ;;


  
%[
*****************************************************************************
Converting between Relations and Terms
*****************************************************************************

dest_rel_term t:term = r,t1,t2 : rel#term#term

Similar to dest_term, but

1. if t is a `tr' definition   (tr(e) == if e then True else False)
   then get its single subterm first. (This permits us to deal
   with defined relations which give a bool term as result rather than a type.)
2. t must have at least two subterms. The last two subterms are returned
   as the relation subterms t1 and t2. These subterms mustn't have any
   binding variables.
3. Any leading bterms are returned as the parameters of the relation.

4. NO LONGER USED:

   if t is an application of a module eq component, (ie. m.eq a b) then 
   return a and b as its subterms, and module_eq as the name of the relation
   class.

5. t is R a b, or t is a R b, where t is formed using `apply` or `infix_ap`
]%



let dest_rel_term_without_check t' =
  ( let t = if is_term `assert` t' then subtermn 1 t' else t' in
% We should allow case 5 above, but more work is needed to
  get such relations to work with everything else.
    if is_term `infix_ap` t  then
       let r = term_to_rel (subtermn 1 t) in
        r, subtermn 2 t, subtermn 3 t 
    else if is_term `apply` t  & is_term `apply` (subtermn 1 t) then
       let r = term_to_rel (subtermn 1 (subtermn 1 t)) in
        r, subtermn 2 (subtermn 1 t), subtermn 2 t 
    else
%
    let (opid,prms),bterms = dest_term t in
    let parm_bts,[[],t1;[],t2] = split_lastn 2 bterms in

      form_rel_with_parms opid prms parm_bts, t1, t2
  ) 
  ?
  failwith `dest_rel_term_without_check`
;;

% 
For backward compatibility. dest_rel_term used to check whether
info existed on relation, and if it didn't, made up some. Seems better
here to prompt the user alert the user failed.
%

letref dest_rel_term_info = null_token;;
let dest_rel_term t = 
  let r,t1,t2 = dest_rel_term_without_check t 
  in
  if can identify_rel r then
    r,t1,t2
  else
  ( dest_rel_term_info := (name_of_rel r)
  ; failwith `dest_rel_term, see dest_rel_term_info`
  )
;;

% for easy testing of relation functions %

let reltm_eq t1 t2 = 
  rel_equal (fst (dest_rel_term t1)) (fst (dest_rel_term t2)) 
;;
let reltm_lt t1 t2 = 
  rel_less (fst (dest_rel_term t1)) (fst (dest_rel_term t2)) 
;;
let reltm_le t1 t2 = 
  rel_less_or_eq (fst (dest_rel_term t1)) (fst (dest_rel_term t2)) 
;;




%
Useful for relations in so lemmas, which are buried under function and
implication terms.
%

let dest_imbedded_rel_term t =
  let (),(),r = dest_simple_formula t 
  in
    dest_rel_term r 
;;


let is_rel_term t = (dest_rel_term t ; true) ? false;;

% 
mk_rel_term (r:rel) (t1:term) (t2:term) = (t : term)

Doesn't handle wrapping of tr around relations with bool result type.
Creates bogus term if parameters are missing from r.
%

let mk_rel_term r t1 t2 = 
  let rname = name_of_rel r in
  let ps = parms_of_rel r in
  if rname = `equal` & null ps then
    failwith `mk_rel_term: missing type for equal term`
  if rname = `identity` then
    failwith `mk_rel_term: cannot make identity term`
  else
    mk_term (rname, prms_of_rel r) (ps @ [[],t1;[],t2])
;;

% 
makes relation term similar to rel_t with the relation
args replaced t1 and t2. 
%

let mk_similar_rel_term rel_t t1 t2 = 
  let wrapped = is_term `assert` rel_t in
  let r,(),() = dest_rel_term rel_t in
  let new_t' = mk_rel_term r t1 t2 in
  if wrapped then mk_simple_term `assert` [new_t'] else new_t'
;;

 
%;;;
;;;;	to get around circular defs rebind order_ref_families.
;;;%

let insert_equiv_rel l cur_rrrs =
 letrec aux rrrs l =
  if (null l) then rrrs
  else
    let nrrrs = aux rrrs (tl l) in
    let rnam, stronger_rnam = hd l in
    let nnrrrs = insert (rnam,stronger_rnam) nrrrs in
      if is_acyclic_graph nnrrrs
        then nnrrrs
        else nrrrs
  in aux cur_rrrs l
;;

rrrs_assoc :=
    new_ref_state `rrrs_assoc`
       flatten
       insert_equiv_rel
       (nil : (object_id # (tok # tok) list) list)
       (nil : (tok # tok) list)
;;
    	
let add_rel_families l orf =
 letrec aux orf l =  
 if null l then orf 
 else 
   (let lt, le, eq, ge, gt = hd l in
     
    let get_r t = fst (dest_rel_term_without_check t) in
    let rtms = [lt;le;eq;ge;gt] in
    let r_ids = mapfilter (name_of_rel o get_r) rtms in
    let fam_id.other_ids = r_ids in
    let mkrel (t,default_id) = 
      get_r t 
      ? form_untyped_rel (fam_id ^ `_fake_` ^ default_id)
    in
    let rs = map mkrel (zip rtms ``lt le eq ge gt``) in
    let orf1 = remove_alist_entries (aux orf (tl l)) other_ids in
      update_alist orf1 fam_id rs)
 in aux orf l
;;

order_rel_families_assoc := 
 new_ref_state
   `order_rel_families_assoc`
    flatten
    add_rel_families
    (nil : (object_id # (term # term # term # term # term) list) list)
    (nil : (tok # reln list) list)
;;


let add_order_rel_pairs l orrs =
 letrec aux orrs l =
  if null l then orrs
  else
  let stronger_rtm, weaker_rtm = hd l in
  let get_r t = fst (dest_rel_term_without_check t) in
  let stronger_r = get_r stronger_rtm in
  let weaker_r = get_r weaker_rtm in
  let norrs = aux orrs (tl l) in
  if exists (\(r,r').rel_equal r stronger_r & rel_equal r' weaker_r)
            norrs
     then norrs
     else ((stronger_r,weaker_r) . norrs)
   in aux orrs l
;;

order_rel_relative_strengths_assoc := 
  new_ref_state
    `order_rel_relative_strengths_assoc`
    flatten
    add_order_rel_pairs
    (nil : (object_id # (term # term) list) list)
    (nil : (reln # reln) list);;



%[
*****************************************************************************
Updating rel info tables
*****************************************************************************
The user should use dummy{}() terms as placeholders when an arg hasn't
been defined. The declaration here, makes up unique dummy names for the
these missing arguments.

declare_rel_family 
  lt : term
  le : term
  eq : term
  ge : term
  gt : term
  = 
  () : unit

Declares a relation family. The family is given a `family id' which is
the opid of the first non-dummy argument. 

declare_equiv_rel 
  rnam : tok
  stronger_rnam : tok
  = 
  () : unit

Declares an equivalence relation and its relationships to other
equivalence relations. rnam should be the opid of the relation term,
and stronger_rnam should be the opid of an immediately stronger
equivalence relation. Most often, there will be only one such
declaration for each rnam. However, multiple declarations for an rnam
are sometimes needed and are quite acceptable.  

Code removes old names from alist when it is updated.
]%

%
let declare_rel_family lt le eq ge gt = 
  let get_r t = fst (dest_rel_term_without_check t) in
  let rtms = [lt;le;eq;ge;gt] in
  let r_ids = mapfilter (name_of_rel o get_r) rtms in
  let fam_id.other_ids = r_ids in
  let mkrel (t,default_id) = 
    get_r t 
    ? form_untyped_rel (fam_id ^ `_fake_` ^ default_id)
  in
  let rs = map mkrel (zip rtms ``lt le eq ge gt``) in
  let orf1 = remove_alist_entries order_rel_families other_ids in
  let orf2 = update_alist orf1 fam_id rs
  in
    order_rel_families := orf2
    ; ()
;;
% 


%
let declare_equiv_rel rnam stronger_rnam =
  let rrrs = insert (rnam,stronger_rnam) root_rel_relative_strengths in
  if is_acyclic_graph rrrs then
  ( root_rel_relative_strengths := rrrs ; ())
  else 
    ()
;;

declare_equiv_rel `equal` `identity` ;;
%


%
to prompt user to change outdated declarations 
%

let add_equiv_rel_info (name:tok) (stronger_names:tok list) =
  display_message 
     ("Use declare_equiv_rel on: " J tok_to_string name)
;;

let add_order_rel_info 
  (name:tok) (inv_name:tok) (reflexive_p:bool) (stronger_names:tok list)
   =
  display_message 
     ("Use declare_rel_family on: " J tok_to_string name)
;;

%[
*****************************************************************************
Recognizing linear order relations
*****************************************************************************
Usage:

add_lin_order_check_fun 
  fam_id : tok
  f : env -> reln -> bool
  = 
  () : unit
;;

fam_id - family name for order. Usually id of lt relation.
f      - (f e r) should return true iff r is a linear relation
         f can assume that r is in family with id fam_id.
]%

let is_int_rel (e:env) (r:reln) = true ;;

%
letref lin_order_check_funs = [`less_than`,is_int_rel] ;;

let add_lin_order_check_fun fam_id f = 
  lin_order_check_funs :=
     update_alist lin_order_check_funs fam_id f
  ; ()
;;
%


let is_linear_rel e r = 
  let (),fam_id,() = identify_rel r in
  apply_alist (lookup_lin_order_check_funs()) fam_id e r ? false
;;

% only valid for linear order relations.%

let complement_rel_term t = 
  let r,a,b = dest_rel_term t in
  let kind,fam_id,root_id = identify_rel r in
  let kind' = 
    (if kind = 2 then 1 
     if kind = 1 then 2
     if kind = -1 then -2
     if kind = -2 then -1
     else failwith `complement_rel_term`)
  in
  let r' = change_rel fam_id kind r kind' 
  in
    mk_rel_term r' b a
;;


%
To be inserted in sets_1:

let is_oset_rel e r = 
    is_term `loset` (get_type_from_env e (parm_of_rel r)) ? false)
;;

add_lin_order_check_fun `set_lt` is_oset_rel ;;

%

