%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%
**********************************************************************
Tactic auxilary functions
**********************************************************************
%


let mk_var_arg = make_variable_argument ;;
let mk_term_arg = make_term_argument ;;
let mk_le_term_arg t = mk_term_arg (compat_le_term t) ;;
let mk_bterm_arg = make_bound_term_argument ;;
let mk_assumption_list_arg = make_assumption_list_argument ;;
% let mk_parm_list_arg = make_parameter_substitution_list_argument ;; %
let mk_parm_sub_arg tok_parm_list = make_substitution_list_argument tok_parm_list [] ;;

let mk_tok_arg tok = make_parameter_argument (make_token_parameter tok) ;;
let mk_obid_arg obid = make_parameter_argument (make_object_id_parameter obid) ;;
let mk_nat_arg i = make_parameter_argument (make_natural_parameter i) ;;
let mk_int_arg i = make_assumption_index_argument i ;;
let mk_level_exp_arg le = make_parameter_argument (make_level_expression_parameter le);;
  

%
**********************************************************************
New variable functions.
**********************************************************************
%

let new_visible_var root p = maybe_new_proof_var (mkv root) p ;;

let new_invisible_var p =
  new_proof_var basic_invisible_var p ;;

%
new_var_set [[v1-1;v1-2;...;v1-n1]
            ;  .          .
            ;  .          .
            ;  .          .
            ;  .          .
            ;[vm-1;vm-2;...;vm-nm]]
            existing_vars

            =
 
            [w1;...;wm]

variable wi is based on the first non null (maybe invisible) variable in the 
list

[vi-1;...;vi-ni;basic_invisible_var]

 and is made distinct from both all existing_vars and

[w1;...;w(i-1)]
%

let get_distinct_var v p =
  maybe_new_proof_var v p 
;;

let new_var_set list_of_lists_of_options existing_vars =
  fst
  ( map_with_carry
      (\list_of_options vars_declared_so_far.
          let root = (hd (remove_if is_null_var list_of_options) 
                     ? basic_invisible_var) in
          let new_var = maybe_new_var root vars_declared_so_far 
          in
            new_var, new_var.vars_declared_so_far  
      )
      existing_vars
      list_of_lists_of_options
  )
;;

let get_distinct_var_set list_of_lists_of_options p =
  new_var_set list_of_lists_of_options (declared_vars p)
;;

let mk_new_var_args list_of_lists_of_options p =
  map 
    mk_var_arg 
   (get_distinct_var_set list_of_lists_of_options p)
;;



%
************************************************************************
Standardizing apart procedure for terms.
************************************************************************
Here we rename bound variables so that no variable shadows any other 
variable. Procedure doesn't maintain sameness / apartness of binding vars
for different subterms of a term. Might want to add this since some display
forms key off this information. 
%

letrec 
  stdize_apart_term ctxt sub term =
    if is_var_term term then 
      fo_subst sub term
    else 
    ( let op,bterms = dest_term term in
        mk_term op (map (stdize_apart_bterm ctxt sub) bterms)
    )
and 
  stdize_apart_bterm ctxt sub (vars,term) =
    if null vars then
      [], stdize_apart_term ctxt sub term
    else
    ( let v.vs = vars in
      let v',sub' =
        if member v ctxt then
        ( let new_v = new_var v ctxt in
            new_v , (v, mk_var_term new_v).sub
        )
        else
          v,sub
      in
      let vs',term' = stdize_apart_bterm (v'.ctxt) sub' (vs,term) in
        (v'.vs' , term' )
    )
;;

let stdize_apart_vars_of_term ctxt t =
  stdize_apart_term ctxt [] t
;;
              




%
**********************************************************************
Proof recognisers
**********************************************************************
%


let is_type_membership_goal p =
  let c = concl p in
    is_member_term c & is_canonical_non_U_type (subterm_of_term c 1)
;;

let is_universe_membership_goal p =
  let c = concl p in
    is_member_term c & is_U_term (subterm_of_term c 1)
;;

let is_membership_goal p =
  let c = concl p in
    is_member_term c & 
    satisfies_one 
      [is_U_term;is_canonical_type] (subterm_of_term c 1)
;;
  

%
****************************************************************************
General Formula Preprocessing.
****************************************************************************
%

%
A general formula has form:

t =   all x:A. t'
    | P => t'
    | t' <= P    
    | t' & t''
    | t' <=> t''  (defined as t' => t'' & t' <= t'')
    | Q           where Q has some other outermost constructor


Conceptually we regard such a formula as a composition of simple formulae,
each of form:

t =   all x:A. t'
    | P => t'
    | t' <= P    
    | Q           

Each simple formula has zero or more antecedents P and a single consequent Q. 

When one uses a general formula, one usually has a particular
consequent in mind.  We give each consequent an address - a list of
booleans. The address of a consequent specifies which way to turn at
each conjunct to reach the consequent from the outside. A `false'
element in an address means take the left conjunct, a `true' element
means take the right. Addresses are read from left to right.

If one desires a Q with one of the term constructors used in defining the 
structure of a general universal formula, it needs to be wrapped in a guard 
definition. Guard wrappers are removed automatically by functions and tactics
which work on general universal formulae.

A guard term is an abstraction with opid `guard`, no parameters and a single 
subterm with no variables bound.

process_general_formula breaks down a general formula into its component 
simple formulae. dest_general_formula separates out the parts of each simple
formula.


process_general_formula f:term =

simple_list : (bool list # (var # term) list) list

%

% consider (a=b <==> b ^ a) as
 then a=b, b, a are the formulas
 however, the iff is also an interesting formula itself.
% 
letref process_general_test = false;;

let process_general_formula f =

 letrec aux f = 

 let f' = unfold_soft_abs f in

 if is_term `function` f' then
      (let [[],A;[x],B] = bterms_of_term f' in
       map (id # $. (x,A)) (aux B)
      )

 if is_term `product` f' & is_null_var (bvar_of_term f' 2 1) then
  (let [A;B] = subterms_of_term f' in
   map ($. false # id) (aux A) 
   @ 
   map ($. true # id) (aux B) 
  )

 if is_term `guard` f then
   [[],[null_var, dest_guard f]]
 else
   [[],[null_var, f]]
 in

 letrec test_aux f = 

 if is_term `iff` f
    then (([],[null_var, f]) . (test_aux (unfold_soft_abs f)))
 else
 let f' = unfold_soft_abs f in

 if is_term `function` f' then
      (let [[],A;[x],B] = bterms_of_term f' in
       map (id # $. (x,A)) (test_aux B)
      )

 if is_term `product` f' & is_null_var (bvar_of_term f' 2 1) then
  (let [A;B] = subterms_of_term f' in
   map ($. false # id) (test_aux A) 
   @ 
   map ($. true # id) (test_aux B) 
  )

 if is_term `guard` f then
   [[],[null_var, dest_guard f]]
 else
   [[],[null_var, f]]

 in if process_general_test then test_aux f else aux f
;;
   
%
dest_general_formula f returns a list of  4 tuples: 

(addr,bindings,ants,con) : bool list # (var # term) list # term list # term

Each 4 tuple describes a simple formula component of the formula f.
%

let analyze_clause_list cl =
  let h_and_c,binders = divide_list (is_null_var o fst) cl in
  let hs,[c] = split_lastn 1 (map snd h_and_c) 
  in
    binders,hs,c
;;


let dest_general_formula f =
  map
    (id # analyze_clause_list)
    (process_general_formula f)
;;

let get_addr_of_nth_simple_formula f =
  let addr_list =
    map fst (process_general_formula f) in
 \n.
   nth n addr_list
;;


let simple_formula_con f =
 letrec aux f =
   if is_term `all` f then
      aux (snd (hd (tl (bterms_of_term f))))

   if is_term `implies` f then
      aux (snd (hd (tl (bterms_of_term f))))

   if is_term `rev_implies` f then
      aux (snd (hd (bterms_of_term f)))

   if is_term `guard` f then
      aux (snd (hd (bterms_of_term f)))

   else f in 

  aux f
;;


letrec process_simple_formula f =

 if is_term `all` f then

  (let [[],A;[x],B] = bterms_of_term f in
    (x,A) . process_simple_formula B
  )
 if is_term `implies` f then

  (let [A;B] = subterms_of_term f in
    (null_var,A) . process_simple_formula B
  )
 if is_term `rev_implies` f then

  (let [A;B] = subterms_of_term f in
    (null_var,B) . process_simple_formula A
  )
 if is_term `guard` f then

   [null_var, dest_guard f]

 else

   [null_var, f]
;;

let dest_simple_formula f =
      analyze_clause_list (process_simple_formula f)
;;



let nth_simple_formula n f = 
  if n < 0 then
    [[],dest_simple_formula f]
  else
  let simp_fmlas = dest_general_formula f in
  if n = 0 then 
    simp_fmlas
  else
    ([nth n (dest_general_formula f)] ? simp_fmlas)
;;



%[
Specialisations of above functions.
]%

let process_simple_lemma name = 
      process_simple_formula (main_goal_of_theorem name) ;;

let dest_simple_lemma name =
  dest_simple_formula (main_goal_of_theorem name) ;;
let dest_simple_lemma_o name =
  dest_simple_formula (main_goal_of_theorem_o name) ;;


%[
Constructor for formulae.
]%

let mk_simple_formula xAs Bs C = 
  mk_iterated_all xAs (mk_iterated_implies (Bs @ [C])) ;;



%
****************************************************************************
General Existential Formula Preprocessing.
****************************************************************************
%

%
A general existential formula has form:

t =   Exists x:A. t'
    | t and t'
    | t or  t'
    | t' <=> t''  (defined as t' => t'' & t' <= t'')
    | Q           where Q has some other outermost constructor


Conceptually we regard such a formula as a composition of simple formulae,
each of form:

t =   all x:A. t'
    | P => t'
    | t' <= P    
    | Q           

Each simple formula has zero or more antecedents P and a single consequent Q. 

When one uses a general formula, one usually has a particular
consequent in mind.  We give each consequent an address - a list of
booleans. The address of a consequent specifies which way to turn at
each conjunct to reach the consequent from the outside. A `false'
element in an address means take the left conjunct, a `true' element
means take the right. Addresses are read from left to right.

If one desires a Q with one of the term constructors used in defining the 
structure of a general universal formula, it needs to be wrapped in a guard 
definition. Guard wrappers are removed automatically by functions and tactics
which work on general universal formulae.

A guard term is an abstraction with opid `guard`, no parameters and a single 
subterm with no variables bound.

process_general_formula breaks down a general formula into its component 
simple formulae. dest_general_formula separates out the parts of each simple
formula.

NB: As is doesn't do best thing if existentials are nested inside.
%

let proc_general_ex_formula_aux split_on_ors f'' = 
 letrec aux f = 

   let f' = unfold_soft_abs f in

   if is_term `function` f' & is_null_var (bvar_of_term f' 2 1) then
    (let [();B] = subterms_of_term f' 
     in
       (aux B)
    )
   if is_term `product` f' then
   ( if is_null_var (bvar_of_term f' 2 1) then

      (let [A;B] = subterms_of_term f' 
       in
          (aux A) @ (aux B)
      )
     else
      (let [[],A;[x],B] = bterms_of_term f' 
       in
         map (id # $. (x,A)) (aux B)
      )
   )
   if split_on_ors & is_term `union` f' then
    (let [A;B] = subterms_of_term f' in
     map ($. false # id) (aux A) 
     @ 
     map ($. true # id) (aux B) 
    )
   if is_term `guard` f then

     [[],[null_var, dest_guard f]]

   else
     [[],[null_var, f]]
 in
   aux f''
;;

let analyze_ex_clause_list cl =
  let props,binders = divide_list (is_null_var o fst) cl 
  in
    binders,map snd props
;;

let dest_gen_ex_fmla_aux split_on_ors f = 
  map
    (id # analyze_ex_clause_list)
    (proc_general_ex_formula_aux split_on_ors f)
;;

let dest_general_ex_formula = dest_gen_ex_fmla_aux true ;;
let dest_simple_ex_formula t = snd (hd (dest_gen_ex_fmla_aux false t)) ;;

%
****************************************************************************
Combinator typing processing
****************************************************************************
A combinator typing formula has form:

(\x1 ... \xm.op(y1;...;yn))  in  z1:A1 -> ... -> zm:Am -> T

  where subset {y1...yn} {x1... xm}

process_comb_typing inputs such a formula, and returns the triple 

[z1,A1;...;zn,An], op(y1...yn)[z1...zm/x1...xm], T

NB: some of the z might be null vars.E.g. zi. If the corresponding xi
does not occur in the ys, we return the null var. Otherwise we return the
xi in place of the zi.

Currently we don't check if the xi shadows some z1...zi-1. We assume
for the most part that the xs and zs will be the same.
%



let process_comb_typing comb_in_type =
  letrec aux cmb cmb_T cmb_vs ctxt = 
    if is_lambda_term cmb then
    ( let [cmb_v],cmb' = hd (bterms_of_term cmb)
      in let z,A,cmb_T' = dest_function cmb_T
      in let z' = 
         (if is_null_var z then
            (if member cmb_v (free_vars cmb') then cmb_v else null_var)
          else
            z
         )
      in
        aux cmb' cmb_T' (cmb_v.cmb_vs) ((z',A) . ctxt)
    )
    else
      cmb,cmb_T, rev cmb_vs, rev ctxt
  in
  let cmb_type,cmb = dest_member comb_in_type
  in let optm, optype, op_vs, ctxt = aux cmb cmb_type [] []
  in let sub = 
       mapfilter 
         (\x,z,().x, if is_null_var z then fail else mk_var_term z) 
         (zip op_vs ctxt)
  in
    ctxt, fo_subst sub optm, optype
;;


%
****************************************************************************
Auxiliary match functions
****************************************************************************
%

%
Used in forward and backward chaining to decide how to do match.
%

let dest_term_for_match flex t =
  if not flex then t
  if is_equal_term t then (let a,b = equands t in mk_pair_term a b)
  if is_member_term t then (subterm_of_term t 2)
  else t
;;

%
****************************************************************************
recognizers for computationally trivial type
****************************************************************************
A computationally trivial type is one with no interesting computational 
contents. Such types in hyp lists can always be unhidden.
%

%
this function could benefit from caching, or at the very least recognisers
for the most common logical abstractions. (all implies not).

Really only want to handle nots...
%

letref atomic_comp_trivial_abstractions = ``le gt ge squash``
;;

letrec is_atomic_comp_trivial_type t =
  let opid = opid_of_term t in
  if member opid ``not`` then
    is_atomic_comp_trivial_type (subterm_of_term t 1)
  if is_ab_term t then
  ( member opid atomic_comp_trivial_abstractions
    or
    is_atomic_comp_trivial_type (unfold_ab t)
  )
  else
     member opid ``void equal sqequal less_than``  
;;


%
returns a list of prs of addresses and bindings for each atomic
computationally trivial component of t.
%

let dest_comp_trivial_type t =
  map
    (\addr,x_A_prs,Bs,C.
        if is_atomic_comp_trivial_type C then
          addr, map fst x_A_prs
        else
          failwith `dest_comp_trivial_type: type not computationally trivial`
    )
    (dest_general_formula t)
;;
 
let is_comp_trivial_type t = (dest_comp_trivial_type t ; true) ? false
;;


