
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      break-control soft-break type-of-break-control text-of-break-control
	      advance-sequence sequence-stamp new-sequence-stamp
	      time-of-sequence-stamp count-of-sequence-stamp
	      process-id advance-transaction-sequence transaction-stamp 
	      transaction-of-stamp process-id-of-stamp sequence-of-stamp time-of-stamp
	      equal-stamps-p
	      tags-of-message pmsg-of-message stamp-of-message
	      warn-message warn-message-p error-message error-message-p format-message
	      raise-error
	      scan-position scan-bump scan-cur-byte scan-cur-char scan-eof-p scan-prime
	      scan-escape-p scan-at-whitespace-p
	      )))

      
;;;; -docs- (mod bsc)
;;;;
;;;;
;;;;	SequenceStamps:
;;;;
;;;;	The ability to stamp data with a unique time stamp is important in order
;;;;	to implement dependencies among objects. We require the stamp to be
;;;;	globally unique, ie given any two stamps we can test equality, even if the time
;;;;	stamps could have been created on different machines at the same time.
;;;;	Stamps will also have the property of being locally ordered, ie any stamps created
;;;;	within a process can be ordered.
;;;;
;;;;
;;;;	<stamp>			: (<transaction-id> . <sequence-stamp>)
;;;;	<transaction-id>	: (INTEGER{transaction counter} . <process-id>)
;;;;	<sequence-stamp>	: (INTEGER{sequence counter} . <universal-time>)
;;;;	<universal-time>	: INTEGER
;;;;	<process-id>		: <id{<host name>::<process number>::<universal-time>}>
;;;;	  * host name could alternatively be ip address.
;;;;
;;;;	*** Assumes process id is globally unique. 
;;;;
;;;;	process-id ()			: <process-id>
;;;;	init-process-id ()		: NULL
;;;;	 ** initializes the process id. Should be called only once.
;;;;
;;;;	advance-sequence ()		: NULL
;;;;	sequence-stamp ()		: <sequence-stamp>
;;;;	new-sequence-stamp ()		: <sequence-stamp>
;;;;	 ** advances sequence stamp
;;;;	sequence()			: INTEGER
;;;;
;;;;	advance-transaction-sequence 	: NULL
;;;;	transaction-stamp ()		: <stamp>
;;;;	new-transaction-id ()		: <transaction-id>
;;;;
;;;; -doce-


(defvar *sequence-count* 1)

(defun advance-sequence ()
  (incf *sequence-count*)
  (values))

(defun current-sequence ()
  *sequence-count*)

(defun sequence-stamp ()
  (cons *sequence-count* (get-universal-time)))

(defun dummy-sequence-stamp ()
  (cons 0 (get-universal-time)))

(defun static-sequence-stamp ()
  (cons 0 0))


(defun count-of-sequence-stamp (s) (car s))
(defun time-of-sequence-stamp (s) (cdr s))

(defun new-sequence-stamp ()
  (advance-sequence)
  (sequence-stamp))


(defvar *process-id* nil)

(defun init-process-id ()
  (setf *process-id*
	(intern-system (format-string "~x_~x_~x"
				     (local-host-ipaddr) (unix-process-id) (get-universal-time))))
  (values))

(defun process-id () *process-id*)


(defvar *transaction-sequence* 0)
(defvar *transaction-id* nil)

(defmacro with-transaction-id (trans-id &body body)
  `(let ((*transaction-id* *transaction-id*))

    ;; ,trans-id likely to be (new-transaction-id) which modifies *transaction-id* prior to new
    ;;  binding so we must bind first then modify or we screw up outer environment.
    (setf *transaction-id* ,trans-id)
  
    ,@body))

(defun transaction-id () *transaction-id*)

;; (defvar *transaction-stamp* (cons *transaction-id* 0))

(defun advance-transaction-sequence ()
  (incf *transaction-sequence*)
  ;;(break "ats")
  (setf *transaction-id* (cons *transaction-sequence* (process-id)))
  (values))


(defun transaction-stamp ()
  (cons *transaction-id* (sequence-stamp)))

(defun dummy-transaction-stamp ()
  (cons (or *transaction-id* (cons 0 'dummy))
	(dummy-sequence-stamp)))

(defun static-transaction-stamp (name)
  (cons (cons 0 name)
	(static-sequence-stamp)))

(defun new-transaction-stamp ()
  (advance-sequence)
  (transaction-stamp))

(defun future-transaction-stamp (jump)
  (let ((seq (sequence-stamp)))
    (cons *transaction-id* (cons (+ (car seq) jump) (cdr seq)))))


(defun jump-transaction-stamp (stamp)
  (if (<= (sequence-of-stamp stamp) *sequence-count*)
      (raise-error (error-message '(jump-transaction-stamp past) (sequence-of-stamp stamp) *sequence-count*))
    (setf *sequence-count* (cadr stamp))))

(defun check-jump-transaction-stamp (stamp)
  (when (<= (sequence-of-stamp stamp) *sequence-count*)
    (raise-error (error-message '(jump-transaction-stamp past) (sequence-of-stamp stamp) *sequence-count*))))


(defun new-transaction-id ()
  (advance-sequence)
  (advance-transaction-sequence)
  *transaction-id*)

(defun transaction-id-of-stamp (s) (car s))
  

(defun transaction-of-stamp (s) (caar s))
(defun process-id-of-stamp (s) (cdar s))
(defun sequence-of-stamp (s) (cadr s))
(defun time-of-stamp (s) (cddr s))
  
(defun dummy-transaction-stamp-p (stamp)
  (zerop (sequence-of-stamp stamp)))

(defmacro equal-stamps-p (a b) `(equal ,a ,b))
(defun equal-stamps-pf (a b) (equal a b))

(defun in-transaction-p (s &optional (other nil))
  (if other
      (and (eql (transaction-of-stamp s) (transaction-of-stamp other))
	   (eql (process-id-of-stamp s) (process-id-of-stamp other)))
      (and (eql (transaction-of-stamp s) (car *transaction-id*))
	   (eql (process-id-of-stamp s) (cdr *transaction-id*)))))

(defun equal-sequence-p (a b)
  (and (in-transaction-p a b)
       (eql (sequence-of-stamp a) (sequence-of-stamp b))))

(defun print-stamp (s &optional (stream t))
  (format stream "[~a:~a] ~a:"
	  (transaction-of-stamp s)
	  (sequence-of-stamp s)
	  (time-string (time-of-stamp s))))


(defun stamp-to-str (stamp)
  (format-string "~D-~D-~D-~a" 
		 (sequence-of-stamp stamp)
		 (time-of-stamp stamp)
		 (transaction-of-stamp stamp)
		 (process-id-of-stamp stamp)))

(defun datetime-string (utime)
    (mlet* (((s m h date mo yr day b z) (decode-universal-time utime)
	     (declare (ignore z b day s))))
	   (if (>= h 12)
	       (format-string "~a:~2,'0DPM ~a/~a/~a" (if (= h 12) 12 (- h 12)) m mo date yr)
	       (format-string "~a:~2,'0DAM ~a/~a/~a" (if (zerop h) 12 h) m mo date yr))))

;; no : or / to facilitate use as filename. 
(defun sortable-datetime-string (utime)
    (mlet* (((s m h date mo yr day b z) (decode-universal-time utime)
	     (declare (ignore z b day))))
	   (if (>= h 12)
	       (format-string "~a_~2,'0D_~2,'0D-PM-~2,'0D_~2,'0D_~2,'0D" yr mo date (if (= h 12) 12 (- h 12)) m s)
	       (format-string "~a_~2,'0D_~2,'0D-AM-~2,'0D_~2,'0D_~2,'0D" yr mo date (if (zerop h) 12 h) m s))))

(defun time-of-day (utime)
    (mlet* (((s m h date mo yr day b z) (decode-universal-time utime)
	     (declare (ignore date mo yr z b day))))

	   (values h m s)))

(defvar *last-gc* 0)

(defun nightly-gc (&optional forcep hook)
  ;;(format t "ng")
  (let ((ctime (get-universal-time)))
    (let ((h (time-of-day ctime)))
      (when (or forcep
		(and (>= h 2) (< h 7)))
	(when (or forcep
		  (> (- ctime *last-gc*) (* 18 60 60)))
	  (format t ";;; Nightly GC, last ~a, current ~a.~%"
		  (if (zerop *last-gc*)
		      "Never"
		      (datetime-string *last-gc*))
		  (datetime-string ctime))

	  (setf *last-gc* ctime)
	  
	  (when hook (funcall hook "nightly-gc" ctime))

	  #-cmu (gc t)
	  #+cmu (gc :full t)

	  (format t ";;; GC done at ~a.~%" (datetime-string (get-universal-time)))
	  (room))))))

;; check if time is normal workday hours and sleep if so.
;;  intent is for long running refinement to free up cycles during daytime
;;  when they may be needed for interactive use.

;; (list start stop interval) ; military time (hours) ie 1 - 24, minutes
(defvar *daily-sleeper* nil)

(defun daily-sleep (&optional iforcep) 

  (when *daily-sleeper*
    (terpri)

    (let ((slept nil)
	  (start (car *daily-sleeper*))
	  (stop (cadr *daily-sleeper*))
	  (interval (caddr  *daily-sleeper*))
	  (forcep iforcep))

      (do ()
	  ((let ((ctime (get-universal-time)))
	     (let ((h (time-of-day ctime)))
	       ;;(format t "~%Hour is ~a ~a ~a ~a" h start stop (and (>= h start) (< h stop)))
	       (not (or forcep
			(and (>= h start) (< h stop)))))))
	(setf slept t)
	(setf forcep nil)
	(format t "Sleeping ~a~% " (datetime-string (get-universal-time)))
	(sleep interval))
    
      (when slept
	(format t "WakingUp ~a~% " (datetime-string (get-universal-time)))
	(format t "StartGC  ~a~% " (datetime-string (get-universal-time)))
      
	#-cmu (gc t)
	#+cmu (gc :full t)

	(format t "FinishGC ~a~% " (datetime-string (get-universal-time))))
      )))

(defun time-string (utime)
    (mlet* (((s m h date mo yr day b z) (decode-universal-time utime)
	     (declare (ignore z b day mo date yr))))
	   (if (>= h 12)
	       (format-string "~a:~2,'0D:~2,'0DPM" (if (= h 12) 12 (- h 12)) m s)
	       (format-string "~a:~2,'0D:~2,'0DAM" (if (zerop h) 12 h) m s))))



  
;; if a is commit and b is start then returns true when a committed
;; before b started.
;; maybe should compare transaction ids first rather than sequences.
;; may allow clients to stamp broadcasts with stamps derived from servers transaction-id
;; and then still have comparable stamps.
(defun transaction-< (a b)
  ;;(break)
  (unless (eql (process-id-of-stamp a)
	       (process-id-of-stamp b))
    (raise-error (error-message '(stamp < differ))))

  (let ((time-a (time-of-stamp a))
	(time-b (time-of-stamp b)))

    (or (< time-a time-b)
	(when (= time-a time-b)
	  (< (sequence-of-stamp a) (sequence-of-stamp b))))))
	  

;;;;
;;;;	Files:
;;;;
;;;;
;;;;	with-prl-open-file ((stream fname direction) &body body)
;;;;

#+clos
(defmacro handle-file-error (&body body)
  `(handler-case (progn ,@body)
    (file-error (c)
     (raise-error (error-message '(file) (format t "~A~%" c))))))

#+clos
(defmacro handle-stream-error (&body body)
  `(handler-case (progn ,@body)
    (stream-error (c)
     (raise-error (error-message '(stream) (format t "~A~%" c))))
    (file-error (c)
     (raise-error (error-message '(stream file) (format t "~A~%" c))))))

#-clos
(defmacro handle-file-error (&body body)
  `(progn ,@body))

#-clos
(defmacro handle-stream-error (&body body)
  `(progn ,@body))

(defmacro handle-abort (kind &body body)
  (let ((okp (gentemp))
	(r (gentemp)))
  `(let ((,okp nil))
    (let ((,r (catch 'abort
	       (prog1 (progn ,@body)
		 (setf ,okp t)))))
      (if ,okp ,r
	  (progn
	    ;; noticed segfault when gc + stack overflow so thinking
	    ;; that if one is about to repeat the same mistatke,
	    ;; then gc now may save them later.
	    (gc)
	    (raise-error (error-message (cons 'abort ,kind) ,r))))))))

(defvar *abort-break* nil)
(defmacro raise-abort (tag)
  `(progn
    (when *abort-break* (break ,tag))
    (throw 'abort (or ,tag t))))

(defvar *stack-overflow-break-p* nil)

#+allegro
(defmacro handle-stack-overflow (kind &body body)
  `(handle-abort (cons 'stack-overflow ,kind)
    (handler-case (progn ,@body)
      (excl:synchronous-operating-system-signal (c)
	(when *stack-overflow-break-p*
	  (break "stack-overflow"))
	(format t "stack-overflow aborted")
	(raise-abort (format-string "~A" c))))))

#-allegro
(defmacro handle-stack-overflow (kind &body body)
  `(handle-abort (cons 'serious-condition ,kind)
    (handler-case (progn ,@body)
      (serious-condition (c)
	(raise-abort (format-string "~A" c))))))





;;;;	Need to open files for byte io rather than character io to
;;;;	allow files to be shared by lisps on all platforms without
;;;;	concern for character <->code mappings.
;;;;	
;;;;	Unfortunately, lucid run program is restricted to char io.		
;;;;	Thus need to make stream io abstract so as to hide lucid
;;;;	run-program char io.

;;;; -docs- (mod io)
;;;;	
;;;;	Prl streams : an abstract interface for stream byte IO.
;;;;	
;;;;	Allows lisp character streams to be treated as byte streams.
;;;;	This capability is convenient if for some reason you need
;;;;	to use a character stream for IO.
;;;;	
;;;;	new-prl-in-stream (STREAM <closure{read})	: <prl-in-stream>
;;;;	  * read () : BYTE8
;;;;	
;;;;	new-prl-out-stream (STREAM <closure{write})	: <prl-out-stream>
;;;;	  * write (BYTE8) : NULL
;;;;	
;;;;	prl-stream-close(<prl-stream>)			: NULL
;;;;	  * <prl-stream>	: <prl-in-stream> | <prl-out-stream>
;;;;	
;;;;	prl-stream-listen (<prl-in-stream>)		: BOOL
;;;;	prl-stream-read (<prl-in-stream>)		: BYTE8 | nil
;;;;	  * returns nil on eof.
;;;;
;;;;	prl-stream-write (<prl-out-stream> BYTE8)	: NULL
;;;;	prl-stream-finish-output(<prl-out-stream>)	: NULL
;;;;	
;;;;	** vaporware
;;;;	prl-stream-write-array (<prl-out-stream> BYTE8 array 
;;;;				&optionalINT{offset} INT{count})	
;;;;	  : NULL
;;;;	  * write bytes from array starting at offset or start for count
;;;;	    or until end.
;;;;	
;;;;	prl-stream-write-string (<prl-out-stream> STRING
;;;;				&optionalINT{offset} INT{count})	
;;;;	  : NULL
;;;;	  * write chars from string starting at offset or start for count
;;;;	    or until end.
;;;;	  * All characters of the string must satisfy  (standard-char-p ch).
;;;;	    If not then string should have been converted to byte array
;;;;	    and written via write-array. If writing to compressed ascii stream
;;;;	    writing not standard chars via this function will corrupt the
;;;;	    compression resulting in unreadable terms. 
;;;;	  * This function allows for optimization when the underlying stream is
;;;;	    a character stream and the string contains only standard chars.
;;;;	    However, one should not purposely convert data to such strings since
;;;;	    the goal is to use byte streams and byte arrays as the underlying io
;;;;	    media.
;;;;	
;;;;	** on reflection this is misguided since strings need to do escaped
;;;;	   and we just don't have many strings that are already escaped.
;;;;	
;;;;	** we're taking an end run around prl-stream anyways by allowing direct
;;;;	   write to a buffer for the stream. It may make sense for the buffer to be
;;;;	   a string and to have string funcs for inserting into the buffer.
;;;;
;;;;	
;;;;	** Buffering : where convenient it is deemed beneficial to buffer as
;;;;	** close to the source as possible. This reduces the number of calls 
;;;;	** to low level functions.
;;;;	**
;;;;	** However, does the overhead of buffering your self outweigh 
;;;;	** the cost of the calls to prl-stream-write-byte
;;;;	** 
;;;;	** If we macrofy, the buffering could be a win.
;;;;	** 
;;;;	** Best of both worlds : buffer on prl-stream level, but
;;;;	** allow direct access to buffers by callers. ie in some context
;;;;	** caller gets handle to streams buffers and directly (via macros)
;;;;	** inserts bytes into the buffer.
;;;;	
;;;;	** For this purpose a disposable byte array buffers are available.
;;;;	** Once written the buffer is discard by the write function and should
;;;;	** not be reused by the caller.
;;;;	**
;;;;	**
;;;;	** Prl streams could buffer to reduce number of calls to write-byte.
;;;;	** But then must insure flush gets called. Maybe make optional
;;;;	** so that only callers known to flush use buffering. Although
;;;;	** all callers may already behave well.
;;;;	
;;;;	prl-stream-read-array(<prl-in-stream> BYTE8-ARRAY INT{count}) : INT
;;;;	  * int returned is count read.
;;;;	  * this is probably not a useful as write-array as you would normally
;;;;	    be scanning the input a byte at a time anyways.
;;;;	
;;;;	** links parameterized by a prl-stream.
;;;;
;;;; -doce- 

(eval-when (compile)
  (proclaim '(inline
	      prl-stream-close prl-stream-in-stream prl-stream-out-stream
	      prl-stream-write prl-stream-finish-output
	      prl-stream-listen prl-stream-read)))




;;;;	
;;;;	Fast byte buffers :
;;;;	  - provide for macro manipulation.
;;;;	  - meant to be more persistent then byte buffers, ie no copy at end.
;;;;	  - exportable to caller.
;;;;	  - tied to prl-stream.
;;;;	  - copy-seq and declares to allow for compiler optimization.

;; need alternate implementation which just calls accumulators.
;; only reason not to always implicitly buffer is in case
;; someone is not calling finish-output.


(defconstant *byte-buffer-size* 4096)

(defvar *fast-byte-buffer*)
(defvar *fast-byte-buffer-array*)
(defvar *fast-byte-buffer-fill*)
(defvar *fast-byte-buffer-size*)


(defmacro fbb-space (count)
  `(when (> (+ ,count *fast-byte-buffer-fill*) *fast-byte-buffer-size*)
    (fbb-flush *fast-byte-buffer*)
    (setf *fast-byte-buffer-fill* 0)))

(defmacro fbb-single-space ()
  `(when (= *fast-byte-buffer-fill* *fast-byte-buffer-size*)
    (fbb-flush *fast-byte-buffer*)
    (setf *fast-byte-buffer-fill* 0)))

(defmacro fbb-add-byte (b)
  `(progn
    (fbb-single-space)
    #-fbbstring(setf (aref *fast-byte-buffer-array* *fast-byte-buffer-fill*) ,b)
    #+fbbstring(setf (char *fast-byte-buffer-array* *fast-byte-buffer-fill*) (code-char ,b))
    (incf *fast-byte-buffer-fill*)))

;; assume if fbbstring then array is also a string.
(defmacro fbb-add-array (a &optional offset count)
  (let ((aa (gensym))
	(cc (gensym))
	(l (gensym))
	(o (gensym)))
    `(let ((,aa ,a)
	   (,o (or ,offset 0))
	   (,cc ,count)
	   )
      (declare
       #-fbbstring (type (vector (unsigned-byte 8)) ,aa)
       #+fbbstring (type (simple-array character) ,aa)
	(type fixnum ,o)
       )
      (let ((,l (or ,cc (- (length ,aa) ,o))))
	(fbb-space ,l)
	(if (> ,l *fast-byte-buffer-size*)
	    (break)
	    (progn
	      (replace *fast-byte-buffer-array* ,aa
		       :start1 *fast-byte-buffer-fill*
		       :end1 (+ *fast-byte-buffer-fill* ,l)
		       :start2 (or ,offset 0))
	      (incf *fast-byte-buffer-fill* ,l)))))))


;; for the occasional serendipitous case where we have a known escaped standard
;; string. Do not purposely produce such strings instead produce buffers
;; and use add array.
(defmacro fbb-add-standard-string (s &optional offset count)
  (let ((ss (gensym))
	(cc (gensym))
	(l (gensym))
	(o (gensym))
	)
    `(let ((,ss ,s)
	   (,o (or ,offset 0))
	   (,cc ,count)
	   )
      (declare
       (type (simple-array character) ,ss)
       (type fixnum ,o)
       )
      (let ((,l (or ,cc (- (length ,ss) ,o))))
	(fbb-space ,l)
	(if (> ,l *fast-byte-buffer-size*)
	    (break)
	    (progn
	      #+fbbstring (replace *fast-byte-buffer-array* ,ss
				   :start1 *fast-byte-buffer-fill*
				   :end1 (+ *fast-byte-buffer-fill* ,l)
				   :start2 (or ,offset 0))
	      #-fbbstring (dotimes (i ,l)
			    (setf (aref *fast-byte-buffer-array* (+ i *fast-byte-buffer-fill*))
				  (char-code (char ,ss i))))		
	      (incf *fast-byte-buffer-fill* ,l)))))))


(defmacro fbb-add-int-bytes (n)
  `(progn
    (fbb-space 4)
    #+fbbstring (setf (char buf *fast-byte-buffer-fill*) (code-char (ldb (byte 8 0) ,n)))
    #-fbbstring (setf (aref buf *fast-byte-buffer-fill*) (ldb (byte 8 0) ,n))
    #+fbbstring (setf (char buf (incf *fast-byte-buffer-fill*)) (code-char (ldb (byte 16 8) ,n)))
    #-fbbstring (setf (aref buf (incf *fast-byte-buffer-fill*)) (ldb (byte 16 8) ,n))
    #+fbbstring (setf (char buf (incf *fast-byte-buffer-fill*)) (code-char (ldb (byte 24 16) ,n)))
    #-fbbstring (setf (aref buf (incf *fast-byte-buffer-fill*)) (ldb (byte 24 16) ,n))
    #+fbbstring (setf (char buf (incf *fast-byte-buffer-fill*)) (code-char (ldb (byte 32 24) ,n)))
    #-fbbstring (setf (aref buf (incf *fast-byte-buffer-fill*)) (ldb (byte 32 24) ,n))
    (incf *fast-byte-buffer-fill*)))

;; ?? but what if (prl-stream-fbb ,prl-stream) is null??
;; could add checks here, or fail, or make up dummy.

;; if  (prl-stream-fbb ,prl-stream) null then dynamically create buffer and flush on exit!!!

(defmacro with-fbb (prl-stream &body body)
  `(let* ((*fast-byte-buffer* (or (prl-stream-fbb ,prl-stream) (new-fbb ,prl-stream t))))
    (let ((*fast-byte-buffer-array* (fbb-array *fast-byte-buffer*))
	  (*fast-byte-buffer-fill* (fbb-fill *fast-byte-buffer*))
	  (*fast-byte-buffer-size* (fbb-size *fast-byte-buffer*)))
      (declare
       #-fbbstring (type (vector (unsigned-byte 8)) *fast-byte-buffer-array*)
       #+fbbstring (type (simple-array character) *fast-byte-buffer-array*)
       (type fixnum *fast-byte-buffer-fill*)
       (type fixnum *fast-byte-buffer-size*))

      (unwind-protect
	   (progn ,@body)
	(if (fbb-ephemeral *fast-byte-buffer*)
	    (fbb-flush *fast-byte-buffer*)
	    (setf (fbb-fill *fast-byte-buffer*) *fast-byte-buffer-fill*)
	    )))))


;; try to warn if -fbbstring but stream uses character io.
;; try to warn if +fbbstring but stream uses byte io.

(defstruct prl-stream
  stream
  read-f
  write-f
  write-buffer-f
  fbb)

(defstruct fbb
  prl-stream
  array
  fill
  size
  ephemeral
  free
  )

(defvar *fbb-pool*
  (list
   (make-fbb :array
	     #-fbbstring (make-array *byte-buffer-size* :element-type '(unsigned-byte 8))
	     #+fbbstring (make-string *byte-buffer-size*)

	     :fill 0
	     :size *byte-buffer-size*
	     )))

(defun get-fbb ()
  (if *fbb-pool*
      (let ((fbb (pop *fbb-pool*)))
	;;(format t "~%get-fbb")
	(setf (fbb-free fbb) nil)
	fbb)
      (progn
	;;(format t "~%make-fbb")
	(make-fbb :array
		  #-fbbstring (make-array *byte-buffer-size* :element-type '(unsigned-byte 8))
		  #+fbbstring (make-string *byte-buffer-size*)
		  :fill 0
		  :size *byte-buffer-size*
		  ))
      ))

(defun put-fbb (fbb)
  ;;(format t "~%put-fbb")  
  (if (fbb-free fbb)
      (break "fbb free")
      (progn
	(setf (fbb-free fbb) t)
	(push fbb *fbb-pool*)))
  )
    
(defun new-fbb (pstream &optional ephemeral)

  (let ((fbb (get-fbb)))
    (setf (fbb-prl-stream fbb) pstream
	  (fbb-ephemeral fbb) ephemeral
	  (fbb-fill fbb) 0)
    fbb))


(defun fbb-flush (fbb)
  ;;(setf a fbb) (break "ff")
  (let ((pstream (fbb-prl-stream fbb)))
    (funcall (or (prl-stream-write-buffer-f pstream) (prl-stream-default-write-buffer pstream))
	     (prl-stream-stream pstream)
	     (fbb-array fbb)
	     0
	     (or (and (boundp '*fast-byte-buffer-fill*) *fast-byte-buffer-fill*)
		 (fbb-fill fbb)))
    (setf (fbb-fill fbb) 0)))


;;;;	write-buffer-f (prl-stream buffer count &optional offset)
;;;;	  - fbbstring feature indicates buffer implementation.
;;;;	  - expect prl-stream caller to supply more efficient replacement
;;;;	    which uses stream specific write-sequence call.
;;;;	  - if there is not an efficient write-sequence call using a buffer
;;;;	    may still be a win by localizing the accumulation of data from
;;;;	    the io.
;;;;	
;;;;	Expected to be used to flush the fbb, but may be called 
;;;;	more directly if the prl-stream does not have an fbb.
;;;;	

(defun prl-stream-default-write-buffer (prl-stream)
  ;;(format t "default-write-buffer~%")(break "dwb")
  (let ((wb (prl-stream-write-f prl-stream))
	)
    #'(lambda (s buffer offset count)
	(macrolet ((my-write-byte (i)
		     `(let ((item
			     #-fbbstring (aref buffer ,i)
			     #+fbbstring (char buffer ,i)
			     ))

		       (when (null item)
			 (break "pswa")
			 (return-from prl-stream-default-write-buffer))
		       #-fbbstring (funcall wb item s)
		       #+fbbstring (funcall wb (char-code item) s)
		       )))
	
	  (if offset
	      (dotimes (i count) (my-write-byte (+ i offset)))
	      (dotimes (i count) (my-write-byte i))))
	nil)))

(defun new-prl-in-stream (stream &optional (read-f #'read-byte))
  (make-prl-stream :stream stream
		   :read-f read-f
		   :write-f nil))

(defun new-prl-out-stream (stream &optional
			   (write-f #'write-byte)
			   write-buffer-f
			   )

  (let ((pstream (make-prl-stream :stream stream
				  :read-f nil
				  :write-f write-f
				  :write-buffer-f write-buffer-f
				  )))
    (when write-buffer-f
      (setf (prl-stream-fbb pstream) (new-fbb pstream)))
    pstream))
      

(defun new-prl-in-out-stream (stream read-f write-f
				     &optional write-buffer-f)
  (make-prl-stream :stream stream
		   :read-f read-f
		   :write-f write-f
		   :write-buffer-f write-buffer-f))


;; returns nil on eof, byte or char otherwise.
(defun prl-stream-read (stream)
  (handle-stream-error
   (let ((ch (funcall (prl-stream-read-f stream)
		      (prl-stream-stream stream)
		      nil nil)))
     ;;(format t "~%PSR ~a ~a" (code-char ch) ch)
     ch)))

(defun prl-stream-write (item stream)
  ;;(format t "~%PSW ~a ~a" (code-char item) item)
  (handle-stream-error
   (when (null item)
     (break "psw"))
   (let ((fbb (prl-stream-fbb stream)))
     (if fbb
	 (with-fbb stream
	   (fbb-add-byte item))
	 (funcall (prl-stream-write-f stream)
		  item
		  (prl-stream-stream stream))))))

(defun prl-stream-write-buffer (stream buffer &optional offset count)
  (handle-stream-error
   (let ((fbb (prl-stream-fbb stream)))
     (if fbb
	 (with-fbb stream
	   (fbb-add-array buffer offset count))
	 (funcall (prl-stream-write-buffer-f stream)
		  (prl-stream-stream stream)
		  buffer
		  offset
		  count)))))

(defun prl-stream-default-write-string (p s)
  (handle-stream-error
   (let ((wb (prl-stream-write-f p))
	 (stream (prl-stream-stream p)))
     (dotimes (i (length s))
       (funcall wb stream (char-code (char s i)))))))

;;gross kludge
(defun prl-char-stream-write-string (p s)
  (handle-stream-error
   (write-string s (prl-stream-stream p))
   ))


(defun prl-stream-write-string (stream s)
  (handle-stream-error
   (let ((fbb (prl-stream-fbb stream)))
     (if fbb
	 (with-fbb stream
	   (fbb-add-standard-string s))
	 #+fbbstring (if (prl-stream-write-buffer-f stream)
			 (funcall (prl-stream-write-buffer-f stream)
				  (prl-stream-stream stream)
				  s)
			 (prl-stream-default-write-string stream s))
	 #-fbbstring (prl-stream-default-write-string stream s)
	 ))))


(defun prl-stream-close (stream)
  (close (prl-stream-stream stream)))

(defun prl-stream-listen (stream)
  (listen (prl-stream-stream stream)))

(defun prl-stream-finish-output (stream)
  (handle-stream-error
   (let ((fbb (prl-stream-fbb stream)))
     (when fbb
       (fbb-flush fbb)))
   (force-output (prl-stream-stream stream))))
  
(defun prl-stream-free-fbb (stream)
  ;;(format t "~%free fbb")
  (let ((fbb (prl-stream-fbb stream)))
    (when fbb
      (put-fbb fbb))))

  

(defun prl-stream-file-position (stream pos)
  (let ((fbb (prl-stream-fbb stream)))
    (when fbb
      (fbb-flush fbb)))

  (file-position (prl-stream-stream stream) pos))


#+fbbstring
(defun char-stream-write-buffer (stream b offset count)
  (write-string b stream :start offset :end (+ offset count)))


;; RLE NAP : sexpr-p is a kludge, sexpr io has dim future so at some point this
;; RLE NAP : shoud be ripped out.
(defmacro with-prl-open-file ((stream fname in-or-out &optional sexpr-p sexpr-pretty-p) &body body)
  (let ((v (gentemp)))
    (cond
     ((eql in-or-out 'in)
      `(handle-file-error
	(with-open-file (,v ,fname
			    :direction :input
			    :element-type (if ,sexpr-p 'base-character '(unsigned-byte 8))
			    :if-does-not-exist nil)
			(when (null ,v)
			      (raise-error (error-message '(file open) ',in-or-out ,fname)))
			(let ((,stream (new-prl-in-stream ,v
							  (if ,sexpr-p
							      #'read
							    #'read-byte))))
			  ,@body))))
     ((eql in-or-out 'out)
      `(handle-file-error
	(prog1
	    (with-open-file (,v ,fname
				:direction :output
				:if-exists :new-version
				:element-type (if ,sexpr-p
						  'character
						  #+fbbstring 'character
						  #-fbbstring '(unsigned-byte 8)


						  )
				:if-does-not-exist :create)
			    (let ((,stream (new-prl-out-stream ,v
							       (if ,sexpr-p
								   #'(lambda (sexpr stream)
								        (write sexpr
									      :stream stream
									      :pretty ,sexpr-pretty-p
									      :level nil
									      :length nil)
								       (terpri stream)
								       (terpri stream))
								   #'write-byte)
							       (unless ,sexpr-p
								 #+fbbstring #'char-stream-write-buffer
								 #-fbbstring #'byte-stream-write-buffer)
							       )))
			      (prog1 (progn
				       ,@body)
				(prl-stream-finish-output ,stream)
				(prl-stream-free-fbb ,stream)
				))))))
     ((eql in-or-out 'append)
      `(handle-file-error
	(prog1
	    (with-open-file (,v ,fname
				:direction :output
				:if-exists :append
				:element-type '(unsigned-byte 8)
				:if-does-not-exist :create)
			    (let ((,stream (new-prl-out-stream ,v #'write-byte)))
			      ,@body)))))
     ((eql in-or-out 'io)
      `(handle-file-error
	(prog1
	    (with-open-file (,v ,fname
				:direction :io
				:if-exists :append;;lal
				:element-type (if ,sexpr-p 'character '(unsigned-byte 8))
				:if-does-not-exist :create)
			    (let ((,stream (new-prl-in-out-stream ,v
								  #'read-byte
								  #'write-byte)))
			      ,@body)))))
      
     (t ;;(setf a in-or-out)
	(break "???") in-or-out))))

(defmacro with-safe-prl-open-file ((stream fname in-or-out) &body body)
  (let ((v (gentemp)))
    `(cond
      ((eql ,in-or-out 'in)
       (handle-file-error
	(with-open-file (,v ,fname
			    :direction :input
			    :element-type '(unsigned-byte 8)
			    :if-does-not-exist nil)
	  (when (null ,v)
	    (raise-error (error-message '(file open) ,in-or-out ,fname)))
	  (let ((,stream (new-prl-in-stream ,v
					    #'read-byte)))
	    ,@body))))
      ((eql ,in-or-out 'out)
       (handle-file-error
	(prog1
	    ;; with-open-file adds a delay of 120 msec.
	    (with-open-file (,v ,fname
				:direction :output
				:if-exists :new-version
				:element-type
				#+fbbstring 'character
				#-fbbstring '(unsigned-byte 8)
				:if-does-not-exist :create)

	      ;; no time issue here
	      (let ((,stream (new-prl-out-stream ,v
						 #'write-byte
						 #+fbbstring #'char-stream-write-buffer
						 #-fbbstring #'byte-stream-write-buffer
						 )))
		(prog1 (progn
			 ,@body)
		  (prl-stream-finish-output ,stream)
		  (prl-stream-free-fbb ,stream)
		  )))
	  )))
      ((eql ,in-or-out 'append)
       (handle-file-error
	(prog1
	    (with-open-file (,v ,fname
				:direction :output
				:if-exists :append
				:element-type '(unsigned-byte 8)
				:if-does-not-exist :create)
	      (let ((,stream (new-prl-out-stream ,v #'write-byte)))
		,@body)))))
      ((eql ,in-or-out 'io)
       (handle-file-error
	(prog1
	    (with-open-file (,v ,fname
				:direction :io
				:if-exists :append;;lal
				:element-type '(unsigned-byte 8)
				:if-does-not-exist :create)
	      (let ((,stream (new-prl-in-out-stream ,v
						    #'read-byte
						    #'write-byte)))
		,@body)))))
      
      (t ;;(setf a ,in-or-out)
       (break "???") ,in-or-out))))


(defun prl-open-file (in-or-out fname)
  (if (eql in-or-out 'in)
      (handle-file-error
       (new-prl-in-stream (open fname
				:direction :input
				:element-type '(unsigned-byte 8)
				:if-does-not-exist :error)
			  #'read-byte))
      (handle-file-error
       (new-prl-out-stream (open fname
				 :direction :output
				 :if-exists :new-version
				 :element-type '(unsigned-byte 8)
				 :if-does-not-exist :create)
			   #'write-byte))))

(defun prl-close-file (prl-stream)
  (prl-stream-close prl-stream))



;;;;
;;;; Basic IO : 
;;;;

;;;;	ichar : code or token eg 'mouse-left
;;;;
;;;;	Constants : 
;;;;	 isemicolon	#\;
;;;;	 icolon		#\"
;;;;	 ilparen	#\(
;;;;	 irparen	#\)
;;;;	 ilangle	#\<
;;;;	 irangle	#\>
;;;;	 ilcurly	#\{
;;;;	 ircurly	#\}
;;;;	 ilsquare	#\[
;;;;	 irsquare	#\]
;;;;	 inot		#\^
;;;;	 iat		#\@
;;;;	 idot		#\.
;;;;	 idash		#\-
;;;;	 iunder		#\_
;;;;	 icomma		#\,
;;;;	 idollar	#\$
;;;;	 ipercent	#\%
;;;;	 iequals	#\=
;;;;	 iquote		#\'
;;;;	 ibackquote	#\`
;;;;	 istringquote	#\"
;;;;	 islash		#\/
;;;;	 iescape	#\\
;;;;	 ibar		#\|
;;;;	 irubout	#\rubout
;;;;	 isplat		#\#
;;;;	 itab		#\tab
;;;;	 ipage		#\page
;;;;	 ispace		#\space
;;;;	 ireturn	#\return
;;;;	 inewline	#\newline
;;;;	 ilinefeed	#\linefeed
;;;;
;;;;
;;;;
;;;;	explode(tok) : list of single character tokens.
;;;;
;;;;	print-object-to-string (object &optional copyp) : returns string.
;;;;	princ-object-to-string (object &optional copyp) : returns string.
;;;;	 ** unless copyp t, string returned may be destructively modified the next
;;;;	 ** time prin*-object-to-string is called.
;;;;


;;;;	following should be ascii codes.
(defconstant isemicolon (char-code #\;))
(defconstant icolon (char-code #\:))
(defconstant ilparen (char-code #\())
(defconstant irparen (char-code #\)))
(defconstant ilangle (char-code #\<))
(defconstant irangle (char-code #\>))
(defconstant ilcurly (char-code #\{))
(defconstant ircurly (char-code #\}))
(defconstant ilsquare (char-code #\[))
(defconstant irsquare (char-code #\]))
(defconstant inot (char-code #\^))
(defconstant iat (char-code #\@))
(defconstant idot (char-code #\.))
(defconstant idash (char-code #\-))
(defconstant iunder (char-code #\_))
(defconstant icomma (char-code #\,))
(defconstant idollar (char-code #\$))
(defconstant ipercent (char-code #\%))
(defconstant iequals (char-code #\=))
(defconstant iquote (char-code #\'))
(defconstant ibackquote (char-code #\`))
(defconstant istringquote (char-code #\"))
(defconstant islash (char-code #\/))
(defconstant iescape (char-code #\\))
(defconstant ibar (char-code #\|))
(defconstant irubout (char-code #\rubout))
(defconstant ibackspace (char-code #\backspace))
(defconstant isplat (char-code #\#))
(defconstant itilde (char-code #\~))
(defconstant iplus (char-code #\+))
(defconstant iexclaimation (char-code #\!))
(defconstant iampersand (char-code #\&))
(defconstant istar (char-code #\*))
(defconstant ispace (char-code #\space))
(defconstant iquestion (char-code #\?))
(defconstant iequal (char-code #\=))
(defconstant ieot (char-code #\eot))

;;;;

;;;;	Allegro for the PC encodes newline as (control-m) so we do 	
;;;;	conditionalization to hide this brain damage.
;;;;	


(defconstant *standard-character-sbits*
  (standard-character-sbits *standard-character-codes*))

(defun standard-character-code-p (code)
  (test-standard-character-bit code *standard-character-sbits*))

(defun standard-character-p (ch)
  (standard-character-code-p (char-code ch)))

;;;;	
;;;;	These should only be used on raw ascii code input or 16bit char codes.	
;;;;	It is wrong to use on standard-char codes because some do not map
;;;;	to the codes specified here.

(defvar *whitespace-sbits* (standard-character-sbits
				 (list inewline ispace itab ireturn ilinefeed)))


;; ??? : either all whitespace should appear in the following or none. Or do we
;; ???	do we assume that other whitespace
;; ???	
;; ???	so a tab in a string need not be escaped but a tab encountered during
;; ???	whitespace skip will be skipped. I'm not so sure this is good.
;; ???	

(defvar *ascii-escape-sbits*
  (standard-character-sbits
   (list iescape
	 ispace inewline ireturn itab
	 ilparen irparen ilsquare irsquare ilcurly ircurly
	 icolon isemicolon idot icomma
	 )))

(defvar *no-escape-sbits* (standard-character-sbits nil))


(defconstant *break-sbits* (standard-character-sbits (list inewline ilinefeed ireturn)))

(defmacro break-code-p (code) `(test-standard-character-bit ,code *break-sbits*))
(defmacro whitespace-code-p (code)
  `(test-standard-character-bit ,code *whitespace-sbits*))



(defun blank-string (l)
  (make-string l :initial-element #\Space))



(defvar *numeric-digit-code-sbits*
  (standard-character-sbits
   (map 'list #'char-code "0123456789")))

(defun numeric-digit-code-p (code)
  (test-standard-character-bit code *numeric-digit-code-sbits*))


(defconstant *zero-digit-code* (char-code #\0))

(defun numeric-digit-code-to-int (code)
  (- code *zero-digit-code*))


(defvar *hex-upper-alpha-digit-code-sbits*
  (standard-character-sbits
   (map 'list #'char-code "ABCDEFG")))

(defvar *hex-lower-alpha-digit-code-sbits*
  (standard-character-sbits
   (map 'list #'char-code "abcdefg")))

(defun hex-lower-alpha-digit-code-p (code)
  (test-standard-character-bit code *hex-lower-alpha-digit-code-sbits*))

(defun hex-upper-alpha-digit-code-p (code)
  (test-standard-character-bit code *hex-upper-alpha-digit-code-sbits*))


(defconstant *upper-a-digit-code* (char-code #\A))
(defconstant *lower-a-digit-code* (char-code #\a))
 
(defun hex-upper-alpha-digit-code-to-int (code)
  (+ 10 (- code *upper-a-digit-code*)))

(defun hex-lower-alpha-digit-code-to-int (code)
  (+ 10 (- code *lower-a-digit-code*)))


(defvar *hex-code-sbits* 
  (standard-character-sbits
   (map 'list #'char-code "0123456789abcdefABCDEF")))

(defun hex-code-p (code)
  (test-standard-character-bit code *hex-code-sbits*))

(defvar *hex-code-ints*
  (let ((a (make-array 128 :element-type 'fixnum)))
    (dotimes (i 128)
      (setf (aref a i) (cond
			 ((numeric-digit-code-p i)
			  (numeric-digit-code-to-int i))
			 ((hex-lower-alpha-digit-code-p i)
			  (hex-lower-alpha-digit-code-to-int i))
			 ((hex-upper-alpha-digit-code-p i)
			  (hex-upper-alpha-digit-code-to-int i))
			 (t -1))))
    a))

(defun hex-code-to-int (code)
  (aref *hex-code-ints* code))


(defun hex-char-to-int (ch)
  (hex-code-to-int (character-to-code ch)))

(defun hex-char-p (ch)
  (hex-code-p (character-to-code ch)))

(defun hex-string-to-int (s)
  (let ((i 0))
    (dotimes (j (length s))
      (setf i (+ (hex-char-to-int (char s j)) (* i 16))))
    i))

;;;;	Convert integer to character string.

(defvar *standard-character-string-array* 
  (let ((a (make-array 128 :element-type 'string :initial-element "")))
    (dotimes (i 128)
      (when (standard-character-code-p i)
	(setf (aref a i)
	      (make-string 1 :initial-element (aref *standard-ascii-character-array* i)))))

    a))


;; array of length 256 containing byte-arrays of length 2. Each byte-array is hex rep of array index.
(defvar *hex-ascii-digit-byte-array*
  (let ((a (make-array 256
		       :element-type '(vector (unsigned-byte 8))
		       #+cmu :initial-element #+cmu(make-array 2 :element-type '(unsigned-byte 8)) ;; just to shut it up.
		       )))
    (dotimes (i 256)
      (let ((s (format-string "~2,'0X" i))
	    (b (make-array 2 :element-type '(unsigned-byte 8))))
	(setf (aref b 0) (character-to-code (schar s 0))
	      (aref b 1) (character-to-code (schar s 1))
	      (aref a i) b)))
    a))

;; array of length 256 containing strings of length 2. Each string is hex rep of array index.
(defvar *hex-ascii-digit-string-array*
  (let ((a (make-array '(256) :element-type 'string :initial-element "")))
    (dotimes (i 256)
      (let ((s (format-string "~2,'0X" i)))
	(setf (aref a i) s)))
    a))


(defvar *first-byte-address* (byte 8 8))
(defvar *second-byte-address* (byte 8 0))


(defun unicode-first-byte-string (code)
  (aref *hex-ascii-digit-string-array* (ldb *first-byte-address* code)))

(defun unicode-second-byte-string (code)
  (aref *hex-ascii-digit-string-array* (ldb *second-byte-address* code)))

(defun unicode-first-byte-array (code)
  (aref *hex-ascii-digit-byte-array* (ldb *first-byte-address* code)))

(defun unicode-second-byte-array (code)
  (aref *hex-ascii-digit-byte-array* (ldb *second-byte-address* code)))


;;; int codes a character, return character string containing char coded.
;;; unicode without 16bit-chars maps to embedding string.

;;;;	acc-f (CHARACTER | STRING)	: NULL

(defun int-to-character-string (i &optional acc-f)
  (if (<= i #xffff)
      (if (standard-character-code-p i)
	  (if acc-f
	      (funcall acc-f (byte-to-standard-character i))
	      (aref *standard-character-string-array* i))

	  #+16bit-chars
	  (if acc-f
	      (funcall acc-f (code-char i))
	      (make-string 1 :initial-element (code-char i)))

	  #-16bit-chars
	  ;; embedded unicode  2 bytes -> 5 standard-chars.
	  (if acc-f
	      (progn
		(funcall acc-f #\\)
		(let ((b (unicode-first-byte-string i)))
		  (funcall acc-f (schar b 0))
		  (funcall acc-f (schar b 1)))
		(let ((b (unicode-second-byte-string i)))
		  (funcall acc-f (schar b 0))
		  (funcall acc-f (schar b 1))))
	      (let ((s (make-string 5)))
		(setf (char s 0) #\\)
		(let ((b (unicode-first-byte-string i)))
		  (setf (schar s 1) (schar b 0)
			(schar s 2) (schar b 1)))
		(let ((b (unicode-second-byte-string i)))
		  (setf (schar s 3) (schar b 0)
			(schar s 4) (schar b 1)))
		s))
	  )

      (raise-error (error-message '(int character bounds) i))))


;;;
;;;	Currently ML treats characters as tokens.
;;;	Thus there is much interning of characters as tokens and
;;;	vice versa. To speed things up we map ichars to tokens
;;;	via an array lookup.
;;;
;;;;	Note that with unicode embedding length of tok representing a char may be more than one.
;;;;	ML should be okay with this but there may be problems.
;;;;	


(defvar *standard-character-token-array*
  (let ((a (make-array 128 :initial-element nil)))
    (dotimes (i 128)
	     (when (standard-character-code-p i)
		   (setf (aref a i)
			 (intern-system (let ((s (make-string 1)))
					 (setf (char s 0) (aref *standard-ascii-character-array* i))
					 s)))))
    a))


(defun code-to-character-tok (ch)
  (if (standard-character-p ch)
      (aref *standard-character-token-array* (character-to-code ch))
      (raise-error (error-message '(character tok character standard not)))))

(defun character-to-character-tok (ch)
  (if (standard-character-p ch)
      (aref *standard-character-token-array* (character-to-code ch))
      (raise-error (error-message '(character tok character standard not)))))

(defun implode-to-character-tok (i)
  (if (standard-character-code-p i)
      (aref *standard-character-token-array* i)
      (intern-system (int-to-character-string i))))

(defun char-member-p (ch string len)
  (dotimes (i len)
      (when (eql ch (char string i))
	(return-from char-member-p t)))
  nil)



;;;
;;; Non garbage producing way of exploding
;;;

(defvar *object-string* (make-array 100 :element-type 'character :fill-pointer 0
				    :adjustable t))

(defun print-object-to-string (obj &optional copyp)
  (cond
    ((stringp obj) obj)
    ((symbolp obj) (string obj))
    (t (setf (fill-pointer *object-string*) 0)
       (with-output-to-string (s *object-string*)
	 (print obj s))
       (if copyp
	   (copy-seq *object-string*)
	   *object-string*))))

(defun princ-object-to-string (obj &optional copyp)
  (cond
    ((stringp obj) obj)
    ((symbolp obj) (string obj))
    (t (setf (fill-pointer *object-string*) 0)
       (with-output-to-string (s *object-string*)
	 (princ obj s))
       (if copyp
	   (copy-seq *object-string*)
	   *object-string*))))





;;;; -docs- (mod bsc)
;;;;
;;;;	<pmsg>  : <text> | <message> | (<psmg> . <pmsg>) | #break-control[<type> <pmsg>]
;;;;
;;;;
;;;;	<text>  : nil | #\Space | '#\Newline | <string> | <symbol>
;;;;		| <break-control(<type>, <text>)>
;;;;		| (<text> . <text>)
;;;;	<type>	: 'linear | 'break | 'soft | 'multilinear
;;;;
;;;;	 This syntax will be extended by dependent modules.
;;;;
;;;; -doce-

;; rle todo doc pmsg break-control better.
(defstruct break-control
  (type 'soft)
  (pmsg nil))

(defun break-control (type pmsg) (make-break-control :type type :pmsg pmsg))
(defun soft-break () (break-control 'soft 'break))

(defun type-of-break-control (bc) (break-control-type bc))
(defun text-of-break-control (bc) (break-control-pmsg bc))


(defstruct force-xref
  (string nil)
  (id nil)
  (obid nil))

(defun force-xref (string id obid)
  (make-force-xref :string (concatenate 'string "\"" string "\"")
		   :id id :obid obid))

(defun string-of-force-xref (fx) (force-xref-string fx))
(defun id-of-force-xref (fx) (force-xref-id fx))
(defun obid-of-force-xref (fx) (force-xref-obid fx))


;;;;
;;;;	make-sexpr-walker (<*> sexpr) 				: (values next-f addr-f)
;;;;	 ** next-f ()		: <*>
;;;;	 ** addr-f ()		: <sexpr-address>
;;;;	 **
;;;;	 ** * <sexpr-addr> 		: (CAR . <sexpr-addr>)
;;;;	 ** *				: (CDR . <sexpr-addr>)
;;;;	 ** *				: NIL
;;;;	 ** * * <sexpr-addr> is current address into pmsg.


;;; RLE TODO doc this.
(defun reverse-flatten-sexpr (sexpr)
  (let ((result nil))
    (labels ((walk (sexpr)
	       (cond
		 ((null sexpr) nil)
		 ((consp sexpr)
		  (walk (car sexpr))
		  (walk (cdr sexpr)))
		 (t (push sexpr result)))))
      (walk sexpr))
    result))
			
	

;;
;; I'm not certain these addresses are correct.
;;
(defun make-sexpr-walker (sexpr &optional (addrp t))
  (let ((continuations (list sexpr))
	(address nil)
	(firstp t))
    (labels
	((pop-cont ()
	   (back-address)
	   (pop continuations)
	   (unless (null continuations)
	     (when addrp (push 'cdr address))))

	 (back-address ()
	   (when addrp
	     (cond
	       ((null address) nil)
	       ((eql 'car (car address))
		(pop address))
	       (t (pop address)
		  (back-address)))))

	 (break-cons ()
	   (cond
	     ((null continuations) nil)
	     ((null (car continuations))
	      (back-address)
	      (pop continuations)
	      (break-cons))
	     ((consp (car continuations))
	      (let ((hd (pop continuations)))
		(when addrp (push 'car address))
		(push (cdr hd) continuations)
		(push (car hd) continuations))
	      (break-cons))
	     (t nil)))
	   )

      ;; next-f
      (values
       #'(lambda ()
	   (if firstp
	       (setf firstp nil)
	       (pop-cont))
	   (break-cons)
	   (car continuations))

       #'(lambda ()
	   (reverse address))))))

    
;;;;
;;;;	<scan-addr>		: (INTEGER{char-count} . (<line-addr> . <text-addr>))
;;;;	<line-addr>		: (INTEGER{lines} . INTEGER{line-index})
;;;;	<text-addr>		: (INTEGER{string-index} . <sexpr-addr>)
;;;;
;;;;	 ***  char-count is total count of chars scanned.
;;;;	 ***  lines is number of breaks, line-index is number of <ich>'s since last break.
;;;;	 ***  <text-addr> is address within <text> of next char to be scanned.
;;;;
;;;;
;;;;	make-string-scanner (<string>) 				: (values eof-f next-f)
;;;;	 ** eof-f () 		: <bool>
;;;;	 ** next-f ()		: <ch>		
;;;;
;;;;	make-text-scanner(<text> &optional <bool{addrp}> nil)	: (values eof-f next-f addr-f)
;;;;	 ** eof-f () 		: <bool>
;;;;	 ** next-f ()		: <ch>		
;;;;	 ** addr-f ()		: <scan-addr>
;;;;	     * only newline chars count as line breaks for <scan-addr>.

;;;;
;;;;	RLE ??? : seems like a byte-array scanner would be useful. 
;;;;	RLE ??? : might be nice to allow byte array as type of text.
;;;;	RLE ??? : at the moment there is no immediate need so not 
;;;;	RLE ??? : implemented.
;;;;	RLE ??? :



(defun make-string-scanner (string)
  (let ((len (length string))
	(index 0))
    (values #'(lambda ()
		(unless (< index len)
		  (raise-error (error-message 'IO "String Scanner EOF")))
		(prog1 (character-to-code (char string index))
		  (incf index)))
	    #'(lambda ()
		(= len index))
	    #'(lambda ()
		index))))


(defun make-byte-array-scanner (byte-array)
  (let ((len (length byte-array))
	(index 0))
    (values #'(lambda ()
		(unless (< index len)
		  (raise-error (error-message 'IO "String Scanner EOF")))
		(prog1 (aref byte-array index)
		  (incf index)))
	    #'(lambda ()
		(= len index))
	    #'(lambda ()
		index))))

(defun make-byte-list-scanner (byte-list)
  (let ((r byte-list)
	(c 0))

    (values #'(lambda ()
		(when (null r)
		  (raise-error (error-message 'IO "String Scanner EOF")))
		(prog1 (car r)
		  (incf c)
		  (setf r (cdr r))))
	    #'(lambda ()
		(null r))
	    #'(lambda ()
		c))))


;;;
;;; break control is ignored.
;;; 'break and ibreaks treated as line-breaks for count purposes.
;;; 'break 'space counted as character
;;; <symbol> and <string> are treated as strings.
;;; everything else, ie extensions, ie terms treated as a character.
;;;

;;; RLE TODO : does not find break chars which are not standard-char-p.

;;; 
(defun make-text-scanner (text)
  (let ((sexpr-next nil)
	(sexpr-addr nil)
	(cur-item nil)
	(lbreak-count 0)
	(lchar-count 0)
	(char-count 0)
	(string-count 0)
	(string-eoff nil)
	(string-nextf nil))

    (labels
	((sexpr-next ()
	   (setf string-nextf nil)
	   (setf string-count 0)
	   (setf cur-item (funcall sexpr-next))

	   (cond
	     ((null cur-item) nil)

	     ((break-control-p cur-item)
	      (sexpr-next))

	     ((force-xref-p cur-item)
	      (force-ml-xref (id-of-force-xref cur-item)
			     (obid-of-force-xref cur-item))
	      (multiple-value-setq (string-nextf string-eoff)
		(make-string-scanner (string-of-force-xref cur-item)))
	      (when (funcall string-eoff)
		(sexpr-next))
	      )

	     ((or (eql cur-item #\space) (eql cur-item #\newline))
	      nil)

	     ((stringp cur-item)
	      (multiple-value-setq (string-nextf string-eoff)
		(make-string-scanner cur-item))
	      (when (funcall string-eoff)
		(sexpr-next)))

	     ((symbolp cur-item)
	      (multiple-value-setq (string-nextf string-eoff)
		(make-string-scanner (string cur-item)))
	      (when (funcall string-eoff)
		(sexpr-next)))
	     (t nil)))

	 (break-char ()
	   (incf lbreak-count)
	   (setf lchar-count 0)))
    
      (multiple-value-setq (sexpr-next sexpr-addr) (make-sexpr-walker text))
      (sexpr-next)

      (values
       #'(lambda ()
	   (prog1
	       (if string-nextf
		   (let ((code (funcall string-nextf)))
		     (incf string-count)
		     (when (funcall string-eoff)
		       (sexpr-next))
		     (if (= inewline code)
			 (break-char)
			 (incf lchar-count))
		     code)
		   (cond
		     ((null cur-item)
		      (raise-error (error-message '(text scan eof))))
		     ((eql #\newline cur-item)
		      (break-char)
		      (sexpr-next)
		      inewline)
		     ((eql #\space cur-item)
		      (incf lchar-count)
		      (sexpr-next)
		      ispace)
		     (t (incf lchar-count)
			;; apparently cur-item would be a term.
			;; also we must not be expecting break control.
			;; rle todo : should doc input text and output text separately.
			(prog1 cur-item
			  (sexpr-next)))))
	     (incf char-count)))

       ;; eoff
       #'(lambda ()
	   (null cur-item))

       ;; count
       #'(lambda ()
	   (list* char-count
		  (cons lbreak-count lchar-count)
		  (cons string-count "ignoring sexpr address" ;;(funcall sexpr-addr)
			)))))))



;;;;	IO : All IO will be BYTE8 streams.
;;;;	
;;;;	BYTE-S : the ascii integer values of the following characters:
;;;;	  - NEWLINE (10)
;;;;	    SPACE (32)
;;;;	    !\"#$%&'()*+,-./
;;;;	    0123456789
;;;;	    :;<=>?@
;;;;	    ABCDEFGHIJKLMNOPQRSTUVWXYZ
;;;;	    [\\]^_`
;;;;	    abcdefghijklmnopqrstuvwxyz
;;;;	    {|}~
;;;;
;;;;	SSTRING : a string restricted to characters corresponding to BYTE-S.
;;;;
;;;;	The characters correspond to the Lisp standard character set.
;;;;
;;;;	
;;;;	Any BYTE16 character can be encoded in the BYTE8 stream as a \ followed
;;;;	by the 4 byte hexadecimal represention of the BYTE16 value. Ie return
;;;;	would be "\000d".
;;;;	
;;;;	Effort will be expended to correctly interpret other 7bit ascii values
;;;;	as input.
;;;;	
;;;;	The internal representation of characters will be standard-chars or
;;;;	16bit-chars.  16bit-char representation is not worthwhile unless the
;;;;	lisp implementation uses 16bits codes for base-characters and for each
;;;;	16bit code, code = (char-code (code-char code)).
;;;;
;;;;	Characters : exist in three realms.
;;;;	  - literal : IO, ml-compiler, lisp-interpreter IO 
;;;;	  - terms : symbols + parameter-values : SSTRING.
;;;;	      * term syntax character escapes removed during ascii->term
;;;;		conversion and reinserted during term->ascii conversion.  It
;;;;		would be nice if the escapes did not need to be removed and
;;;;		inserted.  However, I do not see any sensible scheme to avoid
;;;;		it.
;;;;	      * ascii compression scheme uses injects flags and codes into
;;;;		ascii stream.  80 bit is used to indicate such use. Thus is is
;;;;		imperative that all characters are limited to 7 bits.
;;;;	  - edit : 16bits values.
;;;;	
;;;;	Streams and files should be completely analogous wrt io.
;;;;	
;;;;
;;;;	PROJECT: 
;;;;	One interesting approach to character set foolishnes (eg (code-char 13)
;;;;	= (c-m) is to not use strings or symbols in terms or in ML, but instead
;;;;	use byte vectors.  This would require some coding on our part to get the
;;;;	symbol and string functionality that we currently use. There is some
;;;;	question if we could do this efficiently. However, we would benefit by
;;;;	avoiding the frequent char-code code-char conversions. Also, then any
;;;;	7bit value would map to self and all other 8bit-16bit values would be
;;;;	escaped. Minor benefit and probably not pratical once we advertise 
;;;;	standard-character set.
;;;;
;;;;	This fix isn't so appealing if you can transfer memory mapped terms
;;;;	within a shared memory system as no (un)marshalling need occur.
;;;;	So, a conditionalized inplementation if practical would be best.
;;;;	
;;;;	
;;;;	int-to-character-string (INT &optional <closure{acc-f}) : SSTRING | NULL
;;;;	  * acc-f (STANDARD-CHAR) : NULL.
;;;;	  * returns SSTRING when acc-f null.
;;;;
;;;;
;;;;	string-to-byte-accumulator (SSTRING <sbits{escape}> <closure{acc-f}>)
;;;;	  : NULL
;;;;	  * acc-f (BYTE-S) : NULL
;;;;	
;;;;	string-to-byte-array (SSTRING <sbits{escape}>)		: <BYTE-S> array
;;;;	  * If 16bit-chars then embeds non standard chars.
;;;;	

;;;;	byte-array : stream and file.
;;;;	standard-character-string : internal.
;;;;	
;;;;	standard-character-string -> byte-array
;;;;	  - string-to-byte-array
;;;;	
;;;;	byte-array -> standard-character-string
;;;;	  - int-to-character-string
;;;;	
;;;;	character-toks : ML 
;;;;	  - character-to-character-tok (STANDARD-CHAR) : SYMBOL
;;;;	      * ml text scanner : fast char to tok through array lookup.
;;;;	  - implode-to-character-tok (INT) : 
;;;;	      *  used only to initialize ML whitespace toks.
;;;;	


;;;;	next-f :	nil -> byte
;;;;	eof-f :		nil -> bool
;;;;	
;;;;	In order to implement eof-f for binary streams we need to read ahead
;;;;	a byte as there is no way to detect eof without actually attempting 
;;;;	a read.
;;;;

(defun make-file-scanner (prl-stream &optional err-flush-f)
  (let ((lbyte-count 1)
	(line-count 1)
	(byte-count 1)
	(eof-p nil)
	(read-ahead nil))

    (values

     ;; next-f
     #'(lambda ()
	 (when eof-p
	   (raise-error (error-message '(IO scanner stream next eof))))

	 (let ((byte (if read-ahead
			 (prog1 read-ahead
			   (setf read-ahead nil))
			 (prl-stream-read prl-stream))))
	
	   (when err-flush-f (funcall err-flush-f))

	   ;;(format t "next      ~a ~a" byte (code-char byte))
	   (when (null byte)
	     (setf eof-p t)
	     (raise-error (error-message '(IO scanner stream next eof))))

	   ;; update address.
	   (incf byte-count)
	   (if (break-code-p byte)
	       (progn
		 (incf line-count)
		 (setf lbyte-count 1))
	       (incf lbyte-count))

	   byte))
	    
     ;; eof-f
     #'(lambda ()
	 (if eof-p
	     t
	     (if read-ahead
		 nil
		 (setf eof-p
		       (null
			(setf read-ahead
			      (prl-stream-read prl-stream))
			;;(let ((cra read-ahead) (ra *** )) (format t "ReadAhead ~a ~a ~a~%" ra (code-char ra) cra))
			)))))

     ;; RLE TODO ERR :
     ;; lucid read program is char stream and not
     ;; binary stream, so have to dummy that up somehow 
     ;; addr-f
     #'(lambda ()
	 (list line-count lbyte-count byte-count)))))


(defvar *scanner* nil)
(defvar *scan-cur-char*)
(defvar *scan-cur-byte*)
(defvar *scan-escape-p*)

(defmacro with-string-scanner ((s) &body body)
  `(let ((*scanner* (multiple-value-list (make-string-scanner ,s)))
	 (*scan-cur-byte* nil)
	 (*scan-escape-p* nil))
    (scan-prime)
    ,@body))

(defmacro with-byte-scanner ((bytes) &body body)
  `(let ((*scanner* (multiple-value-list (make-byte-array-scanner ,bytes)))
	 (*scan-cur-byte* nil)
	 (*scan-escape-p* nil))
    (scan-prime)
    ,@body))

(defmacro with-byte-list-scanner ((bytes) &body body)
  `(let ((*scanner* (multiple-value-list (make-byte-list-scanner ,bytes)))
	 (*scan-cur-byte* nil)
	 (*scan-escape-p* nil))
    (scan-prime)
    ,@body))

(defmacro with-text-scanner ((text) &body body)
  `(let ((*scanner* (multiple-value-list (make-text-scanner ,text)))
	 (*scan-cur-byte* nil)
	 (*scan-escape-p* nil))
    (scan-prime)
    ,@body))

(defmacro with-stream-scanner ((stream &optional err-flush-f at-prime-eof-ok-p) &body body)
  `(let ((*scanner* (multiple-value-list (make-file-scanner ,stream ,err-flush-f)))
	 (*scan-cur-byte* nil)
	 (*scan-escape-p* nil))
    (scan-prime ,at-prime-eof-ok-p)
    ,@body))



;;;;
;;;;  Scan primitives
;;;;


;; (funcall (third *scanner)) will return (char . line) count.
;; RLE TODO MILL callers - scan-string-scanned-text and  scan-string-unscanned-text
(defvar *ascii-echo-p* nil)

(defun scan-position ()
  (funcall (third *scanner*)))

(defun scan-cur-byte ()
  ;;(unless (null *scan-cur-byte*)
  ;;(format t "~a ~a ~%" *scan-cur-byte* (code-char *scan-cur-byte*)))
  ;;(when (and *ascii-echo-p* *process-break*) (format t "~a,~a." *scan-cur-byte* (code-char *scan-cur-byte*)))
  *scan-cur-byte*)

(defun scan-bump ()
  ;;(format t "~a ~a ~%" *scan-cur-byte* (when *scan-cur-byte* (code-char *scan-cur-byte*)))
  (when (and *ascii-echo-p* *process-break*) (format t "~a,~a." *scan-cur-byte* (if *scan-cur-byte* (code-char *scan-cur-byte*) " ")))
  (setf *scan-cur-byte*
	(unless (funcall (cadr *scanner*))
	  (funcall (car *scanner*)))))

(defmacro scan-bump-m ()
  `(progn
    ;;(format t "~a ~a ~%" *scan-cur-byte* (when *scan-cur-byte* (code-char *scan-cur-byte*)))
    (when (and *ascii-echo-p* *process-break*) (format t "~a,~a." *scan-cur-byte* (if *scan-cur-byte* (code-char *scan-cur-byte*) " ")))
    (setf *scan-cur-byte*
     (unless (funcall (cadr *scanner*))
       (funcall (car *scanner*))))))

(defun scan-cur-char ()
  (setf *scan-cur-char*
	(when (and *scan-cur-byte* (standard-character-code-p *scan-cur-byte*))
	      (byte-to-standard-character *scan-cur-byte*))))

(defun scan-eof-p ()
  (null *scan-cur-byte*))

(defun scan-prime (&optional eof-ok-p)
  (if (funcall (cadr *scanner*))
      (unless eof-ok-p
	(raise-error (error-message '(IO scanner prime eof))))
      (scan-next nil)))


(defun scan-escape-p ()
  *scan-escape-p*)


(defun scan-at-whitespace-p ()
  (and (not (scan-escape-p))
       (whitespace-code-p (scan-cur-byte))))

(defun scan-whitespace ()
  (when (scan-at-whitespace-p)
    (scan-next)
    (scan-whitespace)))


(defun scan-next (&optional (skip-whitespace t))
  (scan-bump)
  (setf *scan-escape-p* (eql iescape (scan-cur-byte)))
  (if *scan-escape-p*
      (scan-bump)
    (when skip-whitespace
	  (scan-whitespace))))


(defun scan-at-standard-character-p ()
  (standard-character-code-p (scan-cur-byte)))


(defun scan-error (tags &optional pmsg)
  ;;(break "se")
  (raise-error (error-message (cons 'scan tags)
			      (cons (princ-to-string (scan-position))
				    pmsg))))



;;; rle todo inline new scan char/string funcs.

(defun scan-at-byte-p (byte)
  (unless (scan-escape-p)
    (let ((code (scan-cur-byte)))
      (and code
	   (= byte code)))))

;; avoids read-ahead but messes up eof-p.
(defun scan-rest (code)
  (if (scan-at-byte-p code)
      (setf *scan-cur-char* nil
	    *scan-cur-byte* nil)
      (scan-error '(char)
		  (format-string "Scan rest. Expected ~a[~a] not ~a[~a]." 
				 (int-to-character-string code) code 
				 (int-to-character-string (scan-cur-byte)) (scan-cur-byte)))))
      
(defun scan-byte (byte)
  (if (scan-at-byte-p byte)
      (scan-next)
      (scan-error '(char)
		  (let ((b (scan-cur-byte)))
		    (format-string "Scan Expected ~a[~a] not ~a[~a]." 
				   (int-to-character-string byte) byte
				   (when b 
				     (int-to-character-string b))
				   b)))))


;;;;	When scanning from stream we must decode embedded unicode characters.
;;;;	\hhhh if we are using 16bit chars internally, otherwise the unicode 
;;;;	chars are left embedded.
;;;;	

;; todo delimiter string is now sbits.

(defvar *string-buffer-size* 1024)

(defvar *string-buffer* (make-array *string-buffer-size*
				    :element-type 'standard-char
				    :adjustable t
				    :fill-pointer 0))

;;; MTT
(defvar *unicode-digit-buffer* (make-array 4 :element-type '(unsigned-byte 8)))


(defun ascii-unicode-to-int (d1 d2 d3 d4)
  ;;(break "auti")
  (+ (hex-code-to-int d4)
     (ash (hex-code-to-int d3) 4)
     (ash (hex-code-to-int d2) 8)
     (ash (hex-code-to-int d1) 12)))


(defun scan-string  (escape-sbits &optional anti-p)

  (scan-whitespace)

  (setf (fill-pointer *string-buffer*) 0)
  
  (let ((unicode-index nil))

    (do ()
	((or (and (not (scan-escape-p))
		  (if anti-p
		      (not (test-standard-character-bit (scan-cur-byte) escape-sbits))
		      (test-standard-character-bit (scan-cur-byte) escape-sbits)))
	     (scan-eof-p))

	 ;; flush unicode digits.
	 (when unicode-index
	   (dotimes (i (1+ unicode-index))
	     (vector-push-extend (byte-to-standard-character 
				  (aref *unicode-digit-buffer* i))
				 *string-buffer*
				 )))

	 ;; Copy result to string.
	 (let* ((l (fill-pointer *string-buffer*))
		(s (make-string l)))
	
	   ;;(when (> l 128)
	   ;;(setf b *string-buffer*) (break))
	   
	   (dotimes (i l)
	     (setf (char s i) (aref *string-buffer* i)))

	   (when (> l *string-buffer-size*)
	     (setf (fill-pointer *string-buffer*) 0)
	     (adjust-array *string-buffer* *string-buffer-size*))
	   s))
      
	(let ((code (scan-cur-byte)))
	  (cond
	    (unicode-index
	     (if (hex-code-p code)
		 (progn
		   (setf (aref *unicode-digit-buffer* unicode-index) code)
		   (if (= 3 unicode-index)
		       (progn
			 (setf unicode-index nil)
			 ;; convert unicode digits to 16 bit value.
			 ;; non 16bit-char access prevented via unicode index logic flow.
			 ;;(break "d")
			 (vector-push-extend (code-char
					      (ascii-unicode-to-int (aref *unicode-digit-buffer* 0)
						   (aref *unicode-digit-buffer* 1)
						   (aref *unicode-digit-buffer* 2)
						   (aref *unicode-digit-buffer* 3)))
					     *string-buffer*))
		       (incf unicode-index)))

		 ;; 16bit-chars started collecting unicode digits but not enough digits.
		 (progn
		   ;;(setf d i)
		   (dotimes (i (1+ unicode-index))
		     (vector-push-extend (byte-to-standard-character
					  (aref *unicode-digit-buffer* i))
					 *string-buffer*))
		   (setf unicode-index nil))))
	  

	    ;; embedded unicode.
	    ((and (scan-escape-p) 
		  (hex-code-p code))
	     
	     #+16bit-chars
	     (progn
	       ;;(setf c code)
	       (setf (aref *unicode-digit-buffer* 0) code)
	       (setf unicode-index 1))

	     #-16bit-chars
	     (progn 
	       (vector-push-extend #\\ *string-buffer* *string-buffer-size*)
	       (vector-push-extend (byte-to-standard-character code) *string-buffer* *string-buffer-size*)))

	    (t
	     ;;(setf b code)
	     ;; rle todo : if int-to-char-string called instead then would be more robust with ascii
	     ;; but not standard-chars? which for stream io should not be necessary. ML scanner
	     ;; is more robust in this regard already.
	     (vector-push-extend (byte-to-standard-character code) *string-buffer* *string-buffer-size*))))
	   
      (scan-next nil))))


(defun scan-unicode-istring ()
  (let ((unicode-index nil)
	(acc nil))

    (do ()
	((scan-eof-p)
	 (when unicode-index
	   (push iescape acc)
	   (dotimes (i (1+ unicode-index))
	     (push (aref *unicode-digit-buffer* i) acc)))
	 (nreverse acc))
	   
      (let ((code (scan-cur-byte)))
	(cond
	  (unicode-index
	   (if (hex-code-p code)
	       (progn
		 (setf (aref *unicode-digit-buffer* unicode-index) code)
		 (if (= 3 unicode-index)
		     (progn
		       (setf unicode-index nil)
		       (push (ascii-unicode-to-int (aref *unicode-digit-buffer* 0)
						   (aref *unicode-digit-buffer* 1)
						   (aref *unicode-digit-buffer* 2)
						   (aref *unicode-digit-buffer* 3))
			     acc))
		     (incf unicode-index)))
	       (progn
		 ;;(setf d i)
		 (push iescape acc)
		 (dotimes (i (1+ unicode-index))
		   (push (aref *unicode-digit-buffer* i) acc))
		 (setf unicode-index nil))))
	
	  ((and (scan-escape-p)
		(hex-code-p code))
	   (setf (aref *unicode-digit-buffer* 0) code)
	   (setf unicode-index 1)
	   )

	  (t (push code acc)))

	(scan-next nil)))
    ))



(defun scan-decimal-num ()
  (with-backtrace "ScanInteger"
    (if (numeric-digit-code-p (scan-cur-byte))
	(do ((code (scan-cur-byte) (progn (scan-next nil) (scan-cur-byte)))
	     (num 0))
	    ((not (numeric-digit-code-p code))
	     (scan-whitespace) num)
	  (setf num (+ (* num 10) (numeric-digit-code-to-int code)))
	  )
	(scan-error '(numeral)))))



(defun scan-delimited-list (item-scanner ldelim rdelim  &optional divider-scanner)

  (when ldelim
    (scan-byte ldelim)
    (scan-whitespace))

  (when (scan-at-byte-p rdelim)
    (scan-byte rdelim)
    (return-from scan-delimited-list nil))

  (do ((items (list (funcall item-scanner))))
      ((progn (scan-whitespace) (scan-at-byte-p rdelim))
       (scan-byte rdelim)
       (nreverse items))

    (when divider-scanner
      (funcall divider-scanner))

    (scan-whitespace)
    (push (funcall item-scanner) items)

    (when (scan-eof-p)
      (scan-error '(delimited-list) 
		  (format-string "Missing end delimiter: ~a." rdelim)))))


(defun scan-undelimited-list (item-scanner stopf &optional divider-scanner)

  (when (funcall stopf)
    (return-from scan-undelimited-list nil))

  (do ((items (list (funcall item-scanner))))
      ((progn (scan-whitespace) (funcall stopf))
       (nreverse items))

    (when divider-scanner
      (funcall divider-scanner))

    (scan-whitespace)
    (when (scan-eof-p)
      (scan-error '(eof) "Missing end of undelimited list."))

    (push (funcall item-scanner) items)))



;;;;	
;;;;	BYTE8-arrays : 
;;;;	
;;;;	Parse input from :
;;;;	
;;;;	Accumulate output :
;;;;	
;;;;	Buffer
;;;;	
;;;;	
;;;;	bb-add-byte
;;;;	bb-add-byte-array
;;;;	
;;;;	bb-add-standard-char
;;;;	bb-add-standard-string
;;;;	
;;;;	
;;;;	bb-scan-string		: <bit> array{escapes} -> <byte-buffer> -> SSTRING
;;;;	bb-scan-byte-array	: <bit> array{escapes} -> <byte-buffer> -> BYTE8 array{buffer}
;;;;	
;;;;	
;;;;	

;;;;	expose buffer or call buffer methods.
;;;;	
;;;;	  - expose : fewer calls.
;;;;	
;;;;	methods must have handle to buffer.
;;;;	each method invocation must use handle to access buffer.
;;;;	then method manips buffer.
;;;;	define buffer methods as macros.

;;;;	
;;;;	buffer handle is dynamic or static.
;;;;	must assume static allows compiler to produce faster code.
;;;;	however dynamic is variable reference rather than pointer lookup.
;;;;	
;;;;	again use macro combinations ie (with-bb (bb-handle) (bb-manip)) assuming
;;;;	multiple bb-manips within lexical scope of with-bb. Then bb-handle and bb-maip
;;;;	know private lexical variable holding bb. Ie bb-manip used outside of lexical
;;;;	scope of a with-bb should result in comiler error for unreferenced variable?
;;;;	maybe with-bb defines bb-manip via flet.

;; fill is next available slot.
;; when fill reaches size then flush.



(defun accumulate-int-bytes-f (acc-f n)
  ;;(format t "~a~%" n)
  (funcall acc-f (ldb (byte 8 0) n))
  (funcall acc-f (ldb (byte 8 8) n))
  (funcall acc-f (ldb (byte 8 16) n))
  (funcall acc-f (ldb (byte 8 24) n))
  )

(defun accumulate-int-bytes-buf (buf offset n)
  ;;(setf -a buf -b offset -i n) (break "aibb")
  (setf (aref buf offset) (ldb (byte 8 0) n))
  (setf (aref buf (1+ offset)) (ldb (byte 8 8) n))
  (setf (aref buf (+ 2 offset)) (ldb (byte 8 16) n))
  (setf (aref buf (+ 3 offset)) (ldb (byte 8 24) n))
  )



;;(defvar *int-byte-buf* (make-array 4 :element-type '(unsigned-byte 8)))

(defun scan-int-bytes ()
  (+ (scan-cur-byte)
     (ash (progn (scan-bump) (scan-cur-byte)) 8)
     (ash (progn (scan-bump) (scan-cur-byte)) 16)
     (ash (progn (scan-bump) (scan-cur-byte)) 32)))

(defmacro decimal-bytes-to-int (a len)
  (let ((aa (gensym)))
    `(let ((,aa ,a)
	   (n 0))
      (dotimes (i ,len)
	(setf n (+ (* n 10)
		   (numeric-digit-code-to-int (aref ,aa i)))))
      n)))

;; assumed reversed reversed?
(defmacro hexadecimal-bytes-to-int (a offset l)
  (let ((aa (gensym))
	(oo (gensym))
	(ll (gensym)))
    `(let ((,aa ,a)
	   (,oo ,offset))
      (let ((,ll ,l)
	    (n 0)
	    (m 1))
	(dotimes (i ,ll)
	  ;;(setf -i i -n n -a ,aa -o ,oo)
	  (setf n (+ n (* m (aref ,aa (+ ,oo i)))))
	  (setf m (ash m 8))
	  )
	n))))


(defmacro bytes-to-string (a offset &optional len)
  (let ((aa (gensym))
	(ll (gensym))
	(ss (gensym))
	(oo (gensym)))

    `(let* ((,aa ,a)
	    (,oo ,offset)
	    (,ll (or ,len (- (length ,aa) ,oo)))
	    (,ss (make-string ,ll)))

      (dotimes (i ,ll)
	(setf (char ,ss i) (code-char (aref ,aa (+ ,oo i)))))

      ,ss)))


(defmacro accumulate-byte-m (byte)
  ;;(when (eql byte 142) (break "ab"))
  `(vector-push-extend ,byte *byte-buffer* *byte-buffer-size*))

(defmacro accumulate-char-m (char)
  ;;(when (eql byte 142) (break "ab"))
  `(vector-push-extend (char-code ,char) *byte-buffer* *byte-buffer-size*))

#|
(defun test-bb-byte (count bb b)
  (declare (type (unsigned-byte 8) b))
  
  (with-fbb bb (dotimes (i count)  (fbb-add-byte b))))

(defun test-bb-array (count bb aa)
  (declare
   (type (vector (unsigned-byte 8)) aa))
  
  (with-fbb bb (dotimes (i count)  (fbb-add-array aa))))

(defun test-ap (f a)
  (funcall f a))

(defun test-lambda (i)
  (let ((f #'(lambda (i) (1+ i))))
    (dotimes (j i)
      (test-ap f j))))

(defun test-lambda2 (i)
  (dotimes (j i)
    (test-ap #'(lambda (i) (1+ i)) j)))

(defun test-array-replace (bb aa)
  (declare (type (vector (unsigned-byte 8)) aa))

  (with-fbb bb
    (dotimes (i 9769)
      (replace *fast-byte-buffer-array* aa
	       :start1 0
	       :end1 1024
	       :start2 0))))

|#
	  

;;;
;;;	UNSCAN: structure -> string
;;;

;;;;	byte-buffers :
;;;;	
;;;;	Might want to check length of pool occasionally and contract.
;;;;	
;;;;	could be conditionalized by fbbstring feature but fttb no.
;;;;


(defun new-byte-buffer ()
  (let ((a (make-array *byte-buffer-size* 
		       :element-type '(unsigned-byte 8)
		       :adjustable t
		       :fill-pointer 0)))

    (setf (fill-pointer a) 0)
    a))
    

(defun reset-byte-buffer (b)

    (setf (fill-pointer b) 0)
	    
    (when (> (array-dimension b 0) *byte-buffer-size*)
	  (adjust-array b *byte-buffer-size*))

    b)

(defvar *byte-buffer-pool* (list (new-byte-buffer)))

(defun get-byte-buffer ()
  (or (pop *byte-buffer-pool*)
      (new-byte-buffer)))

(defun release-byte-buffer (b)
  (push (reset-byte-buffer b) *byte-buffer-pool*))



(defvar *byte-buffer*)

(defmacro byte-buffer-space (amt)
  (let ((fp (gensym))
	(nfp (gensym))
	(l  (gensym)))
    `(let* ((,fp (fill-pointer *byte-buffer*))
	   (,nfp (+ ,fp ,amt))
	   (,l (array-dimension *byte-buffer* 0)))
      (unless (> ,l ,nfp)
	;;(setf -nfp ,nfp -l ,l) (break "bbs")
	(adjust-array *byte-buffer* (+ ,l *byte-buffer-size*)))
      (setf (fill-pointer *byte-buffer*) ,nfp) )))

(defun accumulate-byte (byte)
  ;;(when (eql byte 142) (break "ab"))
  (vector-push-extend byte *byte-buffer* *byte-buffer-size*))

(defun accumulate-char (char)
  ;;(when (eql byte 142) (break "ab"))
  (vector-push-extend (char-code char) *byte-buffer* *byte-buffer-size*))

(defun accumulated-length () (fill-pointer *byte-buffer*))

(defun flush-byte-accumulator-to-standard-string ()
  (let* ((l (fill-pointer *byte-buffer*))
	 (s (make-string l)))
    (dotimes (i l)
      ;;(setf -s s -l l -i i)
      (setf (char s i) (byte-to-standard-character (aref *byte-buffer* i))))
    s))


(defun flush-byte-accumulator-to-string ()
  (let* ((l (fill-pointer *byte-buffer*))
	 (s (make-string l)))
    (dotimes (i l)
      ;;(setf -s s -l l -i i)
      (setf (char s i) (code-char (aref *byte-buffer* i))))
    s))

(defun flush-byte-accumulator-to-stream (stream) 
  (let* ((l (fill-pointer *byte-buffer*)))
    (dotimes (i l)
      (prl-stream-write (aref *byte-buffer* i) stream))
    nil))

(defun flush-byte-accumulator () 
  (let* ((l (fill-pointer *byte-buffer*))
	 (a (make-array l :element-type '(unsigned-byte 8))))
    (dotimes (i l)
	     (setf (aref a i) (aref *byte-buffer* i)))
    a))

;; nil : byte-array 
;; t : standard-character-string
;; <stream>
(defmacro with-byte-accumulator ((result-type) &body body)
  `(let ((*byte-buffer* (get-byte-buffer)))
     ,@body
     
     (prog1
	 (case ,result-type
	   (byte-array (flush-byte-accumulator))
	   (string (flush-byte-accumulator-to-string))
	   (standard-string (flush-byte-accumulator-to-standard-string))
	   ((nil) nil)
	   (otherwise (flush-byte-accumulator-to-stream ,result-type)))

       (release-byte-buffer *byte-buffer*))))


;; assumes string is standard-character-string.
(defun string-to-byte-accumulator (s escape-sbits acc-f &optional escapechar)

  (let ((iescape-ch (or (when escapechar (char-code escapechar))
			iescape)))
    (labels 
	((accumulate (code)
	   (funcall acc-f code))

	 (accumulate-unicode-byte (a)
	   (accumulate (aref a 0))
	   (accumulate (aref a 1)))
	 )
		
      (let* ((l (length s)))
	(dotimes (i l)
	  (let ((ch (char s i)))
	    #-16bit-chars
	    (if (standard-character-p ch)
		(let ((code (character-to-code ch)))
		  (when (test-standard-character-bit code escape-sbits)
		    (accumulate iescape-ch))
		  (accumulate code))
		(let ((code (char-code ch)))
		  (accumulate iescape)
		  (accumulate-unicode-byte (unicode-first-byte-array code))
		  (accumulate-unicode-byte (unicode-second-byte-array code))))

	    #+16bit-chars
	    (let ((code (character-to-code ch))
		  (one-less (1- l)))
	      (when (test-standard-character-bit code escape-sbits)
		;; mustn't escape \ when \nnnn'
		(unless (and (eql code iescape-ch)
			     (< i one-less)
			     (hex-code-p (character-to-code (char s (1+ i)))))
		  (accumulate iescape-ch)))
	      (accumulate code))
	    ))

	(values)))))

(defun string-to-byte-array (s escape-sbits)
  (with-byte-accumulator ('byte-array)
   (string-to-byte-accumulator s escape-sbits #'accumulate-byte)))


;; may be worthwhile, since localizes manip of buffer
;; needs to do escaping and unicode encoding.
;; diff from previous function would be avoidance of mulitple calls to accumulate-byte.

;;(defmacro accumulate-string (s escape-sbits) )

;; assume needs no escaping and all chars standard.
(defmacro accumulate-standard-string (s)
  (let ((ss (gensym)))
    `(let* ((,ss ,s)
	    (l (length ,ss)))
      (declare
       (type simple-string ,ss)
       (type (vector (unsigned-byte 8)) *byte-buffer*))
    
      (let* ((fp (fill-pointer *byte-buffer*)))
	(byte-buffer-space l)
	(dotimes (i l)
	  ;;(setf -fp fp -nfp nfp -a i -b l -c (char ,ss i) -d (char-code (char ,ss i))) (break "asts")
	  (setf (aref *byte-buffer* (+ i fp)) (char-code (char ,ss i))))
	))))


(defmacro accumulate-array (a)
  (let ((aa (gensym))
	)
    `(let* ((,aa ,a)
	    (l (length ,aa)))
      (declare 
       (type (vector (unsigned-byte 8)) *byte-buffer*)
       (type (vector (unsigned-byte 8)) ,aa))
      
      (let* ((fp (fill-pointer *byte-buffer*))
	     (nfp (+ fp l)))
	(byte-buffer-space l)

	;;(setf (fill-pointer *byte-buffer*) nfp)
	(replace *byte-buffer* ,aa :start1 fp :end1 nfp)))))


(defmacro accumulate-int-hex-bytes (offset n)
  ;;(setf -a buf -b offset -i n) (break "aibb")
  (let ((nn (gensym))
	(oo (gensym)))
    `(let ((,nn ,n)
	   (,oo ,offset))
      (do ((i 0 (1+ i)))
	  ((zerop ,nn) i)
	(setf (aref *byte-buffer* (+ i ,oo)) (ldb (byte 8 0) ,nn))
	(setf ,nn (ash ,nn -8))))))



;; accumulates into *byte-buffer until encountering an unescaped member of escape-sbits.
;; copy escapes.
;; when buffering input should do some kind of loop inside scan to avoid calls to scan-bump/scan-cur-code
;; and then maybe a (replace ...).
(defun scan-bytes (escape-sbits)
  (let ((escape-p nil))
    (when *scan-escape-p*
      (setf *scan-escape-p* nil) 
      (accumulate-byte-m iescape)
      (accumulate-byte-m *scan-cur-byte*)
      (scan-bump-m))
    (do ((code *scan-cur-byte* (scan-bump-m)))
	((or (scan-eof-p)
	     ;;(progn (setf -a escape-p -b code) nil) 
	     (and (not escape-p)
		  (not (= iescape code))
		  (test-standard-character-bit code escape-sbits))))

      ;;(setf -a code) (break "sb")

      (if escape-p
	  (setf escape-p nil)
	  (when (= code iescape)
	    (setf escape-p t)))

      (accumulate-byte-m code))))

(defmacro scan-bytes-n (l)
  `(dotimes (i ,l)
    (accumulate-byte-m (scan-bump-m)) ))



;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	 escape-string(STRING <bit-vector{escapes}>) : STRING

;;;;	only allows for escaping standared chars.

(defun escape-string (s escape-sbits &optional acc-f)

  #+16bit-chars
  (labels
   ((run (acc-f)
	 (dotimes (i (length s))
		  (let ((ch (char s i)))
		    (when (test-standard-character-bit (char-code ch) escape-sbits)
			  (funcall acc-f #\\))
		    (funcall acc-f ch)))
	 nil))

   (if acc-f
       (run acc-f)
       (with-output-to-string (s)
	 (run #'(lambda (ch)
		  (write-char ch s)))
	 s)))
    
  #-16bit-chars
  (if acc-f
      (string-to-byte-accumulator s escape-sbits 
				  #'(lambda (code)
				      (funcall acc-f (byte-to-standard-character code))))

      (with-byte-accumulator ('standard-string)
	(string-to-byte-accumulator s escape-sbits 
				    #'accumulate-byte)))
  )
		 


;; itemf(<item>)	: *
;; punctuate(<delim>)	: NULL
#|
(defun walk-list-delimited (l itemf punctuate
				  &optional
				  (m-delim)
				  (l-delim)
				  (r-delim))
  (labels
      ((do-rest (l)
	 (if (null l)
	     (when r-delim (funcall punctuate r-delim))
	     (progn (when m-delim (funcall punctuate m-delim))
		    (funcall itemf (car l))
		    (do-rest (cdr l))))))

    (if l
	(progn (when l-delim (funcall punctuate l-delim))
	       (funcall itemf (car l))
	       (do-rest (cdr l)))
	(progn (when l-delim (funcall punctuate l-delim))
	       (when r-delim (funcall punctuate r-delim))))))
|#


(defun walk-list-delimited (l itemf punctuate
				  &optional
				  (m-delim)
				  (l-delim)
				  (r-delim))
  (if l
      (progn (when l-delim (funcall punctuate l-delim))
	     (funcall itemf (car l))
	     (do ((ll (cdr l) (cdr ll)))
		 ((null ll)
		  (when r-delim (funcall punctuate r-delim)))
	       (when m-delim (funcall punctuate m-delim))
	       (funcall itemf (car ll))))
      (progn (when l-delim (funcall punctuate l-delim))
	     (when r-delim (funcall punctuate r-delim)))))


;; itemf (<item>) 	: (values <string> sexpr INTEGER)
;;
(defun list-to-delimited-string-sexpr (l itemf
					 &optional
					 (m-string " ")
					 (l-string "(")
					 (r-string ")"))
  (let ((acc nil)
	(len 0))
    (walk-list-delimited l
			 #'(lambda (item)
			     (mlet* (((ss count) (funcall itemf item)))
				    (if count
					(when len
					  (incf len count))
					(setf len nil))
				    (push ss acc)))
			 #'(lambda (s)
			     (when len
			       (incf len (length s)))
			     (push s acc))
			 m-string
			 l-string
			 r-string)
    (values (nreverse acc) len)))


;;;
;;;	while it may seem pointless to produce a string sexpr to write to a stream
;;;	when one could simply write to the stream instead of producing the sexpr,
;;;	it does give the advantage of having the length available before writing.

;; RLE TODO : thereis a better way to do the followin, I'm sure. At the moment this is
;; only used be level-expr to string, something more integrated might be better.

(defun string-sexpr-to-string (sexpr l)
  (let ((i 0)
	(s (make-array l :fill-pointer 0 :element-type 'character)))

      (with-output-to-string (stream s)

	(labels
	 ((checki (ll)
		  (when (> ll l)
			(system-error '(string-sexpr-to-string length low))))
	  (visit (sexpr)
		 (unless (null sexpr)
			 (if (stringp sexpr)
			     (let ((ll (length sexpr)))
			       (checki (+ i ll))
			       (write-string sexpr stream)
			       (incf i ll))
			   (if (not (consp sexpr))
			       (system-error (list 'string-sexpr-to-string 'atom-not-string sexpr))
			     (progn (visit (car sexpr)) (visit (cdr sexpr))))))))

	 (visit sexpr)
	 (unless (= i l)
		 (system-error '(string-sexpr-to-string length high)))
	 s))))


;;;
;;; 	pmsg-to-string :
;;;


(defun print-pmsg-item (item depth)
  ;;(setf a item) (break "ppi")
  (cond
    ((basic-message-p item)
     (format t "~%~a" (blank-string depth))
     (print-message item depth))
    ((eql #\newline item)
     (format t "~%~a" (blank-string depth)))
    ((eql #\space item) (format t " "))
    (t (format t " ~a" item))))


(defun print-pmsg (pmsg depth &optional (print-item #'print-pmsg-item))
  (cond
    ((null pmsg))
    ((consp pmsg)
     (print-pmsg (car pmsg) depth print-item)
     (print-pmsg (cdr pmsg) depth print-item))
    (t (funcall print-item pmsg depth))))
     
(defun print-basic-message (message &optional (depth 0))
  (print-stamp (stamp-of-message message))
  (format t "[~a]:" (tags-of-message message))
  (print-pmsg (pmsg-of-message message) (1+ depth))
  (terpri)
  )



(defvar *print-message-hook* #'print-basic-message)

(defvar *print-message-suppress* nil)

(defun push-message-suppression (f)
  (setf *print-message-suppress*
	(message-filter-or f *print-message-suppress*)))

(defun push-message-prefix-suppression (tags)
  (push-message-suppression (message-filter-by-prefix tags)))
  
;; looks like should be (not (funcall *print-message-suppress* m))
(defun print-message (m &optional (d 0))
  ;;(setf -m m ) ;;(break "pm")
  (format t "~%print-message ~a " (tags-of-message m))
  (when (or (null *print-message-suppress*)
	    (not (funcall *print-message-suppress* m)))
      (funcall *print-message-hook* m d)
      (terpri)))

(push (cons 'asynch #'print-message) *message-stores*)

(defun message-to-string (message)
  (with-output-to-string (s) (print-message message)))


;; RLE TODO DOC insys :
(defmacro insys (&body body)
  `(prog1 (with-handle-error (('(eval)) ())
	    ,@body)
    (when (messages-p)
       (mapcar #'print-message (messages-flush)))))



;; rle todo : really should be part of some other module?
;; rle todo : or maybe as an extension to bsc module.


;; for non 16bit chars maps tab, return, and linefeed to embedded unicode.
(defun string-to-standard-character-string (s)
  #-16bit-chars (with-output-to-string (news)
		  (let ((f #'(lambda (ch) (write-char ch news))))
		    (dotimes (i (length s))
		      (let ((ch (char s i)))
			(cond
			  ((char= #\tab ch)
			   (int-to-character-string itab f))
			  ((char= #\return ch)
			   (int-to-character-string ireturn f))
			  ((char= #\linefeed ch)
			   (int-to-character-string ilinefeed f))
			  (t (int-to-character-string (char-code ch) f)))))))
  #+16bit-chars s
  )


(defun istring-to-standard-character-string (s)
  (with-output-to-string (news)
    (let ((f #'(lambda (ch) (write-char ch news))))
      (dolist (i s)
	(int-to-character-string i f)))))

;; incr-or-cut if integer makes (mod (length s) incr-or-cut) = 0.
(defun pad-string (s padl &optional incr-or-cut)
  (let ((blanks "                                                                                "))

    (let ((l (length s)))

      (labels
	  ((pad (lim)
	     (cond
	       ((= l lim) s)

	       ((> l lim) 
		(if (integerp incr-or-cut)
		    (pad (+ lim incr-or-cut))
		    (subseq s 0 lim)))
      
	       ((< l lim)
		(let ((pad (min (- lim l) 80)))
		  (concatenate 'string s (subseq blanks 0 pad)))))))

	(pad padl)))))



(defun read-text-file (fname)
  (handle-file-error
   (with-open-file (s fname)
     (let ((acc nil))
       (do ((b (read-line s nil nil) (read-line s nil nil)))
	   ((null b))
	 (push b acc))
       (nreverse acc)))))


(defun read-sexpr-file (fname)
  (handle-file-error
   (with-open-file (s fname)
     (let ((acc nil))
       (do ((b (read s nil nil) (read s nil nil)))
	   ((null b))
	 (push b acc))
       (nreverse acc)))))


