
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

(setq *read-base* 10.)  ;-- make the reader/printer work in decimal
(setq *print-base* 10.)

;;;; RLE TODO DOC somewhere that in [<foo> <goo>] list  [] list is a doc operator.

;;;; -docs- (mod bsc)
;;;;
;;;;	<string> 		: STRING
;;;;	<id>			: SYMBOL
;;;;	<closure>		: CLOSURE
;;;;	<bool>			: T | NIL
;;;;
;;;;	<*>			: absolutely anything.
;;;;	NULL			: (values)	** note NULL is not nil. (is eq though)
;;;;
;;;; -doce-

;;; RLE TODO doc list functions forall-p etc
(defmacro format-string (&rest rest)
  `(format nil ,@rest))


;;; multiple-value let.
(defmacro mlet* (((vars prog &optional declare) &rest rest) &body body)
  (if (null rest)
      (if declare
	  `(multiple-value-bind ,vars ,prog
	    ,declare
	    ,@body)
	  `(multiple-value-bind ,vars ,prog
	    ,@body))
      (if declare
	  `(multiple-value-bind ,vars ,prog
	    ,declare
	    (mlet* (,@rest) ,@body))
	  `(multiple-value-bind ,vars ,prog
	    (mlet* (,@rest) ,@body)))))


;;;;
;;;;
;;;;

;; like dolist except has counter (starts at 0).
(defmacro dotimeslist ((i c l) &body body)
  (let ((myl (gensym)))
    `(do* ((,i 0 (1+ ,i))
	   (,myl ,l (cdr ,myl))
	   (,c (car ,myl) (car ,myl))
	   (collector nil))
      ((null ,myl) (nreverse collector))
      (push (progn ,@body) collector))))


;;;
;;;  These are to replace some/every which do a suprising amount of consing.
;;; RLE TODO look for more to replace.

;;; RLE PERF if we can assume all lists are same length we can improve test.
;;;  maybe have debug flag which causes expansion to check lengths.
;;; in cases where not safe assumption then call more general version of apply-predicate to list pair.

(defmacro forall-p (p &rest lists)
  (let* ((vars (mapcar #'(lambda (ignore)
			   (declare (ignore ignore))
			   (gensym))
		       lists))
	 (do-vars (mapcar #'(lambda (v l) `(,v ,l (cdr ,v))) vars lists))
	 (test (mapcar #'(lambda (v) `(null ,v)) vars))
	 (cars (mapcar #'(lambda (v) `(car ,v)) vars))
	 (f (gensym)))
    `(let ((,f ,p))
      (do (,@do-vars)
	  ((or ,@test) t)
	(unless (funcall ,f ,@cars) (return nil))))))

;; returns t or nil.
(defmacro exists-p (p &rest lists)
  (when (null lists)
    (error "exists-p called with 1 argument, but at least 2 arguments are required"))
  (let* ((vars (mapcar #'(lambda (ignore)
			   (declare (ignore ignore))
			   (gensym))
		       lists))
	 (do-vars (mapcar #'(lambda (v l) `(,v ,l (cdr ,v))) vars lists))
	 (test (mapcar #'(lambda (v) `(null ,v)) vars))
	 (cars (mapcar #'(lambda (v) `(car ,v)) vars))
	 (f (gensym)))
    `(let ((,f ,p))
      (do (,@do-vars)
	  ((or ,@test) nil)
       (when (funcall ,f ,@cars) (return t))))))


(defmacro exists-p-optimized ((v l) &body body)
  (let ((ll (gensym))
	(lll (gensym))
	)
    `(let ((,lll ,l))
      (unless (null ,lll)
	(do ((,ll ,lll (cdr ,ll)))
	    ((or (null ,ll) (let ((,v (car ,ll))) ,@body)) (not (null ,ll))))))))

(defmacro forall-p-optimized ((v l) &body body)
  (let ((ll (gensym))
	(lll (gensym))
	)
    
    `(let ((,lll ,l))
      (or (null ,lll)
	(do ((,ll ,lll (cdr ,ll)))
	    ((or (null ,ll) (null (let ((,v (car ,ll))) ,@body)))
	     (null ,ll)))))))

;; RLE TODO MILL some -> find-first exists-p
(defmacro find-first (p &rest lists)
  (let* ((vars (mapcar #'(lambda (ignore)
			   (declare (ignore ignore))
			   (gensym))
		       lists))
	 (do-vars (mapcar #'(lambda (v l) `(,v ,l (cdr ,v))) vars lists))
	 (test (mapcar #'(lambda (v) `(null ,v)) vars))
	 (cars (mapcar #'(lambda (v) `(car ,v)) vars))
	 (f (gensym))
	 (result (gensym)))
    `(let ((,f ,p))
      (do (,@do-vars
	   (,result nil (funcall ,f ,@cars)))
	  ((or ,result ,@test) ,result)))))

(defmacro find-first-optimized ((v l) &body body)
  (let ((ll (gensym))
	(lll (gensym))
	(result (gensym)))
    
    `(let ((,lll ,l))
      (unless (null ,lll)
	(if (consp ,lll)
	    (do ((,ll ,lll (cdr ,ll))
		 (,result nil (let ((,v (car ,ll))) ,@body)))
		((or ,result (null ,ll)) ,result))
	    (let ((,v ,lll)) ,@body)
	    )))))


;; find tail whose car satisfies predicate p.
(defmacro find-cdr (p l)
  (let ((f (gensym))
	(ll (gensym)))
    `(let ((,f ,p))
      (do ((,ll ,l (cdr ,ll)))
	  ((or (null ,ll) (funcall ,f (car ,ll))) ,ll)))))


;; destructively append a list of lists. Optionally, insert a list between
;; each pair of members. The inserted list is not modified.
(defun nlists-to-list (lists &optional insertion)
  (nconc (car lists)
	 (mapcan #'(lambda (l)
		     (nconc (copy-list insertion) l))
		 (cdr lists))))


(defun replace-in-list (new index list)
  (do ((i 1 (1+ i))
       (l list (cdr l))
       (acc nil))
      ((null l) (nreverse acc))
    (if (= i index)
	(push new acc)
	(push (car l) acc)))
  ;; RLE PERF ??? 
  ;;(replace (copy-list children)
  ;;(list new) :start1 (1- index) :end1 index)
  )

;; applys predicate pairwise to list.
;; nil result if lists not same length.
;; RLE PERF could be made better by making similar to forall-p.
;;(defun apply-predicate-to-list-pair (list-a list-b predicate)
;;  (cond
;;    ((and (null list-a) (null list-b))	t)
;;    ((or (null list-a) (null list-b))	nil)
;;    ((not (funcall predicate (car list-a) (car list-b))) nil)
;;    (t (apply-predicate-to-list-pair (cdr list-a) (cdr list-b) predicate))))

;; would be nice to do eq check here first but without restricting predicate
;; it t on eq might be wrong answer.
(defun apply-predicate-to-list-pair (list-a list-b predicate)
  (do ((list-a list-a (cdr list-a))
       (list-b list-b (cdr list-b)))
      ((or (null list-a) (null list-b)
	   (not (funcall predicate (car list-a) (car list-b))))
       (and (null list-a) (null list-b)))))

(defmacro apply-predicate-to-list-pair-optimized (list-a list-b predicate)
  (let ((a (gensym))
	(b (gensym)))
    `(do ((,a ,list-a (cdr ,a))
	  (,b ,list-b (cdr ,b)))
      ((or (null ,a) (null ,b)
	(not (,predicate (car ,a) (car ,b))))
       (and (null ,a) (null ,b))))))


;; possibly dotted.
(defun apply-predicate-to-dotted-lists (list-a list-b predicate)
  (cond
    ((null list-a)
     (null list-b))
    ((null list-b)
     nil)
    ((not (consp list-a))
     (cond
       ((not (consp list-b))
	(funcall predicate  list-a  list-b))
       ((null (cdr list-b))
	(funcall predicate  list-a (car list-b)))
       (t nil)))
    ((not (consp list-b))
     (cond
       ((null (cdr list-a))
	(funcall predicate (car list-a) list-b))
       (t nil)))
    (t (and (funcall predicate (car list-a) (car list-b))
	    (apply-predicate-to-list-pair (cdr list-a) (cdr list-b) predicate)))))

(defmacro apply-predicate-to-dotted-lists-optimized (list-a list-b predicate)
  (let ((a (gensym))
	(b (gensym)))
    `(do ((,a ,list-a (cdr ,a))
	  (,b ,list-b (cdr ,b)))
      ((or (null ,a) (null ,b)
	(not (consp (cdr ,a))) (not (consp (cdr ,b)))
	(not (,predicate (car ,a) (car ,b))))
       (or
	(and (null ,a) (null ,b))
	(and (not (consp (cdr ,a))) (not (consp (cdr ,b)))
	     (,predicate (cdr ,a) (cdr ,b))
	     (,predicate (car ,a) (car ,b))))))))



(defun one-to-one-lists-p (a b &optional (test #'eql))
  (and (= (length a) (length b))
       (let ((b (copy-list b)))
	 ;; every member of a must occur in b
	 (and (every #'(lambda (a-mem)
			 (let ((found-p nil))
			   (setf b
				 (delete-if #'(lambda (b-mem)
						(when (funcall test a-mem b-mem)
						  (setf found-p t))) ; return t
					    b
					    :count 1))
			   found-p))
		     a)
	      ;; there may be no members of b which do not occur in a.
	      (null b)))))


;; Like case, except does not require literals.
(defmacro constant-case (key &body clauses)
  (let ((key-sym (gensym)))
    `(let ((,key-sym ,key))
       (cond
	 ,@(mapcar #'(lambda (clause)
		       (cond 
			 ((eql (car clause) t)  `(t ,@(cdr clause)))
			 ((listp (car clause))  `((member ,key-sym (list ,@(car clause)))))
			 (t `((eql ,(car clause) ,key-sym) ,@(cdr clause)))))
		   clauses)))))


;; returns sublist of l which predicate is true
;; always a new list.
(defun filter (p l)
  (let ((r nil))
    (dolist (item l)
	    (if (funcall p item) (push item r)))
    (nreverse r)))


;;;;
;;;; Primitive errors.
;;;;
(defmacro handle-error (error-tag error-handler &body body)
  (let ((local-tag (gensym)))
    `(block ,local-tag
       (funcall ,error-handler (catch ,error-tag 
				 (return-from ,local-tag (progn ,@body)))))))

(defmacro process-err (x)
  `(throw 'process-err ,x))
  
(defmacro handle-process-err (error-handler &body body)
  `(handle-error 'process-err ,error-handler ,@body))



;;;; -docs- (mod bsc)
;;;;
;;;;
;;;;
;;;;	Messages :
;;;;	
;;;;	<message>	: #S(message <id{tag}> list <psmg> <stamp>)
;;;;	<tag>		: <id>
;;;;	
;;;;	Expected tags:
;;;;	
;;;;	Error	: a computation terminated abnormally.
;;;;		  System should alert user.
;;;;	Warning	: an unusual circumstance was noted, computation proceeded.
;;;;		  User should examine.
;;;;	Inform	: an assumption was made, eg a default was used.
;;;;		  User might examine if computation did not give expected result.
;;;;
;;;;
;;;;	Message Stores :
;;;;	 
;;;;	 There is a default global message store which accepts all messages.
;;;;	 Dynamic messages stores may be defined to shadow default store or to define
;;;;	 special stores(Eg, broadcast).
;;;;	 
;;;;	message-emit(<message> &optional <id{store}>)	: NULL
;;;;	messages-peek (&optional <id{store}>)		: <message> list
;;;;	 ** list is LIFO.
;;;;	messages-flush (&optional <id{store}>)		: <message> list
;;;;	 ** list is LIFO.  Clears message store.
;;;;	messages-p (&optional <id{store}>)		: <bool>
;;;;	 ** Warning if store is not bound.
;;;;	
;;;;
;;;;	basic-message (<tag> list <pmsg>)		: <message>
;;;;	warn-message (<tag> list &optional <pmsg>)	: <message>
;;;;	error-message (<tag> list &optional <pmsg>)	: <message>
;;;;
;;;;
;;;;	with-message-store((&optional <id{store}>) &body body)
;;;;	 : <macro>
;;;;	with-message-accumulator((<tag> list &optional <id{store} <pmsg{prefix}>) &body body)
;;;;	 : macro.
;;;;	 ** collects emitted msgs and emits single msg into callers env.
;;;;
;;;;  -page-
;;;;
;;;;	Error control: Errors are normally handled dynamically. When an error is
;;;;	 detected, an error message is thrown. 
;;;;	 
;;;;	raise-error(<message>)			 	: THROW(<message>
;;;;	 **  throws <message>
;;;;
;;;;	with-handle-error (((<tag> list &optional <pmsg{prefix}>) &body handler) &body body)
;;;;	 : macro
;;;;	 ** accumulates default messages and emits single message.
;;;;	 ** If error, result from handler is returned.
;;;;	 ** It is expected to be a common idiom that the handler raise an error itself.
;;;;
;;;;	with-tag ((<tag> list) &body body)		: macro	
;;;;	with-backtrace ((<tag>) &body body)		: macro
;;;;	with-ignore(&body body)				: macro
;;;;	 ** unconditionally tosses any messages. Returns nil if error.
;;;;
;;;;	The implementation uses the same error system to report implementation errors.
;;;;
;;;;	system-error(<pmsg>)				: THROW<message>
;;;;	with-system-error((&optional pmsg) &body body)
;;;;	 ** If <message> thrown in body, then wrapped as system-error message.
;;;;
;;;; -doce-

;;; RLE TODO add reserved message store which means don't collect send immediately.
;;; RLE TODO maybe which takes a hook and calls hook rather than storing.
;;; RLE TODO MILL remove with-ignore-x-error



(defstruct basic-message
  tags pmsg stamp)

(defun tags-of-message (m) (basic-message-tags m))
(defun pmsg-of-message (m) (basic-message-pmsg m)) 
(defun stamp-of-message (m) (basic-message-stamp m))


(defmacro basic-message (tags &optional (pmsg nil))
  (let ((ltags (gensym)))
    `(let ((,ltags ,tags))
      (make-basic-message :tags (if (consp ,ltags) ,ltags (unless (null ,ltags) (list ,ltags)))
       :pmsg ,pmsg
       :stamp (transaction-stamp)))))

;; concats tags of message.
;; this is used in ml compiler (bml-tran) to convert message to tok
;; when trap binds failure value. ML expects only tokens, however,
;; we routinely pass messages though pretending their tokens.
(defun message-to-token (m)
  (intern-system (list-to-string (tags-of-message m)
			  #'string
			  "_")))



(defun warn-message (tags &rest pmsg)
  (basic-message (cons 'warning tags) pmsg))

(defun inform-message (tags &rest pmsg)
  (basic-message (cons 'inform tags) pmsg))

(defun warn-message-p (m)
  (let ((tags (tags-of-message m)))
    (or (eql 'warning tags)
	(and (consp tags)
	     (eql 'warning (car (tags-of-message m)))))))

(defun error-message (tags &rest pmsg)
  (basic-message (cons 'error tags) pmsg))

(defun error-message-p (m)
  (let ((tags (tags-of-message m)))
    (or (eql 'error (tags-of-message m))
	(and (consp tags)
	     (eql 'error (car (tags-of-message m)))))))

(defun tag-message (tags pmsg &optional prefix)
  ;;(format t "tm ~a~%" tags)
  ;;(when (equal tags 'ml) (break "tm"))
  (basic-message tags
		 (if prefix 
		     (cons prefix pmsg)
		     pmsg)))

;; an accumulation of messages
;; internal
(defun flush-message (tags &optional prefix store)
  (let ((messages (messages-flush store)))
    (when messages
      (tag-message tags messages prefix))))



(defun format-message (msg)
  (format-string "~a" msg)
  )


;;;;	
;;;;	Message Suppression.
;;;;	  - at print : avoids print overhead
;;;;	  - at link  : avoids communication overhead.
;;;;	  - at emit  : avoids all overhead.
;;;;	
;;;;	Sooner suppressed, less cost.
;;;;	Later suppressed, easier to unsupress.
;;;;	
;;;;	So messages the user might want to see should be suppressed later
;;;;	and message the user never wants to see should be supressed earlier.
;;;;	
;;;;	More general suppression should be done later, and specific earlier.
;;;;	
;;;;	
;;;;	


(defun message-filter-by-prefix (prefix)
  (let ((lprefix (length prefix)))
    #'(lambda (m)
	(and (>= (length (tags-of-message m)) lprefix)
	     (forall-p #'eql prefix (tags-of-message m))))
    ))

(defun message-filter-or (a b)
  (cond
    ((and (null a) (null b)) nil)
    ((null a) b)
    ((null b) a)
    (t  #'(lambda (m)
	    (or (funcall a m) (funcall b m))))))
      

(defvar *message-emit-suppress* nil)

(defun push-message-emit-suppression (f)
  (setf *message-emit-suppress*
	(message-filter-or f *message-emit-suppress*)))

(defun push-message-emit-prefix-suppression (tags)
  (push-message-emit-suppression (message-filter-by-prefix tags)))


(defvar *default-message-store*  nil)
(defvar *message-stores* nil)
(defvar *warn-break* nil)

(defun message-emit (m &optional store escape-p)
  ;;(setf -m m) (break "me")
  (when (and m
	     (or (null *message-emit-suppress*)
		 (not (funcall *message-emit-suppress* m))))

    (cond
      ((and store escape-p)
       (let ((*message-stores* (cdr (member store *message-stores* :key #'car))))
	 (message-emit m store)))
      (t 
       (when (and *warn-break* (warn-message-p m)) (break "warn-message-emit"))
       (if store
	   (let ((message-store (assoc store *message-stores*)))
	     (if message-store
		 (if (eql 'asynch store)
		     (funcall (cdr message-store) m)
		     (push m (cdr message-store)))
		 (push (basic-message 'unbound-message-store (list* "Unbound Message Store" store m))
		       *default-message-store*)))
	   (push m *default-message-store*)))))
  )


;;;;	 * RLE TODO synch messages accumulated and returned. dynamically variable which messages are asynch
;;;;	 * and which synch. or asynch msgs must be wrapped in special wrapper!. Or 'asynch tok must be in
;;;;	  * message tag list, but must keep with-tag like things from hiding asynch.
;;;;	 can asynch just be a distinct store?? or a special store which spews instead of storing.

(defun message-emit-asynch (m)
  ;; RLE TODO
  (message-emit m 'asynch))


(defun messages-peek (&optional store)
  (if store
      (let ((message-store (assoc store *message-stores*)))
	(if message-store
	    (cdr message-store)
	    (progn (message-emit (warn-message '(unbound-message-store)
					       (list* "Unbound Message Store" store)))
		   nil)))
      *default-message-store*))

(defun messages-flush (&optional store)
  ;;(break "mf")
  (if store
      (let ((message-store (assoc store *message-stores*)))
	(if message-store
	    (prog1 (cdr message-store) (setf (cdr message-store) nil))
	    (progn (message-emit (warn-message '(unbound-message-store)
					       (list* "Unbound Message Store" store)))
		   nil)))
      (prog1 *default-message-store*
	(setf *default-message-store* nil))))

(defun messages-p (&optional store)
  (if store
      (let ((message-store (assoc store *message-stores*)))
	(if message-store
	    (and (cdr message-store) t)
	    (progn (message-emit (warn-message '(unbound-message-store)
					       (list* "Unbound Message Store" store)))
		   nil)))
      (and *default-message-store* t)))



;; may not be sufficient as may want an escape to send messages to shadowed redirect.
(defmacro with-asynch-message-redirect (hook &body body)
  `(let  ((*message-stores* (acons 'asynch ,hook *message-stores*)))
    ,@body))

(defmacro with-message-store ((&optional store) &body body)
  (if store
      `(let ((*message-stores* (acons ,store nil *message-stores*)))
	,@body)
      `(let ((*default-message-store* nil))
	,@body)))


;; if messages left in store they are emitted as accumulation in
;; callers environemnt.
;; tag is tag for mesasge

(defmacro with-message-accumulator ((&optional tags store prefix closure) &body body)
  (let ((m (gensym)))
    `(let ((,m nil))
      (unwind-protect (multiple-value-prog1
			  (with-message-store (,store)
			    (unwind-protect (progn ,@body)
				(when (messages-p ,store)
				  (setf ,m (if ,closure
					       (funcall ,closure (flush-message ,tags ,prefix ,store))
					       (flush-message ,tags ,prefix ,store)))))))
	(when ,m (message-emit ,m ,store))))))

;; only executes unwind on error.
(defmacro with-unwind-error ((&body unwind) &body body)
  `(let ((ok nil))
    (unwind-protect (multiple-value-prog1
			(progn ,@body)
			(setf ok t))
      (unless ok
	,@unwind))))

(defvar *process-break* nil)
(defvar *process-break-message* nil)

(defun raise-error (message)
  (when *process-break*
    (setf *process-break-message* message)
    (break "process break"))

  ;;(format t "E")
  (process-err message))


;;; RLE NAP would probably be better if handler was a closure taking msg as arg.
;;; RLE NAP allows handler to decide how to handle msg. This relieves handler of that
;;; RLE NAP duty but it's confusing (disconcerting) to hide the msg handling.

(defmacro with-handle-error (((&optional tags prefix closure) &body handler) &body body)
  `(with-message-accumulator (,tags nil ,prefix ,closure)
    (handle-process-err #'(lambda (message)
			    (message-emit message)
			    (progn ,@handler))
     ,@body)))

(defvar *failure-message* nil)

(defmacro with-handle-error-and-message (((&optional tags prefix) handler) &body body)
  `(with-message-accumulator (,tags nil ,prefix)
    (handle-process-err ,handler
     ,@body)))


(defmacro with-tag (tags &body body)
  `(with-handle-error ((,tags)	; tags msgs in non-error case
		       (raise-error nil))
    (progn ,@body)))

(defmacro with-error-to-warn (&body body)
  `(with-handle-error (() (message-emit (warn-message nil (messages-flush)))
		       nil
		       )
    (prog1 (progn ,@body) )))


;; RLE TODO tag should not be a tag but should be a data string
(defmacro with-backtrace (s &body body)
  `(with-handle-error (('(backtrace) ,s)	; tags msgs in non-error case
		       ;;(break "wb")
		       (raise-error nil)) ;tags in error case.
    (progn ,@body)))


;; lose messages and error when error.
;; if no error, retains messsages.
(defmacro with-ignore (&body body)
  `(with-handle-error (() (messages-flush)
		       nil
		       )
    (prog1 (progn ,@body)
      ;;(messages-flush)
      )))

(defmacro error-p (&body body)
  `(null (with-ignore (progn ,@body)
	  t)))


(defvar *system-kind-tok* (intern-system *system-kind*))

(defun system-error (pmsg)
  (raise-error (error-message (list *system-kind-tok*) pmsg)))

;; wrapper for code were we expect no errors. Ie if there is an error
;; its a nuprl error and not a user error.
(defmacro with-system-error ((tags &optional pmsg) &body body)
  `(with-handle-error (()
		       (raise-error (flush-message (list* 'error *system-kind-tok* ,tags) ,pmsg)))

    ,@body))




;; ^ matches begin
;; $ matches end
(defun string-pattern-search (keyf pattern &optional match-case-p)
  (declare (string pattern))
  (let ((ps pattern)
	(beginp nil)
	(endp nil))

    (when (string= ps "")
      (return-from string-pattern-search
	#'(lambda (o)
	    (declare (ignore o))
	    t)))
	
    (when (char= (char ps 0) #\^)
      (setf beginp t ps (subseq ps 1)))
    (when (char= (char ps (1- (length ps))) #\$)
      (setf endp t ps (subseq ps 0 (1- (length ps)))))

    (let ((pl (length ps)))
      #'(lambda (o)
	  (let* ((s (string (funcall keyf o)))
		 (sl (length s)))
	    (declare (string s))
	    (when (>= sl pl)
	      (cond
		((and beginp endp)
		 (if match-case-p
		     (string= ps s)
		     (string-equal ps s)))

		(beginp
		 (if match-case-p
		     (string= ps s :end2 pl)
		     (string-equal ps s :end2 pl)))

		(endp
		 (if match-case-p
		     (string= ps s :start2 (- sl pl))
		     (string-equal ps s :start2 (- sl pl))))
		(t (let ((n (1+ (- sl pl))))
		     (do ((i 0 (1+ i))
			  (j pl (1+ j)))
			 ((or (>= i n)
			      (if match-case-p
				  (string= ps s :start2 i :end2 j)
				  (string-equal ps s :start2 i :end2 j)))
			  (not (= i n)))))))))))))

;; find substring m in s and return prefix and suffix.
;; if case-p t then case must match.
(defun string-split (m s &optional case-p)
  (let ((i (search m s :start2 0 :test (if case-p #'string= #'string-equal))))
    (when (null i)
      (raise-error (error-message '(string-split search found not) m s))) 
    (cons (subseq s 0 i) (subseq s (+ i (length m))))))



;;;;	Files and paths : 
;;;;	
;;;;	Files will be addressed using the following concepts:
;;;;	  - path : root for a pathname.
;;;;	  - dir list : list of directory names
;;;;	  - filename : name of file
;;;;	  - type : type of file
;;;;	
;;;;	
;;;;	prl-make-pathname (<string>)			: <path>
;;;;	prl-extend-pathname (<path> <string> list)	: <path>
;;;;	prl-make-filename (<path> <string{dir}> list <string{filen}> &optional <string{type}>)
;;;;	  : <filename>
;;;;
;;;;	RLE TODO ??? : would be better to use logical pathnames ala portable utilities
;;;;	RLE TODO ??? : ie implement this on top of that.


(defun prl-add-trailing-separator (path)
  (let* ((l (length path))
	 (len (length *path-separator-string*)))
    (if (string= *path-separator-string* path :start2 (- l len))
	path
	(concatenate 'string path *path-separator-string*))))


(defun path-to-path-string (path)
  (if (stringp path)
      (prl-add-trailing-separator path)
      (if (symbolp path)
	  (case path
	    (nuprl (break "ptps nuprl") *system-path-prefix*)
	    (system *system-path-prefix*)
	    (otherwise (raise-error (error-message '(path token unknown) path))))

	  (raise-error (error-message '(path unknown) (princ-to-string path))))))

(defun prl-make-pathname (p)
  (path-to-path-string p))

(defun prl-remove-trailing-separator (path)
  (let ((lens (length *path-separator-string*))
	(lenp (length path)))
    (if (and (> lenp lens)
	     (string= *path-separator-string* (subseq path (- lenp lens))))
	(subseq path 0 (- lenp lens))
	path)))

(defun prl-extend-pathname (path dirs)
  (flet ((add-separator (x) (concatenate 'string x *path-separator-string*)))
    (reduce #'(lambda (x y) (concatenate 'string x y))
	    `(,(prl-make-pathname path)
	      ,@(mapcar #'add-separator (mapcar #'string dirs))))))

(defun prl-make-filename (path dirs fname &optional type)
  (if type
      (concatenate 'string
		   (if dirs
		       (prl-extend-pathname path dirs)
		       (prl-make-pathname path))
		   fname
		   "."
		   type)
      (concatenate 'string
		   (if dirs
		       (prl-extend-pathname path dirs)
		       (prl-make-pathname path))
		   fname)))



;;;
;;; System path
;;;

(defun complete-system-path (dirs fname &optional type)
  (prl-make-filename *system-path-prefix*
		     dirs
		     fname
		     type))

(defun extend-system-path (dirs)
  (prl-extend-pathname *system-path-prefix* dirs))
      
		   


;;;;
;;;;	Unix
;;;;



#+lucid (load-foreign-libraries nil)

#+allegro-v3.1 (require :foreign)



#+cmu
(progn
  (defunff (getpid) int ())
  (defunff (getuid) int ())
  (defunff (gethostid) unsigned ())
  (defunff (my-gethostname ffi-gethostname) string ())
)

#+cmu 
(defun gethostname ()
  (format t "cmu can't yet get hostname")
  nil)


#+lucid
(progn
  (def-foreign-function (getpid
			 (:return-type :signed-32bit) (:language :c)
			 (:name "_getpid"))
    )
  (def-foreign-function (getuid
			 (:return-type :signed-32bit) (:language :c)
			 (:name "_getuid"))
    )

  (def-foreign-function (gethostid
			 (:return-type :signed-32bit) (:language :c)
			 (:name "_gethostid"))
    )
  (def-foreign-function (gethostname
			 (:return-type  :simple-string) (:language :c)
			 (:name "_my_gethostname"))
    )
  (def-foreign-function (libc-gethostname
			 (:return-type :signed-32bit) (:language :c)
			 (:name "_gethostname"))
    (name (:pointer :character))
    (length :unsigned-32bit))
  )

#+allegro
(progn
  (eval-when (compile)
    (require :foreign))
  (ff:defforeign 'getpid
		 :entry-point (ff:convert-to-lang #-:dlwin "getpid" #+:dlwin "_getpid")
		 :return-type :integer)

  (ff:defforeign 'getuid
	 	 :entry-point (ff:convert-to-lang "getuid")
		 :return-type :integer)

  #+:dlwin
  (ff:defforeign 'gethostid
                 :entry-point (ff:convert-to-lang 'my_gethostid) :return-type :integer)

  #-:dlwin
  (ff:defforeign 'gethostid
                 :entry-point  (ff:convert-to-lang 'gethostid) :return-type :fixnum)

  (ff:defforeign 'ff-gethostname
		 :entry-point (ff:convert-to-lang "my_gethostname")
		 :return-type :fixnum))

#+allegro
(defun gethostname ()
  (ff:char*-to-string (ff-gethostname)))

;;; NIL if not supported by Lisp.
(defun get-environment-variable (str)
  nil
  #+cmu (getenv str)
  #+(or lcl3.0 lcl4.0) (lcl:environment-variable str)
  #+allegro (system:getenv str)
  #+aclpc (getenv str)
  )

;;; may return NIL
(defun local-host ()
 (or (get-environment-variable "HOST")
     (get-environment-variable "HOSTNAME")))

(defun local-hostname ()
  (or (get-environment-variable "HOST")
      (get-environment-variable "HOSTNAME")
      (gethostname)
      ))

(defun local-host-ipaddr ()
  (format-string "~x" (abs (gethostid))))

(defun unix-process-id ()
   #+(or lucid allegro cmu)(getpid)
   #-(or lucid allegro cmu)(error "unix-process-id")
   )



;;;;
;;;;	Misc.
;;;;

#-dontinline
(eval-when (compile)
  (proclaim '(inline onep neg-onep)))

(defun onep (x)
    (= x 1))

(defun neg-onep (x)
    (= x -1))


(defun tok-upcase (s)
  (intern-system (string-upcase (string s))))

;; not widely used, ie not worth the savings.
(defconstant *arbitrary-max-zero-list-length* 15)

(defparameter *zero-lists*
	      (make-array (1+ *arbitrary-max-zero-list-length*)
			  :initial-contents
			  (let ((result (list nil)))
			    (dotimes (x *arbitrary-max-zero-list-length* (nreverse result))
			      (setf result (cons (cons 0 (car result)) result))))))

(defun list-of-zeros (i)
  (if (< i  *arbitrary-max-zero-list-length*)
      (elt *zero-lists* i)
      (do ((j 0 (1+ j))
	   (lz nil (cons 0 lz)))
	  ((= j i) lz))))


;; limited by number of function arg limit in call to concatenate
(defun list-to-string (items item-to-string 
			     &optional (partition ", ") (left-delim "(") (right-delim ")"))
  (concatenate 'string
	       left-delim
	       (if items
		   (funcall item-to-string (car items))
		   "")
	       (apply #'concatenate 'string
				    (mapcan #'(lambda (item)
						(list partition 
						      (funcall item-to-string item)))
					    (cdr items)))
	       right-delim))


;;;; RLE TODO DOC list-divide
(defun list-divide (list recognizer &optional (map-sublist #'(lambda (x) x)) map-divider)
  (do ((acc nil)
       (big-acc nil)
       (l list (cdr l)))
      ((null l)
       (when acc (push (funcall map-sublist (nreverse acc)) big-acc))
       (nreverse big-acc))
    (if (funcall recognizer (car l))
	(progn
	  (when acc
	    (push (funcall map-sublist (nreverse acc)) big-acc))
	  (setf acc nil)
	  (if map-divider
	      (push (funcall map-divider (car l)) big-acc)
	      (push (car l) acc)))
	(push (car l) acc))))

;; TODO RLE might put this after bio to get bio error handling.
;; TODO RLE inline trivial mark functions.


(defun flatten (l)  
  (cond ((null l) '())
	((atom l) (list l))
	(t (append (flatten (car l)) (flatten (cdr l))))))


;;;; -docs- (mod bsc)
;;;;
;;;;	Marks: general purpose structure for abstractly hanging properties 
;;;;		on derived structures. A mark consists of a label and a property or some
;;;;		combination of properties (eg a list of properties).
;;;;		Marks may be added, removed, and retrieved from items.
;;;;	NB: unmarked items return nil when prompted for a property. Ie, a null property
;;;;	    is not easily distinquished from no property. This is considered a feature.
;;;;
;;;;
;;;;	<marks>			: #mark[<mark> list]
;;;;	<label>			: <id>	
;;;;	<prop>			: <*>
;;;;	<mark>			: (<label> . <prop>)
;;;;	
;;;;
;;;;
;;;;	mark(<*> <label> &optional <prop>)		: <prop> | NIL
;;;;	mark-f(<*{item}> <label> f &rest other-args)	: <prop>
;;;;	 ** if item not marked with label, then executes (apply f item other-args)
;;;;	 ** to create property.
;;;;	 ** if already marked then property is not changed and f is not called.
;;;;	markp(<*> <label>)				: <bool>
;;;;	mark-value(<*> <label>) 			: <prop>
;;;;	unmark(<*> <label> &key warn)
;;;;	 ** warns if :warn is t and label finds a count marker.
;;;;	mark-values(<*>)				: <mark> list
;;;;	marks(<*> <mark{new}> list)			: <mark{current}> list
;;;;	 ** appends current to new marks.
;;;;
;;;; 	Count Properties: Property includes a counter. This allows recursive marking
;;;;	 where the item will not be unmarked until the outermost unmark is called.
;;;;	 A non-count property marker takes precedence over a similar count property.
;;;;	 It is left to the user to insure that the uses are not interleaved. This
;;;;	 does allow one to mark an item with a property such that subsequent incf and
;;;;	 decfs are ignored. Count starts at 0 (ie first incf results in 1), when
;;;;	 counter is decremented to 0, the mark is removed.
;;;;
;;;;	<count-prop>		: (<label> . #prop[INTEGER{counter} <prop>])
;;;;
;;;;	mark-incf (<*{item}> <label> &optional (<prop> <*{item}>))	: <prop{old}> | NIL
;;;;	 ** increments counter and sets property unless new property is eq to item.
;;;;	mark-incf-f(<*> <label> f &rest other-args)	: <prop>
;;;;	 ** f is not called if <prop> exists.
;;;;	mark-decf (<*> <label>)				: <prop>
;;;;	 ** decrements counter; unmarks if count goes to zero.
;;;;
;;;;
;;;;	List Properites: 
;;;;
;;;;	<list-prop>		: (<label> . <prop> list)
;;;;	<list-prop>		| #prop(<label> . #prop[INTEGER{counter} <prop> list])
;;;;
;;;;	mark-push (<*> <label> <prop> &key countp)	: <prop> list
;;;;	mark-peek (<*> <label>)				: <prop>
;;;;	mark-pop (<*> <label>)				: <prop>
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Label Memory: a mechanism to allow labeled marks to be active only within
;;;;	 a dynamic extent. User must take care not to use labels which may be
;;;;	 permanent in other contexts. EG, if an item is already marked and then
;;;;	 has a mark remembered for same label. Forgetting marks of label will
;;;;	 unmark original.
;;;;
;;;;	remember-marks-of-label (<label>)
;;;;	remembering-marks-of-label (<label>)		: <bool>
;;;;	any-marks-of-label-remembered (<label)		: <bool>
;;;;	remembered-marks-of-label (<label>)		: <*{item}> list
;;;;	forget-marks-of-label (<label>)			: <*{item}> list.
;;;;	 ** unmarks items in list.
;;;;
;;;;	with-mark ((<label> <*{item}> &optional <prop>) &body body)
;;;;	with-mark-f ((<label> <closure{f}> <*{item}> &optional other-args) &body body)
;;;;	 ** f (<*>)	: <prop>
;;;;	with-map-mark ((<label> <*{item}> list &optional <prop> list) &body body)
;;;;	with-map-mark-f ((<label> <closure{f}> <*{item}> list &optional other-lists)
;;;;			 &body body)
;;;;	 ** f (<*>)	: <prop>
;;;;	with-map-mark-push ((<label> <closure{pushf}> <*{item}> list <closure{popf}>
;;;;			     &rest other-lists)
;;;;			    &body body)
;;;;	 ** pushf (<*>)	: <prop>
;;;;	 ** popf (<*>)	: NULL
;;;;
;;;; -doce-

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      clear-marks markp marks mark-values
	      remembering-marks-of-label remembered-marks-of-label any-marks-of-label-remembered 
	      )))

(defstruct (marks (:copier wimpy-copy-marks))
  (alist nil))

(defstruct count-marker
  (count 1)
  (value nil))

(defun copy-marks (old-marks)
  (mapcar #'(lambda (marker)
	      (if (count-marker-p (cdr marker))
		  (cons (car marker) (copy-count-marker (cdr marker)))
		  (cons (car marker) (cdr marker))))
	  (marks-alist old-marks)))


(defun clear-marks (marks)
  (setf (marks-alist marks) nil))


(defvar *labels-remembered* nil)

(defmacro check-label-memory (label object)
  `(when *labels-remembered*
     (let ((cell (assoc ,label *labels-remembered*)))
       (when cell
	 (push ,object (cdr cell))))))



; returns old value if var was marked with non-nil value
; returns nil if var was not marked or was marked with nil.
(defun mark (marks mark &optional value)
  (let ((marker (assoc mark (marks-alist marks))))
    (if marker
	(prog1
	  (cdr marker)
	  (setf (cdr marker) value))
	(progn
	  (check-label-memory mark marks)
	  (setf (marks-alist marks) (acons mark value (marks-alist marks)))
	  nil))))


(defun mark-f (marks mark f)
  (let ((marker (assoc mark (marks-alist marks))))
    (if marker
	(cdr marker)
	(progn
	  (check-label-memory mark marks)
	  (cdar (setf (marks-alist marks) (acons mark (funcall f marks) (marks-alist marks))))))))


(defun markp (marks mark)
  (assoc mark (marks-alist marks)))

(defun mark-value (marks mark)
  (let ((marker (cdr (assoc mark (marks-alist marks)))))
    (if (typep marker 'count-marker)
	(count-marker-value marker)
	marker)))

;; validity of theorem prover depends on there being a mark.
(defmacro mark-value-must (marks mark)
  `(let ((markv (assoc ,mark (marks-alist ,marks))))
    (unless markv (break "mark-value-must"))
    (cdr markv)))

(defun unmark (marks mark &key warn)
  (setf (marks-alist marks)
	(delete-if #'(lambda (item) (when (eq (car item) mark)
				      (when (and warn (count-marker-p (cdr item)))
					(warn "unmarking a count marker"))
				      t))
		   (marks-alist marks)))
  (values))


(defun marks (marks alist)
  (let ((cur-marks (marks-alist marks)))
    (prog1 cur-marks
      (setf (marks-alist marks) (append cur-marks alist)))))


;; alist arg supersedes marks. returns old.
;; may result is sharing of marks values. ie if counter then counter shared which is not good.
;; so removes count markers, and copies all conses.
(defun marks-union (marks alist)
  (let* ((cur-marks (marks-alist marks))
	 (nmarks (mapcan #'(lambda (m)
			     (unless (count-marker-p (cdr m))
			       (list (cons (car m) (cdr m)))))
			 alist)))
    
    (dolist (m cur-marks)
      (unless (count-marker-p (cdr m))
	(setf nmarks (pushnew (cons (car m) (cdr m)) nmarks :key #'car))))

    (setf (marks-alist marks) nmarks)
    cur-marks))


(defun mark-values (marks)
  (marks-alist marks))


;; returns marks; c for continue.
(defun marks-c (marks alist)
  (marks marks alist)
  marks)

(defun marks-union-c (marks alist)
  (marks-union marks alist)
  marks)


;;; stack marker : marker is a list.

;;; for following functions 
;;;   - old value of mark must be a list.
;;;   - if not previously marked then old value is nil.

(defun mark-push (marks mark value &key countp)
  (let ((marker (assoc mark (marks-alist marks))))
    (if marker 
	(if (typep (cdr marker) 'count-marker)
	    (push value (count-marker-value (cdr marker)))
	    (push value (cdr marker)))
	(let ((stack (list value)))
	  (setf (marks-alist marks)
		(acons mark (if countp
				(make-count-marker :value stack)
				(progn (check-label-memory mark marks)
				       stack))
		       (marks-alist marks)))
	  stack))))

(defun mark-pop (marks mark)
  (let ((marker (assoc mark (marks-alist marks))))
      (when marker 
	(if (typep (cdr marker) 'count-marker)
	    (pop (count-marker-value (cdr marker)))
	    (pop (cdr marker))))))

(defun mark-peek (marks mark)
  (let ((marker (assoc mark (marks-alist marks))))
      (when marker 
	(if (typep (cdr marker) 'count-marker)
	    (car (count-marker-value (cdr marker)))
	    (cadr marker)))))


;;; count markers

;; value is ignored if same as marks.
;; returns old value. nil if marking now.
(defun mark-incf (marks mark &optional (value marks))
  (let ((marker (assoc mark (marks-alist marks))))
    (if marker
	(if (count-marker-p (cdr marker))
	  (prog1
	      (count-marker-value (cdr marker))
	    (incf (count-marker-count (cdr marker)))
	    (unless (eq marks value)
	      (setf (count-marker-value (cdr marker)) value)))
	  (prog1
	      (cdr marker)
	    (unless (eq marks value)
	      (setf (cdr marker) value))))
	(progn
	  (setf (marks-alist marks) (acons mark (make-count-marker 
						 :value (unless (eq marks value) value))
					   (marks-alist marks)))
	  nil))))

(defun mark-incf-f (marks mark f &rest other-args) 
  (let ((marker (assoc mark (marks-alist marks))))
    (if marker
	(if (typep (cdr marker) 'count-marker)
	    (prog1
		(count-marker-value (cdr marker))
	      (incf (count-marker-count (cdr marker))))
	    (cdr marker))
	(cdar (setf (marks-alist marks) (acons mark (make-count-marker 
						     :value (apply f marks other-args))
					       (marks-alist marks)))))))

;; decrements count and returns value.
(defun mark-decf (marks mark)
  (let ((marker (assoc mark (marks-alist marks))))
    (if (typep (cdr marker) 'count-marker)
	(progn
	  (if (eql 1 (count-marker-count (cdr marker)))
	      (setf (marks-alist marks)
		    (delete-if #'(lambda (item) (eq (car item) mark))
			       (marks-alist marks)))
	      (decf (count-marker-count (cdr marker))))
	  (count-marker-value (cdr marker)))
	(cdr marker))))


(defun remember-marks-of-label (label)
  (setf *labels-remembered* (acons label nil *labels-remembered*))
  (values))

(defun remembering-marks-of-label (label)
  (and (assoc label *labels-remembered*) t))

(defun remembered-marks-of-label (label)
  (cdr (assoc label *labels-remembered*)))

(defun any-marks-of-label-remembered (label)
  (and (cdr (assoc label *labels-remembered*)) t))

(defun forget-marks-of-label (label)
  (let ((marked (if (eql (caar *labels-remembered*) label)
		    (prog1 (cdar *labels-remembered*)
		      (setf *labels-remembered* (cdr *labels-remembered*)))
		    (do ((alist *labels-remembered* (cdr alist)))
			((or (eql label (caadr alist)) (null (cdr alist)))
			 (prog1 (cdadr alist)
			   (if alist
			       (setf (cdr alist) (cddr alist))
			       (warn "forgetting marks of label which we have no memory of.")))
			 )))))
    
  (mapc #'(lambda (object) 
	    (unmark object label :warn t))
	marked)
  marked))


(defmacro with-label-memory (label &body body)
  `(unwind-protect
    (progn (remember-marks-of-label ,label)
	   ,@body)
    (forget-marks-of-label ,label)))

;; with  - uses count so that you dont have to worry about embedded calls.

(defmacro with-mark ((mark markee &optional value) &body body)
  `(unwind-protect 
       (progn 
	 ,(if value
	      `(mark-incf ,markee ,mark ,value)
	      `(mark-incf ,markee ,mark))
	 ,@body)
     (mark-decf ,markee ,mark)))
	   
;; map
(defmacro with-map-mark ((mark markees &optional mark-values) &body body)
  `(unwind-protect (progn 
		     ,(if mark-values
			  `(mapc #'(lambda (markee value)
				     (mark-incf markee ,mark value))
				 ,markees
				 ,mark-values)
			  `(mapc #'(lambda (markee)
				     (mark-incf markee ,mark))
				 ,markees))
			  ,@body)
     (mapc #'(lambda (markee)
	       (mark-decf markee ,mark))
	   ,markees)))

;; mark with value from function call 
;;   value is not changed if mark already exists (function is not called)
(defmacro with-mark-f ((mark f markee &rest other-args) &body body)
  `(unwind-protect
       (progn
	 (mark-incf-f ,markee ,mark ,f ,@other-args)
	 ,@body)
     (mark-decf ,markee ,mark)))
	   
;; list - does not change value if mark already exists (function is not called).
;;    if function need be called unconditionally then use with-mark-m with mapcar.
(defmacro with-map-mark-f ((mark f things &rest other-lists) &body body)
  `(unwind-protect (progn (mapc 
			    #'(lambda (thing &rest other-stuff)
				(apply #'mark-incf-f thing ,mark ,f other-stuff))
			    ,things
			    ,@other-lists)
			  ,@body)
     (mapc #'(lambda (thing)
	       (mark-decf thing ,mark))
	   ,things)))


(defmacro with-map-mark-push ((mark pushf objects popf &rest other-lists) &body body)
  (let ((objects-var (gensym)))
    `(let ((,objects-var ,objects))	;; if functions only gets called once
       (unwind-protect 
	   (progn 
	     (mapc #'(lambda (object &rest other-stuff)
		       (mark-push object ,mark (apply ,pushf object other-stuff)))
		   ,objects-var
		   ,@other-lists)
	     ,@body)
	 (mapc (if ,popf
		   #'(lambda (object)
		       (funcall ,popf object (mark-pop object ,mark)))
		   #'(lambda (object)
		       (mark-pop object ,mark)))
	       ,objects-var)))))






;;;;
;;;;	Doubly Linked Lists:
;;;;
;;;;	 elements of doubly linked lists must be structures derived from dlink structure.
;;;;	 There is little error checking done. For example, most functions assum that args
;;;;	 are the appropriate structures.
;;;;
;;;;
;;;;
;;;;	make-dlist ()						: <link>
;;;;	 ** <link>'s prev and next pointers point to <link>.
;;;;
;;;;	next-of-dlink (<link>)					: <link>
;;;;	prev-of-dlink (<link>) 					: <link>
;;;;	
;;;;	dlist-insert(<link> <link{before}>)			: NULL
;;;;	dlist-remove(<link>)					: NULL
;;;;
;;;;	dlist-map(<link{start}> <link{stop}> <closure>)	 	: NULL
;;;;	 ** exclusive.
;;;;	 ** f(<link>) 		: NULL
;;;;	 ** Probably not a good idea to modify dlist during map.
;;;;
;;;;	top-of-dlqueue (dlq) : link or nil
;;;;	endp-dlqueue (link dlq) : bool
;;;;
;;;;	dadd(<link> <link{before}>)
;;;;	 ** adds to end.
;;;;
;;;;	dpop(dlq) : link
;;;;	 ** error to pop from empty queue.
;;;;	dpop-after(link) : link
;;;;	dpop-before(link) : link
;;;;
;;;;	*** the dlq is simply a degenerate link. It might be worthwhile to define an identifiable
;;;;	*** structure and have functions which take dlq args check for appropriate structure.
;;;;

(defun dlink-print-function (dlink stream depth)
  (declare (ignore dlink depth))

  (format stream "dlink"))

;; doubly linked list data structures and functions
(defstruct (dlink (:print-function dlink-print-function))
  (prev nil)
  (next nil))

(defun next-of-dlink (link) (dlink-next link))
(defun prev-of-dlink (link) (dlink-prev link))

(defun make-dlist (&optional q)
  (let ((q (or q (make-dlink))))
    (setf (dlink-prev q) q
	  (dlink-next q) q)
    q))

(defun dlist-insert (link dlist)
  (setf (dlink-next link) (dlink-next dlist))
  (setf (dlink-prev link) dlist)
  (setf (dlink-prev (dlink-next dlist)) link)
  (setf (dlink-next dlist) link))

(defun dlist-remove (link)
  (setf (dlink-next (dlink-prev link)) (dlink-next link)
	(dlink-prev (dlink-next link)) (dlink-prev link)
	(dlink-next link) nil
	(dlink-prev link) nil))
  
(defun dlist-map (start stop f)
  (do ((head (next-of-dlink start) (next-of-dlink head)))
      ((or (null head) (eql stop head)) (values))
    (funcall f head)))


(defun dadd (l q)
  (setf (dlink-prev l) (dlink-prev q))
  (setf (dlink-next l) q)
  (setf (dlink-next (dlink-prev q)) l)
  (setf (dlink-prev q) l))

(defun dpop (l)
  (if (eq (dlink-next l) l)
    (error "popping from empty doubly linked list. (after)")
    (progn
      (setf (dlink-next (dlink-prev l)) (dlink-next l))
      (setf (dlink-prev (dlink-next l)) (dlink-prev l))))
  l)


(defun dpop-after (l)
  (if (eq (dlink-next l) l)
    (error "popping from empty doubly linked list. (after)")
    (let ((r (dlink-next l)))
      (setf (dlink-next l) (dlink-next r))
      (setf (dlink-prev (dlink-next r)) l)
      r)))

(defun dpop-before (l)
  (if (eq (dlink-prev l) l)
    (error "popping from empty doubly linked list. (after)")
    (let ((r (dlink-prev l)))
      (setf (dlink-prev l) (dlink-prev r))
      (setf (dlink-next (dlink-prev r)) l)
      r)))


;; need to optimize for list?
#|
(defun walk-p (p sexpr)
  (labels ((visit-p (s)
	     (cond
	       ((null s) t)
	       ((not (consp s))
		(funcall p s))
	       (t (and (visit-p (car s))
		       (visit-p (cdr s)))))))

    (visit-p sexpr)))
|#

;; tail optimization was not occuring for previous version thus this somewhat more
;; obfuscated version to avoid blowing stack on large lists.
(defun walk-p (p sexpr)
  (labels ((visit (s)
	     (cond
	       ((null s) t)
	       ((not (consp s))
		(unless (funcall p s)
		  (return-from walk-p nil)))
	       (t (do ((l s (cdr l)))
		      ((not (consp l)) (visit l))
		    (visit (car l)))))))

    (visit sexpr)
    t))

(defun walk (f sexpr)
  (walk-p #'(lambda (s) (funcall f s) t)
	  sexpr))



;;;;	
;;;;	Queue.
;;;;	

;;;;	
;;;;	queue-push (q *)	: NULL
;;;;	queue-pop (q)		: * | NULL
;;;;	queue-peek (q)		: * | NULL
;;;;	
;;;;	length-of-queue(q)	: int

;;;;	

;; take from front put on back.

(defstruct queue
  (first nil)
  (last nil)
  (len 0))

(defun new-queue ()
  (make-queue))

(defun set-queue (q l)
  (setf (queue-first q) l
	(queue-last q) (last l)
	(queue-len q) (length l)))
  
(defun queue-push (q item)
  (let ((l (queue-last q))
	(n (cons item nil)))
    (incf (queue-len q))
    (setf (queue-last q) n)
    (if (null l)
	(setf (queue-first q) n)
	(setf (cdr l) n)))
  (values))

(defun queue-push-front (q item)
  (let* ((l (queue-first q))
	 (n (cons item l)))
    (incf (queue-len q))
    (setf (queue-first q) n)
    (when (null l) (setf (queue-last q) n)))

  (values))

(defun queue-pop (q)
  (let ((f (queue-first q)))
    (when f
      (decf (queue-len q))
      (let ((n (cdr f)))
	(setf (queue-first q) n)
	(unless n
	  (setf (queue-last q) nil)))
      (car f))))

(defun queue-peek (q)
  (car (queue-first q)))

(defun queue-peek-ahead (q)
  (cadr (queue-first q)))

(defun queue-peek-last (q)
  (car (queue-last q)))

(defun length-of-queue (q)
  (queue-len q))

(defun queue-list (q)
  (queue-first q))

(defun ratio-to-decimal (nd num denom)
  (if (zerop denom)
      "div-by-0"
      (format-string (format-string "~~~af" nd) (float (/ num denom)))))


;;;
;;; time-space reporting.
;;;

(defun num-secs-to-string (num)
  (cond
    ((< num .01)  ;; .01 = 10,000 microseconds
     (format-string " ~5:DUS" (round (* 1000000 num))))
    ((< num 10)	 ;; 10 = 10,000,000 US
     (format-string " ~5:DMS" (round (* 1000 num))))
    ((< num 600) ; 10 min
     (format-string " ~5:DSC" (round num)))
    ((< num 36000) ; 10 hours
     (format-string " ~5:DMN" (round (/ num 60.0))))
    ((< num 864000) ; 10 days
     (format-string " ~5:DHR" (round (/ num 3600.0))))
    ((< num 315360000) ; 10 years
     (format-string " ~5:DDY" (round (/ num 86400.0))))
    ((< num 315360000000) ; 10,000 years
     (format-string " ~5:DYR" (round (/ num 31536000.0))))
    (t
     (format-string " ~6EYR" (/ num 31536000.0)))))

(defun microseconds-to-string (num)
  (cond
    ((< num 10000)
     (format-string " ~5:DUS" num))
    ((< num 10000000)
     (format-string " ~5:DMS" (round (/ num 1000.0 ))))
    ((< num 600000000) ; 10 min
     (format-string " ~5:DSC" (round (/ num 1000000.0))))
    ((< num 36000000000) ; 10 hours
     (format-string " ~5:DMN" (round (/ num 60000000.0))))
    ((< num 864000000000) ; 10 days
     (format-string " ~5:DHR" (round (/ num 3600000000.0))))
    ((< num 315360000000000) ; 10 years
     (format-string " ~5:DDY" (round (/ num 86400000000.0))))
    ((< num 315360000000000000) ; 10,000 years   - 315,360,000,000,000,000
     (format-string " ~5DYR" (round (/ num 31536000000000.0))))
    (t
     (format-string " ~6EYR" (/ num 31536000000000.0)))))


  
(defun num-to-string (num &optional (adj " "))
  (cond
    ((< num 10240)
     (format-string "~6:D ~a" (round num) adj))

    ;; Kilobytes
    ((< num 10485760)
     (format-string " ~5:DK~a" (round (/ num 1024)) adj))

    ;; megabytes
    ((< num 10737418240)
     (format-string " ~5:DM~a" (round (/ num 1048576)) adj))

    ;; gigabytes
    ((< num 10995116277760)
     (format-string " ~5:DG~a" (round (/ num 1073741824)) adj))

    ;; terabytes
    ((< num 11258999068426240)
     (format-string " ~5:DT~a" (round (/ num 1099511627776)) adj))

    (t (format-string "~6EG~a" (/ num 1073741824.0) adj))))

(defun num-bytes-to-string (num)
  (num-to-string num "B"))




;;;;	
;;;;	<time-stats>
;;;;	
;;;;	with-time (<closure{update-f}> &body)
;;;;	  * update-f (<closure>)
;;;;	with-time-stats (body)		: <time-stats>
;;;;	
;;;;	with-accumlate-time-stats (<closure{update-f}>) &body
;;;;	time-stats-accumulated () : <time-stats>
;;;;	
;;;;	report-time-stats (STRING{hdr} <time-stats>)
;;;;
;;;;	time-stats-collect <closure{callback}>		: <time-stats>
;;;;	  * callback (<closure{update}>)
;;;;	  * update (<time-stats>)
;;;;	
;;;;	with : macro calls caller. result is body.
;;;;	collect : caller calls collect, result is time.
;;;;	

;; possibly makes more sens to have etime and cpu time in micro seconds
;; nil means not measured.
;; 0 means measured to be zero.

(defstruct time-stats
  (etime nil)	;; in microseconds.
  (cputime nil) ;; in microseconds.
  ;;usertime systime
  (pfaults nil)
  ;;diskio netio
  (dyncons nil)
  (dyngc nil)
  (ephcons nil)
  (ephgc nil))

(defun elapsed-of-time-stats (ts) (time-stats-etime ts))
(defun cpu-of-time-stats (ts) (time-stats-cputime ts))
(defun page-faults-of-time-stats (ts) (time-stats-pfaults ts))
(defun dynamic-consing-of-time-stats (ts) (time-stats-dyncons ts))
(defun dynamic-gc-of-time-stats (ts) (time-stats-dyngc ts))
(defun ephemeral-consing-of-time-stats (ts) (time-stats-ephcons ts))
(defun ephemeral-gc-of-time-stats (ts) (time-stats-ephgc ts))



(defun new-time-stats (etime cputime pfaults dyncons dyngc ephcons ephgc)
  (make-time-stats :etime etime
		   :cputime cputime
		   :pfaults pfaults
		   :dyncons dyncons
		   :dyngc dyngc
		   :ephcons ephcons
		   :ephgc ephgc))


(defun float-seconds-to-microseconds (float)
  (round (* 1000000 float)))

(defun internal-time-to-microseconds (intt)
  (round (* (/ intt internal-time-units-per-second) 1000000)))

(defmacro with-mytime (&body body)
  (let ((run (gensym))
	(real (gensym)))
    `(let ((,run (get-internal-run-time))
	   (,real (get-internal-real-time)))
      (progn ,@body)
      (values
       (internal-time-to-microseconds (- (get-internal-real-time) ,real)) ; etime
       (internal-time-to-microseconds (- (get-internal-run-time) ,run))	; cputime
       0				; usertime
       0				; systime
       0				; pfaults
       0				; diskio
       0				; netio
       0				; dyncons
       0				; dyngc
       0				; ephcons
       0				; ephgc
       ))))


(defmacro with-time-stats (&body body)
  (let ((etime (gensym))
	(cputime (gensym))
	(usertime (gensym))
	(systime (gensym))
	(pfaults (gensym))
	(diskio (gensym))
	(netio (gensym))
	(dyncons (gensym))
	(dyngc (gensym))
	(ephcons (gensym))
	(ephgc (gensym)))
  
    `(mlet* (((,etime ,cputime ,usertime ,systime ,pfaults ,diskio ,netio ,dyncons ,dyngc ,ephcons ,ephgc)
	      #+lucid(time1 (progn ,@body))
	      #-lucid(with-mytime ,@body)
	      (declare (ignore ,diskio ,netio ,systime ,usertime))))

      #+lucid
      (new-time-stats
       (float-seconds-to-microseconds ,etime)
       (float-seconds-to-microseconds ,cputime)
       ,pfaults ,dyncons ,dyngc ,ephcons ,ephgc)

      #-lucid
      (new-time-stats ,etime ,cputime ,pfaults ,dyncons ,dyngc ,ephcons ,ephgc)
      )))


(defmacro with-time (update-f &body body)
  (let ((result (gensym)))
    `(let ((,result nil))
      (funcall ,update-f (with-time-stats (setf ,result (multiple-value-list ,@body))))
      (values-list ,result))))


(defun report-time-stats (stream stats &optional (descriptor 'short))
  
  (if (eql descriptor 'short)
      (progn
	(format stream
		"~%Elapsed time ~a,              CPU time    ~a."
		(microseconds-to-string (elapsed-of-time-stats stats))
		(microseconds-to-string (cpu-of-time-stats stats)))
	(format stream
		"~%Dyn consing  ~a (~3:D GC's),  Eph consing ~a (~6:D GC's)."
		(num-bytes-to-string (dynamic-consing-of-time-stats stats))
		(dynamic-gc-of-time-stats stats)
		(num-bytes-to-string (ephemeral-consing-of-time-stats stats))
		(ephemeral-gc-of-time-stats stats))
	(format stream " PFaults ~a." (page-faults-of-time-stats stats)))

      (progn
	(format stream "~%  *****~%  *****  ~a ~%  *****~%" descriptor)
	(format stream
		"~%  CPU time               ~a,                Elapsed time         ~a."
		(microseconds-to-string (cpu-of-time-stats stats))
		(microseconds-to-string (elapsed-of-time-stats stats)))
	(format stream
		"~%  Ephemeral bytes consed ~a (~6:D GC's),  Dynamic bytes consed ~a (~3:D GC's)"
		(num-bytes-to-string (ephemeral-consing-of-time-stats stats))
		(ephemeral-gc-of-time-stats stats)
		(num-bytes-to-string (dynamic-consing-of-time-stats stats))
		(dynamic-gc-of-time-stats stats))
	(format stream "~%  Page Faults  ~a.~%~%" (page-faults-of-time-stats stats)))))


(defmacro with-report-time (descriptor &body body)
  `(with-time #'(lambda (stats) (report-time-stats t stats ,descriptor)) ,@body))


;;;;	RLE TODO : need better name than my-time and implement as close
;;;;	RLE TODO : as possible in other lisp implementations.
#+lucid
(defmacro my-time ((descriptor &optional hook) &body body)
  (let ((result (gensym)))
    `(let ((,result nil))
      (mlet* (((etime cputime usertime systime pfaults diskio netio dyncons dyngc ephcons ephgc)
	       (time1 (setf ,result (progn ,@body)))
	       (declare (ignore diskio netio systime usertime))))

       (when ,hook
	 (funcall ,hook cputime ephcons))

       (unless (eql ,descriptor 'short)
	 (format t "~%  *****~%  *****  ~a ~%  *****~%" ,descriptor))

       (if (eql ,descriptor 'short)
	   (progn
	     (format t
		     "~%Elapsed  ~6F seconds. CPU ~6F seconds."
		     etime  cputime)
	     (format t
		     "~%Dyn consing ~a (~3:D GC's),  Eph consing ~a (~6:D GC's)."
		     (num-bytes-to-string dyncons) dyngc (num-bytes-to-string ephcons) ephgc)
	     (format t " PFaults ~a." pfaults)
	     )
	   (progn
	     (format t
		     "~%  Elapsed time ~6F seconds. CPU time ~6F seconds."
		     etime  cputime)

	     (format t
		     "~%  Dynamic bytes consed ~a (~3:D GC's),  Ephemeral bytes consed ~a (~6:D GC's)."
		     (num-bytes-to-string dyncons) dyngc (num-bytes-to-string ephcons) ephgc)

	     (format t "~%  Page Faults ~a." pfaults)))

       (terpri)

       ,result))))
	        
#-lucid
(defmacro my-time ((descriptor &optional hook) &body body)
  (declare (ignore hook))
  `(progn
    (unless (eql ,descriptor 'short)
     (format t "~%  *****~%  *****  ~a ~%  *****~%" ,descriptor)
     (time (progn ,@body)))))



;;;;
;;;;	Flags
;;;;

;;;;	define-flags : initial value is first value enumerated.
;;;;	flag-set (<flags> <name> <value>) : NULL
;;;;	flag-test-p (<flags> <name>)	: <bool> error unless enumerated values were t nil or nil t.
;;;;	flag-test-value-p (<flags> <name> <value>)	: <bool> 
;;;;	flag-value-q (<flags> <name>)	: <value> 

;;;;	define-flags (<type> ((<name> <value> list) list)) : <flags{field}>
;;;;	
;;;;	adding to end shouldn't screw up previous instances.
;;;;
;;;;	maybe integrate with homegrown defstruct then can use :type options on slots for defstruct.
;;;;	
;;;;	(my-defstruct (normal defstruct)
;;;;	  (flags ....)
;;;;	  ;; some syntax for specifying overloaded fields.
;;;;	  ;; maybe some flags to turn on/off checking of consistency of overloaded fields.
;;;;	  (name initial-value &optional type <normal def stuff>))
;;;;	  
;;;;	but for the time being, first pass require convention that such fields be known as flags
;;;;
;;;;	define-flags ((<string{type}> &optional <alias>)
;;;;	              ((<tok{string}> <tok{string}> list) list))
;;;;	  : macro
;;;;	  * default value for alias is type.
;;;;
;;;;	macro which defuns following functions as side effects , should assume that
;;;;	<type> is name of defstruct
;;;;	
;;;;	
;;;;    <alias>-flag-set-<name> (<structure> <value>)	: <structure>
;;;;	<alias>-flag-<name>-p (<structure>)		: <bool>
;;;;	  * error unless enumerated values were t nil or nil t.
;;;;	<alias>-flag-<name>-<item>-p (<structure>)	: <bool> 
;;;;	<alias>-flag-<name>-q (<structure>)	: <item> 
;;;;	
;;;;	init-<alias>-flags (<structure> (<tok{name}> <tok{item}>) list) 	: <structure>
;;;;	  * to specifiy non-default initial values. names not specified should default.
;;;;	
;;;;	should check for duplicate names.


;;;;
;;;;	Stupid implementation defines structure each name has field, no encoding as bits.
;;;;	Quick and easy but can be improved incrementally.
;;;;

;;;;	
;;;;	if uninitialized, first reference should initialize.
;;;;	
;;;;	flags is bit vector.
;;;;	
;;;;	each flag occupies some interval of the bit vector.
;;;;	
;;;;	the number of bits occupied is ceiling log(base2) of number of possible values of flag.
;;;;	
;;;;	
;;;;	<struct>-flag-<flag>-q (<struct>)
;;;;	  - ldb bits of flag.
;;;;	  - use as index into <struct-flag-values>
;;;;	
;;;;
;;;;	Unfortunately some lisps implement bit-vectors as arrays of words, ie a word for each bit.
;;;;	So we will conditionalize so as to allow for an array of fixnums to be a psuedo bit
;;;;	vector. probably best not to have a field cross a fixnum boundary and then
;;;;	can use ldb dpb to access fields. One assumes a real bit vector is superior due to
;;;;	non-allocattion of fixnum will compute updates. However compiler may be able to 
;;;;	avoid allocation as well. If compiler does avoid allocation then might be a win ove
;;;;	true bit vector due to use of ldb dpb.
;;;;	
;;;;	
;;;;	
;;;;	


#+truebitvector
(defun bit-vector-ldb (offset l bits)
  (declare
   (type simple-bit-vector bits)
   (type fixnum l)
   (type fixnum offset))

  (let ((acc 0))
    (dotimes (i l)
      (when (onep (sbit bits (+ i offset)))
	;;(setf -i i -acc acc -l l -offset offset) (break)
	(setf acc
	      (+ acc (ash 1 i)))))
    acc))

#+truebitvector
(defun bit-vector-dpb (offset l bits v)
  (let ((acc v))
    (declare
     (type simple-bit-vector bits)
     (type fixnum l)
     (type fixnum acc)
     (type fixnum offset))

    (dotimes (i l)
      ;; using sbit instead of bit has a negative impact(ie more) on consing???
      (setf (bit bits (+ offset i)) (mod acc 2)) ;;(- l (1+ i))
      (setf acc (ash acc -1)))))

#+truebitvector
(defmacro define-flags ((type &optional alias) flags)
  (let* ((type-s (string type))
	 (alias-s (if alias
		      (string alias)
		      (string type)))
	 (names (mapcar #'car flags))
	 (values (mapcar #'cdr flags))
	 (bool-p (mapcar #'(lambda (values)
			     (or (null values)
				 (and (or (null (car values))
					  (eql t (car values)))
				      (or (null (cdr values))
					  (and (or (and (eql t (car values)) (null (cadr values)))
						   (and (null (car values)) (eql t (cadr values))))
					       (null (cddr values)))))))
			 values))
			    
	 (structure-field-name (intern (string-upcase (concatenate 'string type-s "-flags"))))
	 
	 (offsets-lengths (let ((acc nil))
			    
			    (do ((v values (cdr v))
				 (b bool-p (cdr b))
				 (o 0 (+ o (integer-length (1- (length (car v)))))))
				((null v))
			      ;;(format t " ~a~%" (integer-length (length (car v) )))
			      (push (cons o (integer-length (1- (length (car v))))) acc))
			    (nreverse acc)))
	 )
    

    `(let (,@(mapcan #'(lambda (name values bool-p)
			 (unless bool-p
			   (list `(,name '(,@values)))))
		     names values bool-p)

	   #|
	   ,@(mapcan #'(lambda (name ol bool-p)
			 (unless bool-p
			   (list `(,(intern (string-upcase (concatenate 'string (string name) "-bitspec")))
				   ',ol))))

		     names offsets-lengths bool-p)
	   |#

	   (ol '(,@offsets-lengths));; temp for debug.
	   )


      (defun ,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags-a")))
	  ()
	
	(let ((bits (make-array ,(reduce #'+ (mapcar #'cdr offsets-lengths))
				:element-type 'bit
				:initial-element 0)))
	  ;; only need to set bools since otherwise want zero'th element.
	  ;; bools the order of the elements is ignored.
	  ,@(mapcan #'(lambda (ol b values)
			(when (and b values (eql (car values) t))
			  `((setf (sbit bits ,(car ol)) 1))
			  ))
		    offsets-lengths bool-p values)
	  bits))
      
      ;; default : used to init bits when flags field of structure is nil. returns bits.
      (defun ,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags")))
	  (s)
	
	(let ((bits (make-array ,(reduce #'+ (mapcar #'cdr offsets-lengths))
				:element-type 'bit
				:initial-element 0)))
	  ;; only need to set bools since otherwise want zero'th element.
	  ;; bools the order of the elements is ignored.
	  ,@(mapcan #'(lambda (ol b values)
			(when (and b values (eql (car values) t))
			  `((setf (sbit bits ,(car ol)) 1))
			  ))
		    offsets-lengths bool-p values)
	  (setf (,structure-field-name s)
		 ,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags-a"))))))

      ,@(mapcan #'(lambda (name values bool-p ol) 
		    `(
		      ;; <type>-flag-set-<name>
	 	      (defun ,(intern (string-upcase (concatenate 'string alias-s "-flag-set-" (string name)))) (s v)

			(unless (if ,bool-p
				    (or (null v) (eql t v))
				    (member v ',values))
			  (system-error (list 'flags 'set ',type ',name (princ-to-string v))))

			(let ((bits (or (,structure-field-name s)
					(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
			  (declare (type simple-bit-vector bits))

			  ,(if bool-p
			       `(bit-vector-dpb ,(car ol) 1 bits (if v 1 0))
			       `(bit-vector-dpb ,(car ol) ,(cdr ol) bits (position v ,name)))))


		      ;;<type>-flag-<name>-p
		      ,@(if bool-p
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-p")))
				  (s)
				(let ((bits (or (,structure-field-name s)
						(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
				  (declare (type simple-bit-vector bits))

				  (onep (sbit bits ,(car ol))))))

			    ;;<type>-flag-<name>-q
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-q")))
				  (s)
				(let ((bits (or (,structure-field-name s)
						(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
				  (declare (type simple-bit-vector bits))

				  (nth (bit-vector-ldb ,(car ol) ,(cdr ol) bits) ,name)))

			      
			      ,@(mapcar #'(lambda (item)
					    `(defun ,(intern (string-upcase
							      (concatenate 'string alias-s
									   "-flag-" (string name) "-" (string item) "-p")))
					      (s)
					    
					      (let ((bits (or (,structure-field-name s)
							      (,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
						(declare (type simple-bit-vector bits))

						(= (bit-vector-ldb ,(car ol) ,(cdr ol) bits) ,(position item values)))))
					values)))
		      ))
		names values bool-p offsets-lengths)


      ;; just set the ones mentioned.
      (defun ,(intern (string-upcase (concatenate 'string "set-" alias-s "-flags"))) (s l)
	(dolist (item l)
	  ;; item is cons of name and value.
	  (cond
	    ,@(mapcar #'(lambda (n)
			  `((eql ',n (car item))
			    (,(intern (string-upcase (concatenate 'string alias-s "-flag-set-" (string n)))) s (cdr item))))
		      names)))
	s)

      (defun ,(intern (string-upcase (concatenate 'string "init-" alias-s "-flags")))
	  (s &optional l)
	
	(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s)
	(,(intern (string-upcase (concatenate 'string "set-" alias-s "-flags"))) s l)
	s)
      ))
  )


;; fttb assume we do not need more bits than a single fixnum provides. True at the moment.



#-truebitvector
(defmacro bit-vector-ldb (bitspec bits)
  #-cmu (declare (type fixnum bits))

  `(ldb ,bitspec ,bits))

#-truebitvector
(defmacro bit-vector-dpb (bitspec bits v)
  #-cmu (declare (type fixnum bits)
	         (type fixnum v))

  `(dpb ,v ,bitspec ,bits))


#-truebitvector
(defun clone-flags (f) f)

#+truebitvector
(defun clone-flags (f)
  (declare (ignore f))
  (break "clone truebitvector flags???"))

#-truebitvector
(defmacro define-flags ((type &optional alias class-p) flags)
  (let* ((type-s (string type))
	 (alias-s (if alias
		      (string alias)
		      (string type)))
	 (names (mapcar #'car flags))
	 (values (mapcar #'cdr flags))
	 (bool-p (mapcar #'(lambda (values)
			     (or (null values)
				 (and (or (null (car values))
					  (eql t (car values)))
				      (or (null (cdr values))
					  (and (or (and (eql t (car values)) (null (cadr values)))
						   (and (null (car values)) (eql t (cadr values))))
					       (null (cddr values)))))))
			 values))
			    
	 (structure-field-name (intern (string-upcase (concatenate 'string type-s "-flags"))))

	 (reader (intern (string-upcase  (concatenate 'string "flags-field-of-" type-s ))))
	 (writer (intern (string-upcase  (concatenate 'string "set-" type-s "-flags-field" ))))

	 (offsets-lengths (let ((acc nil))
			    
			    (do ((v values (cdr v))
				 (b bool-p (cdr b))
				 (o 0 (+ o (if (car b) 1 (integer-length (1- (length (car v))))))))
				((null v))
			      ;;(format t "~a ~a ~a~%" (car v) (car b) (integer-length (1- (length (car v)))))
			      (push (cons o (if (car b) 1 (integer-length (1- (length (car v)))))) acc))
			    (nreverse acc)))
	 )
    

    `(let (,@(mapcan #'(lambda (name values bool-p)
			 (unless bool-p
			   (list `(,name '(,@values)))))
		     names values bool-p)

	   ,@(mapcan #'(lambda (name ol b)
			 (unless b ;;if b
			   (list   ;;`(,(intern (string-upcase (concatenate 'string (string name) "-bitmask")))
			           ;;,(ash 1 (car ol)))
				   `(,(intern (string-upcase (concatenate 'string (string name) "-bitspec")))
				     (byte ,(cdr ol) ,(car ol))))
			       
			       ))
		     names offsets-lengths bool-p)
	   
	   ;;(ol '(,@offsets-lengths));; temp for debug.
	   )

      (defun ,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags-a")))
	  ()

	(let ((bits 0))
	  ;; only need to set bools since otherwise want zero'th element.
	  ;; bools the order of the elements is ignored.
	  ,@(mapcan #'(lambda (b values ol)
			(when (and b values (eql (car values) t))
			  `((setf bits
			     (logior ,(ash 1 (car ol)) bits)
			     )))
			  )
		    bool-p values offsets-lengths)
	  bits))
      
      ;; default : used to init bits when flags field of structure is nil. returns bits.
      (defun ,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags")))
	  (s)

	(let ((bits (,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags-a"))))))
	  ,(if class-p
	       `(,writer bits s)
	       `(setf (,structure-field-name s)
		     bits))))

      (defun ,(intern (string-upcase (concatenate 'string "print-" alias-s "-flags"))) (s)
	(declare (ignore s))
	 ,@(mapcar #'(lambda (name bool-p)
		       (declare (ignore name))
		       (if bool-p
			   nil;;`(format t "~a ~a ~%" ',name (,(intern (string-upcase (concatenate 'string alias-s "-flag-" (string name) "-p"))) s))
			   nil;;`(format t "~a ~a ~%" ',name (,(intern (string-upcase (concatenate 'string alias-s "-flag-" (string name) "-q"))) s))
))
		   names bool-p))

      ,@(mapcan #'(lambda (name values bool-p ol) 
		    `(
		      ;; <type>-flag-set-<name>
	 	      (defun ,(intern (string-upcase (concatenate 'string alias-s "-flag-set-" (string name)))) (s v)

			(unless ,(if bool-p
				     `(or (null v) (eql t v))
				     `(member v ',values))
			  (system-error (list 'flags 'set ',type ',name (princ-to-string v))))

			(let ((bits (or ,(if class-p
					     `(,reader s)
					     `(,structure-field-name s))
					(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
			  (declare (type fixnum bits))

			  (setf bits 
				,(if bool-p
				     `(if v
				       (logior ,(ash 1 (car ol)) bits)
				       (logand ,(- #xfffffff (ash 1 (car ol))) bits)
				       )
				     `(bit-vector-dpb
				       ,(intern (string-upcase (concatenate 'string (string name) "-bitspec")))
				       bits
				       (position v ,name))))
			  ,(if class-p
			       `(,writer bits s)
			       `(setf (,structure-field-name s) bits))
			  ))


		      ;;<type>-flag-<name>-p
		      ,@(if bool-p
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-p")))
				  (s)
				(let ((bits (or ,(if class-p
						     `(,reader s)
						     `(,structure-field-name s))
						(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
				  (declare (type fixnum bits))
				  
				  (not (zerop (logand ,(ash 1 (car ol)) bits))))))

			    ;;<type>-flag-<name>-q
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-q")))
				  (s)
				(let ((bits (or ,(if class-p
						     `(,reader s)
						     `(,structure-field-name s))
						(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
				  (declare (type fixnum bits))

				  (nth (bit-vector-ldb ,(intern (string-upcase (concatenate 'string (string name) "-bitspec"))) bits) ,name)))

			      
			      ,@(mapcar #'(lambda (item)
					    `(defun ,(intern (string-upcase
							      (concatenate 'string alias-s
									   "-flag-" (string name) "-" (string item) "-p")))
					      (s)
					    
					      (let ((bits (or ,(if class-p
								   `(,reader s)
								   `(,structure-field-name s))
							      (,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s))))
						(declare (type fixnum bits))

						(= (bit-vector-ldb ,(intern (string-upcase (concatenate 'string (string name) "-bitspec"))) bits) ,(position item values)))))
					values)))
		      ))
		names values bool-p offsets-lengths)


      ;; just set the ones mentioned.
      (defun ,(intern (string-upcase (concatenate 'string "set-" alias-s "-flags"))) (s l)
	(dolist (item l)
	  ;; item is cons of name and value.
	  (cond
	    ,@(mapcar #'(lambda (n)
			  `((eql ',n (car item))
			    (,(intern (string-upcase (concatenate 'string alias-s "-flag-set-" (string n)))) s (cdr item))))
		      names)))
	s)

      
      (defun ,(intern (string-upcase (concatenate 'string "init-" alias-s "-flags")))
	  (s &optional l)
	
	(,(intern (string-upcase (concatenate 'string "init-default-" alias-s "-flags"))) s)
	(,(intern (string-upcase (concatenate 'string "set-" alias-s "-flags"))) s l)
	s)
      ))
  )





#|
(defmacro define-flags ((type &optional alias) flags)
  (let* ((type-s (string type))
	 (alias-s (if alias
		      (string alias)
		      (string type)))
	 (names (mapcar #'car flags))
	 (values (mapcar #'cdr flags))
	 (bool-p (mapcar #'(lambda (values)
			     (or (null values)
				 (and (or (null (car values))
					  (eql t (car values)))
				      (or (null (cdr values))
					  (and (or (and (eql t (car values)) (null (cadr values)))
						   (and (null (car values)) (eql t (cadr values))))
					       (null (cddr values)))))))
			 values))
			    
	 (structure-field-name (intern (string-upcase (concatenate 'string type-s "-flags")))))

    `(progn
      ,@(mapcan #'(lambda (name values bool-p)
		    `(
		      ;; <type>-flag-set-<name>
		      (defun ,(intern (string-upcase (concatenate 'string alias-s "-flag-set-" (string name)))) (s v)
			(mapc #'(lambda (instance)
				  (when (eql (car instance) ',name)
				    (unless (if ,bool-p
						(or (null v) (eql t v))
						(member v ',values))
				      (system-error (list 'flags 'set ',type ',name (princ-to-string v))))
				    (setf (cdr instance) v)))
			      (,structure-field-name s))
			s)

		      ;;<type>-flag-<name>-p
		      ,@(if bool-p
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-p")))
				  (s)
				(exists-p #'(lambda (instance) (when (eql (car instance) ',name)
								 (cdr instance)))
					  (,structure-field-name s))))
			    `((defun ,(intern (string-upcase (concatenate 'string alias-s
									  "-flag-" (string name) "-q")))
				  (s)
				(let ((v (find-first #'(lambda (instance)
							 (when (eql (car instance) ',name)
							   instance))
						     (,structure-field-name s))))
				  (unless v
				    (system-error '(flags query ,type ,name)))
				  (cdr v)))
			      ,@(mapcar #'(lambda (item)
					    `(defun ,(intern (string-upcase
							      (concatenate 'string alias-s
									   "-flag-" (string name) "-" (string item) "-p")))
					      (s)
					      (eql ',item (cdr (find-first #'(lambda (instance)
									       (when (eql (car instance) ',name)
										 instance))
									   (,structure-field-name s))))))
				 values)))))
	 names values bool-p)
      (defun ,(intern (string-upcase (concatenate 'string "init-" alias-s "-flags")))
	  (s &optional l)
	(setf (,structure-field-name s)
	      (mapcar #'(lambda (name values)
			  (cons name
				(let ((new (assoc name l)))
				  (if new
				      (cdr new)
				      (car values)))))
		      ',names ',values))
	s)
      (defun ,(intern (string-upcase (concatenate 'string "set-" alias-s "-flags")))
	  (s l)
	(setf (,structure-field-name s)
	      (mapcar #'(lambda (flag)
			  (cons (car flag)
				(let ((new (assoc (car flag) l)))
				  (if new
				      (cdr new)
				      (cdr flag)))))
		      (,structure-field-name s)))
	s)
      )))
|#


(defun first-n (n l)
   (if (or (zerop n) (null l))
      nil
      (cons (car l) (first-n (1- n) (cdr l)))))


(defun equal-bags-p (e bagaa bagbb)

  (let ((baga (remove-duplicates bagaa :test e))
	(bagb (remove-duplicates bagbb :test e)))
	
    (and (= (length baga) (length bagb))
	 (forall-p #'(lambda (a)
		       (member a bagb :test e))
		   baga))))


(defun permute-list-by-map (eqp map l)
  (let ((m (if map
               (apply #'max map)
               0)))

    (let ((a (make-array m :initial-element nil)))

      (unless (eql (length l) (length map))
	(raise-error (error-message '(permute-list-by-map lengths)
				    (list (length l) (length map)))))

      (mapc #'(lambda (s i)
		(let ((cur (aref a (1- i))))
		  (if cur
		      (unless (or (null eqp) (funcall eqp s cur))
			(raise-error (error-message '(permute-list-by-map converge equal not) i)))
		      (setf (aref a (1- i)) s))))
	    l
	    map)    

      (let ((acc nil))
	(dotimes (i m)
	  (let ((s (aref a i)))
	    (unless s
	      (raise-error (error-message '(permute-list-by-map hole) (cons (1+ i) map))))
	    (push s acc)))

	(nreverse acc)))))

(defun inverse-permute-list-by-map (map l)
  (let ((m (length map))
	(mx (apply #'max map)))

    (let ((a (make-array m :initial-element nil)))

    (unless (eql (length l) mx)
      (raise-error (error-message '(inverse-permute-list-by-map lengths)
				    (list mx (length l)))))

    (dotimeslist (i ii map)
		 (setf (aref a i) (nth (1- ii) l)))

    (let ((acc nil))
      (dotimes (i m)
	(push (aref a i) acc))
      (nreverse acc)))))

