
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;; -docs- (mod edd)
;;;;
;;;;
;;;;	DTREE : a dtree is text/term hybrid. Dtrees are not shared and are
;;;;		modifiable thereby making them more amenable to edit than are
;;;;		terms. 
;;;;	  - all display and edit of terms is done via dtrees.
;;;;	      * display	: term -> dtree -> text
;;;;	      * edit	: term <-> dtree
;;;;	
;;;;	The DTree Manifesto :
;;;;	  Dtree support is geared toward structured objects.
;;;;	  Long displays which are meant to be scrolled are considered
;;;;	  pathological and will be poorly supported, ie expensive and wasteful.
;;;;
;;;;	Dtrees are moderately expensive but reusable and cost is some function
;;;;	of screen(scroll) space.
;;;;
;;;;	Dtrees are not shared. Avoids complications wrt modification and
;;;;	embedded layout parameters.
;;;;
;;;;	Dtree : 
;;;;	 - instantiation
;;;;	 - layout
;;;;	 - write/point
;;;;     - modification
;;;;
;;;;  -page-
;;;;	 
;;;;	Encoding and indirection.
;;;;	  - encoded : Dtree/Edit data is sometimes encoded in the terms.
;;;;	      * Edit Point/Mark is canonical example.
;;;;	      * Local suppression ??
;;;;	      * Given a comprehensible facilty, there may be other applications.
;;;;		Use opquotes to allow direct editing of encoding terms.
;;;;	  - indirection : terms and parameters may be mapped to other terms
;;;;	    for display.
;;;;	      * Library reference is canonical example.
;;;;	      * Other possible uses: complex parameters such as level expressions,
;;;;		and Meta/Slot tags on parameters.
;;;;	      * Given a comprehensible facilty, there will be other applications.
;;;;	Likely to be implemented through dtree attributes and
;;;;	instantiate/uninstantiate hooks.
;;;;
;;;;	Instantiation of dtrees is lazy. Instantiation is also used to update
;;;;	a dtree when the dtree or relevant dforms have been modified.
;;;;
;;;;  -page-
;;;;	 
;;;;	dforms-of-term(<term>) 			: <dforms> list
;;;;
;;;;	create-dtree(<term>
;;;;		     <dfparms>
;;;;		     <dtree{parent}>	
;;;;		     <bool{modifiable}>		
;;;;		     <bool{traverseable}>)	: <dtree>
;;;;
;;;;	instantiate-dtree(<term>
;;;;			  <dfparms>
;;;;			  <dtree>)	: <dtree>
;;;;
;;;;
;;;;	layout-dtree(<dtree> <width> <height> <mode> <bool{allow-wrap}>)
;;;;	  : (values)
;;;;	 * <width>	: INTEGER | nil
;;;;	 * <height>	: INTEGER | nil
;;;;	   Nil means infinite.
;;;;	 * <mode>	: 'linear | `break | 'soft | 'multilinear
;;;;
;;;;	write-dtree(<dtree>
;;;;		    INTEGER{width}
;;;;		    INTEGER{height}
;;;;		    INTEGER{start-row}
;;;;		    INTEGER{start-col}
;;;;		    <bool{wrap not truncate}>
;;;;		    &options <bool{result-p}>
;;;;			     <closure{enter}>
;;;;			     <closure{exit}>
;;;;			     <closure{character}>
;;;;			     <closure{eol}>		: <string> list
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Instantiation : term-to-dtree.
;;;;
;;;;	Considerations:
;;;;	  - choice of dforms
;;;;	      * implicit
;;;;	      * float
;;;;	      * iteration
;;;;	      * stale dform groups
;;;;	  - lazy
;;;;	  - encoding
;;;;	  - indirection
;;;;
;;;;	Partial dtree data structure:
;;;;
;;;;	<dtree>		: dtree[...
;;;;				<term>
;;;;				<dtree{parent}>
;;;;				<dform> | nil
;;;;				<dfparms>
;;;;				<tag> sexpr {update}
;;;;				<tag> sexpr {attributes}
;;;;				<stamp{dform group}>
;;;;				<dtree> list {children}
;;;;				<dtree> list {permuted children}
;;;;				...
;;;;				]
;;;;			| dtreeleaf[...
;;;;					<token{parameter-type}>
;;;;					<parameter-value>
;;;;					<dtree{parent}>
;;;;				    ...
;;;;				    ]
;;;;			
;;;;
;;;;	<dfparms>	: dfparms[<variable> list {implicit}
;;;;			      	  <tag{cursor}>
;;;;	 			  <term{float}>
;;;;				 ]	
;;;;
;;;;	** Notes:
;;;;	  - the <dfparms> contains data needed to chose dform. It is
;;;;	    stored in dtree to facilatate lazy instantiation and update.
;;;;	  - a null dform field indicates that dtree has not been instantiated.
;;;;	  - dtree children are ordered relative to dform model whereas permuted
;;;;	    dtree children are ordered relative to dform display formats.
;;;;	  - the update tag field is used by modification functions to direct
;;;;	    an update instantiation.
;;;;	  - permuted children list should be identical to children if there is
;;;;	    no permutation. dform can pre-compute if there is a permutation.
;;;;	  - could have a dform but no dfparms after dtree splice.
;;;;	      * maybe even a dtree with no dforms or dfparms after splice as
;;;;		method of inheriting dtree descendants.
;;;;	  - the attributes tag will such tags as:
;;;;	      * traversable
;;;;	      * modifiable 
;;;;
;;;;  -page-
;;;;
;;;;	A display meta variable may occur multiple times in formats of dform.
;;;;	As dtrees contain embedded layout the same dtree may not be used to
;;;;	represent more than one occurence of the variable. One occurence will
;;;;	be chosen to be modifiable, then others will be indirect displays of
;;;;	the modifiable.
;;;;
;;;;	A term may exist as a constant in the formats, its dtree will be
;;;;	traversable but not modifiable. 
;;;;
;;;;
;;;	???
;;;;	The instantiate-dtree function will only instantiate the supplied
;;;;	dtree.  It is left to the callers of the dependent functions to insure
;;;;	that the dtree is instantiated before such functions are called.  Such
;;;;	dependent functions will include but are not limited to :
;;;;	  - children-of-dtree
;;;;	  - permuted-children-of-dtree
;;;;
;;;;	To facilitate choosing of dforms, each term will be lazily tagged with
;;;;	its dform group, ie the group of dforms which may be used to
;;;;	display terms with the same term-sig. The group will be destructively
;;;;	modified when updated, ie once tagged the term always points to the
;;;;	current group. Each dform will have a field indicating whether a dform
;;;;	is stale, ie there may be a better choice now. Also, each dform will have
;;;;	a flag indicating if the dform has been removed from the group.
;;;;
;;;;    When instantiating dtree, if new dfparms for dtree children differ from
;;;;    old then dtree state is tagged.
;;;;
;;;;	DtreeLeaf :
;;;;	  - descriptor need not be stored as it can be accessed via parent.
;;;;	  - modification replaces string or text format, ie no istrings used.
;;;; 	  - investigate possiblity of using indirect display for complex
;;;;	    parameters.
;;;;
;;;; -doct- (mod edd dform)
;;;;
;;;;	Layout :
;;;;
;;;;	Layout procedes recursively, some state must be passed forward through
;;;;	recursion. Such state will be referenced through global variables.
;;;;	Other state is only passed down through recursion. Such state will be
;;;;	args to the layout procedure.
;;;;
;;;;	The indent stack :
;;;;
;;;;	  The indent stack and the line-width are part of the recursive state.
;;;;	  And as such, the parent bindings are restored which children are
;;;;	  returned from.
;;;;	
;;;;	  Pushes and pops affect indent stack. Children can pop parent pushes.
;;;;	  However as the indent stack is part of the recursive state, child
;;;;	  pushes and pops can not affect callers stack.
;;;;
;;;;	Dforms contain the core data to direct layout.
;;;;
;;;;	Dform contains a list of formats produced when dform is preprocessed.
;;;;	  - preprocessing will convert newlines embedded in text to some 
;;;;	    sequence of text and break formats.
;;;;
;;;;	<dfformat>	: <text>
;;;;			| <space>
;;;;			| <push>
;;;;			| <parent-push>
;;;;			| <pop>	
;;;;			| <child>
;;;;			| <break-control>
;;;;			| <break>
;;;;			| <depth>
;;;;
;;;;	<push>		: push[INTEGER]
;;;;
;;;;	<child>		: child-format[ <bool{depth-decrement}>
;;;;					     { default t unless iterated child }
;;;;					]
;;;;	<depth>		: nodepth | new | min | max | + | -
;;;;			set constant  
;;;;			| set constant min
;;;;			| set constant max
;;;;			| set relative +/-
;;;;			| set nodepth
;;;;
;;;;	set-depth (<depth{d}> current)
;;;;
;;;;	if d.nodepth
;;;;	  current <- nil
;;;;	elseif null current
;;;;	  if d.min
;;;;	     current <- d.amt
;;;;	  elseif d.max
;;;;	     current <- d.amt
;;;;	  elseif d.new
;;;;	  else current <- d.amt  	??? +|- 
;;;;	else
;;;;	  if d.min
;;;;	     current <- min current d.amt
;;;;	  elseif d.max
;;;;	     current <- max current d.amt
;;;;	  elseif d.new
;;;;	     current <- d.amt
;;;;	  elseif d.+
;;;;	     current <- current + d.amt
;;;;	  elseif d.-
;;;;	     current <- current - d.amt
;;;;	
;;;;  -page-
;;;;
;;;;
;;;;	Backtracking:
;;;;
;;;;	During layout a failure is detected if the length of a format exceeds
;;;;	the available width.  This triggers backtracking. A search is performed
;;;;	to find all possible breaks and elisions which will gain width at
;;;;	failure point. A heuristic is then applied to produce a plan which
;;;;	identifies breaks and elisions to be used to gain required width. The
;;;;	required width will be determined at failure time and may include a
;;;;	lookahead to find other known unavoidable width consumption.  The
;;;;	layout is then retried with the additional breaks and elision planned
;;;;	by the heuristic.
;;;;	
;;;;	Planning works by marking the relevant formats within the dtree. Layout
;;;;	then recoginizes these marks when called again.
;;;;
;;;;	When backing out of a failure, it is possible that layout could detect
;;;;	that continuing from a certain point is safe as further layout is
;;;;	protected from previous failure by a known break. In this way multiple
;;;;	failures may be detected in a single pass of layout.
;;;;
;;;;
;;;;	Flow of control :
;;;;
;;;;	In nuprl4.1 backtracking was done by visiting the dtree in a
;;;;	continuation style such that all choice points would be on the runtime
;;;;	stack. To allow inheritance, this is no longer possible. So the dtree
;;;;	will be visited recursively, however layout will be called iteratively
;;;;	until it completes without failure.
;;;;
;;;;	During layout some values must still be passed forward. In 4.1, such
;;;;	values were stored in a continuation environment structure to facilitate
;;;;	restart after backtracking.  To avoid having to annotate the tree with
;;;;	these environments such data will be passed as dynamic (dynamic extent
;;;;	within the layout function) variables.  FTTB, these will simply be
;;;;	global variables bound in a top level let.
;;;;	
;;;;  -page-
;;;;
;;;;
;;;;	Identifying winning breaks:
;;;;
;;;;	augmenting break: width at the break is greater than width at point.
;;;;			ie taking break will increase width at point.
;;;;			ie margin is to left of break.
;;;;	break indent : an indent which has the break in its scope.
;;;;	point : a format in the ttree.
;;;; 	point line : the line which the point is on.
;;;;	point break : the actual break immediately prior to the point.
;;;;	precedes : format precedes other format and there are no intervening
;;;;		   actual breaks.
;;;;	
;;;;	Winning breaks : a break is a win when break is augmenting and one of
;;;;			 the following is true:
;;;;	  - break precedes point. (point line break)
;;;;	  - break precedes a point break indent.
;;;;	  - break precedes a planned point break indent.
;;;;	 ** if a point line break is planned then the point break indents not
;;;;	 ** shared with the planned point line break are no longer applicable.
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Identifying winning elisions:
;;;;
;;;;	Consumed width : last - first of ttree.
;;;;	dtree precedes : a dtree precedes a point if any format of ttree
;;;			 precedes point on line.
;;;;
;;;;	Winning elisions : an elision is winning if consumed width is greater
;;;;			   than elision width and :
;;;;	 - dtree contains point.
;;;;	   (gain when elided is consumed width + deficit - elision width.
;;;;	   ie deficit is no longer meaningful)
;;;;	 - dtree precedes point.
;;;;	 - dtree precedes point break indent
;;;;	 - dtree precedes planned point break indent
;;;;
;;;;	Decrementing elision depth can also be assumed to be a win.
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Searching for winning breaks and elisions : 
;;;;
;;;;	Search :
;;;;	  - ADD augmenting elisions preceding point
;;;;	  - ADD augmenting elisions containing point
;;;;	  - ForEach b in (augmenting breaks preceding point)
;;;;	      do SearchIndents b.
;;;;	  - SearchIndents (actual break of point line).
;;;;
;;;;	SearchIndents (break)
;;;;	  - find indents of break.
;;;;	  - ADD augmenting breaks preceding indents.
;;;;	  - ADD augmenting elisions preceding indents.
;;;;
;;;;  -page-
;;;;
;;;;	Heuristics :
;;;;
;;;;	Choice list - In the event of a failure a list of possible breaks and
;;;;	elision which may avert the failure is generated. A heuristic is applied
;;;;	to the list to produce a plan of breaks and elisions to try. There is
;;;;	some interdependence among choices.  For example eliding a dtree makes
;;;;	break choices within dtree moot. Completeness of the interdependencies
;;;;	is not crucial. It may happen that a heuristic believes it has planned
;;;;	enough choices to satisfy the required gain, but that layout will not
;;;;	actually realize that gain. The consequence is simply to backtrack
;;;;	again.
;;;;
;;;;	plan-choice(<choice>)			: <choice> list # INTEGER
;;;;	 * <choice> list returned are the previously planned choices made moot
;;;;	   by plan
;;;;	 * INTEGER returned is gained still required to be planned.
;;;;
;;;;	plan-restart()			: NULL
;;;;	 * unmarks all planned choices.
;;;;
;;;;	choice-moot-p(<choice)		: <bool>
;;;;
;;;;	width-gain-of-choice(<choice>)	: INTEGER
;;;;	 * gain at failure point.
;;;;	height-gain-of-choice(<choice>)	: INTEGER
;;;;	 * for a break choice would expect -1.
;;;;
;;;;	dtree-of-choice(<choice>)	: <dtree>
;;;;
;;;;	moot-of-choice (<choice>)	: <choice> list
;;;;	 * returns those choices which would be made not applicable if choice
;;;;	   were planned.
;;;;
;;;;	type-of-choice(<choice>)	: <dtree>
;;;;	 * one of 'break, 'consistent-break, 'elide-depth or 'elision
;;;;	 * other choices may be added when identified.
;;;;
;;;;	choice-decrement-elide-depth(<choice> INTEGER)	: NULL
;;;;	 * default decr is 1.
;;;;
;;;;	choice-conditions (<choice>)	: <tag> | * list
;;;;	 * It is possible that there exist non-standard consequences of a 
;;;;	   choice. The heuristic can determine such consequences by examining
;;;;	   the result.
;;;;         - Known non-standard consequences:
;;;;		* 'consistent-region-right-shift-failure :
;;;;		  Following conditions are met
;;;;		    - choice is a consistent-break.
;;;;		    - At start of consistent zone margin is to the right of
;;;;		      point.
;;;;		    - a region in zone is layed out such that a portion is to 
;;;;		      left of margin.
;;;;		    - the consistent break will shift region which has portion
;;;;		      to left of margin to the margin and thereby reduce width
;;;;		      such that a region no longer fits.
;;;;
;;;;
;;;;	Other dtree functions heuristics might use:
;;;;
;;;;	depth-of-dtree
;;;;	dtree-ancestor-p 
;;;;
;;;;  -page-
;;;;
;;;;	heuristic (INTEGER{gain}, <choice> list)
;;;;
;;;;	Interdependencies :
;;;;	 - plan of break for an indent, makes other breaks for indent moot.
;;;;	 - if a point line break is planned then the point break indents not
;;;;	   shared with the planned point line break are moot.
;;;;	 - elision makes all breaks contained within elided dtree moot.
;;;;	   If part of consistent break is in elided tree then elision alters
;;;;	   width/height of consistent break choice.
;;;;	 - elision makes all elided dtrees contained within elided dtree moot.
;;;;
;;;;  -page-
;;;;
;;;;	Search Implementation :
;;;;	  - producing the list of choices is not trivial.
;;;;
;;;;	Issues:
;;;;	  - Dynamic vs. Static.
;;;;	      * How much to compute after failure vs how much data to store
;;;;		during layout.
;;;;	  - Reusibility of Static data.
;;;;	  - representation of interdependencies.
;;;;
;;;;	Requirements(-) and Solution hints(*): 
;;;;	  - compute elisions containing point
;;;;	      * trivial - follow dtree up.
;;;;	  - compute elisions preceding point
;;;;	      * compute dtrees preceding point.
;;;;		maintain list of consuming formats to current line,
;;;;		consuming formats should contain pointer to containing dtree.
;;;;	  - compute augmentation for break.
;;;;	      * save augmentation amt in break when layed out.
;;;;		do not inherit dtrees containing augmentations depending on 
;;;;		parents (ie containing parent pops).
;;;;	  - compute augmenting breaks preceding point
;;;;	      * maintain list of breaks on current line
;;;;	  - compute indents containing break
;;;;	      * indent formats should have up pointers
;;;;	      * break format has should have indent pointer
;;;;	  - compute possible breaks preceding indent
;;;;	      * possible breaks have prev pointer.
;;;;	      * indent has prev pointer.
;;;;	  - compute elisions preceding indent
;;;;	      * consuming formats have prev pointer.
;;;;	  - compute choice interdependencies.
;;;;	      * sibling breaks for indent
;;;;	      * breaks within dtree - easy
;;;;	      * dtrees within dtrees - easy.
;;;;	      * need ability to detect which breaks are same zone.
;;;;	      * plan point line makes breaks for non-shared indents moot
;;;;		compute non-shared indents, identify breaks preceding indents.
;;;;	
;;;;	** possible breaks and consuming formats are chained as encountered 
;;;;	** using prev pointer. indent will acquire current chain and then set
;;;;	** current to nil to start new chain.
;;;;	
;;;;	** to facilitate reuse of dtrees, a dummy format will be inserted onto
;;;;	** chain when dtree entered.
;;;;	
;;;;  -page-
;;;;
;;;;	IBE-tree : Indent/Break/Elision tree.
;;;;	
;;;;	<ibe>		: <break>
;;;;			| <indent>
;;;;			| <dtree>
;;;;
;;;;	<chain>		: <break> | <text>
;;;;
;;;;	<break>		: break[<chain>
;;;;				INTEGER{augmentation}
;;;;				<indent-up>
;;;;				<dtree>
;;;;				<tag> sexpr {actual | planned }
;;;;				<zone>
;;;;				]
;;;;
;;;;	<text>		: text[ <chain>
;;;;				<dtree>
;;;;				...]
;;;;
;;;;	<indent-up>	: <indent> | <dtree>
;;;;
;;;;	<indent>	: indent[<indent-up>
;;;;			  	 <chain>
;;;;				 <dtree>
;;;;				 ]
;;;;
;;;;	<zone>		: zone[ <break> list
;;;;				<tag> sexpr {planned}
;;;;				]
;;;;
;;;;  -page-
;;;;
;;;;	Dtree Formats :
;;;;	 - where appropriate, built as extensions to ibe nodes.
;;;;	<format>	: <text>
;;;;			| <space>
;;;;			| <push>
;;;;			| <parent-push>
;;;;			| <pop>	
;;;;			| <break-control>
;;;;			| <break>
;;;;			| <depth>
;;;;			| <child>
;;;;
;;;;	<text>		: text [{ibe}
;;;;				<string>
;;;;				]
;;;;
;;;;	<space>		: space[<bool{true means insert-space}]
;;;;
;;;;	<push>		: indent[{ibe}
;;;;				 INTEGER{amt + or -}
;;;;				]
;;;;
;;;;	<parent-push>	: parent-push[INTEGER{count}]
;;;;
;;;;	<pop>		: pop  [INTEGER{amt + or -}]
;;;;
;;;;	<break-control>	: break-control[<tag{BREAK | SOFT | MULTI | LINEAR}>
;;;;
;;;;	<break>		: break[{ibe}
;;;;				<text>]
;;;;	 ** <text> is text to insert if no break.
;;;;
;;;;	<depth>		: depth[INTEGER{amt}
;;;;				<tag sexpr {MIN | MAX | CONSTANT | RELATIVE}>
;;;;				]
;;;;			| nodepth
;;;;	 ** same as dfformat. eq even.
;;;;	
;;;;	<child>		: dtree[{ibe}
;;;;				<bool{depth decrement}>
;;;;				<format> list
;;;;				]
;;;;			 	 
;;;;  -page-
;;;;
;;;;	Layout
;;;;	 - modifies dtree-formats to build IBE-tree.
;;;;	 - modifies dtree-formats to effect layout.
;;;;	 - need wrap mode which results in all augmenting breaks but no elision
;;;;	   when width exceeded.
;;;;	
;;;;	  * depth control
;;;;	  * break control
;;;;	  * indentation
;;;; 	  * wrap : rather than elide allow lines to exceed width and then
;;;;	    wrap lines when writing.
;;;;	  * truncate : layout in linear mode with height of one. Avoid elision 
;;;;	    by allowing lines to exceed width but truncate when writing.
;;;;	  * ??tabs??
;;;;
;;;;	Layout globals: continuation data, ie data which needs to be passed through each
;;;;	  node of the tree visited.
;;;;
;;;;	  *avail-width*
;;;;	  *elide-depth*		{ t means infinite }
;;;;	  *max-width*
;;;;	  *last-text*
;;;;	  *break-control-stack*	
;;;;	  *line-widths*		{ stack of line widths to check skinny  }
;;;;
;;;;	  *point-line*		{ breaks and consuming text ibes }
;;;;	  *break-chain*		{ break ibe chain }
;;;;	  *indent-stack*	{ stack of indent ibes }
;;;;	
;;;;	RLE ???	did not implement parent-push as could see no purpose to it.
;;;;	RLE ??? it may be necessary to implement ibe *indent-stack*
;;;;	
;;;;	
;;;;  -page-
;;;;
;;;;	Layout (dtree line-width indents depth)
;;;;
;;;;	if depth <= 0
;;;;	   (elide; return)
;;;;
;;;;	save-break-control-top <- peek *break-control-stack*
;;;;	*indent-stack* <- idents
;;;;
;;;;	dtree.line-width <- line-width
;;;;	dtree.avail-width <- *avail-width*
;;;;	dtree.estate <- nil
;;;;
;;;;	visit formats-of-dtree
;;;;	  case type-of-format
;;;;  -page-
;;;;	
;;;;	    text t:
;;;;	      *last-text* <- t
;;;;	      *avail-width* <- *avail-width* - t.length
;;;;	      if *avail-width* < 0
;;;;		 fail avail-width
;;;;  -page-
;;;;	      
;;;;	    space s:
;;;;	      if text-p (peek *last-text*)
;;;;		 if not (space-p (last-char (peek *last-text*)))
;;;;		    s.actual <- true
;;;;		    *last-text* <- s
;;;;  -page-
;;;;	
;;;;	    push p :
;;;;	      line-width <- line-width - p.amt
;;;;	      if line-width < 0
;;;;		 fail absolute-value of line-width
;;;;	      else if line-width > *max-width*
;;;;		 then fail warn? elide dtree.
;;;;	      p.up <- peek *indent-stack*
;;;;	      push p *ident-stack*
;;;;	      p.breaks <- *break-chain*
;;;;	      *break-chain* <- nil
;;;;  -page-
;;;;	      
;;;;	    parent-push p :
;;;;	      do i = 1 to p.count
;;;;		(q <- pop indents
;;;;	         line-width <- line-width - q.amt)
;;;;  -page-
;;;;	      
;;;;	    pop p:
;;;;	      q <- pop *indent-stack*
;;;;	      margin <- margin - q.amt
;;;;	      p.amt <- q.amt
;;;;	      if not q.dtree = dtree
;;;;		 q.dtree.estate <- noestate
;;;;  -page-
;;;;
;;;;	    possible break b :
;;;;	      push b *break-chain*
;;;;	      push b *point-line*
;;;;	      b.augmentation <- (line-width - *avail-width*)
;;;;	      b.zone <- peek *break-control-stack*
;;;;	      push b.text formats-of-dtree
;;;;  -page-
;;;;
;;;;	    actual or planned break b : 
;;;;	      b.indent <- peek *indent-stack*
;;;;	      if skinny(*line-widths*)
;;;;		 fail skinny-width + 1
;;;;		 return
;;;;	      *avail-width* <- line-width
;;;;	      push line-width *line-widths*
;;;;	      *point-line* <- (b . nil)
;;;;	      *last-char* <- #\break
;;;;	      when b.planned 
;;;;	         b.planned <- false
;;;;	         b.actual <- true
;;;;  -page-
;;;;	
;;;;	    push break-control c :
;;;;	      d <- peek *break-control-stack*	 
;;;;	      if c.break
;;;;		if d.break | d.soft
;;;;		   push c *break-control-stack*
;;;;		 elseif d.multi
;;;;		   push d *break-control-stack*
;;;;		 elseif d.linear
;;;;		   fail ??? warn? elide break-zone
;;;;	      elseif c.soft
;;;;		if d.soft | d.break
;;;;		   push c *break-control-stack*
;;;;		else push d *break-control-stack*
;;;;	      elseif c.multi
;;;;		if d.linear
;;;;		   push d *break-control-stack*
;;;;		else push c *break-control-stack*
;;;;  -page-
;;;;
;;;;	    pop break-control :
;;;;	      if null *break-control-stack*
;;;;		 warn break-control error.
;;;;	      pop *break-control-stack*
;;;;  -page-
;;;;
;;;;	    depth d :
;;;;	      set-depth d depth
;;;;  -page-
;;;;
;;;;	    child d :
;;;;		push d *point-line*
;;;;		d.up <- peek *indent-stack*
;;;;		push d *ident-stack*
;;;;	        layout d
;;;;		       line-width
;;;;		       *indent-stack*
;;;;		       (if d.depth-decrement
;;;;			  then depth - 1
;;;;		          else depth)
;;;;  -page-
;;;;
;;;;	{ end of format list }
;;;;	if not save-break-control-top = peek *break-control-stack*
;;;;	   dtree.estate <- noestate.
;;;;
;;;;	** For the sake of clarity, preceding presentation of algorithm does
;;;;	**  not allow for wrapping or truncation.
;;;;
;;;;	** some failures skip heuristics and simply elide certain dtrees to
;;;;	** avoid error conditions.
;;;;	** Eg. break zone in a linear zone or shift margin left past zero.
;;;;
;;;;	
;;;; -doct- (mod edd)
;;;;	
;;;;	Inheritance : inheritance of previous layouts of dtrees is used to save
;;;;		      computational expense after backtracking or modification.
;;;;	  - must be able to recognize that previous layout is suitable.
;;;;	  - must be able to adjust state as though inherited layout had been
;;;;	    executed.
;;;;	      * update IBE tree pointers.
;;;;	      * adjust global counters.
;;;;	
;;;;	<dtree>		: .... 
;;;;			  <depth{at layout}>
;;;;			  INTEGER{line-width at layout}		{fits optimally}
;;;;			  INTEGER{avail-width at layout}	{fits optimally}
;;;;			  <estate>
;;;;			  ]
;;;;
;;;;	** line width, avail-width is saved in estate so that an optimal layout
;;;;	** can easily determine if there was a relevant increase in width since
;;;;	** LAST layout.
;;;;
;;;;	** Instantiation should initialize format list and include formats for 
;;;;	** parens if parenthesization is indicated.
;;;;
;;;;	<estate>	: estate[INTEGER{neg-width}		{ fits }
;;;;				 INTEGER{pos-width}		{ fits }
;;;;				 INTEGER{first-width}		{ fits }
;;;;				 INTEGER{last-width}		{ inherit }
;;;;				 INTEGER list {line-widths}	{ inherit }
;;;;				 <text{last}>			{ inherit }
;;;;				 <format> list {last line}	{ inherit }
;;;;				]
;;;;			| noestate	** if unbalanced or parent pops
;;;;
;;;;	 ** probably best to compute inherit data on demand rather saving it.
;;;;	 ** the avail width must be saved at layout as it can not be recovered
;;;;	 ** after the fact.
;;;;
;;;;	 ** formats in dtree are derived from dform formats. They contain fields
;;;;	 ** for embedded data structures used by layout.
;;;;
;;;; 	In order to test for skinny failure after inheritance the line widths
;;;;	at actual breaks need to be saved. Thus the break format must have the
;;;;	following:
;;;;				INTEGER{line-width}
;;;;
;;;;  -page-
;;;;	
;;;;
;;;;	Child portion of layout algorithm becomes:
;;;;
;;;;	    child d :
;;;;	      if not (compatible-depths-p dtree.depth depth)
;;;;		 | dtree.noestate
;;;;		 | dtree.neg-width > *max-width* - line-width
;;;;		 | dtree.pos-width > *line-width*
;;;;		 | dtree.first-width > *avail-width*
;;;;	      then {ie not inheriting}
;;;;		push d *point-line*
;;;;		d.up <- peek *indent-stack*
;;;;		push d *ident-stack*
;;;;	        layout d
;;;;		       line-width
;;;;		       *indent-stack*
;;;;		       (if d.depth-decrement
;;;;			  then depth - 1
;;;;		          else depth)
;;;;
;;;;	      else {ie inheriting}
;;;;		dtree.up <- *indent-stack*
;;;;		*avail-width* <- *avail-width* - dtree.last-consumed
;;;;		*line-widths* <- append *line-widths* dtree.line-widths
;;;;		%% check skinny!!
;;;;		*last-text* <- dtree.last-text
;;;;		do format in dtree.last-line
;;;;		  push format *point-line*
;;;;	
;;;;	 ** one subtle interaction with inheritance is when a dtree is        
;;;;	 ** inherited where the first format is a space.  The inheritor may   
;;;;	 ** not have the same state as when the estate originally layed       
;;;;	 ** thereby causing a space to be required where previously it was not
;;;;     ** or vice versa.  This needs to be accounted for when checking fit  
;;;;	 ** as well since it may change available width.  This is not         
;;;;	 ** accounted for in the algorithm though it will be in the           
;;;;	 ** implementation.                                                   
;;;; -doce- 



;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	


;;;;	ML will have access to functional portions of dtree structure
;;;;	 ie there will be a subset of the dtree which is not modified.
;;;;	 the subset which is modified will be transitory info like annotations
;;                                                                       l
;;;;	 for layout/dtree-to-term etc.
;;;;	
;;;;	dtree_leaf_p		: bool
;;;;
;;;;	parent_of_dtree		: dtree -> dtree
;;;;	dtree_ancestor_p	: dtree -> dtree -> bool
;;;;	dtree_search_preorder	: dtree -> bool -> (dtree -> bool) -> (dtree -> bool) -> dtree
;;;;	 * true means foward, false reverse
;;;;	 * if result equals input dtree, then no hit found.
;;;;	
;;;;
;;;;
;;;;	Valid when dtree_leaf_p is false:
;;;;
;;;;	term_of_dtree		: dtree -> term
;;;;	children_of_dtree	: dtree -> dtree list
;;;;	
;;;;	dtree_insert_term	: dtree -> term -> dtree
;;;;	dtree_merge_term	: dtree -> term -> dtree
;;;;
;;;;	dtree_insert_dtree	: dtree -> dtree -> dtree
;;;;	dtree_merge_dtree	: dtree -> dtree -> dtree
;;;;	
;;;;	term_to_dtree		: term -> variable list -> dtree
;;;;	
;;;;	Valid when dtree_leaf_p is true:
;;;;
;;;;	parameter_of_dtree_leaf	: dtree -> parameter
;;;;	modify_dtree_leaf	: dtree -> parameter -> dtree



;;;;	dform_of_dtree		: dtree -> object_address # int
;;;;	




;;;;	Persistence of layout required.
;;;;	No sharing of layout allowed.
;;;;	


;;;;	Dforms & Dtrees : 
;;;;
;;;;	Dforms are chosen as the dtree is instantiated top down, the term which
;;;;	a dtree is instantiated from is cached in each dtree node. The term can
;;;;	be reconstructed bottom up by substituting the terms represented by the
;;;;	child trees into the dform's model term.
;;;;	
;;;;	Invariants :
;;;;
;;;;	  - Dtrees are not shared.
;;;;
;;;;	  - The dtree cached term, if available, is lexicographically equivalent
;;;;	    to the dtree's represented term, if avaliable.  At least one will be
;;;;	    available at all times.
;;;;
;;;;	  - Dtrees will not be destructively modified such that they change the
;;;;	    term represented.
;;;;	    * Corollary: The model of a dform of a dtree is a model for the
;;;;	      		 represented term of the dtree.
;;;;	    * Corollary: Up pointers may be modified.
;;;;	
;;;;	  - A dtree can be reused only if it and all of its descendents are not
;;;;	    part of an active dtree.
;;;;	
;;;;	
;;;;	When lazily instantiating a dtree there may be no represented term as
;;;;	there is no dform and children.
;;;;
;;;;	Substantive modification of dtrees is accomplished constructively by
;;;;	substituting a dtree for a subtree. The dtree nodes instantiated by the
;;;;	substitution copy the dform and replace the children as needed, but do
;;;;	not copy the cache term pointers as they no longer correspond to the
;;;;	represented term.  Note that the only nodes instantiated are those on
;;;;	the path from the target of the substitution to the root of the dtree.
;;;;
;;;;	Note that there is no guarantee of the optimal dform.  Substantive and
;;;;	also non-substantive modifications to the dtree, as well as changes to
;;;;	the view and edit states may may result in non-optimal dforms. Methods
;;;;	of forcing optimal dforms at a point in time are described later.
;;;;	
;;;;	One can consider dform applicability at two levels. First, you can use
;;;;	the dform model to build a correct instance of the term (match). Second,
;;;;	the dform could be chosen to display the term(suitable).  Thus the first
;;;;	level of applicability is assured, however the second is not.
;;;;	
;;;;	
;;;;	Dtrees and Modifications :
;;;;
;;;;	Dtrees are modified by replacement of one dtree for another.  Replacing
;;;;	entails rebuilding the dtree nodes on the path from the point of
;;;;	replacement to the root.
;;;;
;;;;	Thus as a dtree is modified there are a succession of dtrees produced
;;;;	one from the previous. Call this an amoeba. The most recent dtree of an
;;;;	amoeba is called an active dtree, the older dtrees are inactive.  An
;;;;	amoeba is a conceptual device, the implementation does not manipulate
;;;;	amoebas directly.
;;;;
;;;;	A subtree of an inactive dtree which shares no descendents with the
;;;;	active dtree could be reused (in another dtree) without violating the
;;;;	sharing invariant.  The dtree at the point of replacement is a candidate
;;;;	for reuse as it is severed cleanly from the tree.
;;;;   
;;;;	Other subtrees of inactive dtrees are not candidates for reuse as they
;;;;	may contain descendents which are still part of the active dtree.
;;;;
;;;;	Replace destructively updates parent pointers.
;;;;	
;;;;	We desire that the active dtree of the ameoba share as much data with
;;;;	previous dtrees of the ameoba.
;;;;	Sharing is allowed within an amoeba, ie an inactive dtree and the active
;;;;	  dtree of the ameoba may share subtrees.
;;;;	Sharing is allowed sequentially among ameobas, ie a subtree cut from a
;;;;	  dtree may be reused in another amoeba after the cut.
;;;;	Sharing is not allowed among active dtrees, ie no simultaneous sharing
;;;;	  of dtrees is allowed.
;;;;	Sharing is not allowed within a dtree. A priori, such sharing would
;;;;	  be simultaneous. 
;;;;	
;;;;	The only use to which a non-reusable inactive dtree might be put is to
;;;;	export the represented term.  The destructive modification invariant
;;;;	guarauntees that the represented term is constant.
;;;;
;;;;	Cached terms are only present in lazy dtrees. Consideration was given to
;;;;	have the cache terms be more persistent. However, they are of limited
;;;;	utility if there is any modification substantive or not. Instead some
;;;;	effort has been expended to instantiate dtrees from dtrees as well as
;;;;	terms. Thus if a dtree has been modified the cached term is not longer
;;;;	applicable and if the dtree has not been modified the dtree can be used
;;;;	to instantiate a new dtree without the overhead of encoding and
;;;;	decoding.
;;;;	
;;;;	Some data in the dtree nodes is carried forward during rebuilding by
;;;;	replacement.  Layout data in particular is carried forward, however it
;;;;	is marked so that it can be detected as stale. It is used, if possible,
;;;;	to choose initial break points for relayout. In this way some layout
;;;;	backtracking may be avoided.  This may be a false economy except in some
;;;;	worst case scenarios. A bad scenario is some skinny tree where a deep
;;;;	leaf is modified. Then most of previous layout is lost.  And it may be
;;;;	some trivial change like changing "goof" to "golf" which could use all
;;;;	the old breaks.  for example a right justified cons tree of words where
;;;;	the rightmost text parameter is changed. Without reusing breaks the
;;;;	whole paragraph would have to be re-formatted.
;;;;
;;;;	As long as status is clear there is no harm in preserving the layout
;;;;	data. The layout could be incrementally improved to capitalize on the
;;;;	preserved data.
;;;;	
;;;;	Instantiation 
;;;;	  - term, instantiate from term, initial or paste.
;;;;	  - refresh : 
;;;;	      * stale : environment change.
;;;;	      * lazy term : extend dtree, assimilate modification.
;;;;	      * lazy dtree : assimilate modification.
;;;;		  - lazy dtrees should be assimilated prior to exiting memo environment.
;;;;
;;;;	A weak form of instantiation is sometimes used in order to avoid
;;;;	changing dforms during edit and to avoid excessive computation
;;;;	instantiating possibly ephemeral (wrt display) dtrees.  Weak
;;;;	instantiation copies the dforms when instantiating from a dtree and also
;;;;	inherits the tags and layout.  When instantiating from terms weak,
;;;;	instantiation sets a global condition which can be tested by dform
;;;;	cond_expr. It is left to the user to define cheaply choosen dforms.
;;;;
;;;;	RLE TODO : define aforementioned condition. Better if dform requires weak condition,
;;;;	RLE TODO : then default dforms can be placed at front of queue but only be chosen
;;;;	RLE TODO : in presence of weak condition ??? Maybe two defaults one in front which can
;;;;	RLE TODO : be suppressed and fires only weakly and one last which cannot be suppressed
;;;;	RLE TODO : and fires unconditionally.




;;;; Need an faq type section:
;;;;	
;;;;	consider, push cut large term, move to another view, paste. three top level evals but would like
;;;;	dtree/layout to be reused. Push should push move dtree into lightweight view which preserves layout
;;;;	

(defconstant *true-ce* (new-token-ce nil 'true))
(defconstant *false-ce* (new-token-ce nil 'false))
(defconstant *slot-ce* (new-token-ce nil 'slot))
(defconstant *tree-ce* (new-token-ce nil 'tree))




;;;;
;;;;	dtree :
;;;;
;;;;	parent,
;;;;	dform,
;;;;	children,
;;;;	;;term, (only point is cache)
;;;;	bits and tags of various sorts,
;;;;	layout status
;;;;	layout
;;;;
;;;;	during replacement children and up pointers modified, term nullified or a bit modified
;;;;	dform and children might (want capability) be modified simultaneously. (nullify layout)
;;;;	dform might be modified. (should nullify layout)
;;;;	

;;;;	
;;;;    layout : important that dtree-replace keep layout as for pres replacing parameter need
;;;;	 option of not making new ptree.
;;;;	
;;;;	


;;;;	
;;;;	Dtree Memos.
;;;;	
;;;;	Tags and labels in memoized dtrees should be ??? if mark is in
;;;;	
;;;;	
;;;;	Memoized dtrees may contain tags and labels reusing of memoized dtree needs
;;;;	to allow for lazy application of a tag and label filters.
;;;;	
;;;;	Memoizing a term should take tag and label filter args, meaning that memoized
;;;;	dtree matches term if filters applied to dtree.
;;;;	
;;;;	Maybe mark dtrees with exported terms and filters used to export.
;;;;	Then have a chance at finding memo if term later imported.
;;;;	
;;;;	RLE TODO : FTTB, forget about memo's, after the fact go back and make
;;;;	RLE TODO : memos work for obvious cases such as reinsertion and paste with cut.
;;;;	


(defvar *dtree-memos*)

(defmacro with-dtree-memos (&body body)
  `(let ((*dtree-memos* nil))
    ,@body))

;;; dtree can memoize self as dtree may be used as psuedo-term.
(defun add-dtree-memo (dtree &optional memo)
  (when (boundp '*dtree-memos*)
    (setf *dtree-memos*
	  (acons (or memo dtree)
		 dtree
		 *dtree-memos*))))

(defun dtree-memo-lookup (item)
  (when (boundp '*dtree-memos*)
    (let ((memo (assoc item *dtree-memos*)))
      (when memo
	(prog1 (cdr memo)
	  (setf (cdr memo) nil))))))



;;;
;;; rle ??? would be good if we could avoid marks on dtree by
;;;	enumerating all possible flags.
;;;


;;;;	Dtree Tags and Labels:
;;;;
;;;;	Most dtrees will not be labeled or tagged. Thus rather than require
;;;;	a field in every dtree, tags will be implemented as marks.
;;;;	 - there will be a flag bit on if dtree marked with tags.
;;;;	     * avoids overhead of looking at marks to test if dtree tagged.
;;;;	
;;;;	
;;;;	
;;;;	Tags structures should not be shared so that they may be destructively
;;;;	updated.
;;;;	  - tags must be copied when dtrees copied.
;;;;	
;;;;	However, they will be implicitly shared by in parameters of terms, as terms 
;;;;	may be shared. Thus :
;;;;	  - tags in parameters may not be destructively updated.
;;;;	  - tags of dtree leaves must not be shared with parameters.
;;;;	      * to allow destructive updates in dtrees.
;;;;	
;;;;	Might be safer to only allow constructive updates to tags structures,
;;;;	but then destructively modify dtree pointers to tags structures.
;;;;	
;;;;	
;;;;	

(defstruct tags
  (tags nil)
  (labels nil))
	   
(defun new-tags (sym label-p)
  (make-tags :tags (unless label-p (list sym))
	     :labels (when label-p (list sym))))

(defun new-tags-and-labels (tags labels)
  (make-tags :tags tags
	     :labels labels))

(defun tags-of-tags (a) (tags-tags a))
(defun labels-of-tags (a) (tags-labels a)) 
(defun null-tags-p (a) (or (null a)
			   (and (null (tags-of-tags a))
				(null (labels-of-tags a)))))

(defun copytags (tags)
  (when tags
   (new-tags-and-labels (tags-of-tags tags)
		       (labels-of-tags tags))))


(defstruct (dtree-spine (:include marks))
  
  (flags nil)
  
  parent
  layout
  )


(define-flags (dtree-spine dtree)
    ((modified t nil)			; ie dform choice stale
     (require-refresh nil t)		; refresh required but not accomplished.
					; twould seem like modified nil and req-refresh t ought to be avoided.
					; ie turning refresh on should no-op if not modified.
					; do we want some force version. Force could be done by exporting term
					; and instantiating from scratch.
     
     (instantiated nil t)		; ie no dform.

     (text-modified nil t)
     (term-modified structure text tag nil)

     (layout-modified structure text tag nil)
     
     (refresh-required nil )
     (layout-required nil)
     (present-required nil)

     (tagged nil)			; tagged or labeled.

     ;; oed :
     (elided nil)
     (non-modifiable nil)
     (irrelevant nil)

     ;; oed leaf :
     (slot nil)
     (empty-instantiation nil)

     ;; kludge
     (refreshing nil)

     (length nil)
     ))


;;;;	Modifiable : not a dtree flag.
;;;;	 
;;;;	Mulitple child formats may refer to same model variable, and thus indirectly the same dtree.
;;;;	We desire to restrict modifiability to a single child format. As the non-modifiable
;;;;	formats refer to the same dtree as the modifiable the bit can not be in the dtree.
;;;;	Thus it is in the formats and in the ptree.
;;;;	
;;;;	Non modifiable flags could be defined for library and constant children, however
;;;;	this is not done for the sake of consistency. Instead they are treated like duplicates.
;;;;	
;;;;	


(defun parent-of-dtree (d) (dtree-spine-parent d))
(defun set-dtree-parent (d p)
  ;;(setf -d d -p p) (break "sdp")
  (setf (dtree-spine-parent d) p))

(defun layout-of-dtree (d) (dtree-spine-layout d))
(defun set-layout-of-dtree (d l) (setf (dtree-spine-layout d) l))

(defun merge-tags (tags-a tags-b)
  (cond
    ((null tags-a) (when tags-b
		     (new-tags-and-labels (tags-of-tags tags-b)
					  (labels-of-tags tags-b))))
    ((null tags-b) (new-tags-and-labels (tags-of-tags tags-a)
					(labels-of-tags tags-a)))
    (t (let ((tags (new-tags-and-labels (tags-of-tags tags-a)
					(labels-of-tags tags-a))))
	 (dolist (tag (tags-of-tags tags-b))
	   (pushnew tag (tags-tags tags)))
	 (dolist (label (labels-of-tags tags-b))
	   (pushnew label (tags-labels tags)))
	 tags))))


(defun tag-dtree (dtree sym label-p)
  ;;(when (eql sym 'point) (format t "~a" -tags)(break "td"))
  (if (dtree-flag-tagged-p dtree)
      (let ((tags (mark-value dtree 'tags)))
	(if label-p
	    (pushnew sym (tags-labels tags))
	    (pushnew sym (tags-tags tags))))
      (progn
	(dtree-flag-set-tagged dtree t)
	(mark dtree 'tags (new-tags sym label-p))))
  dtree)

(defun tag-and-label-dtree (dtree tags labels)
  (if (dtree-flag-tagged-p dtree)
      (let ((tags (mark-value dtree 'tags)))
	(when tags
	  (setf (tags-tags tags) (remove-duplicates (append tags (tags-of-tags tags)))))
	(when labels
	  (setf (tags-labels tags) (remove-duplicates (append labels (labels-of-tags tags))))))
      (progn
	(dtree-flag-set-tagged dtree t)
	(mark dtree 'tags (new-tags-and-labels tags labels))))
  dtree)

;; remove is non-destructive.
(defun untag-dtree (dtree sym label-p)
  (when (dtree-flag-tagged-p dtree)
    (let ((tags (mark-value dtree 'tags)))
      (if label-p
	  (setf (tags-labels tags) (remove sym (labels-of-tags tags)))
	  (setf (tags-tags tags) (remove sym (tags-of-tags tags))))))
  dtree)

      
(defun tags-and-labels-of-dtree (dtree)
  (when (dtree-flag-tagged-p dtree)
    (mark-value dtree 'tags)))

(defun set-dtree-tags-and-labels (dtree tags)
  (if tags
      (if (null-tags-p tags)
	  ;; replace tags with null tags if already tagged.
	  (when (dtree-flag-tagged-p dtree)
	    (mark dtree 'tags tags))

	  ;; actual tags
	  (progn
	    (mark dtree 'tags tags)
	    (dtree-flag-set-tagged dtree t)))

      ;; no tags, remove current.
      (when (dtree-flag-tagged-p dtree)
	(dtree-flag-set-tagged dtree nil)
	(unmark dtree 'tags)))
  dtree)
      

(defun tags-of-dtree (dtree)
  (when (dtree-flag-tagged-p dtree)
    (tags-of-tags (mark-value dtree 'tags))))

(defun labels-of-dtree (dtree)
  (when (dtree-flag-tagged-p dtree)
    (labels-of-tags (mark-value dtree 'tags))))

(defun dtree-labeled-p (label dtree)
  (and (member label (labels-of-dtree dtree)) t))


(defun tags-and-labels-of-parameter-value (v)
  (mark-of-parameter-value v 'tags))

(defun set-parameter-value-tags-and-labels (v tags)
  (if (null-tags-p tags)
      (if (mark-parameter-value-p v)
	  (unmark-parameter-value (copy-parameter-value v) 'tags)
	  v)
      (mark-parameter-value (copy-parameter-value v) 'tags tags)))


(defun tags-and-labels-of-parameter (p)
  (mark-of-parameter-value (value-of-parameter-n p) 'tags))

(defun set-parameter-tags-and-labels (p tags)
  ;;(when (equal '(point) (labels-of-tags tags)) (setf -p p -tags tags) (break "sptal"))
  (let* ((v (value-of-parameter-n p))
	 (newv (if (and (null-tags-p tags) (not (mark-parameter-value-p v)))
		   v
		   (let ((nnewv (copy-parameter-value v)))
		     (if (null-tags-p tags)
			 (unmark-parameter-value nnewv 'tags)
			 (mark-parameter-value nnewv 'tags tags))
		     nnewv))))
    (if (eql v newv)
	p
	(instantiate-parameter newv (type-of-parameter p)))))
	  


(defun labels-of-parameter (p)
  (let ((tags (tags-and-labels-of-parameter p)))
    (when tags
      (labels-of-tags tags))))

(defun tags-of-parameter (p)
  (let ((tags (tags-and-labels-of-parameter p)))
    (when tags
      (tags-of-tags (tags-and-labels-of-parameter p)))))
  
;;;
;;; 	leaf.
;;;

(defstruct (dtree-leaf (:include dtree-spine)
		       (:print-function
			 (lambda (leaf stream depth)
			   (declare (ignore depth))
			   (format stream "DTREE-leaf: ~a"
				   (parameter-to-pretty-string (dtree-leaf-parameter leaf))))))
  parameter

  ;; oed :
  (istr nil)
  (descriptor nil)
  (meta nil)

  ;; assoc list of tags and indices
  (indices nil)

  
  (length nil)
  )


;;;;	
;;;;	OED : should think of istring as a temp edit buffer.
;;;;	
;;;;	  - can modify istring without but not consider it a mod
;;;;	    of the term (in some sense).
;;;;	
;;;;	  - move a label out of dtree-leaf should cause flush
;;;;	    of istring to parameter and be considered a structure mod.
;;;;	  | mark dtree as text-mod and then flush explicitly or 
;;;;	    at next real stucture mod.
;;;;	
;;;;	



(defun indices-of-dtree-leaf (d)

  (when (not (dtree-flag-instantiated-p d))
    (instantiate-dtree-leaf d))
  
  (let ((labels (labels-of-dtree d))
	(indices (dtree-leaf-indices d)))
    
    ;;(when (and (null (dtree-leaf-indices d)) (mark-of-parameter-value (parameter-value (parameter-of-dtree d)) 'edit-indices))
    ;;(setf -d d -p  (parameter-of-dtree d) -pi (mark-of-parameter-value (parameter-of-dtree d) 'edit-indices) -di indices) (break "iodl"))
      
    (if labels
	(append (mapcan #'(lambda (l)
			    (unless (member l indices :key #'car)
			      (list (cons l 0))))
			labels)
		indices)
	indices)))
  

(defun index-of-dtree-leaf (d tag)
  ;;(when (integerp (dtree-leaf-indices d)) (setf (dtree-leaf-indices d) nil)) ; temp
  (when (not (dtree-flag-instantiated-p d))
    (instantiate-dtree-leaf d))
  
  (or (cdr (assoc tag (dtree-leaf-indices d))) 0))

(defun isegment-of-dtree-leaf (d atag btag)
  (let ((i (cdr (assoc atag (dtree-leaf-indices d))))
	(j (cdr (assoc btag (dtree-leaf-indices d)))))

    (subseq (istring-of-dtree-leaf d) (min i j) (max i j))))

(defun instantiate-text-carrier (term string &optional label i)
  (let ((term (instantiate-term (instantiate-operator
				 (id-of-term term)
				 (list (instantiate-parameter-s string
								(type-of-parameter
								 (car (parameters-of-term term))))))
				nil)))
    (when label
      ;;(setf -label label -i i) (break "itc")
      (mark term 'edit-indices (list (cons label i))))
    term))

;; assumes that parameter values amenable to istring.
;; TODO edit indices could be split too.
(defun text-carrier-split (term index label)
  ;;(setf -g term -h index -i label) (break "tcs")
  (let ((istr (istring (value-of-parameter-r (car (parameters-of-term term))))))
    (cons (instantiate-text-carrier term
				  (implode-to-string (subseq istr 0 index))
				  label index)
	  (instantiate-text-carrier term 
				  (implode-to-string (subseq istr index))
				  label 0))))


(defun set-dtree-leaf-indices (d indices)
  (dtree-path-term-modified d 'tag)
  (unless (forall-p #'(lambda (p) (integerp (cdr p)))
		    indices)
    (setf -indices indices) (break "sdli"))
  (setf (dtree-leaf-indices d) indices))

(defun set-dtree-leaf-index (d tag index)
  ;;(setf -d d) (break "sdli")
  (dtree-path-term-modified d 'tag)
  (let ((cell (assoc tag (dtree-leaf-indices d))))
    (if cell
	(setf (cdr cell) index)
	(setf (dtree-leaf-indices d) (cons (cons tag index) (dtree-leaf-indices d)))))
  nil)

(defun parameter-of-dtree (d)
  ;;(when (dtree-flag-text-modified-p d) (break "pod"))
  (dtree-leaf-parameter d))
(defun set-parameter-of-dtree (d l) (setf (dtree-leaf-parameter d) l))
(defun type-id-of-dtree-leaf (d) (type-id-of-parameter (parameter-of-dtree d)))

(defun modify-dtree-leaf-istring (d nistr)

  (when (integerp (dtree-leaf-indices d)) (setf (dtree-leaf-indices d) nil)) ; temp

  (unless (dtree-flag-text-modified-p d)
    (break "mdli")
    (raise-error (error-message '(dtree modify leaf) (implode-to-string nistr))))
  
  ;;(setf -a (dtree-leaf-istr d) -b nistr) (break "mdlib")
  (setf (dtree-leaf-istr d) nistr)
  (if (null nistr)
      (dtree-flag-set-slot d t)
      (dtree-flag-set-slot d nil))
  
  
  (let ((l (length nistr)))
    (setf (dtree-leaf-indices d) (mapcar #'(lambda (label-index)
					     (if (> (cdr label-index) l)
						 (cons (car label-index)
						       l)
						 label-index))
					 (dtree-leaf-indices d)))))

(defun new-dtree-leaf (parameter parent)
  ;;(setf a (type-upcase-id-of-parameter parameter)) (break "ndl")
  (tag-dtree 
   (set-dtree-tags-and-labels
    (init-dtree-flags (make-dtree-leaf :parameter parameter
				       :parent parent)
		      '((modified . nil)
			(instantiated . nil)
			(term-modified . nil)))
    (copytags (tags-and-labels-of-parameter parameter)))
   (type-upcase-id-of-parameter parameter)
   nil))


(defun new-modified-dtree-leaf (dtree parameter &optional type)
  (when *dtree-flag-trace*
	    (format t "new-modified-dtree-leaf layout-modified -> structure~%"))
  (set-dtree-tags-and-labels
   (init-dtree-flags (make-dtree-leaf :parameter parameter
				      :layout (layout-of-dtree dtree)
				      :parent (parent-of-dtree dtree))
		     (if type
			 (list* (cons 'layout-modified  type)
				(cons 'term-modified  type)
				'((modified . t)
				  (instantiated . t)))
			 '((layout-modified . structure)
			   (term-modified . structure)
			   (modified . t)
			   (instantiated . t))))
   (copytags (tags-and-labels-of-parameter parameter))))



(defstruct (dtree (:include dtree-spine)
		  (:print-function
		   (lambda (dtree stream depth)
		     (declare (ignore depth))
		     (format stream "DTREE[~a]: ~a, ~a"
			     (term-sig-of-term (dtree-term dtree))
			     (dtree-dfparms dtree)
			     (dtree-flags dtree)))))

  term		

  dfparms

  (dform nil)
  (dform-pool nil)
  (children (make-array 0) :type vector)

  ;;oed
  minw
  )


(eval-when (compile)
  (proclaim '(function children-of-dtree (dtree) (vector dtree-spine)))
  )

;; if extension is nil children may be parameters, terms, or dtrees.

;;; lazy-p -> one of lazy-dtree or lazy-term has value.

(defun dfparms-of-dtree (d) (dtree-dfparms d))
(defun set-dtree-dfparms (d p) (setf (dtree-dfparms d) p))

(defun term-of-dtree (d)
  (dtree-term d))



(defun dform-of-dtree (d) (dtree-dform d))
(defun dform-pool-of-dtree (d) (dtree-dform-pool d))
(defun children-of-dtree (d) (dtree-children d))

(defun model-of-dtree (d)
  (let ((dform (dform-of-dtree d)))
    (when (null dform)
      (raise-error (error-message '(dtree model dform not))))
    (model-of-dform dform)))

(defun model-term-of-dtree (d)
  (term-of-dform-model (model-of-dtree d)))

(defun set-dtree-dform (d dform pool)
  (setf (dtree-dform d) dform)
  (setf (dtree-dform-pool d) (refresh-definition-pool pool))
  )

(defun mapc-vector (f v)
  (dotimes (i (array-dimension v 0))
    (funcall f (aref v i))))

(defun set-dtree-children (d children &optional set-parent)
  ;;(break "sdc")
  (mapc-vector #'(lambda (c)
		   (if set-parent
		       (set-dtree-parent c d)
		       (unless (eq d c) (break "sdc"))))
	       children)

  (setf (dtree-children d) children)
  (values)
  )
  


(defun new-dtree-lazy-term (term dfparms parent)
  (when (null dfparms) (break "ndlt dfparms"))
  ;;(format t ",")
  (mlet* (((term labels tags) (decode-dtree-tags term)))

	 (let ((dtree (or (dtree-memo-lookup term)
			  (init-dtree-flags (make-dtree :term term
							:dfparms dfparms
							:parent parent
							)
					    '((modified . nil)
					      (instantiated . nil)
					      (layout-modified . nil) ; not instantiated supersedes.
					      (term-modified . nil)
					      )))))


	   (if (or labels tags)
	       (tag-and-label-dtree dtree tags labels)
	       dtree))))
	      
(defun new-dtree (term implicit)
  (new-dtree-lazy-term term
		       (make-dfparms :implicit implicit)
		       nil))


;; called when refreshing dtree in place. Attempting to reuse subtrees.
;; note that it allows modified structure.
(defun new-dtree-refresh (dtree dfparms parent)
  ;;(format t ",")
  (setf -dtree dtree) ;;(break "ndr")
  (when (or (null dfparms)) (break "ndr"))
  ;;(when (eql *ilabel* (id-of-term  (term-of-dtree dtree))) (break "ndr"))
  ;;(format t "~a~%" (id-of-term  (term-of-dtree dtree)))
  (let ((ndtree (set-dtree-tags-and-labels
		 (init-dtree-flags (make-dtree :term (term-of-dtree dtree)
					       :dfparms dfparms
					       :parent parent
					       :dform (dform-of-dtree dtree)
					       :dform-pool (dform-pool-of-dtree dtree)
					       :layout (layout-of-dtree dtree))
				   (list '(modified . t)
					 (cons 'instantiated (dtree-flag-instantiated-p dtree))
					 (cons 'layout-modified (dtree-flag-layout-modified-q dtree))
					 (cons 'text-modified (dtree-flag-text-modified-p dtree))
					 (cons 'term-modified (dtree-flag-term-modified-q dtree))))

		 (copytags (tags-and-labels-of-dtree dtree)))))
    (set-dtree-children ndtree (children-of-dtree dtree) t)
    ;;(setf -ndtree ndtree) (break "ndr")
    ndtree))



;; caller updates children/parent; caller is peforming structural mod.
(defun new-modified-dtree (dtree &optional type)

  ;;(format t "~a~%" (id-of-term  (term-of-dtree dtree)))
  ;;(when (eql *ilabel* (id-of-term  (term-of-dtree dtree)))
  ;;(break "nmd")
  ;;)
  ;;(format t ",")
  (dtree-flag-trace
    (format-string "new-modified-dtree layout-modified -> structure~%"))
  (set-dtree-tags-and-labels
   (init-dtree-flags (make-dtree :term (term-of-dtree dtree)
				 :dform (dform-of-dtree dtree)
				 :dfparms (dfparms-of-dtree dtree)
				 :dform-pool (dform-pool-of-dtree dtree)
				 :layout (layout-of-dtree dtree))
		     (if type
			 (list* (cons 'layout-modified  type)
				(cons 'term-modified  type)
				'((modified . t)
				  (instantiated . t)))
			 '((layout-modified . structure)
			   (term-modified . structure)
			   (modified . t)
			   (instantiated . t))))
   (copytags (tags-and-labels-of-dtree dtree))))


;; caller is building dtree from ptree
(defun new-dform-dtree (dform children tags)
  ;;(format t ",")
  (when (equal tags '(nil)) (setf a tags) (break "nt"))
  (let ((d (init-dtree-flags (make-dtree :dform dform
					 :term (model-term-of-dform dform))
			     '((modified . t)
			       (instantiated . t)
			       (layout-modified . structure) ; ?
			       (term-modified . structure)))))
    (set-dtree-children d children t)
    (if tags
	(tag-and-label-dtree d tags nil)
	d)))



(defun dtree-at-address (dtree address)
  ;;(setf -ddtree dtree -address address) (break "daa")
  (labels
      ((visit (dtree addr)
	 (cond
	   ((null addr)
	    dtree)
	   ((dtree-leaf-p dtree)
	    (break "daa")
	    (raise-error (error-message '(dtree address leaf) address "left:" addr)))
	   (t (let* ((instantiated-p (dtree-flag-instantiated-p dtree))
		     (children (children-of-dtree-c dtree))
		     (l (length children))
		     (i (if instantiated-p
			    (car addr)
			    ;; children-of-dform-c caused instantiation.
			    ;; thus addr is term addr need to find dtree addr.
			    (let ((i 1))
			      (some #'(lambda (mv)
					(if (model-variable-of-part-index-p mv (car addr))
					    i
					    (progn (incf i) nil)))
				    (model-variables-of-dform (dform-of-dtree dtree)))))))
		(if instantiated-p
		    (cond
		      ((zerop i)
		       ;; indices start at 1, or -1.
		       (raise-error (error-message '(dtree address index) "address:" address "left:" addr)))
		      ((< i 0)
		       (let ((j (+ l i 1)))
			 (when (< j 0)
			   (raise-error (error-message '(dtree address out-of-range)
						       l "address:" address "left:" addr)))
			 (visit (aref children j) (cdr addr))))
		      (t;; (> i 0)
		       (when (> i l)
			 (raise-error (error-message '(dtree address out-of-range)
						     l "address:" address "left:" addr)))
		       (visit (aref children (1- i)) (cdr addr))))

		    (let ((i 0))
		      (or (some #'(lambda (mv)
				    (if (model-variable-of-part-index-p mv (car addr))
					(visit (aref children i) (cdr addr))
					(progn (incf i) nil)))
				(model-variables-of-dform (dform-of-dtree dtree)))
			  (progn
			    (message-emit (warn-message '(dtree address hidden)
							"address:" address "left:" addr))
			    dtree))))
		)))))

    (visit dtree address)))
    

;; why : to preserve dform choices across paste.
;; 	 otherwise twould be better to just export term.
;;  cant be lazy: otherwise tags and labels are shared and may be changed.



;;	do we want to allow for a filter for tags and labels??
;; 	user can filter by pasting to lightweight view and then walking dtree.
;;	expensive, could provide filter func diff from walk.
;; 	Still requires two traversals, one to copy and then one to filter.
;;	thus probably worthwhile to allow filter on paste.
;;
;;;;	INVARIANT : tags and labels will not be destructively modified in terms
;;;;	or parameters (including bindings).
;;;;
;; 	RLE NAP : thus in lazy dtree no need to copy term but may need to wrap in a filter.
;;	FTTB : just filter term, not likely to be widespread instantiation by filter.
;; 
;;
;;;; 	RLE TODO : a useful flag to have for a dtree may be a flag indicating if there exists any
;;;;	RLE TODO : descendent with any tag or label. It could be convenient to know that no
;;;;	RLE TODO : descendent has any tag or label. Must be careful tracking bit with lazy terms.
;;;;	RLE TODO : Could then add quick check for descendent cond_expr in test-condition-expression.
;;;;	RLE TODO : Might also be able to subvert some walks.



(defstruct tags-filter
  (flags nil)
  (tag-list nil)
  (label-list nil))

(define-flags (tags-filter)
    ((ephemeral t nil)
     (tag-but t nil)
     (label-but t nil)))


(defun tag-list-of-tags-filter (f) (tags-filter-tag-list f))
(defun label-list-of-tags-filter (f) (tags-filter-label-list f))

;;;;	RLE BEWARE : the but-p bools here are the opposite of doc for ml filter args.
;; ie if but-p true then all but the specified tags are removed.
(defun new-tags-filter (eph-p tag-list tag-but-p label-list label-but-p)
  (init-tags-filter-flags (make-tags-filter :tag-list tag-list
					    :label-list label-list)
			  (list (cons 'ephemeral eph-p)
				(cons 'tag-but tag-but-p)
				(cons 'label-but label-but-p))))


(defvar *filter-all*
  (new-tags-filter t nil t nil t))

;; RLE TODO: unit test.
;; returns t if unchanged otherwise filtered list.
(defun filter-parameter-list (parameters filter labels-p)
  (let ((any-filtered-p nil)
	(tags (if labels-p
		  (label-list-of-tags-filter filter)
		  (tag-list-of-tags-filter filter)))
	(but-p (if labels-p
		   (tags-filter-flag-label-but-p filter)
		   (tags-filter-flag-tag-but-p filter)))
	(eph-p (and (not labels-p) (tags-filter-flag-ephemeral-p filter))))

    (let ((filtered (mapcan #'(lambda (p)
				(let ((filtered-p t))
				  (prog1
				      (unless (and eph-p (tag-parameter-p p))
					(if but-p
					    (when (member (value-of-parameter p) tags)
					      (setf filtered-p nil)
					      (list p))
					    (unless (member (value-of-parameter p) tags)
					      (setf filtered-p nil)
					      (list p))))
				    (when filtered-p (setf any-filtered-p t)))))
			    parameters)))
      (or (not any-filtered-p)
	  filtered))))
    

;; returns modifies tags structure. Does not destructively modify lists in tag structure.

(defun filter-tag-list (l but-p tags eph-p)

  (let ((any-filtered-p nil))
    
    (let ((filtered (mapcan #'(lambda (sym)
				(let ((filter-p t))
				  (prog1
				      (unless (and eph-p (null (symbol-package sym)))
					(if but-p
					    (when (member sym tags)
					      (setf filter-p nil)
					      (list sym))
					    (unless (member sym tags)
					      (setf filter-p nil)
					      (list sym))))
				    (when filter-p (setf any-filtered-p t)))))
			    l)))

    (if any-filtered-p
	filtered
	t))))


(defunml (|make_tag_filter| (ephp tags tbutp labels lbutp))
    (bool -> ((tag list) -> (bool -> ((tok list) -> (bool -> filter)))))

  (new-tags-filter ephp tags (not tbutp) labels (not lbutp)))

(defunml (|filter_term| (filter term))
    (filter -> (term -> term))

    (filter-term-tags term filter))
    

  

;; produces new tags so as to preclude sharing of tags by caller.
(defun filter-tags (tags filter)

  (let* ((label-list (labels-of-tags tags))
	 (tag-list (tags-of-tags tags))
	 (filtered-labels (filter-tag-list label-list
					   (tags-filter-flag-label-but-p filter)
					   (label-list-of-tags-filter filter) 
					   nil))
	 (filtered-tags (filter-tag-list tag-list
					 (tags-filter-flag-tag-but-p filter)
					 (tag-list-of-tags-filter filter)
					 (tags-filter-flag-ephemeral-p filter))))
    
    (new-tags-and-labels (if (eql t filtered-tags)
			     tag-list
			     filtered-tags)
			 (if (eql t filtered-labels)
			     label-list
			     filtered-labels))))




;; work under assumption that hits are rare.
;; recursiviely descends tree.
(defun filter-term-tags (term filter)
  (let ((any-tag-p nil))

    (labels
	((visit-parameter (p)
	   (let ((tags (tags-and-labels-of-parameter p)))
	     ;;(setf -p p) (break "ftt")
	     (if tags
		 (let ((ntags (filter-tags tags filter)))
		   (setf -a ntags -b tags)
		   ;;(unless (eql tags ntags) (break "pt"))
		   (setf any-tag-p t)
		   (if (eql tags ntags)
		       p
		       (set-parameter-tags-and-labels p ntags)))
		 p)))

	 (visit-binding (b)
	   (let ((tags (tags-and-labels-of-parameter-value b)))
	     (if tags
		 (let ((ntags (filter-tags tags filter)))
		   (setf any-tag-p t)
		   (if (eql tags ntags)
		       b
		       (progn 
			 (set-parameter-value-tags-and-labels b ntags)
			 b
			 )))
		 b)))
	     
	 (visit-bound-term (bound-term)
	   (let ((hits nil))
	     (dotimeslist (i b (bindings-of-bound-term bound-term))
			  (let ((nb (visit-binding b)))
			    (unless (eql b nb)
			      (push (cons i nb) hits))))
	     (let* ((term (term-of-bound-term bound-term))
		    (subterm (visit-subterm term)))
	       (if (or hits (not (eql subterm term)))
		   (instantiate-bound-term subterm
					   (let ((hits (nreverse hits))
						 (i 0))
					     (mapcar #'(lambda (b)
							 (prog1
							     (if (and hits (= i (caar hits)))
								 (cdr (pop hits))
								 b)
							   (incf i)))
						     (bindings-of-bound-term bound-term))))
		   bound-term))))

	 (visit-term (term)
	   (let ((phits nil))
	     (dotimeslist (i p (parameters-of-term term))
			  (let ((np (visit-parameter p)))
			    (unless (eql p np)
			      (push (cons i np) phits))))
	     (let ((hits nil))
	       (dotimeslist (i bt (bound-terms-of-term term))
			    (let ((nbt (visit-bound-term bt)))
			      (unless (eql nbt bt)
				(push (cons i nbt) hits))))
	       

	       ;;(setf -a hits -b phits) (break "eh1")

	       (if (or phits hits)
		   (instantiate-term (if phits
					 (instantiate-operator (id-of-term term)
							       (let ((i 0)
								     (hits (nreverse phits)))
								 (mapcar #'(lambda (p)
									     (prog1
										 (if (and hits (= i (caar hits)))
										     (cdr (pop hits))
										     p)
									       (incf i)))
									 (parameters-of-term term))))
					 (operator-of-term term))
				     (if hits
					 (let ((i 0)
					       (hits (nreverse hits)))
					   ;;(setf -a hits) (break "eh")
					   (mapcar #'(lambda (bt)
						       ;;(setf -c i -d hits) (break "eh2")
						       (prog1
							   (if (and hits (= i (caar hits)))
							       (cdr (pop hits))
							       bt)
							 (incf i)))
						   (bound-terms-of-term term)))
					 (bound-terms-of-term term)))
		   term))))

	 (visit-itag-term (term)
	   (setf any-tag-p t)
	   (let ((parameters (filter-parameter-list (parameters-of-term term) filter nil)))
	     (if (eql t parameters)
		 (visit-term term)
		 (if parameters
		     (instantiate-term (instantiate-operator (id-of-term term) parameters)
				       (list (visit-bound-term (car (bound-terms-of-term term)))))
		     (visit-subterm (term-of-itags-term term))))))

	 (visit-ilabel-term (term)
	   (setf any-tag-p t)
	   (let ((parameters (filter-parameter-list (parameters-of-term term) filter t)))
	     (if (eql t parameters)
		 (visit-term term)
		 (if parameters
		     (instantiate-term (instantiate-operator (id-of-term term) parameters)
				       (list (visit-bound-term (car (bound-terms-of-term term)))))
		     (visit-subterm (term-of-itags-term term))))))
	 
	 (visit-subterm (term)
	   (cond
	     ((real-ilabel-term-p term)
	      (visit-ilabel-term term))
	     ((real-itag-term-p term)
	      (visit-itag-term term))
	     (t (visit-term term))))
	 )


      (if (mark-value term 'no-tag)
	  term
	  (let ((nterm (visit-subterm term)))
	    (if any-tag-p
		nterm
		(progn (mark term 'no-tag t) term)))))))

(defun define-edd-ce-primitive-types ()

  (add-primitive-type '|cond_expr|
		      #'(lambda (ce)
			  (declare (ignore ce))
			  "ce"))

  (add-primitive-type '|tag|
		      #'(lambda (tag)
			  (string tag))
		      :member-p #'symbolp
		      :eq-func #'eql)
  
  (add-primitive-type '|filter|
		      #'(lambda (f)
			  (declare (ignore f))
			  "filter")
		      :member-p #'tags-filter-p
		      :eq-func #'eql)
  )

(define-edd-ce-primitive-types)

(defun define-ml-edd-primitive-types ()
  (define-ml-basic-edd-primitive-types)
  (define-edd-ce-primitive-types))
  
(defun map-vector(f v)
  (map 'vector f v))

;;     Modified bit only turned on at top of copied tree.
;;
(defun copy-dtree-r (dtree parent filter)
  (labels ((visit (dtree parent)
	     (let ((tags (tags-and-labels-of-dtree dtree)))
	       (let ((dtree (if (dtree-leaf-p dtree)
				(copy-dtree-leaf dtree)
				;; RLE TODO :  need to set some flag so layout knows it is in a copy.
				(copy-dtree dtree)
				)))

		 ;; copy children
		 (unless (dtree-leaf-p dtree)
		   (if (dtree-flag-instantiated-p dtree)
		       (set-dtree-children dtree
					   (map-vector #'(lambda (c) (visit c dtree))
						       (children-of-dtree dtree)))
		       (progn
			 (setf (dtree-term dtree) (filter-term-tags (term-of-dtree dtree) filter))
			 (set-dtree-parent dtree parent))))

		 (when tags
		   (set-dtree-tags-and-labels dtree
					      ;; relies on filter tags returning new structure to avoid sharing.
					      (filter-tags tags filter)))

		 dtree))))

    (when *dtree-flag-trace*
      (format t "copy-dtree-r modified -> t~%"))
    (dtree-flag-set-modified (visit dtree parent) t)))






;; was t > structure > text > tag > nil.
;; changed to t > structure > tag > text > nil.
#|
(defun dtree-modified-flag-max (a b)
  (case a
    ((t) t)
    (structure (if (eql t b) t 'structure))
    (text (case b 
	    ((t) b)
	    (structure b)
	    (otherwise a)))
    (tag (case b 
	    ((nil) a)
	    (otherwise b)))
    (otherwise b)))
|#

(defun dtree-modified-flag-max (a b)
  (case a
    ((t) t)
    (structure (if (eql t b) t 'structure))
    (tag (case b 
	    ((t) b)
	    (structure b)
	    (otherwise a)))
    (text (case b 
	    ((nil) a)	
	    (otherwise b)))
    (otherwise b)))

(defun dtree-modified-flag-greater-than-p (a b)
  (not (eq b (dtree-modified-flag-max a b))))


(defun dtree-modified-term-update (dtree flag)
  (let* ((dflag (dtree-flag-term-modified-q dtree))
	 (max (dtree-modified-flag-max dflag flag)))

    (when *dtree-flag-trace*
      (format t "dtree-modified-term-update term-modified : ~a -> ~a~%" dflag max))
    (unless (eql max dflag)
      (dtree-flag-set-term-modified dtree max)
      t)))

(defun dtree-modified-layout-update (dtree flag)
  (let* ((dflag (dtree-flag-layout-modified-q dtree))
	 (max (dtree-modified-flag-max dflag flag)))

    (when *dtree-flag-trace*
      (format t "dtree-modified-layout-update layout-modified : ~a -> ~a~%" dflag max ))
    (unless (eql max dflag)

      (dtree-flag-set-layout-modified dtree max)
      t)))

;; flag-update-f : dtree -> flag -> bool {continue}
(defun dtree-path-modified (update-f dtree flag)
  (labels ((visit (dtree)
	     (when dtree
	       (when (funcall update-f dtree flag)
		 (visit (parent-of-dtree dtree))))))
    (visit dtree)))

(defun dtree-path-layout-modified (dtree flag)
  (when *dtree-flag-trace*
    (format t "dtree-path-layout-modified -> ~a~%" flag))
  (dtree-path-modified #'dtree-modified-layout-update
			dtree flag))

(defun dtree-path-term-modified (dtree flag)
  (dtree-path-modified #'dtree-modified-term-update
			dtree flag))

;; nfg for turning off as must check if all children off prior to continuing up
;; not really needed for off as cleaning happens top-down.
(defun dtree-path-text-modified (dtree)

  ;;(setf -d dtree) (break "dptm")
  (when *dtree-flag-trace*
    (format t "dtree-path-text-modified text-modified t~%"))

  (dtree-path-modified #'(lambda (d b)
			   ;;(setf -d d) (break "dptm2")
			   (unless (eql b (dtree-flag-text-modified-p d))
			     (dtree-flag-set-text-modified d b)
			     t))
		       dtree
		       t))


;; sets both layout and term modified fields on path.
;; not called? is it sensible? I was confused about meaning of term-update for
;; a while and this may be from that era.
(defun dtree-path-data-modified (dtree flag)
  (dtree-path-modified #'(lambda (d f)
			     (or (dtree-modified-term-update d f)
				 (dtree-modified-layout-update d f)))
			dtree flag))


(defun dtree-path-layout-required (dtree)
  (dtree-path-modified #'(lambda (d f)
			   (unless (eql (dtree-flag-layout-required-p d) f)
			     (dtree-flag-set-layout-required d f))
			   t)
		       dtree
		       t))


;; could have set-dtree-root-flag which finds root and sets flag.
;; or maybe path is better as probably find flagged ancestor faster
;; than root.

;; visit model term, replacing vars with children.
;; there are no unbound display meta vars.
(defun dtree-model-substitute (model children ok-mods)
  (let ((max-mod nil)
	(indices (preorder-indices-of-dform-model model))
	(i 0))

    (labels
	((visit-parameter (p)
	   (let ((val (value-of-parameter p)))
	     (if (display-meta-variable-id-p val)
		 (mlet* (((p status) (dtree-to-parameter (aref children (aref indices i)) ok-mods)))
			;;(setf -p p)(break "dms")
			(incf i)
			(setf max-mod (dtree-modified-flag-max max-mod status))
			p)
		 p)))

	 (visit-binding (b)
	   (if (display-meta-variable-id-p b)
	       (mlet* (((p status) (dtree-to-parameter (aref children (aref indices i)) ok-mods)))
		      (incf i)
		      (setf max-mod (dtree-modified-flag-max max-mod status))
		      (value-of-parameter p))
	       b))

	 (visit-term (subterm)
	   ;;(setf a model b subterm c children d i)
	   (if (itemplate-term-p subterm)
	       (mlet* (((term status) (dtree-to-term (aref children (aref indices i)) ok-mods)))
		      (incf i)
		      (setf max-mod (dtree-modified-flag-max max-mod status))
		      term)
	       subterm)))

      (let ((term (term-of-dform-model model))
	    (meta-parameter-p (dform-model-flag-meta-parameter-p model))
	    (meta-bound-term-p (dform-model-flag-meta-bound-term-p model)))

	(values (if (or meta-bound-term-p meta-parameter-p)
		    (instantiate-term
		     (if meta-parameter-p
			 (instantiate-operator (id-of-term term)
					       (mapcar #'visit-parameter (parameters-of-term term)))
			 (operator-of-term term))
		     (if meta-bound-term-p
			 (mapcar #'(lambda (bt)
				     (instantiate-bound-term
				      (visit-term (term-of-bound-term bt))
				      (mapcar #'visit-binding
					      (bindings-of-bound-term-n bt))))
				 (bound-terms-of-term term))
			 (bound-terms-of-term term)))
		    term)
		max-mod)))))


(defun require-dtree-tags-p (ok-flag)
  (unless (eql 'tag ok-flag)
    (not (dtree-modified-flag-greater-than-p ok-flag 'tag))))

(defun require-dtree-refresh-p (dtree-status ok-flag)
  (dtree-modified-flag-greater-than-p dtree-status ok-flag))


;; if called with ok-flag 'text returns 
(defun dtree-to-parameter (dtree ok-flag)
  (let* ((dtree-tagged-p (dtree-flag-tagged-p dtree))
	 (do-tags-p (and dtree-tagged-p (require-dtree-tags-p ok-flag)))
	 (layout-modified (and do-tags-p 'tag))
	 )
	
    ;;(and (require-dtree-refresh-p (dtree-flag-term-modified-q dtree) ok-flag)
    ;;(require-dtree-tags-p ok-flag))
 
    ;;(setf -a dtree -b ok-flag -c parameter -d (and (not (or (time-parameter-p parameter) (oid-parameter-p parameter)))))
    ;;(break "dtp")

    (when (or (dtree-flag-text-modified-p dtree) do-tags-p)
      ;; do not need to do this if text-mod and text is ok flag, but shouldn't hurt.
      (setf (dtree-leaf-parameter dtree)
	    (dtree-leaf-to-parameter dtree do-tags-p)
	    ;;(set-parameter-tags-and-labels (parameter-of-dtree dtree) (copytags (tags-and-labels-of-dtree dtree)))
	    )

      ;;(break "dtp")
      (unless (or t (eql ok-flag 'text))
	(when *dtree-flag-trace*
	  (format t "dtree-to-parameter text-modified -> nil~%"))
	(dtree-flag-set-text-modified dtree nil)))

    (when *dtree-flag-trace*
      (format t
	      "dtree-to-parameter layout-modified -> ~a~%dtree-to-parameter term-modified -> ~a~%"
	      ok-flag layout-modified))
    (dtree-flag-set-term-modified dtree ok-flag)
    (dtree-flag-set-layout-modified dtree layout-modified)

    (values (parameter-of-dtree dtree)
	    layout-modified)))


;; look at instantiate rationalize en/de-coding of tags and text if applicable
;; ie make sure exploding of parameter is lazy. might be worth complicating layout
;; by allowing for non-exploded strings.
;; produces parameter including buffered edits. Including tags is optional??
;; dtree -> dtree


(defun dtree-to-lift-parameter (dtree ok-flag)
  (let* ((dtree-tagged-p (dtree-flag-tagged-p dtree))
	 (do-tags-p (and dtree-tagged-p (require-dtree-tags-p ok-flag)))
	 (do-text-mod (require-dtree-refresh-p (dtree-flag-term-modified-q dtree) ok-flag))
	 (parameter (parameter-of-dtree dtree))
	 )
	
    ;;(and (require-dtree-refresh-p (dtree-flag-term-modified-q dtree) ok-flag)
    ;;(require-dtree-tags-p ok-flag))
 
    (setf -a dtree -b ok-flag -c parameter -d (and do-text-mod (not (or (time-parameter-p parameter) (oid-parameter-p parameter))))) (break "dt(")

    (if (and do-text-mod
	     (not (or (time-parameter-p parameter)
		      (oid-parameter-p parameter))))
	(values (setf -ff (dtree-leaf-to-parameter dtree t)) ; does tags to
		(dtree-flag-term-modified-q dtree))
	
	(when do-tags-p

	  (setf (dtree-leaf-parameter dtree)
		(set-parameter-tags-and-labels parameter
					       (copytags (tags-and-labels-of-dtree dtree))))

	  (dtree-flag-trace
	   (format-string "dtree-lift-parameter term-modified -> ~a~%" (if do-text-mod 'text nil)))

	  (dtree-flag-set-term-modified dtree (if do-text-mod 'text nil))

	  (values (parameter-of-dtree dtree)
		  (when (and nil do-tags-p)
		    'tag))))))

;;;;	
;;;;	
;;;;	
;;;;	
;;;;	

;; returns (values <term> <flag{term-modified}>).
(defun dtree-to-term (dtree &optional ok-flag)

  ;;(when (eql '|!message| (id-of-term (dtree-term dtree))) (setf -dtree dtree) (break "dtt"))
  (let ((do-tags-p (and (dtree-flag-tagged-p dtree) (require-dtree-tags-p ok-flag))))

    (when (dtree-flag-text-modified-p dtree)
      (dtree-dlift-text dtree))

    (when (require-dtree-refresh-p (dtree-flag-term-modified-q dtree) ok-flag)

      ;; ie, dtree more modified than ok.
      (mlet* (((term children-max) (if (not (dtree-flag-instantiated-p dtree))
				       (values (term-of-dtree dtree) (dtree-flag-term-modified-q dtree))
				       (dtree-model-substitute (model-of-dtree dtree)
							       (children-of-dtree dtree)
							       ok-flag))))

	       ;; set term and encode tags if required.
	       ;; ??? setting dtree-term to encoded flags seems unlikely to ever be proper.
	       (setf (dtree-term dtree)
		     ;; moved to return value.
		     ;;(if do-tags-p
		     ;;(encode-dtree-tags term
		     ;;(labels-of-dtree dtree)
		     ;;(tags-of-dtree dtree)))
		     term)
	     
	       ;; set mod flag to reflect new term.
	       ;;(setf -term term -d dtree -ok ok-flag -m children-max) (break "dot")
	       ;; if ok and lesser may not be resolved.
	       (when *dtree-flag-trace*
		 (format t "dtree-to-term term-modified -> ~a~%" (dtree-modified-flag-max children-max ok-flag)))
	       (dtree-flag-set-term-modified dtree (dtree-modified-flag-max children-max ok-flag))

	       ;;(when (eql children-max nil) (break "dot"))
	       ;;dont do the following as messes up layout refresh.
	       ;;(dtree-flag-set-layout-modified dtree children-max)
	       ))

    ;;(when (eq dtree -ndtree) (setf -dtree dtree -do-tags-p do-tags-p) (break "dtp")    )
    (values (if do-tags-p
		(encode-dtree-tags (term-of-dtree dtree)
				   (labels-of-dtree dtree)
				   (tags-of-dtree dtree))
		(term-of-dtree dtree))
	    (dtree-flag-term-modified-q dtree))) )
  



;; need to recognize kind of replace, ie structure, variable, text and tag accordingly.

;; assume filtering done by caller.
;; assume any needed copying is done by caller.

;;; utility to substitute one dtree for another.
;;; new inherits nothing from point. point is used only as target for replacement.
(defun dtree-replace (point new)
  (labels ((dreplace (dtree new-child old-child)
	     ;;(setf -drd dtree -drn new-child -dro old-child) (break "dr")
	     (if (null dtree)
		 new-child
		 (dreplace (parent-of-dtree dtree)
			   (let ((new-dtree (new-modified-dtree dtree))) ; structural mod.
			     ;; set children and set children parent pointers.
			     (set-dtree-children new-dtree
						 (map-vector #'(lambda (child)
								 (let ((rchild (if (eql child old-child)
										   new-child
										   (progn (dtree-dlift-text child t)
											  child))))
								   rchild))
							     (children-of-dtree dtree))
						 t)
			     new-dtree)
			   dtree))))
      
    (let ((p (parent-of-dtree point)))

      ;; problemmatical for undo?
      ;; could replace point in orginal with lazy term equivalent to allow for reuse of
      ;; point elsewhere. or leave as is and let undo cleanup handle.
      ;; cant since if then reused how can I tell. cleanup compares all children for to point
      ;; up correctly and any that don't get replaced with lazy
      (set-dtree-parent point nil)
      (add-dtree-memo point)

      (dreplace p (progn (dtree-dlift-text new t) new) point))))






;; history update?
;; coerces pending text edits in dtree-leaf edit(istring) buffers to structure mods.
(defun dtree-leaf-lift-text (dtree)
  ;; layout modified <- tag or text.
  (new-modified-dtree-leaf dtree (dtree-leaf-to-parameter dtree t) 'text))


;; constructively builds modified dtree with term-modified[text]
(defun dtree-lift-text (dtree)
  ;;(break "dlt")
  (labels
      ;; visit returns nil if no refresh done.
      ((visit (dtree)
	 (when (dtree-flag-text-modified-p dtree)
	   (if (dtree-leaf-p dtree)

	       (dtree-leaf-lift-text dtree)

	       (let ((modp nil))
		 (let ((nchildren (map 'vector
				       #'(lambda (child)
					   (let ((rchild (visit child)))
					     (when rchild
					       (setf modp t))
					     (or rchild child)))
				       (children-of-dtree dtree))))
		   
		   (when modp
 		     (let ((ndtree (new-modified-dtree dtree 'text)))
		       (set-dtree-children ndtree nchildren t)
		       ndtree))))))))
			   
       (or (visit dtree) dtree)))


;; destructive converts text-modified -> term-modified[text].
(defun dtree-dlift-text (dtree &optional clear-text-edit-p)

  ;;(setf -d dtree) (break "pddt")
  (when (dtree-flag-text-modified-p dtree)
    (if (dtree-leaf-p dtree)
	(progn
	  ;;(setf -d dtree) (break "ddt")
	
	  ;; assume caller fixes (ie clears) view's label cache.
	  (dtree-flag-trace
	   (format-string "dtree-lift-text !TEXT-EDIT ? ~a~%" (dtree-labeled-p '!TEXT-EDIT dtree)))
	  (when clear-text-edit-p
	    (dtree-flag-trace
	     (format-string "dtree-lift-text clear !TEXT-EDIT~%"))
	    (untag-dtree dtree '!text-edit t))
	
	  (setf (dtree-leaf-parameter dtree) (dtree-leaf-to-parameter dtree t)))
      
	(map-dtree-children #'dtree-dlift-text nil t dtree))


    ;;(setf -d dtree) (break "ddt")
    ;;(format t "dtree-to-parameter text-modified -> nil~%")
    (dtree-flag-trace
     (format-string "dtree-lift-text layout-modified -> text, term-modified -> text~%"))
    (dtree-flag-set-text-modified dtree nil)
    (dtree-flag-set-layout-modified dtree 'text)
    (dtree-flag-set-term-modified dtree 'text)
    
    (values)))
	


;;; assume filtering done by caller.
;;; assume tags-and-labels come from parameter.

(defun dtree-leaf-replace-parameter (leaf parameter)
  (dtree-replace leaf
		 (new-modified-dtree-leaf leaf parameter)))
				
  


;;;;
;;;; Stale dform group ??
;;;;
;;;;	dtree should have group stamp to detect staleness.
;;;;	dtree has pool, stale pool -> stale dform. so when choosing
;;;;	dform remember pool chosen from.
;;;;	should have some global stamp as well then if 
;;;;	global unchanged or group unchanged then not stale.
;;;;	
;;;;	do not do a global if same cost as check group stamp.
;;;;	



(defun dform-pool-of-term (term)
  (let ((dforms (environment-resource 'dforms)))

    ;;(when (or (itext-term-p term) t)
    ;;(setf a term b dforms)
    ;;(break "dpot")
    ;;)

    (let ((pool (mark-value term 'dforms)))
      ;; make sure group came from current table.
      (if (and pool
	       (equal-stamps-p (car pool)
			       (table-stamp-of-definition-table dforms))
	       (not (or (stale-definition-pool-p (cdr pool))
			(itext-term-p term)
			)))
	  (cdr pool)

	  ;; cache
	  
	  ;; (dforms-lookup-pool  (environment-resource 'dforms) -term)
	  (let ((pool (dforms-lookup-pool dforms term)))
	    
	    ;;(setf -pool pool -term term ) (break "dpot")

	    (mark term 'dforms (cons (table-stamp-of-definition-table dforms)
				     pool))
	    pool)))))

(defunml (|dform_pool_of_term| (term))
    (term -> (object_id list))

  (let ((pool (dform-pool-of-term term)))
    (when pool
      (mapcan #'(lambda (dforms)
		  (let ((oid  (oid-of-definition dforms)))
		    (unless (null-oid-p oid)
		      (list oid))))
	      (cddr pool)))))


(defun stale-dtree-dform-p (dtree)
  (if (and nil (dtree-term dtree) (itext-term-p (dtree-term dtree)))
      (progn ;;(break "sddp")
	     t)
      (or (let ((dform (dform-of-dtree dtree)))
	    (or (null dform)
		;;(member 'ChooseDFormEagerly (conditions-of-dform dform))
		))
	  (stale-definition-pool-p (dform-pool-of-dtree dtree)))))
      

;;;;
;;;;	Library indirects:
;;;;
;;;;	!object :
;;;;	  - object address
;;;;	  - kind
;;;;	  - status ??
;;;;	  - view-p : currently being viewed.
;;;;	  - source
;;;;	
;;;;	Refresh : build iobject term
;;;;	  - parameter used as pointer is modified.
;;;;	  - object pointed to is touched.
;;;;
;;;;	** when iobject changed, but no dform change could move tags and labels
;;;;	** from top of old dtree, ie dtree corresponding to !object term.
;;;;	
;;;;	TODO : Incremental improvements
;;;;	 - it would be nice to have greater variety of ways of specifying object.
;;;;	   Eg, abstraction def of some instance.
;;;;	 - ??? allow hook to build object term ???
;;;;	 - need to allow term as pointer to allow for complete object address.
;;;;	

;;;;	RLE TODO : At layout, library indirects should be checked to see if refresh required.
;;;;	RLE TODO : If so, replace dtree corresponding to iobject as modified.
;;;;	RLE TODO : iobject dtree can be destructively replaced during layout as it does not
;;;;	RLE TODO : change the term rep'ed by the dtree.

;;;;	RLE TODO : move preceding doc to more central location.

;; EDD needs to maintain some touch history wrt lib so that it knows
;; when objects are stale.


;; parameter-to-iobject-term (<parameter | term> <term{!object})		: <term{!object}>

;; if first arg is a term then term reps oa.
;; if parameter then make up oa from token of parameter.

;; For sake of compatibility may want to guess if parameter does not resolve to object.
;;  ie it may have been a thm name, so try statement address.
;; return iobject arg if term unchanged.
;; should be prepared for null old-iobject.
(defun parameter-to-iobject-term (lib-parameter old-iobject)
  (declare (ignore lib-parameter old-iobject))
    
  ;; RLE TODO

  ;;fttb: 
  (instantiate-term (instantiate-operator '|iobject|))
  ;; later; preceding should only be used when no object found:
  )


(defvar *iobject-term-hook* nil)

(define-primitive |!object| ((oid . oid)) (term))

(defvar *edd-indirect-hooks* nil)

(defun edd-indirect-add (oid f)
  (setf *edd-indirect-hooks*
	(acons oid f (delete oid *edd-indirect-hooks* :key #'car :test #'equal-oids-p))))


(defunml (|edit_indirect_add| (oid f))
    (object_id -> ((term -> (object_id |#| term)) -> unit))

  (edd-indirect-add oid f)
  (values)
  )

;; term -> oid # term
(define-primitive |!edd_indirect| ((oid . spec)) (arg))

(defun term-to-indirect-term (term old-term)
  (let ((r 
	 (let ((parm (find-if #'oid-parameter-p (parameters-of-term term))))
	   (if (null parm)
	       (fooe)
	       (let ((spec (value-of-parameter-r parm)))
		 (let ((f (cdr (assoc spec *edd-indirect-hooks* :test #'equal-oids-p))))
		   ;;(break "adfas")
		   (if f
		       (let ((oidterm (funmlcall f term)))
			 (setf -oidterm oidterm)
			 (io-echo "N")
			 (let ((oid (car oidterm)))
			   (if (and old-term (iobject-term-p old-term)
				    (equal-oids-p oid (oid-of-iobject-term old-term)))
			       old-term
			       (iobject-term oid (cdr oidterm)))))
		       (fooe)))))))
	) (setf -r r) 
	  r))



;;;;	
;;;;	Test condition expression.
;;;;	
;;;;	
;;;;	


;;;;	
;;;;	Unexpected : predictable, but not expected, eg test parameter-type on a term.
;;;;	
;;;;	Unpredicatable : Eg, if I could say then it wouldn't be upredicatable. Any
;;;;	 error generated by functions called by condition evaluator, but not considered
;;;;	 part of the condition evaluator itself.
;;;;
;;;;	When unexpected conditions encountered a warning will be issued
;;;;	and nil will be returned from point.
;;;;
;;;;	Due to the multitude of messages possible, they can be suppressed
;;;;	prior to issue instead of the normal filtering process. They
;;;;	should only be suppressed for production mode.
;;;;
;;;;	Unpredictable errors will fail to top and be coerced to warnings which can not be suppressed.
;;;;
;;;;

(defvar *cond-expr-warn-p* nil)

;; count incremented for every predictable warning.
;; When warnings suppressed, allows one to check if warnings were issued.

;; Every so often issue a warning if some threshold reached.
(defconstant *cond-expr-warn-threshold* 256)

(defvar *cond-expr-warn-count* 0)

(defun cond-expr-warn (msg)
  (incf *cond-expr-warn-count*)

  (if *cond-expr-warn-p*
      (progn
	(message-emit (warn-message '(condition_expressions) msg))
	nil)
      (when (zerop (mod *cond-expr-warn-count* *cond-expr-warn-threshold*))
	(message-emit (warn-message '(condition_expression threshold) *cond-expr-warn-count*))))
  nil)




(defun dtree-abstraction-match (key dtree)

  ;; term-sig of model matches key.
  (let* ((children (children-of-dtree dtree))
	 (model (model-of-dtree dtree))
	 (indices (parameter-indices-of-dform-model model))
	 (i 0))

    (forall-p #'(lambda (key-parm model-parm)
		  (if (display-meta-parameter-p model-parm)
		      (prog1 (abstraction-match-parameter-p key-parm
							    (parameter-of-dtree
							     (aref children (aref indices i))))
			(incf i))
		      (abstraction-match-parameter-p key-parm model-parm)))
	      (parameters-of-term key)
	      (parameters-of-term (term-of-dform-model model))
	      )))


(defun abstraction-of-dtree (dtree)

  ;; may not be exactly what we want but good enougth fttb.
  (abstraction-of-term (term-of-dtree dtree))

 #|
  (let ((abs (mark-value dtree 'abstraction-definition))
	(abstractions (environment-resource 'abstractions)))
    (or (definition-valid-p abs abstractions)
	(let ((abs (definition-lookup abstractions (model-term-of-dtree dtree)
		     :match-f #'(lambda (term model)
				  (declare (ignore model))
				  (dtree-abstraction-match term dtree)))))
	  (mark dtree 'abstraction-definition abs)
	  abs)))
  |#
 )


(defun test-opid-condition (cond opid)
  ;;(setf -cond cond -opid opid) (break "toc")
  (or (eql cond opid)
      (let ((s (string opid)))
	(if (eql (char s 0) #\!)
	    (string= (string-upcase (subseq s 1)) (string cond))
	    (string= (string-upcase s) (string cond))))))
	


;; treats opid as implicit condition.
(defun test-condition-abs-token (mode condition point)

  ;;(setf -mode mode -condition condition -point point) (break "tcat")

  (if (if (eql mode 'dtree)
	  (dtree-leaf-p point)
	  (parameter-p point))
      
      (cond-expr-warn (error-message '(abstraction leaf) condition))

      (let ((abstraction (if (eql mode 'dtree)
			     (abstraction-of-dtree point)
			     (abstraction-of-term point))))

	(or (test-opid-condition condition
				 (id-of-term (if (eql mode 'dtree)
						 (term-of-dtree point)
						 point)))
	    (and abstraction
		 (member condition (conditions-of-abstraction abstraction))
		 t)))))


(defun test-condition-disp-token (mode condition point extend-p)

  ;;(setf -a mode -b condition -c point -d extend-p) (break "tcdt")
  (cond

    ((eql mode 'term)
     (cond-expr-warn (error-message '(disp term) condition)))

    ((dtree-leaf-p point)
     (cond-expr-warn (error-message '(disp leaf) condition)))

    ((and (not extend-p) (not (dtree-flag-instantiated-p point)))
     (cond-expr-warn (error-message '(disp instantiated not) condition)))

    (t (and (member condition (conditions-of-dform (dform-of-dtree-c point)))
	    t))))


(defun test-condition-tag-token (mode condition point)
  (cond

    ((eql mode 'dtree)
     (member condition (tags-of-dtree point)))

    ((parameter-p point)
     (member condition (tags-of-parameter point)))

    (t (tag-member-of-dtree-tags-term-p point condition))))


(defun test-condition-label-token (mode condition point)
  ;;(setf -pp point) (break "tclt") 
  (cond

    ((eql mode 'dtree)
     (member condition (labels-of-dtree point)))

    ((parameter-p point)
     (member condition (labels-of-parameter point)))

    (t (label-member-of-dtree-tags-term-p point condition))))


(defun test-condition-parameter-type-token (mode condition point)
  ;;(setf -a condition -b point) (break "tcptt")
  (if (not (if (eql mode 'dtree)
	       (dtree-leaf-p point)
	       (parameter-p point)))
      (cond-expr-warn (error-message '(parameter-type leaf not) condition))
      (eql condition (type-id-of-parameter
		      (if (eql mode 'dtree)
			  (parameter-of-dtree point)
			  point)))))


;; following can be tested on term:
;; true false parameter binding display-meta abstraction-meta slot tree instantiated = 
;; top, zoom

(defun dtree-slot-p (dtree)
  (if (dtree-leaf-p dtree)
      (if (dtree-flag-instantiated-p dtree)
	  (dtree-flag-slot-p dtree)
	  (slot-parameter-p (parameter-of-dtree dtree)))
      (iplaceholder-term-p (term-of-dtree dtree))))


(defun test-condition-dtree-token (mode condition point)
  ;;(break "tcdt")
  (case condition

    (true t)
    (false nil)

    (slot (if (eql 'dtree mode)
	      (dtree-slot-p point)
	      (if (parameter-p point)
		  (slot-parameter-p point)
		  (iplaceholder-term-p point))))

    (tree (if (eql 'dtree mode)
	      (not (dtree-leaf-p point))
	      (term-p point)))

    (parameter (if (eql 'dtree mode)
		   (dtree-leaf-p point)
		   (parameter-p point)))

    (tag (if (eql 'dtree mode)
	     nil
	     (real-ilabel-term-p point)))

    (instantiated (and (eql 'dtree mode)
		       (dtree-flag-instantiated-p point)))
    
    ;; RLE TODO lots more todo here.
    (otherwise (cond-expr-warn (error-message '(dtree unknown))))))


;;;;	RLE TODO : add optional arg which controls behaviour when lazy term dtree 
;;;;	RLE TODO : run-int.



;;;;	parent (<point>)		: <point>
;;;;	child (<point> <f> INTEGER)	: <point>
;;;;	  * f(<point>) : <bool>
;;;;	descendent (<point> <f>)	: NULL
;;;;	  * f(<point>) : NULL or throws.
;;;;	ancestor (<point> <f>)	: NULL
;;;;	  * f(<point>) : NULL or throws.
;;;;	dtree (<token <point>) 		: <bool>
;;;;	abs   (<token <point>) 		: <bool>
;;;;	disp  (<token <point>) 		: <bool>
;;;;	tag   (<token <point>) 		: <bool>
;;;;	label (<token <point>) 		: <bool>
;;;;	parameter-type (<token <point>)	: <bool>
;;;;	



;;;; alternate to hooks above:
;;;;	
;;;;
;;;;
;;;;	mode : dtree or term, when instantiate boundary crossed then switch.
;;;;	
;;;;	Invariants: 
;;;;	
;;;;	In term mode, can not go up.
;;;;	Can switch from dtree mode to term mode but not vice versa.
;;;;	In dtree mode, point arg is a dtree.
;;;;	In term mode, point arg is a term or parameter.
;;;;	
;;;;	Implementation: 
;;;;	
;;;;	if dtree mode
;;;;	   dtree hook
;;;;	   term hook (includes parameters)
;;;;	
;;;;	hook : mode X type X arg 
;;;;	  can change mode. child and descendent.
;;;;	
;;;;	Args : expr point
;;;;
;;;;	Type: 
;;;;	  - parent child
;;;;	  - descendent ancestor 
;;;;	  - dtree abs disp tag label
;;;;	  - parameter-type

(defun test-mode ()
  (let ((mode t))
    (declare (special mode))

    (labels
	((foo ()
	   (let ((mode nil))
	     (declare (special mode))
	     (goo)))
	 (goo ()
	   mode
	   ))
      (foo))))

;;;;	RLE TODO : Define error condition that users can use for debug.
;;;;	RLE TODO : ie error will abort cond_expr eval with message. 
;;;;	
;;;;	RLE TODO : Define condition which allow extend-p to be bound.
;;;;	RLE TODO : 


;; when walking need to pass mark as arg for avoid condition-evals.

;; previously this would extend refresh frontier only if extend-p
;;  and extend instantiated frontier unconditionally. Modified
;;  so that extends frontier only if extend-p but flavor of extension
;;  depends of state of dtree.
(defun test-condition-expression (ce mode point 
				     &key extend-p mark (max-depth 100)
				     )

  ;;(setf a ce b point) (break (format-string "tce ~a" ce))
  (let ((mark (or mark point)))

    (labels
	((leaf-p (mode point)
	   (cond ((eql 'dtree mode)
		  (dtree-leaf-p point))
		 ((eql 'term mode)
		  (parameter-p point))
		 (t (system-error (error-message '(condition_expression mode) mode)))))
	 
	 (visit (mode ce point depth)
	   (cond
	     ((> depth max-depth)
	      (format t "condition-expression max-depth hit")
	      nil)
	     ((and-ce-p ce)
	      ;;(setf d ce) (break)
	      (and (setf -landce (visit mode (lexpr-of-and-ce ce) point (1+ depth)))
		   (setf -randce (visit mode (rexpr-of-and-ce ce) point (1+ depth)))))

	     ((or-ce-p ce)
	      (or (visit mode (lexpr-of-or-ce ce) point (1+ depth))
		  (visit mode (rexpr-of-or-ce ce) point (1+ depth))))
	   
	     ((not-ce-p ce)
	      (not (visit mode (expr-of-not-ce ce) point (1+ depth))))
	   
	     ((parent-ce-p ce)
	      (if (eql 'term mode)
		  (cond-expr-warn (error-message '(parent term) point))
		  (let ((p (parent-of-dtree point)))
		    (and p (visit mode (expr-of-parent-ce ce) p (1+ depth))))))

	     ((child-ce-p ce)
	      ;; RLE TODO : some unit testing of this is probably worthwhile.
	      (unless (leaf-p mode point)
		(let* ((mode mode)
		       (point point)
		       (filter (filter-of-child-ce ce))
		       (index (int-of-child-ce ce))
		       (rev-p (< index 0))
		       (incr (if rev-p -1 1))
		       (next nil)
		       (i 0)
		       (f #'(lambda (child)
			      (when (visit mode filter child (1+ depth))
				;;(setf -child child -i i) (break "tcec")
				(incf i incr)
				(when (= i index)
				  (setf next child)
				  t)))))

		  (when (and (eql 'dtree mode)
			     (not (dtree-flag-instantiated-p point))
			     (null extend-p))
		    (setf mode 'term)
		    (setf point (term-of-dtree point)))

		  (if (eql 'dtree mode)
		      (dtree-children-apply point f rev-p)
		      (term-children-apply point f rev-p))
		
		  ;;(setf -ce (expr-of-child-ce ce) -next next) (break "tcecc")
		  (and next
		       (visit mode (expr-of-child-ce ce) next (1+ depth))))))


	     ((descendent-ce-p ce)
	      (let ((expr (expr-of-descendent-ce ce))
		    (found nil))
		
		(labels ((descendent (mode point)
			   ;;(setf -mode mode -point point) (break "tced")
			   (if (visit mode expr point (1+ depth))
			       (setf found t)
			       (unless (leaf-p mode point)
				 (let ((mode mode)
				       (point point))

				   (when (and (eql 'dtree mode)
					      (not (dtree-flag-instantiated-p point))
					      (null extend-p))
				     (setf mode 'term)
				     (setf point (term-of-dtree point)))

				   (if (eql mode 'term)
				       (term-children-apply point
							    #'(lambda (child)
								(descendent mode child)
								found))
				       (dtree-children-apply point
							    #'(lambda (child)
								(descendent mode child)
								found))))))))
		  (descendent mode point))

		found))

	     
	     ((ancestor-ce-p ce)
	      (if (eql 'term mode)
		  (cond-expr-warn (error-message '(ancestor term) point))
		  (let ((expr (expr-of-ancestor-ce ce)))
		    (labels ((ancestor (dtree)
			       (or (visit mode expr dtree (1+ depth))
				   (let ((p (parent-of-dtree dtree)))
				     (when p (ancestor p))))))
		      (ancestor point)))))

	   
	     ;; could have a bind mark which restores mark at exit??
	     ;; ie make mark a global and do (let ((*mark* point))) instead of setf.
	     ((mark-ce-p ce)
	      (setf mark point)
	      (visit mode (expr-of-mark-ce ce) point (1+ depth))
	      )

	     ;; TODO mark-eq-ce-p
	     ;;((mark-eq-ce-p ce)
	     ;;(eql mark point))

	     ((token-ce-p ce)
	      (let ((token (token-of-token-ce ce)))
		(case (modifier-of-token-ce ce)
		  ((nil)		(test-condition-dtree-token mode token point))
		  (abs			(test-condition-abs-token mode token point))
		  (disp			(test-condition-disp-token mode token point extend-p))
		  (tag			(test-condition-tag-token mode token point))
		  (label		(test-condition-label-token mode token point))
		  (parameter-type	(test-condition-parameter-type-token mode token point))
		  (otherwise		(system-error (error-message '(test cond_expr token modifier)
								    (modifier-of-token-ce ce)))))))

	     (t (break "tce"))
	     )))

      (if (null ce)
	  t
	  (with-handle-error (()
			      (message-emit (warn-message '(condition_expression)
							  (flush-message nil)))
			      nil)
	    (visit mode ce point 1))))))


(defun test-condition-expression-summary (ce mode point labels tags)

  (labels
      ((visit (mode ce point)
	 (cond
	   ((and-ce-p ce)
	    (let ((l (visit mode (lexpr-of-and-ce ce) point))
		  (r (visit mode (rexpr-of-and-ce ce) point)))
	      (cond
		((or (null l) (null r))
		 nil)
		((and (eql t l) (eql t r))
		 t)
		(t 'maybe))))

	   ((or-ce-p ce)
	    (let ((l (visit mode (lexpr-of-or-ce ce) point))
		  (r (visit mode (rexpr-of-or-ce ce) point)))
	      (cond
		((or (eql t l) (eql t r))
		 t)
		((and (null l) (null r))
		 nil)
		(t 'maybe))))
	   
	   ((not-ce-p ce)
	    (let ((n (visit mode (expr-of-not-ce ce) point)))
	      (cond
		((null n) t)
		((eql t n) nil)
		(t 'maybe))))
	   
	   ((token-ce-p ce)
	    (let ((token (token-of-token-ce ce)))
	      (case (modifier-of-token-ce ce)
		(tag			(when (member token tags) 'maybe))
		(label		(when (member token labels) 'maybe))
		(otherwise		'maybe))))

	   ;;((parent-ce-p ce)		'maybe)
	   ;;((child-ce-p ce)		'maybe)
	   ;;((descendent-ce-p ce)	'maybe)
	   ;;((ancestor-ce-p ce)	'maybe)
	   (t 'maybe)
	   )))

    (if (null ce)
	t
	(with-handle-error (()
			    (message-emit (warn-message '(condition_expression)
							(flush-message nil)))
			    'maybe)
	  (visit mode ce point)))))


;; if f returns nil then continue otherwise halt

(defun dtree-structure-children-apply (s mode f &optional reverse-p)
  (if (eql 'dtree mode )
      (dtree-children-apply s f reverse-p)
      (term-children-apply s f reverse-p)))


(defun dtree-children-apply (dtree f &optional reverse-p)

  (unless (dtree-leaf-p dtree)
    (let* ((children (children-of-dtree-c dtree))
	   (limit (array-dimension children 0)))
    
      (if reverse-p

	  (do ((i (1- limit) (1- i)))
	      ((or (< i 0)
		   (funcall f (aref children i)))))
	  
	  (do ((i 0 (1+ i)))
	      ((or (= limit i)
		   (funcall f (aref children i))))))))
  nil)

(defun term-child-find (term i)
  (let* ((parameters (parameters-of-term term))
	 (l (length parameters)))
    (if (> l i)
	(nth (1- i) parameters)
	(let ((found nil)
	      (ii (- i l)))
	  (do ((bts (bound-terms-of-term term) (cdr bts)))
	      ((if (null bts)
		   (raise-error (error-message '(term child find out-of-range)
					       i term))
		   (let* ((bindings (bindings-of-bound-term (car bts)))
			  (l (length bindings)))
		     (if (< ii l)
			 (setf found (nth (1- ii) bindings))
			 (if (= (1- ii) l)
			     (setf found (term-of-bound-term (car bts)))
			     (progn (setf ii (- ii l 1)) nil)))))
	       found))))))

(defun term-children-apply (term f &optional reverse-p)
  ;;(setf -term term) (break "tca")  ;;
  (unless (parameter-p term)
    (if reverse-p
	(term-children-reverse-apply term f)
	(let ((halt-p nil))
	  (unless (do ((parameters (parameters-of-term term) (cdr parameters)))
		      ((or (null parameters)
			   halt-p)
		       halt-p)
		    (setf halt-p (funcall f (car parameters))))
      
	    (do ((bound-terms (bound-terms-of-term term) (cdr bound-terms)))
		((or (null bound-terms)
		     halt-p)
		 halt-p)
	      (unless (do ((bindings (bindings-of-bound-term (car bound-terms)) (cdr bindings)))
			  ((or (null bindings)
			       halt-p))
			(setf halt-p (funcall f (instantiate-parameter (car bindings) *variable-type*))))
		(setf halt-p (funcall f (term-of-bound-term (car bound-terms)))))))))

    nil))
    
(defun term-children-reverse-apply (term f)

  (labels

      ((visit-bindings (bindings)
	 (unless (null bindings)
	   (or (visit-bindings (cdr bindings))
	       (funcall f (instantiate-parameter (car bindings) *variable-type*)))))

       (visit-bound-terms (bound-terms)
	 (unless (null bound-terms)
	   (or (visit-bound-terms (cdr bound-terms))
	       (or (funcall f (term-of-bound-term (car bound-terms)))
		   (visit-bindings (bindings-of-bound-term (car bound-terms)))))))

       (visit-parameters (parameters)
	 (unless (null parameters)
	   (or (visit-parameters (cdr parameters))
	       (funcall f (car parameters))))))

    (unless (visit-bound-terms (bound-terms-of-term term))
      (visit-parameters (parameters-of-term term))))

  nil)

  
	 






;;;
;;;	Instantiate
;;; 

;;;;	when choosing dform given a dtree, in case where dform models are
;;;;	structurally similar we can probably avoid building term, but
;;;;	otherwise probably need partial term to do match and test predicates.
;;;;
;;;;	actually some/most predicates might not require term.
;;;;	

;;;;	Some predicates require whole term. ie dummy-check. There is also an alpha-equality
;;;;	check for constant subterms. Do not want to write free-vars and equal-terms
;;;;	for dtree.
;;;;	
;;;;	Constant terms will be small and/or infrequent.
;;;;
;;;;	dummy-checks can be of any size, should not be too frequent, but writing a dtree
;;;;	version of free-vars should be easy. Can leave this as an optimization, ie for
;;;;	for now export term to do dummy check. Once exported, instantiation should proceed
;;;;	from term. Note this is for extending modified tree, as extending lazy terms has 
;;;;	tern and extend lazy dtree does not rechoose dform.


;; actuals -> lazy dtree's in place (ie reuse actuals vector as children vector).
;; actuals are parameters or terms.

;; slots ?? slots should match slots.


(defun iterate-match-parameters-p (m-parameter c-parameter)
  (and (equal-types-p (parameter-type m-parameter)
		      (parameter-type c-parameter))
       (or (display-meta-parameter-p m-parameter)
	   (equal-parameter-values-p (value-of-parameter m-parameter) 
				     (value-of-parameter c-parameter)
				     (parameter-type m-parameter)))))

(defun iterate-match-p (model child)
  ;;(setf -model model -child child) (break "imp")
  (and (let ((m-op (operator-of-term model))
	     (c-op (operator-of-term child)))
	 (and (eql (id-of-operator m-op)
		   (id-of-operator c-op))
	      (apply-predicate-to-list-pair (parameters-of-operator m-op)
					    (parameters-of-operator c-op)
					    #'iterate-match-parameters-p)))
       
       ;; (equal (arities-of-term model) (arities-of-term child))
       (apply-predicate-to-list-pair (bound-terms-of-term model)
				     (bound-terms-of-term child)
				     #'(lambda (mbt cbt)
					 (= (length (bindings-of-bound-term-n mbt))
					    (length (bindings-of-bound-term-n cbt)))))))
		   


;;;;	

;;;;	RLE TODO : Should we eval hidden test on constant.
;;;;	RLE TODO : Constant subterm of model may contain point ??
;;;;	RLE TODO : same true for floats?? but floats are hidden!


;;;;	
;;;;	Want consistency of dform choice across paste.
;;;;	Do Not want a modification in one node to affect any other node.
;;;;	 - exception is same dtree occurring multiple times in permuted children.
;;;;
;;;;	When pasting dtrees copy pasted tree.
;;;;
;;;;	RLE PERF: At some point investigate feasibility of destructive updates.
;;;;	RLE PERF: It seems not allowing sharing should make destructive updates ok.
;;;;
;;;;	
;;;;	All dtrees have term pointer plus flags to indicate status of term wrt dtree.
;;;;	
;;;;	Want to detect stale dforms.
;;;;	Want to detect stale terms wrt export.
;;;;	Want to detect stale terms wrt occurs-free.
;;;;	Want to detect stale terms wrt edit alpha-equality.
;;;;	
;;;;	Have one flag to flag stale dform : any mod turns flag on.
;;;;	Head of copy implicitly should be flagged as stale dform.
;;;;	Lazy dtree as will should be flagged as stale dform.
;;;;
;;;;	Have finer discrimination to flag stale terms :
;;;;
;;;;	  - structure : cut or paste, or parameter meta, slot, or variable change.
;;;;	  - tags : labels or tags on parameter or term changed.
;;;;	  - text : parameter modification where meta bits not changed, slot bit not changed and
;;;;		   type is not variable.
;;;;	
;;;;	Then :
;;;;	  - dummy check can live with text and tags mods.
;;;;	  - float checks can live with tags mods.
;;;;	  - dform-match can live with tags mods.
;;;;	  - group lookup can live with any mod as no mod can change term sig
;;;;	    of term represented by dtree.
;;;;
;;;;	Due partial refresh of term, ie only refresh down through needed mods.
;;;;	However, be sure to update flags if other mods refreshed as well. Ie,
;;;;	may only need to refresh structure mods, but if no tag mods below
;;;;	frontier of structure mods then tag mod flag can be offed.
;;;;
;;;;	
;;;;	FTTB match will require a term, it requires though that all mods except tag
;;;;	mods be resolved. In the dtree refresh case, once matched the children
;;;;	can be reused if the dform is the same (or similar?). Refresh ends when
;;;;	a dtree is encountered whose dform is not stale.
;;;;	
;;;;	RLE PERF : it might be worthwhile to have a dform match which works on the dtree.
;;;;	
;;;;	
;;;;	cond_expr : either we have dtree or lazy term.
;;;;
;;;;	When cond_expr requires a dtree but has a term then
;;;;	  - dtree should be extended
;;;;	      * walk.
;;;;	  - cond_expr should fail
;;;;	      * dform choice.
;;;;
;;;; 	 need to be able to identify ce's which can be tested on terms.
;;;;	 need to be restrict dform tests to such ce's. Of course going up should be ok.
;;;;	 Or just fail and issue warning when dform cond_expr oversteps its bounds.
;;;;
;;;;	After dtree layout, unused (??) dtrees should be cut and replaced with exported term.
;;;;	This can be done lazily.
;;;;
;;;;	Observations:
;;;;	  - Leaves are modified by replacing parameter, ie replacing dtree
;;;;	    thus a leaf should never have a structure or text flag on. Tag flag yes.
;;;;	  - Lazy term is fresh implicitly, ie all term mod flags off.
;;;;	  - Exported terms always contain itags.
;;;;	  - Edit alpha-equality will look through the unquoted itag operators.
;;;;	  - Itags do not affect occurs free check.
;;;;	  - Edit alpha-equality is used by dform match and float up/down,
;;;;	  - Occurs-free is used by dummy check.
;;;;	
;;;;	
;;;;	Lazy dtree is vestigial, we now require copy to prevent shared tags.
;;;;	Head of copy implicitly should be flagged as stale dform.
;;;;	
;;;;	
;;;;	extend-dtree-instantiation(<dtree>)			: <dtree>
;;;;	
;;;;	extend-dtree-refresh(<dtree>)				: <dtree>
;;;;	
;;;;	copy-dtree(<dtree>)					: <dtree>
;;;;	
;;;;	dtree-to-term(<dtree> &optional <bool{text-ok}> <bool{tag-ok}>)
;;;;	  : <term>
;;;;	 * mask indicates which modifications to resolve.
;;;;	
;;;;	 ** we do allow text mode to dtree-leaves now as means of buffering
;;;;	    edit mods. if text-ok is t then text mods are ignored. if nil
;;;;	    then parameter reflects text mods. Thus, text mods to not
;;;;	    require history updates as looked at with text-ok t they are
;;;;	    not modified.
;;;;	
;;;;	dtree-replace(<dtree{point}> <dtree{new}>		: <dtree>
;;;;	  * returned dtree is new root of points tree.
;;;;	
;;;;	test-condition-expression(<cond_expr> <token{mode}> <dtree> <bool{extend-p}>)
;;;;	  : <bool>
;;;;	  * if extend-p true then extends instantiation of dtree when needed.
;;;;	  * otherwise issues warning and returns false.
;;;;	
;;;;	** could have bool to extend refresh before testing conditions??
;;;;	** no requirement for this at the moment.
;;;;	
;;;;	edit-alpha-equality (<term> <term>)			: <bool>
;;;;	 * ignores !tag operators.




;;;;	Dummy test :
;;;;	 OR Var is dummy
;;;;	    Var is display meta dummy  (allows dform to be used on model).
;;;;	    AND	Var :
;;;;		  * is not slot.
;;;;		  * hidden cond_expr evals to true.
;;;;		Term :
;;;;		  * does not contain a slot
;;;;		  * is not an !template term.
;;;;		    (doesn't allow hiding of bindings of template subterms of model?)
;;;;		Var does not occur free in term.
;;;;	   


(defvar *dummy-test-slot-ce* (new-descendent-ce (slot-ce)))

;; hidden tested with other hiddens.
(defun dummy-test-p (var term)
  (and (not (slot-parameter-value-p var))
       (or (dummy-variable-id-p var)
	   (dummy-display-meta-variable-id-p var))
       (and (not (itemplate-term-p (term-from-dtree-tags-term term)))
	    (not (test-condition-expression *dummy-test-slot-ce* 'term term))
	    (occurs-free-p (value-of-parameter-value var) term)
	    )))



;; 'dforms mark of term.
;; dtree-pool
(defun clear-dtree-dform-cache (dtree)
  
  (dtree-mapc dtree
	      #'(lambda (d)
		  (setf -d d)
		  (unless (dtree-leaf-p d)
		    (setf (dtree-dform-pool d) nil)
		    (unmark (term-of-dtree d) 'dforms)))))


(defun parameter-to-indirect-term (lib-parameter old-iobject)
    
  (cond
    ((and (real-parameter-p lib-parameter) (oid-parameter-p lib-parameter))
     (let ((oid (value-of-parameter lib-parameter)))
       (if (and old-iobject
		(iobject-term-p old-iobject)
		(equal-oids-p oid (oid-of-iobject-term old-iobject)))
	   (progn ;;(format t "~%Old IObject~%")
		  old-iobject)
	   (progn ;;(format t "~%New IObject~%")
		  (iobject-term oid (edit-read-source oid))))))

    (t (instantiate-term (instantiate-operator '|iobject|)))
    ))


;; dform -> dtree -> bool
;; if true then dform matched and dtree updated?
(defun dform-match-p (dform dtree pool)

  (setf -dform dform -dtree dtree -pool pool) ;;(break "dmp")
  
  (let* ((model (model-of-dform dform))
	 (dfparms (dfparms-of-dtree dtree)))

    ;; check that iparms, ie real variable subterms of model term,
    ;; are bound by implicit, ie real variable bindings of ancestor model terms.
    (unless (let ((implicit (implicit-of-dfparms dfparms)))
	      (forall-p #'(lambda (iparm)
			    (member iparm implicit))
			(iparms-of-dform-model model)))
      (return-from dform-match-p 1))
  
    ;; parent test.
    (let ((families (families-of-dform dform)))
      (when families
	(unless (member (cursor-of-dfparms dfparms) families)
	  (return-from dform-match-p 2))))


    (let* ((floatup-index (floatup-index-of-dform dform))
	   (dummy-tests (dummy-tests-of-dform-model model))

	   ;; want to avoid dtree-to term for 'text but this is dangerous, better
	   ;; to avoid dtree-fresh if only text mod unless forced.
	   ;;(term (dtree-to-termdtree 'text))
	   (term (progn;;(setf -dtree dtree) (break "cdt")
		   (term-from-dtree-tags-term (dtree-to-term dtree nil))))

	   (variables (variables-of-dform-model model))
	   (match-assoc-list (dform-term-match (term-of-dform-model model) term)))

      (setf -mal match-assoc-list) ;;(when (eql -mal 'fail) (break "mal"))
      ;;(when (eql -ddform dform) (setf -dform dform -term term -dtree dtree) (break "hello"))

      (when (eql 'fail (car match-assoc-list)) (return-from dform-match-p nil))

      (setf -mal2 match-assoc-list)
      
      ;;(format t "~%MAL ~a~%" match-assoc-list)
      ;; float checks and dummy check.
      (let ((children (make-array (num-children-of-dform dform)))
	    (float-term (when floatup-index
			  (term-of-bound-term
			   (cdr (assoc (id-of-dform-model-variable (aref variables floatup-index))
				       match-assoc-list)))))			     
	    (floatup-continued-p (dform-flag-floatup-continued-p dform))
	    (former-dform (dform-of-dtree dtree))
	    (former-children (children-of-dtree dtree)))
	(declare (vector children former-children))

	;;(when (and (= 1 (length former-children)) (eql -mldtree (aref former-children 0) )) (break "ugh"))
		   
	(labels
	    ((usedp (children i dtree)
		    (setf -a i -b dtree -c children) (when nil (break "usedp"))
		    (let ((p nil))
		      (when (> i 0)
			(dotimes (j i)
			  (when (eql dtree (aref children j))
			    (setf p t))))
		      p))

	     (test-child (val mv)
			 (if (parameter-p val)
			     ;; test hidden
			     (when (and (dform-model-variable-flag-hidden-p mv)
					(not (forall-p #'(lambda (ce)
							   (test-condition-expression ce 'term val))
						       (hidden-cond-expr-of-dform dform))))
			       (progn
				 ;;(setf -a mv -b dform -c val) (break "cd")
				 (return-from dform-match-p 5)
				 ))
			
			   (let ((subterm (term-of-bound-term val)))

			     ;; test hidden
			     (when (and (dform-model-variable-flag-hidden-p mv)
					;; rle todo : this is nfg. must traverse termnnn.
					;; rle todo : also hidden cond-expr is list of expr not single expr.
					(not (forall-p #'(lambda (ce)
							   (test-condition-expression ce 'term subterm))
						       (hidden-cond-expr-of-dform dform))))
			       (return-from dform-match-p 6))

			     ;; check predicates on subterms here.
			     (when (dform-model-variable-flag-iterate-p mv)

			       ;; do iterate test.
			       (when (not (iterate-match-p (term-of-dform-model model) subterm))
				 (return-from dform-match-p 7))

			       ;; do floatdown-test
			       (let ((floatdown-index (floatdown-index-of-dform-model model)))
				 (when floatdown-index
				   (unless (edit-equal-terms-p (term-of-bound-term
								(nth floatdown-index (bound-terms-of-term term)))
							       (term-of-bound-term
								(nth floatdown-index
								     (bound-terms-of-term
								      (term-from-dtree-tags-term subterm)))))
				     (return-from dform-match-p 8)))))))
			 )

	     (build-child (term dfparms old-child)
			  (when (null dfparms) (break "huh"))

			  (let ((old-dfparms (and old-child (dfparms-of-dtree old-child))))
			    (cond
			     ;; same dforms, former child and dfparms eq  => reuse former child
			     ((and old-child
				   (eql dform former-dform)
				   (eql dfparms old-dfparms))
			      (set-dtree-parent old-child dtree)
			      old-child)

			     ;; former child => new child (modified) but cp children from old child.
			     ;; ??? not clear why we can not reuse dtree but modify layout data.
			     ((and old-child
				   ;; if we allow this messes up dtree, but not sure why. 1/01
				   (not (> (length (library-formats-of-dform dform)) 0)))
			      ;;(format t "refreshing child ~a ~%" (id-of-term term))
			      (new-dtree-refresh old-child dfparms dtree))
		     
			     ;; otherwise lazy-term child.
			     (t
			      ;; this is not the correct term wrt text since did (dtree-to-term dtree 'text)
			      ;; ie this depends on finding the old child. and reusing the dtree.
			      ;; (setf -a term -b dfparms -c old-child) (break "ndlt")
			      (new-dtree-lazy-term term dfparms dtree)))))
	     )

	  (when floatup-continued-p
	    ;; floatup-continued-p => floatup-index hav value => float-term is a term.
	    (let ((dfparm-float (float-of-dfparms dfparms)))
	      (unless (and dfparm-float (edit-equal-terms-p float-term dfparm-float))
		(return-from dform-match-p 3))))

	  ;; dummy tests
	  (when dummy-tests
	    (mapc #'(lambda (dummy-test)
		      (unless (let* ((bt (nth (car dummy-test) (bound-terms-of-term term))))
				(dummy-test-p (nth (cdr dummy-test) (bindings-of-bound-term bt))
					      (term-of-bound-term bt)))
			(return-from dform-match-p 4)))
		  dummy-tests))

	  (dotimes (i (length variables))
	    (let* ((mv (aref variables i))
		   (val (cdr (assoc (id-of-dform-model-variable mv) match-assoc-list))))

	      (test-child val mv)))

	  (dotimes (i (length variables))
	    (let* ((mv (aref variables i))
		   (val (cdr (assoc (id-of-dform-model-variable mv) match-assoc-list))))

	      (setf (aref children i)

		    (if (parameter-p val)
			
			(or (some #'(lambda (child)
				      ;;(when (eql child -mldtree) (break "oh"))
				      (when (and (dtree-leaf-p child)
						 (eql val (parameter-of-dtree child))
						 (not (usedp children i child)))
					child))
				  former-children)
			    (progn
			      ;;(break "yoohoo")
			      (new-dtree-leaf val dtree)
			      ))
			
		      (let ((subterm (term-of-bound-term val)))


			;; Could use model-eq-test to id array of children for reuse.
			;;
			;; When refreshing we should be encoding tags to assure tags are carried over
			;; when dtree not reused.
			;;
			;; RLE PERF: fttb do subterm eq test and leave model eq test as optimization
			(let* ((child (some #'(lambda (child)
						(when (and (not (dtree-leaf-p child))
							   (eql subterm (term-of-dtree child))
							   (not (usedp children i child)))
						      
						  child))
					    former-children))
			       (child-dfparms (and child (dfparms-of-dtree child))))

			  (build-child subterm
				       (new-dfparms dfparms
						    ;; if some chance of reuse,
						    ;; good bet if we've got this far.
						    (when (and child
							       (eql dform former-dform)
							       (not (dtree-flag-modified-p child)))
						      child-dfparms)
						    dform
						    mv
						    (bindings-of-bound-term val)
						    (or floatup-continued-p float-term))
				       child)))))))


	  ;; Constants and libs : some attempt is made at reuse. Could
	  ;; be more aggresive, but this should preserve dtrees in the
	  ;; obvious cases, ie very similar dforms, without expending
	  ;; too much computation. 

	  ;; Constant:
	  ;;	Allows reuse of former dtree of constant when:
	  ;;   - former-dform and dform contain same number of constants.
	  ;;   - constant terms in dform formats are identical.
	  ;;  If no reuse, tag and labels in dtree are lost.
	  (let* ((constants (constant-formats-of-dform dform))
		 (former-constants (and former-dform (constant-formats-of-dform former-dform)))
		 (possible-p (and former-dform (= (length constants) (length former-constants)))))
	    
	    (dotimes (i (length constants))
	      (let* ((f (aref constants i))
		     (term (term-of-dform-constant-child f))
		     (former-child (and possible-p
					(let ((g (aref former-constants i)))
					  (when (compare-terms-p term
								 (term-of-dform-constant-child g))
					    (aref former-children (dtree-index-of-dform-child g))))))
		     (former-dfparms (and former-child (dfparms-of-dtree former-child))))
			
		(setf (aref children (dtree-index-of-dform-child f))
		      (build-child term
				   (new-dfparms dfparms former-dfparms dform nil nil nil)
				   former-child)))))
 
	  ;; change of parameter pointer should cause this refresh.

	  ;; Library:
	  ;;	Allows reuse of former dtree of library when:
	  ;;   - former-dform and dform contain same number of libs.
	  ;;   - refresh of iobject terms results in identical term.
	  ;;  If no reuse, tag and labels in dtree are lost.
	  (let* ((libs (library-formats-of-dform dform))
		 (former-libs (and former-dform (library-formats-of-dform former-dform)))
		 (possible-p (and former-dform (= (length libs) (length former-libs)))))
	    
	    (dotimes (i (length libs))
	      (let* ((f (aref libs i))
		     (former-child (when possible-p
				     (aref former-children
					   (index-of-dform-library-child (aref former-libs i)))))
		     (former-term (and former-child (dtree-to-term former-child))) ; 'tag?
		     (term (handle-process-err #'(lambda (msg)
						   ;;(setf -libterm term -former-term former-term -children children) ;;(break "libterm")
						   ;;(setf -msg msg) (break "mdp5")
						   (format t "mdp5")
						   (return-from dform-match-p 5))
					       (let ((ldtree (aref children (index-of-dform-library-child f))))
						 (if (dtree-leaf-p ldtree)
						     (parameter-to-indirect-term (parameter-of-dtree ldtree) former-term)
						   (term-to-indirect-term (term-of-dtree ldtree) former-term))))))

		  
		(setf (aref children (dtree-index-of-dform-child f))
		      (build-child term
				   (new-dfparms dfparms
						(when (and former-term (eql former-term term))
						  (dfparms-of-dtree former-child))
						dform nil nil nil)
				   (when (eql former-term term) former-child)))))))
	  
	(map-vector #'(lambda (child)
			(when (and (not (dtree-leaf-p child))
				   (null (dtree-dfparms child)))
			  (setf -c child -d dtree -f dform) (break "dmp")))
		    children)
		      
	(set-dtree-children dtree children t)
	(set-dtree-dform dtree dform pool)
	;;(when (dtree-labeled-p 'point dtree) (break "ahha"))
      
	t))))


(defun dtree-suppressed-dform-p (dform dtree)
  (let ((tags (tags-and-labels-of-dtree dtree)))
    (when tags
      (or (let ((tag (suppress-tag-of-dform dform)))
	    ;; default dforms will not have a tag, and thus cannot be suppressed.
	    (and tag (member tag (tags-of-tags tags))))
	  (let ((label (suppress-label-of-dform dform)))
	    (and label (member label (labels-of-tags tags))))))))
			      

(defvar *dform-ce* nil)

(defmacro with-dform-ce ((ce) &body b)
  `(let ((*dform-ce* (new-or-ce (new-token-ce 'disp 'default) ,ce)))
    ,@b))

(defun test-global-dform-ce (dtree)
  (or (null *dform-ce*)
      (test-condition-expression *dform-ce* 'dtree dtree)))


;; want partial dtree with tags to test local suppression and other cond_expr
(defun choose-dform (dtree &optional force)
    
  ;;(format t "cd~%")
  ;;(setf -dtree dtree) (break "cd")
  ;;(when (real-ilabel-term-p (term-of-dtree (dtree-of-view v)))  (setf -a dtree) (break "cd"))
  ;; but might we passed a dtree that is instantiated.
  (let ((instantiated-p (dtree-flag-instantiated-p dtree)))
    (with-unwind-error ((dtree-flag-set-instantiated dtree instantiated-p))
      (when (or force (stale-dtree-dform-p dtree))
	;;(setf c dtree) (break "cd1")
	(let ((pool (dform-pool-of-term (term-of-dtree dtree))))

	  ;;(setf -dtree dtree -pool pool) (break "cd")
	  
	  (dolist (dforms (list-of-definition-pool pool))
	    (dolist (dform (permuted-list-of-dforms dforms))
	      ;;(setf -dforms dforms a dform b dtree) (break "cd3")
	      (when (and dform
			 ;; suppressed?
			 (not (or (globally-suppressed-dform-p dform)
				  (view-suppressed-dform-p dform (current-view))
				  (dtree-suppressed-dform-p dform dtree)))

		 ;;; RLE PERF : Could factor cond_expr and test parts not requiring match
		 ;;; RLE PERF : be completed before match.
		     
			 ;;(progn
			 ;;(setf d dform e dtree)
			 ;;(break "match")
			 ;; t)
		     
			 ;; sets dtree-dform field
			 (eql t (or (when (eql dform (dform-of-dtree dtree))
				      (setf (dtree-dform-pool dtree) (refresh-definition-pool pool))
				      t)
				    (dform-match-p dform dtree pool)))

			 (dtree-flag-set-instantiated dtree t)

			 (test-global-dform-ce dtree)
			 (forall-p #'(lambda (ce)
				       (test-condition-expression ce 'dtree dtree))
				   (cond-expr-of-dform dform))
			 )

		;; choice stored in dtree by dform-match-p.
		;;(setf -a dtree -b pool -c dforms ) (break "cd")
		(return-from choose-dform) 
		)))
	       
	  (dform-match-p (new-default-dform (term-of-dtree dtree)) dtree nil))))))


;;; assume all dtrees start life as some lazy form.

;;; extend-dtree : 
;;; updates node, sets dfparms of children.

;; seems like there should be a refresh wrt to dform changes which
;; does not force dform choice.  



;;;;	
;;;;	Following functions walks dtree top-down and marks as stale any node satisfing
;;;;	predicate on node.
;;;;	
;;;;	dtree-suppress-dform dform-p
;;;;	dtree-unsuppress-dform term-p
;;;;	


(defun dtree-mapc (dtree f)
  (labels ((visit (d)
	     (funcall f d)
	     (unless (dtree-leaf-p d)
	       (let ((children (children-of-dtree d)))
		 (when children
		   (dotimes (i (array-dimension children 0))
		     (visit (aref children i))))))))
    (visit dtree)))


(defun dtree-suppress-dform (dtree dform)
  (let ((count 0))
    (dtree-mapc dtree
		#'(lambda (d)
		    (unless (dtree-leaf-p d)
		      (when (eql dform (dform-of-dtree dtree))
			(incf count)
			(dtree-path-layout-modified d 'structure)))))
    count))

(defun dtree-unsuppress-dform (dtree dform)
  (let ((count 0)
	(tsig (term-sig-of-term (model-term-of-dform dform))))
    (dtree-mapc dtree
		#'(lambda (d)
		    (unless (dtree-leaf-p d)
		      (when (term-sig-of-term-p tsig (term-of-dtree dtree))
			(incf count)
			(dtree-path-layout-modified d 'structure)))))
    count))
  
  


;;;
;;; Need some guard to catch reference to non instantiated/refreshed dtrees.
;;;
;;;;	instantiated and not reqired-refresh : continue
;;;;	  - note modified state irrelevant.
;;;;	required-refresh : extend silently.
;;;;	not instantiated : depending on caller
;;;;	 - instantiate.
;;;;	 - fail.
;;;;
(defun view-layout-stale-p (v d)
  (or (not (dtree-flag-instantiated-p d))
      (dtree-modified-flag-greater-than-p (dtree-flag-layout-modified-q d) 'tag)
      (view-flag-layout-required-p v)))

;; if root dtree has layout modified then
;; need view to require present or at least cursor-present.
(defun dtree-of-view-c (v)
  (let ((d (dtree-of-view v)))

    (when (view-layout-stale-p v d)
      (when *dtree-flag-trace*
	(format t "dtree-of-view-c layout-required -> t~%"))

      (set-view-label-cache v nil)
      (dtree-flag-set-layout-required d t)
      (set-dtree-dfparms d (make-dfparms :implicit (implicit-of-view-object v)))
      (refresh-dtree d t))
    d))


(defun refresh-dtree (dtree refresh-p)

  (let ((dform (dform-of-dtree dtree))
	(chosenp nil))

    (cond
      ((or (null dform)
	   (not (dtree-flag-instantiated-p dtree)))

       (setf chosenp t)
       (choose-dform dtree))

      ((let ((modq (dtree-flag-layout-modified-q dtree)))
	 (and refresh-p
	      (or (dtree-flag-layout-required-p dtree)
		  (not (or (null modq)
			   (eql 'text modq)
			   (and (eql 'tag modq)
				(member 'ChooseDFormEagerly (conditions-of-dform dform)))
			   ))
		  (stale-dtree-dform-p dtree))))
       #|(and refresh-p
	    (or (let ((modq (dtree-flag-layout-modified-q dtree)))
		  (not (or (null modq)
			   (eql 'text modq))))
		(stale-dtree-dform-p dtree)))|#
       
       ;;(setf -d dtree) (break "dr")

       (setf chosenp t)
       (choose-dform dtree t)))

    (when chosenp
      ;;(setf -d dtree) (break "dr")
      (dtree-flag-trace
	(format-string "refresh-dtree layout-modified -> nil~%"))
      (dtree-flag-set-layout-modified dtree nil)
      (dtree-flag-set-instantiated dtree t)

      (let ((ddform (dform-of-dtree dtree)))
	(unless (eql ddform dform)
	  ;;(setf -a ddform -b dform) (break "rd")
	  ;; set root path
	  (dtree-path-layout-required dtree) )))

    dtree))
    

(defun dform-of-dtree-c (dtree &optional refresh-p)
  ;;(when (itext-term-p (dtree-term dtree)) (setf -dtree dtree) (break "dodc"))

  (refresh-dtree dtree refresh-p)
  (dform-of-dtree dtree))
  

(defun children-of-dtree-c (dtree &optional refresh-p)
  (refresh-dtree dtree refresh-p)
  (children-of-dtree dtree))


;;;;
;;;; 	Parenthesize ?
;;;;


(defun parenthesize-p (dtree parens)

  ;;(setf -dtree dtree -parens parens) (break "pp")
  (and t

  (labels ((passthru (dtree)
	     (when dtree
	       (let ((dform (dform-of-dtree-c dtree)))
		 (if (dform-parens-passthru-p dform)
		     (passthru (parent-of-dtree dtree))
		     dform)))))

    (when parens			; no parenthesization by default.

      (let ((address (or (precedence-address-of-dform-parentheses parens)
			 (let ((pdform (passthru (parent-of-dtree dtree))))
			   (when pdform (precedence-address-of-dform pdform)))))
	    (dform (dform-of-dtree-c dtree)))

	(unless (or (dform-precedence-exception-p dform)
		    (null address))
	  (compare-dform-precedences-p (relation-of-dform-parentheses parens)
				       (precedence-address-of-dform dform)
				       address))))))

)


;;;;	
;;;;	edit-walk
;;;;	
;;;;	

;; 
;; child formats contain indices into dtree child list. child formats are permuted.
;; dtree-children is un-permuted and without dups. In same order as model meta vars?
;;
;; walking a subtree should involve upce limited escape from start.
;;
;;
;; 


(defun path-to-root (d)
  (let ((p (parent-of-dtree d)))
    (cons d (when p (path-to-root p)))))

;; bool -> cond expr -> (view -> bool)
;;	-> cond expr -> cond expr -> label -> bool -> view

;; returns self if no descendent.
(defun dtree-descendent (perp dirp ce dtree &optional abort-f)
  (labels ((aux (d)
	     (let ((child (dtree-child perp dirp ce d abort-f)))
	       (if (null child)
		   d
		   (aux child)))))
    (aux dtree)))


;; this will grow dtree unless floor avoids uninstantiated dtrees.
(defun edit-walk (totalp stopce stopf upce downce start dirp view)

  ;; remember start  
 (let* ((point (dtree-at-label start view))
	 (start-path (path-to-root point))
	 (haltp nil)
	 (topp nil)	;; if top hit twice then stop, ie avoid loop if error.
	 (notdown (new-not-ce downce))
	 (perp (not totalp)))
    
    (setf -start-path start-path -notdown notdown -stopce stopce -upce upce)
    (edit-move-label-to-label view '!walk start)
    
    (labels
	((visit (d)
	   ;;(setf -d d) (break "ewv")
	   (setf point d)
	   (edit-move-label view '!walk point)

	   ;; halt if start point encountered. 
	   (when (eql (car start-path) point)
	     (return-from edit-walk nil))

	   (if (test-condition-expression stopce 'dtree point)
	       (if (or (null stopf) (funcall stopf point view))
		   (progn
		     ;;(setf -d point) (break "ewt")
		     (edit-move-label-to-label view start '!walk)
		     (edit-remove-label view '!walk)
		     (return-from edit-walk t)
		     )
		   (move))
	       (move)))
	 
	 (move ()
	   (setf -point point)
	   ;;(break "m")
	   (let* ((p (unless (when (test-condition-expression upce 'dtree point)
			       ;; halt if avoid-up true and point avoided is not an start path.
			       (unless (member point start-path)
				 (return-from edit-walk nil))
			       t)
		       (parent-of-dtree point)))
		  (s (when p
		       (dtree-find-sibling perp dirp notdown point p
					   #'(lambda (avoided)
					       (when (member avoided start-path)
						 ;; could return from here but worry about weak compilers.
						 (setf haltp t)
						 t))))))
	     (when (null p)
	       (if topp
		   (return-from edit-walk nil)
		   (setf topp t)))
		     
	     (when haltp
	       (return-from edit-walk nil))
	     
	     ;;(setf -s s -p p -point point) (break "ewm")
	     (cond

	       (s (visit (dtree-descendent perp dirp notdown s)))

	       (p (visit p))

	       ;; have no sibling and no parent thus we must be root.
 	       (t (let ((descendent (dtree-descendent perp dirp notdown point
					   #'(lambda (avoided)
					       (when (member avoided start-path)
						 ;; could return from here but worry about weak compilers.
						 (setf haltp t)
						 t)))))
		    ;;(setf -haltp haltp -descendent descendent) (break "mm")
		  (when haltp
		    (return-from edit-walk nil))
		  (visit descendent)))))) )


      (move) )))


;;;;
;;;;	Rather than a contiuation environment, allow globals
;;;;	but require globals to be bound dynamically by some high
;;;;	level caller.



;;; RLE ??? needs work::
;;; might be worthwhile do distinquish a match equivalence relation for dforms.
;;; ie, two dforms are match= if they produce same children when matched to same term
;;; then if two dforms are match= you can safely swap them in the dtree without
;;; violating cache/rep term invariant.


;;;;; Vestigial :::



;;;;
;;;; Following is not applicable:
;;;;

;;;;	share
;;;;	destructive modify
;;;;	copy - constructive modify
;;;;	  - so as to later modify without affecting original.
;;;;	  - so as to make a new item???
;;;;

;;;;	two level data structure, when copy, first level new and sets bit describing status
;;;;	 of second level.
;;;;
;;;;    copy some parts. itself similar, ie want to have some parts shareable some modifiable.
;;;;	interpret diff in diff contexts??


;;;;	Constructive modification of dtree spine inherits layout data.

;;;;	to achieve constructive modification spine must point to layout.
;;;;	   - ??? Is it possible to have layout point to spine.
;;;;	thus, no sharing of layout disallows sharing of spine.

;;;;	two level data structure
;;;;	  - first level contains destructive pointers, second level constructive.
;;;;	
;;;;	

;;;; 	shareable data should be in second level.
;;;;	status of shareable data
;;;;

;;;;	
;;;;	possibly duplicate lhs var instances should share spine but not layout???
;;;;	

;;;;
;;;; 	It may be possible to share some parts of the layout. 
;;;;	might be worth some general support to easily manage mixed constructive/destructive data.
;;;;

;;;;	  - constants : never modified. shareable. indirect-p ???
;;;;	  - critical : not modifiable, cache term. shareable
;;;;	  - malleable : modifiable, but does not change data. Not shareable
;;;;		ie changing dform and children does not change term rep.
;;;;	  - properties : modifiable not critical, layout. Not shareable
;;;;	


;; shareable cannot point to nonshareable. or is that backwards?

;;;;	
;;;;	parameter modification is qualitatively different from term
;;;;	modify as term sig and therefore dform group, does not change. 
;;;;	

;;;;	possible to choose dforms without cache term?? ie from dform and children
;;;;	test predicates, match?.
;;;;	

;;;;	
;;;;	up pointers and constructive updates do not peacefully coexist.
;;;;	 - constructive update builds new dtree nodes to root.
;;;;	 - thus new root, thus all old pointers to old root are stale.
;;;;	   transitively all old pointers point to old root, thus whole tree would
;;;;	   need to be rebuilt.
;;;;	
;;;;	Thus up pointers have to be considered a cache.
;;;;	But want robust, want sharing, want abstract treatment.
;;;;	
;;;;	Destructive mods to dtrees:
;;;;	  - substitute
;;;;	      * target - effectively cut from tree, should be marked for reuse.
;;;;		
;;;;	      * path
;;;;		  - children have stale up pointers

;; old root meaningful? undo

;; assume we have 

  

;; normally expect ptree to be reconstructed from old.

(defun ptree-to-dtree (ptree-new ptree-old dtree-old)
  )

;; but tis possible to build one from a ptree.


(defun infer-typeid-from-labels (labels)
  (or 
   (with-ignore
       (let ((id (find-first
		  #'(lambda (l)
		      (and (type-id-p l)
			   l))
		  labels)))
	 (let ((typeid (when id  (unalias-typeid id))))
	   (when typeid
	     (when (string= (string-upcase (string typeid))
			    (string-upcase (string id)))
	       typeid)))))
   *string-typeid*))

;;ignore embedded labels fttb.
(defun ptext-to-string (ptext)
  (with-byte-accumulator ('string)
    
    (labels
	((glyph-run (g)
	   (let ((a (array-of-glyph-run g)))
	     (dotimes (i (length a))
	       (accumulate-byte (aref a i))))))

      (walk #'(lambda (glyphs)
		(when (glyphs-p glyphs)
		  (let ((runs (runs-of-glyphs glyphs)))
		    (dotimes (i (array-dimension runs 0))
		      (glyph-run (aref runs i))))))
	    (glyphs-of-ptext (text-of-ptree-text ptext))))))



(defun ptree-to-new-dtree (ptree)
  (if (or (ptree-text-pform-p ptree)
	  (ptree-text-p ptree))
      (progn
	(setf a ptree b
	      (new-dtree-leaf
	       (instantiate-parameter-s (ptext-to-string ptree)
					(type-id-to-type
					 (infer-typeid-from-labels (labels-of-ptree ptree))))
	       nil))
	;;(break "ptnd")
	b)

      (if (ptree-pform-p ptree)
	  (let ((dform (pda-lookup (pform-of-ptree-pform ptree))))
	    
	    (if dform
		(let* ((pformats (formats-of-ptree ptree))
		       (l (length pformats))
		       (children (make-array l)))

		  (when (not (= l (num-children-of-dform dform)))
		    (raise-error (error-message '(ptree dtree children length))))

		  (dotimes (i l)
		    (setf (aref children i)
			  (ptree-to-new-dtree (aref pformats i))))

		  ;;(setf a children) (break)
		  (let ((d (new-dform-dtree dform
					    children
					    (set-difference (labels-of-ptree ptree)
							    (conditions-of-dform dform)))))

		    (dotimes (i l)
		      (set-dtree-parent (aref children i) d))
		    d))
		(raise-error (error-message '(ptree dtree pform not)))))
	  (raise-error (error-message '(ptree dtree pform not))))))



