
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;; Lifting of lisp functions for timing into ml

(defunml (|sleep| (n) :error-wrap-p nil)
  (int -> void) 
  (sleep n))

(defunml (|get_internal_run_time| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (unit -> int)

  (get-internal-run-time))

(defunml (|get_internal_real_time| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (unit -> int) 

  (get-internal-real-time))

(dmlc |itus_per_sec| internal-time-units-per-second int)

;;; sun idiotically have their time units as microsecs, though they
;;; only report all times to the nearest 10ms.

(dmlc |timer_precision_in_itus| #+sun 10000 #-sun 1 int)

(defmacro meter-iterated-form (form n)
  `(let ((start (get-internal-run-time)))
    (dotimes (i ,n) ,form)
    (let ((end (get-internal-run-time)))
      (- end start))))


;;; fun must be function of 1 dummy argument.

(defun meter-iterated-fun (fval n)
  (meter-iterated-form (funcall fval) n))


;;; like meter-iterated-fun but for use with ml functions of type void->*.
;;; returns total time in internal-time-units units.

;;; closure access functions from sys/ml/runtime.lisp

(defunml (|meter_iterated_fun| (ml-fun-closure n) :error-wrap-p nil)
  ((unit -> unit) -> (int -> int))
  
  (let* ((ml-fun (closure-func ml-fun-closure))
         (fun-arity (closure-arity ml-fun-closure))
         (args (closure-env ml-fun-closure)))
    (cond ((= fun-arity 1)
           (meter-iterated-form (funcall ml-fun nil) n))
          ((and (= fun-arity 2) (= (length args) 1))
           (let ((arg (first args)))
             (meter-iterated-form (funcall ml-fun nil arg) n)))
          (t
           (meter-iterated-form (ap ml-fun-closure nil) n)))))

(defunml (|meter_iterated_fun_ap| (ml-fun-closure n)  :error-wrap-p nil)
    ((void -> void) -> (int -> int))
  (meter-iterated-form (ap ml-fun-closure nil) n))

  

;;; Utility functions for finding out about lisp versions of ml functions.

(defun ml-apropos-list (name)
  (apropos-list (read-from-string
                 (concatenate 'string *ml-runtime-package-name* "::|" name "|"))))

(defun ml-describe (name)
  (map 'list #'describe (ml-apropos-list name)) nil)

(defun ml-inspect (name)
  (map 'list #'inspect (ml-apropos-list name)))



;; converts time in sec to int giving number of milliseconds.

(defun convert-time-to-int (time) (round (* time 1000)))

#+lucid
(defmacro meter-time (form)
  `(multiple-value-bind
       (real-time
        total-cpu-time
        user-cpu-time
        sys-cpu-time
        page-faults
        disk-io
        net-io
        dynamic-conses
        dynamic-gcs
        ephemeral-conses
        ephemeral-gcs)
       (time1 ,form)
       (declare (ignore total-cpu-time sys-cpu-time page-faults disk-io net-io
                        dynamic-gcs ephemeral-gcs))

     `(,(convert-time-to-int real-time)
       ,(convert-time-to-int user-cpu-time)
       ,dynamic-conses
       . ,ephemeral-conses)))

#+lucid
(defmacro meter-iterated-form1 (form n)
  `(meter-time (dotimes (i ,n) ,form)))

;; returns 4 values
;; 1. real elapsed time
;; 2. user cpu time
;; 3. dynamic conses
;; 4. ephemeral conses
  
#+lucid
(defunml (|meter_iterated_fun_ap_1| (ml-fun-closure n) :error-wrap-p nil)
    ((void -> void) -> (int -> (int |#| (int |#| (int |#| int)))))
  (meter-iterated-form1 (ap ml-fun-closure nil) n))

#-lucid
(defunml (|meter_iterated_fun_ap_1| (ml-fun-closure n) :error-wrap-p nil)
    ((void -> void) -> (int -> (int |#| (int |#| (int |#| int)))))
  (declare (ignore n))
  (ap ml-fun-closure nil)
  (list* 0 0 0 0))


;;; similar;;; similar to above, but passes arg to function, and returns result.

#+lucid
(defunml (|iterated_ap_with_meter| (n f x) :error-wrap-p nil)
    (int -> ((* -> **) -> (* -> (** |#| (int |#| (int |#| (int |#| int)))))))
  (when (< n 1) (breakout evaluation '|iterated_ap_with_meter: invalid count|))
  (let* ((val nil)
         (stats (meter-iterated-form1 (setf val (ap f x)) n)))
    (cons val stats)))

#-lucid
(defunml (|iterated_ap_with_meter| (n f x) :error-wrap-p nil)
    (int -> ((* -> **) -> (* -> (** |#| (int |#| (int |#| (int |#| int)))))))
  (declare (ignore n))
  (list* (ap f x) 0 0 0 0))
