
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	PROJECT : modify portable utilities to compile ml files. 
;;;;	
;;;;	PROJECT : define macro which will translate ml source in place
;;;;		 in lsp files (maybe using eval).
;;;;
;;;;	PROJECT : fixup flags macros.


#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      readlist catch-error
	      gen-internal-name gen-external-name gen-runtime-name
	      is-ml-function is-internal-function is-function is-isom
	      name-for-desc name-for-function mk-value-desc mk-function-desc mk-isom-desc
	      make-closure
	      abstract-type-name-p abstract-type-mkid-p mkid-to-abstract-type-name-list
	      mkid-to-abstract-type-name insensitive-name-to-abstract-type-mkid name-to-abstract-type-mkid
	      constant-to-abstract-type-mkid bearer-to-abstract-type-constant
	      abstract-type-pop-bearer abstract-type-constant-p
	      )))


;;;; -docs- (mod bml)
;;;;
;;;;
;;;;	ML:	
;;;;
;;;;	reset-ml()				: NULL
;;;;	 ** should be called after crash and burn.
;;;;
;;;;
;;;;	Calling ML:
;;;;
;;;;	<ml-result>		: (values <result> <ml-type> <ml-print>)
;;;;	<ml-print>		: <message> | NIL
;;;;
;;;;	 *** <result> is the lisp object result of the evaluation.
;;;;	 *** <ml-type> is the internal lisp representation of the ml type of the result.
;;;;	 *** <ml-print> is message containing print representation of <ml-result> and <ml-type>.
;;;;	 *** <ml-print> is NIL when displayp is NIL.
;;;;
;;;;	ml (<closure{nextf} <closure{eoff}> <closure{addrf}>
;;;;	    &optional (<bool:displayp}> t))
;;;;	 : <ml-result>
;;;;	 ** next-f ()		: <ich>
;;;;	 ** eoff ()		: <bool>
;;;;	 ** addrf()		: <scan-count>
;;;;
;;;;	ml-text (<text> &optional (<bool{displayp}> t))		: <ml-result>
;;;;
;;;;	
;;;;	Failure :
;;;;
;;;;	 Failures can be caught conditionally in ML when:
;;;;	  - fail is result of failwith ML call.
;;;;	    failwith must occur within same ML evaluation, ie if ML is called
;;;;	    recursively the caller can not catch failures within the callee.
;;;;	  - primitive function is defined specially. See defunml.
;;;;	When lisp calls ML, ML failures are transformed into <message> errors.
;;;;	Each lisp/ML barrier passed will wrap <message>.
;;;;
;;;;
;;;;	vml (<closure{nextf} <closure{eoff}> <closure{addrf}>
;;;;	    &optional (<bool:displayp}> t))
;;;;	 : (values <ml-type> <ml-print>)
;;;;	 ** next-f ()		: <ich>
;;;;	 ** eoff ()		: <bool>
;;;;	 ** addrf()		: <scan-count>
;;;;
;;;;	vml-text (<text> &optional (<bool:displayp}> t))	: (values <ml-type> <ml-print>)
;;;;
;;;;
;;;;	ast-ml (<closure{nextf} <closure{eoff}> <closure{addrf}>
;;;;	    &optional (<bool:displayp}> t))
;;;;	 : (values <ml-ast> <ml-type> <ml-print>)
;;;;	 ** next-f ()		: <ich>
;;;;	 ** eoff ()		: <bool>
;;;;	 ** addrf()		: <scan-count>
;;;;
;;;;	ast-ml-text (<text> &optional (<bool:displayp}> t))
;;;;	 : (values <ml-ast> <ml-type> <ml-print>)
;;;;
;;;;  -page-
;;;;
;;;;	Read-eval-print :
;;;;
;;;;	(ml-toploop)		: NIL
;;;;	 ** read-eval-print loop.
;;;;
;;;;	ml_exit : unit -> unit
;;;;
;;;;
;;;;	Extending ML:
;;;;
;;;;	Defining New Primitive Functions:
;;;;
;;;;	 defunml ((<id{function-name}> <id{arg-name}> list
;;;;		   &key <bool{error-wrapper-p}> declare-list)
;;;;		  type-expr
;;;;		  &body body)
;;;;	 : MACRO
;;;;
;;;;	**  type-expr : representation of type of function.
;;;;	**  error-wrap-p : when t transforms prl errors into ML errors.
;;;;	**	If wrap is t, then errors can not be caught conditionally.
;;;;	**	If wrap is symbol other than t, then error will be transformed from message
;;;;	**	into catchable ML error. The symbol specified can be used to catch the error.
;;;;	**	Default is t.
;;;;	**  declare-list : ie lisp declarations for compiler, eg (declare (ignore void))
;;;;	**	When error-wrap-p is t then declarations must be in declare list and not
;;;;	**	in body.
;;;;
;;;;
;;;;
;;;;
;;;;
;;;;	Defining New Primitive Abstract Types :
;;;;
;;;;	add-primitive-type (<id{type} <closure{print}>
;;;;			    &key <closure{member-p}>
;;;;				 (<closure{equal}> #'eq)
;;;;				 <closure{parse}>)
;;;;	 : NULL
;;;;
;;;;	 ** equal(<member> <member>)	: <bool>
;;;;     ** print(<member>)		: <text>
;;;;	 ** parse(<ml-ch>		: (values <bool> <member>)
;;;;	 ** * parse can use (gnc) to get more <ml-ch>'s. Currently, there is no mechanism
;;;;	 ** * for look-ahead. 
;;;;	 ** * if bool is t then <member> must be in type.
;;;;	 ** * if bool is nil the (gnc) must not have been called.	
;;;;	 ** * ML must not be re-entered from within parse.
;;;;	 ** * gnc ()			: <ml-ch>
;;;;
;;;;	add-tok-parse-hook (<id{type}> <closure{parse}>)		: NULL
;;;;	 ** parse(<ml-ch> <ml-ch> list)	: (values <bool> <member>)			
;;;;	 ** this is a kludge to allow for parsing of special token types which close
;;;;	 ** with alternate end ch, ie variables. The hook is called after for each
;;;;	 ** <ml-ch> scanned after the tok-quote `. 
;;;;	 ** The type must be defined before the parse hook is added.
;;;;
;;;;	reset-defined-ml-primitive-types ()				: NULL
;;;;	 ** undefines all defined types.
;;;;
;;;; -doce-

;;;;
;;;; RLE NAP
;;;;  *** Would be nice to document type syntax for defunml and to parse/print type expressions.
;;;;  *** and Define mapping to convert type expr returned by (ml) into something more recognizble.


;**************************************************************************
;*                                                                        *
;*      Projet     Formel                       LCF    Project            *
;*                                                                        *
;**************************************************************************
;*                                                                        *
;*            Inria                         University of Cambridge       *
;*      Domaine de Voluceau                   Computer Laboratory         *
;*      78150  Rocquencourt                    Cambridge CB2 3QG          *
;*            France                                England               *
;*                                                                        *
;**************************************************************************

; F-macro       Macros for the LCF system

(eval-when (compile load eval)
    ; expand a function call
    ;   (function f)    --->    (f arg1 ... argn)
    ;   others      --->    (funcall fun arg1 ... argn)
    (defun call-fun (fun args)
      (cond ((or (atom fun) (not (member (car fun) '(function quote))))
             `(funcall ,fun ,@args))
            (t `(,(cadr fun) ,@args)))) ; call-fun


)   ; eval-when

(eval-when (compile load)
  (proclaim '(special
	      token tokchs toktyp char cflag ptoken ptokchs ptoktyp pchar
	      parsedepth arg1 atom-rtn juxtlevel juxt-rtn msg1 msg2
	      tml-sym tokqt-sym arrow-sym sum-sym prod-sym null-sym
	      exfix-sym neg-sym list-sym div-sym plus-sym mns-sym conc-sym lt-sym gt-sym conj-sym
	      disj-sym lam-sym asgn-sym div-sym pp-sym
	      nulltyptok tokbearer toklbearer escape-sym mul-sym declnconstrs olinprec
	      spec-toks anticnr-tok else-tok metaprec sum-tok
	      arrow-tok prod-tok spec-syms rsvdwds eq-sym trap-syms
	      trap-then-sym trap-loop-sym trapif-then-sym  
	      trapif-loop-sym trapbind-then-sym trapbind-loop-sym 
	      bastypes else-sym condl-sym endcnr-sym
	      %skiplimit
	      start-string-sym end-string-sym string-bearer
	      nullty %it
	      )))

(defun syserror (x)
  (system-error (format-string "Error in LCF or ML system: ~a." x)))



;;;;	
;;;;	Each orb environment has own *global-env*.
;;;;	The default *global-env* is used when no environment is bound.
;;;;	When an environment is created it inherits current *global-env*.
;;;;	Thus all environment share tactics.
;;;;
;;;;	TODO: Thus all share system/tactic reference variables.
;;;;	TODO: Also, they share all symbol-plists (since symbol is unstamped ident?).

(defvar *global-env* nil)

(defun readlist (x) (read-from-string (string (implode-toks x))))
(defun catch-error (x &optional y) (declare (ignore y)) (list x))


; Print a constant string, computing length at compile-time
(defmacro ptoken (str)
  `(pstringlen (quote ,str) ,(length (princ-to-string str))))


(defmacro failwith (tok) `(breakout evaluation ,tok))


; fail with appended error message
(defmacro msg-failwith (tok . msgs)
   `(breakout evaluation (concat ,tok " -- " . ,msgs))
)                               ; msg-failwith


; fail if any of the error messages are not nil
(defmacro cond-failwith (tok . code)
    `(let ((msg (or . ,code)))
        (cond (msg (breakout evaluation (concat ,tok " -- " msg)))))
)                               ; cond-failwith

; Lisp error trapping 
(defmacro errortrap (errorfun . trycode)
   (let ((x (gen-temp-name)))
      `((lambda (,x)
           (cond ((atom ,x) ,(call-fun errorfun (list x)))
                 (t (car ,x))))
        (catch-error (progn . ,trycode))))
)               ; errortrap



; Apply the function to successive list elements
; and return the first non-nil value
; if none, return nil
(defmacro exists (fun . lists)
  (let ((vars (mapcar #'(lambda (ignore) (declare (ignore ignore)) (gen-temp-name)) lists)))
   (let ((inits (mapcar #'(lambda (v l) `(,v ,l (cdr ,v))) vars lists))
         (args (mapcar #'(lambda (v) `(car ,v)) vars)))
    `(do ,inits ((null ,(car vars)) nil)
          (cond (,(call-fun fun args) (return (list ,@args)))))))
)   ; exists



(defmacro forall (fun . lists)
  (let ((vars (mapcar #'(lambda (ignore) (declare (ignore ignore)) (gen-temp-name)) lists)))
   (let ((inits (mapcar #'(lambda (v l) `(,v ,l (cdr ,v))) vars lists))
         (args (mapcar #'(lambda (v) `(car ,v)) vars)))
    `(do ,inits ((null ,(car vars)) t)
          (cond (,(call-fun fun args)) ((return nil))))))
)   ; forall


;; returns symbol with _ -> and upcased alpha.
(defun direct-lisp-name (mlid)
  (intern (map 'string
	       #'(lambda (ch)
		   (if (char= #\_ ch)
		       #\-
		       (char-upcase ch)))
	       (string mlid))))


(defmacro defunml ((name args &key (error-wrap-p t) (declare nil) (direct nil)) type &body body)
  (let ((arity (length args))
	(lisp-name (intern (concatenate 'string "ML-" (string-upcase (string name))))))

    `(progn
      ,(if error-wrap-p
	   `(defun ,lisp-name ,(reverse args)
	     ,@declare
	     (with-handle-error (('(,name))
				 ,(if (eql t error-wrap-p)
				      `(breakout evaluation
						(flush-message '(,name)))
				      `(breakout evaluation ,error-wrap-p)))
	       ,@body))
	   `(defun ,lisp-name ,(reverse args)
	     ,@declare
	     ,@body))

      ,(when direct
	(let ((direct-lisp-name (if (eq direct t)
				    (direct-lisp-name name)
				    direct)))
	  `(defun ,direct-lisp-name ,args
	    (,lisp-name ,@(reverse args)))))

      (declare-ml-fun ',name ,arity ',lisp-name ',type)
      nil)))


(defmacro defmlsubst (name args type &body body)
  (let ((arity (length args))
	ml-name
	lisp-name)
    (if (symbolp name)
	(setf ml-name (setf lisp-name name))
	(setq ml-name (first name) lisp-name (second name)))
    `(progn
       (defun ,lisp-name ,(reverse args) ,@body)
       (declare-ml-fun ',ml-name ,arity ',lisp-name ',type))))

(defmacro dml (ml-fn n lisp-fn mty)
  `(declare-ml-fun ',ml-fn ',n ',lisp-fn ',mty))

(defmacro dmlc (id exp mty)
    `(declare-ml-const (quote ,id) (quote ,exp) (quote ,mty)))

;; Some of these functions are used at macro expansion time so must be present here.
(defvar *temp-symbol-counter* 0)

(defun gen-temp-name (&optional prefix)
  ;; Returns a symbol whose print name includes the print name of prefix if
  ;; provided.
  ;; The type of symbol returned by this function depends on the way ml
  ;; compilation is handled.  If the intermediate lisp code is written out to a
  ;; file then the symbol must be interned.  If not, then gensym suffices.
  (when (and prefix (symbolp prefix))
    (setf prefix (symbol-name prefix)))
  (make-symbol (concatenate 'string (if (null prefix) "ml-" prefix)
			      (format nil "~D" (incf *temp-symbol-counter*)))))

(defun gen-internal-name (name)
  (when (symbolp name) (setf name (symbol-name name)))
  (make-symbol name))

(defvar *timestamp* (format nil "~D" (mod (clock) 10000)))
(defvar *external-symbol-counter* (mod (clock) 127))

(defun gen-external-name (name)
  (when (symbolp name) (setf name (symbol-name name)))
  (intern (concatenate 'string name "%" (string *timestamp*) "%"
			 (format nil "~D" (incf *external-symbol-counter*)))
	 *ml-runtime-package*))

(defun gen-runtime-name (name)
  (when (symbolp name) (setf name (symbol-name name)))
  (intern name *ml-runtime-package*))



(defmacro prllambda (parameters &body body)
  `(function (lambda ,parameters ,@body)))

(defmacro dolists (iterators declarations value &body body)
  ;; A call to dolists is of the form
  ;;
  ;;  (dolists ((vari listi)*)
  ;;           (declaration*)
  ;;           result-form
  ;;     body)
  ;;
  ;; This expands into a do form, which iterates over each of the lists, with
  ;; vari being bound to successive elements of listi.  The auxiliary
  ;; declarations are simply added to the declaration list of the resulting do,
  ;; ie they can be anything acceptable as a do declaration.  The forms in body
  ;; are evaluated at each iteration.  The iteration is terminated when the end
  ;; of any of the listi is reached.  The result is the result of evaluating
  ;; result-form.
  (when (not (every (prllambda (x) (and (consp x) (= (length x) 2) (symbolp (first x))))
		  iterators))
    (error "Incorrect iterator list for dolists ~A" iterators))
  (let* ((list-names (mapcar #'(lambda (foo) (declare (ignore foo)) (gensym)) iterators))
	 (var-names (mapcar #'first iterators))
	 (list-initializers (mapcar (prllambda (name iterator) `(,name ,(second iterator)))
				    list-names
				    iterators))
	 (list-declarations (mapcar (prllambda (name) `(,name (cdr ,name) (cdr ,name)))
				    list-names))
	 (var-declarations (mapcar (prllambda (var-name list-name)
				     `(,var-name (car ,list-name) (car ,list-name)))
				   var-names
				   list-names))
	 (end-test `(not (and ,@list-names)))
	 (end-test-result-name (gensym)))
    `(let ,list-initializers
       (do ,(nconc
	      var-declarations
	      list-declarations
	      (list (list end-test-result-name end-test end-test))
	      ;; This must come last to avoid problems with destructive modification.
	      declarations)
	   (,end-test-result-name ,value)
	 ,@body))))

(defmacro hack-list (function list)
  (let ((var (gensym)))
    `(do ((,var ,list (cdr ,var)))
	 ((null ,var))
       (setf (car ,var) (funcall ,function (car ,var))))))



;;; F-descriptor
;;;
;;; Contains the definition of descriptors and various functions for dealing with
;;; them.

;;; The descriptors have the following form:
;;;
;;;    (id 'VALUE name)
;;;
;;;         The identifier id denotes a non function value.  The value is the
;;;         value of name.
;;;
;;;    (id 'FUNCTION name arity fname all-uses-full-appl
;;;                  params tail-recursion-happened)
;;;
;;;         The identifier id denotes an internal function.  The closure is the
;;;         value of name and the associated lisp function is the function value
;;;         of fname (or name if fname is nil).  The lisp function expects arity
;;;         arguments.  The all-uses-full-appl field is t if in every application
;;;         of this function there are arity arguments.  The following two fields
;;;         are for optimizing tail recursion.  The params field is the list of
;;;         parameter names for the function, and tail-recursion-happened is set
;;;         to true during the processing of the body if any tail recursive call
;;;         of this function was optimized.
;;;
;;;    (id 'ML-FUNCTION name arity)
;;;
;;;         The identifier id denotes a global function (ie one defined by let or
;;;         letrec at the top level).  The closure is the value of name, and the
;;;         associated lisp function is the function value of name.  The number
;;;         of arguments expected by the lisp function is given by arity.
;;;
;;;    (id 'PRIM-FUNCTION name arity fname)
;;;
;;;         The identifier id denotes a primitive function.  The closure is the
;;;         value of name and the associated lisp function is the function value
;;;         of fname.  The lisp function expects arity arguments.
;;;
;;;    (id 'ISOM)
;;;         The id is an isomorphism function for an abstract data type.

(defstruct (desc
		 (:type list))
  id
  kind
  name)

(defstruct (value (:include desc (kind 'VALUE)) (:type list))
)

(defstruct (ml-function
	     (:include desc (kind 'ML-FUNCTION))
	     (:type list)
	     )
  arity)

(defstruct (prim-function
	     (:include ml-function (kind 'PRIM-FUNCTION))
	     (:type list)
	     )
  fname)

(defstruct (aux-function (:include prim-function (kind 'function)) (:type list))
  (all-uses-full-appl t)
  params
  (tail-recursion-happened nil))

(defstruct (isom (:include desc (kind 'ISOM)) (:type list))
)

;;; Some useful access functions for descs

(defun is-ml-function (desc) (eql (desc-kind desc) 'FUNCTION))
(defun is-internal-function (desc)
  (eql (desc-kind desc) 'FUNCTION))
(defun is-function (desc)
  (member (desc-kind desc) '(FUNCTION PRIM-FUNCTION ML-FUNCTION)))
(defun is-isom (desc) (eql (desc-kind desc) 'ISOM))

(defun name-for-desc (desc)
  (desc-name desc))

(defun name-for-function (desc)
  (if (eql (desc-kind desc) 'ML-FUNCTION)
      (desc-name desc)
      (or (prim-function-fname desc) (desc-name desc))))

;;; Some functions to hide the details of constructing descs.

(defun mk-value-desc (id)
  (make-value :id id :name (gen-internal-name id)))

(defun mk-function-desc (id arity)
  (let ((name (gen-internal-name id)))
    (make-aux-function :id id :arity arity :name name :fname name)))

(defun mk-isom-desc (id)
  (make-isom :id id))




;;; Contains the definitions of the nodes that appear in the converted abstract
;;; syntax tree.  Note that as convert cannibalizes old list structure, the
;;; constructor functions implicitly defined herein aren't used.  Provided to
;;; help maintain the sanity of anyone trying to read the translation code.
;;; Not included here are definitions for node kinds that are removed in the
;;; conversion process, e.g. mk-binop or mk-straint.


(defstruct (node (:type list))
  kind)

(defmacro defnode (node-type kind &body fields)
  `(defstruct (,node-type
	       (:type list)
	       (:include node (kind ',kind)))
     ,@fields))

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      const-value var-desc mland-left mland-right mlor-left mlor-right
	      pair-left pair-right binop-op binop-left binop-right ml-list-list
	      assign-varstruct assign-value seq-for-effect seq-value failwith-exp
	      test-conditional test-else arm-test arm-exp else-exp trap-exp
	      trap-conditional trap-else trap-else-binding-id appn-fun appn-args
	      abstr-body abstr-params in-decl in-body decl-varstructs decl-values
	      )))



;;; <const>	: '(nil <value>)
;;;
;;; Any constant term.
;;; mk-boolconst, mk-intconst, mk-tokconst, and parsable abstract type extensions.

(defun const-value (n) (cadr n))


;;; <empty>     : '(<mk-empty>)
;;;
;;; The empty pair


;;; <fail>      : '(<mk-fail>)
;;;
;;; An untagged failure


;;; <var>       : '(<mk-var> <desc>)
;;;
;;; A variable reference.

(defun mk-var (desc) (list 'mk-var desc))

(defun var-desc (n) (cadr n))           ; The descriptor for this reference.


;;; <mland>     : '(<mk-and> <left> <right>)

(defun mland (l r) (list 'mk-and l r)) 

(defun mland-left (n) (cadr n))

(defun mland-right (n) (caddr n))


;;; <mlor>     : '(<mk-or> <left> <right>)

(defun mlor (l r) (list 'mk-or l r))

(defun mlor-left (n) (cadr n))

(defun mlor-right (n) (caddr n))

	 
;;; <pair>     : '(<mk-dupl> <left> <right>)
;;;
;;; A pair

(defun pair-left (n) (cadr n))          ; The tree for the left component.

(defun pair-right (n) (caddr n))        ; The tree for the right component.


;;; <binop>    : '(nil <op> <left> <right>)
;;;
;;; A binary operator.  Occurs only in varstructs.

(defun binop-op (n) (cadr n))

(defun binop-left (n) (caddr n))

(defun binop-right (n) (cadddr n))


;;; <ml-list>  : '(<mk-list> <list>)
;;;
;;; A list

(defun mk-ml-list (list) (list 'mk-list list))

(defun ml-list-list (n) (cadr n))       ; A list of trees denoting the values of the list.


;;; <assign>   : '(<mk-assign> <varstruct> <value>)
;;;
;;; An assignment

(defun assign-varstruct (n) (cadr n))   ; Varstruct for the assignment.

(defun assign-value (n) (caddr n))      ; Value to be assigned to the varstruct.


;;; <seq>       : '(<mk-seq> <for-effect> <value>)
;;;
;;; A sequence of commands

(defun seq-for-effect (n) (cadr n))     ; A list of trees denoting commands that are
                                        ; purely for effect.

(defun seq-value (n) (caddr n))         ; The tree denoting the value to be returned.


;;; <failwith>   : '(<mk-failwith> <exp>)
;;;
;;; A failure

(defun failwith-exp (n) (cadr n))       ; The tree denoting the expression to be thrown.


;;; <test>       : '(<mk-test> <conditional> <else>)
;;;
;;; A conditional expression

(defun mk-test (cond else) (list 'mk-test cond else))

(defun test-conditional (n) (cadr n))   ; A list of arms

(defun test-else (n) (caddr n))         ; An else expression


;;; <arm>        : '(nil <test> <exp>)
;;;
;;; An arm of a conditional expression.
;;; ONCE or ITER

(defun mk-arm (kind test exp) (list kind test exp))

(defun arm-test (n) (cadr n))           ; A tree denoting a boolean value

(defun arm-exp (n) (caddr n))           ; The tree denoting the expression to be executed if
			                ; test evaluates to true.


;;; <else>       : '(nil <exp>)
;;;
;;; The default for a conditional expression.
;;; ONCE, ITER or EMPTY

(defun mk-else (kind expr) (list kind expr))

(defun else-exp (n) (cadr n))           ; The expression to be evaluated.


;;; <trap>       : '(mk-trap> <exp> <conditional> <else> <else-binding-id>)
;;;
;;; A trap

(defun mk-trap (exp cond else else-binding)
  (list 'mk-trap exp cond else else-binding))

(defun trap-exp (n) (cadr n))           ; Denotes the expression during whose evaluation
			                ; failures are to be trapped.

(defun trap-conditional (n) (caddr n))  ; A list of arms.  The test of each arm
			                ; denotes a list of tokens.

(defun trap-else (n) (cadddr n))        ; An else expression.

(defun trap-else-binding-id (n)         ; If non nil, an identifier to  be bound to			                
  (cadr (cdddr n)))                     ; the tag thrown by the failure during the
			                ; evaluation of the else expression.


;;; <appn>        : '(<mk-appn> <fun> <args>)
;;;
;;; An application

(defun mk-appn (fun args) (list 'mk-appn fun args))

(defun appn-fun (n) (cadr n))           ; The function to apply.

(defun appn-args (n) (caddr n))         ; The arguments of the application.



;;; <abstr>       : '(<mk-abstr> <body> <params>)
;;;
;;; An abstraction

(defun mk-abstr (body parms) (list 'mk-abstr body parms))

(defun abstr-body (n) (cadr n))         ; Denotes the body of the abstraction

(defun abstr-params (n) (caddr n))      ; A list of varstructs for the parameters of the
			                ; abstraction.


;;; <in>          : '(<mk-in> <decl> <body>)
;;;
;;; A local declaration

(defun mk-in (decl body) (list 'mk-in decl body))

(defun in-decl (n) (cadr n))            ; Denotes the declaration

(defun in-body (n) (caddr n))           ; Denotes the expression to be evaluated in the scope
                                        ; of the declaration


;;; <decl>        : '(nil <varstructs> <values>)
;;;
;;; A declaration
;;; mk-let, mk-letrec, or mk-letref

(defun mk-decl (kind vars vals) (list kind vars vals))

(defun decl-varstructs (n) (cadr n))    ; A list of varstructs corresponding to decl-values

(defun decl-values (n) (caddr n))       ; a list of values



;;; Macro definitions needed for the runtime system.
;;;
(defun make-closure (f arity)
  (list (cons f arity)))

(defmacro make-ml-closure (vars body)
  `(make-closure (function (lambda ,(reverse `,vars) ,@body)) ,(length `,vars)))

(defmacro update-closure (closure &rest args)
  (let ((cname (gensym)))
    `(let ((,cname ,closure))
       (list* (first ,cname) ,@args (cdr ,cname)))))



;; TODO PERF fast-length
(defmacro ap (f-cl &body args)
  (case (length args)
    ((1)
     `(one-ap ,f-cl ,@args))
    ((2)
     `(two-ap ,f-cl ,@args))
    ((3)
     `(three-ap ,f-cl ,@args))
    (otherwise
     `(general-ap ,f-cl ,@args))))





;;;;
;;;;	Defining New Abstract Types :
;;;;


(defvar *abstract-types* nil)			; alist keyed by "mk-"<type-name>, contains type-name.
(defvar *upcase-abstract-types* nil)
(defvar *abstract-type-print-funcs* nil)

(defvar *abstract-type-parse-funcs* nil)	; list of conses of parse-funcs and bearers.
(defvar *abstract-type-eq-funcs* nil)		; list of conses of memberp and equalp funcs.

(defvar *abstract-type-constant-bearer-assoc* nil)
(defvar *abstract-type-constant-mkid-assoc* nil)

;; kludge to allow variable parse, ie `v'.
(defvar *abstract-type-tok-parse-hook* nil)

(defun reset-defined-ml-primitive-types ()
  (setf *abstract-types* nil)
  (setf *upcase-abstract-types* nil)
  (setf *abstract-type-print-funcs* nil)

  (setf *abstract-type-parse-funcs* nil)
  (setf *abstract-type-eq-funcs* nil)	

  (setf *abstract-type-constant-bearer-assoc* nil)
  (setf *abstract-type-constant-mkid-assoc* nil)

  (values))
  


(defun add-primitive-type (type-name print-func  &key
				     parse-func member-p (eq-func #'eql))
  (let* ((type-string (string type-name))
	 (upcase-type (string-upcase type-string))
	 (mk (intern (string-upcase (concatenate 'string "mk-" type-string)))))

    (setf *abstract-types* (acons (list mk) (list type-name) *abstract-types*))
    (setf *upcase-abstract-types* (acons (list mk) (intern upcase-type) *upcase-abstract-types*))
    (setf *abstract-type-print-funcs* (acons mk print-func *abstract-type-print-funcs*))
    (when parse-func
      (let ((bearer-symbol (intern (concatenate 'string "<" upcase-type ">")))
	    (const (intern (string-upcase (concatenate 'string "mk-" type-string "-const")))))
	
	(push (cons parse-func bearer-symbol) *abstract-type-parse-funcs*)
	(setf *abstract-type-constant-bearer-assoc*
	      (acons bearer-symbol const
		     *abstract-type-constant-bearer-assoc*))
	(setf *abstract-type-constant-mkid-assoc*
	      (acons const (list mk)
		     *abstract-type-constant-mkid-assoc*))))
    (when member-p
      (setf *abstract-type-eq-funcs* (cons (cons member-p eq-func) *abstract-type-eq-funcs*)))))

(defun add-tok-parse-hook (type-name parse-func)
  (let* ((type-string (string type-name))
	 (bearer-symbol (intern (string-upcase (concatenate 'string "<" (string-upcase type-string) ">"))))
	 (const (intern (string-upcase (concatenate 'string "mk-" type-string "-const")))))
    (setf *abstract-type-tok-parse-hook*  (cons parse-func bearer-symbol))
    (setf *abstract-type-constant-bearer-assoc*
	  (acons bearer-symbol const
		 *abstract-type-constant-bearer-assoc*))
    (setf *abstract-type-constant-mkid-assoc*
	  (acons const (list (intern (string-upcase (concatenate 'string "mk-" type-string))))
		 *abstract-type-constant-mkid-assoc*))))
    

(defun abstract-type-name-p (sym)
  (member sym *abstract-types* :key #'cadr))

(defun abstract-type-mkid-p (sym)
  (member sym *abstract-types* :key #'caar))

(defun mkid-to-abstract-type-name-list (mk)
  (let ((name (or (assoc mk *abstract-types* :key #'car)
		  (assoc (node-kind (constant-to-abstract-type-mkid mk))
			 *abstract-types* :key #'car))))
    (cdr name)))

(defun mkid-to-abstract-type-name (mk)
  (let ((name (assoc mk *abstract-types* :key #'car)))
    (cadr name)))

(defun insensitive-name-to-abstract-type-mkid (name)
  (let ((mk (or (rassoc name *abstract-types* :key #'car)
		(rassoc name *upcase-abstract-types*))))
    (car mk)))

(defun name-to-abstract-type-mkid (name)
  (let ((mk (rassoc name *abstract-types* :key #'car)))
    (car mk)))

(defun constant-to-abstract-type-mkid (const)
  (cdr (assoc const *abstract-type-constant-mkid-assoc*)))

(defun bearer-to-abstract-type-constant (bearer)
  (cdr (assoc bearer *abstract-type-constant-bearer-assoc*)))

(defun abstract-type-pop-bearer (bearer)
  (let ((x (get bearer 'bearer-value)))
    (setf (get bearer 'bearer-value) (cdr x))
    (car x)))

(defun abstract-type-constant-p (const)
  (member const *abstract-type-constant-mkid-assoc* :key #'car))


(defun initlean-abstract-type-bearers ()
  (mapc #'(lambda (parse) (setf (get (cdr parse) 'bearer-value) nil))
	*abstract-type-parse-funcs*)
  (when *abstract-type-tok-parse-hook*
    (setf (get (cdr *abstract-type-tok-parse-hook*) 'bearer-value) nil)))


(defun abstract-parse-func (ch)
  (mapc #'(lambda (parse)
	    (mlet* (((parsep value) (funcall (car parse) ch)))
		   (when parsep
		     (let ((bearer (cdr parse)))
		       (setf (get bearer 'bearer-value)
			     (append (get bearer 'bearer-value) (list value)))
		       (return-from abstract-parse-func bearer)))))
	*abstract-type-parse-funcs*)
  nil)
		       
(defun abstract-parse-hook (ch tokchs)
  (when *abstract-type-tok-parse-hook*
    (mlet* (((parsep value) (when *abstract-type-tok-parse-hook*
			      (funcall (car *abstract-type-tok-parse-hook*) ch tokchs))))
	   (when parsep
	     (let ((bearer (cdr *abstract-type-tok-parse-hook*)))
	       (setf (get bearer 'bearer-value)
		     (append (get bearer 'bearer-value)
			     (list value)))
	       bearer)))))


(defun defined-abstract-type-equal-p (x y)
  (dolist (eq-func *abstract-type-eq-funcs*)
    (when (funcall (car eq-func) x)
      (return-from defined-abstract-type-equal-p (funcall (cdr eq-func) x y))))
  nil)



(defun print-abstract-type-instance (mk instance)
  (let ((print-f (cdr (assoc mk *abstract-type-print-funcs*))))
    (when print-f
      (funcall print-f instance))))
  




(eval-when (compile load)
  (proclaim '(special
	      lparen rparen
	      period comma colon scolon lbrkt rbrkt cmntchr
	      )))
	    


(setq cmntchr '%)
(setq lparen '|(|)
(setq rparen '|)|)
(setq period '|.|)
(setq comma '|,|)
(setq colon '|:|)
(setq scolon '|;|)
(setq lbrkt '|[|)
(setq rbrkt '|]|)


;;;;	
;;;;	one place to hold all attributes of ml def.
;;;;	
;;;;	code-table : for code from objects.
;;;;	global-env table :  keyed on id possibly multiple values.
;;;;	  - method of finding def given id (ie get-mltype)
;;;;	dynamic during compile.
;;;;	
;;;;	external-name - <system>-ML-RUNTIME::
;;;;	  - would like to avoid defining external name until eval time.
;;;;	  - might be possible to use unamed lambdas  (compile nil (lambda (x) ...))
;;;;	      * would be interesting to have be able to try both ways to test
;;;;		perf.
;;;;	      * precludes overwriting loads but if perf win may be worth the loss.
;;;;	
;;;;	  
;;;;	ml-definitions will be found in :
;;;;	  - *current-ml-definitions*	: dynamic binding during compilation   
;;;;	  - *global-ml-definiton-table*	: global hash-table
;;;;	  - *global-lookupper*		: hook to look in visible code defs.
;;;;	      * code-defs visible may be limited by ref-envs.
;;;;	  
;;;;	  
;;;;   when loading compiled files need to have expression which adds definitions.
;;;;   should be similar to update of global-env currently.

(defstruct ml-definition
  (id nil)				; system symbol
  (description nil)
  (type nil)
  (refp nil)
  (parse-properties nil)

  (external-counter 0)			; *external-symbol-counter* at def time.
  (external-name nil)			; runtime symbol %iii%jjj%
  (value nil)				; function
  (closure nil)

  (source nil)				; or maybe filename? or token (primitive/builtin).
  )


(defun set-mldef-refp (d)
  (setf (ml-definition-refp d) t))

(defun set-mldef-external-counter (d)
  (setf (ml-definition-external-counter d)  (incf *external-symbol-counter*)))

(defun get-mldef-external-name (d)
  (or (ml-definition-external-name d)
      (setf (ml-definition-external-name d)
	    (intern (format nil "~a%~D%~D" (ml-definition-id d)  *timestamp* (ml-definition-external-counter d))
		    *ml-runtime-package*))))

(defun set-mldef-external-name (d n)
  (setf (ml-definition-external-name d) n))

(defun set-mldef-type (d type)
  (setf (ml-definition-type d) type))

(defun set-mldef-value (d val)
  (setf (ml-definition-value d) val))

(defun set-mldef-closure (d c)
  (setf (ml-definition-closure d) c))

(defun set-mldef-source (d o)
  (setf (ml-definition-source d) o))

(defun set-mldef-description (d desc)
  (setf (ml-definition-description d) desc))

(defun mldef-refp (d) (ml-definition-refp d))
(defun id-of-mldef (d) (ml-definition-id d))
(defun source-of-mldef (d) (ml-definition-source d))
(defun type-of-mldef (d) (ml-definition-type d))
(defun parse-properties-of-mldef (d) (ml-definition-parse-properties d))
(defun description-of-mldef (d) (ml-definition-description d))

;;;;	  
;;;;	During compile we need to add new defs but also want to record
;;;;	  which defs accessed during compilation.
;;;;	  


(defvar *global-lookupper* nil)

(defvar *current-ml-definitions*)
(defvar *current-ml-definitions-stamp* nil)

(defvar *global-ml-definiton-table* (make-hash-table))

;;;;	
;;;;	Desire ability to lift file definitions into object table.
;;;;	  - lift_file_defs : string{fname} list -> unit
;;;;	  - finds all ids in global-ml-definiton-table defined by
;;;;	    listed files and adds them to *current-ml-definitions*
;;;;	
;;;;	
;;;;	

(defmacro with-ml-definitions ((init) &body body)
  `(let ((*current-ml-definitions* ,init)
	 (*current-ml-definitions-stamp* (new-transaction-stamp)))
    ,@body))

(defvar *universal-lookupper* #'(lambda (&rest r) nil))

(defmacro with-global-lookupper ( &body body)
  `(let ((*global-lookupper* (funcall *universal-lookupper*))
	 ;;(*global-lookups* nil)
	 )
    ,@body))

(defun get-current-ml-definitions ()
  *current-ml-definitions*)

;; TODO : recompile tactic code does not expose new versions.
;;   ie acts as though not recompiled. The global-lookupper appears to
;;   be central.
(defun find-mldef (id &optional test)
  ;;(when (eql '|is_all_term| id) (break "fm"))
  (let ((def (or (when (boundp '*current-ml-definitions*)
		   (cdr (assoc id *current-ml-definitions*)))
		 (when (and *global-lookupper*
			    (and (not (eql '|create_rec_module_aux| id))
				 (not (eql '|create_union_aux| id))
				 (not (eql '|obacc_add_wrechain| id)))
			    )
		   (funcall *global-lookupper* id))
		 (let ((l (gethash id *global-ml-definiton-table*)))
		   (or (when test (find-if test l))
		       (car l))))))
    ;;(add-global-lookup id def)
    def))

(defun find-mldef-pp (id)
  (find-mldef id #'(lambda (d) (eql 'parse (source-of-mldef d)))))


(defun new-mldefs (defs)
  ;;(break "nmds")
  (dolist (d defs)
    (let ((id (id-of-mldef d)))
      (setf (gethash id *global-ml-definiton-table*)
	    (cons d (gethash id *global-ml-definiton-table*))))))

(defvar lang1 'ml1)
(defvar lang2 'ml2)
(defvar langlp 'mllp)

(defstruct ml-parse-properties
  (mllp nil)
  (ml1 nil)
  (ml2 nil))

(defun mllp-of-parse-properties (p) (ml-parse-properties-mllp p))
(defun ml1-of-parse-properties (p) (ml-parse-properties-ml1 p))
(defun ml2-of-parse-properties (p) (ml-parse-properties-ml2 p))
(defun new-ml-parse-properties (llp l1 l2)
  (make-ml-parse-properties :mllp llp :ml1 l1 :ml2 l2))

(defvar *equal-parse-properties* nil)
(defvar *period-parse-properties* nil)
(defvar *semicolon-parse-properties* nil)

(defun add-global-mldef (d)
  ;;(break)
  (let ((id (id-of-mldef d)))
    (let ((cur (gethash id *global-ml-definiton-table*)))
      ;;(when cur (setf -cur cur) (break "new-mldef"))
      (setf (gethash id *global-ml-definiton-table*)
	    (cons d cur)))
    d))


(defun new-mldef (d)
  ;;(break "nmd")
  ;;(format t "~%new-mldef ~a " (id-of-mldef d))

    (if (boundp '*current-ml-definitions*)
	(let ((id (id-of-mldef d)))
	  (setf *current-ml-definitions* (cons (cons id d) *current-ml-definitions*))
	  (unless (source-of-mldef d) (set-mldef-source d *current-ml-definitions-stamp*))
	  d)
	(add-global-mldef d)
	))

;; infix proclaimation comes prior to definiton.
(defun maybe-new-mldef (id &optional check-globals-p)
  (let ((def (if (boundp '*current-ml-definitions*)
		 ;; if last current def for id is infix then use infix.
		 (let ((d (cdr (assoc id *current-ml-definitions*))))
		   (when (and d (parse-properties-of-mldef d))
		     d))
		 (when check-globals-p
		   (find-if #'(lambda (d) (eql 'parse (source-of-mldef d)))
			    (gethash id *global-ml-definiton-table*))))))

    (or def
	(new-mldef (make-ml-definition :id id :source *current-ml-definitions-stamp*)))))

(defun new-mldef-pp (id llp l1 l2)
  (new-mldef (make-ml-definition :id id
				 :parse-properties (new-ml-parse-properties llp l1 l2)
				 :source 'parse)))
  

(defvar *last-val-mldef*)
(defvar lastvalname '|it|)

;; overwriting_load
;; lastvalname  / *last-val-mldef*
;; mlval
;; how do mldefs get saved.
;;  ml-eval-check ??
