
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      ml-val ml-ast ml-type ml-fail ml-print
	      )))

(defun init-mldefs ()
   
  (new-mldefs
   (list
    (setf *last-val-mldef*
	  (make-ml-definition :id lastvalname
			      :description (make-value :id '|it| :name '%it)
			      :type nullty
			      ))
    (make-ml-definition :id '|nil|
			:description (make-value :id '|nil| :name nil)
			:type '(mk-listyp *)
			)
    (make-ml-definition :id '|%&|
			:type (makety '((bool |#| bool) -> bool))
			)
    (make-ml-definition :id '|%or|
			:type (makety '((bool |#| bool) -> bool))
			)
    )))

(init-mldefs)

(defun reset-ml ()
  (setq initial%load nil)
  (setq eof '$eof$)
  (setq %mlprindepth 3)
  )


(defmacro with-ml-evaluation (tag &body body)
  `(let ((ok nil))
    (prog1
	(let ((result (tag evaluation
			   (prog1 (progn ,@body)
			     (setf ok t)))))
	  ;; if fell through then error.
	  (if ok
	      result
	      (raise-error (error-message '(ml evaluation ,tag) result)))))))
  


(defvar *ml-val*)
(defvar *ml-ast*)
(defvar *ml-type*)
(defvar *ml-print*)
(defvar *ml-display-p*)

(defun ml-val () *ml-val*)
(defun ml-ast () *ml-ast*)
(defun ml-type () *ml-type*)
(defun ml-fail () *ml-fail*)

(defun ml-print ()
  (when (messages-p)
    (flush-message (if *ml-display-p* nil 'ml))))
     


(defmacro with-ml ((display-p) &body body)
  `(let ((*ml-val* nil)
	 (*ml-ast* nil)
	 (*ml-type* nil)
	 (*ml-display-p* ,display-p)
	 (*ml-fail* nil)
	 (prflag ,(and display-p t)))

    (if *ml-display-p*
	(with-message-accumulator ('(ml))
	  ;;(break "wmlm")
	  ,@body)
	(with-message-accumulator ('())
	  ;;(break "wmln")
	  ,@body)
	)) )
  

(defun ml-scan (next eofp addrf &optional (evalp t))
  (let ((*print-base* 10.)
	(*read-base*  10.)
	(okay nil))

    (with-ml-scanner (next eofp addrf)
      (tag tml-error-tag
	   (tag eoftag
		(tag tmltag
		     (with-global-lookupper
			 (prl-mlloop evalp)))
		(setq okay t)))
	
      (unless okay
	(raise-error *ml-fail*)))))


(defvar *intermediate-lisp*)

(defun prl-mlloop (&optional (evalp t))
  ;;(break "pml")
  (do () (nil nil)
    (tag tmllooptag
	 	 
	 ;;--         (and prflag (top%f) (llterpri))   ;-- not so many ml-lf's
	 (let ((%thisdec nil)
	       (%thistydec nil)
	       (%compfns nil)
	       %pt %ty %pr %head %val
	       (output-list nil)
	       (*ml-will-eval-p* evalp))

	   (initlean)
	   (with-appns (prl-okaypass 'parse))
	   (setq %head (car %pt))
	   (setf *ml-ast* %pt)
	   (if (istmlop)
	       (when evalp (prl-okaypass 'evtmlop))
	       (progn
		 (prl-okaypass 'typecheck)
		 (prl-okaypass 'translation)
		 (setf *intermediate-lisp* %pr)
		 (setf *ml-type* %ty)
		 ;;(break "mlloop")
		 (if evalp
		     (let ((init-time (runtimems)))
		       (updatemldefs) ;; inits ml-definition structures.
		       (updatetypes)
		       (prl-okaypass 'evaluation)
		       ;; rle todo somehow parameterize bml so the term-p is not required here.
		       ;; rle todo otherwise bml depends on trm.
		       (when (and (eql *ml-display-p* 'not-term)
				  (term-p %val))
			 (setf prflag nil))
		       (setf *ml-val* %val)
		       (let ((final-time (runtimems)))
			 (updatevalues)
			 (when (if (eql *ml-display-p* 'not-term)
				   (not (term-p %val))
				   *ml-display-p*)
			   (printresults))
			 (when *ml-display-p*
  			   (printtime final-time init-time) )) )
		     (progn  (updatetypedecs)
			     (updatevalues)
			     (vml-printresults)))))
	   (when *ml-display-p*
	     (message-emit
	      (basic-message '(print) (nreverse output-list)))
	     ;;(break "prlloop")
	     )
	   )))
     ;;(breakout tmltag nil)
     )


;;;;	
;;;;	one place to hold all attributes of ml def.
;;;;	
;;;;	code-table
;;;;	global-env table
;;;;	dynamic during compile.
;;;;	
;;;;	external-name has value which is ml-closure ie function and arity.
;;;;	  - would like to avoid defining external name until eval time.
;;;;	  - might be possible to use unamed lambdas  (compile nil (lambda (x) ...))
;;;;	      * would be interesting to have be able to try both ways to test
;;;;		perf.
;;;;	  

;;; modified version of prl-okaypass in F-tml.l
(defun prl-okaypass (pass)
  (tag ok				; prog/return does not work in Franzlisp
       (let ((b (case pass
		  (parse (tag parse (setf -pt (setq %pt (parseml0))) (breakout ok t)))
		  (typecheck (tag typecheck
				  (setq %ty (typechpt)) (breakout ok t)))
		  (translation (tag translation
				    (setf -pr (setq %pr (tranpt))) (breakout ok t)))
		  (evaluation (tag evaluation
				   (setq %val (evalpr)) (breakout ok t)))
		  (evtmlop (tag evaluation
				(setq %val (evtmlop %pt)) (breakout ok t)))
		  (otherwise (syserror (cons pass '(unknown pass)))))))

	 ;; Fall through to here if pass failed
	 (llterpri)
	 (when (eql pass 'parse)
	   (llprinc (format-string "Scan address: ~a" (ml-line-count))))
	 (setf *ml-fail*
	       (basic-message (list 'error 'ml pass) (cons b (nreverse output-list))))
    
	 (setq %it nil)
	 (last-val-reset)
	 (breakout tml-error-tag nil))))


(defun ml (nextf eoff addrf &optional (displayp t))
  (with-ml (displayp)
    ;;(setf -d displayp)  (break "ml")
    (ml-scan nextf eoff addrf)
    (values (ml-val)
	    (ml-type)
	    (ml-print))))


(defun text-call-ml (ml text displayp)
  (mlet* (((nextf eoff addrf) (make-text-scanner (cons text " ;;"))))
	 (funcall ml
		  nextf eoff
		  ;; skip car for (cons text ";;")
		  #'(lambda ()
		      (let ((addr (funcall addrf)))
			(list* (car addr) (cadr addr) (cdddr addr))))
		  displayp)))


(defun ml-text (text &optional (displayp t))
  (text-call-ml #'ml text displayp))



(defun vml (nextf eoff addrf &optional (displayp t))
  (with-ml (displayp)
    (ml-scan nextf eoff addrf nil)
    (values (ml-type)
	    (ml-print))))

(defun vml-text (text &optional (displayp t))
  (text-call-ml #'vml text displayp))



(defun ast-ml (nextf eoff addrf &optional (displayp t))
  (with-ml (displayp)
    (ml-scan nextf eoff addrf nil)
    (values (ml-ast)
	    (ml-type)
	    (ml-print))))

(defun ast-ml-text (text &optional (displayp t))
  (text-call-ml #'ast-ml text displayp))


(defvar *ml-running-p* nil)
(defvar *ml-prompt-tag* nil)

(defunml (|ml_exit| (unit)
		    :error-wrap-p nil
		    :declare ((declare (ignore unit))))
    (void -> void)

  (setf *ml-running-p* nil)
  (values))



;;;;	RLE TODO : needs to multiplex with stream input so as not to let
;;;;	RLE TODO : lib broadcasts back up.
;;;;	RLE TODO : ie need to not block with read-line



;; for not standard-character-string
(defun break-char-p (ch)
  (member ch '(#\newline #\return #\linefeed) :test #'char=))

(defun ml-get-line (prompt firstp)
  (unless (listen)
    (princ prompt))
  
  (let ((line (do ((line (read-line)
			 (progn (unless (listen) (if firstp (setf firstp nil) (princ prompt)))
				(read-line))))
		  ((not (= 0 (length line))) line))))
    (unless (and (= 1 (length line))
		 (break-char-p (char line 0)))
      line)))

(defun whitespace-char-p (ch)
  (member ch '(#\newline #\return #\linefeed #\tab #\space) :test #'char=))


;;;;	RLE TODO : tabs in input mess up remote ml compile.

(defun ml-get-lines (firstp)
  (let ((prompt (if *ml-prompt-tag*
		    (format-string "ML[~a]> " *ml-prompt-tag*)
		    "ML> ")))
  (do* ((line (ml-get-line prompt firstp)
	      (ml-get-line (blank-string (length prompt))
				 nil))
	(ml-text (list line) (progn (push #\newline ml-text) (push line ml-text))))
       ((when (not (listen))
	  (or (null line)
	      (and line
		   ;; index of last non-whitespace character.
		   (let ((index (do ((i (1- (length line)) (1- i)))
				    ((or (< i 0)
					 (not (whitespace-char-p (char line i))))
				     i))))
		     (and (> index 0)
			  (or (and (char= #\; (char line index))
				   (char= #\; (char line (1- index))))
			      (when (and (null (cdr ml-text))
					 (char= #\. (char line index))
					 (not (char= #\\ (char line (1- index))))
					 (let ((p t))
					   (dotimes (i (1- index))
					     (when (whitespace-char-p (char line i))
					       (setf p t)))
					   p))
				(setf ml-text
				      (intern-nuprl (string-upcase (subseq line 0 index))))
				t)))))))
			 
	(if (symbolp ml-text)
	    ml-text
	    (mapcar #'(lambda (item)
			(if (stringp item)
			    (string-to-standard-character-string item)
			    item))
		    (nreverse ml-text)))))))
		       

(defun ml-toploop (&optional (evaluator #'ml-text))

  (banner)

  (let ((*ml-running-p* t)
	;; It's weird, but it works. Prevents ML> ML> prompt at start.
	(firstp t))
    (do () ((null *ml-running-p*))
      (with-handle-error (('(ml toploop)) ())
	(terpri)
	(let ((input (ml-get-lines firstp)))
	  (setf firstp nil)
	  (when input
	    (message-emit
	     (mlet* (((value type output) (funcall evaluator input)
		      (declare (ignore value type))))
		    output)))))
	
      (when (messages-p)
	(terpri)
	(mapcar #'print-message (messages-flush))))))



(defmacro inml (&body body)
  `(prog1 (with-handle-error (('(ml top)) ())
	    (mlet* (((value type output) (progn ,@body)
		     (declare (ignore type))))
	     (when output
	       (message-emit output))
	     value))

    (when (messages-p)
      (mapcar #'print-message (messages-flush)))))




(defunml (|standard_binary_directories| (unit)
		    :error-wrap-p nil
		    :declare ((declare (ignore unit))))
    (void -> string)

  (let ((s (nuprl-bin-path-list)))
    (when (or (null s)
	      (string= s ""))
      (nuprl-error (error-message '(standard-binary-directory))))

    s))



(defunml (|tty_print| (msg))
    (string -> void)

  (format t "~a~%" msg)
  nil)

(defunml (|tty_string| (msg))
    (string -> void)

  (format t "~a" msg)
  nil)


(defun fail-ml (tok)
  (breakout evaluation tok))
