
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


; object_address_of_abstraction_of_term : term -> object_address
; abstraction_of_term_p : term -> bool

; destruct_assumption : declaration -> variable # term
; make_assumption : variable -> term -> declaration






;;;; -docs- (mod mlt ml)
;;;;
;;;;	Terms:
;;;;
;;;;	The parameter value structures are complicated. Some of this
;;;;	complication is visible in the ML term functions. When writing tactics
;;;;	one should assume real values for parameter values, and let the
;;;;	implementation fail if the assumption is violated in any meaningful way.
;;;;	When writing ML functions to manipulate possibly non-real valued
;;;;	parameter values the full ugliness of the parameter structure is
;;;;	available (see edit parameters below).
;;;;
;;;;	As bindings are parameter values, the ML primitive type variable may
;;;;	have values which are not real variable values.  Some functions will
;;;;	fail if a member of the variable type is not a real value and is used in
;;;;	a manner requiring real values, eg variable_to_string fails if value is
;;;;	actually a slot. However, most functions will work in the presence of
;;;;	non real valued bindings, eg make_term and destruct_term.
;;;;
;;;;
;;;;	Term :
;;;;
;;;;	make_term : (tok # (parameter list)) -> (variable list # term) list -> term
;;;;	maybe_make_term :
;;;;	  term -> ((tok # (parameter list)) -> (variable list # term) list -> term)
;;;;	destruct_term : term -> (tok # parameter list) # ((variable list # term) list)
;;;;
;;;;	make_term_using_parameter_bindings :
;;;;	  (tok # parameter list) -> (parameter list # term) list -> term
;;;;	destruct_term_using_parameter_bindings :
;;;;	  term -> tok # parameter list # (parameter list # term) list
;;;;
;;;;	operator_of_term : term -> tok # parameter list
;;;;	bound_terms_of_term : term -> (variable list # term) list
;;;;
;;;;	equal_operators_of_terms : term -> term -> bool
;;;;
;;;;	operator_id_of_term : term -> tok
;;;;	parameters_of_term : term -> parameter list
;;;;	arity_of_term : term -> int list
;;;;	level_variables_of_term : term -> tok list
;;;;
;;;;	subterm_of_term : term -> int -> term
;;;;	bindings_of_subterm_of_term : term -> int -> variable list
;;;;
;;;;	parameter_of_term : term -> int -> parameter
;;;;	natural_parameter_of_term : term -> int -> int
;;;;	token_parameter_of_term : term -> int -> tok
;;;;	variable_parameter_of_term : term -> int -> variable
;;;;
;;;;  -page-
;;;;
;;;;	Edit parameters : slot, error or nil
;;;;
;;;;	make_edit_parameter : (tok -> (tok -> (tok -> (string -> parameter))))
;;;;	 ** first tok is edit-type, ie `SLOT` `ERROR` or `NIL`
;;;;	 ** second tok is meta-type, ie `ABSTRACTION` or `DISPLAY`
;;;;	 ** third tok is parameter type.
;;;;	 ** string is parameter value.
;;;;	destruct_edit_parameter : (parameter -> (tok # (tok # (tok # string))))
;;;;	edit_type_of_parameter : (parameter -> tok)
;;;;	meta_type_of_parameter : (parameter -> tok)
;;;;	type_of_parameter : parameter -> tok
;;;;	string_of_parameter : (parameter -> string)
;;;;
;;;;	pretty_string_of_parameter : (parameter -> string)
;;;;	 ** may be missing escapes, Not suitable for rebuilding parameter value.
;;;;
;;;;	descriptor_of_slot_parameter : (parameter -> string)
;;;;	 ** Descriptors of quite ephemeral and thus not normally available.
;;;;
;;;;  -page-
;;;;
;;;;	When manipulating non-real valued parameters, the preceding edit parameter functions
;;;;	should be used. The following meta-parameter functions are included for compatability. 
;;;;
;;;;	make_meta_parameter : tok -> variable -> parameter
;;;;	meta_parameter_p : parameter -> bool
;;;;	destruct_meta_parameter : parameter -> variable
;;;;
;;;;	make_abstraction_meta_variable : tok -> variable
;;;;	make_display_meta_variable : tok -> variable
;;;;	meta_variable_type : variable -> tok
;;;;
;;;;  -page-
;;;;
;;;;	Parameters :
;;;;
;;;;	equal_parameters : parameter -> parameter
;;;;
;;;;	destruct_natural_parameter : parameter -> int
;;;;	make_natural_parameter : int -> parameter
;;;;
;;;;	destruct_token_parameter : parameter -> tok
;;;;	make_token_parameter : tok -> parameter
;;;;
;;;;
;;;;	string_to_variable : string -> variable
;;;;	variable_to_string : variable -> string
;;;;
;;;;	tok_to_variable : tok -> variable
;;;;	variable_to_tok : variable -> tok
;;;;	destruct_variable_parameter : parameter -> variable
;;;;	make_variable_parameter : variable -> parameter
;;;;
;;;;	make_level_expression_parameter : level_expression -> parameter
;;;;	level_expression_parameter : parameter -> level_expression
;;;;	make_max_level_expression : level_expression list -> level_expression
;;;;	max_level_expression_p : level_expression -> bool
;;;;	destruct_max_level_expression : level_expression -> level_expression list
;;;;	make_increment_level_expression : level_expression -> int -> level_expression
;;;;	increment_level_expression_p : level_expression -> bool
;;;;	destruct_increment_level_expression : level_expression -> level_expression # int
;;;;	make_constant_level_expression : int -> level_expression
;;;;	constant_level_expression_p : level_expression -> bool
;;;;	destruct_constant_level_expression : level_expression -> int
;;;;	make_variable_level_expression : tok -> level_expression
;;;;	variable_level_expression_p : level_expression -> bool
;;;;	destruct_variable_level_expression : level_expression -> tok
;;;;	substitute_in_level_expression :
;;;;	level_expression -> (tok # level_expression) list -> level_expression
;;;;
;;;;	normalize_level_expression : level_expression -> level_expression
;;;;	equal_level_expression : level_expression -> level_expression -> bool
;;;;	less_level_expression : level_expression -> level_expression -> bool
;;;;	equal_less_level_expression : level_expression -> level_expression -> bool
;;;;
;;;;	destruct_ifname_parameter : parameter -> tok
;;;;	make_ifname_parameter : tok -> parameter
;;;;
;;;;	destruct_opname_parameter : parameter -> tok
;;;;	make_opname_parameter : tok -> parameter
;;;;
;;;;  -page-
;;;;
;;;;	bimodal_first_order_substitute :
;;;;	 bool -> term -> (variable # term) list -> term
;;;;
;;;;	bimodal_second_order_substitute :
;;;;	 bool ->
;;;;	  (tok # parameter) list ->
;;;;	    (variable # (variable list) # term) list -> term -> term
;;;;	 ** The tok # parameter list argument specifies parameter substitutions.
;;;;	 ** The tokens are coerced into <abstraction-meta-variable-id>'s and become
;;;;	 ** part of the substitution enviroment.
;;;;
;;;;	free_variables : term -> variable list
;;;;
;;;;	second_order_free_variables : term -> (variable # int) list
;;;;	 ** includes 0 arity variables.
;;;;
;;;;	first_order_match :
;;;;	 term -> term -> variable list -> (variable # term) list
;;;;	 ** first term is the pattern second is instance.
;;;;
;;;;	compare_terms : term -> term -> bool
;;;;	 ** true if first term is lexicographically less than second term.
;;;;
;;;;	lex_equal_terms : term -> term -> bool
;;;;	 ** compare_equal_terms A B -> t <==> (sexpr A) = (sexpr B)
;;;;	 ** this is used as the built-in equality for terms.
;;;;
;;;;	alpha_equal_terms : term -> term -> bool
;;;;
;;;; -docs- (mod mlt ml)
;;;;
;;;;	Utility functions:
;;;;
;;;;	int_to_char		: int -> tok
;;;;	char_to_int		: tok -> int
;;;;
;;;;	tok_to_string		: tok -> string
;;;;	string_to_tok		: string -> tok
;;;;
;;;;	explode_string		: string -> tok list
;;;;	concatanenate_strings	: string list -> string
;;;;	toks_to_string		: tok list -> string
;;;;
;;;; -doce-


;;; RLE TODO  level-expression which is  a simple variable should be be considered a meta.
;;; change destruct funcs.

;;; fix make meta variable : nfg that variable includes meta variables.
;;; is this a prob for bindings then?? Ie, does it change destruct-term type? or entail extra checking?
;;; let variable type include meta, error, etc and fail when used rather than failing at destruct term.
;;; funcs to build meta values should use using_parameter_bindings version.
;;; trash make_abstraction_meta_variable et al.


;;; ML variable type includes all possible parameter values of the variable parameter.
;;;   ie, meta-values, marks, errors, etc.
;;;
;;; This is done to avoid mapping/fail of bindings at term destruct time and to allow writing
;;; of functions which do not care.
;;;
;;; Functions which examine variable must check for real values.
;;; Must supply method of accessing non-standard values.
;;;
;;; flag funcs which use extra stuff in variables as inappropriate and point to parameter binding
;;; versions as alternative.
;;;
;;;	look into decoupling level-exp and meta-vars completely but allow meta bit on level-exps as error.

;;;; edit-parameter stuff should be similar to sexpr stuff.

(defunml (|make_term| (op bound-terms) :error-wrap-p nil)
    ((tok |#| (parameter list)) -> ((((variable list) |#| term) list) -> term))

  ;;(dolist (bt bound-terms)
  ;;    (dolist (v (car bt))
  ;;    (unless (or (variable-id-p v) (meta-variable-id-p v)) (break "whao"))))
  
  (instantiate-term op bound-terms))

(defun bad-term-p (term)
  (and ;;(boundp term)
       term
       (exists-p #'(lambda (bt)
		     (or (exists-p #'(lambda (v)  (not (or (variable-id-p v) (meta-variable-id-p v))))(car bt))
			 (bad-term-p (cdr bt))))
		 (bound-terms-of-term term))))



(defunml (|maybe_make_term| (term op bound-terms) :error-wrap-p nil)
    (term -> ((tok |#| (parameter list)) -> ((((variable list) |#| term) list) -> term)))
  (maybe-instantiate-term term op bound-terms))

(defunml (|destruct_term| (term) :error-wrap-p nil)
	  (term -> ((tok |#| (parameter list)) |#| (((variable list) |#| term) list)))
  (term-values term))



(defunml (|make_term_using_parameter_bindings| (op bound-terms))
  ((tok |#| (parameter list)) -> ((((parameter list) |#| term) list) -> term))

  (instantiate-term op
		  (mapcar #'(lambda (bound-term)
			      (cons (mapcar #'parameter-value (car bound-term))
				    (cdr bound-term)))
			  bound-terms)))


(defunml (|destruct_term_using_parameter_bindings| (term))
  (term -> ((tok |#| (parameter list)) |#| (((parameter list) |#| term) list)))
  
  (cons (operator-of-term term)
	(mapcar #'(lambda (bound-term)
		    (cons (mapcar #'(lambda (b)
				      (instantiate-parameter b *variable-type*))
				  (bindings-of-bound-term-n bound-term))
			  (term-of-bound-term bound-term)))
		(bound-terms-of-term term))))



(defunml (|operator_of_term| (term) :error-wrap-p nil)
	  (term -> (tok |#| (parameter list)))
  (cons (id-of-operator (operator-of-term term))
	(parameters-of-operator (operator-of-term term))))


(defunml (|bound_terms_of_term| (term) :error-wrap-p nil)
	  (term -> (((variable list) |#| term) list))
  (bound-terms-of-term term))



(defunml (|operator_id_of_term| (term) :error-wrap-p nil)
	  (term -> tok)
  (id-of-operator (operator-of-term term)))

(defunml (|parameters_of_term| (term) :error-wrap-p nil)
	  (term -> (parameter list))
  (parameters-of-operator (operator-of-term term)))

(defunml (|arity_of_term| (term) :error-wrap-p nil)
	  (term -> (int list))
  (arities-of-term term))

(defunml (|level_variables_of_term| (term) :error-wrap-p nil)
  (term -> (tok list))

  (mapcar #'(lambda (abstraction-meta-variable)
	      (intern-system (variable-id-to-string abstraction-meta-variable)))
	  (level-variables-of-term term)))



;; indices start at 1.
(defunml (|subterm_of_term| (term index))
	  (term -> (int -> term))
  (when (>= 0 index)
    (raise-error (error-message '(subterm_of_term zero) term)))

  (let ((result (term-of-bound-term (nth (1- index) (bound-terms-of-term term)))))
    (if (term-p result)
	result
	(raise-error (error-message '(subterm_of_term) term index)))))


(defunml (|bindings_of_boundterm_of_term| (term index) :error-wrap-p nil)
	  (term -> (int -> (variable list)))
  (when (>= 0 index)
    (breakout evaluation '|bindings_of_boundterm_of_term|))
    
  (let ((result (nth (1- index) (bound-terms-of-term term))))
    (if (bound-term-p result)
	(bindings-of-bound-term result)
	(breakout evaluation '|bindings_of_boundterm_of_term|))))




(defunml (|parameter_of_term| (term index) :error-wrap-p nil)
	  (term -> (int -> parameter))
  (when (>= 0 index)
    (breakout evaluation '|parameter_of_term|))
    
  (let ((parm (nth (1- index) (parameters-of-operator (operator-of-term term)))))
    (if (parameter-p parm)
	parm
	(breakout evaluation '|parameter_of_term|)
	;; due to error-wrap-p nil raise-error can't be caught by ML.
	;;(raise-error (error-message '(parameter term) (inatural-term index) term))
	)))



(defunml (|variable_to_string| (v) :error-wrap-p nil)
	  (variable -> string)
  
  (let ((value (value-of-parameter-value v)))
    (unless (symbolp value)
      (breakout evaluation "variable_to_string"))
    (variable-id-to-string value)))

(defunml (|string_to_variable| (str) :error-wrap-p nil)
	  (string -> variable)
  (get-variable-id str))


(defunml (|natural_parameter_of_term| (term index))
	  (term -> (int -> int))
  
  (when (>= 0 index)
    (breakout evaluation '|natural_parameter_of_term|))
    
  (let ((result (nth (1- index) (parameters-of-operator (operator-of-term term)))))
    (if (parameter-p result)
	(if (eql *natural-typeid* (type-id-of-parameter result))
	    (value-of-parameter-r result)
	    (breakout evaluation '|natural_parameter_of_term|))
	(breakout evaluation '|natural_parameter_of_term|))))


(defunml (|token_parameter_of_term| (term index))
	  (term -> (int -> tok))

  (when (>= 0 index)
    (breakout evaluation '|token_parameter_of_term|))
  (let ((result (nth (1- index) (parameters-of-operator (operator-of-term term)))))
    (if (parameter-p result)
	(if (eql *token-typeid* (type-id-of-parameter result))
	    (value-of-parameter-r result)
	    (breakout evaluation '|parameter_of_term|))
	(breakout evaluation '|parameter_of_term|))))


(defunml (|variable_parameter_of_term| (term index))
	  (term -> (int -> variable))
  (when (>= 0 index)
    (breakout evaluation '|variable_parameter_of_term|))
  (let ((result (nth (1- index) (parameters-of-operator (operator-of-term term)))))
    (if (parameter-p result)
	(if (eql *variable-id-typeid* (type-id-of-parameter result))
	    (value-of-parameter-r result)
	    (breakout evaluation '|variable_parameter_of_term|))
	(breakout evaluation '|variable_parameter_of_term|))))


;;;;
;;;; Edit parameters.
;;;;


(defunml (|make_edit_parameter| (edit-type meta-type type string))
    (tok -> (tok -> (tok -> (string -> parameter))))

  (make-edit-parameter edit-type meta-type type string))



;; returns (<tok{edit type}> <tok{meta type}> <tok{parameter type}> .  <string{value}>)


    
(defunml (|destruct_edit_parameter| (parameter))
    (parameter -> (tok |#| (tok |#| (tok |#| string))))
  (edit-parameter-parts parameter))
  
(defunml (|edit_type_of_parameter| (p) :error-wrap-p nil)
    (parameter -> tok)
  (let ((v (value-of-parameter p)))
    (cond
      ((slot-parameter-value-p v) 'slot)
      ((error-parameter-value-p v) 'error)
      (t nil))))


(defunml (|meta_type_of_parameter| (p) :error-wrap-p nil)
    (parameter -> tok)
  (cadr (edit-parameter-parts p)))


(defunml (|type_of_parameter| (parameter) :error-wrap-p nil)
	  (parameter -> tok)
  (type-id-of-parameter parameter))


(defunml (|pretty_string_of_parameter| (p) :error-wrap-p nil)
    (parameter -> string)
  (parameter-to-pretty-string p))

(defunml (|string_of_parameter| (p) :error-wrap-p nil)
    (parameter -> string)
  (cdddr (edit-parameter-parts p)))


(defunml (|descriptor_of_slot_parameter| (p))
    (parameter -> string)
  (if (slot-parameter-value p)
      (string (descriptor-of-slot-parameter-value (value-of-parameter p)))
      (raise-error (error-message '(descriptor_of_slot_parameter)))))


(defunml (|name_of_meta_parameter| (parameter))
    (parameter -> string)

  (let ((value (value-of-parameter parameter)))
    (cond
      ((slot-parameter-value-p value)
       (raise-error (parameter-error-message '(name_of_meta_parameter slot)
					     value (type-of-parameter parameter))))
      ((not (meta-parameter-value-p parameter))
       (raise-error (parameter-error-message '(name_of_meta_parameter not-meta)
					     value (type-of-parameter parameter))))
      (t (string value)))))



(defunml (|parameter_to_string|  (parameter))
  (parameter -> string)
  (parameter-to-string parameter))

;;;
;;; Parameters
;;;

(defunml (|make_meta_parameter| (type var) :error-wrap-p nil)
    (tok -> (variable -> parameter))

  (if (meta-variable-id-p var)
       (instantiate-parameter var (type-id-to-type type))
       (raise-error (parameter-error-message '(make_meta_parameter not-meta)
					     var type))))

(defunml (|meta_parameter_p| (parameter) :error-wrap-p nil)
    (parameter -> bool)
  (if (meta-parameter-p parameter) t nil))

(defunml (|destruct_meta_parameter| (parameter))
    (parameter -> variable)

  (let ((value (value-of-parameter parameter)))
    (cond
      ((slot-parameter-value-p value)
       (raise-error (parameter-error-message '(destruct_meta_parameter slot)
					     value (type-of-parameter parameter))))
      ((not (meta-parameter-value-p value))
       (raise-error (parameter-error-message '(destruct_meta_parameter not-meta)
					     value (type-of-parameter parameter))))
      (t value))))


(defunml (|make_abstraction_meta_variable| (tok) :error-wrap-p nil)
    (tok -> variable)
  (get-abstraction-meta-variable-id tok))

(defunml (|make_display_meta_variable| (tok) :error-wrap-p nil)
    (tok -> variable)
  (get-display-meta-variable-id tok))

(defunml (|meta_variable_type| (var) :error-wrap-p nil)
    (variable -> tok)
  (cond
    ((display-meta-variable-id-p var) 'display)
    ((abstraction-meta-variable-id-p var) 'abstraction)
    (t nil)))
							


;;; natural

(defunml (|destruct_natural_parameter| (parm))
	  (parameter -> int)
  (if (eql (type-id-of-parameter parm) *natural-typeid*)
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_natural_parameter:  improper-parameter-type|)))

(defunml (|make_natural_parameter| (i) :error-wrap-p nil)
	  (int -> parameter)
  (instantiate-parameter-r i *natural-type*))


(defunml (|destruct_time_parameter| (parm))
	  (parameter -> int)
  (if (eql (type-id-of-parameter parm) *time-typeid*)
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_time_parameter:  improper-parameter-type|)))

(defunml (|make_time_parameter| (i) :error-wrap-p nil)
	  (int -> parameter)
  (instantiate-parameter-r i *time-type*))

(defunml (|make_current_time_parameter| (unit) :declare ((declare (ignore unit))))
	  (unit -> parameter)
  (instantiate-parameter-r (get-universal-time) *time-type*))

(defunml (|utime| (unit) :declare ((declare (ignore unit))))
	  (unit -> int)
  (get-universal-time))

(defunml (|datetime_sortable| (time)) 
	  (int -> string)
  (sortable-datetime-string time))


;;;
;;; bool
;;;
(defunml (|destruct_bool_parameter| (parm))
    (parameter -> bool)
      (if (eql (type-id-of-parameter parm) *bool-typeid*)
	  (value-of-parameter-r parm)
	  (breakout evaluation '|destruct_bool_parameter:  improper-parameter-type|)))

(defunml (|make_bool_parameter| (i) :error-wrap-p nil)
    (bool -> parameter)
  (instantiate-parameter i *bool-type*))

;;; token

(defunml (|destruct_token_parameter| (parm))
          (parameter -> tok)
  (if (eql *token-typeid* (type-id-of-parameter parm))
      (value-of-parameter-r parm)
      (progn (break "dtp") (breakout evaluation '|destruct_token_parameter:  improper-parameter-type|)))
  )

(defunml (|make_token_parameter| (token) :error-wrap-p nil)
	  (tok -> parameter)
  (instantiate-parameter token *token-type*))


(defunml (|destruct_quote_parameter| (parm))
          (parameter -> tok)
  (if (eql *quote-typeid* (type-id-of-parameter parm))
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_quote_parameter:  improper-parameter-type|)))

;;; string

(defunml (|destruct_string_parameter| (parm))
          (parameter -> string)
  (if (eql *string-typeid* (type-id-of-parameter parm))
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_string_parameter:  improper-parameter-type|)))

(defunml (|make_string_parameter| (string) :error-wrap-p nil)
	  (string -> parameter)
  (instantiate-parameter string *string-type*))


;;;  variable

;; variable <-> tok conversion

(defunml (|tok_to_variable| (tok) :error-wrap-p nil)
	  (tok -> variable)
  (get-variable-id tok))

(defunml (|variable_to_tok| (var) :error-wrap-p nil)
	  (variable -> tok)
  (let ((value (value-of-parameter-value var)))
    (unless (symbolp value)
      (breakout evaluation '|variable_to_tok|))
    (intern-system (variable-id-to-string var))))


(defunml (|destruct_variable_parameter| (parm))
	  (parameter -> variable)
    (if (eql (type-id-of-parameter parm) *variable-id-typeid*)
	(value-of-parameter-r parm)
	(breakout evaluation '|destruct_variable_parameter:  improper-parameter-type|)))

(defunml (|make_variable_parameter| (variable) :error-wrap-p nil)
	  (variable -> parameter)
  (instantiate-parameter variable *variable-type*))


(defunml (|make_level_expression_parameter| (expr) :error-wrap-p nil)
	  (level_expression -> parameter)
  (instantiate-parameter (unml-level-expression expr)
			   *level-expression-type*))

(defunml (|destruct_level_expression_parameter| (parameter))
          (parameter -> level_expression)
  (if (level-expression-parameter-p parameter)
      (ml-level-expression (value-of-parameter-r parameter))
      (breakout evaluation
		'|destruct_level_expression_parameter:  improper-parameter-type|)))

(defunml (|make_max_level_expression| (exprs) :error-wrap-p nil)
          ((level_expression list) -> level_expression)
   (ml-level-expression
    (make-level-max (mapcar #'unml-level-expression exprs))))

(defunml (|max_level_expression_p| (expr) :error-wrap-p nil)
          (level_expression -> bool)
  (level-max-p (unml-level-expression expr)))

(defunml (|destruct_max_level_expression| (expr) :error-wrap-p nil)
    (level_expression -> (level_expression list))

  (let ((unexpr (unml-level-expression expr)))
    (unless (level-max-p unexpr)
      (breakout evaluation "DestructMaxLevelExpression: expression is not max."))
    (mapcar #'ml-level-expression unexpr)))


(defunml (|make_increment_level_expression| (expr incr) :error-wrap-p nil)
    (level_expression -> (int -> level_expression))
  (ml-level-expression (make-level-increment (unml-level-expression expr) incr)))

(defunml (|increment_level_expression_p| (expr) :error-wrap-p nil)
    (level_expression -> bool)
  (level-increment-p (unml-level-expression expr)))

(defunml (|destruct_increment_level_expression| (expr) :error-wrap-p nil)
    (level_expression -> (level_expression |#| int))
  (let ((unexpr (unml-level-expression expr)))
    (unless (level-increment-p unexpr)
      (breakout evaluation "DestructIncrementLevelExpression: expression is not an increment"))
    (cons (ml-level-expression (expression-of-level-increment unexpr))
	  (increment-of-level-increment unexpr))))

(defunml (|make_constant_level_expression| (i) :error-wrap-p nil)
    (int -> level_expression)
  (ml-level-expression (make-level-constant i)))

(defunml (|constant_level_expression_p| (expr) :error-wrap-p nil)
    (level_expression -> bool)
  (level-constant-p (unml-level-expression expr)))

(defunml (|destruct_constant_level_expression| (expr) :error-wrap-p nil)
    (level_expression -> int)
  (let ((unexpr (unml-level-expression expr)))
    (unless (level-constant-p unexpr)
      (breakout evaluation "DestructLevelExpression: expression is not a constant."))
    (int-of-level-constant unexpr)))


(defunml (|make_variable_level_expression| (tok) :error-wrap-p nil)
    (tok -> level_expression)
  (ml-level-expression (make-level-variable (string tok))))

(defunml (|variable_level_expression_p| (expr) :error-wrap-p nil)
    (level_expression -> bool)
  (level-variable-p (unml-level-expression expr)))

(defunml (|destruct_variable_level_expression| (expr) :error-wrap-p nil)
    (level_expression -> tok)
  (let ((unexpr (unml-level-expression expr)))
    (unless (level-variable-p unexpr)
      (breakout evaluation "DestructLevelExpression: expression is not a variable."))
    (intern-system (level-variable-to-string unexpr))))


(defunml (|substitute_in_level_expression| (expr pairs))
  (level_expression -> (((tok |#| level_expression) list) -> level_expression))

  (ml-level-expression
   (let ((subs (mapcar #'(lambda (pair)
			   (cons (intern-system (string (car pair)))
				 (unml-level-expression (cdr pair))))
		       pairs)))

     (level-expression-visit (unml-level-expression expr)
			     #'(lambda (v)
				 (let ((sub (assoc v subs)))
				   (if  sub
				       (cdr sub)
				     v))))))) 

(defunml (|normalize_level_expression| (expr) :error-wrap-p nil)
    (level_expression -> level_expression)

  (ml-level-expression (normalize-level-expression (unml-level-expression expr))))

(defunml (|equal_level_expression| (expr1 expr2) :error-wrap-p nil)
    (level_expression -> (level_expression -> bool))

  (equal-level-expressions-p (unml-level-expression expr1)
			     (unml-level-expression expr2)))

;; returns true if expr1 < expr2.
(defunml (|less_level_expression| (expr1 expr2) :error-wrap-p nil)
    (level_expression -> (level_expression -> bool))

  (less-level-expression-p (unml-level-expression expr1)
			   (unml-level-expression expr2)))

(defunml (|equal_less_level_expression| (expr1 expr2) :error-wrap-p nil)
    (level_expression -> (level_expression -> bool))

  (equal-less-level-expression-p (unml-level-expression expr1)
				 (unml-level-expression expr2)))



;;;
;;; reflection parameter types :
;;;

;;; ifname

(defunml (|destruct_ifname_parameter| (parm))
    (parameter -> tok)
  (if (eql (type-id-of-parameter parm) *ifname-typeid*)
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_ifname_parameter:  improper-parameter-type|)))


(defunml (|make_ifname_parameter| (token) :error-wrap-p nil)
    (tok -> parameter)
  (instantiate-parameter token *ifname-type*))


;;; opname

(defunml (|destruct_opname_parameter| (parm))
    (parameter -> tok)
  (if (eql (type-id-of-parameter parm) *opname-typeid*)
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_opname_parameter:  improper-parameter-type|)))


(defunml (|make_opname_parameter| (token) :error-wrap-p nil)
    (tok -> parameter)
  (instantiate-parameter token *opname-type*))



;;; parameter list

(defunml (|make_parameter_list_parameter| (parms) :error-wrap-p nil)
    ((parm list) -> parameter)

  (instantiate-parameter-r parms *parameter-list-type*))

(defunml (|destruct_parameter_list_parameter| (parm))
    (parameter -> (parameter list))
  
  (if (eql (type-id-of-parameter parm) *parameter-list-typeid*)
      (value-of-parameter-r parm)
      (breakout evaluation '|destruct_pl_parameter:  improper-parameter-type|)))



;;;
;;;  substitution, alpha-equality etc.
;;;



(defunml (|first_order_match| (ml-pattern ml-instance pattern-ids))
    (term -> (term -> ((variable list) -> ((variable |#| term) list))))
  (match ml-pattern ml-instance pattern-ids))


(defunml (|bimodal_first_order_substitute| (maintenance-p term subs))
    (bool -> (term -> (((variable |#| term) list) -> term)))
  (substitute term
	      subs
	      :maintain-p maintenance-p))


(defunml (|bimodal_second_order_substitute| (maintenance-p parm-subs subs term))
    (bool -> (((tok |#| parameter) list) ->
		    (((variable |#| ((variable list) |#| term)) list) -> (term -> term))))

  ;; RLE PERF don't do sos for simple parameter subst.
  (if maintenance-p
      (if (and (null parm-subs)
	       (forall-p #'(lambda (sub) (null (cadr sub))) subs))
	  (substitute term (mapcar #'(lambda (sub) (cons (car sub) (cddr sub))) subs))
	  (second-order-substitution-with-maintenance
	   term
	   (mapcar #'(lambda (parm-sub)
		       (if (level-expression-parameter-p (cdr parm-sub))
			   parm-sub
			   (cons (get-abstraction-meta-variable-id (car parm-sub))
				 (cdr parm-sub))))
		   parm-subs)
	   subs))
      (second-order-substitute
	term
	(mapcar #'(lambda (parm-sub)
		    (if (level-expression-parameter-p (cdr parm-sub))
			parm-sub
			(cons (get-abstraction-meta-variable-id (car parm-sub))
			      (cdr parm-sub))))
		parm-subs)
	subs)))


(defunml (|free_variables| (term) :error-wrap-p nil)
    (term -> (variable list))
  (free-vars term))

(defunml (|second_order_free_variables| (term) :error-wrap-p nil)
    (term -> ((variable |#| int) list))
  (second-order-free-vars term))

;; true if t1 is lexicographically less than t2. false otherwise.
(defunml (|alpha_equal_terms| (t1 t2) :error-wrap-p nil)
    (term -> (term -> bool))
  (equal-terms-p t1 t2))

(dml |eq_terms| 2 eq (term -> (term -> bool)))

;; true if t1 is lexicographically less than t2. false otherwise.
(defunml (|compare_terms| (t1 t2) :error-wrap-p nil)
    (term -> (term -> bool))
  ;;(setf -t1 t1 -t2 t2) (break "ct")
  (eql 'arg1 (compare-terms t1 t2)))

(defunml (|lex_equal_terms| (t1 t2) :error-wrap-p nil)
    (term -> (term -> bool))
  (compare-terms-p t1 t2))


(defunml (|equal_parameters| (p1 p2) :error-wrap-p nil)
    (parameter -> (parameter -> bool))
  (equal-parameters-p p1 p2))

(defunml (|equal_operators_of_terms| (t1 t2) :error-wrap-p nil)
    (term -> (term -> bool))
  (equal-operators-p (operator-of-term t1) (operator-of-term t2)))


(defunml (|with_substitution_count_bound| (b f a) :error-wrap-p nil)
    (int -> ((* -> **) -> (* -> **)))

  (with-substitution-count-bound (b #'(lambda ()
					(breakout evaluation 'substitution_count_bound)))
    (ap f a)))

(defmacro with-suspend-substitution-count (&body body)
  `(with-substitution-count-bound (0 #'(lambda (&rest rest) (declare (ignore rest)) (break "wssc")))
    ,@body))

(defunml (|with_suspend_substitution_count| (f a) :error-wrap-p nil)
    ((* -> **) -> (* -> **))

  (with-suspend-substitution-count
    (ap f a)))


(defunml (|get_substitution_count| (unit) :declare ((declare (ignore unit))))
    (unit -> int)

  (if (boundp '*subst-count*)
      *subst-count*
      0))

;;;
;;; Misc.
;;;


(defun add-com-lettypes ()
  (ml-text "lettype UNIT = .")
  (ml-text "lettype unit = UNIT"))

(insys (reset-ml)
       (add-com-lettypes))


;;;;	This will map unicode codes to embedded standard chars
;;;;	but next func does not map embedded chars to unicode codes.

(defunml (|int_to_char| (i) :error-wrap-p t)
  (int -> tok)

  (if (standard-character-code-p i)
      (implode-to-character-tok i)
      (intern-system (int-to-character-string i))))

(defunml (|char_to_int| (ch) :error-wrap-p nil)
  (tok -> int)
  (let ((s (string ch)))
    (if (= (length s) 1)
	(char-code (char s 0))
	(breakout evaluation
		  (format-string "char_to_int : Char[~a] should be of length 1." ch)))))



(defun concatenate-string-list (strings)
  (let ((s (make-string (do ((l strings (cdr l))
			     (acc 0 (+ (length (car l)) acc)))
			    ((null l) acc)))))
    
    (let ((i 0))
      (dolist (string strings)
	(dotimes (j (length string))
	  (setf (elt s i) (elt string j))
	  (incf i))))
    s))

(defunml (|tok_to_string| (tok) :error-wrap-p nil)
	  (tok -> string)
  (string tok))

(defunml (|string_to_tok| (str) :error-wrap-p nil)
	  (string -> tok)
  (intern str *system-package*))


(defunml (|explode_string| (string))
	  (string -> (tok list))
  (explode string))


(defunml (|concatenate_strings| (strings))
	  ((string list) -> string)
  (concatenate-string-list strings))  


(defunml (|toks_to_string| (toks))
	  ((tok list) -> string)
  (concatenate-string-list (mapcar #'string toks)))



(defunml (|int_to_string| (i))
    (int -> string)
  (princ-to-string i))

(defunml (|string_to_int| (str))
    (string -> int)
  (mlet* (((i lc) (read-from-string str nil nil)))
	 (if (and (integerp i)
		  (eq lc (length str)))
	     i
	     (breakout evaluation 
		       (format-string "string_to_int : string[~a] does not represent an integer." str)))))

(defunml (|term_to_text| (i))
    (term -> (* list))
  (term-to-text i))
    
(defunml (|first_text_string| (i))
    (term -> string)
  (first-text-string i))


(defunml (|term_to_pretty_string| (term))
    (term -> string)

  (term-to-pretty-string term))


(defunml (|build_description_match| (descterm))
    (term -> (term  -> bool))

  (let ((desc (term-to-description descterm)))
    (make-closure #'(lambda (d) (match-descriptions-p d desc))
		  1)))


(defunml (|map_term| (p f term))
    ((term -> bool) -> ((term -> ((term -> term) -> term)) -> (term -> term)))

  (term-walk-d term
	       #'(lambda (term)
		   (funmlcall p term))
	       #'(lambda (term contf)
		   (funmlcall f term (make-closure contf 1)))))


(defunml (|term_op_count| (term))
    (term -> int)
  (term-op-count term))
			  
