
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;
;;;; -docs- (mod ref top ml)
;;;;	
;;;;	lcomp &optional lisp-p ml-p
;;;;	  * compiles 
;;;;	  * by default bools are t, ie (lcomp nil) will compile ml but not lisp.
;;;;
;;;;	lcomp	: bool{lisp-p} -> bool{ml-p} -> unit
;;;;	
;;;;	nosa 	: int{server-port} -> unit
;;;;	  * orb_start_accept `fdl`
;;;;	
;;;;	mosa 	: int{server-port} -> unit
;;;;	  * orb_start_accept `mathbus`
;;;;	
;;;;	
;;;;	nosa 6289;;
;;;;	mosa 5289;;
;;;;	
;;;; -doce-

;; Primary goal is that compiling results in consistent set of binaries.
;; Secondary is that unneccessary compilation is avoided.
;; One problem : consider a function calling a function defined differently
;;   in distinct systems. Eg lib and edd define dir funcs different but present
;;   unified "interface" to later files. To meet primary goal all files following
;;   the diff func must be recompiled when switching between compiling systems.
;;  on the other hand since the interface is the same it would be nice to be able
;;  to avoid recompiling later files. One cheap method would be to define functions
;;  as globals and then move the diff files later and assign to the earlier globals
;;  
;;  if the earlier bin file happened to be compiled prior to the later diff file
;;  which was compiled by other system, then the need for recompilation will not
;;  be detected.

;; system a defines func f in file f-a
;; system b defines func f in file f-b
;; system a calls f from file g-ab
;; system b calls f from file g-ab
;;
;; problem occurs if bin of file g-ab is loaded into system b after being compiled in system a.
;; file bin-a of g-ab will be loaded into system b if bin of f-b is earlier than bin of g-ab.
;; touch f-a, f-b, g-ab at t0
;; compile system b
;;  f-b  compiles at t1
;;  g-ab compiles at t2
;; compile system a
;;  f-a  compiles at t3
;;  g-ab compiles at t4
;; compile system b
;;  f-b loads
;;  g-ab loads
;;  system b is crap.

;; orb environment in lib needs a few less files.
;; lal added orbm-lib 5/24
(defun do-ml-all-lib-compile (&optional printp (metaprlp t))
  (com-ml-comp)
  (inml (ml-text
	 (list
	  (format-string "load_system true false ~a [\"library\"]"
			 (if printp "true" "false"))
	  "[ ([\"orb\"],"
          "   [\"orbm-bsc\"; \"orbm-lib\"])"
    
	  "; ([\"top\"],"
          "   [\"comm-dag\"; \"comm-graph\"; \"comm-nrf\"; \"topm-lib\";"
	  "    \"comm-ref\"; \"comm-ned\"; \"comm-mp\"])" 

	  "; ([\"lib\"],"
          "   [ \"libm-bsc\"; \"libm-ref\"; \"libm-wm\";"
	  "     \"libm-utl\"; \"libm-tac\"; \"libm-mig\""
	  (if metaprlp "   ;\"libm-mp\"]" "]" )
	  ")]"
	  )))

  (setf *lib-refine* nil
	*lib-ml-code-compile* nil
	))


(defun lmlcomp (&optional (printp t) (metaprlp t))
  (do-ml-all-lib-compile printp metaprlp))

(defun init-lib-base (&optional (expand-p nil))

  (format t ";;;;~Tinit-lib~%")
  (init-com-base)

  (setf *component-kind* '|lib|)
  (update-nightly-hooks 'libstats
			#'(lambda (note ctime) (lib-stats t note ctime)))

  (pushnew :fdl-lib *features*)

  ;; done unconditionally in lib-restart fttb.
  (when expand-p
    ;; LAL laptop (half-gig-expand)
    ;; (very-large-expand)
    (lib-expand (car (get-config-data `memory-profile)))
    ))



(defun print-lib-banner ()
  (format t "~%;;;~%;;;	")
  (format t "~%;;;	FDL Library~%")

  (format t ";;;	~%")
  (let ((port (ml-text "lib_accept_port := config_accept () ? 0"))
	(env (ml-text "lib_env := config_libenv() ? \"\"")))
    (if (not (zerop port))
	(format t ";;;	The library will listen for accepts on port ~a.~%" port)
	(format t ";;;	An accept port has not been specified.~%"))
    (if (not (string= "" env))
	(format t ";;;	The default environment is ~a.~%" env)
	(format t ";;;	A default environment was not specified.~%")))
  (format t ";;;	~%")
    
  (format t ";;;	At startup the default port numbers are read from ")
  (format t "~~/.~a.config"  *system-kind*)
  (format t "~%;;;	using the keywords: sockets, libhost, libenv.~%")

  (format t ";;;	You may override these parameters via setup_accept : int -> unit~%;;;	and set_lib_env: string -> unit.~%")
  
  (format t ";;;	~%")
  (format t ";;;	(top)~%")
  (format t ";;;	go.~%;;;~%")
  )

(defun init-lib-restart (&optional db-init-p (db-ascii-p t))

  (init-com-restart)
  (lib-expand (car (get-config-data 'memory-profile)))
  
  (add-producer-state-snap-func 'library 'lib-producer-bound-terms)

  (when db-init-p
    (format t ";;;;~Tinit-lib db-init~%")
    (when (packagep (find-package "MATHBUS"))
      (funcall (intern "READ-TOKENS" (find-package "MATHBUS"))))
    
    ;; db will init levels if ascii-p otherwise doit here.
    ;; should be lazier method.
    (db-init (db-root-path db-ascii-p (find-db-path)))

    (format t ";;;;~Tinit-lib db-init done~%")
    ))



(defun do-disk-init-lib ()
  (init-lib-base nil)  ; 'dodisk ??
  
  (setf *eval-print-p* t)
  (lmlcomp nil)

  (format t "~%;;;~%;;;	")
  (format t "~%;;;	FDL Library. ~%;;;	If making dist disksaves mv ~~/.fdl.config.dist to ~~/.fdl.config ~%;;;	")
  (format t "~%;;;~%;;;	(in-package \"~a\")(dodisk-lib)" *system-package-name*)
  (format t "~%;;;	(in-package \"~a\")(dodisk-lib t) ; overwrites current~%" *system-package-name*)
  (format t "~%;;;~%;;;	")

  )

(defun do-lib-restart ()
  (init-lib-restart t t)
  (print-lib-banner)
  (eval `(in-package ,*system-package-name*))

  #+cmu
  (read-eval-print-loop)
)

(defun dodisk-lib (&optional overwrite)
  (dodisk #'do-lib-restart overwrite))




(defun do-init-lib (&optional (expand-p t) (db-init-p t))

  (init-lib-base expand-p)
  (lmlcomp nil)

  (set-reload-global-alist)
  (init-lib-restart db-init-p)

  ;;(do-ml-all-lib-compile)

  ;;(init-lib-top)
  ;;(add-command-map 'lib0 #'map-command-lib0)

  (print-lib-banner)
  ;;(in-package *system-package-name*)
  ;;(top)
)



(defun docomp () (make:compile-system 'fdl-lib))


(defmacro withlib (&body body)
  `(insys
    (with-environment ((address-of-environment (match-environment-in-list '(|lib|) *component*)))
      (with-local-transaction 
	  ,@body))))
    
(defun go-lib ()

  (let ((port (config-accept)))
    (unless (accept-active-p port)
      (orb-start-lib-accept port 'fdl nil))

    (let ((libenv (list (intern-system (config-libenv t)))))
      (when libenv
	(unless (match-environment-in-list libenv *component*)
	  (with-environment ((address-of-environment (match-environment-in-list (list '|ORB|) *component*)))
	    (with-appropriate-transaction (nil nil)
	      (without-dependencies
	       (open-environment-by-match libenv)))))))))


(defun stop-lib ()

   (let ((libenv (list (intern-system (ml-text "lib_env")))))
      (when libenv
	 (with-environment ((address-of-environment (match-environment-in-list (list '|lib|)
									       *component*)))
	   (with-transaction (nil)
	     (close-environment (match-environment-in-list libenv *component*) nil nil)))))
   (ml-text "stop_accept ()")
   (progn
     (quit-request-loop)
     (toploop-remove-bus-link)))
	

