
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      compare-terms-p compare-terms 
	      free-vars-of-bound-term sticky-free-vars cache-free-vars
	      isub-apply-term-p isub-apply-term subbee-of-isub-apply-term subbends-of-isub-apply-term
	      second-order-substitute
	      make-sub-from-list
	      )))


;;;;
;;;; -docs- (mod trm)
;;;;
;;;;
;;;;	level-variables-of-term (<term>) : <abstraction-meta-variable-id> list
;;;;
;;;;	Lexicographic (in)equality: basically compares print representation of
;;;;	 terms lexicographically. Some attributes of parameter values are compared
;;;;	 explicitly: 
;;;;	   - error < slot < meta < value
;;;;	   - any two slots are considered equal.
;;;;	   - display-meta < abstraction-meta
;;;;
;;;;	compare-terms-p (<term> <term>)			: <bool>
;;;;	compare-terms (<term> <term>)			: T | 'ARG1 | 'ARG2
;;;;	 ** T if equal, ARG1 if first arg is less than second-arg, otherwise ARG2.
;;;;
;;;;
;;;;	equal-terms-p (<term> <term>)		: <bool>
;;;;	 ** T if terms are alpha-convertible.
;;;;
;;;;	free-vars (<term>)			: <variable-id> list
;;;;	occurs-free-p (<variable-id> <term>)	: <bool>
;;;;
;;;;	sticky-free-vars (<term>)		: <variable-id> list
;;;;	cache-free-vars(<term>)			: <term>
;;;;	 ** These cache result.
;;;;
;;;;	match (<term{pattern}> <term{instance}> <variable-id{pattern}> list)
;;;;	 : (<varible-id> . <term>) list
;;;;	 ** fails if no match.
;;;;	 ** free occurences of variable-ids in pattern are meta vars for match.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Bimodal Substitution: When renaming occurs during substitution
;;;;	  it is sometimes desirable to maintain the sameness and difference
;;;;	  of the bindings of the subterms of a term.  For example, if you
;;;;	  have a term foo(x.y; x.void()) and you substitute x for y the desired
;;;;	  outcome may be foo(@x1.x; @x1.void()) instead of the usual foo(@x1.x; x.void()).
;;;;	  We will call the first case substitution with maintenance. The bimodal
;;;;	  substitution functions take a bool as a first argument which indicates whether
;;;;	  to perform the substitution with(true) maintenance or without(false).
;;;;
;;;;
;;;;	Renaming: the renaming algorithm to be described has several goals.
;;;;	  - implement maintenance as described above.
;;;;	  - heuristically attempt to maintain user variables.
;;;;	    A user variable would be the variable most likely to have been type into a
;;;;	    proof goal. Eg, when expanding an abstraction the variables of the instance
;;;;	    would be user variables.
;;;;	  - renamings dependent only upon local information.
;;;;	    Ie, the same substitution on the same arguments always results
;;;;	    in a lexicographically equivalent result.
;;;;
;;;;
;;;;	Let S be the second-order term to be susbstituted into.
;;;;	Let xij.tj be the first order terms to be substituted in.
;;;;	let (yi si) be the selections.
;;;;	Consider a binding to be the occurrence of a variable in a binding
;;;;	list of a bound term, and consider an occurrence to be a occurence
;;;;	of a variable as a term.
;;;;
;;;;	With Maintenance:
;;;;
;;;;	A variable collection is the name of a free variable, or
;;;;    the name of a binding and a term occurence containing the binding.
;;;;	The preferred name of a binding from S is the selected name if the name
;;;;	is selected, or the original name. The preferred name of a binding from
;;;;	a tj is the original name. The preferred name of a free variable is the
;;;;	original name.
;;;;
;;;;	Step 1:
;;;;
;;;;	Rename each variable collection to unique names. If two bindings in a
;;;;	term occurrence in S both select the same name,
;;;;	then unless they occur as non-identical bindings in the same binding list,
;;;;	they should be renamed to the same name such that they become one
;;;;	variable collection.
;;;;	Associate with each new name a tuple consisting of the preferred name,
;;;;	an indication whether the variable was selected, and an indication
;;;;	whether variable is from S or a tj.
;;;;
;;;;
;;;;	Step 2:
;;;;
;;;;	Perform the substitution to produce term T, note that capture is not
;;;;	possible.
;;;;
;;;;  -page-
;;;;
;;;;	Step 3:
;;;;
;;;;	Consider the following disjoint classes used later to define an ordering:
;;;;	 5. dummy variable collections in T.
;;;;	 4. variable collections from S which were not selected and are not dummy in T.
;;;;	 3. variable collections from S which were selected and are not dummy in T.
;;;;	 2. variable collections from tj and are not dummy in T.
;;;;	 1. Free variables.
;;;;
;;;;	Note: A binding variable collection is dummy if there are no variable
;;;;	 occurences actually bound.
;;;;
;;;;	 - a > b if a occurs in class i and b in class j and i<j
;;;;	         or i=j and a and b have same parent and first binding of a
;;;;	                    occurs to the left of the first binding of b
;;;;	         or i=j and b occurs in scope of a in term T.
;;;;
;;;;	Note: this is a partial order, if a is not > b and b is not > a, then
;;;;	 then when visiting in descending order it will not matter which is
;;;;	 visited first.
;;;;
;;;;	A variant of a name is the name with a @i suffix.
;;;;
;;;;	A variable collection conflicts with another when
;;;;	 the preferred names of the collections are the same and one of the following is true:
;;;;	   - they have the same term occurrence (ie, they bind terms of the same operator).
;;;;	     This is to maintain difference.
;;;;	   - a variable occurence of one is free within the scope of a binding occurence of 
;;;;	     the other.  This is standard capture.
;;;;
;;;;	Renaming proceeds sequentially:
;;;;	 Invariant: If a > b and b is being renamed then a has been assigned a name.
;;;;
;;;;	 If a variable collection conflicts then it is renamed to the least
;;;;	 preferred name variant which does not conflict with the assigned name of
;;;;	 a > variable collection.
;;;;
;;;;	 In descending order, rename those which conflict.
;;;;
;;;;	 Note:
;;;;	  Class 2 variable collections can only conflict with free variable
;;;;	    collections.
;;;;	  Class 4 variable collections can not conflict with each other.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Substitution: 
;;;;	
;;;;	 ** <sub>		: (<variable-id> . <term>)
;;;;	 ** <so-sub>		: (<variable-id> . <bound-term>)
;;;;	 ** <parameter-sub>	: (<abstraction-meta-variable-id> . <parameter>)
;;;;
;;;;	substitute (term <sub> list &key (maintain-p t))		: <term>
;;;;	substitute-parameters-in-term (<term> <parameter-sub> list)	: <term>
;;;;	second-order-substitute (<term> <parameter-sub> list <so-sub> list)	: <term>
;;;;	second-order-substitution-with-maintenance (<term>
;;;;						   <parameter-sub> list
;;;;						   <so-sub> list
;;;;						   &optional <selection> list)
;;;;	 : <term>
;;;;	 ** <selection>		: #selection[<variable-id>
;;;;					     INTEGER{term-index}
;;;;					     INTEGER{binding-index}]
;;;; -doce-
;;;;
;;;; RLE TODO this may need updating:
;;;;
;;;;	Without Maintenance:
;;;;
;;;;	Step 1:
;;;;
;;;;	Consider renaming each individual binding with occurrences to unique names.
;;;;	Associate with each new name a tuple consisting of the original id, the
;;;;	selected id, and an indication whether variable is from S or a tj.
;;;;	Perform the substitution to produce term T.
;;;;
;;;;	Step 2:
;;;;
;;;;	Separate names into one of five classes:
;;;;	 5. dummy bindings in T.
;;;;	 4. bindings from S which were not selected and are not dummy in T.
;;;;	 3. bindings from S which were selected and are not dummy in T.
;;;;	 2. bindings from tj and are not dummy in T.
;;;;	 1. Free variables.
;;;;
;;;;	Step 3:
;;;;
;;;;	A binding conflicts with another binding if
;;;;	 one has an occurrence within the scope of the other.
;;;;	A binding conflicts if captures a free-variable.
;;;;	A binding does not conflicts with another binding if they have the same
;;;;	  parent and their conflict ids are the same.
;;;;
;;;;	 - a > b if a occurs in class i and b in class j and i<j or
;;;;	   i=j and b occurs in scope of a in term T.
;;;;
;;;;	The class 2 bindings need not be visited in descending order.
;;;;
;;;; -doce-


;;;
;;; subst stats.
;;;

(defvar *no-exp* 0)
(defvar *no-trivial* 0)
(defvar *no-simple* 0)
(defvar *no-first* 0)
(defvar *no-second* 0)
(defvar *no-subst1* 0)
(defvar *no-subst1-trivial* 0)
(defvar *no-subst1-renamed* 0)
(defvar *no-subst2-m* 0)
(defvar *no-subst2-renamed-m* 0)
(defvar *no-subst2* 0)
(defvar *no-subst2-renamed* 0)
(defvar *subst-stats* nil)


(defmacro without-subst-stats (&body body)
  `(let ((*no-exp* 0)
	 (*no-trivial* 0)
	 (*no-simple* 0)
	 (*no-first* 0)
	 (*no-second* 0)
	 (*no-subst1* 0)
	 (*no-subst1-trivial* 0)
	 (*no-subst1-renamed* 0)
	 (*no-subst2-m* 0)
	 (*no-subst2-renamed-m* 0)
	 (*no-subst2* 0)
	 (*no-subst2-renamed* 0))

    ,@body))

(defmacro with-subst-stats (&body body)
  `(let ((*no-exp* 0)
	 (*no-trivial* 0)
	 (*no-simple* 0)
	 (*no-first* 0)
	 (*no-second* 0)
	 (*no-subst1* 0)
	 (*no-subst1-trivial* 0)
	 (*no-subst1-renamed* 0)
	 (*no-subst2-m* 0)
	 (*no-subst2-renamed-m* 0)
	 (*no-subst2* 0)
	 (*no-subst2-renamed* 0))

    (prog1
	(progn ,@body)

      (format t "~%  Expansions : ~a, trivial ~a, simple : ~a, first : ~a, second ~a.~%  1st Subst : ~a, renamed ~a, trivial : ~a. ~%  2nd Subst : ~a, renamed ~a.~% ~a ~a ~%"
	      *no-exp* *no-trivial* *no-simple* *no-first* *no-second* *no-subst1* *no-subst1-renamed* *no-subst1-trivial*
	      (+ *no-subst2-m* *no-subst2*)
	      (+ *no-subst2-renamed-m* *no-subst2-renamed*)
	      *no-subst2*
	      *no-subst2-renamed*)
    
      (setf *subst-stats*
	    (list
	     *no-exp* *no-trivial* *no-simple* *no-first* *no-second*
	     *no-subst1* *no-subst1-renamed* *no-subst1-trivial*
	     (+ *no-subst2-m* *no-subst2*) (+ *no-subst2-renamed-m* *no-subst2-renamed*))))))



;;;  ----------------------------------------------------------------------
;;;  ---------------------   alpha equality  ------------------------------
;;;  ----------------------------------------------------------------------

;;; RLE PERF NAP ??? try a throw when not equal detected rather than unwinding with nil.

;;;;	
;;;;	Consider optional arg which is closure testing for transparent operators.
;;;;	A transparent operator will be completely ignored.
;;;;	
;;;;	Consider making equal terms-p a macro so when transparent test is omitted
;;;;	there is no overhead assoc'd with it.

;;;; LAL couldn't find the definition of these

#+cmu
(progn
  (defun alpha-equal-bound-term-lists (listx listy)
    (or (and (null listx) (null listy))
	(and listx listy
	     (let ((bindings-x (bindings-of-bound-term (car listx)))
		   (bindings-y (bindings-of-bound-term (car listy))))
	       (if (and (null bindings-x) (null bindings-y))
		   (alpha-equal-terms (term-of-bound-term (car listx))
				      (term-of-bound-term (car listy)))
		   (and (= (length bindings-x) (length bindings-y))
			(prog2
			    (enter-binding-pairs (bindings-of-bound-term (car listx))
						 (bindings-of-bound-term (car listy)))
			    (alpha-equal-terms (term-of-bound-term (car listx))
					       (term-of-bound-term (car listy)))
			  (exit-binding-pairs (bindings-of-bound-term (car listx))
					      (bindings-of-bound-term (car listy)))))))
	     (alpha-equal-bound-term-lists (cdr listx) (cdr listy)))))


  (defun alpha-equal-terms (termx termy)
    (if (and (variable-p termx)
	     (variable-p termy))
	(equal-bindings (id-of-variable-term termx)
			(id-of-variable-term termy))
	(and (equal-operators-p (operator-of-term termx) (operator-of-term termy))
	     (alpha-equal-bound-term-lists (bound-terms-of-term termx)
					   (bound-terms-of-term termy)))))

  (defun equal-terms (termx termy)
    (or (eq termx termy)
	(and termx termy
	     (with-variable-invocation (alpha-equal-terms termx termy)))))
)
;; end of LAL added

(defun equal-terms-p (termx termy)
  (labels
      ((alpha-equal-bound-term-lists (listx listy)
	 (or (and (null listx) (null listy))
	     (and listx listy
		  (prog1
		      (and (enter-binding-pairs (bindings-of-bound-term-n (car listx))
						(bindings-of-bound-term-n (car listy)))
			   (alpha-equal-terms (term-of-bound-term (car listx))
					      (term-of-bound-term (car listy))))
		    (exit-binding-pairs (bindings-of-bound-term-n (car listx))
					(bindings-of-bound-term-n (car listy))))
		  #+cmu(let ((r (alpha-equal-bound-term-lists (cdr listx) (cdr listy))))
			       r)
		  #-cmu(alpha-equal-bound-term-lists (cdr listx) (cdr listy))
		  )))

       (alpha-equal-terms (termx termy)
	 (if (and (variable-p termx)
		  (variable-p termy))
	     (equal-bindings (id-of-variable-term termx)
			     (id-of-variable-term termy))
	     (and (equal-operators-p (operator-of-term termx) (operator-of-term termy))
		  (alpha-equal-bound-term-lists (bound-terms-of-term termx)
						(bound-terms-of-term termy))))))

    (or (eq termx termy)
	(and termx termy
	     (with-variable-invocation (alpha-equal-terms termx termy)))
        nil)))


(defun equal-terms-with-transparencies-p (termx termy transparent-filter)
  (labels
      ((alpha-equal-bound-term-lists (listx listy)
	 (or (and (null listx) (null listy))
	     (and listx listy
		  (prog1
		      (and (enter-binding-pairs (bindings-of-bound-term-n (car listx))
						(bindings-of-bound-term-n (car listy)))
			   (alpha-equal-terms (funcall transparent-filter (term-of-bound-term (car listx)))
					      (funcall transparent-filter (term-of-bound-term (car listy)))))
		    (exit-binding-pairs (bindings-of-bound-term-n (car listx))
					(bindings-of-bound-term-n (car listy))))
		  #+cmu(let ((r (alpha-equal-bound-term-lists (cdr listx) (cdr listy))))
			 r)
		  #-cmu(alpha-equal-bound-term-lists (cdr listx) (cdr listy))
		  )))

       (alpha-equal-terms (termx termy)
	 (if (and (variable-p termx)
		  (variable-p termy))
	     (equal-bindings (id-of-variable-term termx)
			     (id-of-variable-term termy))
	     (and (equal-operators-p (operator-of-term termx) (operator-of-term termy))
		  (alpha-equal-bound-term-lists (bound-terms-of-term termx)
						(bound-terms-of-term termy))))))

	 
    (or (eq termx termy)
	(and termx termy
	     (with-variable-invocation
		 (alpha-equal-terms (funcall transparent-filter termx)
				    (funcall transparent-filter termy)))))))
	 

;;;  ----------------------------------------------------------------------
;;;  ------------------- lexicographic equality ---------------------------
;;;  ----------------------------------------------------------------------

(defun compare-terms (term-a term-b)
  (let ((lower (lexicographically-compare-terms term-a term-b)))
    (cond 
      ((eq lower term-a) 'arg1)
      ((eq lower term-b) 'arg2)
      ((null lower) (raise-error (error-message '(compare) "null term")))
      (t nil))))

(defun compare-terms-p (term-a term-b)
  (eql nil (compare-terms term-a term-b)))

;; return lower term, t if equal, or nil if one but not both are nil.
(defun lexicographically-compare-terms (term-a term-b)
  (labels
    ((compare-symbols (sym-1 sym-2)
       (unless (eql sym-1 sym-2)
	 (if (string< (string sym-1) (string sym-2))
	     term-a
	     term-b)))
     
     (compare-strings (str-1 str-2)
       (unless (string= str-1 str-2)
	 (if (string< str-1 str-2)
	     term-a
	     term-b)))
     
     ;;(compare-integers (int-1 int-2)
     ;;(unless (eql int-1 int-2)
     ;;(if (< int-1 int-2)
     ;;term-a
     ;;term-b)))
     
     ;; display-meta < abstraction-meta
     (compare-meta-variables (m-1 m-2)
       (cond
	 ((display-meta-variable-id-p m-1)
	  (if (display-meta-variable-id-p m-2)
	      (compare-symbols m-1 m-2)
	      term-a))
	 ((display-meta-variable-id-p m-2)
	  term-b)
	 ((abstraction-meta-variable-id-p m-1)
	  (if (abstraction-meta-variable-id-p m-2)
	      (compare-symbols m-1 m-2)
	      (break "?")))
	 (t (break "?"))))

     ;; error < slot < meta < value
     (compare-parameter-values (a b type)
       (let ((v-1 (value-of-parameter-value a))
	     (v-2 (value-of-parameter-value b)))
	 (cond
	   ((error-parameter-value-p v-1)
	    ;; meta < string
	    (if (error-parameter-value-p v-2)
		(let ((sexpr-1 (sexpr-of-error-parameter-value v-1))
		      (sexpr-2 (sexpr-of-error-parameter-value v-2)))
		  (cond
		    ((consp sexpr-1)
		     (if (consp sexpr-2)
			 (cond
			   ((eql (car sexpr-1) (car sexpr-2))
			    (string= (cdr sexpr-1) (cdr sexpr-2)))
			   ;; display-meta < abstraction-meta
			   ((eql 'd (car sexpr-1)) term-a)
			   (t term-b))))
		    ((consp sexpr-2)
		     term-b)
		    (t (compare-strings sexpr-1 sexpr-2))))
		term-a))
	   ((error-parameter-value-p v-2)
	    term-b)
	   ((slot-parameter-value-p v-1)
	    (if (slot-parameter-value-p v-2)
		(let ((s-1 (descriptor-of-slot-parameter-value v-1))
		      (s-2 (descriptor-of-slot-parameter-value v-2)))
		  (cond
		    ((meta-variable-id-p s-1)
		     (if (meta-variable-id-p s-2)
			 (cond
			   ((display-meta-variable-id-p s-1)
			    (unless (display-meta-variable-id-p s-2)
			      term-a))
			   ((display-meta-variable-id-p s-2)
			    term-b)
			   ;; must be be abstraction metas as both metas and neither display metas.
			   (t nil))
			 term-a))
		    ((meta-variable-id-p s-2)
		     term-b)
		    (t nil)))
		term-a))
	   ((slot-parameter-value-p v-2) term-b)
	   ((meta-variable-id-p v-1)
	    (if (meta-variable-id-p v-2)
		(compare-meta-variables v-1 v-2)
		term-a))
	   ((meta-variable-id-p v-2) term-b)
	   (t (compare-strings (real-parameter-value-to-string v-1 type)
			       (real-parameter-value-to-string v-2 type))))))	    

     (compare-parameter-lists (parameters-1 parameters-2)
       (cond ((and (null parameters-1) (null parameters-2)) nil)
	     ((null parameters-1) term-a)
	     ((null parameters-2) term-b)
	     (t (let ((parameter-1 (car parameters-1))
		      (parameter-2 (car parameters-2)))
		  (or (compare-symbols (type-id-of-parameter parameter-1)
				       (type-id-of-parameter parameter-2))
		      (compare-parameter-values (value-of-parameter parameter-1)
						(value-of-parameter parameter-2)
						(type-of-parameter parameter-1))
		      (compare-parameter-lists (cdr parameters-1) (cdr parameters-2)))))))

     (compare-ops (op-1 op-2)
       (or (compare-symbols (id-of-operator op-1)
			    (id-of-operator op-2))
	   (compare-parameter-lists  (parameters-of-operator op-1)
				     (parameters-of-operator op-2))))

     (compare-terms (term-1 term-2)
       (unless (eql term-1 term-2)
	 (let ((d 
	 (or (compare-ops (operator-of-term term-1)
			  (operator-of-term term-2))
	     (compare-bound-term-lists (bound-terms-of-term term-1)
				       (bound-terms-of-term term-2)))))
	   ;;(when d (break))
	   d
	   )))
     
     (compare-binding-lists (bindings-1 bindings-2)
       (cond ((and (null bindings-1) (null bindings-2)) nil)
	     ((null bindings-1) term-a)
	     ((null bindings-2) term-b)
	     (t (or (compare-parameter-values (car bindings-1) (car bindings-2) *variable-type*)
		    (compare-binding-lists (cdr bindings-1) (cdr bindings-2))))))

     (compare-bound-term-lists (bterms-1 bterms-2)
       (cond ((and (null bterms-1) (null bterms-2)) nil)
	     ((null bterms-1) term-a)
	     ((null bterms-2) term-b)
	     (t (or (compare-binding-lists (bindings-of-bound-term-n (car bterms-1))
					   (bindings-of-bound-term-n (car bterms-2)))
		    (compare-terms (term-of-bound-term (car bterms-1))
				   (term-of-bound-term (car bterms-2)))
		    (compare-bound-term-lists (cdr bterms-1) (cdr bterms-2)))))) )

    (cond
      ((and (null term-a) (null term-b)) t)
      ((or (null term-a) (null term-b)) nil)
      (t (let ((term-comp (compare-terms term-a term-b)))
	   (if (null term-comp)
	       t
	       term-comp)))) ))

;;;  ----------------------------------------------------------------------
;;;  -----------------------    free-vars    ------------------------------
;;;  ----------------------------------------------------------------------
(defun free-vars (term)
  (if (free-vars-p term)
      (free-vars-of-term term)
      (let ((result nil))
	(labels
	  ((internal-free-vars (term)
	     (if (variable-p term)
	       (let ((var (id-of-variable-term term)))
		 (when (and (free-var-p var)
			    (not (markv-p var 'already-free)))
		   (markv var 'already-free)
		   (push var result)))
	       (mapc
		 #'(lambda (bound-term)
		     (enter-bindings (bindings-of-bound-term-n bound-term))
		     (internal-free-vars (term-of-bound-term bound-term))
		     (exit-bindings (bindings-of-bound-term-n bound-term)))
		 (bound-terms-of-term term)))))

	  (with-variable-invocation
	    (internal-free-vars term)))
	
	result)))

(defun free-vars-of-bound-term (bound-term)
  (diff-vars (free-vars (term-of-bound-term bound-term)) 
	     (bindings-of-bound-term-r bound-term)))



(defun cache-free-vars (term)
  (mark-f term 'free-vars #'free-vars)
  term)

(defun sticky-free-vars (term)
  (if (free-vars-p term)
      (free-vars-of-term term)
      (let ((fv (free-vars term)))
	(mark term 'free-vars fv)
	fv)))


;;;  ----------------------------------------------------------------------
;;;  -----------------------  occurs free    ------------------------------
;;;  ----------------------------------------------------------------------

;; takes either a var or a list of vars.
;; if given var returns var if var occurs free in term.
;; if given a list of vars returns those in list which occur free in term.

(defun occurs-free-p (var term)
  (let ((var (value-of-parameter-value var)))
    (labels
	((occurs-free (term)
	   (when (variable-p term)
	     (let ((id (id-of-variable-term term)))
	       (when (and (free-var-p id) (markv-p id 'occurs-free))
		 (return-from occurs-free t))))

	   (exists-p #'(lambda (bound-term)
			 (let ((var-bound nil))
			   (prog2
			       (mapc #'(lambda (var)
					 (let ((var (value-of-parameter-value var)))
					   (when (variable-id-p var)
					     (when (markv-p var 'occurs-free)
					       (setf var-bound t))
					     (enter-binding var))))
				     (bindings-of-bound-term-n bound-term))
			       (unless var-bound
				 (occurs-free (term-of-bound-term bound-term)))
			     (exit-bindings (bindings-of-bound-term-n bound-term)))))
		     (bound-terms-of-term term))) )

      (unless (variable-id-p var)
	(raise-error (error-message '(occurs-free)
				    (parameter-value-to-pretty-string var *variable-type*))))
      
      (if (free-vars-p term);; if term marked with free-vars.
	  (member var (free-vars-of-term term))
	  (with-variable-invocation
	      (markv var 'occurs-free)
	    (occurs-free term))) )))

(defun list-occurs-free-p (vs term)
  (let ((free (sticky-free-vars term)))
    (cond
      ((null vs)  nil)
      ((null (cdr vs)) (occurs-free-p (car vs) term))
      (t (not (null-intersect-vars-p vs free))))))


;;; Performs a one-way match treating the free occurrences of the
;;; pattern-vars as the meta-variables.  Throw foo to 'match if the
;;; match fails, otherwise return the appropriate a-list.

(defun match (pattern instance pattern-var-ids)
  (let ((it (get-it)))
    (labels
	((match-fail ()
	   (raise-error (basic-message '(match))))
	 (match-free-vars (term)
	   (let ((free-vars-accumulator nil))
	     (labels
		 ((internal-free-vars (term)
		    (if (variable-p term)
			(when (free-var-p (id-of-variable-term term))
			  (unless (set-variable-minor-use (id-of-variable-term term))
			    (push (id-of-variable-term term) free-vars-accumulator)))
			(mapc #'(lambda (bound-term)
				  (enter-bindings (bindings-of-bound-term-n bound-term))
				  (internal-free-vars (term-of-bound-term bound-term))
				  (exit-bindings (bindings-of-bound-term-n bound-term)))
			      (bound-terms-of-term term)))))

	       (with-variable-minor-invocation
		   (internal-free-vars term))
	       free-vars-accumulator)))

	 (match-alpha-equality (termx termy)
	   (labels
	       ((alpha-equal-bound-term-lists (listx listy)
		  (or (and (null listx) (null listy))
		      (and listx listy
			   (prog1
			       (and (apply-predicate-to-list-pair
				     (bindings-of-bound-term-n (car listx))
				     (bindings-of-bound-term-n (car listy))
				     #'(lambda (x y) (enter-binding-pair-it x y it)))
				    (alpha-equal-terms (term-of-bound-term (car listx))
						       (term-of-bound-term (car listy))))
			     (mapc #'(lambda (x y) (exit-binding-pair-it x y it))
				   (bindings-of-bound-term-n (car listx))
				   (bindings-of-bound-term-n (car listy))))
			   #+cmu(let ((r (alpha-equal-bound-term-lists (cdr listx) (cdr listy))))
				  r)
			   #-cmu(alpha-equal-bound-term-lists (cdr listx) (cdr listy))
			   )))

		(alpha-equal-terms (termx termy)
		  (if (and (variable-p termx)
			   (variable-p termy))
		      (equal-bindings-it (id-of-variable-term termx)
					 (id-of-variable-term termy)
					 it)
		      (and (equal-operators-p (operator-of-term termx) (operator-of-term termy))
			   (alpha-equal-bound-term-lists (bound-terms-of-term termx)
							 (bound-terms-of-term termy))))))

	     (or (eq termx termy)
		 (and termx termy
		      (alpha-equal-terms termx termy)))))


	 (found-match (var term)
	   (let ((previous (markv var 'match term)))
	     (when (and previous
			(not (match-alpha-equality term previous)))
	       (match-fail))))

	 (pattern-var-occurrence-p (term)
	   (and (variable-p term)
		(left-free-var-p (id-of-variable-term term))
		(markv-p (id-of-variable-term term) 'pattern-var)
		))
    
	 (match-bound-term-lists (p-list i-list)
	   (unless (and (null p-list) (null i-list))
	     (when (or (null p-list)
		       (null i-list)
		       (not (= (length (bindings-of-bound-term-n (car p-list)))
			       (length (bindings-of-bound-term-n (car i-list))))))
	       (match-fail))
	     (unless (enter-binding-pairs (bindings-of-bound-term-n (car p-list))
					  (bindings-of-bound-term-n (car i-list)))
	       (match-fail))
	     (match-term (term-of-bound-term (car p-list))
			 (term-of-bound-term (car i-list)))
	     (exit-binding-pairs (bindings-of-bound-term-n (car p-list))
				 (bindings-of-bound-term-n (car i-list)))
	     #+cmu(let ((r (match-bound-term-lists (cdr p-list) (cdr i-list))))
		    r)
	     #-cmu(match-bound-term-lists (cdr p-list) (cdr i-list))
	     ))

	 (match-term (p-term i-term)
	   (if (pattern-var-occurrence-p p-term)
	       (if (exists-p #'bound-var-p (match-free-vars i-term))
		   (match-fail)
		   (found-match (id-of-variable-term p-term) i-term))

	       (cond
		 ((not (equal-operators-p (operator-of-term p-term)
					  (operator-of-term i-term)))
		  (match-fail))
		 ;; bound variable
		 ((and (variable-p p-term)
		       (not (equal-bindings (id-of-variable-term p-term)
					    (id-of-variable-term i-term))))
		  (match-fail))
		 (t (match-bound-term-lists (bound-terms-of-term p-term)
					    (bound-terms-of-term i-term)))))))

      (with-variable-invocation
	  (mapc #'(lambda (var)
		    (unless (variable-id-p var)
			    (raise-error (error-message '(match arg)
							(parameter-value-to-pretty-string var *variable-type*))))
		    (markv var 'pattern-var))
		pattern-var-ids)
	;;(when pattern-var-ids (break "P"))
	(match-term pattern instance)
	(mapcan #'(lambda (var)
		    (let ((match (markv-value var 'match)))
		      (when match (list (cons var match)))))
		pattern-var-ids)))))
	  

;;; -------------------------------------------------------------------
;;; ----------------------- substitute --------------------------------
;;; -------------------------------------------------------------------

;;; substitute options
;;;  - maintain sameness and difference of binding vars.
;;;  - sub in second order terms.
;;;
;;;  subbee    - term being subbed into
;;;  subber    - variable being subbed for.
;;;  subbend   - item being subbed in for subber.
;;;  



(defmacro subber-of-sub (sub)
  `(car ,sub))
(defmacro subbend-of-sub (sub)
  `(cdr ,sub))

(defun make-sub-from-list (subl)
  (unless (and (variable-p (cdr subl))
	       (eql (car subl) (id-of-variable-term (cdr subl))))
    (list subl)))


(defmacro make-sub (subber subbend)
  `(cons ,subber ,subbend))

(defmacro with-free-vars-of-subbends ((subs) &body body)
  `(with-free-vars-l (mapcar #'(lambda (sub) (subbend-of-sub sub))
			     ,subs)
     ,@body))


(defvar *subst-count*)
(defvar *subst-count-bound*)

(defvar *max-substitution-counts* (list 0))
(defvar *num-substitution-count-breached* 0)

(defmacro with-substitution-count-bound ((b f) &body body)
  (let ((okp (gensym))
	(a (gensym)))
  `(let ((*subst-count* 0)
	 (*subst-count-bound* ,b)
	 (,okp nil))
    (let ((,a (catch 'subst-count-bound
		(prog1
		    (progn ,@body)
		  (setf ,okp t)))))

      ;;(setf -a ,a -okp ,okp) (break "wscb")
      
      (if ,okp
	  (progn
	    (format t "~%substitution-count ~a " *subst-count*)
	    (when (> *subst-count* (car *max-substitution-counts*))
	      (push *subst-count* *max-substitution-counts*)
	      (format t "New MAX~%")
	      (when (> (length *max-substitution-counts*) 16)
		(setf *max-substitution-counts* (butlast *max-substitution-counts* 8))))
	    ,a)
	  (progn
	    (incf *num-substitution-count-breached*)
	    (funcall ,f)))))))

(defun subst-count-incf ()
  ;;(break "sci")
  (when (boundp '*subst-count*) 
    (incf *subst-count*)
    (unless (zerop *subst-count-bound*)
      (when (> *subst-count* *subst-count-bound*)
	(format t "subst-count bound exceeded ~a " *subst-count*)
	(throw 'subst-count-bound nil)
	))))

(defun substitute (term input-subs  &key (maintain-p t))
  (declare (ignore maintain-p))
  (incf *no-subst1*)
  ;; rle todo make sure all calls to substitute good then remove following:
  (mapc #'(lambda (sub)
	    (when (consp (cdr sub))
	      ;;(setf a input-subs b sub)
	      (break "cdr sub is cons")))
	input-subs)
  (if (variable-p term)
      (let ((id (id-of-variable-term term)))
 	(incf *no-subst1-trivial*)
	(or (find-first #'(lambda (sub)
			    (when (eql (car sub) id)
			      (cdr sub)))
			input-subs)
	    term))
      (let ((subs (if (free-vars-p term)
		      (choose-by-vars (mapcan #'make-sub-from-list input-subs)
				      #'car
				      (free-vars-of-term term)
				      #'identity)
		      (mapcan #'make-sub-from-list input-subs))))
	(labels
	    ((substitute-term (term)
	       (if (variable-p term)
		   (let ((id (id-of-variable-term term)))
		     (when (free-var-p id)
		       (let ((sub (markv-value id 'sub)))
			 (when sub
			   (if (forall-p #'free-var-p (free-vars sub))
			       sub
			       (process-err "capture"))))))
		   (let ((bound-terms (bound-terms-of-term term)))
		     (cond
		       ((null bound-terms) nil)
		       ((null (cdr bound-terms))
			(let ((bound-term (car bound-terms)))
			  (let ((new-term (prog2
					      (enter-bindings (bindings-of-bound-term-n bound-term))
					      (substitute-term (term-of-bound-term bound-term))
					    (exit-bindings (bindings-of-bound-term-n bound-term)))))
			    (when new-term
			      (instantiate-term
			       (operator-of-term term)
			       (list (instantiate-bound-term new-term
							     (bindings-of-bound-term-n bound-term))))))))
		       ;; RLE NAP might work better to just collect new bound-terms and their indices within
		       ;; the term, then instantiate new bound-terms list from original and new. Allows sharing
		       ;; of tail of bound-term list.
		       (t (do ((new nil)
			       (first nil)
			       (i 0 (1+ i))
			       (bound-terms (bound-terms-of-term term) (cdr bound-terms)))
			      ((null bound-terms)
			       (when new
				 (instantiate-term (operator-of-term term)
						   (nconc (subseq (bound-terms-of-term term) 0 first)
							  (nreverse new)))))
			    (let ((bound-term (car bound-terms)))
			      (prog2
				  (enter-bindings (bindings-of-bound-term-n bound-term))
				  (let ((sterm (substitute-term (term-of-bound-term bound-term))))
				    (if sterm
					(progn
					  (unless new (setf first i))
					  (push (instantiate-bound-term
						 sterm
						 (bindings-of-bound-term-n bound-term))
						new))
					(when new (push bound-term new))))
				(exit-bindings (bindings-of-bound-term-n bound-term)))))))))))

	  (if subs
	      (handle-process-err #'(lambda (x)
				      (declare (ignore x))
 				      (incf *no-subst1-renamed*)
				      (second-order-substitution-with-maintenance
				       term nil
				       (mapcar #'(lambda (sub)
						   (cons (car sub)
							 (cons nil (cdr sub))))
					       subs))
				      ;; maintain-p
				      ;; rle todo: need streamlined substitute with global renaming.
				      )

				  (with-free-vars-l (mapcar #'(lambda (sub) (subbend-of-sub sub)) subs)
				    (with-variable-invocation
					(progn
					  (mapc #'(lambda (sub)
						    (let ((var (subber-of-sub sub)))
						      (unless (variable-id-p var)
							(raise-error
							 (error-message '(occurs-free)
									(parameter-value-to-pretty-string
									 var
									 *variable-type*))))
						      (unless (markv-p var 'sub)
							(markv var 'sub (subbend-of-sub sub)))))
						subs)
					  (let ((r (or (substitute-term term) term)))
					    (subst-count-incf)
					    r)))))
	      term)))))


	   
							     

;; p[.t-i] p<- x-i.s => sa(x-i.s; .t-1; ...; .t-i)
;; p[] p<- .s => sa(.s)

;; mark vars with arity of subs for each subbend.
;; subbers must include arities.

;;(define-primitive sub-apply  nil (i 0 .. 0) (subbee subbends))
;; expects either a bound-term  and list of bound-terms
;; whose length equals the number of bindings of the first
;; bound-term.

(defparameter *isub-apply* '|!sub_apply|)
(defparameter *isub-apply-op* (instantiate-operator *isub-apply* nil))
    
(defun isub-apply-term-p (term)
  (eql *isub-apply* (id-of-operator (operator-of-term term))))

(defun isub-apply-term (subbee subbends)
  (instantiate-term *isub-apply-op*
		  (cons subbee subbends)))

(defun subbee-of-isub-apply-term (term)
  (car (bound-terms-of-term term)))

(defun subbends-of-isub-apply-term (term)
  (cdr (bound-terms-of-term term)))


(defmacro id-of-subber (subber)
  `(car ,subber))
(defmacro arity-of-subber (subber)
  `(cdr ,subber))
(defmacro make-subber (id arity)
  `(cons ,id ,arity))

;;;
;;; RLE NAP need a better sos without maintenance. OR clean this one up.
;;;

(defun s-substitute (term input-subs)
  (labels
    (
     (subbend-of-subber (subber)
       (cdr (markv-peek (id-of-subber subber) (arity-of-subber subber))))

     (subbend-of-variable-term (term)
       (cdr (markv-peek (id-of-variable-term term)
			(length (bound-terms-of-term term)))))

     (flag-of-subber (subber)
       (car (markv-peek (id-of-subber subber) (arity-of-subber subber))))

     (set-flag-of-subber (subber flag)
       (setf (car (markv-peek (id-of-subber subber) (arity-of-subber subber))) flag))

     (push-subber (id arity subbend)
       (let ((var (value-of-parameter-value id)))
	 (unless (variable-id-p var)
	   (raise-error (error-message '(sos no-maint variable) 
				       (parameter-value-to-pretty-string
					var
					*variable-type*))))	
	 (markv-push var arity (cons nil subbend))))
     
     (pop-subber (subber)
       (markv-pop (id-of-subber subber) (arity-of-subber subber)))

     (intersect-subbers-with-occurences (subbers occurences)
       (prog2
	 (mapc #'(lambda (subber) (set-flag-of-subber subber t))
	       subbers)
	 (mapcan #'(lambda (occurence)
		     (when (flag-of-subber occurence)
		       (list occurence)))
		 occurences)
	 (mapc #'(lambda (subber) (set-flag-of-subber subber nil))
	       subbers)))

;     (second-order-occurences-as-subbers (occurences &optional more-occurences)
;       (nconc
;	 (mapcan #'(lambda (occurence)
;		     (when (markv-p (id-of-subber occurence)
;				    (arity-of-subber occurence))
;		       (list occurence)))
;		 occurences)
;	 (mapcan #'(lambda (occurence)
;		     (when (markv-p (id-of-subber occurence)
;				    (arity-of-subber occurence))
;		       (list occurence)))
;		 more-occurences)))
       
     ;; not maintaining sameness and diff
     (renamings-of-bound-term (bound-term subbers)
       (labels
	 ((project-free-vars (subbers)
	    (unionl-vars 
	      (mapcar #'(lambda (subber)
			  (free-vars-of-bound-term
			    (subbend-of-subber subber)))
		      subbers)))

	  (make-renaming (binding reserved)
	    (let ((new-var (variable-term (get-similar-allowed-id binding reserved))))
	      (mark-incf new-var
			 'free-vars 
			 (list (id-of-variable-term new-var)))
	      (make-sub (make-subber binding 0)
			(instantiate-bound-term new-var))))


	  (zero-arity-subber-ids (subbers)
	    (mapcan #'(lambda (subber)
			(when (null (bindings-of-bound-term-n (subbend-of-subber subber)))
			  (list (id-of-subber subber))))
		    subbers)))
	 
	 (let* ((real-bindings (real-bindings (bindings-of-bound-term-n bound-term)))
		(projected-free-vars (project-free-vars subbers))
		(capturing (intersect-vars real-bindings
					   projected-free-vars )))

	   (when capturing
	     (let ((reserved (list projected-free-vars
				   real-bindings
				   (diff-vars
				    (free-vars (term-of-bound-term bound-term))
				    (zero-arity-subber-ids subbers)))))

	       (mapcar #'(lambda (capture)
			   (make-renaming capture reserved))
		       capturing))))))

     ;; if called then must have subs which are applicable to term.
     (substitute-term (term subbers)
       (when (variable-term-p term)
	 (let ((subbend (subbend-of-variable-term term)))
	   (when subbend
	     (setf term
		   (if (null (bindings-of-bound-term-n subbend))
		       (term-of-bound-term subbend)
		       (isub-apply-term subbend (bound-terms-of-term term)))))))

       (if (null (bound-terms-of-term term))
	   term
	   (instantiate-term 
	     (operator-of-term term)
	     (mapcar #'(lambda (bound-term)
			 (substitute-bound-term 
			   bound-term
			   (intersect-subbers-with-occurences
			     subbers
			     (second-order-occurences-of-bound-term bound-term))))
		     (bound-terms-of-term term)))))


     (substitute-bound-term (bound-term subbers)
       (let ((renamings (renamings-of-bound-term bound-term subbers)))
	 (if (and (null subbers) (null renamings))
	     bound-term
	     ;; substitute
	     (let ((new-subbers 
		     (nconc subbers
			    (intersect-subbers-with-occurences 
			      (mapcar #'(lambda (r)
					  (push-subber (id-of-subber (subber-of-sub r))
						       (arity-of-subber (subber-of-sub r))
						       (subbend-of-sub r))
					  (subber-of-sub r))
				      renamings)
			      (second-order-occurences-of-term 
				(term-of-bound-term bound-term))))))


	       (prog1
		 ;; substitute
		 (instantiate-bound-term

		   (if new-subbers
		       (substitute-term (term-of-bound-term bound-term)
					new-subbers)
		       (term-of-bound-term bound-term))

		   ;; rename-bindings
		   (if renamings
		       (mapcar #'(lambda (var-id)
				   (let* ((var (value-of-parameter-value var-id))
					  (value (find-if #'(lambda (r) 
							     (eql var (id-of-subber
								       (subber-of-sub r))))
							 renamings)))
				     (if value
					 (id-of-variable-term 
					  (term-of-bound-term (subbend-of-sub value)))
					 var-id)))
			       (bindings-of-bound-term-n bound-term))
		       (bindings-of-bound-term-n bound-term)))
	       
		 ;; cleanup
		 (mapc #'(lambda (r)
			   (pop-subber (subber-of-sub r))
			   (mark-decf (term-of-bound-term (subbend-of-sub r))
				      'free-vars))
		       renamings)))))) )

   
    ;; substitute
    (with-second-order-variable-occurence-tree (term)
      ;; mark subbers as such 
      (with-free-vars-l (mapcar #'(lambda (input-sub)
				    (term-of-bound-term (cdr input-sub)))
				input-subs)
	(with-variable-invocation
	  (let ((subbers
		  (intersect-subbers-with-occurences
		    (mapcar #'(lambda (input-sub)
				(push-subber (car input-sub)
					     (length (bindings-of-bound-term-n (cdr input-sub)))
					     (cdr input-sub))
				(cons (car input-sub)
				      (length (bindings-of-bound-term-n (cdr input-sub)))))
			    input-subs)
		    (second-order-occurences-of-term term))))
	    (if subbers
		;; rle NAP change to use make-free-var-tree as uses less space.
		(with-free-var-tree (term)
		  (substitute-term term subbers))
		term))))) ))


(defun sos-reduce (term parm-subs)
  (labels
      ((sreduce (term)
	 (if (isub-apply-term-p term)
	     (substitute (term-of-bound-term (subbee-of-isub-apply-term term))
			 (mapcar #'(lambda (binding bound-term)
				     (list binding
					   (sreduce (term-of-bound-term bound-term))))
				 (bindings-of-bound-term-r (subbee-of-isub-apply-term term))
				 (subbends-of-isub-apply-term term)))
	     (instantiate-term
	      (substitute-in-operator (operator-of-term term) parm-subs)
	      (mapcar #'(lambda (bound-term)
			  (let ((term (sreduce (term-of-bound-term bound-term))))
			    (if (eq term (term-of-bound-term bound-term))
				bound-term
				(instantiate-bound-term
				 term (bindings-of-bound-term-n bound-term)))))
		      (bound-terms-of-term term))))))

    (sreduce term)))
				 

(defun second-order-substitute (term parm-subs subs)
  (subst-count-incf)
  ;;(format t "second-order w/sharing~%")
  (if (and (null parm-subs) (null subs))
      term
      (sos-reduce (s-substitute term subs) parm-subs)))


(defun substitute-parameters-in-term (term subs)
  (labels
      ((visit (term)
	 (abstraction-meta-variable-term-r term)
	 (maybe-instantiate-term
	  term
	  (substitute-in-operator (operator-of-term term) subs)
	  (mapcar #'(lambda (bound-term)
		      (maybe-instantiate-bound-term
		       bound-term
		       (bindings-of-bound-term-n bound-term)
		       (visit (term-of-bound-term bound-term))))
		  (bound-terms-of-term term)))))

    (if subs
	(visit term)
	term)))

(defun level-variables-of-term (term)
  (let ((accumulator nil))
    (labels ((visit (term)
	       (mapc #'(lambda (parameter)
			 (when (level-expression-parameter-p parameter)
			   (walk-parameter parameter
					    #'(lambda (v)
						(unless (member v accumulator)
						  (push v accumulator))
						v))))
		     (parameters-of-term term))
	       (mapc #'(lambda (bound-term)
			 (visit (term-of-bound-term bound-term)))
		     (bound-terms-of-term term))))

      (visit term))

    (nreverse accumulator)))



		       
(defun subterm-of-term (term l) ;; indices start at 0, l is list of indices.
  (if (null l)
      term
      (let ((nterm (term-of-bound-term (nth (car l)
					    (bound-terms-of-term term)))))
	(subterm-of-term nterm (cdr l)))))



;;(proclaim '(function hash-parameters (list) fixnum))

(defun hash-parameters (parms)
  (proclaim `(ftype (function (list) fixnum) hash-parameters))
  (do* ((rot-amt 1 (mod (+ rot-amt 3) 15))
	(restp parms (cdr restp))
	(value 0))
       ((null restp) (hash-rot value 5))
    (setf value 
	  (logxor value (hash-rot (hash-parameter (car restp)) rot-amt)))))


;;(defun hash-operator (op)
;;  (logxor (sxhash (id-of-operator op))
;;	  (hash-parameters (parameters-of-operator op))))


(defun hash-term (term &optional ignore-free-vars-p)
  (labels
    ((hash-parameters (parms)
       (do* ((rot-amt 1 (mod (+ rot-amt 3) 15))
	     (restp parms (cdr restp))
	     (value 0))
	    ((null restp) (hash-rot value 5))
	 (setf value 
	       (logxor value (hash-rot (hash-parameter (car restp)) rot-amt)))))

     (hash-operator (op)
       (logxor (sxhash (id-of-operator op))
	       (hash-parameters (parameters-of-operator op))))

     (hash-trm (term)
       (if (variable-p term)
	   (if (free-var-p (id-of-variable-term term))
	       (if ignore-free-vars-p
		   1
		   (sxhash (id-of-variable-term term)))
	       (depth-of-variable-id (id-of-variable-term term)))
	   (logxor (hash-operator (operator-of-term term))
		   (do* ((rot-amt 1 (mod (+ rot-amt 3) 15))
			 (rest (bound-terms-of-term term) (cdr rest))
			 (value 0))
			((null rest) (hash-rot value 5))
		     (setf value (logxor value 
					 (hash-rot (hash-bound-term (car rest)) rot-amt))))) ))

     (hash-bound-term (bound-term)
       (prog2
	 (enter-bindings (bindings-of-bound-term-n bound-term))
	 (hash-trm (term-of-bound-term bound-term))
	 (exit-bindings (bindings-of-bound-term-n bound-term)))) )

    (with-variable-invocation
      (hash-trm term))))



(defun term-stats (term &optional (printp t))
  (let ((op-count 0)
	(bindings-count 0)
	(null-bindings-count 0)
	(parm-count 0)
	(max-bindings 0)
	(max-depth 1)
	(max-width 1)
	)
    (labels
	((visit-term (term depth)
	   (setf max-depth (max max-depth depth))
	   (visit-op (operator-of-term term))
	   (setf max-width (max max-width (length (bound-terms-of-term term))))
	   (dolist (bt (bound-terms-of-term term))
	     (let ((l (length (bindings-of-bound-term bt))))
	       (incf bindings-count l)
	       (if (zerop l)
		   (incf null-bindings-count)
		   (setf max-bindings (max max-bindings l)))
	       (visit-term (term-of-bound-term bt) (1+ depth)))))

	 (visit-op (op)
	   (incf op-count)
	   (incf parm-count (length (parameters-of-operator op)))
	   ))

      (if (listp term)
	  (dolist (term term)
	    (visit-term term 1))
	  (visit-term term 1))

      (when printp
	(format t "Bindings count  : ~a~%" bindings-count)
	(format t "Parameter count : ~a~%" parm-count)
	(format t "Operator count  : ~a~%" op-count)
	(format t "Null bindings   : ~a~%" null-bindings-count)
	(format t "Max bindings    : ~a~%" max-bindings)
	(format t "Max Depth       : ~a~%" max-depth)
	(format t "Max Width       : ~a~%" max-width))

      (list op-count parm-count bindings-count))))


(defun term-op-count (term)
  (let ((count 1))
    (mapc #'(lambda (bt)
	      (incf count (term-op-count (term-of-bound-term bt))))
	  (bound-terms-of-term term))
    count))

(defun term-op-count-exceeds-p (term threshold)
  (let ((i threshold))
    (labels
	((visit (term)
	   (if (= 0 i)
	       (return-from term-op-count-exceeds-p t)
	       (progn
		 (setf i (1- i))
		 (mapc #'(lambda (bt) (visit (term-of-bound-term bt)))
		       (bound-terms-of-term term))))))
      (visit term)
      nil)))


		       

