
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)



#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      in-use variable-value 
	      evcb-of-variable-id set-evcb-of-variable-id
	      free-var-p bound-var-p depth-of-variable-id value-of-variable-id value-of-binding
	      binding-of-variable-id bindings-of-variable-id value-of-variable-binding
	      left-free-var-p left-bound-var-p right-free-var-p right-bound-var-p
	      enter-binding-pairs exit-binding-pairs
	      markv markv-p markv-value unmarkv markv-push markv-pop markv-peek
	      same-vars duplicate-vars-p 
	      )))

;;;;  ------------------------------------------------------------------------
;;;;  ------------------------- Variables ------------------------------------
;;;;  -doc s-------------------------------------------------------------------
;;;;  -level-ff-
;;;;
;;;;	  An unusual feature of the variable implementation is a method of
;;;;	abstractly accessing a dynamically scoped property-list associated
;;;;	with the variable.  For example, when computing the free variables
;;;;	of a term, one needs only increment/decrement a count when a binding
;;;;	scope is entered/exited, and then a variable is free if an occurence
;;;;    is encountered when the count is zero.
;;;;	  There are several issues :
;;;;	    - accumulation/cleanup of junk on propertry lists.
;;;;	    - cost 
;;;;		- overhead of implementation structures.
;;;;		- efficiency of access.
;;;;	    - recursive, ie within the scope of a property list one
;;;;	      should be able to create and access a new property list.
;;;;
;;;;    implementation uses package, global counter to invalidate/clean junk
;;;;    fails third, but hacks around it. but fast access, medicore overhead.
;;;;
;;;;   alternate implementation would be create new hash table each invoc.
;;;;
;;;;   really make it abstract so either implementation could be used!
;;;;    hard to do with hacks for recursive prob.
;;;;
;;;;  best solution would be hybrid. One one-deep in recursion use
;;;; symbol's value. subsequent recursions would create their own hash
;;;; table.
;;;;
;;;;  -doce-------------------------------------------------------------------

;;;;  -doc-s-------------------------------------------------------------------
;;;;  -level-ff-
;;;;  -level-01-
;;;;  marks 		: filter-marks (variable-id|marks)
;;;;   ; hack to allow variable-ids as input to mark functions. the mark
;;;;   ; functions then use filter-marks to insure they have marks.
;;;;
;;;;
;;;; ;;; Map functions on sets of vars.
;;;;
;;;;  void 		: map-on-vars 	function 
;;;;					variable-ids
;;;;					lists:*...* ... *...*    ;optional
;;;;
;;;;  *s		: mapl-on-vars  function
;;;;					variable-ids
;;;;					lists:*...* ... *...*    ;optional
;;;;
;;;;  *s 		: mapr-on-vars  function
;;;;					variable-ids
;;;;					lists:*...* ... *...*    ;optional
;;;;
;;;;  bool		: mapt-on-vars  function
;;;;					variable-ids
;;;;					lists:*...* ... *...*    ;optional
;;;;
;;;;  bool		: mapn-on-vars  function
;;;;					variable-ids
;;;;					lists:*...* ... *...*    ;optional
;;;;  -level-ff-
;;;;  -page------------------------------------------------------------------
;;;;  -level-08-
;;;;   
;;;; ;;; variable invocations : a variable-invocation is a dynamic scoping
;;;; ;;;  of the use of variable-id objects. Only a single variable invocation
;;;; ;;;  may be active at any one time and any number of objects may be used
;;;; ;;;  during the invocation. This eliminates the need to clean up
;;;; ;;;  data stored in the object during the execution of an algorithm but
;;;; ;;;  requires care to insure that no recursive variable invocations are
;;;; ;;;  made. Most other functions described in this file must be called
;;;; ;;;  from within an invocation. Those which can be called from outside of
;;;; ;;;  an invocation will be noted.
;;;;
;;;;   void		with-variable-invocation (&body body)   
;;;;	; macro which executes body within a variable invocation.
;;;;   
;;;; ;;; minor variable-invocation : a minor invocation is similar but effects
;;;; ;;;  only a boolean flag. Minor invocations may (but do not have to) be 
;;;; ;;;  used within major invocations but not within other minor invocations.
;;;; 
;;;;  void		: with-variable-minor-invocation (&body body)
;;;;	; macro which executes body within a minor variable invocation.
;;;; 
;;;;  bool             	: set-variable-minor-use (id:variable-id)
;;;;	; returns value of flag prior to setting
;;;;	; sets flag to true.
;;;;
;;;;  bool		: reset-variable-minor-use (id:variable-id)
;;;;	; returns value of flag prior to resetting.
;;;;	; sets flag to nil.
;;;;
;;;;    
;;;;  bool		: variable-minor-use-p (id:variable-id)
;;;;	; returns t if id has been checked nil otherwise.	
;;;;	; does not set flag.
;;;;
;;;;  -level-ff-
;;;;  -page------------------------------------------------------------------
;;;;
;;;; ;;; renamings - renamings are constructed from an id by appending 
;;;; ;;;  an @ and an integer to the current id. 
;;;; ;;;  Any ids which are derived from the same id by renamings are 
;;;; ;;;  considered similar.
;;;; ;;; NOTE : (x ^= y & (x similar-to u) & (y similar-to v)) ==> u ^= v.
;;;;
;;;;  variable-id	: get-similar-unused-id	(id:variable-id)
;;;;	; returns variable-id similar to id which has not been used during
;;;;	; the current variable-invocation.
;;;;
;;;;  variable-id	: get-similar-allowed-id 
;;;;				old:variable-id 
;;;; 				reserved:variable-ids
;;;;	; returns id similar to old but not the same as any of the reserved.
;;;;	; may be used outside of an invocation.
;;;;
;;;;  variable-ids	: get-similar-allowed-ids
;;;;				old:variable-ids   
;;;;				reserved:list-of-a-list-of-variable-ids  
;;;;	; returns set of ids which are similar to the old and not the same 
;;;;	;   as any of the reserved.
;;;;	; may be used outside of an invocation.
;;;;
;;;;  -level-01-
;;;;  variable-id	: trim-variable-id (id:variable-id)
;;;;	; trims @x from end of id when x is an integer.
;;;;	; may be used outside of an invocation.
;;;;
;;;;  variable-id	: strip-variable-id (id:variable-id)
;;;;	; repeatedly trims id until it can be trimmed no more.
;;;;	; may be used outside of an invocation.
;;;;
;;;;  -level-08-
;;;;  -page------------------------------------------------------------------
;;;;
;;;; ;;; mark functions : reinterpretion of marks given the well defined
;;;; ;;;   life-span of a variable-invocation obviates the need for unmarking.
;;;;
;;;;   *		: markv (variable-id label:* value:*)
;;;;   bool		: markv-p (variable-id label:*)
;;;;   *		: markv-value (variable-id label:*)
;;;;   void		: unmarkv (variable-id label:*)
;;;;   void		: markv-push (variable-id label:* value:*)
;;;;   *		: markv-pop (variable-id label:*)
;;;;   *		: markv-peep (variable-id label:*)
;;;;
;;;; ;;; note that the usual mark functions may be used with the variable-id
;;;; ;;; the difference is that these are automatically unmarked
;;;; ;;; at the end of a variable invocation, while the usual need to be
;;;; ;;; explicitly unmarked.
;;;;
;;;; ------------------------------------------------------------------------
;;;;  -level-ff-
;;;;
;;;; ;;; Set functions
;;;; ;;; may be used outside of an major invocation, but not inside a minor.
;;;;
;;;;   variable-ids	: unionl-vars (list-of-list-of-vars)
;;;; 
;;;;   variable-ids	: union-vars 
;;;;                      	list-of-vars 
;;;;				more-lists-of-vars	; optional
;;;; 
;;;;   variable-ids	: intersectl-vars (list-of-list-of-vars)
;;;; 
;;;;   variable-ids	: intersect-vars
;;;;                      	list-of-vars 
;;;;				more-lists-of-vars	; optional
;;;; 
;;;;   variable-ids	: diff-vars (list-of-vars list-of-vars)
;;;;	; 1st list - 2nd list.
;;;;  
;;;;   bool		: duplicate-vars-p (list-of-vars)
;;;	; t if any var occurs more than once in list of vars.
;;;;
;;;;  -page------------------------------------------------------------------
;;;;  -level-08-
;;;;     
;;;; ;;; deBruijn functions : when traversing a term the deBruijn functions
;;;; ;;;   allow a consistent approach to stacking the bindings and
;;;; ;;;   associating arbitrary data with a binding. They also track the 
;;;; ;;;   depth of the stack.
;;;; ;;;  there are also functions to assist in traversing a pair of terms
;;;; ;;;   simultaneously.
;;;;
;;;;  Object : binding 
;;;;    
;;;;  *			: value-of-binding (binding)	
;;;;	; returns nil if binding nil.
;;;;
;;;;  integer		: depth-of-variable-id (id:variable-id)
;;;;	; depth of last binding of id.
;;;;
;;;;  *			: value-of-variable-id (id:variable-id)
;;;;	; value of last binding of id.
;;;;
;;;;  bindings	enter-binding (id:variable-id &optional value:*)
;;;;	; returns stack of bindings for id
;;;;
;;;;  void		: enter-bindings (list-of-vars)
;;;;
;;;;  binding		: exit-binding (variable-id)
;;;;	; returns binding exited.
;;;;
;;;;  void		: exit-bindings (list-of-vars)
;;;;
;;;;  bool		: bound-var-p (variable-id)
;;;;
;;;;  bool		: free-var-p (variable-id)
;;;;
;;;;  binding		: binding-of-variable-id (variable-id)
;;;;
;;;;  bindings		: bindings-of-variable-id (variable-id)
;;;;
;;;;  void 		: enter-binding-pair (variable-id variable-id)
;;;;
;;;;  void		: enter-binding-pairs (list-of-vars list-of-vars)
;;;;
;;;;  void		: exit-binding-pair (variable-id variable-id)
;;;;
;;;;  void		: exit-binding-pairs (list-of-vars list-of-vars)
;;;;	; traverses lists in reversed order.
;;;;
;;;;  bool		: equal-bindings (variable-id variable-id)
;;;;	; compares bindings of variable-ds for alpha equality.
;;;;
;;;;  bool		: left-bound-var-p (variable-id)
;;;;
;;;;  bool		: left-free-var-p (variable-id)
;;;;
;;;;  bool		: right-bound-var-p (variable-id)
;;;;
;;;;  bool		: right-free-var-p (variable-id)
;;;;
;;;;  -level-01-
;;;;  -page------------------------------------------------------------------
;;;;
;;;; ;;; Abstraction Expansion functions.
;;;;
;;;;  Object : evcb - expansion variable control block.
;;;;	; data associated with variable-id during an expansion
;;;;
;;;;  evcb		: evcb-of-variable-id (variable-id)
;;;;	; returns the current evcb if set within current var invocation.
;;;;	; returns nil otherwise
;;;;
;;;;  evcb		: set-evcb-of-variable-id (variable-id evcb)
;;;;
;;;;  -level-80-
;;;;  -page------------------------------------------------------------------
;;;;
;;;; ;;; Variable operator
;;;;
;;;;
;;;;  bool		: variable-id-evcb-p (variable-id)
;;;;    ; returns t if evcb set withing current var invocation, nil otherwise.
;;;;
;;;;  Object : variable - injection of variable into operator
;;;;
;;;;  operator		: instantiate-variable-operator
;;;;				variable-id
;;;;				arity	;optional -> second-order variables.
;;;;
;;;;  bool		: variable-operator-p (operator)
;;;;
;;;;  variable-id	; id-of-variable-operator (operator)
;;;;	; operator must satisfy variable-operator-p
;;;;
;;;;
;;;;  -doce-------------------------------------------------------------------



;;;; ---------------------------------------------------------------------
;;;; ----------------------- globals -------------------------------------
;;;; ---------------------------------------------------------------------

(defvar *prl-var-invocation-count* 1)
(defvar *prl-in-var-invocation-p* nil)
(defvar *prl-var-minor-invocation-count* 1)
(defvar *prl-var-debruijn-index* 0)

;;; checks if var has been used in current minor invocation
;;; returns t if in use
;;; returns nil if not but puts in use.
(defun set-variable-minor-use (var-sym)
  (let ((value (value-of-parameter-value var-sym)))
    (unless (symbolp value)
      (raise-error (parameter-error-message (variable value) value *variable-type*)))
    (prog1
	(= (prl-var-current-minor-use (symbol-value value))
	   *prl-var-minor-invocation-count*)
      (setf (prl-var-current-minor-use (symbol-value value)) 
	    *prl-var-minor-invocation-count*))))


(defun reset-variable-minor-use (var-sym)
  (let ((value (value-of-parameter-value var-sym)))
    (unless (symbolp value)
      (raise-error (parameter-error-message (variable value) value *variable-type*)))
    (prog1
	(= (prl-var-current-minor-use (symbol-value value)) *prl-var-minor-invocation-count*)
      (setf (prl-var-current-minor-use (symbol-value value)) 0))))

;;; checks if var has been used in current minor invocation
;;; returns t if in use nil if not.
(defun variable-minor-use-p (var-sym)
  (let ((value (value-of-parameter-value var-sym)))
    (unless (symbolp value)
      (raise-error (parameter-error-message (variable value) value *variable-type*)))
    (= (prl-var-current-minor-use (symbol-value value)) *prl-var-minor-invocation-count*)))


;;; checks if var has been used in current invocation
;;; returns t if in use nil if not.
(defun in-use (var-val)
  (when (not *prl-in-var-invocation-p*)
    (error "variable value being accessed while no var invocation"))
  (when (= *prl-var-invocation-count* (prl-var-current-use var-val))
    var-val))

(defun check-use (var-val)
  (if (in-use var-val)
      t
      (progn
	(clear-marks (prl-var-vmarks var-val))
	(setf (prl-var-current-use var-val) *prl-var-invocation-count*)
	(setf (prl-var-bindings var-val) nil)
	(setf (prl-var-left-indices var-val) nil)
	(setf (prl-var-right-indices var-val) nil)
	(setf (prl-var-evcb var-val) nil)
	nil)))

;;; checks if var has been used in current invocation
;;; returns t if in use
;;; returns nil if not but puts in use.

(defun variable-value (var-sym &optional (check-use-p t))
  (let ((sym-val (variable-prl-var var-sym)))
    (when check-use-p (check-use sym-val))
    sym-val))

(defun maybe-variable-value (var-sym)
  (when (symbolp var-sym)
    (let ((val (symbol-value var-sym)))
      (when (in-use val)
	val))))



;;;;
;;;;	variable invocations
;;;;

(defun start-var-invocation ()

  (when *prl-in-var-invocation-p*
    (error "var invocation started while one was in progress"))

  (when (= most-positive-fixnum *prl-var-invocation-count*)

    (do-symbols (var-sym *system-variable-package-name*) 
      (setf (prl-var-current-use (symbol-value var-sym)) 0))

    (do-symbols (var-sym *system-display-meta-variable-package-name*) 
      (setf (prl-var-current-use (symbol-value var-sym)) 0))

    (do-symbols (var-sym *system-abstraction-meta-variable-package-name*) 
      (setf (prl-var-current-use (symbol-value var-sym)) 0))

    (setf *prl-var-invocation-count* 1))

  (incf *prl-var-invocation-count*)
  (setf *prl-in-var-invocation-p* t)
  (setf *prl-var-debruijn-index* 0))


(defun end-var-invocation ()
  (unless *prl-in-var-invocation-p*
    (error "var invocatation ended without one in progress"))
  (setf *prl-in-var-invocation-p* nil))

(defmacro with-variable-invocation (&body body)
  `(unwind-protect 
       (progn (start-var-invocation)
	      ,@body)
     (end-var-invocation)))

(defvar *minor-var-invocation-p* nil)

(defun start-minor-var-invocation ()
  (when (= most-positive-fixnum *prl-var-minor-invocation-count*)
    (do-symbols (var-sym *system-variable-package-name*)
      (setf (prl-var-current-minor-use (symbol-value var-sym)) 0))
    (do-symbols (var-sym *system-display-meta-variable-package-name*)
      (setf (prl-var-current-minor-use (symbol-value var-sym)) 0))
    (do-symbols (var-sym *system-abstraction-meta-variable-package-name*)
      (setf (prl-var-current-minor-use (symbol-value var-sym)) 0))
    (setf *prl-var-minor-invocation-count* 1))

  (when *minor-var-invocation-p*
    (warn "starting minor variable invocation when one in progress"))
  (setf *minor-var-invocation-p* t)
  (incf *prl-var-minor-invocation-count*))

(defun end-minor-var-invocation ()
  (unless *minor-var-invocation-p*
    (error "ending minor variable-invocation when one not in progress"))
  (setf *minor-var-invocation-p* nil))

(defmacro with-variable-minor-invocation (&body body)
  `(unwind-protect
       (progn (start-minor-var-invocation)
	  ,@body)
     (end-minor-var-invocation)))



;;;; renaming stuff

(defun get-new-id (id number)
  (get-variable-id (concatenate 'string "@" (string id) (format nil "~a" number))))

(defun make-variable-id (id number)
  (get-variable-id (concatenate 'string 
				(string id)
				"@"
				(format nil "~a" number))))

(defun get-similar-unused-id (id)
  (do ((i 0 (1+ i))
       (new-id id (get-variable-id (concatenate 'string "@" (string id) (format nil "~a" i)))))
      ((not (in-use (symbol-value new-id))) new-id)))

(defun set-minor-use-in-variable-id-tree (tree)
  (if (listp tree)
      (mapc #'set-minor-use-in-variable-id-tree tree)
      (set-variable-minor-use tree)))
	    
(defun get-similar-allowed-id (id tree-of-unalloweds &key trim-p)
  (with-variable-minor-invocation
    (set-minor-use-in-variable-id-tree tree-of-unalloweds)
    ;; with trim-p not nil when renaming a previously renamed variable undo the last renaming
    ;; and re-rename, but dont allow the unrenamed name.
    (if (not (variable-minor-use-p id))
	id
	(let ((trimmed (if trim-p (trim-variable-id id) id)))
	  (do ((i 1 (1+ i))
	       (new-id (get-variable-id
			 (concatenate 'string 
				      (string trimmed)
				      "@"
				      (format nil "~a" 0)))
		       (get-variable-id
			 (concatenate 'string 
				      (string trimmed)
				      "@"
				      (format nil "~a" i)))))
	      ((not (variable-minor-use-p new-id)) new-id))))))

(defun get-similar-allowed-ids (ids unalloweds)
  (with-variable-minor-invocation
    (set-minor-use-in-variable-id-tree unalloweds)
    (mapcar #'(lambda (var-id)
		(do ((i 0 (1+ i))
		     (new-id var-id (get-variable-id 
				      (concatenate 'string 
						   (string var-id) 
						   "@"
						   (format nil "~a" i)))))
		    ((not (variable-minor-use-p new-id)) new-id)))
	    ids)))

(defconstant *digit-chars* '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))

(defun strip-variable-id (variable-id)
  (let* ((no-nums (string-right-trim *digit-chars* (string variable-id)))
	 (length-no-nums (length no-nums)))
    (if (and (< 1 length-no-nums (length (string variable-id)))
	     (char= #\@ (char no-nums (1- length-no-nums))))
	(strip-variable-id (subseq no-nums 0 (1- length-no-nums)))
	(get-variable-id variable-id))) )

(defun trim-variable-id (variable-id)
  (let* ((no-nums (string-right-trim *digit-chars* (string variable-id)))
	 (length-no-nums (length no-nums)))
    (if (and (< 1 length-no-nums (length (string variable-id)))
	     (char= #\@ (char no-nums (1- length-no-nums))))
	(get-variable-id (subseq no-nums 0 (1- length-no-nums)))
	variable-id)))

;;;; expansion stuff

    

;; returns evcb
;; requires id to be actual variable
(defun evcb-of-variable-id (id)
  (let ((value (variable-value id)))
    (and value
	 (prl-var-evcb value))))

;; returns evcb
;; requires id to be actual variable
(defun set-evcb-of-variable-id (id evcb)
  (setf (prl-var-evcb (variable-value id)) evcb))


;;;; ---------------------------------------------------------------------
;;;; -------------------- marking functions ------------------------------
;;;; ---------------------------------------------------------------------

;; rle todo  prl-var should be derived from marks and vmarks removed.
;; maybe possible for marks not to be tied invocations in all scenarios.
;; though off the top of my head I don't see much use for marks
;; except with invocations, Could move injections to marks?

; returns old value if var was marked with non-nil value
; returns nil if var was not marked or was marked with nil.
(defun markv (var-sym mark &optional value)
  (mark (prl-var-vmarks (variable-value var-sym)) mark value))

(defun markv-p (var-sym mark)
  (let ((vval (maybe-variable-value var-sym)))
    (when vval
      (markp (prl-var-vmarks vval) mark))))

(defun markv-value (var-sym mark)
  (let ((vval (maybe-variable-value var-sym)))
    (when vval
      (mark-value (prl-var-vmarks vval) mark))))

(defun unmarkv (var-sym mark)
  (let ((vval (maybe-variable-value var-sym)))
    (when vval
      (unmark (prl-var-vmarks vval) mark))))

;;; for following functions 
;;;   - old value of mark must be a list.
;;;   - if not previously marked then old value is nil.

(defun markv-push (var-sym mark value)
  (mark-push (prl-var-vmarks (variable-value var-sym)) mark value))

(defun markv-pop (var-sym mark)
  (let ((var-val (maybe-variable-value var-sym)))
    (when var-val
      (mark-pop (prl-var-vmarks var-val) mark))))

(defun markv-peek (var-sym mark)
  (let ((var-val (maybe-variable-value var-sym)))
    (when var-val
      (mark-peek (prl-var-vmarks var-val) mark))))


;;;; deBruijn functions

;;;
;;; Following require variable-id as args.
;;;

(defun value-of-binding (binding)
  (when binding
    (binding-value binding)))

(defun binding-of-variable-id (var-sym)
  (car (prl-var-bindings (variable-value var-sym))))

(defun bindings-of-variable-id (var-sym)
  (prl-var-bindings (variable-value var-sym)))

(defun value-of-variable-binding (var-sym)
  (let ((b (car (prl-var-bindings (variable-value var-sym)))))
    (when b
      (binding-value b))))
	

(defun free-var-p (var-sym)
  (free-var-val-p (variable-value var-sym)))

(defun bound-var-p (var)
  (not (free-var-p var)))

(defun depth-of-variable-id (id)
  (binding-index (binding-of-variable-id id)))

(defun value-of-variable-id (id)
  (binding-value (binding-of-variable-id id)))



(defun left-free-var-p (var-sym)
  (left-free-var-val-p (variable-value var-sym)))

(defun left-bound-var-p (var-sym)
  (left-bound-var-val-p (variable-value var-sym)))

(defun right-free-var-p (var-sym)
  (right-free-var-val-p (variable-value var-sym)))

(defun right-bound-var-p (var-sym)
  (right-bound-var-val-p (variable-value var-sym)))


;;; 
;;; Following accept any parameter-value of type variable.
;;;

(defun enter-binding (var-sym &optional value)
  (let ((var (value-of-parameter-value var-sym)))
    (when (variable-id-p var)
      (push (make-binding :index (incf *prl-var-debruijn-index*) :value value)
	    (prl-var-bindings (variable-value var-sym))))))

(defun exit-binding (var-sym)
  (let ((var (value-of-parameter-value var-sym)))
    (when (variable-id-p var)
      (decf *prl-var-debruijn-index*)
      (when (> 0 *prl-var-debruijn-index*)
	(error "more bindings have been exited than have been entered"))
      (pop (prl-var-bindings (symbol-value var-sym))))))

(defun enter-bindings (vars)
  (mapc #'enter-binding vars))

(defun exit-bindings (vars)
  (mapc #'exit-binding vars)
  ;(mapr-on-vars #'exit-binding vars)
  )


(defun enter-binding-pair (left-sym right-sym)
  (let ((lv (value-of-parameter-value left-sym))
	(rv (value-of-parameter-value right-sym)))
    (if (or (not (variable-id-p lv))
	    (not (variable-id-p rv)))
	(equal-parameter-values-p lv rv *variable-type*)
	(progn
	  (incf *prl-var-debruijn-index*)
	  (push *prl-var-debruijn-index* (prl-var-left-indices (variable-value left-sym)))
	  (push *prl-var-debruijn-index* (prl-var-right-indices  (variable-value right-sym)))
	  t))))


(defun enter-binding-pairs (left-vars right-vars)
  (apply-predicate-to-list-pair left-vars right-vars  #'enter-binding-pair))

(defun exit-binding-pair (left-sym right-sym)
  (let ((lv (value-of-parameter-value left-sym))
	(rv (value-of-parameter-value right-sym)))
    (unless (or (not (variable-id-p lv))
		(not (variable-id-p rv)))
      (decf *prl-var-debruijn-index*)
      (pop (prl-var-left-indices (symbol-value left-sym)))
      (pop (prl-var-right-indices (symbol-value right-sym))))))

(defun exit-binding-pairs (left-vars right-vars)
  (mapc #'exit-binding-pair left-vars right-vars)
  ;(mapr-on-vars #'exit-binding-pair left-vars right-vars)
  )


;; requires variable ids.
(defun equal-bindings (left-var-sym right-var-sym)
  (let ((left-var-val (symbol-value left-var-sym))
	(right-var-val (symbol-value right-var-sym)))
    (or (and (left-bound-var-val-p left-var-val)
	     (right-bound-var-val-p right-var-val)
	     (= (car (prl-var-left-indices left-var-val))
		(car (prl-var-right-indices right-var-val))))
	(and (left-free-var-val-p left-var-val)
	     (right-free-var-val-p right-var-val)
	     (eql left-var-val right-var-val) ))))



;; PERF this might not be best strategy. 
(defun get-it ()
  (let ((it (gensym)))
    (setf (symbol-value it) 0)
    it))

(defun enter-binding-pair-it (left-sym right-sym it)
  (let ((lv (value-of-parameter-value left-sym))
	(rv (value-of-parameter-value right-sym)))
    (if (or (not (variable-id-p lv))
	    (not (variable-id-p rv)))
	(equal-parameter-values-p lv rv *variable-type*)
	(progn
	  (incf (symbol-value it))
	  (markv-push left-sym it (symbol-value it))
	  (markv-push right-sym it (symbol-value it))
	  t))))

(defun exit-binding-pair-it (left-sym right-sym it)
  (let ((lv (value-of-parameter-value left-sym))
	(rv (value-of-parameter-value right-sym)))
    (unless (or (not (variable-id-p lv))
		(not (variable-id-p rv)))
      (decf (symbol-value it))
      (markv-pop left-sym it)
      (markv-pop right-sym it))))

(defun equal-bindings-it (left-sym right-sym it)
  (let ((left-index (markv-peek left-sym it))
	(right-index (markv-peek right-sym it)))
    (or (and left-index right-index
	     (= left-index right-index))
	(and (null left-index)
	     (null right-index)
	     (eql left-sym right-sym)))))



;; internals for mapping on list of lists.
(defun mapr-on-vars (f vars &rest other-lists)
  (macrolet
    ((cars (lists)
      `(mapcar 'car ,lists))
     (cdrs (lists)
      `(mapcar 'cdr ,lists)))

    (labels 
      ((any-endp (lists)
	(cond
	  ((endp lists) nil)
	  ((endp (car lists)) t)
	  (t (any-endp (cdr lists)))))

       (mapr-on-var-list (vars)
	 (cond
	   ((null vars))
	   (t (mapr-on-var-list (cdr vars))
	      (funcall f (car vars)))))

       (mapr-on-var-lists (var-lists)
	 (cond 
	   ((any-endp var-lists))
	   (t (mapr-on-var-lists (cdrs var-lists))
	      (apply f (cars var-lists))))))

      (if other-lists
	  (mapr-on-var-lists (cons vars other-lists))
	  (mapr-on-var-list vars)))))



;;;  ---------------------------------------------------------------------
;;;  ---------------- set functions --------------------------------------
;;;  ---------------------------------------------------------------------

(defun unionl-vars (var-sets)
  (with-variable-minor-invocation
    (let ((result (car var-sets)))
      (mapc #'(lambda (var-sym)
		(set-variable-minor-use var-sym))
	    result)
        (mapc 
	#'(lambda (var-set)
	    (mapc #'(lambda (var-sym)
		      (unless (set-variable-minor-use var-sym)
			(push var-sym result)))
		  var-set))
	(cdr var-sets))
      result)))

(defun union-vars (var-set &rest more-sets)
  (with-variable-minor-invocation
    (let ((result var-set))
      (mapc #'(lambda (var-sym)
		(set-variable-minor-use var-sym))
	    var-set)
      (mapc 
	#'(lambda (var-set)
	    (mapc #'(lambda (var-sym)
		      (unless (set-variable-minor-use var-sym)
			(push var-sym result)))
		  var-set))
	more-sets)
      result)))

(defun diff-vars (+vars &rest -var-sets)
  (cond
    ((null +vars) nil)
    ((null -var-sets) +vars)
    ((and (equal '(()) -var-sets)) +vars)
    (t (with-variable-minor-invocation
	   (let ((result nil))
	     (mapc #'(lambda (-vars) 
		       (mapc #'(lambda (var-sym) (set-variable-minor-use var-sym))
			     -vars))
		   -var-sets)
	     (mapc #'(lambda (var-sym)
		       (unless (set-variable-minor-use var-sym)
			 (push var-sym result)))
		   +vars)
	     result)))))

(defun same-vars (a b)
  (and (null (diff-vars a b))
       (null (diff-vars b a))))

(defun duplicate-vars-p (vars)
  (with-variable-minor-invocation
    (not (exists-p #'set-variable-minor-use vars))))

(defun null-intersect-vars-p (vars-a vars-b)
  (or (null vars-a)
      (null vars-b)
      (with-variable-minor-invocation
	  (mapc #'(lambda (var-sym) (set-variable-minor-use var-sym))
		vars-a)
	(mapc #'(lambda (var-sym)
		  (when (variable-minor-use-p var-sym)
		    (return-from null-intersect-vars-p nil)))
	      vars-b)
	t)))

(defun intersectl-vars (var-sets)
  (if (null (cdr var-sets))
      (car var-sets)
	      
      (let ((result nil))
	(let ((tail-result (intersectl-vars (cdr var-sets))))
	  (with-variable-minor-invocation
	    (mapc #'(lambda (var-sym) (set-variable-minor-use var-sym))
		  tail-result)
	    (mapc #'(lambda (var-sym)
		      (when (variable-minor-use-p var-sym)
			(push var-sym result)))
		  (car var-sets))))
	result)))

(defun intersect-vars (var-set &rest more-sets)
  (if (null more-sets)
      var-set
      (let ((result nil))
	(let ((tail-result (intersectl-vars more-sets)))
	  (with-variable-minor-invocation
	    (mapc #'(lambda (var-sym) (set-variable-minor-use var-sym))
		  tail-result)
	    (mapc #'(lambda (var-sym)
		      (when (variable-minor-use-p var-sym)
			(push var-sym result)))
		  var-set)))
	result)))


(defun choose-by-vars (choose-from faccessor choose-by baccessor)
  (let ((result nil))
    (with-variable-minor-invocation
      (mapc #'(lambda (item) 
		(set-variable-minor-use (funcall baccessor item)))
	    choose-by)
      (mapc #'(lambda (item)
		(when (variable-minor-use-p (funcall faccessor item))
		  (push item result)))
	    choose-from))
    result))



