
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	definition_insert
;;;;	definition_delete
;;;;	commit/undo
;;;;	


(define-primitive |!commit| ((natural . sequence)  (oid . oid)) (stamp))
(define-primitive |!undo| ((natural . sequence)  (oid . oid)))

;; overloaded. in broadcast, there will be an !object_states term contain possiblly multiple
;; subterms which themselves are replace terms. However, each subterm
;; will have only a single !object_state subterm.
(define-primitive |!definition_replace| ((natural . sequence)) (state))

(define-primitive |!definition_insert| ((natural . sequence)) (definition))
(define-primitive |!definition_delete| ((natural . sequence) (oid . oid)))

(define-primitive |!definition_activate| ((natural . sequence) (oid . oid)))
(define-primitive |!definition_deactivate| ((natural . sequence) (oid . oid)))

(define-primitive |!definition_disallow_collection| ((natural . sequence) (oid . oid)))
(define-primitive |!definition_allow_collection| ((natural . sequence) (oid . oid)))

(defun sequence-of-ibroadcast-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))



(defun touch-record (table definition insert-p)
  ;;(setf -table table) (break "tr")
  (cons (car (tags-of-definition-table table))
	(cons (oid-of-definition definition)
	      (if insert-p
		  '|insert|
		  '|delete|))))

(defun oid-of-touch-record (tr) (cadr tr))
(defun tags-of-touch-record (tr) (car tr))
(defun touch-record-delete-p (tr) (eql '|delete| (cddr tr)))
(defun touch-record-insert-p (tr) (eql '|insert| (cddr tr)))


;;;
;;; term-to-definition
;;;

(define-primitive |!definition| () (dependency data))


(defun oid-of-idefinition-term (d)
  (oid-of-idependency-term (dependency-of-idefinition-term d)))

(defun term-to-definition-aux (term import-f &optional odef)

  ;;(setf -term term -import-f import-f) (break "ttda") 

  (if odef
      (funcall import-f (data-of-idefinition-term term) odef)
      (let ((dependency (term-to-dependency (dependency-of-idefinition-term term))))
	(with-dependency (dependency '(definition import))

	  (let ((def (funcall import-f (data-of-idefinition-term term))))

	    (setf (definition-dependency def) dependency) ; import

	    def)))))
    

(defun term-to-definition (table term &optional def)

  (let ((reusep (and def t)))

    (let ((import-f (or (when reusep
			  (import-aux-f-of-definition-table table))
			(progn (setf reusep nil)
			       (import-f-of-definition-table table)))))

      (unless import-f
	;;(setf -table table) (break "ttd")
	(raise-error (error-message (append (tags-of-definition-table table) '(import undefined)))))

      (unless (idefinition-term-p term)
	;;(break "tdd")
	(raise-error (error-message (append (tags-of-definition-table table) '(import syntax))
				    term)))

      (term-to-definition-aux term import-f (when reusep def)))))



;;;;	apply-broadcast-to-definition-table (<bcast> <ttable>)
;;;;	
;;;;	
;;;;	data table
;;;;	
;;;;	supports add/delete commit/undo
;;;;	 - abstractions/dforms/statments etc.
;;;;	


      ;; auto commit implicit? does not effect touch?
      ;; state of obj may change without dir being touched? yes because dir only sees obid.
      ;; thus object changes but dir not touched and dir window not updated.
      ;; should affect touch history. then collect list of oids then update any window contain any
      ;; oid in list of any window showing contents of oid. Important that it check oids in all windows
      ;; not just dir windows. Need cache of oids for each window? refreshed at state change. and
      ;; marked stale at edit mod.
      ;; not bound by serializability constraints. Ie can replace values not visible to transaction. NO
      ;; or each replace sequence is distinct transaction with one atomic replace list. YES.

(defun definition-replace (table def s i)
  
  (let ((otable (oid-table-of-definition-table table))
	(oid (oid-of-definition def table))
	(not-p (object-state-def-not-p def)))	

    (with-oids ((list oid) '(replace))

      (let ((cur (oid-table-lookup otable oid s t)))
	(when cur
	  (oid-table-delete otable oid s i))

	(unless not-p
	  (oid-table-insert otable oid s i def))

	(with-unwind-error ((unless not-p
			      (oid-table-undo otable oid s i))
			    (when cur
			      (oid-table-undo otable oid s i)))

	  ;; insert-hook.
	  (let ((replace-f (replace-f-of-definition-table table)))
	    (when replace-f (funcall replace-f def cur)))
		  
	  ;; definition table-stamp
	  (setf (definition-table-stamp def)
		(table-stamp-of-definition-table table)) ; safe : insert

	  (unless not-p
	    (oid-table-commit otable oid s i))
	  
	  (when cur
	    (oid-table-commit otable oid s i))

	  (when (definition-table-touch-history-p table)
	    (values (touch-record table def (not not-p)))))))))

  
(defun definition-insert (table def s i &optional note)

  (setf (definition-table-caches table) nil)

  ;;(setf a table b def) (break "insert")
  ;;(when (definition-name def)
    ;;(format t "~s >> ~s~%" (definition-name def) (definition-table-tags def)))
  (let ((otable (oid-table-of-definition-table table))
	(ttable (table-of-definition-table table))
	(oid (oid-of-definition def table)))

    (with-dependency ((dependency-of-definition def) '(definition insert))
      (with-oids ((list oid) '(insert))

	;; insert in oid-table
	(oid-table-insert otable oid s i def note)

	(with-unwind-error ((oid-table-undo otable oid s i))

	  (let ((keys (when ttable (keys-of-definition def table))))

	    ;; insert in ttable
	    (when ttable (dolist (key keys) (def-table-insert ttable key s i def)))

	    ;; finish
	    (with-unwind-error ((when ttable (dolist (key keys) (def-table-delete ttable key s i oid))))

	      ;; visibility
	      (init-visibility-of-definition def table)

	      ;; quick dep ref index.
	      (setf (definition-quickdepref-index def)
		    (incf (definition-table-quickdepref-index table))
		    (definition-table-containing def)
		    table)

	      ;; insert-hook.
	      (let ((insert-f (insert-f-of-definition-table table)))
		(when insert-f (funcall insert-f table def s i)))
		  
	      ;; definition table-stamp
	      (setf (definition-table-stamp def)
		    (table-stamp-of-definition-table table)) ; safe : insert
	      
	      )))))))


(defun definition-delete (table oid s i &optional note)

  (setf (definition-table-caches table) nil)

  (let ((otable (oid-table-of-definition-table table))
	(ttable (table-of-definition-table table)))

    (with-oids ((list oid) '(delete))

      (let ((def (oid-table-lookup otable oid s))
	    (delete-f (delete-f-of-definition-table table)))

	;; call hook prior to delete. seems asymetrical when compared with insert.
	;; but insert /delete are mirror processes. Since we call insert-f last we should call
	;; delete-f first. Basically, want hooks to see tables with defs.
	(when delete-f (funcall delete-f table def s i))

	(with-unwind-error ((let ((undo-f (undo-f-of-definition-table table)))
			      (when undo-f
				;; this is likely to confuse derived table as
				;; would expect undo of delete not insert, callers need to code undo to do insert
				;; if that is what is required.
				(funcall undo-f table def nil s i)
				;;(funcall insert-f def s i)
				)))

	  (let ((keys (when ttable (keys-of-definition def table))))

	    (when ttable
	      (dolist (key keys) (def-table-delete ttable key s i oid)))

	    (with-unwind-error ((when ttable
				  (dolist (key keys) (def-table-insert ttable key s i def))))

	      (oid-table-delete otable oid s i note)

	      (setf (definition-table-stamp def) nil) ; delete
	      )))))))
    
	      
(defun definition-activate (table def)
  ;;(setf a table b def) (break "da")

  (let ((activate-f (activate-f-of-definition-table table)))

    (unless activate-f (raise-error (oid-error-message (list (oid-of-definition def))
						       '(definition activate hook not))))

    (funcall activate-f def)))

(defun definition-deactivate (table def)

  (let ((deactivate-f (deactivate-f-of-definition-table table)))

    (unless deactivate-f (raise-error (oid-error-message (list (oid-of-definition def))
						       '(definition deactivate hook not))))

    (funcall deactivate-f def)))

(defun definition-allow-collection (table def)
  ;;(setf a table b def) (break "da")

  (let ((allow-f (allow-f-of-definition-table table)))

    (unless allow-f (raise-error (oid-error-message (list (oid-of-definition def))
						       '(definition allow hook not))))

    (funcall allow-f def)))

(defun definition-disallow-collection (table def)

  (let ((disallow-f (disallow-f-of-definition-table table)))

    (unless disallow-f (raise-error (oid-error-message (list (oid-of-definition def))
						       '(definition disallow hook not))))

    (funcall disallow-f def)))


(defun definition-commit (table o s i)
  ;;(break "dc")

  (with-system-error ('(definition commit))
    (with-oids ((list o) '(commit))

      (let ((otable (oid-table-of-definition-table table))
	    (ttable (table-of-definition-table table)))
      
	(mlet* (((def insert-p note) (oid-table-commit otable o s i)))
	       
	       (when def
		 (when ttable
		   (dolist (key (keys-of-definition def table))
		     (def-table-commit ttable key s i)))

		 (let ((commit-f (commit-f-of-definition-table table)))
		   (when commit-f
		     (funcall commit-f table def insert-p s i)))

		 (tent-order-push s 'commit)
	       
		 (when (definition-table-touch-history-p table)
		   (values (touch-record table def insert-p)
			   note))))))))


(defun definition-undo (table o s i)
  ;;(break "du")

  (with-system-error ('(definition undo))
    (with-oids ((list o) '(undo))

      (let ((otable (oid-table-of-definition-table table))
	    (ttable (table-of-definition-table table)))
      
	(mlet* (((def insert-p note) (oid-table-undo otable o s i)))

	       (when def
		 (when ttable
		   (dolist (key (keys-of-definition def table))
		     (def-table-undo ttable key s i)))

		 (let ((undo-f (undo-f-of-definition-table table)))
		   (when undo-f (funcall undo-f table def insert-p s i)))

		 (values note)))))))



(define-primitive |!table_log| ((t . kind) (n . size)) (persist))
(define-primitive |!table_log_list| () (list))
(define-primitive |!log_checkpointed| () (stamp log))
(define-primitive |!code_order| () (list))

(defun stamp-of-itable-log-term (term)
  (let ((tt (persist-of-itable-log-term term)))
    (if (idata-persist-term-p tt)
	(stamp-of-idata-persist-term tt)
      tt)))

(defun itable-log-list-aux-p (term)
  (eql *itable-log-list* (id-of-term term)))

(defun apply-log (f iilog)
  ;;(break "a-log")

  (let ((ckp (when (ilog-checkpointed-term-p iilog)
	       (stamp-of-ilog-checkpointed-term iilog))))

  (labels ((visit (ilog)
		  ;;(setf -ilog ilog -ckp ckp) (break "alv")
	     (let ((abortp nil))
	       (unless (and ckp
			    (compare-terms-p (stamp-of-itable-log-term ilog)
					     ckp))
		 (let ((rlog (lite-log-open-read (term-to-stamp (stamp-of-itable-log-term ilog)))))
		   (do ((rec (log-read-record rlog) (log-read-record rlog)))
		       ((or (null rec)
			    abortp))
		       (cond ((itable-log-term-p rec)
			      (visit rec))
			     ((itable-log-list-aux-p rec)
			      (let ((bts (bound-terms-of-term rec)))
				(format t "ReadLog ~a~%" (length bts);;(term-op-count rec)
					)
				(mapc #'(lambda (bt)
					  (setf abortp (funcall f (term-of-bound-term-f bt))))
				      bts)))
			     (t;;(format t "ReadLog ~a~%" (term-op-count rec))
			      (setf abortp (funcall f rec)))))
		   (log-close rlog))))))
  
  (visit (if (ilog-checkpointed-term-p iilog)
		  (log-of-ilog-checkpointed-term iilog)
		iilog)))))

;; let ddg_stamp = lib_eval_to_term (unit_ap (begin_ap "\\ (). apply_alist (table_log_stamps ()) `DDG`"));;
;; (setf -ddg-stamp (ml-text "ddg_stamp"));;
;; (search-log f (itable-log-term 'ddg 0 -ddg-stamp))
(defun search-log (p ds)
  (let ((oids nil))
    (apply-log #'(lambda (d)
		   (let ((oid (oid-of-idependency-term (dependency-of-idefinition-term d))))
		     (push oid oids)
		     (when (funcall p oid d)
		       ;;(setf -d d -oid oid)
		       (break "sl")
		       t)))
	       (itable-log-term nil 0 ds))
    ;; (setf -logoids oids) (break "sle")
    (length -logoids)))


(defun apply-broadcast-to-definition-table (term table env tstamp &optional auto-commit def)

  ;;(setf a term) (break "abtdt")

  ;; shouldn't have to bind tstate ??
  ;; transaction may not be otherwise visible, ??
  (let ((tstate (tstate-by-stamp tstamp)))
    (with-tstate tstate

      (unless (definition-table-flag-touched-p table)
	(definition-table-flag-set-touched table t))

      (case (id-of-term term)

	(|!code_order|
	 (if *delay-broadcast-compiles*
	     (push (oids-of-icut-list (list-of-icode-order-term term)) *broadcast-compiles-orders*)
	     (break "abtdt")))  

	(|!commit| (mlet* (((record note) (definition-commit table (oid-of-icommit-term term)
					    (term-to-stamp (stamp-of-icommit-term term))
					    (sequence-of-ibroadcast-term term))))

			  ;;(setf h record i note j term) (break "abdac")

			  (when record
			    (touch-history-touch record tstate env)

			    (when (integerp note)
			      (dotimes (i (1- note))
				(touch-history-touch (definition-commit table (oid-of-icommit-term term)
						      (term-to-stamp (stamp-of-icommit-term term))
						      (sequence-of-ibroadcast-term term))
						    tstate
						    env))
			      ;;(break "abdac")
			      )))
		   nil)

	;; tis possible that broadcast applies to multiple tables and that they might wish to share
	;; same definition
	(|!definition_insert| (let ((def (term-to-definition table
							     (definition-of-idefinition-insert-term term)
							     def)))
			      
				;; when null def, creates prob for delete and commit.
				;; a statement with no extract results in null def for abstraction table.
				(when def
				  (definition-insert table def
				    tstamp
				    (sequence-of-ibroadcast-term term))
			   
				  (when auto-commit
				    (touch-history-touch (definition-commit table
							    (oid-of-definition def)
							  auto-commit
							  (sequence-of-ibroadcast-term term))
							tstate
							env))
				  def)))
      

	(|!definition_delete| (let ((o (oid-of-idefinition-delete-term term)))
				(definition-delete table o
				  tstamp
				  (sequence-of-ibroadcast-term term))

				(when auto-commit
				  (touch-history-touch (definition-commit table o
							auto-commit
							(sequence-of-ibroadcast-term term))
						      tstate
						      env)))
			      nil)

	((|!definition_activate| |!definition_deactivate|)

	 ;; delete, clone & modify, insert clone

	 (let ((o (oid-of-idefinition-activate-term term)))
	   (let ((def (definition-lookup-by-oid table o nil tstamp)))
	   
	     ;;(setf c o d def) (break "abda1")
	   
	     (definition-delete table o
	       tstamp
	       (sequence-of-ibroadcast-term term)
	       2)

	     ;; unwind-protect  to undo delete if activate fails?
	     (with-unwind-error ((definition-undo table o
				   tstamp
				   (sequence-of-ibroadcast-term term)))
	       (let ((new-def (if (idefinition-activate-term-p term)
				  (definition-activate table def)
				  (definition-deactivate table def))))

		 ;;(setf e new-def f term g tstamp) (break "abda2")

		 (definition-insert table new-def
		   tstamp
		   (sequence-of-ibroadcast-term term)
		   2)))
			   
	     (when auto-commit

	       ;; could skip this touch-push
	       (touch-history-touch (definition-commit table o
				     auto-commit
				     (sequence-of-ibroadcast-term term))
				   tstate
				   env)

	       ;;(break "abda3")

	       (touch-history-touch (definition-commit table o
				     auto-commit
				     (sequence-of-ibroadcast-term term))
				   tstate
				   env)
	       ;;(break "abda4")
	       )))
	 nil)

	((|!definition_allow_collection| |!definition_disallow_collection|)

	 ;; delete, clone & modify, insert clone

	 (let ((o (oid-of-idefinition-allow-collection-term term)))
	   (let ((def (definition-lookup-by-oid table o nil tstamp)))
	   
	     (definition-delete table o
	       tstamp
	       (sequence-of-ibroadcast-term term)
	       2)

	     ;; unwind-protect  to undo delete if (dis)allow fails?
	     (with-unwind-error ((definition-undo table o
				   tstamp
				   (sequence-of-ibroadcast-term term)))
	       (let ((new-def (if (idefinition-allow-collection-term-p term)
				  (definition-allow-collection table def)
				  (definition-disallow-collection table def))))

		 (definition-insert table new-def
		   tstamp
		   (sequence-of-ibroadcast-term term)
		   2)))
			   
	     (when auto-commit

	       ;; could skip this touch-push
	       (touch-history-touch (definition-commit table o
				      auto-commit
				      (sequence-of-ibroadcast-term term))
				    tstate
				    env)

	       (touch-history-touch (definition-commit table o
				      auto-commit
				      (sequence-of-ibroadcast-term term))
				    tstate
				    env)
	       )))
	 nil)

	(|!undo| (let ((note (definition-undo table (oid-of-icommit-term term)
			       tstamp
			       (sequence-of-ibroadcast-term term))))

		   (when (integerp note)
		     (dotimes (i (1- note))
		       (definition-undo table (oid-of-icommit-term term)
			 (current-transaction-stamp)
			 (sequence-of-ibroadcast-term term))))))

	(|!definition_replace|
	 ;; kludge alert, all of the replace is kludge.
	 ;;(with-tstate (with-transaction-id (transaction-id-of-stamp tstamp)
	 ;;(new-tstate t))
	 ;;(tstate-set-begin (transaction-state) tstamp)
	 ;;)
	 ;;(setf -term term) (break "dr")
	 (dolist (bt (bound-terms-of-term (state-of-idefinition-replace-term term)))
	   ;; could be more general log pointer.
	   (let ((data (data-of-idefinition-term (term-of-bound-term bt))))
	     (if (or (itable-log-term-p data)
		     (ilog-checkpointed-term-p data))
		 (apply-log #'(lambda (defterm)
				;;(when *process-break* (setf -defterm defterm -table table) (break "drt"))
				(let ((def (term-to-definition table defterm)))
				  ;; touch history push while reading log
				  ;;   - mucho entries
				  ;;   - should only be happening at configure.
				  ;; see additional comments near tree-listings end-hook
				  (definition-replace table def
				    tstamp
				    (sequence-of-ibroadcast-term term)))
				nil)
			    data)
		 (let ((def (term-to-definition table (term-of-bound-term bt))))
		   (touch-history-touch (definition-replace table def
					 tstamp
					 (sequence-of-ibroadcast-term term))
					tstate
					env)))))
	 nil)
			     
	(otherwise (break "apply definition broadcast"))))))



(defun definition-valid-p (def table)

  (and t

       ;; TODO PERF : cache abs def in term. proofs(termof) too?
       ;; this is not sufficient, as def may be deleted from table but if deleted then null.
       ;; but it may not be valid for stamp. but could possibly store tent in term and pull
       ;; def from tent via current stamp. but then clash confounds. maybe when clash not possible
       ;; there is a solution.
       (when (and def
		  (eql (table-stamp-of-definition-table table) (table-stamp-of-definition def)))
	 def)))
  

;; assumes in transaction binding. not!
(defun apply-passport (env term tstamp auto-commit)
  (if (and *other-broadcasts-stamp* (not (equal-stamps-p *other-broadcasts-stamp* tstamp)))
      (push (list* env term auto-commit) *other-broadcasts-delayed*)
	
  
    ;;(setf b env c term d tstamp e auto-commit) (break "ap1")
    ;;(when (eql 'ddg  (table-type-of-ipassport-term term)) (setf b env c term d tstamp e auto-commit) (break "ap1"))
    (when (permit-broadcast-p term (broadcast-sentry-of-environment env))
      (when *other-broadcasts-stamp* (format t "~%ApplyPassport xxx~%"))
      (if (eql 'transaction (table-type-of-ipassport-term term))
	  (apply-broadcast (broadcast-of-ipassport-term term)
			   'transaction
			   env
			   tstamp
			   nil)
	(let ((def nil))
	  (dolist (table (environment-resources-of-type env
							(table-type-of-ipassport-term term)))
		  ;;(setf a env b term c (table-type-of-ipassport-term term)) (break "ap")
		  (unless (equal-terms-p (stamp-of-ipassport-term term)
					 (table-stamp-term-of-definition-table table))
		    (message-emit (warn-message '(broadcast-eval table stamp) term)))
	    
		  (setf def (apply-broadcast (broadcast-of-ipassport-term term)
					     table
					     env
					     tstamp
					     auto-commit
					     def))))))))


(defun apply-broadcast (term table env tstamp &optional auto-commit def)
  (cond
    ((eql table 'transaction)
     ;;(break)
     (apply-transaction-broadcast term))
    ((definition-table-p table)
     (apply-broadcast-to-definition-table term table env tstamp auto-commit def))
    (t (break "apply-broadcast"))))



(defun allocate-transaction (stamp tag)
  (make-definition-table :table-stamp stamp :table-stamp-term (stamp-to-term stamp)
			 :tags (list 'transaction tag))
  )





;;;;	
;;;;	
;;;;	
;;;;	


(defstruct (name-table (:include definition-table))
  (names (make-hash-table))
  )

(defun names-of-name-table (tt) (name-table-names tt))

(defmacro name-table (stamp tags termp &rest keys &key &allow-other-keys)

  (when (or (member :insert-f keys)
	    (member :delete-f keys)
	    (member :commit-f keys)
	    (member :undo-f   keys))
    (error "Cannot use :insert-f :delete-f :commit-f :undo-f keywords with name-table"))
    
  `(define-definition-table
      ,stamp
    ,tags
    ,termp
    :make-f make-name-table
    :insert-f #'name-table-insert
    :delete-f #'name-table-delete
    :commit-f #'name-table-commit
    :undo-f   #'name-table-undo
    ,@keys))


(defun name-table-insert (table def s i)
  (let ((name (name-of-definition def)))

    (when name
      #+dontdoit(format t "nt insert ~a ~a ~a~%"
			(string-of-oid (oid-of-definition def))
			name
			(tags-of-definition-table table)
			)

      (tok-table-insert (names-of-name-table table)
			name
			(oid-of-definition def)
			s i def))))

(defun name-table-delete (table def s i)
  (let* ((name (name-of-definition def))
	 (defs (when name (tok-table-lookup (names-of-name-table table) name s))))

      (when (and name (not (member def defs))) (break "ntd"))
      (when (and name (member def defs))
	;; it is possible that name is present now but was not at insert.
	;; checking for def helps, but not certain that it is impossible for eq def
	;; to be in table with same name, eq defs should be uncommon.
	(tok-table-delete (names-of-name-table table)
			  name
			  (oid-of-definition def)
			  s i))))

(defun name-table-commit (table def insertp s i)
  (declare (ignore insertp))

  (let ((name (name-of-definition def)))
    (when name
      (tok-table-commit (names-of-name-table table)
			name
			(oid-of-definition def)
			s i))))


(defun name-table-undo (table def insertp s i)
  (declare (ignore insertp))

  (let ((name (name-of-definition def)))
    (when name
      (tok-table-undo (names-of-name-table table)
		      name
		      (oid-of-definition def)
		      s i))))

;; list of defs.
(defun name-table-lookup (table name stamp)
  (tok-table-lookup (names-of-name-table table) name stamp))

(defun name-table-bound-p (table name stamp)
  (tok-table-bound-p (names-of-name-table table) name stamp))

;; f: name -> oid -> def -> unit.
(defun name-table-map (table stamp f)
  (tok-table-map (names-of-name-table table) stamp f))

;; f: name -> defs -> unit.
(defun name-table-map-2 (table stamp f)
  (tok-table-map-2 (names-of-name-table table) stamp f))

(defun name-table-search (table stamp pattern f)
  (let ((matchf (string-pattern-search #'identity pattern t)))
    (name-table-map table stamp
		    #'(lambda (name oid def)
			(declare (ignore def))
			(when (funcall matchf name)
			  (funcall f name oid))) )))

