%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%
%
;;;;	
;;;;	put_term
;;;;	
;;;;	General purpose object manipulation:
;;;;	
;;;;	put_term		: object_id -> term -> unit
;;;;	get_term		: object_id -> term
;;;;	
;;;;	put_property 		: object_id -> token -> term -> unit
;;;;	get_property 		: object_id -> token -> term
;;;;	remove_property 	: object_id -> token -> unit
;;;;	
;;;;	get_properties	 	: object_id -> (token # term) list
;;;;	put_properties	 	: object_id -> (token # term) list -> unit
;;;;	
;;;;	
;;;;	ObjectId DAG : to be a node of the object id DAG, the object must have a description property
;;;;	    with an ObjectIdDAG purpose, and must conform to the following syntax.
;;;;	 
;;;;	
;;;;	<dir>		: <root> | <directory>
;;;;	
;;;;	<root>		: !directory(<name>:t)(<dag-child> !dag_cons ilist)
;;;;	<directory>	: !directory()(<dag-child> !dag_cons ilist)
;;;;	
;;;;	<dag-child>	: !dag_child{<name>:t, <object_id>:o}
;;;;
;;;;	
;;;;	The following functions will build such objects:
;;;;
;;;;	make_root		: token -> object_id
;;;;	remove_root		: object_id -> unit
;;;;
;;;;	make_directory		: object_id -> token -> object_id
;;;;	remove_directory	: object_id -> token -> unit
;;;;	
;;;;	make_leaf		: object_id -> token -> object_id
;;;;	remove_leaf		: object_id -> token -> unit
;;;;	insert_leaf		: object_id -> token (* name *) -> token (* kind *) -> term -> object_id
;;;;
;;;;	insert_object_id	: object_id -> token -> object_id -> unit
;;;;	delete_object_id	: object_id -> token -> unit
;;;;	  * will delete leaf or directory. remove_directory implicitly deactivates
;;;;	    the object while delete simply deletes the reference from the parent.
;;;;	    Thus, if you've made a link with insert, use delete to remove it.
;;;;	    However, if you want to remove the directory and its contents use
;;;;	    remove_directory. delete or remove_leaf are equivalent if not a directory.
;;;;
;;;;	
;;;;	The library will distribute ObjectIdDAGs. The following functions are available
;;;;	in clients to examine the dag.
;;;;	
;;;;	Calls to these functions must be made in scope of with_local_transaction.
;;;;	
;;;;	*Note : improving the transactions for 4.2 is current work in progress.
;;;;	
;;;;	with_local_transaction	: (unit -> *) -> *
;;;;	  - Making a remote call, ie using on of the previous functions will cause an error,
;;;;	    as will a recursive call to with_local_transaction. 
;;;;	
;;;;	roots			: (token # object_id) list
;;;;	root_name		: object_id -> token
;;;;	root			: token -> object_id
;;;;
;;;;	directory_p		: object_id -> bool
;;;;	directory_children	: object_id -> (token # object_id) list
;;;;	
;;;;	directory_term		: object_id -> term {<dir>}
;;;;	  - modifying this term and saving to the object is not the proper
;;;;	    method of updating. Use the aformentioned funcs and refresh
;;;;	    by calling directory_term again.
;;;;
;;;;	
%


% compiles source of code object %
let lib_compile_ap = begin_ap "compile_code ";;
let lib_compile oid = lib_eval (oid_ap lib_compile_ap oid);;

let get_rdb_table_ap = null_ap (itext_term "rdb_get_table ");;
let get_rdb_table oid = lib_eval_to_term (oid_ap get_rdb_table_ap oid);;

let get_inf_term_ap = null_ap (itext_term "get_inf_term");;
let get_inf_term oid = lib_eval_to_term (oid_ap get_inf_term_ap oid);;

let get_inf_term_at_address_ap = null_ap (itext_term "get_inf_term_at_address ");;
let get_inf_term_at_address poid address =
   lib_eval_to_term (oid_ap (term_ap get_inf_term_at_address_ap address) poid);;

let put_property_ap = null_ap (itext_term "put_property ");;
let put_property oid s term =
  lib_eval
    (term_ap (token_ap (oid_ap put_property_ap oid) s) term)
;;

let remove_property_ap = null_ap (itext_term "remove_property ");;
let remove_property oid s =
  lib_eval
    (token_ap (oid_ap remove_property_ap oid) s)
;;

let get_ref_env_term_ap = null_ap (itext_term "get_ref_environment_term ");;
let get_ref_env_term oid =
  lib_eval_to_term (oid_ap get_ref_env_term_ap oid)
;;

let get_property_ap = null_ap (itext_term "get_property ");;
let get_property oid s =
  lib_eval_to_term
    (token_ap (oid_ap get_property_ap oid) s)
;;

%
let kind_of_oid oid = token_of_itoken_term (get_property oid `KIND`);;

let kind_of_object oid =
   token_of_itoken_term
      (lib_eval_to_term (oid_ap (null_ap (itext_term "itoken_term (objc_kind (lib_object_contents oid))")) oid))
;;

let kind_of_object t =
   kind_of_oid (lib_eval_to_object_id
			(token_ap (null_ap (itext_term "lib_find_oid_by_name "))
			 t));;
%

let get_term_and_properties_ap  = null_ap (itext_term "get_term_and_properties ");;
let get_term_and_properties oid toks =
 let r = lib_eval_to_term (tokens_ap (oid_ap get_term_and_properties_ap oid) toks) in
  (ihead r, (map snd (term_to_property_list (itail r))))
;;

let put_name_property oid name = put_property oid `NAME` (itoken_term name);;
let put_name = put_name_property;;
let put_description oid desc = put_property oid `DESCRIPTION` desc;;
let get_description oid = get_property oid `DESCRIPTION`;;

let put_properties_ap =
  null_ap
    (itext_term "\\oid iprops. set_properties oid (term_to_property_list iprops) ")
;;

let put_properties oid props =
  lib_eval
    (term_ap (oid_ap put_properties_ap oid) 
	(property_list_to_term props))
;;

let put_term_and_properties_ap  = null_ap (itext_term "put_term_and_iproperties ");;
let put_term_and_properties oid term props =
  lib_eval
    (term_ap 
      (term_ap (oid_ap put_term_and_properties_ap oid) 
	       term)
      (property_list_to_term props))
;;

let get_properties_ap =
  null_ap
    (itext_term "(\\oid . property_list_to_term (get_properties oid) )")
;;
    
let get_properties oid =
  term_to_property_list
    (lib_eval_to_term
      (oid_ap get_properties_ap oid))
;;

let activate_ap = (null_ap (itext_term "activate "));;
let activate oid = lib_eval (oid_ap activate_ap oid);;

let activate_force_ap = (null_ap (itext_term "activate_force "));;
let activate_f oid = lib_eval (oid_ap activate_force_ap oid);;

let lib_aux1 =  (itext_term "\\oids. map (\\oid. if (`");;


let activate_kind_ap kind =
  (null_ap
    (iml_cons_term
      lib_aux1
      (iml_cons_term (itext_term  (tok_to_string kind))
		     (itext_term "` = (objc_kind (lib_object_contents oid))) then activate oid) oids"))));;

let activate_kind kind oids = lib_eval (oids_ap (activate_kind_ap kind) oids);;

let activate_abs oids = activate_kind `ABS` oids;;


let deactivate_kind_ap kind =
  (null_ap
    (iml_cons_term
      lib_aux1
      (iml_cons_term (itext_term  (tok_to_string kind))
		     (itext_term "` = (objc_kind (lib_object_contents oid))) then deactivate oid) oids"))));;

let deactivate_kind kind oids = lib_eval (oids_ap (deactivate_kind_ap kind) oids);;

let activate_list_ap =
  (null_ap
    (itext_term
      "\\oids. map activate oids"));;
  
let activate_list oids = lib_eval (oids_ap activate_list_ap oids);;

let deactivate_ap = (null_ap (itext_term "deactivate "));;
let deactivate oid = lib_eval (oid_ap deactivate_ap oid);;
let deactivate_by_macro = deactivate o oid_of_ioid_term ;;

let deactivate_list_ap =
  (null_ap
    (itext_term
      "\\oids. map deactivate oids"));;
  
let deactivate_list oids = lib_eval (oids_ap deactivate_list_ap oids);;

let allow_ap = null_ap (itext_term "allow_collection ");;
let allow_collection oid = lib_eval (oid_ap allow_ap oid);;

let disallow_ap = null_ap (itext_term "disallow_collection ");;
let disallow_collection oid = lib_eval (oid_ap disallow_ap oid);;



%
  Remote dag functions
%

let make_root_ap = null_ap (itext_term "dag_make_root ");;
let make_root name = lib_eval_to_object_id (token_ap make_root_ap name);;


let make_directory_ap = null_ap (itext_term "dag_make_directory ");;
let make_directory oid s =
  lib_eval_to_object_id
    (token_ap
      (oid_ap make_directory_ap
		 oid)
      s)
;;

let make_directory_alpha_ap = null_ap (itext_term "dag_make_directory_alpha ");;
let make_directory_alpha oid name =
  lib_eval_to_object_id (token_ap (oid_ap make_directory_alpha_ap oid) name)
;;

let make_oid_alpha_ap = null_ap (itext_term "dag_make_oid_alpha ");;
let make_oid_alpha oid name kind =
  lib_eval_to_object_id (token_ap (token_ap (oid_ap make_oid_alpha_ap oid) name) kind)
;;

let insert_alpha_ap = null_ap (itext_term "dag_insert_alpha ");;
let insert_alpha poid name oid =
  lib_eval (oid_ap (token_ap (oid_ap insert_alpha_ap poid) name) oid)
;;


let make_named_directory_ap = null_ap (itext_term "dag_make_named_directory ");;
let make_named_directory poid link name =
  lib_eval_to_object_id
   (string_ap
    (token_ap
      (oid_ap make_named_directory_ap poid)
      link)
    name)
;;

let make_named_directory_after_ap = null_ap (itext_term "dag_make_named_directory_after ");;
let make_named_directory_after poid place name =
  lib_eval_to_object_id
   (token_ap
    (token_ap
      (oid_ap make_named_directory_after_ap poid)
      place)
    name)
;;

let make_named_directory_before_ap = null_ap (itext_term "dag_make_named_directory_before ");;
let make_named_directory_before poid place name =
  lib_eval_to_object_id
   (token_ap
    (token_ap
      (oid_ap make_named_directory_before_ap poid)
      place)
    name)
;;

let make_named_directory_at_ap = null_ap (itext_term "dag_make_named_directory_at ");;
let make_named_directory_at boa poid place name =
  lib_eval_to_object_id
   (token_ap
    (token_ap
      (oid_ap (bool_ap make_named_directory_at_ap boa) poid) 
      place)
    name)
;;


let remove_root_ap = null_ap (itext_term "dag_remove_root ");;
let remove_root oid =
  lib_eval
    (oid_ap remove_root_ap oid)
;;

let remove_directory_ap = null_ap (itext_term "dag_remove_directory ");;
let remove_directory oid s =
  lib_eval
    (token_ap (oid_ap remove_directory_ap oid) s)
;;

let remove_directory_tree_ap = null_ap (itext_term "dag_remove_directory_tree ");;
let remove_directory_tree oid s =
  lib_eval
    (token_ap (oid_ap remove_directory_tree_ap oid) s)
;;

let remove_tree_ap = null_ap (itext_term "dag_remove_tree ");;
let remove_tree oid =
  lib_eval
    (oid_ap remove_tree_ap oid)
;;

let delete_tree_ap = null_ap (itext_term "dag_delete_tree ");;
let delete_tree oid name =
  lib_eval
    (token_ap (oid_ap delete_tree_ap oid) name)
;;


let make_leaf_ap = null_ap (itext_term "dag_make_leaf ");;
let make_leaf oid name kind =
  lib_eval_to_object_id
   (token_ap (token_ap (oid_ap make_leaf_ap oid) name) kind)
;;

let make_named_leaf_ap = null_ap (itext_term "dag_make_named_leaf ");;
let make_named_leaf oid name kind =
  lib_eval_to_object_id
   (token_ap (token_ap (oid_ap make_named_leaf_ap oid) name) kind)
;;

let make_named_leaf_after_ap = null_ap (itext_term "dag_make_named_leaf_at false ");;
let make_named_leaf_after oid place name kind =
  lib_eval_to_object_id
    (token_ap (token_ap (token_ap (oid_ap make_named_leaf_after_ap oid) place) name) kind)
;;

let remove_leaf_ap = null_ap (itext_term "dag_remove_leaf ");;
let remove_leaf oid name =
  lib_eval
    (token_ap (oid_ap remove_leaf_ap oid) name)
;;

let insert_object_id_ap = null_ap (itext_term "dag_insert ");;
let insert_object_id parent name child =
  lib_eval (oid_ap (token_ap (oid_ap insert_object_id_ap parent) name) child)
;;  

let insert_object_id_alpha_ap = null_ap (itext_term "dag_insert_alpha ");;
let insert_object_id_alpha parent name child =
  lib_eval (oid_ap (token_ap (oid_ap insert_object_id_alpha_ap parent) name) child)
;;  

let insert_object_id_after_ap = null_ap (itext_term "dag_insert_at false ");;
let insert_object_id_after parent place name child =
  lib_eval (oid_ap (token_ap (token_ap (oid_ap insert_object_id_after_ap parent) place) name) child)
;;  

let insert_at_ap = null_ap (itext_term "insert_at ");;
let insert_at boa parent place name child =
  lib_eval (oid_ap (token_ap (token_ap (oid_ap (bool_ap insert_at_ap boa) parent)
				       place) name) child)
;;  

let delete_object_id_ap = null_ap (itext_term "dag_delete ");;
let delete_object_id parent name =
  lib_eval (token_ap (oid_ap delete_object_id_ap parent) name)
;;  


let change_link_name_ap = begin_ap "dag_change_name";;
let change_link_name dir name oid newname =
  lib_eval (token_ap (oid_ap (token_ap (oid_ap change_link_name_ap dir) name) oid) newname)
;;


let insert_leaf_ap = null_ap (itext_term "dag_insert_leaf ");;
let insert_leaf parent name type term =
  lib_eval_to_object_id 
    (term_ap (token_ap (token_ap (oid_ap insert_leaf_ap
					 parent)
				name)
		       type)
	     term)
;;

let insert_named_leaf_ap = null_ap (itext_term "dag_insert_named_leaf ");;
let insert_named_leaf parent name type term =
  lib_eval_to_object_id 
    (term_ap (token_ap (token_ap (oid_ap insert_named_leaf_ap
					 parent)
				name)
		       type)
	     term)
;;

let set_leaf_ap = null_ap (itext_term "dag_set_leaf ");;
let set_leaf parent name type term =
  lib_eval_to_object_id 
    (term_ap (token_ap (token_ap (oid_ap set_leaf_ap
					 parent)
				name)
		       type)
	     term)
;;

let set_directory_children dir t =
 lib_eval (oid_ap (term_ap (begin_ap "ddg_oid_set_children") t) dir);;



% twould be better if took tok arg. %
let library_close_ap =
   null_ap (itext_term "(hd (close_environment (orb_match_local_environment [`lib`]) false false)) ")
;;

let library_close (():unit) =
 (lib_eval_to_string library_close_ap)
;;




let delete_directory_ap = null_ap (itext_term "\\oids. map delete_strong oids");;

let delete_directory thy = 
 let oids = (descendents (root `theories`) [`standard`; thy]) in
   remove_directory (descendent (root `theories`) [`standard`]) thy;
   lib_eval (oids_ap delete_directory_ap oids)
;;


let lib_post_activate_code_ap =
  null_ap
   (itext_term "\\oid. (client_post_activate_code (objc_property (lib_object_contents oid) `DESCRIPTION`) oid)");;

let lib_post_activate_code oid =
 lib_eval (oid_ap lib_post_activate_code_ap oid)
 ;;

 
let statement_set_proofs_ap =
  null_ap (itext_term "\\oids oid. save oid (stm_objc_modify_proofs (lib_object_contents oid) oids)")
 ;;

let statement_set_proofs oid oids =
 lib_eval (oid_ap (oids_ap statement_set_proofs_ap oids) oid);;




let activate_directory_kind kind thy =
 activate_kind kind (descendents (root `theories`) [`standard`; thy])
;;

let deactivate_directory_kind kind thy =
 deactivate_kind kind (descendents (root `theories`) [`standard`; thy])
;;


% libm-ref functions %
let get_inf_term_stm_ap = null_ap (itext_term "\\oid. get_inf_term_stm oid");;
let get_inf_term_stm oid = lib_eval_to_term (oid_ap get_inf_term_stm_ap oid);;

let copy_proof_ap = null_ap (itext_term "copy_proof ");;
let lib_copy_proof oid = lib_eval_to_object_id (oid_ap copy_proof_ap oid);;

let run_proof_ap = null_ap (itext_term "run_proof ");;
let lib_run_proof oid = lib_eval_to_term (oid_ap run_proof_ap oid);;

let iproof_info_cons_op = (`!proof_info_cons`, nil);;
let iproof_info_term status oid name=
 make_term (`!proof_info`, [(make_token_parameter name)]) [([],status); ([],oid)]
;;
							   
let status_of_iproof_info_term term =
  let ((id, [name]),
      [([], status);
       ([], oid)]) = destruct_term term in
   status;;

let oid_of_iproof_info_term term =
  let ((id, [name]),
      [([], status);
       ([], oid)]) = destruct_term term in
   oid_of_ioid_term oid;;

let name_of_iproof_info_term term =
  let ((id, [name]), bterms) = destruct_term term in
   destruct_token_parameter name;;
							   

% for compatabilility %
let make_proof_copy = lib_copy_proof;;

let copy_object_after_ap = null_ap (itext_term "copy_object_at false ");;
let copy_object_after poid place name oid =
  lib_eval_to_object_id (oid_ap (token_ap (token_ap (oid_ap copy_object_after_ap poid) place) name) oid);;

letref ref_prf_debug = let doid = dummy_object_id () in (ivoid_term, doid, [doid]);; 
  
let refine_term_ap = null_ap (itext_term "proof_refine ");;
let refine_term term oid oids = 
 ref_prf_debug := (term, oid, oids);
 lib_eval_to_term (oids_ap (oid_ap (term_ap refine_term_ap term) oid) oids);;

let refinet_term_ap = null_ap (itext_term "proof_refinet ");;
let refinet_term term oid = 
 ref_prf_debug := (term, oid, nil);
 lib_eval_to_term (oid_ap (term_ap refinet_term_ap term) oid);;
 
let remove_prf_ap = null_ap (itext_term "remove_prf_from_stm ");;
let remove_prf oid = lib_eval (oid_ap remove_prf_ap oid);;

let remove_bk_prfs_ap = null_ap (itext_term "remove_backup_prfs ");;
let remove_backup_prfs oid = lib_eval (oid_ap remove_bk_prfs_ap oid);;

let set_first_prf_ap = null_ap (itext_term "set_proof_first ");;
let set_first_prf oid = lib_eval (oid_ap set_first_prf_ap oid);;

let step_refine_term_ap = null_ap (itext_term "proof_step_refine ");;
let step_refine_term term oid oids =
  lib_eval_to_term (oids_ap (oid_ap (term_ap step_refine_term_ap term) oid) oids);;

let step_refinet_term_ap = null_ap (itext_term "proof_step_refinet ");;
let step_refinet_term term oid =
  lib_eval_to_term (oid_ap (term_ap step_refinet_term_ap term) oid);;


let refine_tree_ap = null_ap (itext_term "refine_tree ");;
%lal could optimize this not to call lib for property%
let refine_tree term oid oids =
  lib_eval_to_term (oids_ap (term_ap (oid_ap (term_ap refine_tree_ap term) oid)
				     (get_ref_env_term oid))
			    oids);;


let refine_tttt_tree_ap = null_ap (itext_term "refine_tttt_tree ");;
let refine_tttt_tree term oid oids =
  lib_eval_to_term (oids_ap (oid_ap (oid_ap (term_ap refine_tttt_tree_ap term) oid)
				    (oid_of_ioid_term
				     (get_property oid `reference_environment`)))
			    oids);;


let require_termof oid = lib_eval (oid_ap (begin_ap "require_termof ") oid);;

let refine_undo_ap = null_ap (itext_term "refine_undo ");;
let refine_undo oid = (oid_ap refine_undo_ap oid);;

let lib_backup_proof_ap = null_ap (itext_term "make_backup_prf ");;
let lib_backup_proof goal poid soid name =
  lib_eval_to_object_id (token_ap (oid_ap (oid_ap (term_ap lib_backup_proof_ap goal) poid) soid) name);;

let lib_interior_proof_ap = null_ap (itext_term "make_interior_prf ");;
let lib_interior_proof goal poid name =
  lib_eval_to_term (token_ap (oid_ap (term_ap lib_interior_proof_ap goal) poid) name);;

let lib_primitive_proof_ap = null_ap (itext_term "make_primitive_prf ");;
let lib_primitive_proof oid term oids =
  lib_eval_to_term (oids_ap (term_ap (oid_ap lib_primitive_proof_ap oid) term) oids);;

let lib_primitive_prooft_ap = null_ap (itext_term "make_primitive_prft ");;
let lib_primitive_prooft oid term =
  lib_eval_to_term (term_ap (oid_ap lib_primitive_prooft_ap oid) term);;

let lib_abstraction_ap = null_ap (itext_term "find_obid_of_abstraction ");;

let lib_abstraction_obid_of_term term = lib_eval_to_object_id (term_ap lib_abstraction_ap term);;

let abstraction_obid_of_term term =
  (((obid_of_abstraction_term term)
    ? (lib_abstraction_obid_of_term term))
    ? (raise_error nil ``abstraction_obid_of_term`` [term]; fail)) 
;;


let lib_expand_ap = null_ap (itext_term "expand_term ");;
let lib_expand term = lib_eval_to_term (term_ap lib_expand_ap term);;

let lib_abstraction_lookup_ap = null_ap (itext_term "lib_abstraction_lookup ");;
let lib_abstraction_lookup oid = lib_eval_to_term (oid_ap lib_abstraction_lookup_ap oid);;

%
let dependencies_of_object_ap = null_ap (itext_term "(objc_translation o oc)");;
let dependencies_of_object oid = lib_eval_to_term (oid_ap dependencies_of_object_ap oid);;
%

let xref_of_code_object_ap = null_ap (itext_term "(xref_of_code_source o oc)");;
let xref_of_code_object oid = lib_eval_to_term (oid_ap xref_of_code_object_ap oid);;


let dump_objects_ap = begin_ap "dump_object_list_wprfs ";;
let dump_objects closurep mobilep fname oids =
  lib_eval (oids_ap (string_ap (bool_ap (bool_ap dump_objects_ap closurep) mobilep)
			       fname)
		    oids)
;;
  
%
let dump_tree_oid closurep exportp fname path =
   dump_objects closurep exportp fname (subtree_oids true (descendent_s path));;
%
let dump_tree_aux closurep exportp fname path =
  dump_objects closurep exportp fname (subtree_oids true (descendent_s path));;

let dump_tree_aux_aux avoidp closurep exportp fname path =
  dump_objects closurep exportp fname
    (subtree_oids_avoid true (descendent_s path) avoidp)
;;

let dump_tree fname path = dump_objects false false fname (subtree_oids true (descendent_s path));;
let dump_list fname l    = dump_objects false false fname l;;
let export_list fname l  = dump_objects false true fname l;;

let export_tree fname path = dump_objects false true fname (subtree_oids true (descendent_s path));;
let export_full_tree fname path = dump_objects true true fname (subtree_oids true (descendent_s path));;

let load_objects_ap = begin_ap "load_objects ";;
let load_objects overwritep activatep fname  =
  lib_eval (string_ap (bool_ap (bool_ap load_objects_ap overwritep)
				activatep)
		       fname)
;;


let read_objects = load_objects false false;;
let add_objects = load_objects false true;;
let replace_objects = load_objects true true;;
%let replace_objects_path path = load_objects_path path true true;;%


let read_patch = replace_objects;;


%printing calls%

let get_collection_tree_ap  = null_ap (itext_term "collection_object_tree  ");;
let get_collection_tree oid = lib_eval_to_term (oid_ap get_collection_tree_ap oid);;


let lib_mkobj_ap = begin_ap "lib_mkobj";;
let lib_mkobj dir place kind name props =
 lib_eval_to_object_id
   (term_ap (token_ap (token_ap (token_ap (oid_ap lib_mkobj_ap dir) place) kind)
		      name)
	    (property_list_to_term props))
;;
