%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
**********************************************************************
**********************************************************************
COMPUTE-AUX.ML
**********************************************************************
**********************************************************************
Auxilary functions for direct computation. 


Two types of functions are provided here.

1. A suite of tagging functions for the direct and reverse direct computation
   rules. Including tagging functions for:
    a. Definition unfolding
    b. Non-canonical term reduction.
 
2. Subterm filters to allow the rewrite package to support for example lazy 
   reduction of terms.

See file compute-aux2.ml for functions for definition folding.
]%


%
defined in lisp
~~~~~~~~~~~~~~~
abstraction_definition_of_object : tok -> term # term
abstraction_definition_of_term : term -> term # term
name_of_abstraction_object_of_term : term -> tok
arith_simplify_term : term -> term
arith_simplify_term_top : term -> term
compute : term -> term
do_indicated_computations : term -> term

defined in primitives.ml
~~~~~~~~~~~~~~~~~~~~~~~~
mk_tag_term (i:int) (t:term) = (t':term)
is_tag_term (t:term) = b: bool
dest_tag (t:term) = i,t' : int # term

is_ab_term : term -> bool
opid_of_abstraction : term -> tok

defined in term.ml
~~~~~~~~~~~~~~~~~~
is_prim_redex (t:term) = b:bool
is_principle_arg (t:term) (i:int) = b:bool
%

%[
**********************************************************************
Basic tagging functions.
**********************************************************************
]%
%
Only work on term that they are applied to.

Issues to decide.
1. What is the best way of referring to abstractions?
   a. By library name (+ index?)
   b. By opid (+parameters and arity?)
   c. By some abbreviation. (same as used to retrieve def from library?)

For now the unfolding functions discriminate by opid alone, and the folding
functions work by library object name.

Is it likely that we will have several abstraction definitions for a single
opid?
%


let is_ab_term_with_opids opids t =
   member (opid_of_term t) opids & is_ab_term t 
;;

let tag_one_op = (`tag`, [mk_natural_parm 1]);;
let tag_term t = 
  make_term tag_one_op [[],t]
;;
let tag_prim_redex t =
   if is_prim_redex t then
     tag_term t
   else failwith `tag_prim_redex`
;;

let mk_tag_term_aux_o i dir obidts t =
 make_term (`tag`, [mk_natural_parm i; mk_bool_parm dir])
  [ [],t; [],map_to_ilist (\x.x) icons_op obidts] 
;;

let mk_tag_term_o i dir obids t =
 make_term (`tag`, [mk_natural_parm i; mk_bool_parm dir])
  [ [],t; [],map_to_ilist ioid_term icons_op obids] 
;;
let mk_tag_term_compseq dir compseq t =
 make_term (`tag`, [mk_bool_parm dir])
  [ [],t; [],compseq] 
;;

let mk_obidts_term = map_to_ilist (\x.x) icons_op;;
let tag_op_ot = (`tag`, [mk_natural_parm 0; mk_bool_parm true]);;
let tag_op_of = (`tag`, [mk_natural_parm 0; mk_bool_parm false]);;
let mk_tag_term_ot obidst t = make_term tag_op_ot [ [],t; [],obidst];;
let mk_tag_term_of obidst t = make_term tag_op_of [ [],t; [],obidst];;

let tag_zero_op = (`tag`, [mk_natural_parm 0]);;
let tag_prim_redex0 t =
   if is_prim_redex t then
     mk_tag_term_aux_o 0 true nil t
   else failwith `tag_prim_redex`
;;

% 
so_redex is: so_apply(so_lambda(x1,...,xn.t);a1;...;an) 

For convenience we recognize so_redices when 
the application is a normal apply or infix apply.  
%

let tag_so_redex t =
 
 (if is_term `so_apply` t then
   (let f.args = dest_simple_term_with_opid `so_apply` t in
    if is_term `so_lambda` f then
      mk_tag_term
        (length args + 1)
        (mk_simple_term
           `so_apply`
           (tag_term f . args)
        )
    else
      fail
   )
  if is_term `apply` t then
  ( let f.args = dest_iterated_apply t in
    if is_term `so_lambda` f 
       & length args = length (fst (hd (bterms_of_term f)))
    then
      mk_tag_term
        (length args)
        (mk_iterated_apply
           (tag_term f . args)
        )
    else
      fail
  )
  if is_term `infix_ap` t then
   (let f.args = dest_simple_term_with_opid `infix_ap` t in
    if is_term `so_lambda` f & length (bvars_of_term f 1) = 2 then
      mk_tag_term
        3
        (mk_simple_term
           `infix_ap`
           (tag_term f . args)
        )
    else
      fail
   )
  else
    fail
 ) ? failwith `tag_so_redex`
;;

let tag_beta_redex t =
   if is_beta_redex t then
     tag_term t
   else failwith `tag_beta_redex`
;;

let tag_abs_for_unfold names t =
  if is_ab_term_with_opids names t then
    tag_term t
  else
    failwith `tag_abs_for_unfold`
;;

let tag_any_ab_for_unfold t =
  if is_ab_term t then
    tag_term t
  else
    failwith `tag_any_ab_for_unfold`
;;

let tag_so_ap t =
  if is_term `so_apply` t then
    mk_tag_term (length (bterms_of_term t)) t
  else
    failwith `tag_so_ap`
;;

let tag_extract names t = 
  if is_terms names t & is_extract_term t then
    tag_term t
  else
    failwith `tag_extract`
;;

let tag_any_extract t = 
  if is_extract_term t then 
    tag_term t
  else
    failwith `tag_any_extract`
;;

%[
**********************************************************************
term tree tagging functions.
**********************************************************************
]%
%
From term.ml we have

map_on_immediate_subterms f:term->term t:term = t':term
sweep_up_map f:term->term t:term = t':term
apply_to_nth_immediate_subterm f:term->term n:int t:term = t':term
apply_to_addressed_subterm f:term->term  address:int list  t:term  = t':term

Use these functions sparingly. Most times, users should be constructing
direct computation strategies using the rewrite package.
%

%
(tagged_t:term) = tag_all_prim_redices (t:term)
(tagged_t:term) = tag_all_abs_for_unfold (names:tok list) (t:term)
%

let tag_all_aux f = progresseqf (sweep_up_map f);;

let tag_all_aux_po dir p obids =
  tag_all_aux (\t. if p t then mk_tag_term_o 0 dir obids t else t)
;;

let tag_all_pot p obidts =
 let obidst = mk_obidts_term obidts in
  tag_all_aux (\t. if p t then mk_tag_term_ot obidst t else t)
;;
let tag_all_pof p obidts =
 let obidst = mk_obidts_term obidts in
  tag_all_aux (\t. if p t then mk_tag_term_of obidst t else t)
;;

let tag_all_prim_redices =
  progressf (sweep_up_map tag_prim_redex)
;;
let tag_all_prim_redices0 =
  progressf (sweep_up_map tag_prim_redex0)
;;

let tag_all_beta_redices =
  progressf (sweep_up_map tag_beta_redex)
;;

let tag_all_so_redices =
  progressf (sweep_up_map tag_so_redex)
;;

let tag_all_so_aps =
  progressf (sweep_up_map tag_so_ap)
;;

let tag_all_abs_for_unfold names =
  progressf (sweep_up_map (tag_abs_for_unfold names))
;;

%
(tagged_t:term) = tag_addressed_redex (address:int list) (t:term)
(tagged_t:term) = tag_addressed_ab (address:int list) (t:term)
%

let tag_addressed_redex =
  apply_to_addressed_subterm tag_prim_redex 
;;

let tag_addressed_ab =
  apply_to_addressed_subterm tag_any_ab_for_unfold
;;

%
| Take the tags from the second term, and copy them to the same addresses
| on the first term.  Erase all other tags on the first term.
%
let copy_tags term1 term2 =
    let copy_tag t1 t2 =
	if is_tag_term t2 then
	    if is_tag_term t1 then
		t1
	    else
		tag_term t1
	else
	    if is_tag_term t1 then
		(destroy_tag t1)
	    else
		t1
    in
	sweep_up_map2 copy_tag term1 term2;;

%
(tagged_t:term) = tag_nth_redex (n:int) (t:term)
(tagged_t:term) = tag_nth_ab_with_name (name:tok) (n:int) (t:term)

Not implemented. Use rewrite package instead.
%

%[
**********************************************************************
Direct computation functions that depend on matching.
**********************************************************************
]%

%
Matches term argument against rhs of some abstraction, and instantiates left 
hand side of term with resulting (full) substitution. Tags this term.)

nb names here is name of object in library. NOT the opid. Maybe need function 
which takes opid and returns list of abstractions for that opid.
%

let simple_fold_term lhs rhs t =
 let opid = id_of_term rhs in
 if not (opid = `variable`) & not (opid = id_of_term t) then fail;
   let meta_term_vars = free_vars rhs in
   let meta_parm_vars = parm_vars rhs in
   let sub = ( full_match 0 meta_parm_vars meta_term_vars rhs t 
             ? failwith `fold_and_tag_top_ab: no match`) in
      full_subst sub lhs
;;       


let fold_and_tag_top_ab_with nam =
         let lhs,rhs = (lhs_rhs_of_abstraction_by_name nam 
                        ? failwith `fold_and_tag_top_ab: bad object name`) in
         let meta_term_vars = free_vars rhs in
         let extra_vars = diff (free_vars lhs) meta_term_vars in
         let meta_parm_vars = parm_vars rhs in
          (\extra_ts t.
             let sub = (full_match 0 meta_parm_vars meta_term_vars rhs t 
                        ? failwith `fold_and_tag_top_ab: no match`) in
             let sub1 = zip extra_vars extra_ts @ sub 
                 ? failwith `fold_and_tab_top_ab: wrong number of extra ts`
             in
             let t' = full_subst sub1 lhs in
               tag_term t'
         )
;;

let fold_ab dir lhs pattern =  
  let meta_term_vars = free_vars pattern in
  let extra_vars = diff (free_vars lhs) meta_term_vars in
  let meta_parm_vars = parm_vars pattern in
  if not (null extra_vars) then failwith `fold_ab`;
  (\t. 
    let sub = ( full_match 0 meta_parm_vars meta_term_vars pattern t 
              ? failwith `fold_and_tag_top_ab: no match`) in
     let t' = full_subst sub lhs in
      mk_tag_term_compseq dir t' t)
;;

let fold_and_tag_top_ab_aux_o obid =
  let x,lhs,rhs,r = abstraction_lookup obid 
                     ? failwith `fold_and_tag_top_ab: bad object obid` in
  let meta_term_vars = free_vars rhs in
  let extra_vars = diff (free_vars lhs) meta_term_vars in
  let meta_parm_vars = parm_vars rhs in
    (\extra_ts.
      let sub0 = zip extra_vars extra_ts 
                 ? failwith `fold_and_tab_top_ab: wrong number of extra ts` in
        (\t. let sub = (full_match 0 meta_parm_vars meta_term_vars rhs t 
                       ? failwith `fold_and_tag_top_ab: no match`) 
              in let t' = full_subst (sub0 @ sub) lhs in  tag_term t'))
;;

let fold_and_tag_top_ab nam = fold_and_tag_top_ab_with nam [];;
let fold_and_tag_top_ab_o obid = fold_and_tag_top_ab_aux_o obid [];;

let fold_and_tag_abs names =
  let tag_fun_list =
    map
      fold_and_tag_top_ab
      names
  in
    \t. (first_fun tag_fun_list t
         ?
          ( %tty_print "fold_and_tag_abs:"
	  ; tty_print (tok_to_string (opid_of_term t))
	  ; map (\n. tty_print (tok_to_string n)) names
          ; % failwith `fold_and_tag_abs: no abstractions match`))
;;

let fold_and_tag_abs_o obids =
  let tag_fun_list = map_omitting_failures fold_and_tag_top_ab_o obids
  in
    \t. first_fun tag_fun_list t
        ? (failwith `fold_and_tag_abs: no abstractions match`)
;;

% watch for time of application/failure problems %
let fold_and_tag_single_ab_o obid =
  let tag_fun = fold_and_tag_top_ab_o obid in
    \t. tag_fun t ? (failwith `fold_and_tag_abs: no abstractions match`)
;;    
  

let tag_all_abs_for_fold names = progressf (sweep_up_map (fold_and_tag_abs names));;
let tag_all_abs_for_fold_o obids = progressf (sweep_up_map (fold_and_tag_abs_o obids));;

let tag_at_address_for_fold name =
  apply_to_addressed_subterm (fold_and_tag_abs [name])
;;
let tag_at_address_for_fold_o obid =
  apply_to_addressed_subterm (fold_and_tag_top_ab_o obid)
  ? failwith `tag_at_address_for_fold`
;;
  
let fold_top nam t =  destroy_tag (fold_and_tag_top_ab nam t);;
let fold_top_o obid t =  destroy_tag (fold_and_tag_top_ab_o obid t);;

let folds names t = sweep_up_map destroy_tag (tag_all_abs_for_fold names t);;
let folds_o obids t = sweep_up_map destroy_tag (tag_all_abs_for_fold_o obids t);;

let tag_all_single_ab_for_fold_o obid = progressf (sweep_up_map (fold_and_tag_single_ab_o obid));;
let fold_o obid t =  sweep_up_map destroy_tag (tag_all_single_ab_for_fold_o obid t);;


% unfolds abstractions in obids list %
let sweep_down_term repeatp f term =
 % not using sweep down since sweep down
   will not check top expanded term. %

 letrec auxbts bts =
  if null bts then false, nil
  else let bt = hd bts in
       let bbts,newbts = auxbts (tl bts) in
       let b,newt = aux (term_of_bterm bt) in
       if bbts
          then true, ((if b then (bvars_of_bterm bt, newt) else bt) . newbts)
       else if b then true, ((bvars_of_bterm bt, newt) . (tl bts))
       else false, bts

 and aux t =
   let tt = (f t ? t) in
   let b = not (eq_terms t tt) in
    if (repeatp & b) then true, (snd (aux tt))
    else let bb, nbts = auxbts (bterms_of_term tt) in
          if bb then true, (make_term (operator_of_term tt) nbts)
          else b, tt
 in snd (aux term)
;;


let sweep_up_term repeatp f term =

 letrec dorepeat t = 
  (let tt = (f t) ? t in
    if repeatp & not (eq_terms t tt)
       then dorepeat tt
       else tt) in
       
 letrec auxbts bts =
  if null bts then false, nil
  else let bt = hd bts in
       let bbts,newbts = auxbts (tl bts) in
       let b,newt = aux (term_of_bterm bt) in
       if bbts
          then true, ((if b then (bvars_of_bterm bt, newt) else bt) . newbts)
       else if b then true, ((bvars_of_bterm bt, newt) . (tl bts))
       else false, bts

 and aux t =
   let b, bts =  auxbts (bterms_of_term t) in
   let tt = if b then make_term (operator_of_term t) bts else t in
   let ttt = dorepeat tt in
   let bb = not (eq_terms ttt tt) in
   if b or bb then true, ttt else false, t 

   in snd (aux term)
;;

% unfolds abstractions in obids list %
let unfolds_o obids =
 % not using sweep down since sweep down
   will not check top expanded term. %
 let idoids = map (\obid. opid_of_term (fst (snd (abstraction_lookup obid))), obid) obids in

 let tryexp t =
   let idt = opid_of_term t in
   letrec teaux idoids =
    if null idoids then t else
    let idoid = hd idoids in
    if (fst idoid) = idt
       then (let newt = expand_an_abstraction (snd idoid) t in
              if eq_terms newt t
                 then teaux (tl idoids)
                 else newt)
    else teaux (tl idoids) in
    teaux idoids in

    sweep_down_term true tryexp
;;

