
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


(defun equal-arities-p (arities-l arities-r)
  (equal arities-l arities-r))

;;; Primitive term
(define-primitive |term~| )

;;; Get primitives sequal-term-p, etc.
;; moved to trm-funcs.lisp
(define-primitive |sqequal| nil (leftterm rightterm))

;;; Types obeying sqequality
(define-primitive |sq_type| nil (term))

;;; Subterm macro
(defmacro map-on-subterms (f t1 t2)
  "Map a function over the list of subterms"
  `(mapcar ,f (bound-terms-of-term ,t1) (bound-terms-of-term ,t2)))

;;; Sqequal main rule
(defun sqeq (proof-node &optional rule)
  "Refine sqequal on two matching terms"
  (declare (ignore rule))
  (let ((concl (conclusion-of-proof-node proof-node))
	(assums (assumptions-of-proof-node proof-node)))
    
    (if (sqequal-term-p concl)
	(let* ((terms (bound-terms-of-term concl))
	       (result (do-sqeq (term-of-bound-term (car terms)) (term-of-bound-term (cadr terms)) assums)))
	  (if (eql (car result) 'good)
	      (let ((assumptions (if (exists-p #'hidden-assumption-p assums)
				     (mapcar #'unhide-assumption assums)
				     assums)))
		(list
		 ;; subgoals
		 (mapcar #'(lambda (subgoal)
			     (make-proof-node :assumptions (append assumptions (car subgoal))
					      :conclusion (cdr subgoal)))
			 (cdr result))
		 (axiom-term)))
	      (ref-error (car result))))
	(ref-error "Expected opid sqequal"))))

;;; Sqequal
;;; Two terms are sqequal if
;;;    1. Their opnames are equal
;;;    2. Their parameters are equal
;;;    3. They have the same arity
;;;    4. Their subterms are sqequal
(defun do-sqeq (termx termy assums)
  "Perform sqequal equality reasoning"
  (cond ((not (equal-operators-p (operator-of-term termx) (operator-of-term termy)))
	 '("sqequal: Opid's do not match"))
	((not (equal-arities-p (arities-of-term termx) (arities-of-term termy)))
	 '("sqequal: Arities differ"))
	(t;; Now opid's, parameters, and arities are equal.  Generate subgoals.
	 (cons 'good
	       ;; Get one subgoal for each subterm.
	       (map-on-subterms
		#'(lambda (subterm1 subterm2)
		    (if (bindings-of-bound-term subterm1)
			(let ((unallowed (list (free-vars-of-bound-term subterm1)
					       (free-vars-of-bound-term subterm2)
					       (mapcar #'id-of-assumption assums)
					       (get-dummy-variable-id)))
			      (new nil))
			  
			  (let ((subv (mapcar #'(lambda (v)
						  (let ((nv (get-similar-allowed-id v
										    (if new
											(cons new unallowed)
											unallowed))))
						    (push nv new)
						    nv))
					      (bindings-of-bound-term subterm1))))

			    ;; The result is a cons-cell containing
			    ;;     car: the list of new assumptions (which are the new variables just created)
			    ;;     cdr: a new squiggle-equality on the new subterm.
			    ;;          The new variables have to substituted in for the binding variables.
			    (cons (mapcar #'(lambda (v)
					      (instantiate-assumption v (term~-term)))
					  subv)
				  (sqequal-term (substitute
						 (term-of-bound-term subterm1)
						 (delete-duplicates 
						  (mapcar #'(lambda (v b) (cons b (variable-term v)))
							  subv
							  (bindings-of-bound-term subterm1))
						  :key #'car))
						(substitute
						 (term-of-bound-term subterm2)
						 (delete-duplicates 
						  (mapcar #'(lambda (v b) (cons b (variable-term v)))
							  subv
							  (bindings-of-bound-term subterm2))
						  :key #'car))))))
			  (cons nil
				(sqequal-term (term-of-bound-term subterm1)
					      (term-of-bound-term subterm2)))))
		termx
		termy)))))
  
;; \y,x. x + y...  \a,b. b + a  [a/x b/y] [b/a a/b]
;;; Sqequal main rule
(defun sqeq-equality (proof-node &optional rule)
  "Refine sqequal on two matching terms"
  (let ((concl (conclusion-of-proof-node proof-node)))
    (if (equal-term-p concl)
	(let* ((terms (bound-terms-of-term concl))
	       (sq1 (term-of-bound-term (cadr terms)))
	       (sq2 (term-of-bound-term (caddr terms))))
	  (if (and (sqequal-term-p sq1) (sqequal-term-p sq2))
	      (let* ((assumptions (mapcar #'unhide-assumption (assumptions-of-proof-node proof-node)))
		     (terms1 (bound-terms-of-term sq1))
		     (terms2 (bound-terms-of-term sq2))
		     (result (do-sqeq-equality (free-vars-of-term concl)
					       assumptions
					       (term-of-bound-term (car terms1))
					       (term-of-bound-term (cadr terms1))
					       (term-of-bound-term (car terms2))
					       (term-of-bound-term (cadr terms2)))))
		;;(setq debug-var result)
		(list ;; subgoals
		 (mapcar #'(lambda (concl)
			     (make-proof-node :assumptions assumptions :conclusion concl))
			 result)))
	    (ref-error "Expected opid sqequal")))
      (ref-error "Expected opid equal"))))

;;; Sqequal-equality
;;; Two sqequality terms are equal if
;;;    1. They are intensionally equal
;;;    2. All the free variables in the conclusion belong to sqequal types
;;;
;;; Return a new list of goals
;;;
;;; Conclusion has the form 'a ~ b = c ~ d in U{i}'
(defun do-sqeq-equality (fv assums a b c d)
  "Perform sqequal equality reasoning"
  (labels ((find-assumption (v assums)
             (if (null assums)
		 (ref-error "Illegal assumption")
	         (let* ((assum (car assums))
			(id (id-of-assumption assum)))
		   (if (equal v id)
		       (type-of-assumption assum)
		       (find-assumption v (cdr assums)))))))
     `(,(sqequal-term a c)
       ,(sqequal-term b d)
       ,.(mapcar #'(lambda (v)
		     ;; Have a free variable, show that it is a sqtype, find the assumption
		     (sq-type-term (find-assumption v assums)))
		fv))))
		 

