%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
**********************************************************************
**********************************************************************
ARITH.ML
**********************************************************************
**********************************************************************
Support functions for the arith decision procedure.
**********************************************************************
We need to make sure that relevant primitive term constructors are exposed
to make these decision procedures work properly.

However Arith doesn't seem to handle multiple nots. Need to do some 
rewriting...

integer expressions are terms with outermost opids:   natural_number minus 
add subtract multiply divide mod

atomic integer relations are = 
  less_than: less than.
  le  : less or equal
  gt  : greater than
  ge  : greater or equal.
  equal (with int type) : equal
  nequal (with int type): not equal

less_than and equal are primitives. The rest are abstractions.

Compound integer relations are the above nested in 0 or more not abstractions.

An arith conclusion is a disjunction of 1 or more compound arith exps / 
false terms. We run the Arith decision proc if concl is an arith concl.

Since Arith knows nothing of abstractions. (maybe it should...) we have
to unfold all abstractions which might be relevant.

For the arith decision procedure, we need to do the following:

1. unfold all compound int rels in hyps.
2. unfold all compound int rels in concl as well as the disjunction.
]%


let is_refl_equality t = 
  (let a,b = equands t in alpha_equal_terms a b) ? false ;;

let is_atomic_int_rel t = 
  let opid = opid_of_term t in
    member opid ``less_than le gt ge``
    or
    (member opid ``equal nequal`` & is_int_term (hd (subterms_of_term t)))
;;

letrec is_int_rel_arith t =
  is_atomic_int_rel t 
  or is_terms ``lele lelt`` t
  or (is_int_rel_arith (dest_not t) ? false)
;;
  


% 
do we want this to look inside soft abstractions?? For now it doesn't, since
we have listed the abstract subset types explicitly.
%

letref arith_types = ``int nat nat_plus int_iseg 
  int_seg int_upper int_lower int_nzero``
;;

letrec is_arith_type T = 
    let opid = opid_of_term T in
    if member opid arith_types then true
    if opid = `set` then is_arith_type (hd (subterms_of_term T))
    else
      false
;;

let is_int_proper_subset_type T =
    not is_int_term T & is_arith_type T
;;


%[
**********************************************************************
Arith Tactics
**********************************************************************
]%

%
IntArith is the basic arith wrapper. It normalizes away abstractions in
the concl and relevant hyps so the arith rule can recognise what's what.
%


% 
default level exp is no more than a placeholder. Usually Arith never needs 
the universe term. (Only used when dealing with certain integer subsets??)
%

let PrimArith p = 
  Refine 
    `arith` 
    [mk_term_arg 
      (mk_U_term 
         (snd (dest_lp_term (get_type p (concl p)))
          ?
          mk_level_exp [`i`,0]
         )
      )
    ] p 
;;


let PrimMonotonicity kind i j p =
 let i' = get_pos_hyp_num i p in
 let j' = get_pos_hyp_num j p in
   Refine `monotonicity`
     [mk_int_arg i'; mk_int_arg j'; mk_tok_arg kind]
     p
;;

%
We want IntArith to work if 
1. Contradiction in hyps.
2. non trivial Arithmetic conclusion.

Arith also tends to work when conclusion is reflexive equality, but in a 
way which doesn't make progress and might cause looping in Autotactic.
e.g:

i,j:Int >> i+j = i+j in Int

 By PrimArith

i,j:Int >> i+j = i+j in Int
i,j:Int >> i+j = i+j in Int

We therefore throw in some simple loop checking.
%
  
let OldIntArith p =
  if exists
       (\t.is_int_rel_arith t & is_refl_equality t)
       (dest_iterated_or (concl p))
  then
    fail
  else
  ( TryOnAllHyps 
      (\i p. if exists 
                  is_int_rel_arith  
                  (dest_iterated_and (h i p))
             then
               Repeat (Unfolds ``and not false implies le ge gt nequal lele lelt`` i) p
             else
               Fail p
      )
    THEN Repeat (Unfolds ``not false implies or le ge gt nequal`` 0)
    THEN PrimArith
    THEN AddHiddenLabel `wf`
  ) p
  ?
  failwith `IntArith`
;;

% Changes:
1. Unfolded hyps now refolded.
2. Member terms folded for wf subgoals.
%
let IntArith p =
  if exists
       (\t.is_int_rel_arith t & is_refl_equality t)
       (dest_iterated_or (concl p))
  then
    fail
  else
  %let get_arith_hnum (i,t) = 
    if exists is_int_rel_arith (dest_iterated_and t) then i else fail
  in%
  % in let arith_hnums = mapfilter get_arith_hnum (number (map snd (dest_hyps p)))%

  ( Repeat (Unfolds ``not false implies or le ge gt nequal`` 0)
    THEN
    ComputeClausesOnlyFor 
      (tag_all_abs_for_unfold 
         ``and not false implies le ge gt nequal lele lelt``)
      (PrimArith THEN (FoldTop `member` 0) THEN AddHiddenLabel `wf`)
      (filter_numbered_hyps
         (\h. exists is_int_rel_arith (dest_iterated_and (type_of_declaration h)))
         (hyps p)) %arith_hnums%
  ) p
;;

%
Handling Subsets of the integers. For now recognise these 3 cases:

 1. ... i:S ... >> C
 2. ... a = b in S ... >> C
 3. ... >> a = b in S

where S is a subset of Int.

Take care. BasicAbSetEqTypeHD generates additional subgoals...  %

let OpenIntSubsetsForArith p =
  ( TryOnAllMHyps 
      (\i p. let hyp = h i p in
             if is_int_proper_subset_type hyp then
               RepeatM (BasicAbSetHD i) p
             if (is_int_proper_subset_type (eq_type hyp) ? false) then
               RepeatM (BasicAbSetEqTypeHD [] i) p
             else
               Fail p
      )
    THENM IfOnConcl
          (\t. is_int_proper_subset_type (eq_type t) ? false)
          (RepeatM AbSetEqTypeCD)
          Id
  ) p
;;

%
We let Autotactic top loop take care of trying arith to solve concl eq type
set predicates.
%

let Arith p = 
  ( IntArith
    ORELSE
    (Progress OpenIntSubsetsForArith THENM IntArith)
  ) p
  ?
  failwith `Arith`
;;


let RepeatEqCDForArith p =
  letrec Aux p =
   (let T,a,b = dest_equal (concl p) in
    if alpha_equal_terms a b then
      Id 
    if is_arith_type T & (is_int_exp a or is_int_exp b) then
      Arith 
    else
      EqCD THEN IfLab `subterm` Aux Id
   ) p
  in
  (let t,t' = equands (concl p) in
   if alpha_equal_terms t t' then 
     fail
   if alpha_equal_terms (arith_simplify_complete_term t) (arith_simplify_complete_term t') then
     Progress Aux p
   else 
     fail
  ) ? failwith `RepeatEqCDForArith`
;;

% 
Since Arith is rather expensive, we don't run Arith on subgoals without
arithmetic conclusions, although some arithmetic contradictions will have
this form.

We only run Arith if conclusion isn't
To add:
 1. addition feature.
 2. squash hyp decomps.
 3. arith simplification for all terms, not just equality terms.
%

%
Put clause in arithmetical canonical form.
...Hi...>> C

 By ArithSimp i

`main`: ...Hi'...>> C

And many of the following:

`arith simp subgoal`: ... ... >> C'

We take care to thin original hyp in equality goal to prevent Autotactic
from deciding to have a go at it again when solving the equality.

The SwapEquands puts the simplified T' on the lhs of the equality.
On the rhs, sometimes can get reoccurrence of subgoal:

e.g. ArithSimp applied to a in N(n-1+1):
get equality subgoal from SubstClause

(a in N(n-1+1)) = (a in Nn) in Ui

and MemCD give new subgoal: a in N(n-1+1) !
%

let ArithSimp i p =
 (let T = clause_type i p in
  let T' = arith_simplify_complete_term T in
  if alpha_equal_terms T T' then 
    fail
  else
    
    SubstClause T' i
    THEN IfLabL
    [`equality`,
     Try (Thin i) 
     THEN SwapEquands 0
     THEN RepeatEqCDForArith
     THEN AddHiddenLabel `arith simp`
    ]
 ) p
 ? failwith `ArithSimp`

;;

let ArithReflEqTypeCD p =
 (let c = concl p in
  if is_refl_equality c & is_int_proper_subset_type (eq_type c) then
    AbSetEqTypeCD p
  else
    fail
 ) ? failwith `ArithReflEqTypeCD`
;;

let ArithMemberReflEqTypeCD p = EqToMemberEq (\i.ArithReflEqTypeCD) 0 p
;;


%
NB Need to think about how ArithSimp and Arith are called. Normally Arith
might get run on every subgoal. Also if doing a wf proof, although ArithSimp 
might be useful, its success depends on all appropriate wf lemmas being already
in place.

Use of tactics:

Arith.
  Detects arithmetically true conclusions and arithmetic contradictions in
  hyp list.

ArithOnConcl:
  if conclusion is obviously an arithmetic relation then run Arith. Fails to
  run Arith if there is an arithmetic contradiction in hyps.

ArithMemberEqTypeCD:
  Takes care of cases when concl is a in T or a = b in T where T is some subset
  of Int. Paves the way for Arith to work. Missing: similarly expanding hyp 
  equalities or members which might be of use to Arith.

RepeatEqCDForArith:
  If concl is a = b in T where a and b arithmetically simplify to same, then
  invoke EqCD until get down to a and b being 1. the same or 2. arithmetic
  terms. RepeatMemberReflEqCD or Arith takes over from there.
  Should fail if don't get down to arithmetic terms.

ArithSimp i: if clause i is not in Arith canonical form, then try to put it
  in such a form. hands off to RepeatEqCDForArith.
  It should fail, if RepeatEqCDForArith fails.
  Although Strictly stronger than RepeatEqCDForArith, RepeatEqCDForArith should
  probably still be called first, because it is more efficient. 
%


% see simp-tactics where its added into inclusion tactic %

let SubtypeArithSimp p = 
  let i,A,B = get_subtype_args p 
  in let A' = arith_simplify_complete_term A 
  in let B' = arith_simplify_complete_term B 
  in let ProgA = not alpha_equal_terms A A'
  in let ProgB = not alpha_equal_terms B B'
  in
  if (ProgA or ProgB) & alpha_equal_terms A' B' then
    SeqOnM
    [SubtypeCD 
    ;if ProgA then ArithSimp (-1) else Id
    ;if ProgB then ArithSimp 0 else Id
    ;AddHiddenLabel `subterm`
    ]
    p
  else
    failwith `SubtypeArithSimp: no progress`
;;


%[
**********************************************************************
Preprocessor for Arith (and SupInf)
**********************************************************************
Opens up 
1. hyps such as x:List(n)
2. concls such as t in List(n)

]%

letref types_with_arith_properties = [] : tok list ;;

let note_type_with_arith_properties opid = 
  types_with_arith_properties := 
    insert opid types_with_arith_properties 
  ; ()
;;



