%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%
****************************************************************************
Atomic conversions
****************************************************************************
%

%
****************************************************************************
Take formula and turn into conversion
****************************************************************************
GenFormulaCondC 
  name : tok
  formula : term
  InstFormulaByAddr : int list -> (var # term) list -> Tactic
  formula_selector : int
  direction : tok (`LR` or `RL`)
  control : tok (`always` `quick enabler` `enabler` `tactics`)
  conv_enabler : env -> term list -> bool
  Tacs : tactic list
  hints : (var # term) list
   =
  C : conv

Argument description:

name: name used in constructing failure messages.
formula: general_formula with consequents of form t1 R t2.
InstFormulaByAddr: Should assert formula as last hypothesis and
                   instantiate according to supplied address and bindings.
formula_selector: selects which consequent of general formula to use.
direction: LR is left to right. RL is right to left.
control:
  1. always: rewrite goes through if match of pattern against instance
             succeeds.
  2. quick enabler: those antecedents which can be instantiated based purely
       on result of initial match, are instantiated and fed to enabler 
       function. Rewrite succeeds iff enabler returns true.
  3. enabler: All antecedents instantiated and fed to enabler. Rewrite 
       succeeds iff enabler returns true.
  4. tactics: Tacs are run on subgoals. If they completely prove all subgoals
       then rewrite succeeds.

conv_enabler:
  function which decides whether conditional rewrite goes through.
  Only needed if control options 2 or 3 selected. If option 1 or 4 selected,
  use dummy_conv_enabler .



Tacs: a list of tactics to run on subgoals (1 per antecedent.) .
  If there are too few Tacs, the leftmost of Tacs is used to pad the list 
  on the left side. Tacs are only used if control = `tactics`. The
  rewrite succeeds iff Tacs complete. However if Tacs is [] then
  then rewrite is unconditional, it always goes through.

hints: a list of bindings to complete matches in cases when the matching
algorithm cannot generate all the bindings necessary to instantiate some
rewrite lemma or hypothesis.

Improvements:
  maybe not best to throw all this ad hoc functionality into one tactic.
Could leave out quick enabler mode and separate out tactic mode to
separate function as we did in V3.
%

let GenFormulaCondC 
  name formula InstFormula consequent_num 
  direction control conv_enabler Tacs hints =

  let T = BackThruGenFormula
     get_type_using_env
     (half_match_with_retry get_hard_and_supertype_alts)
     (regular_half_match 1)
     formula 
     InstFormula 
     consequent_num 
  in
  let BackupT addr sub AntTs =
     let vsub,tsub = 
       divide_list
         (\v,t.sub_kind_of_term t = `variable`) sub 
     in
     InstFormula addr tsub
     THENLL
        [`antecedent`,AntTs
        ;`main`,[SoftNthHyp (-1) 
                 ORELSE Inclusion (-1)
                 ORELSE AddDebugLabel
                          `GenFormulaCondC: failed` 
                ]
        ]
  in
  let addr,x_A_prs,Bs,t1_R_t2 = 
        hd (nth_simple_formula consequent_num formula) in
  let R,t1,t2 = dest_rel_term t1_R_t2 in
  let pat_tm_a,pat_tm_b = if direction = `LR` then t1,t2 else t2,t1 in
  let xs = map fst x_A_prs in

  let active_Bs =
    if control = `quick enabler` then
    ( let active_xs = intersection (free_vars pat_tm_a) xs in
        filter (\t.subset (free_vars t) active_xs) Bs
    )
    else
      Bs
  in
  let matcher = 
    if control = `quick_enabler` then
      ( let m = soft_match xs pat_tm_a in \i c p.m i)
    else
      match_in_context_with_ti_and_ms 
        (half_match_with_retry get_hard_and_supertype_alts)
        get_type_using_env
        x_A_prs 
% 
  Seems a bad idea to allow fudging types in the primary matching
  e.g. in msets.thy the all_mset_elim and all_fset_elim rely on 
  matching types exactly 
%
%
        (half_match_with_retry get_hard_and_supertype_alts 1)
%
        (half_match_with_retry get_hardened_pr 1) 
        pat_tm_a 
  in
  let Full_Tacs = 
    (let num_missing_tacs = length Bs - length Tacs in
     if null Tacs then 
       replicate Id num_missing_tacs 
     if num_missing_tacs < 0 then 
       nthtl (-num_missing_tacs) Tacs 
     else 
       replicate (hd Tacs) num_missing_tacs  @ Tacs
    )
  in
  let fixed_R,just_fixer =
      if direction = `LR` then R,id
      else
        invert_rel R, get_inversion_just R  
  in
  \env inst_tm_a.
      let sub = 
        (matcher inst_tm_a hints env 
         ? failwith (`GenFormulaCondC: ` ^ name ^ `_` ^ direction  ^ `: match failed`)) in
      let do_rewrite =
      ( if control = `always` then true
        if control = `tactics` & null Tacs then true
        if control = `enabler` or control = `quick enabler` then
          conv_enabler env (map (subst sub) active_Bs)
        else
          every
             (zip Bs Full_Tacs)
             (\subgoal,Tac. 
                (Complete Tac 
                   (mk_sequent_using_env
                      env
                      (full_subst sub subgoal)
                   )
                   ; true
                )
                ? false
             )
      )
      in
      if do_rewrite then    
        full_subst sub pat_tm_b,
        fixed_R,
        just_fixer (form_tactic_just 
                     ((Using hints T THENML (map DebugTry Full_Tacs)) 
		      ORELSE BackupT addr sub (map DebugTry Full_Tacs)))
      else    
        failwith (`GenFormulaCondC: ` ^ name ^ `_` ^ direction ^ `: conv not applicable`)
;;

let dummy_conv_enabler (e:env) (t:term list) = true
;;

%
****************************************************************************
Take universally quantified lemma and turn into conversion
****************************************************************************
%

let GenLemmaCondC_aux ctrl enabler direction conseq_num hints Tacs lemma_oid  =
 let lemma_name = name_of_lemma lemma_oid in
    GenFormulaCondC
        lemma_name 
	(main_goal_of_theorem_o lemma_oid)
	(InstGenLemmaByAddr lemma_oid)
	conseq_num
	direction
	ctrl
	enabler
	Tacs
	hints
;;
let GenLemmaCondC ctrl enabler direction conseq_num hints Tacs lemma_name =
  GenLemmaCondC_aux  ctrl enabler direction conseq_num hints Tacs
   (lemma_lookup lemma_name)
;;
let GenLemmaCondC_o ctrl enabler direction conseq_num hints Tacs lemma_obid =
  GenLemmaCondC_aux  ctrl enabler direction conseq_num hints Tacs lemma_obid
;;

% Some closures for common cases. %

let GenLemmaWithThenLC = GenLemmaCondC `tactics` dummy_conv_enabler `LR` ;;
let RevGenLemmaWithThenLC = GenLemmaCondC `tactics` dummy_conv_enabler `RL` ;;

let GenLemmaWithThenLC_o = GenLemmaCondC_o `tactics` dummy_conv_enabler `LR` ;;
let RevGenLemmaWithThenLC_o = GenLemmaCondC_o `tactics` dummy_conv_enabler `RL` ;;

let LemmaThenLC = GenLemmaWithThenLC (-1) [];;
let RevLemmaThenLC = RevGenLemmaWithThenLC (-1) [];;

let LemmaThenLC_o = GenLemmaWithThenLC_o (-1) [];;
let RevLemmaThenLC_o = RevGenLemmaWithThenLC_o (-1) [];;

let GenLemmaC n name = GenLemmaWithThenLC n [] [] name ;;
let GenLemmaC_o n = GenLemmaCondC_aux `tactics` dummy_conv_enabler `LR` n [] [] ;;
let RevGenLemmaC n name = RevGenLemmaWithThenLC n [] [] name ;;
let RevGenLemmaC_o n = GenLemmaCondC_aux `tactics` dummy_conv_enabler `RL` n [] [] ;;

let LemmaWithC hints = GenLemmaWithThenLC (-1) hints [] ;;
let RevLemmaWithC hints = RevGenLemmaWithThenLC (-1) hints [] ;;

let LemmaC = LemmaThenLC [] ;;
let LemmaC_o = GenLemmaCondC_aux `tactics` dummy_conv_enabler `LR` (-1) [] [];;
let RevLemmaC = RevLemmaThenLC [] ;;
let RevLemmaC_o =  GenLemmaCondC_aux `tactics` dummy_conv_enabler `RL` (-1) [] [] ;;

% Usage:

(Rev)GenLemmaWithThenLC n hints Tacs lemma_name
(Rev)LemmaThenLC Tacs lemma_name
(Rev)LemmaC lemma_name

where

n = clause consequent number
hints = list of completing bindings for match.
Tacs = list of tactics to run on conditional subgoals
lemma_name = name of lemma
%


%
****************************************************************************
Take universally quantified hypothesis and turn into conversion
****************************************************************************
Use of this can be rather inefficient. See tactic document.

Also, do free var check to avoid rewriting subterms containing 
shadowing variables. 
%




let GenHypCondC ctrl enabler direction conseq_num hints Tacs hyp_num e' =
  let hn = get_pos_env_num hyp_num e' in 
  let (),hterm,() = nth_assum_in_env hn e' in
  let (),xAs,Bs,aRb = hd (nth_simple_formula conseq_num hterm) in
  let (),a,b = dest_rel_term aRb in
  let vs = diff (free_vars (mk_pair_term a b)) (map fst xAs) in
  let c =GenFormulaCondC 
            `hyp_conv` 
            hterm 
            (InstGenHypByAddr hn)
            conseq_num
            direction
            ctrl
            enabler
            Tacs
            hints
  in \ e t.
    let result = c e t in 
    if env_shadows_vars e hyp_num vs then 
      failwith `GenHypCondC: variables shadowed`
    else 
      result 
;;

%
Usage:

GenHypWithThenLC 
  conseq_num : int
  hints: (var#term) list
  Tacs : tactic list
  hyp_num : int
  env : e
  = 
  c : convn
%

let OldGenHypWithThenLC = GenHypCondC `tactics` dummy_conv_enabler `LR` ;;
let OldRevGenHypWithThenLC = GenHypCondC `tactics` dummy_conv_enabler `RL` ;;

let OldHypThenLC = OldGenHypWithThenLC (-1) [];;
let OldRevHypThenLC = OldRevGenHypWithThenLC (-1) [];;

let OldHypC = OldHypThenLC [] ;;
let OldRevHypC = OldRevHypThenLC [] ;;


% 
We add here some smarter versions of the Hyp* conversions
that don't need the extra env argument. Instead, it is cached
from the first application.
%

let GenHypWithThenLCAux direction conseq_num hints Tacs hyp_num = 
  letref HypC = IdC and cache_good = false
  in
  \e t.
   if cache_good then 
      HypC e t 
   else
   (HypC := GenHypCondC 
              `tactics` dummy_conv_enabler direction 
              conseq_num hints Tacs hyp_num e 
    ; cache_good := true
    ; HypC e t
   )
;;

let GenHypWithThenLC = GenHypWithThenLCAux `LR` ;;  
let RevGenHypWithThenLC = GenHypWithThenLCAux `RL` ;;  

let GenHypC n i    = GenHypWithThenLCAux `LR` n [] [] i ;;
let RevGenHypC n i = GenHypWithThenLCAux `RL` n [] [] i ;;

let HypWithC hints i = GenHypWithThenLCAux `LR` (-1) hints [] i ;;
let RevHypWithC hints i = GenHypWithThenLCAux `RL` (-1) hints [] i ;;

let HypThenLC Ts i = GenHypWithThenLCAux `LR` (-1) [] Ts i ;;
let RevHypThenLC Ts i = GenHypWithThenLCAux `RL` (-1) [] Ts i ;;

let HypC i = GenHypWithThenLCAux `LR` (-1) [] [] i ;;
let RevHypC i = GenHypWithThenLCAux `RL` (-1) [] [] i ;;

%
****************************************************************************
Arithmetic simplification.
****************************************************************************
%

let is_int_arith_expression t =
  member
    (opid_of_term t)
    `` minus add substract multiply divide mod ``
;;


let IntSimpC (e:env) t =
  if not is_int_arith_expression t then failwith `IntSimpC` else
  let t' = arith_simplify_term t in
  if alpha_equal_terms t t' then failwith `IntSimpC`  else
    t', untyped_equal_reln, form_tactic_just Arith
;;


%[
****************************************************************************
Creation of on-the-fly rewrite rules.
****************************************************************************
No matching, but allows arbitrary order or equiv rel transformation
at any point in a clause.
]%

let AssertThenC T aRb (e:env) t = 
  let r,a,b = dest_rel_term aRb in
  if alpha_equal_terms a t then
    b
    ,r
    ,form_tactic_just (T ORELSE AddHiddenLabel `rewrite assertion`)
  else
    failwith `AssertThenC`
;;

let AssertC aRb e t = AssertThenC Fail aRb e t ;;


%
****************************************************************************
Turn module property into conversion.
****************************************************************************
Disabled since not used
%

%
let GenModuleCondC 
  ctrl 
  enabler 
  direction 
  conseq_num 
  hints 
  Tacs 
  mod_hyp_num 
  fieldname
  abs_to_unfold
  e' 
  =
  let v,ModTm,() = nth_assum_in_env mod_hyp_num e'
  in let short_id = apply_alist module_name_alist (opid_of_term ModTm)
  in let pi_id = apply_2d_alist 
                    module_fieldname_2d_alist 
                    (opid_of_term ModTm)
                    fieldname
  
  in let pi_tm = mk_simple_term pi_id [mk_var_term v]
  in let pi_type = get_type_using_env e' pi_tm
  in let InstTac addr sub = 
        Assert pi_type 
        THEN IfLabL
        [`assertion`,UseWitness pi_tm THEN AddHiddenLabel `wf`
        ;`main`,Unfolds abs_to_unfold (-1) 
                THEN InstGenFormulaByAddr addr sub
        ]
  in

  GenFormulaCondC 
            `module_conv` 
            (unfolds abs_to_unfold pi_type)
            InstTac
            conseq_num
            direction
            ctrl
            enabler
            Tacs
            hints
;;


let GenModuleWithThenLC = GenModuleCondC `tactics` dummy_conv_enabler `LR` ;;
let RevGenModuleWithThenLC = GenModuleCondC `tactics` dummy_conv_enabler `RL` 
;;

let ModuleThenLC = GenModuleWithThenLC 1 [];;
let RevModuleThenLC = RevGenModuleWithThenLC 1 [];;

let ModuleC = ModuleThenLC [] ;;
let RevModuleC = RevModuleThenLC [] ;;

%

%
****************************************************************************
Conversions for specific module properties...
****************************************************************************
%

%
let ModuleIdentC n e = 
  GenModuleWithThenLC 1 [] [] n `ident` ``ident_p`` e 
  ORELSEC
  GenModuleWithThenLC 2 [] [] n `ident` ``ident_p`` e 
;;

let ModuleAssocC n e = 
  GenModuleWithThenLC 1 [] [] n `assoc` ``assoc_p`` e ;;

let ModuleRevAssocC n e = 
  RevGenModuleWithThenLC 1 [] [] n `assoc` ``assoc_p`` e ;;

%
