%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
****************************************************************************
****************************************************************************
BASIC-TACTICS.ML
****************************************************************************
****************************************************************************
Utility General purpose tactics, building on atomic tactics.
]%


%[
****************************************************************************
Generalised Cuts.
****************************************************************************
]%
letrec AssertL tlist p =
  if null tlist then Id p
  else
    (Assert (hd tlist) THENL [Id; AssertL (tl tlist)] ) p
;;

%[
****************************************************************************
Equality term manipulations
****************************************************************************
]%

let SwapEquands i p =
  let T,a,b = dest_equal (clause_type i p) in
  let i' = get_pos_hyp_num i p in
  if i' = 0 then
  (Assert (mk_equal_term T b a) 
   THENL [AddHiddenLabel `main`;Eq]
  ) p
  else
  (AssertAtHyp i' (mk_equal_term T b a) 
   THENL [Eq ;Thin (i'+1)]
  ) p
    
;;

let SplitEq b p =
  let T,a,c = dest_equal (concl p) in
  ( AssertL [mk_equal_term T a b;mk_equal_term T b c]
    THENL
    [Id
    ;OnHyp (-1) Thin
    ;Eq
    ]
  ) p
;;


%[
****************************************************************************
Alpha Conversion of clauses
****************************************************************************
See general-tactics.ml for generalization of this. (RenameBVars).
]%

let SubstAlphaEqProp P i p =
  let i' = get_pos_hyp_num i p in
  if not (alpha_equal_terms P (clause_type i' p)) then failwith `SubstAlphaEqProp`
  if i' = 0 then
  ( Assert P THENL [AddHiddenLabel `main`; NthHyp (-1)]) p 
  else
  ( AssertAtHyp i' P THENL [NthHyp i';Thin (i'+1)] ) p
;;
   
%[
****************************************************************************
Repeatfunctions.
****************************************************************************
]%



% careful not to elim product type declarations... %

letrec RepeatAndHD i p =
  let i' = get_pos_hyp_num i p in
  if is_term `and` (unfold_all_but_last_soft_ab (h i' p)) then

  Try ( UnfoldTopSoftAbs i'
        THEN ProdHD null_var null_var i'
        THEN RepeatAndHD (i'+1)
        THEN RepeatAndHD i'
      ) p
  else
    Id p
;;

letrec RepeatOrHD i p =
  Try
  ( OnHyp i (UnfoldTop `or`)
    THEN UnionHD null_var null_var i
    THEN RepeatOrHD i
  ) p
;;

%[
**********************************************************************
Primitive Reduction tactics for equands.
**********************************************************************
Put `reduction condition` label on extra subgoal.
]%

let PrimReduceFirstEquand case p =
  ((let opid = opid_of_term (first_equand (concl p)) in

    if opid = `ind` then

    ( if case = `up` then
        Refine `indReduceUp` [] 
      if case = `base` then
        Refine `indReduceBase` [] 
      if case = `down` then
        Refine `indReduceDown` [] 
      else fail
    )
    if opid = `less` then

    ( if case = `true` then
        Refine `lessReduceTrue` [] 
      if case = `false` then
        Refine `lessReduceFalse` [] 
      else fail
    )
    if opid = `int_eq` then

    ( if case = `true` then
        Refine `int_eqReduceTrue` [] 
      if case = `false` then
        Refine `int_eqReduceFalse` [] 
      else fail
    )
    if opid = `atom_eq` then

    ( if case = `true` then
        Refine `atom_eqReduceTrue` [] 
      if case = `false` then
        Refine `atom_eqReduceFalse` [] 
      else fail
    )

    else
      fail

   ) THENL [Id;AddHiddenLabel `reduction condition`]
  ) p
  ? failwith `PrimReduceFirstEquand`
;;

% beef up to only spawn 1 reduction subgoal if both reductions done. %

let PrimReduceEquands case equands p =
  let T,a,b = dest_equal (concl p) in
  if (equands = [1;2] or equands = [2;1]) & a = b then
  (  let [p';()],() = PrimReduceFirstEquand case p in
     let a' = first_equand (concl p') in
       AssertL [mk_equal_term T a' a';mk_equal_term T a a']
       THENLL 
       [`assertion`,
        [% >> a' = a' in T %
         AddHiddenLabel `main`
        ;% a' = a' in T >> a = a' in T%
         PrimReduceFirstEquand case 
         THEN IfLabL
         [% a' = a' in T >> a' = a' in T%
          `main`,NthHyp (-1)
         ;% a' = a' in T >> Reduction Condition %
          `aux`, Thin (-1) THEN AddHiddenLabel `reduction condition`
         ]
        ]
       ;`main`,
        [%a' = a' in T, a = a' in T >> a = a in T%
         Eq
        ]
       ]
  ) p
  else
 ((if member 1 equands then
     PrimReduceFirstEquand case 
   else
     AddHiddenLabel `main`
  ) 
  THENM
  (if member 2 equands then
     OnConcl SwapEquands 
     THEN PrimReduceFirstEquand case 
     THENM OnConcl SwapEquands 
   else
     Id
  )
 ) p
;;

let PrimReduceMember case = 
  UnfoldConclMemberFor (PrimReduceEquands case [1;2]);;


% 
Works on equands of equal term and element of member term.
Works only on conclusion term.
Always spawns only one conditional subgoal.
If empty list given, then works on both equands.
Doesn't have to be abstraction around equands.
%

let CaseReduceEquands case sel_equands p =
  let c = concl p in
  letrec abs_names t = 
  ( if not is_soft_ab t then []
    else (opid_of_term t). abs_names (unfold_soft_ab t)
  )
  in
  let choose_equand i = if i = 1 then fst (equands c) else snd (equands c) in
  let UnfoldEquand i = Repeat (UnfoldSoftAbAtAddr [i+1] 0) in
  let FoldEquand i = 
    Repeat (RevComputeAtAddrUsing 
            (tag_all_abs_for_fold (abs_names (choose_equand i)))
            [i+1]
            0
           )
  in
  ( SeqOnM 
    ( map UnfoldEquand sel_equands
      @ [PrimReduceEquands case sel_equands]
      @ map FoldEquand sel_equands
    )
  ) p
;;


let CaseReduce case equands = 
  let fixed_equands = if null equands then [1;2] else equands in
    UnfoldConclMemberFor (CaseReduceEquands case fixed_equands)
;;


%[
**********************************************************************
Concl Decomposing.
**********************************************************************
]%

%
Takes an optional list of var names to use for new hypotheses. 
Strips away any trailing guard term. Never fails.
%

let GenUnivFmlaCDAux maybe_split_concl strip_guard vars =
 
  letrec Aux vars p = 
    let c = unfold_soft_abs (concl p) in
    let opid = opid_of_term c in
   %mlbreak opid;%
    if opid = `function` then 
    ( let x,(),() = dest_function c in
      if x = null_var then 
      ( UnfoldTopSoftAbs 0
        THEN FunCD null_var THENL 
        [Aux vars ;Id]
      ) p
      else
      ( UnfoldTopSoftAbs 0
        THEN FunCD (hd vars ? null_var)
        THENL [Aux (tl vars ? []) ;Id]
      ) p
    )
      
    if 
      maybe_split_concl 
      & opid = `product` 
      & fst (dest_product c) = null_var
    then
    ( UnfoldTopSoftAbs 0
      THEN Refine `independent_pairFormation` [] 
      THEN Aux vars 
    ) p
    else
    ( If (\p.strip_guard) (Try (UnfoldTop `guard` 0)) Id
      THEN AddHiddenLabel `main`
    ) p
  in
    Aux vars 
;;   


let GenUnivFmlaCD = GenUnivFmlaCDAux true true  ;;
let UnivFmlaCD = GenUnivFmlaCDAux false true ;;

let RepeatAllImpCD = UnivFmlaCD ;;

let UnivCD = UnivFmlaCD [] ;;
let GenUnivCD = GenUnivFmlaCD [] ;;

let WeakUnivCD = GenUnivFmlaCDAux false false [] ;;
let WeakGenUnivCD = GenUnivFmlaCDAux true false [] ;;


letrec CDToVarThen v T p = 
  let n = num_hyps p in
  (D 0 THENM
   If (\p.var_of_hyp (-1) p = v)
      (T (n+1))
      (CDToVarThen v T)
  ) p
;;



%[
**********************************************************************
Hyp Decomposing.
**********************************************************************
Analog of UnivCD and GenUnivCD

No branching unless split_ors is true.
]%

let GenExistHDAux split_ors i p = 
  letrec Aux i p = 
    let opid = opid_of_term (h i p) in

    if opid = `exists` then
      (D i THEN Aux (i+1)) p
    if (opid = `and` or opid = `cand`) then
      (D i THEN Try (Aux (i+1)) THEN Try (Aux i)) p
    if split_ors & opid = `or` then
      (D i THEN Aux i) p
    else
      Id p
  in
  let i' = get_pos_hyp_num i p
  in
    Aux i' p
;;

let ExistHD i = GenExistHDAux false i ;;
let GenExistHD i = GenExistHDAux true i ;;


%[
**********************************************************************
Combination Decomping
**********************************************************************
]%

let RepD p = 
  (UnivCD THENM TryOnAllHyps RepeatAndHD) p
;;
let GenRepD p = 
  (GenUnivCD THENM TryOnAllHyps RepeatAndHD) p
;;

let ExRepD p = 
  (UnivCD THENM TryOnAllHyps ExistHD) p
;;
let GenExRepD p = 
  (GenUnivCD THENM TryOnAllHyps GenExistHD) p
;;



%[
**********************************************************************
Shifting Hypotheses around
**********************************************************************
]%


% j is dest hyp. i is source hyp.  %

let CopyToHyp j i p =
  ( AssertAtHyp j (h i p)
    THENL
    [NthHyp i ;Id]
  ) p
;;

let CopyToEnd i p = CopyToHyp 0 i p
;;

% NB allow j = 0 to indicate position beyond end of hyp list %

let MoveToHyp j i p =
  let i' = get_pos_hyp_num i p in
  let j' = get_pos_hyp_num j p in
  ( CopyToHyp j' i' 
    THEN if j' > i' or j = 0 then Thin i' else Thin (i'+1)
  ) p
;;

let MoveToEnd i = MoveToHyp 0 i
;;


let CopyHD i
  = CopyToEnd i THEN OnHyp (-1) D 
;;


%
The `hopefully trivial` label will get added to a subgoal
if the hypothesis P is hidden.
%

let CopyToConcl i p =
  % ... #i:v:T ... >> C[v] %
  % ... #i:P ... >> C %
  let v,T = dest_hyp i p in
  let is_decl = is_visible_var v in
  ( Assert (if is_decl then (mk_all_term v T (concl p))
            else (mk_implies_term T (concl p))
           )
    THENL
    [% ... #i:P ... >> P => C %
     % ... #i:v:T ... >> All v:T .C[v] %
     AddHiddenLabel `main`
    ;% ... #i:P ... P => C >> C %
     % ... #i:v:T ... All v:T. C[v] >> C[v] %
     (if is_decl then DTerm (mk_var_term v) (-1) else D (-1))
     THEN (Trivial ORELSE AddHiddenLabel `hopefully trivial`)
    ]
  ) p
;;



% moves all hyps dependent on var declared in hyp i to concl. Move in last
to first order.
%
letrec MoveToConcl i p = 

  (MoveDepHypsToConcl i THENM CopyToConcl i THENM Thin i) p 
  
   and MoveDepHypsToConcl i p =
  
  let i' = get_pos_hyp_num i p in
  let ith_decl.rest_of_decls = nthtl (i'-1) (hyps p) in
  let numbered_decls = zip (upto (i'+1) (i' + length rest_of_decls))
                           rest_of_decls
  in
  let var = var_of_declaration ith_decl in
  let depending_hyps =
    map_omitting_failures 
     (\n,d.if member var (free_vars (type_of_declaration d)) then
             n
           else fail
     )
     numbered_decls
  in
    OnMHyps (rev depending_hyps) MoveToConcl p
;;

%
MoveDepHypsToConclFor (T:int->tactic) (i:int)  

a) moves all hyps depending on i to concl
b) Executes tactic T on hyp i  
c) on main subgoals, moves all dep hyps back again. 

NB: Would require more work to get hyps back in same order as before, so
initially, we don't bother.
%

let MoveDepHypsToConclFor T i p =
  SeqOnM
  [FoldTop `guard` 0
  ;MoveDepHypsToConcl i
  ;T i
  ;UnivCD] % UnivCD strips the guard from the concl %
  p
;;

% Move all hyps beyond i to concl for tactic T %

let MoveEndHypsToConclFor T i p =
  let i' = get_pos_hyp_num i p in
  let hs = rev (upto (i'+1) (num_hyps p)) 
  in
    SeqOnM
    [OnHyps hs MoveToConcl
    ;T i
    ;RepeatMFor (length hs) (D 0)] 
    p
;;

let MoveToConclFor T i p = 
  SeqOnM
  [FoldTop `guard` 0
  ;MoveToConcl i
  ;T 
  ;UnivCD] % UnivCD strips the guard from the concl %
  p
;;
  

%
Replace clause i with t.
%


let ReplaceWith t i p = 
  let i' = get_pos_hyp_num i p 
  in
 ( if i' = 0 then 
     Assert t THEN 
     IfLabL [`main`,AddHiddenLabel `aux`
            ;`assertion`,AddHiddenLabel `main`]
   else
     AssertAtHyp i' t THEN
     IfLabL [`main`,Thin (i'+1);`assertion`,AddHiddenLabel `aux`]
 ) p
;;


   
%[
****************************************************************************
Intuitionistic Decomposition.
****************************************************************************
]%

%
Version of the D in which we don't thin
function hyps.
%

letref intuitionistic_prop_opids 
  = ``implies rev_implies all not``
;;

let ID i p
  = 
  if not (i = 0) 
     & member (opid_of_term (h i p)) intuitionistic_prop_opids
  then
  ( CopyToEnd i
    THEN D (-1)
  ) p
  else
   D i p
;;

   
%[
****************************************************************************
Selecting concl disjunct for decomposing.
****************************************************************************
]%
        

% overrides PrimD for or's in concl %

let OrCD n = UnfoldTop `or` 0 THENM UnionCD n ;;

let DisjunctCD p = 
  let n = get_int_arg `n` p in
  let ds = dest_iterated_or (concl p) in
  let m = length ds in
  if not (2 LE m) then
    failwith `DisjunctCD: concl not disjunct`
  if not ((1 LE n) & (n LE m)) then 
    failwith `DisjunctCD: selector out of range` 
  else
  % select the nth of m disjuncts %
  letrec T n m p = 
    if n = 1 & m = 1 then
      Id p
    if n = 1 then
      OrCD 1 p
    else
      (OrCD 2 THENM T (n-1) (m-1)) p
  in
    T n m p
;;

let DisjunctCD_addition = (\i.if i = 0 then DisjunctCD else fail);;


