%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
***************************************************************************
Compute Sequence functions.
***************************************************************************
]%


%
Definitions:

`compute step'

A compute step is a pair i,t : int#term  where i denotes the type of the step
and t is a possibly tagged nuprl term.
There are various types of compute steps:

i  OPi      operation

0  NOP     do nothing. t must be untagged.
1  REV     invoke reverse direct computation rule with t as the using term.
2  FWD     invoke forward direct computation rule with t as the using term.  
3  POS     invoke integer induction unrolling rule for n > 0 case.
4  ZERO    ...                                         n = 0 
5  NEG     ...                                         n < 0
6  FAKE    ...
7  SQEQUAL invoke rewriting using computation rules, with t as the result

Invoking one of the integer induction rules results in an extra subgoal to 
prove the unrolling condition valid.

Invoking the SQEQUAL rule results in a subgoal for each tag in t.


`compute sequence'

A compute sequence is a list of compute steps. The string of types of compute
For a sequence to be valid the sequence of step types must be in the language 
denoted by the regular exp:

       NOP . ((OP1+ v ... v OP11+).NOP)*

v = or
. = concatenation
+ = Kleene plus. (1 or more times)
* = Kleene star. (0 or more times)

i.e. steps of different types should be separated by a single NOP step, and  
all sequences should start and end with a NOP step.

Compute sequences are used in this package as justifications for rewrites.
If a sequence 

  [op1,t1; op2,t2; ... ; opn,tn]

is a justification for a conversion from term ta to term tb, then we must have

1. op1,t1 = NOP,ta
2. opn,tn = NOP,tb
3. All i. 0<i<n. ti >--opi--> t'(i+1)

where a >--op--> b  if executing the operation indicated by op and the tags in
term a results in term b, and t' is t with all tags removed.
%



%
Using integers for operation types rather than tokens makes prioritising
operations easy. (see merge_justifications)
%

let NOP = 0;;
let REV = 1;;
let FWD = 2;;
let POS = 3;;
let ZERO = 4;;
let NEG = 5;;
let FAKE = 6;;
let SQEQUAL = 7;;

%
Checks enabled:
1. appended sequences are compatible
2. invariant maintained when merging compute sequences.
3. only correct subterms are tagged in uses of reduce rules.
%

letref dc_debug = false ;;

let dc_debug_on () =
  dc_debug := true ; () ;;

let dc_debug_off () =
  dc_debug := false ; () ;;

%[
***************************************************************************

Functions for manipulating compute sequences

***************************************************************************
]%

%
Compute a computation sequence to reverse the rwerite.  For direct
computation, we can just invert the tag of the compute sequence.
With SQEQUAL it is more tricky; the tagged term only contains the
term that we want to rewrite *to*, it doesn't contain the term that
is reritten *from*.  With direct computation, we can compute the *from*
term from the tagged term, but with sqequality we must compute the from term
here.  It _would_ be possible to delay this computation until the rewrite is
applied, but we probably reverse compute sequence far less frequently
that we apply reversed compute sequences.

Implementing this:
   1. We compute on the list from beginning to end
   2. When a string of SQEQUAL rewrites occurs, it must be preceded
      by a NOP.  We save the term at each NOP.
   3. At each SQEQUAL, we perform the rewrite, and then swap
      the tags of the terms.
   4. At the next NOP, we check to make sure we did everything correctly.
   5. On other types of rewrites, we don't maintain the term.

Paul's comment:
Alternatively here we could allow reduce operations to be negated, assuming
at some later time they would be negated again! Is this likely?

%

%
| We need a tactic to swap the equality.
%
letref (Ref_SqEqSymmetry: tactic) p = failwith `SqSwapConcl: tactic not initialized`;;

let reverse_compute_seq input_seq =
  %
  | (op, t, tac): the current compute sequence
  | term: the term computed up to this point
  %
  let reverse_with_carry f arg seq =
      letrec aux arg seq result =
          if null seq then
	      result
	  else
	      let h.t = seq in
	      let v,newarg = f h arg in
		  aux newarg t (v.result)
      in
	  aux arg seq []
  in
  let rev_step (op, t, taclist) term =
      if op = NOP then (NOP, t, taclist), t
      if op = REV then (FWD, t, taclist), t
      if op = FWD then (REV, t, taclist), t
      if op = SQEQUAL then
	  (SQEQUAL, copy_tags term t, map (\tac. Ref_SqEqSymmetry THEN tac) taclist), t
      else 
	  failwith `reverse_compute_seq: cannot reverse op ` ^ (tok_of_int op)
  in
      reverse_with_carry rev_step (mk_simple_term `it` []) input_seq
;;

%
Always discard the NOP step at the end of cs1.  Also discard the NOP step at 
the start of cs2 if the last but one step in cs1 is the same type  as the 
second step in cs2. (be careful though, cs1 might not have a last but one
step and cs2 might not have a second step)
%

let append_compute_seqs cs1 cs2 =
  let all_but_end_cs1,[end_cs1] = split_lastn 1 cs1 in
    if dc_debug & not end_cs1 = hd cs2 then
      failwith `append_compute_seqs: sequences not compatible`
    else
      all_but_end_cs1 
      @
      ( (if (fst o last) all_but_end_cs1 = (fst o hd o tl) cs2 then
           tl cs2
         else fail)
      ?
        cs2
      )
;;



%
Now for a fancy recursive routine for merging compute sequences.

Types:
cs_el = int # term # tactic list
cs    = cs_el list
cs_list =  cs list
cs_el_bunch = int # (term list) # ((tactic list) list)

A cs_list stores a ordered set of compute sequences, those input.
A cs_el_bunch results from merging a set of compatible cs_els, one of the front
of each cs in a cs_list. (a set is considered `compatible' if it has only
one type of operation other than NOP).

the merge_compute_seqs function takes as input a cs_list 
and returns a cs_bunch list.
%

let get_head_ops cs_list =
    map (\s.(fst o first) s) cs_list 
    ? failwith `get_head_dirs`
;;
%
If any input cs has only one cs_el, make sure that it is a NOP.
%

let get_next_ops cs_list =
    map
      (\s.(fst o second) s
          ? if (fst o first) s = NOP then NOP else fail
      )
      cs_list 
;;

%
Pop the compute sequences from the list that match "op"
%
let pop_css_with_head_op_of op cs_list =
  map
    (\cs. if (fst o first) cs = op then tl cs else cs)
    cs_list
;;

%
Pop the compute sequences, whose second op match "op"
%
let pop_css_with_next_op_of op cs_list =
  map 
    (\cs. if (fst o second) cs = op then tl cs else fail
          ? cs)
    cs_list
;;

%
Description of operation:

The input is a list of compute sequences. Head steps are the first ones
in each sequence. Next steps are the second ones in each sequence.
In the event that there is no second step, the next step is considered to
be the same as the head step.


Each OP has a priority indicated by its integer value.
 
1. NOP step. (in output_NOP_bunch function)

Invariant: all head steps of input sequences are of NOP type.

Form output bunch from heads of input compute sequences.
Examine next steps. If all NOP then done. Otherwise find highest priority
next step and pop all input sequences with this type next step. Goto OP
step with the next step type as argument.

2. OP step with operation op. (in output_OP_bunch function)

Invariant: all head steps of input sequences are of either op type or NOP type,
           and at least one is of op type.

Form output bunch from heads of input compute sequences.
Pop all input sequences with head of type op. Examine new head steps on 
input. If any non NOP steps then loop. Otherwise goto NOP step.

It is clear that the invariants will always be satisfied if the input 
sequences are well formed.

Theorem:
Even on ill formed input sequences, the merge function will not
recurse forever.

Proof:
On every function call cycle the input sequences grow monotonically smaller. 
Since the input sequences are finite, we cannot loop forever.

There are two possible function call cycles:

1. output_OP_bunch -> output_NOP_bunch -> output_OP_bunch
2. output_OP_bunch -> output_OP_bunch.

  Case 1:
    in output_NOP_bunch -> output_OP_bunch step, we know that at least
    one input cs has a second element which is an OP. (get_next_ops can't 
    generate a fake next element type which is an OP) Therefore at least
    one input cs will get its head popped.

  Case 2:
    We know at least one input cs has an OP type and therefore at least
    one cs will get its head popped.


In debug mode, we check explicitly that the invariants are maintained. This
will guarantee that the cs_el_bunch list output is well formed.
%




let merge_compute_seqs cs_list =

  letrec output_NOP_bunch input_cs_list =

    % Check invariant (all heads should be NOP) %
    ( if dc_debug then
        let head_ops = get_head_ops input_cs_list in
        if not every head_ops (\d.d = NOP) then 
          failwith `output_NOP_bunch: invariant violated`
        else ()
      else ()
    ) ;

    % This bunch will be a NOP step; no need for a tactic %
    (NOP, map (fst o snd o hd) input_cs_list, [] : (tactic list) list)
    .
    ( % Find the highest priority op %
      let op = accumulate max NOP (get_next_ops input_cs_list) in
	  if op = NOP then
	      % All sequences are empty %
	      []
	  else
	      % Bunch the output on the chosen op %
	      let new_cs_list = pop_css_with_next_op_of op input_cs_list in
		  output_OP_bunch op new_cs_list
    )

  and output_OP_bunch op input_cs_list =

    % Check invariant (that there is at least one sequence labelled by "op" %
    ( if dc_debug then
        let head_ops = get_head_ops input_cs_list in
        if not every head_ops (\d.d = NOP or d = op)
           & not some head_ops ($= op) 
        then
             failwith `output_OP_bunch: invariant violated`
        else ()
      else ()
    ) ;

    % This bunch groups all the sequences, and all the tactics %
    (op, map (fst o snd o hd) input_cs_list, map (snd o snd o hd) input_cs_list)
    .
    (% Pop the sequences just bunched, and check if we can continue with this op %
     let new_cs_list = pop_css_with_head_op_of op input_cs_list in
     let new_ops = get_head_ops new_cs_list in
	 if member op new_ops then
	     output_OP_bunch op new_cs_list
	 else
	     output_NOP_bunch new_cs_list
    )

  in
    output_NOP_bunch cs_list
      
    ?? [`output_OP_bunch: invariant violated`] 
         failwith `merge_compute_seqs: OP invariant violated`
    ?? [`output_NOP_bunch: invariant violated`] 
         failwith `merge_compute_seqs: NOP invariant violated`
    ? failwith `merge_compute_seqs: premature end of input seq`
;;





%
Creating basic compute seqs from tagging functions:
%

let mk_unit_fwd_comp_seq tagger t =
  let tagged_t = tagger t in
  let computed_t = do_computations tagged_t 
  in
    (NOP,t).(FWD,tagged_t).[NOP,computed_t]
;;

let mk_iterated_fwd_comp_seq tagger t =
  letrec aux t =
   (let tagged_t = tagger t in
    let computed_t = do_computations tagged_t in
      (FWD,tagged_t,[]). aux computed_t
   ) ? [NOP,t,[]]
  in
  let seq = aux t in
  if fst (hd seq) = NOP then
    seq
  else
    (NOP,t,[] : tactic list).seq
;;

let Compute_o dir obids t =
 do_indicated_computations (mk_tag_term_o 0 dir obids t)
;;
  
