%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
***************************************************************************
Reduction tactics and Definition folding and unfolding tactics
***************************************************************************
Use these sparingly, for the most part, just in other tactics. The user should
resort to the rewrite computation conversions for most direct computation
tasks.

]%

let ComputeWithTaggedTerm t i p =
  if i=0 then 
    Refine `direct_computation` [mk_term_arg t] p
  else
    let i' = get_pos_hyp_num i p in
    Refine `direct_computation_hypothesis` 
            [mk_int_arg i';mk_term_arg t] p
;;

let RevComputeWithTaggedTerm t i p =
  if i=0 then 
    Refine `reverse_direct_computation` 
           [mk_term_arg t] p
  else
    let i' = get_pos_hyp_num i p in
    Refine `reverse_direct_computation_hypothesis` 
           [mk_int_arg i'; mk_term_arg t] p
;;

let ComputeUsing tagger i p =
  ComputeWithTaggedTerm (tagger (clause_type i p)) i p
;;

let RevComputeUsing tagger i p =
  RevComputeWithTaggedTerm (tagger (clause_type i p)) i p
;;

let ComputeAtAddrUsing tagger addr =
  ComputeUsing (apply_to_addressed_subterm tagger addr)
;;
let RevComputeAtAddrUsing tagger addr =
  RevComputeUsing (apply_to_addressed_subterm tagger addr)
;;


% unrestricted top level compute %

let Compute = ComputeUsing (mk_tag_term 0) ;;

let UnfoldsTop names =
  ComputeUsing (tag_abs_for_unfold names) 
;;

let Compute_o p obidts = ComputeUsing (tag_all_pot p obidts);;
let Compute_except_o p obidts = ComputeUsing (tag_all_pof p obidts);;

let Compute_top_o dir obidts =
  ComputeUsing (mk_tag_term_aux_o 0 dir obidts);;

let UnfoldTop name = UnfoldsTop [name]
;;
let UnfoldTopAb = ComputeUsing tag_any_ab_for_unfold
;;

let Unfolds names =
  ComputeUsing (tag_all_abs_for_unfold names) 
;;

let Unfold name = Unfolds [name] ;;

let RepUnfolds names i = 
  Repeat (Unfolds names i)
;;

% question about when library accessed here. Without p we have to load 
library before tactics.
%

let FoldsTop names p = RevComputeUsing (fold_and_tag_abs names) p;;
let FoldTop name = FoldsTop [name];;

let FoldsTop_o obids p = RevComputeUsing (fold_and_tag_abs_o obids) p;;
let FoldTop_o obid = RevComputeUsing (fold_and_tag_single_ab_o obid);;

let Folds names p = RevComputeUsing (tag_all_abs_for_fold names) p;;
let Fold name = Folds [name];;

let Folds_o obids p = RevComputeUsing (tag_all_abs_for_fold_o obids) p;;
let Fold_o obid = RevComputeUsing (tag_all_single_ab_for_fold_o obid);;

let UnfoldAtAddr =
  ComputeAtAddrUsing tag_any_ab_for_unfold 
;;

let FoldAtAddr name =
  RevComputeAtAddrUsing (fold_and_tag_abs [name]) 
;;

% NB The standard reduction tactics do not recognise redices beneath
soft abstractions. 
%


let ReduceAtAddr address =
  ComputeUsing (tag_addressed_redex address) 
;;


let PrimReduce i =
 Repeat (ComputeUsing tag_all_prim_redices0 i)
;;

let ReduceSOAps =
  ComputeUsing tag_all_so_aps 
;;

let ReduceBelowAddr address i =
  Repeat (ComputeUsing
           (apply_to_addressed_subterm tag_all_prim_redices0 address)
           i
         )
;;

let ReduceEquands i =
  ReduceAtAddr [2] i ORTHEN ReduceAtAddr [3] i
;;

%[
***************************************************************************
Compute Sequence application tactic
***************************************************************************
]%

%
| Squiggle compute tactic can't be defined yet.
%
letref (Ref_SqSubstWithTaggedTerm: term -> int -> tactic) t i =
    failwith `SqSubstWithTaggedTerm tactics is not set`;;

% Do the rewrites %
let ExecuteComputeSeq cs i p
  =
  let i' = get_pos_hyp_num i p in
  let FWD_Using using_term =
        ComputeWithTaggedTerm using_term i' in
  let REV_Using using_term = 
        RevComputeWithTaggedTerm using_term i' in
  let SQEQUAL_Using using_term =
        Ref_SqSubstWithTaggedTerm using_term i' in
  
  % set up function to execute conversion operations %

  letrec DoStep comp_seq p' =
    if null comp_seq then Id p' else
    let (op,using_term,tactics) . rest_of_comp_seq = comp_seq in
    let DoRest = DoStep rest_of_comp_seq in
      (if op = NOP then DoRest
       if op = REV then REV_Using using_term THEN DoRest
       if op = FWD then FWD_Using using_term THEN DoRest
       if op = SQEQUAL then SQEQUAL_Using using_term THENL (tactics @ [DoRest])
       else 
         failwith `ExecuteComputeSeq: cannot execute operation # `
                  ^(tok_of_int op)
      ) p'
  in
    DoStep cs p
;;


%[
***************************************************************************
Labelling tactics 
***************************************************************************
We support two kinds of labelling.
1. For communication between tactics. These don't affect the sequent.
2. For communication of info to user. These do affect the sequent and
   are visible.

Here we support labels with only tokens. Would be easy (and compatible
with what is here now) to add a second optional integer parameter.
]%

let AddLabel label =
	 %mlbreak `label`;%
  OnConcl (RevComputeUsing (\t. tag_term (mk_label_term label t)))
;;

let RemoveLabel = 
  Try (OnConcl (UnfoldTop `label`))
;;


% 
here we show the number label if it exists as a suffix to the token label.
%

let UnhideLabel p =
  let suffix = int_to_tok (number_of_proof p) ? empty_token in
  AddLabel (label_of_proof p ^ suffix) p

;;

let AddAnnotation annot p =
  Id (add_annotation_to_proof annot p)
;;

let KeepingAnnotation T p = 
  (T THEN AddAnnotation (get_annotation_of_proof p)) p
;;

%[
***************************************************************************
Debugging labelling tactics
***************************************************************************
If tac_debug = true then DebugTry T will catch failures of T and show 
failure message in a label attached to the conclusion of the subgoal T failed
on. If tac_debug = false then failures are not caught not by DebugTry

Put hidden labels on the debug subgoals so that they are not confused with
main goals.
]%

letref tac_debug = true ;;


% 
For debugging purposes it's useful to add label on subgoal when it
fails undesirably.
%

let AddDebugLabel lab =
  AddLabel lab THEN AddHiddenLabel `debug`
;;

let DebugTry T p = 
  if tac_debug then 
    (T p ?\x AddDebugLabel x p) 
  else 
    T p 
;;

let (FailWith failure_token : tactic) p = 
  failwith failure_token
;;


%[
***************************************************************************
Soft abstraction reduction.
***************************************************************************
Really much of this stuff could be done by the rewrite package...

Should maybe make names consistent with elsewhere.
i.e use UnfoldTopSoftAb rather than UnfoldSoftAb
   and UnfoldSoftAbs rather than UnfoldAllSoftAbs
]%

let tag_all_soft_abs =
  progressf (sweep_up_map tag_soft_ab) 
;;

let UnfoldAllSoftAbs i = 
  Repeat (ComputeUsing tag_all_soft_abs i)
;;

let UnfoldSoftAbsBelowAddr addr i =
  Repeat (ComputeAtAddrUsing tag_all_soft_abs addr i)
;;
  
let UnfoldSoftAbAtAddr addr i p = 
  if is_soft_ab (get_addressed_subterm addr (clause_type i p)) then
    UnfoldAtAddr addr i p
  else failwith `UnfoldSoftAbAtAddr`
;;

let UnfoldSoftAb = UnfoldSoftAbAtAddr []
;;

let UnfoldTopSoftAb = UnfoldSoftAbAtAddr []
;;
let UnfoldTopSoftAbs i = Repeat (UnfoldSoftAbAtAddr [] i) 
;;


let UnfoldSoftEquands i p =
  ( if is_equal_term (clause_type i p) then
    ( UnfoldSoftAbAtAddr [2] i THEN UnfoldSoftAbAtAddr [3] i) p
    else fail
  ) ?
    failwith `UnfoldSoftEquands`
;;


let HardenEquands i p =
  ( if is_equal_term (clause_type i p) then
      Progress
        (Try (UnfoldSoftAbsBelowAddr [2] i) THEN 
         Try (UnfoldSoftAbAtAddr [3] i)
        ) 
        p
    else fail
  ) ?
    failwith `HardenEquands`
;;




let UnfoldSoftEqType i p =
  ( if is_equal_term (clause_type i p) then
      UnfoldSoftAbAtAddr [1] i p
    else fail
  ) ?
    failwith `UnfoldSoftEqType`
;;



%[
Often we want to decompose some soft hyp, but we don't want to thin
the hyp. In this case, we need to refold the abstractions we unfolded.
Take care to propagate annotations which appear on subgoals.
]%

let ComputeOnlyFor tagger T i p =
  let i' = get_pos_hyp_num i p in
  let cs = mk_iterated_fwd_comp_seq tagger (clause_type i' p) in
  let rev_cs = reverse_compute_seq cs in
  ( ExecuteComputeSeq cs i' 
    THEN T
    THEN KeepingAnnotation (ExecuteComputeSeq rev_cs i')
  ) p
;;

let UnfoldSoftAbsFor = ComputeOnlyFor tag_soft_ab ;;


% Variant on ComputeOnlyfor that works on all listed clauses of the
sequent. Used e.g. with Arith %

let ComputeClausesOnlyFor tagger T is p = 
  let get_i_t_pr i = 
    let i' = get_pos_hyp_num i p in i',clause_type i' p

  in let i_t_prs = map get_i_t_pr is
  in let mk_comp_seq t = 
    let cs = mk_iterated_fwd_comp_seq tagger t in length cs = 1 => fail | cs
  in let i_cs_prs = mapfilter (id # mk_comp_seq) i_t_prs
  in let FwdEx (i,cs) = ExecuteComputeSeq cs i
  in let RevEx (i,cs) = Try (ExecuteComputeSeq (reverse_compute_seq cs) i)
  in
  ( Seq (map FwdEx i_cs_prs) 
    THEN T
    THEN KeepingAnnotation (Seq (map RevEx i_cs_prs))
  ) p
;;




%
Tries doing an unfold member at start.
Fix to only fold afterwards if member before?
%


let UnfoldConclMemberFor T p =
  Progress
  ( Try (OnConcl (UnfoldTop `member`)) 
    THEN T
    THEN Try (KeepingAnnotation (OnConcl (FoldTop `member`)))
  )
  p
;;
%

Makes clausal tactics designed for working on equality terms, work on member 
terms too.
%

let EqToMemberEq T i p =
  if is_member_term (clause_type i p) then
    Progress
    ( UnfoldTopAb i  
      THEN T i
      THEN Try (KeepingAnnotation (FoldTop `member` i))
    )
    p
  else
    T i p
;;

%[
***************************************************************************
Second Order term reduction
***************************************************************************
]%

let SOReduce i p =
  ComputeUsing tag_all_so_redices i p;;
   
let SOUnfolds names i = Unfolds names i THEN SOReduce i ;;
let SOUnfold name = SOUnfolds [name];;



%
;;;;	
;;;;	computation_sequence
;;;;	
;;;;	(term # term) # (unit -> term list)
;;;;
;;;;	will need to lookup by (term # term)
;;;;	
;;;;	note_dependency when found on lookup.
%

letref compseq_ref_state =
  new_ref_state
   `compseq`
   id     % adds are singletons %
   append % expects caller to remove redundant updates? %
   nil
   (nil:((term # term) # (unit -> unit) # (unit -> (bool # term) list)) list)
;;


update_ref_state_view
 (\ (). compseq_ref_state)
 (ref_state_view_list_entry
    (\ (a,b), notef, genf. (ipair_term a b)))
;;   

let compseq_add_data oid data = 
 compseq_ref_state := ref_state_set_data compseq_ref_state [oid, data];;

let compseq_add uobid (ep,genf) =
 check_computation_sequence ep (genf ());
 % TODO : verify the genf generates compseq matching ep %
 reset_ref_environment_data uobid;
 add_ref_environment_data uobid `compseq` compseq_add_data
   (ep, (\ (). note_dependency `reference_environment_update` uobid), genf)
;;

let undeclare_compseq oid =
  (compseq_ref_state := ref_state_remove compseq_ref_state oid; ())
 ? ()
;;
add_ref_environment_undeclare_hook `compseq` undeclare_compseq;;

let compseq_do_updates oid edges oids =
 compseq_ref_state := ref_state_do_updates compseq_ref_state oid oids edges
 ; ()
;;


let lookup_compseq (l,r) =
  let csl = ref_state_get compseq_ref_state
             (current_ref_environment_index `compseq`) in
  let n,g = snd (find (\ (a,b),n,g. alpha_equal_terms a l & alpha_equal_terms b r) csl) in
    n();
    g
;;

let verify_compseq ep = can lookup_compseq ep;;
set_verify_compseq_hook verify_compseq;;

