%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
****************************************************************************
****************************************************************************
EQUALITY-TACTICS
****************************************************************************
****************************************************************************
Tactics for simplifying membership and equality conclusion terms. i.e. 
terms of form 

t in T
t = t' in T

by eliminating the outermost constructors of t t' and T.

The main equality decomposition tactics add hidden labels to the subgoals for
the equalities over the subterms of t and t'.


]%

%[
****************************************************************************
Using Lemmas and Hyps for NonReflEqCD
****************************************************************************
]%
%[
Generates tactic to solve goals of form 

  t1 = t2 in T

where the outermost constructor of t1 and t2 is an abstraction term with opid
`opid'. Makes use of typing lemma of form:

All x1:A1...xn:An . B1 => ... => Bm => v in V

where opid_of_term v = `opid'.


NB: The combinator creation and proving functions need updating to 
leave the order of antecedents and declarations in the typing lemma
unchanged. (Currently all antecedents are grouped together at the end
of the declarations. This might make the combinator typing fail.

]%

letref NonReflEqCD_Autotactic = Id ;;




let mk_wf_comb_typing wf_lemma_name =
  let x_A_prs,Bs,v_in_V =
         dest_simple_formula (raw_main_goal_of_theorem wf_lemma_name) in
  let xs = map fst x_A_prs in
  if not all_distinct xs then failwith 
     `NonReflEqCDByLemma: need all vars distinct in ` ^ wf_lemma_name
  else
  let z = maybe_new_var (mkv `z`) xs in
  let squashed_ants = 
        mk_squash_term (if null Bs then true_term 
                        else mk_iterated_and Bs) in
  let V,v = dest_member v_in_V in
  let v_comb = mk_iterated_lambda (xs @ [z]) v in

  % v_comb = \x1,...,xn, z. v %

  let type_of_v_comb = mk_iterated_function 
                         (x_A_prs @ [null_var,squashed_ants]) V 

  % type_of_v_comb = x1:A1->...->xn:An->Squash(B1 /\ ... /\ Bn)-> V %

  in
    mk_member_term type_of_v_comb v_comb
;;



let ProveOpCombTyping wf_lemma_name p' =
  let xAs,(),() = 
    dest_simple_formula (raw_main_goal_of_theorem wf_lemma_name) 
  in
  let xs = map (mk_var_term o fst) xAs 
  in
  letrec Tac vs p =

    if not null vs then
     % x1:A1,...xi-1:Ai-1 >> 
         (\xi,...,xn,z.v) = ..  in xi:Ai->...-> xn:An -> V %

    ( ( PrimEqCD
        ORELSE
        AddDebugLabel
          `ProveOpCombTyping: could not infer type for function domain`
      )  
      
       % `subterm`: x1:A1,...xi:Ai >> 
         (\xi+1,...,xn,z.v) = ..  in xi+1:Ai+1->...-> xn:An -> V %
       % `eq aux`: x1:A1,...xi-1:Ai-1 >> Ai = Ai in U*   %

       THEN IfLab `subterm` (Tac (tl vs)) (AddHiddenLabel `wf`)
    ) p
    else
    DebugTry % in e.g. the improbable event that lambdaeqcd renamed vars... %
    ( % x1:A1,...xn:An z:Squash(Bs) >> v = v in V %

      BasicSquashHD (-1)
      THEN RepeatAndHD (-1)

      % x1:A1,...xn:An, B1,...,Bm >> v = v in V %

      THEN 
      InstLemma wf_lemma_name xs
              
        % x1:A1,...xn:An, B1,...,Bm >> x1 = x1 in A1
           .         .     .     .    .      .    .
           .         .     .     .    .      .    .
           .         .     .     .    .      .    .
          x1:A1,...xn:An, B1,...,Bm >> xn = xn in An
          x1:A1,...xn:An, B1,...,Bm >> B1
           .         .     .     .    .      
           .         .     .     .    .      
           .         .     .     .    .      
          x1:A1,...xn:An, B1,...,Bm >> Bm
          x1:A1,...xn:An, B1,...,Bm v in V >> v = v in V %

      THEN Trivial 
    ) p
  in
  ( UnfoldTop `member` 0 
    THEN Tac (xs @ [void_term])
    THEN NonReflEqCD_Autotactic 
  ) p'
;;

let RETURN = tok_to_string `\R` ;;




let get_wf_comb_typing wf_lemma_name =
  let comb_lemma_name = `comb_for_` ^ wf_lemma_name in
  ( if is_statement comb_lemma_name then 
      ()
    else if lemma_exists_p comb_lemma_name then
      (raise_error [] ``wf_comb_for visible not`` [itoken_term wf_lemma_name])
    else let stm = add_comb_for_thm (lemma_lookup wf_lemma_name)
                    (create_thm_obj_data
		      (mk_wf_comb_typing wf_lemma_name)
		      (itext_term
	  	        (concatenate_strings 
	                    ["ProveOpCombTyping " 
		            ;"`"
			    ;tok_to_string wf_lemma_name
	                    ;"`"
	                    ]))
		      (tok_to_string comb_lemma_name))

	   in if not (statement_proven_p stm) then
	           display_message 
		     (concatenate_strings
	                [RETURN
	                ;"Warning: proof of "
		        ;(tok_to_string comb_lemma_name) 
	                ;" is incomplete"
	                ;RETURN
	                ]
	             )
	     ; ()
  )
  ;
  let T,t = dest_member (raw_main_goal_of_theorem comb_lemma_name) in
    t,T,comb_lemma_name
;;

  

let AbstractOperatorCombinators 
  args_a args_b v_comb T' =

    % args_a = a1,...,an axiom  %
    % args_b = b1,...,bn axiom  %

    let abstracted_ta = mk_iterated_apply (v_comb . args_a) in
    let abstracted_tb = mk_iterated_apply (v_comb . args_b) in

    %    >> ta = tb in T  %
 
    Assert (mk_equal_term T' abstracted_ta abstracted_tb)
 
    THEN IfLabL
    [`assertion`,
      % >> (\x1...xn,z.v) a1...an axiom = (\x1...xn,z.v) b1...bn axiom in T' %

     AddHiddenLabel `main`
    ;
     `main`,
      % (\x1...xn,z.v) a1...an axiom = (\x1...xn,z.v) b1...bn axiom in T' 
         >>
         ta = tb in T %

      DebugTry
      ( OnHyp (-1)(ComputeWithTaggedTerm
                   (   mk_equal_term
                         T'
                         (mk_tag_term
                            (length args_a)
                            abstracted_ta
                         )
                         (mk_tag_term
                            (length args_b)
                            abstracted_tb
                         )
                   )
                  )
      % v[a1...an axiom/x1...xn,z] = v[b1...bn axiom/x1...xn,z] in T' 
         >>
         ta = tb in T %

       THEN Try (SOReduce (-1))

       % ta = tb in T'  
         >>
         ta = tb in T %
       THEN AddHiddenLabel `inclusion`
      )
    ]
;;

%[
assemble_tacs_for_subterms
  xs = [x1;...;xn]
  v = op(ys1.w1;...;ysm.wm)
  t = op(zs1.u1;...;zsm.um)
  Tac
= 
  Taclist = [T1;...;Tn]

xs is a list of the context vars from the typing formula for op.
We generate a tactic for the subgoal corresponding to each x.

v is the pattern term from the typing formula.  Each subterm wi will be:
1. one of x1...xn. if ysi = []
2. a so ap of one of x1...xn. to the variables ysi if ysi =|= []
3. otherwise.

t is the instance of v which we are using to get the match.
We use t to set the choices of variables for the binding names in subterms
of the instantiated v.

There are 3 possibilities for each variable x.
  1. x is a free (maybe so) var term occurring as an immediate subterm of v.
  2. x is a non null variable not free in any subterm of v.
  3. x is a null variable.


This tactic organises processing of the subgoals resulting from backchaining
through a well formedness lemma. It takes care of doing lambda CD's to get
bindings for subterms into the hypothesis list, and labelling/numbering
those subgoals corresponding to subterms of the term op. (The subgoal 
corresponding to the ith subterm gets label `subterm` and number i.)

NB: if a subterm wi is of form 3 above, then we do not designate any
subgoal corresponding to it. In this case, if all the free variables occur
in subgoals of cases 1 or 2, then there will be no problem in this subgoal
getting properly taken care of. Otherwise, the subgoal corresponding to this
subterm is abandoned to Autotactic.

]%


let assemble_tacs_for_subterms xs v t Tac =
 (let subterm_vars =
    map
      (\t. dest_var t ? fst (dest_so_apply_of_var t) ? null_var)
      (subterms_of_term v)
  in
  let alist = zip 
                subterm_vars 
                (zip 
                  (map fst (bterms_of_term t)) 
                  (upto 1 (length subterm_vars))
                )
  in
    map
    (\x.if is_null_var x then
          Tac
        if not is_bound x alist then 
          AddHiddenLabel `implicit subterm`
        else
        let vars,i = apply_alist alist x 
        in
          RepeatLambdaMemEqCDThen vars (AddHiddenLabelAndNumber `subterm` i)
          ORELSE
          AddDebugLabel
            `EqCDByComb: failed to infer type of function domain`
    )
    xs
 ) ? failwith `assemble_tacs_for_subterms: bad combinator`
;;

letrec get_args_from_ctxt_and_sub 
  (ctxt : (var # term) list) 
  (sub  : (var # term) list)
  = 
  if null ctxt then []
  else
  let (z,A).ctxt' = ctxt
  in 
  if is_null_var z then 
     axiom_term . get_args_from_ctxt_and_sub ctxt' sub
  else
    (snd (hd sub)) . get_args_from_ctxt_and_sub ctxt' (tl sub)
;;



let EqCDByComb type_inf_fun le_vars comb_typing CombTac IncTac =

  let type_of_v_comb,v_comb = dest_member comb_typing 
  in let ctxt,v,V = process_comb_typing comb_typing
  in let ctxt_vars = map fst ctxt
  in let x_A_prs' = filter (\v,().not is_null_var v) ctxt
  in let term_vars = map fst x_A_prs' 
  in let x_A_prs  = map (\v.tok_to_var v, level_exp_type) le_vars @ x_A_prs'
  in let matcher = 
     match_in_context_with_ti_and_ms
       (half_match_with_retry get_hard_and_supertype_alts)
       type_inf_fun 
       x_A_prs 
  in

  % tactic starts here. %

  \match_only_equands p.
  ( let T,ta,tb = dest_equal (concl p) in
    let ta_in_T = mk_member_term T ta in
    let explicit_bindings = (get_sub_arg `sub` p ? []) in
    %    >> ta = tb in T  %

    let full_sub_for_ta =
           matcher 
             (match_only_equands 
              => half_match_member_el 1
              |  half_match_member_with_soft_type 1
             )
             (mk_member_term V v)
             ta_in_T 
             explicit_bindings 
             (env_of_sequent p) 
    in
    let le_sub,reg_and_bvar_sub_for_ta =
      divide_list 
        (\v,t.sub_kind_of_term t = `level-expression`)
        full_sub_for_ta 
    in let sub_for_ta,() =
      divide_list
        (\v,t.sub_kind_of_term t = `regular`)
        reg_and_bvar_sub_for_ta 
    in let (),partial_sub_for_tb =
      divide_list 
        (\v,t.sub_kind_of_term t = `level-expression`)
        (full_match 1 le_vars term_vars v tb) 
    in
    let sub_for_tb =
          map
            (\v,term.
               v, (apply_alist partial_sub_for_tb v ? term))
            sub_for_ta
    in
    let v_comb' = full_subst le_sub v_comb 
    in let type_of_v_comb' = full_subst le_sub type_of_v_comb 

    in let args_a = get_args_from_ctxt_and_sub ctxt sub_for_ta  
    in let args_b = get_args_from_ctxt_and_sub ctxt sub_for_tb  
    in let T' = full_subst full_sub_for_ta V in

    % sub_for_ta = suba = x1->a1,...,xn->an %
    % sub_for_tb = subb = x1->b1,...,xn->bn %

    % suba v = ta  and suba V = T'. %
    % subb v = tb  %

    AbstractOperatorCombinators 
      args_a args_b v_comb' T' 

    THEN IfLabL

    [`main`,
      % >> (\x1...xn,z.v) a1...an axiom = (\x1...xn,z.v) b1...bn axiom in T' %

        RepeatSOApEqCDUsing type_of_v_comb'

      % >> (\x1...xn,z.v) = (\x1...xn,z.v) in x1:A1->...xn:An->Sq(Bs)->V

        >> a1 = b1 in A1
           .    .    .
           .    .    .
           .    .    .
        >> an = bn in (An [a1,...,an-1/x1,...,xn-1])

        >> axiom = axiom in Squash(Bs [a1,...,an/x1,...,xn]) %

        THENL
        (
          CombTac le_sub
          .
          assemble_tacs_for_subterms 
            ctxt_vars 
            v 
            ta
            (SquashEqTypeCD THEN AddHiddenLabel `antecedent`)
        )
    ;`inclusion`,
       % ta = tb in T'  
         >>
         ta = tb in T %

         IncTac (-1) 
    ]
  ) p
;;                          

let NonReflEqCDByLemma wf_lemma_name =

  let comb_typing = mk_wf_comb_typing wf_lemma_name in
  let tac = ProveOpCombTyping wf_lemma_name in
  let CombTac le_sub = 
          AssertTermWithSub comb_typing tac le_sub
          THEN UnfoldTop `member` (-1) THEN NthHyp (-1)
  in
    EqCDByComb 
      get_type_using_env
      (level_vars comb_typing)
      comb_typing 
      CombTac 
      Inclusion
;;

%[
****************************************************************************
Using Lemmas and Hyps for ReflEqCD
****************************************************************************
]%

let ReflEqCDByLemma wf_lemma_name =

  let RigidTac = 
    BackThruGenFormula
       get_type_using_env
       (half_match_with_retry get_hard_and_supertype_alts)
       (half_match_member_with_soft_type 1)
       (main_goal_of_theorem wf_lemma_name)
       (InstGenLemmaByAddr (lemma_lookup wf_lemma_name))
       0
  in
  let FlexTac = 
    BackThruGenFormula
       get_type_using_env
       (half_match_with_retry get_hard_and_supertype_alts)
       (half_match_member_el 1)
       (main_goal_of_theorem wf_lemma_name)
       (InstGenLemmaByAddr (lemma_lookup wf_lemma_name))
       0
  in
  
  % need the following extra preprocessing to figure out which tactics pair up
    with which subgoals.
  %

  let context,[(),v_in_V] = 
        split_lastn 1 (process_simple_formula 
                         (raw_main_goal_of_theorem wf_lemma_name))
  in
  let context_vars = map fst context in
  let [V;v] = subterms_of_term v_in_V in
 
  % ta = ta in T %

  \do_inclusion p.
    let T,ta,() = dest_equal (concl p) in
    ( OnConcl (FoldTop `member`)
      THEN if do_inclusion then FlexTac else RigidTac
      %
        need to be careful here. FlexTac might do inclusion reasoning which
        generates 1 or more extra wf subgoals. However these extra subgoals
        will always fall to the right of the subgoals for the assumptions
        and variable declarations in the lemma.
      %
      THEN_OnFirstL 
        assemble_tacs_for_subterms 
          context_vars 
          v 
          ta 
          Id
    ) p
;;

%[
****************************************************************************
Caches for EqCD
****************************************************************************
]%

%
EqCD caches:

Initialization: 
  reset caches to [].

update:
1. from EqCD use:
  if no entry in cache, add entry for current good thms.
  if entry, use it.
2. from history update.
  search touch history for relevant entries. If any, remove cache entry.

Checking.
  If entry in cache, check that # of items is equal to # of good thms.

We incorporate with the update due to EqCD use, a call to update all
caches, and the cache entry retrieval.
%


let extract_opid_from_wf_obid oid =
  let name = name_of_lemma oid in
  let opid_toks,() = split_at_infix ``_ w f`` (explode name)
  in
    implode opid_toks
;;

% replaced by above 10/97
let extract_opid_from_wf_lemma_name name = 
  let opid_toks,() = split_at_infix ``_ w f`` (string_to_toks name)
  in
    implode opid_toks
;;
%

let get_wf_lemma_names opid = 
  match_names_of_statements
    (concatenate_strings ["^"; tok_to_string opid; "_wf"])
;;

%
let get_wf_lemma_names opid =
  opid_alist_names opid `wf`
;;
%
%
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
ReflEqCD_cache
~~~~~~~~~~~~~~
%

%
letref ReflEqCD_cache  = [] : (tok # (bool -> tactic) list) list ;;
letref ReflEqCD_cache_enable = true ;; 


let init_ReflEqCD_cache (():unit) = 
  ReflEqCD_cache := [] ; ()
;;

let history_update_ReflEqCD_cache objs = 
  if not ReflEqCD_cache_enable or not is_bound `STATEMENT` objs then
    ()
  else
  let oid_status_prs = apply_alist objs `STATEMENT` 
  in let opids = 
       mapfilter 
          (\oid,().extract_opid_from_wf_obid oid) 
          oid_status_prs
  in
    ReflEqCD_cache := 
      accumulate
       (\acc x.
         remove_alist_entry acc x ? acc)
       ReflEqCD_cache
       opids 
   ;
   ()
;;
%
%
let history_update_ReflEqCD_cache objs = 
  if not ReflEqCD_cache_enable or not is_bound `THM` objs then
    ()
  else
  let name_status_prs = apply_alist objs `THM` 
  in let opids = 
       mapfilter 
          (\name,().extract_opid_from_wf_lemma_name name) 
          name_status_prs
  in
    ReflEqCD_cache := 
      accumulate
       (\acc x.
         remove_alist_entry acc x ? acc)
       ReflEqCD_cache
       opids
   ;
   ()
;;
%
%
let check_ReflEqCD_cache (():unit) = 
  map
    (\opid,tacs.
     if length tacs = length (get_wf_lemma_names opid) then
       ()
     else
       display_message
       (concatenate_strings
          ["ReflEqCD_cache: bad entry at: ";tok_to_string opid]
       )
    )
    ReflEqCD_cache
 ;
 ()
;;

add_history_update_funs
  `ReflEqCD`
  init_ReflEqCD_cache
  history_update_ReflEqCD_cache
  check_ReflEqCD_cache
;;

%

letref reverse_refl_caches = false;;

let ReflEqCD_tacs_lookup id =
 let tacs =  tac_cache_lookup_wf
	       (\(). ReflEqCD_cache)
	       (\c. ReflEqCD_cache := c)
	       (\index. map ReflEqCDByLemma)
	       id
   in
   if (length tacs) > 1 then tty_print ("Mutliple ReflEqCD wfs for id " J (tok_to_string id));
   if reverse_refl_caches then (rev tacs) else tacs 	       
;;

let get_ReflEqCD_tacs = ReflEqCD_tacs_lookup;;

% LAL original
let get_ReflEqCD_tacs id =
  if ReflEqCD_cache_enable then
    (update_all_caches_from_history ()
     ;
     (apply_alist ReflEqCD_cache id
      ?
      let names = get_wf_lemma_names id in
      let Tacs = map ReflEqCDByLemma names in
        ReflEqCD_cache := add_to_alist_start id Tacs ReflEqCD_cache 
        ; 
        Tacs
     )
    ) 
  else
    map ReflEqCDByLemma (get_wf_lemma_names id) 
;;
%%
let get_ReflEqCD_tacs id =
  if ReflEqCD_cache_enable then
    (update_all_caches_from_history ()
     ;
     (apply_alist ReflEqCD_cache id
      ?
      let names = get_wf_lemma_names id in
      let Tacs = map ReflEqCDByLemma names in
	(ReflEqCD_cache := add_to_alist_start id Tacs ReflEqCD_cache; ())
        ; 
        Tacs
     )
    ) 
  else
    map ReflEqCDByLemma (get_wf_lemma_names id) 
;;
%%
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
NonReflEqCD_cache
~~~~~~~~~~~~~~
%
%
letref NonReflEqCD_cache = [] : (tok # (bool -> tactic) list) list ;;
letref NonReflEqCD_cache_enable = true ;;


let init_NonReflEqCD_cache (():unit) = 
  NonReflEqCD_cache := [] ; ()
;;

let history_update_NonReflEqCD_cache objs = 
  if not NonReflEqCD_cache_enable or not is_bound `STATEMENT` objs then
    ()
  else
  let oid_status_prs = apply_alist objs `STATEMENT` 
  in let opids = 
       mapfilter 
          (\oid,().extract_opid_from_wf_obid oid) 
          oid_status_prs
  in
    NonReflEqCD_cache := 
      accumulate
       (\acc x.
         remove_alist_entry acc x ? acc)
       NonReflEqCD_cache
       opids
   ;
   ()
;;

let check_NonReflEqCD_cache (():unit) = 
  map
    (\opid,tacs.
     if length tacs = length (get_wf_lemma_names opid) then
       ()
     else
       display_message
       (concatenate_strings
          ["NonReflEqCD_cache: bad entry at: ";tok_to_string opid]
       )
    )
    ReflEqCD_cache
 ;
 ()
;;

add_history_update_funs
  `NonReflEqCD`
  init_NonReflEqCD_cache
  history_update_NonReflEqCD_cache
  check_NonReflEqCD_cache
;;

let get_NonReflEqCD_tacs id =
  if NonReflEqCD_cache_enable then
    (update_all_caches_from_history ()
     ;
     (apply_alist NonReflEqCD_cache id
      ?
      let names = get_wf_lemma_names id in
      let Tacs = map NonReflEqCDByLemma names in
        NonReflEqCD_cache := add_to_alist_start id Tacs NonReflEqCD_cache 
        ; 
        Tacs
     )
    ) 
  else
    map NonReflEqCDByLemma (get_wf_lemma_names id) 
;;
%


let NonReflEqCD_tacs_lookup id =
 let tacs = tac_cache_lookup_wf
	     (\(). NonReflEqCD_cache)
	     (\c. NonReflEqCD_cache := c)
	     (\index. map_omitting_failures NonReflEqCDByLemma)
	     id
    in
    if (length tacs) > 1 then tty_print ("Mutliple NonReflEqCD wfs for id " J (tok_to_string id));
    if reverse_refl_caches then (rev tacs) else tacs
;;

let get_NonReflEqCD_tacs = NonReflEqCD_tacs_lookup;;

%[
****************************************************************************
EqCDByLemma
****************************************************************************

We allow for there being more than one typing lemma per opid.

Inclusion reasoning is tried on last appropriate tactic in cache - 
usually corresponding to the last appropriate wf lemma in library.

This is on the assumption that later wf lemmas will be stronger,
since they probably will make use of earlier lemmas. Inclusion 
reasoning stands most chance of success if tried after using the
strongest available lemma.

NB: This could be dangerous in the following situation:
LIBRARY:
...
foo_wf1
...
lemmas-a
...
foo_wf2
...
lemmas-b

lemmas-a would try to use foo_wf2 when not doing a load-fully, even
if the intention was that they only use foo_wf1.

This problem would be solved by some library order checking.

27th Sept 94:
~~~~~~~~~~~~~
Have two wf lemmas:

op_wf:  All x:S  op(x) in T
op_wf2: All x:S'  op(x) in T'

where S' c S and T' c T.  ("c" = proper subtype)

and goal t in S |- op(t) in T'' 

where T'' is computationally equal to T.

Current Inclusion strategy, tries 
1. matching exactly on op_wf2 then op_wf. both fail. 
2. matching with inclusion on op_wf2 this succeeds, since T' c T, but
now get insoluble subgoal  |- t in S'.

Solutions:
 a) try all inclusion strategies first looking only for equalities.
 b) when trying each lemma for inclusion, try *all* of them
    in first-last order, and use first which succeeds.
    (relies on inclusion never failing if inclusion predicate cannot be 
     proven). This hasn't always been the case with Inclusion. It used
     to just spinoff a subgoal with label `inclusion?`

a) means writing new inclusion functions.
   Tried quick hack which weakened inclusion, but defeated by caching
   strategy which remembered goal proven with stronger inclusion! Can't be
   bothered to make caching sensitive to desire for Inclusion to not try too
   hard.
b) Will try it out. 
]%



let EqCDByLemma match_only_equands p =

  let T,a,b = dest_equal (concl p) in
  let ida = opid_of_term a in
  let idb = opid_of_term b in
  if not ida = idb then failwith `EqCDByLemma: equands must have same opid`
  else
  let Tacs =
  %  rev %
      ( if alpha_equal_terms a b then
          get_ReflEqCD_tacs ida
        else
          get_NonReflEqCD_tacs ida
      )
  in
  if null Tacs then failwith `EqCDByLemma no applicable lemmas`
  if match_only_equands then
    first_value (\T.T true p) Tacs
  else
    first_value (\T.T false p) Tacs
;;


let PrimEqCDWithInc p = 
  let T,a,b = dest_equal (concl p) in
  let T' = get_type p a 
           ? get_type p b 
           ? failwith `PrimEqCDWithInc: no alternative type inferred`
  in
  if alpha_equal_terms T T' then 
           failwith `PrimEqCDWithInc: type inferred is same`
  else
  ( Assert (mk_equal_term T' a b) 
    THEN IfLab `assertion` PrimEqCD (Inclusion (-1))
  ) p
;;

% 

Operation on goal a = b in T where opa = opb . 

Need case for termof terms.
%

% 
Option was included in V3, but not in V4 since it 
introduces a quadratic factor into type checking types.
However, there are cases when without it, EqCD is too
eager.

(e.g.

1. have wf lemma: All s:Sigi. con(s) in Ui
2. Sigi is not a subtype of Sigj when for level expressions i,j i <= j

On goal 

   s:Sigi |- con(s) in Ui'

MemCD without cumulativity gives false subgoal:

   s:Sigi |- s in Sigi'
  
%


letref do_cumulativity_in_EqCD = true
;;

% 
Probably have to much functionality here. Inclusion and
Cumulativity reasoning should be lifted out.
%

let EqCDAux1 strength =
  letrec Aux p =
  ( let T,a,b = dest_equal (concl p) in

    if not equal_operators_of_terms a b then 
      failwith `EqCD: outermost equand operators must be equal`

    if is_var_term a then

    (Inclusion (get_decl_num (dest_var a) p) p )

    if is_abstraction_term a then
    ( EqCDByLemma false 
      ORELSE EqCDByLemma true 
      ORELSE (UnfoldSoftEquands 0 THEN Aux)
      ORELSE FailWith `EqCDAux1` 
    ) p 

    else
    ( EqCDByLemma false 
      ORELSE PrimEqCD 
      ORELSE PrimEqCDWithInc 
      ORELSE If (\p.strength = `strong`) RecPrimEqCD Fail
      ORELSE LemmaWitnessEqCD
      ORELSE (UnfoldSoftEqType 0 THEN Aux)
      ORELSE FailWith `EqCDAux1`
    ) p
  )
  in
    (if do_cumulativity_in_EqCD then
       Try CumulativityByTypeInf THEN Aux
     else
       Aux
    )
;;

% Use tokens to make it easy to identify entries %

letref EqCD_additions = [] : (tok # tactic) list ;;

let update_EqCD_additions lab Tac = 
  EqCD_additions := update_alist EqCD_additions lab Tac; ();;


%PERF avoid this map%
let EqCDAux strength =
  First (map snd EqCD_additions)
  ORELSE EqCDAux1 strength
;;


let EqCD p = EqCDAux `regular` p;;
let StrongEqCD p = EqCDAux `strong` p;;
let MemberEqCD p = EqToMemberEq (\i.EqCD) 0 p ;;
let StrongMemberEqCD p = EqToMemberEq (\i.StrongEqCD) 0 p ;;

let MemCD p = MemberEqCD p ;;


%[
1. Often not a good idea to invoke primitive rules for EqCD on recursive
   terms. Need to inhibit this, but still allow lemmas to be used for
   abstractions of recursive primitive terms.

]% 



%[ 
If goal is 

op(xs1.a1;...;xsn.an) = op(ys1.b1;...;ysn.bn) in S 

and Ts= [T1;...;Tn] 

then Ti is run on the subgoal:

x1i,...xki >> ai = bi[xsi/ysi] in Si

There should (better!) be one T for each subterm. Subterm subgoals
should be identifed by a `subterm` token label and a number label
indicating which subterm. If the order of the subgoals corresponds to the
order of the subterms, the number labels can be omitted.

As yet, the tactics which add the numbers don't take advantage of this
default feature. (it results in a small - maybe negligible performance
improvement).

]%

let EqCDThenL Ts =
  EqCD THENLL [`subterm`,Ts]
;;


%[ 
We have a number of options here. They include doing one or more of the
following:

1. Unfolding 1 level of soft abstractions around equands
2. Unfolding 1 level of soft abstractions around eq type.
3. if equands are primitive, do primitive rule.
4. try to match equands and type against lemma. 
   good for prim and ab equands when opids are equal.
5. try to match just equands against lemma. Use inclusion reasoning to fix 
   type.missing cases for rec and quotient types, and for unsquashing new hyps.
   good for prim and ab equands when opids are equal.
6. if eq type is set type. (under soft ab?) then can destruct it.
7. can do cumulativity reasoning on type if is Uterm.
8. invoking inclusion if equands are same vars.
9. invoking Eq decision proc. good even when equand opids are not equal.

Not clear what order we should try to do things in. Strategy for now is
to have EqCD by default to relatively little. If user wants it to do more
for certain equands then he/she must specifically define appropriate 
actions. The aim should be for the basic package to provide convenient 
building blocks.


Not handled yet: termof, quotient and rec terms.

Factors to discriminate on:

If equands are prim / abstract.
if equands are soft / hard abs.
if opids of equands are same or not.
if equands are prim canon or non canon.
if equand   


Strength reasoning:


The IfLab `subterm` is especially important to prevent infinite rec on
inclusion? subgoals produced by inclusion.
]%

let RepeatReflEqCD strength p =
  let T,t,t' = dest_equal (concl p) in
  if alpha_equal_terms t t' then
   (EqCDAux strength
    THEN Repeat (IfLab `subterm` (EqCDAux strength) Id)
   ) p
  else
    failwith `RepeatReflEqCD: equand opids not the same`
;;

let RepeatMemberReflEqCD strength p =
  let MemEqCD = EqToMemberEq (\i.EqCDAux strength) 0 in
  let T,t,t' = dest_member_or_equal (concl p) in
  if alpha_equal_terms t t' then
   (MemEqCD 
    THEN Repeat (IfLab `subterm` MemEqCD Id)
   ) p
  else
    failwith `RepeatMemberReflEqCD: equand opids not the same`
;;


%
let RepeatMemberReflEqCD strength =
  EqToMemberEq (\i.RepeatReflEqCD strength) 0
;;
%

%[
****************************************************************************
Special Purpose Equality tactics.
****************************************************************************
Needs. 
  1. When alternate rule must be invoked.
  2. When soft abstractions alter subterm structure or order. 
     (relevant when rewriting)
  3. When want to fold prop abstractions on subterms.
]%

% NB: iff and rev_implies are handled by explicit lemmas %

let LogicEqCD p = 
  let PropTac p'= KeepingLabel (FoldAtAddr `prop` [1] 0) p'
  in let EqCD p'= EqCDAux1 `regular` p'
  in let c = concl p
  in
  if is_term `prop` (eq_type c) then
  ( let opid = opid_of_term (first_equand c)
    in 
    if member opid ``and or implies`` then
      EqCD THENLL [`subterm`,[PropTac;PropTac]]
    if member opid ``exists all`` then
      EqCD THENLL [`subterm`,[Id;PropTac]]
    else failwith `LogicEqCD`
  ) p
  else
    failwith `LogicEqCD`
;;

let CAndEqCD p =
  if is_term `cand` (first_equand (concl p)) then
  ( UnfoldAtAddr [2] 0
    THEN UnfoldAtAddr [3] 0
    THEN Try (UnfoldAtAddr [1] 0)
    THEN Refine `productEquality` (mk_new_var_args [[]] p) 
    THENL (map (AddHiddenLabelAndNumber `subterm`) [1;2])
  ) p
  else
    failwith `CAndEqCD`
;;

% a f b == (f a) b %

let InfixApEqCD p = 
  if is_term `infix_ap` (first_equand (concl p)) then
  ( UnfoldAtAddr [2] 0
    THEN UnfoldAtAddr [3] 0
    THEN PrimEqCD
    THENLL 
    [`subterm`,
     [PrimEqCD THENLL 
       [`subterm`,
         [ AddHiddenLabelAndNumber `subterm` 1
         ; AddHiddenLabelAndNumber `subterm` 2
         ]
       ]
     ;AddHiddenLabelAndNumber `subterm` 3
     ]
    ]
  ) p
  else
    failwith `InfixApEqCD`
;;

update_EqCD_additions `LogicEqCD` LogicEqCD ;;
update_EqCD_additions `InfixApEqCD` InfixApEqCD ;;
update_EqCD_additions `CAndEqCD` CAndEqCD ;;


%[
****************************************************************************
Tactics for working on equalities and member terms in hyp list.
****************************************************************************
PrimEqHD handles:
  1. pairs in product types 
  2. functions in function types
]%

let EqHDPart ctl i p =
  let i' = get_pos_hyp_num i p in

  letrec Aux p =
    ( PrimEqHD ctl i'
      ORELSE (UnfoldAtAddr [1] i' THEN Aux)
    ) p
  in
    Aux p
;;

let EqHD i = EqHDPart [] i
;;

%NB This is not sufficient, since EqHD can generate more than one equality
    hypothesis
%
let MemberEqHD i p = EqToMemberEq EqHD i p
;;

let MemHD i p = MemberEqHD i p
;;
%[
****************************************************************************
Tactics for working on equalities and member terms in any clause
****************************************************************************
]%

let EqD i =
  if i = 0 then EqCD else EqHD i
;;

let MemberEqD i = EqToMemberEq EqD i ;;
let MemD i = EqToMemberEq EqD i ;;
%[
****************************************************************************
Function Extensionality
****************************************************************************
]%

%
| Maybe want to put a call to get_type in if terms is null, rather than
| assuming function type is type in equality term.
| 
| For normal dependent functions:
| >> f = g in x:A->B
|   By ExtWith [v] ['y:C->D';'z:E->F']
| 
| 1. v:A >> f v = g v in B[v/x]
| 2. >> A = A in U{?}   (`wf` label)
| 3. >> f = f in y:C->D (`wf` label)
| 4. >> g = g in z:E->F (`wf` label)
| 
| If z:E->F is omitted but y:C->D is not, y:C->D is used instead of z:E->F.
| If y:C->D and z:E->F are omitted, then 
| 1. if f (g) is lambda term, then Void->Void is used 
| 2. if not, then type x:A->B is used
| 
| Subgoal 4 is omitted if f = g and at most one using type is supplied.
| 
| For very-dependent functions:
| >> f = g in {h | x:A -> B}
|   By With 'R' (ExtWith [v] ['{F1 | y:C -> D}'; '{F2 | z:E -> F}'])
| 
| 1. >> A = A in U{?}			(`wf` label)
| 2. >> R = R in A -> A -> U{?}		(`wf` label)
| 3. >> !WellFnd{?}(A; u, v. R u v)	(`aux` label)
| 4. v:A >> f v = g v in B[v, f/x, h]
| 5. >> f = f in {F1 | y:C -> D}		(`wf` label)
| 6. >> g = g in {F2 | z:E -> F}		(`wf` label)
| 
| If F2 is omitted, but F1 is not, then F1 is used for F2.
| If both F1 and F2 are omitted, then
|     1. If a relation argument is present (using "With") then
|         a. if both f and g are lambda terms, then F1 = F2 = Void -> Void
|         b. otherwise, F1 = F2 = {h | x:A -> B}
|     2. Otherwise, F1 = F2 = {h | x:A -> B}
| If F1 = F2 = {h | x:A -> B}, then the rule considerably simplifies:
| 
| >> f = g in {h | x:A -> B}
| BY ExtWith [v] [...]
| 1. v:A >> f v = g v in B[v, f/x, h]
| 2. >> f = f in {h | x:A -> B}
| 3. >> g = g in {h | x:A -> B}
|
| If f = g and F1 = F2, then final subgoal is omitted in both cases
%

% This tactic deals with \x.x in void->void subgoals. %
letref EqExtWithLambdaTac = AddHiddenLabel `wf`;;

% This tactic sweeps up subgoals of \x.t in { f | x:Void -> Void } %
let EqExtWithFunctionVoid = mk_function_term null_var void_term void_term;;
let EqExtWithRFunctionVoid = mk_rfunction_term `f' `x' void_term void_term;;

%
| This tactic will prove the goals of the form:
|    \x.t = \y.t' in { f | x:Void -> Void }
| It is set in rfunction.ml
%
letref EqExtWithRLambdaTac = AddHiddenLabel `wf`;;

% Perform the extensionality refinement %
let EqExtWith vars terms p =
  let n = num_hyps p 
  in let T,f,g = dest_equal (concl p)
  in let WFT = (if is_rfunction_term T then
		    Fold `member` 0 THEN AddHiddenLabel `wf`
		else
		    AddHiddenLabel `wf`)
  in let relation = get_term_arg `t1` p ? nth 3 terms ? void_term
  in let mk_type h =
     if (is_lambda_term h) & not (alpha_equal_terms relation void_term) then
	 if is_rfunction_term T then
	     EqExtWithRFunctionVoid, EqExtWithRLambdaTac
	 else
	     EqExtWithFunctionVoid, EqExtWithLambdaTac
     else
	 T, WFT
  in let (F1, LambdaTacf), (F2, LambdaTacg) = 
      if null terms then
	  mk_type f, mk_type g
      if length terms = 1 then
	  (hd terms, WFT), (hd terms, WFT)
      else
	  (hd terms, WFT), (nth 2 terms, WFT)
  in let PreTac, PostTacf, PostTacg, ThinTac =
      if alpha_equal_terms F1 F2 & alpha_equal_terms f g then
	  (\T.Assert (mk_equal_term F1 f f) THEN IfLab `main` T LambdaTacf)
	  , Hypothesis
	  , Hypothesis
	  , Thin (n+1)
      else
	  I, LambdaTacf, LambdaTacg, Id
  in
      if is_function_term T then
	  % Normal dependent function %
	  let A = hd (subterms_of_term T) in
	  let le_arg, WFTac = mk_le_arg_and_wf_tac p A in
	  let new_var_args = mk_new_var_args [vars @ [mkv `x`]] p in
	      PreTac
	      (Refine `functionExtensionality`
	          (le_arg . mk_term_arg F1 . mk_term_arg F2 . new_var_args)
	      THENL
	          [ThinTac
		  ;ThinTac THEN AddHiddenLabel `wf`
		  ;PostTacf
		  ;PostTacg
		  ]) p
      else if is_rfunction_term T then
	  (if alpha_equal_terms F1 T & alpha_equal_terms F2 T then
	      % Simple case of the very-dependent function %
	      let new_var_args = mk_new_var_args [vars @ [mkv `x`]] p in
		  PreTac
		      (Refine `rfunction_simpleExtensionality` new_var_args
		       THENL
		           [ThinTac
			   ;PostTacf
			   ;PostTacg
			   ]) p
	  else if alpha_equal_terms relation void_term then
	      % The general case of a very-dependent function must have a relation %
	      failwith (string_to_tok
"EqExtWith:\n\t\
for proving extensionality of this very-dependent function,\n\t\
a well-founded order on the domain is needed.\n\t\
Use the tactical \"With\" to supply the relation")
	  else
	      % General very-dependent function %
	      let (), (), A, () = dest_rfunction T in
	      let le_arg, WFTac = mk_le_arg_and_wf_tac p A in
	      letrec merge l1 l2 =
	          if null l1 or null l2 then
		      l2
		  else
		      let h1.t1 = l1 in
		      let h2.t2 = l2 in
			  (h1 . h2).(merge t1 t2)
              in
	      let [y; u; v] = mk_new_var_args (merge vars [[`x']; [`u']; [`v']]) p
	      in
		  PreTac
		    (Refine `rfunctionExtensionality`
		       [le_arg
		       ;mk_term_arg F1
		       ;mk_term_arg F2
		       ;u
		       ;v
		       ;mk_term_arg relation
		       ;y
		       ]
		     THENL
		       [ThinTac THEN WFT
		       ;ThinTac THEN WFT
		       ;ThinTac THEN AddHiddenLabel `aux`
		       ;ThinTac
		       ;PostTacf
		       ;PostTacg
		       ]) p)
      else
	  failwith `EqExtWith: not a function`
;;


let ExtWith vs ts = 
  Repeat (UnfoldSoftAbAtAddr [1] 0) 
  THEN EqToMemberEq (\i.EqExtWith vs ts) 0 ;;

let Ext p = ExtWith [get_optional_var_arg `v1` p] [] p
;;

%[
****************************************************************************
Widening and Narrowing Types
****************************************************************************
Work on type of member and equality terms in hyps or concl.
]%
  
%
 ... i. a = b in S ... |- ... 

BY EqTypeHWiden T i 

`main` ... i. a = b in T ... |- ... 
`inclusion` ... i. a = b in S ... |- S c= T 
%


let EqTypeHWiden T i p = 
  let i' = get_pos_hyp_num i p in
  let S,a,b = dest_equal (h i' p) in

  % ... i. a = b in S ... |- ... %
  
 (AssertAtHyp i' (mk_equal_term T a b)
  THEN IfLabL
  [`main`,
  % ... i. a = b in S, a = b in T ... |- ... %
   Thin (i' + 1)
  ;`assertion`,
  % ... i. a = b in S... |-  a = b in T %
   EqTypeCNarrow S THENM 
    % ... i. a = b in S... |-  a = b in S %
    NthHyp i'
  ]
 ) p
;;

% provide access to alternate version of tactic 
  that kicks out wf goal for 2nd equand when passed
  Sel 2 argument.
%
   
let EqTypeCClassWiden1 T p = 
  let swap_p = get_int_arg `n` p = 2 ? false in
  if swap_p then
  ( SwapEquands 0
    THEN EqTypeCClassWiden T
    THENM SwapEquands 0
  ) p
  else
    EqTypeCClassWiden T p
;;
%
 ... i. a = b in S ... |- ... 

BY EqTypeHClassWiden T i 

`main` ... i. a = b in T ... |- ... 
`inclusion` ... i. a = b in S ... |- S c={c} T 
%


let EqTypeHClassWiden T i p = 
  let i' = get_pos_hyp_num i p in
  let S,a,b = dest_equal (h i' p) in

  % ... i. a = b in S ... |- ... %
  
 (AssertAtHyp i' (mk_equal_term T a b)
  THEN IfLabL
  [`main`,
  % ... i. a = b in S, a = b in T ... |- ... %
   Thin (i' + 1)
  ;`assertion`,
  % ... i. a = b in S... |-  a = b in T %
   EqTypeCClassNarrow S THENM 
    % ... i. a = b in S... |-  a = b in S %
    NthHyp i'
  ]
 ) p
;;

%
 ... i. a = b in T ... |- ... 

BY EqTypeHClassNarrow S i 

`main` ... i. a = b in S ... |- ... 
`inclusion` ... i. a = b in T ... |- S c={c} T 
`wf`  ... i. a = b in S ... |- a in S
%


let EqTypeHClassNarrow S i p = 
  let i' = get_pos_hyp_num i p in
  let T,a,b = dest_equal (h i' p) in

  % ... i. a = b in T ... |- ... %
  
 (AssertAtHyp i' (mk_equal_term S a b)
  THEN IfLabL
  [`main`,
  % ... i. a = b in S, a = b in T ... |- ... %
   Thin (i' + 1)
  ;`assertion`,
  % ... i. a = b in T... |-  a = b in S %
   EqTypeCClassWiden1 T THENM 
    % ... i. a = b in T... |-  a = b in T %
    NthHyp i'
  ]
 ) p
;;

let EqTypeWiden T i = if i = 0 then Fail else EqTypeHWiden T i ;;
let EqTypeNarrow S i = if i = 0 then EqTypeCNarrow S else Fail ;;

let EqTypeClWiden T i = 
  if i = 0 then EqTypeCClassWiden1 T else EqTypeHClassWiden T i ;;

let EqTypeClNarrow S i = 
  if i = 0 then EqTypeCClassNarrow S else EqTypeHClassNarrow S i;;

% Drop the Mem and Eq suffices - just a hassle... %

let TypeWiden T = EqToMemberEq (EqTypeWiden T);;
let TypeNarrow S = EqToMemberEq (EqTypeNarrow S);;
let TypeClWiden T = EqToMemberEq (EqTypeClWiden T);;
let TypeClNarrow S = EqToMemberEq (EqTypeClNarrow S);;
